/*
 * Copyright 2005,2006,2011 Sony Corporation
 * Copyright 2018 Sony Imaging Products and Solutions Incorporated.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/serio.h>
#include <linux/tty.h>

// TENTATIVE FROM
#define N_USR1		17
#define N_USR2		18
// TENTATIVE TO

#include <linux/wait.h>
#include <linux/console.h>
#include <linux/poll.h>
#include <linux/version.h>
#include <linux/kthread.h>
#include <linux/udif/delay.h>

#ifndef _LINUX_SCHED_H
#include <linux/sched.h>
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 21)
#include <linux/usb/ch9.h>
#else
#include <linux/usb_ch9.h>
#endif

#include <linux/device.h>

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 29)
#include <linux/usb/gadget.h>
#else
#include <linux/usb_gadget.h>
#endif

#include <linux/byteorder/generic.h>

#include <linux/usb/gcore/usb_event.h>
#include <linux/usb/gcore/usb_gadgetcore.h>




#include <linux/usb/specific_gadget/usbg_cmn.h>
#include <linux/usb/specific_gadget/usbg_type.h>
#include <linux/usb/specific_gadget/usbg_sen.h>

#include <linux/usb/specific_gadget/usb_putpid_config.h>

#include <linux/usb/specific_gadget/usb_kdbg.h>
#define PLOG(fmt, args...) {printk("[usbter %s][%3d]"fmt, __FUNCTION__,current->pid,##args); }

/*
 * Internal flag options for termios setting behavior
 */
#define TERMIOS_FLUSH	1
#define TERMIOS_WAIT	2
#define TERMIOS_TERMIO	4
#define TERMIOS_OLD	8

/*
 * LDISC alias
 */
MODULE_ALIAS_LDISC(N_USR1);

#define FNAME "usbg_sen_ldisc.c"

#define __USBG_SEN_LDISC_PVT__
#include "usbg_sen_ldisc.h"
#include "usbg_sen_main.h"
#include "usbg_sen_conf.h"

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 29)
static struct tty_ldisc_ops usbg_sen_ldisc = {
#else
static struct tty_ldisc usbg_sen_ldisc = {
#endif
	.owner		= THIS_MODULE,
	.name		= "usbgter",
	.open		= usbg_sen_ldisc_open,
	.close		= usbg_sen_ldisc_close,
	.read		= usbg_sen_ldisc_read,
	.write		= usbg_sen_ldisc_write,
	.ioctl		= usbg_sen_ldisc_ioctl,
	.receive_buf	= usbg_sen_ldisc_receive_buf,
	.poll		= usbg_sen_ldisc_poll,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 16)
#else
	.receive_room	= usbg_sen_ldisc_receive_room,
#endif
};

const static struct console usbg_sen_console = {
	.name = "usbgcon",
	.write = usbg_sen_console_write,
	.index = -1,
#ifndef UART_SEN_CONSOLE_DIS
  #if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 9, 0) && defined(CONFIG_PRINTK_RT_AWARE)
	/*
	 * printk calls usbg_sen_console_write() while atomic
	 * the callee uses mutex for accessing terminal buffer
	 * so it shall not enable printk on ldisc
	 */
	.flags = CON_PRINTBUFFER,
  #else
	.flags = CON_ENABLED, /* enable printk msg */
  #endif
#endif
};

static struct func_data* FUNC_DRV;
static struct usbg_ldisc_wait wait;

// For confirmation of whether ldisc started or not.
static wait_queue_head_t gQueueStarted;
static int gnCondStarted = 0;

static int usbg_sen_ldisc_update_ep_data( struct tty_struct *tty );

#define TMP_RX_BUF_MAX_SZ	4096
#define TMP_TX_BUF_MAX_SZ	4096
static unsigned char tmp_rx_bufs[TMP_RX_BUF_MAX_SZ];
static unsigned char tmp_tx_bufs[TMP_TX_BUF_MAX_SZ];
const static struct usbg_ldisc_buf default_rx_buf ={
	.buf = tmp_rx_bufs,
	.rp = tmp_rx_bufs,
	.wp = tmp_rx_bufs,
	.len = 0,
	.size = TMP_RX_BUF_MAX_SZ,
	.read = usbg_ldisc_buf_read,
	.write = usbg_ldisc_buf_write,
	.copy_from = usbg_ldisc_buf_copy_to_user,
	.copy_to = usbg_ldisc_buf_memcpy,
};

const static struct usbg_ldisc_buf default_tx_buf ={
	.buf = tmp_tx_bufs,
	.rp = tmp_tx_bufs,
	.wp = tmp_tx_bufs,
	.len = 0,
	.size = TMP_TX_BUF_MAX_SZ,
	.read = usbg_ldisc_buf_read,
	.write = usbg_ldisc_buf_write,
	.copy_from = usbg_ldisc_buf_memcpy,
	.copy_to = usbg_ldisc_buf_memcpy,
};

static unsigned char* P_LDISC_TX_BUF = NULL;
static unsigned char* P_LDISC_RX_BUF = NULL;

extern struct usbg_sen_ep_data* ptbl_ep_data[];

#ifdef USBG_SEN_SEMAPHORE
 #ifdef CONFIG_OSAL_UDIF
  static  UDIF_MUTEX  usbg_ep_te_lock;         // lock for TE endpoints
  static  UDIF_MUTEX  usbg_rx_buf_lock;        // lock for rx buffer
  static  UDIF_MUTEX  usbg_tx_buf_lock;        // lock for tx buffer
  
  #define USBG_SEN_LOCK_LDISC_INIT()  { udif_mutex_init( &usbg_ep_te_lock        ); \
                                        udif_mutex_init( &usbg_rx_buf_lock       ); \
                                        udif_mutex_init( &usbg_tx_buf_lock       ); }
  
  #define USBG_SEN_EP_TE_LOCK_ON()          { udif_mutex_lock(   &usbg_ep_te_lock ); \
                                              USBG_SEN_INF("### TE_ENDPOINT LOCK ON  ---> \n"); }
  
  #define USBG_SEN_EP_TE_LOCK_OFF()         { udif_mutex_unlock( &usbg_ep_te_lock ); \
                                              USBG_SEN_INF("### TE_ENDPOINT LOCK OFF <--- \n"); }
  
  #define USBG_SEN_RX_BUF_LOCK_ON()         { udif_mutex_lock(   &usbg_rx_buf_lock ); \
                                              USBG_SEN_INF("### RX_BUFFER LOCK ON  ---> \n"); }
  
  #define USBG_SEN_RX_BUF_LOCK_OFF()        { udif_mutex_unlock( &usbg_rx_buf_lock ); \
                                              USBG_SEN_INF("### RX_BUFFER LOCK OFF <--- \n"); }
  
  #define USBG_SEN_TX_BUF_LOCK_ON()         { udif_mutex_lock(   &usbg_tx_buf_lock ); \
                                              USBG_SEN_INF("### TX_BUFFER LOCK ON  ---> \n"); }
  
  #define USBG_SEN_TX_BUF_LOCK_OFF()        { udif_mutex_unlock( &usbg_tx_buf_lock ); \
                                              USBG_SEN_INF("### TX_BUFFER LOCK OFF <--- \n"); }
 #else  // CONFIG_OSAL_UDIF
  #error
 #endif  // CONFIG_OSAL_UDIF
#else  // USBG_SEN_SEMAPHORE
 #error
#endif  // USBG_SEN_SEMAPHORE

/*
 * ldisc operations
 */
static int usbg_sen_ldisc_open(struct tty_struct *tty)
{
	int ret = 0;
	struct usbg_ldisc_driver* p_driver;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 3, 0)
	struct task_struct *tsk;
#else
	int pid = 0;
#endif
	
	PLOG("entry\n");
        
	p_driver = kmalloc(sizeof(struct usbg_ldisc_driver), GFP_ATOMIC);
	if(p_driver==NULL){
        KERR("%s() p_driver==NULL\n", __FUNCTION__);
		ret = -EBUSY;
		goto SUB_RET;
	}
	
	if( tty ){
        
		tty->disc_data = p_driver;
		
	}else{
        
        kfree( p_driver );
        KERR("%s() tty==NULL\n", __FUNCTION__);
        ret = -EBUSY;
        goto SUB_RET;
        
    }
	
	p_driver->tx_buf = kmalloc(sizeof(struct usbg_ldisc_buf), GFP_ATOMIC);
	if(p_driver->tx_buf==NULL){
        KERR("%s() p_driver->tx_buf==NULL\n", __FUNCTION__);
		ret = -EBUSY;
		goto SUB_RET;
	}
	memcpy(p_driver->tx_buf, &default_tx_buf, sizeof(struct usbg_ldisc_buf));
	if(usbg_ldisc_buf_init(p_driver->tx_buf)){
        KERR("%s() usbg_ldisc_buf_init(p_driver->tx_buf)\n", __FUNCTION__);
		ret = -EBUSY;
		goto SUB_RET;
	}
	
	p_driver->rx_buf = kmalloc(sizeof(struct usbg_ldisc_buf), GFP_ATOMIC);
	if(p_driver->rx_buf==NULL){
        KERR("%s() p_driver->rx_buf==NULL\n", __FUNCTION__);
		ret = -EBUSY;
		goto SUB_RET;
	}
	memcpy(p_driver->rx_buf, &default_rx_buf, sizeof(struct usbg_ldisc_buf));
	if(usbg_ldisc_buf_init(p_driver->rx_buf)){
        KERR("%s() usbg_ldisc_buf_init(p_driver->rx_buf)\n", __FUNCTION__);
		ret = -EBUSY;
		goto SUB_RET;
	}

	
	p_driver->con = kmalloc(sizeof(struct console), GFP_ATOMIC);
	if(p_driver->con == NULL){
        KERR("%s() p_driver->con == NULL\n", __FUNCTION__);
		ret = -EBUSY;
		goto SUB_RET;
	}
	memcpy(p_driver->con, &usbg_sen_console, sizeof(struct console));

	p_driver->con->data = p_driver;
	
	p_driver->p_this = p_driver;
	p_driver->usb = FUNC_DRV;
	p_driver->tty = tty;
	
	clear_bit(USBG_LDISC_FLAG_CLOSING, &wait.flag);

	clear_bit(USBG_LDISC_FLAG_SEND_INTR, &wait.flag);
	clear_bit(USBG_LDISC_FLAG_RECEIVE_INTR, &wait.flag);

	USBG_SEN_EP_TE_LOCK_ON();
	set_bit(USBG_SE_FLAG_XFER_REQ,
		&ptbl_ep_data[USBG_EPID_TE_TX]->wait_flag);
	set_bit(USBG_SE_FLAG_XFER_REQ,
		&ptbl_ep_data[USBG_EPID_TE_RX]->wait_flag);
	USBG_SEN_EP_TE_LOCK_OFF();

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 3, 0)
	tsk = kthread_run(tx_thread, (void*)p_driver, "usbgtertx");
	if (IS_ERR(tsk)) {
        KERR("%s() kthread_run() error=%p:\n", __FUNCTION__, tsk);
		ret = -EBUSY;
		goto SUB_RET;
	}
#else
	pid = kernel_thread(tx_thread, (void*)p_driver, CLONE_KERNEL );
	if(pid<0){
        KERR("%s() kernel_thread() error pid=%d:\n", __FUNCTION__, pid);
		ret = -EBUSY;
	}
#endif
	else {
		register_console(p_driver->con);
	}
	ret = usbg_ldisc_receive(p_driver);
	
SUB_RET:
	if(ret)
		ldisc_cleanup(tty);
		
	PLOG("exit\n");
	return ret;
}

static void usbg_sen_ldisc_close(struct tty_struct *tty)
{
	int err = 0;
	
	struct usbg_ldisc_driver* p_driver = NULL;
	struct func_data* p_this = NULL;
	struct usb_ep* ep_rx = NULL;
	struct usbg_sen_ep_data* ep_data_rx = NULL;
	struct usb_ep* ep_tx = NULL;
	struct usbg_sen_ep_data* ep_data_tx = NULL;

	PLOG("entry\n");

	if(tty->disc_data)
		p_driver = (struct usbg_ldisc_driver*)tty->disc_data;

	if(p_driver== NULL){
	    KERR("%s() p_driver== NULL\n", __FUNCTION__);
		goto SUB_RET;
	}

	set_bit(USBG_LDISC_FLAG_CLOSING, &wait.flag);

	if(p_driver->tx_buf){
		set_bit(USBG_LDISC_FLAG_BUFFERED, &p_driver->tx_buf->wait_flag);
		set_bit(USBG_LDISC_FLAG_CLOSING, &p_driver->tx_buf->wait_flag);
		wake_up_interruptible(&p_driver->tx_buf->wait_que);
	}

	if(p_driver->rx_buf){
		set_bit(USBG_LDISC_FLAG_BUFFERED, &p_driver->rx_buf->wait_flag);
		set_bit(USBG_LDISC_FLAG_CLOSING, &p_driver->rx_buf->wait_flag);
		wake_up_interruptible(&p_driver->rx_buf->wait_que);
	}

	USBG_SEN_EP_TE_LOCK_ON();
	p_this=p_driver->usb;
	ep_tx = USBG_SEN_GET_EP(p_this, USBG_EPID_TE_TX);
	if(ep_tx!=NULL){
		ep_data_tx = ep_tx->driver_data;
		err = usb_ep_dequeue(ep_tx, ep_data_tx->usb_req );
	}

	ep_rx = USBG_SEN_GET_EP(p_this, USBG_EPID_TE_RX);
	if(ep_rx!=NULL){
		ep_data_rx = ep_rx->driver_data;
		err = usb_ep_dequeue(ep_rx, ep_data_rx->usb_req );
	}
	USBG_SEN_EP_TE_LOCK_OFF();

	if(p_driver->con)
		unregister_console(p_driver->con);

SUB_RET:
	wait_event_interruptible( wait.que,
				  0==test_bit(USBG_LDISC_FLAG_TX_THREAD,
					      &wait.flag) );
	ldisc_cleanup(tty);
	PLOG("exit\n");
}

static void ldisc_cleanup(struct tty_struct* tty)
{
	struct usbg_ldisc_driver* p_driver;
	if(tty==NULL){
	    KERR("%s() tty==NULL\n", __FUNCTION__);
		return;
	}

	p_driver = (struct usbg_ldisc_driver*)tty->disc_data;
	if(p_driver==NULL){
	    KERR("%s() p_driver==NULL\n", __FUNCTION__);
		return;
	}

	if(p_driver->tx_buf)
		kfree(p_driver->tx_buf);
	
	if(p_driver->rx_buf)
		kfree(p_driver->rx_buf);
	
	if(p_driver->con)
		kfree(p_driver->con);

	kfree(p_driver);
}

static ssize_t usbg_sen_ldisc_read(struct tty_struct *tty, struct file *file,
				   unsigned char __user * buf, size_t nr)
{
	ssize_t len = 0;
	struct usbg_ldisc_driver* p_driver;
	struct usbg_sen_ep_data* ep_data = NULL;
	const char ConChrLF = 0x0A; // LINE FEED...

	if(tty==NULL){
        KERR("%s() tty==NULL\n", __FUNCTION__);
		goto SUB_RET;
	}

	if(0==test_bit(USBG_LDISC_FLAG_COMM_ACTIVE, &wait.flag)){
        KERR("%s() 0==test_bit\n", __FUNCTION__);
		len = -EAGAIN;
		goto SUB_RET;
	}

	p_driver = (struct usbg_ldisc_driver*)tty->disc_data;
	if(p_driver==NULL){
        KERR("%s() p_driver==NULL\n", __FUNCTION__);
		goto SUB_RET;
	}

	if(0==test_bit(USBG_LDISC_FLAG_BUFFERED,&p_driver->rx_buf->wait_flag)){
		ep_data=ptbl_ep_data[USBG_EPID_TE_RX];
		if(wait_event_interruptible(
			p_driver->rx_buf->wait_que,
			test_bit(USBG_LDISC_FLAG_BUFFERED,
				 &p_driver->rx_buf->wait_flag)
			|| test_bit(USBG_LDISC_FLAG_BUFFER_INTR,
				    &p_driver->rx_buf->wait_flag))){
			len = -EAGAIN;
			goto SUB_RET;
		}
		if(test_and_clear_bit(USBG_LDISC_FLAG_BUFFER_INTR,
			     &p_driver->rx_buf->wait_flag)){
			// Do NOT Cause Error
			len = sizeof( ConChrLF );
			if( 0 == copy_to_user( buf, &ConChrLF, sizeof( ConChrLF ) ) ){
				buf += sizeof( ConChrLF );
			}
			goto SUB_RET;
		}
	}
	
	/* read 'n copy data from rx buf to user buf	*/
	USBG_SEN_RX_BUF_LOCK_ON();
	len =p_driver->rx_buf->read(p_driver->rx_buf, buf, nr);
	USBG_SEN_RX_BUF_LOCK_OFF();

	if(len<0){
		/* copy failed */
		KERR("%s() ldisc_read copy failed. len=%ld\n", __FUNCTION__, len);
		len = -EAGAIN;
		goto SUB_RET;
	}

	buf += len;

SUB_RET:
	return len;
}

static ssize_t usbg_sen_ldisc_write(struct tty_struct *tty, struct file *file,
				    const unsigned char *buf, size_t nr)
{
	ssize_t len = 0;
	struct usbg_ldisc_driver* p_driver;
	struct usbg_sen_ep_data* ep_data = NULL;
	
	if(tty==NULL){
	    KERR("%s() tty==NULL\n", __FUNCTION__);
		goto SUB_RET;
	}

	if(0==test_bit(USBG_LDISC_FLAG_COMM_ACTIVE, &wait.flag)){
		len = nr;//-EAGAIN;
		goto SUB_RET;
	}
	
	p_driver = (struct usbg_ldisc_driver*)tty->disc_data;
	if(p_driver==NULL){
		KERR("%s() p_driver=NULL\n", __FUNCTION__);
		goto SUB_RET;
	}
	
	if(test_bit(USBG_LDISC_FLAG_BUFFER_FULL,&p_driver->tx_buf->wait_flag)){
		ep_data=ptbl_ep_data[USBG_EPID_TE_TX];
		if(wait_event_interruptible(
			p_driver->tx_buf->wait_que,
			0==test_bit(USBG_LDISC_FLAG_BUFFER_FULL,
				 &p_driver->tx_buf->wait_flag)
			|| test_bit(USBG_LDISC_FLAG_BUFFER_INTR,
				    &p_driver->tx_buf->wait_flag))){
			len = -EAGAIN;
			goto SUB_RET;
		}
		if(test_and_clear_bit(USBG_LDISC_FLAG_BUFFER_INTR,
			     &p_driver->tx_buf->wait_flag)){
			len = -EAGAIN;
			goto SUB_RET;
		}
	}
	if (0==test_bit(USBG_LDISC_FLAG_COMM_ACTIVE, &wait.flag))
		KWARN("%s write on COMM_ACTIVE=0\n", __FUNCTION__);
	USBG_SEN_TX_BUF_LOCK_ON();
	len = p_driver->tx_buf->write( p_driver->tx_buf,(unsigned char*)buf, nr);
	USBG_SEN_TX_BUF_LOCK_OFF();

	if(len<0){
	    KERR("%s() ldisc_write failed. len=%ld\n", __FUNCTION__, len);
		len = -EAGAIN;
	}

SUB_RET:
	return len;
   }

static void usbg_sen_change_termios(struct tty_struct * tty, struct ktermios * new_termios)
{

	/*
	 *	Perform the actual termios internal changes under lock.
	 */
	 

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 12, 0)
	down_write(&tty->termios_rwsem);
	tty->termios = *new_termios;
	up_write(&tty->termios_rwsem);
#else
	/* FIXME: we need to decide on some locking/ordering semantics
	   for the set_termios notification eventually */
	mutex_lock(&tty->termios_mutex);
	*tty->termios = *new_termios;
	mutex_unlock(&tty->termios_mutex);
#endif
}

static int usbg_sen_set_termios(struct tty_struct * tty, void __user *arg, int opt)
{
	struct ktermios tmp_termios;
	int retval = tty_check_change(tty);

	if (retval){
	    KERR("%s() tty_check_change failed. retval:%d\n", __FUNCTION__, retval);
		return retval;
    }

	memcpy(&tmp_termios, &tty->termios, sizeof(struct ktermios));

	if (opt & TERMIOS_TERMIO) {
		if (user_termio_to_kernel_termios(&tmp_termios,
						(struct termio __user *)arg)){
			KERR("%s() user_termio_to_kernel_termios(%d) failed.\n", __FUNCTION__, __LINE__);
			return -EFAULT;
		}
#ifdef TCGETS2
	} else if (opt & TERMIOS_OLD) {
		if (user_termios_to_kernel_termios_1(&tmp_termios,
						(struct termios __user *)arg)){
			KERR("%s() user_termios_to_kernel_termios_1(%d) failed.\n", __FUNCTION__, __LINE__);
			return -EFAULT;
		}
	} else {
		if (user_termios_to_kernel_termios(&tmp_termios,
						(struct termios2 __user *)arg)){
            KERR("%s() user_termios_to_kernel_termios(%d) failed.\n", __FUNCTION__, __LINE__);
			return -EFAULT;
		}
	}
#else
	} else if (user_termios_to_kernel_termios(&tmp_termios,
					(struct termios __user *)arg)){
	    KERR("%s() user_termios_to_kernel_termios(%d) failed.\n", __FUNCTION__, __LINE__);
		return -EFAULT;
	}
#endif

	usbg_sen_change_termios(tty, &tmp_termios);
	return 0;
}

static int usbg_sen_ldisc_ioctl(struct tty_struct * tty, struct file * file,
				unsigned int cmd, unsigned long arg)
{
	void __user *p = (void __user *)arg;

    /* ioctlコマンドの表示 */
//     PLOG("cmd=0x%4x\n", cmd);
//     PLOG("c_lflag=%lo\n", (unsigned long)tty->termios->c_lflag);
    
    /* ioctlコマンドの解析 */
	switch (cmd) {
    
		case TCSETS:
			return usbg_sen_set_termios(tty, p, TERMIOS_OLD);
#ifndef TCGETS2
		case TCGETS:
			if (kernel_termios_to_user_termios((struct termios __user *)arg, (struct ktermios *)&tty->termios)){
                KERR("%s() kernel_termios_to_user_termios(%d) failed.\n", __FUNCTION__, __LINE__);
				return -EFAULT;
			}
			return 0;
#else
		case TCGETS:
			if (kernel_termios_to_user_termios_1((struct termios __user *)arg, (struct ktermios *)&tty->termios)){
                KERR("%s() kernel_termios_to_user_termios_1(%d) failed.\n", __FUNCTION__, __LINE__);
				return -EFAULT;
			}
			return 0;
		case TCGETS2:
			if (kernel_termios_to_user_termios((struct termios2 __user *)arg, (struct ktermios *)&tty->termios)){
                KERR("%s() kernel_termios_to_user_termios(%d) failed.\n", __FUNCTION__, __LINE__);
				return -EFAULT;
			}
			return 0;
		case TCSETS2:
			return usbg_sen_set_termios(tty, p, 0);
#endif

		default:
		    KERR("%s() Unknown case.\n", __FUNCTION__);
			return -ENOIOCTLCMD;
	}
}


static void usbg_sen_ldisc_receive_buf(struct tty_struct *tty,
				       const unsigned char *cp,
				       char *fp, int count)
{
	return;
}

static unsigned int usbg_sen_ldisc_poll(struct tty_struct *tty,
					struct file *file,
					struct poll_table_struct *poll_table_sen)
{
	unsigned int mask = 0;
	int err = 0;
	struct usbg_ldisc_driver* p_driver
		= (struct usbg_ldisc_driver*)tty->disc_data;


    if( gnCondStarted ){
        
        // Already Started
        
        poll_wait(file, &p_driver->rx_buf->wait_que, poll_table_sen);
        poll_wait(file, &p_driver->tx_buf->wait_que, poll_table_sen);
        
        if(test_bit(USBG_LDISC_FLAG_BUFFERED, &p_driver->rx_buf->wait_flag))
        mask |= POLLIN | POLLRDNORM;
        
        if(0==test_bit(USBG_LDISC_FLAG_BUFFER_FULL, &p_driver->tx_buf->wait_flag))
        mask |= POLLOUT | POLLWRNORM;
        
    }else{
        
        // NOT Started yet
        
        // Wait until next ldisc start ( USB reconnect ) zzzZZZ...
        err = wait_event_interruptible( gQueueStarted, gnCondStarted );
        if( err ){
            KWARN("%s() wait_event_interruptible failed. err=%d\n", __FUNCTION__, err);
            goto SUB_RET;
        }
        
        // WakeUp!
        
        // Update information in the tty structure
        err = usbg_sen_ldisc_update_ep_data( tty );
        
        if( err )
            goto SUB_RET;
        
        mask |= POLLIN | POLLRDNORM | POLLOUT | POLLWRNORM;
        
    }
    
SUB_RET:
	KINFO("%s() %d\n", __FUNCTION__, mask);
	return mask;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 16)
#else
static int usbg_sen_ldisc_receive_room(struct tty_struct *tty)
{
	int ret = -ENOMEM;
	struct usbg_ldisc_driver* p_driver
		= (struct usbg_ldisc_driver*)tty->disc_data;

	USBG_SEN_RX_BUF_LOCK_ON();
	ret = p_driver->rx_buf->size - p_driver->rx_buf->len;
	USBG_SEN_RX_BUF_LOCK_OFF();

	KINFO("%s() %d\n", __FUNCTION__, ret);
	return ret;
}
#endif

/**
  *	console operations
  */
static void usbg_sen_console_write(struct console *co, const char*s,
				   unsigned int count)
{
	struct usbg_ldisc_driver* p_driver;

	if(co==NULL){
	    KERR("%s() co==NULL\n", __FUNCTION__);
		return;
	}

	p_driver = (struct usbg_ldisc_driver*)co->data;

	USBG_SEN_TX_BUF_LOCK_ON();
	p_driver->tx_buf->write( p_driver->tx_buf,(unsigned char*)s, count);
	USBG_SEN_TX_BUF_LOCK_OFF();
}

/**
 *	buffer operation
 */
static int usbg_ldisc_buf_init(struct usbg_ldisc_buf* p_buf)
{
	int ret = -EBUSY;
	if(p_buf){
		
		init_waitqueue_head(&p_buf->wait_que);
		p_buf->wait_flag = 0;
		ret = 0;
	}
	return ret;
}

static ssize_t usbg_ldisc_buf_read(
	struct usbg_ldisc_buf* p_buf,unsigned char* cp, size_t count)
{
	int ret = 0;
	ssize_t len = 0;
	ssize_t part = 0;

	if(test_bit(USBG_LDISC_FLAG_CLOSING, &p_buf->wait_flag)){
		goto SUB_RET;
	}
	
	count = (count < p_buf->len) ? count : p_buf->len;
	
	if(p_buf->len == 0){
		/* no data	*/
		len = 0;
		goto SUB_RET;
	} else if(p_buf->rp < p_buf->wp){
		ret = p_buf->copy_from(cp, p_buf->rp, count);
		if(ret)
			goto SUB_RET;
		p_buf->rp += count;
		p_buf->len -= count;
		len = count;
		goto SUB_RET;
	} else {
		/* copy from wp to the end of buffer		*/
		part = p_buf->size - (ssize_t)(p_buf->rp - p_buf->buf);
		len = (count < part)? count : part;
		ret = p_buf->copy_from(cp, p_buf->rp, len);
		if(ret){
			goto SUB_RET;
		}
		if(len<part)
			p_buf->rp += len;
		else
			p_buf->rp = p_buf->buf;	/* reset rp to top	*/

		p_buf->len -= len;
		count -= len;
		cp += len;

		/* copy from the top of buffer for remains	*/
		if(count && len ){
			p_buf->rp = p_buf->buf;	/* reset rp to top	*/
			ret = p_buf->copy_from(cp, p_buf->rp, count);
			if(ret){
				goto SUB_RET;
			}
			p_buf->rp += count;
			p_buf->len -= count;
			len += count;
		}
	}
SUB_RET :
	if(len>0)
		clear_bit(USBG_LDISC_FLAG_BUFFER_FULL, &p_buf->wait_flag);
	if(p_buf->len == 0)
		clear_bit(USBG_LDISC_FLAG_BUFFERED, &p_buf->wait_flag);

	/* wake up write wait, when room is available	*/
	wake_up_interruptible(&p_buf->wait_que);
	
	return len;
}

static int usbg_ldisc_buf_full_warn = 1;
static ssize_t usbg_ldisc_buf_write(
	struct usbg_ldisc_buf* p_buf,unsigned char* cp, size_t count)
{
	int ret = 0;
	ssize_t len = 0;
	ssize_t part = 0;
	ssize_t room = 0;

	room = p_buf->size - p_buf->len;
	count = (room < count) ? room : count;
	
	if(room==0){
		/* no room available	*/
		len = 0;
		goto SUB_RET;
	} else if(p_buf->rp > p_buf->wp){
		ret = p_buf->copy_to(p_buf->wp, cp, count);
		if(ret){
			goto SUB_RET;
		}
		p_buf->wp += count;
		p_buf->len += count;
		len = count;
	} else {
		part = p_buf->size - (ssize_t)(p_buf->wp - p_buf->buf);
		part = (part<count) ? part : count;
		ret = p_buf->copy_to(p_buf->wp, cp, part);
		if(ret){
			KERR("%s() cpy 1 err\n", __FUNCTION__);
			goto SUB_RET;
		}
		
		p_buf->wp += part;
		if(p_buf->wp - p_buf->buf == p_buf->size)
			p_buf->wp = p_buf->buf; /* reload	*/
		
		p_buf->len += part;
		count -= part;
		len = part;
		cp += part;
		
		if(len && count ){
			ret = p_buf->copy_to(p_buf->wp, cp, count);
			if(ret){
				KERR("%s() cpy 2 err\n", __FUNCTION__);
				goto SUB_RET;
			}
			p_buf->wp += count;
			p_buf->len += count;
			len += count;
			count-=count;
		}
	}
	
SUB_RET :
	/* */
	if(p_buf->len>0)
		set_bit(USBG_LDISC_FLAG_BUFFERED,&p_buf->wait_flag);
	else
		clear_bit(USBG_LDISC_FLAG_BUFFERED,&p_buf->wait_flag);
	
	/* no room to write, warn on bit changes */
	if(p_buf->len == p_buf->size && !test_bit(USBG_LDISC_FLAG_BUFFER_FULL,&p_buf->wait_flag)){
		set_bit(USBG_LDISC_FLAG_BUFFER_FULL,&p_buf->wait_flag);
		if (unlikely(usbg_ldisc_buf_full_warn)) {
			KERR("%s() no room() wrote len=%ld, flag=%lx\n",
			     __FUNCTION__, len,p_buf->wait_flag);
			usbg_ldisc_buf_full_warn = 0;
		}
	}
	
	/* wake up read wait, when some data was written */
	wake_up_interruptible(&p_buf->wait_que);
	
	return len;
}

static ssize_t usbg_ldisc_buf_copy_to_user(unsigned char* p_w,
					   unsigned char* p_r, size_t count)
{
	return copy_to_user((unsigned char __user*)p_w, p_r, count);
}

static ssize_t usbg_ldisc_buf_memcpy(unsigned char* p_w, unsigned char* p_r,
				     size_t count )
{
	memcpy(p_w, p_r, count);
	return 0;
}

/**
  *	thread
  */
static int tx_thread(void* vp)
{
	int ret = 0;
	struct usbg_ldisc_driver* p_driver = (struct usbg_ldisc_driver*)vp;

	set_bit(USBG_LDISC_FLAG_TX_THREAD, &wait.flag);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 3, 0)
#else
	daemonize("usbgtertx");
#endif

	while(1){
		if(p_driver==NULL){
		    KERR("%s() p_driver==NULL\n", __FUNCTION__);
			break;
		}
		if(wait_event_interruptible(
			p_driver->tx_buf->wait_que,
			test_bit(USBG_LDISC_FLAG_BUFFERED,
				 &p_driver->tx_buf->wait_flag))){
			break;
		}
		ret = usbg_ldisc_send(p_driver);
		if(ret)
			break;

		if(p_driver==NULL)
			break;

		if(test_bit(USBG_LDISC_FLAG_CLOSING, &wait.flag)){
			wake_up_interruptible(&wait.que);
			break;
		}
	}
	
	clear_bit(USBG_LDISC_FLAG_TX_THREAD, &wait.flag);
	wake_up_interruptible(&wait.que);

	return ret;
}

/**
  *	xfer operations
  */
static ssize_t usbg_ldisc_send(struct usbg_ldisc_driver* p_driver)
{
	int err = -ENOSYS;
	int len = 0;
	struct func_data* p_this = NULL;
	struct usb_ep* ep = NULL;
	struct usbg_sen_ep_data* ep_data = NULL;
	struct usbg_te_ep_data* ep_data_te;

	if(wait_event_interruptible(
		wait.que,
		0==test_and_set_bit(USBG_LDISC_FLAG_SENDING,
				    &wait.flag))){
		KERR("%s() interrupted %d\n", __FUNCTION__, 0);
		err = -EINTR;
		goto SUB_RET;
	}
	
	if(p_driver==NULL){
		KERR("%s() p_driver=%p\n", __FUNCTION__, p_driver);
		goto SUB_RET;
	}
	
	p_this=(struct func_data*)(p_driver->usb);
	KINFO("%s() p_this=%p\n", __FUNCTION__, p_this);
	if(p_this==NULL){
	    KERR("%s() p_this==NULL\n", __FUNCTION__);
		goto SUB_RET;
	}
	
	USBG_SEN_EP_TE_LOCK_ON();
	/* get ep info	*/
	do {	/* while(0)	*/
		ep = USBG_SEN_GET_EP(p_this, USBG_EPID_TE_TX);
		if(ep==NULL){
            KERR("%s() ep==NULL\n", __FUNCTION__);
			break;
		}
		ep_data = ptbl_ep_data[USBG_EPID_TE_TX];
		if(ep_data==NULL){
            KERR("%s() ep_data==NULL\n", __FUNCTION__);
			break;
		}

		if(test_bit(USBG_LDISC_FLAG_SEND_INTR, &wait.flag)){
            KERR("%s() test_bit() > 0\n", __FUNCTION__);
			err = -EINTR;
			break;
		}
		
		USBG_SEN_TX_BUF_LOCK_ON();
		len = p_driver->tx_buf->read(p_driver->tx_buf,
					     P_LDISC_TX_BUF,
					     DEF_LDISC_TX_REQBUF_SZ );
		USBG_SEN_TX_BUF_LOCK_OFF();
		
		if(len <= 0 ){
            KWARN("%s() len: %d\n", __FUNCTION__, len);
			err = -EAGAIN;
			break;
		}

		ep_data->usb_req->buf = P_LDISC_TX_BUF;
		ep_data->usb_req->dma=(dma_addr_t)NULL;
		ep_data->usb_req->length
			= len;
		ep_data->usb_req->zero = 0;
		ep_data->usb_req->context
			= (void*)p_this;
		ep_data->usb_req->complete
			= comp_usbg_ldisc_send;
        
        if( ep_data->pvt == NULL ){
            KERR("%s() ep_data->pvt == NULL\n", __FUNCTION__);
            break;
        }
        
		((struct usbg_te_ep_data*)ep_data->pvt)->ldisc = p_driver;
		if(test_bit(USBG_SE_FLAG_XFER_REQ,&ep_data->wait_flag)){
			ep_data_te = (struct usbg_te_ep_data*)ep_data->pvt;
			udif_mutex_lock( &ep_data_te->queue_cnt_lock );
			ep_data_te->queue_cnt++;
			udif_mutex_unlock( &ep_data_te->queue_cnt_lock );
			err = usb_ep_queue( ep, ep_data->usb_req, GFP_ATOMIC );
		}
	} while(0);
	USBG_SEN_EP_TE_LOCK_OFF();

SUB_RET:
	if(err){
		clear_bit(USBG_LDISC_FLAG_SENDING, &wait.flag);
	}
	clear_bit(USBG_LDISC_FLAG_SEND_INTR, &wait.flag);
	wake_up_interruptible(&wait.que);
	return err;

}

static void comp_usbg_ldisc_send(struct usb_ep* ep, struct usb_request* req)
{
	struct usbg_sen_ep_data* ep_data;
	struct usbg_te_ep_data* ep_data_te;
	struct usbg_ldisc_driver* p_driver;

	if(ep==NULL || req==NULL){
        KERR("%s() NULL pointer ep:%p, req:%p\n", __FUNCTION__, ep, req);
		return;
	}
	
	ep_data = ptbl_ep_data[USBG_EPID_TE_TX];
	if(ep_data==NULL){
        KERR("%s() ep_data==NULL\n", __FUNCTION__);
		return;
	}
	ep_data_te = (struct usbg_te_ep_data*)ep_data->pvt;
	if(ep_data_te==NULL){
        KERR("%s() ep_data_te==NULL\n", __FUNCTION__);
		return;
	}

	p_driver = (struct usbg_ldisc_driver*)ep_data_te->ldisc;
	if(p_driver==NULL){
        KERR("%s() p_driver==NULL\n", __FUNCTION__);
		return;
	}
	
	if(req->status){
		set_bit(USBG_LDISC_FLAG_SEND_INTR, &wait.flag);
		set_bit(USBG_LDISC_FLAG_BUFFER_INTR,
			&p_driver->tx_buf->wait_flag);
		wake_up_interruptible(&p_driver->tx_buf->wait_que);
	}

	clear_bit(USBG_LDISC_FLAG_SENDING, &wait.flag);
	wake_up_interruptible(&wait.que);

	udif_mutex_lock( &ep_data_te->queue_cnt_lock );
	ep_data_te->queue_cnt--;
	if (ep_data_te->queue_cnt == 0)
		udif_wake_up(&ep_data_te->wait_dequeue); // if all dequeued, wakeup.
	udif_mutex_unlock( &ep_data_te->queue_cnt_lock );
}

static ssize_t usbg_ldisc_receive(struct usbg_ldisc_driver* p_driver)
{
	int err = 0;
	struct func_data* p_this = NULL;
	struct usb_ep* ep = NULL;
	struct usbg_sen_ep_data* ep_data = NULL;
	struct usbg_te_ep_data* ep_data_te = NULL;

	if(wait_event_interruptible(
		wait.que,
		0==test_and_set_bit(USBG_LDISC_FLAG_RECEIVEING,
				    &wait.flag))){
        KERR("%s() wait_event_interruptible() failed.\n", __FUNCTION__);
		err = -EINTR;
		goto SUB_RET;
	}

	if(p_driver==NULL){
        KERR("%s() p_driver==NULL\n", __FUNCTION__);
		goto SUB_RET;
	}

	p_this = (struct func_data*)(p_driver->usb);

	if(p_this==NULL){
        KERR("%s() p_this==NULL\n", __FUNCTION__);
		goto SUB_RET;
	}

	USBG_SEN_EP_TE_LOCK_ON();
	/* get ep info	*/
	do {	/* while(0)	*/
		ep = USBG_SEN_GET_EP(p_this, USBG_EPID_TE_RX);
		if(ep==NULL){
            KERR("%s() ep==NULL\n", __FUNCTION__);
			break;
		}
		ep_data = ptbl_ep_data[USBG_EPID_TE_RX];
		if(ep_data==NULL){
            KERR("%s() ep_data==NULL\n", __FUNCTION__);
			break;
		}
		
		ep_data_te = (struct usbg_te_ep_data*)ep_data->pvt;
		if(ep_data_te==NULL){
            KERR("%s() ep_data_te==NULL\n", __FUNCTION__);
			break;
		}

		if(test_bit(USBG_LDISC_FLAG_RECEIVE_INTR,
			    &wait.flag)){
            KERR("%s() test_bit() > 0\n", __FUNCTION__);
			err = -EINTR;
			break;
		}

		ep_data->usb_req->buf = P_LDISC_RX_BUF;
		ep_data->usb_req->dma=(dma_addr_t)NULL;
		ep_data->usb_req->length = usb_gadgetcore_ep_align_maybe(ep, ep->maxpacket);  /* maxpはFS/HSで変わるので、その時点のep情報から決める */
		KINFO("%s() size=%d, aligned=%d\n", __FUNCTION__, ep->maxpacket, ep_data->usb_req->length);
		ep_data->usb_req->zero = 0;
		ep_data->usb_req->context
			= (void*)p_this;
		ep_data->usb_req->complete
			= comp_usbg_ldisc_receive;

		ep_data_te->ldisc = p_driver;
		KINFO("%s() buf=%p\n", __FUNCTION__, ep_data->usb_req->buf);
		KINFO("%s() len=%x\n", __FUNCTION__, ep_data->usb_req->length);
		
		if(test_bit(USBG_SE_FLAG_XFER_REQ,&ep_data->wait_flag)){
			udif_mutex_lock( &ep_data_te->queue_cnt_lock );
			ep_data_te->queue_cnt++;
			udif_mutex_unlock( &ep_data_te->queue_cnt_lock );
			err = usb_ep_queue( ep, ep_data->usb_req, GFP_ATOMIC );
		}
	} while(0);
	USBG_SEN_EP_TE_LOCK_OFF();

SUB_RET:
	clear_bit(USBG_LDISC_FLAG_RECEIVE_INTR, &wait.flag);
	return err;
}

static void comp_usbg_ldisc_receive(struct usb_ep* ep, struct usb_request* req)
{
	int err = -ENOBUFS;
	int len = 0;
	struct usbg_sen_ep_data* ep_data;
	struct usbg_te_ep_data* ep_data_te;
	struct usbg_ldisc_driver* p_driver;

	if(ep==NULL){
        KERR("%s() ep==NULL\n", __FUNCTION__);
		return;
	}

	ep_data = ptbl_ep_data[USBG_EPID_TE_RX];
	if(ep_data==NULL){
        KERR("%s() ep_data==NULL\n", __FUNCTION__);
		return;
	}
	ep_data_te = (struct usbg_te_ep_data*)ep_data->pvt;
	if(ep_data_te==NULL){
        KERR("%s() ep_data_te==NULL\n", __FUNCTION__);
		return;
	}

	p_driver = (struct usbg_ldisc_driver*)ep_data_te->ldisc;
	if(p_driver==NULL){
        KERR("%s() p_driver==NULL\n", __FUNCTION__);
		return;
	}

	KINFO("%s()    ep=%p, req=%p\n", __FUNCTION__, ep, req);
	KINFO("        status=%x\n",req->status);
	KINFO("        actual=%x\n",req->actual);
	KINFO("        buf=%p\n", req->buf);


	switch(req->status){
	case 0:
	case -EOVERFLOW:
		err = 0;
		len = (ep_data->usb_req->length < ep_data->usb_req->actual)
			? ep_data->usb_req->length : ep_data->usb_req->actual;
		break;
	default :
		set_bit(USBG_LDISC_FLAG_RECEIVE_INTR, &wait.flag);
		set_bit(USBG_LDISC_FLAG_BUFFER_INTR,
			&p_driver->rx_buf->wait_flag);
		wake_up_interruptible(&p_driver->rx_buf->wait_que);
		err = -EINTR;
		goto SUB_RET;
		break;
	}
	if(err){
        KERR("%s() err %d\n", __FUNCTION__, err);
		goto SUB_RET;
	}

	
	if(test_bit(USBG_LDISC_FLAG_CLOSING, &wait.flag)){
		err = -EINTR;
		KERR("%s() test_bit() > 0\n", __FUNCTION__);
		goto SUB_RET;
	}

	if (0==test_bit(USBG_LDISC_FLAG_COMM_ACTIVE, &wait.flag))
		KWARN("%s write on COMM_ACTIVE=0\n", __FUNCTION__);
	USBG_SEN_RX_BUF_LOCK_ON();
	len = p_driver->rx_buf->write(p_driver->rx_buf,
				      req->buf,
				      len );
	USBG_SEN_RX_BUF_LOCK_OFF();
	err = 0;
SUB_RET:
	clear_bit(USBG_LDISC_FLAG_RECEIVEING, &wait.flag);
	wake_up_interruptible(&wait.que);
	if(err==0)
		usbg_ldisc_receive(p_driver);
	
	udif_mutex_lock( &ep_data_te->queue_cnt_lock );
	ep_data_te->queue_cnt--;
	if (ep_data_te->queue_cnt == 0)
		udif_wake_up(&ep_data_te->wait_dequeue); // if all dequeued, wakeup.
	udif_mutex_unlock( &ep_data_te->queue_cnt_lock );
}

static void cleanup_ldisc_usb_request_buf( struct usb_ep* ep,
					   struct usb_request* req){
	if(ep==NULL || req==NULL){
        KERR("%s() NULL pointer ep:%p, req:%p\n", __FUNCTION__, ep, req);
		return;
	}
    
	if(req->buf){
		req->buf=NULL;
	}
}

/**
  *	public method
  */
int usbg_ldisc_init(void)
{
	int ret;
	FUNC_DRV = NULL;

	wait.flag = 0;
	init_waitqueue_head(&wait.que);
	
	
	// init "started" wait queue
    init_waitqueue_head( &gQueueStarted );
    
    // init "stated" condition variable
    gnCondStarted = 0;
	
	USBG_SEN_LOCK_LDISC_INIT();
	
	P_LDISC_TX_BUF = kmalloc(DEF_LDISC_TX_REQBUF_SZ, GFP_ATOMIC);
	P_LDISC_RX_BUF = kmalloc(DEF_LDISC_RX_REQBUF_SZ, GFP_ATOMIC);

	if(P_LDISC_TX_BUF==NULL || P_LDISC_RX_BUF==NULL){
		ret = -ENOMEM;
		if(P_LDISC_TX_BUF){
			kfree(P_LDISC_TX_BUF);
			P_LDISC_TX_BUF=NULL;
		}
		if(P_LDISC_RX_BUF){
			kfree(P_LDISC_RX_BUF);
			P_LDISC_RX_BUF=NULL;
		}
		goto SUB_RET;
	}
	
	ret = tty_register_ldisc(N_USR1,&usbg_sen_ldisc);
SUB_RET:
	return ret;
}

void usbg_ldisc_exit(void)
{
	wait.flag = 0;
	tty_unregister_ldisc(N_USR1);

	if(P_LDISC_TX_BUF){
		kfree(P_LDISC_TX_BUF);
		P_LDISC_TX_BUF=NULL;
	}
	if(P_LDISC_RX_BUF){
		kfree(P_LDISC_RX_BUF);
		P_LDISC_RX_BUF=NULL;
	}
	FUNC_DRV = NULL;
}

int usbg_ldisc_start(struct func_data* p_this)
{
	int err = 0;
	struct usb_ep* ep_tx = NULL;
	struct usb_ep* ep_rx = NULL;
	struct usbg_sen_ep_data* ep_data_tx = NULL;
	struct usbg_sen_ep_data* ep_data_rx = NULL;

	USBG_SEN_EP_TE_LOCK_ON();

	/* get ep info	*/
	ep_tx = p_this->ptbl_ep[USBG_EPID_TE_TX];
	ep_data_tx = ep_tx->driver_data;
	ep_data_tx->usb_req->buf = P_LDISC_TX_BUF;
	if(ep_data_tx->usb_req->buf==NULL){
        KERR("%s() ep_data_tx->usb_req->buf==NULL\n", __FUNCTION__);
		err = -ENOBUFS;
		goto SUB_RET;
	}
	KINFO("%s() tx buf=%p\n", __FUNCTION__, ep_data_tx->usb_req->buf);

	ep_rx = p_this->ptbl_ep[USBG_EPID_TE_RX];
	ep_data_rx = ep_rx->driver_data;
	ep_data_rx->usb_req->buf = P_LDISC_RX_BUF;
	if(ep_data_rx->usb_req->buf==NULL){
        KERR("%s() ep_data_rx->usb_req->buf==NULL\n", __FUNCTION__);
		err = -ENOBUFS;
		goto SUB_RET;
	}
	KINFO("%s() rx buf=%p\n", __FUNCTION__, ep_data_rx->usb_req->buf);
	
	FUNC_DRV = p_this;
	set_bit(USBG_LDISC_FLAG_COMM_ACTIVE, &wait.flag);

	KINFO("%s() rx buf=%p\n", __FUNCTION__, ep_data_rx->usb_req->buf);
	
SUB_RET:
	if(err){
		if(ep_data_tx)
			cleanup_ldisc_usb_request_buf(ep_tx, ep_data_tx->usb_req);
		if(ep_data_rx)
			cleanup_ldisc_usb_request_buf(ep_rx, ep_data_rx->usb_req);
	}
	
	USBG_SEN_EP_TE_LOCK_OFF();
	
	clear_bit(USBG_LDISC_FLAG_SENDING, &wait.flag);
	clear_bit(USBG_LDISC_FLAG_RECEIVEING, &wait.flag);
	
    
    // ldisc stated
    gnCondStarted = 1;
    
    // wakeup read-waiting.
    wake_up_interruptible( &gQueueStarted );
	
	PLOG("err=%d\n",err);
	return err;
}
int usbg_ldisc_stop(struct func_data* p_this)
{
	int ret = 0;
	struct usb_ep* ep_tx = NULL;
	struct usb_ep* ep_rx = NULL;
	struct usbg_sen_ep_data* ep_data_tx = NULL;
	struct usbg_sen_ep_data* ep_data_rx = NULL;
	
	USBG_SEN_EP_TE_LOCK_ON();

	// ldisc stopped
    gnCondStarted = 0;
    
	clear_bit(USBG_LDISC_FLAG_COMM_ACTIVE, &wait.flag);
	wake_up_interruptible(&wait.que);
	
	FUNC_DRV = NULL;
	
	if(p_this==NULL){
        KERR("%s() p_this==NULL\n", __FUNCTION__);
		goto SUB_RET;
	}
	
	ep_tx = p_this->ptbl_ep[USBG_EPID_TE_TX];
	if(ep_tx==NULL){
        KERR("%s() ep_tx==NULL\n", __FUNCTION__);
		goto SUB_RET;
	}
	
	ep_data_tx = ep_tx->driver_data;
	if(ep_data_tx){
		cleanup_ldisc_usb_request_buf(ep_tx, ep_data_tx->usb_req);
	}
	
	ep_rx = p_this->ptbl_ep[USBG_EPID_TE_RX];
	if(ep_rx==NULL){
        KERR("%s() ep_rx==NULL\n", __FUNCTION__);
		goto SUB_RET;
	}
	
	ep_data_rx = ep_rx->driver_data;
	if(ep_data_rx){
		cleanup_ldisc_usb_request_buf(ep_rx, ep_data_rx->usb_req);
	}

SUB_RET:

	USBG_SEN_EP_TE_LOCK_OFF();

	PLOG("err=%d\n",ret);
	return ret;
}

#define USBG_SEN_WAIT_DEQUEUE_WAIT_TIME  1 /* 1 sec */

void usbg_ldisc_cleanup_ep_data( struct usb_ep *ep )
{
	struct usbg_sen_ep_data *ep_data;
	struct usbg_te_ep_data *ep_data_te;
	UDIF_UINT wait_flag = 0;
	UDIF_ERR err;
	
	USBG_SEN_EP_TE_LOCK_ON();

	if(!ep){
        USBG_SEN_EP_TE_LOCK_OFF();
        KERR("%s() !ep\n", __FUNCTION__);
        return;
    }
  
	ep_data = ep->driver_data;

	if(!ep_data){
        USBG_SEN_EP_TE_LOCK_OFF();
        KERR("%s() !ep_data\n", __FUNCTION__);
        return;
    }

	/* dequeue */
	usb_ep_dequeue(ep, ep_data->usb_req);

	if(ep_data->pvt){
		ep_data_te = (struct usbg_te_ep_data*)ep_data->pvt;
		udif_mutex_lock( &ep_data_te->queue_cnt_lock );
		if (0 == ep_data_te->queue_cnt)
			wait_flag = 1;
		udif_mutex_unlock( &ep_data_te->queue_cnt_lock );

		/* wait queue complete callback */
		err = udif_wait_event_timeout(&ep_data_te->wait_dequeue,
			(wait_flag==1),
			USBG_SEN_WAIT_DEQUEUE_WAIT_TIME);
		if (err == UDIF_ERR_TIMEOUT)
			KERR("%s() dequeue timeout.\n", __FUNCTION__);

		kfree(ep_data->pvt);
		ep_data->pvt=NULL;
	}
	
	/* usb_requestオブジェクト削除                  */
	if(ep_data->usb_req){
		usb_ep_free_request(ep, ep_data->usb_req);
	}

	ep->driver_data=NULL;
	
	USBG_SEN_EP_TE_LOCK_OFF();

}

static int usbg_sen_ldisc_update_ep_data( struct tty_struct *tty )
{
    
    int ret = 0;
    struct task_struct *th;
    
    struct usbg_ldisc_driver* p_driver = NULL;
    
    // Extract ldisc driver data from tty struct
    if( tty->disc_data ){
        
        p_driver = (struct usbg_ldisc_driver*)tty->disc_data;
        
    }else{
        KERR("%s():: Invalid argument.\n", __FUNCTION__ );
        ret = -EINVAL;
        goto SUB_RET;
        
    }
    
    // Updata Endpoint data
    p_driver->usb = FUNC_DRV;
    
    // Re-Initialize wait flags
    clear_bit( USBG_LDISC_FLAG_SEND_INTR, &wait.flag );
    clear_bit( USBG_LDISC_FLAG_RECEIVE_INTR, &wait.flag );
    
    USBG_SEN_EP_TE_LOCK_ON();
    set_bit(USBG_SE_FLAG_XFER_REQ,
        &ptbl_ep_data[USBG_EPID_TE_TX]->wait_flag);
    set_bit(USBG_SE_FLAG_XFER_REQ,
        &ptbl_ep_data[USBG_EPID_TE_RX]->wait_flag);
    USBG_SEN_EP_TE_LOCK_OFF();
    
    // Re-create transfer thread
    th = kthread_create( tx_thread, (void*)p_driver, "usb_ldisc_tx" );
    
    if( IS_ERR(th) ){
            
            printk(KERN_ERR "### %s():: kthread_create. error is (%d) \n", __FUNCTION__, IS_ERR(th) );
            ret = -EBUSY;
            goto SUB_RET;
            
    }
    
    // start tx_thread.
    wake_up_process( th );
    
    // Start receiving data
    ret = usbg_ldisc_receive( p_driver );
    
SUB_RET:
    return ret;
    
}
