/*
 * Copyright (c) 2013-2019 The Linux Foundation. All rights reserved.
 *
 * Previously licensed under the ISC license by Qualcomm Atheros, Inc.
 *
 *
 * Permission to use, copy, modify, and/or distribute this software for
 * any purpose with or without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

/*
 * This file was originally distributed by Qualcomm Atheros, Inc.
 * under proprietary terms before Copyright ownership was assigned
 * to the Linux Foundation.
 */

/**
 * @addtogroup WMIAPI
 *@{
 */

/** @file
 * This file specifies the WMI interface for the  Software Architecture.
 *
 * It includes definitions of all the commands and events. Commands are messages
 * from the host to the target. Events and Replies are messages from the target
 * to the host.
 *
 * Ownership of correctness in regards to WMI commands
 * belongs to the host driver and the target is not required to validate
 * parameters for value, proper range, or any other checking.
 *
 * Guidelines for extending this interface are below.
 *
 * 1. Add new WMI commands ONLY within the specified range - 0x9000 - 0x9fff
 * 2. Use ONLY A_UINT32 type for defining member variables within WMI command/event
 *    structures. Do not use A_UINT8, A_UINT16, A_BOOL or enum types within these structures.
 * 3. DO NOT define bit fields within structures. Implement bit fields using masks
 *    if necessary. Do not use the programming language's bit field definition.
 * 4. Define macros for encode/decode of A_UINT8, A_UINT16 fields within the A_UINT32
 *    variables. Use these macros for set/get of these fields. Try to use this to
 *    optimize the structure without bloating it with A_UINT32 variables for every lower
 *    sized field.
 * 5. Do not use PACK/UNPACK attributes for the structures as each member variable is
 *    already 4-byte aligned by virtue of being a A_UINT32 type.
 * 6. Comment each parameter part of the WMI command/event structure by using the
 *    2 stars at the begining of C comment instead of one star to enable HTML document
 *    generation using Doxygen.
 *
 */

#ifndef _WMI_UNIFIED_H_
#define _WMI_UNIFIED_H_


#ifdef __cplusplus
extern "C" {
#endif

#include <wlan_defs.h>
#include <wmi_services.h>
#include <dbglog.h>

#define ATH_MAC_LEN             6               /**< length of MAC in bytes */
#define WMI_EVENT_STATUS_SUCCESS 0 /* Success return status to host */
#define WMI_EVENT_STATUS_FAILURE 1 /* Failure return status to host */

#define MAX_TX_RATE_VALUES      10 /*Max Tx Rates*/
#define MAX_RSSI_VALUES         10 /*Max Rssi values*/
#define WMI_MAX_CHAINS 8

/* The WLAN_MAX_AC macro cannot be changed without breaking
   WMI compatibility. */
//The maximum value of access category
#define WLAN_MAX_AC  4

/*
 * These don't necessarily belong here; but as the MS/SM macros require
 * ar6000_internal.h to be included, it may not be defined as yet.
 */
#define WMI_F_MS(_v, _f)                                            \
            ( ((_v) & (_f)) >> (_f##_S) )

/*
 * This breaks the "good macro practice" of only referencing each
 * macro field once (to avoid things like field++ from causing issues.)
 */
#define WMI_F_RMW(_var, _v, _f)                                     \
            do {                                                    \
                (_var) &= ~(_f);                                    \
                (_var) |= ( ((_v) << (_f##_S)) & (_f));             \
            } while (0)

#define WMI_GET_BITS(_val,_index,_num_bits)                         \
    (((_val) >> (_index)) & ((1 << (_num_bits)) - 1))

#define WMI_SET_BITS(_var,_index,_num_bits,_val) do {               \
    (_var) &= ~(((1 << (_num_bits)) - 1) << (_index));              \
    (_var) |= (((_val) & ((1 << (_num_bits)) - 1)) << (_index));    \
    } while(0)

/**
 * A packed array is an array where each entry in the array is less than
 * or equal to 16 bits, and the entries are stuffed into an A_UINT32 array.
 * For example, if each entry in the array is 11 bits, then you can stuff
 * an array of 4 11-bit values into an array of 2 A_UINT32 values.
 * The first 2 11-bit values will be stored in the first A_UINT32,
 * and the last 2 11-bit values will be stored in the second A_UINT32.
 */
#define WMI_PACKED_ARR_SIZE(num_entries,bits_per_entry) \
    (((num_entries) / (32 / (bits_per_entry))) +            \
    (((num_entries) % (32 / (bits_per_entry))) ? 1 : 0))

#define WMI_RETURN_STRING(str) case ((str)): return (uint8_t *)(# str);

static INLINE A_UINT32 wmi_packed_arr_get_bits(A_UINT32 *arr,
    A_UINT32 entry_index, A_UINT32 bits_per_entry)
{
    A_UINT32 entries_per_uint = (32 / bits_per_entry);
    A_UINT32 uint_index = (entry_index / entries_per_uint);
    A_UINT32 num_entries_in_prev_uints = (uint_index * entries_per_uint);
    A_UINT32 index_in_uint = (entry_index - num_entries_in_prev_uints);
    A_UINT32 start_bit_in_uint = (index_in_uint * bits_per_entry);
    return ((arr[uint_index] >> start_bit_in_uint) &
            (( 1 << bits_per_entry) - 1));
}

static INLINE void wmi_packed_arr_set_bits(A_UINT32 *arr, A_UINT32 entry_index,
    A_UINT32 bits_per_entry, A_UINT32 val)
{
    A_UINT32 entries_per_uint = (32 / bits_per_entry);
    A_UINT32 uint_index = (entry_index / entries_per_uint);
    A_UINT32 num_entries_in_prev_uints = (uint_index * entries_per_uint);
    A_UINT32 index_in_uint = (entry_index - num_entries_in_prev_uints);
    A_UINT32 start_bit_in_uint = (index_in_uint * bits_per_entry);

    arr[uint_index] &= ~(((1 << bits_per_entry) - 1) << start_bit_in_uint);
    arr[uint_index] |=
        ((val & ((1 << bits_per_entry) - 1)) << start_bit_in_uint);
}

/** 2 word representation of MAC addr */
typedef struct _wmi_mac_addr {
    /** upper 4 bytes of  MAC address */
    A_UINT32 mac_addr31to0;
    /** lower 2 bytes of  MAC address */
    A_UINT32 mac_addr47to32;
} wmi_mac_addr;

/** macro to convert MAC address from WMI word format to char array */
#define WMI_MAC_ADDR_TO_CHAR_ARRAY(pwmi_mac_addr,c_macaddr) do {               \
     (c_macaddr)[0] =    ((pwmi_mac_addr)->mac_addr31to0) & 0xff;     \
     (c_macaddr)[1] =  ( ((pwmi_mac_addr)->mac_addr31to0) >> 8) & 0xff; \
     (c_macaddr)[2] =  ( ((pwmi_mac_addr)->mac_addr31to0) >> 16) & 0xff; \
     (c_macaddr)[3] =  ( ((pwmi_mac_addr)->mac_addr31to0) >> 24) & 0xff;  \
     (c_macaddr)[4] =    ((pwmi_mac_addr)->mac_addr47to32) & 0xff;        \
     (c_macaddr)[5] =  ( ((pwmi_mac_addr)->mac_addr47to32) >> 8) & 0xff; \
   } while(0)

/** macro to convert MAC address from char array to WMI word format */
#define WMI_CHAR_ARRAY_TO_MAC_ADDR(c_macaddr,pwmi_mac_addr)  do { \
    (pwmi_mac_addr)->mac_addr31to0  =                                   \
        ( (c_macaddr)[0] | ((c_macaddr)[1] << 8)                           \
               | ((c_macaddr)[2] << 16) | ((c_macaddr)[3] << 24) );         \
    (pwmi_mac_addr)->mac_addr47to32  =                                  \
                  ( (c_macaddr)[4] | ((c_macaddr)[5] << 8));             \
   } while(0)

/*
 * The below function declarations are for implementations on some
 * platforms of the above macros, but in function form, to save code
 * memory by avoiding macro-inlining of a non-trivial amount of code.
 * These function versions of the above macros may not be available
 * on all host and target platforms.
 */
void wmi_mac_addr_to_char_array(wmi_mac_addr *pwmi_mac_addr, A_UINT8 *c_macaddr);
void wmi_char_array_to_mac_addr(A_UINT8 *c_macaddr, wmi_mac_addr *pwmi_mac_addr);

/*
 * wmi command groups.
 */
typedef enum {
    /* 0 to 2 are reserved */
    WMI_GRP_START=0x3,
    WMI_GRP_SCAN=WMI_GRP_START,  /* 0x3 */
    WMI_GRP_PDEV,                /* 0x4 */
    WMI_GRP_VDEV,                /* 0x5 */
    WMI_GRP_PEER,                /* 0x6 */
    WMI_GRP_MGMT,                /* 0x7 */
    WMI_GRP_BA_NEG,              /* 0x8 */
    WMI_GRP_STA_PS,              /* 0x9 */
    WMI_GRP_DFS,                 /* 0xa */
    WMI_GRP_ROAM,                /* 0xb */
    WMI_GRP_OFL_SCAN,            /* 0xc */
    WMI_GRP_P2P,                 /* 0xd */
    WMI_GRP_AP_PS,               /* 0xe */
    WMI_GRP_RATE_CTRL,           /* 0xf */
    WMI_GRP_PROFILE,             /* 0x10 */
    WMI_GRP_SUSPEND,             /* 0x11 */
    WMI_GRP_BCN_FILTER,          /* 0x12 */
    WMI_GRP_WOW,                 /* 0x13 */
    WMI_GRP_RTT,                 /* 0x14 */
    WMI_GRP_SPECTRAL,            /* 0x15 */
    WMI_GRP_STATS,               /* 0x16 */
    WMI_GRP_ARP_NS_OFL,          /* 0x17 */
    WMI_GRP_NLO_OFL,             /* 0x18 */
    WMI_GRP_GTK_OFL,             /* 0x19 */
    WMI_GRP_CSA_OFL,             /* 0x1a */
    WMI_GRP_CHATTER,             /* 0x1b */
    WMI_GRP_TID_ADDBA,           /* 0x1c */
    WMI_GRP_MISC,                /* 0x1d */
    WMI_GRP_GPIO,                /* 0x1e */
    WMI_GRP_FWTEST,              /* 0x1f */
    WMI_GRP_TDLS,                /* 0x20 */
    WMI_GRP_RESMGR,              /* 0x21 */
    WMI_GRP_STA_SMPS,            /* 0x22 */
    WMI_GRP_WLAN_HB,             /* 0x23 */
    WMI_GRP_RMC,                 /* 0x24 */
    WMI_GRP_MHF_OFL,             /* 0x25 */
    WMI_GRP_LOCATION_SCAN,       /* 0x26 */
    WMI_GRP_OEM,                 /* 0x27 */
    WMI_GRP_NAN,                 /* 0x28 */
    WMI_GRP_COEX,                /* 0x29 */
    WMI_GRP_OBSS_OFL,            /* 0x2a */
    WMI_GRP_LPI,                 /* 0x2b */
    WMI_GRP_EXTSCAN,             /* 0x2c */
    WMI_GRP_DHCP_OFL,            /* 0x2d */
    WMI_GRP_IPA,                 /* 0x2e */
    WMI_GRP_MDNS_OFL,            /* 0x2f */
    WMI_GRP_SAP_OFL,             /* 0x30 */
    WMI_GRP_OCB,                 /* 0x31 */
    WMI_GRP_SOC,                 /* 0x32 */
    WMI_GRP_PKT_FILTER,          /* 0x33 */
    WMI_GRP_MAWC,                /* 0x34 */
    WMI_GRP_PMF_OFFLOAD,         /* 0x35 */
    WMI_GRP_BPF_OFFLOAD,         /* 0x36 Berkeley Packet  Filter */
    WMI_GRP_NAN_DATA,            /* 0x37 */
    WMI_GRP_PROTOTYPE,           /* 0x38 */
    WMI_GRP_MONITOR,             /* 0x39 */
    WMI_GRP_REGULATORY,          /* 0x3a */
    WMI_GRP_HW_DATA_FILTER,      /* 0x3b */
    WMI_GRP_WLM,                 /* 0x3c WLAN Latency Manager */
    WMI_GRP_11K_OFFLOAD,         /* 0x3d */
    WMI_GRP_TWT,                 /* 0x3e TWT (Target Wake Time) for STA and AP */
    WMI_GRP_MOTION_DET,          /* 0x3f */
    WMI_GRP_SPATIAL_REUSE,       /* 0x40 */
    WMI_GRP_ESP,                 /* 0x41 Estimate Service Parameters (802.11mc) */
    WMI_GRP_HPCS_PULSE,          /* 0x42 High Precision Clock Synchronization */
} WMI_GRP_ID;

#define WMI_CMD_GRP_START_ID(grp_id) (((grp_id) << 12) | 0x1)
#define WMI_EVT_GRP_START_ID(grp_id) (((grp_id) << 12) | 0x1)

/**
 * Command IDs and commange events
 */
typedef enum {
    /** initialize the wlan sub system */
    WMI_INIT_CMDID=0x1,

    /* Scan specific commands */

    /** start scan request to FW  */
    WMI_START_SCAN_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_SCAN) ,
    /** stop scan request to FW  */
    WMI_STOP_SCAN_CMDID,
    /** full list of channels as defined by the regulatory that will be used by scanner   */
    WMI_SCAN_CHAN_LIST_CMDID,
    /** overwrite default priority table in scan scheduler   */
    WMI_SCAN_SCH_PRIO_TBL_CMDID,
    /** This command to adjust the priority and min.max_rest_time
     * of an on ongoing scan request.
     */
    WMI_SCAN_UPDATE_REQUEST_CMDID,

    /** set OUI to be used in probe request if enabled */
    WMI_SCAN_PROB_REQ_OUI_CMDID,
    /** config adaptive dwell scan */
    WMI_SCAN_ADAPTIVE_DWELL_CONFIG_CMDID,
    /** Only applicable to DBS capable product */
    WMI_SET_SCAN_DBS_DUTY_CYCLE_CMDID,

    /* PDEV(physical device) specific commands */
    /** set regulatorty ctl id used by FW to determine the exact ctl power limits */
    WMI_PDEV_SET_REGDOMAIN_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_PDEV),
    /** set channel. mainly used for supporting monitor mode */
    WMI_PDEV_SET_CHANNEL_CMDID,
    /** set pdev specific parameters */
    WMI_PDEV_SET_PARAM_CMDID,
    /** enable packet log */
    WMI_PDEV_PKTLOG_ENABLE_CMDID,
    /** disable packet log*/
    WMI_PDEV_PKTLOG_DISABLE_CMDID,
    /** set wmm parameters */
    WMI_PDEV_SET_WMM_PARAMS_CMDID,
    /** set HT cap ie that needs to be carried probe requests HT/VHT channels */
    WMI_PDEV_SET_HT_CAP_IE_CMDID,
    /** set VHT cap ie that needs to be carried on probe requests on VHT channels */
    WMI_PDEV_SET_VHT_CAP_IE_CMDID,

    /** Command to send the DSCP-to-TID map to the target */
    WMI_PDEV_SET_DSCP_TID_MAP_CMDID,
    /** set quiet ie parameters. primarily used in AP mode */
    WMI_PDEV_SET_QUIET_MODE_CMDID,
    /** Enable/Disable Green AP Power Save  */
    WMI_PDEV_GREEN_AP_PS_ENABLE_CMDID,
    /** get TPC config for the current operating channel */
    WMI_PDEV_GET_TPC_CONFIG_CMDID,

    /** set the base MAC address for the physical device before a VDEV is created.
     *  For firmware that doesn't support this feature and this command, the pdev
     *  MAC address will not be changed. */
    WMI_PDEV_SET_BASE_MACADDR_CMDID,

    /* eeprom content dump , the same to bdboard data */
    WMI_PDEV_DUMP_CMDID,
     /* set LED configuration  */
    WMI_PDEV_SET_LED_CONFIG_CMDID,
    /* Get Current temprature of chip in Celcius degree*/
    WMI_PDEV_GET_TEMPERATURE_CMDID,
    /* Set LED flashing behavior  */
    WMI_PDEV_SET_LED_FLASHING_CMDID,
    /** Enable/Disable Smart Antenna */
    WMI_PDEV_SMART_ANT_ENABLE_CMDID,
    /** Set Smart Antenna RX antenna*/
    WMI_PDEV_SMART_ANT_SET_RX_ANTENNA_CMDID,
    /** Override the antenna switch table */
    WMI_PDEV_SET_ANTENNA_SWITCH_TABLE_CMDID,
    /** Override the CTL table */
    WMI_PDEV_SET_CTL_TABLE_CMDID,
    /** Override the array gain table */
    WMI_PDEV_SET_MIMOGAIN_TABLE_CMDID,
    /** FIPS test mode command */
    WMI_PDEV_FIPS_CMDID,
    /** get CCK ANI level */
    WMI_PDEV_GET_ANI_CCK_CONFIG_CMDID,
    /** get OFDM ANI level */
    WMI_PDEV_GET_ANI_OFDM_CONFIG_CMDID,
    /** NF Cal Power dBr/dBm */
    WMI_PDEV_GET_NFCAL_POWER_CMDID,
    /** TxPPDU TPC */
    WMI_PDEV_GET_TPC_CMDID,
    /* Set to enable MIB stats collection */
    WMI_MIB_STATS_ENABLE_CMDID,

    /** Set preferred channel list for DBS Mgr */
    WMI_PDEV_SET_PCL_CMDID,
    /** Set HW mode. Eg: single MAC, DBS & SBS, see soc_hw_mode_t for values */
    WMI_PDEV_SET_HW_MODE_CMDID,
    /** Set DFS, SCAN modes and other FW configurations */
    WMI_PDEV_SET_MAC_CONFIG_CMDID,
    /** Set per band and per pdev antenna chains */
    WMI_PDEV_SET_ANTENNA_MODE_CMDID,
    /** Periodic channel stats request command */
    WMI_SET_PERIODIC_CHANNEL_STATS_CONFIG_CMDID,
    /** WMI command for power debug framework */
    WMI_PDEV_WAL_POWER_DEBUG_CMDID,
    /** set per-AC rx reorder timeouts */
    WMI_PDEV_SET_REORDER_TIMEOUT_VAL_CMDID,
    /** WMI command for WOW gpio and type */
    WMI_PDEV_SET_WAKEUP_CONFIG_CMDID,
    /* Get current ANT's per chain's RSSI info */
    WMI_PDEV_GET_ANTDIV_STATUS_CMDID,
    /** WMI command for getting Chip Power Stats */
    WMI_PDEV_GET_CHIP_POWER_STATS_CMDID,
    /** set stats reporting thresholds - see WMI_REPORT_STATS_EVENTID */
    WMI_PDEV_SET_STATS_THRESHOLD_CMDID,
    /** vdev restart request for multiple vdevs */
    WMI_PDEV_MULTIPLE_VDEV_RESTART_REQUEST_CMDID,
    /** Pdev update packet routing command */
    WMI_PDEV_UPDATE_PKT_ROUTING_CMDID,
    /** Get Calibration data version details */
    WMI_PDEV_CHECK_CAL_VERSION_CMDID,
    /* Set Diversity Gain */
    WMI_PDEV_SET_DIVERSITY_GAIN_CMDID,
    /** Get chain RSSI and antena index command */
    WMI_PDEV_DIV_GET_RSSI_ANTID_CMDID,
    /** get bss chan info */
    WMI_PDEV_BSS_CHAN_INFO_REQUEST_CMDID,
    /** update pmk cache info */
    WMI_PDEV_UPDATE_PMK_CACHE_CMDID,
    /**  update fils HLP */
    WMI_PDEV_UPDATE_FILS_HLP_PKT_CMDID,
    /** update ctltable request **/
    WMI_PDEV_UPDATE_CTLTABLE_REQUEST_CMDID,
    /** Command to set beacon OUI **/
    WMI_PDEV_CONFIG_VENDOR_OUI_ACTION_CMDID,
    /** enable/disable per-AC tx queue optimizations */
    WMI_PDEV_SET_AC_TX_QUEUE_OPTIMIZED_CMDID,
    /** enable/disable rx promiscuous mode */
    WMI_PDEV_SET_RX_FILTER_PROMISCUOUS_CMDID,
    /* set a generic direct DMA ring config */
    WMI_PDEV_DMA_RING_CFG_REQ_CMDID,
    /* enable/disable Action frame response as HE TB PPDU */
    WMI_PDEV_HE_TB_ACTION_FRM_CMDID,
    /** filter packet log based on MAC address */
    WMI_PDEV_PKTLOG_FILTER_CMDID,
    /** wmi command for setting rogue ap configuration */
    WMI_PDEV_SET_RAP_CONFIG_CMDID,

    /* VDEV (virtual device) specific commands */
    /** vdev create */
    WMI_VDEV_CREATE_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_VDEV),
    /** vdev delete */
    WMI_VDEV_DELETE_CMDID,
    /** vdev start request */
    WMI_VDEV_START_REQUEST_CMDID,
    /** vdev restart request (RX only, NO TX, used for CAC period)*/
    WMI_VDEV_RESTART_REQUEST_CMDID,
    /** vdev up request */
    WMI_VDEV_UP_CMDID,
    /** vdev stop request */
    WMI_VDEV_STOP_CMDID,
    /** vdev down request */
    WMI_VDEV_DOWN_CMDID,
    /* set a vdev param */
    WMI_VDEV_SET_PARAM_CMDID,
    /* set a key (used for setting per peer unicast and per vdev multicast) */
    WMI_VDEV_INSTALL_KEY_CMDID,

    /* wnm sleep mode command */
    WMI_VDEV_WNM_SLEEPMODE_CMDID,
    WMI_VDEV_WMM_ADDTS_CMDID,
    WMI_VDEV_WMM_DELTS_CMDID,
    WMI_VDEV_SET_WMM_PARAMS_CMDID,
    WMI_VDEV_SET_GTX_PARAMS_CMDID,
    WMI_VDEV_IPSEC_NATKEEPALIVE_FILTER_CMDID,

    WMI_VDEV_PLMREQ_START_CMDID,
    WMI_VDEV_PLMREQ_STOP_CMDID,
    /* TSF timestamp action for specified vdev */
    WMI_VDEV_TSF_TSTAMP_ACTION_CMDID,
    /** set the additional IEs in probe requests for scan or
      *  assoc req etc for frames FW locally generates */
    WMI_VDEV_SET_IE_CMDID,

    WMI_VDEV_RATEMASK_CMDID,
    /** ATF VDEV REQUEST commands. */
    WMI_VDEV_ATF_REQUEST_CMDID,
    /** Command to send the DSCP-to-TID map to the target for VAP */
    WMI_VDEV_SET_DSCP_TID_MAP_CMDID,
    /* Configure filter for Neighbor Rx Pkt (smart mesh selective listening) */
    WMI_VDEV_FILTER_NEIGHBOR_RX_PACKETS_CMDID,
    /** set quiet ie parameters. primarily used in AP mode */
    WMI_VDEV_SET_QUIET_MODE_CMDID,
    /** To set custom aggregation size for per vdev */
    WMI_VDEV_SET_CUSTOM_AGGR_SIZE_CMDID,

    /* DISA feature: Encrypt-decrypt data request */
    WMI_VDEV_ENCRYPT_DECRYPT_DATA_REQ_CMDID,
    /** Command to enable mac randomizaton **/
    WMI_VDEV_ADD_MAC_ADDR_TO_RX_FILTER_CMDID,

    /** WMI commands related to dbg arp stats */
    WMI_VDEV_SET_ARP_STAT_CMDID,
    WMI_VDEV_GET_ARP_STAT_CMDID,

    /** get tx power for the current vdev */
    WMI_VDEV_GET_TX_POWER_CMDID,
    /* limit STA offchannel activity */
    WMI_VDEV_LIMIT_OFFCHAN_CMDID,
    /** To set custom software retries per-AC for vdev */
    WMI_VDEV_SET_CUSTOM_SW_RETRY_TH_CMDID,
    /** To set chainmask configuration for vdev */
    WMI_VDEV_CHAINMASK_CONFIG_CMDID,

    WMI_VDEV_GET_BCN_RECEPTION_STATS_CMDID,
    /* request LTE-Coex info */
    WMI_VDEV_GET_MWS_COEX_INFO_CMDID,

    /* peer specific commands */

    /** create a peer */
    WMI_PEER_CREATE_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_PEER),
    /** delete a peer */
    WMI_PEER_DELETE_CMDID,
    /** flush specific  tid queues of a peer */
    WMI_PEER_FLUSH_TIDS_CMDID,
    /** set a parameter of a peer */
    WMI_PEER_SET_PARAM_CMDID,
    /** set peer to associated state. will cary all parameters determined during assocication time */
    WMI_PEER_ASSOC_CMDID,
    /**add a wds  (4 address ) entry. used only for testing WDS feature on AP products */
    WMI_PEER_ADD_WDS_ENTRY_CMDID,
    /**remove wds  (4 address ) entry. used only for testing WDS feature on AP products */
    WMI_PEER_REMOVE_WDS_ENTRY_CMDID,
    /** set up mcast group infor for multicast to unicast conversion */
    WMI_PEER_MCAST_GROUP_CMDID,
    /** request peer info from FW. FW shall respond with PEER_INFO_EVENTID */
    WMI_PEER_INFO_REQ_CMDID,

    /** request the estimated link speed for the peer. FW shall respond with
     *  WMI_PEER_ESTIMATED_LINKSPEED_EVENTID.
     */
    WMI_PEER_GET_ESTIMATED_LINKSPEED_CMDID,
    /** Set the conditions to report peer justified rate to driver
     * The justified rate means the user-rate is justified by PER.
     */
    WMI_PEER_SET_RATE_REPORT_CONDITION_CMDID,

    /** update a wds (4 address) entry */
    WMI_PEER_UPDATE_WDS_ENTRY_CMDID,
    /** add a proxy sta entry */
    WMI_PEER_ADD_PROXY_STA_ENTRY_CMDID,
    /** Set Smart Antenna TX antenna */
    WMI_PEER_SMART_ANT_SET_TX_ANTENNA_CMDID,
    /** Set Smart Antenna TX train info */
    WMI_PEER_SMART_ANT_SET_TRAIN_INFO_CMDID,
    /** Set SA node config options */
    WMI_PEER_SMART_ANT_SET_NODE_CONFIG_OPS_CMDID,
    /** ATF PEER REQUEST commands */
    WMI_PEER_ATF_REQUEST_CMDID,
    /** bandwidth fairness (BWF) peer configuration request command */
    WMI_PEER_BWF_REQUEST_CMDID,
    /** rx reorder queue setup for peer/tid */
    WMI_PEER_REORDER_QUEUE_SETUP_CMDID,
    /** rx reorder queue remove for peer/tid */
    WMI_PEER_REORDER_QUEUE_REMOVE_CMDID,
    /** specify a limit for rx A-MPDU block size */
    WMI_PEER_SET_RX_BLOCKSIZE_CMDID,
    /**
     * request peer antdiv info from FW. FW shall respond with
     * PEER_ANTDIV_INFO_EVENTID
     */
    WMI_PEER_ANTDIV_INFO_REQ_CMDID,
    /*
     * The WMI_PEER_OPER_MODE_CHANGE_EVENTID def was originally mistakenly
     * placed here, amongst the CMDID defs.
     * The WMI_PEER_OPER_MODE_CHANGE_EVENTID def has been moved to the
     * EVENTID section, but to preserve backwards compatibility, the value
     * here that had been used for WMI_PEER_OPER_MODE_CHANGE_EVENTID
     * is kept reserved/deprecated.
     *
     * This WMI_PEER_RESERVED0_CMDID value can be replaced with an actual
     * WMI peer event message ID, though it will be simpler to instead add
     * new WMI_PEER CMDID defs at the end of the WMI_GRP_PEER WMI_CMD_GRP.
     */
    WMI_PEER_RESERVED0_CMDID,

    /** Peer/Tid/Msduq threshold update */
    WMI_PEER_TID_MSDUQ_QDEPTH_THRESH_UPDATE_CMDID,
    /** TID specific configurations per peer of type
     * wmi_peer_tid_configurations_cmd_fixed_param
     */
    WMI_PEER_TID_CONFIGURATIONS_CMDID,

    /** Peer configuration for Channel Frequency Response (CFR) capture
     * of type wmi_peer_cfr_capture_cmd.
     *
     * On targets that do not use the direct DMA framework,
     * completion of the CFR capture is communicated through
     * HTT_T2H_MSG_TYPE_CFR_DUMP_COMPL_IND.
     * Such targets will set WMI_SERVICE_CFR_CAPTURE_IND_MSG_TYPE_1
     * in WMI Service Ready.
     *
     * On targets that use direct DMA, completion of CFR capture is
     * communicated through WMI_PDEV_DMA_RING_BUF_RELEASE_EVENTID
     * using module ID WMI_DMA_RING_CONFIG_MODULE_RTT.
     * Such targets will set WMI_SERVICE_CFR_CAPTURE_IND_EVT_TYPE_1
     * in WMI Service Ready and enumerate WMI_DMA_RING_CONFIG_MODULE_RTT
     * in the dma_ring_caps entry of WMI_SERVICE_READY_EXT_EVENTID.
     * Additional MAC metadata is provided in WMI_PEER_CFR_CAPTURE_EVENTID.
     */
    WMI_PEER_CFR_CAPTURE_CMDID,

    /** WMI command related to AP channel width switching */
    WMI_PEER_CHAN_WIDTH_SWITCH_CMDID,

    /** WMI command to fetch current tx PN for the peer */
    WMI_PEER_TX_PN_REQUEST_CMDID,

    /** unmap response with peer ids */
    WMI_PEER_UNMAP_RESPONSE_CMDID,


    /* beacon/management specific commands */

    /** transmit beacon by reference . used for transmitting beacon on low latency interface like pcie */
    WMI_BCN_TX_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_MGMT),
    /** transmit beacon by value */
    WMI_PDEV_SEND_BCN_CMDID,
    /** set the beacon template. used in beacon offload mode to setup the
     *  the common beacon template with the FW to be used by FW to generate beacons */
    WMI_BCN_TMPL_CMDID,
    /** set beacon filter with FW */
    WMI_BCN_FILTER_RX_CMDID,
    /* enable/disable filtering of probe requests in the firmware */
    WMI_PRB_REQ_FILTER_RX_CMDID,
    /** transmit management frame by value. will be deprecated */
    WMI_MGMT_TX_CMDID,
    /** set the probe response template. used in beacon offload mode to setup the
     *  the common probe response template with the FW to be used by FW to generate
     *  probe responses */
    WMI_PRB_TMPL_CMDID,

    /** Transmit Mgmt frame by reference */
    WMI_MGMT_TX_SEND_CMDID,
    /** Transmit data frame by reference */
    WMI_OFFCHAN_DATA_TX_SEND_CMDID,
    /** transmit FILS Discovery frame by value */
    WMI_PDEV_SEND_FD_CMDID,
    /** Cmd to enable/disable offloaded beacons */
    WMI_BCN_OFFLOAD_CTRL_CMDID,
    /** Cmd to enable FW handling BSS color change notification from AP. */
    WMI_BSS_COLOR_CHANGE_ENABLE_CMDID,
    /** To configure Beacon offload quiet-ie params */
    WMI_VDEV_BCN_OFFLOAD_QUIET_CONFIG_CMDID,

    /** commands to directly control ba negotiation directly from host. only used in test mode */

    /** turn off FW Auto addba mode and let host control addba */
    WMI_ADDBA_CLEAR_RESP_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_BA_NEG),
    /** send add ba request */
    WMI_ADDBA_SEND_CMDID,
    WMI_ADDBA_STATUS_CMDID,
    /** send del ba */
    WMI_DELBA_SEND_CMDID,
    /** set add ba response will be used by FW to generate addba response*/
    WMI_ADDBA_SET_RESP_CMDID,
    /** send single VHT MPDU with AMSDU */
    WMI_SEND_SINGLEAMSDU_CMDID,

    /** Station power save specific config */
    /** enable/disable station powersave */
    WMI_STA_POWERSAVE_MODE_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_STA_PS),
    /** set station power save specific parameter */
    WMI_STA_POWERSAVE_PARAM_CMDID,
    /** set station mimo powersave mode */
    WMI_STA_MIMO_PS_MODE_CMDID,
    /** config station TX cycle percentage in a beacon interval */
    WMI_STA_TDCC_CONFIG_CMDID,


    /** DFS-specific commands */
    /** enable DFS (radar detection)*/
    WMI_PDEV_DFS_ENABLE_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_DFS),
    /** disable DFS (radar detection)*/
    WMI_PDEV_DFS_DISABLE_CMDID,
    /** enable DFS phyerr/parse filter offload */
    WMI_DFS_PHYERR_FILTER_ENA_CMDID,
    /** enable DFS phyerr/parse filter offload */
    WMI_DFS_PHYERR_FILTER_DIS_CMDID,
    /** enable DFS phyerr processing offload */
    WMI_PDEV_DFS_PHYERR_OFFLOAD_ENABLE_CMDID,
    /** disable DFS phyerr processing offload */
    WMI_PDEV_DFS_PHYERR_OFFLOAD_DISABLE_CMDID,
    /** set ADFS channel config */
    WMI_VDEV_ADFS_CH_CFG_CMDID,
    /** abort ADFS off-channel-availability-check currently in progress */
    WMI_VDEV_ADFS_OCAC_ABORT_CMDID,

    /* Roaming specific  commands */
    /** set roam scan mode */
    WMI_ROAM_SCAN_MODE=WMI_CMD_GRP_START_ID(WMI_GRP_ROAM),
    /** set roam scan rssi threshold below which roam scan is enabled  */
    WMI_ROAM_SCAN_RSSI_THRESHOLD,
    /** set roam scan period for periodic roam scan mode  */
    WMI_ROAM_SCAN_PERIOD,
    /** set roam scan trigger rssi change threshold   */
    WMI_ROAM_SCAN_RSSI_CHANGE_THRESHOLD,
    /** set roam AP profile   */
    WMI_ROAM_AP_PROFILE,
    /** set channel list for roam scans */
    WMI_ROAM_CHAN_LIST,
    /** Stop scan command */
    WMI_ROAM_SCAN_CMD,
    /** roaming sme offload sync complete */
    WMI_ROAM_SYNCH_COMPLETE,
    /** set ric request element for 11r roaming */
    WMI_ROAM_SET_RIC_REQUEST_CMDID,
    /** Invoke roaming forcefully */
    WMI_ROAM_INVOKE_CMDID,
    /** roaming filter cmd to allow further filtering of roaming candidate */
    WMI_ROAM_FILTER_CMDID,
    /** set gateway ip, mac and retries for subnet change detection */
    WMI_ROAM_SUBNET_CHANGE_CONFIG_CMDID,
    /** configure thresholds for MAWC */
    WMI_ROAM_CONFIGURE_MAWC_CMDID,
    /** configure MultiBand Operation(refer WFA MBO spec) parameter */
    WMI_ROAM_SET_MBO_PARAM_CMDID, /* DEPRECATED */
    /** configure packet error rate threshold for triggering roaming */
    WMI_ROAM_PER_CONFIG_CMDID,
    /** configure BSS Transition Management (BTM) offload for roaming */
    WMI_ROAM_BTM_CONFIG_CMDID,
    /** Enable or Disable Fast Initial Link Setup (FILS) feature */
    WMI_ENABLE_FILS_CMDID,
    /** Request for roam scan stats */
    WMI_REQUEST_ROAM_SCAN_STATS_CMDID,
    /** Configure BSS load parameters for roam trigger */
    WMI_ROAM_BSS_LOAD_CONFIG_CMDID,
    /** Configure deauth roam trigger parameters */
    WMI_ROAM_DEAUTH_CONFIG_CMDID,
    /** Configure idle roam trigger parameters */
    WMI_ROAM_IDLE_CONFIG_CMDID,

    /** offload scan specific commands */
    /** set offload scan AP profile   */
    WMI_OFL_SCAN_ADD_AP_PROFILE=WMI_CMD_GRP_START_ID(WMI_GRP_OFL_SCAN),
    /** remove offload scan AP profile   */
    WMI_OFL_SCAN_REMOVE_AP_PROFILE,
    /** set offload scan period   */
    WMI_OFL_SCAN_PERIOD,

    /* P2P specific commands */
    /**set P2P device info. FW will used by FW to create P2P IE to be carried in probe response
     * generated during p2p listen and for p2p discoverability  */
    WMI_P2P_DEV_SET_DEVICE_INFO=WMI_CMD_GRP_START_ID(WMI_GRP_P2P),
    /** enable/disable p2p discoverability on STA/AP VDEVs  */
    WMI_P2P_DEV_SET_DISCOVERABILITY,
    /** set p2p ie to be carried in beacons generated by FW for GO  */
    WMI_P2P_GO_SET_BEACON_IE,
    /** set p2p ie to be carried in probe response frames generated by FW for GO  */
    WMI_P2P_GO_SET_PROBE_RESP_IE,
    /** set the vendor specific p2p ie data. FW will use this to parse the P2P NoA
     *  attribute in the beacons/probe responses received.
     *  Note: This command is currently used only for Apple P2P implementation.
     */
    WMI_P2P_SET_VENDOR_IE_DATA_CMDID,
    /** set the configure of p2p find offload */
    WMI_P2P_DISC_OFFLOAD_CONFIG_CMDID,
    /** set the vendor specific p2p ie data for p2p find offload using */
    WMI_P2P_DISC_OFFLOAD_APPIE_CMDID,
    /** set the BSSID/device name pattern of p2p find offload */
    WMI_P2P_DISC_OFFLOAD_PATTERN_CMDID,
    /** set OppPS related parameters **/
    WMI_P2P_SET_OPPPS_PARAM_CMDID,
    /** set listen offload start related parameters */
    WMI_P2P_LISTEN_OFFLOAD_START_CMDID,
    /** set listen offload stop related parameters */
    WMI_P2P_LISTEN_OFFLOAD_STOP_CMDID,

    /** AP power save specific config */
    /** set AP power save specific param */
    WMI_AP_PS_PEER_PARAM_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_AP_PS),
    /** set AP UAPSD coex pecific param */
    WMI_AP_PS_PEER_UAPSD_COEX_CMDID,
    /** set Enhanced Green AP param */
    WMI_AP_PS_EGAP_PARAM_CMDID,


    /** Rate-control specific commands */
    WMI_PEER_RATE_RETRY_SCHED_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_RATE_CTRL),

    /** WLAN Profiling commands. */
    WMI_WLAN_PROFILE_TRIGGER_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_PROFILE),
    WMI_WLAN_PROFILE_SET_HIST_INTVL_CMDID,
    WMI_WLAN_PROFILE_GET_PROFILE_DATA_CMDID,
    WMI_WLAN_PROFILE_ENABLE_PROFILE_ID_CMDID,
    WMI_WLAN_PROFILE_LIST_PROFILE_ID_CMDID,

    /** Suspend resume command Ids */
    WMI_PDEV_SUSPEND_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_SUSPEND),
    WMI_PDEV_RESUME_CMDID,

    /* Beacon filter commands */
    /** add a beacon filter */
    WMI_ADD_BCN_FILTER_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_BCN_FILTER),
    /** remove a  beacon filter */
    WMI_RMV_BCN_FILTER_CMDID,

    /* WOW Specific WMI commands*/
    /** add pattern for awake */
    WMI_WOW_ADD_WAKE_PATTERN_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_WOW),
    /** deleta a wake pattern */
    WMI_WOW_DEL_WAKE_PATTERN_CMDID,
    /** enable/deisable wake event  */
    WMI_WOW_ENABLE_DISABLE_WAKE_EVENT_CMDID,
    /** enable WOW  */
    WMI_WOW_ENABLE_CMDID,
    /** host woke up from sleep event to FW. Generated in response to WOW Hardware event */
    WMI_WOW_HOSTWAKEUP_FROM_SLEEP_CMDID,
    /* IOAC add keep alive cmd. */
    WMI_WOW_IOAC_ADD_KEEPALIVE_CMDID,
    /* IOAC del keep alive cmd. */
    WMI_WOW_IOAC_DEL_KEEPALIVE_CMDID,
    /* IOAC add pattern for awake */
    WMI_WOW_IOAC_ADD_WAKE_PATTERN_CMDID,
    /* IOAC deleta a wake pattern */
    WMI_WOW_IOAC_DEL_WAKE_PATTERN_CMDID,
    /* D0-WOW enable or disable cmd */
    WMI_D0_WOW_ENABLE_DISABLE_CMDID,
    /* enable extend WoW */
    WMI_EXTWOW_ENABLE_CMDID,
    /* Extend WoW command to configure app type1 parameter */
    WMI_EXTWOW_SET_APP_TYPE1_PARAMS_CMDID,
    /* Extend WoW command to configure app type2 parameter */
    WMI_EXTWOW_SET_APP_TYPE2_PARAMS_CMDID,
    /* enable ICMPv6 Network advertisement filtering */
    WMI_WOW_ENABLE_ICMPV6_NA_FLT_CMDID,
    /*
     * Set a pattern to match UDP packet in WOW mode.
     * If match, construct a tx frame in a local buffer
     * to send through the peer AP to the entity in the
     * IP network that sent the UDP packet to this STA.
     */
    WMI_WOW_UDP_SVC_OFLD_CMDID,

    /* configure WOW host wakeup PIN pattern */
    WMI_WOW_HOSTWAKEUP_GPIO_PIN_PATTERN_CONFIG_CMDID,

    /* Set which action category should wake the host from suspend */
    WMI_WOW_SET_ACTION_WAKE_UP_CMDID,

    /* RTT measurement related cmd */
    /** request to make an RTT measurement */
    WMI_RTT_MEASREQ_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_RTT),
    /** request to report a tsf measurement */
    WMI_RTT_TSF_CMDID,

    /** spectral scan command */
    /** configure spectral scan */
    WMI_VDEV_SPECTRAL_SCAN_CONFIGURE_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_SPECTRAL),
    /** enable/disable spectral scan and trigger */
    WMI_VDEV_SPECTRAL_SCAN_ENABLE_CMDID,

    /* F/W stats */
    /** one time request for stats */
    WMI_REQUEST_STATS_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_STATS),
    /** Push MCC Adaptive Scheduler Stats to Firmware */
    WMI_MCC_SCHED_TRAFFIC_STATS_CMDID,
    /** one time request for txrx stats */
    WMI_REQUEST_STATS_EXT_CMDID,

    /* Link Layer stats */
    /** Request for link layer stats */
    WMI_REQUEST_LINK_STATS_CMDID,
    /** Request for setting params to link layer stats */
    WMI_START_LINK_STATS_CMDID,
    /** Request to clear stats*/
    WMI_CLEAR_LINK_STATS_CMDID,

    /** Request for getting the Firmware Memory Dump */
    WMI_GET_FW_MEM_DUMP_CMDID,

    /** Request to flush of the buffered debug messages */
    WMI_DEBUG_MESG_FLUSH_CMDID,

    /** Cmd to configure the verbose level */
    WMI_DIAG_EVENT_LOG_CONFIG_CMDID,

    /** One time request for wlan stats */
    WMI_REQUEST_WLAN_STATS_CMDID,

    /** Request for getting RCPI of peer */
    WMI_REQUEST_RCPI_CMDID,

    /** One time request for peer stats info */
    WMI_REQUEST_PEER_STATS_INFO_CMDID,

    /** One time request for radio channel stats */
    WMI_REQUEST_RADIO_CHAN_STATS_CMDID,

    /** request for WLM (wlan latency manager) stats */
    WMI_REQUEST_WLM_STATS_CMDID,


    /** ARP OFFLOAD REQUEST*/
    WMI_SET_ARP_NS_OFFLOAD_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_ARP_NS_OFL),

    /** Proactive ARP Response Add Pattern Command*/
    WMI_ADD_PROACTIVE_ARP_RSP_PATTERN_CMDID,

    /** Proactive ARP Response Del Pattern Command*/
    WMI_DEL_PROACTIVE_ARP_RSP_PATTERN_CMDID,

    /** NS offload confid*/
    WMI_NETWORK_LIST_OFFLOAD_CONFIG_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_NLO_OFL),

    /** APFIND Config */
    WMI_APFIND_CMDID,

    /** Passpoint list config  */
    WMI_PASSPOINT_LIST_CONFIG_CMDID,
    /** configure supprssing parameters for MAWC */
    WMI_NLO_CONFIGURE_MAWC_CMDID,

    /* GTK offload Specific WMI commands*/
    WMI_GTK_OFFLOAD_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_GTK_OFL),

    /* CSA offload Specific WMI commands*/
    /** csa offload enable */
    WMI_CSA_OFFLOAD_ENABLE_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_CSA_OFL),
    /** chan switch command */
    WMI_CSA_OFFLOAD_CHANSWITCH_CMDID,

    /* Chatter commands*/
    /* Change chatter mode of operation */
    WMI_CHATTER_SET_MODE_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_CHATTER),
    /** chatter add coalescing filter command */
    WMI_CHATTER_ADD_COALESCING_FILTER_CMDID,
    /** chatter delete coalescing filter command */
    WMI_CHATTER_DELETE_COALESCING_FILTER_CMDID,
    /** chatter coalecing query command */
    WMI_CHATTER_COALESCING_QUERY_CMDID,

    /**addba specific commands */
    /** start the aggregation on this TID */
    WMI_PEER_TID_ADDBA_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_TID_ADDBA),
    /** stop the aggregation on this TID */
    WMI_PEER_TID_DELBA_CMDID,

    /** set station mimo powersave method */
    WMI_STA_DTIM_PS_METHOD_CMDID,
    /** Configure the Station UAPSD AC Auto Trigger Parameters */
    WMI_STA_UAPSD_AUTO_TRIG_CMDID,
    /** Configure the Keep Alive Parameters */
    WMI_STA_KEEPALIVE_CMDID,

    /* Request ssn from target for a sta/tid pair */
    WMI_BA_REQ_SSN_CMDID,
    /* misc command group */
    /** echo command mainly used for testing */
    WMI_ECHO_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_MISC),

	/* !!IMPORTANT!!
	  * If you need to add a new WMI command to the WMI_GRP_MISC sub-group,
	  * please make sure you add it BEHIND WMI_PDEV_UTF_CMDID,
	  * as we MUST have a fixed value here to maintain compatibility between
	  * UTF and the ART2 driver
	*/
	/** UTF WMI commands */
	WMI_PDEV_UTF_CMDID,

    /** set debug log config */
    WMI_DBGLOG_CFG_CMDID,
    /* QVIT specific command id */
    WMI_PDEV_QVIT_CMDID,
    /* Factory Testing Mode request command
     * used for integrated chipsets */
    WMI_PDEV_FTM_INTG_CMDID,
    /* set and get keepalive parameters command */
    WMI_VDEV_SET_KEEPALIVE_CMDID,
    WMI_VDEV_GET_KEEPALIVE_CMDID,
    /* For fw recovery test command */
    WMI_FORCE_FW_HANG_CMDID,
    /* Set Mcast/Bdcast filter */
    WMI_SET_MCASTBCAST_FILTER_CMDID,
    /** set thermal management params **/
    WMI_THERMAL_MGMT_CMDID,
    /** set host auto shutdown params **/
    WMI_HOST_AUTO_SHUTDOWN_CFG_CMDID,
    /** set tpc chainmask config command */
    WMI_TPC_CHAINMASK_CONFIG_CMDID,
    /** set Antenna diversity command */
    WMI_SET_ANTENNA_DIVERSITY_CMDID,
    /** Set OCB Sched Request, deprecated */
    WMI_OCB_SET_SCHED_CMDID,
    /** Set rssi monitoring config command */
    WMI_RSSI_BREACH_MONITOR_CONFIG_CMDID,
    /** Enable/disable Large Receive Offload processing; provide cfg params */
    WMI_LRO_CONFIG_CMDID,
    /** transfer data from host to firmware to write flash */
    WMI_TRANSFER_DATA_TO_FLASH_CMDID,
    /** Command to enable/disable filtering of multicast IP with unicast mac */
    WMI_CONFIG_ENHANCED_MCAST_FILTER_CMDID,
     /** Command to control WISA mode */
    WMI_VDEV_WISA_CMDID,
    /** set debug log time stamp sync up with host */
    WMI_DBGLOG_TIME_STAMP_SYNC_CMDID,
    /** Command for host to set/delete multiple mcast filters */
    WMI_SET_MULTIPLE_MCAST_FILTER_CMDID,
    /** upload a requested section of data from firmware flash to host */
    WMI_READ_DATA_FROM_FLASH_CMDID,
    /* Thermal Throttling SET CONF commands */
    WMI_THERM_THROT_SET_CONF_CMDID,
    /* set runtime dpd recalibration params */
    WMI_RUNTIME_DPD_RECAL_CMDID,
    /* get TX power for input HALPHY parameters */
    WMI_GET_TPC_POWER_CMDID,
    /* Specify when to start monitoring for idle state */
    WMI_IDLE_TRIGGER_MONITOR_CMDID,

    /*  Offload 11k related requests */
    WMI_11K_OFFLOAD_REPORT_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_11K_OFFLOAD),
    /* invoke neighbor report from FW */
    WMI_11K_INVOKE_NEIGHBOR_REPORT_CMDID,

    /* GPIO Configuration */
    WMI_GPIO_CONFIG_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_GPIO),
    WMI_GPIO_OUTPUT_CMDID,

    /* Txbf configuration command */
    WMI_TXBF_CMDID,

    /* FWTEST Commands */
    WMI_FWTEST_VDEV_MCC_SET_TBTT_MODE_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_FWTEST),
    /** set NoA descs **/
    WMI_FWTEST_P2P_SET_NOA_PARAM_CMDID,
    /* UNIT Tests  */
    WMI_UNIT_TEST_CMDID,
    /* set debug and tuning parameters */
    WMI_FWTEST_CMDID,
    /* Q-Boost configuration test commands */
    WMI_QBOOST_CFG_CMDID,

    /** TDLS Configuration */
    /** enable/disable TDLS */
    WMI_TDLS_SET_STATE_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_TDLS),
    /** set tdls peer state */
    WMI_TDLS_PEER_UPDATE_CMDID,
    /** TDLS Offchannel control */
    WMI_TDLS_SET_OFFCHAN_MODE_CMDID,

    /** Resmgr Configuration */
    /** Adaptive OCS is enabled by default in the FW. This command is used to
     * disable FW based adaptive OCS.
     */
    WMI_RESMGR_ADAPTIVE_OCS_ENABLE_DISABLE_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_RESMGR),
    /** set the requested channel time quota for the home channels */
    WMI_RESMGR_SET_CHAN_TIME_QUOTA_CMDID,
    /** set the requested latency for the home channels */
    WMI_RESMGR_SET_CHAN_LATENCY_CMDID,

    /** STA SMPS Configuration */
    /** force SMPS mode */
    WMI_STA_SMPS_FORCE_MODE_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_STA_SMPS),
    /** set SMPS parameters */
    WMI_STA_SMPS_PARAM_CMDID,

    /* Wlan HB commands*/
    /* enalbe/disable wlan HB */
    WMI_HB_SET_ENABLE_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_WLAN_HB),
    /* set tcp parameters for wlan HB */
    WMI_HB_SET_TCP_PARAMS_CMDID,
    /* set tcp pkt filter for wlan HB */
    WMI_HB_SET_TCP_PKT_FILTER_CMDID,
    /* set udp parameters for wlan HB */
    WMI_HB_SET_UDP_PARAMS_CMDID,
    /* set udp pkt filter for wlan HB */
    WMI_HB_SET_UDP_PKT_FILTER_CMDID,

    /* OIC ping keep alive */
    WMI_HB_OIC_PING_OFFLOAD_PARAM_CMDID,
    WMI_HB_OIC_PING_OFFLOAD_SET_ENABLE_CMDID,

    /* WMI commands related to DHCP Lease Renew Offload **/
    WMI_HB_DHCP_LEASE_RENEW_OFFLOAD_CMDID,

    /** Wlan RMC commands*/
    /** enable/disable RMC */
    WMI_RMC_SET_MODE_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_RMC),
    /** configure action frame period */
    WMI_RMC_SET_ACTION_PERIOD_CMDID,
    /** For debug/future enhancement purposes only,
     *  configures/finetunes RMC algorithms */
    WMI_RMC_CONFIG_CMDID,
    /** select manual leader */
    WMI_RMC_SET_MANUAL_LEADER_CMDID,

    /** WLAN MHF offload commands */
    /** enable/disable MHF offload */
    WMI_MHF_OFFLOAD_SET_MODE_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_MHF_OFL),
    /** Plumb routing table for MHF offload */
    WMI_MHF_OFFLOAD_PLUMB_ROUTING_TBL_CMDID,

    /*location scan commands*/
    /*start batch scan*/
    WMI_BATCH_SCAN_ENABLE_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_LOCATION_SCAN),
    /*stop batch scan*/
    WMI_BATCH_SCAN_DISABLE_CMDID,
    /*get batch scan result*/
    WMI_BATCH_SCAN_TRIGGER_RESULT_CMDID,
    /* OEM related cmd */
    WMI_OEM_REQ_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_OEM),
    WMI_OEM_REQUEST_CMDID, /* UNUSED */
    /* OEM related cmd used for Low Power ranging */
    WMI_LPI_OEM_REQ_CMDID,
    WMI_OEM_DMA_RING_CFG_REQ_CMDID,

    /** Nan Request */
    WMI_NAN_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_NAN),

    /** Modem power state command */
    WMI_MODEM_POWER_STATE_CMDID=WMI_CMD_GRP_START_ID(WMI_GRP_COEX),
    WMI_CHAN_AVOID_UPDATE_CMDID,
    WMI_COEX_CONFIG_CMDID,
    WMI_CHAN_AVOID_RPT_ALLOW_CMDID,
    WMI_COEX_GET_ANTENNA_ISOLATION_CMDID,
    WMI_SAR_LIMITS_CMDID,
    WMI_SAR_GET_LIMITS_CMDID,

    /**
     *  OBSS scan offload enable/disable commands
     *  OBSS scan enable CMD will send to FW after VDEV UP, if these conditions are true:
     *  1.  WMI_SERVICE_OBSS_SCAN is reported by FW in service ready,
     *  2.  STA connect to a 2.4Ghz ht20/ht40 AP,
     *  3.  AP enable 20/40 coexistence (OBSS_IE-74 can be found in beacon or association response)
     *  If OBSS parameters from beacon changed, also use enable CMD to update parameters.
     *  OBSS scan disable CMD will send to FW if have enabled when tearing down connection.
     */
    WMI_OBSS_SCAN_ENABLE_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_OBSS_OFL),
    WMI_OBSS_SCAN_DISABLE_CMDID,
    WMI_OBSS_COLOR_COLLISION_DET_CONFIG_CMDID,
    /**LPI commands*/
    /**LPI mgmt snooping config command*/
    WMI_LPI_MGMT_SNOOPING_CONFIG_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_LPI),
    /**LPI scan start command*/
    WMI_LPI_START_SCAN_CMDID,
    /**LPI scan stop command*/
    WMI_LPI_STOP_SCAN_CMDID,

     /** ExtScan commands */
    WMI_EXTSCAN_START_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_EXTSCAN),
    WMI_EXTSCAN_STOP_CMDID,
    WMI_EXTSCAN_CONFIGURE_WLAN_CHANGE_MONITOR_CMDID,
    WMI_EXTSCAN_CONFIGURE_HOTLIST_MONITOR_CMDID,
    WMI_EXTSCAN_GET_CACHED_RESULTS_CMDID,
    WMI_EXTSCAN_GET_WLAN_CHANGE_RESULTS_CMDID,
    WMI_EXTSCAN_SET_CAPABILITIES_CMDID,
    WMI_EXTSCAN_GET_CAPABILITIES_CMDID,
    WMI_EXTSCAN_CONFIGURE_HOTLIST_SSID_MONITOR_CMDID,
    WMI_EXTSCAN_CONFIGURE_MAWC_CMDID,

    /** DHCP server offload commands */
    WMI_SET_DHCP_SERVER_OFFLOAD_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_DHCP_OFL),

    /** IPA Offload features related commands */
    WMI_IPA_OFFLOAD_ENABLE_DISABLE_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_IPA),

    /** mDNS responder offload commands */
    WMI_MDNS_OFFLOAD_ENABLE_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_MDNS_OFL),
    WMI_MDNS_SET_FQDN_CMDID,
    WMI_MDNS_SET_RESPONSE_CMDID,
    WMI_MDNS_GET_STATS_CMDID,

    /* enable/disable AP Authentication offload */
    WMI_SAP_OFL_ENABLE_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_SAP_OFL),
    WMI_SAP_SET_BLACKLIST_PARAM_CMDID,
    WMI_SAP_OBSS_DETECTION_CFG_CMDID,
    /** Out-of-context-of-BSS (OCB) commands */
    WMI_OCB_SET_CONFIG_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_OCB),
    WMI_OCB_SET_UTC_TIME_CMDID,
    WMI_OCB_START_TIMING_ADVERT_CMDID,
    WMI_OCB_STOP_TIMING_ADVERT_CMDID,
    WMI_OCB_GET_TSF_TIMER_CMDID,
    WMI_DCC_GET_STATS_CMDID,
    WMI_DCC_CLEAR_STATS_CMDID,
    WMI_DCC_UPDATE_NDL_CMDID,
    /* System-On-Chip commands */
    WMI_SOC_SET_PCL_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_SOC),
    WMI_SOC_SET_HW_MODE_CMDID,
    WMI_SOC_SET_DUAL_MAC_CONFIG_CMDID,
    WMI_SOC_SET_ANTENNA_MODE_CMDID,

    /* packet filter commands */
    WMI_PACKET_FILTER_CONFIG_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_PKT_FILTER),
    WMI_PACKET_FILTER_ENABLE_CMDID,

     /** Motion Aided WiFi Connectivity (MAWC) commands */
    WMI_MAWC_SENSOR_REPORT_IND_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_MAWC),

    /** WMI commands related to PMF 11w Offload */
    WMI_PMF_OFFLOAD_SET_SA_QUERY_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_PMF_OFFLOAD),

    /** WMI commands related to pkt filter (BPF) offload */
    WMI_BPF_GET_CAPABILITY_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_BPF_OFFLOAD),
    WMI_BPF_GET_VDEV_STATS_CMDID,
    WMI_BPF_SET_VDEV_INSTRUCTIONS_CMDID,
    WMI_BPF_DEL_VDEV_INSTRUCTIONS_CMDID,
    WMI_BPF_SET_VDEV_ACTIVE_MODE_CMDID,

    WMI_BPF_SET_VDEV_ENABLE_CMDID,
    WMI_BPF_SET_VDEV_WORK_MEMORY_CMDID,
    WMI_BPF_GET_VDEV_WORK_MEMORY_CMDID,

    /** WMI commands related to monitor mode. */
    WMI_MNT_FILTER_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_MONITOR),

    /** WMI commands related to regulatory offload */
    WMI_SET_CURRENT_COUNTRY_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_REGULATORY),
    WMI_11D_SCAN_START_CMDID,
    WMI_11D_SCAN_STOP_CMDID,
    WMI_SET_INIT_COUNTRY_CMDID,

    /**
     * Nan Data commands
     * NDI - NAN Data Interface
     * NDP - NAN Data Path
     */
    /* Commands in prototyping phase */
    WMI_NDI_GET_CAP_REQ_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_PROTOTYPE),
    WMI_NDP_INITIATOR_REQ_CMDID,
    WMI_NDP_RESPONDER_REQ_CMDID,
    WMI_NDP_END_REQ_CMDID,
    WMI_NDP_CMDID,

    /** WMI commands related to HW data filtering **/
    WMI_HW_DATA_FILTER_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_HW_DATA_FILTER),

    /** WMI commands related to WLAN latency module **/
    WMI_WLM_CONFIG_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_WLM),

    /** WMI commands related to STA & AP TWT module **/
    WMI_TWT_ENABLE_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_TWT),
    WMI_TWT_DISABLE_CMDID,
    WMI_TWT_ADD_DIALOG_CMDID,
    WMI_TWT_DEL_DIALOG_CMDID,
    WMI_TWT_PAUSE_DIALOG_CMDID,
    WMI_TWT_RESUME_DIALOG_CMDID,

    /** WMI commands related to motion detection **/
    WMI_MOTION_DET_CONFIG_PARAM_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_MOTION_DET),
    WMI_MOTION_DET_BASE_LINE_CONFIG_PARAM_CMDID,
    WMI_MOTION_DET_START_STOP_CMDID,
    WMI_MOTION_DET_BASE_LINE_START_STOP_CMDID,

    /** WMI commands related to OBSS PD Spatial Reuse **/
    WMI_PDEV_OBSS_PD_SPATIAL_REUSE_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_SPATIAL_REUSE),
    WMI_PDEV_OBSS_PD_SPATIAL_REUSE_SET_DEF_OBSS_THRESH_CMDID,

    /** WMI commands related to High Precision Clock Synchronization feature **/
    WMI_HPCS_PULSE_START_CMDID = WMI_CMD_GRP_START_ID(WMI_GRP_HPCS_PULSE),
} WMI_CMD_ID;

typedef enum {
    /** WMI service is ready; after this event WMI messages can be sent/received  */
    WMI_SERVICE_READY_EVENTID=0x1,
    /** WMI is ready; after this event the wlan subsystem is initialized and can process commands. */
    WMI_READY_EVENTID,

    /**
     * Specify what WMI services the target supports
     * (for services beyond what fits in the WMI_SERVICE_READY_EVENT
     * message's wmi_service_bitmap)
     */
    WMI_SERVICE_AVAILABLE_EVENTID,

    /** Scan specific events */
    WMI_SCAN_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_SCAN) ,

    /* PDEV specific events */
    /** TPC config for the current operating channel */
    WMI_PDEV_TPC_CONFIG_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_PDEV) ,
    /** Channel stats event    */
    WMI_CHAN_INFO_EVENTID,

    /** PHY Error specific WMI event */
    WMI_PHYERR_EVENTID,

    /** eeprom dump event  */
    WMI_PDEV_DUMP_EVENTID,

    /** traffic pause event */
    WMI_TX_PAUSE_EVENTID,

    /** DFS radar event  */
    WMI_DFS_RADAR_EVENTID,

    /** track L1SS entry and residency event */
    WMI_PDEV_L1SS_TRACK_EVENTID,

    /** Report current temprature of the chip in Celcius degree */
    WMI_PDEV_TEMPERATURE_EVENTID,

    /** Extension of WMI_SERVICE_READY msg with extra target capability info */
    WMI_SERVICE_READY_EXT_EVENTID,

    /** FIPS test mode event */
    WMI_PDEV_FIPS_EVENTID,

    /** Channel hopping avoidance */
    WMI_PDEV_CHANNEL_HOPPING_EVENTID,

    /** CCK ANI level event */
    WMI_PDEV_ANI_CCK_LEVEL_EVENTID,

    /** OFDM ANI level event */
    WMI_PDEV_ANI_OFDM_LEVEL_EVENTID,

    /** Tx PPDU params */
    WMI_PDEV_TPC_EVENTID,

    /** NF Cal Power in DBR/DBM for all channels */
    WMI_PDEV_NFCAL_POWER_ALL_CHANNELS_EVENTID,

    /** SOC/PDEV events */
    WMI_PDEV_SET_HW_MODE_RESP_EVENTID,
    WMI_PDEV_HW_MODE_TRANSITION_EVENTID,
    WMI_PDEV_SET_MAC_CONFIG_RESP_EVENTID,
    /** Report ANT DIV feature's status */
    WMI_PDEV_ANTDIV_STATUS_EVENTID,
    /** Chip level Power stats */
    WMI_PDEV_CHIP_POWER_STATS_EVENTID,
    /** Power Save Failure Detected */
    WMI_PDEV_CHIP_POWER_SAVE_FAILURE_DETECTED_EVENTID,

    /* Event to report the switch count in csa of one or more VDEVs */
    WMI_PDEV_CSA_SWITCH_COUNT_STATUS_EVENTID,

    /** Report the caldata version to host */
    WMI_PDEV_CHECK_CAL_VERSION_EVENTID,

    /** Report chain RSSI and antenna index to host */
    WMI_PDEV_DIV_RSSI_ANTID_EVENTID,

    /** provide noise floor and cycle counts for a channel */
    WMI_PDEV_BSS_CHAN_INFO_EVENTID,

    /** Response received the ctl table to host */
    WMI_PDEV_UPDATE_CTLTABLE_EVENTID,

    WMI_PDEV_DMA_RING_CFG_RSP_EVENTID,

    WMI_PDEV_DMA_RING_BUF_RELEASE_EVENTID,

    /* VDEV specific events */
    WMI_PDEV_CTL_FAILSAFE_CHECK_EVENTID,

    /* Event to report the switch count in BSS color of one or more VDEVs */
    WMI_PDEV_CSC_SWITCH_COUNT_STATUS_EVENTID,

    /* Event to send cold boot calibration data */
    WMI_PDEV_COLD_BOOT_CAL_DATA_EVENTID,

    /* Event to report a rogue ap info that is detected in fw */
    WMI_PDEV_RAP_INFO_EVENTID,

    /* VDEV specific events */
    /** VDEV started event in response to VDEV_START request */
    WMI_VDEV_START_RESP_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_VDEV),
    /** vdev stopped event , generated in response to VDEV_STOP request */
    WMI_VDEV_STOPPED_EVENTID,
    /* Indicate the set key (used for setting per
     * peer unicast and per vdev multicast)
     * operation has completed */
    WMI_VDEV_INSTALL_KEY_COMPLETE_EVENTID,
    /* NOTE: WMI_VDEV_MCC_BCN_INTERVAL_CHANGE_REQ_EVENTID would be deprecated. Please
     don't use this for any new implementations */
    /* Firmware requests dynamic change to a specific beacon interval for a specific vdev ID in MCC scenario.
     This request is valid only for vdevs operating in soft AP or P2P GO mode */
    WMI_VDEV_MCC_BCN_INTERVAL_CHANGE_REQ_EVENTID,

    /* Return the TSF timestamp of specified vdev */
    WMI_VDEV_TSF_REPORT_EVENTID,

    /* FW response to Host for vdev delete cmdid */
    WMI_VDEV_DELETE_RESP_EVENTID,

    /**
     * DISA feature: FW response to Host with encrypted/decrypted
     * 802.11 DISA frame
     */
    WMI_VDEV_ENCRYPT_DECRYPT_DATA_RESP_EVENTID,
    /** event to report mac randomization success **/
    WMI_VDEV_ADD_MAC_ADDR_TO_RX_FILTER_STATUS_EVENTID,

    /* event for ARP stats collection */
    WMI_VDEV_GET_ARP_STAT_EVENTID,

    /** get tx power event in response to VDEV_GET_TX_POWER request */
    WMI_VDEV_GET_TX_POWER_EVENTID,

    WMI_VDEV_BCN_RECEPTION_STATS_EVENTID,

    /* provide LTE-Coex state */
    WMI_VDEV_GET_MWS_COEX_STATE_EVENTID,

    /* provide LTE-Coex Dynamic Power Back-off info */
    WMI_VDEV_GET_MWS_COEX_DPWB_STATE_EVENTID,

    /* provide LTE-Coex TDM info */
    WMI_VDEV_GET_MWS_COEX_TDM_STATE_EVENTID,

    /* provide LTE-Coex IDRx info */
    WMI_VDEV_GET_MWS_COEX_IDRX_STATE_EVENTID,

    /* provide LTE-Coex antenna sharing info */
    WMI_VDEV_GET_MWS_COEX_ANTENNA_SHARING_STATE_EVENTID,


    /* peer specific events */
    /** FW reauet to kick out the station for reasons like inactivity,lack of response ..etc */
    WMI_PEER_STA_KICKOUT_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_PEER),

    /** Peer Info Event with data_rate, rssi, tx_fail_cnt etc */
    WMI_PEER_INFO_EVENTID,

    /** Event indicating that TX fail count reaching threshold */
    WMI_PEER_TX_FAIL_CNT_THR_EVENTID,
    /** Return the estimate link speed for the Peer specified in the
     * WMI_PEER_GET_ESTIMATED_LINKSPEED_CMDID command.
     */
    WMI_PEER_ESTIMATED_LINKSPEED_EVENTID,
    /* Return the peer state
     * WMI_PEER_SET_PARAM_CMDID, WMI_PEER_AUTHORIZE
     */
    WMI_PEER_STATE_EVENTID,

    /* Peer Assoc Conf event to confirm fw had received PEER_ASSOC_CMD.
     * After that, host will send Mx message.
     * Otherwise, host will pause any Mx(STA:M2/M4) message
     */
    WMI_PEER_ASSOC_CONF_EVENTID,

    /* FW response to Host for peer delete cmdid */
    WMI_PEER_DELETE_RESP_EVENTID,

    /** Valid rate code list for peer  */
    WMI_PEER_RATECODE_LIST_EVENTID,
    WMI_WDS_PEER_EVENTID,
    WMI_PEER_STA_PS_STATECHG_EVENTID,
    /** Peer Ant Div Info Event with rssi per chain, etc */
    WMI_PEER_ANTDIV_INFO_EVENTID,

    /*
     * WMI_PEER_RESERVED_EVENTID
     * These values are used for placeholders, to allow the subsequent
     * WMI_PEER_OPER_MODE_CHANGE_EVENTID constant to have the same value
     * as it had in its original location, when it was mistakenly placed
     * amongst the WMI_PEER CMDID defs.
     *
     * These WMI_PEER_RESERVED values can be replaced with actual WMI peer
     * event message IDs, though it will be simpler to instead add new
     * WMI_PEER EVENTID defs at the end of the WMI_GRP_PEER WMI_EVT_GRP.
     */
    WMI_PEER_RESERVED0_EVENTID,
    WMI_PEER_RESERVED1_EVENTID,
    WMI_PEER_RESERVED2_EVENTID,
    WMI_PEER_RESERVED3_EVENTID,
    WMI_PEER_RESERVED4_EVENTID,
    WMI_PEER_RESERVED5_EVENTID,
    WMI_PEER_RESERVED6_EVENTID,
    WMI_PEER_RESERVED7_EVENTID,
    WMI_PEER_RESERVED8_EVENTID,
    WMI_PEER_RESERVED9_EVENTID,
    WMI_PEER_RESERVED10_EVENTID,
    /** Peer operating mode change indication sent to host to update stats */
    WMI_PEER_OPER_MODE_CHANGE_EVENTID,

    /** WMI Event to deliver CTL Failsafe application */
    WMI_PEER_TX_PN_RESPONSE_EVENTID,

    WMI_PEER_CFR_CAPTURE_EVENTID,

    /* beacon/mgmt specific events */
    /** RX management frame. the entire frame is carried along with the event.  */
    WMI_MGMT_RX_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_MGMT),
    /** software beacon alert event to Host requesting host to Queue a beacon for transmission
        use only in host beacon mode */
    WMI_HOST_SWBA_EVENTID,
    /** beacon tbtt offset event indicating the tsf offset of the tbtt from the theoretical value.
        tbtt offset is normally 0 and will be non zero if there are multiple VDEVs operating in
        staggered beacon transmission mode */
    WMI_TBTTOFFSET_UPDATE_EVENTID,

    /** event after the first beacon is transmitted following
             a change in the template.*/
    WMI_OFFLOAD_BCN_TX_STATUS_EVENTID,
    /** event after the first probe response is transmitted following
             a change in the template.*/
    WMI_OFFLOAD_PROB_RESP_TX_STATUS_EVENTID,
    /** Event for Mgmt TX completion event */
    WMI_MGMT_TX_COMPLETION_EVENTID,
    /** Event for Mgmt TX bundle completion event */
    WMI_MGMT_TX_BUNDLE_COMPLETION_EVENTID,
    /** vdev_map used in WMI_TBTTOFFSET_UPDATE_EVENTID supports max 32 vdevs
     *   Use this event if number of vdevs > 32.
     */
    WMI_TBTTOFFSET_EXT_UPDATE_EVENTID,
    /** Event for offchan data TX completion event */
    WMI_OFFCHAN_DATA_TX_COMPLETION_EVENTID,

    /** software FILS Discovery Frame alert event to Host, requesting host to Queue an FD frame for transmission */
    WMI_HOST_SWFDA_EVENTID,

    /** software beacon alert event to Host requesting host to Queue a beacon for transmission.
     *   Used only in host beacon mode. */
    WMI_HOST_SWBA_V2_EVENTID,

    /* ADDBA Related WMI Events*/
    /** Indication the completion of the prior
       WMI_PEER_TID_DELBA_CMDID(initiator) */
    WMI_TX_DELBA_COMPLETE_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_BA_NEG),
    /** Indication the completion of the prior
       *WMI_PEER_TID_ADDBA_CMDID(initiator) */
    WMI_TX_ADDBA_COMPLETE_EVENTID,

    /* Seq num returned from hw for a sta/tid pair */
    WMI_BA_RSP_SSN_EVENTID,

    /* Aggregation state requested by BTC */
    WMI_AGGR_STATE_TRIG_EVENTID,

    /** Roam event to trigger roaming on host */
    WMI_ROAM_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_ROAM),

    /** matching AP found from list of profiles */
    WMI_PROFILE_MATCH,
    /** roam synch event */
    WMI_ROAM_SYNCH_EVENTID,
    /** roam synch frame event */
    WMI_ROAM_SYNCH_FRAME_EVENTID,
    /** various roam scan stats */
    WMI_ROAM_SCAN_STATS_EVENTID,
    /** Blacklisted AP information event */
    WMI_ROAM_BLACKLIST_EVENTID,

    /** P2P disc found */
    WMI_P2P_DISC_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_P2P),
    /*send noa info to host when noa is changed for beacon tx offload enable*/
    WMI_P2P_NOA_EVENTID,
    /** send p2p listen offload stopped event with different reason */
    WMI_P2P_LISTEN_OFFLOAD_STOPPED_EVENTID,

    /** Send EGAP Info to host */
    WMI_AP_PS_EGAP_INFO_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_AP_PS),

    /* send pdev resume event to host after pdev resume. */
    WMI_PDEV_RESUME_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_SUSPEND),

    /** WOW wake up host event.generated in response to WMI_WOW_HOSTWAKEUP_FROM_SLEEP_CMDID.
        will cary wake reason */
    WMI_WOW_WAKEUP_HOST_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_WOW),
    WMI_D0_WOW_DISABLE_ACK_EVENTID,
    WMI_WOW_INITIAL_WAKEUP_EVENTID,

    /*RTT related event ID*/
    /** RTT measurement report */
    WMI_RTT_MEASUREMENT_REPORT_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_RTT),
    /** TSF measurement report */
    WMI_TSF_MEASUREMENT_REPORT_EVENTID,
    /** RTT error report */
    WMI_RTT_ERROR_REPORT_EVENTID,
    /*STATS specific events*/
    /** txrx stats event requested by host */
    WMI_STATS_EXT_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_STATS),
    /** FW iface link stats Event  */
    WMI_IFACE_LINK_STATS_EVENTID,
    /** FW iface peer link stats Event  */
    WMI_PEER_LINK_STATS_EVENTID,
    /** FW Update radio stats Event  */
    WMI_RADIO_LINK_STATS_EVENTID,

    /**  Firmware memory dump Complete event*/
    WMI_UPDATE_FW_MEM_DUMP_EVENTID,

    /** Event indicating the DIAG logs/events supported by FW */
    WMI_DIAG_EVENT_LOG_SUPPORTED_EVENTID,

    /** Instantaneous RSSI event */
    WMI_INST_RSSI_STATS_EVENTID,

    /** FW update tx power levels event */
    WMI_RADIO_TX_POWER_LEVEL_STATS_EVENTID,
    /** This event is used to report wlan stats to host.
     * It is triggered under 3 conditions:
     * (a) Periodic timer timed out, based on the period specified
     *     by WMI_PDEV_PARAM_STATS_OBSERVATION_PERIOD
     * (b) Whenever any of the (enabled) stats thresholds specified
     *     in the WMI_PDEV_SET_STATS_THRESHOLD_CMD message is exceeded
     *     within the current stats period.
     * (c) In response to the one-time wlan stats request of
     *     WMI_REQUEST_WLAN_STATS_CMDID from host.
     *
     *  If this event is triggered by condition a or b,
     *  the stats counters are cleared at the start of each period.
     *  But if it is triggered by condition c, stats counters won't be cleared.
     */
    WMI_REPORT_STATS_EVENTID,

    /** Event indicating RCPI of the peer requested by host in the
     * WMI_REQUEST_RCPI_CMDID */
    WMI_UPDATE_RCPI_EVENTID,

    /** This event is used to respond to WMI_REQUEST_PEER_STATS_INFO_CMDID
     *  and report peer stats info to host */
    WMI_PEER_STATS_INFO_EVENTID,

    /** This event is used to respond to WMI_REQUEST_RADIO_CHAN_STATS_CMDID
    *  and report radio channel stats to host */
    WMI_RADIO_CHAN_STATS_EVENTID,

    /** This event is used to respond to WMI_REQUEST_WLM_STATS_CMDID
     *  and report WLM (WLAN latency manager) stats info to host */
    WMI_WLM_STATS_EVENTID,


    /* NLO specific events */
    /** NLO match event after the first match */
    WMI_NLO_MATCH_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_NLO_OFL),

    /** NLO scan complete event */
    WMI_NLO_SCAN_COMPLETE_EVENTID,

    /** APFIND specific events */
    WMI_APFIND_EVENTID,

    /** passpoint network match event */
    WMI_PASSPOINT_MATCH_EVENTID,

    /** GTK offload stautus event requested by host */
    WMI_GTK_OFFLOAD_STATUS_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_GTK_OFL),

	/** GTK offload failed to rekey event */
	WMI_GTK_REKEY_FAIL_EVENTID,
    /* CSA IE received event */
    WMI_CSA_HANDLING_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_CSA_OFL),

    /*chatter query reply event*/
    WMI_CHATTER_PC_QUERY_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_CHATTER),

    /** DFS related events */
    WMI_PDEV_DFS_RADAR_DETECTION_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_DFS),
    /** Indicate channel-availability-check completion event to host */
    WMI_VDEV_DFS_CAC_COMPLETE_EVENTID,
    /** Indicate off-channel-availability-check completion event to host */
    WMI_VDEV_ADFS_OCAC_COMPLETE_EVENTID,
    /** echo event in response to echo command */
    WMI_ECHO_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_MISC),

	/* !!IMPORTANT!!
	  * If you need to add a new WMI event ID to the WMI_GRP_MISC sub-group,
	  * please make sure you add it BEHIND WMI_PDEV_UTF_EVENTID,
	  * as we MUST have a fixed value here to maintain compatibility between
	  * UTF and the ART2 driver
	*/
	/** UTF specific WMI event */
	WMI_PDEV_UTF_EVENTID,

    /** event carries buffered debug messages  */
    WMI_DEBUG_MESG_EVENTID,
    /** FW stats(periodic or on shot)  */
    WMI_UPDATE_STATS_EVENTID,
    /** debug print message used for tracing FW code while debugging  */
    WMI_DEBUG_PRINT_EVENTID,
    /** DCS wlan or non-wlan interference event
     */
    WMI_DCS_INTERFERENCE_EVENTID,
    /** VI spoecific event  */
    WMI_PDEV_QVIT_EVENTID,
    /** FW code profile data in response to profile request  */
    WMI_WLAN_PROFILE_DATA_EVENTID,
    /* Factory Testing Mode request event
     * used for integrated chipsets */
    WMI_PDEV_FTM_INTG_EVENTID,
    /* avoid list of frequencies .
     */
    WMI_WLAN_FREQ_AVOID_EVENTID,
    /* Indicate the keepalive parameters */
    WMI_VDEV_GET_KEEPALIVE_EVENTID,
    /* Thermal Management event */
    WMI_THERMAL_MGMT_EVENTID,

    /* Container for QXDM/DIAG events */
    WMI_DIAG_DATA_CONTAINER_EVENTID,

     /* host auto shutdown event */
    WMI_HOST_AUTO_SHUTDOWN_EVENTID,

    /*update mib counters together with WMI_UPDATE_STATS_EVENTID*/
    WMI_UPDATE_WHAL_MIB_STATS_EVENTID,

    /*update ht/vht info based on vdev (rx and tx NSS and preamble)*/
    WMI_UPDATE_VDEV_RATE_STATS_EVENTID,

    WMI_DIAG_EVENTID,

    /** Set OCB Sched Response, deprecated */
    WMI_OCB_SET_SCHED_EVENTID,

    /** event to indicate the flush of the buffered debug messages is complete*/
    WMI_DEBUG_MESG_FLUSH_COMPLETE_EVENTID,

    /** event to report mix/max RSSI breach events */
    WMI_RSSI_BREACH_EVENTID,

    /** event to report completion of data storage into flash memory */
    WMI_TRANSFER_DATA_TO_FLASH_COMPLETE_EVENTID,

    /** event to report SCPC calibrated data to host */
    WMI_PDEV_UTF_SCPC_EVENTID,

    /** event to provide requested data from the target's flash memory */
    WMI_READ_DATA_FROM_FLASH_EVENTID,

    /** event to report rx aggregation failure frame information */
    WMI_REPORT_RX_AGGR_FAILURE_EVENTID,

    /** event to upload a PKGID to host to identify chip for various products */
    WMI_PKGID_EVENTID,

    /* Thermal Throttling stats event id for every pdev and zones, etc */
    WMI_THERM_THROT_STATS_EVENTID,

    /* WMI UNIT TEST event */
    WMI_UNIT_TEST_EVENTID,

    /** event to report result of host configure SAR2 */
    WMI_SAR2_RESULT_EVENTID,

    /** event to get TX power per input HALPHY parameters */
    WMI_GET_TPC_POWER_EVENTID,

    /* GPIO Event */
    WMI_GPIO_INPUT_EVENTID=WMI_EVT_GRP_START_ID(WMI_GRP_GPIO),
    /** upload H_CV info WMI event
     * to indicate uploaded H_CV info to host
     */
    WMI_UPLOADH_EVENTID,

    /** capture H info WMI event
     * to indicate captured H info to host
     */
    WMI_CAPTUREH_EVENTID,
    /* hw RFkill */
    WMI_RFKILL_STATE_CHANGE_EVENTID,

    /* TDLS Event */
    WMI_TDLS_PEER_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_TDLS),

    /** STA SMPS Event */
    /** force SMPS mode */
    WMI_STA_SMPS_FORCE_MODE_COMPLETE_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_STA_SMPS),

    /*location scan event*/
    /*report the firmware's capability of batch scan*/
    WMI_BATCH_SCAN_ENABLED_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_LOCATION_SCAN),
    /*batch scan result*/
    WMI_BATCH_SCAN_RESULT_EVENTID,
    /* OEM Event */
    WMI_OEM_CAPABILITY_EVENTID=WMI_EVT_GRP_START_ID(WMI_GRP_OEM), /*DEPRECATED*/
    WMI_OEM_MEASUREMENT_REPORT_EVENTID, /* DEPRECATED */
    WMI_OEM_ERROR_REPORT_EVENTID, /* DEPRECATED */
    WMI_OEM_RESPONSE_EVENTID,
    WMI_OEM_DMA_RING_CFG_RSP_EVENTID,
    WMI_OEM_DMA_BUF_RELEASE_EVENTID,

    /* NAN Event */
    WMI_NAN_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_NAN),
    WMI_NAN_DISC_IFACE_CREATED_EVENTID,
    WMI_NAN_DISC_IFACE_DELETED_EVENTID,
    WMI_NAN_STARTED_CLUSTER_EVENTID,
    WMI_NAN_JOINED_CLUSTER_EVENTID,

    /* Coex Event */
    WMI_COEX_REPORT_ANTENNA_ISOLATION_EVENTID =
            WMI_EVT_GRP_START_ID(WMI_GRP_COEX),
    WMI_SAR_GET_LIMITS_EVENTID,

    /* LPI Event */
    WMI_LPI_RESULT_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_LPI),
    WMI_LPI_STATUS_EVENTID,
    WMI_LPI_HANDOFF_EVENTID,

     /* ExtScan events */
    WMI_EXTSCAN_START_STOP_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_EXTSCAN),
    WMI_EXTSCAN_OPERATION_EVENTID,
    WMI_EXTSCAN_TABLE_USAGE_EVENTID,
    WMI_EXTSCAN_CACHED_RESULTS_EVENTID,
    WMI_EXTSCAN_WLAN_CHANGE_RESULTS_EVENTID,
    WMI_EXTSCAN_HOTLIST_MATCH_EVENTID,
    WMI_EXTSCAN_CAPABILITIES_EVENTID,
    WMI_EXTSCAN_HOTLIST_SSID_MATCH_EVENTID,

    /* mDNS offload events */
    WMI_MDNS_STATS_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_MDNS_OFL),

    /* SAP Authentication offload events */
    WMI_SAP_OFL_ADD_STA_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_SAP_OFL),
    WMI_SAP_OFL_DEL_STA_EVENTID,
    WMI_SAP_OBSS_DETECTION_REPORT_EVENTID,
    /* OBSS Offloads events */
    WMI_OBSS_COLOR_COLLISION_DETECTION_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_OBSS_OFL),
    /** Out-of-context-of-bss (OCB) events */
    WMI_OCB_SET_CONFIG_RESP_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_OCB),
    WMI_OCB_GET_TSF_TIMER_RESP_EVENTID,
    WMI_DCC_GET_STATS_RESP_EVENTID,
    WMI_DCC_UPDATE_NDL_RESP_EVENTID,
    WMI_DCC_STATS_EVENTID,

    /* System-On-Chip events */
    WMI_SOC_SET_HW_MODE_RESP_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_SOC),
    WMI_SOC_HW_MODE_TRANSITION_EVENTID,
    WMI_SOC_SET_DUAL_MAC_CONFIG_RESP_EVENTID,
    /** Motion Aided WiFi Connectivity (MAWC) events */
    WMI_MAWC_ENABLE_SENSOR_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_MAWC),

    /** pkt filter (BPF) offload relevant events */
    WMI_BPF_CAPABILIY_INFO_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_BPF_OFFLOAD),
    WMI_BPF_VDEV_STATS_INFO_EVENTID,
    WMI_BPF_GET_VDEV_WORK_MEMORY_RESP_EVENTID,

    /* RMC specific event */
    /* RMC manual leader selected event */
    WMI_RMC_NEW_LEADER_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_RMC),

    /** WMI events related to regulatory offload */
    WMI_REG_CHAN_LIST_CC_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_REGULATORY),
    WMI_11D_NEW_COUNTRY_EVENTID,

    /** Events for TWT(Target Wake Time) of STA and AP  */
    WMI_TWT_ENABLE_COMPLETE_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_TWT),
    WMI_TWT_DISABLE_COMPLETE_EVENTID,
    WMI_TWT_ADD_DIALOG_COMPLETE_EVENTID,
    WMI_TWT_DEL_DIALOG_COMPLETE_EVENTID,
    WMI_TWT_PAUSE_DIALOG_COMPLETE_EVENTID,
    WMI_TWT_RESUME_DIALOG_COMPLETE_EVENTID,

    /** Events in Prototyping phase */
    WMI_NDI_CAP_RSP_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_PROTOTYPE),
    WMI_NDP_INITIATOR_RSP_EVENTID,
    WMI_NDP_RESPONDER_RSP_EVENTID,
    WMI_NDP_END_RSP_EVENTID,
    WMI_NDP_INDICATION_EVENTID,
    WMI_NDP_CONFIRM_EVENTID,
    WMI_NDP_END_INDICATION_EVENTID,
    WMI_WLAN_COEX_BT_ACTIVITY_EVENTID,
    WMI_NDL_SCHEDULE_UPDATE_EVENTID,
    WMI_NDP_EVENTID,

    /** WMI events related to motion detection */
    WMI_MOTION_DET_HOST_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_MOTION_DET),
    WMI_MOTION_DET_BASE_LINE_HOST_EVENTID,

    /** WMI events related to Estimation of Service Parameters (802.11mc) */
    WMI_ESP_ESTIMATE_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_ESP),
} WMI_EVT_ID;

/* defines for OEM message sub-types */
#define WMI_OEM_CAPABILITY_REQ     0x01
#define WMI_OEM_CAPABILITY_RSP     0x02
#define WMI_OEM_MEASUREMENT_REQ    0x03
#define WMI_OEM_MEASUREMENT_RSP    0x04
#define WMI_OEM_ERROR_REPORT_RSP   0x05
#define WMI_OEM_NAN_MEAS_REQ       0x06
#define WMI_OEM_NAN_MEAS_RSP       0x07
#define WMI_OEM_NAN_PEER_INFO      0x08
#define WMI_OEM_CONFIGURE_LCR      0x09
#define WMI_OEM_CONFIGURE_LCI      0x0A


/* below message subtype is internal to CLD. Target should
 * never use internal response type
 */
#define WMI_OEM_INTERNAL_RSP       0xdeadbeef

#define WMI_CHAN_LIST_TAG		 0x1
#define WMI_SSID_LIST_TAG		 0x2
#define WMI_BSSID_LIST_TAG		 0x3
#define WMI_IE_TAG			 0x4
#define WMI_SCAN_START_OFFSET_TAG        0x5

typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_channel */
    /** primary 20 MHz channel frequency in mhz */
    A_UINT32 mhz;
    /** Center frequency 1 in MHz*/
    A_UINT32 band_center_freq1;
    /** Center frequency 2 in MHz - valid only for 11acvht 80plus80 mode*/
    A_UINT32 band_center_freq2;
    /** channel info described below */
    A_UINT32 info;
    /** contains min power, max power, reg power and reg class id.  */
    A_UINT32 reg_info_1;
    /** contains antennamax */
    A_UINT32 reg_info_2;
} wmi_channel;

typedef enum{
WMI_CHANNEL_CHANGE_CAUSE_NONE = 0,
WMI_CHANNEL_CHANGE_CAUSE_CSA,
}wmi_channel_change_cause;

/** channel info consists of 6 bits of channel mode */

#define WMI_SET_CHANNEL_MODE(pwmi_channel,val) do { \
     (pwmi_channel)->info &= 0xffffffc0;            \
     (pwmi_channel)->info |= (val);                 \
     } while(0)

#define WMI_GET_CHANNEL_MODE(pwmi_channel) ((pwmi_channel)->info & 0x0000003f )

#define WMI_CHAN_FLAG_HT40_PLUS   6
#define WMI_CHAN_FLAG_PASSIVE     7
#define WMI_CHAN_ADHOC_ALLOWED    8
#define WMI_CHAN_AP_DISABLED      9
#define WMI_CHAN_FLAG_DFS         10
#define WMI_CHAN_FLAG_ALLOW_HT    11  /* HT is allowed on this channel */
#define WMI_CHAN_FLAG_ALLOW_VHT   12  /* VHT is allowed on this channel */
#define WMI_CHANNEL_CHANGE_CAUSE_CSA 13 /*Indicate reason for channel switch */
#define WMI_CHAN_FLAG_HALF_RATE     14  /* Indicates half rate channel */
#define WMI_CHAN_FLAG_QUARTER_RATE  15  /* Indicates quarter rate channel */
#define WMI_CHAN_FLAG_DFS_CFREQ2  16 /* Enable radar event reporting for sec80 in VHT80p80 */
#define WMI_CHAN_FLAG_ALLOW_HE    17 /* HE (11ax) is allowed on this channel */

#define WMI_SET_CHANNEL_FLAG(pwmi_channel,flag) do { \
        (pwmi_channel)->info |=  (1 << flag);      \
     } while(0)

#define WMI_GET_CHANNEL_FLAG(pwmi_channel,flag)   \
        (((pwmi_channel)->info & (1 << flag)) >> flag)

#define WMI_SET_CHANNEL_MIN_POWER(pwmi_channel,val) do { \
     (pwmi_channel)->reg_info_1 &= 0xffffff00;           \
     (pwmi_channel)->reg_info_1 |= (val&0xff);           \
     } while(0)
#define WMI_GET_CHANNEL_MIN_POWER(pwmi_channel) ((pwmi_channel)->reg_info_1 & 0xff )

#define WMI_SET_CHANNEL_MAX_POWER(pwmi_channel,val) do { \
     (pwmi_channel)->reg_info_1 &= 0xffff00ff;           \
     (pwmi_channel)->reg_info_1 |= ((val&0xff) << 8);    \
     } while(0)
#define WMI_GET_CHANNEL_MAX_POWER(pwmi_channel) ( (((pwmi_channel)->reg_info_1) >> 8) & 0xff )

#define WMI_SET_CHANNEL_REG_POWER(pwmi_channel,val) do { \
     (pwmi_channel)->reg_info_1 &= 0xff00ffff;           \
     (pwmi_channel)->reg_info_1 |= ((val&0xff) << 16);   \
     } while(0)
#define WMI_GET_CHANNEL_REG_POWER(pwmi_channel) ( (((pwmi_channel)->reg_info_1) >> 16) & 0xff )
#define WMI_SET_CHANNEL_REG_CLASSID(pwmi_channel,val) do { \
     (pwmi_channel)->reg_info_1 &= 0x00ffffff;             \
     (pwmi_channel)->reg_info_1 |= ((val&0xff) << 24);     \
     } while(0)
#define WMI_GET_CHANNEL_REG_CLASSID(pwmi_channel) ( (((pwmi_channel)->reg_info_1) >> 24) & 0xff )

#define WMI_SET_CHANNEL_ANTENNA_MAX(pwmi_channel,val) do { \
     (pwmi_channel)->reg_info_2 &= 0xffffff00;             \
     (pwmi_channel)->reg_info_2 |= (val&0xff);             \
     } while(0)
#define WMI_GET_CHANNEL_ANTENNA_MAX(pwmi_channel) ((pwmi_channel)->reg_info_2 & 0xff )

/* max tx power is in 1 dBm units */
#define WMI_SET_CHANNEL_MAX_TX_POWER(pwmi_channel,val) do { \
     (pwmi_channel)->reg_info_2 &= 0xffff00ff;              \
     (pwmi_channel)->reg_info_2 |= ((val&0xff)<<8);         \
     } while(0)
#define WMI_GET_CHANNEL_MAX_TX_POWER(pwmi_channel) ( (((pwmi_channel)->reg_info_2)>>8) & 0xff )


/** HT Capabilities*/
#define WMI_HT_CAP_ENABLED                0x0001   /* HT Enabled/ disabled */
#define WMI_HT_CAP_HT20_SGI               0x0002   /* Short Guard Interval with HT20 */
#define WMI_HT_CAP_DYNAMIC_SMPS           0x0004   /* Dynamic MIMO powersave */
#define WMI_HT_CAP_TX_STBC                0x0008   /* B3 TX STBC */
#define WMI_HT_CAP_TX_STBC_MASK_SHIFT     3
#define WMI_HT_CAP_RX_STBC                0x0030   /* B4-B5 RX STBC */
#define WMI_HT_CAP_RX_STBC_MASK_SHIFT     4
#define WMI_HT_CAP_LDPC                   0x0040   /* LDPC supported */
#define WMI_HT_CAP_L_SIG_TXOP_PROT        0x0080   /* L-SIG TXOP Protection */
#define WMI_HT_CAP_MPDU_DENSITY           0x0700   /* MPDU Density */
#define WMI_HT_CAP_MPDU_DENSITY_MASK_SHIFT 8
#define WMI_HT_CAP_HT40_SGI               0x0800
#define WMI_HT_CAP_RX_LDPC                0x1000   /* LDPC RX support */
#define WMI_HT_CAP_TX_LDPC                0x2000   /* LDPC TX support */

/* These macros should be used when we wish to advertise STBC support for
 * only 1SS or 2SS or 3SS. */
#define WMI_HT_CAP_RX_STBC_1SS            0x0010   /* B4-B5 RX STBC */
#define WMI_HT_CAP_RX_STBC_2SS            0x0020   /* B4-B5 RX STBC */
#define WMI_HT_CAP_RX_STBC_3SS            0x0030   /* B4-B5 RX STBC */


#define WMI_HT_CAP_DEFAULT_ALL (WMI_HT_CAP_ENABLED       | \
                                WMI_HT_CAP_HT20_SGI      | \
                                WMI_HT_CAP_HT40_SGI      | \
                                WMI_HT_CAP_TX_STBC       | \
                                WMI_HT_CAP_RX_STBC       | \
				WMI_HT_CAP_LDPC          | \
                                WMI_HT_CAP_TX_LDPC       | \
                                WMI_HT_CAP_RX_LDPC)

/* WMI_VHT_CAP_* these maps to ieee 802.11ac vht capability information
   field. The fields not defined here are not supported, or reserved.
   Do not change these masks and if you have to add new one follow the
   bitmask as specified by 802.11ac draft.
*/

#define WMI_VHT_CAP_MAX_MPDU_LEN_7935            0x00000001
#define WMI_VHT_CAP_MAX_MPDU_LEN_11454           0x00000002
#define WMI_VHT_CAP_MAX_MPDU_LEN_MASK            0x00000003
#define WMI_VHT_CAP_CH_WIDTH_160MHZ              0x00000004
#define WMI_VHT_CAP_CH_WIDTH_80P80_160MHZ        0x00000008
#define WMI_VHT_CAP_RX_LDPC                      0x00000010
#define WMI_VHT_CAP_SGI_80MHZ                    0x00000020
#define WMI_VHT_CAP_SGI_160MHZ                   0x00000040
#define WMI_VHT_CAP_TX_STBC                      0x00000080
#define WMI_VHT_CAP_RX_STBC_MASK                 0x00000300
#define WMI_VHT_CAP_RX_STBC_MASK_SHIFT           8
#define WMI_VHT_CAP_SU_BFORMER                   0x00000800
#define WMI_VHT_CAP_SU_BFORMEE                   0x00001000
#define WMI_VHT_CAP_MAX_CS_ANT_MASK              0x0000E000
#define WMI_VHT_CAP_MAX_CS_ANT_MASK_SHIFT        13
#define WMI_VHT_CAP_MAX_SND_DIM_MASK             0x00070000
#define WMI_VHT_CAP_MAX_SND_DIM_MASK_SHIFT       16
#define WMI_VHT_CAP_MU_BFORMER                   0x00080000
#define WMI_VHT_CAP_MU_BFORMEE                   0x00100000
#define WMI_VHT_CAP_TXOP_PS                      0x00200000
#define WMI_VHT_CAP_MAX_AMPDU_LEN_EXP            0x03800000
#define WMI_VHT_CAP_MAX_AMPDU_LEN_EXP_SHIFT      23
#define WMI_VHT_CAP_RX_FIXED_ANT                 0x10000000
#define WMI_VHT_CAP_TX_FIXED_ANT                 0x20000000
#define WMI_VHT_CAP_TX_LDPC                      0x40000000

/* TEMPORARY:
 * Preserve the incorrect old name as an alias for the correct new name
 * until all references to the old name have been removed from all hosts
 * and targets.
 */
#define WMI_VHT_CAP_MAX_AMPDU_LEN_EXP_SHIT WMI_VHT_CAP_MAX_AMPDU_LEN_EXP_SHIFT

/* These macros should be used when we wish to advertise STBC support for
 * only 1SS or 2SS or 3SS. */
#define WMI_VHT_CAP_RX_STBC_1SS 0x00000100
#define WMI_VHT_CAP_RX_STBC_2SS 0x00000200
#define WMI_VHT_CAP_RX_STBC_3SS 0x00000300

/* TEMPORARY:
 * Preserve the incorrect old name as an alias for the correct new name
 * until all references to the old name have been removed from all hosts
 * and targets.
 */
#define WMI_vHT_CAP_RX_STBC_3SS WMI_VHT_CAP_RX_STBC_3SS

#define WMI_VHT_CAP_DEFAULT_ALL (WMI_VHT_CAP_MAX_MPDU_LEN_11454  |      \
                                 WMI_VHT_CAP_SGI_80MHZ           |      \
                                 WMI_VHT_CAP_TX_STBC             |      \
                                 WMI_VHT_CAP_RX_STBC_MASK        |      \
                                 WMI_VHT_CAP_RX_LDPC             |      \
				 WMI_VHT_CAP_TX_LDPC             |      \
                                 WMI_VHT_CAP_MAX_AMPDU_LEN_EXP   |      \
                                 WMI_VHT_CAP_RX_FIXED_ANT        |      \
                                 WMI_VHT_CAP_TX_FIXED_ANT)

/* Interested readers refer to Rx/Tx MCS Map definition as defined in
   802.11ac
*/
#define WMI_VHT_MAX_MCS_EXT_SS_GET(vht_mcs_map, index) WMI_GET_BITS(vht_mcs_map, 16 + index, 1)
#define WMI_VHT_MAX_MCS_EXT_SS_SET(vht_mcs_map, index, value) WMI_SET_BITS(vht_mcs_map, 16 + index, 1, value)

/* Notification bit for Ext MCS 10/11 support */
#define WMI_VHT_MCS_NOTIFY_EXT_SS_GET(vht_mcs_map) WMI_GET_BITS(vht_mcs_map, 24, 1)
#define WMI_VHT_MCS_NOTIFY_EXT_SS_SET(vht_mcs_map, value) WMI_SET_BITS(vht_mcs_map, 24, 1, value)

#define WMI_VHT_MAX_MCS_4_SS_MASK(r,ss)      ((3 & (r)) << (((ss) - 1) << 1))
#define WMI_VHT_MAX_SUPP_RATE_MASK           0x1fff0000
#define WMI_VHT_MAX_SUPP_RATE_MASK_SHIFT     16

/** 11ax capabilities */
#define WMI_HE_CAP_PPE_PRESENT            0x00000001
#define WMI_HE_CAP_TWT_RESPONDER_SUPPORT  0x00000002
#define WMI_HE_CAP_TWT_REQUESTER_SUPPORT  0x00000004
#define WMI_HE_FRAG_SUPPORT_MASK          0x00000018
#define WMI_HE_FRAG_SUPPORT_SHIFT         3

#define WMI_HE_CAP_1X_LTF_400NS_GI_SUPPORT      0x00000001
#define WMI_HE_CAP_2X_LTF_400NS_GI_SUPPORT      0x00000002
#define WMI_HE_CAP_2X_LTF_160_80_80_SUPPORT     0x00000004

#define WMI_HE_CAP_1X_LTF_400NS_GI_SUPPORT_GET(he_cap_info_dword1) \
    WMI_GET_BITS(he_cap_info_dword1, 0, 1)
#define WMI_HE_CAP_1X_LTF_400NS_GI_SUPPORT_SET(he_cap_info_dword1, value) \
    WMI_SET_BITS(he_cap_info_dword1, 0, 1, value)

#define WMI_HE_CAP_2X_LTF_400NS_GI_SUPPORT_GET(he_cap_info_dword1) \
    WMI_GET_BITS(he_cap_info_dword1, 1, 1)
#define WMI_HE_CAP_2X_LTF_400NS_GI_SUPPORT_SET(he_cap_info_dword1, value) \
    WMI_SET_BITS(he_cap_info_dword1, 1, 1, value)

#define WMI_HE_CAP_2X_LTF_160_80_80_SUPPORT_GET(he_cap_info_dword1) \
    WMI_GET_BITS(he_cap_info_dword1, 2, 1)
#define WMI_HE_CAP_2X_LTF_160_80_80_SUPPORT_SET(he_cap_info_dword1, value) \
    WMI_SET_BITS(he_cap_info_dword1, 2, 1, value)

/* Interested readers refer to Rx/Tx MCS Map definition as defined in 802.11ax
 */
#define WMI_HE_MAX_MCS_4_SS_MASK(r,ss)      ((3 & (r)) << (((ss) - 1) << 1))

/* fragmentation support field value */
enum {
    WMI_HE_FRAG_SUPPORT_LEVEL0, /* No Fragmentation support */
    WMI_HE_FRAG_SUPPORT_LEVEL1, /* support for fragments within a VHT single MPDU, no support for fragments within AMPDU */
    WMI_HE_FRAG_SUPPORT_LEVEL2, /* support for up to 1 fragment per MSDU within a single A-MPDU */
    WMI_HE_FRAG_SUPPORT_LEVEL3, /* support for multiple fragments per MSDU within an A-MPDU */
};

/** NOTE: This defs cannot be changed in the future without breaking WMI compatibility */
#define WMI_MAX_NUM_SS                    MAX_HE_NSS
#define WMI_MAX_NUM_RU                    MAX_HE_RU

/*
 * Figure 8 554ae: -PPE Threshold Info field format
 * we pack PPET16 and PPT8 for four RU's in one element of array.
 *
 * ppet16_ppet8_ru3_ru0 array element 0 holds:
 *     |  PPET8 | PPET16 | PPET8  | PPET16 | PPET8  | PPET16 | PPET8  | PPET16 |
 *rsvd |NSS1,RU4|NSS1,RU4|NSS1,RU3|NSS1,RU3|NSS1,RU2|NSS1,RU2|NSS1,RU1|NSS1,RU1|
 *31:23|  22:20 |  19:17 |  17:15 |  14:12 |  11:9  |   8:6  |   5:3  |   2:0  |
 *
 * ppet16_ppet8_ru3_ru0 array element 1 holds:
 *     | PPET8  | PPET16 | PPET8  | PPET16 | PPET8  | PPET16 | PPET8  | PPET16 |
 *rsvd |NSS2,RU4|NSS2,RU4|NSS2,RU3|NSS2,RU3|NSS2,RU2|NSS2,RU2|NSS2,RU1|NSS2,RU1|
 *31:23|  22:20 |  19:17 |  17:15 |  14:12 |  11:9  |   8:6  |   5:3  |   2:0  |
 *
 * etc.
 */

/*
 * Note that in these macros, "ru" is one-based, not zero-based, while
 * nssm1 is zero-based.
 */
#define WMI_SET_PPET16(ppet16_ppet8_ru3_ru0, ru, nssm1, ppet) \
    do { \
        ppet16_ppet8_ru3_ru0[nssm1] &= ~(7 << (((ru-1)&3)*6));       \
        ppet16_ppet8_ru3_ru0[nssm1] |= ((ppet&7) << (((ru-1)&3)*6)); \
    } while (0)

#define WMI_GET_PPET16(ppet16_ppet8_ru3_ru0, ru, nssm1) \
    ((ppet16_ppet8_ru3_ru0[nssm1] >> (((ru-1)&3)*6))&7)

#define WMI_SET_PPET8(ppet16_ppet8_ru3_ru0, ru, nssm1, ppet) \
    do { \
        ppet16_ppet8_ru3_ru0[nssm1] &= ~(7 << (((ru-1)&3)*6+3));       \
        ppet16_ppet8_ru3_ru0[nssm1] |= ((ppet&7) << (((ru-1)&3)*6+3)); \
    } while (0)

#define WMI_GET_PPET8(ppet16_ppet8_ru3_ru0, ru, nssm1) \
    ((ppet16_ppet8_ru3_ru0[nssm1] >> (((ru-1)&3)*6+3))&7)

typedef struct _wmi_ppe_threshold {
    A_UINT32 numss_m1; /** NSS - 1*/
    union {
        A_UINT32 ru_count; /** RU COUNT OBSOLETE to be removed after few versions */
        A_UINT32 ru_mask; /** RU index mask */
    };
    A_UINT32 ppet16_ppet8_ru3_ru0[WMI_MAX_NUM_SS]; /** ppet8 and ppet16 for max num ss */
} wmi_ppe_threshold;

/* WMI_SYS_CAPS_* refer to the capabilities that system support
*/
#define WMI_SYS_CAP_ENABLE                       0x00000001
#define WMI_SYS_CAP_TXPOWER                      0x00000002

/*
 * WMI Dual Band Simultaneous (DBS) hardware mode list bit-mask definitions.
 * Bits 5:0 are reserved
 */
#define WMI_DBS_HW_MODE_MAC0_TX_STREAMS_BITPOS  (28)
#define WMI_DBS_HW_MODE_MAC0_RX_STREAMS_BITPOS  (24)
#define WMI_DBS_HW_MODE_MAC1_TX_STREAMS_BITPOS  (20)
#define WMI_DBS_HW_MODE_MAC1_RX_STREAMS_BITPOS  (16)
#define WMI_DBS_HW_MODE_MAC0_BANDWIDTH_BITPOS   (12)
#define WMI_DBS_HW_MODE_MAC1_BANDWIDTH_BITPOS   (8)
#define WMI_DBS_HW_MODE_DBS_MODE_BITPOS         (7)
#define WMI_DBS_HW_MODE_AGILE_DFS_MODE_BITPOS   (6)

#define WMI_DBS_HW_MODE_MAC0_TX_STREAMS_MASK    (0xf << WMI_DBS_HW_MODE_MAC0_TX_STREAMS_BITPOS)
#define WMI_DBS_HW_MODE_MAC0_RX_STREAMS_MASK    (0xf << WMI_DBS_HW_MODE_MAC0_RX_STREAMS_BITPOS)
#define WMI_DBS_HW_MODE_MAC1_TX_STREAMS_MASK    (0xf << WMI_DBS_HW_MODE_MAC1_TX_STREAMS_BITPOS)
#define WMI_DBS_HW_MODE_MAC1_RX_STREAMS_MASK    (0xf << WMI_DBS_HW_MODE_MAC1_RX_STREAMS_BITPOS)
#define WMI_DBS_HW_MODE_MAC0_BANDWIDTH_MASK     (0xf << WMI_DBS_HW_MODE_MAC0_BANDWIDTH_BITPOS)
#define WMI_DBS_HW_MODE_MAC1_BANDWIDTH_MASK     (0xf << WMI_DBS_HW_MODE_MAC1_BANDWIDTH_BITPOS)
#define WMI_DBS_HW_MODE_DBS_MODE_MASK           (0x1 << WMI_DBS_HW_MODE_DBS_MODE_BITPOS)
#define WMI_DBS_HW_MODE_AGILE_DFS_MODE_MASK     (0x1 << WMI_DBS_HW_MODE_AGILE_DFS_MODE_BITPOS)

#define WMI_DBS_HW_MODE_MAC0_TX_STREAMS_SET(hw_mode, value) \
    WMI_SET_BITS(hw_mode, WMI_DBS_HW_MODE_MAC0_TX_STREAMS_BITPOS, 4, value)
#define WMI_DBS_HW_MODE_MAC0_RX_STREAMS_SET(hw_mode, value) \
    WMI_SET_BITS(hw_mode, WMI_DBS_HW_MODE_MAC0_RX_STREAMS_BITPOS, 4, value)
#define WMI_DBS_HW_MODE_MAC1_TX_STREAMS_SET(hw_mode, value) \
    WMI_SET_BITS(hw_mode, WMI_DBS_HW_MODE_MAC1_TX_STREAMS_BITPOS, 4, value)
#define WMI_DBS_HW_MODE_MAC1_RX_STREAMS_SET(hw_mode, value) \
    WMI_SET_BITS(hw_mode, WMI_DBS_HW_MODE_MAC1_RX_STREAMS_BITPOS, 4, value)
#define WMI_DBS_HW_MODE_MAC0_BANDWIDTH_SET(hw_mode, value)  \
    WMI_SET_BITS(hw_mode, WMI_DBS_HW_MODE_MAC0_BANDWIDTH_BITPOS, 4, value)
#define WMI_DBS_HW_MODE_MAC1_BANDWIDTH_SET(hw_mode, value)  \
    WMI_SET_BITS(hw_mode, WMI_DBS_HW_MODE_MAC1_BANDWIDTH_BITPOS, 4, value)
#define WMI_DBS_HW_MODE_DBS_MODE_SET(hw_mode, value)        \
    WMI_SET_BITS(hw_mode, WMI_DBS_HW_MODE_DBS_MODE_BITPOS, 1, value)
#define WMI_DBS_HW_MODE_AGILE_DFS_SET(hw_mode, value)       \
    WMI_SET_BITS(hw_mode, WMI_DBS_HW_MODE_AGILE_DFS_MODE_BITPOS, 1, value)

#define WMI_DBS_HW_MODE_MAC0_TX_STREAMS_GET(hw_mode)    \
    ((hw_mode & WMI_DBS_HW_MODE_MAC0_TX_STREAMS_MASK) >> WMI_DBS_HW_MODE_MAC0_TX_STREAMS_BITPOS)
#define WMI_DBS_HW_MODE_MAC0_RX_STREAMS_GET(hw_mode)    \
    ((hw_mode & WMI_DBS_HW_MODE_MAC0_RX_STREAMS_MASK) >> WMI_DBS_HW_MODE_MAC0_RX_STREAMS_BITPOS)
#define WMI_DBS_HW_MODE_MAC1_TX_STREAMS_GET(hw_mode)    \
    ((hw_mode & WMI_DBS_HW_MODE_MAC1_TX_STREAMS_MASK) >> WMI_DBS_HW_MODE_MAC1_TX_STREAMS_BITPOS)
#define WMI_DBS_HW_MODE_MAC1_RX_STREAMS_GET(hw_mode)    \
    ((hw_mode & WMI_DBS_HW_MODE_MAC1_RX_STREAMS_MASK) >> WMI_DBS_HW_MODE_MAC1_RX_STREAMS_BITPOS)
#define WMI_DBS_HW_MODE_MAC0_BANDWIDTH_GET(hw_mode)     \
    ((hw_mode & WMI_DBS_HW_MODE_MAC0_BANDWIDTH_MASK) >> WMI_DBS_HW_MODE_MAC0_BANDWIDTH_BITPOS)
#define WMI_DBS_HW_MODE_MAC1_BANDWIDTH_GET(hw_mode)     \
    ((hw_mode & WMI_DBS_HW_MODE_MAC1_BANDWIDTH_MASK) >> WMI_DBS_HW_MODE_MAC1_BANDWIDTH_BITPOS)
#define WMI_DBS_HW_MODE_DBS_MODE_GET(hw_mode)           \
    ((hw_mode & WMI_DBS_HW_MODE_DBS_MODE_MASK) >> WMI_DBS_HW_MODE_DBS_MODE_BITPOS)
#define WMI_DBS_HW_MODE_AGILE_DFS_GET(hw_mode)          \
    ((hw_mode & WMI_DBS_HW_MODE_AGILE_DFS_MODE_MASK) >> WMI_DBS_HW_MODE_AGILE_DFS_MODE_BITPOS)

#define WMI_DBS_CONC_SCAN_CFG_DBS_SCAN_BITPOS        (31)
#define WMI_DBS_CONC_SCAN_CFG_AGILE_SCAN_BITPOS      (30)
#define WMI_DBS_CONC_SCAN_CFG_AGILE_DFS_SCAN_BITPOS  (29)
#define WMI_DBS_CONC_SCAN_CFG_ASYNC_DBS_SCAN_BITPOS  (28)
#define WMI_DBS_CONC_SCAN_CFG_SYNC_DBS_SCAN_BITPOS   (27)

#define WMI_DBS_CONC_SCAN_CFG_DBS_SCAN_MASK         (0x1 << WMI_DBS_CONC_SCAN_CFG_DBS_SCAN_BITPOS)
#define WMI_DBS_CONC_SCAN_CFG_AGILE_SCAN_MASK       (0x1 << WMI_DBS_CONC_SCAN_CFG_AGILE_SCAN_BITPOS)
#define WMI_DBS_CONC_SCAN_CFG_AGILE_DFS_SCAN_MASK   (0x1 << WMI_DBS_CONC_SCAN_CFG_AGILE_DFS_SCAN_BITPOS)
#define WMI_DBS_CONC_SCAN_CFG_ASYC_DBS_SCAN_MASK    (0x1 << WMI_DBS_CONC_SCAN_CFG_ASYNC_DBS_SCAN_BITPOS)
#define WMI_DBS_CONC_SCAN_CFG_SYNC_DBS_SCAN_MASK    (0x1 << WMI_DBS_CONC_SCAN_CFG_SYNC_DBS_SCAN_BITPOS)

#define WMI_DBS_CONC_SCAN_CFG_DBS_SCAN_SET(scan_cfg, value) \
    WMI_SET_BITS(scan_cfg, WMI_DBS_CONC_SCAN_CFG_DBS_SCAN_BITPOS, 1, value)
#define WMI_DBS_CONC_SCAN_CFG_AGILE_SCAN_SET(scan_cfg, value) \
    WMI_SET_BITS(scan_cfg, WMI_DBS_CONC_SCAN_CFG_AGILE_SCAN_BITPOS, 1, value)
#define WMI_DBS_CONC_SCAN_CFG_AGILE_DFS_SCAN_SET(scan_cfg, value) \
    WMI_SET_BITS(scan_cfg, WMI_DBS_CONC_SCAN_CFG_AGILE_DFS_SCAN_BITPOS, 1, value)
#define WMI_DBS_CONC_SCAN_CFG_ASYNC_DBS_SCAN_SET(scan_cfg, value) \
    WMI_SET_BITS(scan_cfg, WMI_DBS_CONC_SCAN_CFG_ASYNC_DBS_SCAN_BITPOS, 1, value)
#define WMI_DBS_CONC_SCAN_CFG_SYNC_DBS_SCAN_SET(scan_cfg, value) \
    WMI_SET_BITS(scan_cfg, WMI_DBS_CONC_SCAN_CFG_SYNC_DBS_SCAN_BITPOS, 1, value)

#define WMI_DBS_CONC_SCAN_CFG_DBS_SCAN_GET(scan_cfg)    \
    ((scan_cfg & WMI_DBS_CONC_SCAN_CFG_DBS_SCAN_MASK) >> WMI_DBS_CONC_SCAN_CFG_DBS_SCAN_BITPOS)
#define WMI_DBS_CONC_SCAN_CFG_AGILE_SCAN_GET(scan_cfg)    \
    ((scan_cfg & WMI_DBS_CONC_SCAN_CFG_AGILE_SCAN_MASK) >> WMI_DBS_CONC_SCAN_CFG_AGILE_SCAN_BITPOS)
#define WMI_DBS_CONC_SCAN_CFG_AGILE_DFS_SCAN_GET(scan_cfg)    \
    ((scan_cfg & WMI_DBS_CONC_SCAN_CFG_AGILE_DFS_SCAN_MASK) >> WMI_DBS_CONC_SCAN_CFG_AGILE_DFS_SCAN_BITPOS)
#define WMI_DBS_CONC_SCAN_CFG_ASYNC_DBS_SCAN_GET(scan_cfg)    \
    ((scan_cfg & WMI_DBS_CONC_SCAN_CFG_ASYC_DBS_SCAN_MASK) >> WMI_DBS_CONC_SCAN_CFG_ASYNC_DBS_SCAN_BITPOS)
#define WMI_DBS_CONC_SCAN_CFG_SYNC_DBS_SCAN_GET(scan_cfg)    \
    ((scan_cfg & WMI_DBS_CONC_SCAN_CFG_SYNC_DBS_SCAN_MASK) >> WMI_DBS_CONC_SCAN_CFG_SYNC_DBS_SCAN_BITPOS)

#define WMI_DBS_FW_MODE_CFG_DBS_BITPOS                  (31)
#define WMI_DBS_FW_MODE_CFG_AGILE_DFS_BITPOS            (30)
#define WMI_DBS_FW_MODE_CFG_DBS_FOR_CXN_BITPOS          (29)
#define WMI_DBS_FW_MODE_CFG_DBS_FOR_STA_PLUS_STA_BITPOS (28)
#define WMI_DBS_FW_MODE_CFG_DBS_FOR_STA_PLUS_P2P_BITPOS (27)

#define WMI_DBS_FW_MODE_CFG_DBS_MASK                    (0x1 << WMI_DBS_FW_MODE_CFG_DBS_BITPOS)
#define WMI_DBS_FW_MODE_CFG_AGILE_DFS_MASK              (0x1 << WMI_DBS_FW_MODE_CFG_AGILE_DFS_BITPOS)
#define WMI_DBS_FW_MODE_CFG_DBS_FOR_CXN_DFS_MASK        (0x1 << WMI_DBS_FW_MODE_CFG_DBS_FOR_CXN_BITPOS)
#define WMI_DBS_FW_MODE_CFG_DBS_FOR_STA_PLUS_STA_MASK   (0x1 << WMI_DBS_FW_MODE_CFG_DBS_FOR_STA_PLUS_STA_BITPOS)
#define WMI_DBS_FW_MODE_CFG_DBS_FOR_STA_PLUS_P2P_MASK   (0x1 << WMI_DBS_FW_MODE_CFG_DBS_FOR_STA_PLUS_P2P_BITPOS)

#define WMI_DBS_FW_MODE_CFG_DBS_SET(fw_mode, value) \
    WMI_SET_BITS(fw_mode, WMI_DBS_FW_MODE_CFG_DBS_BITPOS, 1, value)
#define WMI_DBS_FW_MODE_CFG_AGILE_DFS_SET(fw_mode, value) \
    WMI_SET_BITS(fw_mode, WMI_DBS_FW_MODE_CFG_AGILE_DFS_BITPOS, 1, value)
#define WMI_DBS_FW_MODE_CFG_DBS_FOR_CXN_SET(fw_mode, value) \
    WMI_SET_BITS(fw_mode, WMI_DBS_FW_MODE_CFG_DBS_FOR_CXN_BITPOS, 1, value)
#define WMI_DBS_FW_MODE_CFG_DBS_FOR_STA_PLUS_STA_SET(fw_mode, value) \
    WMI_SET_BITS(fw_mode, WMI_DBS_FW_MODE_CFG_DBS_FOR_STA_PLUS_STA_BITPOS, 1, value)
#define WMI_DBS_FW_MODE_CFG_DBS_FOR_STA_PLUS_P2P_SET(fw_mode, value) \
    WMI_SET_BITS(fw_mode, WMI_DBS_FW_MODE_CFG_DBS_FOR_STA_PLUS_P2P_BITPOS, 1, value)

#define WMI_DBS_FW_MODE_CFG_DBS_GET(fw_mode)    \
    ((fw_mode & WMI_DBS_FW_MODE_CFG_DBS_MASK) >> WMI_DBS_FW_MODE_CFG_DBS_BITPOS)
#define WMI_DBS_FW_MODE_CFG_AGILE_DFS_GET(fw_mode)    \
    ((fw_mode & WMI_DBS_FW_MODE_CFG_AGILE_DFS_MAS) >> WMI_DBS_FW_MODE_CFG_AGILE_DFS_BITPOS)
#define WMI_DBS_FW_MODE_CFG_DBS_FOR_CXN_GET(fw_mode)    \
    ((fw_mode & WMI_DBS_FW_MODE_CFG_DBS_FOR_CXN_DFS_MASK) >> WMI_DBS_FW_MODE_CFG_DBS_FOR_CXN_BITPOS)
#define WMI_DBS_FW_MODE_CFG_DBS_FOR_STA_PLUS_STA_GET(fw_mode)    \
    ((fw_mode & WMI_DBS_FW_MODE_CFG_DBS_FOR_STA_PLUS_STA_MASK) >> WMI_DBS_FW_MODE_CFG_DBS_FOR_STA_PLUS_STA_BITPOS)
#define WMI_DBS_FW_MODE_CFG_DBS_FOR_STA_PLUS_P2P_GET(fw_mode)    \
    ((fw_mode & WMI_DBS_FW_MODE_CFG_DBS_FOR_STA_PLUS_P2P_MASK) >> WMI_DBS_FW_MODE_CFG_DBS_FOR_STA_PLUS_P2P_BITPOS)

/** NOTE: This structure cannot be extended in the future without breaking WMI compatibility */
typedef struct _wmi_abi_version {
    A_UINT32    abi_version_0;   /** WMI Major and Minor versions */
    A_UINT32    abi_version_1;   /** WMI change revision */
    A_UINT32    abi_version_ns_0;  /** ABI version namespace first four dwords */
    A_UINT32    abi_version_ns_1;  /** ABI version namespace second four dwords */
    A_UINT32    abi_version_ns_2;  /** ABI version namespace third four dwords */
    A_UINT32    abi_version_ns_3;  /** ABI version namespace fourth four dwords */
} wmi_abi_version;

/*
* maximum number of memroy requests allowed from FW.
*/
#define WMI_MAX_MEM_REQS 16

/* !!NOTE!!:
 * This HW_BD_INFO_SIZE cannot be changed without breaking compatibility.
 * Please don't change it.
 */
#define HW_BD_INFO_SIZE       5

/**
 * PDEV ID to identify the physical device,
 * value 0 reserved for SOC level commands/event
 */
#define WMI_PDEV_ID_SOC         0 /* SOC level, applicable to all PDEVs */
#define WMI_PDEV_ID_1ST         1 /* first pdev  (pdev 0) */
#define WMI_PDEV_ID_2ND         2 /* second pdev (pdev 1) */
#define WMI_PDEV_ID_3RD         3 /* third pdev  (pdev 2) */

/*
 * Enum regarding which BDF elements are provided in which elements of the
 * wmi_service_ready_event_fixed_param.hw_bd_info[] array
 */
typedef enum {
    BDF_VERSION = 0,
    REF_DESIGN_ID = 1,
    CUSTOMER_ID = 2,
    PROJECT_ID = 3,
    BOARD_DATA_REV = 4,
} wmi_hw_bd_info_e;

/*
 * Macros to get/set BDF details within the
 * wmi_service_ready_event_fixed_param.hw_bd_info[] array
 */
#define WMI_GET_BDF_VERSION(hw_bd_info)         ((hw_bd_info)[BDF_VERSION])
#define WMI_GET_REF_DESIGN(hw_bd_info)          ((hw_bd_info)[REF_DESIGN_ID])
#define WMI_GET_CUSTOMER_ID(hw_bd_info)         ((hw_bd_info)[CUSTOMER_ID])
#define WMI_GET_PROJECT_ID(hw_bd_info)          ((hw_bd_info)[PROJECT_ID])
#define WMI_GET_BOARD_DATA_REV(hw_bd_info)      ((hw_bd_info)[BOARD_DATA_REV])

#define WMI_SET_BDF_VERSION(hw_bd_info, val)    ((hw_bd_info)[BDF_VERSION]    = (val))
#define WMI_SET_REF_DESIGN(hw_bd_info, val)     ((hw_bd_info)[REF_DESIGN_ID]  = (val))
#define WMI_SET_CUSTOMER_ID(hw_bd_info, val)    ((hw_bd_info)[CUSTOMER_ID]    = (val))
#define WMI_SET_PROJECT_ID(hw_bd_info, val)     ((hw_bd_info)[PROJECT_ID]     = (val))
#define WMI_SET_BOARD_DATA_REV(hw_bd_info, val) ((hw_bd_info)[BOARD_DATA_REV] = (val))

/**
 * The following struct holds optional payload for
 * wmi_service_ready_event_fixed_param,e.g., 11ac pass some of the
 * device capability to the host.
*/
typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_SERVICE_READY_EVENT */
    A_UINT32    fw_build_vers; /* firmware build number */
    wmi_abi_version fw_abi_vers;
    A_UINT32    phy_capability;  /* WMI_PHY_CAPABILITY */
	A_UINT32    max_frag_entry;  /* Maximum number of frag table entries that SW will populate less 1 */
    A_UINT32    num_rf_chains;
    /* The following field is only valid for service type WMI_SERVICE_11AC */
    A_UINT32    ht_cap_info; /* WMI HT Capability */
    A_UINT32    vht_cap_info; /* VHT capability info field of 802.11ac */
    A_UINT32    vht_supp_mcs; /* VHT Supported MCS Set field Rx/Tx same */
    A_UINT32    hw_min_tx_power;
    A_UINT32    hw_max_tx_power;
    /* sys_cap_info:
     * bits  1:0  - RXTX LED + RFKILL enable flags (see WMI_LEDRFKILL_FLAGS)
     * bits 31:2  - reserved (must be set to zero)
     */
    A_UINT32    sys_cap_info;
    A_UINT32    min_pkt_size_enable; /* Enterprise mode short pkt enable */
    /** Max beacon and Probe Response IE offload size (includes
     *  optional P2P IEs) */
    A_UINT32    max_bcn_ie_size;
    /*
     * request to host to allocate a chuck of memory and pss it down to FW via WM_INIT.
     * FW uses this as FW extesnsion memory for saving its data structures. Only valid
     * for low latency interfaces like PCIE where FW can access this memory directly (or)
     * by DMA.
     */
    A_UINT32    num_mem_reqs;
    /* Max No. scan channels target can support
     * If FW is too old and doesn't indicate this number, host side value will default to
     * 0, and host will take the original compatible value (62) for future scan channel
     * setup.
     */
    A_UINT32 max_num_scan_channels;

    /* Hardware board specific ID. Values defined in enum WMI_HWBOARD_ID.
     * Default 0 means tha hw_bd_info[] is invalid(legacy board).
     */
    A_UINT32 hw_bd_id;
    A_UINT32 hw_bd_info[HW_BD_INFO_SIZE]; /* Board specific information. Invalid if hw_hd_id is zero. */

    /*
     * Number of MACs supported, i.e. a DBS-capable device will return 2
     */
    A_UINT32 max_supported_macs;

    /*
     * FW sub-feature capabilities to be used in concurrence with wmi_service_bitmap
     */
    A_UINT32 wmi_fw_sub_feat_caps; //values from enum WMI_FW_SUB_FEAT_CAPS

    /*
     * Number of Dual Band Simultaneous (DBS) hardware modes
     */
    A_UINT32 num_dbs_hw_modes;

   /*
     * txrx_chainmask
     *    [7:0]   - 2G band tx chain mask
     *    [15:8]  - 2G band rx chain mask
     *    [23:16] - 5G band tx chain mask
     *    [31:24] - 5G band rx chain mask
     *
     */
    A_UINT32 txrx_chainmask;

    /*
     * default Dual Band Simultaneous (DBS) hardware mode
     */
    A_UINT32 default_dbs_hw_mode_index;

    /*
     * Number of msdu descriptors target would use
     */
    A_UINT32 num_msdu_desc;

    /* The TLVs for hal_reg_capabilities, wmi_service_bitmap and mem_reqs[] will follow this TLV.
         *     HAL_REG_CAPABILITIES   hal_reg_capabilities;
         *     A_UINT32 wmi_service_bitmap[WMI_SERVICE_BM_SIZE];
         *     wlan_host_mem_req mem_reqs[];
         *     wlan_dbs_hw_mode_list[];
         */
} wmi_service_ready_event_fixed_param;

typedef enum {
    WMI_RXTX_LED_ENABLE         = 0x00000001,
    WMI_RFKILL_ENABLE           = 0x00000002,
} WMI_LEDRFKILL_FLAGS;

#define WMI_SERVICE_SEGMENT_BM_SIZE32 4 /* 4x A_UINT32 = 128 bits */
typedef struct {
    /**
     * TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_service_available_event_fixed_param
     */
    A_UINT32 tlv_header;
    /**
     * The wmi_service_segment offset field specifies the position within
     * the logical bitmap of WMI service flags at which the WMI service
     * flags specified within this message begin.
     * Since the first 128 WMI service flags are specified within the
     * wmi_service_bitmap field of the WMI_SERVICE_READY_EVENT message,
     * the wmi_service_segment_offset value is expected to be 128 or more.
     */
    A_UINT32 wmi_service_segment_offset;
    A_UINT32 wmi_service_segment_bitmap[WMI_SERVICE_SEGMENT_BM_SIZE32];
} wmi_service_available_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_SERVICE_EXT_READY_EVENT */
    /* which WMI_DBS_CONC_SCAN_CFG setting the FW is initialized with */
    A_UINT32 default_conc_scan_config_bits;
    /* which WMI_DBS_FW_MODE_CFG setting the FW is initialized with */
    A_UINT32 default_fw_config_bits;
    wmi_ppe_threshold ppet;
    /*
     * see section 8.4.2.213 from draft r8 of 802.11ax;
     * see WMI_HE_FRAG_SUPPORT enum
     */
    A_UINT32 he_cap_info;
    /*
     * An HT STA shall not allow transmission of more than one MPDU start
     * within the time limit described in the MPDU maximum density field.
     */
    A_UINT32 mpdu_density; /* units are microseconds */
    /*
     * Maximum no of BSSID based RX filters host can program
     * Value 0 means FW hasn't given any limit to host.
     */
    A_UINT32 max_bssid_rx_filters;
    /*
     * Extended FW build version information:
     * bits 27:0  -> reserved
     * bits 31:28 -> CRM sub ID
     */
      A_UINT32 fw_build_vers_ext;
    /* max_nlo_ssids - dynamically negotiated maximum number of SSIDS for NLO
     * This limit is the maximum number of SSIDs that can be configured in the
     * target for Network List Offload (i.e. scanning for a preferred network).
     * If this value is 0x0, the target supports WMI_NLO_MAX_SSIDS (16).
     * If this value is non-zero, the host should send back in the
     * WMI_INIT message's wmi_resource_config.max_nlo_ssids a value that
     * is equal to or less than the target capability limit reported here.
     */
    A_UINT32 max_nlo_ssids;
    /* ref to section 8.4.2.48 Multiple BSSID element
     * The Max BSSID Indicator field contains a value assigned to n,
     * where 2^n is the maximum number of BSSIDs
     */
    A_UINT32 max_bssid_indicator;

    /* 2nd DWORD of HE MAC Capabilities */
    A_UINT32 he_cap_info_ext;
} wmi_service_ready_ext_event_fixed_param;

typedef enum {
    WMI_FW_STA_RTT_INITR =     0x00000001,
    WMI_FW_STA_RTT_RESPR =     0x00000002,
    WMI_FW_P2P_CLI_RTT_INITR = 0x00000004,
    WMI_FW_P2P_CLI_RTT_RESPR = 0x00000008,
    WMI_FW_P2P_GO_RTT_INITR =  0x00000010,
    WMI_FW_P2P_GO_RTT_RESPR =  0x00000020,
    WMI_FW_AP_RTT_INITR =      0x00000040,
    WMI_FW_AP_RTT_RESPR =      0x00000080,
    WMI_FW_NAN_RTT_INITR =     0x00000100,
    WMI_FW_NAN_RTT_RESPR =     0x00000200,
    WMI_FW_SCAN_DBS_POLICY =   0x00000400,
    /*
     * New fw sub feature capabilites before
     * WMI_FW_MAX_SUB_FEAT_CAP
     */
    WMI_FW_MAX_SUB_FEAT_CAP =  0x80000000,
} WMI_FW_SUB_FEAT_CAPS;

typedef enum {
    WMI_HWBD_NONE       = 0,            /* No hw board information is given */
    WMI_HWBD_QCA6174    = 1,            /* Rome(AR6320) */
    WMI_HWBD_QCA2582    = 2,            /* Killer 1525*/
} WMI_HWBD_ID;

#define ATH_BD_DATA_REV_MASK            0x000000FF
#define ATH_BD_DATA_REV_SHIFT           0

#define ATH_BD_DATA_PROJ_ID_MASK        0x0000FF00
#define ATH_BD_DATA_PROJ_ID_SHIFT       8

#define ATH_BD_DATA_CUST_ID_MASK        0x00FF0000
#define ATH_BD_DATA_CUST_ID_SHIFT       16

#define ATH_BD_DATA_REF_DESIGN_ID_MASK  0xFF000000
#define ATH_BD_DATA_REF_DESIGN_ID_SHIFT 24

#define SET_BD_DATA_REV(bd_data_ver, value)     \
    ((bd_data_ver) &= ~ATH_BD_DATA_REV_MASK, (bd_data_ver) |= ((value) << ATH_BD_DATA_REV_SHIFT))

#define GET_BD_DATA_REV(bd_data_ver)            \
    (((bd_data_ver) & ATH_BD_DATA_REV_MASK) >> ATH_BD_DATA_REV_SHIFT)

#define SET_BD_DATA_PROJ_ID(bd_data_ver, value) \
    ((bd_data_ver) &= ~ATH_BD_DATA_PROJ_ID_MASK, (bd_data_ver) |= ((value) << ATH_BD_DATA_PROJ_ID_SHIFT))

#define GET_BD_DATA_PROJ_ID(bd_data_ver)        \
    (((bd_data_ver) & ATH_BD_DATA_PROJ_ID_MASK) >> ATH_BD_DATA_PROJ_ID_SHIFT)

#define SET_BD_DATA_CUST_ID(bd_data_ver, value) \
    ((bd_data_ver) &= ~ATH_BD_DATA_CUST_ID_MASK, (bd_data_ver) |= ((value) << ATH_BD_DATA_CUST_ID_SHIFT))

#define GET_BD_DATA_CUST_ID(bd_data_ver)        \
    (((bd_data_ver) & ATH_BD_DATA_CUST_ID_MASK) >> ATH_BD_DATA_CUST_ID_SHIFT)

#define SET_BD_DATA_REF_DESIGN_ID(bd_data_ver, value)   \
    ((bd_data_ver) &= ~ATH_BD_DATA_REF_DESIGN_ID_MASK, (bd_data_ver) |= ((value) << ATH_BD_DATA_REF_DESIGN_ID_SHIFT))

#define GET_BD_DATA_REF_DESIGN_ID(bd_data_ver)          \
    (((bd_data_ver) & ATH_BD_DATA_REF_DESIGN_ID_MASK) >> ATH_BD_DATA_REF_DESIGN_ID_SHIFT)

#ifdef ROME_LTE_COEX_FREQ_AVOID
typedef struct {
    A_UINT32 start_freq; //start frequency, not channel center freq
    A_UINT32 end_freq;//end frequency
}avoid_freq_range_desc;

typedef struct {
    //bad channel range count, multi range is allowed, 0 means all channel clear
    A_UINT32 num_freq_ranges;
    //multi range with num_freq_ranges, LTE advance multi carrier, CDMA,etc
    avoid_freq_range_desc avd_freq_range[0];
}wmi_wlan_avoid_freq_ranges_event;
#endif

/** status consists of  upper 16 bits fo A_STATUS status and lower 16 bits of module ID that retuned status */
#define WLAN_INIT_STATUS_SUCCESS   0x0
#define WLAN_INIT_STATUS_GEN_FAILED   0x1
#define WLAN_GET_INIT_STATUS_REASON(status)    ((status) & 0xffff)
#define WLAN_GET_INIT_STATUS_MODULE_ID(status) (((status) >> 16) & 0xffff)

typedef A_UINT32 WLAN_INIT_STATUS;

typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ready_event_fixed_param */
    wmi_abi_version fw_abi_vers;
    /*
     * mac_addr is always filled; in addition, there can be a mac_addr_list
     * TLV following this fixed_param TLV to specify additional MAC addresses,
     * for cases where the target specifies one MAC address per pdev
     * (so the host can treat the pdevs within the target as separately
     * as possible) rather than one MAC address for the whole SOC.
     */
    wmi_mac_addr mac_addr;
    A_UINT32    status;
    A_UINT32 num_dscp_table;
    /* num_extra_mac_addr -
     * how many additional MAC addresses besides the above mac_addr
     * are provided in the subsequent mac_addr_list TLV
     */
    A_UINT32 num_extra_mac_addr;
    /*
     * Total number of "real" peers (remote peers of an AP vdev,
     * BSS peer of a STA vdev, TDLS peer of a STA vdev) that FW supports.
     * If 0, then Host can use param_tlv->resource_config->num_peers as
     * total number of peers.
     */
    A_UINT32 num_total_peers;
    /*
     * Number of extra peers that Firmware adds.
     * These are self peers and/or other FW only peers that don't represent
     * a 802.11 transceiver, but instead are used for convenience, e.g. to
     * provide a pseudo-peer object for an AP vdev's bcast/mcast tx queues,
     * to allow each tx queue to belong to a peer object.
     * Peer ID can be up to num_total_peers + num_extra_peers.
     */
    A_UINT32 num_extra_peers;
    /*
     * max_ast_index - max AST index that Firmware can generate
     * max_ast_index = (ast_table_size-1), ast_table_size is dynamically chosen
     * based on num_peers configutation from Host. Hence Host needs to know the
     * max_ast_index that Firmware can generate.
     * A 0x0 value for max_ast_index means the target has not specified a limit.
     */
    A_UINT32 max_ast_index;
/*
 * This fixed_param TLV is followed by these additional TLVs:
 * mac_addr_list[num_extra_mac_addr];
 */
} wmi_ready_event_fixed_param;

typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_resource_config */
/**
 * @brief num_vdev - number of virtual devices (VAPs) to support
 */
    A_UINT32 num_vdevs;
/**
 * @brief num_peers - number of peer nodes to support
 */
    A_UINT32 num_peers;
/*
 * @brief In offload mode target supports features like WOW, chatter and other
 * protocol offloads. In order to support them some functionalities like
 * reorder buffering, PN checking need to be done in target. This determines
 * maximum number of peers suported by target in offload mode
 */
    A_UINT32 num_offload_peers;
/* @brief Number of reorder buffers available for doing target based reorder
 * Rx reorder buffering
 */
    A_UINT32 num_offload_reorder_buffs;
/**
 * @brief num_peer_keys - number of keys per peer
 */
    A_UINT32 num_peer_keys;
/**
 * @brief num_peer_tids - number of TIDs to provide storage for per peer.
 */
    A_UINT32 num_tids;
/**
 * @brief ast_skid_limit - max skid for resolving hash collisions
 * @details
 *     The address search table is sparse, so that if two MAC addresses
 *     result in the same hash value, the second of these conflicting
 *     entries can slide to the next index in the address search table,
 *     and use it, if it is unoccupied.  This ast_skid_limit parameter
 *     specifies the upper bound on how many subsequent indices to search
 *     over to find an unoccupied space.
 */
    A_UINT32 ast_skid_limit;
/**
 * @brief tx_chain_mask - the nominal chain mask for transmit
 * @details
 *     The chain mask may be modified dynamically, e.g. to operate AP tx with
 *     a reduced number of chains if no clients are associated.
 *     This configuration parameter specifies the nominal chain-mask that
 *     should be used when not operating with a reduced set of tx chains.
 */
    A_UINT32 tx_chain_mask;
/**
 * @brief rx_chain_mask - the nominal chain mask for receive
 * @details
 *     The chain mask may be modified dynamically, e.g. for a client to use
 *     a reduced number of chains for receive if the traffic to the client
 *     is low enough that it doesn't require downlink MIMO or antenna
 *     diversity.
 *     This configuration parameter specifies the nominal chain-mask that
 *     should be used when not operating with a reduced set of rx chains.
 */
    A_UINT32 rx_chain_mask;
/**
 * @brief rx_timeout_pri - what rx reorder timeout (ms) to use for the AC
 * @details
 *     Each WMM access class (voice, video, best-effort, background) will
 *     have its own timeout value to dictate how long to wait for missing
 *     rx MPDUs to arrive before flushing subsequent MPDUs that have already
 *     been received.
 *     This parameter specifies the timeout in milliseconds for each class .
 *     NOTE: the number of class (defined as 4) cannot be
 *     changed in the future without breaking WMI compatibility.
 */
    A_UINT32 rx_timeout_pri[4];
/**
 * @brief rx_decap mode - what mode the rx should decap packets to
 * @details
 *     MAC can decap to RAW (no decap), native wifi or Ethernet types
 *     THis setting also determines the default TX behavior, however TX
 *     behavior can be modified on a per VAP basis during VAP init
 */
    A_UINT32 rx_decap_mode;
 /**
  * @brief  scan_max_pending_req - what is the maximum scan requests than can be queued
  */
    A_UINT32 scan_max_pending_req;

    /**
     * @brief maximum VDEV that could use BMISS offload
     */
    A_UINT32 bmiss_offload_max_vdev;

    /**
     * @brief maximum VDEV that could use offload roaming
     */
    A_UINT32 roam_offload_max_vdev;

    /**
     * @brief maximum AP profiles that would push to offload roaming
     */
    A_UINT32 roam_offload_max_ap_profiles;

/**
 * @brief num_mcast_groups - how many groups to use for mcast->ucast conversion
 * @details
 *     The target's WAL maintains a table to hold information regarding which
 *     peers belong to a given multicast group, so that if multicast->unicast
 *     conversion is enabled, the target can convert multicast tx frames to a
 *     series of unicast tx frames, to each peer within the multicast group.
 *     This num_mcast_groups configuration parameter tells the target how
 *     many multicast groups to provide storage for within its multicast
 *     group membership table.
 */
    A_UINT32 num_mcast_groups;

/**
 * @brief num_mcast_table_elems - size to alloc for the mcast membership table
 * @details
 *     This num_mcast_table_elems configuration parameter tells the target
 *     how many peer elements it needs to provide storage for in its
 *     multicast group membership table.
 *     These multicast group membership table elements are shared by the
 *     multicast groups stored within the table.
 */
    A_UINT32 num_mcast_table_elems;

/**
 * @brief mcast2ucast_mode - whether/how to do multicast->unicast conversion
 * @details
 *     This configuration parameter specifies whether the target should
 *     perform multicast --> unicast conversion on transmit, and if so,
 *     what to do if it finds no entries in its multicast group membership
 *     table for the multicast IP address in the tx frame.
 *     Configuration value:
 *     0 -> Do not perform multicast to unicast conversion.
 *     1 -> Convert multicast frames to unicast, if the IP multicast address
 *          from the tx frame is found in the multicast group membership
 *          table.  If the IP multicast address is not found, drop the frame.
 *     2 -> Convert multicast frames to unicast, if the IP multicast address
 *          from the tx frame is found in the multicast group membership
 *          table.  If the IP multicast address is not found, transmit the
 *          frame as multicast.
 */
    A_UINT32 mcast2ucast_mode;


 /**
  * @brief tx_dbg_log_size - how much memory to allocate for a tx PPDU dbg log
  * @details
  *     This parameter controls how much memory the target will allocate to
  *     store a log of tx PPDU meta-information (how large the PPDU was,
  *     when it was sent, whether it was successful, etc.)
  */
    A_UINT32 tx_dbg_log_size;

 /**
  * @brief num_wds_entries - how many AST entries to be allocated for WDS
  */
    A_UINT32 num_wds_entries;

 /**
  * @brief dma_burst_size - MAC DMA burst size, e.g., on Peregrine on PCI
  * this limit can be 0 -default, 1 256B
  */
    A_UINT32 dma_burst_size;

  /**
   * @brief mac_aggr_delim - Fixed delimiters to be inserted after every MPDU
   * to account for interface latency to avoid underrun.
   */
    A_UINT32 mac_aggr_delim;
    /**
     * @brief rx_skip_defrag_timeout_dup_detection_check
     * @details
     *  determine whether target is responsible for detecting duplicate
     *  non-aggregate MPDU and timing out stale fragments.
     *
     *  A-MPDU reordering is always performed on the target.
     *
     *  0: target responsible for frag timeout and dup checking
     *  1: host responsible for frag timeout and dup checking
     */
    A_UINT32 rx_skip_defrag_timeout_dup_detection_check;

    /**
     * @brief vow_config - Configuration for VoW : No of Video Nodes to be supported
     * and Max no of descriptors for each Video link (node).
     */
    A_UINT32 vow_config;

    /**
     * @brief maximum VDEV that could use GTK offload
     */
	A_UINT32 gtk_offload_max_vdev;

    /**
     * @brief num_msdu_desc - Number of msdu descriptors target should use
     */
    A_UINT32 num_msdu_desc; /* Number of msdu desc */
 /**
  * @brief max_frag_entry - Max. number of Tx fragments per MSDU
  * @details
  *     This parameter controls the max number of Tx fragments per MSDU.
  *     This is sent by the target as part of the WMI_SERVICE_READY event
  *     and is overriden by the OS shim as required.
  */
    A_UINT32 max_frag_entries;

    /**
     * @brief num_tdls_vdevs - Max. number of vdevs that can support TDLS
     * @brief num_msdu_desc - Number of vdev that can support beacon offload
     */

    A_UINT32 num_tdls_vdevs; /* number of vdevs allowed to do tdls */

    /**
     * @brief num_tdls_conn_table_entries - Number of peers tracked by tdls vdev
     * @details
     *      Each TDLS enabled vdev can track outgoing transmits/rssi/rates to/of
     *      peers in a connection tracking table for possible TDLS link creation
     *      or deletion. This controls the number of tracked peers per vdev.
     */
    A_UINT32  num_tdls_conn_table_entries; /* number of peers to track per TDLS vdev */
     A_UINT32 beacon_tx_offload_max_vdev;
     A_UINT32 num_multicast_filter_entries;
     A_UINT32 num_wow_filters; /*host can configure the number of wow filters*/

    /**
     * @brief num_keep_alive_pattern - Num of keep alive patterns configured
     * from host.
     */
    A_UINT32 num_keep_alive_pattern;
    /**
     * @brief keep_alive_pattern_size - keep alive pattern size.
     */
    A_UINT32 keep_alive_pattern_size;

    /**
     * @brief max_tdls_concurrent_sleep_sta - Number of tdls sleep sta supported
     * @details
     *      Each TDLS STA can become a sleep STA independently. This parameter
     *      mentions how many such sleep STAs can be supported concurrently.
     */
    A_UINT32 max_tdls_concurrent_sleep_sta;

    /**
     * @brief max_tdls_concurrent_buffer_sta - Number of tdls buffer sta supported
     * @details
     *      Each TDLS STA can become a buffer STA independently. This parameter
     *      mentions how many such buffer STAs can be supported concurrently.
     */
    A_UINT32 max_tdls_concurrent_buffer_sta;

    /**
     * @brief wmi_send_separate - host configures fw to send the wmi separately
     */
    A_UINT32 wmi_send_separate;

    /**
     * @brief num_ocb_vdevs - Number of vdevs used for OCB support
     */
    A_UINT32 num_ocb_vdevs;

    /**
     * @brief num_ocb_channels - The supported number of simultaneous OCB channels
     */
    A_UINT32 num_ocb_channels;

    /**
     * @brief num_ocb_schedules - The supported number of OCB schedule segments
     */
    A_UINT32 num_ocb_schedules;

    /**
     * @brief specific configuration from host, such as per platform configuration
     */
    #define WMI_RSRC_CFG_FLAG_WOW_IGN_PCIE_RST_S 0
    #define WMI_RSRC_CFG_FLAG_WOW_IGN_PCIE_RST_M 0x1

    #define WMI_RSRC_CFG_FLAG_LTEU_SUPPORT_S 1
    #define WMI_RSRC_CFG_FLAG_LTEU_SUPPORT_M 0x2

    #define WMI_RSRC_CFG_FLAG_COEX_GPIO_SUPPORT_S 2
    #define WMI_RSRC_CFG_FLAG_COEX_GPIO_SUPPORT_M 0x4

    #define WMI_RSRC_CFG_FLAG_AUX_RADIO_SPECTRAL_INTF_S 3
    #define WMI_RSRC_CFG_FLAG_AUX_RADIO_SPECTRAL_INTF_M 0x8

    #define WMI_RSRC_CFG_FLAG_AUX_RADIO_CHAN_LOAD_INTF_S 4
    #define WMI_RSRC_CFG_FLAG_AUX_RADIO_CHAN_LOAD_INTF_M 0x10

    #define WMI_RSRC_CFG_FLAG_BSS_CHANNEL_INFO_64_S 5
    #define WMI_RSRC_CFG_FLAG_BSS_CHANNEL_INFO_64_M 0x20

    #define WMI_RSRC_CFG_FLAG_ATF_CONFIG_ENABLE_S 6
    #define WMI_RSRC_CFG_FLAG_ATF_CONFIG_ENABLE_M 0x40

    #define WMI_RSRC_CFG_FLAG_IPHR_PAD_CONFIG_ENABLE_S 7
    #define WMI_RSRC_CFG_FLAG_IPHR_PAD_CONFIG_ENABLE_M 0x80

    #define WMI_RSRC_CFG_FLAG_QWRAP_MODE_ENABLE_S 8
    #define WMI_RSRC_CFG_FLAG_QWRAP_MODE_ENABLE_M 0x100

    #define WMI_RSRC_CFG_FLAG_MGMT_COMP_EVT_BUNDLE_SUPPORT_S 9
    #define WMI_RSRC_CFG_FLAG_MGMT_COMP_EVT_BUNDLE_SUPPORT_M 0x200

    #define WMI_RSRC_CFG_FLAG_TX_MSDU_ID_NEW_PARTITION_SUPPORT_S 10
    #define WMI_RSRC_CFG_FLAG_TX_MSDU_ID_NEW_PARTITION_SUPPORT_M 0x400

    #define WMI_RSRC_CFG_FLAG_TX_PPDU_STATS_ENABLE_S 11
    #define WMI_RSRC_CFG_FLAG_TX_PPDU_STATS_ENABLE_M 0x800

    #define WMI_RSRC_CFG_FLAG_TCL_CCE_DISABLE_S 12
    #define WMI_RSRC_CFG_FLAG_TCL_CCE_DISABLE_M 0x1000

    #define WMI_RSRC_CFG_FLAG_TIM_V2_SUPPORT_ENABLE_S 13
    #define WMI_RSRC_CFG_FLAG_TIM_V2_SUPPORT_ENABLE_M 0x2000

    #define WMI_RSRC_CFG_FLAG_EAPOL_REKEY_MINRATE_SUPPORT_ENABLE_S 14
    #define WMI_RSRC_CFG_FLAG_EAPOL_REKEY_MINRATE_SUPPORT_ENABLE_M 0x4000

    #define WMI_RSRC_CFG_FLAG_EAPOL_AC_OVERRIDE_VALID_S 15
    #define WMI_RSRC_CFG_FLAG_EAPOL_AC_OVERRIDE_VALID_M 0x8000

    /*
     * If the AC override valid bit is set then this field will specify the
     * access category to use for EAPOL frames
     * 0 - WMM_AC_BE
     * 1 - WMM_AC_BK
     * 2 - WMM_AC_VI
     * 3 - WMM_AC_VO
     */
    #define WMI_RSRC_CFG_FLAG_EAPOL_AC_OVERRIDE_S 16
    #define WMI_RSRC_CFG_FLAG_EAPOL_AC_OVERRIDE_M 0x30000

    /*
     * If TX_ACK_RSSI is set, then the target should populate the ack_rssi
     * field within the WMI_MGMT_TX_COMPLETION_EVENT message, the ack_rssi
     * TLV within the WMI_MGMT_TX_BUNDLE_COMPLETION_EVENT message, and the
     * "MSDU ACK RSSI" array within the HTT_T2H TX_COMPL_IND message.
     */
    #define WMI_RSRC_CFG_FLAG_TX_ACK_RSSI_S 18
    #define WMI_RSRC_CFG_FLAG_TX_ACK_RSSI_M 0x40000

    /*
     * If HTT_H2T_NO_HTC_HDR_LEN_IN_MSG_LEN is set, the host will not
     * include the HTC header length in the payload length for all HTT_H2T
     * messages.
     * Otherwise, only when sending HTT_H2T_MSG_TYPE_TX_FRM message,
     * payload length includes HTC header length. Other HTT_H2T messages'
     * payload length does not include HTC header length.
     * The host will only set this HTT_H2T_NO_HTC_HDR_LEN_IN_MSG_LEN flag
     * if the target has set the WMI_SERVICE_HTT_H2T_NO_HTC_HDR_LEN_IN_MSG_LEN
     * flag to indicate its support for this option.
     */
    #define WMI_RSRC_CFG_FLAG_HTT_H2T_NO_HTC_HDR_LEN_IN_MSG_LEN_S 19
    #define WMI_RSRC_CFG_FLAG_HTT_H2T_NO_HTC_HDR_LEN_IN_MSG_LEN_M 0x80000

    #define WMI_RSRC_CFG_FLAG_PEER_UNMAP_RESPONSE_SUPPORT_S 20
    #define WMI_RSRC_CFG_FLAG_PEER_UNMAP_RESPONSE_SUPPORT_M 0x100000

    /*
     * If this HTT_PEER_STATS is set, then the target should use the
     * the HTT_T2H_MSG_TYPE_PEER_STATS_IND message to upload peer stats;
     * else the target should avoid sending the PEER_STATS_IND message.
     */
    #define WMI_RSRC_CFG_FLAG_HTT_PEER_STATS_S 21
    #define WMI_RSRC_CFG_FLAG_HTT_PEER_STATS_M 0x200000

    /*
     * If this BIT is set, then the target should use peer_tid_ext to analyze
     * per peer per tid extended configurations
     */
    #define WMI_RSRC_CFG_FLAG_PEER_TID_EXT_S 22
    #define WMI_RSRC_CFG_FLAG_PEER_TID_EXT_M 0x400000

    /*
     * If the VIDEO_OVER_WIFI_ENABLE flag is set, the target will use a
     * series of algorithmic adjustments to optimize Video performance
     * by reducing latency, reducing latency jitter, and minimizing
     * dropped packets.
     */
    #define WMI_RSRC_CFG_FLAG_VIDEO_OVER_WIFI_ENABLE_S 23
    #define WMI_RSRC_CFG_FLAG_VIDEO_OVER_WIFI_ENABLE_M 0x800000

    /*
     * If the THREE_WAY_COEX_CONFIG_LEGACY flag is set, the target will use
     * the configuration parameters given by Host driver to WLAN FW and
     * apply them along with the existing CoEx Weights Override logic to
     * prioritize the WLAN-BT-Zigbee packets accordingly.
     *
     * The host shall only set the THREE_WAY_COEX_CONFIG_LEGACY_SUPPORT
     * RSRC_CFG flag if the target has set the WMI_SERVICE
     * THREE_WAY_COEX_CONFIG_LEGACY_SUPPORT flag.
     *
     * The logic to send GPM to BT-SOC with BT-ZB priorities remains the same.
     */
    #define WMI_RSRC_CFG_FLAG_THREE_WAY_COEX_CONFIG_LEGACY_SUPPORT_S 24
    #define WMI_RSRC_CFG_FLAG_THREE_WAY_COEX_CONFIG_LEGACY_SUPPORT_M 0x1000000

    /*
     * If the THREE_WAY_COEX_CONFIG_OVERRIDE flag is set, the target will use
     * the configuration parameters given by Host driver to WLAN FW and
     * apply them by OVERRIDing the existing CoEx Weights Override logic to
     * prioritize the WLAN-BT-Zigbee packets accordingly.
     *
     * The host shall only set the THREE_WAY_COEX_CONFIG_OVERRIDE_SUPPORT
     * RSRC_CFG flag if the target has set the WMI_SERVICE
     * THREE_WAY_COEX_CONFIG_OVERRIDE_SUPPORT flag.
     *
     * The logic to send GPM to BT-SOC with BT-ZB priorities remains the same.
     */
    #define WMI_RSRC_CFG_FLAG_THREE_WAY_COEX_CONFIG_OVERRIDE_SUPPORT_S 25
    #define WMI_RSRC_CFG_FLAG_THREE_WAY_COEX_CONFIG_OVERRIDE_SUPPORT_M 0x2000000

    A_UINT32 flag1;

    /** @brief smart_ant_cap - Smart Antenna capabilities information
     * @details
     *        1 - Smart antenna is enabled.
     *        0 - Smart antenna is disabled.
     * In future this can contain smart antenna specifc capabilities.
     */
    A_UINT32 smart_ant_cap;

    /**
     * User can configure the buffers allocated for each AC (BE, BK, VI, VO)
     * during init
     */
    A_UINT32 BK_Minfree;
    A_UINT32 BE_Minfree;
    A_UINT32 VI_Minfree;
    A_UINT32 VO_Minfree;

    /**
     * @brief alloc_frag_desc_for_data_pkt . Controls data packet fragment
     * descriptor memory allocation.
     *   1 - Allocate fragment descriptor memory for data packet in firmware.
     *       If host wants to transmit data packet at its desired rate,
     *       this field must be set.
     *   0 - Don't allocate fragment descriptor for data packet.
     */
    A_UINT32 alloc_frag_desc_for_data_pkt;

    /** how much space to allocate for NDP NS (neighbor solicitation) specs */
    A_UINT32 num_ns_ext_tuples_cfg;
    /**
     * size (in bytes) of the buffer the FW shall allocate per vdev
     * firmware can dynamic allocate memory (or disable)
     * packet filtering feature.
     * 0 - fw chooses its default value
     * -1 (0XFFFFFFFF) - disable APF
     */
    A_UINT32 bpf_instruction_size;

    /**
     * Maximum no of BSSID based RX filters host would program
     * Value 0 means host doesn't given any limit to FW.
     */
    A_UINT32 max_bssid_rx_filters;
    /**
     * Use PDEV ID instead of MAC ID, added for backward compatibility with older host
     * which is using MAC ID. 1 means PDEV ID, 0 means MAC ID.
     */
    A_UINT32 use_pdev_id;

    /** Maximum number of scan clients whose DBS scan duty cycle can be configured */
    A_UINT32 max_num_dbs_scan_duty_cycle;

    /** Maximum number of Multi group key to support */
    A_UINT32 max_num_group_keys;

    /**
     * HTT peer map/unmap V2 format support
     * 0 -> host doesn't support HTT peer map/unmap v2 format.
     * 1 -> host supports HTT peer map/unmap v2 format; the target is
     *      allowed but not required to use peer map/unmap v2 format.
     */
    A_UINT32 peer_map_unmap_v2_support;

    /** Sched config params for all pdevs
     * These tx scheduling configuration parameters are currently only
     * used for internal testing purposes; therefore the non-default
     * values for this field are not currently documented.
     * For regular use, this field should be set to 0x0.
     */
    A_UINT32 sched_params;

    /* Number of MAC on which AP TWT feature is supported */
    A_UINT32 twt_ap_pdev_count;

    /* Max no of STA with which TWT sessions can be formed by the AP */
    A_UINT32 twt_ap_sta_count;

    /* max_nlo_ssids - dynamically negotiated maximum number of SSIDS for NLO
     * This parameter provides the final specification for the maximum number
     * of SSIDs for the target to support for Network List Offload's scanning
     * for preferred networks.
     * This wmi_resource_config.max_nlo_ssids must be <= the max_nlo_ssids
     * field from the target's WMI_SERVICE_READY_EXT_EVENT message.
     * (If the target didn't provide a max_nlo_ssids field in the
     * WMI_SERVICE_READY_EXT message, or if the SERVICE_READY_EXT msg's
     * max_nlo_ssids value was 0x0, the target doesn't support dynamic
     * negotiation of max NLO SSIDs, and WMI_NLO_MAX_SSIDS (=16) applies.)
     * If this wmi_resource_config.max_nlo_ssids field is absent or 0x0,
     * the host does not support dynamic negotiation of max NLO SSIDs.
     * In such a case, the target will respond as follows:
     * If the target supports at least WMI_NLO_MAX_SSIDS, the target will
     * use the statically-configured WMI_NLO_MAX_SSIDS value.
     * If the target supports less than WMI_NLO_MAX_SSIDS, the target will
     * abort its boot-up, due to receiving an invalid/unsupported
     * configuration specification.
     */
    A_UINT32 max_nlo_ssids;

    /**
     * num_packet_filters: the num that host requests fw to support for
     * pktfilter in total, then firmware can dynamic allocate
     * memory(or disable) pktfilter feature.
     *
     * 0 -  fw chooses its default value.
     * -1(0XFFFFFFFF)- disable pktfilter.
     */
    A_UINT32 num_packet_filters;

    /**
     * num_max_sta_vdevs: the max num for the sta vdevs
     * fw will use it to config the memory of offload features that
     * are only for sta vdevs.
     * p2p client should be included.
     *
     *  0 - fw chooses its default value: 'num_vdevs' of this structure.
     */
    A_UINT32 num_max_sta_vdevs;

    /* ref to section 8.4.2.48 Multiple BSSID element
     * The Max BSSID Indicator field contains a value assigned to n,
     * where 2^n is the maximum number of BSSIDs
     */
    A_UINT32 max_bssid_indicator;

    /** @brief ul_resp_config - Configures the 11ax uplink ofdma feature on STA.
     *         I.e. sending uplink response to a trigger frame sent by AP.
     *  @details
     *        0 - fw default behavior, based on chipset
     *        1 - UL_RESP is disabled.
     *        2 - UL_RESP is enabled.
     *        other - reserved.
     */
    A_UINT32 ul_resp_config;
} wmi_resource_config;

#define WMI_RSRC_CFG_FLAG_SET(word32, flag, value) \
    do { \
        (word32) &= ~WMI_RSRC_CFG_FLAG_ ## flag ## _M; \
        (word32) |= ((value) << WMI_RSRC_CFG_FLAG_ ## flag ## _S) & \
            WMI_RSRC_CFG_FLAG_ ## flag ## _M; \
    } while (0)
#define WMI_RSRC_CFG_FLAG_GET(word32, flag) \
    (((word32) & WMI_RSRC_CFG_FLAG_ ## flag ## _M) >> \
    WMI_RSRC_CFG_FLAG_ ## flag ## _S)

#define WMI_RSRC_CFG_FLAG_WOW_IGN_PCIE_RST_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), WOW_IGN_PCIE_RST, (value))
#define WMI_RSRC_CFG_FLAG_WOW_IGN_PCIE_RST_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), WOW_IGN_PCIE_RST)

#define WMI_RSRC_CFG_FLAG_LTEU_SUPPORT_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), LTEU_SUPPORT, (value))
#define WMI_RSRC_CFG_FLAG_LTEU_SUPPORT_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), LTEU_SUPPORT)

#define WMI_RSRC_CFG_FLAG_COEX_GPIO_SUPPORT_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), COEX_GPIO_SUPPORT, (value))
#define WMI_RSRC_CFG_FLAG_COEX_GPIO_SUPPORT_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), COEX_GPIO_SUPPORT)

#define WMI_RSRC_CFG_FLAG_AUX_RADIO_SPECTRAL_INTF_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), AUX_RADIO_SPECTRAL_INTF, (value))
#define WMI_RSRC_CFG_FLAG_AUX_RADIO_SPECTRAL_INTF_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), AUX_RADIO_SPECTRAL_INTF)

#define WMI_RSRC_CFG_FLAG_AUX_RADIO_CHAN_LOAD_INTF_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), AUX_RADIO_CHAN_LOAD_INTF, (value))
#define WMI_RSRC_CFG_FLAG_AUX_RADIO_CHAN_LOAD_INTF_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), AUX_RADIO_CHAN_LOAD_INTF)

#define WMI_RSRC_CFG_FLAG_BSS_CHANNEL_INFO_64_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), BSS_CHANNEL_INFO_64, (value))
#define WMI_RSRC_CFG_FLAG_BSS_CHANNEL_INFO_64_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), BSS_CHANNEL_INFO_64)

#define WMI_RSRC_CFG_FLAG_ATF_CONFIG_ENABLE_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), ATF_CONFIG_ENABLE, (value))
#define WMI_RSRC_CFG_FLAG_ATF_CONFIG_ENABLE_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), ATF_CONFIG_ENABLE)

#define WMI_RSRC_CFG_FLAG_IPHR_PAD_CONFIG_ENABLE_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), IPHR_PAD_CONFIG_ENABLE, (value))
#define WMI_RSRC_CFG_FLAG_IPHR_PAD_CONFIG_ENABLE_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), IPHR_PAD_CONFIG_ENABLE)

#define WMI_RSRC_CFG_FLAG_QWRAP_MODE_ENABLE_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), QWRAP_MODE_ENABLE, (value))
#define WMI_RSRC_CFG_FLAG_QWRAP_MODE_ENABLE_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), QWRAP_MODE_ENABLE)

#define WMI_RSRC_CFG_FLAG_MGMT_COMP_EVT_BUNDLE_SUPPORT_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), MGMT_COMP_EVT_BUNDLE_SUPPORT, (value))
#define WMI_RSRC_CFG_FLAG_MGMT_COMP_EVT_BUNDLE_SUPPORT_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), MGMT_COMP_EVT_BUNDLE_SUPPORT)

#define WMI_RSRC_CFG_FLAG_TX_MSDU_ID_NEW_PARTITION_SUPPORT_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), TX_MSDU_ID_NEW_PARTITION_SUPPORT, (value))
#define WMI_RSRC_CFG_FLAG_TX_MSDU_ID_NEW_PARTITION_SUPPORT_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), TX_MSDU_ID_NEW_PARTITION_SUPPORT)

#define WMI_RSRC_CFG_FLAG_TCL_CCE_DISABLE_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), TCL_CCE_DISABLE, (value))
#define WMI_RSRC_CFG_FLAG_TCL_CCE_DISABLE_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), TCL_CCE_DISABLE)

#define WMI_RSRC_CFG_FLAG_TIM_V2_SUPPORT_ENABLE_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), TIM_V2_SUPPORT_ENABLE, (value))
#define WMI_RSRC_CFG_FLAG_TIM_V2_SUPPORT_ENABLE_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), TIM_V2_SUPPORT_ENABLE)

#define WMI_RSRC_CFG_FLAG_EAPOL_REKEY_MINRATE_SUPPORT_ENABLE_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), EAPOL_REKEY_MINRATE_SUPPORT_ENABLE, (value))
#define WMI_RSRC_CFG_FLAG_EAPOL_REKEY_MINRATE_SUPPORT_ENABLE_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), EAPOL_REKEY_MINRATE_SUPPORT_ENABLE)

#define WMI_RSRC_CFG_FLAG_EAPOL_AC_OVERRIDE_VALID_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), EAPOL_AC_OVERRIDE_VALID, (value))
#define WMI_RSRC_CFG_FLAG_EAPOL_AC_OVERRIDE_VALID_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), EAPOL_AC_OVERRIDE_VALID)

#define WMI_RSRC_CFG_FLAG_EAPOL_AC_OVERRIDE_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), EAPOL_AC_OVERRIDE, (value))
#define WMI_RSRC_CFG_FLAG_EAPOL_AC_OVERRIDE_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), EAPOL_AC_OVERRIDE)

#define WMI_RSRC_CFG_FLAG_TX_ACK_RSSI_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), TX_ACK_RSSI, (value))
#define WMI_RSRC_CFG_FLAG_TX_ACK_RSSI_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), TX_ACK_RSSI)

#define WMI_RSRC_CFG_FLAG_HTT_H2T_NO_HTC_HDR_LEN_IN_MSG_LEN_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), HTT_H2T_NO_HTC_HDR_LEN_IN_MSG_LEN, (value))
#define WMI_RSRC_CFG_FLAG_HTT_H2T_NO_HTC_HDR_LEN_IN_MSG_LEN_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), HTT_H2T_NO_HTC_HDR_LEN_IN_MSG_LEN)

#define WMI_RSRC_CFG_FLAG_PEER_UNMAP_RESPONSE_SUPPORT_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), PEER_UNMAP_RESPONSE_SUPPORT, (value))
#define WMI_RSRC_CFG_FLAG_PEER_UNMAP_RESPONSE_SUPPORT_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), PEER_UNMAP_RESPONSE_SUPPORT)

#define WMI_RSRC_CFG_FLAG_HTT_PEER_STATS_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), HTT_PEER_STATS, (value))
#define WMI_RSRC_CFG_FLAG_HTT_PEER_STATS_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), HTT_PEER_STATS)

#define WMI_RSRC_CFG_FLAG_PEER_TID_EXT_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), PEER_TID_EXT, (value))
#define WMI_RSRC_CFG_FLAG_PEER_TID_EXT_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), PEER_TID_EXT)

#define WMI_RSRC_CFG_FLAG_VIDEO_OVER_WIFI_ENABLE_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), VIDEO_OVER_WIFI_ENABLE, (value))
#define WMI_RSRC_CFG_FLAG_VIDEO_OVER_WIFI_ENABLE_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), VIDEO_OVER_WIFI_ENABLE)

#define WMI_RSRC_CFG_FLAG_THREE_WAY_COEX_CONFIG_LEGACY_SUPPORT_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), THREE_WAY_COEX_CONFIG_LEGACY_SUPPORT, (value))
#define WMI_RSRC_CFG_FLAG_THREE_WAY_COEX_CONFIG_LEGACY_SUPPORT_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), THREE_WAY_COEX_CONFIG_LEGACY_SUPPORT)

#define WMI_RSRC_CFG_FLAG_THREE_WAY_COEX_CONFIG_OVERRIDE_SUPPORT_SET(word32, value) \
    WMI_RSRC_CFG_FLAG_SET((word32), THREE_WAY_COEX_CONFIG_OVERRIDE_SUPPORT, (value))
#define WMI_RSRC_CFG_FLAG_THREE_WAY_COEX_CONFIG_OVERRIDE_SUPPORT_GET(word32) \
    WMI_RSRC_CFG_FLAG_GET((word32), THREE_WAY_COEX_CONFIG_OVERRIDE_SUPPORT)

typedef struct {
    A_UINT32   tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_init_cmd_fixed_param */

    /** The following indicate the WMI versions to be supported by
     *  the host driver. Note that the host driver decide to
     *  "downgrade" its WMI version support and this may not be the
     *  native version of the host driver. */
    wmi_abi_version  host_abi_vers;

    A_UINT32   num_host_mem_chunks; /** size of array host_mem_chunks[] */
/* The TLVs for resource_config, host_mem_chunks[], and hw_mode_config will follow.
     *     wmi_resource_config   resource_config;
     *     wlan_host_memory_chunk host_mem_chunks[];
     *     wmi_pdev_set_hw_mode_cmd_fixed_param hw_mode_config;
     *         Note that the hw_mode_config, in spite of its "pdev" name,
     *         applies to the entire target rather than for a single pdev
     *         within the target.
     *         To avoid specifying a HW mode for the target, the host should
     *         fill hw_mode_config's fields with 0x0.
     */

} wmi_init_cmd_fixed_param;

/**
 * TLV for channel list
 */
typedef struct {
    /** WMI_CHAN_LIST_TAG */
    A_UINT32     tag;
    /** # of channels to scan */
    A_UINT32 num_chan;
    /** channels in Mhz */
    A_UINT32 channel_list[1];
} wmi_chan_list;

/**
 * TLV for bssid list
 */
typedef struct {
    /** WMI_BSSID_LIST_TAG */
    A_UINT32     tag;
    /** number of bssids   */
    A_UINT32 num_bssid;
    /** bssid list         */
    wmi_mac_addr bssid_list[1];
} wmi_bssid_list;

/**
 * TLV for  ie data.
 */
typedef struct {
    /** WMI_IE_TAG */
    A_UINT32     tag;
    /** number of bytes in ie data   */
    A_UINT32 ie_len;
    /** ie data array  (ie_len adjusted to  number of words  (ie_len + 4)/4 )  */
    A_UINT32 ie_data[1];
} wmi_ie_data;

/**
 * TLV used for length/buffer
 */
typedef struct {
    /**
     * TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_tlv_buf_len_param
     */
    A_UINT32 tlv_header;
    A_UINT32 buf_len;       /** Length of buf */
    /**
     * Following this structure is the TLV byte stream of buf
     * of length buf_len:
     * A_UINT8 buf[];
     *
     */
} wmi_tlv_buf_len_param;

/**
 * TLV used for specifying the demensions of a multi-dimensional array
 * that has been stored in a flat buffer
 */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tlv_arrays_len_param */
    /**
     * d1_len, d2_len, d3_len, and d4_len are the lengths of each dimension
     * for a multi-dimensional array.
     * If the length of outer dimension is not 1, the inner dimension
     * shouldn't be 1.
     * If the multi-dimensional array has less than 4 dimensions, the outer
     * dimensions' lengths should be 1.  For example, a buf[3][4] array
     * would have d1_len = 4, d2_len = 3, d3_len = 1, d4_len = 1.
     * The outermost dimension of the array can be inferred from the array
     * length; thus, this struct supports up to 5-D arrays.  For a 5-D array,
     * the outermost (5th) dimension would be
     *     array length / (d1_len * d2_len * d3_len * d4_len)
     *
     * For security (to ensure no out-of-bounds memory access),
     * the receiver shall validate that the product of all dimensions
     * is equal to (or less than) the array length.
     */
    A_UINT32 d1_len;  /* the length of 1st (innermost) dimension array */
    A_UINT32 d2_len;  /* the length of 2nd dimension array */
    A_UINT32 d3_len;  /* the length of 3rd dimension array */
    A_UINT32 d4_len;  /* the length of 4th dimension array */
    /**
     * Following this structure is the TLV multi-dimension array buffer:
     * <type> buf[L1*L2*L3*L4];
     * where, L1, L2, L3, and L4 are the values of
     * d1_len, d2_len, d3_len and d4_len.
     * To access the 4-D element a[i][j][k][l], the buf[] array would be
     * indexed as buf[i*L3*L2*L1 + j*L2*L1 + k*L1 + l].
     */
} wmi_tlv_arrays_len_param;

typedef struct {
    /** Len of the SSID */
    A_UINT32     ssid_len;
    /** SSID */
    A_UINT32     ssid[8];
} wmi_ssid;

typedef struct {
    /** WMI_SSID_LIST_TAG */
    A_UINT32     tag;
    A_UINT32     num_ssids;
    wmi_ssid ssids[1];
} wmi_ssid_list;

typedef struct {
    /** WMI_SCAN_START_OFFSET_TAG */
    A_UINT32     tag;
    /** Number of start TSF offsets */
    A_UINT32     num_offset;
    /** Array of start TSF offsets provided in milliseconds */
    A_UINT32     start_tsf_offset[1];
} wmi_scan_start_offset;

/**
 * WLAN_SCAN_CHAN_MODE Macros defined for A_UINT8 phymode_list[]
 */
/** enum WLAN_PHY_MODE _mode starts from 0, but the WMI message requires
 * 0 to be used to represent unspecified / don't care / default values.
 * Therefore, WMI phy mode = WLAN phy mode + 1.
 */
/** If the received WMI phy mode is 0 then it is ignored by the FW,
 * and the FW will use any mode as long as the frequency matches.
 */
/** The number of phy_mode's (BW+mode) passed in the TLV phymode_list[] must
 * be equal to num_chan.  (Unless the host does not specify phymode_list values
 * at all, in which case the number of phymode_list elements will be zero.)
 * The indexing of the phymode_list[] array corresponds to same index of
 * the chan_list[] array.
 */
#define WMI_SCAN_CHAN_SET_MODE(_c) ((_c) + 1)
#define WMI_SCAN_CHAN_GET_MODE(_c) ((_c) - 1)
#define WMI_SCAN_CHAN_MODE_IS_SET(_c) (_c)

/* prefix used by scan requestor ids on the host */
#define WMI_HOST_SCAN_REQUESTOR_ID_PREFIX 0xA000
/* prefix used by scan request ids generated on the host */
/* host cycles through the lower 12 bits to generate ids */
#define WMI_HOST_SCAN_REQ_ID_PREFIX 0xA000

#define WLAN_SCAN_PARAMS_MAX_SSID    16
#define WLAN_SCAN_PARAMS_MAX_BSSID   4
#define WLAN_SCAN_PARAMS_MAX_IE_LEN  512

/* NOTE: This constant cannot be changed without breaking WMI compatibility */
#define WMI_IE_BITMAP_SIZE 8

typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_start_scan_cmd_fixed_param */
    /** Scan ID (lower 16 bits) MSB 4 bits is used to identify scan client based on enum WMI_SCAN_CLIENT_ID */
    A_UINT32 scan_id;
    /** Scan requestor ID (lower 16 bits) is used by scan client to classify the scan source, reason, ...etc */
    A_UINT32 scan_req_id;
    /** VDEV id(interface) that is requesting scan */
    A_UINT32 vdev_id;
    /** Scan Priority, input to scan scheduler */
    A_UINT32 scan_priority;
    /** Scan events subscription */
    A_UINT32 notify_scan_events;
    /** dwell time in msec on active channels */
    A_UINT32 dwell_time_active;
    /** dwell time in msec on passive channels */
    A_UINT32 dwell_time_passive;
    /** min time in msec on the BSS channel,only valid if atleast one VDEV is active*/
    A_UINT32 min_rest_time;
    /** max rest time in msec on the BSS channel,only valid if at least one VDEV is active*/
    /** the scanner will rest on the bss channel at least min_rest_time. after min_rest_time the scanner
     *  will start checking for tx/rx activity on all VDEVs. if there is no activity the scanner will
     *  switch to off channel. if there is activity the scanner will let the radio on the bss channel
     *  until max_rest_time expires.at max_rest_time scanner will switch to off channel
     *  irrespective of activity. activity is determined by the idle_time parameter.
     */
    A_UINT32  max_rest_time;
    /** time before sending next set of probe requests.
     *   The scanner keeps repeating probe requests transmission with period specified by repeat_probe_time.
     *   The number of probe requests specified depends on the ssid_list and bssid_list
     */
    A_UINT32  repeat_probe_time;
    /** time in msec between 2 consequetive probe requests with in a set. */
    A_UINT32  probe_spacing_time;
    /** data inactivity time in msec on bss channel that will be used by scanner for measuring the inactivity  */
    A_UINT32 idle_time;
    /** maximum time in msec allowed for scan  */
    A_UINT32  max_scan_time;
    /** delay in msec before sending first probe request after switching to a channel */
    A_UINT32  probe_delay;
    /** Scan control flags */
    A_UINT32 scan_ctrl_flags;
    /** Burst duration time in msec*/
    A_UINT32 burst_duration;

    /** # if channels to scan. In the TLV channel_list[] */
    A_UINT32 num_chan;
    /** number of bssids. In the TLV bssid_list[] */
    A_UINT32 num_bssid;
    /** number of ssid. In the TLV ssid_list[] */
    A_UINT32     num_ssids;
    /** number of bytes in ie data. In the TLV ie_data[]. Max len is defined by WLAN_SCAN_PARAMS_MAX_IE_LEN */
    A_UINT32 ie_len;
    /** Max number of probes to be sent */
    A_UINT32 n_probes;
    /** MAC Address to use in Probe Req as SA **/
    wmi_mac_addr mac_addr;
    /** Mask on which MAC has to be randomized **/
    wmi_mac_addr mac_mask;
    /**  ie bitmap to use in probe req **/
    A_UINT32 ie_bitmap[WMI_IE_BITMAP_SIZE];
    /** Number of vendor OUIs. In the TLV vendor_oui[] **/
    A_UINT32 num_vendor_oui;
    /** Scan control flags extended **/
    A_UINT32 scan_ctrl_flags_ext;
    /** dwell time in msec on active 2G channels, if it's not zero */
    A_UINT32 dwell_time_active_2g;

/**
 * TLV (tag length value ) parameters follow the scan_cmd
 * structure. The TLV's are:
 *     A_UINT32 channel_list[num_chan];
 *     wmi_ssid ssid_list[num_ssids];
 *     wmi_mac_addr bssid_list[num_bssid];
 *     A_UINT8 ie_data[ie_len];
 *     wmi_vendor_oui vendor_oui[num_vendor_oui];
 *     A_UINT8 phymode_list[0 or num_chan]; // see WMI_SCAN_CHAN_MODE macros
 */
} wmi_start_scan_cmd_fixed_param;

/**
 * scan control flags.
 */

/** passively scan all channels including active channels */
#define WMI_SCAN_FLAG_PASSIVE        0x1
/** add wild card ssid probe request even though ssid_list is specified. */
#define WMI_SCAN_ADD_BCAST_PROBE_REQ 0x2
/** add cck rates to rates/xrate ie for the generated probe request */
#define WMI_SCAN_ADD_CCK_RATES       0x4
/** add ofdm rates to rates/xrate ie for the generated probe request */
#define WMI_SCAN_ADD_OFDM_RATES      0x8
/** To enable indication of Chan load and Noise floor to host */
#define WMI_SCAN_CHAN_STAT_EVENT     0x10
/** Filter Probe request frames  */
#define WMI_SCAN_FILTER_PROBE_REQ    0x20
/**When set, not to scan DFS channels*/
#define WMI_SCAN_BYPASS_DFS_CHN      0x40
/**When set, certain errors are ignored and scan continues.
* Different FW scan engine may use its own logic to decide what errors to ignore*/
#define WMI_SCAN_CONTINUE_ON_ERROR   0x80
/** Enable promiscous mode for ese */
#define WMI_SCAN_FILTER_PROMISCOUS   0x100
/** allow to send probe req on DFS channel */
#define WMI_SCAN_FLAG_FORCE_ACTIVE_ON_DFS 0x200
/** add TPC content in probe req frame */
#define WMI_SCAN_ADD_TPC_IE_IN_PROBE_REQ  0x400
/** add DS content in probe req frame */
#define WMI_SCAN_ADD_DS_IE_IN_PROBE_REQ   0x800
/** use random mac address for TA for probe request frame and add
 * oui specified by WMI_SCAN_PROB_REQ_OUI_CMDID to the probe req frame.
 * if oui is not set by WMI_SCAN_PROB_REQ_OUI_CMDID  then the flag is ignored*/
#define WMI_SCAN_ADD_SPOOFED_MAC_IN_PROBE_REQ   0x1000
/** allow mgmt transmission during off channel scan */
#define WMI_SCAN_OFFCHAN_MGMT_TX    0x2000
/** allow data transmission during off channel scan */
#define WMI_SCAN_OFFCHAN_DATA_TX    0x4000
/** allow capture ppdu with phy errrors */
#define WMI_SCAN_CAPTURE_PHY_ERROR  0x8000

/** always do passive scan on passive channels */
#define WMI_SCAN_FLAG_STRICT_PASSIVE_ON_PCHN 0x10000
/** set HALF (10MHz) rate support */
#define WMI_SCAN_FLAG_HALF_RATE_SUPPORT      0x20000
/** set Quarter (5MHz) rate support */
#define WMI_SCAN_FLAG_QUARTER_RATE_SUPPORT   0x40000
#define WMI_SCAN_RANDOM_SEQ_NO_IN_PROBE_REQ 0x80000
#define WMI_SCAN_ENABLE_IE_WHTELIST_IN_PROBE_REQ 0x100000

/** for adaptive scan mode using 3 bits (21 - 23 bits) */
#define WMI_SCAN_DWELL_MODE_MASK 0x00E00000
#define WMI_SCAN_DWELL_MODE_SHIFT        21

typedef enum {
    WMI_SCAN_DWELL_MODE_DEFAULT      = 0,
    WMI_SCAN_DWELL_MODE_CONSERVATIVE = 1,
    WMI_SCAN_DWELL_MODE_MODERATE     = 2,
    WMI_SCAN_DWELL_MODE_AGGRESSIVE   = 3,
    WMI_SCAN_DWELL_MODE_STATIC       = 4,
} WMI_SCAN_DWELL_MODE;

#define WMI_SCAN_SET_DWELL_MODE(flag, mode) \
    do { \
        (flag) |= (((mode) << WMI_SCAN_DWELL_MODE_SHIFT) & \
            WMI_SCAN_DWELL_MODE_MASK); \
    } while(0)

#define WMI_SCAN_GET_DWELL_MODE(flag) \
    (((flag) & WMI_SCAN_DWELL_MODE_MASK) >> WMI_SCAN_DWELL_MODE_SHIFT)

/** WMI_SCAN_CLASS_MASK must be the same value as IEEE80211_SCAN_CLASS_MASK */
#define WMI_SCAN_CLASS_MASK 0xFF000000

/*
* Masks identifying types/ID of scans
* Scan_Stop macros should be the same value as below defined in UMAC
* #define IEEE80211_SPECIFIC_SCAN       0x00000000
* #define IEEE80211_VAP_SCAN            0x01000000
* #define IEEE80211_ALL_SCANS           0x04000000
*/
#define WMI_SCAN_STOP_ONE       0x00000000
#define WMI_SCN_STOP_VAP_ALL    0x01000000
#define WMI_SCAN_STOP_ALL       0x04000000

/** extended Scan ctrl flags **/
#define WMI_SCAN_FLAG_EXT_DBS_SCAN_POLICY_MASK 0x00000003 /* Bit 0-1 reserved for DBS scan selection policy.*/

#define WMI_SCAN_DBS_POLICY_DEFAULT             0x0 /** Select duty cycle if configured, else fall back to whatever
                                                        policy scan manager computes */
#define WMI_SCAN_DBS_POLICY_FORCE_NONDBS        0x1 /** Force to select Non-DBS scan */
#define WMI_SCAN_DBS_POLICY_IGNORE_DUTY         0x2 /** Ignore duty cycle even if configured and fall back to whatever
                                                        policy scan manager computes*/
#define WMI_SCAN_DBS_POLICY_RESERVED            0x3
#define WMI_SCAN_DBS_POLICY_MAX                 0x3

/** Enable Reception of Public Action frame with this flag
 * (inside scan_ctrl_flags_ext field of wmi_start_scan_cmd_fixed_param)
 */
#define WMI_SCAN_FLAG_EXT_FILTER_PUBLIC_ACTION_FRAME      0x4

typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_stop_scan_cmd_fixed_param */
    /** requestor requesting cancel  */
    A_UINT32 requestor;
    /** Scan ID */
    A_UINT32 scan_id;
    /**
    * Req Type
    * req_type should be WMI_SCAN_STOP_ONE, WMI_SCN_STOP_VAP_ALL or WMI_SCAN_STOP_ALL
    * WMI_SCAN_STOP_ONE indicates to stop a specific scan with scan_id
    * WMI_SCN_STOP_VAP_ALL indicates to stop all scan requests on a specific
    * vDev with vdev_id and pdev with pdev_id
    * WMI_SCAN_STOP_ALL indicates to stop all scan requests in both Scheduler's queue and Scan Engine
    */
    A_UINT32 req_type;
    /**
    * vDev ID
    * used when req_type equals to WMI_SCN_STOP_VAP_ALL, it indexed the vDev on which to stop the scan
    */
    A_UINT32 vdev_id;
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     * In non-DBDC case host should set it to 0
     */
    A_UINT32 pdev_id;
} wmi_stop_scan_cmd_fixed_param;

#define MAX_NUM_CHAN_PER_WMI_CMD     58    // each WMI cmd can hold 58 channel entries at most
#define APPEND_TO_EXISTING_CHAN_LIST 1

typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_scan_chan_list_cmd_fixed_param */
    A_UINT32 num_scan_chans;  /** no of elements in chan_info[] */
    A_UINT32 flags; /* Flags used to control the behavior of channel list update on target side */
    A_UINT32 pdev_id; /* pdev_id for identifying the MAC. See macros starting with WMI_PDEV_ID_ for values */
    /** Followed by the variable length TLV chan_info:
     *  wmi_channel chan_info[] */
} wmi_scan_chan_list_cmd_fixed_param;

/*
 * Priority numbers must be sequential, starting with 0.
 */
 /* NOTE: WLAN SCAN_PRIORITY_COUNT can't be changed without breaking the compatibility */
typedef enum {
    WMI_SCAN_PRIORITY_VERY_LOW    = 0,
    WMI_SCAN_PRIORITY_LOW,
    WMI_SCAN_PRIORITY_MEDIUM,
    WMI_SCAN_PRIORITY_HIGH,
    WMI_SCAN_PRIORITY_VERY_HIGH,

    WMI_SCAN_PRIORITY_COUNT   /* number of priorities supported */
} wmi_scan_priority;

/* Five Levels for Requested Priority */
/* VERY_LOW LOW  MEDIUM   HIGH  VERY_HIGH */
typedef A_UINT32 WLAN_PRIORITY_MAPPING[WMI_SCAN_PRIORITY_COUNT];

/**
* to keep align with UMAC implementation, we pass only vdev_type but not vdev_subtype when we overwrite an entry for a specific vdev_subtype
* ex. if we need overwrite P2P Client prority entry, we will overwrite the whole table for WLAN_M_STA
* we will generate the new WLAN_M_STA table with modified P2P Client Entry but keep STA entry intact
*/
typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_scan_sch_priority_table_cmd_fixed_param */
    /**
    * used as an index to find the proper table for a specific vdev type in default_scan_priority_mapping_table
    * vdev_type should be one of enum in WLAN_OPMODE which inculdes WLAN_M_IBSS, WLAN_M_STA, WLAN_M_AP and WLAN_M_MONITOR currently
    */
    A_UINT32      vdev_type;
    /**
    * number of rows in mapping_table for a specific vdev
    * for WLAN_M_STA type, there are 3 entries in the table (refer to default_scan_priority_mapping_table definition)
    */
    A_UINT32                    number_rows;
    /**
     * pdev_id for identifying the MAC.  See macros starting with
     * WMI_PDEV_ID_ for values.In non-DBDC case host should
     * set it to 0.
     */
    A_UINT32 pdev_id;
    /**  mapping_table for a specific vdev follows this TLV
      *   WLAN_PRIORITY_MAPPING mapping_table[]; */
}wmi_scan_sch_priority_table_cmd_fixed_param;

/** update flags */
#define WMI_SCAN_UPDATE_SCAN_PRIORITY           0x1
#define WMI_SCAN_UPDATE_SCAN_MIN_REST_TIME      0x2
#define WMI_SCAN_UPDATE_SCAN_MAX_REST_TIME      0x4

typedef struct {
    A_UINT32 tlv_header;
    /** requestor requesting update scan request  */
    A_UINT32 requestor;
    /** Scan ID of the scan request that need to be update */
    A_UINT32 scan_id;
    /** update flags, indicating which of the following fields are valid and need to be updated*/
    A_UINT32 scan_update_flags;
    /** scan priority. Only valid if WMI_SCAN_UPDATE_SCAN_PRIORITY flag is set in scan_update_flag */
    A_UINT32 scan_priority;
    /** min rest time. Only valid if WMI_SCAN_UPDATE_MIN_REST_TIME flag is set in scan_update_flag */
    A_UINT32 min_rest_time;
    /** min rest time. Only valid if WMI_SCAN_UPDATE_MAX_REST_TIME flag is set in scan_update_flag */
    A_UINT32 max_rest_time;
    /**
     * pdev_id for identifying the MAC. See macros starting with
     * WMI_PDEV_ID_ for values. In non-DBDC case host should set it to 0
     */
    A_UINT32 pdev_id;
} wmi_scan_update_request_cmd_fixed_param;

#define WMI_SCAN_PROBE_OUI_SPOOFED_MAC_IN_PROBE_REQ 0x1
#define WMI_SCAN_PROBE_OUI_RANDOM_SEQ_NO_IN_PROBE_REQ 0x2
#define WMI_SCAN_PROBE_OUI_ENABLE_IE_WHITELIST_IN_PROBE_REQ 0x4

typedef struct _wmi_vendor_oui {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vendor_oui */
    A_UINT32 oui_type_subtype; /** Vendor OUI type and subtype, lower 3 bytes is type and highest byte is subtype**/
}wmi_vendor_oui;

typedef struct {
    A_UINT32 tlv_header;
    /** oui to be used in probe request frame when  random mac addresss is
     * requested part of scan parameters. this is applied to both FW internal scans and
     * host initated scans. host can request for random mac address with
     * WMI_SCAN_ADD_SPOOFED_MAC_IN_PROBE_REQ flag.     */
    A_UINT32 prob_req_oui;
    A_UINT32 vdev_id;
    /** Control Flags **/
    A_UINT32 flags;
    /**  ie bitmap to use in probe req **/
    A_UINT32 ie_bitmap[WMI_IE_BITMAP_SIZE];
    /** Number of vendor OUIs. In the TLV vendor_oui[] **/
    A_UINT32 num_vendor_oui;
    /**
     * pdev_id for identifying the MAC. See macros starting with
     * WMI_PDEV_ID_ for values. In non-DBDC case host should set it to 0
     */
    A_UINT32 pdev_id;
    /* Following this tlv, there comes an array of structure of type wmi_vendor_ouiwmi_vendor_oui vendor_oui[];*/

} wmi_scan_prob_req_oui_cmd_fixed_param;

enum wmi_scan_event_type {
    WMI_SCAN_EVENT_STARTED=0x1,
    WMI_SCAN_EVENT_COMPLETED=0x2,
    WMI_SCAN_EVENT_BSS_CHANNEL=0x4,
    WMI_SCAN_EVENT_FOREIGN_CHANNEL = 0x8,
    WMI_SCAN_EVENT_DEQUEUED=0x10,       /* scan request got dequeued */
    WMI_SCAN_EVENT_PREEMPTED=0x20,		/* preempted by other high priority scan */
    WMI_SCAN_EVENT_START_FAILED=0x40,   /* scan start failed */
    WMI_SCAN_EVENT_RESTARTED=0x80,      /* Scan restarted */
    WMI_SCAN_EVENT_FOREIGN_CHANNEL_EXIT = 0x100,
    WMI_SCAN_EVENT_SUSPENDED = 0x200, /* scan request is suspended */
    WMI_SCAN_EVENT_RESUMED = 0x400,   /* scan request is resumed */
    WMI_SCAN_EVENT_MAX=0x8000
};

enum wmi_scan_completion_reason {
    /** scan related events */
    WMI_SCAN_REASON_NONE                = 0xFF,
    WMI_SCAN_REASON_COMPLETED           = 0,
    WMI_SCAN_REASON_CANCELLED           = 1,
    WMI_SCAN_REASON_PREEMPTED           = 2,
    WMI_SCAN_REASON_TIMEDOUT            = 3,
    WMI_SCAN_REASON_INTERNAL_FAILURE    = 4, /* This reason indication failures when performaing scan */
    WMI_SCAN_REASON_SUSPENDED           = 5,
    WMI_SCAN_REASON_DFS_VIOLATION       = 6, /* Failure when tried to SCAN channel in NOL list */
    WMI_SCAN_REASON_MAX,
};

typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_scan_event_fixed_param */
    /** scan event (wmi_scan_event_type) */
    A_UINT32 event;
    /** status of the scan completion event */
    A_UINT32 reason;
    /** channel freq , only valid for FOREIGN channel event*/
    A_UINT32 channel_freq;
    /**id of the requestor whose scan is in progress */
    A_UINT32 requestor;
    /**id of the scan that is in progress */
    A_UINT32 scan_id;
    /**id of VDEV that requested the scan */
    A_UINT32 vdev_id;
    /** TSF Timestamp when the scan event (wmi_scan_event_type) is completed
     * In case of AP it is TSF of the AP vdev
     * In case of STA connected state this is the TSF of the AP
     * In case of STA not connected it will be the free running HW timer
     */
    A_UINT32 tsf_timestamp;
} wmi_scan_event_fixed_param;

/* WMI Diag event */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag is WMITLV_TAG_STRUC_wmi_diag_event_fixed_param */
    A_UINT32 time_stamp; /* Reference timestamp. diag frame contains diff value */
    A_UINT32 count;   /* Number of diag frames added to current event */
    A_UINT32 dropped;
    /* followed by WMITLV_TAG_ARRAY_BYTE */
} wmi_diag_event_fixed_param;

#define WMI_11K_OFFLOAD_BITMAP_NEIGHBOR_REPORT_REQ  0x1

typedef struct {
    A_UINT32 time_offset;                   /* positive offset in secs from the time 11k offload command has been received, 0xFFFFFFFF if offset is not valid */
    A_UINT32 low_rssi_offset;               /* positive offset in dB from current low rssi roaming trigger to send neighbor req, 0xFFFFFFFF if offset is not valid */
    A_UINT32 bmiss_count_trigger;           /* value 1 is to send neighbor report at 1st BMISS, 0xFFFFFFFF if input is not valid */
    A_UINT32 per_threshold_offset;          /* percentage offset from the current per_threshold, 0xFFFFFFFF if input is not valid */
    A_UINT32 neighbor_report_cache_timeout; /* cache timeout in secs after which neighbor cache is not valid in FW, 0xFFFFFFFF if input is not valid */
    A_UINT32 max_neighbor_report_req_cap;   /* 0xFFFFFFFF if input is not valid, else positive number per every roam, these are the maximum number of
                                             * neighbor report requests that will be sent by FW after every roam */
    wmi_ssid ssid;                          /* ssid of current connected AP FW might choose to use this SSID in the neighbor report req frame if it is
                                             * interested in candidate of the same SSID */
} wmi_neighbor_report_offload;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_offload_11k_report_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 offload_11k; /* bitmask to indicate to FW what all 11k features are offloaded */
} wmi_11k_offload_report_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_neighbor_report_offload_tlv_param */
    wmi_neighbor_report_offload neighbor_rep_ofld_params;
} wmi_neighbor_report_11k_offload_tlv_param;

#define WMI_INVOKE_NEIGHBOR_REPORT_FLAGS_SEND_RESP_TO_HOST 0x1

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_invoke_neighbor_report_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 flags;
    wmi_ssid ssid; /* if ssid.len == 0, firmware doesn't include ssid sub-element.
                    * In that case AP gives all the candidates in ESS without SSID filter
                    * If host wants to insert ssid subelement in the neighbor report request frame, then it can specify the ssid here */
} wmi_11k_offload_invoke_neighbor_report_fixed_param;

#define WMI_MAX_PMKID_LEN   16
#define WMI_MAX_PMK_LEN     64

#define WMI_PMK_CACHE_CAT_FLAG_BSSID              0x1
#define WMI_PMK_CACHE_CAT_FLAG_SSID_CACHE_ID      0x2

#define WMI_PMK_CACHE_ACTION_FLAG_ADD_ENTRY       0x1
#define WMI_PMK_CACHE_ACTION_FLAG_DEL_ENTRY       0x2

typedef struct {
    A_UINT32       tlv_header;
    A_UINT32       pmk_len;
    A_UINT8        pmk[WMI_MAX_PMK_LEN];/* for big-endian hosts, manual endian conversion will be needed to keep the array values in their original order,
                                        in spite of the automatic byte-swap applied to WMI messages during download*/
    A_UINT32       pmkid_len;
    A_UINT8        pmkid[WMI_MAX_PMKID_LEN];
    wmi_mac_addr   bssid;
    wmi_ssid       ssid;
    A_UINT32       cache_id;
    A_UINT32       cat_flag;  // whether (bssid) or (ssid,cache_id) is valid
    A_UINT32       action_flag;  // add/delete the entry
} wmi_pmk_cache;

#define WMI_PMK_CACHE_OP_FLAG_FLUSH_ALL       0x1

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_update_pmk_cache_cmd_fixed_param */
    A_UINT32 op_flag;   //option to flush all the cache at once
    A_UINT32 vdev_id;
    A_UINT32 num_cache;
    /**
    * TLV (tag length value) parameters follow the update_pmk_cache cmd
    * structure. The TLV's are:
     * wmi_pmk_cache cache_list[];
    */
} wmi_pdev_update_pmk_cache_cmd_fixed_param;

#define WMI_FILS_MAX_USERNAME_LEN 16
#define WMI_FILS_MAX_REALM_LEN 256
#define WMI_FILS_MAX_RRK_LEN 64
#define WMI_FILS_MAX_RIK_LEN 64

/* for big-endian hosts, manual endian conversion will be needed to keep the array values in their original order,
in spite of the automatic byte-swap applied to WMI messages during download*/

typedef struct {
    A_UINT8     username[WMI_FILS_MAX_USERNAME_LEN];
    A_UINT32    username_length;
    A_UINT32    next_erp_seq_num;
    A_UINT8     rRk[WMI_FILS_MAX_RRK_LEN];
    A_UINT32    rRk_length;
    A_UINT8     rIk[WMI_FILS_MAX_RIK_LEN];
    A_UINT32    rIk_length;
    A_UINT8     realm[WMI_FILS_MAX_REALM_LEN];
    A_UINT32    realm_len;
} wmi_erp_info;

enum wmi_fils_hlp_pkt_type {
    WMI_FILS_HLP_PKT_TYPE_DHCP_DISCOVER = 1,
};

typedef struct {
    A_UINT32      tlv_header;  /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_fils_offload_tlv_param */
    A_UINT32      flags;
    wmi_erp_info  vdev_erp_info;
} wmi_roam_fils_offload_tlv_param;

typedef struct {
    A_UINT32  tlv_header; /** tag WMITLV_TAG_STRUC_wmi_pdev_update_fils_hlp_pkt_cmd_fixed_param**/
    A_UINT32  flags;
    A_UINT32  vdev_id;
    A_UINT32  size;
    A_UINT32  pkt_type; // filled using enum wmi_fils_hlp_pkt_type
 // A_UINT8          fils_hlp_pkt[];
} wmi_pdev_update_fils_hlp_pkt_cmd_fixed_param;

#define WMI_MAX_KEK_LEN 64
#define GTK_OFFLOAD_KEK_EXTENDED_BYTES WMI_MAX_KEK_LEN /*KEK len has been increased to 64 to support FILS security.
                                          To not break backward compatibility, new GTK_OFFLOAD_KEK_EXTENDED_BYTES has been defined without modifying old GTK_OFFLOAD_KEK_BYTES */

typedef struct {
    A_UINT32   tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_fils_synch_tlv_param */
    A_UINT32   update_erp_next_seq_num;// Boolean denoting whether next erp_seq_num changed or not.
    A_UINT32   next_erp_seq_num;
    A_UINT32   kek_len;
    A_UINT8    kek[WMI_MAX_KEK_LEN];
    A_UINT32   pmk_len;
    A_UINT8    pmk[WMI_MAX_PMK_LEN];
    A_UINT8    pmkid[WMI_MAX_PMKID_LEN];
    A_UINT8    realm[WMI_FILS_MAX_REALM_LEN];
    A_UINT32   realm_len;
} wmi_roam_fils_synch_tlv_param;

/*
* If FW has multiple active channels due to MCC(multi channel concurrency),
* then these stats are combined stats for all the active channels.
*/
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_update_whal_mib_stats_event_fixed_param */
    /** ack count, it is an incremental number, not accumulated number */
    A_UINT32 ackRcvBad;
    /** bad rts count, it is an incremental number, not accumulated number */
    A_UINT32 rtsBad;
    /** good rts, it is an incremental number, not accumulated number */
    A_UINT32 rtsGood;
    /** fcs count, it is an incremental number, not accumulated number */
    A_UINT32 fcsBad;
    /** beacon count, it is an incremental number, not accumulated number */
    A_UINT32 noBeacons;
} wmi_update_whal_mib_stats_event_fixed_param;

/*
 * This defines how much headroom is kept in the
 * receive frame between the descriptor and the
 * payload, in order for the WMI PHY error and
 * management handler to insert header contents.
 *
 * This is in bytes.
 */
#define WMI_MGMT_RX_HDR_HEADROOM    sizeof(wmi_comb_phyerr_rx_hdr) + WMI_TLV_HDR_SIZE + sizeof(wmi_single_phyerr_rx_hdr)

/** This event will be used for sending scan results
 * as well as rx mgmt frames to the host. The rx buffer
 * will be sent as part of this WMI event. It would be a
 * good idea to pass all the fields in the RX status
 * descriptor up to the host.
 */
 /* ATH_MAX_ANTENNA value (4) can't be changed without breaking the compatibility */
#define ATH_MAX_ANTENNA 4 /* To support beelinear, which is up to 4 chains */

/** flag indicating that the mgmt frame (probe req/beacon) is received in the context of extscan performed by FW */
#define WMI_MGMT_RX_HDR_EXTSCAN     0x01

/** flag indicating that the mgmt frame (probe req/beacon) is received in the context of matched network by FW ENLO */
#define WMI_MGMT_RX_HDR_ENLO     0x02

#define MAX_ANTENNA_EIGHT  8

typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_mgmt_rx_hdr */
    /** channel on which this frame is received. */
    A_UINT32 channel;
    /** snr information used to cal rssi */
    A_UINT32 snr;
    /** Rate kbps */
    A_UINT32 rate;
    /** rx phy mode WLAN_PHY_MODE */
    A_UINT32 phy_mode;
    /** length of the frame */
    A_UINT32 buf_len;
    /** rx status */
    A_UINT32 status;
    /** RSSI of PRI 20MHz for each chain. */
    A_UINT32 rssi_ctl[ATH_MAX_ANTENNA];
    /** information about the management frame e.g. can give a scan source for a scan result mgmt frame */
    A_UINT32 flags;
    /** combined RSSI, i.e. the sum of the snr + noise floor (dBm units) */
    A_INT32 rssi;
    /** delta between local TSF(TSF timestamp when frame was RXd)
     *  and remote TSF(TSF timestamp in the IE for mgmt frame -
     *  beacon,proberesp for e.g). If remote TSF is not available,
     *  delta set to 0.
     *  Although tsf_delta is stored as A_UINT32, it can be negative,
     *  and thus would need to be sign-extended if added to a value
     *  larger than 32 bits.
     */
    A_UINT32 tsf_delta;

    /* The lower 32 bits of the TSF (rx_tsf_l32) is copied by FW from
     * TSF timestamp in the RX MAC descriptor provided by HW.
     */
    A_UINT32 rx_tsf_l32;

    /* The Upper 32 bits (rx_tsf_u32) is filled by reading the TSF register
     * after the packet is received.
     */
    A_UINT32 rx_tsf_u32;

    /** pdev_id for identifying the MAC the rx mgmt frame was received by
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;

    /* This TLV is followed by array of bytes:
         * // management frame buffer
         *   A_UINT8 bufp[];
         */
/* This TLV is optionally followed by array of struct:
 *  wmi_rssi_ctl_ext rssi_ctl_ext;
 */
} wmi_mgmt_rx_hdr;

/*
 * Instead of universally increasing the RX_HDR_HEADROOM size which may cause problems for older targets,
 * this new ext_hdr can be used for extending the header and will be only applicable for new targets.
 */
typedef struct
{
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_rssi_ctl_ext */
    /** RSSI of PRI 20MHz for each chain, in dB w.r.t. noise floor */
    A_UINT32 rssi_ctl_ext[MAX_ANTENNA_EIGHT - ATH_MAX_ANTENNA];
} wmi_rssi_ctl_ext;

typedef struct {
    /** TSF timestamp */
    A_UINT32 tsf_timestamp;

    /**
     * Current freq1, freq2
     *
     * [7:0]:    freq1[lo]
     * [15:8] :   freq1[hi]
     * [23:16]:   freq2[lo]
     * [31:24]:   freq2[hi]
     */
    A_UINT32 freq_info_1;

    /**
     * Combined RSSI over all chains and channel width for this PHY error
     *
     * [7:0]: RSSI combined
     * [15:8]: Channel width (MHz)
     * [23:16]: PHY error code
     * [24:16]: reserved (future use)
     */
    A_UINT32 freq_info_2;

    /**
     * RSSI on chain 0 through 3
     *
     * This is formatted the same as the PPDU_START RX descriptor
     * field:
     *
     * [7:0]:   pri20
     * [15:8]:  sec20
     * [23:16]: sec40
     * [31:24]: sec80
     */
    A_UINT32 rssi_chain0;
    A_UINT32 rssi_chain1;
    A_UINT32 rssi_chain2;
    A_UINT32 rssi_chain3;

   /**
     * Last calibrated NF value for chain 0 through 3
     *
     * nf_list_1:
     *
     * + [15:0] - chain 0
     * + [31:16] - chain 1
     *
     * nf_list_2:
     *
     * + [15:0] - chain 2
     * + [31:16] - chain 3
     */
    A_UINT32 nf_list_1;
    A_UINT32 nf_list_2;

    /** Length of the frame */
    A_UINT32 buf_len;
} wmi_single_phyerr_rx_hdr;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_single_phyerr_ext_rx_hdr */
    /**
     * RSSI on chain 4 through 7 in dB w.r.t noise floor.
     *
     * This is formatted the same as the PPDU_START RX descriptor
     * field:
     *
     * [7:0]:   pri20
     * [15:8]:  sec20
     * [23:16]: sec40
     * [31:24]: sec80
     */
    A_UINT32 rssi_chain4;
    A_UINT32 rssi_chain5;
    A_UINT32 rssi_chain6;
    A_UINT32 rssi_chain7;
    /**
     * Last calibrated NF value for chain 4 through 7 in dbm
     *
     * nf_list_3:
     * + [15:0] - chain 4
     * + [31:16] - chain 5
     *
     * nf_list_4:
     * + [15:0] - chain 6
     * + [31:16] - chain 7
     *
     * Each chain's noise floor is stored as a sign-extended (negative)
     * value in dBm units.
     */
    A_UINT32 nf_list_3;
    A_UINT32 nf_list_4;
} wmi_single_phyerr_ext_rx_hdr;

#define WMI_UNIFIED_FREQINFO_1_LO   0x000000ff
#define WMI_UNIFIED_FREQINFO_1_LO_S 0
#define WMI_UNIFIED_FREQINFO_1_HI   0x0000ff00
#define WMI_UNIFIED_FREQINFO_1_HI_S 8
#define WMI_UNIFIED_FREQINFO_2_LO   0x00ff0000
#define WMI_UNIFIED_FREQINFO_2_LO_S 16
#define WMI_UNIFIED_FREQINFO_2_HI   0xff000000
#define WMI_UNIFIED_FREQINFO_2_HI_S 24

/*
 * Please keep in mind that these _SET macros break macro side effect
 * assumptions; don't be clever with them.
 */
#define WMI_UNIFIED_FREQ_INFO_GET(hdr, f)                                   \
            ( WMI_F_MS( (hdr)->freq_info_1,                                 \
              WMI_UNIFIED_FREQINFO_##f##_LO )                               \
              | (WMI_F_MS( (hdr)->freq_info_1,                              \
                 WMI_UNIFIED_FREQINFO_##f##_HI ) << 8) )

#define WMI_UNIFIED_FREQ_INFO_SET(hdr, f, v)                                \
        do {                                                                \
            WMI_F_RMW((hdr)->freq_info_1, (v) & 0xff,                       \
              WMI_UNIFIED_FREQINFO_##f##_LO);                               \
            WMI_F_RMW((hdr)->freq_info_1, ((v) >> 8) & 0xff,                \
                WMI_UNIFIED_FREQINFO_##f##_HI);                             \
        } while (0)

#define WMI_UNIFIED_FREQINFO_2_RSSI_COMB    0x000000ff
#define WMI_UNIFIED_FREQINFO_2_RSSI_COMB_S  0
#define WMI_UNIFIED_FREQINFO_2_CHWIDTH      0x0000ff00
#define WMI_UNIFIED_FREQINFO_2_CHWIDTH_S    8
#define WMI_UNIFIED_FREQINFO_2_PHYERRCODE   0x00ff0000
#define WMI_UNIFIED_FREQINFO_2_PHYERRCODE_S 16

#define WMI_UNIFIED_RSSI_COMB_GET(hdr)                                      \
            ( (int8_t) (WMI_F_MS((hdr)->freq_info_2,                        \
                WMI_UNIFIED_FREQINFO_2_RSSI_COMB)))

#define WMI_UNIFIED_RSSI_COMB_SET(hdr, v)                                   \
            WMI_F_RMW((hdr)->freq_info_2, (v) & 0xff,                       \
              WMI_UNIFIED_FREQINFO_2_RSSI_COMB);

#define WMI_UNIFIED_CHWIDTH_GET(hdr)                                        \
            WMI_F_MS((hdr)->freq_info_2, WMI_UNIFIED_FREQINFO_2_CHWIDTH)

#define WMI_UNIFIED_CHWIDTH_SET(hdr, v)                                     \
            WMI_F_RMW((hdr)->freq_info_2, (v) & 0xff,                       \
              WMI_UNIFIED_FREQINFO_2_CHWIDTH);

#define WMI_UNIFIED_PHYERRCODE_GET(hdr)                                     \
            WMI_F_MS((hdr)->freq_info_2, WMI_UNIFIED_FREQINFO_2_PHYERRCODE)

#define WMI_UNIFIED_PHYERRCODE_SET(hdr, v)                                  \
            WMI_F_RMW((hdr)->freq_info_2, (v) & 0xff,                       \
              WMI_UNIFIED_FREQINFO_2_PHYERRCODE);

#define WMI_UNIFIED_CHAIN_0     0x0000ffff
#define WMI_UNIFIED_CHAIN_0_S   0
#define WMI_UNIFIED_CHAIN_1     0xffff0000
#define WMI_UNIFIED_CHAIN_1_S   16
#define WMI_UNIFIED_CHAIN_2     0x0000ffff
#define WMI_UNIFIED_CHAIN_2_S   0
#define WMI_UNIFIED_CHAIN_3     0xffff0000
#define WMI_UNIFIED_CHAIN_3_S   16

#define WMI_UNIFIED_CHAIN_4     0x0000ffff
#define WMI_UNIFIED_CHAIN_4_S   0
#define WMI_UNIFIED_CHAIN_5     0xffff0000
#define WMI_UNIFIED_CHAIN_5_S   16
#define WMI_UNIFIED_CHAIN_6     0x0000ffff
#define WMI_UNIFIED_CHAIN_6_S   0
#define WMI_UNIFIED_CHAIN_7     0xffff0000
#define WMI_UNIFIED_CHAIN_7_S   16

#define WMI_UNIFIED_CHAIN_0_FIELD   nf_list_1
#define WMI_UNIFIED_CHAIN_1_FIELD   nf_list_1
#define WMI_UNIFIED_CHAIN_2_FIELD   nf_list_2
#define WMI_UNIFIED_CHAIN_3_FIELD   nf_list_2
#define WMI_UNIFIED_CHAIN_4_FIELD   nf_list_3
#define WMI_UNIFIED_CHAIN_5_FIELD   nf_list_3
#define WMI_UNIFIED_CHAIN_6_FIELD   nf_list_4
#define WMI_UNIFIED_CHAIN_7_FIELD   nf_list_4

#define WMI_UNIFIED_NF_CHAIN_GET(hdr, c)                                    \
            ((int16_t) (WMI_F_MS((hdr)->WMI_UNIFIED_CHAIN_##c##_FIELD,      \
              WMI_UNIFIED_CHAIN_##c)))

#define WMI_UNIFIED_NF_CHAIN_SET(hdr, c, nf)                                \
            WMI_F_RMW((hdr)->WMI_UNIFIED_CHAIN_##c##_FIELD, (nf) & 0xffff,  \
              WMI_UNIFIED_CHAIN_##c);

/*
 * For now, this matches what the underlying hardware is doing.
 * Update ar6000ProcRxDesc() to use these macros when populating
 * the rx descriptor and then we can just copy the field over
 * to the WMI PHY notification without worrying about breaking
 * things.
 */
#define WMI_UNIFIED_RSSI_CHAN_PRI20     0x000000ff
#define WMI_UNIFIED_RSSI_CHAN_PRI20_S   0
#define WMI_UNIFIED_RSSI_CHAN_SEC20     0x0000ff00
#define WMI_UNIFIED_RSSI_CHAN_SEC20_S   8
#define WMI_UNIFIED_RSSI_CHAN_SEC40     0x00ff0000
#define WMI_UNIFIED_RSSI_CHAN_SEC40_S   16
#define WMI_UNIFIED_RSSI_CHAN_SEC80     0xff000000
#define WMI_UNIFIED_RSSI_CHAN_SEC80_S   24

#define WMI_UNIFIED_RSSI_CHAN_SET(hdr, c, ch, rssi)                         \
            WMI_F_RMW((hdr)->rssi_chain##c, (rssi) & 0xff,                  \
              WMI_UNIFIED_RSSI_CHAN_##ch);

#define WMI_UNIFIED_RSSI_CHAN_GET(hdr, c, ch)                               \
            ((int8_t) (WMI_F_MS((hdr)->rssi_chain##c,                       \
              WMI_UNIFIED_RSSI_CHAN_##ch)))

#define WMI_UNIFIED_CHAIN_RSSI_GET(tlv, chain_idx, band) \
    ((A_INT8) WMI_F_MS((tlv)->chain_rssi[chain_idx], WMI_UNIFIED_RSSI_CHAN_ ## band))

typedef struct {
    /** Phy error event header */
    wmi_single_phyerr_rx_hdr hdr;
    /** frame buffer */
    A_UINT8 bufp[1];
}wmi_single_phyerr_rx_event;

/* PHY ERROR MASK 0 */
/* bits 1:0 defined but not published */
#define WMI_PHY_ERROR_MASK0_RADAR                           (1<<2 )
/* bits 23:3 defined but not published */
#define WMI_PHY_ERROR_MASK0_FALSE_RADAR_EXT                 (1<<24)
/* bits 25:24 defined but not published */
#define WMI_PHY_ERROR_MASK0_SPECTRAL_SCAN                   (1<<26)
/* bits 31:27 defined but not published */

/* PHY ERROR MASK 1 */
/* bits 13:0 defined but not published */
/* bits 31:14 reserved */

/* PHY ERROR MASK 2 */
/* bits 31:0 reserved */

typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_comb_phyerr_rx_hdr */
    /** Phy error phy error count */
    A_UINT32 num_phyerr_events;
    A_UINT32 tsf_l32;
    A_UINT32 tsf_u32;
    A_UINT32 buf_len;
    union {
        A_UINT32 pmac_id; // OBSOLETE - will be removed once all refs are gone
        /** pdev_id for identifying the MAC
         * See macros starting with WMI_PDEV_ID_ for values.
         */
        A_UINT32 pdev_id;
    };
    A_UINT32 rsPhyErrMask0; /* see WMI_PHY_ERROR_MASK0 */
    A_UINT32 rsPhyErrMask1; /* see WMI_PHY_ERROR_MASK1 */
    A_UINT32 rsPhyErrMask2; /* see WMI_PHY_ERROR_MASK2 */
    /* This TLV is followed by array of bytes:
         * // frame buffer - contains multiple payloads in the order:
         * // header - payload, header - payload...
         *  (The header is of type: wmi_single_phyerr_rx_hdr)
         *   A_UINT8 bufp[];
         *     The extension hdr will repeat num_phyerr_events of times
         *     and will have 1:1 mapping with above header. i.e the 1st
         *     ext_rx_hdr will belong to 1st phyerr_rx_hdr and so on.
         *   wmi_single_phyerr_ext_rx_hdr single_phyerr_ext;
         */
} wmi_comb_phyerr_rx_hdr;

/* WMI MGMT TX  */
typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_mgmt_tx_hdr */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /** xmit rate */
    A_UINT32 tx_rate;
    /** xmit power */
    A_UINT32 tx_power;
    /** Buffer length in bytes */
    A_UINT32 buf_len;
    /* This TLV is followed by array of bytes:
         * // management frame buffer
         *   A_UINT8 bufp[];
         */
} wmi_mgmt_tx_hdr;

#define WMI_TX_SEND_PARAM_PWR_GET(tx_param_dword0) WMI_GET_BITS(tx_param_dword0, 0, 8)
#define WMI_TX_SEND_PARAM_PWR_SET(tx_param_dword0, value) WMI_SET_BITS(tx_param_dword0, 0, 8, value)

#define WMI_TX_SEND_PARAM_MCS_MASK_GET(tx_param_dword0) WMI_GET_BITS(tx_param_dword0, 8, 12)
#define WMI_TX_SEND_PARAM_MCS_MASK_SET(tx_param_dword0, value) WMI_SET_BITS(tx_param_dword0, 8, 12, value)

#define WMI_TX_SEND_PARAM_NSS_MASK_GET(tx_param_dword0) WMI_GET_BITS(tx_param_dword0, 20, 8)
#define WMI_TX_SEND_PARAM_NSS_MASK_SET(tx_param_dword0, value) WMI_SET_BITS(tx_param_dword0, 20, 8, value)

#define WMI_TX_SEND_PARAM_RETRY_LIMIT_GET(tx_param_dword0) WMI_GET_BITS(tx_param_dword0, 28, 4)
#define WMI_TX_SEND_PARAM_RETRY_LIMIT_SET(tx_param_dword0, value) WMI_SET_BITS(tx_param_dword0, 28, 4, value)

#define WMI_TX_SEND_PARAM_CHAIN_MASK_GET(tx_param_dword1) WMI_GET_BITS(tx_param_dword1, 0, 8)
#define WMI_TX_SEND_PARAM_CHAIN_MASK_SET(tx_param_dword1, value) WMI_SET_BITS(tx_param_dword1, 0, 8, value)

#define WMI_TX_SEND_PARAM_BW_MASK_GET(tx_param_dword1) WMI_GET_BITS(tx_param_dword1, 8, 7)
#define WMI_TX_SEND_PARAM_BW_MASK_SET(tx_param_dword1, value) WMI_SET_BITS(tx_param_dword1, 8, 7, value)

#define WMI_TX_SEND_PARAM_PREAMBLE_GET(tx_param_dword1) WMI_GET_BITS(tx_param_dword1, 15, 5)
#define WMI_TX_SEND_PARAM_PREAMBLE_SET(tx_param_dword1, value) WMI_SET_BITS(tx_param_dword1, 15, 5, value)

#define WMI_TX_SEND_PARAM_FRAME_TYPE_GET(tx_param_dword1) WMI_GET_BITS(tx_param_dword1, 20, 1)
#define WMI_TX_SEND_PARAM_FRAME_TYPE_SET(tx_param_dword1, value) WMI_SET_BITS(tx_param_dword1, 20, 1, value)

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tx_send_params */

    union {
        struct {
            /* DWORD 0: tx power, tx rate, retry_limit */
            A_UINT32
                /* pwr -
                 * Specify what power the tx frame needs to be transmitted at.
                 * The power a signed (two's complement) value is in units of 0.5 dBm.
                 * The value needs to be appropriately sign-extended when extracting
                 * the value from the message and storing it in a variable that is
                 * larger than A_INT8.  (fw automatically handles this sign-extension.)
                 * If the transmission uses multiple tx chains, this power spec is
                 * the total transmit power, assuming incoherent combination of
                 * per-chain power to produce the total power.
                 */
                 pwr: 8,

                /* mcs_mask -
                 * Specify the allowable values for MCS index (modulation and coding)
                 * to use for transmitting the frame.
                 *
                 * For HT / VHT preamble types, this mask directly corresponds to
                 * the HT or VHT MCS indices that are allowed.  For each bit N set
                 * within the mask, MCS index N is allowed for transmitting the frame.
                 * For legacy CCK and OFDM rates, separate bits are provided for CCK
                 * rates versus OFDM rates, so the host has the option of specifying
                 * that the target must transmit the frame with CCK or OFDM rates
                 * (not HT or VHT), but leaving the decision to the target whether
                 * to use CCK or OFDM.
                 *
                 * For CCK and OFDM, the bits within this mask are interpreted as
                 * follows:
                 *     bit  0 -> CCK 1 Mbps rate is allowed
                 *     bit  1 -> CCK 2 Mbps rate is allowed
                 *     bit  2 -> CCK 5.5 Mbps rate is allowed
                 *     bit  3 -> CCK 11 Mbps rate is allowed
                 *     bit  4 -> OFDM BPSK modulation, 1/2 coding rate is allowed
                 *     bit  5 -> OFDM BPSK modulation, 3/4 coding rate is allowed
                 *     bit  6 -> OFDM QPSK modulation, 1/2 coding rate is allowed
                 *     bit  7 -> OFDM QPSK modulation, 3/4 coding rate is allowed
                 *     bit  8 -> OFDM 16-QAM modulation, 1/2 coding rate is allowed
                 *     bit  9 -> OFDM 16-QAM modulation, 3/4 coding rate is allowed
                 *     bit 10 -> OFDM 64-QAM modulation, 2/3 coding rate is allowed
                 *     bit 11 -> OFDM 64-QAM modulation, 3/4 coding rate is allowed
                 *
                 * The MCS index specification needs to be compatible with the
                 * bandwidth mask specification.  For example, a MCS index == 9
                 * specification is inconsistent with a preamble type == VHT,
                 * Nss == 1, and channel bandwidth == 20 MHz.
                 *
                 * Furthermore, the host has only a limited ability to specify to
                 * the target to select from HT + legacy rates, or VHT + legacy rates,
                 * since this mcs_mask can specify either HT/VHT rates or legacy rates.
                 * If no bits are set, target can choose what MCS type to use.
                 */
                 mcs_mask: 12,

                /* nss_mask -
                 * Specify which numbers of spatial streams (MIMO factor) are permitted.
                 * Each bit in this mask corresponds to a Nss value:
                 *     bit 0: if set, Nss = 1 (non-MIMO) is permitted
                 *     bit 1: if set, Nss = 2 (2x2 MIMO) is permitted
                 *     bit 2: if set, Nss = 3 (3x3 MIMO) is permitted
                 *     bit 3: if set, Nss = 4 (4x4 MIMO) is permitted
                 *     bit 4: if set, Nss = 5 (5x5 MIMO) is permitted
                 *     bit 5: if set, Nss = 6 (6x6 MIMO) is permitted
                 *     bit 6: if set, Nss = 7 (7x7 MIMO) is permitted
                 *     bit 7: if set, Nss = 8 (8x8 MIMO) is permitted
                 * The values in the Nss mask must be suitable for the recipient, e.g.
                 * a value of 0x4 (Nss = 3) cannot be specified for a tx frame to a
                 * recipient which only supports 2x2 MIMO.
                 * If no bits are set, target can choose what NSS type to use.
                 */
                 nss_mask: 8,

                /* retry_limit -
                 * Specify the maximum number of transmissions, including the
                 * initial transmission, to attempt before giving up if no ack
                 * is received.
                 * If the tx rate is specified, then all retries shall use the
                 * same rate as the initial transmission.
                 * If no tx rate is specified, the target can choose whether to
                 * retain the original rate during the retransmissions, or to
                 * fall back to a more robust rate.
                 */
                 retry_limit: 4;

       };
       A_UINT32 tx_param_dword0;
    };

    union {
        struct {
            /* DWORD 1: tx chain mask, preamble_type, tx BW */
            A_UINT32
                /* chain_mask - specify which chains to transmit from
                 * If not set, target will choose what chain_mask to use.
                 */
                 chain_mask: 8,

                /* The bits in this mask correspond to the values as below
                 *     bit  0 -> 5MHz
                 *     bit  1 -> 10MHz
                 *     bit  2 -> 20MHz
                 *     bit  3 -> 40MHz
                 *     bit  4 -> 80MHz
                 *     bit  5 -> 160MHz
                 *     bit  6 -> 80_80MHz
                 * If no bits are set, target can choose what BW to use.
                 */
                 bw_mask: 7,

                /* preamble_type_mask -
                 * Specify which preamble types (CCK, OFDM, HT, VHT) the target
                 * may choose from for transmitting this frame.
                 * Each bit in this mask corresponds to a preamble_type value:
                 *     bit 0: if set, OFDM
                 *     bit 1: if set, CCK
                 *     bit 2: if set, HT
                 *     bit 3: if set, VHT
                 *     bit 4: if set, HE
                 * If no bits are set, target can choose what preamble type to use.
                 */
                 preamble_type: 5,

                /* Data:1 Mgmt:0
                 */
                 frame_type: 1,

                 reserved1_31_21: 11;
       };
       A_UINT32 tx_param_dword1;
    };
} wmi_tx_send_params;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_mgmt_tx_send_cmd_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 desc_id;  /* echoed in tx_compl_event */
    A_UINT32 chanfreq; /* MHz units */
    /* WMI_MGMT_TX_SEND_CMDID is used for both pass by value and
     * pass by reference WMI management frames.
     *
     * a) If the command is for pass by reference,
     *    paddr_lo and padd_hi will hold the address of remote/host buffer
     * b) If the command is for pass by value,
     *    paddr_lo and paddr_hi will be NULL.
     */
    A_UINT32 paddr_lo;
    A_UINT32 paddr_hi;
    A_UINT32 frame_len;
    A_UINT32 buf_len;  /** Buffer length in bytes */
    /*
     * The frame which will have tx_params_valid set will be always be RAW
     * frame, as it will be tx'ed on non-pause tid
     */
    A_UINT32 tx_params_valid;
/* This TLV is followed by array of bytes: First 64 bytes of management frame
 *   A_UINT8 bufp[];
 */
/* This TLV is followed by wmi_tx_send_params
 * wmi_tx_send_params tx_send_params;
 */
} wmi_mgmt_tx_send_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_offchan_data_tx_send_cmd_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 desc_id;  /* echoed in tx_compl_event */
    A_UINT32 chanfreq; /* MHz units */
    A_UINT32 paddr_lo;
    A_UINT32 paddr_hi;
    A_UINT32 frame_len;
    A_UINT32 buf_len;  /** Buffer length in bytes */
    /* The frame which will have tx_params_valid set will be always be RAW
     * frame, as it will be tx'ed on non-pause tid
     */
    A_UINT32 tx_params_valid;

/* This TLV is followed by array of bytes: First 64 bytes of frame
 *   A_UINT8 bufp[];
 */
/* This TLV is followed by wmi_tx_send_params
 * wmi_tx_send_params tx_send_params;
 */
} wmi_offchan_data_tx_send_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_echo_event_fixed_param */
    A_UINT32 value;
} wmi_echo_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_echo_cmd_fixed_param */
    A_UINT32 value;
}wmi_echo_cmd_fixed_param;


typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_regdomain_cmd_fixed_param */

    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;

    /** reg domain code */
    A_UINT32 reg_domain;
    A_UINT32 reg_domain_2G;
    A_UINT32 reg_domain_5G;
    A_UINT32 conformance_test_limit_2G;
    A_UINT32 conformance_test_limit_5G;
    A_UINT32 dfs_domain;
} wmi_pdev_set_regdomain_cmd_fixed_param;

typedef struct {
    /** TRUE for scan start and flase for scan end */
    A_UINT32 scan_start;
} wmi_pdev_scan_cmd;

/* WMI support for setting ratemask in target */

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_config_ratemask_fixed_param */
    A_UINT32 vdev_id;
    /* 0 - cck/ofdm
     * 1 - HT
     * 2 - VHT
     * 3 - HE
     */
    A_UINT32 type;

    A_UINT32 mask_lower32;
    A_UINT32 mask_higher32;
    A_UINT32 mask_lower32_2;
    A_UINT32 mask_higher32_2;
} wmi_vdev_config_ratemask_cmd_fixed_param;

/* nrp action - Filter Neighbor Rx Packets  - add/remove filter */
enum {
    WMI_FILTER_NRP_ACTION_ADD        = 0x1,
    WMI_FILTER_NRP_ACTION_REMOVE     = 0x2,
    WMI_FILTER_NRP_ACTION_GET_LIST   = 0x3,
}; /* nrp - Neighbor Rx Packets */

/* nrp type - Filter Neighbor Rx Packets  - ap/client addr */
enum {
    WMI_FILTER_NRP_TYPE_AP_BSSID     = 0x1,
    WMI_FILTER_NRP_TYPE_STA_MACADDR  = 0x2,
};

/* nrp flag - Filter Neighbor Rx Packets
 * (capture flag, 2 & 3 not initially supported)
 */
enum {
    WMI_FILTER_NRP_CAPTURE_ONLY_RX_PACKETS      = 0x1,
    WMI_FILTER_NRP_CAPTURE_ONLY_TX_PACKETS      = 0x2,
    WMI_FILTER_NRP_CAPTURE_BOTH_TXRX_PACKETS    = 0x3,
};

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_filter_nrp_config_cmd_fixed_param */
    A_UINT32 vdev_id;
    /* AP Bssid or Client Mac-addr */
    wmi_mac_addr addr;
    /* Add/Remove NRF Filter */
    A_UINT32 action; /* WMI_FILTER_NRP_ACTION enum */
    /* client/ap filter */
    A_UINT32 type; /* WMI_FILTER_NRP_TYPE enum */
    /* optional - tx/rx capture */
    A_UINT32 flag; /* WMI_FILTER_NRP_CAPTURE enum */
    /* BSSID index - index of the BSSID register */
    A_UINT32 bssid_idx;
} wmi_vdev_filter_nrp_config_cmd_fixed_param; /* Filter for Neighbor Rx Packets */


/*Command to set/unset chip in quiet mode*/
typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_quiet_cmd_fixed_param */
    A_UINT32 pdev_id; /** pdev_id for identifying the MAC, See macros starting with WMI_PDEV_ID_ for values. */
    A_UINT32 period;		/*period in TUs*/
    A_UINT32 duration;		/*duration in TUs*/
    A_UINT32 next_start;	/*offset in TUs*/
    A_UINT32 enabled;		/*enable/disable*/
} wmi_pdev_set_quiet_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_set_quiet_cmd_fixed_param */
    A_UINT32 vdev_id;    /* Virtual interface ID */
    A_UINT32 period;     /* period in TUs */
    A_UINT32 duration;   /* duration in TUs */
    A_UINT32 next_start; /* offset in TUs */
    A_UINT32 enabled;    /* enable/disable */
} wmi_vdev_set_quiet_cmd_fixed_param;

/*
 * START_STOP flag value: 1 - Start, 0 - Stop
 */
#define WMI_OFFLOAD_QUIET_FLAG_START_STOP              0x00000001
/*
 * ONE_SHOT flag value: 1 - One shot, 0 - Repeat
 * This flag is only relevant if the START_STOP flag == 1 (start).
 */
#define WMI_OFFLOAD_QUIET_FLAG_ONE_SHOT                0x00000002
/*
 * Enable/Disable sending Quiet IE info in SWBA event from the target
 * 0 - Don't include Quiet IE in WMI SWBA Event
 * 1 - Include Quiet IE in WMI SWBA Event
 */
#define WMI_OFFLOAD_QUIET_FLAG_INFO_IN_SWBA_START_STOP 0x00000004

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_bcn_offload_quiet_config_cmd_fixed_param */
    A_UINT32 vdev_id;    /* Virtual interface ID */
    A_UINT32 period;     /* period in TUs */
    A_UINT32 duration;   /* duration in TUs */
    A_UINT32 next_start; /* offset in TUs from beacon */
    A_UINT32 flags;      /* STOP or START (and single vs. repeated) Quiet IE
                          * See WMI_OFFLOAD_QUIET_FLAG_xxx defs.
                          */
} wmi_vdev_bcn_offload_quiet_config_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;   /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_set_custom_aggr_size_cmd_fixed_param */
    A_UINT32 vdev_id;      /* vdev id indicating to which the vdev custom aggregation size will be applied. */
    /* Size for tx aggregation for the vdev mentioned in vdev id
     * (max MPDUs per A-MPDU or max MSDUs per A-MSDU based on aggr_type field)
     */
    A_UINT32 tx_aggr_size;

    A_UINT32 rx_aggr_size; /* Size for rx aggregation (block ack window size limit) for the vdev mentioned in vdev id */

    /*
     * To set TX aggregation size limits per VDEV per AC
     * bits 1:0 (ac):
     *     Access Category (0x0=BE, 0x1=BK, 0x2=VI, 0x3=VO)
     *     If tx_ac_enable bit is not set, tx_aggr_size is applied
     *     for all Access Categories
     * bit 2 (aggr_type):            TX Aggregation Type (0=A-MPDU, 1=A-MSDU)
     * bit 3 (tx_aggr_size_disable): If set tx_aggr_size is invalid
     * bit 4 (rx_aggr_size_disable): If set rx_aggr_size is invalid
     * bit 5 (tx_ac_enable):         If set, above ac bitmap is valid.
     * bits 31:6:                    Reserved bits. should be set to zero.
     */
    A_UINT32 enable_bitmap;
} wmi_vdev_set_custom_aggr_size_cmd_fixed_param;

typedef enum {
    WMI_VDEV_CUSTOM_AGGR_TYPE_AMPDU = 0,
    WMI_VDEV_CUSTOM_AGGR_TYPE_AMSDU = 1,
    WMI_VDEV_CUSTOM_AGGR_TYPE_MAX,
} wmi_vdev_custom_aggr_type_t;

#define WMI_VDEV_CUSTOM_AGGR_AC_BITPOS           0
#define WMI_VDEV_CUSTOM_AGGR_AC_NUM_BITS         2
#define WMI_VDEV_CUSTOM_AGGR_TYPE_BITPOS         2
#define WMI_VDEV_CUSTOM_AGGR_TYPE_NUM_BITS       1
#define WMI_VDEV_CUSTOM_TX_AGGR_SZ_DIS_BITPOS    3
#define WMI_VDEV_CUSTOM_TX_AGGR_SZ_DIS_NUM_BITS  1
#define WMI_VDEV_CUSTOM_RX_AGGR_SZ_DIS_BITPOS    4
#define WMI_VDEV_CUSTOM_RX_AGGR_SZ_DIS_NUM_BITS  1
#define WMI_VDEV_CUSTOM_TX_AC_EN_BITPOS          5
#define WMI_VDEV_CUSTOM_TX_AC_EN_NUM_BITS        1

#define WMI_VDEV_CUSTOM_AGGR_AC_SET(param, value) \
    WMI_SET_BITS(param, WMI_VDEV_CUSTOM_AGGR_AC_BITPOS, \
        WMI_VDEV_CUSTOM_AGGR_AC_NUM_BITS, value)
#define WMI_VDEV_CUSTOM_AGGR_AC_GET(param)         \
    WMI_GET_BITS(param, WMI_VDEV_CUSTOM_AGGR_AC_BITPOS, \
        WMI_VDEV_CUSTOM_AGGR_AC_NUM_BITS)

#define WMI_VDEV_CUSTOM_AGGR_TYPE_SET(param, value) \
    WMI_SET_BITS(param, WMI_VDEV_CUSTOM_AGGR_TYPE_BITPOS, \
        WMI_VDEV_CUSTOM_AGGR_TYPE_NUM_BITS, value)
#define WMI_VDEV_CUSTOM_AGGR_TYPE_GET(param)         \
    WMI_GET_BITS(param, WMI_VDEV_CUSTOM_AGGR_TYPE_BITPOS, \
        WMI_VDEV_CUSTOM_AGGR_TYPE_NUM_BITS)

#define WMI_VDEV_CUSTOM_TX_AGGR_SZ_DIS_SET(param, value) \
    WMI_SET_BITS(param, WMI_VDEV_CUSTOM_TX_AGGR_SZ_DIS_BITPOS, \
        WMI_VDEV_CUSTOM_TX_AGGR_SZ_DIS_NUM_BITS, value)
#define WMI_VDEV_CUSTOM_TX_AGGR_SZ_DIS_GET(param)         \
    WMI_GET_BITS(param, WMI_VDEV_CUSTOM_TX_AGGR_SZ_DIS_BITPOS, \
        WMI_VDEV_CUSTOM_TX_AGGR_SZ_DIS_NUM_BITS)

#define WMI_VDEV_CUSTOM_RX_AGGR_SZ_DIS_SET(param, value) \
    WMI_SET_BITS(param, WMI_VDEV_CUSTOM_RX_AGGR_SZ_DIS_BITPOS, \
        WMI_VDEV_CUSTOM_RX_AGGR_SZ_DIS_NUM_BITS, value)
#define WMI_VDEV_CUSTOM_RX_AGGR_SZ_DIS_GET(param)         \
    WMI_GET_BITS(param, WMI_VDEV_CUSTOM_RX_AGGR_SZ_DIS_BITPOS, \
        WMI_VDEV_CUSTOM_RX_AGGR_SZ_DIS_NUM_BITS)

#define WMI_VDEV_CUSTOM_TX_AC_EN_SET(param, value) \
    WMI_SET_BITS(param, WMI_VDEV_CUSTOM_TX_AC_EN_BITPOS, \
        WMI_VDEV_CUSTOM_TX_AC_EN_NUM_BITS, value)
#define WMI_VDEV_CUSTOM_TX_AC_EN_GET(param)         \
    WMI_GET_BITS(param, WMI_VDEV_CUSTOM_TX_AC_EN_BITPOS, \
        WMI_VDEV_CUSTOM_TX_AC_EN_NUM_BITS)

typedef enum {
    WMI_VDEV_CUSTOM_SW_RETRY_TYPE_NONAGGR = 0,
    WMI_VDEV_CUSTOM_SW_RETRY_TYPE_AGGR = 1,
    WMI_VDEV_CUSTOM_SW_RETRY_TYPE_MAX,
} wmi_vdev_custom_sw_retry_type_t;

typedef struct {
    A_UINT32 tlv_header;   /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_set_custom_sw_retry_th_cmd_fixed_param */
    A_UINT32 vdev_id;      /* vdev id indicating to which the vdev custom software retries will be applied. */
    A_UINT32 ac_type;      /* access category (VI, VO, BE, BK) enum wmi_traffic_ac */
    A_UINT32 sw_retry_type; /* 0 = non-aggr retry, 1 = aggr retry (wmi_vdev_custom_sw_retry_type_t enum) */
    A_UINT32 sw_retry_th;   /* max retry count per AC base on ac_type for the vdev mentioned in vdev id*/
} wmi_vdev_set_custom_sw_retry_th_cmd_fixed_param;

typedef struct {
    /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_chainmask_config_cmd_fixed_param */
    A_UINT32 tlv_header;
    /* vdev id indicating to which the vdev, this chainmask configuration will be applied. */
    A_UINT32 vdev_id;
    /* number of chains to use for transmissions in 2.4 GHz band */
    A_UINT32 num_tx_chains_2g;
    /* number of chains to use for reception in 2.4 GHz band */
    A_UINT32 num_rx_chains_2g;
    /* nss to use for transmissions in 2.4 GHz band */
    A_UINT32 tx_nss_2g;
    /* nss to use for reception in 2.4 GHz band */
    A_UINT32 rx_nss_2g;
    /* number of chains to use for 11b transmissions. Valid only in 2.4 GHz */
    A_UINT32 num_tx_chains_b;
    /* number of chains to use for 11g transmissions. Valid only in 2.4 GHz */
    A_UINT32 num_tx_chains_g;
    /* number of chains to use for transmissions in 5 GHz band */
    A_UINT32 num_tx_chains_5g;
    /* number of chains to use for reception in 5 GHz band */
    A_UINT32 num_rx_chains_5g;
    /* nss to use for transmissions in 5 GHz band */
    A_UINT32 tx_nss_5g;
    /* nss to use for reception in 5 GHz band */
    A_UINT32 rx_nss_5g;
    /* number of chains to use for 11a transmissions. Valid only in 5 GHz */
    A_UINT32 num_tx_chains_a;
    /* If non-zero then use only one chain for TX when connection tx_nss is 1 in 2.4 GHz */
    A_UINT32 disable_tx_mrc_2g;
    /* If non-zero then use only one chain for RX when connection rx_nss is 1 in 2.4 GHz */
    A_UINT32 disable_rx_mrc_2g;
    /* If non-zero then use only one chain for TX when connection tx_nss is 1 in 5 GHz */
    A_UINT32 disable_tx_mrc_5g;
    /* If non-zero then use only one chain for RX when connection rx_nss is 1 in 5 GHz */
    A_UINT32 disable_rx_mrc_5g;
} wmi_vdev_chainmask_config_cmd_fixed_param;

/*
 * Command to enable/disable Green AP Power Save.
 * This helps conserve power during AP operation. When the AP has no
 * stations associated with it, the host can enable Green AP Power Save
 * to request the firmware to shut down all but one transmit and receive
 * chains.
 */
typedef struct {
    A_UINT32 tlv_header;         /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_green_ap_ps_enable_cmd_fixed_param */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
    A_UINT32 enable;              /*1:enable, 0:disable*/
} wmi_pdev_green_ap_ps_enable_cmd_fixed_param;


#define MAX_HT_IE_LEN 32
/* DEPRECATED */
typedef struct {
    A_UINT32 tlv_header;   /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_ht_ie_cmd_fixed_param */
    A_UINT32 reserved0;    /** placeholder for pdev_id of future multiple MAC products. Init. to 0. */
    A_UINT32 ie_len;       /*length of the ht ie in the TLV ie_data[] */
    A_UINT32 tx_streams; /* Tx streams supported for this HT IE */
    A_UINT32 rx_streams; /* Rx streams supported for this HT IE */
    /** The TLV for the HT IE follows:
     *       A_UINT32 ie_data[];
     */
} wmi_pdev_set_ht_ie_cmd_fixed_param;

#define MAX_VHT_IE_LEN 32
/* DEPRECATED */
typedef struct {
    A_UINT32 tlv_header;   /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_vht_ie_cmd_fixed_param */
    A_UINT32 reserved0;    /** placeholder for pdev_id of future multiple MAC products. Init. to 0. */
    A_UINT32 ie_len;          /*length of the vht ie in the TLV ie_data[] */
    A_UINT32 tx_streams; /* Tx streams supported for this HT IE */
    A_UINT32 rx_streams; /* Rx streams supported for this HT IE */
    /** The TLV for the VHT IE follows:
     *       A_UINT32 ie_data[];
     */
} wmi_pdev_set_vht_ie_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;   /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_base_macaddr_cmd_fixed_param */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
    wmi_mac_addr base_macaddr;
} wmi_pdev_set_base_macaddr_cmd_fixed_param;

/*
 * For now, the spectral configuration is global rather than
 * per-vdev.  The vdev is a placeholder and will be ignored
 * by the firmware.
 */
typedef struct {
        A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_spectral_configure_cmd_fixed_param */
        A_UINT32    vdev_id;
        A_UINT32    spectral_scan_count;
        A_UINT32    spectral_scan_period;
        A_UINT32    spectral_scan_priority;
        A_UINT32    spectral_scan_fft_size;
        A_UINT32    spectral_scan_gc_ena;
        A_UINT32    spectral_scan_restart_ena;
        A_UINT32    spectral_scan_noise_floor_ref;
        A_UINT32    spectral_scan_init_delay;
        A_UINT32    spectral_scan_nb_tone_thr;
        A_UINT32    spectral_scan_str_bin_thr;
        A_UINT32    spectral_scan_wb_rpt_mode;
        A_UINT32    spectral_scan_rssi_rpt_mode;
        A_UINT32    spectral_scan_rssi_thr;
        A_UINT32    spectral_scan_pwr_format;
        A_UINT32    spectral_scan_rpt_mode;
        A_UINT32    spectral_scan_bin_scale;
        A_UINT32    spectral_scan_dBm_adj;
        A_UINT32    spectral_scan_chn_mask;
} wmi_vdev_spectral_configure_cmd_fixed_param;

/*
 * Enabling, disabling and triggering the spectral scan
 * is a per-vdev operation.  That is, it will set channel
 * flags per vdev rather than globally; so concurrent scan/run
 * and multiple STA (eg p2p, tdls, multi-band STA) is possible.
 */
typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_spectral_enable_cmd_fixed_param */
    A_UINT32    vdev_id;
    /* 0 - ignore; 1 - trigger, 2 - clear trigger */
    A_UINT32    trigger_cmd;
    /* 0 - ignore; 1 - enable, 2 - disable */
    A_UINT32    enable_cmd;
} wmi_vdev_spectral_enable_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_get_tx_power_cmd_fixed_param */
    A_UINT32 vdev_id;
} wmi_vdev_get_tx_power_cmd_fixed_param;

#define WMI_BEACON_CTRL_TX_DISABLE  0
#define WMI_BEACON_CTRL_TX_ENABLE   1
#define WMI_BEACON_CTRL_SWBA_EVENT_DISABLE  2
#define WMI_BEACON_CTRL_SWBA_EVENT_ENABLE   3

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_bcn_offload_ctrl_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    A_UINT32 bcn_ctrl_op; /* fw default 1 (see WMI_BEACON_CTRL defs) */
} wmi_bcn_offload_ctrl_cmd_fixed_param;

/** common structure used for wmi_vedv_get_tx_power_event */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_tx_power_event_fixed_param */
    A_UINT32 tx_power; /** units: 0.5 dBm, per-chain tx power */
    A_UINT32 vdev_id; /** unique id identifying the VDEV, generated by the caller */
} wmi_vdev_get_tx_power_event_fixed_param;

typedef enum {
    /** Limit the offchannel duration */
    WMI_VDEV_LIMIT_OFFCHAN_ENABLE   = 0x1,
    /** Skip DFS channels from Scan channel list.
     *  valid for both host scans and FW scans */
    WMI_VDEV_LIMIT_OFFCHAN_SKIP_DFS = 0x2,
} wmi_vdev_limit_offchan_flags;

typedef struct {
    A_UINT32 tlv_header; /* WMITLV_TAG_STRUC_wmi_vdev_limit_offchan_cmd_fixed_param */
    /** Limit the duration of offchannel events requested by the vdev corresponding to the specified vdev_id */
    A_UINT32 vdev_id;
    /** see enum wmi_vdev_limit_offchan_flags */
    A_UINT32 flags;
    /** max offchannel time allowed in msec when WMI_VDEV_LIMIT_OFFCHAN_ENABLE flag is set */
    A_UINT32 max_offchan_time;
    /** rest time in msec on the BSS channel */
    A_UINT32 rest_time;
} wmi_vdev_limit_offchan_cmd_fixed_param;

/* information sub element id for QSBW, expected value is 0x02 */
#define WMI_CSA_EVENT_QSBW_ISE_ID_MASK    0x000000FF
/* length of QSBW ISE data, expected value is 0x02 */
#define WMI_CSA_EVENT_QSBW_ISE_LEN_MASK   0x0000FF00
/* capabilities, 0x01 for 5MHz, 0x02 for 10MHz, 0x01|0x2 for both
 * (see WMI_CSA_EVENT_QSBW_ISE bitmask defs)
 */
#define WMI_CSA_EVENT_QSBW_ISE_CAP_MASK   0x00FF0000
/* notification from AP, 0x01 for 5MHz, 0x02 for 10MHz
 * (see WMI_CSA_EVENT_QSBW_ISE bitmask defs)
 */
#define WMI_CSA_EVENT_QSBW_ISE_NOTIF_MASK 0xFF000000

#define WMI_CSA_EVENT_QSBW_ISE_ID 0x02
#define WMI_CSA_EVENT_QSBW_ISE_LEN 0x02

#define WMI_CSA_EVENT_QSBW_ISE_5M_BITMASK  0x01
#define WMI_CSA_EVENT_QSBW_ISE_10M_BITMASK  0x02

#define WMI_CSA_EVENT_QSBW_ISE_CAP_5M(qsbw_ise) \
        (((qsbw_ise) >> 16) & WMI_CSA_EVENT_QSBW_ISE_5M_BITMASK)
#define WMI_CSA_EVENT_QSBW_ISE_CAP_10M(qsbw_ise) \
        (((qsbw_ise) >> 16) & WMI_CSA_EVENT_QSBW_ISE_10M_BITMASK)
#define WMI_CSA_EVENT_QSBW_ISE_NOTIF_5M(qsbw_ise) \
        (((qsbw_ise) >> 24) & WMI_CSA_EVENT_QSBW_ISE_5M_BITMASK)
#define WMI_CSA_EVENT_QSBW_ISE_NOTIF_10M(qsbw_ise) \
        (((qsbw_ise) >> 24) & WMI_CSA_EVENT_QSBW_ISE_10M_BITMASK)

#define WMI_CSA_EVENT_QSBW_ISE_EXTRACT_ID(qsbw_ise) \
        ((qsbw_ise) & WMI_CSA_EVENT_QSBW_ISE_ID_MASK)
#define WMI_CSA_EVENT_QSBW_ISE_EXTRACT_LEN(qsbw_ise)  \
        (((qsbw_ise) & WMI_CSA_EVENT_QSBW_ISE_LEN_MASK) >> 8)
#define WMI_CSA_EVENT_QSBW_ISE_EXTRACT_CAP(qsbw_ise)  \
        (((qsbw_ise) & WMI_CSA_EVENT_QSBW_ISE_CAP_MASK) >> 16)
#define WMI_CSA_EVENT_QSBW_ISE_EXTRACT_NOTIF(qsbw_ise)  \
        (((qsbw_ise) & WMI_CSA_EVENT_QSBW_ISE_NOTIF_MASK) >> 24)

typedef enum {
WMI_CSA_IE_PRESENT = 0x00000001,
WMI_XCSA_IE_PRESENT = 0x00000002,
WMI_WBW_IE_PRESENT = 0x00000004,
WMI_CSWARP_IE_PRESENT = 0x00000008,
WMI_QSBW_ISE_PRESENT  = 0x00000010,
}WMI_CSA_EVENT_IES_PRESENT_FLAG;

/* wmi CSA receive event from beacon frame */
typedef struct{
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_csa_event_fixed_param */
    A_UINT32 i_fc_dur;
//  Bit 0-15: FC
//  Bit 16-31: DUR
    wmi_mac_addr i_addr1;
    wmi_mac_addr i_addr2;
    /* NOTE: size of array of csa_ie[], xcsa_ie[], and wb_ie[] cannot be
         * changed in the future without breaking WMI compatibility */
    A_UINT32 csa_ie[2];
    A_UINT32 xcsa_ie[2];
    A_UINT32 wb_ie[2];
    A_UINT32 cswarp_ie;
    A_UINT32 ies_present_flag; //WMI_CSA_EVENT_IES_PRESENT_FLAG
    A_UINT32 qsbw_ise;
}wmi_csa_event_fixed_param;

typedef enum {
    WAL_PEER_MCAST2UCAST_DISABLED    = 0,
    WAL_PEER_MCAST2UCAST_DROP_EMPTY  = 1, //Drop the frames if match is not found
    WAL_PEER_MCAST2UCAST_MCAST_EMPTY = 2, // Send as mcast if match is not found
} WMI_PEER_MCAST2UCAST_MODE;

typedef enum {
    PKT_PWR_SAVE_NAP_ENABLE     = 0x00000001,
    PKT_PWR_SAVE_LS_ENABLE      = 0x00000002,
    PKT_PWR_SAVE_DS_ENABLE      = 0x00000004,

    PKT_PWR_SAVE_BTCOEX_ENABLE  = 0x00000008,

    PKT_PWR_SAVE_FSM_ENABLE     = 0x80000000,
} WMI_PDEV_PKT_PWR_SAVE_LEVEL;

/** MACROs to get user setting for enabling/disabling Secondary Rate Feature set
 * Bit-0    : Enable/Disable Control for "PPDU Secondary Retry Support"
 * Bit-1    : Enable/Disable Control for "RTS Black/White-listing Support"
 * Bit-2    : Enable/Disable Control for "Higher MCS retry restriction on XRETRY failures"
 * Bit 3-5  : "Xretry threshold" to use
 * Bit 6~31 : reserved for future use.
 */
#define WMI_PDEV_PARAM_SECONDARY_RATE_ENABLE_BIT_S      0
#define WMI_PDEV_PARAM_SECONDARY_RATE_ENABLE_BIT        0x00000001
#define WMI_PDEV_PARAM_RTS_BL_WL_ENABLE_BIT_S           1
#define WMI_PDEV_PARAM_RTS_BL_WL_ENABLE_BIT             0x00000002
#define WMI_PDEV_PARAM_HIGHER_MCS_XRETRY_RESTRICTION_S  2
#define WMI_PDEV_PARAM_HIGHER_MCS_XRETRY_RESTRICTION    0x00000004

#define WMI_PDEV_PARAM_XRETRY_THRESHOLD_S               3
#define WMI_PDEV_PARAM_XRETRY_THRESHOLD                 0x00000038

#define WMI_PDEV_PARAM_IS_SECONDARY_RATE_ENABLED(word32)            WMI_F_MS(word32, WMI_PDEV_PARAM_SECONDARY_RATE_ENABLE_BIT)
#define WMI_PDEV_PARAM_IS_RTS_BL_WL_ENABLED(word32)                 WMI_F_MS(word32, WMI_PDEV_PARAM_RTS_BL_WL_ENABLE_BIT)
#define WMI_PDEV_PARAM_IS_HIGHER_MCS_XRETRY_RESTRICTION_SET(word32) WMI_F_MS(word32, WMI_PDEV_PARAM_HIGHER_MCS_XRETRY_RESTRICTION)
#define WMI_PDEV_PARAM_GET_XRETRY_THRESHOLD(word32)                 WMI_F_MS(word32, WMI_PDEV_PARAM_XRETRY_THRESHOLD)

typedef enum {
    /** TX chain mask */
    WMI_PDEV_PARAM_TX_CHAIN_MASK = 0x1,
    /** RX chain mask */
    WMI_PDEV_PARAM_RX_CHAIN_MASK,                     /* 0x2 */
    /** TX power limit for 2G Radio */
    WMI_PDEV_PARAM_TXPOWER_LIMIT2G,                   /* 0x3 */
    /** TX power limit for 5G Radio */
    WMI_PDEV_PARAM_TXPOWER_LIMIT5G,                   /* 0x4 */
    /** TX power scale */
    WMI_PDEV_PARAM_TXPOWER_SCALE,                     /* 0x5 */
    /** Beacon generation mode . 0: host, 1: target   */
    WMI_PDEV_PARAM_BEACON_GEN_MODE,                   /* 0x6 */
    /** Beacon generation mode . 0: staggered 1: bursted   */
    WMI_PDEV_PARAM_BEACON_TX_MODE,                    /* 0x7 */
    /** Resource manager off chan mode .
     * 0: turn off off chan mode. 1: turn on offchan mode
     */
    WMI_PDEV_PARAM_RESMGR_OFFCHAN_MODE,               /* 0x8 */
    /** Protection mode  0: no protection 1:use CTS-to-self 2: use RTS/CTS */
    WMI_PDEV_PARAM_PROTECTION_MODE,                   /* 0x9 */
    /** Dynamic bandwidth 0: disable 1: enable */
    WMI_PDEV_PARAM_DYNAMIC_BW,                        /* 0xa */
    /** Non aggregrate/ 11g sw retry threshold.0-disable */
    WMI_PDEV_PARAM_NON_AGG_SW_RETRY_TH,               /* 0xb */
    /** aggregrate sw retry threshold. 0-disable*/
    WMI_PDEV_PARAM_AGG_SW_RETRY_TH,                   /* 0xc */
    /** Station kickout threshold (non of consecutive failures).0-disable */
    WMI_PDEV_PARAM_STA_KICKOUT_TH,                    /* 0xd */
    /** Aggerate size scaling configuration per AC */
    WMI_PDEV_PARAM_AC_AGGRSIZE_SCALING,               /* 0xe */
    /** LTR enable */
    WMI_PDEV_PARAM_LTR_ENABLE,                        /* 0xf */
    /** LTR latency for BE, in us */
    WMI_PDEV_PARAM_LTR_AC_LATENCY_BE,                 /* 0x10 */
    /** LTR latency for BK, in us */
    WMI_PDEV_PARAM_LTR_AC_LATENCY_BK,                 /* 0x11 */
    /** LTR latency for VI, in us */
    WMI_PDEV_PARAM_LTR_AC_LATENCY_VI,                 /* 0x12 */
    /** LTR latency for VO, in us  */
    WMI_PDEV_PARAM_LTR_AC_LATENCY_VO,                 /* 0x13 */
    /** LTR AC latency timeout, in ms */
    WMI_PDEV_PARAM_LTR_AC_LATENCY_TIMEOUT,            /* 0x14 */
    /** LTR platform latency override, in us */
    WMI_PDEV_PARAM_LTR_SLEEP_OVERRIDE,                /* 0x15 */
    /** LTR-M override, in us */
    WMI_PDEV_PARAM_LTR_RX_OVERRIDE,                   /* 0x16 */
    /** Tx activity timeout for LTR, in us */
    WMI_PDEV_PARAM_LTR_TX_ACTIVITY_TIMEOUT,           /* 0x17 */
    /** L1SS state machine enable */
    WMI_PDEV_PARAM_L1SS_ENABLE,                       /* 0x18 */
    /** Deep sleep state machine enable */
    WMI_PDEV_PARAM_DSLEEP_ENABLE,                     /* 0x19 */
    /** RX buffering flush enable */
    WMI_PDEV_PARAM_PCIELP_TXBUF_FLUSH,                /* 0x1a */
    /** RX buffering matermark */
    WMI_PDEV_PARAM_PCIELP_TXBUF_WATERMARK,            /* 0x1b */
    /** RX buffering timeout enable */
    WMI_PDEV_PARAM_PCIELP_TXBUF_TMO_EN,               /* 0x1c */
    /** RX buffering timeout value */
    WMI_PDEV_PARAM_PCIELP_TXBUF_TMO_VALUE,            /* 0x1d */
    /** pdev level stats update period in ms */
    WMI_PDEV_PARAM_PDEV_STATS_UPDATE_PERIOD,          /* 0x1e */
    /** vdev level stats update period in ms */
    WMI_PDEV_PARAM_VDEV_STATS_UPDATE_PERIOD,          /* 0x1f */
    /** peer level stats update period in ms */
    WMI_PDEV_PARAM_PEER_STATS_UPDATE_PERIOD,          /* 0x20 */
    /** beacon filter status update period */
    WMI_PDEV_PARAM_BCNFLT_STATS_UPDATE_PERIOD,        /* 0x21 */
    /** QOS Mgmt frame protection MFP/PMF 0: disable, 1: enable */
    WMI_PDEV_PARAM_PMF_QOS,                           /* 0x22 */
    /** Access category on which ARP frames are sent */
    WMI_PDEV_PARAM_ARP_AC_OVERRIDE,                   /* 0x23 */
    /** DCS configuration */
    WMI_PDEV_PARAM_DCS,                               /* 0x24 */
    /** Enable/Disable ANI on target */
    WMI_PDEV_PARAM_ANI_ENABLE,                        /* 0x25 */
    /** configure the ANI polling period */
    WMI_PDEV_PARAM_ANI_POLL_PERIOD,                   /* 0x26 */
    /** configure the ANI listening period */
    WMI_PDEV_PARAM_ANI_LISTEN_PERIOD,                 /* 0x27 */
    /** configure OFDM immunity level */
    WMI_PDEV_PARAM_ANI_OFDM_LEVEL,                    /* 0x28 */
    /** configure CCK immunity level */
    WMI_PDEV_PARAM_ANI_CCK_LEVEL,                     /* 0x29 */
    /** Enable/Disable CDD for 1x1 STAs in rate control module */
    WMI_PDEV_PARAM_DYNTXCHAIN,                        /* 0x2a */
    /** Enable/Disable proxy STA */
    WMI_PDEV_PARAM_PROXY_STA,                         /* 0x2b */
    /** Enable/Disable low power state when all VDEVs are inactive/idle. */
    WMI_PDEV_PARAM_IDLE_PS_CONFIG,                    /* 0x2c */
    /** Enable/Disable power gating sleep */
    WMI_PDEV_PARAM_POWER_GATING_SLEEP,                /* 0x2d */
    /** Enable/Disable Rfkill */
    WMI_PDEV_PARAM_RFKILL_ENABLE,                     /* 0x2e */
    /** Set Bursting DUR */
    WMI_PDEV_PARAM_BURST_DUR,                         /* 0x2f */
    /** Set Bursting ENABLE */
    WMI_PDEV_PARAM_BURST_ENABLE,                      /* 0x30 */
    /** HW rfkill config */
    WMI_PDEV_PARAM_HW_RFKILL_CONFIG,                  /* 0x31 */
    /** Enable radio low power features */
    WMI_PDEV_PARAM_LOW_POWER_RF_ENABLE,               /* 0x32 */
   /** L1SS entry and residency time track */
    WMI_PDEV_PARAM_L1SS_TRACK,                        /* 0x33 */
    /** set hyst at runtime, requirement from SS */
    WMI_PDEV_PARAM_HYST_EN,                           /* 0x34 */
    /** Enable/ Disable POWER COLLAPSE */
    WMI_PDEV_PARAM_POWER_COLLAPSE_ENABLE,             /* 0x35 */
   /** configure LED system state */
    WMI_PDEV_PARAM_LED_SYS_STATE,                     /* 0x36 */
   /** Enable/Disable LED */
    WMI_PDEV_PARAM_LED_ENABLE,                        /* 0x37 */
    /** set DIRECT AUDIO time latency */
    WMI_PDEV_PARAM_AUDIO_OVER_WLAN_LATENCY, /* DEPRECATED */ /* 0x38 */
    /** set DIRECT AUDIO Feature ENABLE */
    WMI_PDEV_PARAM_AUDIO_OVER_WLAN_ENABLE, /* DEPRECATED */  /* 0x39 */
    /** pdev level whal mib stats update enable */
    WMI_PDEV_PARAM_WHAL_MIB_STATS_UPDATE_ENABLE,      /* 0x3a */
    /** ht/vht info based on vdev */
    WMI_PDEV_PARAM_VDEV_RATE_STATS_UPDATE_PERIOD,     /* 0x3b */
    /** Set CTS channel BW for dynamic BW adjustment feature */
    WMI_PDEV_PARAM_CTS_CBW,                           /* 0x3c */
    /** Set GPIO pin info used by WNTS */
    WMI_PDEV_PARAM_WNTS_CONFIG,                       /* 0x3d */
    /** Enable/Disable hardware adaptive early rx feature */
    WMI_PDEV_PARAM_ADAPTIVE_EARLY_RX_ENABLE,          /* 0x3e */
    /** The minimum early rx duration, to ensure early rx duration is non-zero */
    WMI_PDEV_PARAM_ADAPTIVE_EARLY_RX_MIN_SLEEP_SLOP,  /* 0x3f */
    /** Increasing/decreasing step used by hardware */
    WMI_PDEV_PARAM_ADAPTIVE_EARLY_RX_INC_DEC_STEP,    /* 0x40 */
    /** The fixed early rx duration when adaptive early rx is disabled */
    WMI_PDEV_PARAM_EARLY_RX_FIX_SLEEP_SLOP,           /* 0x41 */
    /** Enable/Disable bmiss based adaptive beacon timeout feature */
    WMI_PDEV_PARAM_BMISS_BASED_ADAPTIVE_BTO_ENABLE,   /* 0x42 */
    /** The minimum beacon timeout duration, to ensure beacon timeout duration is non-zero */
    WMI_PDEV_PARAM_BMISS_BTO_MIN_BCN_TIMEOUT,         /* 0x43 */
    /** Increasing/decreasing step used by hardware */
    WMI_PDEV_PARAM_BMISS_BTO_INC_DEC_STEP,            /* 0x44 */
    /** The fixed beacon timeout duration when bmiss based adaptive beacon timeout is disabled */
    WMI_PDEV_PARAM_BTO_FIX_BCN_TIMEOUT,               /* 0x45 */
    /** Enable/Disable Congestion Estimator based adaptive beacon timeout feature */
    WMI_PDEV_PARAM_CE_BASED_ADAPTIVE_BTO_ENABLE,      /* 0x46 */
    /** combo value of ce_id, ce_threshold, ce_time, refer to WMI_CE_BTO_CE_ID_MASK */
    WMI_PDEV_PARAM_CE_BTO_COMBO_CE_VALUE,             /* 0x47 */
    /** 2G TX chain mask */
    WMI_PDEV_PARAM_TX_CHAIN_MASK_2G,                  /* 0x48 */
    /** 2G RX chain mask */
    WMI_PDEV_PARAM_RX_CHAIN_MASK_2G,                  /* 0x49 */
    /** 5G TX chain mask */
    WMI_PDEV_PARAM_TX_CHAIN_MASK_5G,                  /* 0x4a */
    /** 5G RX chain mask */
    WMI_PDEV_PARAM_RX_CHAIN_MASK_5G,                  /* 0x4b */
    /* Set tx chain mask for CCK rates */
    WMI_PDEV_PARAM_TX_CHAIN_MASK_CCK,                 /* 0x4c */
    /* Set tx chain mask for 1SS stream */
    WMI_PDEV_PARAM_TX_CHAIN_MASK_1SS,                 /* 0x4d */
    /* Enable/Disable CTS2Self for P2P GO when Non-P2P Client is connected */
    WMI_PDEV_PARAM_CTS2SELF_FOR_P2P_GO_CONFIG,        /* 0x4e */
    /** TX power backoff in dB: tx power -= param value
     * Host passes values(DB) to Halphy, Halphy reduces the power table by
     * the values. Safety check will happen in Halphy
     */
    WMI_PDEV_PARAM_TXPOWER_DECR_DB,                   /* 0x4f */
    /** enable and disable aggregate burst along with duration */
    WMI_PDEV_PARAM_AGGR_BURST,                        /* 0x50 */
    /** Set the global RX decap mode */
    WMI_PDEV_PARAM_RX_DECAP_MODE,                     /* 0x51 */
    /** Enable/Disable Fast channel reset */
    WMI_PDEV_PARAM_FAST_CHANNEL_RESET,                /* 0x52 */
    /** Default antenna for Smart antenna */
    WMI_PDEV_PARAM_SMART_ANTENNA_DEFAULT_ANTENNA,     /* 0x53 */
    /** Set the user-specified antenna gain */
    WMI_PDEV_PARAM_ANTENNA_GAIN,                      /* 0x54 */
    /** Set the user-specified RX filter */
    WMI_PDEV_PARAM_RX_FILTER,                         /* 0x55 */
    /** configure the user-specified MCAST tid for managed mcast feature
     *  0-15 is the valid range. 0xff will clear the tid setting */
    WMI_PDEV_SET_MCAST_TO_UCAST_TID,                  /* 0x56 */
    /** Enable/Disable Proxy sta mode */
    WMI_PDEV_PARAM_PROXY_STA_MODE,                    /* 0x57 */
    /** configure the mcast2ucast mode for the pdev->peer_mcast.
     *  See WMI_PEER_MCAST2UCAST_MODE for possible values */
    WMI_PDEV_PARAM_SET_MCAST2UCAST_MODE,              /* 0x58 */
    /** Sets the Mcast buffers for cloning, to support Mcast enhancement */
    WMI_PDEV_PARAM_SET_MCAST2UCAST_BUFFER,            /* 0x59 */
    /** Remove the Mcast buffers added by host */
    WMI_PDEV_PARAM_REMOVE_MCAST2UCAST_BUFFER,         /* 0x5a */
    /** En/disable station power save state indication */
    WMI_PDEV_PEER_STA_PS_STATECHG_ENABLE,             /* 0x5b */
    /** Access category on which ARP frames are sent */
    WMI_PDEV_PARAM_IGMPMLD_AC_OVERRIDE,               /* 0x5c */
    /** allow or disallow interbss frame forwarding */
    WMI_PDEV_PARAM_BLOCK_INTERBSS,                    /* 0x5d */
    /** Enable/Disable reset */
    WMI_PDEV_PARAM_SET_DISABLE_RESET_CMDID,           /* 0x5e */
    /** Enable/Disable/Set MSDU_TTL in milliseconds. */
    WMI_PDEV_PARAM_SET_MSDU_TTL_CMDID,                /* 0x5f */
    /** Set global PPDU duration limit (usec). */
    WMI_PDEV_PARAM_SET_PPDU_DURATION_CMDID,           /* 0x60 */
    /** set txbf sounding period of vap in milliseconds */
    WMI_PDEV_PARAM_TXBF_SOUND_PERIOD_CMDID,           /* 0x61 */
    /** Set promiscuous mode */
    WMI_PDEV_PARAM_SET_PROMISC_MODE_CMDID,            /* 0x62 */
    /** Set burst mode */
    WMI_PDEV_PARAM_SET_BURST_MODE_CMDID,              /* 0x63 */
    /** enable enhanced stats */
    WMI_PDEV_PARAM_EN_STATS,                          /* 0x64 */
    /** Set mu-grouping policy */
    WMI_PDEV_PARAM_MU_GROUP_POLICY,                   /* 0x65 */
    /** Channel Hopping Enable */
    WMI_PDEV_PARAM_NOISE_DETECTION,                   /* 0x66 */
    /** Set Channel Hopping NF threshold in dBm */
    WMI_PDEV_PARAM_NOISE_THRESHOLD,                   /* 0x67 */
    /** Set PAPRD policy */
    WMI_PDEV_PARAM_DPD_ENABLE,                        /* 0x68 */
    /** Enable/disable mcast/bcast echo, used by ProxySTA */
    WMI_PDEV_PARAM_SET_MCAST_BCAST_ECHO,              /* 0x69 */
    /** ATF enable/disable strict schedule */
    WMI_PDEV_PARAM_ATF_STRICT_SCH,                    /* 0x6a */
    /** ATF set access category duration, B0-B29 duration, B30-B31: AC */
    WMI_PDEV_PARAM_ATF_SCHED_DURATION,                /* 0x6b */
    /** Default antenna polarization */
    WMI_PDEV_PARAM_ANT_PLZN,                          /* 0x6c */
    /** Set mgmt retry limit */
    WMI_PDEV_PARAM_MGMT_RETRY_LIMIT,                  /* 0x6d */
    /** Set CCA sensitivity level in dBm */
    WMI_PDEV_PARAM_SENSITIVITY_LEVEL,                 /* 0x6e */
    /** Set 2G positive and negative Tx power in 0.5dBm units */
    WMI_PDEV_PARAM_SIGNED_TXPOWER_2G,                 /* 0x6f */
    /** Set 5G positive and negative Tx power in 0.5dBm
     *  units */
    WMI_PDEV_PARAM_SIGNED_TXPOWER_5G,                 /* 0x70 */
    /** Enable/disble AMSDU for tids */
    WMI_PDEV_PARAM_ENABLE_PER_TID_AMSDU,              /* 0x71 */
    /** Enable/disable AMPDU for tids */
    WMI_PDEV_PARAM_ENABLE_PER_TID_AMPDU,              /* 0x72 */
    /** Set CCA threshold in dBm */
    WMI_PDEV_PARAM_CCA_THRESHOLD,                     /* 0x73 */
    /** RTS Fixed rate setting */
    WMI_PDEV_PARAM_RTS_FIXED_RATE,                    /* 0x74 */
    /** Pdev reset */
    WMI_PDEV_PARAM_PDEV_RESET,                        /* 0x75 */
    /** wapi mbssid offset **/
    WMI_PDEV_PARAM_WAPI_MBSSID_OFFSET,                /* 0x76 */
    /** ARP DEBUG source address*/
    WMI_PDEV_PARAM_ARP_DBG_SRCADDR,                   /* 0x77 */
    /** ARP DEBUG destination address*/
    WMI_PDEV_PARAM_ARP_DBG_DSTADDR,                   /* 0x78 */
    /** ATF enable/disable obss noise scheduling */
    WMI_PDEV_PARAM_ATF_OBSS_NOISE_SCH,                /* 0x79 */
    /** ATF obss noise scaling factor */
    WMI_PDEV_PARAM_ATF_OBSS_NOISE_SCALING_FACTOR,     /* 0x7a */
    /**
     * TX power reduction scaling exponent - final tx power is the
     * nominal tx power (A_MIN(reg_pow,ctl,etc..)) divided by
     * 2^(scale exponent).  For example:
     * If this scale exponent is  0, the power is unchanged (divided by 2^0)
     * If this factor is 1, the power is scaled down by 2^1, i.e. 3 dB
     * If this factor is 2, the power is scaled down by 2^2, i.e. 6 dB
     * If this factor is 3, the power is scaled down by 2^3, i.e. 9 dB
     */
    WMI_PDEV_PARAM_CUST_TXPOWER_SCALE,                /* 0x7b */
    /** ATF enabe/disabe dynamically */
    WMI_PDEV_PARAM_ATF_DYNAMIC_ENABLE,                /* 0x7c */
    /** Set tx retry limit for control frames. 0 = disable, 31 = max */
    WMI_PDEV_PARAM_CTRL_RETRY_LIMIT,                  /* 0x7d */
    /** Set propagation delay for 2G / 5G band.
     * The propagation delay is fundamentally a per-peer property, but
     * the target may not support per-peer settings for ack timeouts.
     * This pdev parameter allows the MAC-level ack timeout to be set to
     * a value suitable for the worst-case propagation delay of any peer
     * within that pdev.
     * Units are microseconds.
     */
    WMI_PDEV_PARAM_PROPAGATION_DELAY,                 /* 0x7e */
    /**
     * Host can enable/disable ANT DIV feature
     * if it's been enabled in BDF
     */
    WMI_PDEV_PARAM_ENA_ANT_DIV,                       /* 0x7f */
    /** Host can force one chain to select a specific ANT */
    WMI_PDEV_PARAM_FORCE_CHAIN_ANT,                   /* 0x80 */
    /**
     * Start a cycle ANT self test periodically.
     * In the test, the FW would select each ANT pair
     * one by one, the cycle time could be configured
     * via WMI_PDEV_PARAM_ANT_DIV_SELFTEST_INTVL
     */
    WMI_PDEV_PARAM_ANT_DIV_SELFTEST,                  /* 0x81 */
    /**
     * Configure the cycle time of ANT self test,
     * the unit is micro second. Per the timer
     * limitation, too small value could be not so
     * accurate.
     */
    WMI_PDEV_PARAM_ANT_DIV_SELFTEST_INTVL,            /* 0x82 */
    /**
     * wlan stats observation period, the unit is millisecond.
     * The value of 0 is used to turn off periodic stats report.
     */
    WMI_PDEV_PARAM_STATS_OBSERVATION_PERIOD,          /* 0x83 */
    /**
     * Set tx_ppdu_delay[] bin size to specify how many
     * milliseconds each bin of the wmi_tx_stats.tx_ppdu_delay[]
     * histogram represents.
     */
    WMI_PDEV_PARAM_TX_PPDU_DELAY_BIN_SIZE_MS,         /* 0x84 */
    /** set wmi_tx_stats.tx_ppdu_delay[] array length */
    WMI_PDEV_PARAM_TX_PPDU_DELAY_ARRAY_LEN,           /* 0x85 */
    /** set wmi_tx_stats.tx_mpdu_aggr[] array length */
    WMI_PDEV_PARAM_TX_MPDU_AGGR_ARRAY_LEN,            /* 0x86 */
    /** set wmi_rx_stats.rx_mpdu_aggr[] array length */
    WMI_PDEV_PARAM_RX_MPDU_AGGR_ARRAY_LEN,            /* 0x87 */
    /** Set TX delay value in TX sch module, unit is microseconds */
    WMI_PDEV_PARAM_TX_SCH_DELAY,                      /* 0x88 */
    /** Set RTS enable for SIFS bursting */
    WMI_PDEV_PARAM_ENABLE_RTS_SIFS_BURSTING,          /* 0x89 */
    /** Set Maximum number of MPDUs in an AMPDU*/
    WMI_PDEV_PARAM_MAX_MPDUS_IN_AMPDU,                /* 0x8a */

    /** Enable/disable peer stats info mechanism
     * A zero value disables; a non-zero value enables.
     */
    WMI_PDEV_PARAM_PEER_STATS_INFO_ENABLE,            /* 0x8b */

    /** Configure Fast PWR Transition mode
     * 0x0 -> inidcates Fast PWR transition disabled
     * 0x1 -> indicates Static mode enabled
     * 0x2 -> indicates Dynamic mode enabled
     */
    WMI_PDEV_PARAM_FAST_PWR_TRANSITION,               /* 0x8c */

    /** Enable/disable radio channel stats mechanism
     *  A zero value disables; a non-zero value enables.
     */
    WMI_PDEV_PARAM_RADIO_CHAN_STATS_ENABLE,           /* 0x8d */
    /** Enable/disable radio diagnosis feature
     *  which allows retrieving the status of radio.
     *  A zero value disables; a non-zero value enables.
     */
    WMI_PDEV_PARAM_RADIO_DIAGNOSIS_ENABLE,            /* 0x8e */
    /** Enable/Disable mesh mcast traffic
     * 1 - Allow mesh mcast traffic
     * 0 - Disallow mesh mcast traffic
     */
    WMI_PDEV_PARAM_MESH_MCAST_ENABLE,                 /* 0x8f */
    /** Enable/Disable smart chainmask scheme
      * 1 - Enable smart chainmask scheme
      * 0 - Disable smart chainmask scheme
      */
    WMI_PDEV_PARAM_SMART_CHAINMASK_SCHEME,            /* 0x90 */
    /** Enable/Disable alternate chainmask scheme
     * 1 - Enable alternate chainmask scheme
     * 0 - Disable alternate chainmask scheme
     */
    WMI_PDEV_PARAM_ALTERNATIVE_CHAINMASK_SCHEME,      /* 0x91 */
    /** User configured parameters for antenna diversity algorithm
     *  BIT[25..13]: Probe period (milliseconds units)
     *  BIT[12..0]: Stay period (milliseconds units)
     */
    WMI_PDEV_PARAM_ANT_DIV_USRCFG,                    /* 0x92 */
    /** pdev packet power save levels,
     *  refer to WMI_PDEV_PKT_PWR_SAVE_LEVEL
     */
    WMI_PDEV_PARAM_PACKET_POWER_SAVE_LEVEL,           /* 0x93 */
    /** Define IOT pattern to be enabled/disabled
      * bit values: 0 - disable, 1 - enable
      * BIT[0..31]: each bit represents an IOT pattern
      * -----
      * Bit 0 - avoid SMPS with certain APs
      * Bits 31:1 - reserved
      */
    WMI_PDEV_PARAM_SET_IOT_PATTERN,                   /* 0x94 */
    /** ACK timeout - change wireless packet ack timeout configuration,
     *  units are microseconds
     */
    WMI_PDEV_PARAM_ACK_TIMEOUT,                       /* 0x95 */
    /** Number of TX chains to use for a/b/g rates.
     *  bit 0~15  : 11b mode TX chain number.
     *  bit 16~31 : 11ag mode TX chain number.
     */
    WMI_PDEV_PARAM_ABG_MODE_TX_CHAIN_NUM,             /* 0x96 */
    /** Enable/Disable cck txfir override
     *  bit 0 - enable (1) or disable (0) CCK tx FIR
     *  bits 31:1 - unused / reserved (set to 0)
     */
    WMI_PDEV_PARAM_ENABLE_CCK_TXFIR_OVERRIDE,         /* 0x97 */
    /** Enable/Disable DTIM Synth
     *   1- Enable DTIM Synth
     *   0- Disable DTIM Synth
     */
    WMI_PDEV_PARAM_DTIM_SYNTH,                        /* 0x98 */
    /** Configure auto detect power failure feature.
     *   0 - FW will trigger crash if power failure happens.
     *   1 - FW will send a failure notification to host, and the host
     *       framework determines how to respond to the power failure
     *   2 - Silently rejuvenate if power failure occurs.
     *   3 - Feature disabled.
     */
    WMI_PDEV_AUTO_DETECT_POWER_FAILURE,               /* 0x99 */
    /** Configure operating voltage corner mode based on phymode and bw.
     * bit 0-1 -   operating voltage corner mode for 11a/b.
     * bit 2-3 -   operating voltage corner mode for 11g.
     * bit 4-5 -   operating voltage corner mode for 11n, 20MHz, 1x1.
     * bit 6-7 -   operating voltage corner mode for 11n, 20MHz, 2x2.
     * bit 8-9 -   operating voltage corner mode for 11n, 40MHz, 1x1.
     * bit 10-11 - operating voltage corner mode for 11n, 40MHz, 2x2.
     * bit 12-13 - operating voltage corner mode for 11ac, 20MHz, 1x1.
     * bit 14-15 - operating voltage corner mode for 11ac, 20MHz, 2x2.
     * bit 16-17 - operating voltage corner mode for 11ac, 40MHz, 1x1.
     * bit 18-19 - operating voltage corner mode for 11ac, 40MHz, 2x2.
     * bit 20-21 - operating voltage corner mode for 11ac, 80MHz, 1x1.
     * bit 22-23 - operating voltage corner mode for 11ac, 80MHz, 2x2.
     * bit 24-25 - operating voltage corner mode for 11ac, 160MHz, 1x1.
     * bit 26-27 - operating voltage corner mode for 11ac, 160MHz, 2x2.
     * ---------------------------------------------
     * 00 - Static voltage corner SVS
     * 01 - static voltage corner LOW SVS
     * 10 - Dynamic voltage corner selection based on TPUT
     * 11 - Dynamic voltage corner selection based on TPUT and Tx Flush counters
     */
    WMI_PDEV_UPDATE_WDCVS_ALGO,                       /* 0x9a */

    /* Enable/Disable data stall detection */
    WMI_PDEV_PARAM_DATA_STALL_DETECT_ENABLE,          /* 0x9b */
    /* GCMP Support indication to FW */
    WMI_PDEV_PARAM_GCMP_SUPPORT_ENABLE,               /* 0x9c */
    /** Enable/Disable chain selection optimization for one chain dtim
     *   non-zero - Enable optimization and use this non-zero value as the
     *              chain imbalance threshold for optimization to kick in
     *              (units = dB)
     *   0- Disable optimization
     */
    WMI_PDEV_PARAM_1CH_DTIM_OPTIMIZED_CHAIN_SELECTION,/* 0x9d */
    /*
     * Override default FW behavior and explicitly enable / disable
     * the use of CCK for PPDU transmissions.
     *
     * When CCK transmissions are disabled, the default OFDM legacy
     * rate will be used instead.
     */
    WMI_PDEV_PARAM_CCK_TX_ENABLE,                     /* 0x9e */
    /*
     * Set the user-specified antenna gain, but in 0.5 dB units.
     * This is a finer-granularity version of WMI_PDEV_PARAM_ANTENNA_GAIN.
     * E.g. to set a gain of 15.5 dB, a value of 31 could be provided as the
     * value accompanying the PDEV_PARAM_ANTENNA_GAIN_HALF_DB parameter type.
     */
    WMI_PDEV_PARAM_ANTENNA_GAIN_HALF_DB,              /* 0x9f */
    /*
     * Global Enable/Disable control for Secondary Retry Feature Set
     *
     * Bit-0  : Enable/Disable Control for "PPDU Secondary Retry Support"
     * Bit-1  : Enable/Disable Control for "RTS Black/White-listing Support"
     * Bit-2  : Enable/Disable Control for "Higher MCS retry restriction on XRETRY failures"
     * Bit 3-5: "Xretry threshold" to use
     */
    WMI_PDEV_PARAM_SECONDARY_RETRY_ENABLE,            /* 0xA0 */
    /** Set global uplink triggered PPDU duration limit (usec). */
    WMI_PDEV_PARAM_SET_UL_PPDU_DURATION,              /* 0xA1 */
    /** Set target buffer status report trigger interval (ms) */
    WMI_PDEV_PARAM_SET_UL_BSR_TRIG_INTERVAL,          /* 0xA2 */
    /** Use simplified equal RU allocation for DL and UL OFDMA */
    WMI_PDEV_PARAM_EQUAL_RU_ALLOCATION_ENABLE,        /* 0xA3 */
    /** Enable/disable MWS-COEX 4G (LTE) Quick FTDM.
     * 0 - Don't allow quick FTDM Policy (Default)
     * 1 - Allow quick FTDM policy.
     */
    WMI_PDEV_PARAM_MWSCOEX_4G_ALLOW_QUICK_FTDM,       /* 0xA4 */
    /** Set MWS-COEX 5G-NR power limit.
     * 0:    Don't apply user specific power limit,
     *       use internal power limit (Default)
     * 1-2:  invalid value (ignored)
     * 3-21: apply the specified value as the external power limit, in dBm
     * >21:  invalid value (ignored)
     */
    WMI_PDEV_PARAM_MWSCOEX_SET_5GNR_PWR_LIMIT,        /* 0xA5 */
    /** Set max msdus available for cong ctrl in target */
    WMI_PDEV_PARAM_SET_CONG_CTRL_MAX_MSDUS,           /* 0xA6 */
    /*
     * Configures the Estimated Throughput Calculation indication (802.11mc) settings.
     * The accompanying A_UINT32 parameter, in units of seconds, specifies how often FW needs to send the ESP estimation indication to the host.
     * Value 0: Disable this feature
     * Non zero Value: Periodicity (seconds)
     */
    WMI_PDEV_PARAM_ESP_INDICATION_PERIOD,             /* 0xA7 */

    /*
     * Enable/Disable periodic peer CFR capture
     * WMI_PEER_CFR_CAPTURE_ENABLE - Enable per peer periodic CFR capture
     * WMI_PEER_CFR_CAPTURE_DISABLE - Disable per peer periodic CFR capture
     */
    WMI_PDEV_PARAM_PER_PEER_PERIODIC_CFR_ENABLE,

    /*
     * Set the base timer for the periodic CFR capture. By default this is 10ms.
     * The period ('periodicity' param in wmi_peer_cfr_capture_cmd) of
     * CFR measurment of other peers will be in multiples of this base timer.
     * The unit is in milliseconds.
     */
    WMI_PDEV_PARAM_PERIODIC_CFR_BASE_TIMER,

    /*
     * Once the periodic capture is enabled using
     * WMI_PDEV_PARAM_PER_PEER_PERIODIC_CFR_ENABLE, the timer starts running in
     * the target. This parameter will ensure that the timer stops if there are
     * no active peers in the capture list. Once the peers are added again to
     * the capture list, the timer will not start again. The timer has to be
     * started again using WMI_PDEV_PARAM_PER_PEER_PERIODIC_CFR_ENABLE.
     * Value 1: Enable this feature
     * Value 0: Disable this feature
     */
    WMI_PDEV_PARAM_ENABLE_OPTIMIZED_PERIODIC_CFR_TIMER,

    /*
     * Configures a portion of the Estimated Service Params indication
     * (802.11mc) settings, which together specify estimated throughput.
     * The accompanying A_UINT32 parameter is the ESP BA WINDOW size advertised
     * Value 0: Disable this feature
     * Non zero Value: ESP BA WINDOW size
     */
    WMI_PDEV_PARAM_ESP_BA_WINDOW,

    /*
     * Configures a portion of the Estimated Service Params indication
     * (802.11mc) settings, which together specify estimated throughput.
     * The accompanying A_UINT32 parameter is the air time fraction to be
     * advertised in the ESP IE
     * Value 0: Disable this feature
     * Non zero Value: Air time fraction in percentage
     */
    WMI_PDEV_PARAM_ESP_AIRTIME_FRACTION,

    /*
     * Configures a portion of the Estimated Service Params indication
     * (802.11mc) settings, which together specify estimated throughput.
     * The accompanying A_UINT32 parameter specifies PPDU duration in units
     * of milliseconds.
     * Value 0: Disable this feature
     * Non zero Value: PPDU duration in milliseconds
     */
    WMI_PDEV_PARAM_ESP_PPDU_DURATION,

    /*
     * Enable/Disable NOL(Non Occupancy list) in firmware
     *   1- Use NOL in firmware
     *   0- Don't use NOL in firmware
     */
    WMI_PDEV_PARAM_USE_NOL,

    /*
     * Allow / Not Allow RU26 in any user's RU allocation field in UL OFDMA
     * trigger frames sent by AP
     *  1 - Allow RU26
     *  0 - Do not allow RU26
     */
    WMI_PDEV_PARAM_UL_RU26_ALLOWED,

    /*
     * Enable/Disable sub channel marking
     *  1 - Enable sub channel marking
     *  0 - Disable sub channel marking (default value)
     */
    WMI_PDEV_PARAM_SUB_CHANNEL_MARKING,

    /*
     * Enable/Disable/Set MGMT_TTL in milliseconds.
     * non_zero - Enable, with the specified value
     * 0 - Disable
     */
    WMI_PDEV_PARAM_SET_MGMT_TTL,

    /*
     * Enable/Disable/Set PROBE_RESP_TTL in milliseconds.
     * non_zero - Enable, with the specified value
     * 0 - Disable
     */
    WMI_PDEV_PARAM_SET_PROBE_RESP_TTL,

    /*
     * TBTT_CTRL_CFG
     * BITS 0 - 2 (refer to WMI_TBTT_CTRL_CFG enum)
     *   0  - DEFAULT -> HW_TBTT
     *   1  - SW_TBTT -> HW_TBTT disabled,
     *        software would truncate BURST near TBTT
     *   2  - IGNORE_TBTT
     *
     * BITS 3 - 31 Reserved, must be set to 0x0
     */
    WMI_PDEV_PARAM_SET_TBTT_CTRL,

    /*
     * BITS0 ~1 (refer to enum)
     * 0 - default --> always update
     * 1 - ignore to update
     * 2 - update if larger than threshould
     * 3 - update if less or equal than threshould
     *
     * BITS 2 ~ 31 Threshould
     */
    WMI_PDEV_PARAM_NAV_OVERRIDE_CONFIG,

    /* Set global MU PPDU duration for DL (usec units) */
    WMI_PDEV_PARAM_SET_MU_PPDU_DURATION,

    /*
     * Enable / disable test mode configuration.
     * By default FW will always send triggers dynamically (mix of BSR/Basic).
     * The below testmode are only used for certain tests.
     * A value of 1 in a given bit enables corresponding test mode.
     * bit | test mode
     * ---------------
     *  0  | FW only sends BSR triggers.
     *  1  | FW only sends Basic triggers.
     *  2  | If set, FW enables MU-RTS trigger.
     *     | If cleared, FW uses implementation-specific internal default setting.
     *  3  | FW enables unicast embedded trigger in HE MU PPDU.
     * 4-31| reserved.
     */
    WMI_PDEV_PARAM_SET_TEST_CMD_UL_TRIGGER_TYPE_ENABLE,

    /*
     * Configure test command to set LSIG len.
     * Value 0: Dynamic LSIG based on STA's qdepth.
     * Non zero Value: LSIG length to be configured
     *                 as part of trigger frame.
     *   bit   |
     * ---------------
     *  0 - 30 | Bits storing the host supplied <value>.
     *  31     | If set: The legitimate value closest to the value specified in
     *         |   in bits 30:0 is directly set in UL len in trigger frame.
     *         |   The FW performs calculations to determine which legitimate
     *         |   value is closest to the specified value, if the specified
     *         |   value is not already legitimate.
     *         | If not set: The value in lower bits is the duration (in ms),
     *         |   from which the UL len is derived.
     *         |   Example: if host sets 2000 (2ms), then UL Len in trigger
     *         |   will be derived to accomodate the given duration.
     */
    WMI_PDEV_PARAM_SET_TEST_CMD_UL_TRIGGER_LSIG_LEN,

    /*
     * Configure test cmd for fixed rate setting used for UL Trigger
     * (only Basic/BSR triggers).
     * The top nibble is used to select which format to use for encoding
     * the rate specification: 0xVXXXXXXX, V must be 1 for the UL
     * format.
     * If V == 0b0001: format is: 0x1000RRRR.
     *                 This will be output of WMI_ASSEMBLE_RATECODE_V1
     *
     */
    WMI_PDEV_PARAM_SET_TEST_CMD_UL_TRIGGER_FIXED_RATE,

    /*
     * Configure test command to set the mac padding duration.
     *  0 - FW set Mac Padding to 0us
     *  1 - FW set Mac Padding to 8us
     *  2 - FW set Mac Padding to 16us
     */
    WMI_PDEV_PARAM_SET_TEST_CMD_UL_MAC_PADDING,

    /*
     * Configure test command to set the fc duration in BSR trigger frame.
     *  value 0 - FW calulates the duration(default).
     *  Non zero Value: Duration to be configured.
     */
    WMI_PDEV_PARAM_SET_TEST_CMD_UL_BSR_FC_DURATION,

    /* Parameter used for configuring TWT scheduling properties
     * bit | config_mode
     * -----------------
     *  0  | Disables DL MU for TWT peers within TWT SP
     *  1  | Disables UL MU for TWT peers within TWT SP
     *  2  | Disables scheduling from WMM sched context for TWT peers
     *  3  | If set, FW only sends Basic triggers in TWT SP.
     * 4-31| reserved.
     */
    WMI_PDEV_PARAM_SET_TEST_CMD_TWT_SCHED_CONFIG,

} WMI_PDEV_PARAM;

#define WMI_PDEV_ONLY_BSR_TRIG_IS_ENABLED(trig_type) WMI_GET_BITS(trig_type, 0, 1)
#define WMI_PDEV_ONLY_BSR_TRIG_ENABLE(trig_type) WMI_SET_BITS(trig_type, 0, 1, 1)
#define WMI_PDEV_ONLY_BSR_TRIG_DISABLE(trig_type) WMI_SET_BITS(trig_type, 0, 1, 0)

#define WMI_PDEV_ONLY_BASIC_TRIG_IS_ENABLED(trig_type) WMI_GET_BITS(trig_type, 1, 1)
#define WMI_PDEV_ONLY_BASIC_TRIG_ENABLE(trig_type) WMI_SET_BITS(trig_type, 1, 1, 1)
#define WMI_PDEV_ONLY_BASIC_TRIG_DISABLE(trig_type) WMI_SET_BITS(trig_type, 1, 1, 0)

#define WMI_PDEV_MU_RTS_IS_ENABLED(trig_type) WMI_GET_BITS(trig_type, 2, 1)
#define WMI_PDEV_MU_RTS_ENABLE(trig_type) WMI_SET_BITS(trig_type, 2, 1, 1)
#define WMI_PDEV_MU_RTS_DISABLE(trig_type) WMI_SET_BITS(trig_type, 2, 1, 0)

#define WMI_PDEV_EMBEDDED_TRIGGER_IS_ENABLED(trig_type) WMI_GET_BITS(trig_type, 3, 1)
#define WMI_PDEV_EMBEDDED_TRIGGER_ENABLE(trig_type) WMI_SET_BITS(trig_type, 3, 1, 1)
#define WMI_PDEV_EMBEDDED_TRIGGER_DISABLE(trig_type) WMI_SET_BITS(trig_type, 3, 1, 0)

#define WMI_PDEV_TWT_SCHED_CFG_IS_DL_MU_IS_ENABLED(twt_sched_cfg) WMI_GET_BITS(twt_sched_cfg, 0, 1)
#define WMI_PDEV_TWT_SCHED_CFG_DL_MU_ENABLE(twt_sched_cfg) WMI_SET_BITS(twt_sched_cfg, 0, 1, 1)
#define WMI_PDEV_TWT_SCHED_CFG_DL_MU_DISABLE(twt_sched_cfg) WMI_SET_BITS(twt_sched_cfg, 0, 1, 0)

#define WMI_PDEV_TWT_SCHED_CFG_IS_UL_MU_IS_ENABLED(twt_sched_cfg) WMI_GET_BITS(twt_sched_cfg, 1, 1)
#define WMI_PDEV_TWT_SCHED_CFG_UL_MU_ENABLE(twt_sched_cfg) WMI_SET_BITS(twt_sched_cfg, 1, 1, 1)
#define WMI_PDEV_TWT_SCHED_CFG_UL_MU_DISABLE(twt_sched_cfg) WMI_SET_BITS(twt_sched_cfg, 1, 1, 0)

#define WMI_PDEV_TWT_SCHED_CFG_IS_WMM_IS_ENABLED(twt_sched_cfg) WMI_GET_BITS(twt_sched_cfg, 2, 1)
#define WMI_PDEV_TWT_SCHED_CFG_WMM_ENABLE(twt_sched_cfg) WMI_SET_BITS(twt_sched_cfg, 2, 1, 1)
#define WMI_PDEV_TWT_SCHED_CFG_WMM_DISABLE(twt_sched_cfg) WMI_SET_BITS(twt_sched_cfg, 2, 1, 0)

#define WMI_PDEV_TWT_SCHED_CFG_IS_USE_ONLY_BASIC_TRIGGER_IS_ENABLED(twt_sched_cfg) WMI_GET_BITS(twt_sched_cfg, 3, 1)
#define WMI_PDEV_TWT_SCHED_CFG_USE_ONLY_BASIC_TRIGGER_ENABLE(twt_sched_cfg) WMI_SET_BITS(twt_sched_cfg, 3, 1, 1)
#define WMI_PDEV_TWT_SCHED_CFG_USE_ONLY_BASIC_TRIGGER_DISABLE(twt_sched_cfg) WMI_SET_BITS(twt_sched_cfg, 3, 1, 0)

#define WMI_PDEV_LSIG_LEN_DURATION_ENABLE(lsig_len) WMI_SET_BITS(lsig_len, 0, 31, 1)
#define WMI_PDEV_LSIG_LEN_DURATION_DISABLE(lsig_len) WMI_SET_BITS(lsig_len, 0, 31, 0)
#define WMI_PDEV_LSIG_LEN_DURATION_GET(lsig_len) WMI_GET_BITS(lsig_len, 0, 30)
#define WMI_PDEV_LSIG_LEN_DURATION_SET(lsig_len, value) WMI_SET_BITS(lsig_len, 0, 30, value)

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_param_cmd_fixed_param */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
    /** parameter id   */
    A_UINT32 param_id;/* WMI_PDEV_PARAM */
    /** parameter value */
    A_UINT32 param_value;
} wmi_pdev_set_param_cmd_fixed_param;

/* param values for WMI_PDEV_PARAM_SET_TBTT_CTRL's TBTT_CTRL_CFG bit-field */
typedef enum {
    WMI_TBTT_CTRL_HW_TRUNCATE = 0,
    WMI_TBTT_CTRL_SW_TRUNCATE,
    WMI_TBTT_CTRL_IGNORE_TBTT,

    WMI_TBTT_CTRL_MAX = 0x7,
} WMI_TBTT_CTRL_CFG;

/** MACRO to set / get TBTT_CTRL_CFG bit-field within
 *  WMI_PDEV_PARAM_SET_TBTT_CTRL:
 *      bits 0~2 : TBTT_CTRL_CFG
 *      bits 3~31: Reserved (set to 0x0)
 */
#define WMI_PDEV_PARAM_TBTT_CTRL_CFG_S  0
#define WMI_PDEV_PARAM_TBTT_CTRL_CFG    0x00000007

#define WMI_PDEV_PARAM_GET_TBTT_CTRL_CFG(word32) \
    WMI_F_MS(word32, WMI_PDEV_PARAM_TBTT_CTRL_CFG)
#define WMI_PDEV_PARAM_SET_TBTT_CTRL_CFG(word32, value) \
    WMI_F_RMW(word32,value,WMI_PDEV_PARAM_TBTT_CTRL_CFG)

typedef enum {
    /** Set the loglevel */
    WMI_DBGLOG_LOG_LEVEL = 0x1,
    /** Enable VAP level debug */
    WMI_DBGLOG_VAP_ENABLE,
    /** Disable VAP level debug */
    WMI_DBGLOG_VAP_DISABLE,
    /** Enable MODULE level debug */
    WMI_DBGLOG_MODULE_ENABLE,
    /** Disable MODULE level debug */
    WMI_DBGLOG_MODULE_DISABLE,
    /** Enable MODULE level debug */
    WMI_DBGLOG_MOD_LOG_LEVEL,
    /** set type of the debug output */
    WMI_DBGLOG_TYPE,
    /** Enable Disable debug */
    WMI_DBGLOG_REPORT_ENABLE
} WMI_DBG_PARAM;

/** MACRO define to set / get 11b and 11ag mode TX chain number:
 *  bit 0~15 : 11b mode TX chain number.
 *  bit 16~31: 11ag mode TX chain number.
 */
#define WMI_PDEV_PARAM_11B_TX_CHAIN_NUM_S  0
#define WMI_PDEV_PARAM_11B_TX_CHAIN_NUM    0x0000FFFF
#define WMI_PDEV_PARAM_11AG_TX_CHAIN_NUM_S 16
#define WMI_PDEV_PARAM_11AG_TX_CHAIN_NUM   0xFFFF0000

#define WMI_PDEV_PARAM_GET_11B_TX_CHAIN_NUM(word32) WMI_F_MS(word32, WMI_PDEV_PARAM_11B_TX_CHAIN_NUM)
#define WMI_PDEV_PARAM_SET_11B_TX_CHAIN_NUM(word32, value)  WMI_F_RMW(word32,value,WMI_PDEV_PARAM_11B_TX_CHAIN_NUM)

#define WMI_PDEV_PARAM_GET_11AG_TX_CHAIN_NUM(word32) WMI_F_MS(word32, WMI_PDEV_PARAM_11AG_TX_CHAIN_NUM)
#define WMI_PDEV_PARAM_SET_11AG_TX_CHAIN_NUM(word32, value) WMI_F_RMW(word32,value,WMI_PDEV_PARAM_11AG_TX_CHAIN_NUM)

/* param_value for param_id WMI_PDEV_PARAM_CTS_CBW */
typedef enum {
    WMI_CTS_CBW_INVALID = 0,
    WMI_CTS_CBW_20,
    WMI_CTS_CBW_40,
    WMI_CTS_CBW_80,
    WMI_CTS_CBW_80_80,
    WMI_CTS_CBW_160,
} WMI_CTS_CBW;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_get_tpc_config_cmd_fixed_param */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
    /** parameter   */
    A_UINT32 param;
} wmi_pdev_get_tpc_config_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* WMITLV_TAG_STRUC_wmi_pdev_div_get_rssi_antid_fixed_param */
    /** pdev_id for identifying the MAC */
    A_UINT32 pdev_id;
    /** RSSI (rssi_chain_x_pri20) on each chain (units: dB above noise floor) */
    A_UINT32 chain_rssi[WMI_MAX_CHAINS];
    /** index of the last-used antenna for each chain */
    A_UINT32 ant_id[WMI_MAX_CHAINS];
    /** mac address of diversity peer */
    wmi_mac_addr macaddr;
} wmi_pdev_div_get_rssi_antid_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* WMITLV_TAG_STRUC_wmi_pdev_bss_chan_info_request_fixed_param */
    A_UINT32 param;   /* 1 = read only, 2= read and clear */
} wmi_pdev_bss_chan_info_request_fixed_param;

typedef struct {
    A_UINT32 tlv_header;    /* WMITLV_TAG_STRUC_wmi_pdev_update_ctltable_request_fixed_param */
    A_UINT32 total_len;     /* the total number of ctl table bytes to be transferred */
    A_UINT32 len;           /* the number of ctltable bytes in current payload */
    A_UINT32 seq;           /* the number of current transfers */
/*
 * This TLV is followed by the following additional TLVs:
 * ARRAY_BYTE TLV of ctltable_data
 */
} wmi_pdev_update_ctltable_request_fixed_param;

#define WMI_ESP_ESTIMATE_GET_BE(airtime)         WMI_GET_BITS(airtime, 0, 8)
#define WMI_ESP_ESTIMATE_SET_BE(airtime, value)  WMI_SET_BITS(airtime, 0, 8, value)

#define WMI_ESP_ESTIMATE_GET_BK(airtime)         WMI_GET_BITS(airtime, 8, 8)
#define WMI_ESP_ESTIMATE_SET_BK(airtime, value)  WMI_SET_BITS(airtime, 8, 8, value)

#define WMI_ESP_ESTIMATE_GET_VI(airtime)         WMI_GET_BITS(airtime, 16, 8)
#define WMI_ESP_ESTIMATE_SET_VI(airtime, value)  WMI_SET_BITS(airtime, 16, 8, value)

#define WMI_ESP_ESTIMATE_GET_VO(airtime)         WMI_GET_BITS(airtime, 24, 8)
#define WMI_ESP_ESTIMATE_SET_VO(airtime, value)  WMI_SET_BITS(airtime, 24, 8, value)

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_esp_estimate_event_fixed_param */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     * In non-DBDC case host should set it to 0
     */
    A_UINT32 pdev_id;
    /*
     * Percentage of air time available for each AC
     * BIT[0-7]   : AC_BE
     * BIT[8-15]  : AC_BK
     * BIT[16-23] : AC_VI
     * BIT[24-31] : AC_VO
     */
    A_UINT32 ac_airtime_percentage;
} wmi_esp_estimate_event_fixed_param;

#define WMI_FAST_DIVERSITY_BIT_OFFSET 0
#define WMI_SLOW_DIVERSITY_BIT_OFFSET 1

#define WMI_SLOW_DIVERSITY_CH0_WEIGHT_SHIFT 2
#define WMI_SLOW_DIVERSITY_CH0_WEIGHT_MASK \
		(0xf << WMI_SLOW_DIVERSITY_CH0_WEIGHT_SHIFT)
#define WMI_SLOW_DIVERSITY_CH0_WEIGHT_GET_BITS(word32) \
	(((word32) & WMI_SLOW_DIVERSITY_CH0_WEIGHT_MASK) >> \
		WMI_SLOW_DIVERSITY_CH0_WEIGHT_SHIFT)
#define WMI_SLOW_DIVERSITY_CH0_WEIGHT_SET_BITS(word32, value) \
	do { \
		(word32) &= ~WMI_SLOW_DIVERSITY_CH0_WEIGHT_MASK; \
		(word32) |= ((value) << WMI_SLOW_DIVERSITY_CH0_WEIGHT_SHIFT) & \
			WMI_SLOW_DIVERSITY_CH0_WEIGHT_MASK; \
	} while (0)

#define WMI_SLOW_DIVERSITY_CH1_WEIGHT_SHIFT 6
#define WMI_SLOW_DIVERSITY_CH1_WEIGHT_MASK \
		(0xf << WMI_SLOW_DIVERSITY_CH1_WEIGHT_SHIFT)
#define WMI_SLOW_DIVERSITY_CH1_WEIGHT_GET_BITS(word32) \
	(((word32) & WMI_SLOW_DIVERSITY_CH1_WEIGHT_MASK) >> \
		WMI_SLOW_DIVERSITY_CH1_WEIGHT_SHIFT)
#define WMI_SLOW_DIVERSITY_CH1_WEIGHT_SET_BITS(word32, value) \
	do { \
		(word32) &= ~WMI_SLOW_DIVERSITY_CH1_WEIGHT_MASK; \
		(word32) |= ((value) << WMI_SLOW_DIVERSITY_CH1_WEIGHT_SHIFT) & \
			WMI_SLOW_DIVERSITY_CH1_WEIGHT_MASK; \
	} while (0)

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_antenna_diversity_cmd_fixed_param */
    union {
        A_UINT32 mac_id; // OBSOLETE - will be removed once all refs are gone
        /** pdev_id for identifying the MAC
         * See macros starting with WMI_PDEV_ID_ for values.
         */
        A_UINT32 pdev_id;
    };
    /**
     * The following "value" field is divided into bit fields as follows:
     *     bits | purpose
     *     -----+---------------------------------------
     *        0 | enable/disable FAST diversity
     *        1 | enable/disable SLOW diversity
     *      5:2 | chain0 slow-diversity weighting factor
     *      9:6 | chain1 slow-diversity weighting factor
     *     31:10| currenty unused (set to 0x0)
     * Refer to the above WMI_[FAST/SLOW]_DIVERSITY constants.
     */
    A_UINT32 value;
} wmi_pdev_set_antenna_diversity_cmd_fixed_param;

#define WMI_MAX_RSSI_THRESHOLD_SUPPORTED 3

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_rssi_breach_monitor_config_cmd_fixed_param */
    A_UINT32 vdev_id; /* vdev_id, where RSSI monitoring will take place */
    A_UINT32 request_id; /* host will configure request_id and firmware echo this id in RSSI_BREACH_EVENT */
    A_UINT32 enabled_bitmap; /* bit [0-2] = low_rssi_breach_enabled[0-2] enabled, bit [3-5] = hi_rssi_breach_enabled[0-2] */
    A_UINT32 low_rssi_breach_threshold[WMI_MAX_RSSI_THRESHOLD_SUPPORTED]; /* unit dBm. host driver to make sure [0] > [1] > [2] */
    A_UINT32 hi_rssi_breach_threshold[WMI_MAX_RSSI_THRESHOLD_SUPPORTED]; /* unit dBm. host driver to make sure [0] < [1] < [2] */
    A_UINT32 lo_rssi_reenable_hysteresis; /* unit dBm. once low rssi[] breached, same event bitmap will be generated only after signal gets better than this level. This value is adopted for all low_rssi_breach_threshold[3] */
    A_UINT32 hi_rssi_reenable_histeresis;/* unit dBm. once hi rssi[] breached, same event bitmap will be generated only after signal gets worse than this level. This value is adopted for all hi_rssi_breach_threshold[3] */
    A_UINT32 min_report_interval; /* After last event is generated, we wait until this interval to generate next event  */
    A_UINT32 max_num_report; /* this is to suppress number of event to be generated */
} wmi_rssi_breach_monitor_config_fixed_param;

typedef struct {
    /** parameter   */
    A_UINT32 param;
} wmi_pdev_dump_cmd;

typedef enum {
    PAUSE_TYPE_CHOP =           0x1, /** for MCC (switch channel), only vdev_map is valid */
    PAUSE_TYPE_PS =             0x2, /** for peer station sleep in sap mode, only peer_id is valid */
    PAUSE_TYPE_UAPSD =          0x3, /** for uapsd, only peer_id and tid_map are valid. */
    PAUSE_TYPE_P2P_CLIENT_NOA = 0x4, /** only vdev_map is valid, actually only one vdev id is set at one time */
    PAUSE_TYPE_P2P_GO_PS =      0x5, /** only vdev_map is valid, actually only one vdev id is set at one time */
    PAUSE_TYPE_STA_ADD_BA =     0x6, /** only peer_id and tid_map are valid, actually only one tid is set at one time */
    PAUSE_TYPE_AP_PS =          0x7, /** for pausing AP vdev when all the connected clients are in PS. only vdev_map is valid */
    PAUSE_TYPE_IBSS_PS =        0x8, /** for pausing IBSS vdev when all the peers are in PS. only vdev_map is valid */
    PAUSE_TYPE_CHOP_TDLS_OFFCHAN =  0x9, /** for TDLS offchannel MCC (switch channel), only vdev_map is valid, TDLS connection tracker needs to be notified */
    PAUSE_TYPE_HOST =           0x15,/** host is requesting vdev pause */
} wmi_tx_pause_type;

typedef enum {
    ACTION_PAUSE =     0x0,
    ACTION_UNPAUSE =   0x1,
} wmi_tx_pause_action;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 pause_type;
    A_UINT32 action;
    A_UINT32 vdev_map;
    A_UINT32 peer_id;
    A_UINT32 tid_map;
} wmi_tx_pause_event_fixed_param;

typedef enum {
    WMI_MGMT_TX_COMP_TYPE_COMPLETE_OK = 0,
    WMI_MGMT_TX_COMP_TYPE_DISCARD,
    WMI_MGMT_TX_COMP_TYPE_INSPECT,
    WMI_MGMT_TX_COMP_TYPE_COMPLETE_NO_ACK,
    WMI_MGMT_TX_COMP_TYPE_MAX,
} WMI_MGMT_TX_COMP_STATUS_TYPE;

typedef struct {
    A_UINT32    tlv_header;
    A_UINT32    desc_id; /* from tx_send_cmd */
    A_UINT32    status;  /* WMI_MGMT_TX_COMP_STATUS_TYPE */
    /** pdev_id for identifying the MAC that transmitted the mgmt frame
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32    pdev_id;
    /* ppdu_id
     * Hardware PPDU ID for tracking the completion stats
     * A ppdu_id value of 0x0 is invalid, and should be ignored.
     */
    A_UINT32    ppdu_id;
    /* ack_rssi
     * TX mgmt ack RSSI report to host.
     * Only valid when status == COMPLETE_OK and the ACK_RSSI report is enabled
     * ACK RSSI is reported as SNR dB, i.e. how many dB the RSSI is above
     * the noise floor.
     */
    A_UINT32    ack_rssi;
} wmi_mgmt_tx_compl_event_fixed_param;

typedef struct {
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_offchan_data_tx_compl_event_fixed_param */
    A_UINT32    desc_id; /* from tx_send_cmd */
    A_UINT32    status;  /* same status as WMI_MGMT_TX_COMP_STATUS_TYPE */
    /** pdev_id for identifying the MAC that transmitted the mgmt frame
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32    pdev_id;
    /* ppdu_id
     * Hardware PPDU ID for tracking the completion stats
     * A ppdu_id value of 0x0 is invalid, and should be ignored.
     */
    A_UINT32    ppdu_id;
} wmi_offchan_data_tx_compl_event_fixed_param;

typedef struct {
    A_UINT32    tlv_header;
    A_UINT32    num_reports;
    /* tlv for completion
     * A_UINT32 desc_ids[num_reports]; <- from tx_send_cmd
     * A_UINT32 status[num_reports];   <- WMI_MGMT_TX_COMP_STATUS_TYPE
     * A_UINT32 ppdu_id[num_reports];  <- list of PPDU IDs
     * A_UINT32 ack_rssi[num_reports]; <- list of ack RSSI
     *                                    RSSI units = dB w.r.t. noise floor
     */
} wmi_mgmt_tx_compl_bundle_event_fixed_param;

#define WMI_TPC_RATE_MAX            160
/* WMI_TPC_TX_NUM_CHAIN macro can't be changed without breaking the WMI compatibility */
#define WMI_TPC_TX_NUM_CHAIN        4

typedef enum {
    /* bits 0-7 for table flags */
    WMI_TPC_CONFIG_EVENT_FLAG_TABLE_CDD  = 0x1,
    WMI_TPC_CONFIG_EVENT_FLAG_TABLE_STBC = 0x2,
    WMI_TPC_CONFIG_EVENT_FLAG_TABLE_TXBF = 0x4,

    /* bits 8-11 for interface version flags */
    WMI_TPC_CONFIG_EVENT_FLAG_IF_MASK = 0x0F00,
    WMI_TPC_CONFIG_EVENT_FLAG_IF_V1   = 0x0100,
} WMI_TPC_CONFIG_EVENT_FLAG;

typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_tpc_config_event_fixed_param  */
    A_UINT32 regDomain;
    A_UINT32 chanFreq;
    A_UINT32 phyMode;
    A_UINT32 twiceAntennaReduction;
    A_UINT32 twiceMaxRDPower;
    A_INT32  twiceAntennaGain;
    A_UINT32 powerLimit;
    A_UINT32 rateMax;
    A_UINT32 numTxChain;
    A_UINT32 ctl;
    A_UINT32 flags;
    /* WMI_TPC_TX_NUM_CHAIN macro can't be changed without breaking the WMI compatibility */
    A_INT8  maxRegAllowedPower[WMI_TPC_TX_NUM_CHAIN];
    A_INT8  maxRegAllowedPowerAGCDD[WMI_TPC_TX_NUM_CHAIN][WMI_TPC_TX_NUM_CHAIN];
    A_INT8  maxRegAllowedPowerAGSTBC[WMI_TPC_TX_NUM_CHAIN][WMI_TPC_TX_NUM_CHAIN];
    A_INT8  maxRegAllowedPowerAGTXBF[WMI_TPC_TX_NUM_CHAIN][WMI_TPC_TX_NUM_CHAIN];
     /** pdev_id for identifying the MAC
      * See macros starting with WMI_PDEV_ID_ for values.
      */
    A_UINT32 pdev_id;
/* This TLV is followed by further TLVs:
 * 1.  byte array:
 *      A_UINT8 ratesArray[];
 * 2a. multi-dimensional array dimension spec:
 *      wmi_tlv_arrays_len_param ctlPwrTbl_param[0 or 1]
 *
 * 2b. ctl power table
 *      A_UINT8 ctlPwrTbl_buf[bf][modes][nss], i.e.
 *      A_UINT8 ctlPwrTbl_buf[d3_len * d2_len * d1_len]
 *      bf is [0 or 1] for [on or off].
 *      For 2G, modes is [0,   1,      2,       or 3] for
 *                       [cck, legacy, (v)ht20, or (v)ht40]
 *      For 5G, modes is [0,      1,       2,       or 3] for
 *                       [legacy, (v)ht20, (v)ht40, or vht80]
 *      nss is [0, 1, 2, or 3] for the number of streams 1~4.
 */
} wmi_pdev_tpc_config_event_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_div_rssi_antid_event_fixed_param */
    A_UINT32 tlv_header;
    /** how many elements in the MAX_CHAINS arrays below contain valid info */
    A_UINT32 num_chains_valid;
    /** RSSI (rssi_chain_x_pri20) on each chain (units: dB above noise floor) */
    A_UINT32 chain_rssi[WMI_MAX_CHAINS];
    /** index of the last-used antenna for each chain */
    A_UINT32 ant_id[WMI_MAX_CHAINS];
    /** mac address of diversity peer */
    wmi_mac_addr macaddr;
    /* EVM value for stream0 and stream1 20Mhz, dB units */
    A_INT32 chain_evm[WMI_MAX_CHAINS];
} wmi_pdev_div_rssi_antid_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header;                /* WMITLV_TAG_STRUC_wmi_pdev_bss_chan_info_event_fixed_param */
    A_UINT32 freq;                      /* Units in MHz */
    A_INT32 noise_floor;                /* units are dBm */

    /* rx clear - how often the channel was unused */
    A_UINT32 rx_clear_count_low;        /* low 31 bits of rx_clear cnt in 64bits format */
    A_UINT32 rx_clear_count_high;       /* high 31 bits of rx_clear cnt in 64bits format */

    /* cycle count - elapsed time during the measured period, in clock ticks */
    A_UINT32 cycle_count_low;           /* low 31 bits of cycle cnt in 64bits format */
    A_UINT32 cycle_count_high;          /* high 31 bits of cycle cnt in 64bits format */

    /* tx cycle count - elapsed time spent in tx, in clock ticks */
    A_UINT32 tx_cycle_count_low;        /* low 31 bits of tx_cycle cnt in 64bits format */
    A_UINT32 tx_cycle_count_high;       /* high 31 bits of tx_cycle cnt in 64bits format */

    /* rx cycle count - elapsed time spent in rx, in clock ticks */
    A_UINT32 rx_cycle_count_low;        /* low 31 bits of rx_cycle cnt in 64bits format */
    A_UINT32 rx_cycle_count_high;       /* high 31 bits of rx_cycle cnt in 64bits format */

    A_UINT32 rx_bss_cycle_count_low;    /* low 31 bits of rx cycle cnt for my bss in 64bits format */
    A_UINT32 rx_bss_cycle_count_high;   /* high 31 bits of rx_cycle cnt for my bss in 64bits format */
} wmi_pdev_bss_chan_info_event_fixed_param;

typedef struct {
    /* WMI event response update ctltable request to host */
    A_UINT32 tlv_header;                /* WMITLV_TAG_STRUC_wmi_pdev_update_ctltable_event_fixed_param */
    A_UINT32 total_len;                 /* the total number of bytes to be transferred */
    A_UINT32 len;                       /* the number of FW received bytes from host */
    A_UINT32 seq;                       /* the number of current transfers */
} wmi_pdev_update_ctltable_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_l1ss_track_event_fixed_param  */
    A_UINT32 periodCnt;
    A_UINT32 L1Cnt;
    A_UINT32 L11Cnt;
    A_UINT32 L12Cnt;
    A_UINT32 L1Entry;
    A_UINT32 L11Entry;
    A_UINT32 L12Entry;
    /** pdev_id for identifying the MAC
      * See macros starting with WMI_PDEV_ID_ for values.
      */
    A_UINT32 pdev_id;
} wmi_pdev_l1ss_track_event_fixed_param;

typedef struct {
    A_UINT32 len;
    A_UINT32 msgref;
    A_UINT32 segmentInfo;
} wmi_pdev_seg_hdr_info;


/*
 * Transmit power scale factor.
 *
 */
typedef enum {
    WMI_TP_SCALE_MAX    = 0,        /* no scaling (default) */
    WMI_TP_SCALE_50     = 1,        /* 50% of max (-3 dBm) */
    WMI_TP_SCALE_25     = 2,        /* 25% of max (-6 dBm) */
    WMI_TP_SCALE_12     = 3,        /* 12% of max (-9 dBm) */
    WMI_TP_SCALE_MIN    = 4,        /* min, but still on   */
    WMI_TP_SCALE_SIZE   = 5,        /* max num of enum     */
} WMI_TP_SCALE;

#define WMI_MAX_DEBUG_MESG (sizeof(A_UINT32) * 32)

typedef struct {
   /** message buffer, NULL terminated */
   char bufp[WMI_MAX_DEBUG_MESG];
} wmi_debug_mesg_event;

enum {
    /** P2P device */
    VDEV_SUBTYPE_P2PDEV=0,
    /** P2P client */
    VDEV_SUBTYPE_P2PCLI,
    /** P2P GO */
    VDEV_SUBTYPE_P2PGO,
    /** BT3.0 HS */
    VDEV_SUBTYPE_BT,
};

typedef struct {
    /** idnore power , only use flags , mode and freq */
   wmi_channel  chan;
}    wmi_pdev_set_channel_cmd;

typedef enum {
    WMI_PKTLOG_EVENT_RX  = 0x1,
    WMI_PKTLOG_EVENT_TX  = 0x2,
    WMI_PKTLOG_EVENT_RCF = 0x4, /* Rate Control Find */
    WMI_PKTLOG_EVENT_RCU = 0x8, /* Rate Control Update */
    /* 0x10 used by deprecated DBG_PRINT */
    WMI_PKTLOG_EVENT_SMART_ANTENNA = 0x20, /* To support Smart Antenna */
    WMI_PKTLOG_EVENT_SW = 0x40, /* To support SW defined events */
} WMI_PKTLOG_EVENT;

typedef enum {
    WMI_PKTLOG_ENABLE_AUTO  = 0, /* (default) FW will decide under what conditions to enable pktlog */
    WMI_PKTLOG_ENABLE_FORCE = 1, /* pktlog unconditionally enabled */
} WMI_PKTLOG_ENABLE;

typedef enum {
    WMI_PKTLOG_FILTER_IN  = 0, /* capture only for the MAC addresses in pktlog_mac_addr_list*/
    WMI_PKTLOG_FILTER_OUT = 1, /* capture for all MAC addresses except those in pktlog_mac_addr_list */
} WMI_PKTLOG_FILTER_TYPE;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_pktlog_enable_cmd_fixed_param */
    /** pdev_id for identifying the MAC
      * See macros starting with WMI_PDEV_ID_ for values.
      */
    A_UINT32 pdev_id;
    A_UINT32 evlist; /* WMI_PKTLOG_EVENT */
    A_UINT32 enable; /* WMI_PKTLOG_ENABLE */
} wmi_pdev_pktlog_enable_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_pktlog_disable_cmd_fixed_param */
    /** pdev_id for identifying the MAC
      * See macros starting with WMI_PDEV_ID_ for values.
      */
    A_UINT32 pdev_id;
} wmi_pdev_pktlog_disable_cmd_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
    *  WMITLV_TAG_STRUC_wmi_pdev_pktlog_filter_info */
    A_UINT32 tlv_header;
    /** mac addr of the peer to be filtered */
    wmi_mac_addr peer_mac_address;
} wmi_pdev_pktlog_filter_info;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_pktlog_filter_cmd_fixed_param */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
    /** 0 - disable filtering, 1 - enable filtering */
    A_UINT32 enable;
    A_UINT32 filter_type; /* WMI_PKTLOG_FILTER_TYPE */
    A_UINT32 num_of_mac_addresses;
    /* This TLV is followed by another TLV of array of structs
     * wmi_pdev_pktlog_filter_info pdev_pktlog_filter_info[];
     */
} wmi_pdev_pktlog_filter_cmd_fixed_param;

typedef enum {
    WMI_ROGUE_AP_ON_STA_PS  = 1, /* rogue ap on sta ps module */
} WMI_ROGUE_AP_TYPE;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_rap_config_fixed_param */
    /** pdev_id for identifying the MAC, the default value is WMI_PDEV_ID_SOC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;

    A_UINT32 type;
    /** Enable detection of rogue ap in the ps module
     *
     * 0 -> disabled
     * 1 -> enabled (default)
     */
    A_UINT32 sta_ps_detection_enabled;
/* This TLV is followed by rap_param for each rogue ap:
 *     wmi_pdev_set_rap_config_on_sta_ps_tlv_param rap_param[];
 */
} wmi_pdev_set_rap_config_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_rap_config_on_sta_ps_tlv_param */
    /** bssid of rogue ap */
    wmi_mac_addr bssid;
} wmi_pdev_set_rap_config_on_sta_ps_tlv_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_mib_stats_enable_cmd_fixed_param */
    /** pdev_id for identifying the MAC
      * See macros starting with WMI_PDEV_ID_ for values.
      */
    A_UINT32 pdev_id;
    A_UINT32 enable_Mib; /** enable for mib stats collection. Stats are delivered to host in wmi_mib_stats structure.
                          * If enable_Mib=1, stats collection is enabled. If enable_Mib=0, stats collection does not happen */
} wmi_mib_stats_enable_cmd_fixed_param;

/** Customize the DSCP (bit) to TID (0-7) mapping for QOS.
 *  NOTE: This constant cannot be changed without breaking
 *  WMI Compatibility. */

#define WMI_DSCP_MAP_MAX    (64)
    /*
     * @brief dscp_tid_map_cmdid - command to send the dscp to tid map to the target
     * @details
     * Create an API for sending the custom DSCP-to-TID map to the target
     * If this is a request from the user space or from above the UMAC
     * then the best place to implement this is in the umac_if_offload of the OL path.
     * Provide a place holder for this API in the ieee80211com (ic).
     *
     * This API will be a function pointer in the ieee80211com (ic). Any user space calls for manually setting the DSCP-to-TID mapping
     * in the target should be directed to the function pointer in the ic.
     *
     * Implementation details of the API to send the map to the target are as described-
     *
     * 1. The function will have 2 arguments- struct ieee80211com, DSCP-to-TID map.
     *    DSCP-to-TID map is a one dimensional u_int32_t array of length 64 to accomodate
     *    64 TID values for 2^6 (64) DSCP ids.
     *    Example:
     *      A_UINT32 dscp_tid_map[WMI_DSCP_MAP_MAX] = {
	 *									0, 0, 0, 0, 0, 0, 0, 0,
	 *									1, 1, 1, 1, 1, 1, 1, 1,
	 *									2, 2, 2, 2, 2, 2, 2, 2,
	 *									3, 3, 3, 3, 3, 3, 3, 3,
	 *									4, 4, 4, 4, 4, 4, 4, 4,
	 *									5, 5, 5, 5, 5, 5, 5, 5,
	 *									6, 6, 6, 6, 6, 6, 6, 6,
	 *									7, 7, 7, 7, 7, 7, 7, 7,
	 *								  };
     *
     * 2. Request for the WMI buffer of size equal to the size of the DSCP-to-TID map.
     *
     * 3. Copy the DSCP-to-TID map into the WMI buffer.
     *
     * 4. Invoke the wmi_unified_cmd_send to send the cmd buffer to the target with the
     *    WMI_PDEV_SET_DSCP_TID_MAP_CMDID. Arguments to the wmi send cmd API
     *    (wmi_unified_send_cmd) are wmi handle, cmd buffer, length of the cmd buffer and
     *    the WMI_PDEV_SET_DSCP_TID_MAP_CMDID id.
     *
     */

/* DEPRECATED - use VDEV level command instead
 * (wmi_vdev_set_dscp_tid_map_cmd_fixed_param)
 */
typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_dscp_tid_map_cmd_fixed_param */
    A_UINT32 reserved0;      /** placeholder for pdev_id of future multiple MAC products. Init. to 0. */
 /* map indicating DSCP to TID conversion */
    A_UINT32 dscp_to_tid_map[WMI_DSCP_MAP_MAX];
} wmi_pdev_set_dscp_tid_map_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_set_dscp_tid_map_cmd_fixed_param */
    A_UINT32 vdev_id;
    /** map indicating DSCP to TID conversion */
    A_UINT32 dscp_to_tid_map[WMI_DSCP_MAP_MAX];
    A_UINT32 enable_override;
} wmi_vdev_set_dscp_tid_map_cmd_fixed_param;

enum WMI_WAKE_GPIO_TYPE {
    WMI_WAKE_GPIO_LOW = 1,
    WMI_WAKE_GPIO_HIGH = 2,
    WMI_WAKE_GPIO_RISING_EDGE = 3,
    WMI_WAKE_GPIO_FALLING_EDGE = 4,
};

/**
 * Set GPIO numbers used to wakeup host and wakeup target.
 */
typedef struct {
    /**
     * TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_WMI_PDEV_SET_WAKEUP_CONFIG_CMDID_fixed_param
     */
    A_UINT32 tlv_header;
    /* gpio num used to wakeup host, 0xff disable wakeup gpio */
    A_UINT32 host_wakeup_gpio;
    /* refer to WMI_WAKE_GPIO_TYPE */
    A_UINT32 host_wakeup_type;
    /* gpio num used to wakeup target, 0xff disable wakeup gpio */
    A_UINT32 target_wakeup_gpio;
    /* refer to WMI_WAKE_GPIO_TYPE */
    A_UINT32 target_wakeup_type;
} WMI_PDEV_SET_WAKEUP_CONFIG_CMDID_fixed_param;

/** Fixed rate (rate-code) for broadcast/ multicast data frames */
/* @brief bcast_mcast_data_rate - set the rates for the bcast/ mcast frames
 * @details
 * Create an API for setting the custom rate for the MCAST and BCAST frames
 * in the target. If this is a request from the user space or from above the UMAC
 * then the best place to implement this is in the umac_if_offload of the OL path.
 * Provide a place holder for this API in the ieee80211com (ic).
 *
 * Implementation details of the API to set custom rates for MCAST and BCAST in
 * the target are as described-
 *
 * 1. The function will have 3 arguments-
 *    vap structure,
 *    MCAST/ BCAST identifier code,
 *    8 bit rate code
 *
 * The rate-code is a 1-byte field in which:for given rate, nss and preamble
 * b'7-b-6 indicate the preamble (0 OFDM, 1 CCK, 2, HT, 3 VHT)
 * b'5-b'4 indicate the NSS (0 - 1x1, 1 - 2x2, 2 - 3x3)
 * b'3-b'0 indicate the rate, which is indicated as follows:
 *          OFDM :     0: OFDM 48 Mbps
 *                     1: OFDM 24 Mbps
 *                     2: OFDM 12 Mbps
 *                     3: OFDM 6 Mbps
 *                     4: OFDM 54 Mbps
 *                     5: OFDM 36 Mbps
 *                     6: OFDM 18 Mbps
 *                     7: OFDM 9 Mbps
 *         CCK (pream == 1)
 *                     0: CCK 11 Mbps Long
 *                     1: CCK 5.5 Mbps Long
 *                     2: CCK 2 Mbps Long
 *                     3: CCK 1 Mbps Long
 *                     4: CCK 11 Mbps Short
 *                     5: CCK 5.5 Mbps Short
 *                     6: CCK 2 Mbps Short
 *         HT/VHT (pream == 2/3)
 *                     0..7: MCS0..MCS7 (HT)
 *                     0..9: MCS0..MCS9 (VHT)
 *
 * 2. Invoke the wmi_unified_vdev_set_param_send to send the rate value
 *    to the target.
 *    Arguments to the API are-
 *    wmi handle,
 *    VAP interface id (av_if_id) defined in ol_ath_vap_net80211,
 *    WMI_VDEV_PARAM_BCAST_DATA_RATE/ WMI_VDEV_PARAM_MCAST_DATA_RATE,
 *    rate value.
 */
typedef enum {
    WMI_SET_MCAST_RATE,
    WMI_SET_BCAST_RATE
} MCAST_BCAST_RATE_ID;

typedef struct {
    MCAST_BCAST_RATE_ID rate_id;
    A_UINT32 rate;
} mcast_bcast_rate;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_wmm_params */
    A_UINT32 cwmin;
    A_UINT32 cwmax;
    A_UINT32 aifs;
    A_UINT32 txoplimit;
    A_UINT32 acm;
    A_UINT32 no_ack;
} wmi_wmm_params;

/* DEPRECATED - use VDEV level command instead
 * (wmi_vdev_set_wmm_params_cmd_fixed_param)
 */

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_wmm_params_cmd_fixed_param */
    A_UINT32 reserved0;      /** placeholder for pdev_id of future multiple MAC products. Init. to 0. */
    A_UINT32 dg_type;

    /* The TLVs for the 4 AC follows:
     *     wmi_wmm_params wmm_params_ac_be;
     *     wmi_wmm_params wmm_params_ac_bk;
     *     wmi_wmm_params wmm_params_ac_vi;
     *     wmi_wmm_params wmm_params_ac_vo;
     */
} wmi_pdev_set_wmm_params_cmd_fixed_param;

typedef enum {
    WMI_REQUEST_PEER_STAT           = 0x0001,
    WMI_REQUEST_AP_STAT             = 0x0002,
    WMI_REQUEST_PDEV_STAT           = 0x0004,
    WMI_REQUEST_VDEV_STAT           = 0x0008,
    WMI_REQUEST_BCNFLT_STAT         = 0x0010,
    WMI_REQUEST_VDEV_RATE_STAT      = 0x0020,
    WMI_REQUEST_INST_STAT           = 0x0040,
    WMI_REQUEST_MIB_STAT            = 0x0080,
    WMI_REQUEST_RSSI_PER_CHAIN_STAT = 0x0100,
    WMI_REQUEST_CONGESTION_STAT     = 0x0200,
    WMI_REQUEST_PEER_EXTD_STAT      = 0x0400,
    WMI_REQUEST_BCN_STAT            = 0x0800,
    WMI_REQUEST_BCN_STAT_RESET      = 0x1000,
    WMI_REQUEST_PEER_EXTD2_STAT     = 0x2000,
} wmi_stats_id;

/*
 * cfg_retry_count is set to max number of times the AP should try sending
 * QoS Null frames to the STA for measuring the instantaneous RSSI
 *
 * retry_count is used to maintain the number of times the AP has tried sending
 * QoS Null frames to the STA for measuring the instantaneous RSSI
 */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_request_stats_cmd_sub_struc_param */
    A_UINT32 cfg_retry_count;
    A_UINT32 retry_count;
} wmi_inst_rssi_stats_params;

typedef struct {
    A_UINT32   tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_request_stats_cmd_fixed_param */
    wmi_stats_id stats_id;
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    A_UINT32 pdev_id; /** pdev_id for identifying the MAC.  See macros starting with WMI_PDEV_ID_ for values. In non-DBDC case host should set it to 0. */
/*
 * This TLV is (optionally) followed by other TLVs:
 *     wmi_inst_rssi_stats_params inst_rssi_params;
 */
} wmi_request_stats_cmd_fixed_param;

/* stats type bitmap  */
#define WMI_LINK_STATS_RADIO         0x00000001
#define WMI_LINK_STATS_IFACE         0x00000002
#define WMI_LINK_STATS_ALL_PEER      0x00000004
#define WMI_LINK_STATS_PER_PEER      0x00000008


/* wifi clear statistics bitmap  */
#define WIFI_STATS_RADIO              0x00000001 /** all radio statistics */
#define WIFI_STATS_RADIO_CCA          0x00000002 /** cca_busy_time (within radio statistics) */
#define WIFI_STATS_RADIO_CHANNELS     0x00000004 /** all channel statistics (within radio statistics) */
#define WIFI_STATS_RADIO_SCAN         0x00000008 /** all scan statistics (within radio statistics) */
#define WIFI_STATS_IFACE              0x00000010 /** all interface statistics */
#define WIFI_STATS_IFACE_TXRATE       0x00000020 /** all tx rate statistics (within interface statistics) */
#define WIFI_STATS_IFACE_AC           0x00000040 /** all ac statistics (within interface statistics) */
#define WIFI_STATS_IFACE_CONTENTION   0x00000080 /** all contention (min, max, avg) statistics (within ac statisctics) */
#define WMI_STATS_IFACE_ALL_PEER      0x00000100 /** All peer stats on this interface */
#define WMI_STATS_IFACE_PER_PEER      0x00000200 /** Clear particular peer stats depending on the peer_mac */

/** Default value for stats if the stats collection has not started */
#define WMI_STATS_VALUE_INVALID       0xffffffff

#define WMI_DIAG_ID_GET(diag_events_logs)                         WMI_GET_BITS(diag_events_logs, 0, 16)
#define WMI_DIAG_ID_SET(diag_events_logs, value)                  WMI_SET_BITS(diag_events_logs, 0, 16, value)
#define WMI_DIAG_TYPE_GET(diag_events_logs)                       WMI_GET_BITS(diag_events_logs, 16, 1)
#define WMI_DIAG_TYPE_SET(diag_events_logs, value)                WMI_SET_BITS(diag_events_logs, 16, 1, value)
#define WMI_DIAG_ID_ENABLED_DISABLED_GET(diag_events_logs)        WMI_GET_BITS(diag_events_logs, 17, 1)
#define WMI_DIAG_ID_ENABLED_DISABLED_SET(diag_events_logs, value) WMI_SET_BITS(diag_events_logs, 17, 1, value)

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_diag_event_log_config_fixed_param */
    A_UINT32 num_of_diag_events_logs;
/* The TLVs will follow.
 *    A_UINT32 diag_events_logs_list[]; 0-15 Bits Diag EVENT/LOG ID,
 *                                      Bit 16 - DIAG type EVENT/LOG, 0 - Event, 1 - LOG
 *                                      Bit 17 Indicate if the DIAG type is Enabled/Disabled.
 */
} wmi_diag_event_log_config_fixed_param;

#define WMI_DIAG_FREQUENCY_GET(diag_events_logs)          WMI_GET_BITS(diag_events_logs, 17, 1)
#define WMI_DIAG_FREQUENCY_SET(diag_events_logs, value)   WMI_SET_BITS(diag_events_logs, 17, 1, value)
#define WMI_DIAG_EXT_FEATURE_GET(diag_events_logs)        WMI_GET_BITS(diag_events_logs, 18, 1)
#define WMI_DIAG_EXT_FEATURE_SET(diag_events_logs, value) WMI_SET_BITS(diag_events_logs, 18, 1, value)

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 num_of_diag_events_logs;
/* The TLVs will follow.
 *    A_UINT32 diag_events_logs_list[]; 0-15 Bits Diag EVENT/LOG ID,
 *                                      Bit 16 - DIAG type EVENT/LOG, 0 - Event, 1 - LOG
 *                                      Bit 17 - Frequncy of the DIAG EVENT/LOG High Frequency -1, Low Frequency - 0
 *                                      Bit 18 - Set if the EVENTS/LOGs are used for EXT DEBUG Framework
 */
} wmi_diag_event_log_supported_event_fixed_params;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_debug_mesg_flush_fixed_param*/
    A_UINT32 reserved0; /** placeholder for future */
} wmi_debug_mesg_flush_fixed_param;

typedef struct {
    A_UINT32 tlv_header;        /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_debug_mesg_fw_data_stall_param */
    A_UINT32 vdev_id_bitmap;    /** bitmap representation for vdev_id's where data stall happened */
    A_UINT32 data_stall_type;   /** wlan_dbg_data_stall_type_e */
    /** reason_code1:
     * The information stored in reason_code1 varies based on the data stally
     * type values:
     * data_stall_type      | reason_code1
     * -----------------------------------------------------
     * HWSCHED_CMD_FLUSH    | flush req reason (0-40)
     * RX_REFILL_FAILED     | ring_id (0-7)
     * RX_FCS_LEN_ERROR     | exact error type
     */
    A_UINT32 reason_code1;
    A_UINT32 reason_code2;      /** on which tid/hwq stall happened */
    A_UINT32 recovery_type;     /** wlan_dbg_data_stall_recovery_type_e */
} wmi_debug_mesg_fw_data_stall_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_debug_mesg_flush_complete_fixed_param*/
    A_UINT32 reserved0; /** placeholder for future */
} wmi_debug_mesg_flush_complete_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_rssi_breach_fixed_param */
    /* vdev_id, where RSSI breach event occurred */
    A_UINT32 vdev_id;
    /* request id */
    A_UINT32 request_id;
    /* bitmap[0-2] is corresponding to low_rssi[0-2]. bitmap[3-5] is corresponding to hi_rssi[0-2]*/
    A_UINT32 event_bitmap;
    /* rssi at the time of RSSI breach. Unit dBm */
    A_UINT32 rssi;
    /* bssid of the monitored AP's */
    wmi_mac_addr bssid;
} wmi_rssi_breach_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_fw_mem_dump */
    /** unique id identifying the segment */
    A_UINT32 seg_id;
    /** Start address of the segment to be read */
    A_UINT32 seg_start_addr_lo;
    A_UINT32 seg_start_addr_hi;
    /** Length of the segment to be read */
    A_UINT32 seg_length;
    /** Host bufeer address to which the segment will be read and dumped */
    A_UINT32 dest_addr_lo;
    A_UINT32 dest_addr_hi;
} wmi_fw_mem_dump;

/* Command to get firmware memory dump*/
typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_get_fw_mem_dump_fixed_param */
    /** unique id identifying the request */
    A_UINT32 request_id;
    /** number of memory dump segments */
    A_UINT32 num_fw_mem_dump_segs;
/**
 * This TLV is followed by another TLV
 *     wmi_fw_mem_dump fw_mem_dump[];
 */
} wmi_get_fw_mem_dump_fixed_param;

/** Event to indicate the completion of fw mem dump */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_update_fw_mem_dump_fixed_param */
    /** unique id identifying the request, given in the request stats command */
    A_UINT32 request_id;
    /*In case of Firmware memory dump */
    A_UINT32 fw_mem_dump_complete;
} wmi_update_fw_mem_dump_fixed_param;

typedef enum {
    WMI_ROAMING_IDLE = 0,
    WMI_ROAMING_ACTIVE = 1,
} wmi_roam_state;

/* access categories */
typedef enum {
   WMI_AC_VO  = 0,
   WMI_AC_VI  = 1,
   WMI_AC_BE  = 2,
   WMI_AC_BK  = 3,
   WMI_AC_MAX = 4,
} wmi_traffic_ac;

typedef enum {
    WMI_STA_STATS = 0,
    WMI_SOFTAP_STATS = 1,
    WMI_IBSS_STATS = 2,
    WMI_P2P_CLIENT_STATS = 3,
    WMI_P2P_GO_STATS = 4,
    WMI_NAN_STATS = 5,
    WMI_MESH_STATS = 6,
} wmi_link_iface_type;

/* channel operating width */
typedef enum {
    WMI_CHAN_WIDTH_20    = 0,
    WMI_CHAN_WIDTH_40    = 1,
    WMI_CHAN_WIDTH_80    = 2,
    WMI_CHAN_WIDTH_160   = 3,
    WMI_CHAN_WIDTH_80P80 = 4,
    WMI_CHAN_WIDTH_5     = 5,
    WMI_CHAN_WIDTH_10    = 6,
} wmi_channel_width;

/*Clear stats*/
typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_clear_link_stats_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** stop_stats_collection_req = 1 will imply stop the statistics collection */
    A_UINT32 stop_stats_collection_req;
    /** identifies what stats to be cleared */
    A_UINT32 stats_clear_req_mask;
    /** identifies which peer stats to be cleared. Valid only while clearing PER_REER */
    wmi_mac_addr peer_macaddr;
} wmi_clear_link_stats_cmd_fixed_param;

/* Link Stats configuration params. Trigger the link layer statistics collection*/
typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_start_link_stats_cmd_fixed_param */
    /** threshold to classify the pkts as short or long */
    A_UINT32 mpdu_size_threshold;
    /** set for field debug mode. Driver should collect all statistics regardless of performance impact.*/
    A_UINT32 aggressive_statistics_gathering;
} wmi_start_link_stats_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_request_link_stats_cmd_fixed_param */
    /** Type of stats required. This is a bitmask WMI_LINK_STATS_RADIO, WMI_LINK_STATS_IFACE */
    A_UINT32 stats_type;
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** unique id identifying the request, generated by the caller */
    A_UINT32 request_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
} wmi_request_link_stats_cmd_fixed_param;

#define WLM_STATS_REQ_LINK          0x00000001
typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_request_wlm_stats_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** WLM event request bitmask
     * Used by host to customize the wlm stats report.
     * Filled with a combination of WLM_STATS_xxx values.
     */
    A_UINT32 request_bitmask;
} wmi_request_wlm_stats_cmd_fixed_param;

/* channel statistics */
typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_channel_stats */
    /** Channel width (20, 40, 80, 80+80, 160) enum wmi_channel_width*/
    A_UINT32 channel_width;
    /** Primary 20 MHz channel */
    A_UINT32 center_freq;
    /** center frequency (MHz) first segment */
    A_UINT32 center_freq0;
    /** center frequency (MHz) second segment */
    A_UINT32 center_freq1;
    /** msecs the radio is awake (32 bits number accruing over time) */
    A_UINT32 radio_awake_time;
    /** msecs the CCA register is busy (32 bits number accruing over time) */
    A_UINT32 cca_busy_time;
} wmi_channel_stats;

/*
 * Each step represents 0.5 dB.  The starting value is 0 dBm.
 * Thus the TPC levels cover 0 dBm to 31.5 dBm inclusive in 0.5 dB steps.
 */
#define MAX_TPC_LEVELS 64

/* radio statistics */
typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_radio_link_stats */
    /** Wifi radio (if multiple radio supported) */
    A_UINT32 radio_id;
    /** msecs the radio is awake (32 bits number accruing over time) */
    A_UINT32 on_time;
    /** msecs the radio is transmitting (32 bits number accruing over time) */
    A_UINT32 tx_time;
    /** msecs the radio is in active receive (32 bits number accruing over time) */
    A_UINT32 rx_time;
    /** msecs the radio is awake due to all scan (32 bits number accruing over time) */
    A_UINT32 on_time_scan;
    /** msecs the radio is awake due to NAN (32 bits number accruing over time) */
    A_UINT32 on_time_nbd;
    /** msecs the radio is awake due to G?scan (32 bits number accruing over time) */
    A_UINT32 on_time_gscan;
    /** msecs the radio is awake due to roam?scan (32 bits number accruing over time) */
    A_UINT32 on_time_roam_scan;
    /** msecs the radio is awake due to PNO scan (32 bits number accruing over time) */
    A_UINT32 on_time_pno_scan;
    /** msecs the radio is awake due to HS2.0 scans and GAS exchange (32 bits number accruing over time) */
    A_UINT32 on_time_hs20;
    /** number of channels */
    A_UINT32 num_channels;
    /** tx time per TPC level - DEPRECATED
     * This field is deprecated.
     * It is superseded by the WMI_RADIO_TX_POWER_LEVEL_STATS_EVENTID message.
     */
    A_UINT32 tx_time_per_tpc[MAX_TPC_LEVELS];
    /** msecs the radio is awake due to Host initiated scan (accruing over time) */
    A_UINT32 on_time_host_scan;
    /** msecs the radio is awake due to LPI scan (accruing over time) */
    A_UINT32 on_time_lpi_scan;
} wmi_radio_link_stats;

/** tx time per power level statistics */
typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tx_power_level_stats_evt_fixed_param */
    /** total number of tx power levels */
    A_UINT32 total_num_tx_power_levels;
    /** number of tx power levels that are carried in this event */
    A_UINT32 num_tx_power_levels;
    /** offset of current stats
     * If ((num_tx_power_levels + power_level_offset)) ==
     *     total_num_tx_power_levels)
     * this message completes the report of tx time per power levels.
     * Otherwise, additional WMI_RADIO_TX_POWER_LEVEL_STATS_EVENTID messages
     * will be sent by the target to deliver the remainder of the tx time
     * per power level stats.
     */
    A_UINT32 power_level_offset;
    /* radio id for this tx time per power level statistics (if multiple radio supported) */
    A_UINT32 radio_id;
/*
 * This TLV will be followed by a TLV containing a variable-length array of
 * A_UINT32 with tx time per power level data
 *  A_UINT32 tx_time_per_power_level[num_tx_power_levels]
 * The tx time is in units of milliseconds.
 * The power levels are board-specific values; a board-specific translation
 * has to be applied to determine what actual power corresponds to each
 * power level.
 * Just as the host has a BDF file available, the host should also have
 * a data file available that provides the power level to power translations.
 */
} wmi_tx_power_level_stats_evt_fixed_param;

/** Radio statistics (once started) do not stop or get reset unless wifi_clear_link_stats is invoked */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_stats_event_fixed_param */
    /** unique id identifying the request, given in the request stats command */
    A_UINT32 request_id;
    /** Number of radios*/
    A_UINT32 num_radio;
    /** more_data will be set depending on the number of radios */
    A_UINT32 more_radio_events;
/*
 * This TLV is followed by another TLV of array of bytes
 *   size of(struct wmi_radio_link_stats);
 *
 * This TLV is followed by another TLV of array of bytes
 *   num_channels * size of(struct wmi_channel_stats)
 */

} wmi_radio_link_stats_event_fixed_param;

/* per rate statistics */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_rate_stats */
    /** rate information
     * The rate-code is a 1-byte field in which:for given rate, nss and preamble
     * b'7-b-6 indicate the preamble (0 OFDM, 1 CCK, 2, HT, 3 VHT)
     * b'5-b'4 indicate the NSS (0 - 1x1, 1 - 2x2, 2 - 3x3)
     * b'3-b'0 indicate the rate, which is indicated as follows:
     *          OFDM :     0: OFDM 48 Mbps
     *                     1: OFDM 24 Mbps
     *                     2: OFDM 12 Mbps
     *                     3: OFDM 6 Mbps
     *                     4: OFDM 54 Mbps
     *                     5: OFDM 36 Mbps
     *                     6: OFDM 18 Mbps
     *                     7: OFDM 9 Mbps
     *         CCK (pream == 1)
     *                     0: CCK 11 Mbps Long
     *                     1: CCK 5.5 Mbps Long
     *                     2: CCK 2 Mbps Long
     *                     3: CCK 1 Mbps Long
     *                     4: CCK 11 Mbps Short
     *                     5: CCK 5.5 Mbps Short
     *                     6: CCK 2 Mbps Short
     *         HT/VHT (pream == 2/3)
     *                     0..7: MCS0..MCS7 (HT)
     *                     0..9: MCS0..MCS9 (VHT)
     */
    A_UINT32 rate;
    /** units of 100 Kbps */
    A_UINT32 bitrate;
    /** number of successfully transmitted data pkts (ACK rcvd) */
    A_UINT32 tx_mpdu;
    /** number of received data pkts */
    A_UINT32 rx_mpdu;
    /** number of data packet losses (no ACK) */
    A_UINT32 mpdu_lost;
    /** total number of data pkt retries */
    A_UINT32 retries;
    /** number of short data pkt retries */
    A_UINT32 retries_short;
    /** number of long data pkt retries */
    A_UINT32 retries_long;
} wmi_rate_stats;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_link_stats */
    /** peer type (AP, TDLS, GO etc.) enum wmi_peer_type*/
    A_UINT32 peer_type;
    /** mac address */
    wmi_mac_addr peer_mac_address;
    /** peer wmi_CAPABILITY_XXX */
    A_UINT32 capabilities;
    /** number of rates */
    A_UINT32 num_rates;
} wmi_peer_link_stats;

/** PEER statistics (once started) reset and start afresh after each connection */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_stats_event_fixed_param */
    /** unique id identifying the request, given in the request stats command */
    A_UINT32 request_id;
    /** number of peers accomidated in this particular event  */
    A_UINT32 num_peers;
    /** Indicates the fragment number  */
    A_UINT32 peer_event_number;
    /** Indicates if there are more peers which will be sent as seperate peer_stats event */
    A_UINT32 more_data;

/**
 * This TLV is followed by another TLV
 * num_peers * size of(struct wmi_peer_stats)
 * num_rates * size of(struct wmi_rate_stats). num_rates is the sum of the rates of all the peers.
 */
} wmi_peer_stats_event_fixed_param;

/* per access category statistics */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_wmm_ac_stats */
    /** access category (VI, VO, BE, BK) enum wmi_traffic_ac*/
    A_UINT32 ac_type;
    /** number of successfully transmitted unicast data pkts (ACK rcvd) */
    A_UINT32 tx_mpdu;
    /** number of received unicast mpdus */
    A_UINT32 rx_mpdu;
    /** number of succesfully transmitted multicast data packets */
    /** STA case: implies ACK received from AP for the unicast packet in which mcast pkt was sent */
    A_UINT32 tx_mcast;
    /** number of received multicast data packets */
    A_UINT32 rx_mcast;
    /** number of received unicast a-mpdus */
    A_UINT32 rx_ampdu;
    /** number of transmitted unicast a-mpdus */
    A_UINT32 tx_ampdu;
    /** number of data pkt losses (no ACK) */
    A_UINT32 mpdu_lost;
    /** total number of data pkt retries */
    A_UINT32 retries;
    /** number of short data pkt retries */
    A_UINT32 retries_short;
    /** number of long data pkt retries */
    A_UINT32 retries_long;
    /** data pkt min contention time (usecs) */
    A_UINT32 contention_time_min;
    /** data pkt max contention time (usecs) */
    A_UINT32 contention_time_max;
    /** data pkt avg contention time (usecs) */
    A_UINT32 contention_time_avg;
    /** num of data pkts used for contention statistics */
    A_UINT32 contention_num_samples;
    /** number of pending pkts */
    A_UINT32 tx_pending_msdu;
} wmi_wmm_ac_stats;

/* interface statistics */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_iface_link_stats */
    /** access point beacon received count from connected AP */
    A_UINT32 beacon_rx;
    /** access point mgmt frames received count from connected AP (including Beacon) */
    A_UINT32 mgmt_rx;
    /** action frames received count */
    A_UINT32 mgmt_action_rx;
    /** action frames transmit count */
    A_UINT32 mgmt_action_tx;
    /** access Point Beacon and Management frames RSSI (averaged) */
    A_UINT32 rssi_mgmt;
    /** access Point Data Frames RSSI (averaged) from connected AP */
    A_UINT32 rssi_data;
    /** access Point ACK RSSI (averaged) from connected AP */
    A_UINT32 rssi_ack;
    /** number of peers */
    A_UINT32 num_peers;
    /** Indicates how many peer_stats events will be sent depending on the num_peers. */
    A_UINT32 num_peer_events;
    /** number of ac */
    A_UINT32 num_ac;
    /** Roaming Stat */
    A_UINT32 roam_state;
    /** Average Beacon spread offset is the averaged time delay between TBTT and beacon TSF */
    /** Upper 32 bits of averaged 64 bit beacon spread offset */
    A_UINT32 avg_bcn_spread_offset_high;
    /** Lower 32 bits of averaged 64 bit beacon spread offset */
    A_UINT32 avg_bcn_spread_offset_low;
    /** Takes value of 1 if AP leaks packets after sending an ACK for PM=1 otherwise 0 */
    A_UINT32 is_leaky_ap;
    /** Average number of frames received from AP after receiving the ACK for a frame with PM=1 */
    A_UINT32 avg_rx_frms_leaked;
    /** Rx leak watch window currently in force to minimize data loss because of leaky AP. Rx leak window is the
        time driver waits before shutting down the radio or switching the channel and after receiving an ACK for
        a data frame with PM bit set) */
    A_UINT32 rx_leak_window;
    A_UINT32 tx_rts_succ_cnt;
    A_UINT32 tx_rts_fail_cnt;
    A_UINT32 tx_ppdu_succ_cnt;
    A_UINT32 tx_ppdu_fail_cnt;
    /** msecs the interface is in Connected state (accruing over time) */
    A_UINT32 connected_duration;
    /** msecs the interface is in DisConnected state (accruing over time) */
    A_UINT32 disconnected_duration;
    /** msecs the interface is doing RTT ranging (accruing over time) */
    A_UINT32 rtt_ranging_duration;
    /** msecs the interface is in RTT responder mode (accruing over time) */
    A_UINT32 rtt_responder_duration;
    /** Number of Probes (Tx) sent on the interface (accruing over time) */
    A_UINT32 num_probes_tx;
    /** Number of Beacon misses on this interface (accruing over time) */
    A_UINT32 num_beacon_miss;
} wmi_iface_link_stats;

typedef enum {
    WMI_OFFLOAD_STATS_TYPE_SOC_BCAST   = 0,
    WMI_OFFLOAD_STATS_TYPE_SOC_MCAST   = 1,
    WMI_OFFLOAD_STATS_TYPE_SOC_UCAST   = 2,
    WMI_OFFLOAD_STATS_TYPE_ARP         = 3,
    WMI_OFFLOAD_STATS_TYPE_NS          = 4,
    WMI_OFFLOAD_STATS_TYPE_APF_BCAST   = 5,
    WMI_OFFLOAD_STATS_TYPE_APF_MCAST   = 6,
    WMI_OFFLOAD_STATS_TYPE_APF_UCAST   = 7,
    /* Add New offload stat type here */
    WMI_OFFLOAD_STATS_TYPE_MAX,
} wmi_offload_stats_type;

typedef struct {
   A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_iface_offload_stats */
   /** Type of offload stat. enum wmi_offload_stats_type **/
   A_UINT32 type;
   /** Number of (MSDUs) frames Received **/
   A_UINT32 rx_count;
   /** Number of frames Dropped **/
   A_UINT32 drp_count;
   /** Number of frames for which FW Responded (Valid for ARP and NS only). (or)
     * Number of frames forwarded to Host (Valid for stats type except ARP and NS). **/
   A_UINT32 fwd_count;
} wmi_iface_offload_stats;

/** Interface statistics (once started) reset and start afresh after each connection */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_iface_link_stats_event_fixed_param */
    /** unique id identifying the request, given in the request stats command */
    A_UINT32 request_id;
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** Number of offload stats **/
    A_UINT32 num_offload_stats;
/*
 * This TLV is followed by other TLVs
 *   wmi_iface_link_stats iface_link_stats;
 *   num_ac * size of(struct wmi_wmm_ac_stats)
 *   wmi_iface_offload_stats iface_offload_stats[num_offload_stats]
 */
} wmi_iface_link_stats_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_wlm_stats_event_fixed_param */
    /** bitmask listing which WLM stats are provided.
     * Copied from wlm_stats_cmd_fixed_param.
     * (This field is provided for convenience rather than necessity, since
     * the recipient can directly check which TLV arrays have non-zero length.)
     */
    A_UINT32 request_bitmask;
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /**
     * timestamp of event report, in microseconds units
     * This timestamp is for debugging purposes only.
     * It can be used to correlate this WLM stats event data with
     * other WLM information uploaded through other means.
     */
    A_UINT32 timestamp;
    /**
     * Interval between two consecutive WLM stats query requests,
     * in microseconds units.
     * This interval is used for converting the scan_period and pwr_on_period
     * values from within wmi_wlm_link_stats from percentage units to time
     * units.
     */
    A_UINT32 req_interval;
/*
 * This TLV is followed by an A_UINT32 array TLV carrying an opaque payload.
 */
} wmi_wlm_stats_event_fixed_param;

/** Suspend option */
enum {
    WMI_PDEV_SUSPEND, /* suspend */
    WMI_PDEV_SUSPEND_AND_DISABLE_INTR, /* suspend and disable all interrupts */
};

typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_suspend_cmd_fixed_param  */
    /* suspend option sent to target */
    A_UINT32 pdev_id; /** pdev_id for identifying the MAC, See macros starting with WMI_PDEV_ID_ for values. */
    A_UINT32 suspend_opt;
} wmi_pdev_suspend_cmd_fixed_param;

typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_resume_cmd_fixed_param  */
    A_UINT32 pdev_id; /** pdev_id for identifying the MAC, See macros starting with WMI_PDEV_ID_ for values. */
} wmi_pdev_resume_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_rate_stats_event_fixed_param,  */
    A_UINT32 num_vdev_stats; /* number of vdevs */
}wmi_vdev_rate_stats_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len, tag equals WMITLV_TAG_STRUC_wmi_vdev_rate_ht_info*/
    A_UINT32 vdevid; /* Id of the wlan vdev*/
    A_UINT32 tx_nss; /* Bit 28 of tx_rate_kbps has this info - based on last data packet transmitted*/
    A_UINT32 rx_nss; /* Bit 24 of rx_rate_kbps - same as above*/
    A_UINT32 tx_preamble; /* Bits 30-29 from tx_rate_kbps */
    A_UINT32 rx_preamble; /* Bits 26-25 from rx_rate_kbps */
} wmi_vdev_rate_ht_info;

typedef struct {
	/**
	 * TLV tag and len, tag equals
	 * WMITLV_TAG_STRUC_wmi_rx_aggr_failure_event_fixed_param
	 */
	A_UINT32 tlv_header;
	A_UINT32 num_failure_info; /* How many holes on rx aggregation */
} wmi_rx_aggr_failure_event_fixed_param;

typedef struct {
	/**
	 * TLV tag and len, tag equals
	 * WMITLV_wmi_rx_aggr_failure_info
	 */
	A_UINT32 tlv_header;
	A_UINT32 start_seq; /* start sequence number of the hole */
	A_UINT32 end_seq; /* end sequence number of the hole */
} wmi_rx_aggr_failure_info;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tx_pn_request_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    A_UINT32 key_type; /* use standard cipher types - see WMI_CIPHER_ defs */
} wmi_peer_tx_pn_request_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tx_pn_response_event_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    A_UINT32 key_type; /* use standard cipher types - see WMI_CIPHER_ defs */
    /** Packet Number
    * The PN is provided in little endian order, with bits 7:0 of the PN
    * residing in pn[0].
    * The key_type indirectly specifies the packet number length, and thus
    * how many bytes within pn[] are filled with valid data.
    */
    A_UINT8 pn[16];
} wmi_peer_tx_pn_response_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_stats_event_fixed_param */
    wmi_stats_id stats_id;
    /** number of pdev stats event structures (wmi_pdev_stats) 0 or 1 */
    A_UINT32 num_pdev_stats;
    /** number of vdev stats event structures  (wmi_vdev_stats) 0 or max vdevs */
    A_UINT32 num_vdev_stats;
    /** number of peer stats event structures  (wmi_peer_stats) 0 or max peers */
    A_UINT32 num_peer_stats;
    A_UINT32 num_bcnflt_stats;
    /** number of chan stats event structures  (wmi_chan_stats) 0 to MAX MCC CHANS */
    A_UINT32 num_chan_stats;
    /** number of MIB stats event structures (wmi_mib_stats) */
    A_UINT32 num_mib_stats;
    A_UINT32 pdev_id; /** pdev_id for identifying the MAC.  See macros starting with WMI_PDEV_ID_ for values. In non-DBDC case host should set it to 0. */
    /** number of beacon stats event structures (wmi_bcn_stats) */
    A_UINT32 num_bcn_stats;
    /** number of extended peer stats event structures (wmi_peer_extd_stats) */
    A_UINT32 num_peer_extd_stats;
    /** number of extd2 peer stats event structures (wmi_peer_extd2_stats) */
    A_UINT32 num_peer_extd2_stats;
    /** last_event
     * The most significant bit is set to 1 to indicate whether the last_event
     * field contains valid data.  The least significant bit is set to 1 to
     * indicate this is the final WMI_STATS_EVENT in a series.
     */
    A_UINT32 last_event;

    /* This TLV is followed by another TLV of array of bytes
         *   A_UINT8 data[];
         *  This data array contains
         *   num_pdev_stats * size of(struct wmi_pdev_stats)
         *   num_vdev_stats * size of(struct wmi_vdev_stats)
         *   num_peer_stats * size of(struct wmi_peer_stats)
         *   num_bcnflt_stats * size_of()
         *   num_chan_stats * size of(struct wmi_chan_stats)
         *   num_mib_stats * size of(struct wmi_mib_stats)
         *   num_bcn_stats * size of(struct wmi_bcn_stats)
         */
    /* If WMI_REQUEST_PEER_EXTD_STAT is set in stats_id,
         * the data[] array also contains num_peer_stats * size of wmi_peer_extd_stats
         * following the information elements listed above.
         */
} wmi_stats_event_fixed_param;

/* WLAN channel CCA stats bitmap  */
#define WLAN_STATS_IDLE_TIME_SHIFT             0
#define WLAN_STATS_IDLE_TIME_TIME              0x00000001

#define WLAN_STATS_TX_TIME_SHIFT               1
#define WLAN_STATS_TX_TIME_MASK                0x00000002

#define WLAN_STATS_RX_IN_BSS_TIME_SHIFT        2
#define WLAN_STATS_RX_IN_BSS_TIME_MASK         0x00000004

#define WLAN_STATS_RX_OUT_BSS_TIME_SHIFT       3
#define WLAN_STATS_RX_OUT_BSS_TIME_MASK        0x00000008

#define WLAN_STATS_RX_BUSY_TIME_SHIFT          4
#define WLAN_STATS_RX_BUSY_TIME_MASK           0x00000010

#define WLAN_STATS_RX_IN_BAD_COND_TIME_SHIFT   5
#define WLAN_STATS_RX_IN_BAD_COND_TIME_MASK    0x00000020

#define WLAN_STATS_TX_IN_BAD_COND_TIME_SHIFT   6
#define WLAN_STATS_TX_IN_BAD_COND_TIME_MASK    0x00000040

#define WLAN_STATS_WLAN_NOT_AVAIL_TIME_SHIFT   7
#define WLAN_STATS_WLAN_NOT_AVAIL_TIME_MASK    0x00000080

/* WLAN peer signal stats bitmap  */
#define WLAN_STATS_PER_CHAIN_SNR_SHIFT         0
#define WLAN_STATS_PER_CHAIN_SNR_MASK          0x00000001

#define WLAN_STATS_PER_CHAIN_NF_SHIFT          1
#define WLAN_STATS_PER_CHAIN_NF_MASK           0x00000002

/* WLAN TX stats bitmap  */
#define WLAN_STATS_TX_MSDU_CNT_SHIFT           0
#define WLAN_STATS_TX_MSDU_CNT_MASK            0x00000001

#define WLAN_STATS_TX_MPDU_CNT_SHIFT           1
#define WLAN_STATS_TX_MPDU_CNT_MASK            0x00000002

#define WLAN_STATS_TX_PPDU_CNT_SHIFT           2
#define WLAN_STATS_TX_PPDU_CNT_MASK            0x00000004

#define WLAN_STATS_TX_BYTES_SHIFT              3
#define WLAN_STATS_TX_BYTES_MASK               0x00000008

#define WLAN_STATS_TX_MSDU_DROP_CNT_SHIFT      4
#define WLAN_STATS_TX_MSDU_DROP_CNT_MASK       0x00000010

#define WLAN_STATS_TX_DROP_BYTES_SHIFT         5
#define WLAN_STATS_TX_DROP_BYTES_MASK          0x00000020

#define WLAN_STATS_TX_MPDU_RETRY_CNT_SHIFT     6
#define WLAN_STATS_TX_MPDU_RETRY_CNT_MASK      0x00000040

#define WLAN_STATS_TX_MPDU_FAIL_CNT_SHIFT      7
#define WLAN_STATS_TX_MPDU_FAIL_CNT_MASK       0x00000080

#define WLAN_STATS_TX_PPDU_FAIL_CNT_SHIFT      8
#define WLAN_STATS_TX_PPDU_FAIL_CNT_MASK       0x00000100

#define WLAN_STATS_TX_MPDU_AGGR_SHIFT          9
#define WLAN_STATS_TX_MPDU_AGGR_MASK           0x00000200

#define WLAN_STATS_TX_SUCC_MCS_SHIFT           10
#define WLAN_STATS_TX_SUCC_MCS_MASK            0x00000400

#define WLAN_STATS_TX_FAIL_MCS_SHIFT           11
#define WLAN_STATS_TX_FAIL_MCS_MASK            0x00000800

#define WLAN_STATS_TX_PPDU_DELAY_SHIFT         12
#define WLAN_STATS_TX_PPDU_DELAY_MASK          0x00001000

/* WLAN RX stats bitmap  */
#define WLAN_STATS_MAC_RX_MPDU_CNT_SHIFT     0
#define WLAN_STATS_MAC_RX_MPDU_CNT_MASK      0x00000001

#define WLAN_STATS_MAC_RX_BYTES_SHIFT        1
#define WLAN_STATS_MAC_RX_BYTES_MASK         0x00000002

#define WLAN_STATS_PHY_RX_PPDU_CNT_SHIFT     2
#define WLAN_STATS_PHY_RX_PPDU_CNT_MASK      0x00000004

#define WLAN_STATS_PHY_RX_BYTES_SHIFT        3
#define WLAN_STATS_PHY_RX_BYTES_MASK         0x00000008

#define WLAN_STATS_RX_DISORDER_CNT_SHIFT     4
#define WLAN_STATS_RX_DISORDER_CNT_MASK      0x00000010

#define WLAN_STATS_RX_RETRY_CNT_SHIFT        5
#define WLAN_STATS_RX_RETRY_CNT_MASK         0x00000020

#define WLAN_STATS_RX_DUP_CNT_SHIFT          6
#define WLAN_STATS_RX_DUP_CNT_MASK           0x00000040

#define WLAN_STATS_RX_DISCARD_CNT_SHIFT      7
#define WLAN_STATS_RX_DISCARD_CNT_MASK       0x00000080

#define WLAN_STATS_RX_MPDU_AGGR_SHIFT        8
#define WLAN_STATS_RX_MPDU_AGGR_MASK         0x00000100

#define WLAN_STATS_RX_MCS_SHIFT              9
#define WLAN_STATS_RX_MCS_MASK               0x00000200

#define WLAN_STATS_STA_PS_INDS_SHIFT         10
#define WLAN_STATS_STA_PS_INDS_MASK          0x00000400

#define WLAN_STATS_STA_PS_DURS_SHIFT         11
#define WLAN_STATS_STA_PS_DURS_MASK          0x00000800

#define WLAN_STATS_RX_PROBE_REQS_SHIFT       12
#define WLAN_STATS_RX_PROBE_REQS_MASK        0x00001000

#define WLAN_STATS_RX_OTH_MGMTS_SHIFT        13
#define WLAN_STATS_RX_OTH_MGMTS_MASK         0x00002000

typedef struct
{
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_chan_cca_stats */
    A_UINT32 vdev_id;
    /** Percentage of idle time, no TX, no RX, no interference */
    A_UINT32 idle_time;
    /** Percentage of time transmitting packets */
    A_UINT32 tx_time;
    /** Percentage of time receiving packets in current BSSs */
    A_UINT32 rx_in_bss_time;
    /** Percentage of time receiving packets not in current BSSs */
    A_UINT32 rx_out_bss_time;
    /** Percentage of time interference detected. */
    A_UINT32 rx_busy_time;
    /** Percentage of time receiving packets with errors
     * or packets flagged as retransmission or seqnum discontinued. */
    A_UINT32 rx_in_bad_cond_time;
    /** Percentage of time the device transmitted packets that haven't been ACKed. */
    A_UINT32 tx_in_bad_cond_time;
    /** Percentage of time the chip is unable to work in normal conditions. */
    A_UINT32 wlan_not_avail_time;
} wmi_chan_cca_stats;

/** Thresholds of cca stats, stands for percentages of stats variation.
 *  Check wmi_chan_cca_stats for each stats's meaning.
 */
typedef struct
{
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_chan_cca_stats_thresh */
    A_UINT32 idle_time;           /* units = percent */
    A_UINT32 tx_time;             /* units = percent */
    A_UINT32 rx_in_bss_time;      /* units = percent */
    A_UINT32 rx_out_bss_time;     /* units = percent */
    A_UINT32 rx_busy_time;        /* units = percent */
    A_UINT32 rx_in_bad_cond_time; /* units = percent */
    A_UINT32 tx_in_bad_cond_time; /* units = percent */
    A_UINT32 wlan_not_avail_time; /* units = percent */
} wmi_chan_cca_stats_thresh;

typedef struct
{
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_signal_stats */
    A_UINT32 vdev_id;
    A_UINT32 peer_id;
    /** per chain SNR in current bss, units are dB */
    A_INT32 per_chain_snr[WMI_MAX_CHAINS];
    /** per chain background noise, units are dBm */
    A_INT32 per_chain_nf[WMI_MAX_CHAINS];
    /** per antenna rx MPDUs */
    A_UINT32 per_antenna_rx_mpdus[WMI_MAX_CHAINS];
    /** per antenna tx MPDUs */
    A_UINT32 per_antenna_tx_mpdus[WMI_MAX_CHAINS];
    /** num of valid chains for per antenna rx/tx MPDU cnts*/
    A_UINT32 num_chains_valid;
} wmi_peer_signal_stats;

/** Thresholds of signal stats, stand for percentage of stats variation.
 *  Check wmi_peer_signal_stats for each stats's meaning.
 */
typedef struct
{
    /**
     * TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_peer_signal_stats_thresh
     */
    A_UINT32 tlv_header;
    A_UINT32 per_chain_snr; /* units = dB */
    A_UINT32 per_chain_nf; /* units = dBm */
} wmi_peer_signal_stats_thresh;

typedef struct
{
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tx_stats */
    /** Number of total TX MSDUs on MAC layer in the period */
    A_UINT32 tx_msdu_cnt;
    /** Number of total TX MPDUs on MAC layer in the period */
    A_UINT32 tx_mpdu_cnt;
    /** Number of total TX PPDUs on MAC layer in the period */
    A_UINT32 tx_ppdu_cnt;
    /** Bytes of tx data on MAC layer in the period */
    A_UINT32 tx_bytes;
    /** Number of TX MSDUs cancelled due to any reason in the period,
     * such as WMM limitation/bandwidth limitation/radio congestion */
    A_UINT32 tx_msdu_drop_cnt;
    /** Bytes of dropped TX packets in the period */
    A_UINT32 tx_drop_bytes;
    /** Number of unacked transmissions of MPDUs */
    A_UINT32 tx_mpdu_retry_cnt;
    /** Number of MPDUs have not been ACKed despite retried */
    A_UINT32 tx_mpdu_fail_cnt;
    /** Number of PPDUs which received no block ack */
    A_UINT32 tx_ppdu_fail_cnt;
    /* This TLV is followed by TLVs below: :
     *     A_UINT32 tx_mpdu_aggr[tx_mpdu_aggr_array_len];
     *     A_UINT32 tx_succ_mcs[tx_succ_mcs_array_len];
     *     A_UINT32 tx_fail_mcs[tx_fail_mcs_array_len];
     *     A_UINT32 tx_delay[tx_ppdu_delay_array_len];
     */
} wmi_tx_stats;

/** Thresholds of tx stats, stand for percentage of stats variation.
 *  Check wmi_tx_stats for each stats's meaning.
 */
typedef struct
{
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tx_stats_thresh */
    A_UINT32 tx_msdu_cnt;
    A_UINT32 tx_mpdu_cnt;
    A_UINT32 tx_ppdu_cnt;
    A_UINT32 tx_bytes;
    A_UINT32 tx_msdu_drop_cnt;
    A_UINT32 tx_drop_bytes;
    A_UINT32 tx_mpdu_retry_cnt;
    A_UINT32 tx_mpdu_fail_cnt;
    A_UINT32 tx_ppdu_fail_cnt;
    A_UINT32 tx_mpdu_aggr;
    A_UINT32 tx_succ_mcs;
    A_UINT32 tx_fail_mcs;
    A_UINT32 tx_ppdu_delay;
} wmi_tx_stats_thresh;

typedef struct
{
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_ac_tx_stats */
    A_UINT32 vdev_id;
    A_UINT32 peer_id;
    /* The TLVs for the 4 AC follows:
     *     wmi_tx_stats tx_stats[];   wmi_tx_stats for BE/BK/VI/VO
     */
} wmi_peer_ac_tx_stats;

typedef struct
{
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_rx_stats */
    /** Number of RX MPDUs on MAC layer */
    A_UINT32 mac_rx_mpdu_cnt;
    /** Bytes of RX packets on MAC layer */
    A_UINT32 mac_rx_bytes;
    /** Number of RX PPDU on PHY layer */
    A_UINT32 phy_rx_ppdu_cnt;
    /** Bytes of RX packets on PHY layer */
    A_UINT32 phy_rx_bytes;
    /** Number of discontinuity in seqnum */
    A_UINT32 rx_disorder_cnt;
    /** Number of RX MPDUs flagged as retransmissions */
    A_UINT32 rx_mpdu_retry_cnt;
    /** Number of RX MPDUs identified as duplicates */
    A_UINT32 rx_mpdu_dup_cnt;
    /** Number of RX MPDUs discarded */
    A_UINT32 rx_mpdu_discard_cnt;
    /* This TLV is followed by TLVs below:
     *     A_UINT32 rx_mpdu_aggr[rx_mpdu_aggr_array_len];
     *     A_UINT32 rx_mcs[rx_msdu_mcs_array_len];
     */
} wmi_rx_stats;

/** Thresholds of rx stats, stands for percentage of stats variation.
 *  Check wmi_rx_stats for each stats's meaning.
 */
typedef struct
{
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_rx_stats_thresh */
    A_UINT32 mac_rx_mpdu_cnt;
    A_UINT32 mac_rx_bytes;
    A_UINT32 phy_rx_ppdu_cnt;
    A_UINT32 phy_rx_bytes;
    A_UINT32 rx_disorder_cnt;
    A_UINT32 rx_mpdu_retry_cnt;
    A_UINT32 rx_mpdu_dup_cnt;
    A_UINT32 rx_mpdu_discard_cnt;
    A_UINT32 rx_mpdu_aggr;
    A_UINT32 rx_mcs;
    A_UINT32 sta_ps_inds;
    A_UINT32 sta_ps_durs;
    A_UINT32 rx_probe_reqs;
    A_UINT32 rx_oth_mgmts;
} wmi_rx_stats_thresh;

typedef struct
{
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_ac_rx_stats */
    A_UINT32 vdev_id;
    A_UINT32 peer_id;
    /** How many times STAs go to sleep */
    A_UINT32 sta_ps_inds;
    /** Total sleep time of STAs, milliseconds units */
    A_UINT32 sta_ps_durs;
    /** Number of probe requests received */
    A_UINT32 rx_probe_reqs;
    /** Number of other management frames received, not including probe requests */
    A_UINT32 rx_oth_mgmts;
    /* The TLVs for the 4 AC follows:
     *     wmi_rx_stats rx_stats[];  wmi_rx_stats for BE/BK/VI/VO
     */
} wmi_peer_ac_rx_stats;

typedef struct
{
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_stats_period */
    /*
     * This struct provides the timestamps from a low-frequency timer
     * for the start and end of a stats period.
     * Each timestamp is reported twice, with different units.
     * The _msec timestamp is in millisecond units.
     * The _count timestamp is in clock tick units.
     * The timestamp is reported in clock ticks as well as in milliseconds
     * so that if the stats start and end times fall within the same
     * millisecond, the clock tick timestamps can still be used to
     * determine what fraction of a millisecond the stats period occupied.
     */
    A_UINT32 start_low_freq_msec;
    A_UINT32 start_low_freq_count;
    A_UINT32 end_low_freq_msec;
    A_UINT32 end_low_freq_count;
} wmi_stats_period;

typedef enum {
    /** Periodic timer timed out, based on the period specified
     *  by WMI_PDEV_PARAM_STATS_OBSERVATION_PERIOD
     */
    TRIGGER_COND_ID_TIMER_TIMED_OUT  =  0x1,
    /** Any of the (enabled) stats thresholds specified
     *  in the WMI_PDEV_SET_STATS_THRESHOLD_CMD message is exceeded
     *  within the current stats period.
     */
    TRIGGER_COND_ID_THRESH_EXCEEDED  =  0x2,
    /** In Response to the one-time wlan stats request of
     *  WMI_REQUEST_WLAN_STATS_CMDID from host.
     */
    TRIGGER_COND_ID_ONE_TIME_REQUEST =  0x3,
} wmi_report_stats_event_trigger_cond_id;

typedef struct
{
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_stats_interference */

    /** For cases where a single rx chain has options to be connected to
     * different rx antennas, show which rx antennas were in use during
     * receipt of a given PPDU.
     * This sa_ant_matrix provides a bitmask of the antennas used while
     * receiving this frame.
     */
    A_UINT32 sa_ant_matrix;

    /** Count how many times the hal_rxerr_phy is marked, in this time period.
     * The counter value is reset each period. The host specifies the period
     * via WMI_PDEV_PARAM_STATS_OBSERVATION_PERIOD.
     */
    A_UINT32 phyerr_count;

    /** The timestamp at which the WMI event is reported.
     * In targets that have a WBTIMER_1 timer, this timestamp is taken
     * from WBTIMER_1.
     */
    A_UINT32 timestamp;
} wmi_stats_interference;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_report_stats_event_fixed_param */
    /** Indicate what triggered this event, check wmi_report_stats_event_trigger_cond_id for details */
    A_UINT32 trigger_cond_id;
    /** Bitmap to indicate changed channel CCA stats which exceeded the thresholds */
    A_UINT32 cca_chgd_bitmap;
    /** Bitmap to indicate changed peer signal stats which exceeded the thresholds */
    A_UINT32 sig_chgd_bitmap;
    /** Bitmap to indicate changed TX counters which exceeded the thresholds */
    A_UINT32 tx_chgd_bitmap;
    /** Bitmap to indicate changed RX counters which exceeded the thresholds */
    A_UINT32 rx_chgd_bitmap;
    /** number of per channel CCA stats structures (wmi_chan_cca_stats), 0 to max vdevs*/
    A_UINT32 num_chan_cca_stats;
    /** number of per peer signal stats structures (wmi_peer_signal_stats), 0 to max peers*/
    A_UINT32 num_peer_signal_stats;
    /** number of per peer ac TX stats structures (wmi_peer_ac_tx_stats), 0 to max peers*/
    A_UINT32 num_peer_ac_tx_stats;
    /** Array length of tx_mpdu_aggr[] which is histogram of MPDU aggregation size(1 to 7 and 8+).
     *  The array indicates number of MPDUs sent on specified aggregation size
     *  (per number of MPDUs per AMPDUs / 1 to 7 and 8+).
     *  Array length can be set per WMI_PDEV_PARAM_TX_MPDU_AGGR_ARRAY_LEN */
    A_UINT32 tx_mpdu_aggr_array_len;
    /** Array length of tx_succ_mcs[] which is histogram of encoding rate.
     *  The array indicates number of acked PPDUs sent at a specific rate */
    A_UINT32 tx_succ_mcs_array_len;
    /** Array length of tx_fail_mcs[] which is histogram of encoding rate.
     *  The array indicates number of unacked PPDUs sent at a specific rate */
    A_UINT32 tx_fail_mcs_array_len;
    /** tx_ppdu_delay[]is a histogram of delays on MAC layer.
     * The array counts numbers of PPDUs encountering different TX time delays.
     * TX delay here means time interval between the time a PPDU is queued
     * to the MAC HW for transmission and the time the lower layers of
     * tx FW return a tx status.
     *
     * The bin size tx_ppdu_delay_bin_size_ms specifies how many
     * milliseconds. Each bin of the tx_ppdu_delay histogram represents.
     * By default the bin size is 10ms.
     * tx_ppdu_delay[0] -> delays between 0-9 ms
     * tx_ppdu_delay[1] -> delays between 10-19 ms
     * ...
     * tx_ppdu_delay[9] -> delays between 90-99 ms
     * tx_ppdu_delay[10] -> delays >= 100 ms
     * Bin size can be set per WMI_PDEV_PARAM_TX_PPDU_DELAY_BIN_SIZE_MS.
     */
    A_UINT32 tx_ppdu_delay_bin_size_ms;
    /** Array length of tx_ppdu_delay[]. It can be set per WMI_PDEV_PARAM_TX_PPDU_DELAY_ARRAY_LEN */
    A_UINT32 tx_ppdu_delay_array_len;
    /** number of per peer ac RX stats structures (wmi_peer_ac_rx_stats), 0 to max peers*/
    A_UINT32 num_peer_ac_rx_stats;
    /** Array length of rx_mpdu_aggr[] which is histogram of MPDU aggregation size(1 to 7 and 8+).
     *  It can be set per WMI_PDEV_PARAM_RX_MPDU_AGGR_ARRAY_LEN */
    A_UINT32 rx_mpdu_aggr_array_len;
    /** Array size of rx_mcs[] which is histogram of encoding rate.
     *  The array indicates number of PPDUs received at a specific rate */
    A_UINT32 rx_mcs_array_len;
    /** Array size of stats_period[] which contains several stats periods. */
    A_UINT32 stats_period_array_len;

    /**
     * This TLV is followed by TLVs below:
     *    wmi_chan_cca_stats       chan_cca_stats[];         Array length is specified by num_chan_cca_stats
     *    wmi_peer_signal_stats    peer_signal_stats[];      Array length is specified by num_peer_signal_stats
     *    wmi_peer_ac_tx_stats     peer_ac_tx_stats[];       Array length is specified by num_peer_ac_tx_stats
     *    wmi_tx_stats             tx_stats[][];             Array length is num_peer_ac_tx_stats * WLAN_MAX_AC, array index is (peer_index * WLAN_MAX_AC + ac_index)
     *    A_UINT32                 tx_mpdu_aggr[][][];       Array length is num_peer_ac_tx_stats * WLAN_MAX_AC * tx_mpdu_aggr_array_len,
     *                                                       array index is (peer_index * WLAN_MAX_AC + ac_index) * tx_mpdu_aggr_array_len + A-MPDU aggregation index
     *    A_UINT32                 tx_succ_mcs[][][];        Array length is num_peer_ac_tx_stats * WLAN_MAX_AC * tx_succ_mcs_array_len,
     *                                                       array index is (peer_index * WLAN_MAX_AC + ac_index) * tx_succ_mcs_array_len + MCS index
     *    A_UINT32                 tx_fail_mcs[][][];        Array length is num_peer_ac_tx_stats * WLAN_MAX_AC * tx_fail_mcs_array_len,
     *                                                       array index is (peer_index * WLAN_MAX_AC + ac_index) * tx_fail_mcs_array_len + MCS index
     *    A_UINT32                 tx_ppdu_delay[][][];           Array length is num_peer_ac_tx_stats * WLAN_MAX_AC * tx_ppdu_delay_array_len,
     *                                                       array index is (peer_index * WLAN_MAX_AC + ac_index) * tx_ppdu_delay_array_len + tx delay index
     *    wmi_peer_ac_rx_stats     peer_ac_rx_stats[];       Array length is specified by num_peer_ac_rx_stats
     *    wmi_rx_stats             rx_stats[][];             Array length is num_peer_ac_rx_stats * WLAN_MAX_AC, array index is (peer_index * WLAN_MAX_AC + ac_index)
     *    A_UINT32                 rx_mpdu_aggr[][][];       Array length is num_peer_ac_rx_stats * WLAN_MAX_AC * rx_mpdu_aggr_array_len,
     *                                                       array index is (peer_index * WLAN_MAX_AC + ac_index) * rx_mpdu_aggr_array_len + A-MPDU aggregation index
     *    A_UINT32                 rx_mcs[][][];             Array length is (num_peer_ac_rx_stats * WLAN_MAX_AC) * rx_mcs_array_len,
     *                                                       array index is (peer_index * WLAN_MAX_AC + ac_index) * rx_mcs_array_len + MCS index
     *    wmi_stats_period         stats_period[];           Array length is specified by stats_period_array_len
     *    wmi_stats_interference   stats_interference[];     Array length is determied by dividing array level TLV header's length value by array-element TLV header's length value.
     **/
} wmi_report_stats_event_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_peer_stats_info */
    A_UINT32 tlv_header;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /** bytes (size of MPDUs) transmitted to this peer */
    struct {
        /* lower 32 bits of the tx_bytes value */
        A_UINT32 low_32;
        /* upper 32 bits of the tx_bytes value */
        A_UINT32 high_32;
    } tx_bytes;
    /** packets (MSDUs) transmitted to this peer */
    struct {
        /* lower 32 bits of the tx_packets value */
        A_UINT32 low_32;
        /* upper 32 bits of the tx_packets value */
        A_UINT32 high_32;
    } tx_packets;
    /** bytes (size of MPDUs) received from this peer */
    struct {
        /* lower 32 bits of the rx_bytes value */
        A_UINT32 low_32;
        /* upper 32 bits of the rx_bytes value */
        A_UINT32 high_32;
    } rx_bytes;
    /** packets (MSDUs) received from this peer */
    struct {
        /* lower 32 bits of the rx_packets value */
        A_UINT32 low_32;
        /* upper 32 bits of the rx_packets value */
        A_UINT32 high_32;
    } rx_packets;
    /** cumulative retry counts (MPDUs) */
    A_UINT32 tx_retries;
    /** number of failed transmissions (MPDUs) (retries exceeded, no ACK) */
    A_UINT32 tx_failed;
    /** rate information, it is output of WMI_ASSEMBLE_RATECODE_V1
     *  (in format of 0x1000RRRR)
     * The rate-code is a 4-bytes field in which,
     * for given rate, nss and preamble
     *
     * b'31-b'29 unused / reserved
     * b'28      indicate the version of rate-code (1 = RATECODE_V1)
     * b'27-b'11 unused / reserved
     * b'10-b'8  indicate the preamble (0 OFDM, 1 CCK, 2 HT, 3 VHT)
     * b'7-b'5   indicate the NSS (0 - 1x1, 1 - 2x2, 2 - 3x3, 3 - 4x4)
     * b'4-b'0   indicate the rate, which is indicated as follows:
     *          OFDM :     0: OFDM 48 Mbps
     *                     1: OFDM 24 Mbps
     *                     2: OFDM 12 Mbps
     *                     3: OFDM 6 Mbps
     *                     4: OFDM 54 Mbps
     *                     5: OFDM 36 Mbps
     *                     6: OFDM 18 Mbps
     *                     7: OFDM 9 Mbps
     *         CCK (pream == 1)
     *                     0: CCK 11 Mbps Long
     *                     1: CCK 5.5 Mbps Long
     *                     2: CCK 2 Mbps Long
     *                     3: CCK 1 Mbps Long
     *                     4: CCK 11 Mbps Short
     *                     5: CCK 5.5 Mbps Short
     *                     6: CCK 2 Mbps Short
     *         HT/VHT (pream == 2/3)
     *                     0..7: MCS0..MCS7 (HT)
     *                     0..9: MCS0..MCS9 (11AC VHT)
     *                     0..11: MCS0..MCS11 (11AX VHT)
     */
    /** rate-code of the last transmission */
    A_UINT32 last_tx_rate_code;
    /** rate-code of the last received PPDU */
    A_UINT32 last_rx_rate_code;
    /** bitrate of the last transmission, in units of kbps */
    A_UINT32 last_tx_bitrate_kbps;
    /** bitrate of the last received PPDU, in units of kbps */
    A_UINT32 last_rx_bitrate_kbps;
    /** combined RSSI of the last received PPDU, in unit of dBm */
    A_INT32 peer_rssi;
} wmi_peer_stats_info;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_peer_stats_info_event_fixed_param */
    A_UINT32 tlv_header;
    /** VDEV to which the peers belong to */
    A_UINT32 vdev_id;
    /** number of peers in peer_stats_info[] */
    A_UINT32 num_peers;
    /** flag to indicate if there are more peers which will
     *  be sent a following seperate peer_stats_info event */
    A_UINT32 more_data;
    /* This TLV is followed by another TLV of array of structs
     * wmi_peer_stats_info peer_stats_info[];
     */
} wmi_peer_stats_info_event_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_radio_chan_stats */
     A_UINT32 tlv_header;
     /** primary channel freq of the channel whose stats is sent */
     A_UINT32 chan_mhz;
     /** accumulation of time the radio is tuned to this channel,
     *  in units of microseconds */
     A_UINT32 on_chan_us;
     /** accumulation of the TX PPDU duration over the measurement period,
     *  in units of microseconds */
     A_UINT32 tx_duration_us;
     /** accumulation of the RX PPDU duration over the measurement period,
     *  in units of microseconds */
     A_UINT32 rx_duration_us;
     /** ratio of channel busy time to on_chan_us, in units of percent */
     A_UINT32 chan_busy_ratio;
     /** ratio of on_chan_us to the measurement period, in units of percent */
     A_UINT32 on_chan_ratio;
     /** measurement period, in units of microseconds */
     A_UINT32 measurement_period_us;
     /** MPDUs transmitted on this channel */
     A_UINT32 tx_mpdus;
     /** MSDUs transmitted on this channel */
     A_UINT32 tx_msdus;
     /** MPDUS successfully received on this channel */
     A_UINT32 rx_succ_mpdus;
     /** Failed MPDUs (CRC failures) received on this channel */
     A_UINT32 rx_fail_mpdus;
} wmi_radio_chan_stats;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_radio_chan_stats_event_fixed_param */
     A_UINT32 tlv_header;
     /** number of channel stats in radio_chan_stats[] */
     A_UINT32 num_chans;
     /* This TLV is followed by another TLV of array of structs
     * wmi_radio_chan_stats radio_chan_stats[];
     */
} wmi_radio_chan_stats_event_fixed_param;

/**
 *  PDEV statistics
 *
 *  This struct incorporates the wlan_dbg_stats struct, which is
 *  conditionally defined, based on the AR900B flag.
 *  The below _v1 struct variant is the unconditional definition
 *  that matches what would be conditionally defined by builds that
 *  don't use the AR900B flag.  The _v2 struct variant is the
 *  unconditional definition that matches what would be conditionally
 *  defined by builds that use the AR900B flag.
 *  The _v2 struct def can be used within host or target builds
 *  that don't use the AR900B flag, but needs to interoperate with a
 *  target or host build that does use the AR900B flag.
 *  Similarly, the _v1 struct def can be used by a host or target build
 *  that does use the AR900B flag, but needs to interoperate with a
 *  target or host build that doesn't use the AR900B flag.
 *
 *  For backwards compatibility, wmi_pdev_stats is still (conditionally)
 *  defined, as an alias for either the _v1 or _v2 variant.
 */
typedef struct {
    /** Channel noise floor */
    A_INT32 chan_nf;
    /** TX frame count */
    A_UINT32 tx_frame_count;
    /** RX frame count */
    A_UINT32 rx_frame_count;
    /** rx clear count */
    A_UINT32 rx_clear_count;
    /** cycle count */
    A_UINT32 cycle_count;
    /** Phy error count */
    A_UINT32 phy_err_count;
    /** Channel Tx Power */
    A_UINT32 chan_tx_pwr;
    /** WAL dbg stats */
    struct wlan_dbg_stats_v1 pdev_stats;
} wmi_pdev_stats_v1;

typedef struct {
    /** Channel noise floor */
    A_INT32 chan_nf;
    /** TX frame count */
    A_UINT32 tx_frame_count;
    /** RX frame count */
    A_UINT32 rx_frame_count;
    /** rx clear count */
    A_UINT32 rx_clear_count;
    /** cycle count */
    A_UINT32 cycle_count;
    /** Phy error count */
    A_UINT32 phy_err_count;
    /** Channel Tx Power */
    A_UINT32 chan_tx_pwr;
    /** WAL dbg stats */
    struct wlan_dbg_stats_v2 pdev_stats;
} wmi_pdev_stats_v2;

#if defined(AR900B)
#define wmi_pdev_stats wmi_pdev_stats_v2
#else
#define wmi_pdev_stats wmi_pdev_stats_v1
#endif

/**
 *  VDEV statistics
 *  @todo
 *  add all VDEV stats here
 */

typedef struct {
    A_INT32 bcn_snr;
    A_INT32 dat_snr;
} wmi_snr_info;

typedef struct {
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32         vdev_id;
    wmi_snr_info     vdev_snr;
    A_UINT32         tx_frm_cnt[WLAN_MAX_AC];/* Total number of packets(per AC) that were successfully transmitted(with and without retries, including multi-cast, broadcast) */
    A_UINT32         rx_frm_cnt;/* Total number of packets that were successfully received (after appropriate filter rules including multi-cast, broadcast)*/
    A_UINT32         multiple_retry_cnt[WLAN_MAX_AC];/*The number of MSDU packets and MMPDU frames per AC
                                                       that the 802.11 station successfully transmitted after more than one retransmission attempt*/
    A_UINT32         fail_cnt[WLAN_MAX_AC]; /*Total number packets(per AC) failed to transmit */
    A_UINT32         rts_fail_cnt;/*Total number of RTS/CTS sequence failures for transmission of a packet*/
    A_UINT32         rts_succ_cnt;/*Total number of RTS/CTS sequence success for transmission of a packet*/
    A_UINT32         rx_err_cnt;/*The receive error count. HAL will provide the RxP FCS error global */
    A_UINT32         rx_discard_cnt;/* The sum of the receive error count and dropped-receive-buffer error count. (FCS error)*/
    A_UINT32         ack_fail_cnt;/*Total number packets failed transmit because of no ACK from the remote entity*/
    A_UINT32         tx_rate_history[MAX_TX_RATE_VALUES];/*History of last ten transmit rate, in units of 500 kbit/sec*/
    A_UINT32         bcn_rssi_history[MAX_RSSI_VALUES];/*History of last ten Beacon rssi of the connected Bss*/
} wmi_vdev_stats;

/*
 * vdev ext stats with additonal bcn stats
 * (Due to backward compatiblity requirements, these new stats fields cannot be
 * added inside wmi_vdev_stats.)
 */
typedef struct {
    A_UINT32 vdev_id;
    A_UINT32 tx_bcn_succ_cnt; /* Total number of beacon frame transmitted successfully */
    A_UINT32 tx_bcn_outage_cnt; /* Total number of failed beacons */
} wmi_bcn_stats;

/**
 *  peer statistics.
 */
typedef struct {
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /** rssi */
    A_UINT32  peer_rssi;
    /** last tx data rate used for peer */
    A_UINT32  peer_tx_rate;
    /** last rx data rate used for peer */
    A_UINT32  peer_rx_rate;
} wmi_peer_stats;

/**
 *  Peer extension statistics
 */
typedef struct {
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /* lower 32 bits of rx duration in microseconds */
    A_UINT32 rx_duration;
    /** Total TX bytes (including dot11 header) sent to peer */
    A_UINT32 peer_tx_bytes;
    /** Total RX bytes (including dot11 header) received from peer */
    A_UINT32 peer_rx_bytes;
    /** last TX ratecode */
    A_UINT32 last_tx_rate_code;
    /** TX power used by peer - units are 0.5 dBm */
    A_INT32 last_tx_power;

    /* Total number of received multicast & broadcast data frames corresponding to this peer */
    A_UINT32 rx_mc_bc_cnt; /* 1 in the MSB of rx_mc_bc_cnt represents a valid data */
    /* upper 32 bits of rx duration in microseconds */
    A_UINT32 rx_duration_u32; /* 1 in the most significant bit indicates this field contains valid data */
    A_UINT32 reserved[2]; /** for future use - add new peer stats here */
} wmi_peer_extd_stats;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_extd2_stats */
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /*
     * The following rx_bytes field (lower/upper pair) counts only the
     * MSDU bytes (after 802.11 decap, if applicable), and thus doesn't
     * count the 802.11 header, unlike the wmi_peer_extd_stats.peer_rx_bytes
     * and wmi_peer_stats_info.rx_bytes fields.
     */
    /** Lower 32 bits of the rx_bytes (size of MSDUs) excluding dot11 header from this peer */
    A_UINT32 rx_bytes_l32;
    /** Upper 32 bits of the rx_bytes (size of MSDUs) excluding dot11 header from this peer */
    A_UINT32 rx_bytes_u32;
    /** Number of MPDUS received with FCS error from this peer */
    A_UINT32 rx_fcs_err;
    /** Number of MPDUs(both data and non data) received from this peer */
    A_UINT32 rx_mpdus;
    /** nss of last tx data to peer */
    A_UINT32 last_tx_nss;
    /** nss of last rx data from peer */
    A_UINT32 last_rx_nss;
    /** chain mask used for last tx data to peer */
    A_UINT32 last_tx_chain_mask;
    /** chain mask used for last rx data from peer */
    A_UINT32 last_rx_chain_mask;
} wmi_peer_extd2_stats;

typedef struct {
    /** Primary channel freq of the channel for which stats are sent */
    A_UINT32 chan_mhz;
    /** Time spent on the channel */
    A_UINT32 sampling_period_us;
    /** Aggregate duration over a sampling period for which channel activity was observed */
    A_UINT32 rx_clear_count;
    /** Accumalation of the TX PPDU duration over a sampling period */
    A_UINT32 tx_duration_us;
    /** Accumalation of the RX PPDU duration over a sampling period */
    A_UINT32 rx_duration_us;
} wmi_chan_stats;

typedef struct {
    A_UINT32 tx_mpdu_grp_frag_cnt;       /*dot11TransmittedFragmentCount */
    A_UINT32 tx_msdu_grp_frm_cnt;        /*dot11GroupTransmittedFrameCount */
    A_UINT32 tx_msdu_fail_cnt;           /*dot11FailedCount*/
    A_UINT32 rx_mpdu_frag_cnt;           /*dot11ReceivedFragmentCount*/
    A_UINT32 rx_msdu_grp_frm_cnt;        /*dot11GroupReceivedFrameCount*/
    A_UINT32 rx_mpdu_fcs_err;            /*dot11FCSErrorCount*/
    A_UINT32 tx_msdu_frm_cnt;            /*dot11TransmittedFrameCount*/
    A_UINT32 tx_msdu_retry_cnt;          /*dot11RetryCount*/
    A_UINT32 rx_frm_dup_cnt;             /*dot11FrameDuplicateCount */
    A_UINT32 tx_rts_success_cnt;         /*dot11RTSSuccessCount*/
    A_UINT32 tx_rts_fail_cnt;            /*dot11RTSFailureCount*/
    A_UINT32 tx_Qos_mpdu_grp_frag_cnt;   /*dot11QosTransmittedFragmentCount */
    A_UINT32 tx_Qos_msdu_fail_UP;        /*dot11QosFailedCount */
    A_UINT32 tx_Qos_msdu_retry_UP;       /*dot11QosRetryCount */
    A_UINT32 rx_Qos_frm_dup_cnt_UP;      /*dot11QosFrameDuplicateCount*/
    A_UINT32 tx_Qos_rts_success_cnt_UP;  /*dot11QosRTSSuccessCount*/
    A_UINT32 tx_Qos_rts_fail_cnt_UP;     /*dot11QosRTSFailureCount*/
    A_UINT32 rx_Qos_mpdu_frag_cnt_UP;    /*dot11QosReceivedFragmentCount*/
    A_UINT32 tx_Qos_msdu_frm_cnt_UP;     /*dot11QosTransmittedFrameCount*/
    A_UINT32 rx_Qos_msdu_discard_cnt_UP; /*dot11QosDiscardedFrameCount*/
    A_UINT32 rx_Qos_mpdu_cnt;            /*dot11QosMPDUsReceivedCount*/
    A_UINT32 rx_Qos_mpdu_retryBit_cnt;   /*dot11QosRetriesReceivedCount*/
    A_UINT32 rsna_Mgmt_discard_CCMP_replay_err_cnt; /*dot11RSNAStatsRobustMgmtCCMPReplays*/
    A_UINT32 rsna_TKIP_icv_err_cnt;      /*dot11RSNAStatsTKIPICVErrors*/
    A_UINT32 rsna_TKIP_replay_err_cnt;   /*dot11RSNAStatsTKIPReplays*/
    A_UINT32 rsna_CCMP_decrypt_err_cnt;  /*dot11RSNAStatsCCMPDecryptErrors*/
    A_UINT32 rsna_CCMP_replay_err_cnt;   /*dot11RSNAStatsCCMPReplays*/
    A_UINT32 tx_ampdu_cnt;               /*dot11TransmittedAMPDUCount*/
    A_UINT32 tx_mpdu_cnt_in_ampdu;       /*dot11TransmittedMPDUsInAMPDUCount*/
    union {
        A_UINT64 counter; /* for use by target only */
        struct {
            A_UINT32 low;
            A_UINT32 high;
        } upload; /* for use by host */
    } tx_octets_in_ampdu;                /*dot11TransmittedOctetsInAMPDUCount*/
    A_UINT32 rx_ampdu_cnt;               /*dot11AMPDUReceivedCount*/
    A_UINT32 rx_mpdu_cnt_in_ampdu;       /*dot11MPDUInReceivedAMPDUCount*/
    union {
        A_UINT64 counter; /* for use by target only */
        struct {
            A_UINT32 rx_octets_in_ampdu_low;
            A_UINT32 rx_octets_in_ampdu_high;
        } upload; /* for use by host */
    } rx_octets_in_ampdu;                /*dot11ReceivedOctetsInAMPDUCount*/
    A_UINT32 reserved_1;
    A_UINT32 reserved_2;
    A_UINT32 reserved_3;
    A_UINT32 reserved_4;
} wmi_mib_stats;

typedef struct {
    /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_rssi_stats */
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_INT32  rssi_avg_beacon[WMI_MAX_CHAINS];
    A_INT32  rssi_avg_data[WMI_MAX_CHAINS];
    wmi_mac_addr peer_macaddr;
} wmi_rssi_stats;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_congestion_stats */
    A_UINT32 vdev_id;
   /* congestion -
    * This field holds the congestion percentage = (busy_time/total_time)*100
    * for the interval from when the vdev was started to the current time
    * (or the time at which the vdev was stopped).
    */
    A_UINT32 congestion;
} wmi_congestion_stats;

typedef struct {
    /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_per_chain_rssi_stats */
    A_UINT32 tlv_header;
    A_UINT32 num_per_chain_rssi_stats;
   /* This TLV is followed by another TLV of array of structs:
    * wmi_rssi_stats rssi_stats[num_per_chain_rssi_stats];
    */
} wmi_per_chain_rssi_stats;

/* vdev control flags (per bits) */
#define VDEV_FLAGS_NON_MBSSID_AP      0x00000001   /* legacy AP */
#define VDEV_FLAGS_TRANSMIT_AP        0x00000002   /* indicate if this vdev is transmitting AP */
#define VDEV_FLAGS_NON_TRANSMIT_AP    0x00000004   /* explicitly indicate this vdev is non-transmitting AP */

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_create_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** VDEV type (AP,STA,IBSS,MONITOR) */
    A_UINT32 vdev_type;
    /** VDEV subtype (P2PDEV, P2PCLI, P2PGO, BT3.0)*/
    A_UINT32 vdev_subtype;
    /** VDEV MAC address */
    wmi_mac_addr vdev_macaddr;
    /** Number of configured txrx streams */
    A_UINT32 num_cfg_txrx_streams;
    /**
     * pdev_id for identifying the MAC,
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
    /** control flags for this vdev */
    A_UINT32 flags;
    /**  vdevid of transmitted AP (mbssid case) */
    A_UINT32 vdevid_trans;
/* This TLV is followed by another TLV of array of structures
 *   wmi_vdev_txrx_streams cfg_txrx_streams[];
 */
} wmi_vdev_create_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_txrx_streams */
    /* band - Should take values from wmi_channel_band_mask */
    A_UINT32 band;
    /* max supported tx streams per given band for this vdev */
    A_UINT32 supported_tx_streams;
    /* max supported rx streams per given band for this vdev */
    A_UINT32 supported_rx_streams;
} wmi_vdev_txrx_streams;

/* wmi_p2p_noa_descriptor structure can't be modified without breaking the compatibility for WMI_HOST_SWBA_EVENTID */
typedef struct {
    A_UINT32   tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_p2p_noa_descriptor */
    A_UINT32   type_count; /** 255: continuous schedule, 0: reserved */
    A_UINT32   duration ;  /** Absent period duration in micro seconds */
    A_UINT32   interval;   /** Absent period interval in micro seconds */
    A_UINT32   start_time; /** 32 bit tsf time when in starts */
} wmi_p2p_noa_descriptor;

/** values for vdev_type */
#define WMI_VDEV_TYPE_AP         0x1
#define WMI_VDEV_TYPE_STA        0x2
#define WMI_VDEV_TYPE_IBSS       0x3
#define WMI_VDEV_TYPE_MONITOR    0x4

/** VDEV type is for social wifi interface.This VDEV is Currently mainly needed
* by FW to execute the NAN specific WMI commands and also implement NAN specific
* operations like Network discovery, service provisioning and service
* subscription  ..etc. If FW needs NAN VDEV then Host should issue VDEV create
* WMI command to create this VDEV once during initialization and host is not
* expected to use any VDEV specific WMI commands on this VDEV.
**/
#define WMI_VDEV_TYPE_NAN        0x5

#define WMI_VDEV_TYPE_OCB        0x6

/* NAN Data Interface */
#define WMI_VDEV_TYPE_NDI        0x7

/*
 * Param values to be sent for WMI_VDEV_PARAM_SGI command
 * which are used in 11ax systems
 */
#define WMI_SGI_LEGACY         0x1 /* for HT and VHT   */
#define WMI_SGI_HE_400_NS      0x2 /* for HE 400 nsec  */
#define WMI_SGI_HE_800_NS      0x4 /* for HE 800 nsec  */
#define WMI_SGI_HE_1600_NS     0x8 /* for HE 1600 nsec */
#define WMI_SGI_HE_3200_NS    0x10 /* for HE 3200 nsec */

/*
 * Param values to be sent for WMI_VDEV_PARAM_HE_LTF command
 * which are used in 11ax systems
 */
#define WMI_HE_LTF_DEFAULT 0x0
#define WMI_HE_LTF_1X      0x1
#define WMI_HE_LTF_2X      0x2
#define WMI_HE_LTF_4X      0x3

/** values for vdev_subtype */
#define WMI_UNIFIED_VDEV_SUBTYPE_P2P_DEVICE 0x1
#define WMI_UNIFIED_VDEV_SUBTYPE_P2P_CLIENT 0x2
#define WMI_UNIFIED_VDEV_SUBTYPE_P2P_GO     0x3
#define WMI_UNIFIED_VDEV_SUBTYPE_PROXY_STA  0x4
#define WMI_UNIFIED_VDEV_SUBTYPE_MESH       0x5
/* new subtype for 11S mesh is required as 11S functionality differs
 * in many ways from proprietary mesh
 * 11S uses 6-addr frame format and supports peering between mesh
 * stations and dynamic best path selection between mesh stations.
 * While in proprietary mesh, neighboring mesh station MAC is manually
 * added to AST table for traffic flow between mesh stations
 */
#define WMI_UNIFIED_VDEV_SUBTYPE_MESH_11S   0x6
/** values for vdev_start_request flags */
/** Indicates that AP VDEV uses hidden ssid. only valid for
 *  AP/GO */
#define WMI_UNIFIED_VDEV_START_HIDDEN_SSID  (1<<0)
/** Indicates if robust management frame/management frame
 *  protection is enabled. For GO/AP vdevs, it indicates that
 *  it may support station/client associations with RMF enabled.
 *  For STA/client vdevs, it indicates that sta will
 *  associate with AP with RMF enabled. */
#define WMI_UNIFIED_VDEV_START_PMF_ENABLED  (1<<1)

/*
 * Host is sending bcn_tx_rate to override the beacon tx rates.
 */
#define WMI_UNIFIED_VDEV_START_BCN_TX_RATE_PRESENT (1<<2)
/** Indicates if LDPC RX will be advertized inside HT/VHT Capabilities IE
 * of assoc request/response
 */
#define WMI_UNIFIED_VDEV_START_LDPC_RX_ENABLED  (1<<3)

/* BSS color 0-6 */
#define WMI_HEOPS_COLOR_GET_D2(he_ops) WMI_GET_BITS(he_ops, 0, 6)
#define WMI_HEOPS_COLOR_SET_D2(he_ops, value) WMI_SET_BITS(he_ops, 0, 6, value)

/* Default PE Duration subfield indicates the PE duration in units of 4 us */
#define WMI_HEOPS_DEFPE_GET_D2(he_ops) WMI_GET_BITS(he_ops, 6, 3)
#define WMI_HEOPS_DEFPE_SET_D2(he_ops, value) WMI_SET_BITS(he_ops, 6, 3, value)

/* TWT required */
#define WMI_HEOPS_TWT_REQUIRED_GET_D2(he_ops) WMI_GET_BITS(he_ops, 9, 1)
#define WMI_HEOPS_TWT_REQUIRED_SET_D2(he_ops, value) WMI_SET_BITS(he_ops, 9, 1, value)
/* DEPRECATED, use WMI_HEOPS_TWT_REQUIRED_GET instead */
#define WMI_HEOPS_TWT_GET_D2(he_ops) \
    WMI_HEOPS_TWT_REQUIRED_GET_D2(he_ops)
/* DEPRECATED, use WMI_HEOPS_TWT_REQUIRED_SET instead */
#define WMI_HEOPS_TWT_SET_D2(he_ops, value) \
    WMI_HEOPS_TWT_REQUIRED_SET_D2(he_ops, value)

/* RTS threshold in units of 32 us,0 - always use RTS 1023 - this is disabled */
#define WMI_HEOPS_RTSTHLD_GET_D2(he_ops) WMI_GET_BITS(he_ops, 10, 10)
#define WMI_HEOPS_RTSTHLD_SET_D2(he_ops, value) WMI_SET_BITS(he_ops, 10, 10, value)

/* Partial BSS Color field indicates whether BSS applies an AID assignment rule using partial BSS color bits */
#define WMI_HEOPS_PARTBSSCOLOR_GET_D2(he_ops) WMI_GET_BITS(he_ops, 20, 1)
#define WMI_HEOPS_PARTBSSCOLOR_SET_D2(he_ops, value) WMI_SET_BITS(he_ops, 20, 1, value)

/* MAX BSS supported by MultiBSS element */
#define WMI_HEOPS_MAXBSSID_GET_D2(he_ops) WMI_GET_BITS(he_ops, 21, 8)
#define WMI_HEOPS_MAXBSSID_SET_D2(he_ops, value) WMI_SET_BITS(he_ops, 21, 8, value)

/* Tx BSSID Indicator indicates whether HE AP corresponds to transmitted BSSID */
#define WMI_HEOPS_TXBSSID_GET_D2(he_ops) WMI_GET_BITS(he_ops, 29, 1)
#define WMI_HEOPS_TXBSSID_SET_D2(he_ops, value) WMI_SET_BITS(he_ops, 29, 1, value)

/* when set to 1 disables use of BSS color */
#define WMI_HEOPS_BSSCOLORDISABLE_GET_D2(he_ops) WMI_GET_BITS(he_ops, 30, 1)
#define WMI_HEOPS_BSSCOLORDISABLE_SET_D2(he_ops, value) WMI_SET_BITS(he_ops, 30, 1, value)

/**--- HEOPS_DUALBEACON: DO NOT USE - DEPRECATED ---*/
/* When set to 1 HE AP transmits beacons using two PHY formats,
 * one in non-HE format and other in an HE_EXT_SU PHY format
 */
#define WMI_HEOPS_DUALBEACON_GET_D2(he_ops) (0)
#define WMI_HEOPS_DUALBEACON_SET_D2(he_ops, value) {;}

#define WMI_MAX_HECAP_PHY_SIZE                 (3)

/* Dual Band both 2.4 GHz and 5 GHz Supported */
#define WMI_HECAP_PHY_DB_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 0, 1)
#define WMI_HECAP_PHY_DB_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 0, 1, value)

/*
 * B0: Indicates STA support 40 MHz channel width in 2.4 GHz
 * B1: Indicates STA support 40 MHz and 80 MHz channel width in 5 GHz
 * B2: Indicates STA supports 160 MHz channel width in 5 GHz
 * B3: Indicates STA supports 160/80+80 MHz channel width in 5 GHz
 * B4: If B1 is set to 0, then B5 indicates support of 242/106/52/26-tone
 *     RU mapping in 40 MHz channel width in 2.4 GHz. Otherwise Reserved.
 * B5: If B2, B3, and B4 are set to 0, then B6 indicates support of
 *     242-tone RU mapping in 40 MHz and 80
 * MHz channel width in 5 GHz. Otherwise Reserved.
 * B6: Reserved
 */
#define WMI_HECAP_PHY_CBW_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 1, 7)
#define WMI_HECAP_PHY_CBW_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 1, 7, value)

/*
 * B0: Indicates STA supports reception of preamble puncturing in 80 MHz,
 *     where in the preamble only the secondary 20 MHz is punctured
 * B1: Indicates STA supports reception of preamble puncturing in 80 MHz,
 *     where in the preamble only one of the two 20 MHz sub-channels in the
 *     secondary 40 MHz is punctured
 * B2: Indicates STA supports reception of preamble puncturing in 160 MHz
 *     or 80+80 MHz, where in the primary 80 MHz of the preamble only the
 *     secondary 20 MHz is punctured
 * B3: Indicates STA supports reception of preamble puncturing in 160 MHz
 *     or 80+80 MHz, where in the primary 80 MHz of the preamble, the
 *     primary 40 MHz is present
 */
#define WMI_HECAP_PHY_PREAMBLEPUNCRX_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 8, 4)
#define WMI_HECAP_PHY_PREAMBLEPUNCRX_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 8, 4, value)

/* Indicates transmitting STA is a Class A (1) or a Class B (0) device */
#define WMI_HECAP_PHY_COD_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 12, 1)
#define WMI_HECAP_PHY_COD_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 12, 1, value)

/* Indicates support of transmission and reception of LDPC encoded packets */
#define WMI_HECAP_PHY_LDPC_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 13, 1)
#define WMI_HECAP_PHY_LDPC_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 13, 1, value)

/* Below 2 macros are for maintaining backward compatability - Deprecated use WMI_HECAP_PHY_LDPC instead */
#define WMI_HECAP_PHY_TXLDPC_GET_D2(he_cap_phy) WMI_HECAP_PHY_LDPC_GET(he_cap_phy)
#define WMI_HECAP_PHY_TXLDPC_SET_D2(he_cap_phy, value) WMI_HECAP_PHY_LDPC_SET(he_cap_phy, value)
/* Below 2 macros are for maintaining backward compatability - Deprecated use WMI_HECAP_PHY_LDPC instead */
#define WMI_HECAP_PHY_RXLDPC_GET_D2(he_cap_phy) WMI_HECAP_PHY_LDPC_GET(he_cap_phy)
#define WMI_HECAP_PHY_RXLDPC_SET_D2(he_cap_phy, value) WMI_HECAP_PHY_LDPC_SET(he_cap_phy, value)

/*
 * B0: Indicates support of reception of 1x LTF and 0.8us guard interval duration for HE SU PPDUs.
 */
#define WMI_HECAP_PHY_LTFGIFORHE_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 14, 1)
#define WMI_HECAP_PHY_LTFGIFORHE_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 14, 1, value)

/*
 * When the Doppler Rx subfield is 1, indicates the maximum number of space-
 * time streams supported for reception when midamble is used in the Data field.
 */
#define WMI_HECAP_PHY_MIDAMBLERXMAXNSTS_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 15, 2)
#define WMI_HECAP_PHY_MIDAMBLERXMAXNSTS_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 15, 2, value)
/*
 * B0: For a transmitting STA acting as beamformee, it indicates support of
 *     NDP reception using 4x LTF and 3.2 us guard interval duration
 */
#define WMI_HECAP_PHY_LTFGIFORNDP_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 17, 1)
#define WMI_HECAP_PHY_LTFGIFORNDP_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 17, 1, value)

/* indicates support for the transmission of HE PPDUs using STBC with one spatial stream for <= 80MHz Tx*/
#define WMI_HECAP_PHY_TXSTBC_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 18, 1)
#define WMI_HECAP_PHY_TXSTBC_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 18, 1, value)

/* indicates support for the reception of HE PPDUs using STBC with one spatial stream for <= 80MHz Tx*/
#define WMI_HECAP_PHY_RXSTBC_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 19, 1)
#define WMI_HECAP_PHY_RXSTBC_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 19, 1, value)

/* indicates transmitting STA supports transmitting HE PPDUs with Doppler procedure */
#define WMI_HECAP_PHY_TXDOPPLER_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 20, 1)
#define WMI_HECAP_PHY_TXDOPPLER_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 20, 1, value)

/* indicates transmitting STA supports receiving HE PPDUs with Doppler procedure */
#define WMI_HECAP_PHY_RXDOPPLER_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 21, 1)
#define WMI_HECAP_PHY_RXDOPPLER_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 21, 1, value)

/*
 * If the transmitting STA is an AP:
 *     indicates STA supports of reception of full bandwidth UL MU-MIMO
 *     transmission.
 * If the transmitting STA is a non-AP STA:
 *     indicates STA supports of transmission of full bandwidth UL MU-MIMO
 *     transmission.
 */
#define WMI_HECAP_PHY_UL_MU_MIMO_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 22, 1)
#define WMI_HECAP_PHY_UL_MU_MIMO_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 22, 1, value)

/*
 * If the transmitting STA is an AP:
 *     indicates STA supports of reception of UL MUMIMO transmission on an
 *     RU in an HE MU PPDU where the RU does not span the entire PPDU bandwidth.
 * If the transmitting STA is a non-AP STA:
 *     indicates STA supports of transmission of UL MU-MIMO transmission on an
 *     RU in an HE MU PPDU where the RU does not span the entire PPDU bandwidth.
 */
#define WMI_HECAP_PHY_ULMUMIMOOFDMA_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 23, 1)
#define WMI_HECAP_PHY_ULMUMIMOOFDMA_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 23, 1, value)

/* Tx DCM
 * B0:B1
 *     00: Does not support DCM
 *     01: BPSK
 *     10: QPSK
 *     11: 16-QAM
 * B2 signals maximum number of spatial streams with DCM
 *     0: 1 spatial stream
 *     1: 2 spatial streams
 */
#define WMI_HECAP_PHY_DCMTX_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 24, 3)
#define WMI_HECAP_PHY_DCMTX_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 24, 3, value)

/* Rx DCM
 * B0:B1
 *     00: Does not support DCM
 *     01: BPSK
 *     10: QPSK
 *     11: 16-QAM
 * B2 signals maximum number of spatial streams with DCM
 *     0: 1 spatial stream
 *     1: 2 spatial streams
 */
#define WMI_HECAP_PHY_DCMRX_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 27, 3)
#define WMI_HECAP_PHY_DCMRX_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 27, 3, value)

/*
 * Indicates that the STA supports the reception of an HE MU PPDU payload
 * over full bandwidth and partial bandwidth (106-tone RU within 20 MHz).
 */
#define WMI_HECAP_PHY_ULHEMU_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 30, 1)
#define WMI_HECAP_PHY_ULHEMU_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 30, 1, value)

/* Indicates support for operation as an SU beamformer */
#define WMI_HECAP_PHY_SUBFMR_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[0], 31, 1)
#define WMI_HECAP_PHY_SUBFMR_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[0], 31, 1, value)

/* Indicates support for operation as an SU beamformee */
#define WMI_HECAP_PHY_SUBFME_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 0, 1)
#define WMI_HECAP_PHY_SUBFME_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 0, 1, value)

/* Indicates support for operation as an MU Beamformer */
#define WMI_HECAP_PHY_MUBFMR_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 1, 1)
#define WMI_HECAP_PHY_MUBFMR_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 1, 1, value)

/*
 * Num STS -1 for <= 80MHz (min val 3)
 * The maximum number of space-time streams minus 1 that the STA can
 * receive in an HE NDP
 */
#define WMI_HECAP_PHY_BFMESTSLT80MHZ_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 2, 3)
#define WMI_HECAP_PHY_BFMESTSLT80MHZ_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 2, 3, value)

/*
 * Num STS -1 for > 80MHz (min val 3)
 * The maximum number of space-time streams minus 1 that the STA can
 * receive in an HE NDP
 */
#define WMI_HECAP_PHY_BFMESTSGT80MHZ_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 5, 3)
#define WMI_HECAP_PHY_BFMESTSGT80MHZ_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 5, 3, value)

/*
 * Number Of Sounding Dimensions For <= 80 MHz
 * If SU beamformer capable, set to the maximum supported value of the
 * TXVECTOR parameter NUM_STS minus 1.
 * Otherwise, reserved.
 */
#define WMI_HECAP_PHY_NUMSOUNDLT80MHZ_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 8, 3)
#define WMI_HECAP_PHY_NUMSOUNDLT80MHZ_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 8, 3, value)

/*
 * Number Of Sounding Dimensions For > 80 MHz
 * If SU beamformer capable, set to the maximum supported value of the
 * TXVECTOR parameter NUM_STS minus 1.
 * Otherwise, reserved.
 */
#define WMI_HECAP_PHY_NUMSOUNDGT80MHZ_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 11, 3)
#define WMI_HECAP_PHY_NUMSOUNDGT80MHZ_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 11, 3, value)

/*
 * Indicates if the HE beamformee is capable of feedback with tone
 * grouping of 16 in the HE Compressed Beamforming Report field for
 * a SU-type feedback.
 */
#define WMI_HECAP_PHY_NG16SUFEEDBACKLT80_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 14, 1)
#define WMI_HECAP_PHY_NG16SUFEEDBACKLT80_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 14, 1, value)

/*
 * Indicates if the HE beamformee is capable of feedback with tone
 * grouping of 16 in the HE Compressed Beamforming Report field for
 * a MU-type feedback.
 */
#define WMI_HECAP_PHY_NG16MUFEEDBACKGT80_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 15, 1)
#define WMI_HECAP_PHY_NG16MUFEEDBACKGT80_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 15, 1, value)

/*
 * Indicates if HE beamformee is capable of feedback with codebook
 * size {4, 2} in the HECompressed Beamforming Report field for
 * a SU-type feedback.
 */
#define WMI_HECAP_PHY_CODBK42SU_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 16, 1)
#define WMI_HECAP_PHY_CODBK42SU_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 16, 1, value)

/*
 * Indicates if HE beamformee is capable of feedback with codebook
 * size {7, 5} in the HE Compressed Beamforming Report field for
 * a MU-type feedback.
 */
#define WMI_HECAP_PHY_CODBK75MU_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 17, 1)
#define WMI_HECAP_PHY_CODBK75MU_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 17, 1, value)

/*
 * Beamforming Feedback With Trigger Frame
 * If the transmitting STA is an AP STA:
 * B0: indicates support of reception of SU-Type partial(1) and full bandwidth feedback(0)
 * B1: indicates support of reception of MU-Type partial(1) bandwidth feedback
 * B2: indicates support of reception of CQI-Only partial and full bandwidth feedback
 * If the transmitting STA is a non-AP STA:
 * B0: indicates support of transmission of SU-Type partial(1) and full bandwidth(0) feedback
 * B1: indicates support of transmission of MU-Type partial(1) bandwidth feedback
 * B2: indicates support of transmission of CQI-Onlypartial (1)and full bandwidth feedback
 */
#define WMI_HECAP_PHY_BFFEEDBACKTRIG_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 18, 3)
#define WMI_HECAP_PHY_BFFEEDBACKTRIG_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 18, 3, value)

/* Indicates the support of transmission and reception of an HE extended range SU PPDU payload transmitted
 * over the right 106-tone RU or partial BW ER
 */
#define WMI_HECAP_PHY_HEERSU_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 21, 1)
#define WMI_HECAP_PHY_HEERSU_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 21, 1, value)

/* Indicates that the non-AP STA supports reception of a DL MU-MIMO transmission on an RU in an HE MU PPDU
 * where the RU does not span the entire PPDU bandwidth.
 */
#define WMI_HECAP_PHY_DLMUMIMOPARTIALBW_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 22, 1)
#define WMI_HECAP_PHY_DLMUMIMOPARTIALBW_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 22, 1, value)

/* Indicates whether or not the PPE Threshold field is present */
#define WMI_HECAP_PHY_PETHRESPRESENT_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 23, 1)
#define WMI_HECAP_PHY_PETHRESPRESENT_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 23, 1, value)

/* Indicates that the STA supports SRP-based SR operation */
#define WMI_HECAP_PHY_SRPSPRESENT_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 24, 1)
#define WMI_HECAP_PHY_SRPPRESENT_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 24, 1, value)

/* Indicates that the STA supports a power boost factor ar for the r-th RU in the range [0.5, 2] */
#define WMI_HECAP_PHY_PWRBOOSTAR_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 25, 1)
#define WMI_HECAP_PHY_PWRBOOSTAR_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 25, 1, value)

/* Indicates support for the reception of 4x LTF and 0.8us guard interval duration for HE SU PPDUs. */
#define WMI_HECAP_PHY_4XLTFAND800NSECSGI_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 26, 1)
#define WMI_HECAP_PHY_4XLTFAND800NSECSGI_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 26, 1, value)

/* For a transmitting STA acting as a beamformee, it indicates the maximum Nc for beamforming sounding
 * feedback supported If SU beamformee capable, then set to the maximum Nc for beamforming sounding feedback
 * minus 1. Otherwise, reserved.
 */
#define WMI_HECAP_PHY_MAXNC_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 27, 3)
#define WMI_HECAP_PHY_MAXNC_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 27, 3, value)

/* Indicates support for the transmission of an HE PPDU that has a bandwidth greater than 80 MHz and is using
 * STBC with one spatial stream
 */
#define WMI_HECAP_PHY_STBCTXGT80_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 30, 1)
#define WMI_HECAP_PHY_STBCTXGT80_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 30, 1, value)

/* Indicates support for the reception of an HE PPDU that has a bandwidth greater than 80 MHz and is using
 * STBC with one spatial stream
 */
#define WMI_HECAP_PHY_STBCRXGT80_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 31, 1)
#define WMI_HECAP_PHY_STBCRXGT80_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 31, 1, value)

/* Indicates support for the reception of an HE ER SU PPDU with 4x LTF and 0.8 us guard interval duration */
#define WMI_HECAP_PHY_ERSU4X800NSECGI_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[2], 0, 1)
#define WMI_HECAP_PHY_ERSU4X800NSECGI_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[2], 0, 1, value)

/*
 * Indicates support of 26-, 52-, and 106-tone mapping for a 20 MHz operating non-AP HE STA that is the
 * receiver of a 40 MHz HE MU PPDU in 2.4 GHz band, or the transmitter of a 40 MHz HE TB PPDU in 2.4GHz band.
 */
#define WMI_HECAP_PHY_HEPPDU20IN40MHZ2G_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[2], 1, 1)
#define WMI_HECAP_PHY_HEPPDU20IN40MHZ2G_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[2], 1, 1, value)

/*
 * Indicates support of 26-, 52-, and 106-tone mapping for a 20 MHz operating non-AP HE STA that is the
 * receiver of a 80+80 MHz or a 160 MHz HE MU PPDU, or the transmitter of a 80+80 MHz or 160 MHz HE TB PPDU.
 */
#define WMI_HECAP_PHY_HEPPDU20IN160OR80P80MHZ_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[2], 2, 1)
#define WMI_HECAP_PHY_HEPPDU20IN160OR80P80MHZ_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[2], 2, 1, value)

/*
 * Indicates supports of 160 MHz OFDMA for a non-AP HE STA that sets bit B1 of Channel Width Set to 1, and
 * sets B2 and B3 of Channel Width Set each to 0, when operating with 80 MHz channel width. The capability
 * bit is applicable while receiving a 80+80 MHz or a 160 MHz HE MU PPDU, or transmitting a 80+80 MHz or a
 * 160 MHz HE TB PPDU.
 */
#define WMI_HECAP_PHY_HEPPDU80IN160OR80P80MHZ_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[2], 3, 1)
#define WMI_HECAP_PHY_HEPPDU80IN160OR80P80MHZ_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[2], 3, 1, value)

/* Indicates support for the reception of an HE ER SU PPDU with 1x LTF and 0.8 us guard interval duration */
#define WMI_HECAP_PHY_ERSU1X800NSECGI_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[2], 4, 1)
#define WMI_HECAP_PHY_ERSU1X800NSECGI_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[2], 4, 1, value)

/*
 * When the Doppler Rx subfield is 1, indicates support for receiving midambles with 2x HE-LTF, 1x HE-LTF
 * in HE SU PPDU if the HE SU PPDU With 1x HE-LTF And 0.8 s GI subfield is set to 1, and 1x HE-LTF in
 * HE ER SU PPDU if the HE ER SU PPDU With 1x HELTF And 0.8 s GI subfield is set to 1.
 */
#define WMI_HECAP_PHY_MIDAMBLERX2XAND1XHELTF_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[2], 5, 1)
#define WMI_HECAP_PHY_MIDAMBLERX2XAND1XHELTF_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[2], 5, 1, value)

/*HTC + HE Support  Set to 1 if STA supports reception of HE Variant HT control Field*/
#define WMI_HECAP_MAC_HECTRL_GET_D2(he_cap) WMI_GET_BITS(he_cap, 0, 1)
#define WMI_HECAP_MAC_HECTRL_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 0, 1, value)

/* set to 1 to for TWT Requestor support*/
#define WMI_HECAP_MAC_TWTREQ_GET_D2(he_cap) WMI_GET_BITS(he_cap, 1, 1)
#define WMI_HECAP_MAC_TWTREQ_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 1, 1, value)

/* set to 1 to for TWT Responder support*/
#define WMI_HECAP_MAC_TWTRSP_GET_D2(he_cap) WMI_GET_BITS(he_cap, 2, 1)
#define WMI_HECAP_MAC_TWTRSP_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 2, 1, value)

/* Level of frag support
   Set to 0 for no support for dynamic fragmentation.
   Set to 1 for support for dynamic fragments that are contained within a S-MPDU
   Set to 2 for support for dynamic fragments that are contained within a Single MPDU and support for up to
        one dynamic fragment for each MSDU and each MMPDU within an A-MPDU or multi-TID A-MPDU.
   Set to 3 for support for dynamic fragments that are contained within a Single MPDU and support for multiple
        dynamic fragments for each MSDU within an AMPDU or multi-TID AMPDU and up to one dynamic fragment
        for each MMPDU in a multi-TID A-MPDU that is not a Single MPDU
*/
#define WMI_HECAP_MAC_HEFRAG_GET_D2(he_cap) WMI_GET_BITS(he_cap, 3, 2)
#define WMI_HECAP_MAC_HEFRAG_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 3, 2, value)

/* The maximum number of fragmented MSDUs, Nmax,defined by this field is Nmax = 2 Maximum Number Of FMPDUs*/
#define WMI_HECAP_MAC_MAXFRAGMSDU_GET_D2(he_cap) WMI_GET_BITS(he_cap, 5, 3)
#define WMI_HECAP_MAC_MAXFRAGMSDU_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 5, 3, value)

/* 0 =  no restriction on the minimum payload , 1 = 128 octets min, 2 = 256 octets min, 3 = 512 octets min */
#define WMI_HECAP_MAC_MINFRAGSZ_GET_D2(he_cap) WMI_GET_BITS(he_cap, 8, 2)
#define WMI_HECAP_MAC_MINFRAGSZ_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 8, 2, value)

/*0 = no additional processing time, 1 = 8us,2 = 16us */
#define WMI_HECAP_MAC_TRIGPADDUR_GET_D2(he_cap) WMI_GET_BITS(he_cap, 10, 2)
#define WMI_HECAP_MAC_TRIGPADDUR_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 10, 2, value)

/*number of TIDs minus 1 of QoS Data frames that HE STA can aggregate in  multi-TID AMPDU*/
#define WMI_HECAP_MAC_MTID_GET_D2(he_cap) WMI_GET_BITS(he_cap, 12, 3)
#define WMI_HECAP_MAC_MTID_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 12, 3, value)

/*
 * Indicates support by a STA to receive an ack-enabled A-MPDU in which an A-MSDU is carried in
 * a QoS Data frame for which no block ack agreement exists.
 */
#define WMI_HECAP_MAC_AMSDUINAMPDU_GET_D2(he_cap) WMI_GET_BITS(he_cap, 15, 1)
#define WMI_HECAP_MAC_AMSDUINAMPDU_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 15, 1, value)

/*--- HECAP_MAC_HELKAD: DO NOT USE - DEPRECATED ---*/
/*0=No Feedback,2=Unsolicited,3=Both*/
#define WMI_HECAP_MAC_HELKAD_GET_D2(he_cap) (0)
#define WMI_HECAP_MAC_HELKAD_SET_D2(he_cap, value) {;}

/* bit 16 reserved. */

/*Set to 1 for reception of AllAck support*/
#define WMI_HECAP_MAC_AACK_GET_D2(he_cap) WMI_GET_BITS(he_cap, 17, 1)
#define WMI_HECAP_MAC_AACK_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 17, 1, value)

/*Set to 1 if the STA supports reception of the UL MU Response Scheduling A-Control field*/
#define WMI_HECAP_MAC_ULMURSP_GET_D2(he_cap) WMI_GET_BITS(he_cap, 18, 1)
#define WMI_HECAP_MAC_ULMURSP_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 18, 1, value)

/*Set to 1 if the STA supports the BSR A-Control field functionality.*/
#define WMI_HECAP_MAC_BSR_GET_D2(he_cap) WMI_GET_BITS(he_cap, 19, 1)
#define WMI_HECAP_MAC_BSR_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 19, 1, value)

/*Set to 1 when the STA supports broadcast TWT functionality.*/
#define WMI_HECAP_MAC_BCSTTWT_GET_D2(he_cap) WMI_GET_BITS(he_cap, 20, 1)
#define WMI_HECAP_MAC_BCSTTWT_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 20, 1, value)

/*Set to 1 if STA supports rx of Multi-STA BA that has 32-bit Block Ack Bitmap*/
#define WMI_HECAP_MAC_32BITBA_GET_D2(he_cap) WMI_GET_BITS(he_cap, 21, 1)
#define WMI_HECAP_MAC_32BITBA_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 21, 1, value)

/*Set to 1 if the STA supports MU cascading operation*/
#define WMI_HECAP_MAC_MUCASCADE_GET_D2(he_cap) WMI_GET_BITS(he_cap, 22, 1)
#define WMI_HECAP_MAC_MUCASCADE_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 22, 1, value)

/*Set to 1 when the STA supports reception of this multi-TID A-MPDU format*/
#define WMI_HECAP_MAC_ACKMTIDAMPDU_GET_D2(he_cap) WMI_GET_BITS(he_cap, 23, 1)
#define WMI_HECAP_MAC_ACKMTIDAMPDU_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 23, 1, value)

/*Set to 1 when the STA supports its reception*/
#define WMI_HECAP_MAC_GROUPMSTABA_GET_D2(he_cap) WMI_GET_BITS(he_cap, 24, 1)
#define WMI_HECAP_MAC_GROUPMSTABA_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 24, 1, value)

/*Set to 1 if the STA supports reception of the OMI A-Control field*/
#define WMI_HECAP_MAC_OMI_GET_D2(he_cap) WMI_GET_BITS(he_cap, 25, 1)
#define WMI_HECAP_MAC_OMI_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 25, 1, value)

/*1 if OFDMA Random Access Supported*/
#define WMI_HECAP_MAC_OFDMARA_GET_D2(he_cap) WMI_GET_BITS(he_cap, 26, 1)
#define WMI_HECAP_MAC_OFDMARA_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 26, 1, value)

/* Maximum AMPDU Length Exponent.
 * If the HE STA includes a VHT Capabilities element, the Maximum A-MPDU Length Exponent subfield in
 * HE Capabilities element combined with the Maximum A-MPDU Length Exponent subfield in VHT
 * Capabilities element indicate the maximum length of A-MPDU that the STA can Receive where EOF
 * padding is not included in this limit.
*/
#define WMI_HECAP_MAC_MAXAMPDULEN_EXP_GET_D2(he_cap) WMI_GET_BITS(he_cap, 27, 2)
#define WMI_HECAP_MAC_MAXAMPDULEN_EXP_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 27, 2, value)

/*A-MSDU Fragmentation Support*/
#define WMI_HECAP_MAC_AMSDUFRAG_GET_D2(he_cap) WMI_GET_BITS(he_cap, 29, 1)
#define WMI_HECAP_MAC_AMSDUFRAG_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 29, 1, value)

/*Flexible TWT Schedule Support*/
#define WMI_HECAP_MAC_FLEXTWT_GET_D2(he_cap) WMI_GET_BITS(he_cap, 30, 1)
#define WMI_HECAP_MAC_FLEXTWT_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 30, 1, value)

/*Rx Control Frame to MultiBSS*/
#define WMI_HECAP_MAC_MBSS_GET_D2(he_cap) WMI_GET_BITS(he_cap, 31, 1)
#define WMI_HECAP_MAC_MBSS_SET_D2(he_cap, value) WMI_SET_BITS(he_cap, 31, 1, value)

/* BSRP A-MPDU Aggregation
 * maintaining compatability since we dont support this now so not wasting memory
 */
#define WMI_HECAP_MAC_BSRPAMPDU_GET_D2(he_cap) (0)
#define WMI_HECAP_MAC_BSRPAMPDU_SET_D2(he_cap, value) {;}

/* Quiet Time Period (QTP) operation
 * maintaining compatability since we dont support this now so not wasting memory
 */
#define WMI_HECAP_MAC_QTP_GET_D2(he_cap) (0)
#define WMI_HECAP_MAC_QTP_SET_D2(he_cap, value) {;}

/* support by an AP for receiving an (A-)MPDU that contains a BQR in the
 * A-Control subfield and support by a non-AP STA for generating an (A-)MPDU
 * that contains a BQR in the A-Control subfield
 * maintaining compatability since we dont support this now so not wasting memory
 */
#define WMI_HECAP_MAC_ABQR_GET_D2(he_cap) (0)
#define WMI_HECAP_MAC_ABQR_SET_D2(he_cap, value) {;}

/*Indicates support by the STA for the role of SR Responder.*/
#define WMI_HECAP_MAC_SRRESP_GET_D2(he_cap) (0)
#define WMI_HECAP_MAC_SRRESP_SET_D2(he_cap, value) {;}

/* Indicates support for an AP to encode OPS information to TIM element of the FILS Discovery
 * frames or TIM frames as described in AP operation for opportunistic power save.
 * Indicates support for a non-AP STA to receive the opportunistic power save encoded TIM elements
 */
#define WMI_HECAP_MAC_OPS_GET_D2(he_cap) (0)
#define WMI_HECAP_MAC_OPS_SET_D2(he_cap, value) {;}

/* Indicates support for a non-AP STA to follow the NDP feedback report procedure and respond to
 * the NDP Feedback Report Poll Trigger frame.
 */
#define WMI_HECAP_MAC_NDPFDBKRPT_GET_D2(he_cap) (0)
#define WMI_HECAP_MAC_NDPFDBKRPT_SET_D2(he_cap, value) {;}

/* BELOW MACROS ARE DEPRECATED Also we are not defining bits for capabilities
 * beyond bit 31 we donot support as it adds additional dword to our struct which may be later
 * removed by standard
 */
#define WMI_HECAP_MAC_MBAHECTRL_GET_D2(he_cap) (0) /* DO NOT USE - DEPRECATED*/
#define WMI_HECAP_MAC_MBAHECTRL_SET_D2(he_cap, value) {;} /* DO NOT USE - DEPRECATED*/

#define WMI_HECAP_MAC_MURTS_GET_D2(he_cap) (0) /* DO NOT USE - DEPRECATED*/
#define WMI_HECAP_MAC_MURTS_SET_D2(he_cap, value) {;} /* DO NOT USE - DEPRECATED*/

/*Deprecate use  WMI_HECAP_PHY_PREAMBLEPUNCRX instead*/
#define WMI_HECAP_PHY_CBMODE_GET_D2(he_cap_phy) WMI_HECAP_PHY_CBMODE_GET(he_cap_phy)
#define WMI_HECAP_PHY_CBMODE_SET_D2(he_cap_phy, value) WMI_HECAP_PHY_CBMODE_SET(he_cap_phy, value)

/* Below 2 macros are for maintaining backward compatability - Deprecated use WMI_HECAP_PHY_LTFGIFORHE_GET instead */
#define WMI_HECAP_PHY_OLTF_GET_D2(he_cap_phy) WMI_HECAP_PHY_LTFGIFORHE_GET(he_cap_phy)
#define WMI_HECAP_PHY_OLTF_SET_D2(he_cap_phy, value) WMI_HECAP_PHY_LTFGIFORHE_SET(he_cap_phy, value)

/*DEPRECATED - USE WMI_HECAP_PHY_BFMENLTSGT80MHZ*/
#define WMI_HECAP_PHY_SUBFMESTS_GET_D2(he_cap_phy) WMI_HECAP_PHY_BFMESTSLT80MHZ_GET(he_cap_phy)
#define WMI_HECAP_PHY_SUBFMESTS_SET_D2(he_cap_phy, value) WMI_HECAP_PHY_BFMESTSLT80MHZ_SET(he_cap_phy, value)

/*DEPRECATED - use WMI_HECAP_PHY_PETHRESPRESENT**/
#define WMI_HECAP_PHY_PADDING_GET_D2(he_cap_phy) WMI_HECAP_PHY_PETHRESPRESENT_GET(he_cap_phy)
#define WMI_HECAP_PHY_PADDING_SET_D2(he_cap_phy, value) WMI_HECAP_PHY_PETHRESPRESENT_SET(he_cap_phy, value)


/**DO NOT USE - DEPRECATED*/
#define WMI_HECAP_PHY_DLOFMAMUMIMO_GET_D2(he_cap_phy) (0)
#define WMI_HECAP_PHY_DLOFDMAMUMIO_SET_D2(he_cap_phy, value) {;}

/*DO NOT USE - DEPRECATED**/
#define WMI_HECAP_PHY_32GI_GET_D2(he_cap_phy) (0)
#define WMI_HECAP_PHY_32GI_SET_D2(he_cap_phy, value) {;}

/*DO NOT USE - DEPRECATED**/
#define WMI_HECAP_PHY_NOSUNDIMENS_GET_D2(he_cap_phy) (0)
#define WMI_HECAP_PHY_NOSUNDIMENS_SET_D2(he_cap_phy, value) {;}

/*DO NOT USE - DEPRECATED**/
#define WMI_HECAP_PHY_40MHZNSS_GET_D2(he_cap_phy)(0)
#define WMI_HECAP_PHY_40MHZNSS_SET_D2(he_cap_phy, value) {;}


/* START TEMPORARY WORKAROUND -
 * Leave legacy names as aliases for new names, until all references to the
 * legacy names have been removed.
 */
#define WMI_HECAP_PHY_ULOFDMA_GET_D2 WMI_HECAP_PHY_ULMUMIMOOFDMA_GET
#define WMI_HECAP_PHY_ULOFDMA_SET_D2 WMI_HECAP_PHY_ULMUMIMOOFDMA_SET
/* END TEMPORARY WORKAROUND */

/* DEPRECATED - use WMI_HECAP_PHY_DCMRX or WMI_HECAP_PHY_DCMTX */
#define WMI_HECAP_PHY_DCM_GET_D2(he_cap_phy) WMI_HECAP_PHY_DCMRX_GET(he_cap_phy)
#define WMI_HECAP_PHY_DCM_SET_D2(he_cap_phy, value) WMI_HECAP_PHY_DCMRX_SET(he_cap_phy, value)

/*
 * The maximum value for NSTS-1<=80MHz,(min val 3)total that can be sent
 * to the STA in a DL MU-MIMO transmission on full or partial bandwidth
 */
#define WMI_HECAP_PHY_NSTSLT80MHZ_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 5, 3)
#define WMI_HECAP_PHY_NSTSLT80MHZ_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 5, 3, value)


/*
 * The maximum value for NSTS-1 > 80MHz (min val 3) total that can be sent
 * to the STA in a DL MU-MIMO transmission on full or partial bandwidth
 */
#define WMI_HECAP_PHY_NSTSGT80MHZ_GET_D2(he_cap_phy) WMI_GET_BITS(he_cap_phy[1], 8, 3)
#define WMI_HECAP_PHY_NSTSGT80MHZ_SET_D2(he_cap_phy, value) WMI_SET_BITS(he_cap_phy[1], 8, 3, value)

/*
 * Indicates the spatial multiplexing power save mode after receiving a
 * Trigger frame that is in operation immediately after (re)association.
 */
#define WMI_HECAP_MAC_DYNSMPWRSAVE_GET_D2(he_cap2) (0)
#define WMI_HECAP_MAC_DYNSMPWRSAVE_SET_D2(he_cap2, value) {;}

/* Indicates support for Punctured Sounding */
#define WMI_HECAP_MAC_PUNCSOUNDING_GET_D2(he_cap2) (0)
#define WMI_HECAP_MAC_PUNCSOUNDING_SET_D2(he_cap2, value) {;}

/*
 * Indicates support for receiving a Trigger frame in an HT PPDU and
 * receiving a Trigger frame in a VHT PPDU
 */
#define WMI_HECAP_MAC_HTVHTTRIGRX_GET_D2(he_cap2) (0)
#define WMI_HECAP_MAC_HTVHTTRIGRX_SET_D2(he_cap2, value) {;}

#define WMI_GET_HW_RATECODE_PREAM_V1(_rcode)     (((_rcode) >> 8) & 0x7)
#define WMI_GET_HW_RATECODE_NSS_V1(_rcode)       (((_rcode) >> 5) & 0x7)
#define WMI_GET_HW_RATECODE_RATE_V1(_rcode)      (((_rcode) >> 0) & 0x1F)
#define WMI_ASSEMBLE_RATECODE_V1(_rate, _nss, _pream) \
	(((1) << 28) | ((_pream) << 8) | ((_nss) << 5) | (_rate))

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_start_request_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** requestor id identifying the caller module */
    A_UINT32 requestor_id;
    /** beacon interval from received beacon */
    A_UINT32 beacon_interval;
    /** DTIM Period from the received beacon */
    A_UINT32 dtim_period;
    /** Flags */
    A_UINT32 flags;
    /** ssid field. Only valid for AP/GO/IBSS/BTAmp VDEV type. */
    wmi_ssid ssid;
    /** beacon/probe reponse xmit rate. Applicable for SoftAP. */
    /** This field will be invalid and ignored unless the */
    /** flags field has the WMI_UNIFIED_VDEV_START_BCN_TX_RATE_PRESENT bit. */
    /** When valid, this field contains the fixed tx rate for the beacon */
    /** and probe response frames send by the GO or SoftAP */
    A_UINT32 bcn_tx_rate;
    /** beacon/probe reponse xmit power. Applicable for SoftAP. */
    A_UINT32 bcn_txPower;
    /** number of p2p NOA descriptor(s) from scan entry */
    A_UINT32  num_noa_descriptors;
    /** Disable H/W ack. This used by WMI_VDEV_RESTART_REQUEST_CMDID.
          During CAC, Our HW shouldn't ack ditected frames */
    A_UINT32  disable_hw_ack;
    /** This field will be invalid unless the Dual Band Simultaneous (DBS) feature is enabled. */
    /** The DBS policy manager indicates the preferred number of transmit streams. */
    A_UINT32 preferred_tx_streams;
    /** This field will be invalid unless the Dual Band Simultaneous (DBS) feature is enabled. */
    /** the DBS policy manager indicates the preferred number of receive streams. */
    A_UINT32 preferred_rx_streams;
    A_UINT32 he_ops; /* refer to WMI_HEOPS_xxx macros */
    A_UINT32 cac_duration_ms;  /* in milliseconds */
    A_UINT32 regdomain;
    /* The TLVs follows this structure:
         *     wmi_channel chan;   //WMI channel
         *     wmi_p2p_noa_descriptor  noa_descriptors[]; //actual p2p NOA descriptor from scan entry
         */
} wmi_vdev_start_request_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_delete_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
} wmi_vdev_delete_cmd_fixed_param;

typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_up_cmdid_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** aid (assoc id) received in association response for STA VDEV  */
    A_UINT32 vdev_assoc_id;
    /** bssid of the BSS the VDEV is joining  */
    wmi_mac_addr vdev_bssid;
    /** bssid of transmitted AP (mbssid case) */
    wmi_mac_addr trans_bssid;
    /** the profile index of the connected non-trans ap (mbssid case). 0 means invalid */
    A_UINT32 profile_idx;
    /** the total profile numbers of non-trans aps (mbssid case). 0 means legacy AP */
    A_UINT32 profile_num;
} wmi_vdev_up_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_stop_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
} wmi_vdev_stop_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_down_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
} wmi_vdev_down_cmd_fixed_param;

typedef struct {
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
} wmi_vdev_standby_response_cmd;

typedef struct {
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
} wmi_vdev_resume_response_cmd;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_set_param_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** parameter id   */
    A_UINT32 param_id;
    /** parameter value */
    A_UINT32 param_value;
} wmi_vdev_set_param_cmd_fixed_param;

typedef struct {
    A_UINT32 key_seq_counter_l;
    A_UINT32 key_seq_counter_h;
} wmi_key_seq_counter;

#define  WMI_CIPHER_NONE         0x0  /* clear key */
#define  WMI_CIPHER_WEP          0x1
#define  WMI_CIPHER_TKIP         0x2
#define  WMI_CIPHER_AES_OCB      0x3
#define  WMI_CIPHER_AES_CCM      0x4
#define  WMI_CIPHER_WAPI         0x5
#define  WMI_CIPHER_CKIP         0x6
#define  WMI_CIPHER_AES_CMAC     0x7
#define  WMI_CIPHER_ANY          0x8
#define  WMI_CIPHER_AES_GCM      0x9
#define  WMI_CIPHER_AES_GMAC     0xa
#define  WMI_CIPHER_WAPI_GCM_SM4 0xb

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_install_key_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** MAC address used for installing   */
    wmi_mac_addr peer_macaddr;
    /** key index */
    A_UINT32 key_ix;
    /** key flags */
    A_UINT32 key_flags;
    /** key cipher, defined above */
    A_UINT32 key_cipher;
    /** key rsc counter */
    wmi_key_seq_counter key_rsc_counter;
    /** global key rsc counter */
    wmi_key_seq_counter key_global_rsc_counter;
    /** global key tsc counter */
    wmi_key_seq_counter key_tsc_counter;
    /** WAPI key rsc counter */
    A_UINT8 wpi_key_rsc_counter[16];
    /** WAPI key tsc counter */
    A_UINT8 wpi_key_tsc_counter[16];
    /** key length */
    A_UINT32 key_len;
    /** key tx mic length */
    A_UINT32 key_txmic_len;
    /** key rx mic length */
    A_UINT32 key_rxmic_len;
    /** specifies if the group_key_ix is valid, filled by the sender */
    A_UINT32 is_group_key_ix_valid;
    /** Multi group key ID */
    A_UINT32 group_key_ix;
        /*
         * Following this struct are this TLV.
         *     // actual key data
         *     A_UINT8  key_data[]; // contains key followed by tx mic followed by rx mic
         */
} wmi_vdev_install_key_cmd_fixed_param;

/** Preamble types to be used with VDEV fixed rate configuration */
typedef enum {
    WMI_RATE_PREAMBLE_OFDM,
    WMI_RATE_PREAMBLE_CCK,
    WMI_RATE_PREAMBLE_HT,
    WMI_RATE_PREAMBLE_VHT,
    WMI_RATE_PREAMBLE_HE,
} WMI_RATE_PREAMBLE;

/** Value to disable fixed rate setting */
#define WMI_FIXED_RATE_NONE    (0xff)

#define WMI_GI_400_NS 1
#define WMI_GI_800_NS 0
#define WMI_GI_1600_NS 2
#define WMI_GI_3200_NS 3

/** OCE(Optimized Connectivity_Experience) Feature flags */
#define WMI_VDEV_OCE_PROBE_REQUEST_RATE_FEATURE_BITMAP                 0x1
#define WMI_VDEV_OCE_PROBE_RESPONSE_RATE_FEATURE_BITMAP                0x2
#define WMI_VDEV_OCE_BEACON_RATE_FEATURE_BITMAP                        0x4
#define WMI_VDEV_OCE_PROBE_REQUEST_DEFERRAL_FEATURE_BITMAP             0x8
#define WMI_VDEV_OCE_FILS_DISCOVERY_FRAME_FEATURE_BITMAP               0x10
#define WMI_VDEV_OCE_ESP_FEATURE_BITMAP                                0x20
#define WMI_VDEV_OCE_REASSOC_REJECT_FEATURE_BITMAP                     0x40

/** the definition of different VDEV parameters */
typedef enum {
    /** RTS Threshold */
    WMI_VDEV_PARAM_RTS_THRESHOLD = 0x1,
    /** Fragmentation threshold */
    WMI_VDEV_PARAM_FRAGMENTATION_THRESHOLD,                   /* 0x2 */
    /** beacon interval in TUs */
    WMI_VDEV_PARAM_BEACON_INTERVAL,                           /* 0x3 */
    /** Listen interval in TUs */
    WMI_VDEV_PARAM_LISTEN_INTERVAL,                           /* 0x4 */
    /** muticast rate in Mbps */
    WMI_VDEV_PARAM_MULTICAST_RATE,                            /* 0x5 */
    /** management frame rate in Mbps */
    WMI_VDEV_PARAM_MGMT_TX_RATE,                              /* 0x6 */
    /** slot time (long vs short) */
    WMI_VDEV_PARAM_SLOT_TIME,                                 /* 0x7 */
    /** preamble (long vs short) */
    WMI_VDEV_PARAM_PREAMBLE,                                  /* 0x8 */
    /** SWBA time (time before tbtt in msec) */
    WMI_VDEV_PARAM_SWBA_TIME,                                 /* 0x9 */
    /** time period for updating VDEV stats */
    WMI_VDEV_STATS_UPDATE_PERIOD,                             /* 0xa */
    /** age out time in msec for frames queued for station in power save*/
    WMI_VDEV_PWRSAVE_AGEOUT_TIME,                             /* 0xb */
    /** Host SWBA interval (time in msec before tbtt for SWBA event generation) */
    WMI_VDEV_HOST_SWBA_INTERVAL,                              /* 0xc */
    /** DTIM period (specified in units of num beacon intervals) */
    WMI_VDEV_PARAM_DTIM_PERIOD,                               /* 0xd */
    /** scheduler air time limit for this VDEV. used by off chan scheduler  */
    WMI_VDEV_OC_SCHEDULER_AIR_TIME_LIMIT,                     /* 0xe */
    /** enable/dsiable WDS for this VDEV  */
    WMI_VDEV_PARAM_WDS,                                       /* 0xf */
    /** ATIM Window */
    WMI_VDEV_PARAM_ATIM_WINDOW,                               /* 0x10 */
    /** BMISS max */
    WMI_VDEV_PARAM_BMISS_COUNT_MAX,                           /* 0x11 */
    /** BMISS first time */
    WMI_VDEV_PARAM_BMISS_FIRST_BCNT,                          /* 0x12 */
    /** BMISS final time */
    WMI_VDEV_PARAM_BMISS_FINAL_BCNT,                          /* 0x13 */
    /** WMM enables/disabled */
    WMI_VDEV_PARAM_FEATURE_WMM,                               /* 0x14 */
    /** Channel width */
    WMI_VDEV_PARAM_CHWIDTH,                                   /* 0x15 */
    /** Channel Offset */
    WMI_VDEV_PARAM_CHEXTOFFSET,                               /* 0x16 */
    /** Disable HT Protection */
    WMI_VDEV_PARAM_DISABLE_HTPROTECTION,                      /* 0x17 */
    /** Quick STA Kickout */
    WMI_VDEV_PARAM_STA_QUICKKICKOUT,                          /* 0x18 */
    /** Rate to be used with Management frames */
    WMI_VDEV_PARAM_MGMT_RATE,                                 /* 0x19 */
    /** Protection Mode */
    WMI_VDEV_PARAM_PROTECTION_MODE,                           /* 0x1a */
    /** Fixed rate setting
     * The top nibble is used to select which format to use for encoding
     * the rate specification: 0xVXXXXXXX
     * If V == 0b0000: format is same as before: 0x000000RR
     * If V == 0b0001: format is: 0x1000RRRR.
     *                 This will be output of WMI_ASSEMBLE_RATECODE_V1
     * The host shall use the new V1 format (and set V = 0x1) if the target
     * indicates 802.11ax support via the WMI_SERVICE_11AX flag, or if the
     * system is configured with Nss > 4 (either at compile time within the
     * host driver, or through WMI_SERVICE_READY PHY capabilities provided
     * by the target).
     */
    WMI_VDEV_PARAM_FIXED_RATE,                                /* 0x1b */
    /**
     * 11AX: GI =
     *	   WMI_GI_400_NS, WMI_GI_800_NS, WMI_GI_1600_NS, or WMI_GI_3200_NS
     * 11N: SGI=WMI_GI_400_NS
     */
    WMI_VDEV_PARAM_SGI,                                       /* 0x1c */
    /** Enable LDPC */
    WMI_VDEV_PARAM_LDPC,                                      /* 0x1d */
    /** Enable Tx STBC */
    WMI_VDEV_PARAM_TX_STBC,                                   /* 0x1e */
    /** Enable Rx STBC */
    WMI_VDEV_PARAM_RX_STBC,                                   /* 0x1f */
    /** Intra BSS forwarding  */
    WMI_VDEV_PARAM_INTRA_BSS_FWD,                             /* 0x20 */
    /** Setting Default xmit key for Vdev */
    WMI_VDEV_PARAM_DEF_KEYID,                                 /* 0x21 */
    /** NSS width */
    WMI_VDEV_PARAM_NSS,                                       /* 0x22 */
    /** Set the custom rate for the broadcast data frames */
    WMI_VDEV_PARAM_BCAST_DATA_RATE,                           /* 0x23 */
    /** Set the custom rate (rate-code) for multicast data frames */
    WMI_VDEV_PARAM_MCAST_DATA_RATE,                           /* 0x24 */
    /** Tx multicast packet indicate Enable/Disable */
    WMI_VDEV_PARAM_MCAST_INDICATE,                            /* 0x25 */
    /** Tx DHCP packet indicate Enable/Disable */
    WMI_VDEV_PARAM_DHCP_INDICATE,                             /* 0x26 */
    /** Enable host inspection of Tx unicast packet to unknown destination */
    WMI_VDEV_PARAM_UNKNOWN_DEST_INDICATE,                     /* 0x27 */

    /* The minimum amount of time AP begins to consider STA inactive */
    WMI_VDEV_PARAM_AP_KEEPALIVE_MIN_IDLE_INACTIVE_TIME_SECS,  /* 0x28 */

    /* An associated STA is considered inactive when there is no recent TX/RX
     * activity and no downlink frames are buffered for it. Once a STA exceeds
     * the maximum idle inactive time, the AP will send an 802.11 data-null as
     * a keep alive to verify the STA is still associated. If the STA does ACK
     * the data-null, or if the data-null is buffered and the STA does not
     * retrieve it, the STA will be considered unresponsive (see
     * WMI_VDEV_AP_KEEPALIVE_MAX_UNRESPONSIVE_TIME_SECS). */
    WMI_VDEV_PARAM_AP_KEEPALIVE_MAX_IDLE_INACTIVE_TIME_SECS,  /* 0x29 */

    /* An associated STA is considered unresponsive if there is no recent
     * TX/RX activity and downlink frames are buffered for it. Once a STA
     * exceeds the maximum unresponsive time, the AP will send a
     * WMI_STA_KICKOUT event to the host so the STA can be deleted. */
    WMI_VDEV_PARAM_AP_KEEPALIVE_MAX_UNRESPONSIVE_TIME_SECS,   /* 0x2a */

    /* Enable NAWDS : MCAST INSPECT Enable, NAWDS Flag set */
    WMI_VDEV_PARAM_AP_ENABLE_NAWDS,                           /* 0x2b */
    /** Enable/Disable RTS-CTS */
    WMI_VDEV_PARAM_ENABLE_RTSCTS,                             /* 0x2c */
    /* Enable TXBFee/er */
    WMI_VDEV_PARAM_TXBF,                                      /* 0x2d */

    /**Set packet power save */
    WMI_VDEV_PARAM_PACKET_POWERSAVE,                          /* 0x2e */

    /**Drops un-encrypted packets if any received in an encryted connection
     * otherwise forwards to host
     */
    WMI_VDEV_PARAM_DROP_UNENCRY,                              /* 0x2f */

    /*
     * Set TX encap type.
     *
     * enum wmi_pkt_type is to be used as the parameter
     * specifying the encap type.
     */
    WMI_VDEV_PARAM_TX_ENCAP_TYPE,                             /* 0x30 */

    /*
     * Try to detect stations that woke-up and exited power save but did not
     * successfully transmit data-null with PM=0 to AP. When this happens,
     * STA and AP power save state are out-of-sync. Use buffered but
     * undelivered MSDU to the STA as a hint that the STA is really awake
     * and expecting normal ASAP delivery, rather than retrieving BU with
     * PS-Poll, U-APSD trigger, etc.
     *
     * 0 disables out-of-sync detection. Maximum time is 255 seconds.
     */
    WMI_VDEV_PARAM_AP_DETECT_OUT_OF_SYNC_SLEEPING_STA_TIME_SECS,  /* 0x31 */

    /* Enable/Disable early rx dynamic adjust feature.
     * Early-rx dynamic adjust is a advance power save feature.
     * Early-rx is a wakeup duration before exact TBTT,which is deemed necessary to provide a cushion for various
     * timing discrepancies in the system.
     * In current code branch, the duration is set to a very conservative fix value to make sure the drift impact is minimum.
     * The fix early-tx will result in the unnessary power consume, so a dynamic early-rx adjust algorithm can be designed
     * properly to minimum the power consume.*/
    WMI_VDEV_PARAM_EARLY_RX_ADJUST_ENABLE,                    /* 0x32 */

    /* set target bmiss number per sample cycle if bmiss adjust was chosen.
     * In this adjust policy,early-rx is adjusted by comparing the current bmiss rate to target bmiss rate
     * which can be set by user through WMI command.
     */
    WMI_VDEV_PARAM_EARLY_RX_TGT_BMISS_NUM,                    /* 0x33 */

    /* set sample cycle(in the unit of beacon interval) if bmiss adjust was chosen */
    WMI_VDEV_PARAM_EARLY_RX_BMISS_SAMPLE_CYCLE,               /* 0x34 */

    /* set slop_step */
    WMI_VDEV_PARAM_EARLY_RX_SLOP_STEP,                        /* 0x35 */

    /* set init slop */
    WMI_VDEV_PARAM_EARLY_RX_INIT_SLOP,                        /* 0x36 */

    /* pause adjust enable/disable */
    WMI_VDEV_PARAM_EARLY_RX_ADJUST_PAUSE,                     /* 0x37 */

    /* Set channel pwr limit value of the vdev the minimal value of all
     * vdevs operating on this channel will be set as channel tx power
     * limit, which is used to configure ratearray
     */
    WMI_VDEV_PARAM_TX_PWRLIMIT,                               /* 0x38 */

    /* set the count of snr value for calculation in snr monitor */
    WMI_VDEV_PARAM_SNR_NUM_FOR_CAL,                           /* 0x39 */

    /** Roaming offload */
    WMI_VDEV_PARAM_ROAM_FW_OFFLOAD,                           /* 0x3a */

    /** Enable Leader request RX functionality for RMC */
    WMI_VDEV_PARAM_ENABLE_RMC,                                /* 0x3b */

   /* IBSS does not have deauth/disassoc, vdev has to detect peer gone event
     * by himself. If the beacon lost time exceed this threshold, the peer is
     * thought to be gone. */
    WMI_VDEV_PARAM_IBSS_MAX_BCN_LOST_MS,                      /* 0x3c */

    /** max rate in kpbs, transmit rate can't go beyond it */
    WMI_VDEV_PARAM_MAX_RATE,                                  /* 0x3d */

    /* enable/disable drift sample. 0: disable; 1: clk_drift; 2: ap_drift; 3 both clk and ap drift*/
    WMI_VDEV_PARAM_EARLY_RX_DRIFT_SAMPLE,                     /* 0x3e */

    /* set Tx failure count threshold for the vdev */
    WMI_VDEV_PARAM_SET_IBSS_TX_FAIL_CNT_THR,                  /* 0x3f */

    /* set ebt resync timeout value, in the unit of TU */
    WMI_VDEV_PARAM_EBT_RESYNC_TIMEOUT,                        /* 0x40 */

    /* Enable Aggregation State Trigger Event */
    WMI_VDEV_PARAM_AGGR_TRIG_EVENT_ENABLE,                    /* 0x41 */

    /* This parameter indicates whether IBSS station can enter into power save
    * mode by sending Null frame (with PM=1). When not allowed, IBSS station has to stay
    * awake all the time and should never set PM=1 in its transmitted frames.
    * This parameter is meaningful/valid only when WMI_VDEV_PARAM_ATIM_WINDOW_LENGTH
    * is non-zero. */
    WMI_VDEV_PARAM_IS_IBSS_POWER_SAVE_ALLOWED,                /* 0x42 */

    /* This parameter indicates if this station can enter into power collapse
    * for the remaining beacon interval after the ATIM window.
    * This parameter is meaningful/valid only when WMI_VDEV_PARAM_IS_IBSS_POWER_SAVE_ALLOWED
    * is set to TRUE. */
    WMI_VDEV_PARAM_IS_POWER_COLLAPSE_ALLOWED,                 /* 0x43 */

    /* This parameter indicates whether IBSS station exit power save mode and
    * enter power active state (by sending Null frame with PM=0 in the immediate ATIM Window)
    * whenever there is a TX/RX activity. */
    WMI_VDEV_PARAM_IS_AWAKE_ON_TXRX_ENABLED,                  /* 0x44 */

    /* If Awake on TX/RX activity is enabled, this parameter indicates
    * the data inactivity time in number of beacon intervals after which
    * IBSS station reenters power save by sending Null frame with PM=1. */
    WMI_VDEV_PARAM_INACTIVITY_CNT,                            /* 0x45 */

    /* Inactivity time in msec after which TX Service Period (SP) is
    * terminated by sending a Qos Null frame with EOSP.
    * If value is 0, TX SP is terminated with the last buffered packet itself
    * instead of waiting for the inactivity timeout. */
    WMI_VDEV_PARAM_TXSP_END_INACTIVITY_TIME_MS,               /* 0x46 */

    /** DTIM policy */
    WMI_VDEV_PARAM_DTIM_POLICY,                               /* 0x47 */

    /* When IBSS network is initialized, PS-supporting device
    * does not enter protocol sleep state during first
    * WMI_VDEV_PARAM_IBSS_PS_WARMUP_TIME_SECS seconds. */
    WMI_VDEV_PARAM_IBSS_PS_WARMUP_TIME_SECS,                  /* 0x48 */

    /* Enable/Disable 1 RX chain usage during the ATIM window */
    WMI_VDEV_PARAM_IBSS_PS_1RX_CHAIN_IN_ATIM_WINDOW_ENABLE,   /* 0x49 */

    /* RX Leak window is the time driver waits before shutting down
     * the radio or switching the channel and after receiving an ACK
     * for a data frame with PM bit set) */
    WMI_VDEV_PARAM_RX_LEAK_WINDOW,                            /* 0x4a */

    /** Averaging factor(16 bit value) is used in the calculations to
     * perform averaging of different link level statistics like average
     * beacon spread or average number of frames leaked */
    WMI_VDEV_PARAM_STATS_AVG_FACTOR,                          /* 0x4b */

    /** disconnect threshold, once the consecutive error for specific peer
     * exceed this threhold, FW will send kickout event to host */
    WMI_VDEV_PARAM_DISCONNECT_TH,                             /* 0x4c */

    /** The rate_code of RTS_CTS changed by host. Now FW can support
     * more non-HT rates rather than 1Mbps or 6Mbps */
    WMI_VDEV_PARAM_RTSCTS_RATE,                               /* 0x4d */

    /** This parameter indicates whether using a long duration RTS-CTS
     * protection when a SAP goes off channel in MCC mode */
    WMI_VDEV_PARAM_MCC_RTSCTS_PROTECTION_ENABLE,              /* 0x4e */

    /** This parameter indicates whether using a broadcast probe response
     * to increase the detectability of SAP in MCC mode */
    WMI_VDEV_PARAM_MCC_BROADCAST_PROBE_ENABLE,                /* 0x4f */

    /** This parameter indicates the power backoff in percentage
     * currently supports 100%, 50%, 25%, 12.5%, and minimum
     * Host passes 0, 1, 2, 3, 4 to Firmware
     * 0 --> 100% --> no changes, 1 --> 50% --> -3dB,
     * 2 --> 25% --> -6dB, 3 --> 12.5% --> -9dB, 4 --> minimum --> -32dB
     */
    WMI_VDEV_PARAM_TXPOWER_SCALE,                             /* 0x50 */

    /** TX power backoff in dB: tx power -= param value
     * Host passes values(DB) to Halphy, Halphy reduces the power table
     * by the values.  Safety check will happen in Halphy.
     */
    WMI_VDEV_PARAM_TXPOWER_SCALE_DECR_DB,                     /* 0x51 */

    /** Multicast to Unicast conversion setting */
    WMI_VDEV_PARAM_MCAST2UCAST_SET,                           /* 0x52 */

    /** Total number of HW retries */
    WMI_VDEV_PARAM_RC_NUM_RETRIES,                            /* 0x53 */

    /** Max tx percentage for cabq */
    WMI_VDEV_PARAM_CABQ_MAXDUR,                               /* 0x54 */

    /** MFPTEST settings */
    WMI_VDEV_PARAM_MFPTEST_SET,                               /* 0x55 */

    /** RTS Fixed rate setting */
    WMI_VDEV_PARAM_RTS_FIXED_RATE,                            /* 0x56 */

    /** VHT SGI MASK */
    WMI_VDEV_PARAM_VHT_SGIMASK,                               /* 0x57 */

    /** VHT80 Auto Rate MASK */
    WMI_VDEV_PARAM_VHT80_RATEMASK,                            /* 0x58 */

    /** set Proxy STA features for this vap */
    WMI_VDEV_PARAM_PROXY_STA,                                 /* 0x59 */

    /** set virtual cell mode - enable/disable */
    WMI_VDEV_PARAM_VIRTUAL_CELL_MODE,                         /* 0x5a */

    /** Set receive packet type */
    WMI_VDEV_PARAM_RX_DECAP_TYPE,                             /* 0x5b */

    /** Set ratemask with specific Bandwidth and NSS */
    WMI_VDEV_PARAM_BW_NSS_RATEMASK,                           /* 0x5c */

    /** Set SENSOR Support */
    WMI_VDEV_PARAM_SENSOR_AP,                                 /* 0x5d */

    /** Set beacon rate */
    WMI_VDEV_PARAM_BEACON_RATE,                               /* 0x5e */

    /** Enable CTS to self for DTIM beacon */
    WMI_VDEV_PARAM_DTIM_ENABLE_CTS,                           /* 0x5f */

    /** Disable station kickout at Vap level */
    WMI_VDEV_PARAM_STA_KICKOUT,                               /* 0x60 */

    /* VDEV capabilities */
    WMI_VDEV_PARAM_CAPABILITIES, /* see capabilities defs below */ /* 0x61 */

    /**
     * Increment TSF in micro seconds to avoid beacon collision on mesh VAP.
     * The host must ensure that either no other vdevs share the TSF with
     * this vdev, or else that it is acceptable to apply this TSF adjustment
     * to all vdevs sharing the TSF.
     */
    WMI_VDEV_PARAM_TSF_INCREMENT,                             /* 0x62 */
    /*
     * Disable/Enable AMPDU of vdev per AC:
     * bit | AC
     * --------
     *  0  | VO
     *  1  | VI
     *  2  | BE
     *  3  | BK
     * A value of 0 in a given bit disables A-MPDU aggregation for
     * that AC; a value of 1 enables A-MPDU aggregation
     */
    WMI_VDEV_PARAM_AMPDU_PER_AC,                              /* 0x63 */

    /**
     * Vdev level rx filter of from-ds / to-ds / no-ds / ta / ra frames.
     * Used mainly for mesh-vap.
     * The parameter value delivered with the RX_FILTER vdev param contains
     * a bit-or mask of wmi_vdev_param_filter enum values.
     */
    WMI_VDEV_PARAM_RX_FILTER,                                 /* 0x64 */

    /** vdev-specific mgmt tx power in dBm units (signed integer value) */
    WMI_VDEV_PARAM_MGMT_TX_POWER,                             /* 0x65 */

    /** Vdev level non aggregration/11g sw retry threshold. 0-disable, min:0, max:31, default:15 */
    WMI_VDEV_PARAM_NON_AGG_SW_RETRY_TH,                       /* 0x66 */
    /** Vdev level aggregration sw retry threshold. 0-disable, min:0, max:31, default:15 */
    WMI_VDEV_PARAM_AGG_SW_RETRY_TH,                           /* 0x67 */

    /** disable dynamic bw RTS **/
    WMI_VDEV_PARAM_DISABLE_DYN_BW_RTS,                        /* 0x68 */

    /** per ssid (vdev) based ATF strict/fair scheduling policy
     *  param values are WMI_ATF_SSID_FAIR_SCHED or WMI_ATF_SSID_STRICT_SCHED
     */
    WMI_VDEV_PARAM_ATF_SSID_SCHED_POLICY,                     /* 0x69 */

    /** Enable or disable Dual carrier modulation
     * valid values: 0-Disable DCM, 1-Enable DCM.
     */
    WMI_VDEV_PARAM_HE_DCM,                                    /* 0x6a */

    /** Enable or disable Extended range
     * valid values: 0-Disable ER, 1-Enable ER.
     */
    WMI_VDEV_PARAM_HE_RANGE_EXT,                              /* 0x6b */

    /* enable or disable BCAST probe response feature */
    WMI_VDEV_PARAM_ENABLE_BCAST_PROBE_RESPONSE,               /* 0x6c */

    /* param to specify probe request Tx delay during Fast Initial Link Setup */
    WMI_VDEV_PARAM_FILS_MAX_CHANNEL_GUARD_TIME, /* units = milliseconds */ /* 0x6d */

    /* enable or disable NOA for P2P GO */
    WMI_VDEV_PARAM_DISABLE_NOA_P2P_GO,                        /* 0x6e */

    /** Per band user management frame fix rate setting
     *  BIT 31: enable (1) or disable (0) mgmt fix rate for 5G
     *  BIT 30: enable (1) or disable (0) mgmt fix rate for 2G
     *
     *  BIT 23: 11ax (1) or legacy (0) rate code
     *  BITS [22..12]: rate code for 5G
     *
     *  BIT 11: 11ax (1) or legacy (0) rate code
     *  BITS [10..0]: rate code for 2G
     */
    WMI_VDEV_PARAM_PER_BAND_MGMT_TX_RATE,                     /* 0x6f */
    /* This should be called before WMI_VDEV_PARAM_TXBF */
    WMI_VDEV_PARAM_11AX_TXBF,                                 /* 0x70 */

    /** This parameter indicates whether VDEV is SMPS intolerant.
     * I.e. - SMPS action frame cannot be transmitted by the VDEV to
     * dynamically change the RX NSS.
     *
     * valid values: 1 - VDEV is SMPS intolerant, 0 - VDEV is SMPS tolerant
     */
    WMI_VDEV_PARAM_SMPS_INTOLERANT,                           /* 0x71 */

    /** specify probe request Tx delay for scans triggered on this VDEV */
    WMI_VDEV_PARAM_PROBE_DELAY, /* units = milliseconds */    /* 0x72 */

    /** specify the time gap between each set of probe request transmissions.
     * The number of probe requests in each set depends on the ssid_list and
     * bssid_list in the scan request.
     * This parameter will be applied only for scans triggered on this VDEV.
     */
    WMI_VDEV_PARAM_REPEAT_PROBE_TIME, /* units = milliseconds */ /* 0x73 */

    /** specify the HE LTF setting that should be used for fixed rate
     * transmissions.
     *
     * Expects values of WMI_HE_LTF_DEFAULT, WMI_HE_LTF_1X, WMI_HE_LTF_2X,
     * or WMI_HE_LTF_4X.
     */
    WMI_VDEV_PARAM_HE_LTF,                                    /* 0x74 */

    /** VDEV parameter to configure the number of TX chains to use for
     *  a/b/g rates.
     *  bit 0~15  : 11b mode TX chain number.
     *  bit 16~31 : 11ag mode TX chain number.
     *
     *  valid values:
     *      Should not exceed the maximum number of supported TX Chains
     *      0 - Used to reset the vote. Acts as Don't Care
     */
    WMI_VDEV_PARAM_ABG_MODE_TX_CHAIN_NUM,                     /* 0x75 */

    /**
     * Enable or disable the multi group key feature on this VDEV.
     * used for NAN APP and VLAN Tagging
     */
    WMI_VDEV_PARAM_ENABLE_MULTI_GROUP_KEY,                    /* 0x76 */

    /*
     * Specify the total number of multi group key on this VDEV.
     */
    WMI_VDEV_PARAM_NUM_GROUP_KEYS,                            /* 0x77 */

    /** VDEV parameter to enable or disable various OCE features */
    WMI_VDEV_PARAM_ENABLE_DISABLE_OCE_FEATURES,              /* 0x78 */

    /*
     * Set/Clear 3 least-significant bits to
     * Disable or Enable rate drop down for MGMT, SU data and MU data pkts
     *
     * bit 0 -> If set MGMT Pkt rate drop down is enabled else disabled
     * bit 1 -> If set SU data Pkt rate drop down is enabled else disabled
     * bit 2 -> If set MU data Pkt rate drop down is enabled else disabled
     * bits 31:3 -> Reserved bits. should be set to zero.
     */
    WMI_VDEV_PARAM_RATE_DROPDOWN_BMAP,                       /* 0x79 */

    /** VDEV parameter to config modulated DTIM count */
    WMI_VDEV_PARAM_MODDTIM_CNT,                              /* 0x7a */

    /**
     * VDEV parameter to config max listen interval,
     * when modulated DTIM is enabled.
     * Units are 100TU.
     */
    WMI_VDEV_PARAM_MAX_LI_OF_MODDTIM,                        /* 0x7b */

    /** VDEV parameter to config dynamic DTIM count */
    WMI_VDEV_PARAM_DYNDTIM_CNT,                              /* 0x7c */

    /** VDEV parameter to enable or disable RTT responder role
      * valid values: 0-Disable responder role 1-Enable responder role
      */
    WMI_VDEV_PARAM_ENABLE_DISABLE_RTT_RESPONDER_ROLE,        /* 0x7d */

    /** Parameter to configure BA mode.
     * Valid values: 0- Auto mode,
     *               1- Manual mode(addba req not sent).
     *               2- buffer size 64
     *               3- buffer size 256
     */
    WMI_VDEV_PARAM_BA_MODE,                                 /* 0x7e */

    /**
     * VDEV parameter to force to set modulate DTIM count as listen interval,
     * no matter whether WoW is enabled
     * Default: Disabled.
     * Valid values: 0- Disabled,
     *               1- Enabled.
     */
    WMI_VDEV_PARAM_FORCED_MODDTIM_ENABLE,                   /* 0x7f */

    /** specify the setting that are valid for auto rate transmissions.
     * bits 7:0 (LTF): When bitmask is set, then corresponding LTF value is
     *                 used for auto rate.
     *     BIT0   = 1 (WMI_HE_LTF_1X)
     *     BIT1   = 1 (WMI_HE_LTF_2X)
     *     BIT2   = 1 (WMI_HE_LTF_4X)
     *     BIT3-7 = Reserved bits.
     * bits 15:8 (SGI): When bitmask is set, then corresponding SGI value is
     *                 used for auto rate.
     *     BIT8     = 1 (400 NS)
     *     BIT9     = 1 (800 NS)
     *     BIT10    = 1 (1600 NS)
     *     BIT11    = 1 (3200 NS)
     *     BIT12-15 = Reserved bits.
     * bits 31:16: Reserved bits. should be set to zero.
     */
    WMI_VDEV_PARAM_AUTORATE_MISC_CFG,                       /* 0x80 */

    /** VDEV parameter to enable or disable RTT initiator mac address
      * randomization.
      * Default: Disabled.
      * valid values: 0-Disable random mac 1-Enable random mac
      */
    WMI_VDEV_PARAM_ENABLE_DISABLE_RTT_INITIATOR_RANDOM_MAC, /* 0x81 */

    /**
     * For each AC, configure how many tx retries to send without RTS
     * before enabling RTS
     *  bits 0:7    :BE
     *  bits 8:15   :BK
     *  bits 16:23  :VI
     *  bits 24:31  :VO
     * A value of 0 in specific AC means default configuration for that AC.
     */
    WMI_VDEV_PARAM_TX_RETRIES_BEFORE_RTS_PER_AC,           /* 0x82 */

    /**
     * Parameter to enable/disable AMSDU aggregation size auto-selection logic.
     * We have logic where AMSDU aggregation size is dynamically decided
     * based on MCS. That logic is enabled by default.
     * For certain tests, we need a method to disable this optimization,
     * and base AMSDU size only on the peer's capability rather than our logic.
     * A value of 0 means disable internal optimization,
     * 1 means enable internal optimzation.
     */
    WMI_VDEV_PARAM_AMSDU_AGGREGATION_SIZE_OPTIMIZATION,    /* 0x83 */

    /**
     * In RAW mode, FW will not know whether the encryption is enabled
     * on this vdev or not.
     * Because of this, FW will not program the right info into the
     * RawNwifi TLV resulting in the connection failure in RAW mode.
     * So to program the right info, FW should know whether the security
     * is enabled on this VDEV.
     * Host will send this VDEV param command (With Value = 1) in case of
     * RAW secure mode.
     */
    WMI_VDEV_PARAM_RAW_IS_ENCRYPTED,                       /* 0x84 */

    /**
     * Dynamically enable/disable green tx (GTX) on supported rates.
     * Host will set this param to 1 for enabling GTX & 0 for disabling it.
     * Note: If GTX was already running (since the global GTX control
     * resides with default BDF setting) & host wants to disable GTX,
     * the VDEV does not exercise any more TPC changes on the GTX supported
     * rates & goes to a default GTX SM where all PPDU's sent on default TPC.
     * Whenever, host wants to reenable GTX, the enable command resumes the
     * GTX functionality & TPC convergence to lower power levels can be
     * attained as long as PER on GTX supported rates is within the pre-defined
     * PER margin configured through the BDF.
     */
    WMI_VDEV_PARAM_GTX_ENABLE,                             /* 0x85 */

    /*
     * Enable/Disable multicast buffer.
     * A FLAG to enable & disable buffering of multicast frames at AP
     * when stations are in Power Save mode.
     * Once AP disables buffering of multicast frame,
     * clients which goes into Power save mode will not receive these frames.
     * by default MCAST buffering will be enabled.
     * (CABQ = Content After Beacon Queue = multicast queue)
     * Host will send this VDEV param command,
     * With Value = 1 means fw will disable the MCAST buffering
     * With Value = 0 means fw will enable the MCAST buffering.
     */
    WMI_VDEV_PARAM_DISABLE_CABQ,                          /* 0x86 */

    /**
      * For SU and MU sounding
      * switch between su ac/ax sounding and mu ac/ax sounding
      * switch between triggered/ non-triggered on ax sounding enabled.
      * each bit toggles the corresponding modes by enabling/disabling
      *
      * Bit 1 doesn't carry any operation for now and may change later,
      * so reserved.
      *
      *-----------------------
      * bit(0)   |    mode
      *-----------------------
      *       0  |  AC
      *       1  |  AX
      *-----------------------
      *
      * bit(1)   |  Reserved
      *
      *-----------------------
      * bit(2)   |    mode
      *-----------------------
      *       0  |  SU
      *       1  |  MU
      *-----------------------
      * bit(3)   |    mode
      *-----------------------
      *       0  |  non -triggered
      *       1  |  triggered
      */
    WMI_VDEV_PARAM_SET_HE_SOUNDING_MODE,                  /* 0x87 */

    /** Fixed rate setting used in UL Trigger
     * The top nibble is used to select which format to use for encoding
     * the rate specification: 0xVXXXXXXX, V must be 1 for the UL
     * format.
     * If V == 0b0001: format is: 0x1000RRRR.
     *                 This will be output of WMI_ASSEMBLE_RATECODE_V1
     *
     * This parameter controls the UL OFDMA and UL MU-MIMO vdev fixed rate.
     */
    WMI_VDEV_PARAM_UL_FIXED_RATE,                         /* 0x88 */

    /**
     * Uplink MU-MIMO & OFDMA GI configuration used in UL Trigger
     * 11AX: GI =
     *     WMI_GI_400_NS, WMI_GI_800_NS, WMI_GI_1600_NS, or WMI_GI_3200_NS
     * 11N: SGI=WMI_GI_400_NS
     */
    WMI_VDEV_PARAM_UL_GI,                                 /* 0x89 */

    /** Enable/Disable LDPC in UL Trigger */
    WMI_VDEV_PARAM_UL_LDPC,                               /* 0x8A */

    /** Max NSS allowed in UL Trigger */
    WMI_VDEV_PARAM_UL_NSS,                                /* 0x8B */

    /** Enable/Disable STBC in UL Trigger */
    WMI_VDEV_PARAM_UL_STBC,                               /* 0x8C */

    /** specify the HE LTF setting that should be used for fixed rate
     * uplink transmissions.
     *
     * Expects values of WMI_HE_LTF_DEFAULT, WMI_HE_LTF_1X, WMI_HE_LTF_2X,
     * or WMI_HE_LTF_4X.
     */
    WMI_VDEV_PARAM_UL_HE_LTF,                             /* 0x8D */

    /** Uplink OFDMA PPDU bandwidth (0: 20MHz, 1: 40MHz, 2: 80Mhz, 3: 160MHz)*/
    WMI_VDEV_PARAM_UL_PPDU_BW,                            /* 0x8E */

    /** Enable/Disable FW handling MU EDCA change from AP (1: En, 0:Dis)  */
    WMI_VDEV_PARAM_MU_EDCA_FW_UPDATE_EN,                  /* 0x8F */

    /** Update dot11ObssNbruToleranceTime in fw. Param value: seconds */
    WMI_VDEV_PARAM_UPDATE_OBSS_RU_TOLERANCE_TIME,         /* 0x90 */

    /** Parameter used when MTU size is sent by the host
     * In particular, this configuration message is used for cases where the
     * encapsulation header results in a larger max frame size than the
     * typical 802.3 + SNAP/LLC frame.
     */
    WMI_VDEV_PARAM_MAX_MTU_SIZE,                          /* 0x91 */

    /** Send every nth beacon to host
     * if value of n is 0, it means this Nth beacon upload is disabled
     */
    WMI_VDEV_PARAM_NTH_BEACON_TO_HOST,                    /* 0x92 */


    /*=== ADD NEW VDEV PARAM TYPES ABOVE THIS LINE ===
     * The below vdev param types are used for prototyping, and are
     * prone to change.
     */
    WMI_VDEV_PARAM_PROTOTYPE = 0x8000,
        /* 11AX SPECIFIC defines */
        /* USE this for BSS color change */
        WMI_VDEV_PARAM_BSS_COLOR,                             /* 0x8001 */

        /*
         * Enable / disable trigger access for a AP vdev's peers.
         * For a STA mode vdev this will enable/disable triggered access
         * and enable/disable Multi User mode of operation.
         * A value of 0 in a given bit disables corresponding mode.
         * bit | hemu mode
         * ---------------
         *  0  | HE SUBFEE
         *  1  | HE SUBFER
         *  2  | HE MUBFEE
         *  3  | HE MUBFER
         *  4  | DL OFDMA, for AP its DL Tx OFDMA for Sta its Rx OFDMA
         *  5  | UL OFDMA, for AP its Tx OFDMA trigger for Sta its Rx OFDMA
         *     |           trigger receive & UL response
         *  6  | UL MUMIMO
         */
        WMI_VDEV_PARAM_SET_HEMU_MODE,                         /* 0x8002 */
        WMI_VDEV_PARAM_HEOPS_0_31,                            /* 0x8003 */
        WMI_VDEV_PARAM_OBSSPD,                                /* 0x8004 */
    /*=== END VDEV_PARAM_PROTOTYPE SECTION ===*/
} WMI_VDEV_PARAM;

#define WMI_VDEV_HE_SUBFEE_IS_ENABLED(hemu_mode) WMI_GET_BITS(hemu_mode, 0, 1)
#define WMI_VDEV_HE_SUBFEE_ENABLE(hemu_mode) WMI_SET_BITS(hemu_mode, 0, 1, 1)
#define WMI_VDEV_HE_SUBFEE_DISABLE(hemu_mode) WMI_SET_BITS(hemu_mode, 0, 1, 0)

#define WMI_VDEV_HE_SUBFER_IS_ENABLED(hemu_mode) WMI_GET_BITS(hemu_mode, 1, 1)
#define WMI_VDEV_HE_SUBFER_ENABLE(hemu_mode) WMI_SET_BITS(hemu_mode, 1, 1, 1)
#define WMI_VDEV_HE_SUBFER_DISABLE(hemu_mode) WMI_SET_BITS(hemu_mode, 1, 1, 0)

#define WMI_VDEV_HE_MUBFEE_IS_ENABLED(hemu_mode) WMI_GET_BITS(hemu_mode, 2, 1)
#define WMI_VDEV_HE_MUBFEE_ENABLE(hemu_mode) WMI_SET_BITS(hemu_mode, 2, 1, 1)
#define WMI_VDEV_HE_MUBFEE_DISABLE(hemu_mode) WMI_SET_BITS(hemu_mode, 2, 1, 0)

#define WMI_VDEV_HE_MUBFER_IS_ENABLED(hemu_mode) WMI_GET_BITS(hemu_mode, 3, 1)
#define WMI_VDEV_HE_MUBFER_ENABLE(hemu_mode) WMI_SET_BITS(hemu_mode, 3, 1, 1)
#define WMI_VDEV_HE_MUBFER_DISABLE(hemu_mode) WMI_SET_BITS(hemu_mode, 3, 1, 0)

#define WMI_VDEV_HE_DLOFDMA_IS_ENABLED(hemu_mode) WMI_GET_BITS(hemu_mode, 4, 1)
#define WMI_VDEV_HE_DLOFDMA_ENABLE(hemu_mode) WMI_SET_BITS(hemu_mode, 4, 1, 1)
#define WMI_VDEV_HE_DLOFDMA_DISABLE(hemu_mode) WMI_SET_BITS(hemu_mode, 4, 1, 0)

#define WMI_VDEV_HE_ULOFDMA_IS_ENABLED(hemu_mode) WMI_GET_BITS(hemu_mode, 5, 1)
#define WMI_VDEV_HE_ULOFDMA_ENABLE(hemu_mode) WMI_SET_BITS(hemu_mode, 5, 1, 1)
#define WMI_VDEV_HE_ULOFDMA_DISABLE(hemu_mode) WMI_SET_BITS(hemu_mode, 5, 1, 0)

#define WMI_VDEV_HE_ULMUMIMO_IS_ENABLED(hemu_mode) WMI_GET_BITS(hemu_mode, 6, 1)
#define WMI_VDEV_HE_ULMUMIMO_ENABLE(hemu_mode) WMI_SET_BITS(hemu_mode, 6, 1, 1)
#define WMI_VDEV_HE_ULMUMIMO_DISABLE(hemu_mode) WMI_SET_BITS(hemu_mode, 6, 1, 0)

#define WMI_VDEV_HE_AX_SOUNDING_IS_ENABLED(mode) WMI_GET_BITS(mode, 0, 1)
#define WMI_VDEV_HE_MU_SOUNDING_IS_ENABLED(mode) WMI_GET_BITS(mode, 2, 1)
#define WMI_VDEV_HE_AX_TRIG_SOUNDING_IS_ENABLED(mode) WMI_GET_BITS(mode, 3, 1)

/* vdev capabilities bit mask */
#define WMI_VDEV_BEACON_SUPPORT  0x1
#define WMI_VDEV_WDS_LRN_ENABLED 0x2
#define WMI_VDEV_VOW_ENABLED     0x4

#define WMI_VDEV_IS_BEACON_SUPPORTED(param) ((param) & WMI_VDEV_BEACON_SUPPORT)
#define WMI_VDEV_IS_WDS_LRN_ENABLED(param) ((param) & WMI_VDEV_WDS_LRN_ENABLED)
#define WMI_VDEV_IS_VOW_ENABLED(param) ((param) & WMI_VDEV_VOW_ENABLED)

/* Per VAP rate dropdown masks */
#define WMI_VDEV_MGMT_RATE_DROPDOWN_M        0x01
#define WMI_VDEV_MGMT_RATE_DROPDOWN_S           0
#define WMI_VDEV_MGMT_RATE_DROPDOWN (WMI_VDEV_MGMT_RATE_DROPDOWN_M << WMI_VDEV_MGMT_RATE_DROPDOWN_S)
#define WMI_VDEV_MGMT_RATE_DROPDOWN_GET(x) WMI_F_MS(x, WMI_VDEV_MGMT_RATE_DROPDOWN)
#define WMI_VDEV_MGMT_RATE_DROPDOWN_SET(x,z) WMI_F_RMW(x, z, WMI_VDEV_MGMT_RATE_DROPDOWN)

#define WMI_VDEV_SU_DATA_RATE_DROPDOWN_M     0x01
#define WMI_VDEV_SU_DATA_RATE_DROPDOWN_S        1
#define WMI_VDEV_SU_DATA_RATE_DROPDOWN (WMI_VDEV_SU_DATA_RATE_DROPDOWN_M << WMI_VDEV_SU_DATA_RATE_DROPDOWN_S)
#define WMI_VDEV_SU_DATA_RATE_DROPDOWN_GET(x) WMI_F_MS(x, WMI_VDEV_SU_DATA_RATE_DROPDOWN)
#define WMI_VDEV_SU_DATA_RATE_DROPDOWN_SET(x,z) WMI_F_RMW(x, z, WMI_VDEV_SU_DATA_RATE_DROPDOWN)

#define WMI_VDEV_MU_DATA_RATE_DROPDOWN_M     0x01
#define WMI_VDEV_MU_DATA_RATE_DROPDOWN_S        2
#define WMI_VDEV_MU_DATA_RATE_DROPDOWN (WMI_VDEV_MU_DATA_RATE_DROPDOWN_M << WMI_VDEV_MU_DATA_RATE_DROPDOWN_S)
#define WMI_VDEV_MU_DATA_RATE_DROPDOWN_GET(x) WMI_F_MS(x, WMI_VDEV_MU_DATA_RATE_DROPDOWN)
#define WMI_VDEV_MU_DATA_RATE_DROPDOWN_SET(x,z) WMI_F_RMW(x, z, WMI_VDEV_MU_DATA_RATE_DROPDOWN)

/* TXBF capabilities masks */
#define WMI_TXBF_CONF_SU_TX_BFEE_S 0
#define WMI_TXBF_CONF_SU_TX_BFEE_M 0x1
#define WMI_TXBF_CONF_SU_TX_BFEE (WMI_TXBF_CONF_SU_TX_BFEE_M << WMI_TXBF_CONF_SU_TX_BFEE_S)
#define WMI_TXBF_CONF_SU_TX_BFEE_GET(x) WMI_F_MS(x,WMI_TXBF_CONF_SU_TX_BFEE)
#define WMI_TXBF_CONF_SU_TX_BFEE_SET(x,z) WMI_F_RMW(x,z,WMI_TXBF_CONF_SU_TX_BFEE)

#define WMI_TXBF_CONF_MU_TX_BFEE_S 1
#define WMI_TXBF_CONF_MU_TX_BFEE_M 0x1
#define WMI_TXBF_CONF_MU_TX_BFEE (WMI_TXBF_CONF_MU_TX_BFEE_M << WMI_TXBF_CONF_MU_TX_BFEE_S)
#define WMI_TXBF_CONF_MU_TX_BFEE_GET(x) WMI_F_MS(x,WMI_TXBF_CONF_MU_TX_BFEE)
#define WMI_TXBF_CONF_MU_TX_BFEE_SET(x,z) WMI_F_RMW(x,z,WMI_TXBF_CONF_MU_TX_BFEE)

#define WMI_TXBF_CONF_SU_TX_BFER_S 2
#define WMI_TXBF_CONF_SU_TX_BFER_M 0x1
#define WMI_TXBF_CONF_SU_TX_BFER (WMI_TXBF_CONF_SU_TX_BFER_M << WMI_TXBF_CONF_SU_TX_BFER_S)
#define WMI_TXBF_CONF_SU_TX_BFER_GET(x) WMI_F_MS(x,WMI_TXBF_CONF_SU_TX_BFER)
#define WMI_TXBF_CONF_SU_TX_BFER_SET(x,z) WMI_F_RMW(x,z,WMI_TXBF_CONF_SU_TX_BFER)

#define WMI_TXBF_CONF_MU_TX_BFER_S 3
#define WMI_TXBF_CONF_MU_TX_BFER_M 0x1
#define WMI_TXBF_CONF_MU_TX_BFER (WMI_TXBF_CONF_MU_TX_BFER_M << WMI_TXBF_CONF_MU_TX_BFER_S)
#define WMI_TXBF_CONF_MU_TX_BFER_GET(x) WMI_F_MS(x,WMI_TXBF_CONF_MU_TX_BFER)
#define WMI_TXBF_CONF_MU_TX_BFER_SET(x,z) WMI_F_RMW(x,z,WMI_TXBF_CONF_MU_TX_BFER)

#define WMI_TXBF_CONF_STS_CAP_S 4
#define WMI_TXBF_CONF_STS_CAP_M 0x7
#define WMI_TXBF_CONF_STS_CAP (WMI_TXBF_CONF_STS_CAP_M << WMI_TXBF_CONF_STS_CAP_S)
#define WMI_TXBF_CONF_STS_CAP_GET(x) WMI_F_MS(x,WMI_TXBF_CONF_STS_CAP);
#define WMI_TXBF_CONF_STS_CAP_SET(x,z) WMI_F_RMW(x,z,WMI_TXBF_CONF_STS_CAP)

#define WMI_TXBF_CONF_IMPLICIT_BF_S 7
#define WMI_TXBF_CONF_IMPLICIT_BF_M 0x1
#define WMI_TXBF_CONF_IMPLICIT_BF (WMI_TXBF_CONF_IMPLICIT_BF_M << WMI_TXBF_CONF_IMPLICIT_BF_S)
#define WMI_TXBF_CONF_IMPLICIT_BF_GET(x) WMI_F_MS(x,WMI_TXBF_CONF_IMPLICIT_BF)
#define WMI_TXBF_CONF_IMPLICIT_BF_SET(x,z) WMI_F_RMW(x,z,WMI_TXBF_CONF_IMPLICIT_BF)

#define WMI_TXBF_CONF_BF_SND_DIM_S 8
#define WMI_TXBF_CONF_BF_SND_DIM_M 0x7
#define WMI_TXBF_CONF_BF_SND_DIM (WMI_TXBF_CONF_BF_SND_DIM_M << WMI_TXBF_CONF_BF_SND_DIM_S)
#define WMI_TXBF_CONF_BF_SND_DIM_GET(x) WMI_F_MS(x,WMI_TXBF_CONF_BF_SND_DIM)
#define WMI_TXBF_CONF_BF_SND_DIM_SET(x,z) WMI_F_RMW(x,z,WMI_TXBF_CONF_BF_SND_DIM)

/* commands for 11ax TXBF capabilities */

#define WMI_TXBF_CONF_11AX_SU_TX_BFER_GET(x) WMI_GET_BITS((x,0,1)
#define WMI_TXBF_CONF_11AX_SU_TX_BFER_SET(x,z)  WMI_SET_BITS(x,0,1,z)

#define WMI_TXBF_CONF_11AX_SU_TX_BFEE_GET(x) WMI_GET_BITS((x,1,1)
#define WMI_TXBF_CONF_11AX_SU_TX_BFEE_SET(x,z) WMI_SET_BITS(x,1,1,z)

#define WMI_TXBF_CONF_11AX_MU_TX_BFER_GET(x) WMI_GET_BITS((x,2,1)
#define WMI_TXBF_CONF_11AX_MU_TX_BFER_SET(x,z)  WMI_SET_BITS(x,2,1,z)

#define WMI_TXBF_CONF_11AX_BFEE_NDP_STS_LT_EQ_80_GET(x) WMI_GET_BITS((x,3,3)
#define WMI_TXBF_CONF_11AX_BFEE_NDP_STS_LT_EQ_80_SET(x,z) WMI_SET_BITS(x,3,3,z)

#define WMI_TXBF_CONF_11AX_NSTS_LT_EQ_80_GET(x) WMI_GET_BITS((x,6,3)
#define WMI_TXBF_CONF_11AX_NSTS_LT_EQ_80_SET(x,z) WMI_SET_BITS(x,6,3,z)

#define WMI_TXBF_CONF_11AX_TX_BFEE_NDP_STS_GT_80_GET(x) WMI_GET_BITS((x,9,3)
#define WMI_TXBF_CONF_11AX_TX_BFEE_NDP_STS_GT_80_SET(x,z) WMI_SET_BITS(x,9,3,z)

#define WMI_TXBF_CONF_11AX_NSTS_GT_80_GET(x) WMI_GET_BITS((x,12,3)
#define WMI_TXBF_CONF_11AX_NSTS_GT_80_SET(x,z) WMI_SET_BITS(x,12,3,z)

#define WMI_TXBF_CONF_AX_BFER_SND_DIM_LT_EQ_80_SND_DIM_GET(x) WMI_GET_BITS((x,15,3)
#define WMI_TXBF_CONF_AX_BFER_SND_DIM_LT_EQ_80_SND_DIM_SET(x,z) WMI_SET_BITS(x,15,3,z)

#define WMI_TXBF_CONF_AX_BFER_SND_DIM_GT_80_SND_DIM_GET(x) WMI_GET_BITS((x,18,3)
#define WMI_TXBF_CONF_AX_BFER_SND_DIM_GT_80_SND_DIM_SET(x,z) WMI_SET_BITS(x,18,3,z)

#define WMI_TXBF_CONF_AX_SU_BFEE_NG16_FDBK_GET(x) WMI_GET_BITS((x,21,1)
#define WMI_TXBF_CONF_AX_SU_BFEE_NG16_FDBK_SET(x,z) WMI_SET_BITS(x,21,1,z)

#define WMI_TXBF_CONF_AX_MU_BFEE_NG16_FDBK_GET(x) WMI_GET_BITS((x,22,1)
#define WMI_TXBF_CONF_AX_MU_BFEE_NG16_FDBK_SET(x,z) WMI_SET_BITS(x,22,1,z)

#define WMI_TXBF_CONF_AX_SU_BFEE_CDBK_4_2_GET(x) WMI_GET_BITS((x,23,1)
#define WMI_TXBF_CONF_AX_SU_BFEE_CDBK_4_2_SET(x,z) WMI_SET_BITS(x,23,1,z)

#define WMI_TXBF_CONF_AX_MU_BFEE_CDBK_7_5_GET(x) WMI_GET_BITS((x,24,1)
#define WMI_TXBF_CONF_AX_MU_BFEE_CDBK_7_5_SET(x,z) WMI_SET_BITS(x,24,1,z)

#define WMI_TXBF_CONF_AX_FDBK_TRIG_GET(x) WMI_GET_BITS((x,25,1)
#define WMI_TXBF_CONF_AX_FDBK_TRIG_SET(x,z) WMI_SET_BITS(x,25,1,z)

/* TXBF capabilities */
typedef struct {
    A_UINT32 txbf_cap;
} wmi_vdev_txbf_cap;

/* vdev rx filters (for mesh) */
typedef enum {
    WMI_VDEV_RX_ALLOW_ALL_FRAMES  =  0x0, /* Don't drop any frames - Default */
    WMI_VDEV_RX_FILTER_OUT_FROMDS =  0x1, /* Drop FromDS frames */
    WMI_VDEV_RX_FILTER_OUT_TODS   =  0x2, /* Drop ToDS frames */
    WMI_VDEV_RX_FILTER_OUT_NODS   =  0x4, /* Drop NODS frames */
    WMI_VDEV_RX_FILTER_OUT_RA     =  0x8, /* Drop RA frames */
    WMI_VDEV_RX_FILTER_OUT_TA     = 0x10, /* Drop TA frames */
} wmi_vdev_param_filter;

/* Length of ATIM Window in TU */
#define WMI_VDEV_PARAM_ATIM_WINDOW_LENGTH WMI_VDEV_PARAM_ATIM_WINDOW

enum wmi_pkt_type {
    WMI_PKT_TYPE_RAW = 0,
    WMI_PKT_TYPE_NATIVE_WIFI = 1,
    WMI_PKT_TYPE_ETHERNET = 2,
};

/*******************************************************************
 * wmi_vdev_txbf_en is DEPRECATED in favor of wmi_vdev_txbf_cap
 * Do not use it!
 *******************************************************************/

typedef struct {
    A_UINT8     sutxbfee : 1,
                mutxbfee : 1,
                sutxbfer : 1,
                mutxbfer : 1,
#if defined(AR900B)
                txb_sts_cap : 3,
                implicit_bf:1;
#else
                reserved : 4;
#endif
} wmi_vdev_txbf_en;

/** Upto 8 bits are available for Roaming module to be sent along with
WMI_VDEV_PARAM_ROAM_FW_OFFLOAD WMI_VDEV_PARAM **/
/* Enable Roaming FW offload LFR1.5/LFR2.0 implementation */
#define WMI_ROAM_FW_OFFLOAD_ENABLE_FLAG                          0x1
/* Enable Roaming module in FW to do scan based on Final BMISS */
#define WMI_ROAM_BMISS_FINAL_SCAN_ENABLE_FLAG                    0x2

        /** slot time long */
        #define WMI_VDEV_SLOT_TIME_LONG                                  0x1
        /** slot time short */
        #define WMI_VDEV_SLOT_TIME_SHORT                                 0x2
        /** preablbe long */
        #define WMI_VDEV_PREAMBLE_LONG                                   0x1
        /** preablbe short */
        #define WMI_VDEV_PREAMBLE_SHORT                                  0x2

/** the definition of different START/RESTART Event response  */
typedef enum {
    /* Event respose of START CMD */
    WMI_VDEV_START_RESP_EVENT = 0,
    /* Event respose of RESTART CMD */
    WMI_VDEV_RESTART_RESP_EVENT,
} WMI_START_EVENT_PARAM;

typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_start_response_event_fixed_param  */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** requestor id that requested the VDEV start request */
    A_UINT32 requestor_id;
    /* Respose of Event type START/RESTART */
    WMI_START_EVENT_PARAM resp_type;
    /** status of the response */
    A_UINT32 status;
    /** Vdev chain mask */
    A_UINT32 chain_mask;
    /** Vdev mimo power save mode */
    A_UINT32 smps_mode;
    union {
        A_UINT32 mac_id; // OBSOLETE - will be removed once all refs are gone
        /** pdev_id for identifying the MAC
         * See macros starting with WMI_PDEV_ID_ for values.
         */
        A_UINT32 pdev_id;
    };
    /** Configured Transmit Streams **/
    A_UINT32 cfgd_tx_streams;
    /** Configured Receive Streams **/
    A_UINT32 cfgd_rx_streams;
} wmi_vdev_start_response_event_fixed_param;

typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_stopped_event_fixed_param  */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
} wmi_vdev_stopped_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_delete_resp_event_fixed_param  */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
} wmi_vdev_delete_resp_event_fixed_param;

/** common structure used for simple events (stopped, resume_req, standby response) */
typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag would be equivalent to actual event  */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
} wmi_vdev_simple_event_fixed_param;


/** VDEV start response status codes */
#define WMI_VDEV_START_RESPONSE_STATUS_SUCCESS 0x0  /** VDEV succesfully started */
#define WMI_VDEV_START_RESPONSE_INVALID_VDEVID  0x1  /** requested VDEV not found */
#define WMI_VDEV_START_RESPONSE_NOT_SUPPORTED  0x2  /** unsupported VDEV combination */
#define WMI_VDEV_START_RESPONSE_DFS_VIOLATION  0x3  /** DFS_VIOLATION since channel in the NOL is selected */

/** Beacon processing related command and event structures */
typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_bcn_tx_hdr */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** xmit rate */
    A_UINT32 tx_rate;
    /** xmit power */
    A_UINT32 txPower;
    /** beacon buffer length in bytes */
    A_UINT32 buf_len;
    /* This TLV is followed by array of bytes:
               * // beacon frame buffer
               *   A_UINT8 bufp[];
               */
} wmi_bcn_tx_hdr;

/* Beacon filter */
#define WMI_BCN_FILTER_ALL   0 /* Filter all beacons */
#define WMI_BCN_FILTER_NONE  1 /* Pass all beacons */
#define WMI_BCN_FILTER_RSSI  2 /* Pass Beacons RSSI >= RSSI threshold */
#define WMI_BCN_FILTER_BSSID 3 /* Pass Beacons with matching BSSID */
#define WMI_BCN_FILTER_SSID  4 /* Pass Beacons with matching SSID */

typedef struct {
    /** Filter ID */
    A_UINT32 bcn_filter_id;
    /** Filter type - wmi_bcn_filter */
    A_UINT32 bcn_filter;
    /** Buffer len */
    A_UINT32 bcn_filter_len;
    /** Filter info (threshold, BSSID, RSSI) */
    A_UINT8 *bcn_filter_buf;
} wmi_bcn_filter_rx_cmd;

/** Capabilities and IEs to be passed to firmware */
typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_bcn_prb_info */
    /** Capabilities */
    A_UINT32 caps;
    /** ERP info */
    A_UINT32 erp;
    /** Advanced capabilities */
    /** HT capabilities */
    /** HT Info */
    /** ibss_dfs */
    /** wpa Info */
    /** rsn Info */
    /** rrm info */
    /** ath_ext */
    /** app IE */
} wmi_bcn_prb_info;

 typedef struct {
     A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_bcn_tmpl_cmd_fixed_param */
     /** unique id identifying the VDEV, generated by the caller */
     A_UINT32 vdev_id;
     /** TIM IE offset from the beginning of the template. */
     A_UINT32 tim_ie_offset;
     /** beacon buffer length. data is in TLV data[] */
     A_UINT32 buf_len;
    /** CSA IE switch count offset from the beginning of data[]
     *  Value 0 indicates CSA IE is not present in beacon template.
     */
    A_UINT32 csa_switch_count_offset; /* units = bytes */
    /** Extended CSA IE switch count offset from the beginning of data[]
     *  Value 0 indicates CSA IE is not present in beacon template.
     */
    A_UINT32 ext_csa_switch_count_offset; /* units = bytes */

    /** Specify when to send the CSA switch count status from FW to host.
     * See WMI_CSA_EVENT_BMAP* below for more information.
     * E.g. if CSA switch count event is needed to be sent when the switch count
     * is 0, 1, 4 and 5, set the bitmap to (0X80000033)
     */
    A_UINT32 csa_event_bitmap;
    /** offset (in octets/bytes) of MBSSID IE in beacon frame */
    A_UINT32 mbssid_ie_offset;
    /** offset (in octets/bytes) of ESP IE in beacon frame */
    A_UINT32 esp_ie_offset;
    /** CSC IE color switch count offset from the beginning of data[]
     *  Value 0 indicates CSC IE is not present in beacon template.
     */
    A_UINT32 csc_switch_count_offset; /* units = bytes */
    /** Specify when to send the CSC switch count status from FW to host.
     * See WMI_CSC_EVENT_BMAP* below for more information.
     * E.g. if CSA switch count event is needed to be sent when the switch count
     * is 0, 1, 4 and 5, set the bitmap to (0X80000033)
     */
    A_UINT32 csc_event_bitmap;

 /*
  * The TLVs follows:
  *    wmi_bcn_prb_info bcn_prb_info; //beacon probe capabilities and IEs
  *    A_UINT8  data[]; //Variable length data
  */
} wmi_bcn_tmpl_cmd_fixed_param;

#define WMI_CSA_EVENT_BMAP_VALID_MASK           0X80000000  /* Follow bitmap for sending the CSA switch count event */
#define WMI_CSA_EVENT_BMAP_SWITCH_COUNT_ZERO    0           /* Send only when the switch count becomes zero, added for backward compatibility
                                                            Same can also be achieved by setting bitmap to 0X80000001 */
#define WMI_CSA_EVENT_BMAP_ALL                  0XFFFFFFFF  /* Send CSA switch count event for every update to switch count */

 typedef struct {
     A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_prb_tmpl_cmd_fixed_param */
     /** unique id identifying the VDEV, generated by the caller */
     A_UINT32 vdev_id;
     /** beacon buffer length. data is in TLV data[] */
     A_UINT32 buf_len;
     /*
                 * The TLVs follows:
                 *    wmi_bcn_prb_info bcn_prb_info; //beacon probe capabilities and IEs
                 *    A_UINT8  data[]; //Variable length data
                 */
 } wmi_prb_tmpl_cmd_fixed_param;

#define WMI_CSC_EVENT_BMAP_VALID_MASK           0X80000000  /* Follow bitmap for sending the CSC switch count event */
#define WMI_CSC_EVENT_BMAP_SWITCH_COUNT_ZERO    0           /* Send only when the switch count becomes zero, added for backward compatibility
                                                            Same can also be achieved by setting bitmap to 0X80000001 */
#define WMI_CSC_EVENT_BMAP_ALL                  0XFFFFFFFF  /* Send CSC switch count event for every update to switch count */

typedef struct {
  /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_offload_bcn_tx_status_event_fixed_param */
    A_UINT32 tlv_header;
  /** unique id identifying the VDEV */
    A_UINT32 vdev_id;
  /** bcn tx status, values defined in enum WMI_FRAME_TX_STATUS */
    A_UINT32 tx_status;
} wmi_offload_bcn_tx_status_event_fixed_param;

enum wmi_sta_ps_mode {
    /** enable power save for the given STA VDEV */
    WMI_STA_PS_MODE_DISABLED = 0,
    /** disable power save  for a given STA VDEV */
    WMI_STA_PS_MODE_ENABLED = 1,
};

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_sta_powersave_mode_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;

    /** Power save mode
     *
     * (see enum wmi_sta_ps_mode)
     */
    A_UINT32 sta_ps_mode;
} wmi_sta_powersave_mode_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_sta_tdcc_config_cmd_fixed_param */
    /**  Set a max tx period: percentage of one beacon interval. range: 0 - 100  */
    A_UINT32 tx_cycle_percentage;
    /**  Enable/disable TX Duty Cycle Control powersave */
    A_UINT32 enabled;
} wmi_sta_tdcc_config_cmd_fixed_param;

enum wmi_csa_offload_en{
    WMI_CSA_OFFLOAD_DISABLE = 0,
    WMI_CSA_OFFLOAD_ENABLE = 1,
};

typedef struct{
   A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_csa_offload_enable_cmd_fixed_param  */
   A_UINT32 vdev_id;
   A_UINT32 csa_offload_enable;
} wmi_csa_offload_enable_cmd_fixed_param;

typedef struct{
   A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_csa_offload_chanswitch_cmd_fixed_param  */
   A_UINT32 vdev_id;
   /*
    * The TLVs follows:
    *    wmi_channel chan;
    */
} wmi_csa_offload_chanswitch_cmd_fixed_param;

/**
 * This parameter controls the policy for retrieving frames from AP while the
 * STA is in sleep state.
 *
 * Only takes affect if the sta_ps_mode is enabled
 */
enum wmi_sta_ps_param_rx_wake_policy {
    /* Wake up when ever there is an  RX activity on the VDEV. In this mode
     * the Power save SM(state machine) will come out of sleep by either
     * sending null frame (or) a data frame (with PS==0) in response to TIM
     * bit set in the received beacon frame from AP.
     */
    WMI_STA_PS_RX_WAKE_POLICY_WAKE = 0,

    /* Here the power save state machine will not wakeup in response to TIM
     * bit, instead it will send a PSPOLL (or) UASPD trigger based on UAPSD
     * configuration setup by WMISET_PS_SET_UAPSD  WMI command.  When all
     * access categories are delivery-enabled, the station will send a UAPSD
     * trigger frame, otherwise it will send a PS-Poll.
     */
    WMI_STA_PS_RX_WAKE_POLICY_POLL_UAPSD = 1,
};

/** Number of tx frames/beacon  that cause the power save SM to wake up.
 *
 * Value 1 causes the SM to wake up for every TX. Value 0 has a special
 * meaning, It will cause the SM to never wake up. This is useful if you want
 * to keep the system to sleep all the time for some kind of test mode . host
 * can change this parameter any time.  It will affect at the next tx frame.
 */
enum wmi_sta_ps_param_tx_wake_threshold {
    WMI_STA_PS_TX_WAKE_THRESHOLD_NEVER = 0,
    WMI_STA_PS_TX_WAKE_THRESHOLD_ALWAYS = 1,

    /* Values greater than one indicate that many TX attempts per beacon
     * interval before the STA will wake up
     */
};

/**
 * The maximum number of PS-Poll frames the FW will send in response to
 * traffic advertised in TIM before waking up (by sending a null frame with PS
 * = 0). Value 0 has a special meaning: there is no maximum count and the FW
 * will send as many PS-Poll as are necessary to retrieve buffered BU. This
 * parameter is used when the RX wake policy is
 * WMI_STA_PS_RX_WAKE_POLICY_POLL_UAPSD and ignored when the RX wake
 * policy is WMI_STA_PS_RX_WAKE_POLICY_WAKE.
 */
enum wmi_sta_ps_param_pspoll_count {
    WMI_STA_PS_PSPOLL_COUNT_NO_MAX = 0,
    /* Values greater than 0 indicate the maximum numer of PS-Poll frames FW
     * will send before waking up.
     */
};

/*
 * This will include the delivery and trigger enabled state for every AC.
 * This is the negotiated state with AP. The host MLME needs to set this based
 * on AP capability and the state Set in the association request by the
 * station MLME.Lower 8 bits of the value specify the UAPSD configuration.
 */
#define WMI_UAPSD_AC_TYPE_DELI 0
#define WMI_UAPSD_AC_TYPE_TRIG 1

#define WMI_UAPSD_AC_BIT_MASK(ac,type) (type ==  WMI_UAPSD_AC_TYPE_DELI)?(1<<(ac<<1)):(1<<((ac<<1)+1))

enum wmi_sta_ps_param_uapsd {
    WMI_STA_PS_UAPSD_AC0_DELIVERY_EN = (1 << 0),
    WMI_STA_PS_UAPSD_AC0_TRIGGER_EN  = (1 << 1),
    WMI_STA_PS_UAPSD_AC1_DELIVERY_EN = (1 << 2),
    WMI_STA_PS_UAPSD_AC1_TRIGGER_EN  = (1 << 3),
    WMI_STA_PS_UAPSD_AC2_DELIVERY_EN = (1 << 4),
    WMI_STA_PS_UAPSD_AC2_TRIGGER_EN  = (1 << 5),
    WMI_STA_PS_UAPSD_AC3_DELIVERY_EN = (1 << 6),
    WMI_STA_PS_UAPSD_AC3_TRIGGER_EN  = (1 << 7),
};

enum wmi_sta_powersave_param {
    /**
     * Controls how frames are retrievd from AP while STA is sleeping
     *
     * (see enum wmi_sta_ps_param_rx_wake_policy)
     */
    WMI_STA_PS_PARAM_RX_WAKE_POLICY = 0,

    /**
     * The STA will go active after this many TX
     *
     * (see enum wmi_sta_ps_param_tx_wake_threshold)
     */
    WMI_STA_PS_PARAM_TX_WAKE_THRESHOLD = 1,

    /**
     * Number of PS-Poll to send before STA wakes up
     *
     * (see enum wmi_sta_ps_param_pspoll_count)
     *
     */
    WMI_STA_PS_PARAM_PSPOLL_COUNT = 2,

    /**
     * TX/RX inactivity time in msec before going to sleep.
     *
     * The power save SM will monitor tx/rx activity on the VDEV, if no
     * activity for the specified msec of the parameter the Power save SM will
     * go to sleep.
     */
    WMI_STA_PS_PARAM_INACTIVITY_TIME = 3,

    /**
     * Set uapsd configuration.
     *
     * (see enum wmi_sta_ps_param_uapsd)
     */
    WMI_STA_PS_PARAM_UAPSD = 4,
    /**
     * Number of PS-Poll to send before STA wakes up in QPower Mode
     */
    WMI_STA_PS_PARAM_QPOWER_PSPOLL_COUNT = 5,

    /**
     * Enable QPower
     */
    WMI_STA_PS_ENABLE_QPOWER = 6,

    /**
     * Number of TX frames before the entering the Active state
     */
    WMI_STA_PS_PARAM_QPOWER_MAX_TX_BEFORE_WAKE = 7,

    /**
     * QPower SPEC PSPOLL interval
     */
    WMI_STA_PS_PARAM_QPOWER_SPEC_PSPOLL_WAKE_INTERVAL = 8,

    /**
     * Max SPEC PSPOLL to be sent when the PSPOLL response has
     * no-data bit set
     */
    WMI_STA_PS_PARAM_QPOWER_SPEC_MAX_SPEC_NODATA_PSPOLL = 9,

    /**
     * Max value of ITO reset when there is no tx-rx
     * after AP has set the TIM bit
     */
    WMI_STA_PS_PARAM_MAX_RESET_ITO_COUNT_ON_TIM_NO_TXRX = 10,
};

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_sta_powersave_param_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** station power save parameter (see enum wmi_sta_powersave_param) */
    A_UINT32 param;
    A_UINT32 value;
} wmi_sta_powersave_param_cmd_fixed_param;

 /** No MIMO power save */
#define WMI_STA_MIMO_PS_MODE_DISABLE
 /** mimo powersave mode static*/
#define WMI_STA_MIMO_PS_MODE_STATIC
 /** mimo powersave mode dynamic */
#define WMI_STA_MIMO_PS_MODE_DYNAMI

typedef struct {
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** mimo powersave mode as defined above */
    A_UINT32 mimo_pwrsave_mode;
} wmi_sta_mimo_ps_mode_cmd;


/** U-APSD configuration of peer station from (re)assoc request and TSPECs */
enum wmi_ap_ps_param_uapsd {
    WMI_AP_PS_UAPSD_AC0_DELIVERY_EN = (1 << 0),
    WMI_AP_PS_UAPSD_AC0_TRIGGER_EN  = (1 << 1),
    WMI_AP_PS_UAPSD_AC1_DELIVERY_EN = (1 << 2),
    WMI_AP_PS_UAPSD_AC1_TRIGGER_EN  = (1 << 3),
    WMI_AP_PS_UAPSD_AC2_DELIVERY_EN = (1 << 4),
    WMI_AP_PS_UAPSD_AC2_TRIGGER_EN  = (1 << 5),
    WMI_AP_PS_UAPSD_AC3_DELIVERY_EN = (1 << 6),
    WMI_AP_PS_UAPSD_AC3_TRIGGER_EN  = (1 << 7),
};

/** U-APSD maximum service period of peer station */
enum wmi_ap_ps_peer_param_max_sp {
    WMI_AP_PS_PEER_PARAM_MAX_SP_UNLIMITED = 0,
    WMI_AP_PS_PEER_PARAM_MAX_SP_2 = 1,
    WMI_AP_PS_PEER_PARAM_MAX_SP_4 = 2,
    WMI_AP_PS_PEER_PARAM_MAX_SP_6 = 3,

    /* keep last! */
    MAX_WMI_AP_PS_PEER_PARAM_MAX_SP,
};

/** param values for WMI_AP_PS_PEER_PARAM_SIFS_RESP_FRMTYPE */
enum wmi_ap_ps_param_sifs_resp_frmtype {
    WMI_SIFS_RESP_PSPOLL    = (1 << 0),
    WMI_SIFS_RESP_UAPSD     = (1 << 1),
    WMI_SIFS_RESP_QBST_EXP  = (1 << 2),
    WMI_SIFS_RESP_QBST_DATA = (1 << 3),
    WMI_SIFS_RESP_QBST_BAR  = (1 << 4),
};

/**
 * AP power save parameter
 * Set a power save specific parameter for a peer station
 */
enum wmi_ap_ps_peer_param {
    /** Set uapsd configuration for a given peer.
     *
     * This will include the delivery and trigger enabled state for every AC.
     * The host  MLME needs to set this based on AP capability and stations
     * request Set in the association request  received from the station.
     *
     * Lower 8 bits of the value specify the UAPSD configuration.
     *
     * (see enum wmi_ap_ps_param_uapsd)
     * The default value is 0.
     */
    WMI_AP_PS_PEER_PARAM_UAPSD = 0,

    /**
     * Set the service period for a UAPSD capable station
     *
     * The service period from wme ie in the (re)assoc request frame.
     *
     * (see enum wmi_ap_ps_peer_param_max_sp)
     */
    WMI_AP_PS_PEER_PARAM_MAX_SP = 1,

    /** Time in seconds for aging out buffered frames for STA in power save */
    WMI_AP_PS_PEER_PARAM_AGEOUT_TIME = 2,

    /**
     * Specify frame types that are considered SIFS RESP trigger frame
     * (see enum wmi_ap_ps_param_sifs_resp_frmtype)
     */
    WMI_AP_PS_PEER_PARAM_SIFS_RESP_FRMTYPE = 3,

    /** Specifies the trigger state of TID. Valid only for UAPSD frame type  */
    WMI_AP_PS_PEER_PARAM_SIFS_RESP_UAPSD = 4,

    /** Specifies the WNM sleep state of a STA */
    WMI_AP_PS_PEER_PARAM_WNM_SLEEP = 5,
};

typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ap_ps_peer_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /** AP powersave param (see enum wmi_ap_ps_peer_param) */
    A_UINT32 param;
    /** AP powersave param value (see defines) */
    A_UINT32 value;
} wmi_ap_ps_peer_cmd_fixed_param;

/** Configure peer station 11v U-APSD coexistance
 *
 * Two parameters from uaspd coexistence ie info (as specified in 11v) are
 * sent down to FW along with this command.
 *
 * The semantics of these fields are described in the following text extracted
 * from 802.11v.
 *
 * ---  If the non-AP STA specified a non-zero TSF 0 Offset value in the
 *      U-APSD Coexistence element, the AP should not transmit frames to the
 *      non-AP STA outside of the U-APSD Coexistence Service Period, which
 *      begins when the AP receives the U-APSD trigger frame and ends after
 *      the transmission period specified by the result of the following
 *      calculation:
 *
 *          End of transmission period = T + (Interval . ((T . TSF 0 Offset) mod Interval))
 *
 *      Where T is the time the U-APSD trigger frame was received at the AP
 *      Interval is the UAPSD Coexistence element Duration/Interval field
 *      value (see 7.3.2.91) or upon the successful transmission of a frame
 *      with EOSP bit set to 1, whichever is earlier.
 *
 *
 * ---  If the non-AP STA specified a zero TSF 0 Offset value in the U-APSD
 *      Coexistence element, the AP should not transmit frames to the non-AP
 *      STA outside of the U-APSD Coexistence Service Period, which begins
 *      when the AP receives a U-APSD trigger frame and ends after the
 *      transmission period specified by the result of the following
 *      calculation: End of transmission period = T + Duration
 */
typedef struct {
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /** Enable U-APSD coexistence support for this peer
     *
     * 0 -> disabled (default)
     * 1 -> enabled
     */
    A_UINT32 enabled;
    /** Duration/Interval as defined by 11v U-ASPD coexistance */
    A_UINT32 duration_interval;
    /** Upper 32 bits of 64-bit TSF offset */
    A_UINT32 tsf_offset_high;
    /** Lower 32 bits of 64-bit TSF offset */
    A_UINT32 tsf_offset_low;
} wmi_ap_powersave_peer_uapsd_coex_cmd;

typedef enum {
    WMI_AP_PS_EGAP_F_ENABLE_PHYERR_DETECTION      = 0x0001,
    WMI_AP_PS_EGAP_F_ENABLE_PWRSAVE_BY_PS_STATE   = 0x0002,
    WMI_AP_PS_EGAP_F_ENABLE_PWRSAVE_BY_INACTIVITY = 0x0004,

    WMI_AP_PS_EGAP_FLAG_MAX = 0x8000
} wmi_ap_ps_egap_flag_type;

/**
 * configure ehanced green ap parameters
 */
typedef struct {
    /*
     * TLV tag and len; tag equals
     * wmi_ap_powersave_egap_param_cmd_fixed_param
     */
    A_UINT32 tlv_header;
    /** Enable enhanced green ap
     * 0 -> disabled
     * 1 -> enabled
     */
    A_UINT32 enable;
    /** The param indicates a duration that all STAs connected
     * to S-AP have no traffic.
     */
    A_UINT32 inactivity_time; /* in unit of milliseconds */
    /** The param indicates a duration that all STAs connected
     * to S-AP have no traffic, after all STAs have entered powersave.
     */
    A_UINT32 wait_time; /* in unit of milliseconds */
    /** The param is used to turn on/off some functions within E-GAP.
     */
    A_UINT32 flags; /* wmi_ap_ps_egap_flag_type bitmap */
} wmi_ap_ps_egap_param_cmd_fixed_param;

typedef enum {
    WMI_AP_PS_EGAP_STATUS_IDLE        = 1,
    WMI_AP_PS_EGAP_STATUS_PWRSAVE_OFF = 2,
    WMI_AP_PS_EGAP_STATUS_PWRSAVE_ON  = 3,

    WMI_AP_PS_EGAP_STATUS_MAX = 15
} wmi_ap_ps_egap_status_type;

/**
 * send ehanced green ap status to host
 */
typedef struct {
    /** TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_ap_ps_egap_info_chainmask_list */
    A_UINT32 tlv_header;
    union {
        A_UINT32 mac_id; // OBSOLETE - will be removed once all refs are gone
        /** pdev_id for identifying the MAC
         * See macros starting with WMI_PDEV_ID_ for values.
         */
        A_UINT32 pdev_id;
    };
    /** The param indicates the current tx chainmask with the mac id. */
    A_UINT32 tx_chainmask;
    /** The param indicates the current rx chainmask with the mac id. */
    A_UINT32 rx_chainmask;
} wmi_ap_ps_egap_info_chainmask_list;

typedef struct {
    /*
     * TLV tag and len; tag equals
     * wmi_ap_powersave_egap_param_cmd_fixed_param
     */
    A_UINT32 tlv_header;
    /** Enhanced green ap status (WMI_AP_PS_EGAP_STATUS). */
    A_UINT32 status;
/* This TLV is followed by
 *     wmi_ap_ps_egap_info_chainmask_list chainmask_list[];
 */
} wmi_ap_ps_egap_info_event_fixed_param;

        /* 128 clients = 4 words */
        /* WMI_TIM_BITMAP_ARRAY_SIZE can't be modified without breaking the compatibility */
        #define WMI_TIM_BITMAP_ARRAY_SIZE 4

        typedef struct {
            A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tim_info  */
            /** TIM bitmap len (in bytes) */
            A_UINT32 tim_len;
            /** TIM Partial Virtual Bitmap */
            A_UINT32 tim_mcast;
            A_UINT32 tim_bitmap[WMI_TIM_BITMAP_ARRAY_SIZE];
            A_UINT32 tim_changed;
            A_UINT32 tim_num_ps_pending;
            /** Use the vdev_id only if vdev_id_valid is set */
            A_UINT32 vdev_id_valid;
            /** unique id identifying the VDEV */
            A_UINT32 vdev_id;
        } wmi_tim_info;

        typedef struct {
            A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tim_info  */
            /** TIM Partial Virtual Bitmap */
            A_UINT32 tim_mcast;
            A_UINT32 tim_changed;
            A_UINT32 tim_num_ps_pending;
            /** Use the vdev_id only if vdev_id_valid is set */
            A_UINT32 vdev_id_valid;
            /** unique id identifying the VDEV */
            A_UINT32 vdev_id;
            /** TIM bitmap len (in bytes) */
            A_UINT32 tim_len;
            /* followed by WMITLV_TAG_ARRAY_BYTE holding the TIM bitmap */
        } wmi_tim_info_v2;

        typedef struct {
            /** Flag to enable quiet period IE support */
            A_UINT32   is_enabled;
            /** Quiet start */
            A_UINT32   tbttcount;
            /** Beacon intervals between quiets*/
            A_UINT32   period;
            /** TUs of each quiet*/
            A_UINT32   duration;
            /** TUs of from TBTT of quiet start*/
            A_UINT32   offset;
        } wmi_quiet_info;

/* WMI_P2P_MAX_NOA_DESCRIPTORS can't be modified without breaking the compatibility */
#define WMI_P2P_MAX_NOA_DESCRIPTORS 4   /* Maximum number of NOA Descriptors supported */

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_p2p_noa_info  */
    /** Bit  0:     Flag to indicate an update in NOA schedule
     *  Bits 7-1:   Reserved
     *  Bits 15-8:  Index (identifies the instance of NOA sub element)
     *  Bit  16:    Opp PS state of the AP
     *  Bits 23-17: Ctwindow in TUs
     *  Bits 31-24: Number of NOA descriptors
     */
    A_UINT32 noa_attributes;
    wmi_p2p_noa_descriptor noa_descriptors[WMI_P2P_MAX_NOA_DESCRIPTORS];
    /** Use the vdev_id only if vdev_id_valid is set */
    A_UINT32 vdev_id_valid;
    /** unique id identifying the VDEV */
    A_UINT32 vdev_id;
} wmi_p2p_noa_info;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_quiet_offload_info  */
    A_UINT32 vdev_id;   /* unique id identifying the VDEV */
    A_UINT8  tbttcount; /* quiet start */
    A_UINT8  period;    /* beacon intervals between quiets */
    A_UINT16 duration;  /* TUs of each quiet */
    A_UINT16 offset;    /* TUs of from TBTT of quiet start */
} wmi_quiet_offload_info;

#define WMI_UNIFIED_NOA_ATTR_MODIFIED       0x1
#define WMI_UNIFIED_NOA_ATTR_MODIFIED_S     0

#define WMI_UNIFIED_NOA_ATTR_IS_MODIFIED(hdr)                       \
            WMI_F_MS((hdr)->noa_attributes, WMI_UNIFIED_NOA_ATTR_MODIFIED)

#define WMI_UNIFIED_NOA_ATTR_MODIFIED_SET(hdr)                      \
            WMI_F_RMW((hdr)->noa_attributes, 0x1,                    \
              WMI_UNIFIED_NOA_ATTR_MODIFIED);

#define	WMI_UNIFIED_NOA_ATTR_INDEX			0xff00
#define	WMI_UNIFIED_NOA_ATTR_INDEX_S		8

#define WMI_UNIFIED_NOA_ATTR_INDEX_GET(hdr)                            \
            WMI_F_MS((hdr)->noa_attributes, WMI_UNIFIED_NOA_ATTR_INDEX)

#define WMI_UNIFIED_NOA_ATTR_INDEX_SET(hdr, v)                      \
            WMI_F_RMW((hdr)->noa_attributes, (v) & 0xff,            \
        WMI_UNIFIED_NOA_ATTR_INDEX);

#define	WMI_UNIFIED_NOA_ATTR_OPP_PS			0x10000
#define	WMI_UNIFIED_NOA_ATTR_OPP_PS_S		16

#define WMI_UNIFIED_NOA_ATTR_OPP_PS_GET(hdr)                         \
            WMI_F_MS((hdr)->noa_attributes, WMI_UNIFIED_NOA_ATTR_OPP_PS)

#define WMI_UNIFIED_NOA_ATTR_OPP_PS_SET(hdr)                         \
            WMI_F_RMW((hdr)->noa_attributes, 0x1,                    \
        WMI_UNIFIED_NOA_ATTR_OPP_PS);

#define	WMI_UNIFIED_NOA_ATTR_CTWIN			0xfe0000
#define	WMI_UNIFIED_NOA_ATTR_CTWIN_S		17

#define WMI_UNIFIED_NOA_ATTR_CTWIN_GET(hdr)                          \
            WMI_F_MS((hdr)->noa_attributes, WMI_UNIFIED_NOA_ATTR_CTWIN)

#define WMI_UNIFIED_NOA_ATTR_CTWIN_SET(hdr, v)                       \
            WMI_F_RMW((hdr)->noa_attributes, (v) & 0x7f,             \
        WMI_UNIFIED_NOA_ATTR_CTWIN);

#define	WMI_UNIFIED_NOA_ATTR_NUM_DESC		0xff000000
#define	WMI_UNIFIED_NOA_ATTR_NUM_DESC_S		24

#define WMI_UNIFIED_NOA_ATTR_NUM_DESC_GET(hdr)                       \
            WMI_F_MS((hdr)->noa_attributes, WMI_UNIFIED_NOA_ATTR_NUM_DESC)

#define WMI_UNIFIED_NOA_ATTR_NUM_DESC_SET(hdr, v)                    \
            WMI_F_RMW((hdr)->noa_attributes, (v) & 0xff,             \
                WMI_UNIFIED_NOA_ATTR_NUM_DESC);

typedef struct {
    /** TIM info */
    wmi_tim_info tim_info;
    /** P2P NOA info */
    wmi_p2p_noa_info p2p_noa_info;
/* TBD: More info elements to be added later */
} wmi_bcn_info;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_host_swba_event_fixed_param  */
    /** bitmap identifying the VDEVs, generated by the caller */
    A_UINT32 vdev_map;
    /** how many vdev's info is included in this message
     *  If this field is zero, then the number of vdevs is specified by
     *  the number of bits set in the vdev_map bitmap.
     */
    A_UINT32 num_vdevs;
/* This TLV is followed by tim_info and p2p_noa_info for each vdev:
 *     wmi_tim_info tim_info[];
 *     wmi_p2p_noa_info p2p_noa_info[];
 *     wmi_quiet_offload_info quiet_offload_info[0/1];
 *
 */
} wmi_host_swba_event_fixed_param;

#define WMI_MAX_AP_VDEV 16

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_host_swfda_event_fixed_param  */
    /** vdev_id identifying the VDEV for which FILS should be generated */
    A_UINT32 vdev_id;
    /** time (in TU) at which current FILS Discovery frame is scheduled for Tx */
    A_UINT32 fils_tt;
    /** next TBTT time (in TU) for this vdev  */
    A_UINT32 tbtt;
} wmi_host_swfda_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tbtt_offset_event_fixed_param  */
    /** bimtap of VDEVs that has tbtt offset updated */
    A_UINT32 vdev_map;
/* The TLVs for tbttoffset_list will follow this TLV.
 *     tbtt offset list in the order of the LSB to MSB in the vdev_map bitmap
 *     A_UINT32 tbttoffset_list[WMI_MAX_AP_VDEV];
 */
} wmi_tbtt_offset_event_fixed_param;

typedef struct {
    /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tbtt_offset_info */
    A_UINT32 tlv_header;
    /** unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /** tbttoffset in TUs */
    A_UINT32 tbttoffset;
} wmi_tbtt_offset_info;

        /** Use this event if number of vdevs > 32 */
        typedef struct {
            /*
             * TLV tag and len;
             * tag equals WMITLV_TAG_STRUC_wmi_tbtt_offset_ext_event_fixed_param
             */
            A_UINT32 tlv_header;
            A_UINT32 num_vdevs;
            /*
             * The TLVs for tbttoffset will follow this TLV.
             * Of size num_vdevs * wmi_tbtt_offset_info
             */
        } wmi_tbtt_offset_ext_event_fixed_param;

        /* Peer Specific commands and events */

typedef struct {
    A_UINT32 percentage; /* in unit of 12.5% */
    A_UINT32 min_delta;  /* in unit of Mbps */
} rate_delta_t;

#define PEER_RATE_REPORT_COND_FLAG_DELTA        0x01
#define PEER_RATE_REPORT_COND_FLAG_THRESHOLD    0x02
#define MAX_NUM_OF_RATE_THRESH                  4

typedef struct {
    A_UINT32 val_cond_flags;     /* PEER_RATE_REPORT_COND_FLAG_DELTA, PEER_RATE_REPORT_COND_FLAG_THRESHOLD
                                    Any of these two conditions or both of them can be set. */
    rate_delta_t rate_delta;
    A_UINT32 rate_threshold[MAX_NUM_OF_RATE_THRESH];  /* In unit of Mbps. There are at most 4 thresholds.
                                                         If the threshold count is less than 4, set zero to
                                                         the one following the last threshold */
} report_cond_per_phy_t;


enum peer_rate_report_cond_phy_type {
    PEER_RATE_REPORT_COND_11B = 0,
    PEER_RATE_REPORT_COND_11A_G,
    PEER_RATE_REPORT_COND_11N,
    PEER_RATE_REPORT_COND_11AC,
    PEER_RATE_REPORT_COND_MAX_NUM
};

typedef struct {
    A_UINT32 tlv_header;                     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_rate_report_condtion_fixed_param */
    A_UINT32 enable_rate_report;             /* 1= enable, 0=disable */
    A_UINT32 report_backoff_time;            /* in unit of msecond */
    A_UINT32 report_timer_period;            /* in unit of msecond */
    /* In the following field, the array index means the phy type,
     * please see enum peer_rate_report_cond_phy_type for detail */
    report_cond_per_phy_t cond_per_phy[PEER_RATE_REPORT_COND_MAX_NUM];
} wmi_peer_set_rate_report_condition_fixed_param;

/* Peer Type:
 * NB: This can be left DEFAULT for the normal case, and f/w will determine BSS type based
 * on address and vdev opmode. This is largely here to allow host to indicate that
 * peer is explicitly a TDLS peer
 */
enum wmi_peer_type {
    WMI_PEER_TYPE_DEFAULT = 0,    /* Generic/Non-BSS/Self Peer */
    WMI_PEER_TYPE_BSS = 1,        /* Peer is BSS Peer entry */
    WMI_PEER_TYPE_TDLS = 2,       /* Peer is a TDLS Peer */
    WMI_PEER_TYPE_OCB = 3,        /* Peer is a OCB Peer */
    WMI_PEER_TYPE_NAN_DATA = 4,   /* Peer is NAN DATA */
    WMI_PEER_TYPE_TRANS_BSS = 5,  /* For creating BSS peer when connecting with non-transmit AP */
    WMI_PEER_TYPE_HOST_MAX = 127, /* Host <-> Target Peer type is assigned up to 127 */
                                  /* Reserved from 128 - 255 for target internal use.*/
    WMI_PEER_TYPE_ROAMOFFLOAD_TEMP = 128, /* Temporarily created during offload roam */
};

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_create_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /** peer type: see enum values above */
    A_UINT32 peer_type;
} wmi_peer_create_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_delete_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
} wmi_peer_delete_cmd_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_unmap_response_cmd_fixed_param */
    A_UINT32 tlv_header;
    /*
     * Following this struct is the TLV:
     * A_UINT32 peer_ids[]; <-- variable-length array of peer_ids
     *                          that have been unmapped by the host
     */
} wmi_peer_unmap_response_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_set_rx_blocksize_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /**
     * maximum block ack window size to use during a rx block ack negotiation,
     * i.e. the maximum number of MPDUs per A-MPDU that will be received
     */
    A_UINT32  rx_block_ack_win_limit;
} wmi_peer_set_rx_blocksize_cmd_fixed_param;

        typedef struct {
            A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_flush_tids_cmd_fixed_param */
            /** unique id identifying the VDEV, generated by the caller */
            A_UINT32 vdev_id;
            /** peer MAC address */
            wmi_mac_addr peer_macaddr;
            /** tid bitmap identifying the tids to flush */
            A_UINT32 peer_tid_bitmap;
        } wmi_peer_flush_tids_cmd_fixed_param;

        typedef struct {
            /** rate mode . 0: disable fixed rate (auto rate)
             *   1: legacy (non 11n) rate  specified as ieee rate 2*Mbps
             *   2: ht20 11n rate  specified as mcs index
             *   3: ht40 11n rate  specified as mcs index
             */
            A_UINT32  rate_mode;
             /** 4 rate values for 4 rate series. series 0 is stored in byte 0 (LSB)
              *  and series 3 is stored at byte 3 (MSB) */
            A_UINT32  rate_series;
             /** 4 retry counts for 4 rate series. retry count for rate 0 is stored in byte 0 (LSB)
              *  and retry count for rate 3 is stored at byte 3 (MSB) */
            A_UINT32  rate_retries;
        } wmi_fixed_rate;

        typedef struct {
            /** unique id identifying the VDEV, generated by the caller */
            A_UINT32 vdev_id;
            /** peer MAC address */
            wmi_mac_addr peer_macaddr;
            /** fixed rate */
            wmi_fixed_rate peer_fixed_rate;
        } wmi_peer_fixed_rate_cmd;

        #define WMI_MGMT_TID    17

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_addba_clear_resp_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
} wmi_addba_clear_resp_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_addba_send_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /** Tid number */
    A_UINT32 tid;
    /** Buffer/Window size*/
    A_UINT32 buffersize;
} wmi_addba_send_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_delba_send_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /** Tid number */
    A_UINT32 tid;
    /** Is Initiator */
    A_UINT32 initiator;
    /** Reason code */
    A_UINT32 reasoncode;
} wmi_delba_send_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_addba_setresponse_cmd_fixed_param */
    /** unique id identifying the vdev, generated by the caller */
    A_UINT32 vdev_id;
    /** peer mac address */
    wmi_mac_addr peer_macaddr;
    /** Tid number */
    A_UINT32 tid;
    /** status code */
    A_UINT32 statuscode;
} wmi_addba_setresponse_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_send_singleamsdu_cmd_fixed_param */
    /** unique id identifying the vdev, generated by the caller */
    A_UINT32 vdev_id;
    /** peer mac address */
    wmi_mac_addr peer_macaddr;
    /** Tid number */
    A_UINT32 tid;
} wmi_send_singleamsdu_cmd_fixed_param;

/* Type of Station DTIM Power Save method */
enum {
    /* For NORMAL DTIM, the parameter is the number of beacon intervals and
     * also the same value as the listen interval. For this method, the
     * station will wake up based on the listen interval. If this
     * listen interval is not equal to DTIM, then the station may
     * miss certain DTIM beacons. If this value is 1, then the
     * station will wake up for every beacon.
     */
    WMI_STA_DTIM_PS_NORMAL_DTIM = 0x01,
    /* For MODULATED_DTIM, parameter is a multiple of DTIM beacons to skip.
     * When this value is 1, then the station will wake at every DTIM beacon.
     * If this value is >1, then the station will skip certain DTIM beacons.
     * This value is the multiple of DTIM intervals that the station will
     * wake up to receive the DTIM beacons.
     */
    WMI_STA_DTIM_PS_MODULATED_DTIM = 0x02,
};

/* Parameter structure for the WMI_STA_DTIM_PS_METHOD_CMDID */
typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_sta_dtim_ps_method_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** Station DTIM Power Save method as defined above */
    A_UINT32 dtim_pwrsave_method;
    /** DTIM PS value. Contents depends on the method */
    A_UINT32 value;
    /** Modulated DTIM value */
    A_UINT32 MaxLIModulatedDTIM;
} wmi_sta_dtim_ps_method_cmd_fixed_param;

    /*
    * For Station UAPSD Auto Trigger feature, the Firmware monitors the
    * uAPSD uplink and downlink traffic for each uAPSD enabled WMM ACs.
    * If there is no uplink/download for the specified service interval (field service_interval),
    * firmware will auto generate a QOS-NULL trigger for that WMM-AP with the TID value
    * specified in the UP (field user_priority).
    * Firmware also monitors the responses for these QOS-NULL triggers.
    * If the peer does not have any delivery frames, it will respond with
    * QOS-NULL (EOSP=1). This feature of only using service interval is assumed to be mandatory for all
    * firmware implementation. For this basic implementation, the suspend_interval and delay_interval
    * are unused and should be set to 0.
    * When service_interval is 0, then the firmware will not send any trigger frames. This is for
    * certain host-based implementations that don't want this firmware offload.
    * Note that the per-AC intervals are required for some usage scenarios. This is why the intervals
    * are given in the array of ac_param[]. For example, Voice service interval may defaults to 20 ms
    * and rest of the AC default to 300 ms.
    *
    * The service bit, WMI_STA_UAPSD_VAR_AUTO_TRIG, will indicate that the more advanced feature
    * of variable auto trigger is supported. The suspend_interval and delay_interval is used in
    * the more advanced monitoring method.
    * If the PEER does not have any delivery enabled data frames (non QOS-NULL) for the
    * suspend interval (field suspend_interval), firmware will change its auto trigger interval
    * to delay interval (field delay_interval). This way, when there is no traffic, the station
    * will save more power by waking up less and sending less trigger frames.
    * The (service_interval < suspend_interval) and (service_interval < delay_interval).
    * If this variable auto trigger is not required, then the suspend_interval and delay_interval
    * should be 0.
    */
typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_sta_uapsd_auto_trig_param */
    /** WMM Access category from 0 to 3 */
    A_UINT32 wmm_ac;
    /** User priority to use in trigger frames. It is the TID
     *  value. This field needs to be specified and may not be
     *  equivalent to AC since some implementation may use the TSPEC
     *  to enable UAPSD and negotiate a particular user priority. */
    A_UINT32 user_priority;
    /** service interval in ms */
    A_UINT32 service_interval;
    /** Suspend interval in ms */
    A_UINT32 suspend_interval;
    /** delay interval in ms */
    A_UINT32 delay_interval;
} wmi_sta_uapsd_auto_trig_param;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_sta_uapsd_auto_trig_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer mac address */
    wmi_mac_addr peer_macaddr;
    /** Number of AC to specify */
    A_UINT32 num_ac;
    /*
         * Following this struc is the TLV:
         *     wmi_sta_uapsd_auto_trig_param ac_param[]; //Variable number of AC parameters (defined by field num_ac)
         */

} wmi_sta_uapsd_auto_trig_cmd_fixed_param;

/** mimo powersave state */
#define WMI_PEER_MIMO_PS_STATE                          0x1
/** enable/disable AMPDU . initial value (enabled) */
#define WMI_PEER_AMPDU                                  0x2
/** authorize/unauthorize peer. initial value is unauthorized (0)  */
#define WMI_PEER_AUTHORIZE                              0x3
/** peer channel bandwidth */
#define WMI_PEER_CHWIDTH                                0x4
/** peer NSS */
#define WMI_PEER_NSS                                    0x5
/** USE 4 ADDR */
#define WMI_PEER_USE_4ADDR                              0x6
/* set group membership status */
#define WMI_PEER_MEMBERSHIP                0x7
#define WMI_PEER_USERPOS                0x8
/*
 * A critical high-level protocol is being used with this peer. Target
 * should take appropriate measures (if possible) to ensure more
 * reliable link with minimal latency. This *may* include modifying the
 * station power save policy, enabling more RX chains, increased
 * priority of channel scheduling, etc.
 *
 * NOTE: This parameter should only be considered a hint as specific
 * behavior will depend on many factors including current network load
 * and vdev/peer configuration.
 *
 * For STA VDEV this peer corresponds to the AP's BSS peer.
 * For AP VDEV this peer corresponds to the remote peer STA.
 */
#define WMI_PEER_CRIT_PROTO_HINT_ENABLED                0x9
/* set Tx failure count threshold for the peer - Currently unused */
#define WMI_PEER_TX_FAIL_CNT_THR                        0xA
/* Enable H/W retry and Enable H/W Send CTS2S before Data */
#define WMI_PEER_SET_HW_RETRY_CTS2S                     0xB

/* Set peer advertised IBSS atim window length */
#define WMI_PEER_IBSS_ATIM_WINDOW_LENGTH                0xC

/** peer phy mode */
#define WMI_PEER_PHYMODE                                0xD
/** Use FIXED Pwr */
#define WMI_PEER_USE_FIXED_PWR                          0xE
/** Set peer fixed rate
 * The top nibble is used to select which format to use for encoding
 * the rate specification: 0xVXXXXXXX
 * If V == 0b0000: format is same as before: 0x000000RR
 * If V == 0b0001: format is: 0x1000RRRR.
 *                 This will be output of WMI_ASSEMBLE_RATECODE_V1
 * The host shall use the new V1 format (and set V = 0x1) if the target
 * indicates 802.11ax support via the WMI_SERVICE_11AX flag, or if the
 * system is configured with Nss > 4 (either at compile time within the
 * host driver, or through WMI_SERVICE_READY PHY capabilities provided
 * by the target).
 */
#define WMI_PEER_PARAM_FIXED_RATE                       0xF
/** Whitelist peer TIDs */
#define WMI_PEER_SET_MU_WHITELIST                       0x10
/** Set peer max tx rate (MCS) in adaptive rate ctrl */
#define WMI_PEER_SET_MAX_TX_RATE                        0x11
/** Set peer minimal tx rate (MCS) in adaptive rate ctrl */
#define WMI_PEER_SET_MIN_TX_RATE                        0x12
/**
 * default ring routing for peer data packets,
 * param_value = bit 0 for hash based routing enabled or not
 *               (value 1 is enabled, value 0 is disabled)
 *               bits 1:5 are for ring 32 (i.e. ring id value
 *               selected from 0 to 31 values)
 *               bit 8 for peer based ring selection enabled or not
 *               (value 1 is enabled, value 0 is disabled
 *                bits 9-15 are valid when bit 8 is set to 1)
 *               bit 9 is for ring selection enabled for filter-pass
 *               unicast or not (value 1 is enabled, value 0 is disabled)
 *               bit 10 is for ring selection enabled for filter-pass
 *               mcast or not (value 1 is enabled, value 0 is disabled)
 *               bit 11 is for ring selection enabled for filter-pass
 *               BAR or not (value 1 is enabled, value 0 is disabled)
 *               bit 12-13 is for source ring selection value
 *               (value 0 for wbm2rxdma buf ring,
 *                value 1 for fw2rxdma buf ring,
 *                value 2 for sw2rxdma buf ring,
 *                value 3 for no buf ring)
 *               bit 14-15 is for destination ring selection value
 *               (value 0 for wbm release ring,
 *                value 1 for rxdma2fw ring,
 *                value 2 for rxdma2sw ring,
 *                value 3 for rxdma2reo ring)
 */
#define WMI_PEER_SET_DEFAULT_ROUTING                    0x13
/* peer NSS for VHT160 - Extended NSS support */
#define WMI_PEER_NSS_VHT160                             0x14
/* peer NSS for VHT160 - Extended NSS support */
#define WMI_PEER_NSS_VHT80_80                           0x15
/* Peer SU TXBF sounding interval */
#define WMI_PEER_PARAM_SU_TXBF_SOUNDING_INTERVAL        0x16
/* Peer MU TXBF sounding interval */
#define WMI_PEER_PARAM_MU_TXBF_SOUNDING_INTERVAL        0x17
/* Peer TXBF sounding enable or disable */
#define WMI_PEER_PARAM_TXBF_SOUNDING_ENABLE             0x18
/* Per peer 11ax OFDMA enable or disable */
#define WMI_PEER_PARAM_OFDMA_ENABLE                     0x19
/* Per peer 11ax/11ac MU enable or disable */
#define WMI_PEER_PARAM_MU_ENABLE                        0x1a
/** Set peer fixed rate used in UL Trigger
 * The top nibble is used to select which format to use for encoding
 * the rate specification: 0xVXXXXXXX, V must be 1 for this parameter.
 * If V == 0b0001: format is: 0x1000RRRR.
 *                 This will be output of WMI_ASSEMBLE_RATECODE_V1
 *
 * This parameter controls the UL OFDMA and UL MU-MIMO peer fixed rate.
 */
#define WMI_PEER_PARAM_UL_FIXED_RATE                    0x1b
/** send specific OMI to peer via QoS-null frame
 * param_value = follow 11ax spec definition
 *               bit0:VHT(1), bit1:HE(1), bit2-31:A-Control
 */
#define WMI_PEER_PARAM_XMIT_OMI                         0x1c
#define WMI_PEER_RARAM_XMIT_OMI WMI_PEER_PARAM_XMIT_OMI /* alias due to prior typo */

/* Disable burst and assist */
#define WMI_PEER_PARAM_DISABLE_AGGRESSIVE_TX            0x1d
/* Enable 11r FT Roaming */
#define  WMI_PEER_PARAM_ENABLE_FT                       0x1e

#define  WMI_PEER_PARAM_NEED_PTK_4_WAY                  0x1f
/* update peer flag for gtk 2 way handshake */
#define  WMI_PEER_PARAM_NEED_GTK_2_WAY                  0x20
/* update peer flag for M4 sent */
#define  WMI_PEER_PARAM_M4_SENT                         0x21

/* Per peer MISC stats enable or disable */
#define  WMI_PEER_PARAM_MISC_STATS_ENABLE               0x22

/** mimo ps values for the parameter WMI_PEER_MIMO_PS_STATE  */
#define WMI_PEER_MIMO_PS_NONE                          0x0
#define WMI_PEER_MIMO_PS_STATIC                        0x1
#define WMI_PEER_MIMO_PS_DYNAMIC                       0x2

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_set_param_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /** parameter id   */
    A_UINT32 param_id;
    /** parametr value */
    A_UINT32 param_value;
} wmi_peer_set_param_cmd_fixed_param;

typedef union {
/*
 * The A_UINT16 "mode" and "tx_rate" fields can only be directly used
 * by the target or a little-endian host.
 * A big-endian host needs to use the WMI_PEER_MAX_MIN_TX_xxx_GET/SET
 * macros on the A_UINT32 "value" field.
 */
	struct {
		A_UINT16 mode; /* 0:CCK, 1:OFDM, 2:HT, 3:VHT (see WMI_RATE_PREAMBLE) */
		A_UINT16 tx_rate; /* see per-mode specs below */
	};
	A_UINT32 value; /* for use by big-endian host */
} wmi_peer_max_min_tx_rate;

/*
 * Any access to the mode/tx_rate in an big endian system should use
 * the below Macros on the wmi_peer_max_min_tx_rate.value field.
 */
#define WMI_PEER_MAX_MIN_TX_MODE_GET(value32) WMI_GET_BITS(value32, 0, 16)
#define WMI_PEER_MAX_MIN_TX_MODE_SET(value32, tx_mode) WMI_SET_BITS(value32, 0, 16, tx_mode)

#define WMI_PEER_MAX_MIN_TX_RATE_GET(value32) WMI_GET_BITS(value32, 16, 16)
#define WMI_PEER_MAX_MIN_TX_RATE_SET(value32, tx_mode) WMI_SET_BITS(value32, 16, 16, tx_mode)

/*   CCK max/min tx Rate description
 *   tx_rate = 0:  1   Mbps
 *   tx_rate = 1:  2   Mbps
 *   tx_rate = 2:  5.5 Mbps
 *   tx_rate = 3: 11   Mbps
 *   tx_rate = else: invalid
 */
enum {
    WMI_MAX_CCK_TX_RATE_1M,       /* up to 1M CCK Rate avaliable */
    WMI_MAX_CCK_TX_RATE_2M,       /* up to 2M CCK Rate avaliable */
    WMI_MAX_CCK_TX_RATE_5_5M,     /* up to 5.5M CCK Rate avaliable */
    WMI_MAX_CCK_TX_RATE_11M,      /* up to 11M CCK Rate avaliable */
    WMI_MAX_CCK_TX_RATE = WMI_MAX_CCK_TX_RATE_11M,
};

/*   OFDM max/min tx Rate description
 *   tx_rate = 0:  6 Mbps
 *   tx_rate = 1:  9 Mbps
 *   tx_rate = 2: 12 Mbps
 *   tx_rate = 3: 18 Mbps
 *   tx_rate = 4: 24 Mbps
 *   tx_rate = 5: 32 Mbps
 *   tx_rate = 6: 48 Mbps
 *   tx_rate = 7: 54 Mbps
 *   tx_rate = else: invalid
 */
enum {
    WMI_MAX_OFDM_TX_RATE_6M,      /* up to 6M OFDM Rate avaliable */
    WMI_MAX_OFDM_TX_RATE_9M,      /* up to 9M OFDM Rate avaliable */
    WMI_MAX_OFDM_TX_RATE_12M,     /* up to 12M OFDM Rate avaliable */
    WMI_MAX_OFDM_TX_RATE_18M,     /* up to 18M OFDM Rate avaliable */
    WMI_MAX_OFDM_TX_RATE_24M,     /* up to 24M OFDM Rate avaliable */
    WMI_MAX_OFDM_TX_RATE_36M,     /* up to 36M OFDM Rate avaliable */
    WMI_MAX_OFDM_TX_RATE_48M,     /* up to 48M OFDM Rate avaliable */
    WMI_MAX_OFDM_TX_RATE_54M,     /* up to 54M OFDM Rate avaliable */
    WMI_MAX_OFDM_TX_RATE = WMI_MAX_OFDM_TX_RATE_54M,
};

/*   HT max/min tx rate description
 *    tx_rate = 0~7 : MCS Rate 0~7
 *    tx_rate=else : invalid.
 */
#define WMI_MAX_HT_TX_MCS 0x07

/*   VHT max/min tx rate description
 *    tx_rate = 0~9 : MCS Rate 0~9
 *    tx_rate=else : invalid.
 */
#define WMI_MAX_VHT_TX_MCS 0x09


#define MAX_SUPPORTED_RATES 128

typedef struct {
    /** total number of rates */
    A_UINT32 num_rates;
    /**
     * rates (each 8bit value) packed into a 32 bit word.
     * the rates are filled from least significant byte to most
     * significant byte.
     */
    A_UINT32 rates[(MAX_SUPPORTED_RATES/4)+1];
} wmi_rate_set;

/* NOTE: It would bea good idea to represent the Tx MCS
 * info in one word and Rx in another word. This is split
 * into multiple words for convenience
 */
typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vht_rate_set */
    A_UINT32 rx_max_rate; /* Max Rx data rate */
    A_UINT32 rx_mcs_set; /* Negotiated RX VHT rates */
    A_UINT32 tx_max_rate; /* Max Tx data rate */
    /*
     *  bit [15:0]  indicates MCS 0 to 9
     *  bit [23:16] indicates MCS 10 & 11
     *  bit [24]    indicates whether MCS 10 & 11 is notified in bit [23:16]
     */
    A_UINT32 tx_mcs_set; /* Negotiated TX VHT rates */
    A_UINT32 tx_max_mcs_nss;  /* b0-b3: max mcs idx; b4-b7: max nss */
} wmi_vht_rate_set;

/* NOTE: It would bea good idea to represent the Tx MCS
 * info in one word and Rx in another word. This is split
 * into multiple words for convenience
 * currently this is being defined in IEEE802.11ax so this is same as wmi_vht_rate_set and is  sub change in future and may include BW as well
 */
typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_he_rate_set */
    /* HE Supported MCS Set field Rx
     * - 2 bits are used for each NSS chain.Max of 8 NSS can be encoded with
     *    value 0 - MCS 0-7 enabled for this NSS
     *    value 1 - MCS 0-9 enabled for this NSS
     *    value 2 - MCS 0-11 enabled for this NSS
     *    value 3 - NSS disabled
     * - WMI_HE_MAX_MCS_4_SS_MASK macro can be used for encoding this info
     */
    A_UINT32 rx_mcs_set; /* Negotiated RX HE rates (i.e. rate this node can RX from peer)*/
    /* HE Supported MCS Set field Tx
     * - 2 bits are used for each NSS chain.Max of 8 NSS can be encoded with
     *    value 0 - MCS 0-7 enabled for this NSS
     *    value 1 - MCS 0-9 enabled for this NSS
     *    value 2 - MCS 0-11 enabled for this NSS
     *    value 3 - NSS disabled
     * - WMI_HE_MAX_MCS_4_SS_MASK macro can be used for encoding this info
     */
    A_UINT32 tx_mcs_set; /* Negotiated TX HE rates(i.e. rate this node can TX to peer) */
} wmi_he_rate_set;

/*
 * IMPORTANT: Make sure the bit definitions here are consistent
 * with the ni_flags definitions in wlan_peer.h
 */
#define WMI_PEER_AUTH           0x00000001  /* Authorized for data */
#define WMI_PEER_QOS            0x00000002  /* QoS enabled */
#define WMI_PEER_NEED_PTK_4_WAY 0x00000004  /* Needs PTK 4 way handshake for authorization */
#define WMI_PEER_NEED_GTK_2_WAY 0x00000010  /* Needs GTK 2 way handshake after 4-way handshake */
#define WMI_PEER_HE             0x00000400  /* HE Enabled */
#define WMI_PEER_APSD           0x00000800  /* U-APSD power save enabled */
#define WMI_PEER_HT             0x00001000  /* HT enabled */
#define WMI_PEER_40MHZ          0x00002000  /* 40MHz enabld */
#define WMI_PEER_STBC           0x00008000  /* STBC Enabled */
#define WMI_PEER_LDPC           0x00010000  /* LDPC ENabled */
#define WMI_PEER_DYN_MIMOPS     0x00020000  /* Dynamic MIMO PS Enabled */
#define WMI_PEER_STATIC_MIMOPS  0x00040000  /* Static MIMO PS enabled */
#define WMI_PEER_SPATIAL_MUX    0x00200000  /* SM Enabled */
#define WMI_PEER_TWT_REQ        0x00400000  /* TWT Requester Support bit in Extended Capabilities element */
#define WMI_PEER_TWT_RESP       0x00800000  /* TWT Responder Support bit in Extended Capabilities element */
#define WMI_PEER_MULTI_BSSID    0x01000000  /* Multiple BSSID Support bit in Extended Capabilities element */
#define WMI_PEER_VHT            0x02000000  /* VHT Enabled */
#define WMI_PEER_80MHZ          0x04000000  /* 80MHz enabld */
#define WMI_PEER_PMF            0x08000000  /* Robust Management Frame Protection enabled */
/** CAUTION TODO: Place holder for WLAN_PEER_F_PS_PRESEND_REQUIRED = 0x10000000. Need to be clean up */
#define WMI_PEER_IS_P2P_CAPABLE 0x20000000  /* P2P capable peer */
#define WMI_PEER_160MHZ         0x40000000  /* 160 MHz enabled */
#define WMI_PEER_SAFEMODE_EN    0x80000000  /* Fips Mode Enabled */

/**
 * Peer rate capabilities.
 *
 * This is of interest to the ratecontrol
 * module which resides in the firmware. The bit definitions are
 * consistent with that defined in if_athrate.c.
 *
 * @todo
 * Move this to a common header file later so there is no need to
 * duplicate the definitions or maintain consistency.
 */
#define WMI_RC_DS_FLAG          0x01    /* Dual stream flag */
#define WMI_RC_CW40_FLAG        0x02    /* CW 40 */
#define WMI_RC_SGI_FLAG         0x04    /* Short Guard Interval */
#define WMI_RC_HT_FLAG          0x08    /* HT */
#define WMI_RC_RTSCTS_FLAG      0x10    /* RTS-CTS */
#define WMI_RC_TX_STBC_FLAG     0x20    /* TX STBC */
#define WMI_RC_TX_STBC_FLAG_S   5       /* TX STBC */
#define WMI_RC_RX_STBC_FLAG     0xC0    /* RX STBC ,2 bits */
#define WMI_RC_RX_STBC_FLAG_S   6       /* RX STBC ,2 bits */
#define WMI_RC_WEP_TKIP_FLAG    0x100   /* WEP/TKIP encryption */
#define WMI_RC_TS_FLAG          0x200   /* Three stream flag */
#define WMI_RC_UAPSD_FLAG       0x400   /* UAPSD Rate Control */

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_assoc_complete_cmd_fixed_param */
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /** VDEV id */
    A_UINT32 vdev_id;
    /** assoc = 1 reassoc = 0 */
    A_UINT32 peer_new_assoc;
    /** peer associd (16 bits) */
    A_UINT32 peer_associd;
    /** peer station flags: see definition above */
    A_UINT32 peer_flags;
    /** negotiated capabilities (lower 16 bits)*/
    A_UINT32 peer_caps;
    /** Listen interval */
    A_UINT32 peer_listen_intval;
    /** HT capabilties of the peer */
    A_UINT32 peer_ht_caps;
    /** maximum rx A-MPDU length */
    A_UINT32 peer_max_mpdu;
    /** mpdu density of the peer in usec(0 to 16) */
    A_UINT32 peer_mpdu_density;
    /** peer rate capabilties see flags above */
    A_UINT32 peer_rate_caps;
    /** num spatial streams */
    A_UINT32 peer_nss;
    /** VHT capabilties of the peer */
    A_UINT32 peer_vht_caps;
    /** phy mode */
    A_UINT32 peer_phymode;
    /** HT Operation Element of the peer. Five bytes packed in 2
         *  INT32 array and filled from lsb to msb.
         *  Note that the size of array peer_ht_info[] cannotbe changed
         *  without breaking WMI Compatibility. */
    A_UINT32 peer_ht_info[2];
    /** total number of negotiated legacy rate set. Also the sizeof
     *  peer_legacy_rates[] */
    A_UINT32 num_peer_legacy_rates;
    /** total number of negotiated ht rate set. Also the sizeof
     *  peer_ht_rates[] */
    A_UINT32 num_peer_ht_rates;
    /**
     * Bitmap providing the mapping of bandwidths to max Rx NSS for this peer
     * in VHT160 / VHT80+80 Mode.
     * As per the New IEEE 802.11 Update, the AP & Peer could advertise and
     * handshake with the Max Rx NSS differing for different bandwidths
     * instead of a single max Rx NSS Value.
     * Some QCA chipsets have to advertise a different max Rx NSS value for
     * 160 MHz and 80MHz.
     *
     *  bit[2:0] : Represents value of Rx NSS for VHT 160 MHz
     *  bit[5:3] : Represents value of Rx NSS for VHT 80_80 MHz -
     *             Extended NSS support
     *  bit[30:6]: Reserved
     *  bit[31]  : MSB(0/1): 1 in case of valid data sent from Host
     */
    A_UINT32 peer_bw_rxnss_override;

    /* 802.11ax capabilities */
    wmi_ppe_threshold peer_ppet;
    A_UINT32 peer_he_cap_info; /* protocol-defined HE / 11ax capability flags */
    A_UINT32 peer_he_ops; /* HE operation contains BSS color */
    A_UINT32 peer_he_cap_phy[WMI_MAX_HECAP_PHY_SIZE];
    A_UINT32 peer_he_mcs; /* Indicates number of HE MCS TLV present */

    /* 2nd DWORD of 11ax MAC Capabilities */
    A_UINT32 peer_he_cap_info_ext;

    /*
     * bit 0     : Indicated support for RX 1xLTF + 0.4us
     * bit 1     : Indicates support for RX 2xLTF + 0.4us
     * bit 2     : Indicates support for 2xLTF in 160/80+80 MHz HE PPDU
     * bit[31:3] : Reserved
     * Refer to WMI_HE_CAP_xx_LTF_xxx_SUPPORT_GET/SET macros
     */
    A_UINT32 peer_he_cap_info_internal;

    /* Following this struct are the TLV's:
         *     A_UINT8 peer_legacy_rates[];
         *     A_UINT8 peer_ht_rates[];
         *     wmi_vht_rate_set peer_vht_rates; //VHT capabilties of the peer
         *     WMI_he_rate_set_peer_he_rates; <-- HE capabilities of the peer
         */
} wmi_peer_assoc_complete_cmd_fixed_param;

/* WDS Entry Flags */
#define WMI_WDS_FLAG_STATIC    0x1    /* Disable aging & learning */

typedef struct {
    A_UINT32     tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_add_wds_entry_cmd_fixed_param */
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /** wds MAC addr */
    wmi_mac_addr wds_macaddr;
    /* Flags associated with WDS entry - see WMI_WDS_FLAG defs */
    A_UINT32 flags;
    A_UINT32 vdev_id;
} wmi_peer_add_wds_entry_cmd_fixed_param;

#define WMI_CHAN_INFO_START_RESP  0
#define WMI_CHAN_INFO_END_RESP    1

#define WMI_CHAN_InFO_START_RESP WMI_CHAN_INFO_START_RESP
#define WMI_CHAN_InFO_END_RESP   WMI_CHAN_INFO_END_RESP

typedef struct {
    A_UINT32     tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_remove_wds_entry_cmd_fixed_param */
    /** wds MAC addr */
    wmi_mac_addr wds_macaddr;
    A_UINT32 vdev_id;
} wmi_peer_remove_wds_entry_cmd_fixed_param;


typedef struct {
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
} wmi_peer_q_empty_callback_event;

/*
 * Command to update an already existing WDS entry. Different address setting
 * combinations are possible.
 *
 * Valid wds and peer      -> Associated WDS entry peer ptr & flags will be updated.
 * Valid wds and null peer -> Associated WDS entry flags will be updated.
 * Null wds and Valid peer -> Flags will be updated for all WDS entries behind the peer.
 * Null wds and peer       -> Flags will be updated for all WDS entries.
 */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_update_wds_entry_cmd_fixed_param */
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /** wds MAC addr */
    wmi_mac_addr wds_macaddr;
    /* Flags associated with WDS entry */
    A_UINT32 flags;
    A_UINT32 vdev_id;
} wmi_peer_update_wds_entry_cmd_fixed_param;

/**
 * Channel info WMI event
 */
typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_chan_info_event_fixed_param */
     /** Error code */
    A_UINT32 err_code;
   /** Channel freq */
    A_UINT32 freq;
    /** Read flags */
    A_UINT32 cmd_flags;
    /** Noise Floor value */
    A_UINT32 noise_floor;
    /** rx clear count */
    A_UINT32   rx_clear_count;
    /** cycle count */
    A_UINT32   cycle_count;
    /** channel tx power per range in 0.5dBm steps */
    A_UINT32 chan_tx_pwr_range;
    /** channel tx power per throughput */
    A_UINT32 chan_tx_pwr_tp;
    /** rx frame count (cumulative) */
    A_UINT32   rx_frame_count;
    /** BSS rx cycle count */
    A_UINT32 my_bss_rx_cycle_count;
    /** b-mode data rx time (units are microseconds) */
    A_UINT32 rx_11b_mode_data_duration;
    /** tx frame count */
    A_UINT32 tx_frame_cnt;
    /** mac clock */
    A_UINT32 mac_clk_mhz;
    /** unique id identifying the VDEV */
    A_UINT32 vdev_id;
} wmi_chan_info_event_fixed_param;

/**
 * Non wlan interference event
 */
typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wlan_dcs_cw_int */
    A_UINT32 channel; /* either number or freq in mhz*/
} wlan_dcs_cw_int;
#define ath_dcs_cw_int /* DEPRECATED */ wlan_dcs_cw_int /* alias */

/**
 *  wlan_dcs_im_tgt_stats
 *
 */
typedef struct _wlan_dcs_im_tgt_stats {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wlan_dcs_im_tgt_stats_t */
    /** current running TSF from the TSF-1 */
    A_UINT32                  reg_tsf32;

    /** Known last frame rssi, in case of multiple stations, if
     *  and at different ranges, this would not gaurantee that
     *  this is the least rssi.
     */
    A_UINT32                  last_ack_rssi;

    /**  Sum of all the failed durations in the last one second interval.
     */
    A_UINT32                  tx_waste_time;
    /** count how many times the hal_rxerr_phy is marked, in this
     *  time period
     */
    A_UINT32                rx_time;
    A_UINT32                 phyerr_cnt;

    /**
         *  WLAN IM stats from target to host
         *
         *  Below statistics are sent from target to host periodically.
         *  These are collected at target as long as target is running
         *  and target chip is not in sleep.
         *
         */

    /** listen time from ANI */
    A_INT32   listen_time;

    /** tx frame count, MAC_PCU_TX_FRAME_CNT_ADDRESS */
    A_UINT32   reg_tx_frame_cnt;

    /** rx frame count, MAC_PCU_RX_FRAME_CNT_ADDRESS */
    A_UINT32   reg_rx_frame_cnt;

    /** rx clear count, MAC_PCU_RX_CLEAR_CNT_ADDRESS */
    A_UINT32   reg_rxclr_cnt;

    /** total cycle counts MAC_PCU_CYCLE_CNT_ADDRESS */
    A_UINT32   reg_cycle_cnt;            /* delta cycle count */

    /** extenstion channel rx clear count  */
    A_UINT32   reg_rxclr_ext_cnt;

    /** OFDM phy error counts, MAC_PCU_PHY_ERR_CNT_1_ADDRESS */
    A_UINT32   reg_ofdm_phyerr_cnt;

    /** CCK phy error count, MAC_PCU_PHY_ERR_CNT_2_ADDRESS */
    A_UINT32   reg_cck_phyerr_cnt;        /* CCK err count since last reset, read from register */

    /** Channel noise floor (units are dBm) */
    A_INT32 chan_nf;

    /** BSS rx cycle count */
    A_UINT32 my_bss_rx_cycle_count;
} wlan_dcs_im_tgt_stats_t;

/**
 *  wmi_dcs_interference_event_t
 *
 *  Right now this is event and stats together. Partly this is
 *  because cw interference is handled in target now. This
 *  can be done at host itself, if we can carry the NF alone
 *  as a stats event. In future this would be done and this
 *  event would carry only stats.
 */
typedef struct {
    A_UINT32    tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_dcs_interference_event_fixed_param */
    /**
     * Type of the event present, either the cw interference event, or the wlan_im stats
     */
    A_UINT32    interference_type;      /* type of interference, wlan or cw */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
    /*
     * Following this struct are these TLVs. Note that they are both array of structures
     * but can have at most one element. Which TLV is empty or has one element depends
     * on the field interference_type. This is to emulate an union with cw_int and wlan_stat
     * elements (not arrays).     union { wlan_dcs_cw_int cw_int; wlan_dcs_im_tgt_stats_t   wlan_stat; } int_event;
     *
     *        //cw_interference event
     *       wlan_dcs_cw_int            cw_int[];  this element
     *       // wlan im interfernce stats
     *       wlan_dcs_im_tgt_stats_t   wlan_stat[];
     */
} wmi_dcs_interference_event_fixed_param;

enum wmi_peer_mcast_group_action {
    wmi_peer_mcast_group_action_add = 0,
    wmi_peer_mcast_group_action_del = 1
};
#define WMI_PEER_MCAST_GROUP_FLAG_ACTION_M   0x1
#define WMI_PEER_MCAST_GROUP_FLAG_ACTION_S   0
#define WMI_PEER_MCAST_GROUP_FLAG_WILDCARD_M 0x2
#define WMI_PEER_MCAST_GROUP_FLAG_WILDCARD_S 1
#define WMI_PEER_MCAST_GROUP_FLAG_SRC_FILTER_EXCLUDE_M 0x4 /* flag to exclude an ip while filtering. set to exclude */
#define WMI_PEER_MCAST_GROUP_FLAG_SRC_FILTER_EXCLUDE_S 2
#define WMI_PEER_MCAST_GROUP_FLAG_IPV6_M 0x8 /* flag to say ipv4/ipv6. Will be set for ipv6 */
#define WMI_PEER_MCAST_GROUP_FLAG_IPV6_S 3
#define WMI_PEER_MCAST_GROUP_FLAG_DELETEALL_M 0x10 /* delete all mcast table entries. */
#define WMI_PEER_MCAST_GROUP_FLAG_DELETEALL_S 4

/* multicast group membership commands */
/* TODO: Converting this will be tricky since it uses an union.
   Also, the mac_addr is not aligned. We will convert to the wmi_mac_addr */
typedef struct{
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_mcast_group_cmd_fixed_param */
    A_UINT32 flags;
    wmi_mac_addr ucast_mac_addr;
    /* in network byte order */
    /* for ipv4, bytes (12-15) should contain ip address and other lower bytes 0. ipv6 should have all bytes valid */
    A_UINT8 mcast_ip_addr[16];
    /* for ipv6, all 16 bytes has to be valid; for ipv4 last 4 bytes(12-15) has to be valid, rest all 0s */
    A_UINT8 mcast_ip_mask[16];/* zero out lower bytes if ipv4*/
    /* number of address filters - irrespective of ipv4/ipv6 addresses */
    A_UINT32 num_filter_addr;
    /* this array should contain the src IPs that are to be filtered during find
       The array should be packed.
       If there are 2 ipv4 addresses, there should be 8 bytes and rest all 0s */
    A_UINT8 filter_addr[64]; /* 16 ipv4 addresses or 4 ipv6 addresses */
    A_UINT8  vdev_id; /* vdev of this mcast group */
} wmi_peer_mcast_group_cmd_fixed_param;


/** Offload Scan and Roaming related  commands */
/** The FW performs 2 different kinds of offload scans independent
 *  of host. One is Roam scan which is primarily performed  on a
 *  station VDEV after association to look for a better AP that
 *  the station VDEV can roam to. The second scan is connect scan
 *  which is mainly performed when the station is not associated
 *  and to look for a matching AP profile from a list of
 *  configured profiles. */

/* flags for roam_scan_mode_cmd
 * indicate the status (success/fail) of wmi_roam_scan_mode cmd through WMI_ROAM_EVENTID */
#define WMI_ROAM_SCAN_MODE_FLAG_REPORT_STATUS  0x1

/**
 * WMI_ROAM_SCAN_MODE: Set Roam Scan mode
 *   the roam scan mode is one of the periodic, rssi change, both, none.
 *   None        : Disable Roam scan. No Roam scan at all.
 *   Periodic    : Scan periodically with a configurable period.
 *   Rssi change : Scan when ever rssi to current AP changes by the threshold value
 *                 set by WMI_ROAM_SCAN_RSSI_CHANGE_THRESHOLD command.
 *   Both        : Both of the above (scan when either period expires or rss to current AP changes by X amount)
 *
 */
typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_scan_mode_fixed_param */
    A_UINT32 roam_scan_mode;
    A_UINT32 vdev_id;
    A_UINT32 flags; /* see WMI_ROAM_SCAN_MODE_FLAG defs */
    /*
     * Minimum duration allowed between two consecutive roam scans.
     * Roam scan is not allowed, if duration between two consecutive
     * roam scans is less than this time.
     */
    A_UINT32 min_delay_btw_scans; /* In msec */
    /*
     * Bitmask (with enum WMI_ROAM_TRIGGER_REASON_ID identifying the bit
     * positions) showing for which roam_trigger_reasons the
     * min_delay_btw_scans constraint should be applied.
     * 0x0 means there is no time restrictions between successive roam scans.
     */
    A_UINT32 min_delay_roam_trigger_reason_bitmask;
} wmi_roam_scan_mode_fixed_param;

#define WMI_ROAM_SCAN_MODE_NONE        0x0
#define WMI_ROAM_SCAN_MODE_PERIODIC    0x1
#define WMI_ROAM_SCAN_MODE_RSSI_CHANGE 0x2
#define WMI_ROAM_SCAN_MODE_BOTH        0x3
/* Note: WMI_ROAM_SCAN_MODE_ROAMOFFLOAD is one bit not conflict with LFR2.0 SCAN_MODE. */
#define WMI_ROAM_SCAN_MODE_ROAMOFFLOAD 0x4


typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_scan_cmd_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 command_arg;
} wmi_roam_scan_cmd_fixed_param;

#define WMI_ROAM_SCAN_STOP_CMD 0x1

/**
 * WMI_ROAM_SCAN_RSSI_THRESHOLD : set scan rssi threshold
 *  scan rssi threshold is the rssi threshold below which the FW will start running Roam scans.
 * Applicable when WMI_ROAM_SCAN_MODE is not set to none.
 */
typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_scan_rssi_threshold_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** roam scan rssi threshold */
    A_UINT32 roam_scan_rssi_thresh;
    /** When using Hw generated beacon RSSI interrupts */
    A_UINT32 roam_rssi_thresh_diff;
    /** 5G scan max count */
    A_UINT32 hirssi_scan_max_count;
    /** 5G scan rssi change threshold value */
    A_UINT32 hirssi_scan_delta;
    /** 5G scan upper bound */
    A_UINT32 hirssi_upper_bound;
    /** roam scan rssi threshold for 5G band.
     *  offset from roam_scan_rssi_thres, in dB units
     */
    A_INT32 rssi_thresh_offset_5g;
    /* The TLVs will follow.
     * wmi_roam_scan_extended_threshold_param extended_param;
     * wmi_roam_earlystop_rssi_thres_param earlystop_param;
     * wmi_roam_dense_thres_param dense_param;
     * wmi_roam_bg_scan_roaming_param bg_scan_param;
     */
} wmi_roam_scan_rssi_threshold_fixed_param;

/**
 * WMI_ROAM_BTM_CONFIG_CMDID : set BTM (BSS Transition Management. 802.11v) offload config
 *  Applicable only when WMI_ROAM_SCAN_MODE is enabled with WMI_ROAM_SCAN_MODE_ROAMOFFLOAD
 */

/**
 *  btm_config.flags
 *  BIT 0     : Enable/Disable the BTM offload.
 *  BIT 1-2   : Action on non matching candidate with cache. Used WMI_ROAM_BTM_OFLD_NON_MATCHING_CND_XXX
 *  BIT 3-5   : Roaming handoff decisions. Use WMI_ROAM_BTM_OFLD_CNDS_MATCH_XXX
 *  BIT 6     : Enable/Disable solicited BTM
 *  BIT 7     : Roam BTM candidates based on the roam score instead of BTM preferred value
 *  BIT 8     : BTM query preference over 11k neighbor report request
 *  BIT 9-31  : Reserved
 */
#define WMI_ROAM_BTM_SET_ENABLE(flags, val)                        WMI_SET_BITS(flags, 0, 1, val)
#define WMI_ROAM_BTM_GET_ENABLE(flags)                             WMI_GET_BITS(flags, 0, 1)
#define WMI_ROAM_BTM_SET_NON_MATCHING_CND_ACTION(flags, val)       WMI_SET_BITS(flags, 1, 2, val)
#define WMI_ROAM_BTM_GET_NON_MATCHING_CND_ACTION(flags)            WMI_GET_BITS(flags, 1, 2)
#define WMI_ROAM_BTM_SET_CNDS_MATCH_CONDITION(flags, val)          WMI_SET_BITS(flags, 3, 3, val)
#define WMI_ROAM_BTM_GET_CNDS_MATCH_CONDITION(flags)               WMI_GET_BITS(flags, 3, 3)
#define WMI_ROAM_BTM_SET_SOLICITED_BTM_ENABLE(flags, val)          WMI_SET_BITS(flags, 6, 1, val)
#define WMI_ROAM_BTM_GET_SOLICITED_BTM_ENABLE(flags)               WMI_GET_BITS(flags, 6, 1)
#define WMI_ROAM_BTM_SET_CNDS_SELECT_BASED_ON_SCORE(flags, val)    WMI_SET_BITS(flags, 7, 1, val)
#define WMI_ROAM_BTM_GET_CNDS_SELECT_BASED_ON_SCORE(flags)         WMI_GET_BITS(flags, 7, 1)
#define WMI_ROAM_BTM_SET_BTM_QUERY_PREFERENCE_OVER_11K(flags, val) WMI_SET_BITS(flags, 8, 1, val)
#define WMI_ROAM_BTM_GET_BTM_QUERY_PREFERENCE_OVER_11K(flags)      WMI_GET_BITS(flags, 8, 1)
#define WMI_ROAM_BTM_SET_BTM_QUERY_WITH_CANDIDATE_LIST(flags, val) WMI_SET_BITS(flags, 9, 1, val)
#define WMI_ROAM_BTM_GET_BTM_QUERY_WITH_CANDIDATE_LIST(flags)      WMI_GET_BITS(flags, 9, 1)


/** WMI_ROAM_BTM_SET_NON_MATCHING_CNDS_ACTION definition: When BTM candidate is not matched with cache by WMI_ROAM_BTM_SET_CNDS_MATCH_CONDITION, determine what to do */
#define WMI_ROAM_BTM_NON_MATCHING_CNDS_SCAN_CONSUME      0 /** Invoke roam scan and consume within firmware. Applicable only when ROAM_SCAN_MODE is enabled. If ROAM_SCAN_MODE is disabled, firmware won't scan and forward it to host */
#define WMI_ROAM_BTM_NON_MATCHING_CNDS_NO_SCAN_FORWARD   1 /** Does not allow roam scan and forward BTM frame to host */
/** reserved upto 3 */

/** WMI_ROAM_BTM_SET_CNDS_MATCH_CONDITION definition: This is used to invoke WMI_ROAM_BTM_SET_NON_MATCHING_CND_ACTION when compared with cache. i.e this condition is not applied with fresh scan result */
#define WMI_ROAM_BTM_CNDS_MATCH_EXACT                    0 /** Exactly matched with roam candidate list to BTM candidates */
#define WMI_ROAM_BTM_CNDS_MATCH_AT_LEAST_TOP             1 /** At least one or more top priority bssid matched */
/** reserved upto 7 */

typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_btm_config_fixed_param */
    A_UINT32 tlv_header;
    /** unique id identifying the VDEV on which BTM is enabled/disabled */
    A_UINT32 vdev_id;
    /** BTM configuration control flags */
    A_UINT32 flags;
    /*  BTM query timeout, unit: milliseconds
     *  valid value range: 1-10000,
     *  default value: 100 will be set if invalid value is given
     */
    A_UINT32 solicited_timeout_ms;
    /*  Maximum attempt of solicited BTM
     *  If continuous failure reach to this value, solicited BTM to current
     *  ESS will be disabled.
     *  valid value range: 1 - (2^32-1). (2^32)-1 means sending forever
     *  Default value: 3 will be set if invalid value is given
     */
    A_UINT32 max_attempt_cnt;
    /*  Time to stick to current AP after BTM, unit: seconds
     *  valid value range: 0 -(2^16-1).
     *  Either 0 or (2^16)-1 means stick to AP forever.
     *  Default value: 300 will be set if invalid value is given
     */
     A_UINT32 stick_time_seconds;
    /*  Disassoc time threshold in milli seconds
     *  This time threshold allows the target to judge whether the STA
     *  should can move to another AP immediately, or if the STA has time
     *  to calculate roaming candidates.
     *  If the disassoc_timer_threshold value is 0x0, the field should be
     *  disregarded.
     */
    A_UINT32 disassoc_timer_threshold;
    /*
     * Bitmask (with enum WMI_ROAM_TRIGGER_REASON_ID identifying the bit
     * positions) showing for which roam_trigger_reasons the
     * btm query needs to be sent.
     * If roam trigger reasons are unspecified, btm_bitmap will be 0x0.
     */
    A_UINT32 btm_bitmap;
} wmi_btm_config_fixed_param;

#define WMI_ROAM_5G_BOOST_PENALIZE_ALGO_FIXED  0x0
#define WMI_ROAM_5G_BOOST_PENALIZE_ALGO_LINEAR 0x1
#define WMI_ROAM_5G_BOOST_PENALIZE_ALGO_LOG    0x2
#define WMI_ROAM_5G_BOOST_PENALIZE_ALGO_EXP    0x3

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_scan_extended_threshold_param */
    A_UINT32 boost_threshold_5g; /** RSSI threshold above which 5GHz RSSI is favored */
    A_UINT32 penalty_threshold_5g; /** RSSI threshold below which 5GHz RSSI is penalized */
    A_UINT32 boost_algorithm_5g; /** 0 == fixed, 1 == linear, 2 == logarithm ..etc */
    A_UINT32 boost_factor_5g; /** factor by which 5GHz RSSI is boosted */
    A_UINT32 penalty_algorithm_5g; /** 0 == fixed, 1 == linear, 2 == logarithm ..etc */
    A_UINT32 penalty_factor_5g; /** factor by which 5GHz RSSI is penalized */
    A_UINT32 max_boost_5g; /** maximum boost that can be applied to a 5GHz RSSI */
    A_UINT32 max_penalty_5g; /** maximum penality that can be applied to a 5GHz RSSI */
    A_UINT32 good_rssi_threshold; /**  RSSI below which roam is kicked in by background scan, although rssi is still good */
} wmi_roam_scan_extended_threshold_param;

/**
 * WMI_ROAM_SCAN_PERIOD: period for roam scan.
 *  Applicable when the scan mode is Periodic or both.
 */
typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_scan_period_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** roam scan period value */
    A_UINT32 roam_scan_period;
    /** Aging for Roam scans */
    A_UINT32 roam_scan_age;
} wmi_roam_scan_period_fixed_param;

/**
 * WMI_ROAM_SCAN_RSSI_CHANGE_THRESHOLD : rssi delta to trigger the roam scan.
 *   Rssi change threshold used when mode is Rssi change (or) Both.
 *   The FW will run the roam scan when ever the rssi changes (up or down) by the value set by this parameter.
 *   Note scan is triggered based on the rssi threshold condition set by WMI_ROAM_SCAN_RSSI_THRESHOLD
 */
typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_scan_rssi_change_threshold_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** roam scan rssi change threshold value */
    A_UINT32 roam_scan_rssi_change_thresh;
    /** When using Hw generated beacon RSSI interrupts */
    A_UINT32 bcn_rssi_weight;
    /** Minimum delay between two 5G scans */
    A_UINT32 hirssi_delay_btw_scans;
} wmi_roam_scan_rssi_change_threshold_fixed_param;

#define WMI_ROAM_SCAN_CHAN_LIST_TYPE_NONE 0x1
#define WMI_ROAM_SCAN_CHAN_LIST_TYPE_STATIC 0x2
#define WMI_ROAM_SCAN_CHAN_LIST_TYPE_DYNAMIC 0x3
/**
 * TLV for roaming channel list
 */
typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_chan_list_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** WMI_CHAN_LIST_TAG */
    A_UINT32 chan_list_type;
    /** # if channels to scan */
    A_UINT32 num_chan;
/**
 * TLV (tag length value ) parameters follow the wmi_roam_chan_list
 * structure. The TLV's are:
 *     A_UINT32 channel_list[];
 **/
} wmi_roam_chan_list_fixed_param;

/** Authentication modes */
enum {
    WMI_AUTH_NONE   , /* no upper level auth */
    WMI_AUTH_OPEN   , /* open */
    WMI_AUTH_SHARED , /* shared-key */
    WMI_AUTH_8021X  , /* 802.1x */
    WMI_AUTH_AUTO   , /* Auto */
    WMI_AUTH_WPA    , /* WPA */
    WMI_AUTH_RSNA   , /* WPA2/RSNA */
    WMI_AUTH_CCKM   , /* CCKM */
    WMI_AUTH_WAPI   ,/* WAPI */
    WMI_AUTH_AUTO_PSK,
    WMI_AUTH_WPA_PSK,
    WMI_AUTH_RSNA_PSK,
    WMI_AUTH_WAPI_PSK,
    WMI_AUTH_FT_RSNA, /* 11r FT */
    WMI_AUTH_FT_RSNA_PSK,
    WMI_AUTH_RSNA_PSK_SHA256,
    WMI_AUTH_RSNA_8021X_SHA256,
    WMI_AUTH_CCKM_WPA,
    WMI_AUTH_CCKM_RSNA,
    WMI_AUTH_RSNA_FILS_SHA256,
    WMI_AUTH_RSNA_FILS_SHA384,
    WMI_AUTH_RSNA_SUITE_B_8021X_SHA256,
    WMI_AUTH_RSNA_SUITE_B_8021X_SHA384,
    WMI_AUTH_FT_RSNA_SAE,
    WMI_AUTH_FT_RSNA_SUITE_B_8021X_SHA384,
};

typedef enum {
    WMI_SCAN_CLIENT_NLO = 0x1,  /* 1 */
    WMI_SCAN_CLIENT_EXTSCAN,    /* 2 */
    WMI_SCAN_CLIENT_ROAM,       /* 3 */
    WMI_SCAN_CLIENT_P2P,        /* 4 */
    WMI_SCAN_CLIENT_LPI,        /* 5 */
    WMI_SCAN_CLIENT_NAN,        /* 6 */
    WMI_SCAN_CLIENT_ANQP,       /* 7 */
    WMI_SCAN_CLIENT_OBSS,       /* 8 */
    WMI_SCAN_CLIENT_PLM,        /* 9 */
    WMI_SCAN_CLIENT_HOST,       /* 10 */
} WMI_SCAN_CLIENT_ID;

typedef struct {
    /** authentication mode (defined above)  */
    A_UINT32               rsn_authmode;
    /** unicast cipher set */
    A_UINT32               rsn_ucastcipherset;
    /** mcast/group cipher set */
    A_UINT32               rsn_mcastcipherset;
    /** mcast/group management frames cipher set */
    A_UINT32               rsn_mcastmgmtcipherset;
} wmi_rsn_params;

/** looking for a wps enabled AP */
#define WMI_AP_PROFILE_FLAG_WPS    0x1
/** looking for a secure AP  */
#define WMI_AP_PROFILE_FLAG_CRYPTO 0x2
/** looking for a PMF enabled AP */
#define WMI_AP_PROFILE_FLAG_PMF    0x4

/* the value used in wmi_roam_cnd_scoring_param->disable_bitmap */
#define WLAN_ROAM_SCORING_RSSI_DISABLE                  0x00000001
#define WLAN_ROAM_SCORING_HT_DISABLE                    0x00000002
#define WLAN_ROAM_SCORING_VHT_DISABLE                   0x00000004
#define WLAN_ROAM_SCORING_BW_DISABLE                    0x00000008
#define WLAN_ROAM_SCORING_BAND_DISABLE                  0x00000010
#define WLAN_ROAM_SCORING_NSS_DISABLE                   0x00000020
#define WLAN_ROAM_SCORING_CHAN_CONGESTION_DISABLE       0x00000040
#define WLAN_ROAM_SCORING_BEAMFORMING_DISABLE           0x00000080
#define WLAN_ROAM_SCORING_PCL_DISABLE                   0x00000100
#define WLAN_ROAM_SCORING_HE_DISABLE                    0x00000200
#define WLAN_ROAM_SCORING_OCE_WAN_DISABLE               0x00000400
#define WLAN_ROAM_SCORING_DISABLE_ALL                   0xFFFFFFFF
#define WLAN_ROAM_SCORING_DEFAULT_PARAM_ALLOW           0x0

#define WLAN_ROAM_MAX_SELECTION_SCORE                   10000

#define WLAN_ROAM_SCORE_20MHZ_BW_INDEX                  0
#define WLAN_ROAM_SCORE_40MHZ_BW_INDEX                  1
#define WLAN_ROAM_SCORE_80MHZ_BW_INDEX                  2
#define WLAN_ROAM_SCORE_160MHZ_BW_INDEX                 3
#define WLAN_ROAM_SCORE_MAX_BW_INDEX                    4
#define WMI_ROAM_GET_BW_SCORE_PERCENTAGE(value32, bw_index)                   WMI_GET_BITS(value32, (8 * (bw_index)), 8)
#define WMI_ROAM_SET_BW_SCORE_PERCENTAGE(value32, score_pcnt, bw_index)       WMI_SET_BITS(value32, (8 * (bw_index)), 8, score_pcnt)

#define WLAN_ROAM_SCORE_NSS_1x1_INDEX                   0
#define WLAN_ROAM_SCORE_NSS_2x2_INDEX                   1
#define WLAN_ROAM_SCORE_NSS_3x3_INDEX                   2
#define WLAN_ROAM_SCORE_NSS_4x4_INDEX                   3
#define WLAN_ROAM_SCORE_MAX_NSS_INDEX                   4
#define WMI_ROAM_GET_NSS_SCORE_PERCENTAGE(value32, nss_index)                 WMI_GET_BITS(value32, (8 * (nss_index)), 8)
#define WMI_ROAM_SET_NSS_SCORE_PERCENTAGE(value32, score_pcnt, nss_index)     WMI_SET_BITS(value32, (8 * (nss_index)), 8, score_pcnt)

#define WLAN_ROAM_SCORE_BAND_2G_INDEX                   0
#define WLAN_ROAM_SCORE_BAND_5G_INDEX                   1
/* 2 and 3 are reserved */
#define WLAN_ROAM_SCORE_MAX_BAND_INDEX                  4
#define WMI_ROAM_GET_BAND_SCORE_PERCENTAGE(value32, band_index)                 WMI_GET_BITS(value32, (8 * (band_index)), 8)
#define WMI_ROAM_SET_BAND_SCORE_PERCENTAGE(value32, score_pcnt, band_index)     WMI_SET_BITS(value32, (8 * (band_index)), 8, score_pcnt)

#define WLAN_ROAM_SCORE_MAX_CHAN_CONGESTION_SLOT        16
#define WLAN_ROAM_SCORE_DEFAULT_CONGESTION_SLOT         0

#define WLAN_ROAM_SCORE_MAX_OCE_WAN_SLOT                16
#define WLAN_ROAM_SCORE_DEFAULT_OCE_WAN_SLOT            0

/**
    best_rssi_threshold: Roamable AP RSSI equal or better than this threshold, full rssi score 100. Units in dBm.
    good_rssi_threshold: Below this threshold, scoring linear percentage between rssi_good_pcnt and 100. Units in dBm.
    bad_rssi_threshold:  Between good and bad rssi threshold, scoring linear percentage between rssi_bad_pcnt and rssi_good_pcnt. Units in dBm.
    good_rssi_pcnt: Used to assigned scoring percentage of each slot between best to good rssi threshold. Units in percentage.
    bad_rssi_pcnt: Used to assigned scoring percentage of each slot between good to bad rssi threshold. Units in percentage.
    good_bucket_size : bucket size of slot in good zone.  Units in dB.
    bad_bucket_size : bucket size of slot in bad zone. Units in dB.
    rssi_pref_5g_rssi_thresh: Below rssi threshold, 5G AP have given preference of band percentage. Units in dBm.
*/
/**
    The graph below explains how RSSI scoring percentage is calculated
    as the RSSI improves.  In the graph, the derived parameters good_buckets
    and bad_buckets are used.  These derived parameters are related to the
    specified parameters as follows:
        good_buckets =
            (best_rssi_threshold - good_rssi_threshold) / good_bucket_size
        bad_buckets =
            (good_rssi_threshold - bad_rssi_threshold) / bad_bucket_size

      |     (x0,y0) (x0 = best_rssi, y0 = 100)
p 100 |-------o
e     |       |<--------- (100 - good_rssi_pct)/good_buckets
r     |       |___  ,---- good_bucket_size
c     |           | |
e     |           |_V_
n     |               |
t     |               |___o (x1,y1) (x1 = good_rssi, y1 = good_rssi_pcnt)
   80 |                   |
%     |                   |<------ (good_rssi_pcnt - bad_rssi_pcnt)/bad_buckets
      |                   |_____
      |                         |  ,-- bad_bucket_size
      |                         |  |
      |                         |__v__
      |                               |
      |                               |
   40 |                               o------------
      |                             (x2,y2) (x2 = bad_rssi, y2 = bad_rssi_pcnt)
      +------o------------o-----------o------------->
            -50         -70          -80          rssi dBm

| excellent  |  good      | bad       | poor
| zone       |  zone      | zone      | zone
             V            V           V
        BEST_THRES   GOOD_THRES     BAD_THRES
 */
typedef struct {
    A_INT32  best_rssi_threshold;
    A_INT32  good_rssi_threshold;
    A_INT32  bad_rssi_threshold;
    A_UINT32 good_rssi_pcnt;
    A_UINT32 bad_rssi_pcnt;
    A_UINT32 good_bucket_size;
    A_UINT32 bad_bucket_size;
    A_INT32  rssi_pref_5g_rssi_thresh;
} wmi_roam_cnd_rssi_scoring;

/**
    Use macro WMI_ROAM_CND_GET/SET_BW_SCORE_PERCENTAGE to get and set the value respectively.
    BITS 0-7   :- It contains scoring percentage of 20MHz   BW
    BITS 8-15  :- It contains scoring percentage of 40MHz   BW
    BITS 16-23 :- It contains scoring percentage of 80MHz   BW
    BITS 24-31 :- It contains scoring percentage of 1600MHz BW

    The value of each index must be 0-100
 */
typedef struct {
    A_UINT32 score_pcnt;
} wmi_roam_cnd_bw_scoring;

/**
    Use macro WMI_ROAM_CND_GET/SET_BAND_SCORE_PERCENTAGE to get and set the value respectively.
    BITS 0-7   :- It contains scoring percentage of 2G
    BITS 8-15  :- It contains scoring percentage of 5G
    BITS 16-23 :- reserved
    BITS 24-31 :- reserved

    The value of each index must be 0-100
 */
typedef struct {
    A_UINT32 score_pcnt;
} wmi_roam_cnd_band_scoring;

/**
    Use macro WMI_ROAM_CND_GET/SET_NSS_SCORE_PERCENTAGE to get and set the value respectively.
        BITS 0-7   :- It contains scoring percentage of 1x1
        BITS 8-15  :- It contains scoring percentage of 2x2
        BITS 16-23 :- It contains scoring percentage of 3x3
        BITS 24-31 :- It contains scoring percentage of 4x4

    The value of each index must be 0-100
 */
typedef struct {
    A_UINT32 score_pcnt;
} wmi_roam_cnd_nss_scoring;

/**
    score_pcnt: Contains roam score percentage of each slot of respective channel_congestion_pcnt.It is also used same BITS for slot(0-3)
                BITS 0-7   :- the scoring pcnt when AP is not advertise QBSS/ESP info
                BITS 8-15 :-  SLOT_1
                BITS 16-23 :- SLOT_2
                BITS 24-31 :- SLOT_3
                BITS 32-      ...

    num_slot will equally divide 100. e.g, if num_slot = 4
    slot 0 = 0-25%, slot 1 = 26-50% slot 2 = 51-75%, slot 3 = 76-100%
*/
typedef struct {
    A_UINT32 num_slot; /* max 15 */
    A_UINT32 score_pcnt3_to_0;
    A_UINT32 score_pcnt7_to_4;
    A_UINT32 score_pcnt11_to_8;
    A_UINT32 score_pcnt15_to_12;
} wmi_roam_cnd_esp_qbss_scoring;

/**
    score_pcnt: Contains roam score percentage of each slot of respective channel_congestion_pcnt.It is also used same BITS for slot(0-3)
                BITS 0-7   :- the scoring pcnt when AP is not advertise QBSS/ESP info
                BITS 8-15 :-  SLOT_1
                BITS 16-23 :- SLOT_2
                BITS 24-31 :- SLOT_3
                BITS 32-      ...

    num_slot will equally divide 100. e.g, if num_slot = 4
    slot 0 = 0-25%, slot 1 = 26-50% slot 2 = 51-75%, slot 3 = 76-100%
*/
typedef struct {
    A_UINT32 num_slot; /* max 15  */
    A_UINT32 score_pcnt3_to_0;
    A_UINT32 score_pcnt7_to_4;
    A_UINT32 score_pcnt11_to_8;
    A_UINT32 score_pcnt15_to_12;
} wmi_roam_cnd_oce_wan_scoring;

/**
    disable_bitmap :- Each bit will be either allow(0)/disallow(1) to considered the roam score param.
    rssi_weightage_pcnt :- RSSI weightage out of total score in percentage
    ht_weightage_pcnt :- HT weightage out of total score in percentage.
    vht_weightage_pcnt :- VHT weightage out of total score in percentage.
    he_weightage_pcnt :- 11ax weightage out of total score in percentage.
    bw_weightage_pcnt :- Bandwidth weightage out of total score in percentage.
    band_weightage_pcnt :- Band(2G/5G) weightage out of total score in percentage.
    nss_weightage_pcnt:- NSS(1x1 / 2x2) weightage out of total score in percentage.
    esp_qbss_weightage_pcnt: ESP/QBSS weightage out of total score in percentage.
    beamforming_weightage_pcnt :- Beamforming weightage out of total score in percentage.
    pcl_weightage_pcnt :- PCL weightage out of total score in percentage.
    oce_wan_weightage_pcnt :- OCE WAN metrics weightage out of total score in percentage.
    rssi_scoring :- RSSI scoring information.
    bw_scoring :- channel BW scoring percentage information.
    band_scoring : - band scording percentage information.
    nss_scoring :- NSS scoring percentage information.
    esp_qbss_scoring :- ESP/QBSS scoring percentage information
    oce_wan_scoring : OCE WAN metrics percentage information
    roam_score_delta_pcnt :- consider scanned AP as roam eligible candidate only if scanned AP score is at least roam_score_delta % better than connected AP score
    roam_score_delta_mask :- roam trigger bitmap for which roam_score_delta needs to apply. The WMI_ROAM_TRIGGER_REASON_ID enum values identify which bit within the mask is used for which roam trigger cause.
*/
typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_cnd_scoring_param */
    A_UINT32 disable_bitmap;
    A_INT32 rssi_weightage_pcnt;
    A_INT32 ht_weightage_pcnt;
    A_INT32 vht_weightage_pcnt;
    A_INT32 he_weightage_pcnt;
    A_INT32 bw_weightage_pcnt;
    A_INT32 band_weightage_pcnt;
    A_INT32 nss_weightage_pcnt;
    A_INT32 esp_qbss_weightage_pcnt;
    A_INT32 beamforming_weightage_pcnt;
    A_INT32 pcl_weightage_pcnt;
    A_INT32 oce_wan_weightage_pcnt;
    wmi_roam_cnd_rssi_scoring rssi_scoring;
    wmi_roam_cnd_bw_scoring bw_scoring;
    wmi_roam_cnd_band_scoring band_scoring;
    wmi_roam_cnd_nss_scoring nss_scoring;
    wmi_roam_cnd_esp_qbss_scoring esp_qbss_scoring;
    wmi_roam_cnd_oce_wan_scoring oce_wan_scoring;
    A_UINT32 roam_score_delta_pcnt;
    A_UINT32 roam_score_delta_mask;
} wmi_roam_cnd_scoring_param;

/** To match an open AP, the rs_authmode should be set to WMI_AUTH_NONE
 *  and WMI_AP_PROFILE_FLAG_CRYPTO should be clear.
 *  To match a WEP enabled AP, the rs_authmode should be set to WMI_AUTH_NONE
 *  and WMI_AP_PROFILE_FLAG_CRYPTO should be set .
 */

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ap_profile */
    /** flags as defined above */
    A_UINT32  flags;
	/**
     * rssi threshold value in dB: the value of the candidate AP should
	 * higher by this threshold than the rssi of the currrently associated AP.
	 */
	A_UINT32 rssi_threshold;
	/**
	 * ssid vlaue to be matched.
	 */
    wmi_ssid ssid;

    /**
	 * security params to be matched.
	 */
    /** authentication mode (defined above)  */
    A_UINT32               rsn_authmode;
    /** unicast cipher set */
    A_UINT32               rsn_ucastcipherset;
    /** mcast/group cipher set */
    A_UINT32               rsn_mcastcipherset;
    /** mcast/group management frames cipher set */
    A_UINT32               rsn_mcastmgmtcipherset;
    /**
    * rssi_abs_threshold value: the value of the candidate AP should
    * higher than this absolute RSSI threshold.
    * Zero means no absolute minimum RSSI is required.
    * units are the offset from the noise floor in dB.
    */
    A_UINT32 rssi_abs_thresh;
    /**
     * bg_rssi_threshold value in dB: For background scan the value of
     * the candidate AP should be higher by this threshold than the rssi
     * of the currrently associated AP.
     */
    A_UINT32 bg_rssi_threshold;
} wmi_ap_profile;

/** Support early stop roaming scanning when finding a strong candidate AP
 * A 'strong' candidate is
 * 1) Is eligible candidate
 *    (all conditions are met in existing candidate selection).
 * 2) Its rssi is better than earlystop threshold.
 *    Earlystop threshold will be relaxed as each channel is scanned.
 */
typedef struct {
    A_UINT32 tlv_header;
    /* Minimum RSSI threshold value for early stop, unit is dB above NF. */
    A_UINT32 roam_earlystop_thres_min;
    /* Maminum RSSI threshold value for early stop, unit is dB above NF. */
    A_UINT32 roam_earlystop_thres_max;
} wmi_roam_earlystop_rssi_thres_param;

typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_dense_thres_param */
    A_UINT32 tlv_header;
    /** rssi threshold offset under trffic and dense env */
    A_UINT32 roam_dense_rssi_thres_offset;
    /** minimum number of APs to determine dense env */
    A_UINT32 roam_dense_min_aps;
    /** initial dense status detected by host at the time of initial connection */
    A_UINT32 roam_dense_status;
    /* traffic threshold to enable aggressive roaming in dense env; units are percent of medium occupancy, 0 - 100 */
    A_UINT32 roam_dense_traffic_thres;
} wmi_roam_dense_thres_param;

/* Definition for flags in wmi_roam_bg_scan_roaming_param
 * Bit 0: BG roaming enabled when we connect to 2G AP only and roaming to 5G AP only.
 * Bit 1-31: Reserved
 */
#define WMI_ROAM_BG_SCAN_FLAGS_2G_TO_5G_ONLY   1

typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_bg_scan_roaming_param */
    A_UINT32 tlv_header;
    /** rssi threshold in dBm below which roaming will be triggered during background scan(non-roam scan). 0 will disable this threshold */
    A_UINT32 roam_bg_scan_bad_rssi_thresh;
    /** bitmap for which scan client will enable/disable background roaming. bit position is mapped to the enum WMI_SCAN_CLIENT_ID. 1 = enable, 0 = disable */
    A_UINT32 roam_bg_scan_client_bitmap;
    /** roam scan rssi threshold for 2G band.
     *  offset from roam_bg_scan_bad_rssi_thresh, in dB units
     */
    A_INT32 bad_rssi_thresh_offset_2g;
    /* flags for background roaming */
    A_UINT32 flags;
} wmi_roam_bg_scan_roaming_param;

/** Beacon filter wmi command info */

#define BCN_FLT_MAX_SUPPORTED_IES    256
#define BCN_FLT_MAX_ELEMS_IE_LIST    BCN_FLT_MAX_SUPPORTED_IES/32

typedef struct bss_bcn_stats {
    A_UINT32    vdev_id;
    A_UINT32    bss_bcnsdropped;
    A_UINT32    bss_bcnsdelivered;
}wmi_bss_bcn_stats_t;

typedef struct bcn_filter_stats {
    A_UINT32    bcns_dropped;
    A_UINT32    bcns_delivered;
    A_UINT32    activefilters;
    wmi_bss_bcn_stats_t bss_stats;
}wmi_bcnfilter_stats_t;

typedef struct wmi_add_bcn_filter_cmd {
    A_UINT32    tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_add_bcn_filter_cmd_fixed_param */
    A_UINT32    vdev_id;
    /*
     * Following this structure is the TLV:
     *    A_UINT32   ie_map[BCN_FLT_MAX_ELEMS_IE_LIST];
    */
} wmi_add_bcn_filter_cmd_fixed_param;

typedef struct wmi_rmv_bcn_filter_cmd {
    A_UINT32    tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_rmv_bcn_filter_cmd_fixed_param */
    A_UINT32    vdev_id;
}wmi_rmv_bcn_filter_cmd_fixed_param;

#define WMI_BCN_SEND_DTIM_ZERO         1
#define WMI_BCN_SEND_DTIM_BITCTL_SET   2
typedef struct wmi_bcn_send_from_host {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_bcn_send_from_host_cmd_fixed_param  */
    A_UINT32 vdev_id;
    A_UINT32 data_len;
    union {
        A_UINT32 frag_ptr; /* Physical address of the frame */
        A_UINT32 frag_ptr_lo; /* LSBs of physical address of the frame */
    };
    A_UINT32 frame_ctrl; /* farme ctrl to setup PPDU desc */
    A_UINT32 dtim_flag;   /* to control CABQ traffic */
    A_UINT32 bcn_antenna;   /* Antenna for beacon transmission */
    A_UINT32 frag_ptr_hi; /* MSBs of physical address of the frame */
}wmi_bcn_send_from_host_cmd_fixed_param;

/* cmd to support bcn snd for all vaps at once */
typedef struct wmi_pdev_send_bcn {
    A_UINT32                       num_vdevs;
    wmi_bcn_send_from_host_cmd_fixed_param   bcn_cmd[1];
} wmi_pdev_send_bcn_cmd_t;

typedef struct wmi_fd_send_from_host {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_fd_send_from_host_cmd_fixed_param  */
    A_UINT32 vdev_id;
    A_UINT32 data_len;
    union {
        A_UINT32 frag_ptr; /* Physical address of the frame */
        A_UINT32 frag_ptr_lo; /* LSBs of physical address of the frame */
    };
    A_UINT32 frag_ptr_hi; /* MSBs of physical address of the frame */
    A_UINT32 frame_ctrl; /* frame ctrl to setup PPDU desc */
} wmi_fd_send_from_host_cmd_fixed_param;

/*
 * WMI command structure for FILS feature enable/disable
 */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_enable_fils_cmd_fixed_param  */
    /* VDEV identifier */
    A_UINT32 vdev_id;
    A_UINT32 fd_period; /* non-zero - enable Fils Discovery frames with this period (in TU), 0 - disable FD frames */
} wmi_enable_fils_cmd_fixed_param;

 /*
  * WMI_ROAM_AP_PROFILE:  AP profile of connected AP for roaming.
  */
typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_ap_profile_fixed_param */
    /** id of AP criteria */
	A_UINT32 id;

    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;

/*
 * Following this structure is the TLV:
 * wmi_ap_profile ap_profile; //AP profile info
 * wmi_roam_cnd_scoring_param roam_cnd_scoring_param
 */
} wmi_roam_ap_profile_fixed_param;

/**
 * WMI_OFL_SCAN_ADD_AP_PROFILE: add an AP profile.
 */
typedef struct {
    /** id of AP criteria */
	A_UINT32 id;

    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;

    /** AP profile info */
    wmi_ap_profile ap_profile;

} wmi_ofl_scan_add_ap_profile;

/**
 * WMI_OFL_SCAN_REMOVE_AP_CRITERIA: remove an ap profile.
 */
typedef struct {
    /** id of AP criteria */
	A_UINT32 id;
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
} wmi_ofl_scan_remove_ap_profile;

/**
 * WMI_OFL_SCAN_PERIOD: period in msec for offload scan.
 *  0 will disable ofload scan and a very low value will perform a continous
 *  scan.
 */
typedef struct {
	/** offload scan period value, used for scans used when not connected */
	A_UINT32 ofl_scan_period;
} wmi_ofl_scan_period;

/* Do not modify XXX_BYTES or XXX_LEN below as it is fixed by standard */
#define ROAM_OFFLOAD_PMK_BYTES       (32)
#define ROAM_OFFLOAD_PSK_MSK_BYTES   (32)
#define ROAM_OFFLOAD_KRK_BYTES       (16)
#define ROAM_OFFLOAD_BTK_BYTES       (32)
#define ROAM_OFFLOAD_R0KH_ID_MAX_LEN (48)
#define ROAM_OFFLOAD_NUM_MCS_SET     (16)

/* This TLV will be filled only in case roam offload
 * for wpa2-psk/okc/ese/11r is enabled */
typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_offload_fixed_param */
    A_UINT32 rssi_cat_gap;   /* gap for every category bucket */
    A_UINT32 prefer_5g;      /* prefer select 5G candidate */
    A_UINT32 select_5g_margin;
    A_UINT32 reassoc_failure_timeout;     /* reassoc failure timeout */
    A_UINT32 capability;
    A_UINT32 ht_caps_info;
    A_UINT32 ampdu_param;
    A_UINT32 ht_ext_cap;
    A_UINT32 ht_txbf;
    A_UINT32 asel_cap;
    A_UINT32 qos_enabled;
    A_UINT32 qos_caps;
    A_UINT32 wmm_caps;
    A_UINT32 mcsset[ROAM_OFFLOAD_NUM_MCS_SET>>2]; /* since this 4 byte aligned,
                                                   * we don't declare it as
                                                   * tlv array */
    A_UINT32 handoff_delay_for_rx; /* In msec. Delay Hand-Off by this duration
                                    * to receive pending Rx frames from
                                    * current BSS */
    A_UINT32 max_mlme_sw_retries; /* maximum number of software retries for preauth and reassoc req */
    A_UINT32 no_ack_timeout; /* In msec. duration to wait before another SW retry made if no ack seen for previous frame */
    A_UINT32 roam_candidate_validity_time; /* In msec. validity duration of each entry in roam cache.  If the value is 0x0, this field should be disregarded. */
} wmi_roam_offload_tlv_param;

/* flags for 11i offload */
#define WMI_ROAM_OFFLOAD_FLAG_OKC_ENABLED       0   /* okc is enabled */
/* pmk caching is disabled */
#define WMI_ROAM_OFFLOAD_FLAG_PMK_CACHE_DISABLED 1
/* from bit 2 to bit 31 are reserved */

#define WMI_SET_ROAM_OFFLOAD_OKC_ENABLED(flag) do { \
        (flag) |=  (1 << WMI_ROAM_OFFLOAD_FLAG_OKC_ENABLED);      \
     } while(0)

#define WMI_SET_ROAM_OFFLOAD_OKC_DISABLED(flag) do { \
        (flag) &=  ~(1 << WMI_ROAM_OFFLOAD_FLAG_OKC_ENABLED);      \
     } while(0)

#define WMI_GET_ROAM_OFFLOAD_OKC_ENABLED(flag)   \
        ((flag) & (1 << WMI_ROAM_OFFLOAD_FLAG_OKC_ENABLED))

#define WMI_SET_ROAM_OFFLOAD_PMK_CACHE_ENABLED(flag) \
    do { \
        (flag) &=  ~(1 << WMI_ROAM_OFFLOAD_FLAG_PMK_CACHE_DISABLED); \
    } while (0)

#define WMI_SET_ROAM_OFFLOAD_PMK_CACHE_DISABLED(flag) \
    do { \
        (flag) |=  (1 << WMI_ROAM_OFFLOAD_FLAG_PMK_CACHE_DISABLED); \
    } while (0)

#define WMI_GET_ROAM_OFFLOAD_PMK_CACHE_DISABLED(flag) \
    ((flag) & (1 << WMI_ROAM_OFFLOAD_FLAG_PMK_CACHE_DISABLED))

/* This TLV will be  filled only in case of wpa-psk/wpa2-psk */
/* This TLV will be  filled only in case of wpa-psk/wpa2-psk */
typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_11i_offload_fixed_param */
    A_UINT32 flags;          /** flags. see WMI_ROAM_OFFLOAD_FLAG_ above */
    A_UINT32 pmk[ROAM_OFFLOAD_PMK_BYTES>>2]; /* pmk offload. As this 4 byte aligned, we don't declare it as tlv array */
    A_UINT32 pmk_len; /**the length of pmk. in normal case it should be 32, but for LEAP, is should be 16*/
} wmi_roam_11i_offload_tlv_param;

/* This TLV will be  filled only in case of 11R*/
typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_11r_offload_fixed_param */
    A_UINT32 mdie_present;
    A_UINT32 mdid;
    A_UINT32 r0kh_id[ROAM_OFFLOAD_R0KH_ID_MAX_LEN>>2];
    A_UINT32 r0kh_id_len;
    A_UINT32 psk_msk[ROAM_OFFLOAD_PSK_MSK_BYTES>>2]; /* psk/msk offload. As this 4 byte aligned, we don't declare it as tlv array */
    A_UINT32 psk_msk_len; /**length of psk_msk*/
    A_UINT32 psk_msk_ext_len; /**length of psk_msk_ext*/
    A_UINT32 psk_msk_ext[ROAM_OFFLOAD_PSK_MSK_BYTES>>2];
} wmi_roam_11r_offload_tlv_param;

/* This TLV will be filled only in case of ESE */
typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_ese_offload_fixed_param */
    A_UINT32 krk[ROAM_OFFLOAD_KRK_BYTES>>2]; /* KRK offload. As this 4 byte aligned, we don't declare it as tlv array */
    A_UINT32 btk[ROAM_OFFLOAD_BTK_BYTES>>2]; /* BTK offload. As this 4 byte aligned, we don't declare it as tlv array */
} wmi_roam_ese_offload_tlv_param;

typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_blacklist_with_timeout_tlv_param */
    A_UINT32 tlv_header;
    /** Blaclisted AP mac address */
    wmi_mac_addr bssid;
    /** How much time in milli seconds to keep AP in blacklist */
    A_UINT32 timeout;
} wmi_roam_blacklist_with_timeout_tlv_param;

/** WMI_ROAM_BLACKLIST_EVENT: generated whenever STA needs to move AP to blacklist for a particluar time
 *  Ex: AP which sends BTM request with disassoc imminent is set should be
 *  moved to blacklist until disassociation timer expires
 */
typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_blacklist_event_fixed_param */
    A_UINT32 vdev_id;
    /* This TLV is followed by further TLVs:
     *     wmi_roam_blacklist_with_timeout_tlv_param blacklist_with_timeout[]
     */
} wmi_roam_blacklist_event_fixed_param;

/** WMI_ROAM_EVENT: roam event triggering the host roam logic.
 * generated when ever a better AP is found in the recent roam scan (or)
 * when beacon miss is detected (or) when a DEAUTH/DISASSOC is received
 * from the current AP.
 */
typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_event_fixed_param  */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** reason for roam event */
    A_UINT32 reason;
    /** associated AP's rssi calculated by FW when reason code is WMI_ROAM_REASON_LOW_RSSI*/
    A_UINT32 rssi;
    /** roam notification */
    A_UINT32 notif;
    /** roam notification param
     *  Refer to WMI_ROAM_NOTIF_ defs to interpret the notif_params value.
     */
    A_UINT32 notif_params;
} wmi_roam_event_fixed_param;

/* roam_reason: bits 0-3 */
#define WMI_ROAM_REASON_INVALID   0x0 /** invalid reason. Do not interpret reason field */
#define WMI_ROAM_REASON_BETTER_AP 0x1 /** found a better AP */
#define WMI_ROAM_REASON_BMISS     0x2 /** beacon miss detected */
#define WMI_ROAM_REASON_DEAUTH    0x2 /** deauth/disassoc received */
#define WMI_ROAM_REASON_LOW_RSSI  0x3 /** connected AP's low rssi condition detected */
#define WMI_ROAM_REASON_SUITABLE_AP 0x4 /** found another AP that matches
                                          SSID and Security profile in
                                          WMI_ROAM_AP_PROFILE, found during scan
                                          triggered upon FINAL_BMISS **/
#define WMI_ROAM_REASON_HO_FAILED  0x5  /** LFR3.0 roaming failed, indicate the disconnection to host */
/** WMI_ROAM_REASON_INVOKE_ROAM_FAIL:
 * Result code of WMI_ROAM_INVOKE_CMDID.
 * Any roaming failure before reassociation will be indicated to host
 * with this reason.
 * Any roaming failure after reassociation will be indicated to host with
 * WMI_ROAM_REASON_HO_FAILED no matter WMI_ROAM_INVOKE_CMDID is called or not.
 */
#define WMI_ROAM_REASON_INVOKE_ROAM_FAIL 0x6
#define WMI_ROAM_REASON_RSO_STATUS       0x7
#define WMI_ROAM_REASON_BTM              0x8 /** Roaming because of BTM request received */
/* reserved up through 0xF */

/* subnet status: bits 4-5 */
typedef enum {
    WMI_ROAM_SUBNET_CHANGE_STATUS_UNKNOWN = 0,
    WMI_ROAM_SUBNET_CHANGE_STATUS_UNCHANGED,
    WMI_ROAM_SUBNET_CHANGE_STATUS_CHANGED,
} wmi_roam_subnet_change_status;

#define WMI_ROAM_SUBNET_CHANGE_STATUS_MASK      0x30
#define WMI_ROAM_SUBNET_CHANGE_STATUS_SHIFT     4

#define WMI_SET_ROAM_SUBNET_CHANGE_STATUS(roam_reason, status) \
    do { \
        (roam_reason) |= \
            (((status) << WMI_ROAM_SUBNET_CHANGE_STATUS_SHIFT) & \
             WMI_ROAM_SUBNET_CHANGE_STATUS_MASK); \
    } while (0)

#define WMI_GET_ROAM_SUBNET_CHANGE_STATUS(roam_reason) \
    (((roam_reason) & WMI_ROAM_SUBNET_CHANGE_STATUS_MASK) >> \
     WMI_ROAM_SUBNET_CHANGE_STATUS_SHIFT)

#define WMI_ROAM_REQUEST_HOST_HW_MODE_CHANGE_MASK     0x40
#define WMI_ROAM_REQUEST_HOST_HW_MODE_CHANGE_SHIFT    6

#define WMI_SET_ROAM_REQUEST_HOST_HW_MODE_CHANGE(roam_reason, status) \
    do { \
        (roam_reason) |= \
            (((status) << WMI_ROAM_REQUEST_HOST_HW_MODE_CHANGE_SHIFT) & \
             WMI_ROAM_REQUEST_HOST_HW_MODE_CHANGE_MASK); \
    } while (0)

#define WMI_GET_ROAM_REQUEST_HOST_HW_MODE_CHANGE(roam_reason) \
    (((roam_reason) & WMI_ROAM_REQUEST_HOST_HW_MODE_CHANGE_MASK) >> \
     WMI_ROAM_REQUEST_HOST_HW_MODE_CHANGE_SHIFT)

/* roaming notification */
#define WMI_ROAM_NOTIF_INVALID           0x0 /** invalid notification. Do not interpret notif field  */
#define WMI_ROAM_NOTIF_ROAM_START        0x1 /** indicate that roaming is started. sent only in non WOW state */
#define WMI_ROAM_NOTIF_ROAM_ABORT        0x2 /** indicate that roaming is aborted. sent only in non WOW state */
#define WMI_ROAM_NOTIF_ROAM_REASSOC      0x3 /** indicate that reassociation is done. sent only in non WOW state */
#define WMI_ROAM_NOTIF_SCAN_MODE_SUCCESS 0x4 /** indicate that roaming scan mode is successful */
#define WMI_ROAM_NOTIF_SCAN_MODE_FAIL    0x5 /** indicate that roaming scan mode is failed due to internal roaming state */
#define WMI_ROAM_NOTIF_DISCONNECT        0x6 /** indicate that roaming not allowed due BTM req */
#define WMI_ROAM_NOTIF_SUBNET_CHANGED    0x7 /** indicate that subnet has changed */
#define WMI_ROAM_NOTIF_SCAN_START        0x8 /** indicate roam scan start, notif_params to be sent as WMI_ROAM_TRIGGER_REASON_ID */

/**whenever RIC request information change, host driver should pass all ric related information to firmware (now only support tsepc)
* Once, 11r roaming happens, firmware can generate RIC request in reassoc request based on these informations
*/
typedef struct
{
    A_UINT32 tlv_header;      /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ric_request_fixed_param */
    A_UINT32 vdev_id;         /**unique id identifying the VDEV, generated by the caller*/
    A_UINT32 num_ric_request; /**number of ric request ie send to firmware.(max value is 2 now)*/
    A_UINT32 is_add_ric;      /**support add ric or delete ric*/
}wmi_ric_request_fixed_param;

/**tspec element: refer to 8.4.2.32 of 802.11 2012 spec
* these elements are used to construct tspec field in RIC request, which allow station to require specific TS when 11r roaming
*/
typedef struct{
    A_UINT32                         tlv_header;
    A_UINT32                         ts_info; /** bits value of TS Info field.*/
    A_UINT32                         nominal_msdu_size; /**Nominal MSDU Size field*/
    A_UINT32                         maximum_msdu_size; /**The Maximum MSDU Size field*/
    A_UINT32                         min_service_interval; /**The Minimum Service Interval field*/
    A_UINT32                         max_service_interval; /**The Maximum Service Interval field*/
    A_UINT32                         inactivity_interval; /**The Inactivity Interval field*/
    A_UINT32                         suspension_interval; /**The Suspension Interval field*/
    A_UINT32                         svc_start_time; /**The Service Start Time field*/
    A_UINT32                         min_data_rate; /**The Minimum Data Rate field*/
    A_UINT32                         mean_data_rate; /**The Mean Data Rate field*/
    A_UINT32                         peak_data_rate; /**The Peak Data Rate field*/
    A_UINT32                         max_burst_size; /**The Burst Size field*/
    A_UINT32                         delay_bound; /**The Delay Bound field*/
    A_UINT32                         min_phy_rate; /**The Minimum PHY Rate field*/
    A_UINT32                         surplus_bw_allowance; /**The Surplus Bandwidth Allowance field*/
    A_UINT32                         medium_time; /**The Medium Time field,in units of 32 us/s.*/
} wmi_ric_tspec;

/* flags for roam_invoke_cmd */
/* add this channel into roam cache channel list after this command is finished */
#define WMI_ROAM_INVOKE_FLAG_ADD_CH_TO_CACHE       0
/* indicate to host of failure if WMI_ROAM_INVOKE_CMDID. */
#define WMI_ROAM_INVOKE_FLAG_REPORT_FAILURE        1
/* during host-invoked roaming, don't send null data frame to AP */
#define WMI_ROAM_INVOKE_FLAG_NO_NULL_FRAME_TO_AP   2
/* from bit 3 to bit 31 are reserved */

#define WMI_SET_ROAM_INVOKE_ADD_CH_TO_CACHE(flag) do { \
        (flag) |=  (1 << WMI_SET_ROAM_INVOKE_ADD_CH_TO_CACHE);      \
     } while(0)

#define WMI_CLEAR_ROAM_INVOKE_ADD_CH_TO_CACHE(flag) do { \
        (flag) &=  ~(1 << WMI_SET_ROAM_INVOKE_ADD_CH_TO_CACHE);      \
     } while(0)

#define WMI_GET_ROAM_INVOKE_ADD_CH_TO_CACHE(flag)   \
        ((flag) & (1 << WMI_SET_ROAM_INVOKE_ADD_CH_TO_CACHE))


#define WMI_ROAM_INVOKE_SCAN_MODE_FIXED_CH      0   /* scan given channel only */
#define WMI_ROAM_INVOKE_SCAN_MODE_CACHE_LIST    1   /* scan cached channel list */
#define WMI_ROAM_INVOKE_SCAN_MODE_FULL_CH       2   /* scan full channel */
#define WMI_ROAM_INVOKE_SCAN_MODE_SKIP          3   /* no scan is performed. use beacon/probe resp given by the host */

#define WMI_ROAM_INVOKE_AP_SEL_FIXED_BSSID      0   /* roam to given BSSID only */
#define WMI_ROAM_INVOKE_AP_SEL_ANY_BSSID        1   /* roam to any BSSID */

/** WMI_ROAM_INVOKE_CMD: command to invoke roaming forcefully
 *
 * if <roam_scan_ch_mode> is zero and <channel_no> is not given, roaming is not executed.
 * if <roam_ap_sel_mode> is zero and <BSSID) is not given, roaming is not executed
 *
 * This command can be used to add specific channel into roam cached channel list by following
 * <roam_scan_ch_mode> = 0
 * <roam_ap_sel_mode> = 0
 * <roam_delay> = 0
 * <flag> |= WMI_ROAM_INVOKE_FLAG_ADD_CH_TO_CACHE
 * <BSSID> = do not fill (there will be no actual roaming because of ap_sel_mode is zero, but no BSSID is given)
 * <channel_no> = channel list to be added
 */
typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_invoke_fixed_param */
    A_UINT32 vdev_id; /** Unique id identifying the VDEV on which roaming is invoked */
    A_UINT32 flags;   /** flags. see WMI_ROAM_INVOKE_FLAG_ above */
    A_UINT32 roam_scan_mode; /** see WMI_ROAM_INVOKE_SCAN_ above */
    A_UINT32 roam_ap_sel_mode; /** see WMI_ROAM_INVOKE_AP_SEL_ above */
    A_UINT32 roam_delay; /** 0 = immediate roam, 1-2^32 = roam after this delay (msec) */
    A_UINT32 num_chan; /** # if channels to scan. In the TLV channel_list[] */
    A_UINT32 num_bssid;  /** number of bssids. In the TLV bssid_list[] */
    A_UINT32 num_buf; /** number of buffers In the TLV bcn_prb_buf_list[] */
    /**
     * TLV (tag length value ) parameters follows roam_invoke_req
     * The TLV's are:
     *     A_UINT32 channel_list[];
     *     wmi_mac_addr bssid_list[];
     *     wmi_tlv_buf_len_param bcn_prb_buf_list[];
     *     A_UINT8 bcn_prb_frm[];
     */
} wmi_roam_invoke_cmd_fixed_param;

/* Definition for op_bitmap */
enum {
    ROAM_FILTER_OP_BITMAP_BLACK_LIST =   0x1,
    ROAM_FILTER_OP_BITMAP_WHITE_LIST =   0x2,
    ROAM_FILTER_OP_BITMAP_PREFER_BSSID = 0x4,
    ROAM_FILTER_OP_BITMAP_LCA_DISALLOW = 0x8,
    ROAM_FILTER_OP_BITMAP_RSSI_REJECTION_OCE = 0x10,
};

/** lca_enable_source_bitmap */
#define WMI_ROAM_LCA_DISALLOW_SOURCE_PER         0x1
#define WMI_ROAM_LCA_DISALLOW_SOURCE_BMISS       0x2
#define WMI_ROAM_LCA_DISALLOW_SOURCE_LOW_RSSI    0x4
#define WMI_ROAM_LCA_DISALLOW_SOURCE_HIGH_RSSI   0x8
#define WMI_ROAM_LCA_DISALLOW_SOURCE_PERIODIC    0x10
#define WMI_ROAM_LCA_DISALLOW_SOURCE_MAWC        0x20 /* MAWC = Motion Aided Wifi connectivity */
#define WMI_ROAM_LCA_DISALLOW_SOURCE_DENSE       0x40
#define WMI_ROAM_LCA_DISALLOW_SOURCE_BACKGROUND  0x80
#define WMI_ROAM_LCA_DISALLOW_SOURCE_FORCED      0x100

typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_rejection_list_config_param */
    A_UINT32 tlv_header;
     /** BSSID of AP, who reject (re-)assoc due to low RSSI */
    wmi_mac_addr bssid;
    /** Disallowed AP for certain duration, in units of milliseconds */
    A_UINT32 remaining_disallow_duration;
   /** AP will be allowed for candidate, when AP RSSI better than expected RSSI units in dBm */
    A_INT32 requested_rssi;
} wmi_roam_rejection_list_config_param;
typedef wmi_roam_rejection_list_config_param wmi_roam_rssi_rejection_oce_config_param; /* retain old struct name as an alias for the new name */

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_filter_list_fixed_param */
    A_UINT32 vdev_id; /** Unique id identifying the VDEV on which roaming filter is adopted */
    A_UINT32 flags; /** flags for filter */
    A_UINT32 op_bitmap; /** 32 bit bitmap to be set on. bit0 = first param, bit 1 = second param...etc. Can be or'ed */
    A_UINT32 num_bssid_black_list; /* number of blacklist in the TLV variable bssid_black_list */
    A_UINT32 num_ssid_white_list; /* number of whitelist in the TLV variable ssid_white_list */
    A_UINT32 num_bssid_preferred_list; /* only for lfr 3.0. number of preferred list & factor in the TLV */
    A_UINT32 num_rssi_rejection_ap; /** number of list of AP who rejected STA due to low RSSI */
    /**
     * TLV (tag length value ) parameters follows roam_filter_list_cmd
     * The TLV's are:
     *     wmi_mac_addr bssid_black_list[];
     *     wmi_ssid ssid_white_list[];
     *     wmi_mac_addr bssid_preferred_list[];
     *     A_UINT32 bssid_preferred_factor[];
     *     wmi_roam_lca_disallow_config_tlv_param lca_disallow_param[0/1] (opt)
     *     wmi_roam_rejection_list_config_param rssi_rejection_list[]
     */
} wmi_roam_filter_fixed_param;

typedef struct {
    A_UINT32 tlv_header;        /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_lca_disallow_config_tlv_param */
    A_UINT32 disallow_duration; /** How long LCA AP will be disallowed before it can be a roaming candidate again, in units of seconds */
    A_UINT32 rssi_channel_penalization; /** How much RSSI will be penalized if candidate(s) are found in the same channel as disallowed AP's, in units of db */
    A_UINT32 num_disallowed_aps; /** How many APs the target should maintain in its LCA (Last Connected AP) list */
    A_UINT32 disallow_lca_enable_source_bitmap; /** disallow LCA logic is enabled only when trigger sources are matched with corresponding bit (see WMI_ROAM_LCA_DISALLOW_SOURCE constants) */
} wmi_roam_lca_disallow_config_tlv_param;

typedef struct {
    A_UINT8 address[4]; /* IPV4 address in Network Byte Order */
} WMI_IPV4_ADDR;

typedef struct _WMI_IPV6_ADDR {
    A_UINT8 address[16]; /* IPV6 in Network Byte Order */
} WMI_IPV6_ADDR;

/* flags for subnet change detection */
#define WMI_ROAM_SUBNET_CHANGE_FLAG_IP4_ENABLED 0
#define WMI_ROAM_SUBNET_CHANGE_FLAG_IP6_ENABLED 1
/* bit 2 to bit 31 are reserved */

/* set IPv4 enabled/disabled flag and get the flag */
#define WMI_SET_ROAM_SUBNET_CHANGE_FLAG_IP4_ENABLED(flag) do { \
    (flag) |= (1 << WMI_ROAM_SUBNET_CHANGE_FLAG_IP4_ENABLED); \
} while (0)

#define WMI_SET_ROAM_SUBNET_CHANGE_FLAG_IP4_DISABLED(flag) do { \
    (flag) &= ~(1 << WMI_ROAM_SUBNET_CHANGE_FLAG_IP4_ENABLED); \
} while (0)

#define WMI_GET_ROAM_SUBNET_CHANGE_FLAG_IP4_ENABLED(flag) \
    ((flag) & (1 << WMI_ROAM_SUBNET_CHANGE_FLAG_IP4_ENABLED))

/* set IPv6 enabled flag, disabled and get the flag */
#define WMI_SET_ROAM_SUBNET_CHANGE_FLAG_IP6_ENABLED(flag) do { \
    (flag) |= (1 << WMI_ROAM_SUBNET_CHANGE_FLAG_IP6_ENABLED); \
} while (0)

#define WMI_SET_ROAM_SUBNET_CHANGE_FLAG_IP6_DISABLED(flag) do { \
    (flag) &= ~(1 << WMI_ROAM_SUBNET_CHANGE_FLAG_IP6_ENABLED); \
} while (0)

#define WMI_GET_ROAM_SUBNET_CHANGE_FLAG_IP6_ENABLED(flag) \
    ((flag) & (1 << WMI_ROAM_SUBNET_CHANGE_FLAG_IP6_ENABLED))

/**
 * WMI_ROAM_SUBNET_CHANGE_CONFIG : Pass the gateway IP and MAC addresses
 *  to FW. FW uses these parameters for subnet change detection.
 */
typedef struct {
    A_UINT32      tlv_header; /** TLV tag and len; tag equals
WMITLV_TAG_STRUC_wmi_roam_subnet_change_config_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32      vdev_id;
    /** IPv4/IPv6 enabled/disabled */
    /** This flag sets the WMI_SET_ROAM_SUBNET_CHANGE_FLAG_xxx_ENABLED/
DISABLED */
    A_UINT32      flag;
    /** Gateway MAC address */
    wmi_mac_addr  inet_gw_mac_addr;
    /** IP addresses */
    WMI_IPV4_ADDR inet_gw_ip_v4_addr;
    WMI_IPV6_ADDR inet_gw_ip_v6_addr;
    /** Number of software retries for ARP/Neighbor solicitation request */
    A_UINT32      max_retries;
    /** timeout in milliseconds for each ARP request*/
    A_UINT32      timeout;
    /** number of skipped aps **/
    A_UINT32      num_skip_subnet_change_detection_bssid_list;
/**
 * TLV (tag length value ) parameters follows roam_subnet_change_config_cmd
 * structure. The TLV's are:
 *     wmi_mac_addr skip_subnet_change_detection_bssid_list [];
 **/
} wmi_roam_subnet_change_config_fixed_param;

/** WMI_PROFILE_MATCH_EVENT: offload scan
 * generated when ever atleast one of the matching profiles is found
 * in recent NLO scan. no data is carried with the event.
 */

/** P2P specific commands */

/**
 * WMI_P2P_DEV_SET_DEVICE_INFO : p2p device info, which will be used by
 * FW to generate P2P IE tobe carried in probe response frames.
 * FW will respond to probe requests while in listen state.
 */
typedef struct {
    /* number of secondary device types,supported */
    A_UINT32 num_secondary_dev_types;
    /**
     * followed by 8 bytes of primary device id and
     * num_secondary_dev_types * 8 bytes of secondary device
     * id.
     */
} wmi_p2p_dev_set_device_info;

/** WMI_P2P_DEV_SET_DISCOVERABILITY: enable/disable discoverability
 *  state. if enabled, an active STA/AP will respond to P2P probe requests on
 *  the operating channel of the VDEV.
 */

typedef struct {
    /* 1:enable disoverability, 0:disable discoverability */
    A_UINT32 enable_discoverability;
}  wmi_p2p_set_discoverability;

/** WMI_P2P_GO_SET_BEACON_IE: P2P IE to be added to
 *  beacons generated by FW. used in FW beacon mode.
 *  the FW will add this IE to beacon in addition to the beacon
 *  template set by WMI_BCN_TMPL_CMDID command.
 */
typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_p2p_go_set_beacon_ie_fixed_param  */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /* ie length */
    A_UINT32 ie_buf_len;
   /* Following this structure is the TLV byte stream of ie data of length ie_buf_len:
       *     A_UINT8 ie_data[];    // length in byte given by field num_data.
       */

}  wmi_p2p_go_set_beacon_ie_fixed_param;

/** WMI_P2P_GO_PROBE_RESP_IE: P2P IE to be added to
 *  probe response generated by FW. used in FW beacon mode.
 *  the FW will add this IE to probe response in addition to the probe response
 *  template set by WMI_PRB_TMPL_CMDID command.
 */
typedef struct {
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /* ie length */
    A_UINT32 ie_buf_len;
    /*followed by  byte stream of ie data of length ie_buf_len */
}  wmi_p2p_go_set_probe_resp_ie;

/** WMI_P2P_SET_VENDOR_IE_DATA_CMDID: Vendor specific P2P IE data, which will
 *  be used by the FW to parse the P2P NoA attribute in beacons, probe resposes
 *  and action frames received by the P2P Client.
 */
typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_p2p_set_vendor_ie_data_cmd_fixed_param */
    /** OS specific P2P IE OUI (3 bytes) + OUI type (1 byte)  */
    A_UINT32 p2p_ie_oui_type;
    /** OS specific NoA Attribute ID */
    A_UINT32 p2p_noa_attribute;
}  wmi_p2p_set_vendor_ie_data_cmd_fixed_param;

/*----P2P disc offload definition  ----*/

typedef struct {
    A_UINT32        pattern_type;
    /**
     * TLV (tag length value )  paramerters follow the pattern structure.
     * TLV can contain bssid list, ssid list and
     * ie. the TLV tags are defined above;
     */
}wmi_p2p_disc_offload_pattern_cmd;

typedef struct {
    /* unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
	/* mgmt type of the ie*/
	A_UINT32 mgmt_type;
    /* ie length */
    A_UINT32 ie_buf_len;
    /*followed by  byte stream of ie data of length ie_buf_len */
}wmi_p2p_disc_offload_appie_cmd;

typedef struct {
    /* enable/disable p2p find offload*/
    A_UINT32      enable;
	/* unique id identifying the VDEV, generated by the caller */
	A_UINT32      vdev_id;
	/* p2p find type */
	A_UINT32      disc_type;
	/* p2p find perodic */
	A_UINT32      perodic;
	/* p2p find listen channel */
	A_UINT32      listen_channel;
	/* p2p find full channel number */
    A_UINT32      num_scan_chans;
	/**
     * TLV (tag length value )  paramerters follow the pattern structure.
     * TLV  contain channel list
     */
}wmi_p2p_disc_offload_config_cmd;

/*----P2P OppPS definition  ----*/
typedef struct {
	/* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_p2p_set_oppps_cmd_fixed_param  */
    A_UINT32 tlv_header;
    /* unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /* OppPS attributes */
	/** Bit  0: Indicate enable/disable of OppPS
	 *	Bits 7-1:	Ctwindow in TUs
	 *	Bits 31-8:  Reserved
	 */
	A_UINT32	oppps_attr;
} wmi_p2p_set_oppps_cmd_fixed_param;

#define	WMI_UNIFIED_OPPPS_ATTR_ENALBED		0x1
#define	WMI_UNIFIED_OPPPS_ATTR_ENALBED_S	0

#define WMI_UNIFIED_OPPPS_ATTR_IS_ENABLED(hdr)                   \
			WMI_F_MS((hdr)->oppps_attr, WMI_UNIFIED_OPPPS_ATTR_ENALBED)

#define WMI_UNIFIED_OPPPS_ATTR_ENABLED_SET(hdr)                 \
            WMI_F_RMW((hdr)->oppps_attr, 0x1,                \
            WMI_UNIFIED_OPPPS_ATTR_ENALBED);

#define	WMI_UNIFIED_OPPPS_ATTR_CTWIN		0xfe
#define	WMI_UNIFIED_OPPPS_ATTR_CTWIN_S		1

#define WMI_UNIFIED_OPPPS_ATTR_CTWIN_GET(hdr)                        \
            WMI_F_MS((hdr)->oppps_attr, WMI_UNIFIED_OPPPS_ATTR_CTWIN)

#define WMI_UNIFIED_OPPPS_ATTR_CTWIN_SET(hdr, v)                \
            WMI_F_RMW((hdr)->oppps_attr, (v) & 0x7f,            \
            WMI_UNIFIED_OPPPS_ATTR_CTWIN);

typedef enum p2p_lo_start_ctrl_flags_e {
    /* flush prob. req when host is awake */
    P2P_LO_START_CTRL_FLAG_FLUSH_LISTEN_RESULT = 1 << 0,
} p2p_lo_start_ctrl_flags;

#define P2P_LO_PER_DEV_TYPE_LEN     8
#define P2P_LO_DEV_TYPES_COUNT_MAX  10
#define P2P_LO_DEV_TYPES_LEN_MAX    (P2P_LO_PER_DEV_TYPE_LEN * P2P_LO_DEV_TYPES_COUNT_MAX)
#define P2P_LO_PROB_RESP_MAX_LEN    512

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 ctl_flags; /* refer to enum_p2p_lo_start_ctrl_flags_e */
    A_UINT32 channel;   /* MHz */
    A_UINT32 period;    /* ms */
    A_UINT32 interval;  /* ms, interval should be larger than period */
    A_UINT32 count;     /* 0 means forever */
    /*
     * device_types_len specifies the number of bytes in the
     * device_types_data[] byte-array TLV that follows this TLV.
     * The data in device_types_data[] is in 8-byte elements, so
     * device_types_len will be a multiple of 8.
     * Refer to P2P_LO_DEV_TYPES_LEN_MAX
     */
    A_UINT32 device_types_len;
    /*
     * prob_resp_len specifies the number of bytes in the
     * prob_resp_data[] byte-array TLV that follows this TLV.
     * Refer to P2P_LO_PROB_RESP_MAX_LEN
     */
    A_UINT32 prob_resp_len;
    /*
     * Two other TLVs follow this TLV:
     * A_UINT8 device_types_data[device_types_len];
     * A_UINT8 prob_resp_data[prob_resp_len];
     *     The information in device_types_data[] and prob_resp_data[]
     *     needs to be provided to the target in over-the-air byte order.
     *     On a big-endian host, if device_types_data and prob_resp_data
     *     are initially in the correct over-the-air byte order,
     *     the automatic byteswap for endianness-correction during WMI
     *     message download will mess up the byte order.
     *     Thus, a big-endian host needs to explicitly byte-swap the bytes
     *     within each quad-byte segment of device_types_data[] and
     *     prob_resp_data[], so that the automatic byte-swap applied during
     *     WMI download from a big-endian host to the little-endian target
     *     will restore device_types_data and prob_resp_data into the
     *     correct byte ordering.
     */
} wmi_p2p_lo_start_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
} wmi_p2p_lo_stop_cmd_fixed_param;

typedef enum p2p_lo_stopped_reason_e {
    P2P_LO_STOPPED_REASON_COMPLETE = 0,   /* finished as scheduled */
    P2P_LO_STOPPED_REASON_RECV_STOP_CMD,  /* host stops it */
    P2P_LO_STOPPED_REASON_INVALID_LO_PAR, /* invalid listen offload par */
    P2P_LO_STOPPED_REASON_FW_NOT_SUPPORT, /* vdev cannot support it right now */
} p2p_lo_stopped_reason;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 reason;/* refer to p2p_lo_stopped_reason_e */
} wmi_p2p_lo_stopped_event_fixed_param;

typedef enum {
    WMI_MNT_FILTER_CONFIG_MANAGER,
    WMI_MNT_FILTER_CONFIG_CONTROL,
    WMI_MNT_FILTER_CONFIG_DATA,
    WMI_MNT_FILTER_CONFIG_ALL,
    WMI_MNT_FILTER_CONFIG_UNKNOWN,
} WMI_MNT_FILTER_CONFIG_TYPE;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 clear_or_set;
    A_UINT32 configure_type; /* see WMI_MNT_FILTER_CONFIG_TYPE */
} wmi_mnt_filter_cmd_fixed_param;

typedef struct {
    A_UINT32 time32;     //upper 32 bits of time stamp
    A_UINT32 time0;      //lower 32 bits of time stamp
} A_TIME64;

typedef enum wmi_peer_sta_kickout_reason {
    WMI_PEER_STA_KICKOUT_REASON_UNSPECIFIED = 0,        /* default value to preserve legacy behavior */
    WMI_PEER_STA_KICKOUT_REASON_XRETRY = 1,
    WMI_PEER_STA_KICKOUT_REASON_INACTIVITY = 2,
    WMI_PEER_STA_KICKOUT_REASON_IBSS_DISCONNECT = 3,
    WMI_PEER_STA_KICKOUT_REASON_TDLS_DISCONNECT = 4,    /* TDLS peer has disappeared. All tx is failing */
    WMI_PEER_STA_KICKOUT_REASON_SA_QUERY_TIMEOUT = 5,
} PEER_KICKOUT_REASON;

typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_sta_kickout_event_fixed_param  */
    /** peer mac address */
    wmi_mac_addr peer_macaddr;
    /** Reason code, defined as above */
    A_UINT32 reason;
    /** RSSI of the last bcn (averaged) in dB. 0 means Noise Floor value */
    A_UINT32 rssi;
} wmi_peer_sta_kickout_event_fixed_param;

#define WMI_WLAN_PROFILE_MAX_HIST     3
#define WMI_WLAN_PROFILE_MAX_BIN_CNT 32

typedef struct _wmi_wlan_profile_t {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_wlan_profile_t */
    A_UINT32 id;
    A_UINT32 cnt;
    A_UINT32 tot;
    A_UINT32 min;
    A_UINT32 max;
    A_UINT32 hist_intvl;
    A_UINT32 hist[WMI_WLAN_PROFILE_MAX_HIST];
} wmi_wlan_profile_t;

typedef struct _wmi_wlan_profile_ctx_t {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_wlan_profile_ctx_t */
    A_UINT32 tot; /* time in us */
    A_UINT32 tx_msdu_cnt;
    A_UINT32 tx_mpdu_cnt;
    A_UINT32 tx_ppdu_cnt;
    A_UINT32 rx_msdu_cnt;
    A_UINT32 rx_mpdu_cnt;
    A_UINT32 bin_count;
} wmi_wlan_profile_ctx_t;


typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_wlan_profile_trigger_cmd_fixed_param */
    A_UINT32 enable;
} wmi_wlan_profile_trigger_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_wlan_profile_get_prof_data_cmd_fixed_param */
    A_UINT32 value;
} wmi_wlan_profile_get_prof_data_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_wlan_profile_set_hist_intvl_cmd_fixed_param */
    A_UINT32 profile_id;
    A_UINT32 value;
} wmi_wlan_profile_set_hist_intvl_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_wlan_profile_enable_profile_id_cmd_fixed_param */
    A_UINT32 profile_id;
    A_UINT32 enable;
} wmi_wlan_profile_enable_profile_id_cmd_fixed_param;

/*Wifi header is upto 26, LLC is 8, with 14 byte duplicate in 802.3 header, that's 26+8-14=20.
146-128=18. So this means it is converted to non-QoS header. Riva FW take care of the QOS/non-QOS
when comparing wifi header.*/
/* NOTE: WOW_DEFAULT_BITMAP_PATTERN_SIZE(_DWORD) and WOW_DEFAULT_BITMASK_SIZE(_DWORD) can't be changed without breaking the compatibility */
#define WOW_DEFAULT_BITMAP_PATTERN_SIZE      146
#define WOW_DEFAULT_BITMAP_PATTERN_SIZE_DWORD 37 //Convert WOW_DEFAULT_EVT_BUF_SIZE into Int32 size
#define WOW_DEFAULT_BITMASK_SIZE             146
#define WOW_DEFAULT_BITMASK_SIZE_DWORD        37
#define WOW_MAX_BITMAP_FILTERS               32
#define WOW_DEFAULT_MAGIG_PATTERN_MATCH_CNT  16
#define WOW_EXTEND_PATTERN_MATCH_CNT         16
#define WOW_SHORT_PATTERN_MATCH_CNT           8
#define WOW_DEFAULT_EVT_BUF_SIZE             148  /* Maximum 148 bytes of the data is copied starting from header incase if the match is found.
                                                                                    The 148 comes from (128 - 14 )  payload size  + 8bytes LLC + 26bytes MAC header*/
#define WOW_DEFAULT_IOAC_PATTERN_SIZE  6
#define WOW_DEFAULT_IOAC_PATTERN_SIZE_DWORD 2
#define WOW_DEFAULT_IOAC_RANDOM_SIZE  6
#define WOW_DEFAULT_IOAC_RANDOM_SIZE_DWORD 2
#define WOW_DEFAULT_IOAC_KEEP_ALIVE_PKT_SIZE   120
#define WOW_DEFAULT_IOAC_KEEP_ALIVE_PKT_SIZE_DWORD 30
#define WOW_DEFAULT_IOAC_SOCKET_PATTERN_SIZE  32
#define WOW_DEFAULT_IOAC_SOCKET_PATTERN_SIZE_DWORD 8
#define WOW_DEFAULT_IOAC_KEEP_ALIVE_PKT_REV_SIZE       32
#define WOW_DEFAULT_IOAC_KEEP_ALIVE_PKT_REV_SIZE_DWORD 8
#define WOW_DEFAULT_IOAC_SOCKET_PATTERN_ACKNAK_SIZE  128
#define WOW_DEFAULT_IOAC_SOCKET_PATTERN_ACKNAK_SIZE_DWORD 32

typedef enum pattern_type_e {
    WOW_PATTERN_MIN = 0,
    WOW_BITMAP_PATTERN = WOW_PATTERN_MIN,
    WOW_IPV4_SYNC_PATTERN,
    WOW_IPV6_SYNC_PATTERN,
    WOW_WILD_CARD_PATTERN,
    WOW_TIMER_PATTERN,
    WOW_MAGIC_PATTERN,
    WOW_IPV6_RA_PATTERN,
    WOW_IOAC_PKT_PATTERN,
    WOW_IOAC_TMR_PATTERN,
    WOW_IOAC_SOCK_PATTERN,
    WOW_PATTERN_MAX
}WOW_PATTERN_TYPE;

typedef enum event_type_e {
    WOW_BMISS_EVENT = 0,
    WOW_BETTER_AP_EVENT,
    WOW_DEAUTH_RECVD_EVENT,
    WOW_MAGIC_PKT_RECVD_EVENT,
    WOW_GTK_ERR_EVENT,
    WOW_FOURWAY_HSHAKE_EVENT,
    WOW_EAPOL_RECVD_EVENT,
    WOW_NLO_DETECTED_EVENT,
    WOW_DISASSOC_RECVD_EVENT,
    WOW_PATTERN_MATCH_EVENT,
    WOW_CSA_IE_EVENT,
    WOW_PROBE_REQ_WPS_IE_EVENT,
    WOW_AUTH_REQ_EVENT,
    WOW_ASSOC_REQ_EVENT,
    WOW_HTT_EVENT,
    WOW_RA_MATCH_EVENT,
    WOW_HOST_AUTO_SHUTDOWN_EVENT,
    WOW_IOAC_MAGIC_EVENT,
    WOW_IOAC_SHORT_EVENT,
    WOW_IOAC_EXTEND_EVENT,
    WOW_IOAC_TIMER_EVENT,
    WOW_DFS_PHYERR_RADAR_EVENT,
    WOW_BEACON_EVENT,
    WOW_CLIENT_KICKOUT_EVENT,
    WOW_NAN_EVENT,
    WOW_EXTSCAN_EVENT,
    WOW_IOAC_REV_KA_FAIL_EVENT,
    WOW_IOAC_SOCK_EVENT,
    WOW_NLO_SCAN_COMPLETE_EVENT,
    WOW_NAN_DATA_EVENT,
    WOW_NAN_RTT_EVENT, /* DEPRECATED, UNUSED */
    WOW_OEM_RESPONSE_EVENT = WOW_NAN_RTT_EVENT, /* reuse deprecated event value */
    WOW_TDLS_CONN_TRACKER_EVENT,
    WOW_CRITICAL_LOG_EVENT,
    WOW_CHIP_POWER_FAILURE_DETECT_EVENT,
    WOW_11D_SCAN_EVENT,
    WOW_SAP_OBSS_DETECTION_EVENT,
    WOW_BSS_COLOR_COLLISION_DETECT_EVENT,
    WOW_TKIP_MIC_ERR_FRAME_RECVD_EVENT,
} WOW_WAKE_EVENT_TYPE;

typedef enum wake_reason_e {
    WOW_REASON_UNSPECIFIED =-1,
    WOW_REASON_NLOD = 0,
    WOW_REASON_AP_ASSOC_LOST,
    WOW_REASON_LOW_RSSI,
    WOW_REASON_DEAUTH_RECVD,
    WOW_REASON_DISASSOC_RECVD,
    WOW_REASON_GTK_HS_ERR,
    WOW_REASON_EAP_REQ,
    WOW_REASON_FOURWAY_HS_RECV,
    WOW_REASON_TIMER_INTR_RECV,
    WOW_REASON_PATTERN_MATCH_FOUND,
    WOW_REASON_RECV_MAGIC_PATTERN,
    WOW_REASON_P2P_DISC,
    WOW_REASON_WLAN_HB,
    WOW_REASON_CSA_EVENT,
    WOW_REASON_PROBE_REQ_WPS_IE_RECV,
    WOW_REASON_AUTH_REQ_RECV,
    WOW_REASON_ASSOC_REQ_RECV,
    WOW_REASON_HTT_EVENT,
    WOW_REASON_RA_MATCH,
    WOW_REASON_HOST_AUTO_SHUTDOWN,
    WOW_REASON_IOAC_MAGIC_EVENT,
    WOW_REASON_IOAC_SHORT_EVENT,
    WOW_REASON_IOAC_EXTEND_EVENT,
    WOW_REASON_IOAC_TIMER_EVENT,
    WOW_REASON_ROAM_HO,
    WOW_REASON_DFS_PHYERR_RADADR_EVENT,
    WOW_REASON_BEACON_RECV,
    WOW_REASON_CLIENT_KICKOUT_EVENT,
    WOW_REASON_NAN_EVENT,
    WOW_REASON_EXTSCAN,
    WOW_REASON_RSSI_BREACH_EVENT,
    WOW_REASON_IOAC_REV_KA_FAIL_EVENT,
    WOW_REASON_IOAC_SOCK_EVENT,
    WOW_REASON_NLO_SCAN_COMPLETE,
    WOW_REASON_PACKET_FILTER_MATCH,
    WOW_REASON_ASSOC_RES_RECV,
    WOW_REASON_REASSOC_REQ_RECV,
    WOW_REASON_REASSOC_RES_RECV,
    WOW_REASON_ACTION_FRAME_RECV,
    WOW_REASON_BPF_ALLOW,
    WOW_REASON_NAN_DATA,
    WOW_REASON_NAN_RTT, /* DEPRECATED, UNUSED */
    WOW_REASON_OEM_RESPONSE_EVENT = WOW_REASON_NAN_RTT, /* reuse deprecated reason value */
    WOW_REASON_TDLS_CONN_TRACKER_EVENT,
    WOW_REASON_CRITICAL_LOG,
    WOW_REASON_P2P_LISTEN_OFFLOAD,
    WOW_REASON_NAN_EVENT_WAKE_HOST,
    WOW_REASON_CHIP_POWER_FAILURE_DETECT,
    WOW_REASON_11D_SCAN,
    WOW_REASON_THERMAL_CHANGE,
    WOW_REASON_OIC_PING_OFFLOAD,
    WOW_REASON_WLAN_DHCP_RENEW,
    WOW_REASON_SAP_OBSS_DETECTION,
    WOW_REASON_BSS_COLOR_COLLISION_DETECT,
    WOW_REASON_TKIP_MIC_ERR_FRAME_RECVD_DETECT,
    WOW_REASON_WLAN_MD, /* motion detected */
    WOW_REASON_WLAN_BL, /* baselining done */
    WOW_REASON_NTH_BCN_OFLD, /* nth beacon forward to host */

    WOW_REASON_DEBUG_TEST = 0xFF,
} WOW_WAKE_REASON_TYPE;


typedef enum {
    WOW_IFACE_PAUSE_ENABLED,
    WOW_IFACE_PAUSE_DISABLED
} WOW_IFACE_STATUS;

enum {
    /* some win10 platfrom will not assert pcie_reset for wow.*/
    WMI_WOW_FLAG_IGNORE_PCIE_RESET          = 0x00000001,
    /* WMI_WOW_FLAG_SEND_PM_PME
     * Some platforms have issues if the PM_PME message is sent after WoW,
     * so don't send PM_PME after WoW unless the host uses this flag
     * to request it.
     */
    WMI_WOW_FLAG_SEND_PM_PME                = 0x00000002,
    /* Flag to indicate unit test */
    WMI_WOW_FLAG_UNIT_TEST_ENABLE           = 0x00000004,
    /* Force HTC wakeup */
    WMI_WOW_FLAG_DO_HTC_WAKEUP              = 0x00000008,
    /* Enable L1SS sleep for PCIE DRV case */
    WMI_WOW_FLAG_ENABLE_DRV_PCIE_L1SS_SLEEP = 0x00000010,
};

typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_wow_enable_cmd_fixed_param  */
    A_UINT32    enable;
    A_UINT32 pause_iface_config;
    A_UINT32 flags;  /* WMI_WOW_FLAG enums */
} wmi_wow_enable_cmd_fixed_param;

typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_wow_hostwakeup_from_sleep_cmd_fixed_param  */
    /** Reserved for future use */
    A_UINT32    reserved0;
} wmi_wow_hostwakeup_from_sleep_cmd_fixed_param;

#define WOW_ICMPV6_NA_FILTER_DISABLE 0
#define WOW_ICMPV6_NA_FILTER_ENABLE 1

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_wow_enable_icmpv6_na_flt_cmd_fixed_param  */
    A_UINT32 vdev_id;
    A_UINT32 enable; /* WOW_ICMPV6_NA_FILTER_ENABLE/DISABLE */
} wmi_wow_enable_icmpv6_na_flt_cmd_fixed_param;

typedef struct bitmap_pattern_s {
    A_UINT32     tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WOW_BITMAP_PATTERN_T */
    A_UINT32     patternbuf[WOW_DEFAULT_BITMAP_PATTERN_SIZE_DWORD];
    A_UINT32     bitmaskbuf[WOW_DEFAULT_BITMASK_SIZE_DWORD];
    A_UINT32     pattern_offset;
    A_UINT32     pattern_len;
    A_UINT32     bitmask_len;
    A_UINT32     pattern_id;  /* must be less than max_bitmap_filters */
}WOW_BITMAP_PATTERN_T;

typedef struct ipv4_sync_s {
    A_UINT32     tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WOW_IPV4_SYNC_PATTERN_T */
    A_UINT32     ipv4_src_addr;
    A_UINT32     ipv4_dst_addr;
    A_UINT32     tcp_src_prt;
    A_UINT32     tcp_dst_prt;
}WOW_IPV4_SYNC_PATTERN_T;

typedef struct ipv6_sync_s {
    A_UINT32     tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WOW_IPV6_SYNC_PATTERN_T */
    A_UINT32     ipv6_src_addr[4];
    A_UINT32     ipv6_dst_addr[4];
    A_UINT32     tcp_src_prt;
    A_UINT32     tcp_dst_prt;
}WOW_IPV6_SYNC_PATTERN_T;

typedef struct WOW_MAGIC_PATTERN_CMD
{
    A_UINT32     tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WOW_MAGIC_PATTERN_CMD */
	wmi_mac_addr macaddr;
}WOW_MAGIC_PATTERN_CMD;

typedef enum wow_ioac_pattern_type {
    WOW_IOAC_MAGIC_PATTERN = 1,
    WOW_IOAC_SHORT_PATTERN,
    WOW_IOAC_EXTEND_PATTERN,
} WOW_IOAC_PATTERN_TYPE;

typedef struct ioac_sock_pattern_s {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WOW_IOAC_SOCK_PATTERN_T */
    A_UINT32 id;
    A_UINT32 local_ipv4;
    A_UINT32 remote_ipv4;
    A_UINT32 local_port;
    A_UINT32 remote_port;
    A_UINT32 pattern_len; /* units = bytes */
    A_UINT32 pattern[WOW_DEFAULT_IOAC_SOCKET_PATTERN_SIZE_DWORD];
    WMI_IPV6_ADDR local_ipv6;
    WMI_IPV6_ADDR remote_ipv6;
    A_UINT32 ack_nak_len;
    A_UINT32 ackpkt[WOW_DEFAULT_IOAC_SOCKET_PATTERN_ACKNAK_SIZE_DWORD];
    A_UINT32 nakpkt[WOW_DEFAULT_IOAC_SOCKET_PATTERN_ACKNAK_SIZE_DWORD];
} WOW_IOAC_SOCK_PATTERN_T;

typedef struct ioac_pkt_pattern_s {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WOW_IOAC_PKT_PATTERN_T */
    A_UINT32 pattern_type;
    A_UINT32 pattern[WOW_DEFAULT_IOAC_PATTERN_SIZE_DWORD];
    A_UINT32 random[WOW_DEFAULT_IOAC_RANDOM_SIZE_DWORD];
    A_UINT32 pattern_len;
    A_UINT32 random_len;
} WOW_IOAC_PKT_PATTERN_T;

typedef struct ioac_tmr_pattern_s {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WOW_IOAC_TMR_PATTERN_T */
    A_UINT32 wake_in_s;
    A_UINT32 vdev_id;
} WOW_IOAC_TMR_PATTERN_T;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_WOW_IOAC_ADD_KEEPALIVE_CMD_fixed_param */
    A_UINT32 nID;
} WMI_WOW_IOAC_ADD_KEEPALIVE_CMD_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_WOW_IOAC_DEL_KEEPALIVE_CMD_fixed_param */
    A_UINT32 nID;
} WMI_WOW_IOAC_DEL_KEEPALIVE_CMD_fixed_param;

typedef struct ioac_keepalive_s {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_WOW_IOAC_KEEPALIVE_T */
    A_UINT32 keepalive_pkt_buf[WOW_DEFAULT_IOAC_KEEP_ALIVE_PKT_SIZE_DWORD];
    A_UINT32 keepalive_pkt_len;
    A_UINT32 period_in_ms;
    A_UINT32 vdev_id;
    A_UINT32 max_loss_cnt;
    A_UINT32 local_ipv4;
    A_UINT32 remote_ipv4;
    A_UINT32 local_port;
    A_UINT32 remote_port;
    A_UINT32 recv_period_in_ms;
    A_UINT32 rev_ka_size;
    A_UINT32 rev_ka_data[WOW_DEFAULT_IOAC_KEEP_ALIVE_PKT_REV_SIZE_DWORD];
    WMI_IPV6_ADDR local_ipv6;
    WMI_IPV6_ADDR remote_ipv6;
} WMI_WOW_IOAC_KEEPALIVE_T;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_WOW_IOAC_ADD_PATTERN_CMD_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 pattern_type;
/*
 * Following this struct are these TLVs. Note that they are all array of structures
 * but can have at most one element. Which TLV is empty or has one element depends
 * on the field pattern_type. This is to emulate an union.
 *     WOW_IOAC_PKT_PATTERN_T pattern_info_pkt[];
 *     WOW_IOAC_TMR_PATTERN_T pattern_info_tmr[];
 */
} WMI_WOW_IOAC_ADD_PATTERN_CMD_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_WOW_IOAC_DEL_PATTERN_CMD_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 pattern_type;
    A_UINT32 pattern_id;
} WMI_WOW_IOAC_DEL_PATTERN_CMD_fixed_param;

typedef struct {
    A_UINT32        tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_WOW_ADD_PATTERN_CMD_fixed_param */
    A_UINT32        vdev_id;
    A_UINT32        pattern_id;
    A_UINT32        pattern_type;
    /*
     * Following this struct are these TLVs. Note that they are all array of structures
     * but can have at most one element. Which TLV is empty or has one element depends
     * on the field pattern_type. This is to emulate an union.
     *     WOW_BITMAP_PATTERN_T       pattern_info_bitmap[];
     *     WOW_IPV4_SYNC_PATTERN_T    pattern_info_ipv4[];
     *     WOW_IPV6_SYNC_PATTERN_T    pattern_info_ipv6[];
     *     WOW_MAGIC_PATTERN_CMD      pattern_info_magic_pattern[];
     *     A_UINT32                   pattern_info_timeout[];
     *     A_UINT32                   ra_ratelimit_interval;
     */
}WMI_WOW_ADD_PATTERN_CMD_fixed_param;

typedef struct {
    A_UINT32        tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_WOW_DEL_PATTERN_CMD_fixed_param */
    A_UINT32        vdev_id;
    A_UINT32        pattern_id;
    A_UINT32        pattern_type;
}WMI_WOW_DEL_PATTERN_CMD_fixed_param;

#define WMI_WOW_MAX_EVENT_BM_LEN 4

typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_WOW_ADD_DEL_EVT_CMD_fixed_param  */
    A_UINT32    vdev_id;
    A_UINT32    is_add;
    union {
        A_UINT32 event_bitmap;
        A_UINT32 event_bitmaps[WMI_WOW_MAX_EVENT_BM_LEN];
    };
}WMI_WOW_ADD_DEL_EVT_CMD_fixed_param;

/*
 * This structure is used to set the pattern to check UDP packet in WOW mode.
 * If match, construct a tx frame in a local buffer to send through the peer
 * AP to the entity in the IP network that sent the UDP packet to this STA.
 */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_WOW_UDP_SVC_OFLD_CMD_fixed_param  */
    A_UINT32 vdev_id;
    A_UINT32 enable; /* 1: enable, 0: disable*/
    /* dest_port -
     * bits 7:0  contain the LSB of the UDP dest port,
     * bits 15:8 contain the MSB of the UDP dest port
     */
    A_UINT32 dest_port;
    A_UINT32 pattern_len; /* length in byte of pattern[] */
    A_UINT32 response_len; /* length in byte of response[] */
/* Following this struct are the TLV's:
 *  A_UINT8 pattern[];  // payload of UDP packet to be checked, network byte order
 *  A_UINT8 response[]; // payload of UDP packet to be response, network byte order
 */
} WMI_WOW_UDP_SVC_OFLD_CMD_fixed_param;

/*
 * This structure is used to set the pattern for WOW host wakeup pin pulse
 * pattern confirguration.
 */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_WOW_HOSTWAKEUP_PIN_PATTERN_CONFIG_CMD_fixed_param  */
    A_UINT32 enable; /* 1: enable, 0: disable */
    A_UINT32 pin; /* pin for host wakeup */
    A_UINT32 interval_low; /* interval for keeping low voltage, unit: ms */
    A_UINT32 interval_high; /* interval for keeping high voltage, unit: ms */
    A_UINT32 repeat_cnt; /* repeat times for pulse (0xffffffff means forever) */
    A_UINT32 init_state; /* Sense of the GPIO pin used for host wakeups.
                          * If init_state is 0, a low --> high transition
                          * causes a host wakeup interrupt.
                          * If init_state is 1, a high --> low transition
                          * causes a host wakeup interrrupt.
                          */
} WMI_WOW_HOSTWAKEUP_GPIO_PIN_PATTERN_CONFIG_CMD_fixed_param;

#define MAX_SUPPORTED_ACTION_CATEGORY           256
#define MAX_SUPPORTED_ACTION_SUBCATEGORY        32
#define MAX_SUPPORTED_ACTION_CATEGORY_ELE_LIST  (MAX_SUPPORTED_ACTION_CATEGORY/32)

typedef enum {
    WOW_ACTION_WAKEUP_OPERATION_RESET = 0,
    WOW_ACTION_WAKEUP_OPERATION_SET,
    WOW_ACTION_WAKEUP_OPERATION_ADD_SET,
    WOW_ACTION_WAKEUP_OPERATION_DELETE_SET,
} WOW_ACTION_WAKEUP_OPERATION;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_WOW_SET_ACTION_WAKE_UP_CMD_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 operation; /* 0 reset to fw default, 1 set the bits, 2 add the setting bits, 3 delete the setting bits */
    A_UINT32 action_category_map[MAX_SUPPORTED_ACTION_CATEGORY_ELE_LIST];
    /* This fixed_param TLV is followed by these additional TLV's
     * action_bitmaps_per_category -
     * Each element is a 32-bit bitmap indicating which subcategories
     * for that particular action category are considered for WoW wakeup
     * (if the subcategory's bit is 0) or ignored for WoW wakeup (if the
     * subcategory's bit is 1).
     *
     * A_UINT32 action_bitmaps_per_category[]; <-- variable length array
     */
} WMI_WOW_SET_ACTION_WAKE_UP_CMD_fixed_param;

typedef struct  wow_event_info_s {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WOW_EVENT_INFO_fixed_param  */
    A_UINT32    vdev_id;
    A_UINT32    flag;                              /*This is current reserved.*/
    A_INT32     wake_reason;
    A_UINT32    data_len;
}WOW_EVENT_INFO_fixed_param;

typedef struct wow_initial_wakeup_event_s {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WOW_INITIAL_WAKEUP_EVENT_fixed_param  */
    A_UINT32 vdev_id;
} WOW_INITIAL_WAKEUP_EVENT_fixed_param;

typedef enum {
    WOW_EVENT_INFO_TYPE_PACKET = 0x0001,
    WOW_EVENT_INFO_TYPE_BITMAP,
    WOW_EVENT_INFO_TYPE_GTKIGTK,
}WOW_EVENT_INFO_TYPE;

typedef struct  wow_event_info_section_s {
    A_UINT32    data_type;
    A_UINT32    data_len;
}WOW_EVENT_INFO_SECTION;

typedef struct  wow_event_info_section_packet_s {
    A_UINT8     packet[WOW_DEFAULT_EVT_BUF_SIZE];
}WOW_EVENT_INFO_SECTION_PACKET;

typedef struct  wow_event_info_section_bitmap_s {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WOW_EVENT_INFO_SECTION_BITMAP  */
    A_UINT32    flag;                              /*This is current reserved.*/
    A_UINT32    value;                         /*This could be the pattern id for bitmap pattern.*/
    A_UINT32    org_len;                      /*The length of the orginal packet.*/
}WOW_EVENT_INFO_SECTION_BITMAP;

/**
 * This command is sent from WLAN host driver to firmware to
 * enable or disable D0-WOW. D0-WOW means APSS suspend with
 * PCIe link and DDR being active.
 *
 *
 * Entering D0-WOW Mode (based on kernel suspend request):
 *    host->target: WMI_DO_WOW_ENABLE_DISABLE_CMDID (enable = 1)
 *    target: Take action (e.g. dbglog suspend)
 *    target->host: HTC_ACK (HTC_MSG_SEND_SUSPEND_COMPLETE message)
 *
 * Exiting D0-WOW mode (based on kernel resume OR target->host message received)
 *    host->target: WMI_DO_WOW_ENABLE_DISABLE_CMDID (enable = 0)
 *    target: Take action (e.g. dbglog resume)
 *    target->host: WMI_D0_WOW_DISABLE_ACK_EVENTID
 *
 * This command is applicable only on the PCIE LL systems
 * Host can enter either D0-WOW or WOW mode, but NOT both at same time
 * Decision to enter D0-WOW or WOW is based on active interfaces
 *
 */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_d0_wow_enable_disable_cmd_fixed_param  */
    A_UINT32 enable;     /* 1 = enable, 0 = disable */
} wmi_d0_wow_enable_disable_cmd_fixed_param;

typedef enum extend_wow_type_e {
    EXTWOW_TYPE_APP_TYPE1,   /* extend wow type: only enable wakeup for app type1 */
    EXTWOW_TYPE_APP_TYPE2,   /* extend wow type: only enable wakeup for app type2 */
    EXTWOW_TYPE_APP_TYPE1_2, /* extend wow type: enable wakeup for app type1&2 */
    EXTWOW_TYPE_APP_PULSETEST,
    EXTWOW_DISABLED = 255,
} EXTWOW_TYPE;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals wmi_extwow_enable_cmd_fixed_param  */
    A_UINT32 vdev_id;
    A_UINT32 type;
    A_UINT32 wakeup_pin_num;
    A_UINT32 swol_pulsetest_type;
    A_UINT32 swol_pulsetest_application;
} wmi_extwow_enable_cmd_fixed_param;

#define SWOL_INDOOR_MAC_ADDRESS_INDEX_MAX 8
#define SWOL_INDOOR_KEY_LEN 16

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals wmi_extwow_set_app_type1_params_cmd_fixed_param  */
    A_UINT32 vdev_id;
    wmi_mac_addr  wakee_mac;
    A_UINT8  ident[8];
    A_UINT8  passwd[16];
    A_UINT32 ident_len;
    A_UINT32 passwd_len;

    /* indoor check parameters */
    /* key for mac addresses specified in swol_indoor_key_mac
     * Big-endian hosts need to byte-swap the bytes within each 4-byte
     * segment of this array, so the bytes will return to their original
     * order when the entire WMI message contents are byte-swapped to
     * convert from big-endian to little-endian format.
     */
    A_UINT8 swol_indoor_key[SWOL_INDOOR_MAC_ADDRESS_INDEX_MAX][SWOL_INDOOR_KEY_LEN];
    /* key length for specified mac address index
     * Big-endian hosts need to byte-swap the bytes within each 4-byte
     * segment of this array, so the bytes will return to their original
     * order when the entire WMI message contents are byte-swapped to
     * convert from big-endian to little-endian format.
     */
    A_UINT8 swol_indoor_key_len[SWOL_INDOOR_MAC_ADDRESS_INDEX_MAX];
    /* mac address array allowed to wakeup host*/
    wmi_mac_addr swol_indoor_key_mac[SWOL_INDOOR_MAC_ADDRESS_INDEX_MAX];
    /* app mask for the mac addresses specified in swol_indoor_key_mac */
    A_UINT32 swol_indoor_app_mask[SWOL_INDOOR_MAC_ADDRESS_INDEX_MAX];
    A_UINT32 swol_indoor_waker_check; /* whether to do indoor waker check */
    A_UINT32 swol_indoor_pw_check;    /* whether to check password */
    A_UINT32 swol_indoor_pattern;     /* wakeup pattern */
    A_UINT32 swol_indoor_exception;   /* wakeup when exception happens */
    A_UINT32 swol_indoor_exception_app;
    A_UINT32 swol_assist_enable;      /* whether to enable IoT mode */
} wmi_extwow_set_app_type1_params_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;  /* TLV tag and len; tag equals wmi_extwow_set_app_type2_params_cmd_fixed_param  */
    A_UINT32 vdev_id;

    A_UINT8  rc4_key[16];
    A_UINT32 rc4_key_len;

    /** ip header parameter */
    A_UINT32 ip_id;         /* NC id */
    A_UINT32 ip_device_ip;  /* NC IP address */
    A_UINT32 ip_server_ip;  /* Push server IP address */

    /** tcp header parameter */
    A_UINT16 tcp_src_port;  /* NC TCP port */
    A_UINT16 tcp_dst_port;  /* Push server TCP port */
    A_UINT32 tcp_seq;
    A_UINT32 tcp_ack_seq;

    A_UINT32 keepalive_init;  /* Initial ping interval */
    A_UINT32 keepalive_min;   /* Minimum ping interval */
    A_UINT32 keepalive_max;   /* Maximum ping interval */
    A_UINT32 keepalive_inc;   /* Increment of ping interval */

    wmi_mac_addr gateway_mac;
    A_UINT32 tcp_tx_timeout_val;
    A_UINT32 tcp_rx_timeout_val;

    /** add extra parameter for backward-compatible */
    /*
     * For all byte arrays, natural order is used.  E.g.
     * rc4_write_sandbox[0] holds the 1st RC4 S-box byte,
     * rc4_write_sandbox[1] holds the 2nd RC4 S-box byte, etc.
     */

    /* used to encrypt transmit packet such as keep-alive */
    A_UINT8  rc4_write_sandbox[256];
    A_UINT32 rc4_write_x;
    A_UINT32 rc4_write_y;

    /* used to decrypt received packet such as wow data */
    A_UINT8  rc4_read_sandbox[256];
    A_UINT32 rc4_read_x;
    A_UINT32 rc4_read_y;

    /* used to caculate HMAC hash for transmit packet such as keep-alive */
    A_UINT8  ssl_write_seq[8];
    A_UINT8  ssl_sha1_write_key[64];
    A_UINT32 ssl_sha1_write_key_len;

    /* used to calculate HAMC hash for receive packet such as wow data */
    A_UINT8  ssl_read_seq[8];
    A_UINT8  ssl_sha1_read_key[64];
    A_UINT32 ssl_sha1_read_key_len;

    /* optional element for specifying TCP options data to include in
     * transmit packets such as keep-alive
     */
    A_UINT32 tcp_options_len;
    A_UINT8  tcp_options[40];

    A_UINT32 async_id; /* keep-alive request id */
} wmi_extwow_set_app_type2_params_cmd_fixed_param;



#define WMI_RXERR_CRC               0x01    /* CRC error on frame */
#define WMI_RXERR_DECRYPT           0x08    /* non-Michael decrypt error */
#define WMI_RXERR_MIC               0x10    /* Michael MIC decrypt error */
#define WMI_RXERR_KEY_CACHE_MISS    0x20    /* No/incorrect key matter in h/w */

typedef enum {
    PKT_PWR_SAVE_PAID_MATCH =           0x00000001,
    PKT_PWR_SAVE_GID_MATCH =            0x00000002,
    PKT_PWR_SAVE_EARLY_TIM_CLEAR =      0x00000004,
    PKT_PWR_SAVE_EARLY_DTIM_CLEAR =     0x00000008,
    PKT_PWR_SAVE_EOF_PAD_DELIM =        0x00000010,
    PKT_PWR_SAVE_MACADDR_MISMATCH =     0x00000020,
    PKT_PWR_SAVE_DELIM_CRC_FAIL =       0x00000040,
    PKT_PWR_SAVE_GID_NSTS_ZERO =        0x00000080,
    PKT_PWR_SAVE_RSSI_CHECK =           0x00000100,
    PKT_PWR_SAVE_5G_EBT =               0x00000200,
    PKT_PWR_SAVE_2G_EBT =               0x00000400,
    PKT_PWR_SAVE_BSS_COLOR_MISMATCH =   0x00000800,
    PKT_PWR_SAVE_UL_FLAG =              0x00001000,
    PKT_PWR_SAVE_STA_ID_MISMATCH =      0x00002000,
    PKT_PWR_SAVE_MACADDR_MISMATCH_FCS = 0x00004000,

    PKT_PWR_SAVE_ENABLE =               0x80000000,
} WMI_PKT_PWR_SAVE_TYPE;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ftm_intg_cmd_fixed_param */
    A_UINT32 num_data;       /** length in byte of data[]. */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
    /* This structure is used to send Factory Test Mode [FTM] command
     * from host to firmware for integrated chips which are binary blobs.
     * Following this structure is the TLV:
     *     A_UINT8 data[];    // length in byte given by field num_data.
     */
}wmi_ftm_intg_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ftm_intg_event_fixed_param */
    A_UINT32 num_data;       /** length in byte of data[]. */
    /* This structure is used to receive Factory Test Mode [FTM] event
     * from firmware to host for integrated chips which are binary blobs.
     * Following this structure is the TLV:
     *     A_UINT8 data[];    // length in byte given by field num_data.
     */
}wmi_ftm_intg_event_fixed_param;

#define WMI_MAX_NS_OFFLOADS           2
#define WMI_MAX_ARP_OFFLOADS          2

#define WMI_ARPOFF_FLAGS_VALID              (1 << 0)    /* the tuple entry is valid */
#define WMI_ARPOFF_FLAGS_MAC_VALID          (1 << 1)    /* the target mac address is valid */
#define WMI_ARPOFF_FLAGS_REMOTE_IP_VALID    (1 << 2)    /* remote IP field is valid */

typedef struct {
    A_UINT32          tlv_header;              /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_ARP_OFFLOAD_TUPLE */
    A_UINT32           flags;                  /* flags */
    A_UINT8           target_ipaddr[4];        /* IPV4 addresses of the local node*/
    A_UINT8           remote_ipaddr[4];        /* source address of the remote node requesting the ARP (qualifier) */
    wmi_mac_addr      target_mac;              /* mac address for this tuple, if not valid, the local MAC is used */
} WMI_ARP_OFFLOAD_TUPLE;

#define WMI_NSOFF_FLAGS_VALID           (1 << 0)    /* the tuple entry is valid */
#define WMI_NSOFF_FLAGS_MAC_VALID       (1 << 1)    /* the target mac address is valid */
#define WMI_NSOFF_FLAGS_REMOTE_IP_VALID (1 << 2)    /* remote IP field is valid */
#define WMI_NSOFF_FLAGS_IS_IPV6_ANYCAST (1 << 3)    /* whether the configured IPv6 address is anycast */

#define WMI_NSOFF_MAX_TARGET_IPS    2

typedef struct {
    A_UINT32          tlv_header;                /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_NS_OFFLOAD_TUPLE */
    A_UINT32           flags;                     /* flags */
    /* NOTE: This size of array target_ipaddr[] cannot be changed without breaking WMI compatibility. */
    WMI_IPV6_ADDR     target_ipaddr[WMI_NSOFF_MAX_TARGET_IPS]; /* IPV6 target addresses of the local node  */
    WMI_IPV6_ADDR     solicitation_ipaddr;       /* multi-cast source IP addresses for receiving solicitations */
    WMI_IPV6_ADDR     remote_ipaddr;             /* address of remote node requesting the solicitation (qualifier) */
    wmi_mac_addr      target_mac;                /* mac address for this tuple, if not valid, the local MAC is used */
} WMI_NS_OFFLOAD_TUPLE;

typedef struct {
    A_UINT32 tlv_header;      /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_SET_ARP_NS_OFFLOAD_CMD_fixed_param */
    A_UINT32 flags;
    A_UINT32 vdev_id;
    A_UINT32 num_ns_ext_tuples;
    /* Following this structure are the TLVs:
     *  WMI_NS_OFFLOAD_TUPLE    ns_tuples[WMI_MAX_NS_OFFLOADS];
     *  WMI_ARP_OFFLOAD_TUPLE   arp_tuples[WMI_MAX_ARP_OFFLOADS];
     *  WMI_NS_OFFLOAD_TUPLE  ns_ext_tuples[]; //size based on num_ns_ext_tuples
     */
} WMI_SET_ARP_NS_OFFLOAD_CMD_fixed_param;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 pattern_id;
    A_UINT32 timeout;
    A_UINT32 length;
    /* Following this would be the pattern
       A_UINT8 pattern[] of length specifed by length
       field in the structure. */
} WMI_ADD_PROACTIVE_ARP_RSP_PATTERN_CMD_fixed_param;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 pattern_id;
} WMI_DEL_PROACTIVE_ARP_RSP_PATTERN_CMD_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_tid_addba_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /** Tid number */
    A_UINT32 tid;
    /** Initiator (1) or Responder (0) for this aggregation */
    A_UINT32 initiator;
    /** size of the negotiated window */
    A_UINT32 window_size;
    /** starting sequence number (only valid for initiator) */
    A_UINT32 ssn;
    /** timeout field represents the time to wait for Block Ack in
    *   initiator case and the time to wait for BAR in responder
    *   case. 0 represents no timeout. */
    A_UINT32 timeout;
    /* BA policy: immediate ACK (0) or delayed ACK (1) */
    A_UINT32 policy;
} wmi_peer_tid_addba_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_tid_delba_cmd */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /** Tid number */
    A_UINT32 tid;
    /** Initiator (1) or Responder (0) for this aggregation */
    A_UINT32 initiator;
} wmi_peer_tid_delba_cmd_fixed_param;

typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tx_addba_complete_event_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /** Tid number */
    A_UINT32 tid;
    /** Event status */
    A_UINT32 status;
} wmi_tx_addba_complete_event_fixed_param;

typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tx_delba_complete_event_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /** Tid number */
    A_UINT32 tid;
    /** Event status */
    A_UINT32 status;
} wmi_tx_delba_complete_event_fixed_param;
/*
 * Structure to request sequence numbers for a given
 * peer station on different TIDs. The TIDs are
 * indicated in the tidBitMap, tid 0 would
 * be represented by LSB bit 0. tid 1 would be
 * represented by LSB bit 1 etc.
 * The target will retrieve the current sequence
 * numbers for the peer on all the TIDs requested
 * and send back a response in a WMI event.
 */
typedef struct
{
    A_UINT32    tlv_header;  /* TLV tag and len; tag equals
                                WMITLV_TAG_STRUC_wmi_ba_req_ssn_cmd_sub_struct_param */
    wmi_mac_addr peer_macaddr;
    A_UINT32 tidBitmap;
} wmi_ba_req_ssn;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals
     WMITLV_TAG_STRUC_wmi_ba_req_ssn_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** Number of requested SSN In the TLV wmi_ba_req_ssn[] */
    A_UINT32 num_ba_req_ssn;
/* Following this struc are the TLV's:
 *     wmi_ba_req_ssn ba_req_ssn_list; All peer and tidBitMap for which the ssn is requested
 */
} wmi_ba_req_ssn_cmd_fixed_param;

/*
 * Max transmit categories
 *
 * Note: In future if we need to increase WMI_MAX_TC definition
 * It would break the compatibility for WMI_BA_RSP_SSN_EVENTID.
 */
#define WMI_MAX_TC  8

/*
 * Structure to send response sequence numbers
 * for a give peer and tidmap.
 */
typedef struct
{
    A_UINT32    tlv_header;  /* TLV tag and len; tag equals
                                WMITLV_TAG_STRUC_wmi_ba_req_ssn_event_sub_struct_param */
    wmi_mac_addr peer_macaddr;
    /* A boolean to indicate if ssn is present */
    A_UINT32 ssn_present_for_tid[WMI_MAX_TC];
    /* The ssn from target, valid only if
     * ssn_present_for_tid[tidn] equals 1
     */
    A_UINT32 ssn_for_tid[WMI_MAX_TC];
} wmi_ba_event_ssn;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals
     WMITLV_TAG_STRUC_wmi_ba_rsp_ssn_event_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** Event status, success or failure of the overall operation */
    A_UINT32 status;
    /** Number of requested SSN In the TLV wmi_ba_req_ssn[] */
    A_UINT32 num_ba_event_ssn;
/* Following this struc are the TLV's:
 *     wmi_ba_event_ssn ba_event_ssn_list; All peer and tidBitMap for which the ssn is requested
 */
} wmi_ba_rsp_ssn_event_fixed_param;


enum wmi_aggr_state_req_type {
    WMI_DISABLE_AGGREGATION,
    WMI_ENABLE_AGGREGATION
};

/*
 * This event is generated by the COEX module
 * when esco call is begins the coex module in fw genrated this event to host to
 * disable the RX aggregation and after completion of the esco call fw will indicate to
 * enable back the Rx aggregation .
*/

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_aggr_state_trig_event_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
   /** req_type  contains values from enum
     *  wmi_aggr_state_req_type; 0 (disable) 1(enable) */
    A_UINT32  req_type;
}wmi_aggr_state_trig_event_fixed_param;

typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_install_key_complete_event_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** MAC address used for installing   */
    wmi_mac_addr peer_macaddr;
    /** key index */
    A_UINT32 key_ix;
    /** key flags */
    A_UINT32 key_flags;
    /** Event status */
    A_UINT32 status;
} wmi_vdev_install_key_complete_event_fixed_param;

typedef enum _WMI_NLO_AUTH_ALGORITHM {
    WMI_NLO_AUTH_ALGO_80211_OPEN        = 1,
    WMI_NLO_AUTH_ALGO_80211_SHARED_KEY  = 2,
    WMI_NLO_AUTH_ALGO_WPA               = 3,
    WMI_NLO_AUTH_ALGO_WPA_PSK           = 4,
    WMI_NLO_AUTH_ALGO_WPA_NONE          = 5,
    WMI_NLO_AUTH_ALGO_RSNA              = 6,
    WMI_NLO_AUTH_ALGO_RSNA_PSK          = 7,
} WMI_NLO_AUTH_ALGORITHM;

typedef enum _WMI_NLO_CIPHER_ALGORITHM {
    WMI_NLO_CIPHER_ALGO_NONE           = 0x00,
    WMI_NLO_CIPHER_ALGO_WEP40          = 0x01,
    WMI_NLO_CIPHER_ALGO_TKIP           = 0x02,
    WMI_NLO_CIPHER_ALGO_CCMP           = 0x04,
    WMI_NLO_CIPHER_ALGO_WEP104         = 0x05,
    WMI_NLO_CIPHER_ALGO_BIP            = 0x06,
    WMI_NLO_CIPHER_ALGO_WPA_USE_GROUP  = 0x100,
    WMI_NLO_CIPHER_ALGO_RSN_USE_GROUP  = 0x100,
    WMI_NLO_CIPHER_ALGO_WEP            = 0x101,
} WMI_NLO_CIPHER_ALGORITHM;

/* SSID broadcast  type passed in NLO params */
typedef enum _WMI_NLO_SSID_BcastNwType
{
  WMI_NLO_BCAST_UNKNOWN      = 0,
  WMI_NLO_BCAST_NORMAL       = 1,
  WMI_NLO_BCAST_HIDDEN       = 2,
} WMI_NLO_SSID_BcastNwType;

#define WMI_NLO_MAX_SSIDS    16
#define WMI_NLO_MAX_CHAN     48

#define WMI_NLO_CONFIG_STOP                             (0x1 << 0)
#define WMI_NLO_CONFIG_START                            (0x1 << 1)
#define WMI_NLO_CONFIG_RESET                            (0x1 << 2)
#define WMI_NLO_CONFIG_SLOW_SCAN                        (0x1 << 4)
#define WMI_NLO_CONFIG_FAST_SCAN                        (0x1 << 5)
#define WMI_NLO_CONFIG_SSID_HIDE_EN                     (0x1 << 6)
/* This bit is used to indicate if EPNO or supplicant PNO is enabled. Only one of them can be enabled at a given time */
#define WMI_NLO_CONFIG_ENLO                             (0x1 << 7)
#define WMI_NLO_CONFIG_SCAN_PASSIVE                     (0x1 << 8)
#define WMI_NLO_CONFIG_ENLO_RESET                       (0x1 << 9)
#define WMI_NLO_CONFIG_SPOOFED_MAC_IN_PROBE_REQ         (0x1 << 10)
#define WMI_NLO_CONFIG_RANDOM_SEQ_NO_IN_PROBE_REQ       (0x1 << 11)
#define WMI_NLO_CONFIG_ENABLE_IE_WHITELIST_IN_PROBE_REQ (0x1 << 12)
#define WMI_NLO_CONFIG_ENABLE_CNLO_RSSI_CONFIG          (0x1 << 13)

/* Whether directed scan needs to be performed (for hidden SSIDs) */
#define WMI_ENLO_FLAG_DIRECTED_SCAN      1
/* Whether PNO event shall be triggered if the network is found on A band */
#define WMI_ENLO_FLAG_A_BAND             2
/* Whether PNO event shall be triggered if the network is found on G band */
#define WMI_ENLO_FLAG_G_BAND             4
/* Whether strict matching is required (i.e. firmware shall not match on the entire SSID) */
#define WMI_ENLO_FLAG_STRICT_MATCH       8

/* Code for matching the beacon AUTH IE - additional codes TBD */
/* open */
#define WMI_ENLO_AUTH_CODE_OPEN  1
/* WPA_PSK or WPA2PSK */
#define WMI_ENLO_AUTH_CODE_PSK   2
/* any EAPOL */
#define WMI_ENLO_AUTH_CODE_EAPOL 4

/* NOTE: wmi_nlo_ssid_param structure can't be changed without breaking the compatibility */
typedef struct wmi_nlo_ssid_param
{
	A_UINT32 valid;
    wmi_ssid ssid;
} wmi_nlo_ssid_param;

/* NOTE: wmi_nlo_enc_param structure can't be changed without breaking the compatibility */
typedef struct wmi_nlo_enc_param
{
	A_UINT32 valid;
    A_UINT32 enc_type;
} wmi_nlo_enc_param;

/* NOTE: wmi_nlo_auth_param structure can't be changed without breaking the compatibility */
typedef struct wmi_nlo_auth_param
{
	A_UINT32 valid;
    A_UINT32 auth_type;
} wmi_nlo_auth_param;

/* NOTE: wmi_nlo_bcast_nw_param structure can't be changed without breaking the compatibility */
typedef struct wmi_nlo_bcast_nw_param
{
    A_UINT32 valid;
    /* If WMI_NLO_CONFIG_EPNO is not set. Supplicant PNO is enabled. The value should be true/false
       Otherwise EPNO is enabled. bcast_nw_type would be used as a bit flag contains WMI_ENLO_FLAG_XXX */
    A_UINT32 bcast_nw_type;
} wmi_nlo_bcast_nw_param;

/* NOTE: wmi_nlo_rssi_param structure can't be changed without breaking the compatibility */
typedef struct wmi_nlo_rssi_param
{
    A_UINT32 valid;
    A_INT32 rssi;
} wmi_nlo_rssi_param;

typedef struct nlo_configured_parameters {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_nlo_configured_parameters */
    wmi_nlo_ssid_param ssid;
    wmi_nlo_enc_param enc_type;
    wmi_nlo_auth_param auth_type;
    wmi_nlo_rssi_param rssi_cond;
    wmi_nlo_bcast_nw_param bcast_nw_type; /* indicates if the SSID is hidden or not */
} nlo_configured_parameters;

/* Support channel prediction for PNO scan after scanning top_k_num channels
 * if stationary_threshold is met.
 */
typedef struct nlo_channel_prediction_cfg {
    A_UINT32 tlv_header;
    /* Enable or disable this feature. */
    A_UINT32 enable;
    /* Top K channels will be scanned before deciding whether to further scan
     * or stop. Minimum value is 3 and maximum is 5. */
    A_UINT32 top_k_num;
    /* Preconfigured stationary threshold.
     * Lesser value means more conservative. Bigger value means more aggressive.
     * Maximum is 100 and mininum is 0. */
    A_UINT32 stationary_threshold;
    /* Periodic full channel scan in milliseconds unit.
     * After full_scan_period_ms since last full scan, channel prediction
     * scan is suppressed and will do full scan.
     * This is to help detecting sudden AP power-on or -off. Value 0 means no
     * full scan at all (not recommended).
     */
    A_UINT32 full_scan_period_ms;
} nlo_channel_prediction_cfg;

typedef struct enlo_candidate_score_params_t {
    A_UINT32 tlv_header;   /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_enlo_candidate_score_param */
    A_INT32 min5GHz_rssi;  /* minimum 5GHz RSSI for a BSSID to be considered (units = dBm) */
    A_INT32 min24GHz_rssi; /* minimum 2.4GHz RSSI for a BSSID to be considered (units = dBm) */
    A_UINT32 initial_score_max; /* the maximum score that a network can have before bonuses */
    /* current_connection_bonus:
     * only report when there is a network's score this much higher
     * than the current connection
     */
    A_UINT32 current_connection_bonus;
    A_UINT32 same_network_bonus; /* score bonus for all networks with the same network flag */
    A_UINT32 secure_bonus; /* score bonus for networks that are not open */
    A_UINT32 band5GHz_bonus; /* 5GHz RSSI score bonus (applied to all 5GHz networks) */
} enlo_candidate_score_params;

typedef struct connected_nlo_bss_band_rssi_pref_t {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_connected_nlo_bss_band_rssi_pref */
    /** band which needs to get preference over other band - see wmi_set_vdev_ie_band enum */
    A_UINT32 band;
    /* Amount of RSSI preference (in dB) that can be given to a band */
    A_INT32  rssi_pref;
} connected_nlo_bss_band_rssi_pref;

typedef struct connected_nlo_rssi_params_t {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_connected_nlo_rssi_params */
    /* Relative rssi threshold (in dB) by which new BSS should have better rssi than
     * the current connected BSS.
     */
    A_INT32  relative_rssi;
    /* The amount of rssi preference (in dB) that can be given to a 5G BSS over 2.4G BSS. */
    A_INT32  relative_rssi_5g_pref;
} connected_nlo_rssi_params;

typedef struct wmi_nlo_config {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_nlo_config_cmd_fixed_param */
    A_UINT32    flags;
    A_UINT32    vdev_id;
    A_UINT32    fast_scan_max_cycles;
    A_UINT32    active_dwell_time;
    A_UINT32    passive_dwell_time; /* PDT in msecs */
    A_UINT32    probe_bundle_size;
    A_UINT32    rest_time;  /* ART = IRT */
    A_UINT32    max_rest_time; /* Max value that can be reached after SBM */
    A_UINT32    scan_backoff_multiplier;  /* SBM */
    A_UINT32    fast_scan_period; /* SCBM */
    A_UINT32    slow_scan_period; /* specific to windows */
    A_UINT32    no_of_ssids;
    A_UINT32    num_of_channels;
    A_UINT32    delay_start_time; /* NLO scan start delay time in milliseconds */
    /** MAC Address to use in Probe Req as SA **/
    wmi_mac_addr mac_addr;
    /** Mask on which MAC has to be randomized **/
    wmi_mac_addr mac_mask;
    /** IE bitmap to use in Probe Req **/
    A_UINT32 ie_bitmap[WMI_IE_BITMAP_SIZE];
    /** Number of vendor OUIs. In the TLV vendor_oui[] **/
    A_UINT32 num_vendor_oui;
    /** Number of connected NLO band preferences **/
    A_UINT32 num_cnlo_band_pref;

    /* The TLVs will follow.
        * nlo_configured_parameters nlo_list[];
        * A_UINT32 channel_list[num_of_channels];
        * nlo_channel_prediction_cfg ch_prediction_cfg;
        * enlo_candidate_score_params candidate_score_params;
        * wmi_vendor_oui vendor_oui[num_vendor_oui];
        * connected_nlo_rssi_params cnlo_rssi_params;
        * connected_nlo_bss_band_rssi_pref cnlo_bss_band_rssi_pref[num_cnlo_band_pref];
        */

} wmi_nlo_config_cmd_fixed_param;

typedef struct wmi_nlo_event
{
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_nlo_event */
    A_UINT32    vdev_id;
}wmi_nlo_event;


/* WMI_PASSPOINT_CONFIG_SET
 * Sets a list for passpoint networks for PNO purposes;
 * it should be matched against any passpoint networks found
 * during regular PNO scan.
 */
#define WMI_PASSPOINT_CONFIG_SET       (0x1 << 0)
/* WMI_PASSPOINT_CONFIG_RESET
 * Reset passpoint network list -
 * no Passpoint networks should be matched after this.
 */
#define WMI_PASSPOINT_CONFIG_RESET     (0x1 << 1)

#define PASSPOINT_REALM_LEN                  256
#define PASSPOINT_ROAMING_CONSORTIUM_ID_LEN  5
#define PASSPOINT_ROAMING_CONSORTIUM_ID_NUM  16
#define PASSPOINT_PLMN_ID_LEN                3
#define PASSPOINT_PLMN_ID_ALLOC_LEN /* round up to A_UINT32 boundary */ \
    (((PASSPOINT_PLMN_ID_LEN + 3) >> 2) << 2)

/*
 * Confirm PASSPOINT_REALM_LEN is a multiple of 4, so the
 *     A_UINT8 realm[PASSPOINT_REALM_LEN]
 * array will end on a 4-byte boundary.
 * (This 4-byte alignment simplifies endianness-correction byte swapping.)
 */
A_COMPILE_TIME_ASSERT(
    check_passpoint_realm_size,
    (PASSPOINT_REALM_LEN % sizeof(A_UINT32)) == 0);

/*
 * Confirm the product of PASSPOINT_ROAMING_CONSORTIUM_ID_NUM and
 * PASSPOINT_ROAMING_CONSORTIUM_ID_LEN is a multiple of 4, so the
 * roaming_consortium_ids array below will end on a 4-byte boundary.
 * (This 4-byte alignment simplifies endianness-correction byte swapping.)
 */
A_COMPILE_TIME_ASSERT(
    check_passpoint_roaming_consortium_ids_size,
    ((PASSPOINT_ROAMING_CONSORTIUM_ID_NUM*PASSPOINT_ROAMING_CONSORTIUM_ID_LEN) % sizeof(A_UINT32)) == 0);

/* wildcard ID to allow an action (reset) to apply to all networks */
#define WMI_PASSPOINT_NETWORK_ID_WILDCARD 0xFFFFFFFF
typedef struct wmi_passpoint_config {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals wmi_passpoint_config_cmd_fixed_param */
    /* (network) id
     * identifier of the matched network, report this in event
     * This id can be a wildcard (WMI_PASSPOINT_NETWORK_ID_WILDCARD)
     * that indicates the action should be applied to all networks.
     * Currently, the only action that is applied to all networks is "reset".
     * If a non-wildcard ID is specified, that particular network is configured.
     * If a wildcard ID is specified, all networks are reset.
     */
    A_UINT32 id;
    A_UINT32 req_id;
    A_UINT8  realm[PASSPOINT_REALM_LEN]; /*null terminated UTF8 encoded realm, 0 if unspecified*/
    A_UINT8  roaming_consortium_ids[PASSPOINT_ROAMING_CONSORTIUM_ID_NUM][PASSPOINT_ROAMING_CONSORTIUM_ID_LEN]; /*roaming consortium ids to match, 0s if unspecified*/
                                                                                                              /*This would be bytes-stream as same as defition of realm id in 802.11 standard*/
    A_UINT8  plmn[PASSPOINT_PLMN_ID_ALLOC_LEN]; /*PLMN id mcc/mnc combination as per rules, 0s if unspecified */
} wmi_passpoint_config_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;    /* TLV tag and len; tag equals wmi_passpoint_event_hdr */
    A_UINT32 id;            /* identifier of the matched network */
    A_UINT32 vdev_id;
    A_UINT32 timestamp;     /* time since boot (in microsecond) when the result was retrieved*/
    wmi_ssid ssid;
    wmi_mac_addr    bssid;  /* bssid of the network */
    A_UINT32 channel_mhz;   /* channel frequency in MHz */
    A_UINT32 rssi;          /* rssi value */
    A_UINT32 rtt;           /* timestamp in nanoseconds*/
    A_UINT32 rtt_sd;        /* standard deviation in rtt */
    A_UINT32 beacon_period; /* beacon advertised in the beacon */
    A_UINT32 capability;    /* capabilities advertised in the beacon */
    A_UINT32 ie_length;     /* size of the ie_data blob */
    A_UINT32 anqp_length;   /* length of ANQP blob */
/* Following this structure is the byte stream of ie data of length ie_buf_len:
 *  A_UINT8 ie_data[];      // length in byte given by field ie_length, blob of ie data in beacon
 *  A_UINT8 anqp_ie[];      // length in byte given by field anqp_len, blob of anqp data of IE
 *  Implicitly, combing ie_data and anqp_ie into a single bufp, and the bytes stream of each ie should be same as BEACON/Action-frm  by 802.11 spec.
 */
} wmi_passpoint_event_hdr;


#define GTK_OFFLOAD_OPCODE_MASK				0xFF000000
/** Enable GTK offload, and provided parameters KEK,KCK and replay counter values */
#define GTK_OFFLOAD_ENABLE_OPCODE			0x01000000
/** Disable GTK offload */
#define GTK_OFFLOAD_DISABLE_OPCODE			0x02000000
/** Read GTK offload parameters, generates WMI_GTK_OFFLOAD_STATUS_EVENT */
#define GTK_OFFLOAD_REQUEST_STATUS_OPCODE	0x04000000
enum wmi_chatter_mode {
    /* Chatter enter/exit happens
     * automatically based on preset
     * params
     */
    WMI_CHATTER_MODE_AUTO,
    /* Chatter enter is triggered
     * manually by the user
     */
    WMI_CHATTER_MODE_MANUAL_ENTER,
    /* Chatter exit is triggered
     * manually by the user
     */
    WMI_CHATTER_MODE_MANUAL_EXIT,
    /* Placeholder max value, always last*/
    WMI_CHATTER_MODE_MAX
};

enum wmi_chatter_query_type {
    /*query coalescing filter match counter*/
    WMI_CHATTER_QUERY_FILTER_MATCH_CNT,
    WMI_CHATTER_QUERY_MAX
};

typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_chatter_set_mode_cmd_fixed_param  */
    A_UINT32 chatter_mode;
} wmi_chatter_set_mode_cmd_fixed_param;

/** maximum number of filter supported*/
#define CHATTER_MAX_COALESCING_RULES     11
/** maximum number of field tests per filter*/
#define CHATTER_MAX_FIELD_TEST           5
/** maximum field length in number of DWORDS*/
#define CHATTER_MAX_TEST_FIELD_LEN32     2

/** field test kinds*/
#define CHATTER_COALESCING_TEST_EQUAL            1
#define CHATTER_COALESCING_TEST_MASKED_EQUAL     2
#define CHATTER_COALESCING_TEST_NOT_EQUAL        3

/** packet type*/
#define CHATTER_COALESCING_PKT_TYPE_UNICAST      (1 << 0)
#define CHATTER_COALESCING_PKT_TYPE_MULTICAST    (1 << 1)
#define CHATTER_COALESCING_PKT_TYPE_BROADCAST    (1 << 2)

/** coalescing field test*/
typedef struct _chatter_pkt_coalescing_hdr_test {
    /** offset from start of mac header, for windows native wifi host driver
     * should assume standard 802.11 frame format without QoS info and address4
     * FW would account for any non-stand fields for final offset value.
     */
    A_UINT32    offset;
    A_UINT32    length;	    /* length of test field*/
    A_UINT32    test;       /*equal, not equal or masked equal*/
    A_UINT32    mask[CHATTER_MAX_TEST_FIELD_LEN32];    /*mask byte stream*/
    A_UINT32    value[CHATTER_MAX_TEST_FIELD_LEN32];   /*value byte stream*/
} chatter_pkt_coalescing_hdr_test;

/** packet coalescing filter*/
typedef struct _chatter_pkt_coalescing_filter {
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_chatter_pkt_coalescing_filter */
    A_UINT32    filter_id;  /*unique id assigned by OS*/
    A_UINT32    max_coalescing_delay; /*max miliseconds 1st pkt can be hold*/
    A_UINT32    pkt_type; /*unicast/multicast/broadcast*/
    A_UINT32    num_of_test_field;    /*number of field test in table*/
    chatter_pkt_coalescing_hdr_test   test_fields[CHATTER_MAX_FIELD_TEST]; /*field test tbl*/
} chatter_pkt_coalescing_filter;

/** packet coalescing filter add command*/
typedef struct {
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_chatter_coalescing_add_filter_cmd_fixed_param */
    A_UINT32    num_of_filters;
    /* Following this tlv, there comes an array of structure of type chatter_pkt_coalescing_filter
     chatter_pkt_coalescing_filter rx_filter[1];*/
} wmi_chatter_coalescing_add_filter_cmd_fixed_param;
/** packet coalescing filter delete command*/
typedef struct {
    A_UINT32    tlv_header;  /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_chatter_coalescing_delete_filter_cmd_fixed_param */
    A_UINT32    filter_id; /*filter id which will be deleted*/
} wmi_chatter_coalescing_delete_filter_cmd_fixed_param;
/** packet coalescing query command*/
typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_chatter_coalescing_query_cmd_fixed_param */
    A_UINT32    type;    /*type of query*/
} wmi_chatter_coalescing_query_cmd_fixed_param;
/** chatter query reply event*/
typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_chatter_query_reply_event_fixed_param  */
    A_UINT32    type;    /*query type*/
    A_UINT32    filter_match_cnt; /*coalescing filter match counter*/
} wmi_chatter_query_reply_event_fixed_param;

/* NOTE: This constants GTK_OFFLOAD_KEK_BYTES, GTK_OFFLOAD_KCK_BYTES, and GTK_REPLAY_COUNTER_BYTES
 * cannot be changed without breaking WMI compatibility. */
#define GTK_OFFLOAD_KEK_BYTES       16
#define GTK_OFFLOAD_KCK_BYTES       16
/* NOTE: GTK_REPLAY_COUNTER_BYTES, WMI_MAX_KEY_LEN, IGTK_PN_SIZE cannot be changed in the future without breaking WMI compatibility */
#define GTK_REPLAY_COUNTER_BYTES    8
#define WMI_MAX_KEY_LEN             32
#define IGTK_PN_SIZE                6

typedef struct {
    A_UINT32    tlv_header;  /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_GTK_OFFLOAD_STATUS_EVENT_fixed_param */
    A_UINT32    vdev_id;     /** unique id identifying the VDEV */
    A_UINT32    flags;       /* status flags */
    A_UINT32    refresh_cnt; /* number of successful GTK refresh exchanges since last SET operation */
    /*
     * As with all WMI messages, this message uses little-endian byte
     * ordering within each A_UINT32 field.
     * If a big-endian host is using automatic swapping of the bytes within
     * each 4-byte A_UINT32 to automatically correct the endianness of the
     * A_UINT32 fields as the message is uploaded from target --> host, the
     * big-endian host will have to undo the automatic byte swapping for the
     * below A_UINT8 fields, to restore them to their original order.
     */
    A_UINT8     replay_counter[GTK_REPLAY_COUNTER_BYTES]; /* current replay counter */
    A_UINT8     igtk_keyIndex; /* Use if IGTK_OFFLOAD is defined */
    A_UINT8     igtk_keyLength; /* Use if IGTK_OFFLOAD is defined */
    A_UINT8     igtk_keyRSC[IGTK_PN_SIZE];  /* key replay sequence counter */ /* Use if IGTK_OFFLOAD is defined */
    A_UINT8     igtk_key[WMI_MAX_KEY_LEN]; /* Use if IGTK_OFFLOAD is defined */
    A_UINT8 gtk_keyIndex; /* GTK key index */
    A_UINT8 gtk_keyLength; /* GTK key length */
    A_UINT8 gtk_keyRSC[GTK_REPLAY_COUNTER_BYTES]; /* GTK key replay sequence counter */
    A_UINT8 gtk_key[WMI_MAX_KEY_LEN]; /* GTK key data */
} WMI_GTK_OFFLOAD_STATUS_EVENT_fixed_param;

typedef struct {
    A_UINT32    tlv_header;                     /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_GTK_OFFLOAD_CMD_fixed_param */
    A_UINT32	vdev_id;					/** unique id identifying the VDEV */
    A_UINT32    flags;                          /* control flags, GTK offload command use high byte  */
    /* The size of following 3 arrays cannot be changed without breaking WMI compatibility. */
    A_UINT8     KEK[GTK_OFFLOAD_KEK_BYTES];     /* key encryption key */
    A_UINT8     KCK[GTK_OFFLOAD_KCK_BYTES];     /* key confirmation key */
    A_UINT8     replay_counter[GTK_REPLAY_COUNTER_BYTES];  /* replay counter for re-key */
}WMI_GTK_OFFLOAD_CMD_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_gtk_offload_extended_tlv_param */
    A_UINT32 vdev_id; /** unique id identifying the VDEV */
    A_UINT32 flags; /* control flags, GTK offload command use high byte  */
    A_UINT32 kek_len;
    A_UINT8  KEK[GTK_OFFLOAD_KEK_EXTENDED_BYTES]; /* key encryption key */
    A_UINT8  KCK[GTK_OFFLOAD_KCK_BYTES]; /* key confirmation key */
    A_UINT8  replay_counter[GTK_REPLAY_COUNTER_BYTES]; /* replay counter for re-key */
} wmi_gtk_offload_fils_tlv_param;

typedef struct {
    A_UINT32 tlv_header;  /** TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_PMF_OFFLOAD_SET_SA_QUERY_CMD_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 sa_query_retry_interval;  /* in msec */
    A_UINT32 sa_query_max_retry_count;
} WMI_PMF_OFFLOAD_SET_SA_QUERY_CMD_fixed_param;

typedef enum {
    WMI_STA_KEEPALIVE_METHOD_NULL_FRAME = 1,                   /* 802.11 NULL frame */
    WMI_STA_KEEPALIVE_METHOD_UNSOLICITED_ARP_RESPONSE = 2,     /* ARP response */
    WMI_STA_KEEPALIVE_METHOD_ETHERNET_LOOPBACK = 3, /*ETHERNET LOOPBACK*/
    WMI_STA_KEEPALIVE_METHOD_GRATUITOUS_ARP_REQUEST = 4, /* gratuitous ARP req*/
} WMI_STA_KEEPALIVE_METHOD;

typedef struct {
    A_UINT32                 tlv_header;               /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_STA_KEEPALVE_ARP_RESPONSE */
    WMI_IPV4_ADDR            sender_prot_addr;         /* Sender protocol address */
    WMI_IPV4_ADDR            target_prot_addr;         /* Target protocol address */
    wmi_mac_addr             dest_mac_addr;            /* destination MAC address */
} WMI_STA_KEEPALVE_ARP_RESPONSE;

typedef struct  {
    A_UINT32 tlv_header;                    /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_STA_KEEPALIVE_CMD_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 enable;                        /* 1 - Enable, 0 - disable */
    A_UINT32 method;                        /* keep alive method */
    A_UINT32 interval;                      /* time interval in seconds  */
    /*
        * NOTE: following this structure is the TLV for ARP Resonse:
        *     WMI_STA_KEEPALVE_ARP_RESPONSE arp_resp; // ARP response
        */
} WMI_STA_KEEPALIVE_CMD_fixed_param;

typedef struct  {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 action;
} WMI_VDEV_WNM_SLEEPMODE_CMD_fixed_param;
typedef WMI_VDEV_WNM_SLEEPMODE_CMD_fixed_param WMI_STA_WNMSLEEP_CMD;

typedef struct {
    A_UINT32 tlv_header;           /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_set_keepalive_cmd_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 keepaliveInterval;   /* seconds */
    A_UINT32 keepaliveMethod;
} wmi_vdev_set_keepalive_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;           /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_get_keepalive_cmd_fixed_param */
    A_UINT32    vdev_id;
} wmi_vdev_get_keepalive_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_get_keepalive_event_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 keepaliveInterval;   /* seconds */
    A_UINT32 keepaliveMethod;   /* seconds */
} wmi_vdev_get_keepalive_event_fixed_param;

typedef enum {
    WMI_CLEAR_ARP_STATS_COLLECTED = 0x0,
    WMI_START_ARP_STATS_COLLECTION,
} WMI_ARP_STATS_ACTION;

typedef enum {
    WMI_ARP_STATS_RX_PKT_TYPE_ARP = 0x1,
} WMI_ARP_STATS_RX_PKT_TYPE;

typedef enum {
    WMI_BA_SESSION_ESTABLISHMENT_STATUS_SUCCESS = 0x0,
    WMI_BA_SESSION_ESTABLISHMENT_STATUS_FAILURE,
} WMI_ARP_STATS_BA_SESSION_ESTABLISH_STATUS;

typedef enum {
    WMI_CONNECTION_STATUS_FAILURE = 0x0,
    WMI_CONNECTION_STATUS_SUCCESS,
} WMI_ARP_STATS_CONNECTION_STATUS;

/* ARP stats set (configure) req */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_set_arp_stats_cmd_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 set_clr;  /* WMI_ARP_STATS_ACTION */
    A_UINT32 pkt_type; /* WMI_ARP_STATS_RX_PKT_TYPE */
    A_UINT32 ipv4;     /* target will maintain ARP stats (only) for frames containing this IP address */
} wmi_vdev_set_arp_stats_cmd_fixed_param;

/* ARP stats get req */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_get_arp_stats_cmd_fixed_param */
    A_UINT32 vdev_id;
} wmi_vdev_get_arp_stats_cmd_fixed_param;

/* per vdev based ARP stats */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_get_arp_stats_event_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 arp_req_enqueue;
    A_UINT32 arp_req_tx_success;
    A_UINT32 arp_req_tx_failure; /* number of times a tx MPDU containing a unicast ARP request went unacked */
    A_UINT32 arp_rsp_recvd;
    A_UINT32 out_of_order_arp_rsp_drop_cnt;
    A_UINT32 dad_detected; /* duplicate address detection */
    A_UINT32 connect_status; /* WMI_ARP_STATS_CONNECTION_STATUS */
    A_UINT32 ba_session_establishment_status; /* WMI_ARP_STATS_BA_SESSION_ESTABLISH_STATUS */
} wmi_vdev_get_arp_stats_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /*  TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_set_connectivity_check_stats */
    A_UINT32 pkt_type_bitmap;  /* WMI_IP_CONNECTIVITY_STATS_RX_PKT_TYPE_BITMAP - only for DNS, TCP and ICMP */
    A_UINT32 tcp_src_port;  /* target will maintain TCP stats (only) for frames with this src port */
    A_UINT32 tcp_dst_port; /* target will maintain TCP stats (only) for frames with this dst port */
    A_UINT32 icmp_ipv4; /* target will maintain ICMPv4 stats (only) for frames containing this IP address */
} wmi_vdev_set_connectivity_check_stats;

/*  per vdev dns/icmp/tcp based stats*/
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_get_connectivity_check_stats */
    A_UINT32 tcp_ack_recvd; /* number of tcp syn ack's received by FW */
    A_UINT32 icmpv4_rsp_recvd; /* number of icmpv4 responses received by FW */
} wmi_vdev_get_connectivity_check_stats;

#define IPSEC_NATKEEPALIVE_FILTER_DISABLE 0
#define IPSEC_NATKEEPALIVE_FILTER_ENABLE  1

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 action;
} WMI_VDEV_IPSEC_NATKEEPALIVE_FILTER_CMD_fixed_param;

typedef WMI_VDEV_IPSEC_NATKEEPALIVE_FILTER_CMD_fixed_param WMI_VDEV_IPSEC_NATKEEPALIVE_FILTER_CMD;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 mcc_tbttmode;
    wmi_mac_addr mcc_bssid;
} wmi_vdev_mcc_set_tbtt_mode_cmd_fixed_param;

#define WMI_MAX_VENDOR_OUI_ACTION_SUPPORTED_PER_ACTION 10
#define WMI_MAX_VENDOR_OUI_DATA_LENGTH                 20

typedef enum
{
    WMI_VENDOR_OUI_ACTION_CONNECTION_1X1 = 0, /* Connect in 1X1 only */
    WMI_VENDOR_OUI_ACTION_ITO_EXTENSION = 1,  /* Extend the Immediate Time-Out (ITO) if data is not received from AP after beacon with TIM bit set */
    WMI_VENDOR_OUI_ACTION_CCKM_1X1 = 2,       /* TX (only) CCKM rates with 1 chain only */
    WMI_VENDOR_OUI_ACTION_ALT_ITO = 3,        /* inactivity time-out */
    WMI_VENDOR_OUI_ACTION_SWITCH_TO_11N_MODE = 4, /* Switch from 11ac to 11n mode to avoid IOT issues with ONM frame */
    /* WMI_VENDOR_OUI_ACTION_CONNECTION_1X1_NUM_TX_RX_CHAINS_1
     * Connect in 1x1 only and Use only one chain for both Tx and Rx
     * to avoid IOT issues due to change in number of Tx and Rx chains
     */
    WMI_VENDOR_OUI_ACTION_CONNECTION_1X1_NUM_TX_RX_CHAINS_1 = 5,

    /* Disable burst and assist, and restrict A-MPDU size to 32 */
    WMI_VENDOR_OUI_ACTION_DISABLE_AGGRESSIVE_TX = 6,

    /* Add any action before this line */
    WMI_VENDOR_OUI_ACTION_MAX_ACTION_ID
} wmi_vendor_oui_action_id;

typedef struct {
    A_UINT32 tlv_header;
    /** vendor OUI actions */
    A_UINT32 action_id;  /* enum wmi_vendor_oui_action_id */
    A_UINT32 total_num_vendor_oui; /* total number of OUI present in ini for all actions.
                                   ** For first command, this value will be used for allocating memory in FW accordingly */
    A_UINT32 num_vendor_oui_ext; /* the number of wmi_vendor_oui_ext for action_id */
    /* followed by TLVs, an array of structure of type wmi_vendor_oui_ext
    ** wmi_vendor_oui_ext vendor_oui_ext[num_vendor_oui_ext];
    */
    /* followed by A_UINT8 data[] of concatenated data for each vendor_oui_ext[] element
    ** first byte contains the index i of structure vendor_oui_ext[]
    ** for which data is coming and length of the data is extracted as
    ** wmi_vendor_oui_ext[i].buf_data_length. No padding between data of
    ** N th OUI and (N+1) th OUI.
    ** For example, if vendor_oui_ext[0].buf_data_length is 18, then
    ** data[0] will hold the index value 0, data[1] through data[17]
    ** will hold the OUI data for this first OUI, data[18] will hold
    ** the index value 1, and the OUI data for the second OUI will
    ** begin at data[19].
    */
} wmi_pdev_config_vendor_oui_action_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_ac_tx_queue_optimized_cmd_fixed_param */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
    /** AC number */
    A_UINT32 ac; /* refer to wmi_traffic_ac */
    /**
     * Enable/disable tx queue optimizations (such as dropping stale tx frms)
     * for the specified AC.
     */
    A_UINT32 ac_tx_queue_optimize_enable;
} wmi_pdev_set_ac_tx_queue_optimized_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_rx_filter_promiscuous_cmd_fixed_param */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
    /** Enable/disable rx filter promiscuous */
    A_UINT32 rx_filter_promiscuous_enable;
} wmi_pdev_set_rx_filter_promiscuous_cmd_fixed_param;

typedef enum {
    WMI_BEACON_INFO_PRESENCE_OUI_EXT            = 1 <<  0,
    WMI_BEACON_INFO_PRESENCE_MAC_ADDRESS        = 1 <<  1,
    WMI_BEACON_INFO_PRESENCE_AP_CAPABILITY_NSS  = 1 <<  2,
    WMI_BEACON_INFO_PRESENCE_AP_CAPABILITY_HT   = 1 <<  3,
    WMI_BEACON_INFO_PRESENCE_AP_CAPABILITY_VHT  = 1 <<  4,
    WMI_BEACON_INFO_PRESENCE_AP_CAPABILITY_BAND = 1 <<  5,
} wmi_beacon_info_presence_items;

typedef struct _wmi_vendor_oui_ext {
    A_UINT32 tlv_header;
    A_UINT32 buf_data_length;        /* length of data in bytes for this OUI including index byte */
    A_UINT32 info_presence_bit_mask; /* see enum wmi_beacon_info_presence_items */
    A_UINT32 oui_header_length;      /* either 3 or 5 bytes */
    A_UINT32 oui_data_length;        /* length of oui_data to compare in beacon which follows OUI header. Max length is capped to WMI_MAX_VENDOR_OUI_DATA_LENGTH bytes */
    A_UINT32 mac_address_length;     /* MAC address length in bytes
                                     ** (This value will always be 6,
                                     ** but is explicitly specified for sake
                                     ** of uniformity and completeness).
                                     */
    A_UINT32 capability_data_length; /* length of capability in bytes */
} wmi_vendor_oui_ext;

#define WMI_INFO_CAPABILITY_NSS_MASK                       0x0f
#define WMI_INFO_CAPABILITY_NSS_OFFSET                     0
#define WMI_INFO_CAPABILITY_HT_ENABLE_MASK                 0x10
#define WMI_INFO_CAPABILITY_HT_ENABLE_OFFSET               4
#define WMI_INFO_CAPABILITY_VHT_ENABLE_MASK                0x20
#define WMI_INFO_CAPABILITY_VHT_ENABLE_OFFSET              5
#define WMI_INFO_CAPABILITY_BAND_MASK                      0xc0
#define WMI_INFO_CAPABILITY_BAND_OFFSET                    6

/* definition of WMI_INFO_CAPABILITY_NSS_MASK */
#define WMI_INFO_CAPABILITY_NSS_1X1                        1
#define WMI_INFO_CAPABILITY_NSS_2X2                        2
#define WMI_INFO_CAPABILITY_NSS_3X3                        3
#define WMI_INFO_CAPABILITY_NSS_4X4                        4

/* definition of WMI_INFO_CAPABILITY_BAND_MASK */
#define WMI_INFO_CAPABILITY_2G_BAND_MASK                   (1 << 0)
#define WMI_INFO_CAPABILITY_5G_BAND_MASK                   (1 << 1)

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;        /* home vdev id */
    A_UINT32 meas_token;     /* from measure request frame */
    A_UINT32 dialog_token;
    A_UINT32 number_bursts;  /* zero keep sending until cancel, bigger than 0 means times e.g. 1,2 */
    A_UINT32 burst_interval; /* unit in mill seconds, interval between consecutive burst*/
    A_UINT32 burst_cycle;   /* times cycle through within one burst */
    A_UINT32 tx_power;      /* for path frame */
    A_UINT32 off_duration;  /* uint in mill seconds, channel off duraiton for path loss frame sending */
    wmi_mac_addr dest_mac; /* multicast DA, for path loss frame */
    A_UINT32 num_chans;
} wmi_vdev_plmreq_start_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 meas_token; /* same value from req*/
} wmi_vdev_plmreq_stop_cmd_fixed_param;

typedef struct {
	/* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_p2p_set_noa_cmd_fixed_param  */
    A_UINT32 tlv_header;
    /* unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /* enable/disable NoA */
	A_UINT32 enable;
	/** number of NoA desc. In the TLV noa_descriptor[] */
	A_UINT32 num_noa;
	/**
     * TLV (tag length value )  paramerters follow the pattern structure.
     * TLV  contain NoA desc with num of num_noa
     */
} wmi_p2p_set_noa_cmd_fixed_param;

typedef struct {
    /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_unit_test_cmd_fixed_param  */
    A_UINT32 tlv_header;
    /* unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /* Identify the wlan module */
    A_UINT32 module_id;
    /* Num of test arguments passed */
    A_UINT32 num_args;
    /* unique id identifying the unit test cmd, generated by the caller */
    A_UINT32 diag_token;
/**
 * TLV (tag length value ) parameters follow the wmi_unit_test_cmd_fixed_param
 * structure. The TLV's are:
 *   A_UINT32 args[];
 **/
} wmi_unit_test_cmd_fixed_param;

/** Roaming offload SYNCH_COMPLETE from host when host finished sync logic
 * after it received WMI_ROAM_SYNCH_EVENTID.
 */
typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_synch_complete_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
} wmi_roam_synch_complete_fixed_param;


typedef enum {
    RECOVERY_SIM_ASSERT          = 0x01,
    RECOVERY_SIM_NO_DETECT       = 0x02,
    RECOVERY_SIM_CTR_EP_FULL     = 0x03,
    RECOVERY_SIM_EMPTY_POINT     = 0x04,
    RECOVERY_SIM_STACK_OV        = 0x05,
    RECOVERY_SIM_INFINITE_LOOP   = 0x06,
    RECOVERY_SIM_PCIE_LINKDOWN   = 0x07,
    RECOVERY_SIM_SELF_RECOVERY   = 0x08,
} RECOVERY_SIM_TYPE;

/* WMI_FORCE_FW_HANG_CMDID */
typedef struct {
    A_UINT32 tlv_header;           /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_FORCE_FW_HANG_CMD_fixed_param */
    A_UINT32 type;     /*0:unused 1: ASSERT, 2: not respond detect command,3:  simulate ep-full(),4:...*/
    A_UINT32 delay_time_ms;   /*0xffffffff means the simulate will delay for random time (0 ~0xffffffff ms)*/
}WMI_FORCE_FW_HANG_CMD_fixed_param;

typedef enum {
    WMI_MCAST_FILTER_SET = 1,
    WMI_MCAST_FILTER_DELETE
} WMI_SET_SINGLE_MCAST_FILTER_OP;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 index;
    A_UINT32 action;
    wmi_mac_addr mcastbdcastaddr;
} WMI_SET_MCASTBCAST_FILTER_CMD_fixed_param;

typedef enum {
    WMI_MULTIPLE_MCAST_FILTER_CLEAR = 1, /* clear all previous mc list */
    /* clear all previous mc list, and set new list */
    WMI_MULTIPLE_MCAST_FILTER_SET,
    WMI_MULTIPLE_MCAST_FILTER_DELETE,    /* delete one/multiple mc list */
    WMI_MULTIPLE_MCAST_FILTER_ADD        /* add one/multiple mc list */
} WMI_MULTIPLE_MCAST_FILTER_OP;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 operation;  /* refer WMI_MULTIPLE_MCAST_FILTER_OP */
    /* number of elements in the subsequent mcast addr list */
    A_UINT32 num_mcastaddrs;
    /**
     * TLV (tag length value) parameters follow the
     * structure. The TLV's are:
     * wmi_mac_addr mcastaddr_list[num_mcastaddrs];
     */
} WMI_SET_MULTIPLE_MCAST_FILTER_CMD_fixed_param;


/* WMI_DBGLOG_TIME_STAMP_SYNC_CMDID */
typedef enum {
    WMI_TIME_STAMP_SYNC_MODE_MS, /* millisecond units */
    WMI_TIME_STAMP_SYNC_MODE_US, /* microsecond units */
} WMI_TIME_STAMP_SYNC_MODE;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_dbglog_time_stamp_sync_cmd_fixed_param */
    A_UINT32 mode; /* 0: millisec, 1: microsec (see WMI_TIME_STAMP_SYNC_MODE) */
    A_UINT32 time_stamp_low; /* lower 32 bits of remote time stamp */
    A_UINT32 time_stamp_high; /* higher 32 bits of remote time stamp */
} WMI_DBGLOG_TIME_STAMP_SYNC_CMD_fixed_param;


/* GPIO Command and Event data structures */

/* WMI_GPIO_CONFIG_CMDID */
enum {
    WMI_GPIO_PULL_NONE,
    WMI_GPIO_PULL_UP,
    WMI_GPIO_PULL_DOWN,
};

enum {
    WMI_GPIO_INTTYPE_DISABLE,
    WMI_GPIO_INTTYPE_RISING_EDGE,
    WMI_GPIO_INTTYPE_FALLING_EDGE,
    WMI_GPIO_INTTYPE_BOTH_EDGE,
    WMI_GPIO_INTTYPE_LEVEL_LOW,
    WMI_GPIO_INTTYPE_LEVEL_HIGH
};

typedef struct {
    A_UINT32 tlv_header;           /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_gpio_config_cmd_fixed_param */
    A_UINT32 gpio_num;             /* GPIO number to be setup */
    A_UINT32 input;                /* 0 - Output/ 1 - Input */
    A_UINT32 pull_type;            /* Pull type defined above */
    A_UINT32 intr_mode;            /* Interrupt mode defined above (Input) */
    /* mux_config_val - configure pin MUX
     * A value of 0x0 for this field means to use the default MUX configuration.
     * Valid non-zero values are as follows:
     * Rome:
     *     0x4 - use the pin as GPIO (rather than UART)
     */
    A_UINT32 mux_config_val;
} wmi_gpio_config_cmd_fixed_param;

/* WMI_GPIO_OUTPUT_CMDID */
typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_gpio_output_cmd_fixed_param */
    A_UINT32 gpio_num;    /* GPIO number to be setup */
    A_UINT32 set;         /* Set the GPIO pin*/
} wmi_gpio_output_cmd_fixed_param;

/* WMI_GPIO_INPUT_EVENTID */
typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_gpio_input_event_fixed_param */
    A_UINT32 gpio_num;    /* GPIO number which changed state */
} wmi_gpio_input_event_fixed_param;

/* WMI_P2P_DISC_EVENTID */
enum {
    P2P_DISC_SEARCH_PROB_REQ_HIT = 0,   /* prob req hit the p2p find pattern */
    P2P_DISC_SEARCH_PROB_RESP_HIT,      /* prob resp hit the p2p find pattern */
};

enum {
    P2P_DISC_MODE_SEARCH = 0, /* do search when p2p find offload*/
    P2P_DISC_MODE_LISTEN,     /* do listen when p2p find offload*/
	P2P_DISC_MODE_AUTO,       /* do listen and search when p2p find offload*/
};

enum {
    P2P_DISC_PATTERN_TYPE_BSSID = 0,  /* BSSID pattern */
    P2P_DISC_PATTERN_TYPE_DEV_NAME,   /* device name pattern */
};

typedef struct {
    A_UINT32    vdev_id;
    A_UINT32    reason;    /* P2P DISC wake up reason*/
} wmi_p2p_disc_event;

typedef WMI_GTK_OFFLOAD_STATUS_EVENT_fixed_param WOW_EVENT_INFO_SECTION_GTKIGTK;

typedef enum {
    WMI_FAKE_TXBFER_SEND_NDPA,
    WMI_FAKE_TXBFER_SEND_MU,
    WMI_FAKE_TXBFER_NDPA_FBTYPE,
    WMI_FAKE_TXBFER_NDPA_NCIDX,
    WMI_FAKE_TXBFER_NDPA_POLL,
    WMI_FAKE_TXBFER_NDPA_BW,
    WMI_FAKE_TXBFER_NDPA_PREAMBLE,
    WMI_FAKE_TXBFER_NDPA_RATE,
    WMI_FAKE_TXBFER_NDP_BW,
    WMI_FAKE_TXBFER_NDP_NSS,
    WMI_TXBFEE_ENABLE_UPLOAD_H,
    WMI_TXBFEE_ENABLE_CAPTURE_H,
    WMI_TXBFEE_SET_CBF_TBL,
    WMI_TXBFEE_CBF_TBL_LSIG,
    WMI_TXBFEE_CBF_TBL_SIGA1,
    WMI_TXBFEE_CBF_TBL_SIGA2,
    WMI_TXBFEE_CBF_TBL_SIGB,
    WMI_TXBFEE_CBF_TBL_PAD,
    WMI_TXBFEE_CBF_TBL_DUR,
    WMI_TXBFEE_SU_NCIDX,
    WMI_TXBFEE_CBIDX,
    WMI_TXBFEE_NGIDX,
} WMI_TXBF_PARAM_ID;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_txbf_cmd_fixed_param */
    /** parameter id   */
    A_UINT32 param_id;
    /** parameter value */
    A_UINT32 param_value;
} wmi_txbf_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_upload_h_hdr */
    A_UINT32     h_length;
    A_UINT32     cv_length;
    /* This TLV is followed by array of bytes:
     * // h_cv info buffer
     *   A_UINT8 bufp[];
     */
} wmi_upload_h_hdr;

typedef struct {
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_capture_h_event_hdr */
    A_UINT32 svd_num;
    A_UINT32 tone_num;
    A_UINT32 reserved;
} wmi_capture_h_event_hdr;

typedef struct {
    A_UINT32    tlv_header;                 /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_avoid_freq_range_desc */
    A_UINT32    start_freq;  //start frequency, not channel center freq
    A_UINT32    end_freq;   //end frequency
} wmi_avoid_freq_range_desc;

typedef struct {
    A_UINT32    tlv_header;                 /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_avoid_freq_ranges_event_fixed_param */
   //bad channel range count, multi range is allowed, 0 means all channel clear
   A_UINT32  num_freq_ranges;

   /* The TLVs will follow.
       * multi range with num_freq_ranges, LTE advance multi carrier, CDMA,etc
       *     wmi_avoid_freq_range_desc avd_freq_range[];     // message buffer, NULL terminated
       */
} wmi_avoid_freq_ranges_event_fixed_param;

enum {
    WMI_SAR2_SUCCESS                = 0,
    WMI_SAR2_INVALID_ANTENNA_INDEX  = 1,
    WMI_SAR2_INVALID_TABLE_INDEX    = 2,
    WMI_SAR2_STATE_ERROR            = 4,
    WMI_SAR2_BDF_NO_TABLE           = 8,
};

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_sar2_result_event_fixed_param  */
    A_UINT32 result; /* refer to the above WMI_SAR2_ result definitions */
} wmi_sar2_result_event_fixed_param;

typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_gtk_rekey_fail_event_fixed_param  */
    /** Reserved for future use */
    A_UINT32    reserved0;
    A_UINT32 vdev_id;
} wmi_gtk_rekey_fail_event_fixed_param;

typedef enum WLAN_COEX_EVENT {
    WLAN_COEX_EVENT_BT_NONE                     = 0,
    WLAN_COEX_EVENT_BT_A2DP_PROFILE_ADD         = 1,
    WLAN_COEX_EVENT_BT_A2DP_PROFILE_REMOVE      = 2,
    WLAN_COEX_EVENT_BT_VOICE_PROFILE_ADD        = 3,
    WLAN_COEX_EVENT_BT_VOICE_PROFILE_REMOVE     = 4,
    WLAN_COEX_EVENT_BT_SCAN_START               = 5,
    WLAN_COEX_EVENT_BT_SCAN_STOP                = 6,
}WLAN_COEX_EVENT;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 coex_profile_evt; //uses the enum values form WLAN_COEX_EVENT
} wmi_coex_bt_activity_event_fixed_param;

enum wmm_ac_downgrade_policy {
    WMM_AC_DOWNGRADE_DEPRIO,
    WMM_AC_DOWNGRADE_DROP,
    WMM_AC_DOWNGRADE_INVALID,
};

/* WMM EDCA Params type */
#define WMM_PARAM_TYPE_LEGACY        0
/* Relaxed EDCA parameters for 11ax to be used in case of triggered access */
#define WMM_PARAM_TYPE_11AX_EDCA     1

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 cwmin;
    A_UINT32 cwmax;
    A_UINT32 aifs;
    union {
        A_UINT32 txoplimit;
        A_UINT32 mu_edca_timer;
    };
    A_UINT32 acm;
    A_UINT32 no_ack;
} wmi_wmm_vparams;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    wmi_wmm_vparams wmm_params[4]; /* 0 be, 1 bk, 2 vi, 3 vo */
    A_UINT32 wmm_param_type; /* see WMM_PARAM_TYPE_xxx defs */
} wmi_vdev_set_wmm_params_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32              gtxRTMask[2]; /* for HT and VHT rate masks */
    A_UINT32              userGtxMask; /* host request for GTX mask */
    A_UINT32              gtxPERThreshold; /* default: 10% */
    A_UINT32              gtxPERMargin; /* default: 2% */
    A_UINT32              gtxTPCstep; /* default: 1 */
    A_UINT32              gtxTPCMin; /* default: 5 */
    A_UINT32              gtxBWMask; /* 20/40/80/160 Mhz */
} wmi_vdev_set_gtx_params_cmd_fixed_param;

typedef struct
{
    A_UINT32 tlv_header;
    A_UINT32  vdev_id;
    A_UINT32 ac;
    A_UINT32 medium_time_us; /* per second unit, the Admitted time granted, unit in micro seconds */
    A_UINT32 downgrade_type;
} wmi_vdev_wmm_addts_cmd_fixed_param;

typedef struct
{
    A_UINT32 tlv_header;
    A_UINT32  vdev_id;
    A_UINT32 ac;
} wmi_vdev_wmm_delts_cmd_fixed_param;

/* DEPRECATED */
typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_dfs_enable_cmd_fixed_param  */
    /** Reserved for future use */
    A_UINT32    reserved0;
} wmi_pdev_dfs_enable_cmd_fixed_param;

/* DEPRECATED */
typedef struct {
    A_UINT32    tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_dfs_disable_cmd_fixed_param  */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
} wmi_pdev_dfs_disable_cmd_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_dfs_phyerr_filter_ena_cmd_fixed_param
     */
    A_UINT32 tlv_header;

    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
} wmi_dfs_phyerr_filter_ena_cmd_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_dfs_phyerr_filter_dis_cmd_fixed_param
     */
    A_UINT32 tlv_header;
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
} wmi_dfs_phyerr_filter_dis_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 pdev_id;
} wmi_pdev_dfs_phyerr_offload_enable_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 pdev_id;
} wmi_pdev_dfs_phyerr_offload_disable_cmd_fixed_param;

typedef enum {
    QUICK_OCAC = 0,
    EXTENSIVE_OCAC,
} WMI_ADFS_OCAC_MODE;

typedef struct {
    A_UINT32  tlv_header;
    A_UINT32  vdev_id;
    A_UINT32  ocac_mode;  /* WMI_ADFS_OCAC_MODE */
    A_UINT32  min_duration_ms; /* in milliseconds */
    A_UINT32  max_duration_ms; /* in milliseconds */
    A_UINT32  chan_freq;   /* in MHz */
    A_UINT32  chan_width;  /* in MHz */
    A_UINT32  center_freq; /* in MHz */
} wmi_vdev_adfs_ch_cfg_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
} wmi_vdev_adfs_ocac_abort_cmd_fixed_param;

typedef enum {
    IN_SERVICE_MODE = 0,
    OCAC_MODE,
} WMI_DFS_RADAR_DETECTION_MODE;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 pdev_id;
    /* In-service mode or O-CAC mode */
    A_UINT32 detection_mode; /* WMI_DFS_RADAR_DETECTION_MODE */
    A_UINT32 chan_freq;  /* in MHz */
    A_UINT32 chan_width; /* in MHz */
    A_UINT32 detector_id;
    A_UINT32 segment_id;
    A_UINT32 timestamp;
    A_UINT32 is_chirp;
    A_INT32  freq_offset; /* in MHz */
    A_INT32  sidx; /* segment index (where was the radar within the channel) */
} wmi_pdev_dfs_radar_detection_event_fixed_param;

typedef enum {
    OCAC_COMPLETE = 0,
    OCAC_ABORT,
} WMI_VDEV_OCAC_COMPLETE_STATUS;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 chan_freq;   /* in MHz */
    A_UINT32 chan_width;  /* in MHz */
    A_UINT32 center_freq; /* in MHz */
    A_UINT32 status;   /* WMI_VDEV_OCAC_COMPLETE_STATUS */
} wmi_vdev_adfs_ocac_complete_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
} wmi_vdev_dfs_cac_complete_event_fixed_param;

/** TDLS COMMANDS */

/* WMI_TDLS_SET_STATE_CMDID */
/* TDLS State */
enum wmi_tdls_state {
    /** TDLS disable */
    WMI_TDLS_DISABLE,
    /** TDLS enabled - no firmware connection tracking/notifications */
    WMI_TDLS_ENABLE_PASSIVE,
    /** TDLS enabled - with firmware connection tracking/notifications */
    WMI_TDLS_ENABLE_ACTIVE,
    /** TDLS enabled - firmware waits for peer mac for connection tracking */
    WMI_TDLS_ENABLE_ACTIVE_EXTERNAL_CONTROL,
    /** TDLS enabled - TDLS connection tracking is done in host */
    WMI_TDLS_ENABLE_CONNECTION_TRACKER_IN_HOST,
};

/* TDLS Options */
#define WMI_TDLS_OFFCHAN_EN             (1 << 0) /** TDLS Off Channel support */
#define WMI_TDLS_BUFFER_STA_EN          (1 << 1) /** TDLS Buffer STA support */
#define WMI_TDLS_SLEEP_STA_EN           (1 << 2) /** TDLS Sleep STA support (not currently supported) */

typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tdls_set_state_cmd_fixed_param  */
    A_UINT32 tlv_header;
    /** unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /** Enable/Disable TDLS (wmi_tdls_state) */
    A_UINT32 state;
    /** Duration (in ms) over which to calculate tx/rx threshold to trigger TDLS Discovery */
    A_UINT32 notification_interval_ms;
    /** number of packets OVER which notify/suggest TDLS Discovery:
     *  if current tx pps counter / notification interval >= threshold
     *  then a notification will be sent to host to advise TDLS Discovery */
    A_UINT32 tx_discovery_threshold;
    /** number of packets UNDER which notify/suggest TDLS Teardown:
     *  if current tx pps counter / notification interval < threshold
     *  then a notification will be sent to host to advise TDLS Tear down */
    A_UINT32 tx_teardown_threshold;
    /** Absolute RSSI value under which notify/suggest TDLS Teardown */
    A_INT32 rssi_teardown_threshold;
    /** Peer RSSI < (AP RSSI + delta) will trigger a teardown */
    A_INT32 rssi_delta;
    /** TDLS Option Control
     * Off-Channel, Buffer STA, (later)Sleep STA support */
    A_UINT32 tdls_options;
    /* Buffering time in number of beacon intervals */
    A_UINT32 tdls_peer_traffic_ind_window;
    /* Wait time for PTR frame */
    A_UINT32 tdls_peer_traffic_response_timeout_ms;
    /* Self PUAPSD mask */
    A_UINT32 tdls_puapsd_mask;
    /* Inactivity timeout */
    A_UINT32 tdls_puapsd_inactivity_time_ms;
    /* Max of rx frame during SP */
    A_UINT32 tdls_puapsd_rx_frame_threshold;
    /**Duration (in ms) over which to check whether TDLS link needs to be torn down */
    A_UINT32 teardown_notification_ms;
    /**STA kickout threshold for TDLS peer */
    A_UINT32 tdls_peer_kickout_threshold;
    /* TDLS discovery WAKE timeout in ms.
     * DUT will wake until this timeout to receive TDLS discovery response
     * from peer.
     * If tdls_discovery_wake_timeout is 0x0, the DUT will choose autonomously
     * what wake timeout value to use.
     */
    A_UINT32 tdls_discovery_wake_timeout;
} wmi_tdls_set_state_cmd_fixed_param;

/* WMI_TDLS_PEER_UPDATE_CMDID */

enum wmi_tdls_peer_state {
    /** tx peer TDLS link setup now starting, traffic to DA should be
     * paused (except TDLS frames) until state is moved to CONNECTED (or
     * TEARDOWN on setup failure) */
    WMI_TDLS_PEER_STATE_PEERING,
    /** tx peer TDLS link established, running (all traffic to DA unpaused) */
    WMI_TDLS_PEER_STATE_CONNECTED,
    /** tx peer TDLS link tear down started (link paused, any frames
     * queued for DA will be requeued back through the AP)*/
    WMI_TDLS_PEER_STATE_TEARDOWN,
    /** Add peer mac into connection table */
    WMI_TDLS_PEER_ADD_MAC_ADDR,
    /** Remove peer mac from connection table */
    WMI_TDLS_PEER_REMOVE_MAC_ADDR,
};

/* NB: These defines are fixed, and cannot be changed without breaking WMI compatibility */
#define WMI_TDLS_MAX_SUPP_OPER_CLASSES 32
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tdls_peer_capabilities  */
    A_UINT32 tlv_header;
    /* Peer's QoS Info - for U-APSD */
    /* AC FLAGS  - accessed through macros below */
    /* Ack, SP, More Data Ack - accessed through macros below */
    A_UINT32 peer_qos;
    /*TDLS Peer's U-APSD Buffer STA Support*/
    A_UINT32 buff_sta_support;
    /*TDLS off channel related params */
    A_UINT32 off_chan_support;
    A_UINT32 peer_curr_operclass;
    A_UINT32 self_curr_operclass;
    /* Number of channels available for off channel operation */
    A_UINT32 peer_chan_len;
    A_UINT32 peer_operclass_len;
    A_UINT8  peer_operclass[WMI_TDLS_MAX_SUPP_OPER_CLASSES];
    /* Is peer initiator or responder of TDLS setup request */
    A_UINT32 is_peer_responder;
    /* Preferred off channel number as configured by user */
    A_UINT32 pref_offchan_num;
    /* Preferred off channel bandwidth as configured by user */
    A_UINT32 pref_offchan_bw;

    /** Followed by the variable length TLV peer_chan_list:
     *  wmi_channel peer_chan_list[].
     *  Array size would be peer_chan_len.
     *  This array is intersected channels which is supported by both peer
     *  and DUT. freq1 in chan_info shall be same as mhz, freq2 shall be 0.
     *  FW shall compute BW for an offchan based on peer's ht/vht cap
     *  received in peer_assoc cmd during change STA operation
     */
} wmi_tdls_peer_capabilities;

#define WMI_TDLS_QOS_VO_FLAG           0
#define WMI_TDLS_QOS_VI_FLAG           1
#define WMI_TDLS_QOS_BK_FLAG           2
#define WMI_TDLS_QOS_BE_FLAG           3
#define WMI_TDLS_QOS_ACK_FLAG          4
#define WMI_TDLS_QOS_SP_FLAG           5
#define WMI_TDLS_QOS_MOREDATA_FLAG     7

#define WMI_TDLS_PEER_SET_QOS_FLAG(ppeer_caps,flag) do { \
        (ppeer_caps)->peer_qos |=  (1 << flag);      \
     } while(0)
#define WMI_TDLS_PEER_GET_QOS_FLAG(ppeer_caps,flag)   \
        (((ppeer_caps)->peer_qos & (1 << flag)) >> flag)

#define WMI_SET_TDLS_PEER_VO_UAPSD(ppeer_caps) \
    WMI_TDLS_PEER_SET_QOS_FLAG(ppeer_caps, WMI_TDLS_QOS_VO_FLAG)
#define WMI_GET_TDLS_PEER_VO_UAPSD(ppeer_caps) \
    WMI_TDLS_PEER_GET_QOS_FLAG(ppeer_caps, WMI_TDLS_QOS_VO_FLAG)
#define WMI_SET_TDLS_PEER_VI_UAPSD(ppeer_caps) \
    WMI_TDLS_PEER_SET_QOS_FLAG(ppeer_caps, WMI_TDLS_QOS_VI_FLAG)
#define WMI_GET_TDLS_PEER_VI_UAPSD(ppeer_caps) \
    WMI_TDLS_PEER_GET_QOS_FLAG(ppeer_caps, WMI_TDLS_QOS_VI_FLAG)
#define WMI_SET_TDLS_PEER_BK_UAPSD(ppeer_caps) \
    WMI_TDLS_PEER_SET_QOS_FLAG(ppeer_caps, WMI_TDLS_QOS_BK_FLAG)
#define WMI_GET_TDLS_PEER_BK_UAPSD(ppeer_caps) \
    WMI_TDLS_PEER_GET_QOS_FLAG(ppeer_caps, WMI_TDLS_QOS_BK_FLAG)
#define WMI_SET_TDLS_PEER_BE_UAPSD(ppeer_caps) \
    WMI_TDLS_PEER_SET_QOS_FLAG(ppeer_caps, WMI_TDLS_QOS_BE_FLAG)
#define WMI_GET_TDLS_PEER_BE_UAPSD(ppeer_caps) \
    WMI_TDLS_PEER_GET_QOS_FLAG(ppeer_caps, WMI_TDLS_QOS_BE_FLAG)
#define WMI_SET_TDLS_PEER_ACK_UAPSD(ppeer_caps) \
    WMI_TDLS_PEER_SET_QOS_FLAG(ppeer_caps, WMI_TDLS_QOS_ACK_FLAG)
#define WMI_GET_TDLS_PEER_ACK_UAPSD(ppeer_caps) \
    WMI_TDLS_PEER_GET_QOS_FLAG(ppeer_caps, WMI_TDLS_QOS_ACK_FLAG)
/* SP has 2 bits */
#define WMI_SET_TDLS_PEER_SP_UAPSD(ppeer_caps,val) do { \
     (ppeer_caps)->peer_qos |=  (((val)&0x3) << WMI_TDLS_QOS_SP_FLAG); \
     } while(0)
#define WMI_GET_TDLS_PEER_SP_UAPSD(ppeer_caps) \
    (((ppeer_caps)->peer_qos & (0x3 << WMI_TDLS_QOS_SP_FLAG)) >> WMI_TDLS_QOS_SP_FLAG)

#define WMI_SET_TDLS_PEER_MORE_DATA_ACK_UAPSD(ppeer_caps) \
    WMI_TDLS_PEER_SET_QOS_FLAG(ppeer_caps, WMI_TDLS_QOS_MOREDATA_FLAG)
#define WMI_GET_TDLS_PEER_MORE_DATA_ACK_UAPSD(ppeer_caps) \
    WMI_TDLS_PEER_GET_QOS_FLAG(ppeer_caps, WMI_TDLS_QOS_MOREDATA_FLAG)


#define WMI_TDLS_SELF_SET_QOS_FLAG(pset_cmd,flag) do { \
        (pset_cmd)->tdls_puapsd_mask |=  (1 << flag);      \
    } while(0)
#define WMI_TDLS_SELF_GET_QOS_FLAG(pset_cmd,flag)   \
        (((pset_cmd)->tdls_puapsd_mask & (1 << flag)) >> flag)

#define WMI_SET_TDLS_SELF_VO_UAPSD(pset_cmd) \
    WMI_TDLS_SELF_SET_QOS_FLAG(pset_cmd, WMI_TDLS_QOS_VO_FLAG)
#define WMI_GET_TDLS_SELF_VO_UAPSD(pset_cmd) \
    WMI_TDLS_SELF_GET_QOS_FLAG(pset_cmd, WMI_TDLS_QOS_VO_FLAG)
#define WMI_SET_TDLS_SELF_VI_UAPSD(pset_cmd) \
    WMI_TDLS_SELF_SET_QOS_FLAG(pset_cmd, WMI_TDLS_QOS_VI_FLAG)
#define WMI_GET_TDLS_SELF_VI_UAPSD(pset_cmd) \
    WMI_TDLS_SELF_GET_QOS_FLAG(pset_cmd, WMI_TDLS_QOS_VI_FLAG)
#define WMI_SET_TDLS_SELF_BK_UAPSD(pset_cmd) \
    WMI_TDLS_SELF_SET_QOS_FLAG(pset_cmd, WMI_TDLS_QOS_BK_FLAG)
#define WMI_GET_TDLS_SELF__BK_UAPSD(pset_cmd) \
    WMI_TDLS_SELF_GET_QOS_FLAG(pset_cmd, WMI_TDLS_QOS_BK_FLAG)
#define WMI_SET_TDLS_SELF_BE_UAPSD(pset_cmd) \
    WMI_TDLS_SELF_SET_QOS_FLAG(pset_cmd, WMI_TDLS_QOS_BE_FLAG)
#define WMI_GET_TDLS_SELF_BE_UAPSD(pset_cmd) \
    WMI_TDLS_SELF_GET_QOS_FLAG(pset_cmd, WMI_TDLS_QOS_BE_FLAG)
#define WMI_SET_TDLS_SELF_ACK_UAPSD(pset_cmd) \
    WMI_TDLS_SELF_SET_QOS_FLAG(pset_cmd, WMI_TDLS_QOS_ACK_FLAG)
#define WMI_GET_TDLS_SELF_ACK_UAPSD(pset_cmd) \
    WMI_TDLS_SELF_GET_QOS_FLAG(pset_cmd, WMI_TDLS_QOS_ACK_FLAG)
/* SP has 2 bits */
#define WMI_SET_TDLS_SELF_SP_UAPSD(pset_cmd,val) do { \
     (pset_cmd)->tdls_puapsd_mask |=  (((val)&0x3) << WMI_TDLS_QOS_SP_FLAG); \
    } while(0)
#define WMI_GET_TDLS_SELF_SP_UAPSD(pset_cmd) \
    (((pset_cmd)->tdls_puapsd_mask & (0x3 << WMI_TDLS_QOS_SP_FLAG)) >> WMI_TDLS_QOS_SP_FLAG)

#define WMI_SET_TDLS_SELF_MORE_DATA_ACK_UAPSD(pset_cmd) \
    WMI_TDLS_SELF_SET_QOS_FLAG(pset_cmd, WMI_TDLS_QOS_MOREDATA_FLAG)
#define WMI_GET_TDLS_SELF_MORE_DATA_ACK_UAPSD(pset_cmd) \
    WMI_TDLS_SELF_GET_QOS_FLAG(pset_cmd, WMI_TDLS_QOS_MOREDATA_FLAG)

typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tdls_peer_update_cmd_fixed_param  */
    A_UINT32    tlv_header;
    /** unique id identifying the VDEV */
    A_UINT32                    vdev_id;
    /** peer MAC address */
    wmi_mac_addr                peer_macaddr;
    /** new TDLS state for peer (wmi_tdls_peer_state) */
    A_UINT32                    peer_state;
    /* The TLV for wmi_tdls_peer_capabilities will follow.
     *     wmi_tdls_peer_capabilities  peer_caps;
     */
    /** Followed by the variable length TLV chan_info:
     *  wmi_channel chan_info[] */
} wmi_tdls_peer_update_cmd_fixed_param;

/* WMI_TDLS_SET_OFFCHAN_MODE_CMDID */


/* bitmap  20, 40, 80 or 160 MHz wide channel */
#define WMI_TDLS_OFFCHAN_20MHZ                  0x1   /*  20 MHz wide channel */
#define WMI_TDLS_OFFCHAN_40MHZ                  0x2   /*  40 MHz wide channel */
#define WMI_TDLS_OFFCHAN_80MHZ                  0x4   /*  80 MHz wide channel */
#define WMI_TDLS_OFFCHAN_160MHZ                 0x8   /*  160 MHz wide channel */


enum wmi_tdls_offchan_mode {
    WMI_TDLS_ENABLE_OFFCHANNEL,
    WMI_TDLS_DISABLE_OFFCHANNEL
};

typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tdls_set_offchan_mode_cmd_fixed_param  */
    A_UINT32   tlv_header;
    /** unique id identifying the VDEV */
    A_UINT32   vdev_id;
    /** Enable/Disable TDLS offchannel */
    A_UINT32 offchan_mode;
    /** peer MAC address */
    wmi_mac_addr   peer_macaddr;
    /* Is peer initiator or responder of TDLS setup request */
    A_UINT32 is_peer_responder;
    /* off channel number*/
    A_UINT32 offchan_num;
    /* off channel bandwidth bitmap, e.g. WMI_OFFCHAN_20MHZ */
    A_UINT32 offchan_bw_bitmap;
    /* operating class for offchan */
    A_UINT32 offchan_oper_class;
} wmi_tdls_set_offchan_mode_cmd_fixed_param;

/** TDLS EVENTS */
enum wmi_tdls_peer_notification {
    /** tdls discovery recommended for peer (based
     * on tx bytes per second > tx_discover threshold) */
    WMI_TDLS_SHOULD_DISCOVER,
    /** tdls link tear down recommended for peer
     * due to tx bytes per second below tx_teardown_threshold
     * NB: this notification sent once */
    WMI_TDLS_SHOULD_TEARDOWN,
    /** tx peer TDLS link tear down complete */
    WMI_TDLS_PEER_DISCONNECTED,
    /** TDLS/BT role change notification for connection tracker */
    WMI_TDLS_CONNECTION_TRACKER_NOTIFICATION,
};

enum wmi_tdls_peer_reason {
    /** tdls teardown recommended due to low transmits */
    WMI_TDLS_TEARDOWN_REASON_TX,
    /** tdls link tear down recommended due to poor RSSI */
    WMI_TDLS_TEARDOWN_REASON_RSSI,
    /** tdls link tear down recommended due to offchannel scan */
    WMI_TDLS_TEARDOWN_REASON_SCAN,
    /** tdls peer disconnected due to peer deletion */
    WMI_TDLS_DISCONNECTED_REASON_PEER_DELETE,
    /** tdls peer disconnected due to PTR timeout */
    WMI_TDLS_TEARDOWN_REASON_PTR_TIMEOUT,
    /** tdls peer disconnected due wrong PTR format */
    WMI_TDLS_TEARDOWN_REASON_BAD_PTR,
    /** tdls peer not responding */
    WMI_TDLS_TEARDOWN_REASON_NO_RESPONSE,
    /** tdls entered buffer STA role, TDLS connection tracker needs to handle this */
    WMI_TDLS_ENTER_BUF_STA,
    /** tdls exited buffer STA role, TDLS connection tracker needs to handle this */
    WMI_TDLS_EXIT_BUF_STA,
    /** BT entered busy mode, TDLS connection tracker needs to handle this */
    WMI_TDLS_ENTER_BT_BUSY_MODE,
    /** BT exited busy mode, TDLS connection tracker needs to handle this */
    WMI_TDLS_EXIT_BT_BUSY_MODE,
    /*
     * TDLS module received a scan start event, TDLS connection tracker
     * needs to handle this
     */
    WMI_TDLS_SCAN_STARTED_EVENT,
    /*
     * TDLS module received a scan complete event, TDLS connection tracker
     * needs to handle this
     */
    WMI_TDLS_SCAN_COMPLETED_EVENT,
};

/* WMI_TDLS_PEER_EVENTID */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_tdls_peer_event_fixed_param  */
    A_UINT32    tlv_header;
    /** peer MAC address */
    wmi_mac_addr    peer_macaddr;
    /** TDLS peer status (wmi_tdls_peer_notification)*/
    A_UINT32        peer_status;
    /** TDLS peer reason (wmi_tdls_peer_reason) */
    A_UINT32        peer_reason;
    /** unique id identifying the VDEV */
    A_UINT32        vdev_id;
} wmi_tdls_peer_event_fixed_param;

/* NOTE: wmi_vdev_mcc_bcn_intvl_change_event_fixed_param would be deprecated. Please
 don't use this for any new implementations */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_mcc_bcn_intvl_change_event_fixed_param  */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /* New beacon interval to be used for the specified VDEV suggested by firmware */
    A_UINT32 new_bcn_intvl;
} wmi_vdev_mcc_bcn_intvl_change_event_fixed_param;

/* WMI_RESMGR_ADAPTIVE_OCS_ENABLE_DISABLE_CMDID */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_resmgr_adaptive_ocs_enable_disable_cmd_fixed_param */
    A_UINT32 tlv_header;
    /** 1: enable fw based adaptive ocs,
     *  0: disable fw based adaptive ocs
     */
    A_UINT32 enable;
    /** This field contains the MAC identifier in order to lookup the appropriate OCS instance. */
    union {
        A_UINT32 mac_id; // OBSOLETE - will be removed once all refs are gone
        /** pdev_id for identifying the MAC
         * See macros starting with WMI_PDEV_ID_ for values.
         */
        A_UINT32 pdev_id;
    };
} wmi_resmgr_adaptive_ocs_enable_disable_cmd_fixed_param;

/* WMI_RESMGR_SET_CHAN_TIME_QUOTA_CMDID */
typedef struct {
    /* Frequency of the channel for which the quota is set */
    A_UINT32 chan_mhz;
    /* Requested channel time quota expressed as percentage */
    A_UINT32 channel_time_quota;
} wmi_resmgr_chan_time_quota;

typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_resmgr_set_chan_time_quota_cmd_fixed_param */
    A_UINT32 tlv_header;
    /** number of channel time quota command structures
     * (wmi_resmgr_chan_time_quota) 1 or 2
     */
    A_UINT32 num_chans;
/* This TLV is followed by another TLV of array of bytes
 * A_UINT8 data[];
 * This data array contains
 * num_chans * size of(struct wmi_resmgr_chan_time_quota)
 */
} wmi_resmgr_set_chan_time_quota_cmd_fixed_param;

/* WMI_RESMGR_SET_CHAN_LATENCY_CMDID */
typedef struct {
    /* Frequency of the channel for which the latency is set */
    A_UINT32 chan_mhz;
    /* Requested channel latency in milliseconds */
    A_UINT32 latency;
} wmi_resmgr_chan_latency;

typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_resmgr_set_chan_latency_cmd_fixed_param */
    A_UINT32 tlv_header;
    /** number of channel latency command structures
     * (wmi_resmgr_chan_latency) 1 or 2
     */
    A_UINT32 num_chans;
/* This TLV is followed by another TLV of array of bytes
 * A_UINT8 data[];
 * This data array contains
 * num_chans * size of(struct wmi_resmgr_chan_latency)
 */
} wmi_resmgr_set_chan_latency_cmd_fixed_param;

/* WMI_STA_SMPS_FORCE_MODE_CMDID */

/** STA SMPS Forced Mode */
typedef enum {
    WMI_SMPS_FORCED_MODE_NONE = 0,
    WMI_SMPS_FORCED_MODE_DISABLED,
    WMI_SMPS_FORCED_MODE_STATIC,
    WMI_SMPS_FORCED_MODE_DYNAMIC
} wmi_sta_smps_forced_mode;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_sta_smps_force_mode_cmd_fixed_param */
    A_UINT32 tlv_header;
    /** Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /** The mode of SMPS that is to be forced in the FW. */
    A_UINT32 forced_mode;
} wmi_sta_smps_force_mode_cmd_fixed_param;

/** wlan HB commands */
#define WMI_WLAN_HB_ITEM_UDP            0x1
#define WMI_WLAN_HB_ITEM_TCP            0x2
#define WMI_WLAN_HB_MAX_FILTER_SIZE     32 /* should be equal to WLAN_HB_MAX_FILTER_SIZE, must be a multiple of 4 bytes */

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_hb_set_enable_cmd_fixed_param */
    A_UINT32 tlv_header;     /** TLV header*/
    A_UINT32 vdev_id;        /** Vdev ID */
    A_UINT32 enable;         /** 1: Enable, 0: Disable`*/
    A_UINT32 item;           /** 1: UDP, 2: TCP */
    A_UINT32 session;        /** Session ID from HOST */
} wmi_hb_set_enable_cmd_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_hb_set_tcp_params_cmd_fixed_param */
    A_UINT32 tlv_header;      /** TLV header*/
    A_UINT32 vdev_id;         /** Vdev ID */
    A_UINT32 srv_ip;          /** Server IP address (IPv4) */
    A_UINT32 dev_ip;          /** Device IP address (IPv4) */
    A_UINT32 seq;             /** TCP Sequence no */
    A_UINT32 src_port;        /** Source port */
    A_UINT32 dst_port;        /** Destination port */
    A_UINT32 interval;        /** Keep alive interval */
    A_UINT32 timeout;         /** Timeout if keep alive fails */
    A_UINT32 session;         /** Session ID from HOST */
    wmi_mac_addr gateway_mac; /** Server Mac Address */
} wmi_hb_set_tcp_params_cmd_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_hb_set_tcp_pkt_filter_cmd_fixed_param */
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 length;
    A_UINT32 offset;
    A_UINT32 session;
    A_UINT8  filter[WMI_WLAN_HB_MAX_FILTER_SIZE];
} wmi_hb_set_tcp_pkt_filter_cmd_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_hb_set_udp_params_cmd_fixed_param */
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 srv_ip;
    A_UINT32 dev_ip;
    A_UINT32 src_port;
    A_UINT32 dst_port;
    A_UINT32 interval;
    A_UINT32 timeout;
    A_UINT32 session;
    wmi_mac_addr gateway_mac;
} wmi_hb_set_udp_params_cmd_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_hb_set_udp_pkt_filter_cmd_fixed_param */
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 length;
    A_UINT32 offset;
    A_UINT32 session;
    A_UINT8  filter[WMI_WLAN_HB_MAX_FILTER_SIZE];
} wmi_hb_set_udp_pkt_filter_cmd_fixed_param;

/** wlan HB events */
typedef enum {
    WMI_WLAN_HB_REASON_UNKNOWN      = 0,
    WMI_WLAN_HB_REASON_TCP_TIMEOUT  = 1,
    WMI_WLAN_HB_REASON_UDP_TIMEOUT  = 2,
} WMI_HB_WAKEUP_REASON;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_hb_ind_event_fixed_param */
    A_UINT32 vdev_id;    /** unique id identifying the VDEV */
    A_UINT32 session;    /** Session ID from HOST */
    A_UINT32 reason;     /** wakeup reason */
} wmi_hb_ind_event_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_oic_set_enable_cmd_fixed_param */
    A_UINT32 tlv_header;           /** TLV Header */
    A_UINT32 vdev_id;              /** Vdev ID */
    A_UINT32 session;              /** Session number from the HOST */
    A_UINT32 srv_ip;               /** IPv4 address of the OCF server */
    A_UINT32 dev_ip;               /** IPv4 address of the device */
    A_UINT32 tcp_tx_seq;           /** TCP sequence number */
    A_UINT32 src_port;             /** Source port */
    A_UINT32 dst_port;             /** Destination port */
    A_UINT32 protocol;             /** Protocol used: TCP:0, UDP:1 */
    A_UINT32 wlan_hb_session;      /** Linked WLAN HB session. If a keepalive is configured for the TCP session, the session ID of the TCP keepalive */
    A_UINT32 timeout_retries;      /** timeout[31:16]: TCP ACK timeout, time to wait for a TCP ACK in ms
                                       retries[15:0]: Number of TCP level retries of OIC ping request */
    wmi_mac_addr peer_macaddr;     /** MAC address of the OCF server */
    A_UINT32 oic_ping_pkt0;        /** OIC ping packet content [Byte03:Byte00] */
    A_UINT32 oic_ping_pkt1;        /** OIC ping packet content [Byte07:Byte04] */
    A_UINT32 oic_ping_pkt2;        /** OIC ping packet content [Byte11:Byte08] */
    A_UINT32 oic_ping_pkt3;        /** OIC ping packet content [Byte15:Byte12] */

    A_UINT32 tls_cipher_suite_version; /** Cipher suite [31:16] as defined in https://www.iana.org/assignments/tls-parameters/tls-parameters.xhtml
                                       TLS version  [15:00] */
    A_UINT32 tls_tx_seq0;          /** Tx sequence number [31:00], incremented after every TLS packet transmission */
    A_UINT32 tls_tx_seq1;          /** Tx sequence number [63:32] */
    A_UINT32 tls_rx_seq0;          /** Rx sequence number [31:00], incremented after every TLS packet reception */
    A_UINT32 tls_rx_seq1;          /** Rx sequence number [63:32] */
    A_UINT32 tls_tx_key0;          /** client_write_key[Byte03:Byte00] refer Section 6.3 RFC 5246 */
    A_UINT32 tls_tx_key1;          /** client_write_key[Byte07:Byte04] */
    A_UINT32 tls_tx_key2;          /** client_write_key[Byte11:Byte08] */
    A_UINT32 tls_tx_key3;          /** client_write_key[Byte15:Byte12] */
    A_UINT32 tls_rx_key0;          /** server_write_key[Byte03:Byte00] */
    A_UINT32 tls_rx_key1;          /** server_write_key[Byte07:Byte04] */
    A_UINT32 tls_rx_key2;          /** server_write_key[Byte11:Byte08] */
    A_UINT32 tls_rx_key3;          /** server_write_key[Byte15:Byte12] */
    A_UINT32 tls_MAC_write_key0;   /** client_write_MAC_key[Byte03:Byte00] refer Section 6.3 RFC 5246 */
    A_UINT32 tls_MAC_write_key1;   /** client_write_MAC_key[Byte07:Byte04] */
    A_UINT32 tls_MAC_write_key2;   /** client_write_MAC_key[Byte11:Byte08] */
    A_UINT32 tls_MAC_write_key3;   /** client_write_MAC_key[Byte15:Byte12] */
    A_UINT32 tls_MAC_write_key4;   /** client_write_MAC_key[Byte19:Byte16] */
    A_UINT32 tls_MAC_write_key5;   /** client_write_MAC_key[Byte23:Byte20] */
    A_UINT32 tls_MAC_write_key6;   /** client_write_MAC_key[Byte27:Byte24] */
    A_UINT32 tls_MAC_write_key7;   /** client_write_MAC_key[Byte31:Byte28] */
    A_UINT32 tls_MAC_read_key0;    /** server_write_MAC_key[Byte03:Byte00] refer Section 6.3 RFC 5246 */
    A_UINT32 tls_MAC_read_key1;    /** server_write_MAC_key[Byte07:Byte04] */
    A_UINT32 tls_MAC_read_key2;    /** server_write_MAC_key[Byte11:Byte08] */
    A_UINT32 tls_MAC_read_key3;    /** server_write_MAC_key[Byte15:Byte12] */
    A_UINT32 tls_MAC_read_key4;    /** server_write_MAC_key[Byte19:Byte16] */
    A_UINT32 tls_MAC_read_key5;    /** server_write_MAC_key[Byte23:Byte20] */
    A_UINT32 tls_MAC_read_key6;    /** server_write_MAC_key[Byte27:Byte24] */
    A_UINT32 tls_MAC_read_key7;    /** server_write_MAC_key[Byte31:Byte28] */
    A_UINT32 tls_client_IV0;       /** CBC Mode: CBC_residue [Byte03:Byte00] refer section 6.2.3.2. CBC Block Cipher in RFC 5246
                                   GCM Mode: GCMNonce.salt [Byte03:Byte00] refer Section 3 of RFC 5288 */
    A_UINT32 tls_client_IV1;       /** CBC Mode: CBC_residue [Byte7:Byte4] */
    A_UINT32 tls_client_IV2;       /** CBC Mode: CBC_residue [Byte11:Byte8] */
    A_UINT32 tls_client_IV3;       /** CBC Mode: CBC_residue [Byte15:Byte12] */
    A_UINT32 tls_server_IV0;       /** CBC Mode: CBC_residue [Byte3:Byte0] refer section 6.2.3.2. CBC Block Cipher in RFC 5246
                                   GCM Mode: GCMNonce.salt [Byte4: Byte0] refer Section 3 of RFC 5288 */
    A_UINT32 tls_server_IV1;       /** CBC Mode: CBC_residue [Byte7:Byte4] */
    A_UINT32 tls_server_IV2;       /** CBC Mode: CBC_residue [Byte11:Byte8] */
    A_UINT32 tls_server_IV3;       /** CBC Mode: CBC_residue [Byte15:Byte12] */
} wmi_oic_ping_offload_params_cmd_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_oic_set_enable_cmd_fixed_param */
    A_UINT32 tlv_header; /** TLV Header*/
    A_UINT32 vdev_id;    /** Interface number */
    A_UINT32 session;    /** Session ID*/
    A_UINT32 enable;     /** 1: Enable , 0: Disable */
} wmi_oic_ping_offload_set_enable_cmd_fixed_param;

/** wlan OIC events */
typedef enum {
    WMI_WLAN_OIC_REASON_UNKNOWN      = 0, /** Unknown */
    WMI_WLAN_OIC_REASON_HOST_WAKE    = 1, /** No error , but host is woken up due to other reasons */
    WMI_WLAN_OIC_REASON_TCP_TIMEOUT  = 2, /** TCP Timeout */
    WMI_WLAN_OIC_REASON_PING_TIMEOUT = 3, /** OIC Ping resposnse timeout */
    WMI_WLAN_OIC_REASON_TLS_ERROR    = 4, /** TLS decryption error */
} WMI_OIC_WAKEUP_REASON;

typedef struct {
    A_UINT32 tlv_header;           /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_oic_ind_event_fixed_param */
    A_UINT32 vdev_id;              /** unique id identifying the VDEV */
    A_UINT32 session;              /** Session ID from driver */
    A_UINT32 reason;               /** wakeup reason as per WMI_OIC_WAKEUP_REASON */
    A_UINT32 tcp_tx_seq;           /** Current TCP sequence number */
    A_UINT32 tcp_ack_num;          /** Current TCP Acknowledgement number */
    A_UINT32 tls_tx_seq0;          /** Tx sequence number [31:00], incremented after every TLS packet transmission */
    A_UINT32 tls_tx_seq1;          /** Tx sequence number [63:32] */
    A_UINT32 tls_rx_seq0;          /** Rx sequence number [31:00], incremented after every TLS packet reception */
    A_UINT32 tls_rx_seq1;          /** Rx sequence number [63:32] */
    A_UINT32 tls_client_IV0;       /** CBC Mode: CBC_residue [Byte03:Byte00] refer section 6.2.3.2. CBC Block Cipher in RFC 5246 */
    A_UINT32 tls_client_IV1;       /** CBC Mode: CBC_residue [Byte7:Byte4] */
    A_UINT32 tls_client_IV2;       /** CBC Mode: CBC_residue [Byte11:Byte8] */
    A_UINT32 tls_client_IV3;       /** CBC Mode: CBC_residue [Byte15:Byte12] */
    A_UINT32 tls_server_IV0;       /** CBC Mode: CBC_residue [Byte3:Byte0] refer section 6.2.3.2. CBC Block Cipher in RFC 5246 */
    A_UINT32 tls_server_IV1;       /** CBC Mode: CBC_residue [Byte7:Byte4] */
    A_UINT32 tls_server_IV2;       /** CBC Mode: CBC_residue [Byte11:Byte8] */
    A_UINT32 tls_server_IV3;       /** CBC Mode: CBC_residue [Byte15:Byte12] */
} wmi_oic_ping_handoff_event;

typedef struct {
   /** TLV tag and len; tag equals
    * WMITLV_TAG_STRUC_wmi_dhcp_lease_renew_offload_fixed_param */
    A_UINT32 tlv_header;     /** TLV Header*/
    A_UINT32 vdev_id;        /** Vdev ID */
    A_UINT32 enable;         /** 1: Enable 0: Disable*/
    A_UINT32 srv_ip;         /** DHCP Server IP address (IPv4) */
    A_UINT32 client_ip;      /** Device IP address (IPv4) */
    wmi_mac_addr srv_mac;    /** DHCP Server MAC address */
    A_UINT32 parameter_list; /** Optional Parameter list. RFC 1533 gives the complete set of options defined for use with DHCP */
} wmi_dhcp_lease_renew_offload_cmd_fixed_param;

/** WLAN DHCP Lease Renew Events */
typedef enum {
    WMI_WLAN_DHCP_RENEW_REASON_UNKNOWN  = 0, /** Unknown */
    WMI_WLAN_DHCP_RENEW_REASON_ACK_TIMEOUT  = 1, /** DHCP ACK Timeout */
    WMI_WLAN_DHCP_RENEW_REASON_NACK     = 2, /** DHCP error */
} WMI_DHCP_RENEW_WAKEUP_REASON;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_dhcp_renew_ind_event_fixed_param */
    A_UINT32 vdev_id;    /** unique id identifying the VDEV */
    A_UINT32 reason;     /** wakeup reason as per enum WMI_DHCP_RENEW_WAKEUP_REASON*/
} wmi_dhcp_lease_renew_event;

/** WMI_STA_SMPS_PARAM_CMDID */
typedef enum {
    /** RSSI threshold to enter Dynamic SMPS mode from inactive mode */
    WMI_STA_SMPS_PARAM_UPPER_RSSI_THRESH = 0,
    /** RSSI threshold to enter Stalled-D-SMPS mode from D-SMPS mode or
     * to enter D-SMPS mode from Stalled-D-SMPS mode */
    WMI_STA_SMPS_PARAM_STALL_RSSI_THRESH = 1,
    /** RSSI threshold to disable SMPS modes */
    WMI_STA_SMPS_PARAM_LOWER_RSSI_THRESH = 2,
    /** Upper threshold for beacon-RSSI. Used to reduce RX chainmask. */
    WMI_STA_SMPS_PARAM_UPPER_BRSSI_THRESH = 3,
    /** Lower threshold for beacon-RSSI. Used to increase RX chainmask. */
    WMI_STA_SMPS_PARAM_LOWER_BRSSI_THRESH = 4,
    /** Enable/Disable DTIM 1chRx feature */
    WMI_STA_SMPS_PARAM_DTIM_1CHRX_ENABLE = 5
} wmi_sta_smps_param;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_sta_smps_param_cmd_fixed_param */
    A_UINT32 tlv_header;
    /** Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /** SMPS parameter (see wmi_sta_smps_param) */
    A_UINT32 param;
    /** Value of SMPS parameter */
    A_UINT32 value;
} wmi_sta_smps_param_cmd_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_mcc_sched_sta_traffic_stats */
    A_UINT32 tlv_header;
    /* TX stats */
    A_UINT32 txBytesPushed;
    A_UINT32 txPacketsPushed;
    /* RX stats */
    A_UINT32 rxBytesRcvd;
    A_UINT32 rxPacketsRcvd;
    A_UINT32 rxTimeTotal;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
} wmi_mcc_sched_sta_traffic_stats;

typedef struct {
    /** TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_mcc_sched_traffic_stats_cmd_fixed_param */
    A_UINT32 tlv_header;
    /** Duration over which the host stats were collected */
    A_UINT32 duration;
    /** Number of stations filled in following stats array */
    A_UINT32 num_sta;
    /* Following this struct are the TLVs:
     * wmi_mcc_sched_sta_traffic_stats mcc_sched_sta_traffic_stats_list;
     */
} wmi_mcc_sched_traffic_stats_cmd_fixed_param;

typedef struct
{
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals  WMITLV_TAG_STRUC_wmi_batch_scan_enable_cmd_fixed_param */
    /* unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /*Batch scan enable command parameters*/
    A_UINT32 scanInterval;
    A_UINT32 numScan2Batch;
    A_UINT32 bestNetworks;
    A_UINT32 rfBand;
    A_UINT32 rtt;
} wmi_batch_scan_enable_cmd_fixed_param;

typedef struct
{
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_batch_scan_enabled_event_fixed_param  */
    A_UINT32 supportedMscan;
} wmi_batch_scan_enabled_event_fixed_param;

typedef struct
{
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals  WMITLV_TAG_STRUC_wmi_batch_scan_disable_cmd_fixed_param */
/* unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    A_UINT32   param;
} wmi_batch_scan_disable_cmd_fixed_param;

typedef struct
{
    A_UINT32 tlv_header;     /* TLV tag and len; tag equals  WMITLV_TAG_STRUC_wmi_batch_scan_trigger_result_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    A_UINT32 param;
} wmi_batch_scan_trigger_result_cmd_fixed_param;

typedef struct
{
    A_UINT32 tlv_header;
    wmi_mac_addr   bssid;     /* BSSID */
    wmi_ssid   ssid;     /* SSID */
    A_UINT32   ch;           /* Channel */
    A_UINT32   rssi;         /* RSSI or Level */
    /* Timestamp when Network was found. Used to calculate age based on timestamp in GET_RSP msg header */
    A_UINT32  timestamp;
} wmi_batch_scan_result_network_info;

typedef struct
{
    A_UINT32 tlv_header;
    A_UINT32 scanId;                         /* Scan List ID. */
    /* No of AP in a Scan Result. Should be same as bestNetwork in SET_REQ msg */
    A_UINT32 numNetworksInScanList;
    A_UINT32 netWorkStartIndex;  /* indicate the start index of network info*/
} wmi_batch_scan_result_scan_list;

#define LPI_IE_BITMAP_BSSID                  0x00000001     // if this bit is set, bssid of the scan response frame is sent as the first IE in the data buffer sent to LOWI LP.
#define LPI_IE_BITMAP_IS_PROBE               0x00000002     // send true or false based on scan response frame being a Probe Rsp or not
#define LPI_IE_BITMAP_SSID                   0x00000004     // send ssid from received scan response frame
#define LPI_IE_BITMAP_RSSI                   0x00000008     // send RSSI value reported by HW for the received scan response after adjusting with noise floor
#define LPI_IE_BITMAP_CHAN                   0x00000010     // send channel number from the received scan response
#define LPI_IE_BITMAP_AP_TX_PWR              0x00000020     // send Tx power from TPC IE of scan rsp
#define LPI_IE_BITMAP_TX_RATE                0x00000040     // send rate of the received frame as reported by HW.
#define LPI_IE_BITMAP_80211_MC_SUPPORT       0x00000080     // send true or false based on the received scan rsp was from a 11mc supported AP or not.
#define LPI_IE_BITMAP_TSF_TIMER_VALUE        0x00000100     // send timestamp reported in the received scan rsp frame.
#define LPI_IE_BITMAP_AGE_OF_MEASUREMENT     0x00000200     // (current system time - received time) = duration of time scan rsp frame data is kept in the buffer before sending to LOWI LP.
/*
 * TEMPORARY alias of incorrect old name the correct name.
 * This alias will be removed once all references to the old name have been fixed.
 */
#define LPI_IE_BITMAP_AGE_OF_MESAUREMENT LPI_IE_BITMAP_AGE_OF_MEASUREMENT
#define LPI_IE_BITMAP_CONN_STATUS            0x00000400     // If an infra STA is active and connected to an AP, true value is sent else false.
#define LPI_IE_BITMAP_MSAP_IE                0x00000800     // info on the vendor specific proprietary IE MSAP
#define LPI_IE_BITMAP_SEC_STATUS             0x00001000     // we indicate true or false based on if the AP has WPA or RSN security enabled
#define LPI_IE_BITMAP_DEVICE_TYPE            0x00002000     // info about the beacons coming from an AP or P2P or NAN device.
#define LPI_IE_BITMAP_CHAN_IS_PASSIVE        0x00004000     // info on whether the scan rsp was received from a passive channel
#define LPI_IE_BITMAP_DWELL_TIME             0x00008000     // send the scan dwell time of the channel on which the current scan rsp frame was received.
#define LPI_IE_BITMAP_BAND_CENTER_FREQ1      0x00010000     // the center frequencies in case AP is supporting wider channels than 20 MHz
#define LPI_IE_BITMAP_BAND_CENTER_FREQ2      0x00020000     // same as above
#define LPI_IE_BITMAP_PHY_MODE               0x00040000     // PHY mode indicates a, b, ,g, ac and other combinations
#define LPI_IE_BITMAP_SCAN_MODULE_ID         0x00080000     // scan module id indicates the scan client who originated the scan
#define LPI_IE_BITMAP_SCAN_ID                0x00100000     // extscan inserts the scan cycle count for this value; other scan clients can insert the scan id of the scan, if needed.
#define LPI_IE_BITMAP_FLAGS                  0x00200000     // reserved as a bitmap to indicate more scan information; one such use being to indicate if the on-going scan is interrupted or not
#define LPI_IE_BITMAP_CACHING_REQD           0x00400000     // extscan will use this field to indicate if this frame info needs to be cached in LOWI LP or not
#define LPI_IE_BITMAP_REPORT_CONTEXT_HUB     0x00800000     // extscan will use this field to indicate to LOWI LP whether to report result to context hub or not.
#define LPI_IE_BITMAP_CHRE_RADIO_CHAIN       0x01000000     /* include radio chain and rssi per chain information if this bit is set - for CHRE */

/* 0x02000000, 0x04000000, and 0x08000000 are unused / available */

#define LPI_IE_BITMAP_CHRE_ESS               0x10000000     /* ESS capability info for CHRE */
#define LPI_IE_BITMAP_CHRE_SEC_MODE          0x20000000     /* Security capability info for CHRE */
#define LPI_IE_BITMAP_CHRE_SUPPORTED_RATE    0x40000000     /* Highest MCS corresponding NCC for TX and RX */
#define LPI_IE_BITMAP_ALL                    0xFFFFFFFF

typedef struct {
    A_UINT32 tlv_header;
    /**A_BOOL indicates LPI mgmt snooping enable/disable*/
    A_UINT32 enable;
    /**LPI snooping mode*/
    A_UINT32 snooping_mode;
    /** LPI interested IEs in snooping context */
    A_UINT32 ie_bitmap;
} wmi_lpi_mgmt_snooping_config_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_start_scan_cmd_fixed_param */
    /** Scan ID */
    A_UINT32 scan_id;
    /** Scan requestor ID */
    A_UINT32 scan_req_id;
    /** VDEV id(interface) that is requesting scan */
    A_UINT32 vdev_id;
    /** LPI interested IEs in scan context */
    A_UINT32 ie_bitmap;
    /** Scan Priority, input to scan scheduler */
    A_UINT32 scan_priority;
    /** dwell time in msec on active channels */
    A_UINT32 dwell_time_active;
    /** dwell time in msec on passive channels */
    A_UINT32 dwell_time_passive;
    /** min time in msec on the BSS channel,only valid if atleast one VDEV is active*/
    A_UINT32 min_rest_time;
    /** max rest time in msec on the BSS channel,only valid if at least one VDEV is active*/
    /** the scanner will rest on the bss channel at least min_rest_time. after min_rest_time the scanner
     *  will start checking for tx/rx activity on all VDEVs. if there is no activity the scanner will
     *  switch to off channel. if there is activity the scanner will let the radio on the bss channel
     *  until max_rest_time expires.at max_rest_time scanner will switch to off channel
     *  irrespective of activity. activity is determined by the idle_time parameter.
     */
    A_UINT32 max_rest_time;
    /** time before sending next set of probe requests.
     *   The scanner keeps repeating probe requests transmission with period specified by repeat_probe_time.
     *   The number of probe requests specified depends on the ssid_list and bssid_list
     */
    A_UINT32 repeat_probe_time;
    /** time in msec between 2 consequetive probe requests with in a set. */
    A_UINT32 probe_spacing_time;
    /** data inactivity time in msec on bss channel that will be used by scanner for measuring the inactivity  */
    A_UINT32 idle_time;
    /** maximum time in msec allowed for scan  */
    A_UINT32 max_scan_time;
    /** delay in msec before sending first probe request after switching to a channel */
    A_UINT32 probe_delay;
    /** Scan control flags */
    A_UINT32 scan_ctrl_flags;
    /** Burst duration time in msec*/
    A_UINT32 burst_duration;

    /** # if channels to scan. In the TLV channel_list[] */
    A_UINT32 num_chan;
    /** number of bssids. In the TLV bssid_list[] */
    A_UINT32 num_bssid;
    /** number of ssid. In the TLV ssid_list[] */
    A_UINT32 num_ssids;
    /** number of bytes in ie data. In the TLV ie_data[] */
    A_UINT32 ie_len;
    /** Scan control flags extended (see WMI_SCAN_FLAG_EXT_xxx) */
    A_UINT32 scan_ctrl_flags_ext;

/**
 * TLV (tag length value ) parameters follow the scan_cmd
 * structure. The TLV's are:
 *     A_UINT32 channel_list[];
 *     wmi_ssid ssid_list[];
 *     wmi_mac_addr bssid_list[];
 *     A_UINT8 ie_data[];
 */
} wmi_lpi_start_scan_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_stop_scan_cmd_fixed_param */
    /** Scan requestor ID */
    A_UINT32 scan_req_id;
    /** Scan ID */
    A_UINT32 scan_id;
    /**
     * Req Type
     * req_type should be WMI_SCAN_STOP_ONE, WMI_SCN_STOP_VAP_ALL or WMI_SCAN_STOP_ALL
     * WMI_SCAN_STOP_ONE indicates to stop a specific scan with scan_id
     * WMI_SCN_STOP_VAP_ALL indicates to stop all scan requests on a specific vDev with vdev_id
     * WMI_SCAN_STOP_ALL indicates to stop all scan requests in both Scheduler's queue and Scan Engine
     */
    A_UINT32 req_type;
    /**
     * vDev ID
     * used when req_type equals to WMI_SCN_STOP_VAP_ALL, it indexed the vDev on which to stop the scan
     */
    A_UINT32 vdev_id;
} wmi_lpi_stop_scan_cmd_fixed_param;

typedef enum {
   WMI_LPI_DEVICE_TYPE_AP = 1,
   WMI_LPI_DEVICE_TYPE_P2P = 2,
   WMI_LPI_DEVICE_TYPE_NAN = 3,
}wmi_lpi_device_type;

typedef struct
{
    A_UINT32 tlv_header;
    /** Scan requestor ID */
    A_UINT32 scan_req_id;
    A_UINT32 ie_bitmap;
    A_UINT32 data_len;
} wmi_lpi_result_event_fixed_param;

typedef enum {
   /** User scan Request completed */
   WMI_LPI_STATUS_SCAN_REQ_COMPLED = 0,
   /** User Request was never serviced */
   WMI_LPI_STATUS_DROPPED_REQ = 1,
   /** Illegal channel Req */
   WMI_LPI_STATUS_ILLEGAL_CHAN_REQ = 2,
   /** Illegal Operation Req */
   WMI_LPI_STATUS_ILLEGAL_OPER_REQ = 3,
   /** Request Aborted */
   WMI_LPI_STATUS_REQ_ABORTED = 4,
   /** Request Timed Out */
   WMI_LPI_STATUS_REQ_TIME_OUT = 5,
   /** Medium Busy, already there
    * is a scan is going on */
   WMI_LPI_STATUS_MEDIUM_BUSY = 6,
   /** Extscan is the scan client whose scan complete event is triggered */
   WMI_LPI_STATUS_EXTSCAN_CYCLE_AND_SCAN_REQ_COMPLETED = 7,
}wmi_lpi_staus;

typedef struct
{
    A_UINT32      tlv_header;
    wmi_lpi_staus status;
    /** Scan requestor ID */
    A_UINT32      scan_req_id;
}  wmi_lpi_status_event_fixed_param;


typedef struct
{
    A_UINT32      tlv_header;
    wmi_mac_addr  bssid;
    wmi_ssid      ssid;
    A_UINT32      freq;
    A_UINT32      rssi;
    A_UINT32      vdev_id;
}  wmi_lpi_handoff_event_fixed_param;

typedef struct
{
    A_UINT32 tlv_header;
    A_UINT32 timestamp;   /*timestamp of batch scan event*/
    A_UINT32 numScanLists;  /*number of scan in this event*/
    A_UINT32 isLastResult;  /*is this event a last event of the whole batch scan*/
}  wmi_batch_scan_result_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_p2p_noa_event_fixed_param  */
    A_UINT32 vdev_id;
    /* This TLV is followed by p2p_noa_info for vdev :
     *     wmi_p2p_noa_info p2p_noa_info;
     */
} wmi_p2p_noa_event_fixed_param;

#define WMI_RFKILL_CFG_RADIO_LEVEL_OFFSET      6
#define WMI_RFKILL_CFG_RADIO_LEVEL_MASK      0x1

#define WMI_RFKILL_CFG_GPIO_PIN_NUM_OFFSET      0
#define WMI_RFKILL_CFG_GPIO_PIN_NUM_MASK      0x3f

#define WMI_RFKILL_CFG_PIN_AS_GPIO_OFFSET      7
#define WMI_RFKILL_CFG_PIN_AS_GPIO_MASK      0xf

typedef struct {
    /** TLV tag and len; tag equals
     * */
    A_UINT32 tlv_header;
    /** gpip pin number */
    A_UINT32 gpio_pin_num;
    /** gpio interupt type */
    A_UINT32 int_type;
    /** RF radio status */
    A_UINT32 radio_state;
} wmi_rfkill_mode_param;

typedef enum {
    WMI_SET_LED_SYS_POWEROFF,
    WMI_SET_LED_SYS_S3_SUSPEND,
    WMI_SET_LED_SYS_S4_S5,
    WMI_SET_LED_SYS_DRIVER_DISABLE,
    WMI_SET_LED_SYS_WAKEUP,
    WMI_SET_LED_SYS_ALWAYS_ON, //just for test!
    WMI_SET_LED_SYS_POWERON,
} wmi_led_sys_state_param;

typedef enum {
    WMI_CONFIG_LED_TO_VDD = 0,
    WMI_CONFIG_LED_TO_GND = 1,
} wmi_config_led_connect_type;

typedef enum {
   WMI_CONFIG_LED_NOT_WITH_BT = 0,
   WMI_CONFIG_LED_WITH_BT = 1,
} wmi_config_led_with_bt_flag;

typedef enum {
   WMI_CONFIG_LED_DISABLE = 0,
   WMI_CONFIG_LED_ENABLE  = 1,
} wmi_config_led_enable_flag;

typedef enum {
    WMI_CONFIG_LED_HIGH_UNSPECIFIED = 0,
    WMI_CONFIG_LED_HIGH_OFF         = 1,
    WMI_CONFIG_LED_HIGH_ON          = 2,
} wmi_config_led_on_flag;

typedef enum {
    WMI_CONFIG_LED_UNSPECIFIED = 0,
    WMI_CONFIG_LED_ON          = 1,
    WMI_CONFIG_LED_OFF         = 2,
    WMI_CONFIG_LED_DIM         = 3,
    WMI_CONFIG_LED_BLINK       = 4,
    WMI_CONFIG_LED_TXRX        = 5,
} wmi_config_led_operation_type;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_pdev_set_led_config_cmd_fixed_param   */
    A_UINT32 tlv_header;
    /* Set GPIO pin */
    A_UINT32 led_gpio_pin;
    /* Set connect type defined in wmi_config_led_connect_type */
    A_UINT32 connect_type;
    /* Set flag defined in wmi_config_led_with_bt_flag*/
    A_UINT32 with_bt;
    /* Set LED enablement defined in wmi_config_led_enable_flag */
    A_UINT32 led_enable;
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
    /* see wmi_config_led_operation_type enum */
    A_UINT32 led_operation_type;
    /* see wmi_config_led_on_flag enum */
    A_UINT32 led_on_flag;      /* configure high/low on/off sense */
    A_UINT32 led_on_interval;  /* for blink function; unit: ms */
    A_UINT32 led_off_interval; /* for blink function; unit: ms */
    A_UINT32 led_repeat_cnt;   /* for blink function: how many blinks */
} wmi_pdev_set_led_config_cmd_fixed_param;

#define WMI_WNTS_CFG_GPIO_PIN_NUM_OFFSET 0
#define WMI_WNTS_CFG_GPIO_PIN_NUM_MASK   0xff

/** WMI_PEER_INFO_REQ_CMDID
 *   Request FW to provide peer info */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_peer_info_req_cmd_fixed_param   */
    A_UINT32 tlv_header;
    /** In order to get the peer info for a single peer, host shall
     *  issue the peer_mac_address of that peer. For getting the
     *  info all peers, the host shall issue 0xFFFFFFFF as the mac
     *  address. The firmware will return the peer info for all the
     *  peers on the specified vdev_id */
    wmi_mac_addr peer_mac_address;
    /** vdev id */
    A_UINT32 vdev_id;
} wmi_peer_info_req_cmd_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_peer_info */
    A_UINT32 tlv_header;
    /** mac addr of the peer */
    wmi_mac_addr peer_mac_address;
    /** data_rate of the peer */
    A_UINT32 data_rate;
    /** rssi of the peer */
    A_UINT32 rssi;
    /** tx fail count */
    A_UINT32 tx_fail_cnt;
} wmi_peer_info;

/** FW response with the peer info */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_peer_info_event_fixed_param   */
    A_UINT32 tlv_header;
    /** number of peers in peer_info */
    A_UINT32 num_peers;
    /* Set to 1 only if vdev_id field is valid */
    A_UINT32 valid_vdev_id;
    /* VDEV to which the peer belongs to */
    A_UINT32 vdev_id;
    /* This TLV is followed by another TLV of array of structs
     * wmi_peer_info peer_info[];
     */
} wmi_peer_info_event_fixed_param;

/**
 * WMI_PEER_ANTDIV_INFO_REQ_CMDID
 * Request FW to provide peer info
 */
typedef struct {
    /**
     * TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_peer_antdiv_info_req_cmd_fixed_param
     */
    A_UINT32 tlv_header;
    /**
     * In order to get the peer antdiv info for a single peer, host shall
     * issue the peer_mac_address of that peer. For getting the
     * info all peers, the host shall issue 0xFFFFFFFF as the mac
     * address. The firmware will return the peer info for all the
     * peers on the specified vdev_id
     */
    wmi_mac_addr peer_mac_address;
    /** vdev id */
    A_UINT32 vdev_id;
} wmi_peer_antdiv_info_req_cmd_fixed_param;

/** FW response with the peer antdiv info */
typedef struct {
    /** TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_peer_antdiv_info_event_fixed_param
     */
    A_UINT32 tlv_header;
    /** number of peers in peer_info */
    A_UINT32 num_peers;
    /** VDEV to which the peer belongs to */
    A_UINT32 vdev_id;
    /**
     * This TLV is followed by another TLV of array of structs
     * wmi_peer_antdiv_info peer_antdiv_info[];
     */
} wmi_peer_antdiv_info_event_fixed_param;

typedef struct {
    /**
     * TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_peer_antdiv_info
     */
    A_UINT32 tlv_header;
    /** mac addr of the peer */
    wmi_mac_addr peer_mac_address;
    /**
     * per chain rssi of the peer, for up to 8 chains.
     * Each chain's entry reports the RSSI for different bandwidths:
     * bits 7:0   -> primary 20 MHz
     * bits 15:8  -> secondary 20 MHz of  40 MHz channel (if applicable)
     * bits 23:16 -> secondary 40 MHz of  80 MHz channel (if applicable)
     * bits 31:24 -> secondary 80 MHz of 160 MHz channel (if applicable)
     * Each of these 8-bit RSSI reports is in dB units, with respect to
     * the noise floor.
     * 0x80 means invalid.
     * All unused bytes within used chain_rssi indices shall be set
     * to 0x80.
     * All unused chain_rssi indices shall be set to 0x80808080.
     */
    A_INT32 chain_rssi[8];
} wmi_peer_antdiv_info;

typedef struct {
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_msduq_qdepth_thresh_update */

    /** tid_number */
    A_UINT32 tid_num;

    /** msduq_mask to set the value
     * bit 0 - HI-PRI msdu flowq qdepth threshold need to update if set
     * bit 1 - LOW-PRI msdu flowq qdepth threshold need to update if set
     * bit 2 - UDP msdu flowq qdepth threshold need to update if set
     * bit 3 - NON-UDP msdu flowq qdepth threshold need to update if set
     * rest of bits are reserved and set to 0.
     */
    A_UINT32 msduq_update_mask;

    /** Qdepth threshold value
     * If number of msdus in a queue excess over qdepth_thresh_value value
     * while queuing msdu's then we drop new msdus.
     * (Though dropping older (stale) data rather than newer data might be
     * preferable, the dropping is performed by MAC HW, and there’s no option
     * to configure the HW to do head dropping rather than tail dropping.)
     */
    A_UINT32 qdepth_thresh_value;
} wmi_msduq_qdepth_thresh_update;

/** WMI_PEER_TID_MSDUQ_QDEPTH_THRESH_UPDATE_CMDID
 *   Request FW to update msduq qdepth threshold per TID per peer */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_peer_tid_msduq_qdepth_thresh_update_cmd_fixed_param
     */
    A_UINT32 tlv_header;

    /** pdev id
     * The pdev_id can be determined from the vdev_id, but the pdev_id
     * is explicitly provided so it can be used for sanity checking.
     */
    A_UINT32 pdev_id;

    /** vdev id */
    A_UINT32 vdev_id;

    /**
     * To set the peer msduq qdepth threshold update for a single peer,
     * the host shall send mac address for which peer need to be updated.
     */
    wmi_mac_addr peer_mac_address;

    /** number of payload update tlvs */
    A_UINT32 num_of_msduq_updates;

    /** Followed by the variable length TLV msduq_qdepth_updates:
     *  wmi_msduq_qdepth_th_update msduq_qdepth_thshd_update_list[]
     */
} wmi_peer_tid_msduq_qdepth_thresh_update_cmd_fixed_param;

/**
 * ACK policy to be followed for the TID
 */
typedef enum {
    /** Used when the host does not want to configure the ACK policy */
    WMI_PEER_TID_CONFIG_ACK_POLICY_IGNORE,
    /** Allow ACK for the TID */
    WMI_PEER_TID_CONFIG_ACK,
    /** Do not expect ACK for the TID */
    WMI_PEER_TID_CONFIG_NOACK,
} WMI_PEER_TID_CONFIG_ACK_POLICY;

/**
 * Aggregation control policy for the TID
 */
typedef enum {
    /** Used when the host does not want to configure the aggregation policy */
    WMI_PEER_TID_CONFIG_AGGR_CONTROL_IGNORE,
    /** Enable aggregation for the TID */
    WMI_PEER_TID_CONFIG_AGGR_CONTROL_ENABLE,
    /** Disable aggregation for the TID */
    WMI_PEER_TID_CONFIG_AGGR_CONTROL_DISABLE,
} WMI_PEER_TID_CONFIG_AGGR_CONTROL;

/**
 * Rate control policy for the TID
 */
typedef enum {
    /** Used when the host does not want to configure the rate control policy */
    WMI_PEER_TID_CONFIG_RATE_CONTROL_IGNORE,
    /** Auto rate control */
    WMI_PEER_TID_CONFIG_RATE_CONTROL_AUTO,
    /** Fixed rate control */
    WMI_PEER_TID_CONFIG_RATE_CONTROL_FIXED_RATE,
    /** Set the Default lowest rate (6Mbps in 5GHZ and 1Mbps in 2GHZ) */
    WMI_PEER_TID_CONFIG_RATE_CONTROL_DEFAULT_LOWEST_RATE,
    /**
     * Set the highest rate cap allowed for this TID.
     * Rate cap is specified in rate code format,
     * i.e. NSS and MCS combined as shown below:
     * b'5-b'4 indicate the NSS (0 - 1x1, 1 - 2x2, 2 - 3x3, 3 - 4x4)
     * b'3-b'0 indicate the MCS
     */
    WMI_PEER_TID_CONFIG_RATE_UPPER_CAP,
} WMI_PEER_TID_CONFIG_RATE_CONTROL;

/**
 * SW retry threshold for the TID
 */
typedef enum {
    /** Used when the host does not want to configure the SW retry threshold */
    WMI_PEER_TID_SW_RETRY_IGNORE = 0,
    WMI_PEER_TID_SW_RETRY_MIN = 1,
    WMI_PEER_TID_SW_RETRY_MAX = 30,
    /** No SW retry for the TID */
    WMI_PEER_TID_SW_RETRY_NO_RETRY = 0xFFFFFFFF,
} WMI_PEER_TID_CONFIG_SW_RETRY_THRESHOLD;

/*
 * values for tid_config_supported_bitmap field,
 * in wmi_peer_tid_configurations_cmd structure.
 */
typedef enum {
      /* Used to indicate that disable_rts_cts field is valid */
      WMI_PEER_TID_DISABLE_RTS_CTS_VALID = 0x00000001,
} WMI_PEER_TID_EXT_CONFIG_VALID_BITMAP;

/**
 * Command format for the TID configuration
 */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_peer_tid_configurations_cmd_fixed_param
     */
    A_UINT32 tlv_header;

    /** vdev id */
    A_UINT32 vdev_id;

    /** peer MAC address */
    wmi_mac_addr peer_mac_address;

    /** TID number, generated by the caller.
     * Valid range for QoS TID : 0-15
     * Valid range for non QOS/Mgmt TID: 16-19
     * Any other TID number is invalid.
     */
    A_UINT32 tid_num;

    /** ACK policy - of type WMI_PEER_TID_CONFIG_ACK_POLICY */
    A_UINT32 ack_policy;

    /** Aggregation control - of type WMI_PEER_TID_CONFIG_AGGR_CONTROL */
    A_UINT32 aggr_control;

    /** Rate control - of type WMI_PEER_TID_CONFIG_RATE_CONTROL */
    A_UINT32 rate_control;

    /** Fixed rate control parameters - of type WMI_PEER_PARAM_FIXED_RATE.
     * This is applicable only when rate_control is
     * WMI_PEER_TID_CONFIG_RATE_CONTROL_FIXED_RATE
     */
    A_UINT32 rcode_rcflags;

    /** MPDU SW retry threshold - of type WMI_PEER_TID_CONFIG_SW_RETRY_THRESHOLD
     * This SW retry threshold limits the total number of retransmits of
     * nacked or unacked MPDUs, but it is up to the FW to decide what
     * tx rate to use during each retransmission.
     */
    A_UINT32 sw_retry_threshold;

    /*--- Start of extended structure ---*/
    /* Bitmap to indicate which fields in the extended structure are valid.
     * Bitmap values correspond to enum WMI_PEER_TID_EXT_CONFIG_VALID_BITMAP
     */
    A_UINT32 tid_config_supported_bitmap;

    /* Knob to enable/disable RTS/CTS per TID */
    A_UINT32 disable_rts_cts;
} wmi_peer_tid_configurations_cmd_fixed_param;

/* The below enable/disable macros are used for both per peer CFR capture
 * control (as in wmi_peer_cfr_capture_cmd) and control of the entire periodic
 * CFR capture feature (as in WMI_PDEV_PARAM_PER_PEER_PERIODIC_CFR_ENABLE)
 */
#define WMI_PEER_CFR_CAPTURE_ENABLE   1
#define WMI_PEER_CFR_CAPTURE_DISABLE  0

#define WMI_PEER_CFR_ONE_SHOT_REQUEST 0
#define WMI_PEER_CFR_PERIODICITY_MIN  10 /* 10ms */
#define WMI_PEER_CFR_PERIODICITY_MAX  10*60*1000 /* 10 minutes */

/* Bandwidth of peer CFR captures */
typedef enum {
    WMI_PEER_CFR_CAPTURE_BW_20MHZ    = 0,
    WMI_PEER_CFR_CAPTURE_BW_40MHZ    = 1,
    WMI_PEER_CFR_CAPTURE_BW_80MHZ    = 2,
    WMI_PEER_CFR_CAPTURE_BW_160MHZ   = 3,
    WMI_PEER_CFR_CAPTURE_BW_80_80MHZ = 4,
    WMI_PEER_CFR_CAPTURE_BW_MAX,
} WMI_PEER_CFR_CAPTURE_BW;

/* Peer CFR capture method */
typedef enum {
    /* Send null frame on the requested bw and capture CFR on ACK */
    WMI_PEER_CFR_CAPTURE_METHOD_NULL_FRAME = 0,
    /* New methods to be added here */
    WMI_PEER_CFR_CAPTURE_METHOD_MAX,
} WMI_PEER_CFR_CAPTURE_METHOD;

/*
 * Peer command structure to configure the CFR capture
 */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_peer_cfr_capture_cmd_fixed_param
     */
    A_UINT32 tlv_header;

    /* WMI_PEER_CFR_CAPTURE_ENABLE: Enable CFR capture for the peer
     * WMI_PEER_CFR_CAPTURE_DISABLE: Disable CFR capture for the peer
     */
    A_UINT32 request;
    /* Peer MAC address. In AP mode, this is the address of the connected peer
     * for which CFR capture is needed. In case of STA mode, this is the address
     * of the AP to which the STA is connected
     */
    wmi_mac_addr mac_addr;
    /* vdev id */
    A_UINT32 vdev_id;
    /* Periodicity of measurement in ms.
     * WMI_PEER_CFR_ONE_SHOT_REQUEST: One-shot request i.e., Only one CFR
     * capture for the request and no periodic CFR captures.
     * The min value is WMI_PEER_CFR_PERIODICITY_MIN
     * The max value is WMI_PEER_CFR_PERIODICITY_MAX
     */
    A_UINT32 periodicity;
    /* BW of measurement - of type WMI_PEER_CFR_CAPTURE_BW */
    A_UINT32 bandwidth;
    /* Method used to capture CFR - of type WMI_PEER_CFR_CAPTURE_METHOD */
    A_UINT32 capture_method;
} wmi_peer_cfr_capture_cmd_fixed_param;

typedef enum {
    WMI_PEER_IND_SMPS = 0x0, /* spatial multiplexing power save */
    WMI_PEER_IND_OMN,        /* operating mode notification */
    WMI_PEER_IND_OMI,        /* operating mode indication */
} WMI_PEER_OPER_MODE_IND;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_peer_oper_mode_change */
    A_UINT32 tlv_header;
    /** mac addr of the peer */
    wmi_mac_addr peer_mac_address;
    /** Peer type indication WMI_PEER_OPER_MODE_IND. */
    A_UINT32 ind_type;
    /** new_rxnss valid for all peer_operating mode ind. */
    A_UINT32 new_rxnss;
    /** new_bw  valid for peer_operating mode ind. OMN/OMI
     *  value of this bw is as per 11ax/ac standard:
     *  0 = 20MHz,1 = 40MHz, 2= 80MHz, 3 = 160MHz
     */
    A_UINT32 new_bw;
    /** new_txnss valid for peer_operating mode ind. OMI */
    A_UINT32 new_txnss;
    /** new_disablemu: disable mu mode
     *  valid for peer_operating mode ind. OMI
     */
    A_UINT32 new_disablemu;
} wmi_peer_oper_mode_change_event_fixed_param;

/** FW response when tx failure count has reached threshold
 *  for a peer */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_peer_tx_fail_cnt_thr_event_fixed_param */
    A_UINT32 tlv_header;
    /** vdev id*/
    A_UINT32 vdev_id;
    /** mac address */
    wmi_mac_addr peer_mac_address;
    /** tx failure count- will eventually be removed and not used * */
    A_UINT32 tx_fail_cnt;
    /** seq number of the nth tx_fail_event */
     A_UINT32 seq_no;
} wmi_peer_tx_fail_cnt_thr_event_fixed_param;

enum wmi_rmc_mode {
    /** Disable RMC */
    WMI_RMC_MODE_DISABLED = 0,
    /** Enable RMC */
    WMI_RMC_MODE_ENABLED = 1,
};

/** Enable RMC transmitter functionality. Upon
 *  receiving this, the FW shall mutlicast frames with
 *  reliablity. This is a vendor
 *  proprietary feature. */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_rmc_set_mode_cmd_fixed_param   */
    A_UINT32 tlv_header;
    /** vdev id*/
    A_UINT32 vdev_id;
    /** enable_rmc contains values from enum wmi_rmc_mode;
     *  Default value: 0 (disabled) */
    A_UINT32 enable_rmc;
} wmi_rmc_set_mode_cmd_fixed_param;

/** Configure transmission periodicity of action frames in a
 *  RMC network for the multicast transmitter */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_rmc_set_action_period_cmd_fixed_param */
    A_UINT32 tlv_header;
    /** vdev id */
    A_UINT32 vdev_id;
    /** time period in milliseconds. Default: 300 ms.
     An action frame indicating the current leader is transmitted by the
     RMC transmitter once every 'periodity_msec' */
    A_UINT32 periodicity_msec;
} wmi_rmc_set_action_period_cmd_fixed_param;

/** Optimise Leader selection process in RMC functionality. For
 *  Enhancement/Debug purposes only */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_rmc_config_cmd_fixed_param   */
    A_UINT32 tlv_header;
    /** vdev id */
    A_UINT32 vdev_id;
    /** flags ::
     *  0x0001 - Enable beacon averaging
     *  0x0002 - Force leader selection
     *  0x0004 - Enable Timer based leader switch
     *  0x0008 - Use qos/NULL based for multicast reliability */
    A_UINT32 flags;
    /**  control leader change timeperiod (in seconds) */
    A_UINT32 peridocity_leader_switch;
    /** control activity timeout value for data rx (in seconds) */
    A_UINT32 data_activity_timeout;
    /** mac address of leader */
    wmi_mac_addr forced_leader_mac_addr;
} wmi_rmc_config_cmd_fixed_param;

/** MHF is generally implemented in
 *  the kernel. To decrease system power consumption, the
 *  driver can enable offloading this to the chipset. In
 *  order for the offload, the firmware needs the routing table.
 *  The host shall plumb the routing table into FW. The firmware
 *  shall perform an IP address lookup and forward the packet to
 *  the next hop using next hop's mac address. This is a vendor
 *  proprietary feature. */
enum wmi_mhf_ofl_mode {
    /** Disable MHF offload */
    WMI_MHF_OFL_MODE_DISABLED = 0,
    /** Enable MHF offload */
    WMI_MHF_OFL_MODE_ENABLED = 1,
};

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_mhf_offload_set_mode_cmd_fixed_param */
    A_UINT32 tlv_header;
    /** vdev id*/
    A_UINT32 vdev_id;
    /** enable_mhf_ofl contains values from enum
     *  wmi_mhf_ofl_mode; Default value: 0 (disabled) */
    A_UINT32 enable_mhf_ofl;
} wmi_mhf_offload_set_mode_cmd_fixed_param;

enum wmi_mhf_ofl_table_action {
   /** Create MHF offload table in FW */
   WMI_MHF_OFL_TBL_CREATE = 0,
   /** Append to existing MHF offload table */
   WMI_MHF_OFL_TBL_APPEND = 1,
   /** Flush entire MHF offload table in FW */
   WMI_MHF_OFL_TBL_FLUSH = 2,
};

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_mhf_offload_plumb_routing_table_cmd_fixed_param */
    A_UINT32 tlv_header;
    /** vdev id*/
    A_UINT32 vdev_id;
    /** action corresponds to values from enum
     *  wmi_mhf_ofl_table_action */
    A_UINT32 action;
    /** number of entries in the table */
    A_UINT32 num_entries;
/** Followed by the variable length TLV
 *  wmi_mhf_offload_routing_table_entry entries[] */
}wmi_mhf_offload_plumb_routing_table_cmd;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_mhf_offload_routing_table_entry */
    A_UINT32 tlv_header;
    /** Destination node's IP address */
    WMI_IPV4_ADDR dest_ipv4_addr;
    /** Next hop node's MAC address */
    wmi_mac_addr next_hop_mac_addr;
}wmi_mhf_offload_routing_table_entry;

enum {
    WMI_DFS_RADAR_PULSE_FLAG_MASK_PSIDX_DIFF_VALID = 0x00000001,
};

typedef struct {
    /** tlv tag and len, tag equals
     * WMITLV_TAG_STRUC_wmi_dfs_radar_event */
    A_UINT32 tlv_header;

    /** full 64 tsf timestamp get from MAC tsf timer indicates
     * the time that the radar event uploading to host, split
     * it to high 32 bit and lower 32 bit in fulltsf_high and
     * full_tsf_low
     */
    A_UINT32 upload_fullts_low;
    A_UINT32 upload_fullts_high;

    /** timestamp indicates the time when DFS pulse is detected
     * equal to ppdu_end_ts - radar_pusle_summary_ts_offset
     */
    A_UINT32 pulse_detect_ts;

    /** the duaration of the pulse in us */
    A_UINT32 pulse_duration;

    /** the center frequency of the radar pulse detected, KHz */
    A_UINT32 pulse_center_freq;

    /** bandwidth of current DFS channel, MHz */
    A_UINT32 ch_bandwidth;

    /** center channel frequency1 of current DFS channel, MHz */
    A_UINT16 ch_center_freq1;

    /** center channel frequency2 of current DFS channel, MHz,
     * reserved for 160 BW mode
     */
    A_UINT16 ch_center_freq2;

    /** flag to indicate if this pulse is chirp */
    A_UINT8  pulse_is_chirp;

    /** RSSI recorded in the ppdu */
    A_UINT8  rssi;

    /** extened RSSI info */
    A_UINT8  rssi_ext;

    union {
        A_UINT8 pmac_id; // OBSOLETE - will be removed once all refs are gone
        /** pdev_id for identifying the MAC
         * See macros starting with WMI_PDEV_ID_ for values.
         */
        A_UINT8 pdev_id;
    };

    /** index of peak magnitude bin (signed) */
    A_INT32 peak_sidx;
    /** Max pulse chirp velocity in delta bins over chirp FFT interval */
    A_INT32 pulse_delta_peak;

    /** Max pulse chirp velocity variance in delta bins */
    A_INT32 pulse_delta_diff;
    /** the difference in the FFT peak index between short FFT and the first long FFT
     * psidx_diff = (first_long_fft_psidx - 4*first_short_fft_psidx),
     */
    A_INT32 psidx_diff;

    /** pulse_flags: see WMI_DFS_RADAR_PULSE_FLAG_MASK enum values
     *  0x0001 - set if psidx_diff is valid
     */
    A_UINT32 pulse_flags;

} wmi_dfs_radar_event_fixed_param;

enum {
    /* DEFAULT - target chooses what action to take, based on its thermal
     * management policy
     * Targets which throttle tx (and potentially rx) based on thermal
     * management thresholds specified by the host will shut down tx
     * if the temperature exceeds upper_thresh_degreeC.
     * Targets which simply inform the host about threshold breaches will
     * send a notification message to the host if the temperature exceeds
     * upper_thresh_degreeC.
     * Conversely, if the temperature was above upper_thresh_degreeC but
     * then drops to below lower_threshold_degreeC, the target will either
     * resume tx, or notify the host.
     */
    WMI_THERMAL_MGMT_ACTION_DEFAULT = 0,
    /* HALT_TRAFFIC -
     * If the temperature rises above upper_thresh_degreeC, the target will
     * halt tx.
     * If the temperature falls back below lower_thresh_degreeC, the target
     * will resume tx.
     */
    WMI_THERMAL_MGMT_ACTION_HALT_TRAFFIC = 1,
    /* NOTIFY_HOST - the target will notify the host if the temperature
     * either rises above upper_thresh_degreeC or falls below
     * lower_thresh_degreeC.
     */
    WMI_THERMAL_MGMT_ACTION_NOTIFY_HOST = 2,
};

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_thermal_mgmt_cmd_fixed_param  */

    /*Thermal thresholds*/
    A_UINT32 lower_thresh_degreeC; /* in degree C*/
    A_UINT32 upper_thresh_degreeC; /* in degree C*/

    /*Enable/Disable Thermal Monitoring for Mitigation*/
    A_UINT32 enable;

    /* action: what the target should do when a thermal upper/lower threshold
     * is crossed.
     * Refer to the WMI_THERMAL_MGMT_ACTION enum.
     */
    A_UINT32 action;
    A_UINT32 threshold_warning_degreeC;
    A_UINT32 sample_rate_ms;
} wmi_thermal_mgmt_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_thermal_mgmt_event_fixed_param  */

    A_UINT32 temperature_degreeC;/* temperature in degree C*/
} wmi_thermal_mgmt_event_fixed_param;

/**
 * This command is sent from WLAN host driver to firmware to
 * request firmware to configure auto shutdown timer in fw
 * 0 - Disable <1-19600>-Enabled and timer value is seconds (86400 seconds = 1 day maximum>
 */
typedef struct {
    A_UINT32 tlv_header;    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_host_auto_shutdown_cfg_cmd_param  */
    A_UINT32 timer_value;   /** timer value; 0=disable */
} wmi_host_auto_shutdown_cfg_cmd_fixed_param;

enum wmi_host_auto_shutdown_reason {
    WMI_HOST_AUTO_SHUTDOWN_REASON_UNKNOWN = 0,
    WMI_HOST_AUTO_SHUTDOWN_REASON_TIMER_EXPIRY = 1,
    WMI_HOST_AUTO_SHUTDOWN_REASON_MAX,
};

/* WMI_HOST_AUTO_SHUTDOWN_EVENTID  */
typedef struct{
    A_UINT32    tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_host_auto_shutdown_event_fixed_param  */
    A_UINT32    shutdown_reason; /* value: wmi_host_auto_shutdown_reason */
} wmi_host_auto_shutdown_event_fixed_param;


/** New WMI command to support TPC CHAINMASK ADJUSTMENT ACCORDING TO a set of conditions specified in the command.
 *  fw will save c tpc offset/chainmask along with conditions and adjust tpc/chainmask when condition meet.
 *  This command is only used by some customer for  verification test.  It is not for end-user.
 *
 *  array of wmi_tpc_chainmask_config structures are passed with the command to specify multiple conditions.
 *
 *  The set of conditions include bt status, stbc status, band, phy_mode, 1stream/2streams, channel, rate. when all these conditions meet,
 *  the output(tpc_offset,chainmask) will be applied on per packet basis. ack_offset is applied based on channel condtion only. When multiple
 *  conditions has the same channel ,then the first ack_offset will be applied. It is better for host driver to make sure the
 *  <channel, ack_offset> pair is unique.
 *
 *  the conditions (bt status, stbc status, band, phy_mode, 1steam/2streams, tpc_offset, ack_offset, chainmask) are combinedi into a single word
 *  called basic_config_info by bitmap
 *  to save memory. And channel & rate info will be tracked by 'channel' field and 'rate0', 'rate1' field because of its large combination.
 *
 *  'rate bit' or 'channel bit' field of basic_config_info indicate validity of the channel and rate fields.if rate bit is 0 then the rate field
 *   is ignored.
 *  disable will remove preious conditions from FW.
 *  conditions from the later command will over write conditions stored from a previous command.
 *
 */

#define WMI_TPC_CHAINMASK_CONFIG_BT_ON_OFF    0   /** dont' care the bt status */
#define WMI_TPC_CHAINMASK_CONFIG_BT_ON        1   /** apply only when bt on */
#define WMI_TPC_CHAINMASK_CONFIG_BT_OFF       2   /** apply only when bt off  */
#define WMI_TPC_CHAINMASK_CONFIG_BT_RESV1     3   /** reserved  */

#define WMI_TPC_CHAINMASK_CONFIG_CHAINMASK_DONT_CARE   0   /**  don't care the chainmask */
#define WMI_TPC_CHAINMASK_CONFIG_CHAINMASK_CHAIN0      1   /**  force to use Chain0 to send */
#define WMI_TPC_CHAINMASK_CONFIG_CHAINMASK_CHAIN1      2   /**  force to use Chain1 to send */
#define WMI_TPC_CHAINMASK_CONFIG_CHAINMASK_CHAIN0_CHAIN1  3   /** force to use Chain0 & Chain1 to send */

#define WMI_TPC_CHAINMASK_CONFIG_STBC_ON_OFF  0   /**  don't care about stbc  */
#define WMI_TPC_CHAINMASK_CONFIG_STBC_ON      1   /**  apply only when stbc on */
#define WMI_TPC_CHAINMASK_CONFIG_STBC_OFF     2   /**  apply only when stbc off */
#define WMI_TPC_CHAINMASK_CONFIG_STBC_RESV1   3   /**  reserved */

#define WMI_TPC_CHAINMASK_CONFIG_BAND_2G      0   /**  2G */
#define WMI_TPC_CHAINMASK_CONFIG_BAND_5G      1   /**  5G */

#define WMI_TPC_CHAINMASK_CONFIG_PHY_MODE_11B_2G    0        /** 11b 2G */
#define WMI_TPC_CHAINMASK_CONFIG_PHY_MODE_11G_2G    1        /** 11g 2G */
#define WMI_TPC_CHAINMASK_CONFIG_PHY_MODE_11N_2G    2        /** 11n 2G */
#define WMI_TPC_CHAINMASK_CONFIG_PHY_MODE_11N_11AC_2G   3    /** 11n + 11ac 2G */
#define WMI_TPC_CHAINMASK_CONFIG_PHY_MODE_11A_5G    4        /** 11a 5G */
#define WMI_TPC_CHAINMASK_CONFIG_PHY_MODE_11N_5G    5        /** 11n 5G */
#define WMI_TPC_CHAINMASK_CONFIG_PHY_MODE_11AC_5G   6        /** 11ac 5G */
#define WMI_TPC_CHAINMASK_CONFIG_PHY_MODE_11N_11AC_5G  7     /** 11n + 11ac 5G */

#define WMI_TPC_CHAINMASK_CONFIG_STREAM_1           0    /** 1 stream  */
#define WMI_TPC_CHAINMASK_CONFIG_STREAM_2           1    /** 2 streams */

#define WMI_TPC_CHAINMASK_CONFIG_CHANNEL_OFF        0    /** channel field is ignored */
#define WMI_TPC_CHAINMASK_CONFIG_CHANNEL_ON         1    /** channel field needs to be checked */

#define WMI_TPC_CHAINMASK_CONFIG_RATE_OFF           0    /** rate field is ignored */
#define WMI_TPC_CHAINMASK_CONFIG_RATE_ON            1    /** rate field needs to be checked */

/**  Bit map definition for basic_config_info starts   */
#define WMI_TPC_CHAINMASK_CONFIG_TPC_OFFSET_S   0
#define WMI_TPC_CHAINMASK_CONFIG_TPC_OFFSET     (0x1f << WMI_TPC_CHAINMASK_CONFIG_TPC_OFFSET_S)
#define WMI_TPC_CHAINMASK_CONFIG_TPC_OFFSET_GET(x)     WMI_F_MS(x,WMI_TPC_CHAINMASK_CONFIG_TPC_OFFSET)
#define WMI_TPC_CHAINMASK_CONFIG_TPC_OFFSET_SET(x,z)   WMI_F_RMW(x,(z) & 0x1f,WMI_TPC_CHAINMASK_CONFIG_TPC_OFFSET)

#define WMI_TPC_CHAINMASK_CONFIG_ACK_OFFSET_S      5
#define WMI_TPC_CHAINMASK_CONFIG_ACK_OFFSET        (0x1f << WMI_TPC_CHAINMASK_CONFIG_ACK_OFFSET_S)
#define WMI_TPC_CHAINMASK_CONFIG_ACK_OFFSET_GET(x)     WMI_F_MS(x,WMI_TPC_CHAINMASK_CONFIG_ACK_OFFSET)
#define WMI_TPC_CHAINMASK_CONFIG_ACK_OFFSET_SET(x,z)   WMI_F_RMW(x, (z) & 0x1f, WMI_TPC_CHAINMASK_CONFIG_ACK_OFFSET)

#define WMI_TPC_CHAINMASK_CONFIG_CHAINMASK_S  10
#define WMI_TPC_CHAINMASK_CONFIG_CHAINMASK   (0x3 << WMI_TPC_CHAINMASK_CONFIG_CHAINMASK_S)
#define WMI_TPC_CHAINMASK_CONFIG_CHAINMASK_GET(x)   WMI_F_MS(x,WMI_TPC_CHAINMASK_CONFIG_CHAINMASK)
#define WMI_TPC_CHAINMASK_CONFIG_CHAINMASK_SET(x,z)  WMI_F_RMW(x, (z)&0x3, WMI_TPC_CHAINMASK_CONFIG_CHAINMASK)

#define WMI_TPC_CHAINMASK_CONFIG_BT_S       12
#define WMI_TPC_CHAINMASK_CONFIG_BT         (0x3 << WMI_TPC_CHAINMASK_CONFIG_BT_S)
#define WMI_TPC_CHAINMASK_CONFIG_BT_GET(x)     WMI_F_MS(x,WMI_TPC_CHAINMASK_CONFIG_BT)
#define WMI_TPC_CHAINMASK_CONFIG_BT_SET(x,z)   WMI_F_RMW(x, (z)&0x3, WMI_TPC_CHAINMASK_CONFIG_BT)

#define WMI_TPC_CHAINMASK_CONFIG_STBC_S     14
#define WMI_TPC_CHAINMASK_CONFIG_STBC       (0x3 << WMI_TPC_CHAINMASK_CONFIG_STBC_S)
#define WMI_TPC_CHAINMASK_CONFIG_STBC_GET(x)     WMI_F_MS(x,WMI_TPC_CHAINMASK_CONFIG_STBC)
#define WMI_TPC_CHAINMASK_CONFIG_STBC_SET(x,z)   WMI_F_RMW(x, (z)& 0x3, WMI_TPC_CHAINMASK_CONFIG_STBC)

#define WMI_TPC_CHAINMASK_CONFIG_BAND_S     16
#define WMI_TPC_CHAINMASK_CONFIG_BAND       (0x1 << WMI_TPC_CHAINMASK_CONFIG_BAND_S)
#define WMI_TPC_CHAINMASK_CONFIG_BAND_GET(x)  WMI_F_MS(x,WMI_TPC_CHAINMASK_CONFIG_BAND)
#define WMI_TPC_CHAINMASK_CONFIG_BAND_SET(x,z) WMI_F_RMW(x, (z) &0x1, WMI_TPC_CHAINMASK_CONFIG_BAND)

#define WMI_TPC_CHAINMASK_CONFIG_STREAM_S   17
#define WMI_TPC_CHAINMASK_CONFIG_STREAM     (0x1 << WMI_TPC_CHAINMASK_CONFIG_STREAM_S)
#define WMI_TPC_CHAINMASK_CONFIG_STREAM_GET(x)  WMI_F_MS(x,WMI_TPC_CHAINMASK_CONFIG_STREAM)
#define WMI_TPC_CHAINMASK_CONFIG_STREAM_SET(x,z)  WMI_F_RMW(x, (z)&0x1, WMI_TPC_CHAINMASK_CONFIG_STREAM)

#define WMI_TPC_CHAINMASK_CONFIG_PHY_MODE_S     18
#define WMI_TPC_CHAINMASK_CONFIG_PHY_MODE       (0x7 << WMI_TPC_CHAINMASK_CONFIG_PHY_MODE_S)
#define WMI_TPC_CHAINMASK_CONFIG_PHY_MODE_GET(x) WMI_F_MS(x,WMI_TPC_CHAINMASK_CONFIG_PHY_MODE)
#define WMI_TPC_CHAINAMSK_CONFIG_PHY_MODE_SET(x,z)  WMI_F_RMW(x, (z)&0x7, WMI_TPC_CHAINMASK_CONFIG_PHY_MODE)

#define WMI_TPC_CHAINMASK_CONFIG_CHANNEL_S     21
/*
 * The deprecated old name (WMI_TPC_CHAINMASK_CONFIG_CHANNEL_EXIST)
 * is temporarily maintained as an alias for the correct name
 * (WMI_TPC_CHAINMASK_CONFIG_CHANNEL)
 */
#define WMI_TPC_CHAINMASK_CONFIG_CHANNEL_EXIST WMI_TPC_CHAINMASK_CONFIG_CHANNEL
#define WMI_TPC_CHAINMASK_CONFIG_CHANNEL       (0x1 << WMI_TPC_CHAINMASK_CONFIG_CHANNEL_S)
#define WMI_TPC_CHAINMASK_CONFIG_CHANNEL_GET(x)  WMI_F_MS(x,WMI_TPC_CHAINMASK_CONFIG_CHANNEL)
#define WMI_TPC_CHAINMASK_CONFIG_CHANNEL_SET(x,z)  WMI_F_RMW(x, (z)&0x1, WMI_TPC_CHAINMASK_CONFIG_CHANNEL)

#define WMI_TPC_CHAINMASK_CONFIG_RATE_S  22
/*
 * The deprecated old name (WMI_TPC_CHAINMASK_CONFIG_RATE_EXIST)
 * is temporarily maintained as an alias for the correct name
 * (WMI_TPC_CHAINMASK_CONFIG_RATE)
 */
#define WMI_TPC_CHAINMASK_CONFIG_RATE_EXIST WMI_TPC_CHAINMASK_CONFIG_RATE
#define WMI_TPC_CHAINMASK_CONFIG_RATE    (0x1 << WMI_TPC_CHAINMASK_CONFIG_RATE_S)
#define WMI_TPC_CHAINMASK_CONFIG_RATE_GET(x)   WMI_F_MS(x, WMI_TPC_CHAINMASK_CONFIG_RATE)
#define WMI_TPC_CHAINMASK_CONFIG_RATE_SET(x,z)  WMI_F_RMW(x, (z)&0x1, WMI_TPC_CHAINMASK_CONFIG_RATE)

/**  Bit map definition for basic_config_info ends   */

typedef struct{
    A_UINT32 tlv_header;
    /** Basic condition defined as bit map above, bitmap is chosen to save memory.
     * Bit0  ~ Bit4: tpc offset which will be adjusted if condtion matches, the unit is 0.5dB.  bit4 indicates signed
     * Bit5  ~ Bit9: ack offset which will be adjusted if condtion matches, the unit is 0.5dB.  bit9 indicates signed
     * Bit10 ~ Bit11: chainmask  b'00: don't care, b'01: force to use chain0, b'10: force to use chain1, b'11: force to use chain0&chain1
     * Bit12 ~ Bit13: bt condition  b'00: don't care, b'01: apply only when bt on, b'10: apply only when bt off,  b'11: reserved
     * Bit14 ~ Bit15: stbc condition  b'00: don't care, b'01: apply only when stbc on, b'10: apply only when stbc off, b'11: reserved
     * Bit16 : band condition  b'0: 2G,  b'1: 5G
     * Bit17 : stream condition:  b'0: 1 stream, b'1: 2 streams
     * Bit18 ~ Bit20: phy mode condition: b'000: 11b 2g, b'001: 11g 2g, b'010: 11n 2g, b'011: 11n+11ac 2g, b'100: 11a, b'101: 11n 5g, b'110: 11ac 5g, b'111: 11n+11ac 5g
     * Bit21 : channel bit, if this bit is 0, then the following channel field is ignored
     * Bit22 : rate bit, if this bit is 0, then the following rate0&rate1 is ignored.
     * Bit23 ~ Bit31:  reserved
     */
    A_UINT32 basic_config_info;

    /** channel mapping bit rule: The lower bit corresponds with smaller channel.
     *  it depends on Bit14 of basic_config_info
     *  Total 24 channels for 5G
     *  36    40    44    48    52    56    60    64   100   104   108   112   116   120   124   128   132   136   140   149   153   157   161   165
     *  Total 14 channels for 2G
     *  1 ~ 14
     */
    A_UINT32 channel;

    /** rate mapping bit rule:  The lower bit corresponds with lower rate.
     *  it depends on Bit16 ~ Bit18 of basic_config_info, "phy mode condition"
     *  Legacy rates , 11b, 11g, 11A
     *  11n one stream ( ht20, ht40 ) 8+8
     *  11n two streams ( ht20, ht40 ) 8+8
     *  11ac one stream ( vht20, vht40, vht80 ) 10+10+10
     *  11ac two streams (vht20, vht40, vht80 ) 10+10+10
     */
    A_UINT32 rate0;
    /** For example, for 11b, when rate0 equals 0x3, it means if actual_rate in [ "1Mbps", "2Mbps"] connection, the rate condition is true.
     *  For example, for 11g/11a, when rate0 equals 0xf0,it means "54Mbps", "48Mbps", "36Mbps", "24Mb's" is selected, while "18Mbps", "12Mbps", "9Mbps", "6Mbps" is not selected
     */

    /** only used for "11n+11ac" combined phy_mode, (WMI_TPC_CHAINMASK_CONFIG_PHY_MODE_11N_11AC_2G , WMI_TPC_CHAINMASK_CONFIG_PHY_MODE_11N_11AC_5G) in this case, 11n rates begins on rate0, while 11ac rates begins on rate1
     */
    A_UINT32 rate1;
} wmi_tpc_chainmask_config;

#define WMI_TPC_CHAINMASK_CONFIG_DISABLE   0   /** control the off for the tpc & chainmask*/
#define WMI_TPC_CHAINMASK_CONFIG_ENABLE    1   /** control the on for the tpc & chainmask*/

typedef struct{
    A_UINT32 tlv_header;
    A_UINT32 enable;  /** enable to set tpc & chainmask when condtions meet, 0: disabled,   1: enabled.  */
    A_UINT32 num_tpc_chainmask_configs;
    /** following this structure is num_tpc_chainmask_configs number of wmi_tpc_chainmask_config  */
} wmi_tpc_chainmask_config_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header;        /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_oem_dma_ring_cfg_req_fixed_param */
    A_UINT32 pdev_id;
    /**
     * Bits 31:0:   base address of ring [31:0]
     */
    A_UINT32 base_addr_lo;
    /**
     * Bits 3:0:    base address of ring [35:32]
     * Bits 31:4:   reserved
     */
    A_UINT32 base_addr_hi;
    /**
     * Bits 31:0:   address of head index [31:0]
     */
    A_UINT32 head_idx_addr_lo;
    /**
     * Bits 3:0:    address of head index [35:32]
     * Bits 31:4:   reserved
     */
    A_UINT32 head_idx_addr_hi;
    /**
     * Bits 31:0:   address of tail index [31:0]
     */
    A_UINT32 tail_idx_addr_lo;
    /**
     * Bits 3:0:    address of tail index [35:32]
     * Bits 31:4:   reserved
     */
    A_UINT32 tail_idx_addr_hi;
    A_UINT32 num_ptr;           /** Number of pointers in the ring */
} wmi_oem_dma_ring_cfg_req_fixed_param;

#define WMI_OEM_DMA_RING_ADDR_LO_S 0
#define WMI_OEM_DMA_RING_ADDR_LO 0xffffffff

#define WMI_OEM_DMA_RING_ADDR_LO_GET(dword) WMI_F_MS(dword, WMI_OEM_DMA_RING_ADDR_LO)
#define WMI_OEM_DMA_RING_ADDR_LO_SET(dword, val) WMI_F_RMW(dword, val, WMI_OEM_DMA_RING_ADDR_LO)

#define WMI_OEM_DMA_RING_ADDR_HI_S 0
#define WMI_OEM_DMA_RING_ADDR_HI 0xf

#define WMI_OEM_DMA_RING_ADDR_HI_GET(dword) WMI_F_MS(dword, WMI_OEM_DMA_RING_ADDR_HI)
#define WMI_OEM_DMA_RING_ADDR_HI_SET(dword, val) WMI_F_RMW(dword, val, WMI_OEM_DMA_RING_ADDR_HI)

typedef struct {
    A_UINT32 tlv_header;    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_oem_dma_ring_cfg_rsp_fixed_param */
    A_UINT32 pdev_id;
    A_UINT32 cfg_status;    /** Configuration status; see A_STATUS */
} wmi_oem_dma_ring_cfg_rsp_fixed_param;

typedef struct {
    A_UINT32 tlv_header;    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_oem_indirect_data */
    A_UINT32 pdev_id;       /** ID of pdev whose OEM DMA ring produced the data */
    /**
     * Bits 31:0:   address of data [31:0]
     */
    A_UINT32 addr_lo;
    /**
     * Bits 3:0:    address of data [35:32]
     * Bits 11:4:   reserved
     * Bits 31:12:  opaque host context data [19:0]
     */
    A_UINT32 addr_hi;
    A_UINT32 len;           /** Length of data in bytes */
} wmi_oem_indirect_data;

#define WMI_OEM_DMA_DATA_ADDR_LO_S 0
#define WMI_OEM_DMA_DATA_ADDR_LO 0xffffffff

#define WMI_OEM_DMA_DATA_ADDR_LO_GET(dword) WMI_F_MS(dword, WMI_OEM_DMA_DATA_ADDR_LO)
#define WMI_OEM_DMA_DATA_ADDR_LO_SET(dword, val) WMI_F_RMW(dword, val, WMI_OEM_DMA_DATA_ADDR_LO)

#define WMI_OEM_DMA_DATA_ADDR_HI_S 0
#define WMI_OEM_DMA_DATA_ADDR_HI 0xf

#define WMI_OEM_DMA_DATA_ADDR_HI_GET(dword) WMI_F_MS(dword, WMI_OEM_DMA_DATA_ADDR_HI)
#define WMI_OEM_DMA_DATA_ADDR_HI_SET(dword, val) WMI_F_RMW(dword, val, WMI_OEM_DMA_DATA_ADDR_HI)

#define WMI_OEM_DMA_DATA_ADDR_HI_HOST_DATA_S 12
#define WMI_OEM_DMA_DATA_ADDR_HI_HOST_DATA 0xfffff

#define WMI_OEM_DMA_DATA_ADDR_HI_HOST_DATA_GET(dword) WMI_F_MS(dword, WMI_OEM_DMA_DATA_ADDR_HI_HOST_DATA)
#define WMI_OEM_DMA_DATA_ADDR_HI_HOST_DATA_SET(dword, val) WMI_F_RMW(dword, val, WMI_OEM_DMA_DATA_ADDR_HI_HOST_DATA)

typedef struct {
    A_UINT32 tlv_header;    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_oem_dma_buf_release_hdr */
    A_UINT32 pdev_id;       /** ID of pdev whose OEM DMA ring produced the data */
} wmi_oem_dma_buf_release_fixed_param;

typedef struct {
    /**
     * Bits 31:0:   address of data [31:0]
     */
    A_UINT32 addr_lo;
    /**
     * Bits 3:0:    address of data [35:32]
     * Bits 11:4:   reserved
     * Bits 31:12:  host context data [19:0]
     */
    A_UINT32 addr_hi;
} wmi_oem_dma_buf_release_entry;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_nan_cmd_param */
    A_UINT32 data_len; /** length in byte of data[]. */
    /* This structure is used to send REQ binary blobs
     * from application/service to firmware where Host drv is pass through .
     * Following this structure is the TLV:
     *     A_UINT8 data[];    // length in byte given by field data_len.
     */
} wmi_nan_cmd_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_nan_host_config_param */
    A_UINT32 nan_2g_disc_disable:1; /** This bit when set to 1 indicate NAN 2G discovery should be disabled */
    A_UINT32 nan_5g_disc_disable:1; /** This bit when set to 1 indicate NAN 5G discovery should be disabled */
    A_UINT32 reserved:30;
} wmi_nan_host_config_param_PROTOTYPE;
#define wmi_nan_host_config_param wmi_nan_host_config_param_PROTOTYPE

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_nan_event_hdr */
    A_UINT32 data_len; /** length in byte of data[]. */
    /* This structure is used to send REQ binary blobs
     * from firmware to application/service where Host drv is pass through .
     * Following this structure is the TLV:
     *     A_UINT8 data[];    // length in byte given by field data_len.
     */
} wmi_nan_event_hdr;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_nan_event_info */
    A_UINT32 mac_id; /* MAC ID associated with NAN primary discovery channel; Valid only for NAN enable resp message identified by NAN_MSG_ID_ENABLE_RSP */
    A_UINT32 status:1; /** This bit when set to 0 indicates status is successful; Valid only for NAN enable resp message identified by NAN_MSG_ID_ENABLE_RSP */
    A_UINT32 reserved:31;
} wmi_nan_event_info_PROTOTYPE;

#define wmi_nan_event_info wmi_nan_event_info_PROTOTYPE

/**
 * Event to indicate NAN discovery interface created
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_nan_disc_iface_created_event_fixed_param */
    A_UINT32 tlv_header;
    /** Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /** NAN interface MAC address */
    wmi_mac_addr nan_interface_macaddr;
} wmi_nan_disc_iface_created_event_fixed_param_PROTOTYPE;

#define wmi_nan_disc_iface_created_event_fixed_param wmi_nan_disc_iface_created_event_fixed_param_PROTOTYPE

/**
 * Event to indicate NAN discovery interface deleted
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_nan_disc_iface_deleted_event_fixed_param */
    A_UINT32 tlv_header;
    /** Unique id identifying the VDEV */
    A_UINT32 vdev_id;
} wmi_nan_disc_iface_deleted_event_fixed_param_PROTOTYPE;

#define wmi_nan_disc_iface_deleted_event_fixed_param wmi_nan_disc_iface_deleted_event_fixed_param_PROTOTYPE

/**
 * Event to indicate NAN device started new cluster
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_nan_started_cluster_event_fixed_param */
    A_UINT32 tlv_header;
    /** Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /** NAN Cluster ID */
    A_UINT32 nan_cluster_id;
} wmi_nan_started_cluster_event_fixed_param_PROTOTYPE;

#define wmi_nan_started_cluster_event_fixed_param wmi_nan_started_cluster_event_fixed_param_PROTOTYPE

/**
 * Event to indicate NAN device joined to cluster
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_nan_joined_cluster_event_fixed_param */
    A_UINT32 tlv_header;
    /** Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /** NAN Cluster ID */
    A_UINT32 nan_cluster_id;
} wmi_nan_joined_cluster_event_fixed_param_PROTOTYPE;

#define wmi_nan_joined_cluster_event_fixed_param wmi_nan_joined_cluster_event_fixed_param_PROTOTYPE

/** NAN DATA CMD's */

/**
 * NAN Data get capabilities req
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndi_get_cap_req_fixed_param */
    A_UINT32 tlv_header;
    /** unique id generated in upper layer for the transaction */
    A_UINT32 transaction_id;
} wmi_ndi_get_cap_req_fixed_param_PROTOTYPE;

#define wmi_ndi_get_cap_req_fixed_param wmi_ndi_get_cap_req_fixed_param_PROTOTYPE

/**
 * NDP Response code
 */
typedef enum {
    NDP_RSP_CODE_REQUEST_ACCEPT = 0x00,
    NDP_RSP_CODE_REQUEST_REJECT = 0x01,
    NDP_RSP_CODE_REQUEST_DEFER  = 0x02,
} wmi_ndp_rsp_code_PROTOTYPE;

#define wmi_ndp_rsp_code wmi_ndp_rsp_code_PROTOTYPE

/**
* NDP Channel configuration type
*/
typedef enum {
    WMI_NDP_CHANNEL_NOT_REQUESTED = 0, /* Channel will not configured */
    WMI_NDP_REQUEST_CHANNEL_SETUP = 1, /* Channel will be provided and is optional/hint */
    WMI_NDP_FORCE_CHANNEL_SETUP = 2/* NDP must start on the provided channel */
} wmi_ndp_channel_cfg_PROTOTYPE;

/*
 * The WMI_NDP_IPV6_INTF_ADDR_LEN macro cannot be changed without breaking
 * WMI compatibility.
 */
#define WMI_NDP_IPV6_INTF_ADDR_LEN    16

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndp_transport_ip_param */
    /* Presence of ipv6_intf_addr */
    A_UINT32 ipv6_addr_present;
    /* Presence of transport Port */
    A_UINT32 trans_port_present;
    /* Presence of  transport Protocol */
    A_UINT32 trans_proto_present;
    /* ipv6 Interface address */
    A_UINT8  ipv6_intf_addr[WMI_NDP_IPV6_INTF_ADDR_LEN];
    /* Transport Port */
    A_UINT32 transport_port;
    /* Transport Protocol */
    A_UINT32 transport_protocol;
} wmi_ndp_transport_ip_param;

#define wmi_ndp_channel_cfg wmi_ndp_channel_cfg_PROTOTYPE

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndp_channel_info */
    A_UINT32 mac_id; /* mac_id associated with ndp channel at same index */
} wmi_ndp_channel_info_PROTOTYPE;

#define wmi_ndp_channel_info wmi_ndp_channel_info_PROTOTYPE

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndp_event_param */
    A_UINT32 vdev_id; /* NDI VDEV ID */
    A_UINT32 ndp_termination_in_progress:1; /** This bit when set to 1 indicates to termination of all NDPs associated with NDI vdev ID is started */
    A_UINT32 reserved:31;
} wmi_ndp_event_param_PROTOTYPE;

#define wmi_ndp_event_param wmi_ndp_event_param_PROTOTYPE

/**
 * NDP Initiator requesting a data session
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndp_initiator_req_fixed_param */
    A_UINT32 tlv_header;
    /** Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /** unique id generated in upper layer for the transaction */
    A_UINT32 transaction_id;
    /** Unique Instance Id identifying the Responder's service */
    A_UINT32 service_instance_id;
    /** Discovery MAC addr of the publisher/peer */
    wmi_mac_addr peer_discovery_mac_addr;
    /** Actual number of bytes in TLV ndp_cfg */
    A_UINT32 ndp_cfg_len;
    /** Actual number of bytes in TLV ndp_app_info */
    A_UINT32 ndp_app_info_len;
    /** NDP channel configuration type defined in wmi_ndp_channel_cfg */
    A_UINT32 ndp_channel_cfg;
    /**  NAN Cipher Suite Shared Key */
    A_UINT32 nan_csid;
    /** Actual number of bytes in TLV ndp_pmk */
    A_UINT32 nan_pmk_len;
    /** Actual number of bytes in TLV ndp_passphrase */
    A_UINT32 nan_passphrase_len;
    /** Actual number of bytes in TLV nan_servicename */
    A_UINT32 nan_servicename_len;
    /**
     * TLV (tag length value ) parameters follow the ndp_initiator_req
     * structure. The TLV's are:
     * wmi_channel channel;
     * A_UINT8 ndp_cfg[];
     * A_UINT8 ndp_app_info[];
     * A_UINT8 ndp_pmk[];
     * A_INT8 ndp_passphrase[];
     * A_INT8 nan_servicename[];
     * wmi_ndp_transport_ip_param ndp_transport_ip_param;
     */
} wmi_ndp_initiator_req_fixed_param_PROTOTYPE;

#define wmi_ndp_initiator_req_fixed_param wmi_ndp_initiator_req_fixed_param_PROTOTYPE

/**
 * Initiate a data response on the responder side
 * for data request indication from the peer
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndp_responder_req_fixed_param */
    A_UINT32 tlv_header;
    /** Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /** unique id generated in upper layer for the transaction */
    A_UINT32 transaction_id;
    /**
     * Unique token Id generated on the initiator/responder
     * side used for a NDP session between two NAN devices
     */
    A_UINT32 ndp_instance_id;
    /** Response Code defined in wmi_ndp_rsp_code */
    A_UINT32 rsp_code;
    /** Number of bytes in TLV ndp_cfg */
    A_UINT32 ndp_cfg_len;
    /** Number of bytes in TLV ndp_app_info */
    A_UINT32 ndp_app_info_len;
    /**  NAN Cipher Suite Shared Key */
    A_UINT32 nan_csid;
    /** Actual number of bytes in TLV ndp_pmk */
    A_UINT32 nan_pmk_len;
    /** Actual number of bytes in TLV ndp_passphrase */
    A_UINT32 nan_passphrase_len;
    /** Actual number of bytes in TLV nan_servicename */
    A_UINT32 nan_servicename_len;
    /**
     * TLV (tag length value ) parameters follow the ndp_responder_req
     * structure. The TLV's are:
     * A_UINT8 ndp_cfg[];
     * A_UINT8 ndp_app_info[];
     * A_UINT8 ndp_pmk[];
     * A_INT8 ndp_passphrase[];
     * A_INT8 nan_servicename[];
     * wmi_ndp_transport_ip_param ndp_transport_ip_param;
     */
} wmi_ndp_responder_req_fixed_param_PROTOTYPE;

#define wmi_ndp_responder_req_fixed_param wmi_ndp_responder_req_fixed_param_PROTOTYPE

/**
 * NDP end type
 */
typedef enum {
    WMI_NDP_END_TYPE_UNSPECIFIED = 0x00,
    WMI_NDP_END_TYPE_PEER_UNAVAILABLE = 0x01,
    WMI_NDP_END_TYPE_OTA_FRAME = 0x02,
} wmi_ndp_end_type_PROTOTYPE;

#define wmi_ndp_end_type wmi_ndp_end_type_PROTOTYPE

/**
 * NDP end reason code
 */
typedef enum {
    WMI_NDP_END_REASON_UNSPECIFIED = 0x00,
    WMI_NDP_END_REASON_INACTIVITY = 0x01,
    WMI_NDP_END_REASON_PEER_DATA_END = 0x02,
} wmi_ndp_end_reason_code_PROTOTYPE;

#define wmi_ndp_end_reason_code wmi_ndp_end_reason_code_PROTOTYPE

/**
 * NDP end request
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndp_end_req */
    A_UINT32 tlv_header;
    /** NDP instance id */
    A_UINT32 ndp_instance_id;
} wmi_ndp_end_req_PROTOTYPE;

#define wmi_ndp_end_req wmi_ndp_end_req_PROTOTYPE

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndp_cmd_param */
    A_UINT32 vdev_id; /* NDI VDEV ID */
    A_UINT32 ndp_disable:1; /** This bit when set to 1 indicates to terminate all NDPs associated with NDI vdev ID */
    A_UINT32 reserved:31;
} wmi_ndp_cmd_param_PROTOTYPE;

#define wmi_ndp_cmd_param wmi_ndp_cmd_param_PROTOTYPE

/**
 * NDP End request
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndp_end_req_fixed_param */
    A_UINT32 tlv_header;
    /** unique id generated in upper layer for the transaction */
    A_UINT32 transaction_id;
    /**
     * TLV (tag length value ) parameters follow the ndp_end_req
     * structure. The TLV's are:
     * wmi_ndp_end_req ndp_end_req_list[];
     */
} wmi_ndp_end_req_fixed_param_PROTOTYPE;

#define wmi_ndp_end_req_fixed_param wmi_ndp_end_req_fixed_param_PROTOTYPE

/* NAN DATA RSP EVENTS */

/**
 * Event to indicate NAN Data Interface capabilities cmd
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndi_cap_rsp_event_fixed_param */
    A_UINT32 tlv_header;
    /** Copy of transaction_id received in wmi_ndi_get_cap_req */
    A_UINT32 transaction_id;
    /** Max ndi interface support */
    A_UINT32 max_ndi_interfaces;
    /** Max ndp sessions can support */
    A_UINT32 max_ndp_sessions;
    /** Max number of peer's per ndi */
    A_UINT32 max_peers_per_ndi;
    /** which combination of bands is supported - see NAN_DATA_SUPPORTED_BAND enums */
    A_UINT32 nan_data_supported_bands;
} wmi_ndi_cap_rsp_event_fixed_param_PROTOTYPE;

#define wmi_ndi_cap_rsp_event_fixed_param wmi_ndi_cap_rsp_event_fixed_param_PROTOTYPE

/**
 * NDP command response code
 */
typedef enum {
    NDP_CMD_RSP_STATUS_SUCCESS = 0x00,
    NDP_CMD_RSP_STATUS_ERROR = 0x01,
} wmi_ndp_cmd_rsp_status_PROTOTYPE;

#define wmi_ndp_cmd_rsp_status wmi_ndp_cmd_rsp_status_PROTOTYPE

/**
 * Event response for wmi_ndp_initiator_req
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndp_initiator_rsp_event_fixed_param */
    A_UINT32 tlv_header;
    /** Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /** Copy of transaction_id received in wmi_ndp_initiator_req */
    A_UINT32 transaction_id;
    /** Response status defined in wmi_ndp_cmd_rsp_status*/
    A_UINT32 rsp_status;
    A_UINT32 reason_code;
    /**
     * Unique token Id generated on the initiator/responder
     * side used for a NDP session between two NAN devices
     */
    A_UINT32 ndp_instance_id;
} wmi_ndp_initiator_rsp_event_fixed_param_PROTOTYPE;

#define wmi_ndp_initiator_rsp_event_fixed_param wmi_ndp_initiator_rsp_event_fixed_param_PROTOTYPE

/**
 * Event response for wmi_ndp_responder_req cmd
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndp_responder_rsp_event_fixed_param */
    A_UINT32 tlv_header;
    /** Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /** Copy of transaction_id received in wmi_ndp_responder_req */
    A_UINT32 transaction_id;
    /** Response status defined in wmi_ndp_cmd_rsp_status*/
    A_UINT32 rsp_status;
    A_UINT32 reason_code;
    /**
     * Unique token Id generated on the initiator/responder
     * side used for a NDP session between two NAN devices
     */
    A_UINT32 ndp_instance_id;
    /** NDI mac address of the peer */
    wmi_mac_addr peer_ndi_mac_addr;
    /** Host can create peer if this entry is TRUE */
    A_UINT32 create_peer;
} wmi_ndp_responder_rsp_event_fixed_param_PROTOTYPE;

#define wmi_ndp_responder_rsp_event_fixed_param wmi_ndp_responder_rsp_event_fixed_param_PROTOTYPE

/**
 * Active ndp instance id
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_active_ndp_instance_id */
    A_UINT32 tlv_header;
    /** NDP instance id */
    A_UINT32 ndp_instance_id;
} wmi_active_ndp_instance_id_PROTOTYPE;

#define wmi_active_ndp_instance_id wmi_active_ndp_instance_id_PROTOTYPE

/**
 * NDP end response per ndi
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndp_end_rsp_per_ndi */
    A_UINT32 tlv_header;
    /** Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /** Peer MAC addr */
    wmi_mac_addr peer_mac_addr;
    /** Number of active ndps on this ndi */
    A_UINT32 num_active_ndps_on_ndi;
} wmi_ndp_end_rsp_per_ndi_PROTOTYPE;

#define wmi_ndp_end_rsp_per_ndi wmi_ndp_end_rsp_per_ndi_PROTOTYPE

/**
 * Event response for wmi_ndp_end_req cmd
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndp_end_rsp_event_fixed_param */
    A_UINT32 tlv_header;
    /** Copy of transaction_id received in wmi_ndp_end_req */
    A_UINT32 transaction_id;
    /** Response status defined in wmi_ndp_cmd_rsp_status*/
    A_UINT32 rsp_status;
    A_UINT32 reason_code;
    /**
     * TLV (tag length value ) parameters follow the ndp_end_rsp
     * structure. The TLV's are:
     * wmi_ndp_end_rsp_per_ndi ndp_end_rsp_per_ndis[];
     * wmi_active_ndp_instance_id active_ndp_instances_id[];
     */
} wmi_ndp_end_rsp_event_fixed_param_PROTOTYPE;

#define wmi_ndp_end_rsp_event_fixed_param wmi_ndp_end_rsp_event_fixed_param_PROTOTYPE

/** NAN DATA EVENTS */

/**
 * NDP self role
 */
typedef enum {
    WMI_NDP_INITIATOR_ROLE,
    WMI_NDP_RESPONDER_ROLE,
} wmi_ndp_self_role_PROTOTYPE;

#define wmi_ndp_self_role wmi_ndp_self_role_PROTOTYPE

/**
 * NDP accept policy
 */
typedef enum {
    WMI_NDP_ACCEPT_POLICY_NONE,
    WMI_NDP_ACCEPT_POLICY_ALL,
} wmi_ndp_accept_policy_PROTOTYPE;

#define wmi_ndp_accept_policy wmi_ndp_accept_policy_PROTOTYPE

/**
 * Event indication received on the responder side when a NDP Initiator request/
 * NDP session is initiated on the Initiator side (self role will be NDP_RESPONDER_ROLE)
 *
 * Event indication received on the initiator side when a
 * NDP responder request on the Initiator side (self role will be NDP_INITIATOR_ROLE)
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndp_indication_event_fixed_param */
    A_UINT32 tlv_header;
    /** Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /** Self NDP Role defined in wmi_ndp_self_role */
    A_UINT32 self_ndp_role;
    /** Accept policy defined in wmi_ndp_accept_policy */
    A_UINT32 accept_policy;
    /** Unique Instance Id corresponding to a service/session. */
    A_UINT32 service_instance_id;
    /** Discovery MAC addr of the peer/initiator */
    wmi_mac_addr peer_discovery_mac_addr;
    /** NDI mac address of the peer */
    wmi_mac_addr peer_ndi_mac_addr;
    /**
     * Unique token Id generated on the initiator/responder
     * side used for a NDP session between two NAN devices
     */
    A_UINT32 ndp_instance_id;
    /** Number of bytes in TLV wmi_ndp_cfg */
    A_UINT32 ndp_cfg_len;
    /** Number of bytes in TLV wmi_ndp_app_info */
    A_UINT32 ndp_app_info_len;
    /** Peer NAN Cipher Suite Shared Key */
    A_UINT32 nan_csid;
    /** Actual number of bytes in TLV nan_scid */
    A_UINT32 nan_scid_len;
    /** Self NDI mac address */
    wmi_mac_addr self_ndi_mac_addr;
    /**
     * TLV (tag length value ) parameters follow the ndp_indication
     * structure. The TLV's are:
     * A_UINT8 ndp_cfg[];
     * A_UINT8 ndp_app_info[];
     * A_UINT8 nan_scid[];
     * wmi_ndp_transport_ip_param ndp_transport_ip_param;
     */
} wmi_ndp_indication_event_fixed_param_PROTOTYPE;

#define wmi_ndp_indication_event_fixed_param wmi_ndp_indication_event_fixed_param_PROTOTYPE

/**
 * Event indication of data confirm is received on both
 * initiator and responder side confirming a NDP session
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndp_confirm_event_fixed_param */
    A_UINT32 tlv_header;
    /** Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /**
     * Unique token Id generated on the initiator/responder
     * side used for a NDP session between two NAN devices
     */
    A_UINT32 ndp_instance_id;
    /** NDI mac address of the peer (required to derive target ipv6 address) */
    wmi_mac_addr peer_ndi_mac_addr;
    /** Response Code defined in wmi_ndp_rsp_code */
    A_UINT32 rsp_code;
    /** Number of bytes in TLV wmi_ndp_cfg */
    A_UINT32 ndp_cfg_len;
    /** Number of bytes in TLV wmi_ndp_app_info */
    A_UINT32 ndp_app_info_len;
    /** Reason Code */
    A_UINT32 reason_code;
    /** Number of active ndps on this peer */
    A_UINT32 num_active_ndps_on_peer;
    /** Number of channels on this peer */
    A_UINT32 num_ndp_channels;
    /**
     * TLV (tag length value ) parameters follow the ndp_confirm
     * structure. The TLV's are:
     * A_UINT8 ndp_cfg[];
     * A_UINT8 ndp_app_info[];
     * wmi_channel ndp_channel_list[];
     * A_UINT32 nss_list[]; // Nss indexing should match with channel indexing,
     *                      // since Nss is associated with the channel
     * wmi_ndp_transport_ip_param ndp_transport_ip_param;
     */
} wmi_ndp_confirm_event_fixed_param_PROTOTYPE;

#define wmi_ndp_confirm_event_fixed_param wmi_ndp_confirm_event_fixed_param_PROTOTYPE

/**
 * Event indication received on the initiator/responder side terminating a NDP session
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndp_end_indication */
    A_UINT32 tlv_header;
    /** type defined in  wmi_ndp_end_type */
    A_UINT32 type;
    /** Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /** reason_code  defined in  wmi_ndp_end_reason_code */
    A_UINT32 reason_code;
    /** NDP instance id */
    A_UINT32 ndp_instance_id;
    /** NDI MAC addr of the peer */
    wmi_mac_addr peer_ndi_mac_addr;
    /** Number of active ndps on this peer */
    A_UINT32 num_active_ndps_on_peer;
} wmi_ndp_end_indication_PROTOTYPE;

#define wmi_ndp_end_indication wmi_ndp_end_indication_PROTOTYPE

typedef struct
{
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ndl_schedule_update_fixed_param */
    /** Unique id identifying the VDEV */
    A_UINT32 vdev_id;
   /** peer MAC address */
    wmi_mac_addr peer_macaddr;
   /** Flags:
    *  Bit   0    -> Nss updated
    *  Bit   1    -> channel list updated
    *  Bits  2-31 -> Reserved
    */
    A_UINT32 flags;
    /** num of channels */
    A_UINT32 num_channels;
    /** num of ndp instances */
    A_UINT32 num_ndp_instances;
    /**
     * TLV (tag length value) parameters follow the ndl_schedule_update
     * structure. The TLV's are:
     * A_UINT32 ndp_instance_list[];
     * wmi_channel ndl_channel_list[];
     * A_UINT32 nss_list[]; // Nss indexing should match with channel indexing,
     *                      // since Nss is associate with a channel
     */
} wmi_ndl_schedule_update_fixed_param_PROTOTYPE;

#define wmi_ndl_schedule_update_fixed_param wmi_ndl_schedule_update_fixed_param_PROTOTYPE

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 num_data;
    /* followed by WMITLV_TAG_ARRAY_BYTE */
} wmi_diag_data_container_event_fixed_param;

enum {
    WMI_PDEV_PARAM_TXPOWER_REASON_NONE = 0,
    WMI_PDEV_PARAM_TXPOWER_REASON_SAR,
    WMI_PDEV_PARAM_TXPOWER_REASON_MAX
};

#define PDEV_PARAM_TXPOWER_VALUE_MASK  0x000000FF
#define PDEV_PARAM_TXPOWER_VALUE_SHIFT 0

#define PDEV_PARAM_TXPOWER_REASON_MASK  0x0000FF00
#define PDEV_PARAM_TXPOWER_REASON_SHIFT 8

#define SET_PDEV_PARAM_TXPOWER_VALUE(txpower_param, value)     \
    ((txpower_param) &= ~PDEV_PARAM_TXPOWER_VALUE_MASK, (txpower_param) |= ((value) << PDEV_PARAM_TXPOWER_VALUE_SHIFT))

#define SET_PDEV_PARAM_TXPOWER_REASON(txpower_param, value)     \
    ((txpower_param) &= ~PDEV_PARAM_TXPOWER_REASON_MASK, (txpower_param) |= ((value) << PDEV_PARAM_TXPOWER_REASON_SHIFT))

#define GET_PDEV_PARAM_TXPOWER_VALUE(txpower_param)     \
    (((txpower_param) & PDEV_PARAM_TXPOWER_VALUE_MASK) >> PDEV_PARAM_TXPOWER_VALUE_SHIFT)

#define GET_PDEV_PARAM_TXPOWER_REASON(txpower_param)     \
    (((txpower_param) & PDEV_PARAM_TXPOWER_REASON_MASK) >> PDEV_PARAM_TXPOWER_REASON_SHIFT)

#define PDEV_PARAM_SMART_CHAINMASK_SCHEME_DECISION_MASK 0x00000001
#define PDEV_PARAM_SMART_CHAINMASK_SCHEME_DECISION_SHIFT 0

#define SET_PDEV_SMART_CHAINMASK_SCHEME_DECISION(param, value) \
    do { \
        (param) &= ~PDEV_PARAM_SMART_CHAINMASK_SCHEME_DECISION_MASK; \
        (param) |= (value) << PDEV_PARAM_SMART_CHAINMASK_SCHEME_DECISION_SHIFT; \
    while (0)

#define GET_PDEV_SMART_CHAINMASK_SCHEME_DECISION(param)     \
    (((param) & PDEV_PARAM_SMART_CHAINMASK_SCHEME_DECISION_MASK) >> PDEV_PARAM_SMART_CHAINMASK_SCHEME_DECISION_SHIFT)

/**
 * This command is sent from WLAN host driver to firmware to
 * notify the current modem power state. Host would receive a
 * message from modem when modem is powered on. Host driver
 * would then send this command to firmware. Firmware would then
 * power on WCI-2 (UART) interface for LTE/MWS Coex.
 *
 * This command is only applicable for APQ platform which has
 * modem on the platform. If firmware doesn't support MWS Coex,
 * this command can be dropped by firmware.
 *
 * This is a requirement from modem team that WCN can't toggle
 * UART before modem is powered on.
 */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_modem_power_state_cmd_param */
    A_UINT32 tlv_header;

    /** Modem power state parameter */
    A_UINT32 modem_power_state;
} wmi_modem_power_state_cmd_param;

enum {
    WMI_MODEM_STATE_OFF = 0,
    WMI_MODEM_STATE_ON
};

/**
 * This command is sent from WLAN host driver to firmware to
 * notify the updated Specific Absorption Rate (SAR) limits.
 * A critical regulation for FCC compliance, OEMs require methods to set
 * limits on TX power of WLAN/WWAN.
 * Host would receive instructions on what to set the limits per
 * band/chain/modulation to, it would then interpret and send the limits
 * to FW using this WMI message.
 * Since it is possible to have too many commands to fit into one message,
 * FW will keep receiving the messages, until it finds one with
 * commit_limits = 1, at which point it will apply all the received
 * specifications.
*/
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_sar_limits_cmd_param */
    A_UINT32 tlv_header;

    /** when set to WMI_SAR_FEATURE_ON, enable SAR feature;
     *  with BDF (SET_0 to 4) or WMI
     * if set to WMI_SAR_FEATURE_OFF, disable feature;
     * if set to WMI_SAR_FEATURE_NO_CHANGE, do not alter state of feature;
     */
    A_UINT32 sar_enable;

    /** number of items in sar_limits[] */
    A_UINT32 num_limit_rows;

    /** once received and is set to 1, FW will calculate the power limits
     * and send set_power command to apply them.
     * Otherwise just update local values stored in FW until a future msg
     * with commit_limits=1 arrives.
    */
    A_UINT32 commit_limits;

    /**
     * TLV (tag length value) parameters follow the sar_limit_cmd_row
     * structure. The TLV's are:
     * wmi_sar_limit_cmd_row sar_limits[];
     */
} wmi_sar_limits_cmd_fixed_param;

enum wmi_sar_feature_state_flags {
    WMI_SAR_FEATURE_OFF = 0,
    WMI_SAR_FEATURE_ON_SET_0,
    WMI_SAR_FEATURE_ON_SET_1,
    WMI_SAR_FEATURE_ON_SET_2,
    WMI_SAR_FEATURE_ON_SET_3,
    WMI_SAR_FEATURE_ON_SET_4,
    WMI_SAR_FEATURE_NO_CHANGE,
    WMI_SAR_FEATURE_ON_USER_DEFINED,
    WMI_SAR_FEATURE_ON_SAR_V2_0,
};

typedef struct {
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_sar_limit_cmd_row */

    /** Current values: WMI_SAR_2G_ID, WMI_SAR_5G_ID. Can be extended by adding
     * new band_id values .
     */
    A_UINT32 band_id;

    A_UINT32 chain_id;

    /** Current values: WMI_SAR_MOD_CCK, WMI_SAR_MOD_OFDM */
    A_UINT32 mod_id;

    /**
     * To be backwards-compatible with older code use a union with
     * limit_value & limit_index as alternate names / interpretations
     * of the same message information element.
     * The older code still uses limit_value, while the new code will
     * use limit_index.
     * The interpretation of the field as value or index depends on
     * WMI_SAR_FEATURE_ON_*
     * WMI_SAR_FEATURE_ON_SAR_V2_0 will use it as index, other case
     * still use it as value.
     */
    union {
        /** actual power limit value, in steps of 0.5 dBm */
        A_UINT32 limit_value;
        A_UINT32 limit_index;
    };

    /** in case the OEM doesn't care about one of the qualifiers from above,
     * the bit for that qualifier within the validity_bitmap can be set to 0
     * so that limit is applied to all possible cases of this qualifier
     * (i.e. if a qualifier's validity_bitmap flag is 0, the qualifier is
     * treated as a wildcard).
     * Current masks:
     *     WMI_SAR_BAND_ID_VALID_MASK
     *     WMI_SAR_CHAIN_ID_VALID_MASK
     *     WMI_SAR_MOD_ID_VALID_MASK
     * Example: if !WMI_IS_SAR_MOD_ID_VALID(bitmap),
     *     it means apply same limit_value to both WMI_SAR_MOD_CCK and
     *     WMI_SAR_MOD_OFDM cases.
     */

     A_UINT32 validity_bitmap;
} wmi_sar_limit_cmd_row;

enum wmi_sar_band_id_flags {
     WMI_SAR_2G_ID = 0,
     WMI_SAR_5G_ID
};

enum wmi_sar_mod_id_flags {
     WMI_SAR_MOD_CCK = 0,
     WMI_SAR_MOD_OFDM
};

/**
 * This message is sent from FW to WLAN host to inform the host of the
 * updated Specific Absorption Rate (SAR) limits currently in use.
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_sar_get_limits_event_param */
    A_UINT32 tlv_header;

    /** when set to WMI_SAR_FEATURE_ON_*, SAR feature is enabled
     *  with BDF (SET_0 to 4) or WMI
     * if set to WMI_SAR_FEATURE_OFF, feature is disabled;
     */
    A_UINT32 sar_enable;

    /**
     * number of items in sar_limits[].
     *  used when sar_enable == WMI_SAR_FEATURE_ON_USER_DEFINED.
     *  Should be zero if any of the BDF sets is activated.
     */
    A_UINT32 num_limit_rows;

    /**
     * TLV (tag length value) parameters follow the sar_get_limit_event_row
     * structure. The TLV's are:
     * wmi_sar_get_limit_event_row sar_limits[num_limit_rows];
     */
} wmi_sar_get_limits_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_sar_get_limit_event_row */

    /** Current values: WMI_SAR_2G_ID, WMI_SAR_5G_ID. Can be extended by adding
     * new band_id values .
     */
    A_UINT32 band_id;

    A_UINT32 chain_id;

    /** Current values: WMI_SAR_MOD_CCK, WMI_SAR_MOD_OFDM */
    A_UINT32 mod_id;

    /** actual power limit value, in steps of 0.5 dBm */
    A_UINT32 limit_value;
} wmi_sar_get_limit_event_row;

#define WMI_SAR_BAND_ID_VALID_MASK      (0x1)
#define WMI_SAR_CHAIN_ID_VALID_MASK     (0x2)
#define WMI_SAR_MOD_ID_VALID_MASK       (0x4)

#define WMI_SET_SAR_BAND_ID_VALID(bitmap)    ((bitmap) |= WMI_SAR_BAND_ID_VALID_MASK)
#define WMI_SET_SAR_CHAIN_ID_VALID(bitmap)   ((bitmap) |= WMI_SAR_CHAIN_ID_VALID_MASK)
#define WMI_SET_SAR_MOD_ID_VALID(bitmap)     ((bitmap) |= WMI_SAR_MOD_ID_VALID_MASK)

#define WMI_IS_SAR_BAND_ID_VALID(bitmap)     ((bitmap) & WMI_SAR_BAND_ID_VALID_MASK)
#define WMI_IS_SAR_CHAIN_ID_VALID(bitmap)    ((bitmap) & WMI_SAR_CHAIN_ID_VALID_MASK)
#define WMI_IS_SAR_MOD_ID_VALID(bitmap)      ((bitmap) & WMI_SAR_MOD_ID_VALID_MASK)

/**
 * This command is sent from WLAN host driver to firmware to
 * get current Specific Absorption Rate (SAR) limits status from firmware.
 * The command does not require any parameters as of now.
 */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_sar_get_limits_cmd_param */
    A_UINT32 tlv_header;
    /** currently no parameters are required. Reserved bit field for future use added */
    /*  All bits need to be set to 0 while it is a reserved field. */
    A_UINT32 reserved;
} wmi_sar_get_limits_cmd_fixed_param;

#define WMI_ROAM_AUTH_STATUS_CONNECTED       0x1 /** connected, but not authenticated */
#define WMI_ROAM_AUTH_STATUS_AUTHENTICATED   0x2 /** connected and authenticated */

/** WMI_ROAM_SYNCH_EVENT: roam synch event triggering the host propagation logic
    generated whenever firmware roamed to new AP silently and
    (a) If the host is awake, FW sends the event to the host immediately .
    (b) If host is in sleep then either
        (1) FW waits until  host sends WMI_PDEV_RESUME_CMDID or WMI_WOW_HOSTWAKEUP_FROM_SLEEP_CMDID
    command to FW (part of host wake up sequence  from low power mode) before sending the event host.
        (2) data/mgmt frame is received from roamed AP, which needs to return to host
*/

#define GTK_OFFLOAD_KCK_EXTENDED_BYTES 32

typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_key_material */
    A_UINT32 tlv_header;

    A_UINT8  kck[GTK_OFFLOAD_KCK_BYTES]; /* EAPOL-Key Key Confirmation Key (KCK) */
    A_UINT8  kek[GTK_OFFLOAD_KEK_BYTES]; /* EAPOL-Key Key Encryption Key (KEK) */
    A_UINT8  replay_counter[GTK_REPLAY_COUNTER_BYTES];
} wmi_key_material;

typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_key_material_ext */
    A_UINT32 tlv_header;
    A_UINT8  key_buffer[GTK_OFFLOAD_KEK_EXTENDED_BYTES+GTK_OFFLOAD_KCK_EXTENDED_BYTES+GTK_REPLAY_COUNTER_BYTES]; /*the split of kck, kek should be known to host based on akmp*/
} wmi_key_material_ext;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_synch_event_fixed_param  */
    /** Unique id identifying the VDEV on which roaming is done by firmware */
    A_UINT32 vdev_id;
    /** auth_status: connected or authorized */
    A_UINT32 auth_status;
     /*
      *  roam_reason:
      *  bits 0-3 for roam reason   see WMI_ROAM_REASON_XXX
      *  bits 4-5 for subnet status see WMI_ROAM_SUBNET_CHANGE_STATUS_XXX.
      *  bit  6   for HW mode status, set 1 to indicate host to schedule
      *           HW mode change, see WMI_ROAM_REQUEST_HOST_HW_MODE_CHANGE.
      */
    A_UINT32 roam_reason;
    /** associated AP's rssi calculated by FW when reason code is WMI_ROAM_REASON_LOW_RSSI. not valid if roam_reason is BMISS */
    A_UINT32 rssi;
    /** MAC address of roamed AP */
    wmi_mac_addr bssid;     /* BSSID */
    /** whether the frame is beacon or probe rsp */
    A_UINT32 is_beacon;
    /** the length of beacon/probe rsp */
    A_UINT32 bcn_probe_rsp_len;
    /** the length of reassoc rsp */
    A_UINT32 reassoc_rsp_len;
    /** the length of reassoc req */
    A_UINT32 reassoc_req_len;
    /**
     * TLV (tag length value ) parameters follows roam_synch_event
     * The TLV's are:
     *     A_UINT8 bcn_probe_rsp_frame[];  length identified by bcn_probe_rsp_len
     *     A_UINT8 reassoc_rsp_frame[];  length identified by reassoc_rsp_len
     *     wmi_channel chan;
     *     wmi_key_material key;
     *     A_UINT32 status; subnet changed status not being used currently.
     *         will pass the information using roam_status.
     *     A_UINT8 reassoc_req_frame[];  length identified by reassoc_req_len
     *     wmi_key_material_ext key_ext
     **/
} wmi_roam_synch_event_fixed_param;

/**
 * The WMI_ROAM_SYNCH_FRAME_EVENTID message is used in conjunction with the
 * WMI_ROAM_SYNCH_EVENTID message.  The former will be sent 1st followed by
 * the latter for cases where the WMI_ROAM_SYNCH_EVENTID message size would
 * exceed 2K.  The more_frag field in the WMI_ROAM_SYNCH_FRAME_EVENTID informs
 * the host whether more WMI_ROAM_SYNCH_FRAME_EVENTID messages would follow;
 * after the WMI_ROAM_SYNCH_FRAME_EVENTID messages the target sends the
 * WMI_ROAM_SYNCH_EVENTID with bcn_probe_rsp_len, reassoc_rsp_len, and
 * reassoc_rsp_len set to 0.
 */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_roam_synch_frame_event_fixed_param  */
    /** Unique id identifying the VDEV on which roaming is done by firmware */
    A_UINT32 vdev_id;
    /** More frags to follow? */
    A_UINT32 more_frag;
    /** whether the frame is beacon or probe rsp */
    A_UINT32 is_beacon;
    /** the length of beacon/probe rsp */
    A_UINT32 bcn_probe_rsp_len;
    /** the length of reassoc rsp */
    A_UINT32 reassoc_rsp_len;
    /** the length of reassoc req */
    A_UINT32 reassoc_req_len;
    /**
     * TLV (tag length value) parameters follows roam_synch_event
     * The TLV's are:
     *     A_UINT8 bcn_probe_rsp_frame[bcn_probe_rsp_len];
     *     A_UINT8 reassoc_rsp_frame[reassoc_rsp_len];
     *     A_UINT8 reassoc_req_frame[reassoc_req_len];
     */
} wmi_roam_synch_frame_event_fixed_param;

#define WMI_PEER_ESTIMATED_LINKSPEED_INVALID    0xFFFFFFFF

typedef struct {
    /* TLV tag and len; tag equals WMITLV_TAG_STRUC_ wmi_peer_get_estimated_linkspeed_cmd_fixed_param */
    A_UINT32 tlv_header;
    /** MAC address of the peer for which the estimated link speed is required. */
    wmi_mac_addr peer_macaddr;
    /* Set to 1 only if vdev_id field is valid */
    A_UINT32 valid_vdev_id;
    /* VDEV to which the peer belongs to */
    A_UINT32 vdev_id;
} wmi_peer_get_estimated_linkspeed_cmd_fixed_param;

typedef struct {
    /* TLV tag and len; tag equals WMITLV_TAG_STRUC_ wmi_peer_estimated_linkspeed_event_fixed_param */
    A_UINT32 tlv_header;
    /** MAC address of the peer for which the estimated link speed is required.
     */
    wmi_mac_addr peer_macaddr;
  /* Estimated link speed in kbps.
   * When est_linkspeed_kbps is not valid, the value is set to WMI_PEER_ESTIMATED_LINKSPEED_INVALID.
   */
    A_UINT32 est_linkspeed_kbps;
    /* Set to 1 only if vdev_id field is valid */
    A_UINT32 valid_vdev_id;
    /* VDEV to which the peer belongs to */
    A_UINT32 vdev_id;
} wmi_peer_estimated_linkspeed_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals */
    /* vdev ID */
    A_UINT32 vdev_id;
    A_UINT32 data_len; /** length in byte of data[]. */
    /* This structure is used to send REQ binary blobs
     * from application/service to firmware where Host drv is pass through .
     * Following this structure is the TLV:
     *     A_UINT8 data[];    // length in byte given by field data_len.
     */
} wmi_req_stats_ext_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_stats1_event_fix_param */
    A_UINT32 vdev_id; /** vdev ID */
    A_UINT32 data_len; /** length in byte of data[]. */
    /* This structure is used to send REQ binary blobs
     * from firmware to application/service where Host drv is pass through .
     * Following this structure is the TLV:
     *     A_UINT8 data[];    // length in byte given by field data_len.
     */
} wmi_stats_ext_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_delete_resp_event_fixed_param  */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
} wmi_peer_delete_resp_event_fixed_param;

typedef struct {
    /* TLV tag and len; tag equals WMITLV_TAG_STRUC_ wmi_peer_state_event_fixed_param */
    A_UINT32 tlv_header;
    A_UINT32 vdev_id; /* vdev ID */
    /* MAC address of the peer for which the estimated link speed is required.*/
    wmi_mac_addr peer_macaddr;
    A_UINT32 state; /* peer state */
} wmi_peer_state_event_fixed_param;

typedef struct {
    /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_assoc_conf_event_fixed_param */
    A_UINT32 tlv_header;
    /* unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /* peer MAC address */
    wmi_mac_addr peer_macaddr;
} wmi_peer_assoc_conf_event_fixed_param;

enum {
    WMI_2G4_HT40_OBSS_SCAN_PASSIVE = 0,    /** scan_type: passive */
    WMI_2G4_HT40_OBSS_SCAN_ACTIVE, /** scan_type: active */
};

typedef struct {
    /**
     * TLV tag and len;
     * tag equals WMITLV_TAG_STRUC_wmi_obss_scan_enalbe_cmd_fixed_param
     */
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    /**
     * active or passive. if active all the channels are actively scanned.
     *  if passive then all the channels are passively scanned
     */
    A_UINT32 scan_type;
    /**
     * FW can perform multiple scans with in a  OBSS scan interval.
     * For each scan,
     *  if the scan is passive then obss_scan_passive_dwell is minimum dwell to be used for each channel  ,
     *  if the scan is active then obss_scan_active_dwell is minimum dwell to be used for each channel .
     *   The unit for these 2 parameters is TUs.
     */
    A_UINT32 obss_scan_passive_dwell;
    A_UINT32 obss_scan_active_dwell;
    /**
     * OBSS scan interval . FW needs to perform one or more OBSS scans within this interval and fulfill the
     *  both min and total per channel dwell time requirement
     */
    A_UINT32 bss_channel_width_trigger_scan_interval;
    /**
     * FW can perform multiple scans with in a  OBSS scan interval.
     * For each scan,
     * the total per channel dwell time across all scans with in OBSS scan interval should be
     * atleast obss_scan_passive_total_per channel for passive scas and obss_scan_active_total_per channel
     * for active scans and ,
     *   The unit for these 2 parameters is TUs.
     */
    A_UINT32 obss_scan_passive_total_per_channel;
    A_UINT32 obss_scan_active_total_per_channel;
    A_UINT32 bss_width_channel_transition_delay_factor; /** parameter to check exemption from scan */
    A_UINT32 obss_scan_activity_threshold; /** parameter to check exemption from scan */
    /** following two parameters used by FW to fill IEs when sending 20/40 coexistence action frame to AP */
    A_UINT32 forty_mhz_intolerant; /** STA 40M bandwidth intolerant capability */
    A_UINT32 current_operating_class; /** STA current operating class */
    /** length of 2.4GHz channel list to scan at, channel list in tlv->channels[] */
    A_UINT32 channel_len;
    /** length of optional ie data to append to probe reqest when active scan, ie data in tlv->ie_field[] */
    A_UINT32 ie_len;
} wmi_obss_scan_enable_cmd_fixed_param;

typedef struct {
    /**
    * TLV tag and len;
    * tag equals WMITLV_TAG_STRUC_wmi_obss_scan_disalbe_cmd_fixed_param
    */
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
} wmi_obss_scan_disable_cmd_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_offload_prb_rsp_tx_status_event_fixed_param */
    A_UINT32 tlv_header;
    /** unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /** prb rsp tx status, values defined in enum WMI_FRAME_TX_STATUS */
    A_UINT32 tx_status;
}wmi_offload_prb_rsp_tx_status_event_fixed_param;

typedef enum {
    WMI_FRAME_TX_OK,            /* frame tx ok */
    WMI_FRAME_TX_XRETRY,        /* excessivley retried */
    WMI_FRAME_TX_DROP,          /* frame dropped by FW due to resources */
    WMI_FRAME_TX_FILTERED,      /* frame filtered by hardware */
} WMI_FRAME_TX_STATUS;

/**
 * This command is sent from WLAN host driver to firmware to
 * request firmware to send the latest channel avoidance range
 * to host.
 *
 * This command is only applicable for APQ platform which has
 * modem on the platform. If firmware doesn't support MWS Coex,
 * this command can be dropped by firmware.
 *
 * Host would send this command to firmware to request a channel
 * avoidance information update.
 */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_chan_avoid_update_cmd_param */
    A_UINT32 tlv_header;
} wmi_chan_avoid_update_cmd_param;

/* ExtScan operation mode */
typedef enum {
   WMI_EXTSCAN_MODE_NONE          = 0x0000,
   WMI_EXTSCAN_MODE_START         = 0x0001,    // ExtScan/TableMonitoring operation started
   WMI_EXTSCAN_MODE_STOP          = 0x0002,    // ExtScan/TableMonitoring operation stopped
   WMI_EXTSCAN_MODE_IGNORED       = 0x0003,    // ExtScan command ignored due to error
} wmi_extscan_operation_mode;

/* Channel Mask */
typedef enum {
   WMI_CHANNEL_BAND_UNSPECIFIED = 0x0000,
   WMI_CHANNEL_BAND_24          = 0x0001,    // 2.4 channel
   WMI_CHANNEL_BAND_5_NON_DFS   = 0x0002,    // 5G Channels (No DFS channels)
   WMI_CHANNEL_BAND_DFS         = 0x0004,    // DFS channels
} wmi_channel_band_mask;

typedef enum {
    WMI_EXTSCAN_CYCLE_STARTED_EVENT     = 0x0001,
    WMI_EXTSCAN_CYCLE_COMPLETED_EVENT   = 0x0002,
    WMI_EXTSCAN_BUCKET_STARTED_EVENT    = 0x0004,
    WMI_EXTSCAN_BUCKET_COMPLETED_EVENT  = 0x0008,
    WMI_EXTSCAN_BUCKET_FAILED_EVENT     = 0x0010,
    WMI_EXTSCAN_BUCKET_OVERRUN_EVENT    = 0x0020,
    WMI_EXTSCAN_THRESHOLD_NUM_SCANS     = 0x0040,
    WMI_EXTSCAN_THRESHOLD_PERCENT       = 0x0080,

    WMI_EXTSCAN_EVENT_MAX               = 0x8000
} wmi_extscan_event_type;

#define WMI_EXTSCAN_CYCLE_EVENTS_MASK    (WMI_EXTSCAN_CYCLE_STARTED_EVENT   | \
                                          WMI_EXTSCAN_CYCLE_COMPLETED_EVENT)

#define WMI_EXTSCAN_BUCKET_EVENTS_MASK   (WMI_EXTSCAN_BUCKET_STARTED_EVENT   | \
                                          WMI_EXTSCAN_BUCKET_COMPLETED_EVENT | \
                                          WMI_EXTSCAN_BUCKET_FAILED_EVENT    | \
                                          WMI_EXTSCAN_BUCKET_OVERRUN_EVENT)

typedef enum {
    WMI_EXTSCAN_NO_FORWARDING         = 0x0000,
    WMI_EXTSCAN_FORWARD_FRAME_TO_HOST = 0x0001
} wmi_extscan_forwarding_flags;

typedef enum {
    WMI_EXTSCAN_USE_MSD                 = 0x0001,    // Use Motion Sensor Detection */
    WMI_EXTSCAN_EXTENDED_BATCHING_EN    = 0x0002,    // Extscan LPASS extended batching feature is supported and enabled
} wmi_extscan_configuration_flags;

typedef enum {
    WMI_EXTSCAN_BUCKET_CACHE_RESULTS    = 0x0001,    // Cache the results of bucket whose configuration flags has this bit set
    WMI_EXTSCAN_REPORT_EVENT_CONTEXT_HUB = 0x0002,   // Report ext scan results to context hub or not.
} wmi_extscan_bucket_configuration_flags;

typedef enum {
    WMI_EXTSCAN_STATUS_OK    = 0,
    WMI_EXTSCAN_STATUS_ERROR = 0x80000000,
    WMI_EXTSCAN_STATUS_INVALID_PARAMETERS,
    WMI_EXTSCAN_STATUS_INTERNAL_ERROR
} wmi_extscan_start_stop_status;

typedef struct {
    /** Request ID - to identify command. Cannot be 0 */
    A_UINT32     request_id;
    /** Requestor ID - client requesting ExtScan */
    A_UINT32     requestor_id;
    /** VDEV id(interface) that is requesting scan */
    A_UINT32     vdev_id;
} wmi_extscan_command_id;

typedef struct {
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_ARRAY_STRUC */
    /** channel number */
    A_UINT32    channel;

    /** dwell time in msec - use defaults if 0 */
    A_UINT32    min_dwell_time;
    A_UINT32    max_dwell_time;
    /** passive/active channel and other flags */
    A_UINT32    control_flags;                        // 0 => active, 1 => passive scan; ignored for DFS
} wmi_extscan_bucket_channel;

/* Scan Bucket specification */
typedef struct {
    A_UINT32        tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_ARRAY_STRUC */
    /** Bucket ID  - 0-based */
    A_UINT32        bucket_id;
    /** ExtScan events subscription - events to be reported to client (see wmi_extscan_event_type) */
    A_UINT32        notify_extscan_events;
    /** Options to forward scan results - see wmi_extscan_forwarding_flags */
    A_UINT32        forwarding_flags;
    /** ExtScan configuration flags - wmi_extscan__bucket_configuration_flags */
    A_UINT32        configuration_flags;
    /** DEPRECATED member: multiplier to be applied to the periodic scan's base period */
    A_UINT32        base_period_multiplier;
    /** dwell time in msec on active channels - use defaults if 0 */
    A_UINT32        min_dwell_time_active;
    A_UINT32        max_dwell_time_active;
    /** dwell time in msec on passive channels - use defaults if 0 */
    A_UINT32        min_dwell_time_passive;
    A_UINT32        max_dwell_time_passive;
    /** see wmi_channel_band_mask; when equal to WMI_CHANNEL_UNSPECIFIED, use channel list */
    A_UINT32        channel_band;
    /** number of channels (if channel_band is WMI_CHANNEL_UNSPECIFIED) */
    A_UINT32        num_channels;
   /** scan period upon start or restart of the bucket - periodicity of the bucket to begin with */
    A_UINT32        min_period;
    /** period above which exponent is not applied anymore */
    A_UINT32        max_period;
    /** back off value to be applied to bucket's periodicity after exp_max_step_count scan cycles
      * new_bucket_period = last_bucket_period + last_exponent_period * exp_backoff
      */
    A_UINT32        exp_backoff;
    /** number of scans performed at a given periodicity after which exponential back off value is
       * applied to current periodicity to obtain a newer one
       */
    A_UINT32        exp_max_step_count;
/** Followed by the variable length TLV chan_list:
 *  wmi_extscan_bucket_channel chan_list[] */
} wmi_extscan_bucket;

typedef struct {
    A_UINT32     tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_extscan_start_cmd_fixed_param */
    /** Request ID - to identify command. Cannot be 0 */
    A_UINT32     request_id;
    /** Requestor ID - client requesting ExtScan */
    A_UINT32     requestor_id;
    /** VDEV id(interface) that is requesting scan */
    A_UINT32     vdev_id;
    /** table ID - to allow support for multiple simultaneous requests */
    A_UINT32     table_id;
    /** Base period (milliseconds) used by scan buckets to define periodicity of the scans */
    A_UINT32     base_period;
    /** Maximum number of iterations to run - one iteration is the scanning of the least frequent bucket */
    A_UINT32     max_iterations;
    /** Options to forward scan results - see wmi_extscan_forwarding_flags */
    A_UINT32     forwarding_flags;
    /** ExtScan configuration flags - wmi_extscan_configuration_flags */
    A_UINT32     configuration_flags;
    /** ExtScan events subscription - bitmask indicating which events should be send to client (see wmi_extscan_event_type) */
    A_UINT32     notify_extscan_events;
    /** Scan Priority, input to scan scheduler */
    A_UINT32     scan_priority;
    /** Maximum number of BSSIDs to cache on each scan cycle */
    A_UINT32     max_bssids_per_scan_cycle;
    /** Minimum RSSI value to report */
    A_UINT32     min_rssi;
    /** Maximum table usage in percentage */
    A_UINT32     max_table_usage;
    /** default dwell time in msec on active channels */
    A_UINT32     min_dwell_time_active;
    A_UINT32     max_dwell_time_active;
    /** default dwell time in msec on passive channels */
    A_UINT32     min_dwell_time_passive;
    A_UINT32     max_dwell_time_passive;
    /** min time in msec on the BSS channel,only valid if atleast one VDEV is active*/
    A_UINT32     min_rest_time;
    /** max rest time in msec on the BSS channel,only valid if at least one VDEV is active*/
    /** the scanner will rest on the bss channel at least min_rest_time. after min_rest_time the scanner
     *  will start checking for tx/rx activity on all VDEVs. if there is no activity the scanner will
     *  switch to off channel. if there is activity the scanner will let the radio on the bss channel
     *  until max_rest_time expires.at max_rest_time scanner will switch to off channel
     *  irrespective of activity. activity is determined by the idle_time parameter.
     */
    A_UINT32     max_rest_time;
    /** time before sending next set of probe requests.
     *   The scanner keeps repeating probe requests transmission with period specified by repeat_probe_time.
     *   The number of probe requests specified depends on the ssid_list and bssid_list
     */
    /** Max number of probes to be sent */
    A_UINT32     n_probes;
    /** time in msec between 2 sets of probe requests. */
    A_UINT32     repeat_probe_time;
    /** time in msec between 2 consequetive probe requests with in a set. */
    A_UINT32     probe_spacing_time;
    /** data inactivity time in msec on bss channel that will be used by scanner for measuring the inactivity  */
    A_UINT32     idle_time;
    /** maximum time in msec allowed for scan  */
    A_UINT32     max_scan_time;
    /** delay in msec before sending first probe request after switching to a channel */
    A_UINT32     probe_delay;
    /** Scan control flags */
    A_UINT32     scan_ctrl_flags;
    /** Burst duration time in msec*/
    A_UINT32     burst_duration;

    /** number of bssids in the TLV bssid_list[] */
    A_UINT32     num_bssid;
    /** number of ssid in the TLV ssid_list[] */
    A_UINT32     num_ssids;
    /** number of bytes in TLV ie_data[] */
    A_UINT32     ie_len;
    /** number of buckets in the TLV bucket_list[] */
    A_UINT32     num_buckets;
    /** in number of scans, send notifications to host after these many scans */
    A_UINT32    report_threshold_num_scans;
    /** number of channels in channel_list[] determined by the
        sum of wmi_extscan_bucket.num_channels in array  */

/**
 * TLV (tag length value ) parameters follow the extscan_cmd
 * structure. The TLV's are:
 *     wmi_ssid                   ssid_list[];
 *     wmi_mac_addr               bssid_list[];
 *     A_UINT8                    ie_data[];
 *     wmi_extscan_bucket         bucket_list[];
 *     wmi_extscan_bucket_channel channel_list[];
 */
} wmi_extscan_start_cmd_fixed_param;

typedef struct {
    A_UINT32     tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_extscan_stop_cmd_fixed_param */
    /** Request ID - to match running command. 0 matches any request */
    A_UINT32     request_id;
    /** Requestor ID - client requesting stop */
    A_UINT32     requestor_id;
    /** VDEV id(interface) that is requesting scan */
    A_UINT32     vdev_id;
    /** table ID - to allow support for multiple simultaneous requests */
    A_UINT32     table_id;
} wmi_extscan_stop_cmd_fixed_param;

enum wmi_extscan_get_cached_results_flags {
    WMI_EXTSCAN_GET_CACHED_RESULTS_FLAG_NONE        = 0x0000,
    WMI_EXTSCAN_GET_CACHED_RESULTS_FLAG_FLUSH_TABLE = 0x0001
};

typedef struct {
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_extscan_get_cached_results_cmd_fixed_param */
    /** request ID - used to correlate command with events */
    A_UINT32    request_id;
    /** Requestor ID - client that requested results */
    A_UINT32    requestor_id;
    /** VDEV id(interface) that is requesting scan */
    A_UINT32    vdev_id;
    /** table ID - to allow support for multiple simultaneous requests */
    A_UINT32    table_id;
    /** maximum number of results to be returned  */
    A_UINT32    max_results;
    /** flush BSSID list - wmi_extscan_get_cached_results_flags */
    A_UINT32    control_flags;    // enum wmi_extscan_get_cached_results_flags
} wmi_extscan_get_cached_results_cmd_fixed_param;

typedef struct {
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_extscan_get_wlan_change_results_cmd_fixed_param */
    /** request ID - used to correlate command with events */
    A_UINT32    request_id;
    /** Requestor ID - client that requested results */
    A_UINT32    requestor_id;
    /** VDEV id(interface) that is requesting scan */
    A_UINT32    vdev_id;
    /** table ID - to allow support for multiple simultaneous requests */
    A_UINT32    table_id;
} wmi_extscan_get_wlan_change_results_cmd_fixed_param;

typedef struct {
    A_UINT32        tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_ARRAY_STRUC */
    /**bssid */
    wmi_mac_addr    bssid;
    /**channel number */
    A_UINT32        channel;
    /**upper RSSI limit */
    A_UINT32        upper_rssi_limit;
    /**lower RSSI limit */
    A_UINT32        lower_rssi_limit;
} wmi_extscan_wlan_change_bssid_param;

typedef struct {
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_extscan_configure_wlan_change_monitor_cmd_fixed_param */
    /** Request ID - to identify command. Cannot be 0 */
    A_UINT32    request_id;
    /** Requestor ID - client requesting wlan change monitoring */
    A_UINT32    requestor_id;
    /** VDEV id(interface) that is requesting scan */
    A_UINT32    vdev_id;
    /** table ID - to allow support for multiple simultaneous tables */
    A_UINT32    table_id;
    /** operation mode: start/stop */
    A_UINT32    mode;    // wmi_extscan_operation_mode
    /** number of rssi samples to store */
    A_UINT32    max_rssi_samples;
    /** number of samples to use to calculate RSSI average */
    A_UINT32    rssi_averaging_samples;
    /** number of scans to confirm loss of contact with RSSI */
    A_UINT32    lost_ap_scan_count;
    /** number of out-of-range BSSIDs necessary to send event */
    A_UINT32    max_out_of_range_count;
    /** total number of bssid signal descriptors (in all pages) */
    A_UINT32    total_entries;
    /** index of the first bssid entry found in the TLV wlan_change_descriptor_list*/
    A_UINT32    first_entry_index;
    /** number of bssid signal descriptors in this page */
    A_UINT32    num_entries_in_page;
    /* Following this structure is the TLV:
     *     wmi_extscan_wlan_change_bssid_param wlan_change_descriptor_list[];    // number of elements given by field num_page_entries.
     */
} wmi_extscan_configure_wlan_change_monitor_cmd_fixed_param;

typedef struct {
    A_UINT32        tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_ARRAY_STRUC */
    /**bssid */
    wmi_mac_addr    bssid;
    /**RSSI min threshold for reporting */
    A_UINT32        min_rssi;
    /**Deprecated entry - channel number */
    A_UINT32        channel;
    /** RSSI max threshold for reporting */
    A_UINT32        max_rssi;
} wmi_extscan_hotlist_entry;

typedef struct {
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_extscan_configure_hotlist_monitor_cmd_fixed_param */
    /** Request ID - to identify command. Cannot be 0 */
    A_UINT32    request_id;
    /** Requestor ID - client requesting hotlist monitoring */
    A_UINT32    requestor_id;
    /** VDEV id(interface) that is requesting scan */
    A_UINT32    vdev_id;
    /** table ID - to allow support for multiple simultaneous tables */
    A_UINT32    table_id;
    /** operation mode: start/stop */
    A_UINT32    mode;    // wmi_extscan_operation_mode
    /**total number of bssids (in all pages) */
    A_UINT32    total_entries;
    /**index of the first bssid entry found in the TLV wmi_extscan_hotlist_entry*/
    A_UINT32    first_entry_index;
    /**number of bssids in this page */
    A_UINT32    num_entries_in_page;
    /** number of consecutive scans to confirm loss of contact with AP */
    A_UINT32    lost_ap_scan_count;
    /* Following this structure is the TLV:
     *     wmi_extscan_hotlist_entry hotlist[];    // number of elements given by field num_page_entries.
     */
} wmi_extscan_configure_hotlist_monitor_cmd_fixed_param;

typedef struct {
    A_UINT32        tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_ARRAY_STRUC */
    /**ssid */
    wmi_ssid        ssid;
    /**band */
    A_UINT32        band;
    /**RSSI threshold for reporting */
    A_UINT32        min_rssi;
    A_UINT32        max_rssi;
} wmi_extscan_hotlist_ssid_entry;

typedef struct {
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_extscan_configure_hotlist_ssid_monitor_cmd_fixed_param */
    /** Request ID - to identify command. Cannot be 0 */
    A_UINT32    request_id;
    /** Requestor ID - client requesting hotlist ssid monitoring */
    A_UINT32    requestor_id;
    /** VDEV id(interface) that is requesting scan */
    A_UINT32    vdev_id;
    /** table ID - to allow support for multiple simultaneous tables */
    A_UINT32    table_id;
    /** operation mode: start/stop */
    A_UINT32    mode;    // wmi_extscan_operation_mode
    /**total number of ssids (in all pages) */
    A_UINT32    total_entries;
    /**index of the first ssid entry found in the TLV wmi_extscan_hotlist_ssid_entry*/
    A_UINT32    first_entry_index;
    /**number of ssids in this page */
    A_UINT32    num_entries_in_page;
    /** number of consecutive scans to confirm loss of an ssid **/
    A_UINT32    lost_ap_scan_count;
    /* Following this structure is the TLV:
     *     wmi_extscan_hotlist_ssid_entry hotlist_ssid[];    // number of elements given by field num_page_entries.
     */
} wmi_extscan_configure_hotlist_ssid_monitor_cmd_fixed_param;


typedef struct {
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_ARRAY_STRUC */
    /** table ID - to allow support for multiple simultaneous tables */
    A_UINT32    table_id;
    /** size in bytes of scan cache entry */
    A_UINT32    scan_cache_entry_size;
    /** maximum number of scan cache entries */
    A_UINT32    max_scan_cache_entries;
    /** maximum number of buckets per extscan request */
    A_UINT32    max_buckets;
    /** maximum number of BSSIDs that will be stored in each scan (best n/w as per RSSI) */
    A_UINT32    max_bssid_per_scan;
    /** table usage level at which indication must be sent to host */
    A_UINT32    max_table_usage_threshold;
} wmi_extscan_cache_capabilities;

typedef struct {
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_ARRAY_STRUC */
    /** table ID - to allow support for multiple simultaneous tables */
    A_UINT32    table_id;
    /** size in bytes of wlan change entry */
    A_UINT32    wlan_change_entry_size;
    /** maximum number of entries in wlan change table */
    A_UINT32    max_wlan_change_entries;
    /** number of RSSI samples used for averaging RSSI */
    A_UINT32    max_rssi_averaging_samples;
    /** number of BSSID/RSSI entries (BSSID pointer, RSSI, timestamp) that device can hold */
    A_UINT32    max_rssi_history_entries;
} wmi_extscan_wlan_change_monitor_capabilities;

typedef struct {
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_ARRAY_STRUC */
    /** table ID - to allow support for multiple simultaneous tables */
    A_UINT32    table_id;
    /** size in bytes of hotlist entry */
    A_UINT32    wlan_hotlist_entry_size;
    /** maximum number of entries in wlan change table */
    A_UINT32    max_hotlist_entries;
} wmi_extscan_hotlist_monitor_capabilities;

typedef struct {
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_extscan_set_capabilities_cmd_fixed_param */
    /** Request ID - matches request ID used to start hot list monitoring */
    A_UINT32    request_id;
    /** Requestor ID - client requesting stop */
    A_UINT32    requestor_id;
    /** number of extscan caches */
    A_UINT32    num_extscan_cache_tables;
    /** number of wlan change lists */
    A_UINT32    num_wlan_change_monitor_tables;
    /** number of hotlists */
    A_UINT32    num_hotlist_monitor_tables;
    /** if one sided rtt data collection is supported */
    A_UINT32    rtt_one_sided_supported;
    /** if 11v data collection is supported */
    A_UINT32    rtt_11v_supported;
    /** if 11mc data collection is supported */
    A_UINT32    rtt_ftm_supported;
    /** number of extscan cache capabilities (one per table)  */
    A_UINT32    num_extscan_cache_capabilities;
    /** number of wlan change  capabilities (one per table)  */
    A_UINT32    num_extscan_wlan_change_capabilities;
    /** number of extscan hotlist capabilities (one per table)  */
    A_UINT32    num_extscan_hotlist_capabilities;
    /* Following this structure is the TLV:
     *     wmi_extscan_cache_capabilities               extscan_cache_capabilities; // number of capabilities given by num_extscan_caches
     *     wmi_extscan_wlan_change_monitor_capabilities wlan_change_capabilities;   // number of capabilities given by num_wlan_change_monitor_tables
     *     wmi_extscan_hotlist_monitor_capabilities     hotlist_capabilities;       // number of capabilities given by num_hotlist_monitor_tables
     */
} wmi_extscan_set_capabilities_cmd_fixed_param;

typedef struct {
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_extscan_get_capabilities_cmd_fixed_param */
    /** Request ID - matches request ID used to start hot list monitoring */
    A_UINT32    request_id;
    /** Requestor ID - client requesting capabilities */
    A_UINT32    requestor_id;
} wmi_extscan_get_capabilities_cmd_fixed_param;

typedef struct {
    A_UINT32     tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_extscan_start_stop_event_fixed_param */
    /** Request ID of the operation that was started/stopped */
    A_UINT32     request_id;
    /** Requestor ID of the operation that was started/stopped */
    A_UINT32     requestor_id;
    /** VDEV id(interface) of the operation that was started/stopped */
    A_UINT32     vdev_id;
    /** extscan WMI command */
    A_UINT32     command;
    /** operation mode: start/stop */
    A_UINT32     mode;      // wmi_extscan_operation_mode
    /**success/failure */
    A_UINT32     status;    // enum wmi_extscan_start_stop_status
    /** table ID - to allow support for multiple simultaneous requests */
    A_UINT32     table_id;
} wmi_extscan_start_stop_event_fixed_param;

typedef struct {
    A_UINT32     tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_extscan_operation_event_fixed_param */
    /** Request ID of the extscan operation that is currently running */
    A_UINT32     request_id;
    /** Requestor ID of the extscan operation that is currently running */
    A_UINT32     requestor_id;
    /** VDEV id(interface) of the extscan operation that is currently running */
    A_UINT32     vdev_id;
    /** scan event (wmi_scan_event_type) */
    A_UINT32     event;    // wmi_extscan_event_type
    /** table ID - to allow support for multiple simultaneous requests */
    A_UINT32     table_id;
    /**number of buckets */
    A_UINT32     num_buckets;
    /* Following this structure is the TLV:
     *     A_UINT32    bucket_id[];    // number of elements given by field num_buckets.
     */
} wmi_extscan_operation_event_fixed_param;

/* Types of extscan tables */
typedef enum {
    EXTSCAN_TABLE_NONE    = 0,
    EXTSCAN_TABLE_BSSID   = 1,
    EXTSCAN_TABLE_RSSI    = 2,
} wmi_extscan_table_type;

typedef struct {
    A_UINT32     tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_extscan_table_usage_event_fixed_param */
    /** Request ID of the extscan operation that is currently running */
    A_UINT32     request_id;
    /** Requestor ID of the extscan operation that is currently running */
    A_UINT32     requestor_id;
    /** VDEV id(interface) of the extscan operation that is currently running */
    A_UINT32     vdev_id;
    /** table ID - to allow support for multiple simultaneous tables */
    A_UINT32     table_id;
    /**see wmi_extscan_table_type for table reporting usage */
    A_UINT32     table_type;
    /**number of entries in use */
    A_UINT32     entries_in_use;
    /**maximum number of entries in table */
    A_UINT32     maximum_entries;
} wmi_extscan_table_usage_event_fixed_param;

typedef enum {
    WMI_SCAN_STATUS_INTERRUPTED = 1      /* Indicates scan got interrupted i.e. aborted or pre-empted for a long time (> 1sec)
                                            this can be used to discard scan results */
} wmi_scan_status_flags;

typedef struct {
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_ARRAY_STRUC */
    /** RSSI */
    A_UINT32    rssi;
    /** time stamp in milliseconds */
    A_UINT32    tstamp;
    /** Extscan cycle during which this entry was scanned */
    A_UINT32    scan_cycle_id;
    /** flag to indicate if the given result was obtained as part of interrupted (aborted/large time gap preempted) scan */
    A_UINT32    flags;
    /** Bitmask of buckets (i.e. sets of channels) scanned */
    A_UINT32    buckets_scanned;
} wmi_extscan_rssi_info;

typedef struct {
    A_UINT32        tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_ARRAY_STRUC */
    /**bssid */
    wmi_mac_addr    bssid;
    /**ssid */
    wmi_ssid        ssid;
    /**channel number */
    A_UINT32        channel;
    /* capabilities */
    A_UINT32        capabilities;
    /* beacon interval in TUs */
    A_UINT32        beacon_interval;
    /**time stamp in milliseconds - time last seen */
    A_UINT32        tstamp;
    /**flags - _tExtScanEntryFlags */
    A_UINT32        flags;
    /**RTT in ns */
    A_UINT32        rtt;
    /**rtt standard deviation */
    A_UINT32        rtt_sd;
    /* rssi information */
    A_UINT32        number_rssi_samples;
    /** IE length */
    A_UINT32        ie_length;             // length of IE data
} wmi_extscan_wlan_descriptor;

typedef struct {
    A_UINT32     tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_extscan_cached_results_event_fixed_param */
    /** Request ID of the WMI_EXTSCAN_GET_CACHED_RESULTS_CMDID */
    A_UINT32     request_id;
    /** Requestor ID of the WMI_EXTSCAN_GET_CACHED_RESULTS_CMDID */
    A_UINT32     requestor_id;
    /** VDEV id(interface) of the WMI_EXTSCAN_GET_CACHED_RESULTS_CMDID */
    A_UINT32     vdev_id;
    /** Request ID of the extscan operation that is currently running */
    A_UINT32     extscan_request_id;
    /** Requestor ID of the extscan operation that is currently running */
    A_UINT32     extscan_requestor_id;
    /** VDEV id(interface) of the extscan operation that is currently running */
    A_UINT32     extscan_vdev_id;
    /** table ID - to allow support for multiple simultaneous tables */
    A_UINT32     table_id;
    /**current time stamp in seconds. Used to provide a baseline for the relative timestamps returned for each block and entry */
    A_UINT32     current_tstamp;
    /**total number of bssids (in all pages) */
    A_UINT32     total_entries;
    /**index of the first bssid entry found in the TLV wmi_extscan_wlan_descriptor*/
    A_UINT32     first_entry_index;
    /**number of bssids in this page */
    A_UINT32     num_entries_in_page;
    /** number of buckets scanned**/
    A_UINT32     buckets_scanned;
    /* Followed by the variable length TLVs
     *     wmi_extscan_wlan_descriptor    bssid_list[]
     *     wmi_extscan_rssi_info          rssi_list[]
     *     A_UINT8                        ie_list[]
     */
} wmi_extscan_cached_results_event_fixed_param;

typedef enum {
    EXTSCAN_WLAN_CHANGE_FLAG_NONE         = 0x00,
    EXTSCAN_WLAN_CHANGE_FLAG_OUT_OF_RANGE = 0x01,
    EXTSCAN_WLAN_CHANGE_FLAG_AP_LOST      = 0x02,
} wmi_extscan_wlan_change_flags;

typedef struct {
    A_UINT32        tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_ARRAY_STRUC */
    /**bssid */
    wmi_mac_addr    bssid;
    /**time stamp in milliseconds */
    A_UINT32        tstamp;
    /**upper RSSI limit */
    A_UINT32        upper_rssi_limit;
    /**lower RSSI limit */
    A_UINT32        lower_rssi_limit;
    /** channel */
    A_UINT32        channel;    /* in MHz */
    /**current RSSI average */
    A_UINT32        rssi_average;
    /**flags - wmi_extscan_wlan_change_flags */
    A_UINT32        flags;
    /**legnth of RSSI history to follow (number of values) */
    A_UINT32        num_rssi_samples;
} wmi_extscan_wlan_change_result_bssid;

typedef struct {
    A_UINT32     tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_extscan_wlan_change_results_event_fixed_param */
    /** Request ID of the WMI_EXTSCAN_GET_WLAN_CHANGE_RESULTS_CMDID command that requested the results */
    A_UINT32     request_id;
    /** Requestor ID of the WMI_EXTSCAN_GET_WLAN_CHANGE_RESULTS_CMDID command that requested the results */
    A_UINT32     requestor_id;
    /** VDEV id(interface) of the WMI_EXTSCAN_GET_WLAN_CHANGE_RESULTS_CMDID command that requested the results */
    A_UINT32     vdev_id;
    /** Request ID of the WMI_EXTSCAN_CONFIGURE_WLAN_CHANGE_MONITOR_CMDID command that configured the table */
    A_UINT32     config_request_id;
    /** Requestor ID of the WMI_EXTSCAN_CONFIGURE_WLAN_CHANGE_MONITOR_CMDID command that configured the table */
    A_UINT32     config_requestor_id;
    /** VDEV id(interface) of the WMI_EXTSCAN_CONFIGURE_WLAN_CHANGE_MONITOR_CMDID command that configured the table */
    A_UINT32     config_vdev_id;
    /** table ID - to allow support for multiple simultaneous tables */
    A_UINT32     table_id;
    /**number of entries with RSSI out of range or BSSID not detected */
    A_UINT32     change_count;
    /**total number of bssid signal descriptors (in all pages) */
    A_UINT32     total_entries;
    /**index of the first bssid signal descriptor entry found in the TLV wmi_extscan_wlan_descriptor*/
    A_UINT32     first_entry_index;
    /**number of bssids signal descriptors in this page */
    A_UINT32     num_entries_in_page;
    /* Following this structure is the TLV:
     *     wmi_extscan_wlan_change_result_bssid bssid_signal_descriptor_list[];    // number of descriptors given by field num_entries_in_page.
     * Following this structure is the list of RSSI values (each is an A_UINT8):
     *     A_UINT8 rssi_list[];    // last N RSSI values.
     */
} wmi_extscan_wlan_change_results_event_fixed_param;

enum _tExtScanEntryFlags
{
    WMI_HOTLIST_FLAG_NONE           = 0x00,
    WMI_HOTLIST_FLAG_PRESENCE       = 0x01,
    WMI_HOTLIST_FLAG_DUPLICATE_SSID = 0x80,
};

typedef struct {
    A_UINT32     tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_extscan_hotlist_match_event_fixed_param */
    /** Request ID of the WMI_EXTSCAN_CONFIGURE_HOTLIST_MONITOR_CMDID that configured the table */
    A_UINT32     config_request_id;
    /** Requestor ID of the WMI_EXTSCAN_CONFIGURE_HOTLIST_MONITOR_CMDID that configured the table */
    A_UINT32     config_requestor_id;
    /** VDEV id(interface) of the WMI_EXTSCAN_CONFIGURE_HOTLIST_MONITOR_CMDID that configured the table */
    A_UINT32     config_vdev_id;
    /** table ID - to allow support for multiple simultaneous tables */
    A_UINT32     table_id;
    /**total number of bssids (in all pages) */
    A_UINT32     total_entries;
    /**index of the first bssid entry found in the TLV wmi_extscan_wlan_descriptor*/
    A_UINT32     first_entry_index;
    /**number of bssids in this page */
    A_UINT32     num_entries_in_page;
    /* Following this structure is the TLV:
     *     wmi_extscan_wlan_descriptor hotlist_match[];    // number of descriptors given by field num_entries_in_page.
     */
} wmi_extscan_hotlist_match_event_fixed_param;

typedef struct {
    A_UINT32     tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_extscan_hotlist_match_event_fixed_param */
    /** Request ID of the WMI_EXTSCAN_CONFIGURE_HOTLIST_SSID_MONITOR_CMDID that configured the table */
    A_UINT32     config_request_id;
    /** Requestor ID of the WMI_EXTSCAN_CONFIGURE_HOTLIST_SSID_MONITOR_CMDID that configured the table */
    A_UINT32     config_requestor_id;
    /** VDEV id(interface) of the WMI_EXTSCAN_CONFIGURE_HOTLIST_SSID_MONITOR_CMDID that configured the table */
    A_UINT32     config_vdev_id;
    /** table ID - to allow support for multiple simultaneous tables */
    A_UINT32     table_id;
    /**total number of ssids (in all pages) */
    A_UINT32     total_entries;
    /**index of the first ssid entry found in the TLV wmi_extscan_wlan_descriptor*/
    A_UINT32     first_entry_index;
    /**number of ssids in this page */
    A_UINT32     num_entries_in_page;
    /* Following this structure is the TLV:
     *     wmi_extscan_wlan_descriptor hotlist_match[];    // number of descriptors given by field num_entries_in_page.
     */
} wmi_extscan_hotlist_ssid_match_event_fixed_param;

typedef struct {
    A_UINT32     tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_extscan_capabilities_event_fixed_param */
    /** Request ID of the WMI_EXTSCAN_GET_CAPABILITIES_CMDID */
    A_UINT32     request_id;
    /** Requestor ID of the WMI_EXTSCAN_GET_CAPABILITIES_CMDID */
    A_UINT32     requestor_id;
    /** VDEV id(interface) of the WMI_EXTSCAN_GET_CAPABILITIES_CMDID */
    A_UINT32     vdev_id;
    /** number of extscan caches */
    A_UINT32     num_extscan_cache_tables;
    /** number of wlan change lists */
    A_UINT32     num_wlan_change_monitor_tables;
    /** number of hotlists */
    A_UINT32     num_hotlist_monitor_tables;
    /** if one sided rtt data collection is supported */
    A_UINT32     rtt_one_sided_supported;
    /** if 11v data collection is supported */
    A_UINT32     rtt_11v_supported;
    /** if 11mc data collection is supported */
    A_UINT32     rtt_ftm_supported;
    /** number of extscan cache capabilities (one per table)  */
    A_UINT32     num_extscan_cache_capabilities;
    /** number of wlan change  capabilities (one per table)  */
    A_UINT32     num_extscan_wlan_change_capabilities;
    /** number of extscan hotlist capabilities (one per table)  */
    A_UINT32     num_extscan_hotlist_capabilities;
    /* max number of roaming ssid whitelist firmware can support */
    A_UINT32 num_roam_ssid_whitelist;
    /* max number of blacklist bssid firmware can support */
    A_UINT32 num_roam_bssid_blacklist;
    /* max number of preferred list firmware can support */
    A_UINT32 num_roam_bssid_preferred_list;
    /* max number of hotlist ssids firmware can support */
    A_UINT32 num_extscan_hotlist_ssid;
    /* max number of epno networks firmware can support */
    A_UINT32 num_epno_networks;

    /* Following this structure are the TLVs describing the capabilities of of the various types of lists. The FW theoretically
     * supports multiple lists of each type.
     *
     *     wmi_extscan_cache_capabilities               extscan_cache_capabilities[] // capabilities of extscan cache (BSSID/RSSI lists)
     *     wmi_extscan_wlan_change_monitor_capabilities wlan_change_capabilities[]   // capabilities of wlan_change_monitor_tables
     *     wmi_extscan_hotlist_monitor_capabilities     hotlist_capabilities[]       // capabilities of hotlist_monitor_tables
     */
} wmi_extscan_capabilities_event_fixed_param;

/* WMI_D0_WOW_DISABLE_ACK_EVENTID  */
typedef struct{
    A_UINT32    tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_d0_wow_disable_ack_event_fixed_param  */
    A_UINT32    reserved0; /* for future need */
} wmi_d0_wow_disable_ack_event_fixed_param;

/** WMI_PDEV_RESUME_EVENTID : generated in response to WMI_PDEV_RESUME_CMDID */
typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_resume_event_fixed_param  */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
} wmi_pdev_resume_event_fixed_param;


/** value representing all modules */
#define WMI_DEBUG_LOG_MODULE_ALL 0xffff

/* param definitions */

/**
  * Log level for a given module. Value contains both module id and log level.
  * here is the bitmap definition for value.
  * module Id   : 16
  *     Flags   :  reserved
  *     Level   :  8
  * if odule Id  is WMI_DEBUG_LOG_MODULE_ALL then  log level is  applied to all modules (global).
  * WMI_DEBUG_LOG_MIDULE_ALL will overwrites per module level setting.
  */
#define WMI_DEBUG_LOG_PARAM_LOG_LEVEL      0x1

#define WMI_DBGLOG_SET_LOG_LEVEL(val,lvl) do { \
        (val) |=  (lvl & 0xff);                \
     } while(0)

#define WMI_DBGLOG_GET_LOG_LEVEL(val) ((val) & 0xff)

#define WMI_DBGLOG_SET_MODULE_ID(val,mid) do { \
        (val) |=  ((mid & 0xffff) << 16);        \
     } while(0)

#define WMI_DBGLOG_GET_MODULE_ID(val) (( (val) >> 16) & 0xffff)

/**
  * Enable the debug log for a given vdev. Value is vdev id
  */
#define WMI_DEBUG_LOG_PARAM_VDEV_ENABLE    0x2


/**
  * Disable the debug log for a given vdev. Value is vdev id
  * All the log level  for a given VDEV is disabled except the ERROR log messages
  */

#define WMI_DEBUG_LOG_PARAM_VDEV_DISABLE   0x3

/**
  * set vdev enable bitmap. value is the vden enable bitmap
  */
#define WMI_DEBUG_LOG_PARAM_VDEV_ENABLE_BITMAP    0x4

/**
  * set a given log level to all the modules specified in the module bitmap.
  * and set the log levle for all other modules to DBGLOG_ERR.
  *  value: log levelt to be set.
  *  module_id_bitmap : identifies the modules for which the log level should be set and
  *                      modules for which the log level should be reset to DBGLOG_ERR.
  */
#define WMI_DEBUG_LOG_PARAM_MOD_ENABLE_BITMAP    0x5

#define NUM_MODULES_PER_ENTRY ((sizeof(A_UINT32)) << 3)

#define WMI_MODULE_ENABLE(pmid_bitmap,mod_id) \
    ( (pmid_bitmap)[(mod_id)/NUM_MODULES_PER_ENTRY] |= \
         (1 << ((mod_id)%NUM_MODULES_PER_ENTRY)) )

#define WMI_MODULE_DISABLE(pmid_bitmap,mod_id)     \
    ( (pmid_bitmap)[(mod_id)/NUM_MODULES_PER_ENTRY] &=  \
      ( ~(1 << ((mod_id)%NUM_MODULES_PER_ENTRY)) ) )

#define WMI_MODULE_IS_ENABLED(pmid_bitmap,mod_id) \
    ( ((pmid_bitmap)[(mod_id)/NUM_MODULES_PER_ENTRY ] &  \
       (1 << ((mod_id)%NUM_MODULES_PER_ENTRY)) ) != 0)

#define MAX_MODULE_ID_BITMAP_WORDS 16 /* 16*32=512 module ids. should be more than sufficient */
typedef struct {
        A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_debug_log_config_cmd_fixed_param */
        A_UINT32 dbg_log_param; /** param types are defined above */
        A_UINT32 value;
        /* The below array will follow this tlv ->fixed length module_id_bitmap[]
        A_UINT32 module_id_bitmap[MAX_MODULE_ID_BITMAP_WORDS];
     */
} wmi_debug_log_config_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_get_temperature_cmd_fixed_param  */
    A_UINT32 param;     /* Reserved for future use */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
} wmi_pdev_get_temperature_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_temperature_event_fixed_param */
    A_INT32  value;     /* temprature value in Celcius degree */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
} wmi_pdev_temperature_event_fixed_param;

typedef enum {
    ANTDIV_HW_CFG_STATUS,
    ANTDIV_SW_CFG_STATUS,
    ANTDIV_MAX_STATUS_TYPE_NUM
} ANTDIV_STATUS_TYPE;

typedef struct {
    /**
     * TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_pdev_get_antdiv_status_cmd_fixed_param
     */
    A_UINT32 tlv_header;
    /* Status event ID - see ANTDIV_STATUS_TYPE */
    A_UINT32 status_event_id;
    /**
     * pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
} wmi_pdev_get_antdiv_status_cmd_fixed_param;

typedef struct {
    /**
     * TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_pdev_antdiv_status_event_fixed_param
     */
    A_UINT32 tlv_header;
    /* ANT DIV feature enabled or not */
    A_UINT32 support;
    A_UINT32 chain_num; /* how many chain supported */
    /* how many ANT supported, 32 max */
    A_UINT32 ant_num;
    /**
     * Each entry is for a tx/rx chain, and contains a bitmap identifying
     * the antennas attached to that tx/rx chain.
     */
    A_UINT32 selectable_ant_mask[8];
    /**
     * pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
} wmi_pdev_antdiv_status_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_set_dhcp_server_offload_cmd_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 enable;
    A_UINT32 srv_ipv4; /* server IP */
    A_UINT32 start_lsb; /* starting address assigned to client */
    A_UINT32 num_client; /* number of clients we support */
} wmi_set_dhcp_server_offload_cmd_fixed_param;

typedef enum {
    AP_RX_DATA_OFFLOAD             = 0x00,
    STA_RX_DATA_OFFLOAD            = 0x01,
} wmi_ipa_offload_types;

/**
 * This command is sent from WLAN host driver to firmware for
 * enabling/disabling IPA data-path offload features.
 *
 *
 * Enabling data path offload to IPA(based on host INI configuration), example:
 *    when STA interface comes up,
 *    host->target: WMI_IPA_OFFLOAD_ENABLE_DISABLE_CMD,
 *                  (enable = 1, vdev_id = STA vdev id, offload_type = STA_RX_DATA_OFFLOAD)
 *
 * Disabling data path offload to IPA, example:
 *    host->target: WMI_IPA_OFFLOAD_ENABLE_DISABLE_CMD,
 *                  (enable = 0, vdev_id = STA vdev id, offload_type = STA_RX_DATA_OFFLOAD)
 *
 *
 * This command is applicable only on the PCIE LL systems
 *
 */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ipa_offload_enable_disable_cmd_fixed_param */
    A_UINT32 offload_type; /* wmi_ipa_offload_types enum values */
    A_UINT32 vdev_id;
    A_UINT32 enable; /* 1 == enable, 0 == disable */
} wmi_ipa_offload_enable_disable_cmd_fixed_param;

typedef enum {
    WMI_LED_FLASHING_PATTERN_NOT_CONNECTED    = 0,
    WMI_LED_FLASHING_PATTERN_CONNECTED   = 1,
    WMI_LED_FLASHING_PATTERN_RESERVED    = 2,
} wmi_set_led_flashing_type;

/**
The state of the LED GPIO control is determined by two 32 bit values(X_0 and X_1) to produce a 64 bit value.
Each 32 bit value consists of 4 bytes, where each byte defines the number of 50ms intervals that the GPIO will
remain at a predetermined state. The 64 bit value provides 8 unique GPIO timing intervals. The pattern starts
with the MSB of X_0 and continues to the LSB of X_1. After executing the timer interval of the LSB of X_1, the
pattern returns to the MSB of X_0 and repeats. The GPIO state for each timing interval  alternates from Low to
High and the first interval of the pattern represents the time when the GPIO is Low. When a timing interval of
Zero is reached, it is skipped and moves on to the next interval.
*/
typedef struct{
    A_UINT32    tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_set_led_flashing_cmd_fixed_param  */
    A_UINT32    pattern_id; /* pattern identifier */
    A_UINT32    led_x0; /* led flashing parameter0 */
    A_UINT32    led_x1; /* led flashing parameter1 */
    A_UINT32    gpio_num; /* GPIO number */
} wmi_set_led_flashing_cmd_fixed_param;

/**
 * The purpose of the multicast Domain Name System (mDNS) is to resolve host names to IP addresses
 * within small networks that do not include a local name server.
 * It utilizes essentially the same programming interfaces, packet formats and operating semantics
 * as the unicast DNS, and the advantage is zero configuration service while no need for central or
 * global server.
 * Based on mDNS, the DNS-SD (Service Discovery) allows clients to discover a named list of services
 * by type in a specified domain using standard DNS queries.
 * Here, we provide the ability to advertise the available services by responding to mDNS queries.
 */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_mdns_offload_cmd_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 enable;
} wmi_mdns_offload_cmd_fixed_param;

#define WMI_MAX_MDNS_FQDN_LEN         64
#define WMI_MAX_MDNS_RESP_LEN         512
#define WMI_MDNS_FQDN_TYPE_GENERAL    0
#define WMI_MDNS_FQDN_TYPE_UNIQUE     1

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_mdns_set_fqdn_cmd_fixed_param */
    A_UINT32 vdev_id;
    /** type of fqdn, general or unique */
    A_UINT32 type;
    /** length of fqdn */
    A_UINT32 fqdn_len;
    /* Following this structure is the TLV byte stream of fqdn data of length fqdn_len
     * A_UINT8  fqdn_data[]; // fully-qualified domain name to check if match with the received queries
     */
} wmi_mdns_set_fqdn_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_mdns_set_resp_cmd_fixed_param */
    A_UINT32 vdev_id;
    /** Answer Resource Record count */
    A_UINT32 AR_count;
    /** length of response */
    A_UINT32 resp_len;
    /* Following this structure is the TLV byte stream of resp data of length resp_len
     * A_UINT8  resp_data[]; // responses consisits of Resource Records
     */
} wmi_mdns_set_resp_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_mdns_get_stats_cmd_fixed_param */
    A_UINT32 vdev_id;
} wmi_mdns_get_stats_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_mdns_stats_event_fixed_param */
    A_UINT32 vdev_id;
    /** curTimestamp in milliseconds */
    A_UINT32 curTimestamp;
    /** last received Query in milliseconds */
    A_UINT32 lastQueryTimestamp;
    /** last sent Response in milliseconds */
    A_UINT32 lastResponseTimestamp;
    /** stats of received queries */
    A_UINT32 totalQueries;
    /** stats of macth queries */
    A_UINT32 totalMatches;
    /** stats of responses */
    A_UINT32 totalResponses;
    /** indicate the current status of mDNS offload */
    A_UINT32 status;
} wmi_mdns_stats_event_fixed_param;

/**
 * The purpose of the SoftAP authenticator offload is to offload the association and 4-way handshake process
 * down to the firmware. When this feature is enabled, firmware can process the association/disassociation
 * request and create/remove connection even host is suspended.
 * 3 major components are offloaded:
 *     1. ap-mlme. Firmware will process auth/deauth, association/disassociation request and send out response.
 *     2. 4-way handshake. Firmware will send out m1/m3 and receive m2/m4.
 *     3. key installation. Firmware will generate PMK from the psk info which is sent from the host and install PMK/GTK.
 * Current implementation only supports WPA2 CCMP.
 */

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_sap_ofl_enable_cmd_fixed_param */
    /** VDEV id(interface) of the WMI_SAP_OFL_ENABLE_CMDID */
    A_UINT32 vdev_id;
    /** enable/disable sap auth offload */
    A_UINT32 enable;
    /** sap ssid */
    wmi_ssid ap_ssid;
    /** authentication mode (defined above) */
    A_UINT32 rsn_authmode;
    /** unicast cipher set */
    A_UINT32 rsn_ucastcipherset;
    /** mcast/group cipher set */
    A_UINT32 rsn_mcastcipherset;
    /** mcast/group management frames cipher set */
    A_UINT32 rsn_mcastmgmtcipherset;
    /** sap channel */
    A_UINT32 channel;
    /** length of psk */
    A_UINT32 psk_len;
    /* Following this structure is the TLV byte stream of wpa passphrase data of length psk_len
     * A_UINT8  psk[];
     */
} wmi_sap_ofl_enable_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_sap_ofl_add_sta_event_fixed_param */
    /** VDEV id(interface) of the WMI_SAP_OFL_ADD_STA_EVENTID */
    A_UINT32 vdev_id;
    /** aid (association id) of this station */
    A_UINT32 assoc_id;
    /** peer station's mac addr */
    wmi_mac_addr peer_macaddr;
    /** length of association request frame */
    A_UINT32 data_len;
    /* Following this structure is the TLV byte stream of a whole association request frame of length data_len
     * A_UINT8 bufp[];
     */
} wmi_sap_ofl_add_sta_event_fixed_param;

typedef enum {
    SAP_OFL_DEL_STA_FLAG_NONE       = 0x00,
    SAP_OFL_DEL_STA_FLAG_RECONNECT  = 0x01,
} wmi_sap_ofl_del_sta_flags;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_sap_ofl_del_sta_event_fixed_param */
    /** VDEV id(interface) of the WMI_SAP_OFL_DEL_STA_EVENTID */
    A_UINT32 vdev_id;
    /** aid (association id) of this station */
    A_UINT32 assoc_id;
    /** peer station's mac addr */
    wmi_mac_addr peer_macaddr;
    /** disassociation reason */
    A_UINT32 reason;
    /** flags - wmi_sap_ofl_del_sta_flags */
    A_UINT32 flags;
} wmi_sap_ofl_del_sta_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_sap_set_blacklist_param_cmd_fixed_param */
    A_UINT32 vdev_id;
    /* Number of client failure connection attempt */
    A_UINT32 num_retry;
    /* Time in milliseconds to record the client's failure connection attempts */
    A_UINT32 retry_allow_time_ms;
    /* Time in milliseconds to drop the connection request if client is blacklisted */
    A_UINT32 blackout_time_ms;
} wmi_sap_set_blacklist_param_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_apfind_cmd_param */
    A_UINT32 data_len; /** length in byte of data[]. */
    /** This structure is used to send REQ binary blobs
     * from application/service to firmware where Host drv is pass through .
     * Following this structure is the TLV:
     *     A_UINT8 data[];    // length in byte given by field data_len.
     */
} wmi_apfind_cmd_param;

typedef enum apfind_event_type_e {
    APFIND_MATCH_EVENT = 0,
    APFIND_WAKEUP_EVENT,
} APFIND_EVENT_TYPE;

typedef struct {
    A_UINT32 tlv_header; /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_apfind_event_hdr */
    A_UINT32 event_type; /** APFIND_EVENT_TYPE */
    A_UINT32 data_len; /** length in byte of data[]. */
    /** This structure is used to send event binary blobs
     * from firmware to application/service and Host drv.
     * Following this structure is the TLV:
     *     A_UINT8 data[];    // length in byte given by field data_len.
     */
} wmi_apfind_event_hdr;

/* SAP obss detection offload types */
typedef enum {
    WMI_SAP_OBSS_DETECTION_MODE_DISABLED = 0, /* fw to disable the detection */
    WMI_SAP_OBSS_DETECTION_MODE_PRESENT_NOTIFY = 1, /* if the matching beacon is present, notify host immediately */
    WMI_SAP_OBSS_DETECTION_MODE_ABSENT_TIMEOUT_NOTIFY = 2,/* if the matching beacon is absent for the timeout period, notify host */
} WMI_SAP_OBSS_DETECTION_MODE;

typedef struct wmi_sap_obss_detection_cfg_cmd_s {
    A_UINT32 tlv_header; /* tag = WMITLV_TAG_STRUC_wmi_sap_obss_detection_cfg_cmd_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 detect_period_ms;

    /* detect whether there is 11b ap/ibss */
    A_UINT32 b_ap_detect_mode;  /* refer WMI_SAP_OBSS_DETECTION_MODE */

    /* detect whether there is 11b sta connected with other APs */
    A_UINT32 b_sta_detect_mode;

    /* detect whether there is 11g AP */
    A_UINT32 g_ap_detect_mode;

    /* detect whether there is legacy 11a traffic */
    A_UINT32 a_detect_mode;

    /* detect whether there is ap which is ht legacy mode  */
    A_UINT32 ht_legacy_detect_mode;

    /* detect whether there is ap which is ht mixed mode : has 11b/11g sta */
    A_UINT32 ht_mixed_detect_mode;

    /* detect whether there is ap which has 20M only station */
    A_UINT32 ht_20mhz_detect_mode;

} wmi_sap_obss_detection_cfg_cmd_fixed_param;

typedef enum {
    WMI_SAP_OBSS_DETECTION_EVENT_REASON_NOT_SUPPORT = 0,
    WMI_SAP_OBSS_DETECTION_EVENT_REASON_PRESENT_NOTIFY,
    WMI_SAP_OBSS_DETECTION_EVENT_REASON_ABSENT_TIMEOUT,
} WMI_SAP_OBSS_DETECTION_EVENT_REASON;

/* WMI_SAP_OBSS_DETECTION_MATCH_MASK is one or more of the following shift bits */
#define WMI_SAP_OBSS_DETECTION_MATCH_BIT_11B_AP_S       0
#define WMI_SAP_OBSS_DETECTION_MATCH_BIT_11B_STA_S      1
#define WMI_SAP_OBSS_DETECTION_MATCH_BIT_11G_AP_S       2
#define WMI_SAP_OBSS_DETECTION_MATCH_BIT_11A_S          3
#define WMI_SAP_OBSS_DETECTION_MATCH_BIT_HT_LEGACY_S    4
#define WMI_SAP_OBSS_DETECTION_MATCH_BIT_HT_MIXED_S     5
#define WMI_SAP_OBSS_DETECTION_MATCH_BIT_HT_20MHZ_S     6

typedef struct wmi_sap_obss_detection_info_evt_s {
    A_UINT32 tlv_header; /* tag = WMITLV_TAG_STRUC_wmi_sap_obss_detection_info_evt_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 reason;   /* refer WMI_SAP_OBSS_DETECTION_EVENT_REASON */
    A_UINT32 matched_detection_masks;  /* bit(s) from WMI_SAP_OBSS_DETECTION_MATCH_MASK */
    wmi_mac_addr matched_bssid_addr;  /* valid when reason is WMI_SAP_OBSS_DETECTION_EVENT_REASON_PRESENT_NOTIFY */
} wmi_sap_obss_detection_info_evt_fixed_param;

/** WMI command to enable STA FW handle bss color change notification from AP */
typedef struct  {
    A_UINT32 tlv_header; /* tag equals WMITLV_TAG_STRUC_wmi_bss_color_change_enable_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 enable;
} wmi_bss_color_change_enable_fixed_param;

typedef enum  {
    WMI_BSS_COLOR_COLLISION_DISABLE = 0,
    WMI_BSS_COLOR_COLLISION_DETECTION,
    WMI_BSS_COLOR_FREE_SLOT_TIMER_EXPIRY,
    WMI_BSS_COLOR_FREE_SLOT_AVAILABLE,
} WMI_BSS_COLOR_COLLISION_EVT_TYPE;

/** Command to enable OBSS Color collision detection for both STA and AP mode */
typedef struct  {
    A_UINT32 tlv_header;                /* tag equals WMITLV_TAG_STRUC_wmi_obss_color_collision_det_config_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 flags;                     /* proposed for future use cases */
    A_UINT32 evt_type;                  /* WMI_BSS_COLOR_COLLISION_EVT_TYPE */
    A_UINT32 current_bss_color;
    A_UINT32 detection_period_ms;       /* scan interval for both AP and STA mode */
    A_UINT32 scan_period_ms;            /* scan period for passive scan to detect collision */
    A_UINT32 free_slot_expiry_time_ms;  /* FW to notify host at timer expiry after which Host disables bss color */
} wmi_obss_color_collision_det_config_fixed_param;

/** WMI event to notify host on OBSS Color collision detection, free slot available for AP mode */
typedef struct  {
    A_UINT32 tlv_header;                    /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_obss_color_collision_evt_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 evt_type;                      /* WMI_BSS_COLOR_COLLISION_EVT_TYPE */
    A_UINT32 bss_color_bitmap_bit0to31;     /* Bit set indicating BSS color present */
    A_UINT32 bss_color_bitmap_bit32to63;    /* Bit set indicating BSS color present */
} wmi_obss_color_collision_evt_fixed_param;

/**
 * OCB DCC types and structures.
 */

/**
 * DCC types as described in ETSI TS 102 687
 * Type                   Format            stepSize    referenceValue  numBits
 * -------------------------------------------------------------------------
 * ndlType_acPrio         INTEGER (0...7)   1           number          3
 * ndlType_controlLoop    INTEGER (0...7)   1           0               3
 * ndlType_arrivalRate    INTEGER (0..8191) 0.01 /s     0               13
 * ndlType_channelLoad    INTEGER (0..1000) 0.1 %       0 %             10
 * ndlType_channelUse     INTEGER (0..8000) 0.0125 %    0 %             13
 * ndlType_datarate       INTEGER (0..7)                Table 8         3
 * ndlType_distance       INTEGER (0..4095) 1 m         0               12
 * ndlType_numberElements INTEGER (0..63)               number          6
 * ndlType_packetDuration INTEGER (0..2047) TSYM        0               11
 * ndlType_packetInterval INTEGER (0..1023) 10 ms       0               10
 * ndlType_pathloss       INTEGER (0..31)   0.1         1.0             5
 * ndlType_rxPower        INTEGER (0..127)  -0.5 dB     -40 dBm         7
 * ndlType_snr            INTEGER (0..127)  0.5 dB      -10 dB          7
 * ndlType_timing         INTEGER (0..4095) 10 ms       0               12
 * ndlType_txPower        INTEGER (0..127)  0.5 dB      -20 dBm         7
 * ndlType_ratio          INTEGER (0..100)  1 %         0 %             7
 * ndlType_exponent       INTEGER (0..100)  0.1         0               7
 * ndlType_queueStatus    Enumeration                   Table A.2       1
 * ndlType_dccMechanism   Bitset                        Table A.2       6
 *
 * NOTE: All of following size macros (SIZE_NDLTYPE_ACPRIO through SIZE_BYTE)
 * cannot be changed without breaking WMI compatibility.
 *
 * NOTE: For each of the types, one additional bit is allocated. This
 *  leftmost bit is used to indicate that the value is invalid. */
#define SIZE_NDLTYPE_ACPRIO         (1 + 3 )
#define SIZE_NDLTYPE_CONTROLLOOP    (1 + 3 )
#define SIZE_NDLTYPE_ARRIVALRATE    (1 + 13)
#define SIZE_NDLTYPE_CHANNELLOAD    (1 + 10)
#define SIZE_NDLTYPE_CHANNELUSE     (1 + 13)
#define SIZE_NDLTYPE_DATARATE       (1 + 3 )
#define SIZE_NDLTYPE_DISTANCE       (1 + 12)
#define SIZE_NDLTYPE_NUMBERELEMENTS (1 + 6 )
#define SIZE_NDLTYPE_PACKETDURATION (1 + 11)
#define SIZE_NDLTYPE_PACKETINTERVAL (1 + 10)
#define SIZE_NDLTYPE_PATHLOSS       (1 + 5 )
#define SIZE_NDLTYPE_RXPOWER        (1 + 7 )
#define SIZE_NDLTYPE_SNR            (1 + 7 )
#define SIZE_NDLTYPE_TIMING         (1 + 12)
#define SIZE_NDLTYPE_TXPOWER        (1 + 7 )
#define SIZE_NDLTYPE_RATIO          (1 + 7 )
#define SIZE_NDLTYPE_EXPONENT       (1 + 7 )
#define SIZE_NDLTYPE_QUEUESTATUS    (1 + 1 )
#define SIZE_NDLTYPE_DCCMECHANISM   (1 + 6 )
#define SIZE_BYTE                   (8)

#define INVALID_ACPRIO          ((1 << SIZE_NDLTYPE_ACPRIO) - 1)
#define INVALID_CONTROLLOOP     ((1 << SIZE_NDLTYPE_CONTROLLOOP) - 1)
#define INVALID_ARRIVALRATE     ((1 << SIZE_NDLTYPE_ARRIVALRATE) - 1)
#define INVALID_CHANNELLOAD     ((1 << SIZE_NDLTYPE_CHANNELLOAD) - 1)
#define INVALID_CHANNELUSE      ((1 << SIZE_NDLTYPE_CHANNELUSE) - 1)
#define INVALID_DATARATE        ((1 << SIZE_NDLTYPE_DATARATE) - 1)
#define INVALID_DISTANCE        ((1 << SIZE_NDLTYPE_DISTANCE) - 1)
#define INVALID_NUMBERELEMENTS  ((1 << SIZE_NDLTYPE_NUMBERELEMENTS) - 1)
#define INVALID_PACKETDURATION  ((1 << SIZE_NDLTYPE_PACKETDURATION) - 1)
#define INVALID_PACKETINTERVAL  ((1 << SIZE_NDLTYPE_PACKETINTERVAL) - 1)
#define INVALID_PATHLOSS        ((1 << SIZE_NDLTYPE_PATHLOSS) - 1)
#define INVALID_RXPOWER         ((1 << SIZE_NDLTYPE_RXPOWER) - 1)
#define INVALID_SNR             ((1 << SIZE_NDLTYPE_SNR) - 1)
#define INVALID_TIMING          ((1 << SIZE_NDLTYPE_TIMING) - 1)
#define INVALID_TXPOWER         ((1 << SIZE_NDLTYPE_TXPOWER) - 1)
#define INVALID_RATIO           ((1 << SIZE_NDLTYPE_RATIO) - 1)
#define INVALID_EXPONENT        ((1 << SIZE_NDLTYPE_EXPONENT) - 1)
#define INVALID_QUEUESTATS      ((1 << SIZE_NDLTYPE_QUEUESTATUS) - 1)
#define INVALID_DCCMECHANISM    ((1 << SIZE_NDLTYPE_DCCMECHANISM) - 1)

/** The MCS_COUNT macro cannot be modified without breaking
 *  WMI compatibility. */
#define MCS_COUNT               (8)

/** Flags for ndlType_dccMechanism. */
typedef enum {
    DCC_MECHANISM_TPC = 1,
    DCC_MECHANISM_TRC = 2,
    DCC_MECHANISM_TDC = 4,
    DCC_MECHANISM_DSC = 8,
    DCC_MECHANISM_TAC = 16,
    DCC_MECHANISM_RESERVED = 32,
    DCC_MECHANISM_ALL = 0x3f,
} wmi_dcc_ndl_type_dcc_mechanism;

/** Values for ndlType_queueStatus. */
typedef enum {
    DCC_QUEUE_CLOSED = 0,
    DCC_QUEUE_OPEN = 1,
} wmi_dcc_ndl_type_queue_status;

/** For ndlType_acPrio, use the values in wmi_traffic_ac. */

/** Values for ndlType_datarate */
typedef enum {
    DCC_DATARATE_3_MBPS = 0,
    DCC_DATARATE_4_5_MBPS = 1,
    DCC_DATARATE_6_MBPS = 2,
    DCC_DATARATE_9_MBPS = 3,
    DCC_DATARATE_12_MBPS = 4,
    DCC_DATARATE_18_MBPS = 5,
    DCC_DATARATE_24_MBPS = 6,
    DCC_DATARATE_27_MBPS = 7,
} wmi_dcc_ndl_type_datarate;

/** Data structure for active state configuration. */
typedef struct {
    /** TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_dcc_ndl_active_state_config */
    A_UINT32 tlv_header;
    /**
     * NDL_asStateId, ndlType_numberElements, 1+6 bits.
     * NDL_asChanLoad, ndlType_channelLoad, 1+10 bits.
     */
    A_UINT32 state_info;
    /**
     * NDL_asDcc(AC_BK), ndlType_dccMechanism, 1+6 bits.
     * NDL_asDcc(AC_BE), ndlType_dccMechanism, 1+6 bits.
     * NDL_asDcc(AC_VI), ndlType_dccMechanism, 1+6 bits.
     * NDL_asDcc(AC_VO), ndlType_dccMechanism, 1+6 bits.
     */
    A_UINT32 as_dcc[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC, SIZE_NDLTYPE_DCCMECHANISM)];

    /**
     * NDL_asTxPower(AC_BK), ndlType_txPower, 1+7 bits.
     * NDL_asTxPower(AC_BE), ndlType_txPower, 1+7 bits.
     * NDL_asTxPower(AC_VI), ndlType_txPower, 1+7 bits.
     * NDL_asTxPower(AC_VO), ndlType_txPower, 1+7 bits.
     */
    A_UINT32 as_tx_power_ac[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC, SIZE_NDLTYPE_TXPOWER)];
    /**
     * NDL_asPacketInterval(AC_BK), ndlType_packetInterval, 1+10 bits.
     * NDL_asPacketInterval(AC_BE), ndlType_packetInterval, 1+10 bits.
     * NDL_asPacketInterval(AC_VI), ndlType_packetInterval, 1+10 bits.
     * NDL_asPacketInterval(AC_VO), ndlType_packetInterval, 1+10 bits.
     */
    A_UINT32 as_packet_interval_ac[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC, SIZE_NDLTYPE_PACKETINTERVAL)];
    /**
     * NDL_asDatarate(AC_BK), ndlType_datarate, 1+3 bits.
     * NDL_asDatarate(AC_BE), ndlType_datarate, 1+3 bits.
     * NDL_asDatarate(AC_VI), ndlType_datarate, 1+3 bits.
     * NDL_asDatarate(AC_VO), ndlType_datarate, 1+3 bits.
     */
    A_UINT32 as_datarate_ac[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC, SIZE_NDLTYPE_DATARATE)];
    /**
     * NDL_asCarrierSense(AC_BK), ndlType_rxPower, 1+7 bits.
     * NDL_asCarrierSense(AC_BE), ndlType_rxPower, 1+7 bits.
     * NDL_asCarrierSense(AC_VI), ndlType_rxPower, 1+7 bits.
     * NDL_asCarrierSense(AC_VO), ndlType_rxPower, 1+7 bits.
     */
    A_UINT32 as_carrier_sense_ac[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC, SIZE_NDLTYPE_RXPOWER)];
} wmi_dcc_ndl_active_state_config;

#define WMI_NDL_AS_STATE_ID_GET(ptr)                    WMI_GET_BITS((ptr)->state_info, 0, 7)
#define WMI_NDL_AS_STATE_ID_SET(ptr,val)                WMI_SET_BITS((ptr)->state_info, 0, 7, val)
#define WMI_NDL_AS_CHAN_LOAD_GET(ptr)                   WMI_GET_BITS((ptr)->state_info, 7, 11)
#define WMI_NDL_AS_CHAN_LOAD_SET(ptr,val)               WMI_SET_BITS((ptr)->state_info, 7, 11, val)
#define WMI_NDL_AS_DCC_GET(ptr,acprio)                  wmi_packed_arr_get_bits((ptr)->as_dcc, acprio, SIZE_NDLTYPE_DCCMECHANISM)
#define WMI_NDL_AS_DCC_SET(ptr,acprio,val)              wmi_packed_arr_set_bits((ptr)->as_dcc, acprio, SIZE_NDLTYPE_DCCMECHANISM, val)
#define WMI_NDL_AS_TX_POWER_GET(ptr,acprio)             wmi_packed_arr_get_bits((ptr)->as_tx_power_ac, acprio, SIZE_NDLTYPE_TXPOWER)
#define WMI_NDL_AS_TX_POWER_SET(ptr,acprio,val)         wmi_packed_arr_set_bits((ptr)->as_tx_power_ac, acprio, SIZE_NDLTYPE_TXPOWER, val)
#define WMI_NDL_AS_PACKET_INTERVAL_GET(ptr,acprio)      wmi_packed_arr_get_bits((ptr)->as_packet_interval_ac, acprio, SIZE_NDLTYPE_PACKETINTERVAL)
#define WMI_NDL_AS_PACKET_INTERVAL_SET(ptr,acprio,val)  wmi_packed_arr_set_bits((ptr)->as_packet_interval_ac, acprio, SIZE_NDLTYPE_PACKETINTERVAL, val)
#define WMI_NDL_AS_DATARATE_GET(ptr,acprio)             wmi_packed_arr_get_bits((ptr)->as_datarate_ac, acprio, SIZE_NDLTYPE_DATARATE)
#define WMI_NDL_AS_DATARATE_SET(ptr,acprio,val)         wmi_packed_arr_set_bits((ptr)->as_datarate_ac, acprio, SIZE_NDLTYPE_DATARATE, val)
#define WMI_NDL_AS_CARRIER_SENSE_GET(ptr,acprio)        wmi_packed_arr_get_bits((ptr)->as_carrier_sense_ac, acprio, SIZE_NDLTYPE_RXPOWER)
#define WMI_NDL_AS_CARRIER_SENSE_SET(ptr,acprio,val)    wmi_packed_arr_set_bits((ptr)->as_carrier_sense_ac, acprio, SIZE_NDLTYPE_RXPOWER, val)

/** Data structure for EDCA/QOS parameters. */
typedef struct
{
    /** TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_qos_parameter */
    A_UINT32 tlv_header;
    /** Arbitration Inter-Frame Spacing. Range: 2-15 */
    A_UINT32 aifsn;
    /** Contention Window minimum. Range: 1 - 10 */
    A_UINT32 cwmin;
    /** Contention Window maximum. Range: 1 - 10 */
    A_UINT32 cwmax;
} wmi_qos_parameter;

/** Data structure for information specific to a channel. */
typedef struct {
    /** TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_ocb_channel */
    A_UINT32 tlv_header;
    A_UINT32 bandwidth; /* MHz units */
    wmi_mac_addr mac_address;
} wmi_ocb_channel;

/** Data structure for an element of the schedule array. */
typedef struct {
    /** TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_ocb_schedule_element */
    A_UINT32 tlv_header;
    A_UINT32 channel_freq; /* MHz units */
    A_UINT32 total_duration; /* ms units */
    A_UINT32 guard_interval; /* ms units */
} wmi_ocb_schedule_element;

/** Data structure for OCB configuration. */
typedef struct {
    /** TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_ocb_set_config_cmd_fixed_param */
    A_UINT32 tlv_header;
    /** VDEV id(interface) that is being configured */
    A_UINT32 vdev_id;
    A_UINT32 channel_count;
    A_UINT32 schedule_size;
    A_UINT32 flags;
    /**
     * Max duration of continuing multichannel operation without
     * receiving a TA frame (units = seconds)
     */
    A_UINT32 ta_max_duration;

    /** This is followed by a TLV array of wmi_channel. */
    /** This is followed by a TLV array of wmi_ocb_channel. */
    /** This is followed by a TLV array of wmi_qos_parameter. */
    /** This is followed by a TLV array of wmi_dcc_ndl_chan. */
    /** This is followed by a TLV array of wmi_dcc_ndl_active_state_config. */
    /** This is followed by a TLV array of wmi_ocb_schedule_element. */
} wmi_ocb_set_config_cmd_fixed_param;

#define EXPIRY_TIME_IN_TSF_TIMESTAMP_OFFSET     0
#define EXPIRY_TIME_IN_TSF_TIMESTAMP_MASK       1

#define WMI_OCB_EXPIRY_TIME_IN_TSF(ptr) \
    (((ptr)->flags & EXPIRY_TIME_IN_TSF_TIMESTAMP_MASK) >> EXPIRY_TIME_IN_TSF_TIMESTAMP_OFFSET)

/** Data structure for the response to the WMI_OCB_SET_CONFIG_CMDID command. */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_ocb_set_config_resp_event_fixed_param */
    A_UINT32 tlv_header;
    /* VDEV identifier */
    A_UINT32 vdev_id;
    A_UINT32 status;
} wmi_ocb_set_config_resp_event_fixed_param;

/* SIZE_UTC_TIME and SIZE_UTC_TIME_ERROR cannot be modified without breaking
   WMI compatibility. */
#define SIZE_UTC_TIME (10) // The size of the utc time in bytes.
#define SIZE_UTC_TIME_ERROR (5) // The size of the utc time error in bytes.

/** Data structure to set the UTC time. */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_ocb_set_utc_time_cmd_fixed_param */
    A_UINT32 tlv_header;
    /* VDEV identifier */
    A_UINT32 vdev_id;
    /** 10 bytes of the utc time. */
    A_UINT32 utc_time[WMI_PACKED_ARR_SIZE(SIZE_UTC_TIME,SIZE_BYTE)];
    /** 5 bytes of the time error. */
    A_UINT32 time_error[WMI_PACKED_ARR_SIZE(SIZE_UTC_TIME_ERROR,SIZE_BYTE)];
} wmi_ocb_set_utc_time_cmd_fixed_param;

#define WMI_UTC_TIME_GET(ptr,byte_index)        wmi_packed_arr_get_bits((ptr)->utc_time, byte_index, SIZE_BYTE)
#define WMI_UTC_TIME_SET(ptr,byte_index,val)    wmi_packed_arr_set_bits((ptr)->utc_time, byte_index, SIZE_BYTE, val)
#define WMI_TIME_ERROR_GET(ptr,byte_index)      wmi_packed_arr_get_bits((ptr)->time_error, byte_index, SIZE_BYTE)
#define WMI_TIME_ERROR_SET(ptr,byte_index,val)  wmi_packed_arr_set_bits((ptr)->time_error, byte_index, SIZE_BYTE, val)

/** Data structure start the timing advertisement. The template for the
 *  timing advertisement frame follows this structure in the WMI command.
 */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_ocb_start_timing_advert_cmd_fixed_param */
    A_UINT32 tlv_header;
    /* VDEV identifier */
    A_UINT32 vdev_id;
    /** Number of times the TA is sent every 5 seconds. */
    A_UINT32 repeat_rate;
    /** The frequency on which to transmit. */
    A_UINT32 channel_freq; /* MHz units */
    /** The offset into the template of the timestamp. */
    A_UINT32 timestamp_offset;
    /** The offset into the template of the time value. */
    A_UINT32 time_value_offset;
    /** The length of the timing advertisement template. The
     *  template is in the TLV data. */
    A_UINT32 timing_advert_template_length;

    /** This is followed by a binary array containing the TA template. */
} wmi_ocb_start_timing_advert_cmd_fixed_param;

/** Data structure to stop the timing advertisement. */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_ocb_stop_timing_advert_cmd_fixed_param */
    A_UINT32 tlv_header;
    /* VDEV identifier */
    A_UINT32 vdev_id;
    A_UINT32 channel_freq; /* MHz units */
} wmi_ocb_stop_timing_advert_cmd_fixed_param;

/** Data structure for the request for WMI_OCB_GET_TSF_TIMER_CMDID. */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_ocb_get_tsf_timer_cmd_fixed_param */
    A_UINT32 tlv_header;
    /* VDEV identifier */
    A_UINT32 vdev_id;
    A_UINT32 reserved;
} wmi_ocb_get_tsf_timer_cmd_fixed_param;

/** Data structure for the response to WMI_OCB_GET_TSF_TIMER_CMDID. */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_ocb_get_tsf_timer_resp_event_fixed_param */
    A_UINT32 tlv_header;
    /* VDEV identifier */
    A_UINT32 vdev_id;
    A_UINT32 tsf_timer_high;
    A_UINT32 tsf_timer_low;
} wmi_ocb_get_tsf_timer_resp_event_fixed_param;

/** Data structure for DCC stats configuration per channel. */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_dcc_ndl_stats_per_channel */
    A_UINT32 tlv_header;

    /** The channel for which this applies, 16 bits. */
    /** The dcc_stats_bitmap, 8 bits. */
    A_UINT32 chan_info;

    /** Demodulation model parameters. */
    /**
     * NDL_snrBackoff(MCS0), ndlType_snr, 1+7 bits.
     * NDL_snrBackoff(MCS1), ndlType_snr, 1+7 bits.
     * NDL_snrBackoff(MCS2), ndlType_snr, 1+7 bits.
     * NDL_snrBackoff(MCS3), ndlType_snr, 1+7 bits.
     * NDL_snrBackoff(MCS4), ndlType_snr, 1+7 bits.
     * NDL_snrBackoff(MCS5), ndlType_snr, 1+7 bits.
     * NDL_snrBackoff(MCS6), ndlType_snr, 1+7 bits.
     * NDL_snrBackoff(MCS7), ndlType_snr, 1+7 bits.
     */
    A_UINT32 snr_backoff_mcs[WMI_PACKED_ARR_SIZE(MCS_COUNT,SIZE_NDLTYPE_SNR)];

    /** Communication ranges. */
    /**
     * tx_power, ndlType_txPower, 1+7 bits.
     * datarate, ndlType_datarate, 1+3 bits.
     */
    A_UINT32 tx_power_datarate;
    /**
     * NDL_carrierSenseRange, ndlType_distance, 1+12 bits.
     * NDL_estCommRange, ndlType_distance, 1+12 bits.
     */
    A_UINT32 carrier_sense_est_comm_range;

    /** Channel load measures. */
    /**
     * dccSensitivity, ndlType_rxPower, 1+7 bits.
     * carrierSense, ndlType_rxPower, 1+7 bits.
     * NDL_channelLoad, ndlType_channelLoad, 1+10 bits.
     */
    A_UINT32 dcc_stats;
    /**
     * NDL_packetArrivalRate, ndlType_arrivalRate, 1+13 bits.
     * NDL_packetAvgDuration, ndlType_packetDuration, 1+11 bits.
     */
    A_UINT32 packet_stats;
    /**
     * NDL_channelBusyTime, ndlType_channelLoad, 1+10 bits.
     */
    A_UINT32 channel_busy_time;

    /** Transmit packet statistics. */
    /**
     * NDL_txPacketArrivalRate(AC_BK), ndlType_arrivalRate, 1+13 bits.
     * NDL_txPacketArrivalRate(AC_BE), ndlType_arrivalRate, 1+13 bits.
     * NDL_txPacketArrivalRate(AC_VI), ndlType_arrivalRate, 1+13 bits.
     * NDL_txPacketArrivalRate(AC_VO), ndlType_arrivalRate, 1+13 bits.
     */
    A_UINT32 tx_packet_arrival_rate_ac[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC,SIZE_NDLTYPE_ARRIVALRATE)];
    /**
     * NDL_txPacketAvgDuration(AC_BK), ndlType_packetDuration, 1+11 bits.
     * NDL_txPacketAvgDuration(AC_BE), ndlType_packetDuration, 1+11 bits.
     * NDL_txPacketAvgDuration(AC_VI), ndlType_packetDuration, 1+11 bits.
     * NDL_txPacketAvgDuration(AC_VO), ndlType_packetDuration, 1+11 bits.
     */
    A_UINT32 tx_packet_avg_duration_ac[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC,SIZE_NDLTYPE_PACKETDURATION)];
    /**
     * NDL_txChannelUse(AC_BK), ndlType_channelUse, 1+13 bits.
     * NDL_txChannelUse(AC_BE), ndlType_channelUse, 1+13 bits.
     * NDL_txChannelUse(AC_VI), ndlType_channelUse, 1+13 bits.
     * NDL_txChannelUse(AC_VO), ndlType_channelUse, 1+13 bits.
     */
    A_UINT32 tx_channel_use_ac[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC,SIZE_NDLTYPE_CHANNELUSE)];
    /**
     * NDL_txSignalAvgPower(AC_BK), ndlType_txPower, 1+7 bits.
     * NDL_txSignalAvgPower(AC_BE), ndlType_txPower, 1+7 bits.
     * NDL_txSignalAvgPower(AC_VI), ndlType_txPower, 1+7 bits.
     * NDL_txSignalAvgPower(AC_VO), ndlType_txPower, 1+7 bits.
     */
    A_UINT32 tx_signal_avg_power_ac[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC,SIZE_NDLTYPE_TXPOWER)];
} wmi_dcc_ndl_stats_per_channel;

#define WMI_NDL_STATS_SNR_BACKOFF_GET(ptr,mcs)      wmi_packed_arr_get_bits((ptr)->snr_backoff_mcs, mcs, SIZE_NDLTYPE_SNR)
#define WMI_NDL_STATS_SNR_BACKOFF_SET(ptr,mcs,val)  wmi_packed_arr_set_bits((ptr)->snr_backoff_mcs, mcs, SIZE_NDLTYPE_SNR, val)

#define WMI_NDL_STATS_CHAN_FREQ_GET(ptr)            WMI_GET_BITS((ptr)->chan_info, 0, 16)
#define WMI_NDL_STATS_CHAN_FREQ_SET(ptr,val)        WMI_SET_BITS((ptr)->chan_info, 0, 16, val)
#define WMI_NDL_STATS_DCC_STATS_BITMAP_GET(ptr)     WMI_GET_BITS((ptr)->chan_info, 16, 8)
#define WMI_NDL_STATS_DCC_STATS_BITMAP_SET(ptr,val) WMI_SET_BITS((ptr)->chan_info, 16, 8, val)

#define WMI_NDL_STATS_SNR_BACKOFF_GET(ptr,mcs)      wmi_packed_arr_get_bits((ptr)->snr_backoff_mcs, mcs, SIZE_NDLTYPE_SNR)
#define WMI_NDL_STATS_SNR_BACKOFF_SET(ptr,mcs,val)  wmi_packed_arr_set_bits((ptr)->snr_backoff_mcs, mcs, SIZE_NDLTYPE_SNR, val)

#define WMI_TX_POWER_GET(ptr)                       WMI_GET_BITS((ptr)->tx_power_datarate, 0, 8)
#define WMI_TX_POWER_SET(ptr,val)                   WMI_SET_BITS((ptr)->tx_power_datarate, 0, 8, val)
#define WMI_TX_DATARATE_GET(ptr)                    WMI_GET_BITS((ptr)->tx_power_datarate, 0, 4)
#define WMI_TX_DATARATE_SET(ptr,val)                WMI_SET_BITS((ptr)->tx_power_datarate, 0, 4, val)
#define WMI_NDL_CARRIER_SENSE_RANGE_GET(ptr)        WMI_GET_BITS((ptr)->carrier_sense_est_comm_range, 0, 13)
#define WMI_NDL_CARRIER_SENSE_RANGE_SET(ptr,val)    WMI_SET_BITS((ptr)->carrier_sense_est_comm_range, 0, 13, val)
#define WMI_NDL_EST_COMM_RANGE_GET(ptr)             WMI_GET_BITS((ptr)->carrier_sense_est_comm_range, 13, 13)
#define WMI_NDL_EST_COMM_RANGE_SET(ptr,val)         WMI_SET_BITS((ptr)->carrier_sense_est_comm_range, 13, 13, val)

#define WMI_DCC_SENSITIVITY_GET(ptr)                WMI_GET_BITS((ptr)->dcc_stats, 0, 8)
#define WMI_DCC_SENSITIVITY_SET(ptr,val)            WMI_SET_BITS((ptr)->dcc_stats, 0, 8, val)
#define WMI_CARRIER_SENSE_GET(ptr)                  WMI_GET_BITS((ptr)->dcc_stats, 8, 8)
#define WMI_CARRIER_SENSE_SET(ptr,val)              WMI_SET_BITS((ptr)->dcc_stats, 8, 8, val)
#define WMI_NDL_CHANNEL_LOAD_GET(ptr)               WMI_GET_BITS((ptr)->dcc_stats, 16, 11)
#define WMI_NDL_CHANNEL_LOAD_SET(ptr,val)           WMI_SET_BITS((ptr)->dcc_stats, 16, 11, val)
#define WMI_NDL_PACKET_ARRIVAL_RATE_GET(ptr)        WMI_GET_BITS((ptr)->packet_stats, 0, 14)
#define WMI_NDL_PACKET_ARRIVAL_RATE_SET(ptr,val)    WMI_SET_BITS((ptr)->packet_stats, 0, 14, val)
#define WMI_NDL_PACKET_AVG_DURATION_GET(ptr)        WMI_GET_BITS((ptr)->packet_stats, 14, 12)
#define WMI_NDL_PACKET_AVG_DURATION_SET(ptr,val)    WMI_SET_BITS((ptr)->packet_stats, 14, 12, val)
#define WMI_NDL_CHANNEL_BUSY_TIME_GET(ptr)          WMI_GET_BITS((ptr)->channel_busy_time, 0, 11)
#define WMI_NDL_CHANNEL_BUSY_TIME_SET(ptr,val)      WMI_SET_BITS((ptr)->channel_busy_time, 0, 11, val)

#define WMI_NDL_TX_PACKET_ARRIVAL_RATE_GET(ptr,acprio)          wmi_packed_arr_get_bits((ptr)->tx_packet_arrival_rate_ac, acprio, SIZE_NDLTYPE_ARRIVALRATE)
#define WMI_NDL_TX_PACKET_ARRIVAL_RATE_SET(ptr,acprio,val)      wmi_packed_arr_set_bits((ptr)->tx_packet_arrival_rate_ac, acprio, SIZE_NDLTYPE_ARRIVALRATE, val)
#define WMI_NDL_TX_PACKET_AVG_DURATION_GET(ptr,acprio)          wmi_packed_arr_get_bits((ptr)->tx_packet_avg_duration_ac, acprio, SIZE_NDLTYPE_PACKETDURATION)
#define WMI_NDL_TX_PACKET_AVG_DURATION_SET(ptr,acprio,val)      wmi_packed_arr_set_bits((ptr)->tx_packet_avg_duration_ac, acprio, SIZE_NDLTYPE_PACKETDURATION, val)
#define WMI_NDL_TX_CHANNEL_USE_GET(ptr,acprio)                  wmi_packed_arr_get_bits((ptr)->tx_channel_use_ac, acprio, SIZE_NDLTYPE_CHANNELUSE)
#define WMI_NDL_TX_CHANNEL_USE_SET(ptr,acprio,val)              wmi_packed_arr_set_bits((ptr)->tx_channel_use_ac, acprio, SIZE_NDLTYPE_CHANNELUSE, val)
#define WMI_NDL_TX_SIGNAL_AVG_POWER_GET(ptr,acprio)             wmi_packed_arr_get_bits((ptr)->tx_signal_avg_power_ac, acprio, SIZE_NDLTYPE_TXPOWER)
#define WMI_NDL_TX_SIGNAL_AVG_POWER_SET(ptr,acprio,val)         wmi_packed_arr_set_bits((ptr)->tx_signal_avg_power_ac, acprio, SIZE_NDLTYPE_TXPOWER, val)

/** Bitmap for DCC stats. */
typedef enum {
    DCC_STATS_DEMODULATION_MODEL = 1,
    DCC_STATS_COMMUNICATION_RANGES = 2,
    DCC_STATS_CHANNEL_LOAD_MEASURES = 4,
    DCC_STATS_TRANSMIT_PACKET_STATS = 8,
    DCC_STATS_TRANSMIT_MODEL_PARAMETER = 16,
    DCC_STATS_ALL = 0xff,
} wmi_dcc_stats_bitmap;

/** Data structure for getting the DCC stats. */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_dcc_get_stats_cmd_fixed_param */
    A_UINT32 tlv_header;

    /* VDEV identifier */
    A_UINT32 vdev_id;

    /** The number of channels for which stats are being requested. */
    A_UINT32 num_channels;

    /** This is followed by a TLV array of wmi_dcc_channel_stats_request. */
} wmi_dcc_get_stats_cmd_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_dcc_channel_stats_request */
    A_UINT32 tlv_header;

    /** The channel for which this applies. */
    A_UINT32 chan_freq; /* MHz units */

    /** The DCC stats being requested. */
    A_UINT32 dcc_stats_bitmap;
} wmi_dcc_channel_stats_request;

/** Data structure for the response with the DCC stats. */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_dcc_get_stats_resp_event_fixed_param */
    A_UINT32 tlv_header;
    /* VDEV identifier */
    A_UINT32 vdev_id;
    /** Number of channels in the response. */
    A_UINT32 num_channels;
    /** This is followed by a TLV array of wmi_dcc_ndl_stats_per_channel. */
} wmi_dcc_get_stats_resp_event_fixed_param;

/** Data structure for clearing the DCC stats. */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_dcc_clear_stats_cmd_fixed_param */
    A_UINT32 tlv_header;
    /* VDEV identifier */
    A_UINT32 vdev_id;
    A_UINT32 dcc_stats_bitmap;
} wmi_dcc_clear_stats_cmd_fixed_param;

/** Data structure for the pushed DCC stats */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_dcc_stats_event_fixed_param */
    A_UINT32 tlv_header;
    /* VDEV identifier */
    A_UINT32 vdev_id;
    /** The number of channels in the response. */
    A_UINT32 num_channels;

    /** This is followed by a TLV array of wmi_dcc_ndl_stats_per_channel. */
} wmi_dcc_stats_event_fixed_param;

/** Data structure for updating NDL per channel. */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_dcc_ndl_chan */
    A_UINT32 tlv_header;

    /**
     * Channel frequency, 16 bits
     * NDL_numActiveState, ndlType_numberElements, 1+6 bits
     */
    A_UINT32 chan_info;

    /**
     *  NDL_minDccSampling, 10 bits.
     *      Maximum time interval between subsequent checks of the DCC rules.
     */
    A_UINT32 ndl_min_dcc_sampling;

    /**
     * dcc_enable, 1 bit.
     * dcc_stats_enable, 1 bit.
     * dcc_stats_interval, 16 bits.
     */
    A_UINT32 dcc_flags;

    /** General DCC configuration. */
    /**
     * NDL_timeUp, ndlType_timing, 1+12 bits.
     * NDL_timeDown, ndlType_timing, 1+12 bits.
     */
    A_UINT32 general_config;

    /** Transmit power thresholds. */
    /**
     * NDL_minTxPower, ndlType_txPower, 1+7 bits.
     * NDL_maxTxPower, ndlType_txPower, 1+7 bits.
     */
    A_UINT32 min_max_tx_power; /* see "ETSI TS 102 687" table above for units */
    /**
     * NDL_defTxPower(AC_BK), ndlType_txPower, 1+7 bits.
     * NDL_defTxPower(AC_BE), ndlType_txPower, 1+7 bits.
     * NDL_defTxPower(AC_VI), ndlType_txPower, 1+7 bits.
     * NDL_defTxPower(AC_VO), ndlType_txPower, 1+7 bits.
     */
    /* see "ETSI TS 102 687" table above for units */
    A_UINT32 def_tx_power_ac[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC,SIZE_NDLTYPE_TXPOWER)];

    /** Packet timing thresholds. */
    /**
     * NDL_maxPacketDuration(AC_BK), ndlType_packetDuration, 1+11 bits.
     * NDL_maxPacketDuration(AC_BE), ndlType_packetDuration, 1+11 bits.
     * NDL_maxPacketDuration(AC_VI), ndlType_packetDuration, 1+11 bits.
     * NDL_maxPacketDuration(AC_VO), ndlType_packetDuration, 1+11 bits.
     */
    A_UINT32 max_packet_duration_ac[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC,SIZE_NDLTYPE_PACKETDURATION)];
    /**
     * NDL_minPacketInterval, ndlType_packetInterval, 1+10 bits.
     * NDL_maxPacketInterval, ndlType_packetInterval, 1+10 bits.
     */
    A_UINT32 min_max_packet_interval;
    /**
     * NDL_defPacketInterval(AC_BK), ndlType_packetInterval, 1+10 bits.
     * NDL_defPacketInterval(AC_BE), ndlType_packetInterval, 1+10 bits.
     * NDL_defPacketInterval(AC_VI), ndlType_packetInterval, 1+10 bits.
     * NDL_defPacketInterval(AC_VO), ndlType_packetInterval, 1+10 bits.
     */
    A_UINT32 def_packet_interval_ac[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC,SIZE_NDLTYPE_PACKETINTERVAL)];

    /** Packet datarate thresholds. */
    /**
     * NDL_minDatarate, ndlType_datarate, 1+3 bits.
     * NDL_maxDatarate, ndlType_datarate, 1+3 bits.
     */
    A_UINT32 min_max_datarate;
    /**
     * NDL_defDatarate(AC_BK), ndlType_datarate, 1+3 bits.
     * NDL_defDatarate(AC_BE), ndlType_datarate, 1+3 bits.
     * NDL_defDatarate(AC_VI), ndlType_datarate, 1+3 bits.
     * NDL_defDatarate(AC_VO), ndlType_datarate, 1+3 bits.
     */
    A_UINT32 def_datarate_ac[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC,SIZE_NDLTYPE_DATARATE)];

    /** Receive signal thresholds. */
    /**
     * NDL_minCarrierSense, ndlType_rxPower, 1+7 bits.
     * NDL_maxCarrierSense, ndlType_rxPower, 1+7 bits.
     * NDL_defCarrierSense, ndlType_rxPower, 1+7 bits.
     */
    A_UINT32 min_max_def_carrier_sense;

    /** Receive model parameter. */
    /**
     * NDL_defDccSensitivity, ndlType_rxPower, 1+7 bits.
     * NDL_maxCsRange, ndlType_distance, 1+12 bits.
     * NDL_refPathLoss, ndlType_pathloss, 1+5 bits.
     */
    A_UINT32 receive_model_parameter;

    /**
     * NDL_minSNR, ndlType_snr, 1+7 bits.
     */
    A_UINT32 receive_model_parameter_2;

    /** Demodulation model parameters. */
    /**
     * NDL_snrBackoff(MCS0), ndlType_snr, 1+7 bits.
     * NDL_snrBackoff(MCS1), ndlType_snr, 1+7 bits.
     * NDL_snrBackoff(MCS2), ndlType_snr, 1+7 bits.
     * NDL_snrBackoff(MCS3), ndlType_snr, 1+7 bits.
     * NDL_snrBackoff(MCS4), ndlType_snr, 1+7 bits.
     * NDL_snrBackoff(MCS5), ndlType_snr, 1+7 bits.
     * NDL_snrBackoff(MCS6), ndlType_snr, 1+7 bits.
     * NDL_snrBackoff(MCS7), ndlType_snr, 1+7 bits.
     */
    A_UINT32 snr_backoff_mcs[WMI_PACKED_ARR_SIZE(MCS_COUNT,SIZE_NDLTYPE_SNR)];

    /** Transmit model parameters. */
    /**
     * NDL_tmPacketArrivalRate(AC_BK), ndlType_arrivalRate, 1+13 bits.
     * NDL_tmPacketArrivalRate(AC_BE), ndlType_arrivalRate, 1+13 bits.
     * NDL_tmPacketArrivalRate(AC_VI), ndlType_arrivalRate, 1+13 bits.
     * NDL_tmPacketArrivalRate(AC_VO), ndlType_arrivalRate, 1+13 bits.
     */
    A_UINT32 tm_packet_arrival_rate_ac[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC, SIZE_NDLTYPE_ARRIVALRATE)];
    /**
     * NDL_tmPacketAvgDuration(AC_BK), ndlType_packetDuration, 1+11 bits.
     * NDL_tmPacketAvgDuration(AC_BE), ndlType_packetDuration, 1+11 bits.
     * NDL_tmPacketAvgDuration(AC_VI), ndlType_packetDuration, 1+11 bits.
     * NDL_tmPacketAvgDuration(AC_VO), ndlType_packetDuration, 1+11 bits.
     */
    A_UINT32 tm_packet_avg_duration_ac[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC, SIZE_NDLTYPE_PACKETDURATION)];
    /**
     * NDL_tmSignalAvgPower(AC_BK), ndlType_txPower, 1+7 bits.
     * NDL_tmSignalAvgPower(AC_BE), ndlType_txPower, 1+7 bits.
     * NDL_tmSignalAvgPower(AC_VI), ndlType_txPower, 1+7 bits.
     * NDL_tmSignalAvgPower(AC_VO), ndlType_txPower, 1+7 bits.
     */
    A_UINT32 tm_signal_avg_power_ac[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC, SIZE_NDLTYPE_TXPOWER)];
    /**
     * NDL_tmMaxChannelUse, ndlType_channelUse, 1+13 bits.
     */
    A_UINT32 tm_max_channel_use;
    /**
     * NDL_tmChannelUse(AC_BK), ndlType_channelUse, 1+13 bits.
     * NDL_tmChannelUse(AC_BE), ndlType_channelUse, 1+13 bits.
     * NDL_tmChannelUse(AC_VI), ndlType_channelUse, 1+13 bits.
     * NDL_tmChannelUse(AC_VO), ndlType_channelUse, 1+13 bits.
     */
    A_UINT32 tm_channel_use_ac[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC, SIZE_NDLTYPE_CHANNELUSE)];

    /** Channel load thresholds. */
    /**
     * NDL_minChannelLoad, ndlType_channelLoad, 1+10 bits.
     * NDL_maxChannelLoad, ndlType_channelLoad, 1+10 bits.
     */
    A_UINT32 min_max_channel_load;

    /** Transmit queue parameters. */
    /**
     * NDL_numQueue, ndlType_acPrio, 1+3 bits.
     * NDL_refQueueStatus(AC_BK), ndlType_queueStatus, 1+1 bit.
     * NDL_refQueueStatus(AC_BE), ndlType_queueStatus, 1+1 bit.
     * NDL_refQueueStatus(AC_VI), ndlType_queueStatus, 1+1 bit.
     * NDL_refQueueStatus(AC_VO), ndlType_queueStatus, 1+1 bit.
     */
    A_UINT32 transmit_queue_parameters;

    /**
     * NDL_refQueueLen(AC_BK), ndlType_numberElements, 1+6 bits.
     * NDL_refQueueLen(AC_BE), ndlType_numberElements, 1+6 bits.
     * NDL_refQueueLen(AC_VI), ndlType_numberElements, 1+6 bits.
     * NDL_refQueueLen(AC_VO), ndlType_numberElements, 1+6 bits.
     */
    A_UINT32 numberElements[WMI_PACKED_ARR_SIZE(WLAN_MAX_AC, SIZE_NDLTYPE_NUMBERELEMENTS)];

} wmi_dcc_ndl_chan;

#define WMI_CHAN_FREQ_GET(ptr)                  WMI_GET_BITS((ptr)->chan_info, 0, 16)
#define WMI_CHAN_FREQ_SET(ptr,val)              WMI_SET_BITS((ptr)->chan_info, 0, 16, val)
#define WMI_NDL_NUM_ACTIVE_STATE_GET(ptr)       WMI_GET_BITS((ptr)->chan_info, 16, 7)
#define WMI_NDL_NUM_ACTIVE_STATE_SET(ptr,val)   WMI_SET_BITS((ptr)->chan_info, 16, 7, val)

#define WMI_NDL_MIN_DCC_SAMPLING_GET(ptr)       WMI_GET_BITS((ptr)->ndl_min_dcc_sampling, 0, 10)
#define WMI_NDL_MIN_DCC_SAMPLING_SET(ptr,val)   WMI_SET_BITS((ptr)->ndl_min_dcc_sampling, 0, 10, val)
#define WMI_NDL_MEASURE_INTERVAL_GET(ptr)       WMI_GET_BITS((ptr)->ndl_min_dcc_sampling, 10, 16)
#define WMI_NDL_MEASURE_INTERVAL_SET(ptr,val)   WMI_SET_BITS((ptr)->ndl_min_dcc_sampling, 10, 16, val)

#define WMI_NDL_DCC_ENABLE_GET(ptr)             WMI_GET_BITS((ptr)->dcc_flags, 0, 1)
#define WMI_NDL_DCC_ENABLE_SET(ptr,val)         WMI_SET_BITS((ptr)->dcc_flags, 0, 1, val)
#define WMI_NDL_DCC_STATS_ENABLE_GET(ptr)       WMI_GET_BITS((ptr)->dcc_flags, 1, 1)
#define WMI_NDL_DCC_STATS_ENABLE_SET(ptr,val)   WMI_SET_BITS((ptr)->dcc_flags, 1, 1, val)
#define WMI_NDL_DCC_STATS_INTERVAL_GET(ptr)     WMI_GET_BITS((ptr)->dcc_flags, 2, 16)
#define WMI_NDL_DCC_STATS_INTERVAL_SET(ptr,val) WMI_SET_BITS((ptr)->dcc_flags, 2, 16, val)

#define WMI_NDL_TIME_UP_GET(ptr)                WMI_GET_BITS((ptr)->general_config, 0, 13)
#define WMI_NDL_TIME_UP_SET(ptr,val)            WMI_SET_BITS((ptr)->general_config, 0, 13, val)
#define WMI_NDL_TIME_DOWN_GET(ptr)              WMI_GET_BITS((ptr)->general_config, 13, 13)
#define WMI_NDL_TIME_DOWN_SET(ptr,val)          WMI_SET_BITS((ptr)->general_config, 13, 13, val)

#define WMI_NDL_MIN_TX_POWER_GET(ptr)       WMI_GET_BITS((ptr)->min_max_tx_power, 0, 8)
#define WMI_NDL_MIN_TX_POWER_SET(ptr,val)   WMI_SET_BITS((ptr)->min_max_tx_power, 0, 8, val)
#define WMI_NDL_MAX_TX_POWER_GET(ptr)       WMI_GET_BITS((ptr)->min_max_tx_power, 8, 8)
#define WMI_NDL_MAX_TX_POWER_SET(ptr,val)   WMI_SET_BITS((ptr)->min_max_tx_power, 8, 8, val)

#define WMI_NDL_DEF_TX_POWER_GET(ptr,acprio)        wmi_packed_arr_get_bits((ptr)->def_tx_power_ac, acprio, SIZE_NDLTYPE_TXPOWER)
#define WMI_NDL_DEF_TX_POWER_SET(ptr,acprio,val)    wmi_packed_arr_set_bits((ptr)->def_tx_power_ac, acprio, SIZE_NDLTYPE_TXPOWER, val)

#define WMI_NDL_MAX_PACKET_DURATION_GET(ptr,acprio)     wmi_packed_arr_get_bits((ptr)->max_packet_duration_ac, acprio, SIZE_NDLTYPE_PACKETDURATION)
#define WMI_NDL_MAX_PACKET_DURATION_SET(ptr,acprio,val) wmi_packed_arr_set_bits((ptr)->max_packet_duration_ac, acprio, SIZE_NDLTYPE_PACKETDURATION, val)
#define WMI_NDL_MIN_PACKET_INTERVAL_GET(ptr)            WMI_GET_BITS((ptr)->min_max_packet_interval, 0, 11)
#define WMI_NDL_MIN_PACKET_INTERVAL_SET(ptr,val)        WMI_SET_BITS((ptr)->min_max_packet_interval, 0, 11, val)
#define WMI_NDL_MAX_PACKET_INTERVAL_GET(ptr)            WMI_GET_BITS((ptr)->min_max_packet_interval, 11, 11)
#define WMI_NDL_MAX_PACKET_INTERVAL_SET(ptr,val)        WMI_SET_BITS((ptr)->min_max_packet_interval, 11, 11, val)
#define WMI_NDL_DEF_PACKET_INTERVAL_GET(ptr,acprio)     wmi_packed_arr_get_bits((ptr)->def_packet_interval_ac, acprio, SIZE_NDLTYPE_PACKETINTERVAL)
#define WMI_NDL_DEF_PACKET_INTERVAL_SET(ptr,acprio,val) wmi_packed_arr_set_bits((ptr)->def_packet_interval_ac, acprio, SIZE_NDLTYPE_PACKETINTERVAL, val)

#define WMI_NDL_MIN_DATARATE_GET(ptr)               WMI_GET_BITS((ptr)->min_max_datarate, 0, 4)
#define WMI_NDL_MIN_DATARATE_SET(ptr,val)           WMI_SET_BITS((ptr)->min_max_datarate, 0, 4, val)
#define WMI_NDL_MAX_DATARATE_GET(ptr)               WMI_GET_BITS((ptr)->min_max_datarate, 4, 4)
#define WMI_NDL_MAX_DATARATE_SET(ptr,val)           WMI_SET_BITS((ptr)->min_max_datarate, 4, 4, val)
#define WMI_NDL_DEF_DATARATE_GET(ptr,acprio)        wmi_packed_arr_get_bits((ptr)->def_datarate_ac, acprio, SIZE_NDLTYPE_DATARATE)
#define WMI_NDL_DEF_DATARATE_SET(ptr,acprio,val)    wmi_packed_arr_set_bits((ptr)->def_datarate_ac, acprio, SIZE_NDLTYPE_DATARATE, val)

#define WMI_NDL_MIN_CARRIER_SENSE_GET(ptr)      WMI_GET_BITS((ptr)->min_max_def_carrier_sense, 0, 8)
#define WMI_NDL_MIN_CARRIER_SENSE_SET(ptr,val)  WMI_SET_BITS((ptr)->min_max_def_carrier_sense, 0, 8, val)
#define WMI_NDL_MAX_CARRIER_SENSE_GET(ptr)      WMI_GET_BITS((ptr)->min_max_def_carrier_sense, 8, 8)
#define WMI_NDL_MAX_CARRIER_SENSE_SET(ptr,val)  WMI_SET_BITS((ptr)->min_max_def_carrier_sense, 8, 8, val)
#define WMI_NDL_DEF_CARRIER_SENSE_GET(ptr)      WMI_GET_BITS((ptr)->min_max_def_carrier_sense, 16, 8)
#define WMI_NDL_DEF_CARRIER_SENSE_SET(ptr,val)  WMI_SET_BITS((ptr)->min_max_def_carrier_sense, 16, 8, val)

#define WMI_NDL_DEF_DCC_SENSITIVITY_GET(ptr)        WMI_GET_BITS((ptr)->receive_model_parameter, 0, 8)
#define WMI_NDL_DEF_DCC_SENSITIVITY_SET(ptr,val)    WMI_SET_BITS((ptr)->receive_model_parameter, 0, 8, val)
#define WMI_NDL_MAX_CS_RANGE_GET(ptr)               WMI_GET_BITS((ptr)->receive_model_parameter, 8, 13)
#define WMI_NDL_MAX_CS_RANGE_SET(ptr,val)           WMI_SET_BITS((ptr)->receive_model_parameter, 8, 13, val)
#define WMI_NDL_REF_PATH_LOSS_GET(ptr)              WMI_GET_BITS((ptr)->receive_model_parameter, 21, 6)
#define WMI_NDL_REF_PATH_LOSS_SET(ptr,val)          WMI_SET_BITS((ptr)->receive_model_parameter, 21, 6, val)

#define WMI_NDL_MIN_SNR_GET(ptr)                    WMI_GET_BITS((ptr)->receive_model_parameter_2, 0, 8)
#define WMI_NDL_MIN_SNR_SET(ptr,val)                WMI_SET_BITS((ptr)->receive_model_parameter_2, 0, 8, val)

#define WMI_NDL_SNR_BACKOFF_GET(ptr,mcs)        wmi_packed_arr_get_bits((ptr)->snr_backoff_mcs, mcs, SIZE_NDLTYPE_SNR)
#define WMI_NDL_SNR_BACKOFF_SET(ptr,mcs,val)    wmi_packed_arr_set_bits((ptr)->snr_backoff_mcs, mcs, SIZE_NDLTYPE_SNR, val)

#define WMI_NDL_TM_PACKET_ARRIVAL_RATE_GET(ptr,acprio)      wmi_packed_arr_get_bits((ptr)->tm_packet_arrival_rate_ac, acprio, SIZE_NDLTYPE_ARRIVALRATE)
#define WMI_NDL_TM_PACKET_ARRIVAL_RATE_SET(ptr,acprio,val)  wmi_packed_arr_set_bits((ptr)->tm_packet_arrival_rate_ac, acprio, SIZE_NDLTYPE_ARRIVALRATE, val)
#define WMI_NDL_TM_PACKET_AVG_DURATION_GET(ptr,acprio)      wmi_packed_arr_get_bits((ptr)->tm_packet_avg_duration_ac, acprio, SIZE_NDLTYPE_PACKETDURATION)
#define WMI_NDL_TM_PACKET_AVG_DURATION_SET(ptr,acprio,val)  wmi_packed_arr_set_bits((ptr)->tm_packet_avg_duration_ac, acprio, SIZE_NDLTYPE_PACKETDURATION, val)
#define WMI_NDL_TM_SIGNAL_AVG_POWER_GET(ptr,acprio)         wmi_packed_arr_get_bits((ptr)->tm_signal_avg_power_ac, acprio, SIZE_NDLTYPE_TXPOWER)
#define WMI_NDL_TM_SIGNAL_AVG_POWER_SET(ptr,acprio,val)     wmi_packed_arr_set_bits((ptr)->tm_signal_avg_power_ac, acprio, SIZE_NDLTYPE_TXPOWER, val)
#define WMI_NDL_TM_MAX_CHANNEL_USE_GET(ptr)                 WMI_GET_BITS((ptr)->tm_max_channel_use, 0, 14)
#define WMI_NDL_TM_MAX_CHANNEL_USE_SET(ptr,val)             WMI_SET_BITS((ptr)->tm_max_channel_use, 0, 14, val)
#define WMI_NDL_TM_CHANNEL_USE_GET(ptr,acprio)              wmi_packed_arr_get_bits((ptr)->tm_channel_use_ac, acprio, SIZE_NDLTYPE_CHANNELUSE)
#define WMI_NDL_TM_CHANNEL_USE_SET(ptr,acprio,val)          wmi_packed_arr_set_bits((ptr)->tm_channel_use_ac, acprio, SIZE_NDLTYPE_CHANNELUSE, val)

#define WMI_NDL_MIN_CHANNEL_LOAD_GET(ptr)       WMI_GET_BITS((ptr)->min_max_channel_load, 0, 11)
#define WMI_NDL_MIN_CHANNEL_LOAD_SET(ptr,val)   WMI_SET_BITS((ptr)->min_max_channel_load, 0, 11, val)
#define WMI_NDL_MAX_CHANNEL_LOAD_GET(ptr)       WMI_GET_BITS((ptr)->min_max_channel_load, 11, 11)
#define WMI_NDL_MAX_CHANNEL_LOAD_SET(ptr,val)   WMI_SET_BITS((ptr)->min_max_channel_load, 11, 11, val)

#define WMI_NDL_NUM_QUEUE_GET(ptr)                      WMI_GET_BITS((ptr)->transmit_queue_parameters, 0, 4)
#define WMI_NDL_NUM_QUEUE_SET(ptr,val)                  WMI_SET_BITS((ptr)->transmit_queue_parameters, 0, 4, val)
#define WMI_NDL_REF_QUEUE_STATUS_GET(ptr,acprio)        WMI_GET_BITS((ptr)->transmit_queue_parameters, (4 + (acprio * 2)), 2)
#define WMI_NDL_REF_QUEUE_STATUS_SET(ptr,acprio,val)    WMI_SET_BITS((ptr)->transmit_queue_parameters, (4 + (acprio * 2)), 2, val)
#define WMI_NDL_REF_QUEUE_LEN_GET(ptr,acprio)           wmi_packed_arr_get_bits((ptr)->numberElements, acprio, SIZE_NDLTYPE_NUMBERELEMENTS)
#define WMI_NDL_REF_QUEUE_LEN_SET(ptr,acprio,val)       wmi_packed_arr_set_bits((ptr)->numberElements, acprio, SIZE_NDLTYPE_NUMBERELEMENTS, val)

/** Data structure for updating the NDL. */
typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_dcc_update_ndl_cmd_fixed_param */
    A_UINT32 tlv_header;

    /* VDEV identifier */
    A_UINT32 vdev_id;

    /** The number of channels in the request. */
    A_UINT32 num_channel;

    /** This is followed by a TLV array of wmi_dcc_ndl_chan. */
    /** This is followed by a TLV array of wmi_dcc_ndl_active_state_config. */
} wmi_dcc_update_ndl_cmd_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_dcc_update_ndl_resp_event_fixed_param */
    A_UINT32 tlv_header;
    /* VDEV identifier */
    A_UINT32 vdev_id;
    A_UINT32 status;
} wmi_dcc_update_ndl_resp_event_fixed_param;

/* Actions for TSF timestamp */
typedef enum {
    TSF_TSTAMP_CAPTURE_REQ = 1,
    TSF_TSTAMP_CAPTURE_RESET = 2,
    TSF_TSTAMP_READ_VALUE = 3,
    TSF_TSTAMP_QTIMER_CAPTURE_REQ = 4,
} wmi_tsf_tstamp_action;

typedef struct {
    /** TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_vdev_tsf_tstamp_action_cmd_fixed_param */
    A_UINT32 tlv_header;
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /* action type, refer to wmi_tsf_tstamp_action */
    A_UINT32 tsf_action;
} wmi_vdev_tsf_tstamp_action_cmd_fixed_param;

typedef struct {
    /* TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_vdev_tsf_report_event_fixed_param */
    A_UINT32 tlv_header;
    /* VDEV identifier */
    A_UINT32 vdev_id;
    /* low 32bit of tsf */
    A_UINT32 tsf_low;
    /* high 32 bit of tsf */
    A_UINT32 tsf_high;
    /* low 32 bits of qtimer */
    A_UINT32 qtimer_low;
    /* high 32 bits of qtimer */
    A_UINT32 qtimer_high;
    /*
     * tsf_id: TSF ID for the current vdev
     * This field should be ignored unless the tsf_id_valid flag is set.
     */
    A_UINT32 tsf_id;
    A_UINT32 tsf_id_valid;
    /*
     * mac_id: MAC identifier
     * This field should be ignored unless the mac_id_valid flag is set.
     */
    A_UINT32 mac_id;
    A_UINT32 mac_id_valid;
} wmi_vdev_tsf_report_event_fixed_param;

/**
 * ie_id values:
 * 0 to 255 are used for individual IEEE802.11 Information Element types
 */
#define WMI_SET_VDEV_IE_ID_SCAN_SET_DEFAULT_IE 256

/* source values: */
#define WMI_SET_VDEV_IE_SOURCE_HOST     0x0

/* band values: */
typedef enum {
    WMI_SET_VDEV_IE_BAND_ALL = 0,
    WMI_SET_VDEV_IE_BAND_2_4GHZ,
    WMI_SET_VDEV_IE_BAND_5GHZ,
} wmi_set_vdev_ie_band;

typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_set_ie_cmd_fixed_param */
    A_UINT32 tlv_header;
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** unique id to identify the ie_data as defined by ieee 802.11 spec */
    A_UINT32 ie_id; /** ie_len corresponds to num of bytes in ie_data[] */
    A_UINT32 ie_len;
    /** source of this command */
    A_UINT32 ie_source; /* see WMI_SET_VDEV_IE_SOURCE_ defs */
    /** band for this IE - se wmi_set_vdev_ie_band enum */
    A_UINT32 band;
   /**
    * Following this structure is the TLV byte stream of ie data of length ie_buf_len:
    * A_UINT8 ie_data[]; */
} wmi_vdev_set_ie_cmd_fixed_param;

/* DISA feature related data structures */
#define MAX_MAC_HEADER_LEN 32
typedef enum {
	WMI_ENCRYPT_DECRYPT_FLAG_INVALID,
	WMI_ENCRYPT = 1,
	WMI_DECRYPT = 2,
} WMI_ENCRYPT_DECRYPT_FLAG;

typedef struct {
	/**
	 * TLV tag and len; tag equals
	 * WMITLV_TAG_STRUC_wmi_vdev_encrypt_decrypt_data_req_cmd_fixed_param
	 */
	A_UINT32 tlv_header;
	/** unique id identifying the VDEV, generated by the caller */
	A_UINT32 vdev_id;
	A_UINT32 key_flag; /* WMI_ENCRYPT_DECRYPT_FLAG */
	A_UINT32 key_idx;
	A_UINT32 key_cipher;
	A_UINT32 key_len; /* units = bytes */
	A_UINT32 key_txmic_len; /* units = bytes */
	A_UINT32 key_rxmic_len; /* units = bytes */
	/** Key: This array needs to be provided in little-endian order */
	A_UINT8 key_data[WMI_MAX_KEY_LEN];
	/**
	 * Packet number: This array needs to be provided in little-endian
	 *                order.
	 * If the PN is less than 8 bytes, the PN data shall be placed into this
	 * pn[] array starting at byte 0, leaving the MSBs empty.
	 */
	A_UINT8 pn[8];
	/**
	 * 802.11 MAC header to be typecast to struct ieee80211_qosframe_addr4
	 *  This array needs to be provided in little-endian order.
	 */
	A_UINT8 mac_hdr[MAX_MAC_HEADER_LEN];
	A_UINT32 data_len; /** Payload length, units = bytes */
	/**
	 * Following this struct are this TLV:
	 *     A_UINT8  data[]; <-- actual data to be encrypted,
	 *                          needs to be provided in little-endian order
	 */
} wmi_vdev_encrypt_decrypt_data_req_cmd_fixed_param;

/**
 * This event is generated in response to
 * WMI_VDEV_ENCRYPT_DECRYPT_DATA_REQ_CMDID from HOST.
 * On receiving WMI command WMI_VDEV_ENCRYPT_DECRYPT_DATA_REQ_CMDID from
 * HOST with DISA test vectors, DISA frame will prepared and submitted to HW,
 * then on receiving the tx completion for the DISA frame this WMI event
 * will be delivered to HOST with the encrypted frame.
 */
typedef struct {
	/**
	 * TLV tag and len; tag equals
	 * WMITLV_TAG_STRUC_wmi_vdev_encrypt_decrypt_data_resp_event_fixed_param
	 */
	A_UINT32 tlv_header;
	/* VDEV identifier */
	A_UINT32 vdev_id;
	A_INT32 status; /*  0: success, -1: Failure, */
	/* 802.11 header length + encrypted payload length (units = bytes) */
	A_UINT32 data_length;
	/**
	 * Following this struct is this TLV:
	 *    A_UINT8 enc80211_frame[]; <-- Encrypted 802.11 frame;
	 *        802.11 header + encrypted payload,
	 *        provided in little-endian order
	 */
} wmi_vdev_encrypt_decrypt_data_resp_event_fixed_param;

/* DEPRECATED - use wmi_pdev_set_pcl_cmd_fixed_param instead */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_soc_set_pcl_cmd_fixed_param */
    /** Set Preferred Channel List  **/

    /** # of channels to scan */
    A_UINT32 num_chan;
/**
 * TLV (tag length value ) parameters follow the wmi_soc_set_pcl_cmd
 * structure. The TLV's are:
 *     A_UINT32 channel_list[];
 **/
} wmi_soc_set_pcl_cmd_fixed_param;

/* Values for channel_weight */
typedef enum {
    WMI_PCL_WEIGHT_DISALLOW  = 0,
    WMI_PCL_WEIGHT_LOW       = 1,
    WMI_PCL_WEIGHT_MEDIUM    = 2,
    WMI_PCL_WEIGHT_HIGH      = 3,
    WMI_PCL_WEIGHT_VERY_HIGH = 4,
} wmi_pcl_chan_weight;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_pcl_cmd_fixed_param */
    /** Set Preferred Channel List  **/

    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;

    /** # of channels to scan */
    A_UINT32 num_chan;
/**
 * TLV (tag length value ) parameters follow the wmi_soc_set_pcl_cmd
 * structure. The TLV's are:
 *     A_UINT32 channel_weight[];  channel order & size will be as per the list provided in WMI_SCAN_CHAN_LIST_CMDID
 **/
} wmi_pdev_set_pcl_cmd_fixed_param;

/* DEPRECATED - use wmi_pdev_set_hw_mode_cmd_fixed_param instead */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_soc_set_hw_mode_cmd_fixed_param */
    /**  Set Hardware Mode  **/

    /* Hardware Mode Index */
    A_UINT32 hw_mode_index;
} wmi_soc_set_hw_mode_cmd_fixed_param;

typedef struct {
    /* TLV tag and len tag equals WMITLV_TAG_STRUC_wmi_pdev_band_to_mac */
    A_UINT32 tlv_header;
    /** pdev_id for identifying the MACC
     * See macros starting with WMI_PDEV_ID_ for values..
     */
    A_UINT32 pdev_id;
    /* start frequency in MHz */
    A_UINT32 start_freq;
    /* end frequency in MHz */
    A_UINT32 end_freq;
} wmi_pdev_band_to_mac;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_hw_mode_cmd_fixed_param */
    /**  Set Hardware Mode  **/

    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;

    /* Hardware Mode Index */
    A_UINT32 hw_mode_index;
    /* Number of band to mac TLVs */

    A_UINT32 num_band_to_mac;

    /* Followed by TLVs of typee
     * num_band_to_mac * wmi_pdev_band_to_mac.
     */
} wmi_pdev_set_hw_mode_cmd_fixed_param;

/* DEPRECATED - use wmi_pdev_set_dual_mac_config_cmd_fixed_param instead */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_soc_set_dual_mac_config_cmd_fixed_param */
    /**  Set Dual MAC Firmware Configuration  **/

    /* Concurrent scan configuration bits */
    A_UINT32 concurrent_scan_config_bits;
    /* Firmware mode configuration bits */
    A_UINT32 fw_mode_config_bits;
} wmi_soc_set_dual_mac_config_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_mac_config_cmd_fixed_param */
    /**  Set Dual MAC Firmware Configuration  **/

    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;

    /* Concurrent scan configuration bits */
    A_UINT32 concurrent_scan_config_bits;
    /* Firmware mode configuration bits */
    A_UINT32 fw_mode_config_bits;
} wmi_pdev_set_mac_config_cmd_fixed_param;

typedef struct { /* DEPRECATED */
    A_UINT32 num_tx_chains;
    A_UINT32 num_rx_chains;
    A_UINT32 reserved[2];
} soc_num_tx_rx_chains;

typedef struct {
    A_UINT32 num_tx_chains_2g;
    A_UINT32 num_rx_chains_2g;
    A_UINT32 num_tx_chains_5g;
    A_UINT32 num_rx_chains_5g;
} band_num_tx_rx_chains;

typedef union {
    soc_num_tx_rx_chains soc_txrx_chain_setting;
    band_num_tx_rx_chains band_txrx_chain_setting;
} antenna_num_tx_rx_chains;

typedef enum {
    ANTENNA_MODE_DISABLED = 0x0,
    ANTENNA_MODE_LOW_POWER_LOCATION_SCAN = 0x01,
    /* reserved */
} antenna_mode_reason;

/* DEPRECATED - use wmi_pdev_set_antenna_mode_cmd_fixed_param instead */
typedef struct {
    /*
     * TLV tag and len;
     *  tag equals WMITLV_TAG_STRUC_wmi_soc_set_antenna_mode_cmd_fixed_param
     */
    A_UINT32 tlv_header;

    /* the reason for setting antenna mode, refer antenna_mode_reason */
    A_UINT32 reason;

    /*
     * The above reason parameter will select whether the following union
     * is soc_num_tx_rx_chains or band_num_tx_rx_chains.
     */
    antenna_num_tx_rx_chains num_txrx_chains_setting;
} wmi_soc_set_antenna_mode_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len;  tag equals WMITLV_TAG_STRUC_wmi_pdev_set_antenna_mode_cmd_fixed_param */

    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;

    /* Bits 0-15 is the number of RX chains and 16-31 is the number of TX chains */
    A_UINT32 num_txrx_chains;
} wmi_pdev_set_antenna_mode_cmd_fixed_param;

/** Data structure for information specific to a VDEV to MAC mapping. */
/* DEPRECATED - use wmi_pdev_set_hw_mode_response_vdev_mac_entry instead */
typedef struct {
    /** TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_soc_set_hw_mode_response_vdev_mac_entry */
    A_UINT32 tlv_header;
    A_UINT32 vdev_id; /* VDEV ID */
    A_UINT32 mac_id; /* MAC ID */
} wmi_soc_set_hw_mode_response_vdev_mac_entry;

/** Data structure for information specific to a VDEV to MAC mapping. */
typedef struct {
    /** TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_pdev_set_hw_mode_response_vdev_mac_entry */
    A_UINT32 tlv_header;

    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;

    A_UINT32 vdev_id;
} wmi_pdev_set_hw_mode_response_vdev_mac_entry;

/* DEPRECATED - use wmi_pdev_set_hw_mode_response_event_fixed_param instead */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_soc_set_hw_mode_response_event_fixed_param */
    /**  Set Hardware Mode Response Event  **/

    /* Status of set_hw_mode command */
    /*
     * Values for Status:
     *  0 - OK; command successful
     *  1 - EINVAL; Requested invalid hw_mode
     *  2 - ECANCELED; HW mode change canceled
     *  3 - ENOTSUP; HW mode not supported
     *  4 - EHARDWARE; HW mode change prevented by hardware
     *  5 - EPENDING; HW mode change is pending
     *  6 - ECOEX; HW mode change conflict with Coex
     */
    A_UINT32 status;
    /* Configured Hardware Mode */
    A_UINT32 cfgd_hw_mode_index;
    /* Number of Vdev to Mac entries */
    A_UINT32 num_vdev_mac_entries;

/**
 * TLV (tag length value ) parameters follow the soc_set_hw_mode_response_event
 * structure. The TLV's are:
 *      A_UINT32 wmi_soc_set_hw_mode_response_vdev_mac_entry[];
 */
} wmi_soc_set_hw_mode_response_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_hw_mode_response_event_fixed_param */
    /**  Set Hardware Mode Response Event  **/

    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;

    /* Status of set_hw_mode command */
    /*
     * Values for Status:
     *  0 - OK; command successful
     *  1 - EINVAL; Requested invalid hw_mode
     *  2 - ECANCELED; HW mode change canceled
     *  3 - ENOTSUP; HW mode not supported
     *  4 - EHARDWARE; HW mode change prevented by hardware
     *  5 - EPENDING; HW mode change is pending
     *  6 - ECOEX; HW mode change conflict with Coex
     */
    A_UINT32 status;
    /* Configured Hardware Mode */
    A_UINT32 cfgd_hw_mode_index;
    /* Number of Vdev to Mac entries */
    A_UINT32 num_vdev_mac_entries;
/**
 * TLV (tag length value ) parameters follow the soc_set_hw_mode_response_event
 * structure. The TLV's are:
 *      A_UINT32 wmi_soc_set_hw_mode_response_vdev_mac_entry[];
 */
} wmi_pdev_set_hw_mode_response_event_fixed_param;

/* DEPRECATED - use wmi_pdev_hw_mode_transition_event_fixed_param instead */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_soc_hw_mode_transition_event_fixed_param */
    /**  Hardware Mode Transition Event **/

    /* Original or old Hardware mode */
    A_UINT32 old_hw_mode_index;
    /* New Hardware Mode */
    A_UINT32 new_hw_mode_index;
    /* Number of Vdev to Mac entries */
    A_UINT32 num_vdev_mac_entries;

/**
 * TLV (tag length value ) parameters follow the soc_set_hw_mode_response_event
 * structure. The TLV's are:
 *      A_UINT32 wmi_soc_set_hw_mode_response_vdev_mac_entry[];
 */
} wmi_soc_hw_mode_transition_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_hw_mode_transition_event_fixed_param */
    /**  Hardware Mode Transition Event **/

    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;

    /* Original or old Hardware mode */
    A_UINT32 old_hw_mode_index;
    /* New Hardware Mode */
    A_UINT32 new_hw_mode_index;
    /* Number of Vdev to Mac entries */
    A_UINT32 num_vdev_mac_entries;

/**
 * TLV (tag length value ) parameters follow the soc_set_hw_mode_response_event
 * structure. The TLV's are:
 *      A_UINT32 wmi_soc_set_hw_mode_response_vdev_mac_entry[];
 */
} wmi_pdev_hw_mode_transition_event_fixed_param;

/**
 * This command is sent from WLAN host driver to firmware for
 * plugging in reorder queue desc to lithium hw.
 *
 * Example: plug-in queue desc for TID 5
 *    host->target: WMI_PEER_REORDER_QUEUE_SETUP_CMDID,
 *                  (vdev_id = PEER vdev id,
 *                   peer_macaddr = PEER mac addr,
 *                   tid = 5,
 *                   queue_ptr_lo = queue desc addr lower 32 bits,
 *                   queue_ptr_hi = queue desc addr higher 32 bits,
 *                   queue_no = 16-bit number assigned by host for queue,
 *                              stored in bits 15:0 of queue_no field)
 */
typedef struct {
    /* TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_peer_reorder_queue_setup_cmd_fixed_param
     */
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    wmi_mac_addr peer_macaddr; /* peer mac address */
    A_UINT32 tid; /* 0 to 15 = QoS TIDs, 16 = non-qos TID */
    A_UINT32 queue_ptr_lo; /* lower 32 bits of queue desc adddress */
    A_UINT32 queue_ptr_hi; /* upper 32 bits of queue desc adddress */
    A_UINT32 queue_no; /* 16-bit number assigned by host for queue,
                          stored in bits 15:0 of queue_no field */
    A_UINT32 ba_window_size_valid; /* Is ba_window_size valid?
                                    * 0 = Invalid, 1 = Valid */
    A_UINT32 ba_window_size; /* Valid values: 0 to 256
                              * Host sends the message when BA session is
                              * established or terminated for the TID. */
} wmi_peer_reorder_queue_setup_cmd_fixed_param;

/**
 * This command is sent from WLAN host driver to firmware for
 * removing one or more reorder queue desc to lithium hw.
 *
 * Example: remove queue desc for all TIDs
 *    host->target: WMI_PEER_REORDER_REMOVE_CMDID,
 *                  (vdev_id = PEER vdev id,
 *                   peer_macaddr = PEER mac addr,
 *                   tid = 0x1FFFF,
 */
typedef struct {
    /* TLV tag and len;
     * tag equals WMITLV_TAG_STRUC_wmi_peer_reorder_queue_remove_cmd_fixed_param
     */
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    wmi_mac_addr peer_macaddr; /* peer mac address */
    A_UINT32 tid_mask; /* bits 0 to 15 = QoS TIDs, bit 16 = non-qos TID */
} wmi_peer_reorder_queue_remove_cmd_fixed_param;


/* DEPRECATED - use wmi_pdev_set_mac_config_response_event_fixed_param instead */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_soc_set_dual_mac_config_response_event_fixed_param */
    /**  Set Dual MAC Config Response Event  **/

    /* Status for set_dual_mac_config command */
    /*
     * Values for Status:
     *  0 - OK; command successful
     *  1 - EINVAL; Requested invalid hw_mode
     *  3 - ENOTSUP; HW mode not supported
     *  4 - EHARDWARE; HW mode change prevented by hardware
     *  6 - ECOEX; HW mode change conflict with Coex
     */
    A_UINT32 status;

} wmi_soc_set_dual_mac_config_response_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_mac_config_response_event_fixed_param */
    /**  Set Dual MAC Config Response Event  **/

    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;

    /* Status for set_dual_mac_config command */
    /*
     * Values for Status:
     *  0 - OK; command successful
     *  1 - EINVAL; Requested invalid hw_mode
     *  3 - ENOTSUP; HW mode not supported
     *  4 - EHARDWARE; HW mode change prevented by hardware
     *  6 - ECOEX; HW mode change conflict with Coex
     */
     A_UINT32 status;
} wmi_pdev_set_mac_config_response_event_fixed_param;

typedef enum {
    MAWC_MOTION_STATE_UNKNOWN,
    MAWC_MOTION_STATE_STATIONARY,
    MAWC_MOTION_STATE_WALK,
    MAWC_MOTION_STATE_TRANSIT,
} MAWC_MOTION_STATE;

typedef enum {
    MAWC_SENSOR_STATUS_OK,
    MAWC_SENSOR_STATUS_FAILED_TO_ENABLE,
    MAWC_SENSOR_STATUS_SHUTDOWN,
} MAWC_SENSOR_STATUS;

typedef struct {
    /* TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_mawc_sensor_report_ind_cmd_fixed_param */
    A_UINT32 tlv_header;
    /** new motion state, MAWC_MOTION_STATE */
    A_UINT32 motion_state;
    /** status code of sensor, MAWC_SENSOR_STATUS */
    A_UINT32 sensor_status;
} wmi_mawc_sensor_report_ind_cmd_fixed_param;

/* MBO flag field definition */
/* Bit 0: 0 - Allow to connect to both MBO and non-MBO AP
 *        1 - Allow to connect to MBO AP only
 * Bit 1-31 : reserved.
 */
#define WMI_ROAM_MBO_FLAG_MBO_ONLY_MODE  (1<<0) /* DEPRECATED */

typedef struct {
    /* TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_roam_set_mbo_fixed_param */
    A_UINT32 tlv_header;
    /** vdev id */
    A_UINT32 vdev_id;
    /** enable or disable MBO */
    A_UINT32 enable;
    /** MBO flags, refer to definition of MBO flags*/
    A_UINT32 flags;
} wmi_roam_set_mbo_fixed_param; /* DEPRECATED */

typedef struct {
    /* TLV tag and len; tag equals
     * WMITLV_TAG_ARRAY_STRUC */
    A_UINT32 tlv_header;
    /** Current operating class number */
    A_UINT32 cur_op_class;
    /** Country string of current reg domain,
     * the expected value should be same as country str defined
     * in country IE.
     * 3 octets (COUNTRY_STR) + 1 octet (always 0)
     * The ordering of this array must be maintained,
     * even when a big-endian host's WMI messages undergo
     * automatic byte reordering for conversion to the
     * little-endian ordering required by the target.
     * On big-endian hosts, this array may need to be byte-swapped
     * by the host, so the subsequent automatic byte-swap
     * will result in the correct final byte order.
     * global operating class: set country_str[0]=0
     */
    A_UINT8 country_str[4];
    /** Supported operating class number in current regdomain */
    A_UINT32 supp_op_class_num;
    /* The TLVs will follow. */
    /* A_UINT32 supp_op_class_list[] */
} wmi_supported_operating_class_param;

typedef struct {
    /* TLV tag and len; tag equals
     * WMITLV_TAG_ARRAY_STRUC */
    A_UINT32 tlv_header;
    /** non preferred channel attribute length */
    A_UINT32 non_prefer_ch_attr_len;
    /* The TLVs will follow. */
    /** A_UINT8 non_prefer_ch_attr[];*/
} wmi_mbo_non_preferred_channel_report_param;

typedef struct {
    /* TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_mawc_enable_sensor_event_fixed_param */
    A_UINT32 tlv_header;
    /* enable(1) or disable(0) */
    A_UINT32 enable;
} wmi_mawc_enable_sensor_event_fixed_param;

typedef struct {
    /* TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_extscan_configure_mawc_cmd_fixed_param */
    A_UINT32 tlv_header;
    /* Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /* enable(1) or disable(0) MAWC */
    A_UINT32 enable;
    /* ratio of skipping suppressing scan, skip one out of x */
    A_UINT32 suppress_ratio;
} wmi_extscan_configure_mawc_cmd_fixed_param;

typedef struct {
   /*
    * TLV tag and len; tag equals
    * WMITLV_TAG_STRUC_wmi_roam_per_config_fixed_param
    */
    A_UINT32 tlv_header;
   /* Unique id identifying the VDEV */
    A_UINT32 vdev_id;
   /* enable(1) or disable(0) packet error rate trigger for roaming */
    A_UINT32 enable;
   /* high_rate_thresh, low_rate_thresh, pkt_err_rate_thresh_pct:
    * If PER monitoring as a trigger for roaming is enabled,
    * it is controlled by high_rate_thresh, low_rate_thresh, and
    * pkt_err_rate_thresh_pct.
    * PER monitoring is performed only when the time-averaged throughput
    * is less than high_rate_thresh.
    * During PER monitoring, the target keeps track of the PHY rate for
    * each of the first N PPDUs within a time window.
    * If the number of PPDUs with PHY rate < low_rate_thresh exceeds
    * N * pkt_err_rate_thresh_pct / 100, roaming will be triggered.
    *
    * This PER monitoring as a trigger for roaming is performed
    * concurrently but independently for rx and tx.
    */
    A_UINT32 high_rate_thresh; /* units = Kbps */
    A_UINT32 low_rate_thresh; /* units = Kbps */
    A_UINT32 pkt_err_rate_thresh_pct;
   /*
    * rest time after associating to a new AP before
    * starting to monitor PER as a roaming trigger,
    * (units are seconds)
    */
    A_UINT32 per_rest_time;
    /* This is the total time for which PER monitoring will be run.
     * After completion of time windows, the average PER over the window
     * will be computed.
     * The parameter value stores specifications for both TX and RX
     * monitor times.
     * The two least-significant bytes (0 & 1) hold the RX monitor time;
     * the two most-significant bytes (2 & 3) hold the TX monitor time.
     */
    A_UINT32 pkt_err_rate_mon_time; /* units = seconds */
    /* Minimum roamable AP RSSI for candidate selection for PER based roam */
    A_INT32 min_candidate_rssi; /* units = dBm */
} wmi_roam_per_config_fixed_param;

typedef struct {
    /* TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_nlo_configure_mawc_cmd_fixed_param */
    A_UINT32 tlv_header;
    /* Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /* enable(1) or disable(0) MAWC */
    A_UINT32 enable;
    /* ratio of exponential backoff, next = current + current*ratio/100 */
    A_UINT32 exp_backoff_ratio;
    /* initial scan interval(msec) */
    A_UINT32 init_scan_interval;
    /* max scan interval(msec) */
    A_UINT32 max_scan_interval;
} wmi_nlo_configure_mawc_cmd_fixed_param;

typedef struct {
    /* TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_roam_configure_mawc_cmd_fixed_param */
    A_UINT32 tlv_header;
    /* Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /* enable(1) or disable(0) MAWC */
    A_UINT32 enable;
    /* data traffic load (kBps) to register CMC */
    A_UINT32 traffic_load_threshold;
    /* RSSI threshold (dBm) to scan for Best AP */
    A_UINT32 best_ap_rssi_threshold;
    /* high RSSI threshold adjustment in Stationary to suppress scan */
    A_UINT32 rssi_stationary_high_adjust;
    /* low RSSI threshold adjustment in Stationary to suppress scan */
    A_UINT32 rssi_stationary_low_adjust;
} wmi_roam_configure_mawc_cmd_fixed_param;

#define WMI_PACKET_FILTER_COMPARE_DATA_LEN_DWORD     2
#define WMI_PACKET_FILTER_MAX_CMP_PER_PACKET_FILTER  5

typedef enum {
    PACKET_FILTER_TYPE_INVALID = 0,
    PACKET_FILTER_TYPE_FILTER_PKT,
    PACKET_FILTER_TYPE_RESERVE_PKT, // not used
    PACKET_FILTER_TYPE_MAX_ENUM_SIZE
} WMI_PACKET_FILTER_FILTER_TYPE;

typedef enum {
    PACKET_FILTER_PROTO_TYPE_INVALID = 0,

    /* L2 header */
    PACKET_FILTER_PROTO_TYPE_MAC,
    PACKET_FILTER_PROTO_TYPE_SNAP,

    /* L3 header (EtherType) */
    PACKET_FILTER_PROTO_TYPE_IPV4,
    PACKET_FILTER_PROTO_TYPE_IPV6,

    /* L4 header (IP protocol) */
    PACKET_FILTER_PROTO_TYPE_UDP,
    PACKET_FILTER_PROTO_TYPE_TCP,
    PACKET_FILTER_PROTO_TYPE_ICMPV6,

    PACKET_FILTER_PROTO_TYPE_MAX
} WMI_PACKET_FILTER_PROTO_TYPE;

typedef enum {
    PACKET_FILTER_CMP_TYPE_INVALID = 0,
    PACKET_FILTER_CMP_TYPE_EQUAL,
    PACKET_FILTER_CMP_TYPE_MASK_EQUAL,
    PACKET_FILTER_CMP_TYPE_NOT_EQUAL,
    PACKET_FILTER_CMP_TYPE_MASK_NOT_EQUAL,
    PACKET_FILTER_CMP_TYPE_ADDRTYPE,
    PACKET_FILTER_CMP_TYPE_MAX
} WMI_PACKET_FILTER_CMP_TYPE;

typedef enum {
    PACKET_FILTER_SET_INACTIVE = 0,
    PACKET_FILTER_SET_ACTIVE
} WMI_PACKET_FILTER_ACTION;

typedef enum {
    PACKET_FILTER_SET_DISABLE = 0,
    PACKET_FILTER_SET_ENABLE
} WMI_PACKET_FILTER_RUNTIME_ENABLE;

typedef struct {
    A_UINT32  proto_type;
    A_UINT32  cmp_type;
    A_UINT32  data_length; /* Length of the data to compare (units = bytes) */
    A_UINT32  data_offset; /* from start of the respective frame header (
units = bytes) */
    A_UINT32  compareData[WMI_PACKET_FILTER_COMPARE_DATA_LEN_DWORD]; /* Data to compare, little-endian order */
    A_UINT32  dataMask[WMI_PACKET_FILTER_COMPARE_DATA_LEN_DWORD]; /* Mask to be applied on rcvd packet data before compare, little-endian order */
} WMI_PACKET_FILTER_PARAMS_TYPE;

typedef struct {
    A_UINT32  tlv_header;
    A_UINT32  vdev_id;
    A_UINT32  filter_id;
    A_UINT32  filter_action; /* WMI_PACKET_FILTER_ACTION */
    A_UINT32  filter_type;
    A_UINT32  num_params; /* how many entries in paramsData are valid */
    A_UINT32  coalesce_time; // not currently used - fill with 0x0
    WMI_PACKET_FILTER_PARAMS_TYPE  paramsData[
WMI_PACKET_FILTER_MAX_CMP_PER_PACKET_FILTER];
    /*
     * deprecated0:
     * This field simply provides filler space to retain the original message
     * format while reducing WMI_PACKET_FILTER_MAX_CMP_PER_PACKET_FILTER
     * from 10 to 5.
     */
    WMI_PACKET_FILTER_PARAMS_TYPE  deprecated0[5];
} WMI_PACKET_FILTER_CONFIG_CMD_fixed_param;

/* enable / disable all filters within the specified vdev */
typedef struct {
    A_UINT32  tlv_header;
    A_UINT32  vdev_id;
    A_UINT32  enable; /* WMI_PACKET_FILTER_RUNTIME_ENABLE */
} WMI_PACKET_FILTER_ENABLE_CMD_fixed_param;

#define WMI_LRO_INFO_TCP_FLAG_VALS_BITPOS  0
#define WMI_LRO_INFO_TCP_FLAG_VALS_NUMBITS 9

#define WMI_LRO_INFO_TCP_FLAG_VALS_SET(tcp_flag_u32, tcp_flag_values) \
    WMI_SET_BITS(tcp_flag_u32, \
    WMI_LRO_INFO_TCP_FLAG_VALS_BITPOS, \
    WMI_LRO_INFO_TCP_FLAG_VALS_NUMBITS, \
    tcp_flag_values)
#define WMI_LRO_INFO_TCP_FLAG_VALS_GET(tcp_flag_u32) \
    WMI_GET_BITS(tcp_flag_u32, \
    WMI_LRO_INFO_TCP_FLAG_VALS_BITPOS, \
    WMI_LRO_INFO_TCP_FLAG_VALS_NUMBITS)

#define WMI_LRO_INFO_TCP_FLAGS_MASK_BITPOS  9
#define WMI_LRO_INFO_TCP_FLAGS_MASK_NUMBITS 9

#define WMI_LRO_INFO_TCP_FLAGS_MASK_SET(tcp_flag_u32, tcp_flags_mask) \
    WMI_SET_BITS(tcp_flag_u32, \
    WMI_LRO_INFO_TCP_FLAGS_MASK_BITPOS, \
    WMI_LRO_INFO_TCP_FLAGS_MASK_NUMBITS, \
    tcp_flags_mask)
#define WMI_LRO_INFO_TCP_FLAGS_MASK_GET(tcp_flag_u32) \
    WMI_GET_BITS(tcp_flag_u32, \
    WMI_LRO_INFO_TCP_FLAGS_MASK_BITPOS, \
    WMI_LRO_INFO_TCP_FLAGS_MASK_NUMBITS)

typedef struct {
    A_UINT32 tlv_header;
    /**
     * @brief lro_enable - indicates whether lro is enabled
     * [0] LRO Enable
     */
    A_UINT32 lro_enable;
    /**
     * @brief tcp_flag_u32 - mask of which TCP flags to check and
     *      values to check for
     * [8:0] TCP flag values - If the TCP flags from the packet do not match
     *       the values in this field after masking with TCP flags mask below,
     *       LRO eligible will not be set
     * [17:9] TCP flags mask - Mask field for comparing the TCP values
     *       provided above with the TCP flags field in the received packet
     * Use WMI_LRO_INFO_TCP_FLAG_VALS and WMI_LRO_INFO_TCP_FLAGS_MASK
     * macros to isolate the mask field and values field that are packed
     * into this u32 "word".
     */
    A_UINT32 tcp_flag_u32;
    /**
     * @brief toeplitz_hash_ipv4 - contains seed needed to compute
     * the flow id 5-tuple toeplitz hash for IPv4 packets. Contains
     * bytes 0 to 3
     *
     * In this and all the below toeplitz_hash fields, the bytes are
     * specified in little-endian order.  For example:
     *     toeplitz_hash_ipv4_0_3 bits 7:0   holds seed byte 0
     *     toeplitz_hash_ipv4_0_3 bits 15:8  holds seed byte 1
     *     toeplitz_hash_ipv4_0_3 bits 23:16 holds seed byte 2
     *     toeplitz_hash_ipv4_0_3 bits 31:24 holds seed byte 3
     */
    A_UINT32 toeplitz_hash_ipv4_0_3;

    /**
     * @brief toeplitz_hash_ipv4 - contains seed needed to compute
     * the flow id 5-tuple toeplitz hash for IPv4 packets. Contains
     * bytes 4 to 7
     */
    A_UINT32 toeplitz_hash_ipv4_4_7;

    /**
     * @brief toeplitz_hash_ipv4 - contains seed needed to compute
     * the flow id 5-tuple toeplitz hash for IPv4 packets. Contains
     * bytes 8 to 11
     */
    A_UINT32 toeplitz_hash_ipv4_8_11;

    /**
     * @brief toeplitz_hash_ipv4 - contains seed needed to compute
     * the flow id 5-tuple toeplitz hash for IPv4 packets. Contains
     * bytes 12 to 15
     */
    A_UINT32 toeplitz_hash_ipv4_12_15;

    /**
     * @brief toeplitz_hash_ipv4 - contains seed needed to compute
     * the flow id 5-tuple toeplitz hash for IPv4 packets. Contains
     * byte 16
     */
    A_UINT32 toeplitz_hash_ipv4_16;

    /**
     * @brief toeplitz_hash_ipv6 - contains seed needed to compute
     * the flow id 5-tuple toeplitz hash for IPv6 packets. Contains
     * bytes 0 to 3
     */
    A_UINT32 toeplitz_hash_ipv6_0_3;

    /**
     * @brief toeplitz_hash_ipv6 - contains seed needed to compute
     * the flow id 5-tuple toeplitz hash for IPv6 packets. Contains
     * bytes 4 to 7
     */
    A_UINT32 toeplitz_hash_ipv6_4_7;

    /**
     * @brief toeplitz_hash_ipv6 - contains seed needed to compute
     * the flow id 5-tuple toeplitz hash for IPv6 packets. Contains
     * bytes 8 to 11
     */
    A_UINT32 toeplitz_hash_ipv6_8_11;

    /**
     * @brief toeplitz_hash_ipv6 - contains seed needed to compute
     * the flow id 5-tuple toeplitz hash for IPv6 packets. Contains
     * bytes 12 to 15
     */
    A_UINT32 toeplitz_hash_ipv6_12_15;

    /**
     * @brief toeplitz_hash_ipv6 - contains seed needed to compute
     * the flow id 5-tuple toeplitz hash for IPv6 packets. Contains
     * bytes 16 to 19
     */
    A_UINT32 toeplitz_hash_ipv6_16_19;

    /**
     * @brief toeplitz_hash_ipv6 - contains seed needed to compute
     * the flow id 5-tuple toeplitz hash for IPv6 packets. Contains
     * bytes 20 to 22
     */
    A_UINT32 toeplitz_hash_ipv6_20_23;

    /**
     * @brief toeplitz_hash_ipv6 - contains seed needed to compute
     * the flow id 5-tuple toeplitz hash for IPv6 packets. Contains
     * bytes 24 to 27
     */
    A_UINT32 toeplitz_hash_ipv6_24_27;

    /**
     * @brief toeplitz_hash_ipv6 - contains seed needed to compute
     * the flow id 5-tuple toeplitz hash for IPv6 packets. Contains
     * bytes 28 to 31
     */
    A_UINT32 toeplitz_hash_ipv6_28_31;

    /**
     * @brief toeplitz_hash_ipv6 - contains seed needed to compute
     * the flow id 5-tuple toeplitz hash for IPv6 packets. Contains
     * bytes 32 to 35
     */
    A_UINT32 toeplitz_hash_ipv6_32_35;

    /**
     * @brief toeplitz_hash_ipv6 - contains seed needed to compute
     * the flow id 5-tuple toeplitz hash for IPv6 packets. Contains
     * bytes 36 to 39
     */
    A_UINT32 toeplitz_hash_ipv6_36_39;

    /**
     * @brief toeplitz_hash_ipv6 - contains seed needed to compute
     * the flow id 5-tuple toeplitz hash for IPv6 packets. Contains
     * byte 40
     */
    A_UINT32 toeplitz_hash_ipv6_40;

    /**
     * @brief pdev_id - identifies the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
} wmi_lro_info_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_transfer_data_to_flash_cmd_fixed_param  */
    A_UINT32 offset; /* flash offset to write, starting from 0 */
    A_UINT32 length; /* vaild data length in buffer, unit: byte */
} wmi_transfer_data_to_flash_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_transfer_data_to_flash_complete_event_fixed_param */
    /** Return status. 0 for success, non-zero otherwise */
    A_UINT32 status;
} wmi_transfer_data_to_flash_complete_event_fixed_param;

typedef struct {
    /* TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_read_data_from_flash_cmd_fixed_param
     */
    A_UINT32 tlv_header;
    A_UINT32 offset; /* flash offset to read, starting from 0 */
    A_UINT32 length; /* data length to read, unit: byte */
} wmi_read_data_from_flash_cmd_fixed_param;

typedef struct {
    /* TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_read_data_from_flash_event_fixed_param
     */
    A_UINT32 tlv_header;
    A_UINT32 status; /* Return status. 0 for success, non-zero otherwise */
    A_UINT32 offset; /* flash offset reading from, starting from 0 */
    A_UINT32 length; /* length of data being reported, unit: byte */
} wmi_read_data_from_flash_event_fixed_param;

typedef enum {
    ENHANCED_MCAST_FILTER_DISABLED,
    ENHANCED_MCAST_FILTER_ENABLED
} ENHANCED_MCAST_FILTER_CONFIG;

/*
 * Command to enable/disable filtering of multicast IP with unicast mac
 */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_config_enhanced_mcast_filter_fixed_param */
    /* Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /* 1 = enable 0 = disable (see ENHANCED_MCAST_FILTER_CONFIG) */
    A_UINT32 enable;
} wmi_config_enhanced_mcast_filter_cmd_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_wisa_cmd_fixed_param */
    A_UINT32 tlv_header;
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id;
    /** WISA enable / disable mode */
    A_UINT32 wisa_mode;
} wmi_vdev_wisa_cmd_fixed_param;

/*
 * This structure is used to report SMPS force mode set complete to host.
 */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_sta_smps_force_mode_complete_event_fixed_param */
    /* Unique id identifying the VDEV */
    A_UINT32 vdev_id;
    /* Return status. 0 for success, non-zero otherwise */
    A_UINT32 status;
} wmi_sta_smps_force_mode_complete_event_fixed_param;

/*
 * This structure is used to report SCPC calibrated data to host.
 */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_scpc_event_fixed_param */
    /** number of BDF patches. Each patch contains offset, length and data */
    A_UINT32 num_patch;
    /*  This TLV is followed by another TLV of array of bytes
     *  A_UINT8 data[];
     *  This data array contains, for example
     *  patch1 offset(byte3~0),   patch1 data length(byte7~4),   patch1 data(byte11~8)
     *  patch2 offset(byte15~12), patch2 data length(byte19~16), patch2 data(byte47~20)
     *
     */
} wmi_scpc_event_fixed_param;

typedef enum {
FW_ACTIVE_BPF_MODE_DISABLE =         (1 << 1),
FW_ACTIVE_BPF_MODE_FORCE_ENABLE =    (1 << 2),
FW_ACTIVE_BPF_MODE_ADAPTIVE_ENABLE = (1 << 3),
} FW_ACTIVE_BPF_MODE;

/* bpf interface structure */
typedef struct wmi_bpf_get_capability_cmd_s {
    A_UINT32 tlv_header;
    A_UINT32 reserved;  /* reserved for future use - must be filled with 0x0 */
} wmi_bpf_get_capability_cmd_fixed_param;

typedef struct wmi_bpf_capability_info_evt_s {
    A_UINT32 tlv_header;
    A_UINT32 bpf_version; /* fw's implement version */
    A_UINT32 max_bpf_filters; /* max filters that fw supports */
    A_UINT32 max_bytes_for_bpf_inst; /* the maximum bytes that can be used as bpf instructions */
    A_UINT32 fw_active_bpf_support_mcbc_modes; /* multicast/broadcast - refer to FW_ACTIVE_BPF_MODE, it can be 'or' of them */
    A_UINT32 fw_active_bpf_support_uc_modes; /* unicast - refer to FW_ACTIVE_BPF_MODE, it can be 'or' of them */
} wmi_bpf_capability_info_evt_fixed_param;

/* bit 0 of flags: report counters */
#define WMI_BPF_GET_VDEV_STATS_FLAG_CTR_S  0
#define WMI_BPF_GET_VDEV_STATS_FLAG_CTR_M  0x1
typedef struct wmi_bpf_get_vdev_stats_cmd_s {
    A_UINT32 tlv_header;
    A_UINT32 flags;
    A_UINT32 vdev_id;
} wmi_bpf_get_vdev_stats_cmd_fixed_param;

typedef struct wmi_bpf_vdev_stats_info_evt_s {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 num_filters;
    A_UINT32 num_checked_pkts;
    A_UINT32 num_dropped_pkts;
} wmi_bpf_vdev_stats_info_evt_fixed_param;

typedef struct wmi_bpf_set_vdev_instructions_cmd_s {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 filter_id;
    A_UINT32 bpf_version;  /* host bpf version */
    A_UINT32 total_length;
    A_UINT32 current_offset;
    A_UINT32 current_length;
    /*
     * The TLV follows:
     *    A_UINT8  buf_inst[]; //Variable length buffer for the instuctions
     */
} wmi_bpf_set_vdev_instructions_cmd_fixed_param;

#define BPF_FILTER_ID_ALL  0xFFFFFFFF
typedef struct wmi_bpf_del_vdev_instructions_cmd_s {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 filter_id;  /* BPF_FILTER_ID_ALL means delete all */
} wmi_bpf_del_vdev_instructions_cmd_fixed_param;

typedef struct wmi_bpf_set_vdev_active_mode_cmd_s {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 mcbc_mode; /* refer to FW_ACTIVE_BPF_MODE */
    A_UINT32 uc_mode; /* refer to FW_ACTIVE_BPF_MODE */
} wmi_bpf_set_vdev_active_mode_cmd_fixed_param;

typedef struct wmi_bpf_set_vdev_enable_cmd_s {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 is_enabled; /* fw assume host default enables */
} wmi_bpf_set_vdev_enable_cmd_fixed_param;

typedef struct wmi_bpf_set_vdev_work_memory_cmd_s {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 bpf_version; /* bpf instruction version */
    A_UINT32 program_len; /* the program length may be changed by this command */
    A_UINT32 addr_offset; /* start writing addr in the working memory */
    A_UINT32 length;      /* the writing size of this command (byte units) */
/*
 * The TLV follows:
 *  A_UINT8 buf_inst[]; <-- Variable length buffer with the data to write
 */
} wmi_bpf_set_vdev_work_memory_cmd_fixed_param;

typedef struct wmi_bpf_get_vdev_work_memory_cmd_s {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 addr_offset; /* start reading addr in the working memory */
    A_UINT32 length;      /* reading size from addr (byte units) */
} wmi_bpf_get_vdev_work_memory_cmd_fixed_param;

typedef struct wmi_bpf_get_vdev_work_memory_resp_evt_s {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 offset;    /* read memory offset from start_addr */
    A_UINT32 length;    /* read memory length of this command */
    A_UINT32 fragment;  /* 1 means more data will come, 0 means last fragment */
/*
 * The TLV follows:
 *  A_UINT8 buf_inst[]; <-- Variable length buffer containing the read data
 */
} wmi_bpf_get_vdev_work_memory_resp_evt_fixed_param;

#define AES_BLOCK_LEN           16  /* in bytes */
#define FIPS_KEY_LENGTH_128     16  /* in bytes */
#define FIPS_KEY_LENGTH_256     32  /* in bytes */
#define FIPS_ENCRYPT_CMD        0
#define FIPS_DECRYPT_CMD        1
#define FIPS_ENGINE_AES_CTR     0
#define FIPS_ENGINE_AES_MIC     1
#define FIPS_ERROR_OPER_TIMEOUT 1

/* WMI_PDEV_FIPS_CMDID */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_fips_cmd_fixed_param */
    union {
        A_UINT32 mac_id; // OBSOLETE - will be removed once all refs are gone
        /** pdev_id for identifying the MAC
         * See macros starting with WMI_PDEV_ID_ for values.
         */
        A_UINT32 pdev_id;
    };
    A_UINT32 fips_cmd;            /* FIPS_ENCRYPT or FIPS_DECRYPT */
    A_UINT32 mode;                /* FIPS_ENGINE_AES_CTR or FIPS_ENGINE_AES_MIC */
    A_UINT32 key_len;             /* FIPS_KEY_LENGTH_128 or FIPS_KEY_LENGTH_256 (units = bytes) */
    A_UINT8 key[WMI_MAX_KEY_LEN]; /* Key */
    A_UINT32 data_len;            /* data length */
   /*
    * Following this structure is the TLV:
    * A_UINT32 data[1];               // In Data (keep this in the end)
    */
} wmi_pdev_fips_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_smart_ant_enable_cmd_fixed_param */
    union {
        A_UINT32 mac_id; // OBSOLETE - will be removed once all refs are gone
        /** pdev_id for identifying the MAC
         * See macros starting with WMI_PDEV_ID_ for values.
         */
        A_UINT32 pdev_id;
    };
    A_UINT32 enable;              /* 1:enable, 0:disable */
    A_UINT32 mode;                /* 1:GPIO parallel mode, 0:GPIO serial mode */
    A_UINT32 rx_antenna;          /* rx antenna */
    A_UINT32 tx_default_antenna;  /* tx default antenna */
   /*
    * Following this structure is the TLV:
    * wmi_pdev_smart_ant_gpio_handle
    */
} wmi_pdev_smart_ant_enable_cmd_fixed_param;

/** GPIO pins/function values to control antennas */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_smart_ant_gpio_handle */
    A_UINT32 gpio_pin;   /* For serial: index 0-strobe index 1-data, For Parallel: per stream */
    A_UINT32 gpio_func;  /* GPIO function values for Smart Antenna */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
} wmi_pdev_smart_ant_gpio_handle;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_smart_ant_set_rx_antenna_cmd_fixed_param */
    union {
        A_UINT32 mac_id; // OBSOLETE - will be removed once all refs are gone
        /** pdev_id for identifying the MAC
         * See macros starting with WMI_PDEV_ID_ for values.
         */
        A_UINT32 pdev_id;
    };
    A_UINT32 rx_antenna;
} wmi_pdev_smart_ant_set_rx_antenna_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_smart_ant_set_tx_antenna_cmd_fixed_param */
    /** unique id identifying the vdev, generated by the caller */
    A_UINT32 vdev_id; /* ID of the vdev this peer belongs to */
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
   /*
    * Following this structure is the TLV:
    * wmi_peer_smart_ant_set_tx_antenna_series
    */
} wmi_peer_smart_ant_set_tx_antenna_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_smart_ant_set_tx_antenna_series */
    /* antenna array */
    A_UINT32 antenna_series;
} wmi_peer_smart_ant_set_tx_antenna_series;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_smart_ant_set_train_antenna_param */
    /* rate array */
    A_UINT32  train_rate_series;
    /* antenna array */
    A_UINT32 train_antenna_series;
    /* Rate flags */
    /* TODO: For future use? */
    A_UINT32 rc_flags;
} wmi_peer_smart_ant_set_train_antenna_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_smart_ant_set_train_antenna_cmd_fixed_param */
    /** unique id identifying the VDEV, generated by the caller */
    A_UINT32 vdev_id; /* ID of the vdev this peer belongs to */
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /* num packets; 0-stop training */
    A_UINT32 num_pkts;
   /*
    * Following this structure is the TLV:
    * wmi_peer_smart_ant_set_train_antenna_param
    */
} wmi_peer_smart_ant_set_train_antenna_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_smart_ant_set_node_config_ops_cmd_fixed_param */
    /** unique id identifying the vdev, generated by the caller */
    A_UINT32 vdev_id; /* ID of the vdev this peer belongs to */
    /** peer MAC address */
    wmi_mac_addr peer_macaddr;
    /* command id*/
    A_UINT32 cmd_id;
    /* number of arguments passed */
    A_UINT32 args_count;
   /*
    * Following this structure is the TLV:
    * A_UINT32 args[];               // argument list
    */
} wmi_peer_smart_ant_set_node_config_ops_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_ant_ctrl_chain */
    A_UINT32 antCtrlChain;
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
} wmi_pdev_set_ant_ctrl_chain;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_ant_switch_tbl_cmd_fixed_param */
    A_UINT32 mac_id;     /* MAC ID */
    A_UINT32 antCtrlCommon1;
    A_UINT32 antCtrlCommon2;
   /*
    * Following this structure is the TLV:
    * wmi_pdev_set_ant_ctrl_chain
    */
} wmi_pdev_set_ant_switch_tbl_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_ctl_table_cmd_fixed_param */
    union {
        A_UINT32 mac_id; // OBSOLETE - will be removed once all refs are gone
        /** pdev_id for identifying the MAC
         * See macros starting with WMI_PDEV_ID_ for values.
         */
        A_UINT32 pdev_id;
    };
    /** len of CTL info */
    A_UINT32 ctl_len;
    /* ctl array (len adjusted to  number of words).
     * Following this structure is the TLV:
     * A_UINT32 ctl_info[1];
     */
} wmi_pdev_set_ctl_table_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_mimogain_table_cmd_fixed_param */
    union {
        A_UINT32 mac_id; // OBSOLETE - will be removed once all refs are gone
        /** pdev_id for identifying the MAC
         * See macros starting with WMI_PDEV_ID_ for values.
         */
        A_UINT32 pdev_id;
    };
    A_UINT32 mimogain_info; /* see WMI_MIMOGAIN macros */
    /*
     * Bit 7:0 len of array gain table
     * Bit 8   bypass multi chain gain or not
     */
    /* array gain table(s) (len adjusted to number of words).
     * Following this structure is the TLV:
     * A_UINT32 arraygain_tbl[1];
     */
} wmi_pdev_set_mimogain_table_cmd_fixed_param;

#define WMI_MIMOGAIN_ARRAY_GAIN_LEN_S 0
#define WMI_MIMOGAIN_ARRAY_GAIN_LEN (0xff << WMI_MIMOGAIN_ARRAY_GAIN_LEN_S)
#define WMI_MIMOGAIN_ARRAY_GAIN_LEN_GET(x) WMI_F_MS(x,WMI_MIMOGAIN_ARRAY_GAIN_LEN)
#define WMI_MIMOGAIN_ARRAY_GAIN_LEN_SET(x,z) WMI_F_RMW(x,z,WMI_MIMOGAIN_ARRAY_GAIN_LEN)

#define WMI_MIMOGAIN_MULTI_CHAIN_BYPASS_S 8
#define WMI_MIMOGAIN_MULTI_CHAIN_BYPASS (0x1 << WMI_MIMOGAIN_MULTI_CHAIN_BYPASS_S)
#define WMI_MIMOGAIN_MULTI_CHAIN_BYPASS_GET(x) WMI_F_MS(x,WMI_MIMOGAIN_MULTI_CHAIN_BYPASS)
#define WMI_MIMOGAIN_MULTI_CHAIN_BYPASS_SET(x,z) WMI_F_RMW(x,z,WMI_MIMOGAIN_MULTI_CHAIN_BYPASS)


typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_fwtest_set_param_cmd_fixed_param */
    /** parameter id   */
    A_UINT32 param_id;
    /** parameter value */
    A_UINT32 param_value;
} wmi_fwtest_set_param_cmd_fixed_param;

#define WMI_ATF_DENOMINATION   1000 /* Expressed in 1 part in 1000 (permille) */

#define WMI_ATF_SSID_FAIR_SCHED     0   /** Fair ATF scheduling for vdev */
#define WMI_ATF_SSID_STRICT_SCHED   1   /** Strict ATF scheduling for vdev */

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_atf_peer_info */
    A_UINT32 tlv_header;
    wmi_mac_addr peer_macaddr;
    A_UINT32 atf_units; /* Based on 1 part in 1000 (per mille) */
    A_UINT32 atf_groupid; /* Group Id of the peers for ATF SSID grouping */
    A_UINT32 atf_units_reserved; /* Peer congestion threshold for future use */
    A_UINT32 vdev_id;
    A_UINT32 pdev_id;
} wmi_atf_peer_info;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_atf_request_fixed_param */
    A_UINT32 num_peers;
    /*
     * Following this structure is the TLV:
     * struct wmi_atf_peer_info peer_info[num_peers];
     */
} wmi_peer_atf_request_fixed_param;

/* Structure for Bandwidth Fairness peer information */
typedef struct {
    /** TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_bwf_peer_info */
    A_UINT32 tlv_header;
    wmi_mac_addr peer_macaddr;
    A_UINT32 bwf_guaranteed_bandwidth; /* BWF guaranteed_bandwidth for the peers in mbps */
    A_UINT32 bwf_max_airtime; /* BWF Maximum airtime percentage that can be allocated to the peer to meet the guaranteed_bandwidth */
    A_UINT32 bwf_peer_priority; /* BWF priority of the peer to allocate the tokens dynamically */
    A_UINT32 vdev_id;
    A_UINT32 pdev_id;
} wmi_bwf_peer_info;

/* Structure for Bandwidth Fairness peer request */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_bwf_request_fixed_param */
    A_UINT32 num_peers;
    /*
     * Following this structure is the TLV:
     * struct wmi_bwf_peer_info peer_info[num_peers];
     */
} wmi_peer_bwf_request_fixed_param;

/* Equal distribution of ATF air time within an VDEV. */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_atf_request_fixed_param */
    A_UINT32 vdev_id;
    A_UINT32 peer_atf_units; /* Per peer ATF units (per mille). */
    A_UINT32 pdev_id;
} wmi_vdev_atf_request_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_get_ani_cck_config_cmd_fixed_param */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
    /** parameter   */
    A_UINT32 param;
} wmi_pdev_get_ani_cck_config_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_get_ani_ofdm_config_cmd_fixed_param */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
    /** parameter   */
    A_UINT32 param;
} wmi_pdev_get_ani_ofdm_config_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_QBOOST_CFG_CMD_fixed_param */
    A_UINT32 vdev_id; /* ID of the vdev this peer belongs to */
    A_UINT32 qb_enable;
    wmi_mac_addr peer_macaddr;
} WMI_QBOOST_CFG_CMD_fixed_param;

#define WMI_INST_STATS_INVALID_RSSI 0

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_inst_rssi_stats_resp_fixed_param */
    A_UINT32 iRSSI; /* dBm above the noise floor */
    /* peer MAC address */
    wmi_mac_addr peer_macaddr;
    A_UINT32 vdev_id; /* ID of the vdev this peer belongs to */
} wmi_inst_rssi_stats_resp_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_cck_ofdm_rate_info */
    A_UINT32 ratecode_legacy; /* Rate code for CCK OFDM */
} wmi_peer_cck_ofdm_rate_info;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_mcs_rate_info */
    A_UINT32 ratecode_20; /* Rate code for 20MHz BW */
    A_UINT32 ratecode_40; /* Rate code for 40MHz BW */
    A_UINT32 ratecode_80; /* Rate code for 80MHz BW */
} wmi_peer_mcs_rate_info;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_ratecode_list_event_fixed_param */
    wmi_mac_addr peer_macaddr;
    A_UINT32 ratecount; /* Max Rate count for each mode */
    A_UINT32 vdev_id; /* ID of the vdev this peer belongs to */
    /*
     * Following this structure are the TLV:
     * struct wmi_peer_cck_ofdm_rate_info;
     * struct wmi_peer_mcs_rate_info;
     */
} wmi_peer_ratecode_list_event_fixed_param;

typedef struct wmi_wds_addr_event {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_wds_addr_event_fixed_param */
    A_UINT32 event_type[4];
    wmi_mac_addr peer_mac;
    wmi_mac_addr dest_mac;
    A_UINT32 vdev_id; /* ID of the vdev this peer belongs to */
} wmi_wds_addr_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_peer_sta_ps_statechange_event_fixed_param */
    wmi_mac_addr peer_macaddr;
    A_UINT32 peer_ps_state;
} wmi_peer_sta_ps_statechange_event_fixed_param;

/* WMI_PDEV_FIPS_EVENTID */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_fips_event_fixed_param */
    union {
        A_UINT32 mac_id; // OBSOLETE - will be removed once all refs are gone
        /** pdev_id for identifying the MAC
         * See macros starting with WMI_PDEV_ID_ for values.
         */
        A_UINT32 pdev_id;
    };
    A_UINT32 error_status; /* Error status: 0 (no err), 1, or OPER_TIMEOUT */
    A_UINT32 data_len;     /* Data length */
   /*
    * Following this structure is the TLV:
    *    A_UINT32 data[1];               // Out Data (keep this in the end)
    */
} wmi_pdev_fips_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_channel_hopping_event_fixed_param */
    union {
        A_UINT32 mac_id;     /* MAC ID */
        /** pdev_id for identifying the MAC
         * See macros starting with WMI_PDEV_ID_ for values.
         */
        A_UINT32 pdev_id;
    };
    /* Noise threshold iterations with high values */
    A_UINT32 noise_floor_report_iter;
    /* Total noise threshold iterations */
    A_UINT32 noise_floor_total_iter;
} wmi_pdev_channel_hopping_event_fixed_param;

enum {
    WMI_PDEV_RESERVE_AST_ENTRY_OK,
    WMI_PDEV_RESERVE_AST_ENTRY_HASH_COLLISION,
    WMI_PDEV_RESERVE_AST_ENTRY_CROSSING_AXI_BOUNDARY,
};

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_get_tpc_cmd_fixed_param */
    A_UINT32 mac_id;     /* MAC ID */
    A_UINT32 rate_flags;
    /**
    * FLAG_ONE_CHAIN     0x00000001  - one chain mask
    * FLAG_TWO_CHAIN     0x00000003  - two chain mask
    * FLAG_THREE_CHAIN   0x00000007  - three chain mask
    * FLAG_FOUR_CHAIN    0x0000000F  - four chain mask
    * FLAG_FIVE_CHAIN    0x0000001F  - five chain mask
    * FLAG_SIX_CHAIN     0x0000003F  - six chain mask
    * FLAG_SEVEN_CHAIN   0x0000007F  - seven chain mask
    * FLAG_EIGHT_CHAIN   0x000000FF  - eight chain mask
    * FLAG_STBC          0x00000100  - STBC is set
    * FLAG_40MHZ         0x00000200  - bits 9-10 used for BW:
    * FLAG_80MHZ         0x00000400    (bw >> 9) & 3 will give
    * FLAG_160MHZ        0x00000600    1 -> 40 MHz, 2 -> 80 MHz, 3 -> 160 MHz
    * FLAG_TXBF          0x00000800 - Tx Bf enabled
    * FLAG_RTSENA        0x00001000 - RTS enabled
    * FLAG_CTSENA        0x00002000 - CTS enabled
    * FLAG_LDPC          0x00004000 - LDPC set
    * FLAG_SGI           0x00008000 - Short gaurd interval
    *                   (0x00010000-0x00080000 unused)
    *------------------
    * 0x00100000-0x00700000 used for SU/MU/OFDMA tx mode
    * FLAG_SU            0x00100000 - SU Data
    * FLAG_DL_MU_MIMO_AC 0x00200000 - DL AC MU data
    * FLAG_DL_MU_MIMO_AX 0x00300000 - DL AX MU data
    * FLAG_DL_OFDMA      0x00400000 - DL OFDMA data
    * FLAG_UL_OFDMA      0x00500000 - UL OFDMA data
    * FLAG_UL_MU_MIMO    0x00600000 - UL MU data
    *------------------
    * */
    A_UINT32 nss;
    /**
    * NSS 0x0 - 0x7
    * */
    A_UINT32 preamble;
    /**
    * PREAM_OFDM - 0x0
    * PREAM_CCK  - 0x1
    * PREAM_HT   - 0x2
    * PREAM_VHT  - 0x3
    * PREAM_HE   - 0x4
    * */
    A_UINT32 hw_rate;
    /**
     * *** HW_OFDM_RATE ***
     * OFDM_48_MBPS - 0x0
     * OFDM_24_MBPS - 0x1
     * OFDM_12_MBPS - 0x2
     * OFDM_6_MBPS  - 0x3
     * OFDM_54_MBPS - 0x4
     * OFDM_36_MBPS - 0x5
     * OFDM_18_MBPS - 0x6
     * OFDM_9_MBPS  - 0x7
     *
     * *** HW_CCK_RATE ***
     * CCK_11_LONG_MBPS   - 0x0
     * CCK_5_5_LONG_MBPS  - 0x1
     * CCK_2_LONG_MBPS    - 0x2
     * CCK_1_LONG_MBPS    - 0x3
     * CCK_11_SHORT_MBPS  - 0x4
     * CCK_5_5_SHORT_MBPS - 0x5
     * CCK_2_SHORT_MBPS   - 0x6
     *
     * *** HW_HT / VHT_RATE ***
     * MCS 0x0 - 0xb
     * */
} wmi_pdev_get_tpc_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals
                            WMITLV_TAG_STRUC_wmi_pdev_get_chip_power_stats_cmd_fixed_param */
    /**
     * pdev_id for identifying the MAC See macros
     * starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
} wmi_pdev_get_chip_power_stats_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_get_bcn_recv_stats_cmd_fixed_param */
    /** VDEV identifier */
    A_UINT32 vdev_id;
} wmi_vdev_get_bcn_recv_stats_cmd_fixed_param;

/*
 * wmi mws-coex command IDs
 */
typedef enum {
    WMI_MWS_COEX_STATE = 0x01,
    WMI_MWS_COEX_DPWB_STATE,
    WMI_MWS_COEX_TDM_STATE,
    WMI_MWS_COEX_IDRX_STATE,
    WMI_MWS_COEX_ANTENNA_SHARING_STATE,
} wmi_mws_coex_cmd_id;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_get_mws_coex_state_cmd_fixed_param */
    /** VDEV identifier */
    A_UINT32 vdev_id;
    /** Command ID (type: wmi_mws_coex_cmd_id) */
    A_UINT32 cmd_id;
} wmi_vdev_get_mws_coex_info_cmd_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_tpc_event_fixed_param */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
   /*
    * Following this structure is the TLV:
    *    A_UINT32 tpc[2];
    *    tpc[0] - maximum power per rate
    *    tpc[1] - minimum power per rate
    *    Currently this event only provides max and min power limits
    *    for a single rate.  In the future this event may be expanded
    *    to provide the information for multiple rates.
    *    At that time, the format of the data will be provided.
    */
} wmi_pdev_tpc_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_nfcal_power_all_channels_event_fixed_param */
    union {
        A_UINT32 mac_id;     /* MAC ID */
        /** pdev_id for identifying the MAC
         * See macros starting with WMI_PDEV_ID_ for values.
         */
        A_UINT32 pdev_id;
    };
    A_UINT32 nfdBr_len;
    A_UINT32 nfdBm_len;
    A_UINT32 freqNum_len;
   /*
    * Following this structure is the TLV:
    * WMITLV_TAG_STRUC_wmi_pdev_nfcal_power_all_channels_nfdBr;
    * WMITLV_TAG_STRUC_wmi_pdev_nfcal_power_all_channels_nfdBm;
    * WMITLV_TAG_STRUC_wmi_pdev_nfcal_power_all_channels_freqNum;
    */
} wmi_pdev_nfcal_power_all_channels_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_nfcal_power_all_channels_nfdBr */
    A_UINT32 nfdBr;
} wmi_pdev_nfcal_power_all_channels_nfdBr;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_nfcal_power_all_channels_nfdBm */
    A_UINT32 nfdBm;
} wmi_pdev_nfcal_power_all_channels_nfdBm;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_nfcal_power_all_channels_freqNum */
    A_UINT32 freqNum;
} wmi_pdev_nfcal_power_all_channels_freqNum;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ani_cck_event_fixed_param */
    A_UINT32 cck_level;
} wmi_ani_cck_event_fixed_param;

typedef enum wmi_power_debug_reg_fmt_type {
    /* WMI_POWER_DEBUG_REG_FMT_TYPE_ROME -> Dumps following 12 Registers
     *     SOC_SYSTEM_SLEEP
     *     WLAN_SYSTEM_SLEEP
     *     RTC_SYNC_FORCE_WAKE
     *     MAC_DMA_ISR
     *     MAC_DMA_TXRX_ISR
     *     MAC_DMA_ISR_S1
     *     MAC_DMA_ISR_S2
     *     MAC_DMA_ISR_S3
     *     MAC_DMA_ISR_S4
     *     MAC_DMA_ISR_S5
     *     MAC_DMA_ISR_S6
     *     MAC_DMA_ISR_S7
     */
    WMI_POWER_DEBUG_REG_FMT_TYPE_ROME,
    WMI_POWER_DEBUG_REG_FMT_TYPE_MAX = 0xf,
} WMI_POWER_DEBUG_REG_FMT_TYPE;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals
    WMITLV_TAG_STRUC_wmi_chip_power_stats_event_fixed_param */
    A_UINT32 cumulative_sleep_time_ms; /* maximum range is 35 hours, due to conversion from internal 0.03215 ms units to ms */
    A_UINT32 cumulative_total_on_time_ms; /* maximum range is 35 hours, due to conversion from internal 0.03215 ms units to ms */
    A_UINT32 deep_sleep_enter_counter; /* count of number of times chip enterred deep sleep */
    A_UINT32 last_deep_sleep_enter_tstamp_ms; /* Last Timestamp when Chip went to deep sleep */
    A_UINT32 debug_register_fmt; /* WMI_POWER_DEBUG_REG_FMT_TYPE enum, describes debug registers being dumped as part of the event */
    A_UINT32 num_debug_register; /* number of debug registers being sent to host */
    /*
     * Following this structure is the TLV:
     * A_UINT32 debug_registers[num_debug_registers];
     */
} wmi_pdev_chip_power_stats_event_fixed_param;

typedef struct {
    A_UINT32 tlv_header;  /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_get_bcn_recv_stats_fixed_param */
    A_UINT32 vdev_id;
    /* total_bcn_cnt
     * total beacon count (tbtt instances)
     * After this value reaches 255 it saturates and stays at 255.
     * This field is used to determine which of the 256 bits in the
     * bmiss_bitmap field are valid.
     */
    A_UINT32 total_bcn_cnt;
    /* total_bmiss_cnt
     * Total beacon miss count in last 255 beacons, max value is 255.
     * This value is the number of bits set within bmiss_bitmap.
     */
    A_UINT32 total_bmiss_cnt;
    /* bmiss_bitmap
     * This bitmap indicates the status of the last 255 beacons.
     * If a bit is set, that means the corresponding beacon was missed.
     * Bit 0 of bmiss_bitmap[0] represents the most recent beacon.
     * The total_bcn_cnt field indicates how many bits within bmiss_bitmap
     * are valid.
     */
    A_UINT32 bmiss_bitmap[8];
} wmi_vdev_bcn_recv_stats_fixed_param;

typedef struct {
    A_UINT32 tlv_header;  /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_get_mws_coex_state_fixed_param */
    A_UINT32 vdev_id;

    /* LTE-WLAN coexistence scheme bitmap
     * Indicates the final schemes applied for the currrent Coex scenario.
     * Bit 0 - TDM policy
     * Bit 1 - Forced TDM policy
     * Bit 2 - Dynamic Power Back-off policy
     * Bit 3 - Channel Avoidance policy
     * Bit 4 - Static Power Back-off policy
     */
    A_UINT32 coex_scheme_bitmap;

    /* Active conflict count
     * Indicates the number of Active conflicts for the current WLAN and LTE frequency combinations.
     */
    A_UINT32 active_conflict_count;

    /* Potential conflict count
     * Indicates the number of Potential conflicts for the current WLAN and LTE frequency combinations.
     */
    A_UINT32 potential_conflict_count;

    /* Bitmap of the group-0 WLAN channels to be avoided during LTE-WLAN coex operation.
     * Indicates the WLAN channels to be avoided in b/w WLAN CH-1 and WLAN CH-14.
     */
    A_UINT32 chavd_group0_bitmap;

    /* Bitmap of the group-1 WLAN channels to be avoided during LTE-WLAN coex operation.
     * Indicates the WLAN channels to be avoided in b/w WLAN CH-36 and WLAN CH-64.
     */
    A_UINT32 chavd_group1_bitmap;

    /* Bitmap of the group-2 WLAN channels to be avoided during LTE-WLAN coex operation.
     * Indicates the WLAN channels to be avoided in b/w WLAN CH-100 and WLAN CH-140.
     */
    A_UINT32 chavd_group2_bitmap;

    /* Bitmap of the group-3 WLAN channels to be avoided during LTE-WLAN coex operation.
     * Indicates the WLAN channels to be avoided in b/w WLAN CH-149 and WLAN CH-165.
     */
    A_UINT32 chavd_group3_bitmap;
} wmi_vdev_get_mws_coex_state_fixed_param;

typedef struct {
    A_UINT32 tlv_header;  /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_get_mws_coex_dpwb_state_fixed_param */
    A_UINT32 vdev_id;

    /* Current state of the Dynamic Power Back-off state machine
     * MWSCOEX_PWB_UNINIT_STATE                        = 0, PWB state machine is in un-intialized state.
     * MWSCOEX_PWB_WLAN_ON_SINR_START_STATE            = 1, SINR measurement starts when WLAN is on
     * MWSCOEX_PWB_WLAN_ON_WAIT_RESP_STATE             = 2, Waiting for SINR response when WLAN is on
     * MWSCOEX_PWB_WLAN_OFF_AWAIT_STATE                = 3, WLAN off state for buffer between SINR on/off measurement.
     * MWSCOEX_PWB_WLAN_OFF_SINR_START_STATE           = 4, SINR measurement starts when WLAN is off
     * MWSCOEX_PWB_WLAN_OFF_WAIT_RESP_STATE            = 5, Waiting for SINR response when WLAN is off
     * MWSCOEX_PWB_WLAN_OFF_SINR_STOP_STATE            = 6, SINR measurement stops when WLAN is off
     * MWSCOEX_PWB_FORCED_TDM_STATE                    = 7, Entered Forced TDM state.
     * MWSCOEX_PWB_HALTED_STATE                        = 8, Power back-off algorithm halted.
     * MWSCOEX_PWB_WLAN_ALWAYS_ON_SINR_START_STATE     = 9, SINR measurement starts in WLAN always on state.
     * MWSCOEX_PWB_WLAN_ALWAYS_ON_SINR_STOP_STATE      = 10, SINR measurement stops in WLAN always on state.
     */
    A_UINT32 current_dpwb_state;

    /* P(N+1) value in dBm i.e. Tx power to be applied in the next Dynamic Power Back-off cycle,
     * where P(N) is the power applied during current cycle.
     * ranges from 3dBm to 21 dBM
     */
    A_INT32 pnp1_value;

    /* Indicates the duty cycle of current LTE frame.
     * Duty cycle: Number of UL slots with uplink data and allocated RBs.
     */
    A_UINT32 lte_dutycycle;

    /* LTE SINR value in dB, when WLAN is ON. */
    A_INT32 sinr_wlan_on;

    /* LTE SINR value in dB, when WLAN is OFF. */
    A_INT32 sinr_wlan_off;

    /* LTE blocks with error for the current bler report.
     * Number of LTE blocks with error for a given number (block_count)  of LTE blocks.
     */
    A_UINT32 bler_count;

    /* Number of LTE blocks considered for bler count report.
     * Bler repot will be generated after the reception of every "block_count" number of blocks.
     */
    A_UINT32 block_count;

    /* WLAN RSSI level
     * WLAN RSSI is devided in to 3 levels i.e. Good/Moderate/Low (configurable inside f/w)
     * 0-Good, 1-Moderate, 2-Low
     */
    A_UINT32 wlan_rssi_level;

    /* WLAN RSSI value in dBm considered in Dynamic Power back-off algorithm
     * Dynamic power back-off algorithm considers either Rx data frame RSSI/Beacon RSSI based on some constraints.
     */
    A_INT32 wlan_rssi;

    /* Indicates whether any TDM policy triggered from Dynamic power back-off policy.
     * 1 - TDM triggered.
     * 0 - TDM not triggered.
     */
    A_UINT32 is_tdm_running;
} wmi_vdev_get_mws_coex_dpwb_state_fixed_param;

typedef struct {
    A_UINT32 tlv_header;  /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_get_mws_coex_tdm_state_fixed_param */
    A_UINT32 vdev_id;
    /* Time Division Multiplexing (TDM) LTE-Coex Policy type.
     * There are totally 4 types of TDM policies(1-SINR TDM, 2-RSSI TDM, 3-LOW RX RATE TDM, 4-STICKY TDM)
     * Bit 0 - SINR TDM policy.
     * Bit 1 - RSSI TDM policy.
     * Bit 2 - Low Rx rate TDM policy
     * Bit 3 - Sticky TDM policy
     */
    A_UINT32 tdm_policy_bitmap;

    /* TDM LTE/WLAN sub-frame bitmap
     * Indicates the bitmap of LTE/WLAN sub-frames.
     * value 0: WLAN slot.
     * value 1: LTE slot.
     */
    A_UINT32 tdm_sf_bitmap;
} wmi_vdev_get_mws_coex_tdm_state_fixed_param;

typedef struct {
    A_UINT32 tlv_header;  /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_get_mws_coex_idrx_state_fixed_param */
    A_UINT32 vdev_id;

    /* SUB0 LTE-coex tech.
     *------------------------
     * TECH          TECH_ID
     *------------------------
     * LTE              0
     * TDSCDMA          1
     * GSM1             2
     * ONEX             3
     * HDR              4
     * WCDMA            5
     * GSM2             6
     * GSM3             7
     * WCDMA2           8
     * LTE2             9
     * Indicates the type of WWAN technology used as SUB0 i.e. SIM slot 1
     */
    A_UINT32 sub0_techid;

    /* SUB0 mitigation policy.
     * Indicates the mitigation policy used to coexist with WLAN.
     * 1 - Tx blanking
     * 2 - Static power back-off
     */
    A_UINT32 sub0_policy;

    /* Set if SUB0 is in link critical state.
     * Link critical will be set, if continuous page miss happens or RSSI is below -100 dBm at LTE side.
     */
    A_UINT32 sub0_is_link_critical;

    /* LTE SUB0 imposed static power applied to WLAN due to LTE-WLAN coex.
     * Value of static power applied during LTE page cycle ranges from 3-21 dBm.
     */
    A_INT32 sub0_static_power;

    /* LTE SUB0 RSSI value in dBm */
    A_INT32 sub0_rssi;

    /* SUB1 LTE-coex tech.
     *------------------------
     * TECH          TECH_ID
     *------------------------
     * LTE              0
     * TDSCDMA          1
     * GSM1             2
     * ONEX             3
     * HDR              4
     * WCDMA            5
     * GSM2             6
     * GSM3             7
     * WCDMA2           8
     * LTE2             9
     * Indicates the type of WWAN technology used as SUB1 i.e. SIM slot 2
     */
    A_UINT32 sub1_techid;

    /* SUB1 mitigation policy.
     * Indicates the mitigation policy used to coexist with WLAN.
     * 1 - Tx blanking
     * 2 - Static power back-off
     */
    A_UINT32 sub1_policy;

    /* Set if SUB1 is in link critical state.
     * Link critical will be set, if continuous page miss happens or RSSI is below -100 dBm at LTE side.
     */
    A_UINT32 sub1_is_link_critical;

    /* LTE SUB1 imposed static power applied to WLAN due to LTE-WLAN coex.
     * Value of static power applied during LTE page cycle ranges from 3-21 dBm.
     */
    A_INT32 sub1_static_power;

    /* LTE SUB1 RSSI value in dBm */
    A_INT32 sub1_rssi;
} wmi_vdev_get_mws_coex_idrx_state_fixed_param;

typedef struct {
    A_UINT32 tlv_header;  /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_vdev_get_mws_coex_antenna_sharing_state_fixed_param */
    A_UINT32 vdev_id;

    /* BDF values of Coex flags.
     * coexflag 0x1 = MWS Coex enabled
     * coexflag 0x3 = MWS Coex enabled + Antenna sharing enabled for WLAN operating in 2.4GHz band.
     */
    A_UINT32 coex_flags;

    /* BDF values of Coex Antenna sharing config
     * coex_config 0x0 = no Antenna sharing
     * coexconfig 0x1 = switched based Antenna sharing
     * coexconfig 0x2 = splitter based Antenna sharing
     */
    A_UINT32 coex_config;

    /* Tx Chain mask value
     * Bit 0: Tx chain-0
     * Bit 1: Tx Chain-1
     * value: 0x1 - Operating in 1X1
     * value: 0x3 - Operating in 2X2
     */
    A_UINT32 tx_chain_mask;

    /* Rx Chain mask value
     * Bit 0: Rx chain-0
     * Bit 1: Rx Chain-1
     * value: 0x1 - Operating in 1X1
     * value: 0x3 - Operating in 2X2
     */
    A_UINT32 rx_chain_mask;

    /* Currently active Rx Spatial streams
     * Bit 0: Rx Spatial Stream-0
     * Bit 1: Rx Spatial Stream-1
     */
    A_UINT32 rx_nss;

    /* Forced MRC policy type
     * BTC_FORCED              (0x01)
     * RSSI_FORCED             (0x02)
     * MODEM_ACQ_FORCED        (0x04)
     */
    A_UINT32 force_mrc;

    /* RSSI value considered for MRC
     * 1: Data RSSI
     * 2: Beacon RSSI
     */
    A_UINT32 rssi_type;

    /* RSSI value measured at Chain-0 in dBm */
    A_INT32 chain0_rssi;

    /* RSSI value measured at Chain-1 in dBm */
    A_INT32 chain1_rssi;

    /* RSSI value of two chains combined in dBm */
    A_INT32 combined_rssi;

    /* Absolute imbalance between two Rx chains in dB */
    A_UINT32 imbalance;

    /* RSSI threshold defined for the above imbalance value in dBm.
     * Based on the degree of imbalance between the rx chains, different
     * RSSI thresholds are used to determine whether MRC (Maximum-Ratio
     * Combining) use of multiple rx chains is suitable.
     * This field shows the RSSI threshold below which MRC is used.
     */
    A_INT32 mrc_threshold;

    /* Antenna grant duration to WLAN, in milliseconds */
    A_UINT32 grant_duration;
} wmi_vdev_get_mws_coex_antenna_sharing_state_fixed_param;

typedef enum wmi_chip_power_save_failure_reason_code_type {
    WMI_PROTOCOL_POWER_SAVE_FAILURE_REASON,
    WMI_HW_POWER_SAVE_FAILURE_REASON,
    WMI_CSS_LOCKED_POWER_FAILURE_REASON,
    WMI_MAC0_LOCKED_POWER_FAILURE_REASON,
    WMI_MAC1_LOCKED_POWER_FAILURE_REASON,
    WMI_POWER_SAVE_FAILURE_REASON_MAX = 0xf,
} WMI_POWER_SAVE_FAILURE_REASON_TYPE;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_chip_power_save_failure_detected_fixed_param */
    A_UINT32 power_save_failure_reason_code; /* Chip power save failuire reason as defined in WMI_POWER_SAVE_FAILURE_REASON_TYPE */
    A_UINT32 protocol_wake_lock_bitmap[4]; /* bitmap with bits set for modules (from WLAN_MODULE_ID enum) voting against sleep for prolonged duration */
} wmi_chip_power_save_failure_detected_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_ani_ofdm_event_fixed_param */
    A_UINT32 ofdm_level;
} wmi_ani_ofdm_event_fixed_param;

/* When a bit is set it specifies the particular WLAN traffic type is high priority.
 * BT low priority traffic has higher priority than WLAN low priority traffic and has
 * lower priority when compared to WLAN high priority traffic.
 * BT high priority traffic has higher priority than WLAN low/high priority traffic.
 */
#define WMI_PDEV_BE_PRIORITY_BIT     (1<<0)
#define WMI_PDEV_BK_PRIORITY_BIT     (1<<1)
#define WMI_PDEV_VI_PRIORITY_BIT     (1<<2)
#define WMI_PDEV_VO_PRIORITY_BIT     (1<<3)
#define WMI_PDEV_BEACON_PRIORITY_BIT (1<<4)
#define WMI_PDEV_MGMT_PRIORITY_BIT   (1<<5)
#define WMI_PDEV_IS_BE_PRIORITY_SET(val) ((val) & WMI_PDEV_BE_PRIORITY_BIT)
#define WMI_PDEV_IS_BK_PRIORITY_SET(val) ((val) & WMI_PDEV_BK_PRIORITY_BIT)
#define WMI_PDEV_IS_VI_PRIORITY_SET(val) ((val) & WMI_PDEV_VI_PRIORITY_BIT)
#define WMI_PDEV_IS_VO_PRIORITY_SET(val) ((val) & WMI_PDEV_VO_PRIORITY_BIT)
#define WMI_PDEV_IS_BEACON_PRIORITY_SET(val) ((val) & WMI_PDEV_BEACON_PRIORITY_BIT)
#define WMI_PDEV_IS_MGMT_PRIORITY_SET(val) ((val) & WMI_PDEV_MGMT_PRIORITY_BIT)

typedef enum wmi_coex_algo_type  {
    WMI_COEX_ALGO_UNCONS_FREERUN  = 0,
    WMI_COEX_ALGO_FREERUN         = 1,
    WMI_COEX_ALGO_OCS             = 2,
} WMI_COEX_ALGO_TYPE;

typedef enum wmi_coex_config_type {
    /* config interval (arg1 BT, arg2 WLAN) for P2P + PAGE */
    WMI_COEX_CONFIG_PAGE_P2P_TDM        =  1,
    /* config interval (arg1 BT, arg2 WLAN) for STA + PAGE */
    WMI_COEX_CONFIG_PAGE_STA_TDM        =  2,
    /* config interval (arg1 BT, arg2 WLAN) for SAP + PAGE */
    WMI_COEX_CONFIG_PAGE_SAP_TDM        =  3,
    /* config during WLAN connection */
    WMI_COEX_CONFIG_DURING_WLAN_CONN    =  4,
    /* config to enable/disable BTC */
    WMI_COEX_CONFIG_BTC_ENABLE          =  5,
    /* config of COEX debug setting */
    WMI_COEX_CONFIG_COEX_DBG            =  6,
    /* config interval (ms units) (arg1 BT, arg2 WLAN) for P2P + STA + PAGE */
    WMI_COEX_CONFIG_PAGE_P2P_STA_TDM    =  7,
    /* config interval (ms units) (arg1 BT, arg2 WLAN) for P2P + INQUIRY */
    WMI_COEX_CONFIG_INQUIRY_P2P_TDM     =  8,
    /* config interval (ms units) (arg1 BT, arg2 WLAN) for STA + INQUIRY */
    WMI_COEX_CONFIG_INQUIRY_STA_TDM     =  9,
    /* config interval (ms units) (arg1 BT, arg2 WLAN) for SAP + INQUIRY */
    WMI_COEX_CONFIG_INQUIRY_SAP_TDM     = 10,
    /*
     * config interval (ms units) (arg1 BT, arg2 WLAN) for P2P + STA +
     * INQUIRY
     */
    WMI_COEX_CONFIG_INQUIRY_P2P_STA_TDM = 11,
    /* config wlan total tx power when bt coex (arg1 is wlan_tx_power_limit, in 0.5dbm units) */
    WMI_COEX_CONFIG_TX_POWER            = 12,
    /* config  whether enable PTA and GPIO number (arg1 is pta_enable, arg2 is GPIO number used as /BT_ACTIVE/BT_PRIORITY/WLAN_DENY,8 bit for each) */
    WMI_COEX_CONFIG_PTA_CONFIG          = 13,
    WMI_COEX_CONFIG_AP_TDM              = 14, /* config interval (arg1 duty cycle in ms, arg2 WLAN duration in ms) for AP */
    WMI_COEX_CONFIG_WLAN_SCAN_PRIORITY  = 15, /* config to set WLAN priority during Off Channel Scan */
    WMI_COEX_CONFIG_WLAN_PKT_PRIORITY   = 16, /* config to set WLAN priority for BE/BK/VO/VI/Beacon/Management frame */
    WMI_COEX_CONFIG_PTA_INTERFACE       = 17, /* config PTA interface,
                                                 arg1 PTA num,
                                                 arg2 mode (2-wire/3-wire/PTA),
                                                 arg3 first slot time in microsec,
                                                 arg4 BT priority time in microsec,
                                                 arg5 PTA algorithm (WMI_COEX_ALGO_TYPE),
                                                 arg6 PTA priority */
    WMI_COEX_CONFIG_BTC_DUTYCYCLE       = 18, /* config interval (ms units) (arg1 WLAN pause duration, arg2 WLAN unpause duration) for WLAN UL + BT Rx */
    WMI_COEX_CONFIG_HANDOVER_RSSI       = 19, /* config to set WLAN RSSI (dBm units) at which device switches from Hybrid to TDD coex mode */
#ifdef FEATURE_COEX_PTA_CONFIG_ENABLE
    /* config BT info to FW when coexistence enable */
    WMI_COEX_CONFIG_PTA_BT_INFO         = 20,
#endif
    WMI_COEX_CONFIG_SINK_WLAN_TDM       = 21, /* config interval (ms units) (arg1 BT, arg2 WLAN) for A2DP SINK + WLAN */
    WMI_COEX_CONFIG_COEX_ENABLE_MCC_TDM = 22, /* config disable/enable COEX TDM for MCC */
    WMI_COEX_CONFIG_LOWRSSI_A2DPOPP_TDM = 23, /* config interval (ms units) (arg1 BT, arg2 WLAN) for STA + A2dp + OPP + LOWRSSI */
    WMI_COEX_CONFIG_BTC_MODE            = 24, /* config BTC mode, arg1 mode: 0 TDD/1 FDD/2 Hybrid*/
    WMI_COEX_CONFIG_ANTENNA_ISOLATION   = 25, /* config isolation between BT and WLAN chains
                                               * The arguments are interpreted differently
                                               * depending on whether the target suppports
                                               * WMI_SERVICE_COEX_SUPPORT_UNEQUAL_ISOLATION
                                               * If (not COEX_SUPPORT_UNEQUAL_ISOLATION) or arg2 == 0:
                                               *     arg1 => isolation between BT and WLAN chains,
                                               *             dB units,
                                               *             same isolation for all chains
                                               * Else:
                                               *     arg1 bits  7:0  - chain 0 isolation, in dB
                                               *     arg1 bits 15:8  - chain 1 isolation, in dB
                                               *     arg1 bits 23:16 - chain 2 isolation, in dB
                                               *     arg1 bits 31:24 - chain 3 isolation, in dB
                                               * arg2 - 0 => Equal isolation b/w BT and each WLAN chain (default)
                                               *        1 => Different isolation b/w BT and each WLAN chain
                                               */
    WMI_COEX_CONFIG_BT_LOW_RSSI_THRESHOLD = 26,/*config BT low rssi threshold (dbm units)*/
    WMI_COEX_CONFIG_BT_INTERFERENCE_LEVEL = 27,/*config bt interference level (dbm units)
                                                 arg1 low - lower limit
                                                 arg2 low - upper limit
                                                 arg3 medium - lower limit
                                                 arg4 medium - upper limit
                                                 arg5 high - lower limit
                                                 arg6 high - upper limit */
    WMI_COEX_CONFIG_WLAN_OVER_ZBLOW        = 28, /* config to boost WiFi traffic over 15.4 Low prio traffic */
    WMI_COEX_CONFIG_WLAN_MGMT_OVER_BT_A2DP = 29, /* config to raise WLAN priority higher than BT in coex scenario of SAP + BT or 15.4 */
    WMI_COEX_CONFIG_WLAN_CONN_OVER_LE      = 30, /* config to elevate Wifi priority over BLE during WLAN association */
    WMI_COEX_CONFIG_LE_OVER_WLAN_TRAFFIC   = 31, /* config to elevate BLE traffic over WiFi traffic */
    WMI_COEX_CONFIG_THREE_WAY_COEX_RESET   = 32, /* config to reset the weights to default  */
    /* WMI_COEX_CONFIG_THREE_WAY_DELAY_PARA
     * config to T_PRIO T_DELAY parameter for each case
     *   arg1 - wlan/bt state
     *       0: beacon tx
     *       1: wlan connecting
     *       2: wlan in dhcp
     *       3: a2dp critical
     *       4: eSCO
     *   arg2 - t_prio for low priority traffic (microsecond units)
     *   arg3 - t_delay for low priority traffic (microsecond units)
     *   arg4 - t_prio for high priority traffic (microsecond units)
     *   arg5 - t_delay for high priority traffic (microsecond units)
     */
    WMI_COEX_CONFIG_THREE_WAY_DELAY_PARA   = 33,
    /* WMI_COEX_CONFIG_THREE_WAY_COEX_START
     * config to set coex parameters from WLAN host to adjust priorities
     * among wlan/bt/zb
     * arg1 - priority level 1, the serialized coex scenorio ID will be put here
     * arg2 - priority level 2, same parameters rules as arg1
     * arg3 - priority level 3, same parameters rules as arg1
     * arg4 - priority level 4, same parameters rules as arg1
     */
    WMI_COEX_CONFIG_THREE_WAY_COEX_START   = 34,
    /* WMI_COEX_CONFIG_MPTA_HELPER_ENABLE
     * config to enable(1)/disable(0) mpta-helper function
     */
    WMI_COEX_CONFIG_MPTA_HELPER_ENABLE     = 35,
    /* WMI_COEX_CONFIG_MPTA_HELPER_ZIGBEE_STATE
     * config zigbee state
     * arg1: zigbee state
     *       (idle form-network wait-join join-network network-up HMI)
     */
    WMI_COEX_CONFIG_MPTA_HELPER_ZIGBEE_STATE = 36,
    /* WMI_COEX_CONFIG_MPTA_HELPER_INT_OCS_PARAMS
     * config ocs wlan/nonwlan params after MPTA interrupt fired
     * arg1: wlan duration (ms units) in Shape-OCS
     * arg2: nonwlan duration (ms units) in Shape-OCS
     */
    WMI_COEX_CONFIG_MPTA_HELPER_INT_OCS_PARAMS = 37,
    /* WMI_COEX_CONFIG_MPTA_HELPER_MON_OCS_PARAMS
     * config ocs wlan/nonwlan params during monitor period after
     * interrupt period finished
     * arg1: wlan duration (ms units) in Shape-OCS
     * arg2: nonwlan duration (ms units) in Shape-OCS
     */
    WMI_COEX_CONFIG_MPTA_HELPER_MON_OCS_PARAMS   = 38,
    /* WMI_COEX_CONFIG_MPTA_HELPER_INT_MON_DURATION
     * config ocs duration in interrupt period and monitor period
     * arg1: duration (ms units) in interrupt period
     * arg2: duration (ms units) in monitor period
     */
    WMI_COEX_CONFIG_MPTA_HELPER_INT_MON_DURATION = 39,
    /* WMI_COEX_CONFIG_MPTA_HELPER_ZIGBEE_CHANNEL
     * config zigbee channel 11 - 26
     */
    WMI_COEX_CONFIG_MPTA_HELPER_ZIGBEE_CHANNEL   = 40,
    /* WMI_COEX_CONFIG_MPTA_HELPER_WLAN_MUTE_DURATION
     * config msw mute duration (ms units) after MPTA interrupt fired
     */
    WMI_COEX_CONFIG_MPTA_HELPER_WLAN_MUTE_DURATION   = 41,
} WMI_COEX_CONFIG_TYPE;

typedef struct {
    A_UINT32 tlv_header;
    A_UINT32 vdev_id;
    A_UINT32 config_type; /* wmi_coex_config_type enum */
    A_UINT32 config_arg1;
    A_UINT32 config_arg2;
    A_UINT32 config_arg3;
    A_UINT32 config_arg4;
    A_UINT32 config_arg5;
    A_UINT32 config_arg6;
} WMI_COEX_CONFIG_CMD_fixed_param;

/**
 * This command is sent from WLAN host driver to firmware to
 * request firmware to enable/disable channel avoidance report
 * to host.
 *
 */
enum {
    WMI_MWSCOEX_CHAN_AVD_RPT_DISALLOW = 0,
    WMI_MWSCOEX_CHAN_AVD_RPT_ALLOW = 1
};

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_CHAN_AVOID_RPT_ALLOW_CMD_fixed_param */
    /** Allow/disallow flag - see WMI_MWSCOEX_CHAN_AVD_RPT enum */
    A_UINT32 rpt_allow;
} WMI_CHAN_AVOID_RPT_ALLOW_CMD_fixed_param;

/*
 * Periodic channel stats WMI command structure
 * WMI_SET_PERIODIC_CHANNEL_STATS_CONFIG_CMDID
 */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_set_periodic_channel_stats_config_fixed_param */
    /** 1 = enable, 0 = disable */
    A_UINT32 enable;
    /** periodic stats duration (units are milliseconds) */
    A_UINT32 stats_period;
    /*
     * pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
} wmi_set_periodic_channel_stats_config_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_wal_power_debug_cmd_fixed_param */
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     */
    A_UINT32 pdev_id;
    /* Identify the wlan module */
    A_UINT32 module_id;
    /* Num of elements in the following args[] array */
    A_UINT32 num_args;
/**
 * Following this structure are the TLVs:
 *   A_UINT32 args[];
 **/
} wmi_pdev_wal_power_debug_cmd_fixed_param;

typedef struct {
    /**
     * TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_pdev_set_reorder_timeout_val_cmd_fixed_param
     */
    A_UINT32 tlv_header;
    /**
     * @brief rx_timeout_pri - what rx reorder timeout (ms) to use for the AC
     * @details
     *     Each WMM access category (voice, video, best-effort, background)
     *     will have its own timeout value to dictate how long to wait for
     *     missing rx MPDUs to arrive before releasing subsequent MPDUs that
     *     have already been received.
     *     This parameter specifies the timeout in milliseconds for each
     *     access category.
     *     The array elements are indexed by the WMI_AC enum to identify
     *     which array element corresponds to which AC / traffic type.
     */
    A_UINT32 rx_timeout_pri[WMI_AC_MAX];
} wmi_pdev_set_reorder_timeout_val_cmd_fixed_param;

/**
 * wlan stats shall be understood as per period.
 * Generally, it is reported periodically based on the period specified by host.
 * But if the variation of one stats of compared to the
 * pervious notification exceeds a threshold,
 * FW will report the wlan stats immediately.
 * The values of the stats becomes the new reference to compute variations.
 * This threshold can be a global setting or per category.
 * Host can enable/disable the mechanism for any stats per bitmap.
 * TX/RX thresholds (percentage value) are shared across ACs,
 * and TX/RX stats comprisons are processed per AC of each peer.
 * For example, if bit 0 (stand for tx_mpdus) of tx_thresh_bitmap is set to 1,
 * and the detailed tx_mpdus threshold value is set to 10%,
 * suppose tx_mpdus value of BE of peer 0 is 100 in first period,
 * and it reaches 110 during the second period,
 * FW will generate and send out a wlan stats event immediately.
 */
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_pdev_set_stats_threshold_cmd_fixed_param */
    /** Indicate if threshold mechnism is enabled or disabled.
     *  It is disabled by default.
     *  Host can enable and disable it dynamically.
     */
    A_UINT32 enable_thresh;
    /** use_thresh_bitmap equals 0 means gbl_thresh is used.
     *  when use_thresh_bitmap equals 1, ignore gbl_thresh and use stats bitmap indicated thresholds.
     */
    A_UINT32 use_thresh_bitmap;
    /** global threshold, valid when use_thresh_bitmap equals 0.
     *  It takes effect for all counters.
     *  If use_thresh_bitmap ==0 && gbl_thresh == 0, disable threshold mechanism.
     */
    A_UINT32 gbl_thresh;
    /** Enable/disable bitmap for threshold mechanism of CCA stats */
    A_UINT32 cca_thresh_enable_bitmap;
    /** Enable/disable bitmap for threshold mechanism of signal stats */
    A_UINT32 signal_thresh_enable_bitmap;
    /** Enable/disable bitmap for threshold mechanism of TX stats */
    A_UINT32 tx_thresh_enable_bitmap;
    /** Enable/disable bitmap for threshold mechanism of RX stats */
    A_UINT32 rx_thresh_enable_bitmap;
    /* This TLV is followed by TLVs below:
     *    wmi_chan_cca_stats_thresh     cca_thresh;
     *    wmi_peer_signal_stats_thresh  signal_thresh;
     *    wmi_tx_stats_thresh           tx_thresh;
     *    wmi_rx_stats_thresh           rx_thresh;
     */
} wmi_pdev_set_stats_threshold_cmd_fixed_param;

typedef enum {
    WMI_REQUEST_WLAN_TX_STAT     = 0x01,
    WMI_REQUEST_WLAN_RX_STAT     = 0x02,
    WMI_REQUEST_WLAN_CCA_STAT    = 0x04,
    WMI_REQUEST_WLAN_SIGNAL_STAT = 0x08,
} wmi_wlan_stats_id;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_request_wlan_stats_cmd_fixed_param */
    wmi_wlan_stats_id stats_id;
} wmi_request_wlan_stats_cmd_fixed_param;

typedef enum {
    WMI_REQUEST_ONE_PEER_STATS_INFO = 0x01, /* request stats of one specified peer */
    WMI_REQUEST_VDEV_ALL_PEER_STATS_INFO = 0x02, /* request stats of all peers belong to specified VDEV */
} wmi_peer_stats_info_request_type;

/** It is required to issue WMI_PDEV_PARAM_PEER_STATS_INFO_ENABLE
*  (with a non-zero value) before issuing the first REQUEST_PEER_STATS_INFO.
*/
typedef struct {
    /** TLV tag and len; tag equals
    * WMITLV_TAG_STRUC_wmi_request_peer_stats_info_cmd_fixed_param */
    A_UINT32 tlv_header;
    /** request_type to indicate if only stats of
    *  one peer or all peers of the VDEV are requested,
    *  see wmi_peer_stats_info_request_type.
    */
    A_UINT32 request_type;
    /** VDEV identifier */
    A_UINT32 vdev_id;
    /** this peer_macaddr is only used if request_type == ONE_PEER_STATS_INFO */
    wmi_mac_addr peer_macaddr;
    /** flag to indicate if FW needs to reset requested peers stats */
    A_UINT32 reset_after_request;
} wmi_request_peer_stats_info_cmd_fixed_param;

typedef enum {
    WMI_REQUEST_ONE_RADIO_CHAN_STATS = 0x01, /* request stats of one specified channel */
    WMI_REQUEST_ALL_RADIO_CHAN_STATS = 0x02, /* request stats of all channels */
} wmi_radio_chan_stats_request_type;

/** It is required to issue WMI_PDEV_PARAM_RADIO_CHAN_STATS_ENABLE
 *  (with a non-zero value) before issuing the first REQUEST_RADIO_CHAN_STATS.
 */
typedef struct {
    /** TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_request_radio_chan_stats_cmd_fixed_param */
     A_UINT32 tlv_header;
     /** request_type to indicate if only stats of
     *  one channel or all channels are requested,
     *  see wmi_radio_chan_stats_request_type.
     */
     A_UINT32 request_type;
     /** Frequency of channel whose stats is requested,
     *  only used when request_type == WMI_REQUEST_ONE_RADIO_CHAN_STATS
     */
     A_UINT32 chan_mhz;
     /** flag to indicate if FW needs to reset requested stats of specified channel/channels */
     A_UINT32 reset_after_request;
} wmi_request_radio_chan_stats_cmd_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
     * WMITLV_TAG_STRUC_wmi_rmc_set_leader_cmd_fixed_param */
    A_UINT32 tlv_header;
    /* VDEV identifier */
    A_UINT32 vdev_id;
    /* Leader's mac address */
    wmi_mac_addr leader_mac_addr;
} wmi_rmc_set_leader_cmd_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
    * WMITLV_TAG_STRUC_wmi_rmc_manual_leader_event_fixed_param */
    A_UINT32 tlv_header;
    /* VDEV identifier */
    A_UINT32 vdev_id;
    /*  0: success
     *  1: selected leader not found in network, able to find using auto selection
     * -1: error
     * non zero value should be return to userspace in case of failure */
    A_INT32 status;
    /* bssid of new leader */
    wmi_mac_addr leader_mac_addr;
} wmi_rmc_manual_leader_event_fixed_param;

typedef enum {
    WLAN_2G_CAPABILITY = 0x1,
    WLAN_5G_CAPABILITY = 0x2,
} WLAN_BAND_CAPABILITY;

typedef enum wmi_hw_mode_config_type {
	/* Only one PHY is active. */
	WMI_HW_MODE_SINGLE      = 0,
	/**
	 * Both PHYs are active in different bands, one in 2G
	 * and another in 5G.
	 */
	WMI_HW_MODE_DBS         = 1,
	/**
	 * Both PHYs are in passive mode (only rx) in same band;
	 * no tx allowed.
	 */
	WMI_HW_MODE_SBS_PASSIVE = 2,
	/**
	 * Both PHYs are active in the same band.
	 * Support for both PHYs within one band is planned for 5G only
	 * (as indicated in WMI_MAC_PHY_CAPABILITIES),
	 * but could be extended to other bands in the future.
	 * The separation of the band between the two PHYs needs to be
	 * communicated separately.
	 */
	WMI_HW_MODE_SBS         = 3,
	/**
	 * 3 PHYs, with 2 on the same band doing SBS
	 * as in WMI_HW_MODE_SBS, and 3rd on the other band
	 */
	WMI_HW_MODE_DBS_SBS     = 4,
	WMI_HW_MODE_DBS_OR_SBS  = 5, /* One PHY is on 5G and the other PHY can be in 2G or 5G. */
} WMI_HW_MODE_CONFIG_TYPE;

#define WMI_SUPPORT_11B_GET(flags) WMI_GET_BITS(flags, 0, 1)
#define WMI_SUPPORT_11B_SET(flags, value) WMI_SET_BITS(flags, 0, 1, value)

#define WMI_SUPPORT_11G_GET(flags) WMI_GET_BITS(flags, 1, 1)
#define WMI_SUPPORT_11G_SET(flags, value) WMI_SET_BITS(flags, 1, 1, value)

#define WMI_SUPPORT_11A_GET(flags) WMI_GET_BITS(flags, 2, 1)
#define WMI_SUPPORT_11A_SET(flags, value) WMI_SET_BITS(flags, 2, 1, value)

#define WMI_SUPPORT_11N_GET(flags) WMI_GET_BITS(flags, 3, 1)
#define WMI_SUPPORT_11N_SET(flags, value) WMI_SET_BITS(flags, 3, 1, value)

#define WMI_SUPPORT_11AC_GET(flags) WMI_GET_BITS(flags, 4, 1)
#define WMI_SUPPORT_11AC_SET(flags, value) WMI_SET_BITS(flags, 4, 1, value)

#define WMI_SUPPORT_11AX_GET(flags) WMI_GET_BITS(flags, 5, 1)
#define WMI_SUPPORT_11AX_SET(flags, value) WMI_SET_BITS(flags, 5, 1, value)

#define WMI_MAX_MUBFEE_GET(flags) WMI_GET_BITS(flags, 28, 4)
#define WMI_MAX_MUBFEE_SET(flags, value) WMI_SET_BITS(flags, 28, 4, value)

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_MAC_PHY_CAPABILITIES */
    /* hw_mode_id - identify a particular set of HW characteristics, as specified
     * by the subsequent fields. WMI_MAC_PHY_CAPABILITIES element must be mapped
     * to its parent WMI_HW_MODE_CAPABILITIES element using hw_mode_id.
     * No particular ordering of WMI_MAC_PHY_CAPABILITIES elements should be assumed,
     * though in practice the elements may always be ordered by hw_mode_id */
    A_UINT32 hw_mode_id;
    /* pdev_id starts with 1. pdev_id 1 => phy_id 0, pdev_id 2 => phy_id 1 */
    A_UINT32 pdev_id;
    /* phy id. Starts with 0 */
    A_UINT32 phy_id;
    /* supported modulations and number of MU beamformees */
    union {
        struct {
            A_UINT32 supports_11b:1,
                     supports_11g:1,
                     supports_11a:1,
                     supports_11n:1,
                     supports_11ac:1,
                     supports_11ax:1,

                     unused:22,

                     /* max MU beamformees supported per MAC */
                     max_mubfee:4;
        };
        A_UINT32 supported_flags;
    };
    /* supported bands, enum WLAN_BAND_CAPABILITY */
    A_UINT32 supported_bands;
    /* ampdu density 0 for no restriction, 1 for 1/4 us, 2 for 1/2 us,
     * 3 for 1 us,4 for 2 us, 5 for 4 us, 6 for 8 us,7 for 16 us */
    A_UINT32 ampdu_density;
    /* max bw supported 2G, enum wmi_channel_width */
    A_UINT32 max_bw_supported_2G;
    /* WMI HT Capability, WMI_HT_CAP defines */
    A_UINT32 ht_cap_info_2G;
    /* VHT capability info field of 802.11ac, WMI_VHT_CAP defines */
    A_UINT32 vht_cap_info_2G;
    /* VHT Supported MCS Set field Rx/Tx same
     * The max VHT-MCS for n SS subfield (where n = 1,...,8) is encoded as follows
     * - 0 indicates support for VHT-MCS 0-7 for n spatial streams
     * - 1 indicates support for VHT-MCS 0-8 for n spatial streams
     * - 2 indicates support for VHT-MCS 0-9 for n spatial streams
     * - 3 indicates that n spatial streams is not supported
     * - bit [15:0] Each NSS takes 2 bit.
     * - bit [23:16] Indicates support for VHT-MCS 10 and 11 for
     *   n spatial streams
     *       - bit 16 - for NSS 1
     *       - bit 17 - for NSS 2
     *       - .
     *       - .
     *       - bit 23 - for NSS 8
     *       - bit 24 - indicate whether the VHT-MCS 10-11 specs in bits 23:16
     *                  are valid
     *   Refer to the WMI_VHT_MAX_MCS_EXT_SS_GET/SET macros.
     */
    A_UINT32 vht_supp_mcs_2G;
    /*HE capability info field of 802.11ax, WMI_HE_CAP defines */
    A_UINT32 he_cap_info_2G;
    /* HE Supported MCS Set field Rx/Tx same
     * - 2 bits are used for each NSS chain.Max of 8 NSS can be encoded with
     *     bit 2-0 indicating max HE MCS of NSS1
     *     bit 5-3 indicating max HE MCS of NSS2 and so on
     * - The max HE-MCS for n SS subfield (where n = 1,...,8) in case of
     *     HE BW less than or equal to 80MHZ is encoded as follows
     * - 0 indicates support for VHT-MCS 0-7 for n spatial streams
     * - 1 indicates support for VHT-MCS 0-9 for n spatial streams
     * - 2 indicates support for VHT-MCS 0-11 for n spatial streams
     * - 3 indicates that n spatial streams is not supported
     * - WMI_HE_MAX_MCS_4_SS_MASK macro can be used for encoding this info
     */
    A_UINT32 he_supp_mcs_2G;
    /* Valid Transmit chain mask */
    A_UINT32 tx_chain_mask_2G;
    /* Valid Receive chain mask */
    A_UINT32 rx_chain_mask_2G;
    /* max bw supported 5G, enum wmi_channel_width */
    A_UINT32 max_bw_supported_5G;
    /* WMI HT Capability, WMI_HT_CAP defines */
    A_UINT32 ht_cap_info_5G;
    /* VHT capability info field of 802.11ac, WMI_VHT_CAP defines */
    A_UINT32 vht_cap_info_5G;
    /* VHT Supported MCS Set field Rx/Tx same
     * The max VHT-MCS for n SS subfield (where n = 1,...,8) is encoded as follows
     * - 0 indicates support for VHT-MCS 0-7 for n spatial streams
     * - 1 indicates support for VHT-MCS 0-8 for n spatial streams
     * - 2 indicates support for VHT-MCS 0-9 for n spatial streams
     * - 3 indicates that n spatial streams is not supported
     * - bit [15:0] Each NSS takes 2 bit.
     * - bit [23:16] Indicates support for VHT-MCS 10 and 11 for
     *   n spatial streams
     *       - bit 16 - for NSS 1
     *       - bit 17 - for NSS 2
     *       - .
     *       - .
     *       - bit 23 - for NSS 8
     *       - bit 24 - indicate whether the VHT-MCS 10-11 specs in bits 23:16
     *                  are valid
     *   Refer to the WMI_VHT_MAX_MCS_EXT_SS_GET/SET macros.
     */
    A_UINT32 vht_supp_mcs_5G;
    /*HE capability info field of 802.11ax, WMI_HE_CAP defines */
    A_UINT32 he_cap_info_5G;
    /* HE Supported MCS Set field Rx/Tx same
     * - 2 bits are used for each NSS chain.Max of 8 NSS can be encoded with
     *     bit 2-0 indicating max HE MCS of NSS1
     *     bit 5-3 indicating max HE MCS of NSS2 and so on
     * - The max HE-MCS for n SS subfield (where n = 1,...,8) in case of
     *     HE BW less than or equal to 80MHZ is encoded as follows
     * - 0 indicates support for VHT-MCS 0-7 for n spatial streams
     * - 1 indicates support for VHT-MCS 0-9 for n spatial streams
     * - 2 indicates support for VHT-MCS 0-11 for n spatial streams
     * - 3 indicates that n spatial streams is not supported
     * - WMI_HE_MAX_MCS_4_SS_MASK macro can be used for encoding this info
     * - The max HE-MCS for n SS subfield (where n = 1,...,8) in case of
     *     HE BW equal to 80+80 or 160 MHZ encoding is same as above just the
     *     lower 16 bits are used for lower 80MHz NSS-MCS supported combo and
     *     upper 16 bits are used for upper 80MHz NSS-MCS supported combo
     */
    A_UINT32 he_supp_mcs_5G;
    /* Valid Transmit chain mask */
    A_UINT32 tx_chain_mask_5G;
    /* Valid Receive chain mask */
    A_UINT32 rx_chain_mask_5G;
    /* HE capability phy field of 802.11ax, WMI_HE_CAP defines */
    A_UINT32 he_cap_phy_info_2G[WMI_MAX_HECAP_PHY_SIZE];
    A_UINT32 he_cap_phy_info_5G[WMI_MAX_HECAP_PHY_SIZE];
    wmi_ppe_threshold he_ppet2G;
    wmi_ppe_threshold he_ppet5G;
    /* chainmask table to be used for the MAC */
    A_UINT32 chainmask_table_id;
    /* PDEV ID to LMAC ID mapping */
    A_UINT32 lmac_id;
    /* 2nd DWORD of HE capability info field of 802.11ax, support Draft 3+ */
    A_UINT32 he_cap_info_2G_ext;
    A_UINT32 he_cap_info_5G_ext;
    /*
     * bit 0     : Indicated support for RX 1xLTF + 0.4us
     * bit 1     : Indicates support for RX 2xLTF + 0.4us
     * bit 2     : Indicates support for 2xLTF in 160/80+80 MHz HE PPDU
     * bit[31:3] : Reserved
     * Refer to WMI_HE_CAP_xx_LTF_xxx_SUPPORT_GET/SET macros
     */
    A_UINT32 he_cap_info_internal;
} WMI_MAC_PHY_CAPABILITIES;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_HW_MODE_CAPABILITIES */
    /* hw_mode_id - identify a particular set of HW characteristics,
     * as specified by the subsequent fields */
    A_UINT32 hw_mode_id;
    /**
     * BIT0 represents phy_id 0, BIT1 represent phy_id 1 and so on.
     * Number of bits set in phy_id_map represents number of
     * WMI_MAC_PHY_CAPABILITIES TLV's, one for each active PHY for current HW
     * mode identified by hw_mode_id. For example, for DBS/SBS mode there will
     * be 2 WMI_MAC_PHY_CAPABILITIES TLVs and for single MAC modes it will be
     * 1 WMI_MAC_PHY_CAPABILITIES TLVs
     */
    A_UINT32 phy_id_map;
    /**
     * hw_mode_config_type
     * Identify a particular type of HW mode such as SBS, DBS etc.
     * Refer to WMI_HW_MODE_CONFIG_TYPE values.
     */
    A_UINT32 hw_mode_config_type;
} WMI_HW_MODE_CAPABILITIES;

/*
 * The following macros are for the bitfields witihin the supported flags field
 * of WMI_MAC_PHY_CHAINMASK_CAPABILITY:
 * Capabilities for the chainmask
 */

#define WMI_SUPPORT_CHAN_WIDTH_20_GET(flags) WMI_GET_BITS(flags, 0, 1)
#define WMI_SUPPORT_CHAN_WIDTH_20_SET(flags, value) WMI_SET_BITS(flags, 0, 1, value)

#define WMI_SUPPORT_CHAN_WIDTH_40_GET(flags) WMI_GET_BITS(flags, 1, 1)
#define WMI_SUPPORT_CHAN_WIDTH_40_SET(flags, value) WMI_SET_BITS(flags, 1, 1, value)

#define WMI_SUPPORT_CHAN_WIDTH_80_GET(flags) WMI_GET_BITS(flags, 2, 1)
#define WMI_SUPPORT_CHAN_WIDTH_80_SET(flags, value) WMI_SET_BITS(flags, 2, 1, value)

#define WMI_SUPPORT_CHAN_WIDTH_160_GET(flags) WMI_GET_BITS(flags, 3, 1)
#define WMI_SUPPORT_CHAN_WIDTH_160_SET(flags, value) WMI_SET_BITS(flags, 3, 1, value)

#define WMI_SUPPORT_CHAN_WIDTH_80P80_GET(flags) WMI_GET_BITS(flags, 4, 1)
#define WMI_SUPPORT_CHAN_WIDTH_80P80_SET(flags, value) WMI_SET_BITS(flags, 4, 1, value)

#define WMI_SUPPORT_CHAIN_MASK_2G_GET(flags) WMI_GET_BITS(flags, 27, 1)
#define WMI_SUPPORT_CHAIN_MASK_2G_SET(flags, value) WMI_SET_BITS(flags, 27, 1, value)

#define WMI_SUPPORT_CHAIN_MASK_5G_GET(flags) WMI_GET_BITS(flags, 28, 1)
#define WMI_SUPPORT_CHAIN_MASK_5G_SET(flags, value) WMI_SET_BITS(flags, 28, 1, value)

#define WMI_SUPPORT_CHAIN_MASK_TX_GET(flags) WMI_GET_BITS(flags, 29, 1)
#define WMI_SUPPORT_CHAIN_MASK_TX_SET(flags, value) WMI_SET_BITS(flags, 29, 1, value)

#define WMI_SUPPORT_CHAIN_MASK_RX_GET(flags) WMI_GET_BITS(flags, 30, 1)
#define WMI_SUPPORT_CHAIN_MASK_RX_SET(flags, value) WMI_SET_BITS(flags, 30, 1, value)

#define WMI_SUPPORT_CHAIN_MASK_ADFS_GET(flags) WMI_GET_BITS(flags, 31, 1)
#define WMI_SUPPORT_CHAIN_MASK_ADFS_SET(flags, value) WMI_SET_BITS(flags, 31, 1, value)

/* Definition of valid chainmask and associated capabilities */
typedef struct {
    /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_MAC_PHY_CHAINMASK_CAPABILITY */
    A_UINT32 tlv_header;
    /* supported flags: Capabilities for this chianmask*/
    union {
        struct {
            A_UINT32 supports_chan_width_20:1,
                     supports_chan_width_40:1,
                     supports_chan_width_80:1,
                     supports_chan_width_160:1,
                     supports_chan_width_80P80:1,
                     reserved:22, /* bits 26:5 */
                     chain_mask_2G:1,
                     chain_mask_5G:1,
                     chain_mask_tx:1,
                     chain_mask_rx:1,
                     supports_aDFS:1; /* agile DFS */
        };
        A_UINT32 supported_flags;
    };
    A_UINT32 chainmask;
} WMI_MAC_PHY_CHAINMASK_CAPABILITY;

typedef struct {
    /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_MAC_PHY_CHAINMASK_COMBO */
    A_UINT32 tlv_header;
    A_UINT32 chainmask_table_id;
    /* Number of vaild Chainmask in the table */
    A_UINT32 num_valid_chainmask;
    /*
     * This TLV is followed by the below TLVs:
     * WMI_MAC_PHY_CHAINMASK_CAPABILITY mac_phy_chainmask_caps[num_valid_chainmask]
     */
} WMI_MAC_PHY_CHAINMASK_COMBO;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_SOC_MAC_PHY_HW_MODE_CAPS */
    /* num HW modes */
    A_UINT32 num_hw_modes;
    /* number of unique chainmask combo tables */
    A_UINT32 num_chainmask_tables;
    /*
     * This TLV is followed by the below TLVs:
     *
     * WMI_HW_MODE_CAPABILITIES soc_hw_mode_caps[num_hw_modes]
     *
     * (intervening TLVs, e.g. HW_MODE_CAPS, MAC_PHY_CAPS, HAL_REG_CAPS)
     *
     * WMI_MAC_PHY_CHAINMASK_COMBO mac_phy_chainmask_combo[num_chainmask_tables]
     * // number of chainmasks specified in mac_phy_chainmask_combo[0]
     * WMI_MAC_PHY_CHAINMASK_CAPABILITY mac_phy_chainmask_caps[num_valid_chainmask0]
     * // number of chainmasks specified in mac_phy_chainmask_combo[1]
     * WMI_MAC_PHY_CHAINMASK_CAPABILITY mac_phy_chainmask_caps[num_valid_chainmask1]
     * // number of chainmasks specified in mac_phy_chainmask_combo[2]
     * WMI_MAC_PHY_CHAINMASK_CAPABILITY mac_phy_chainmask_caps[num_valid_chainmask2]
     * etc.
     */
} WMI_SOC_MAC_PHY_HW_MODE_CAPS;

/*Below are Reg caps per PHY. Please note PHY ID starts with 0.*/
typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_HAL_REG_CAPABILITIES_EXT */
    /* phy id */
    A_UINT32 phy_id;
    /* regdomain value specified in EEPROM */
    A_UINT32 eeprom_reg_domain;
    /* regdomain */
    A_UINT32 eeprom_reg_domain_ext;
    /* CAP1 capabilities bit map, see REGDMN_CAP1_ defines */
    A_UINT32 regcap1;
    /* REGDMN EEPROM CAP, see REGDMN_EEPROM_EEREGCAP_ defines */
    A_UINT32 regcap2;
    /* REGDMN MODE, see REGDMN_MODE_ enum */
    A_UINT32 wireless_modes;
    A_UINT32 low_2ghz_chan;
    A_UINT32 high_2ghz_chan;
    A_UINT32 low_5ghz_chan;
    A_UINT32 high_5ghz_chan;
} WMI_HAL_REG_CAPABILITIES_EXT;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_SOC_HAL_REG_CAPABILITIES */
    A_UINT32 num_phy;
    // num_phy WMI_HAL_REG_CAPABILITIES_EXT TLV's
} WMI_SOC_HAL_REG_CAPABILITIES;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_OEM_DMA_RING_CAPABILITIES */
    A_UINT32 pdev_id;
    A_UINT32 min_num_ptr;
    /* Minimum number of pointers in the OEM DMA ring for this pdev */
    A_UINT32 min_buf_size;
    /* Minimum size in bytes of each buffer in the OEM DMA ring */
    A_UINT32 min_buf_align;
    /* Minimum alignment in bytes of each buffer in the OEM DMA ring */
} WMI_OEM_DMA_RING_CAPABILITIES;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_WMI_SAR_CAPABILITIES*/
    /* sar version in bdf */
    A_UINT32 active_version;
} WMI_SAR_CAPABILITIES;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_scan_adaptive_dwell_parameters_tlv */
    /** global default adaptive dwell mode, used when WMI_SCAN_DWELL_MODE_DEFAULT */
    A_UINT32 default_adaptive_dwell_mode;
   /** the weight to calculate the average low pass filter for channel congestion. 0-100 */
    A_UINT32 adapative_lpf_weight;
   /** interval to monitor passive scan in msec */
    A_UINT32 passive_monitor_interval_ms;
   /** % of wifi activity to switch from passive to active 0-100 */
    A_UINT32 wifi_activity_threshold_pct;
} wmi_scan_adaptive_dwell_parameters_tlv;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_scan_adaptive_dwell_config_fixed_param */
    /* globally enable/disable adaptive dwell */
    A_UINT32 enable;
    /**
     * pdev_id for identifying the MAC. See macros starting with
     * WMI_PDEV_ID_ for values. In non-DBDC case host should set it to 0
     */
    A_UINT32 pdev_id;
    /**
     * followed by TLV (tag length value) parameters array
     * The TLV's are:
     * wmi_scan_adaptive_dwell_parameters_tlv param[]; (0 or 1 elements)
     */
} wmi_scan_adaptive_dwell_config_fixed_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_scan_dbs_duty_cycle_param_tlv */
    /**  module_id **/
    A_UINT32 module_id;
   /** number of dbs scans */
    A_UINT32 num_dbs_scans;
   /** number of non-dbs scans */
    A_UINT32 num_non_dbs_scans;
} wmi_scan_dbs_duty_cycle_tlv_param;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_scan_dbs_duty_cycle_fixed_param */
    /* number of scan client dutycycle param elements */
    A_UINT32 num_clients;
    /** pdev_id for identifying the MAC
     * See macros starting with WMI_PDEV_ID_ for values.
     * In non-DBDC case host should set it to 0
     */
    A_UINT32 pdev_id;
/**
 * followed by TLV (tag length value) parameters array
 * The TLV's are:
 * wmi_scan_selection_duty_cycle_tlv_param[num_clients];
 */
} wmi_scan_dbs_duty_cycle_fixed_param;

typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_therm_throt_level_config_info */
    A_UINT32 tlv_header;
    /**
      * temperature sensor value in celsius when to exit to lower zone,
      * this value can be lower than HWM of lower zone as zone overlapping
      * is permitted by design to provide hysteresis
      */
    A_UINT32 temp_lwm;
    /**
      * temperature sensor value in celsius when to exit to higher zone,
      * this value can be higher than LWM of higher zone as zone overlapping
      * is permitted by design to provide hysteresis
      */
    A_UINT32 temp_hwm;
    A_UINT32 dc_off_percent; /* duty cycle off percent 0-100. 0 means no off, 100 means no on (shutdown the phy) */
    /** Disable only the transmit queues in firmware that have lower priority than value defined by prio
       *    Prioritization:
       *    0 = disable all data tx queues, No Prioritization defined
       *    1 = disable BK tx queue
       *    2 = disable BK+BE tx queues
       *    3 = disable BK+BE+VI tx queues
       */
    A_UINT32 prio;
} wmi_therm_throt_level_config_info;

typedef struct {
    A_UINT32 tlv_header; /* TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_therm_throt_config_request_fixed_param */
    A_UINT32 pdev_id;          /* config for each pdev */
    A_UINT32 enable;           /* 0:disable, 1:enable */
    A_UINT32 dc;               /* duty cycle in ms */
    A_UINT32 dc_per_event;     /* how often (after how many duty cycles) the FW sends stats to host */
    A_UINT32 therm_throt_levels; /* Indicates the number of thermal zone configuration */
    /*
     * Following this structure is the TLV:
     * struct wmi_therm_throt_level_config_info therm_throt_level_config_info[therm_throt_levels];
     */
} wmi_therm_throt_config_request_fixed_param;

/** FW response with the stats event id for every pdev and zones */
typedef struct {
    /*  TLV tag and len; tag equals WMITLV_TAG_STRUC_wmi_therm_throt_stats_event_fixed_param */
    A_UINT32 tlv_header;
    A_UINT32 pdev_id;            /* stats for corresponding pdev*/
    A_UINT32 temp;               /* Temperature reading in celsius */
    A_UINT32 level;              /* current thermal throttling level */
    A_UINT32 therm_throt_levels; /* number of levels in therm_throt_level_stats_info */
    /* This TLV is followed by another TLV of array of structs
     * wmi_therm_throt_level_stats_info therm_throt_level_stats_info[therm_throt_levels];
     */
} wmi_therm_throt_stats_event_fixed_param;



typedef struct {
    /** TLV tag and len; tag equals
     *  WMITLV_TAG_STRUC_wmi_therm_throt_level_stats_info */
    A_UINT32 tlv_header;
    A_UINT32 level_count; /* count of each time thermal throttling entered this state */
    A_UINT32 dc_count;    /* total number of duty cycles spent in this state. */
                          /* this number increments by one each time we are in this state and we finish one full duty cycle. */
} wmi_therm_throt_level_stats_info;

typedef enum {
    WMI_REG_EXT_FCC_MIDBAND = 0,
    WMI_REG_EXT_JAPAN_MIDBAND = 1,
    WMI_REG_EXT_FCC_DFS_HT40 = 2,
    WMI_REG_EXT_JAPAN_NONDFS_HT40 = 3,
    WMI_REG_EXT_JAPAN_DFS_HT40 = 4,
    WMI_REG_EXT_FCC_CH_144 = 5,
} WMI_REG_EXT_BITMAP;

#ifdef WMI_CMD_STRINGS
static INLINE A_UINT8 *wmi_id_to_name(A_UINT32 wmi_command)
{
	switch (wmi_command) {
		/* initialize the wlan sub system */
		WMI_RETURN_STRING(WMI_INIT_CMDID);

		/* Scan specific commands */

		/* start scan request to FW  */
		WMI_RETURN_STRING(WMI_START_SCAN_CMDID);
		/* stop scan request to FW  */
		WMI_RETURN_STRING(WMI_STOP_SCAN_CMDID);
		/* full list of channels as defined by the regulatory
		* that will be used by scanner   */
		WMI_RETURN_STRING(WMI_SCAN_CHAN_LIST_CMDID);
		/* overwrite default priority table in scan scheduler */
		WMI_RETURN_STRING(WMI_SCAN_SCH_PRIO_TBL_CMDID);
		/* This command to adjust the priority and min.max_rest_time
		* of an on ongoing scan request.
		*/
		WMI_RETURN_STRING(WMI_SCAN_UPDATE_REQUEST_CMDID);

		/* PDEV(physical device) specific commands */
		/* set regulatorty ctl id used by FW to determine the exact
		* ctl power limits */
		WMI_RETURN_STRING(WMI_PDEV_SET_REGDOMAIN_CMDID);
		/* set channel. mainly used for supporting monitor mode */
		WMI_RETURN_STRING(WMI_PDEV_SET_CHANNEL_CMDID);
		/* set pdev specific parameters */
		WMI_RETURN_STRING(WMI_PDEV_SET_PARAM_CMDID);
		/* enable packet log */
		WMI_RETURN_STRING(WMI_PDEV_PKTLOG_ENABLE_CMDID);
		/* disable packet log*/
		WMI_RETURN_STRING(WMI_PDEV_PKTLOG_DISABLE_CMDID);
		/* set wmm parameters */
		WMI_RETURN_STRING(WMI_PDEV_SET_WMM_PARAMS_CMDID);
		/* set HT cap ie that needs to be carried probe requests
		* HT/VHT channels */
		WMI_RETURN_STRING(WMI_PDEV_SET_HT_CAP_IE_CMDID);
		/* set VHT cap ie that needs to be carried on probe
		* requests on VHT channels */
		WMI_RETURN_STRING(WMI_PDEV_SET_VHT_CAP_IE_CMDID);

		/* Command to send the DSCP-to-TID map to the target */
		WMI_RETURN_STRING(WMI_PDEV_SET_DSCP_TID_MAP_CMDID);
		/* set quiet ie parameters. primarily used in AP mode */
		WMI_RETURN_STRING(WMI_PDEV_SET_QUIET_MODE_CMDID);
		/* Enable/Disable Green AP Power Save  */
		WMI_RETURN_STRING(WMI_PDEV_GREEN_AP_PS_ENABLE_CMDID);
		/* get TPC config for the current operating channel */
		WMI_RETURN_STRING(WMI_PDEV_GET_TPC_CONFIG_CMDID);

		/* set the base MAC address for the physical device before
		* a VDEV is created. For firmware that does not support
		* this feature and this command, the pdev MAC address will
		* not be changed. */
		WMI_RETURN_STRING(WMI_PDEV_SET_BASE_MACADDR_CMDID);

		/* eeprom content dump , the same to bdboard data */
		WMI_RETURN_STRING(WMI_PDEV_DUMP_CMDID);

		/* VDEV(virtual device) specific commands */
		/* vdev create */
		WMI_RETURN_STRING(WMI_VDEV_CREATE_CMDID);
		/* vdev delete */
		WMI_RETURN_STRING(WMI_VDEV_DELETE_CMDID);
		/* vdev start request */
		WMI_RETURN_STRING(WMI_VDEV_START_REQUEST_CMDID);
		/* vdev restart request (RX only, NO TX, used for CAC period)*/
		WMI_RETURN_STRING(WMI_VDEV_RESTART_REQUEST_CMDID);
		/* vdev up request */
		WMI_RETURN_STRING(WMI_VDEV_UP_CMDID);
		/* vdev stop request */
		WMI_RETURN_STRING(WMI_VDEV_STOP_CMDID);
		/* vdev down request */
		WMI_RETURN_STRING(WMI_VDEV_DOWN_CMDID);
		/* set a vdev param */
		WMI_RETURN_STRING(WMI_VDEV_SET_PARAM_CMDID);
		/* set a key (used for setting per peer unicast
		* and per vdev multicast) */
		WMI_RETURN_STRING(WMI_VDEV_INSTALL_KEY_CMDID);

		/* wnm sleep mode command */
		WMI_RETURN_STRING(WMI_VDEV_WNM_SLEEPMODE_CMDID);
		WMI_RETURN_STRING(WMI_VDEV_WMM_ADDTS_CMDID);
		WMI_RETURN_STRING(WMI_VDEV_WMM_DELTS_CMDID);
		WMI_RETURN_STRING(WMI_VDEV_SET_WMM_PARAMS_CMDID);
		WMI_RETURN_STRING(WMI_VDEV_SET_GTX_PARAMS_CMDID);
		WMI_RETURN_STRING(WMI_VDEV_IPSEC_NATKEEPALIVE_FILTER_CMDID);

		WMI_RETURN_STRING(WMI_VDEV_PLMREQ_START_CMDID);
		WMI_RETURN_STRING(WMI_VDEV_PLMREQ_STOP_CMDID);
		WMI_RETURN_STRING(WMI_VDEV_TSF_TSTAMP_ACTION_CMDID);
		WMI_RETURN_STRING(WMI_VDEV_SET_IE_CMDID);

		/* peer specific commands */

		/** create a peer */
		WMI_RETURN_STRING(WMI_PEER_CREATE_CMDID);
		/** delete a peer */
		WMI_RETURN_STRING(WMI_PEER_DELETE_CMDID);
		/** flush specific  tid queues of a peer */
		WMI_RETURN_STRING(WMI_PEER_FLUSH_TIDS_CMDID);
		/** set a parameter of a peer */
		WMI_RETURN_STRING(WMI_PEER_SET_PARAM_CMDID);
		/* set peer to associated state. will cary all parameters
		* determined during assocication time */
		WMI_RETURN_STRING(WMI_PEER_ASSOC_CMDID);
		/* add a wds  (4 address ) entry. used only for testing
		* WDS feature on AP products */
		WMI_RETURN_STRING(WMI_PEER_ADD_WDS_ENTRY_CMDID);
		/* remove wds  (4 address ) entry. used only for testing WDS
		* feature on AP products */
		WMI_RETURN_STRING(WMI_PEER_REMOVE_WDS_ENTRY_CMDID);
		/* set up mcast info for multicast to unicast conversion */
		WMI_RETURN_STRING(WMI_PEER_MCAST_GROUP_CMDID);
		/* request peer info from FW to get PEER_INFO_EVENTID */
		WMI_RETURN_STRING(WMI_PEER_INFO_REQ_CMDID);
		/* unmap response for a peer */
		WMI_RETURN_STRING(WMI_PEER_UNMAP_RESPONSE_CMDID);

		/* beacon/management specific commands */

		/* transmit beacon by reference. used for transmitting beacon
		* on low latency interface like pcie */
		WMI_RETURN_STRING(WMI_BCN_TX_CMDID);
		/* transmit beacon by value */
		WMI_RETURN_STRING(WMI_PDEV_SEND_BCN_CMDID);
		/* set the beacon template. used in beacon offload mode to setup
		* the common beacon template with the FW to be used by FW to
		* generate beacons */
		WMI_RETURN_STRING(WMI_BCN_TMPL_CMDID);
		/* set beacon filter with FW */
		WMI_RETURN_STRING(WMI_BCN_FILTER_RX_CMDID);
		/* enable/disable filtering of probe requests in the firmware */
		WMI_RETURN_STRING(WMI_PRB_REQ_FILTER_RX_CMDID);
		/* transmit management frame by value. will be deprecated */
		WMI_RETURN_STRING(WMI_MGMT_TX_CMDID);
		/* set the probe response template. used in beacon offload mode
		* to setup the common probe response template with the FW to
		* be used by FW to generate probe responses */
		WMI_RETURN_STRING(WMI_PRB_TMPL_CMDID);

		/* commands to directly control ba negotiation directly from
		* host. only used in test mode */

		/* turn off FW Auto addba mode and let host control addba */
		WMI_RETURN_STRING(WMI_ADDBA_CLEAR_RESP_CMDID);
		/* send add ba request */
		WMI_RETURN_STRING(WMI_ADDBA_SEND_CMDID);
		WMI_RETURN_STRING(WMI_ADDBA_STATUS_CMDID);
		/* send del ba */
		WMI_RETURN_STRING(WMI_DELBA_SEND_CMDID);
		/* set add ba response will be used by FW to generate
		* addba response*/
		WMI_RETURN_STRING(WMI_ADDBA_SET_RESP_CMDID);
		/* send single VHT MPDU with AMSDU */
		WMI_RETURN_STRING(WMI_SEND_SINGLEAMSDU_CMDID);

		/* Station power save specific config */
		/* enable/disable station powersave */
		WMI_RETURN_STRING(WMI_STA_POWERSAVE_MODE_CMDID);
		/* set station power save specific parameter */
		WMI_RETURN_STRING(WMI_STA_POWERSAVE_PARAM_CMDID);
		/* set station mimo powersave mode */
		WMI_RETURN_STRING(WMI_STA_MIMO_PS_MODE_CMDID);

		/* DFS-specific commands */
		/* enable DFS (radar detection)*/
		WMI_RETURN_STRING(WMI_PDEV_DFS_ENABLE_CMDID);
		/* disable DFS (radar detection)*/
		WMI_RETURN_STRING(WMI_PDEV_DFS_DISABLE_CMDID);
		/* enable DFS phyerr/parse filter offload */
		WMI_RETURN_STRING(WMI_DFS_PHYERR_FILTER_ENA_CMDID);
		/* enable DFS phyerr/parse filter offload */
		WMI_RETURN_STRING(WMI_DFS_PHYERR_FILTER_DIS_CMDID);

		/* Roaming specific  commands */
		/* set roam scan mode */
		WMI_RETURN_STRING(WMI_ROAM_SCAN_MODE);
		/* set roam scan rssi threshold below which roam
		* scan is enabled  */
		WMI_RETURN_STRING(WMI_ROAM_SCAN_RSSI_THRESHOLD);
		/* set roam scan period for periodic roam scan mode  */
		WMI_RETURN_STRING(WMI_ROAM_SCAN_PERIOD);
		/* set roam scan trigger rssi change threshold   */
		WMI_RETURN_STRING(WMI_ROAM_SCAN_RSSI_CHANGE_THRESHOLD);
		/* set roam AP profile   */
		WMI_RETURN_STRING(WMI_ROAM_AP_PROFILE);
		/* set channel list for roam scans */
		WMI_RETURN_STRING(WMI_ROAM_CHAN_LIST);
		/* offload scan specific commands */
		/* set offload scan AP profile   */
		WMI_RETURN_STRING(WMI_OFL_SCAN_ADD_AP_PROFILE);
		/* remove offload scan AP profile   */
		WMI_RETURN_STRING(WMI_OFL_SCAN_REMOVE_AP_PROFILE);
		/* set offload scan period   */
		WMI_RETURN_STRING(WMI_OFL_SCAN_PERIOD);

		/* P2P specific commands */
		/* set P2P device info. FW will used by FW to create P2P IE
		* to be carried in probe response generated during p2p listen
		* and for p2p discoverability  */
		WMI_RETURN_STRING(WMI_P2P_DEV_SET_DEVICE_INFO);
		/* enable/disable p2p discoverability on STA/AP VDEVs  */
		WMI_RETURN_STRING(WMI_P2P_DEV_SET_DISCOVERABILITY);
		/* set p2p ie to be carried in beacons generated by FW for GO */
		WMI_RETURN_STRING(WMI_P2P_GO_SET_BEACON_IE);
		/* set p2p ie to be carried in probe response frames generated
		* by FW for GO  */
		WMI_RETURN_STRING(WMI_P2P_GO_SET_PROBE_RESP_IE);
		/* set the vendor specific p2p ie data.
		* FW will use this to parse the P2P NoA
		* attribute in the beacons/probe responses received.
		*/
		WMI_RETURN_STRING(WMI_P2P_SET_VENDOR_IE_DATA_CMDID);
		/* set the configure of p2p find offload */
		WMI_RETURN_STRING(WMI_P2P_DISC_OFFLOAD_CONFIG_CMDID);
		/* set the vendor specific p2p ie data for p2p find offload */
		WMI_RETURN_STRING(WMI_P2P_DISC_OFFLOAD_APPIE_CMDID);
		/* set the BSSID/device name pattern of p2p find offload */
		WMI_RETURN_STRING(WMI_P2P_DISC_OFFLOAD_PATTERN_CMDID);
		/* set OppPS related parameters **/
		WMI_RETURN_STRING(WMI_P2P_SET_OPPPS_PARAM_CMDID);

		/* AP power save specific config
		* set AP power save specific param */
		WMI_RETURN_STRING(WMI_AP_PS_PEER_PARAM_CMDID);
		/* set AP UAPSD coex pecific param */
		WMI_RETURN_STRING(WMI_AP_PS_PEER_UAPSD_COEX_CMDID);

		/* Rate-control specific commands */
		WMI_RETURN_STRING(WMI_PEER_RATE_RETRY_SCHED_CMDID);

		/* WLAN Profiling commands. */
		WMI_RETURN_STRING(WMI_WLAN_PROFILE_TRIGGER_CMDID);
		WMI_RETURN_STRING(WMI_WLAN_PROFILE_SET_HIST_INTVL_CMDID);
		WMI_RETURN_STRING(WMI_WLAN_PROFILE_GET_PROFILE_DATA_CMDID);
		WMI_RETURN_STRING(WMI_WLAN_PROFILE_ENABLE_PROFILE_ID_CMDID);
		WMI_RETURN_STRING(WMI_WLAN_PROFILE_LIST_PROFILE_ID_CMDID);

		/* Suspend resume command Ids */
		WMI_RETURN_STRING(WMI_PDEV_SUSPEND_CMDID);
		WMI_RETURN_STRING(WMI_PDEV_RESUME_CMDID);

		/* Beacon filter commands */
		/* add a beacon filter */
		WMI_RETURN_STRING(WMI_ADD_BCN_FILTER_CMDID);
		/* remove a  beacon filter */
		WMI_RETURN_STRING(WMI_RMV_BCN_FILTER_CMDID);

		/* WOW Specific WMI commands */
		/* add pattern for awake */
		WMI_RETURN_STRING(WMI_WOW_ADD_WAKE_PATTERN_CMDID);
		/* deleta a wake pattern */
		WMI_RETURN_STRING(WMI_WOW_DEL_WAKE_PATTERN_CMDID);
		/* enable/deisable wake event  */
		WMI_RETURN_STRING(WMI_WOW_ENABLE_DISABLE_WAKE_EVENT_CMDID);
		/* enable WOW  */
		WMI_RETURN_STRING(WMI_WOW_ENABLE_CMDID);
		/* host woke up from sleep event to FW. Generated in response
		* to WOW Hardware event */
		WMI_RETURN_STRING(WMI_WOW_HOSTWAKEUP_FROM_SLEEP_CMDID);

		/* RTT measurement related cmd */
		/* reques to make an RTT measurement */
		WMI_RETURN_STRING(WMI_RTT_MEASREQ_CMDID);
		/* reques to report a tsf measurement */
		WMI_RETURN_STRING(WMI_RTT_TSF_CMDID);

		/* spectral scan command */
		/* configure spectral scan */
		WMI_RETURN_STRING(WMI_VDEV_SPECTRAL_SCAN_CONFIGURE_CMDID);
		/* enable/disable spectral scan and trigger */
		WMI_RETURN_STRING(WMI_VDEV_SPECTRAL_SCAN_ENABLE_CMDID);

		/* F/W stats */
		/* one time request for stats */
		WMI_RETURN_STRING(WMI_REQUEST_STATS_CMDID);
		/* Push MCC Adaptive Scheduler Stats to Firmware */
		WMI_RETURN_STRING(WMI_MCC_SCHED_TRAFFIC_STATS_CMDID);

		/* ARP OFFLOAD REQUEST*/
		WMI_RETURN_STRING(WMI_SET_ARP_NS_OFFLOAD_CMDID);

		/* Proactive ARP Response Add Pattern Command*/
		WMI_RETURN_STRING(WMI_ADD_PROACTIVE_ARP_RSP_PATTERN_CMDID);

		/* Proactive ARP Response Del Pattern Command*/
		WMI_RETURN_STRING(WMI_DEL_PROACTIVE_ARP_RSP_PATTERN_CMDID);

		/* NS offload confid*/
		WMI_RETURN_STRING(WMI_NETWORK_LIST_OFFLOAD_CONFIG_CMDID);

		/* GTK offload Specific WMI commands */
		WMI_RETURN_STRING(WMI_GTK_OFFLOAD_CMDID);

		/* CSA offload Specific WMI commands */
		/* csa offload enable */
		WMI_RETURN_STRING(WMI_CSA_OFFLOAD_ENABLE_CMDID);
		/* chan switch command */
		WMI_RETURN_STRING(WMI_CSA_OFFLOAD_CHANSWITCH_CMDID);

		/* Chatter commands */
		/* Change chatter mode of operation */
		WMI_RETURN_STRING(WMI_CHATTER_SET_MODE_CMDID);
		/* chatter add coalescing filter command */
		WMI_RETURN_STRING(WMI_CHATTER_ADD_COALESCING_FILTER_CMDID);
		/* chatter delete coalescing filter command */
		WMI_RETURN_STRING(WMI_CHATTER_DELETE_COALESCING_FILTER_CMDID);
		/* chatter coalecing query command */
		WMI_RETURN_STRING(WMI_CHATTER_COALESCING_QUERY_CMDID);

		/* addba specific commands */
		/* start the aggregation on this TID */
		WMI_RETURN_STRING(WMI_PEER_TID_ADDBA_CMDID);
		/* stop the aggregation on this TID */
		WMI_RETURN_STRING(WMI_PEER_TID_DELBA_CMDID);

		/* set station mimo powersave method */
		WMI_RETURN_STRING(WMI_STA_DTIM_PS_METHOD_CMDID);
		/* Configure the Station UAPSD AC Auto Trigger Parameters */
		WMI_RETURN_STRING(WMI_STA_UAPSD_AUTO_TRIG_CMDID);
		/* Configure the Keep Alive Parameters */
		WMI_RETURN_STRING(WMI_STA_KEEPALIVE_CMDID);

		/* Request ssn from target for a sta/tid pair */
		WMI_RETURN_STRING(WMI_BA_REQ_SSN_CMDID);
		/* misc command group */
		/* echo command mainly used for testing */
		WMI_RETURN_STRING(WMI_ECHO_CMDID);

		/* !!IMPORTANT!!
		* If you need to add a new WMI command to the
		* WMI_RETURN_STRING(WMI_GRP_MISC) sub-group,
		* please make sure you add it BEHIND
		* WMI_RETURN_STRING(WMI_PDEV_UTF_CMDID);
		* as we MUST have a fixed value here to maintain compatibility between
		* UTF and the ART2 driver
		*/
		/* UTF WMI commands */
		WMI_RETURN_STRING(WMI_PDEV_UTF_CMDID);

		/* set debug log config */
		WMI_RETURN_STRING(WMI_DBGLOG_CFG_CMDID);
		/* QVIT specific command id */
		WMI_RETURN_STRING(WMI_PDEV_QVIT_CMDID);
		/* Factory Testing Mode request command
		* used for integrated chipsets */
		WMI_RETURN_STRING(WMI_PDEV_FTM_INTG_CMDID);
		/* set and get keepalive parameters command */
		WMI_RETURN_STRING(WMI_VDEV_SET_KEEPALIVE_CMDID);
		WMI_RETURN_STRING(WMI_VDEV_GET_KEEPALIVE_CMDID);
		/* For fw recovery test command */
		WMI_RETURN_STRING(WMI_FORCE_FW_HANG_CMDID);
		/* Set Mcast/Bdcast filter */
		WMI_RETURN_STRING(WMI_SET_MCASTBCAST_FILTER_CMDID);
		/* set thermal management params */
		WMI_RETURN_STRING(WMI_THERMAL_MGMT_CMDID);
		WMI_RETURN_STRING(WMI_RSSI_BREACH_MONITOR_CONFIG_CMDID);
		WMI_RETURN_STRING(WMI_LRO_CONFIG_CMDID);
		WMI_RETURN_STRING(WMI_TRANSFER_DATA_TO_FLASH_CMDID);
		WMI_RETURN_STRING(WMI_CONFIG_ENHANCED_MCAST_FILTER_CMDID);
		WMI_RETURN_STRING(WMI_VDEV_WISA_CMDID);
		WMI_RETURN_STRING(WMI_SCAN_ADAPTIVE_DWELL_CONFIG_CMDID);
		WMI_RETURN_STRING(WMI_WOW_SET_ACTION_WAKE_UP_CMDID);
		WMI_RETURN_STRING(WMI_MAWC_SENSOR_REPORT_IND_CMDID);
		WMI_RETURN_STRING(WMI_ROAM_CONFIGURE_MAWC_CMDID);
		WMI_RETURN_STRING(WMI_NLO_CONFIGURE_MAWC_CMDID);
		WMI_RETURN_STRING(WMI_EXTSCAN_CONFIGURE_MAWC_CMDID);
		/* GPIO Configuration */
		WMI_RETURN_STRING(WMI_GPIO_CONFIG_CMDID);
		WMI_RETURN_STRING(WMI_GPIO_OUTPUT_CMDID);

		/* Txbf configuration command */
		WMI_RETURN_STRING(WMI_TXBF_CMDID);

		/* FWTEST Commands */
		WMI_RETURN_STRING(WMI_FWTEST_VDEV_MCC_SET_TBTT_MODE_CMDID);
		/* set NoA descs */
		WMI_RETURN_STRING(WMI_FWTEST_P2P_SET_NOA_PARAM_CMDID);

		/* TDLS Configuration */
		/* enable/disable TDLS */
		WMI_RETURN_STRING(WMI_TDLS_SET_STATE_CMDID);
		/* set tdls peer state */
		WMI_RETURN_STRING(WMI_TDLS_PEER_UPDATE_CMDID);

		/* Resmgr Configuration */
		/* Adaptive OCS is enabled by default in the FW.
		* This command is used to disable FW based adaptive OCS.
		*/
		WMI_RETURN_STRING
			(WMI_RESMGR_ADAPTIVE_OCS_ENABLE_DISABLE_CMDID);
		/* set the requested channel time quota for the home channels */
		WMI_RETURN_STRING(WMI_RESMGR_SET_CHAN_TIME_QUOTA_CMDID);
		/* set the requested latency for the home channels */
		WMI_RETURN_STRING(WMI_RESMGR_SET_CHAN_LATENCY_CMDID);

		/* STA SMPS Configuration */
		/* force SMPS mode */
		WMI_RETURN_STRING(WMI_STA_SMPS_FORCE_MODE_CMDID);
		/* set SMPS parameters */
		WMI_RETURN_STRING(WMI_STA_SMPS_PARAM_CMDID);

		/* Wlan HB commands */
		/* enalbe/disable wlan HB */
		WMI_RETURN_STRING(WMI_HB_SET_ENABLE_CMDID);
		/* set tcp parameters for wlan HB */
		WMI_RETURN_STRING(WMI_HB_SET_TCP_PARAMS_CMDID);
		/* set tcp pkt filter for wlan HB */
		WMI_RETURN_STRING(WMI_HB_SET_TCP_PKT_FILTER_CMDID);
		/* set udp parameters for wlan HB */
		WMI_RETURN_STRING(WMI_HB_SET_UDP_PARAMS_CMDID);
		/* set udp pkt filter for wlan HB */
		WMI_RETURN_STRING(WMI_HB_SET_UDP_PKT_FILTER_CMDID);

		/* Wlan RMC commands*/
		/* enable/disable RMC */
		WMI_RETURN_STRING(WMI_RMC_SET_MODE_CMDID);
		/* configure action frame period */
		WMI_RETURN_STRING(WMI_RMC_SET_ACTION_PERIOD_CMDID);
		/* For debug/future enhancement purposes only,
		* configures/finetunes RMC algorithms */
		WMI_RETURN_STRING(WMI_RMC_CONFIG_CMDID);

		/* WLAN MHF offload commands */
		/* enable/disable MHF offload */
		WMI_RETURN_STRING(WMI_MHF_OFFLOAD_SET_MODE_CMDID);
		/* Plumb routing table for MHF offload */
		WMI_RETURN_STRING(WMI_MHF_OFFLOAD_PLUMB_ROUTING_TBL_CMDID);

		/* location scan commands */
		/* start batch scan */
		WMI_RETURN_STRING(WMI_BATCH_SCAN_ENABLE_CMDID);
		/* stop batch scan */
		WMI_RETURN_STRING(WMI_BATCH_SCAN_DISABLE_CMDID);
		/* get batch scan result */
		WMI_RETURN_STRING(WMI_BATCH_SCAN_TRIGGER_RESULT_CMDID);
		/* OEM related cmd */
		WMI_RETURN_STRING(WMI_OEM_REQ_CMDID);
		WMI_RETURN_STRING(WMI_OEM_REQUEST_CMDID);
		/* NAN request cmd */
		WMI_RETURN_STRING(WMI_NAN_CMDID);
		/* Modem power state cmd */
		WMI_RETURN_STRING(WMI_MODEM_POWER_STATE_CMDID);
		WMI_RETURN_STRING(WMI_REQUEST_STATS_EXT_CMDID);
		WMI_RETURN_STRING(WMI_OBSS_SCAN_ENABLE_CMDID);
		WMI_RETURN_STRING(WMI_OBSS_SCAN_DISABLE_CMDID);
		WMI_RETURN_STRING(WMI_PEER_GET_ESTIMATED_LINKSPEED_CMDID);
		WMI_RETURN_STRING(WMI_ROAM_SCAN_CMD);
		WMI_RETURN_STRING(WMI_PDEV_SET_LED_CONFIG_CMDID);
		WMI_RETURN_STRING(WMI_HOST_AUTO_SHUTDOWN_CFG_CMDID);
		WMI_RETURN_STRING(WMI_CHAN_AVOID_UPDATE_CMDID);
		WMI_RETURN_STRING(WMI_COEX_CONFIG_CMDID);
		WMI_RETURN_STRING(WMI_WOW_IOAC_ADD_KEEPALIVE_CMDID);
		WMI_RETURN_STRING(WMI_WOW_IOAC_DEL_KEEPALIVE_CMDID);
		WMI_RETURN_STRING(WMI_WOW_IOAC_ADD_WAKE_PATTERN_CMDID);
		WMI_RETURN_STRING(WMI_WOW_IOAC_DEL_WAKE_PATTERN_CMDID);
		WMI_RETURN_STRING(WMI_REQUEST_LINK_STATS_CMDID);
		WMI_RETURN_STRING(WMI_START_LINK_STATS_CMDID);
		WMI_RETURN_STRING(WMI_CLEAR_LINK_STATS_CMDID);
		WMI_RETURN_STRING(WMI_GET_FW_MEM_DUMP_CMDID);
		WMI_RETURN_STRING(WMI_LPI_MGMT_SNOOPING_CONFIG_CMDID);
		WMI_RETURN_STRING(WMI_LPI_START_SCAN_CMDID);
		WMI_RETURN_STRING(WMI_LPI_STOP_SCAN_CMDID);
		WMI_RETURN_STRING(WMI_EXTSCAN_START_CMDID);
		WMI_RETURN_STRING(WMI_EXTSCAN_STOP_CMDID);
		WMI_RETURN_STRING(WMI_EXTSCAN_CONFIGURE_WLAN_CHANGE_MONITOR_CMDID);
		WMI_RETURN_STRING(WMI_EXTSCAN_CONFIGURE_HOTLIST_MONITOR_CMDID);
		WMI_RETURN_STRING(WMI_EXTSCAN_GET_CACHED_RESULTS_CMDID);
		WMI_RETURN_STRING(WMI_EXTSCAN_GET_WLAN_CHANGE_RESULTS_CMDID);
		WMI_RETURN_STRING(WMI_EXTSCAN_SET_CAPABILITIES_CMDID);
		WMI_RETURN_STRING(WMI_EXTSCAN_GET_CAPABILITIES_CMDID);
		WMI_RETURN_STRING(WMI_EXTSCAN_CONFIGURE_HOTLIST_SSID_MONITOR_CMDID);
		WMI_RETURN_STRING(WMI_ROAM_SYNCH_COMPLETE);
		WMI_RETURN_STRING(WMI_D0_WOW_ENABLE_DISABLE_CMDID);
		WMI_RETURN_STRING(WMI_EXTWOW_ENABLE_CMDID);
		WMI_RETURN_STRING(WMI_EXTWOW_SET_APP_TYPE1_PARAMS_CMDID);
		WMI_RETURN_STRING(WMI_EXTWOW_SET_APP_TYPE2_PARAMS_CMDID);
		WMI_RETURN_STRING(WMI_UNIT_TEST_CMDID);
		WMI_RETURN_STRING(WMI_ROAM_SET_RIC_REQUEST_CMDID);
		WMI_RETURN_STRING(WMI_PDEV_GET_TEMPERATURE_CMDID);
		WMI_RETURN_STRING(WMI_SET_DHCP_SERVER_OFFLOAD_CMDID);
		WMI_RETURN_STRING(WMI_TPC_CHAINMASK_CONFIG_CMDID);
		WMI_RETURN_STRING(WMI_IPA_OFFLOAD_E