/*  exception.c  - arm64 specific part of Exception Monitor
 *
 * Copyright 2018 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */
#include <linux/version.h>
#include <linux/interrupt.h>
#include <linux/mm.h>
#include <linux/fs.h>
#include <linux/sched/task_stack.h>
#include <linux/slab.h>
#include <linux/elf.h>
#include <linux/uio.h>
#include <linux/exception.h>

#include <asm/uaccess.h>
#include <asm/esr.h>
#include <asm/exception_monitor.h>

#ifdef DEBUG
#define dbg(fmt, argv...) em_dump_write(fmt, ##argv)
#else
#define dbg(fmt, argv...) do{}while(0)
#endif

#define ARRAY_NUM(x) (sizeof(x)/sizeof((x)[0]))

extern struct pt_regs *em_regs;
extern unsigned int em_errcode;
extern int not_interrupt;

/*
 * indicate if fp supported
 */
static int fp_valid;
/*
 * for callstack
 */
#define LIB_MAX 100
#define LIB_NAME_SIZE 64
static char libname[LIB_MAX][LIB_NAME_SIZE];

//#define ELF_INFO_MAX (1 + ARRAY_NUM(target_module) + LIB_MAX)
#define ELF_INFO_MAX (LIB_MAX + 2)
static struct _elf_info elf_info[ELF_INFO_MAX];


#define MAX_CALLSTACK    100
static struct callstack callstack[MAX_CALLSTACK + 2];


static ssize_t em_read_file(struct file *filp, char *buf, size_t len)
{
	struct iovec iov = { .iov_base = (void __user *)buf, .iov_len = len };
	struct kiocb kiocb;
	struct iov_iter iter;
	ssize_t ret;

	if (filp->f_op->read) {
		return filp->f_op->read(filp, (char __user *)buf, len, &filp->f_pos);
	}
	init_sync_kiocb(&kiocb, filp);
	kiocb.ki_pos = filp->f_pos;
	iov_iter_init(&iter, READ, &iov, 1, len);
	ret = filp->f_op->read_iter(&kiocb, &iter);
	filp->f_pos = kiocb.ki_pos;
	return ret;
}

static void em_get_entry_name(struct file *fp, Elf64_Off offset, char *str)
{
	fp->f_op->llseek(fp, offset, 0);
	em_read_file(fp, str, CALLSTACK_STR_SZ);
	str[CALLSTACK_STR_SZ] = '\0';
}

static int
em_get_entry_near(unsigned int index, Elf64_Shdr * sym, Elf64_Shdr * str,
		  struct callstack *cs)
{
	int n, j;
	struct file *fp;
	Elf64_Sym ent;
	Elf64_Off offset = 0x0;
	Elf64_Addr value = 0x0;
	Elf64_Word name = 0x0;
	unsigned long addr_base;
	int flag = 0;

	fp = elf_info[index].file;
	n = sym->sh_size / sizeof(Elf64_Sym);
	addr_base = (index) ? elf_info[index].addr_offset : 0;

	if (fp->f_op->llseek(fp, sym->sh_offset, 0) < 0)
		return 0;
	for (j = 0; j < n; j++) {
		if (fp->f_op->llseek(fp, sym->sh_offset + j*sizeof(Elf64_Sym), 0) < 0)
			break;
		if (em_read_file(fp, (char *)&ent, sizeof(Elf64_Sym)) != sizeof(Elf64_Sym))
			break;
		if ((ELF_ST_TYPE(ent.st_info) == STT_FUNC) &&
		    ((ent.st_value + addr_base) <= cs->caller) &&
		    ((ent.st_value + addr_base) >= value)) {
			cs->elf_info = & elf_info[index];
			cs->entry = ent.st_value + addr_base;
			cs->size = ent.st_size;

			offset = str->sh_offset;
			name = ent.st_name;
			em_get_entry_name(fp, offset + name, cs->entry_str);
			if(cs->entry_str[0] != '$'){
				value = ent.st_value + addr_base;
				flag = 1;
			}
		}
	}

	if (!flag) {
		return 0;
	}

	return 1;
}

static void em_find_symbol_near(int elf_cnt, struct callstack *cs)
{
	int i;
	int len ;

	if (cs->caller == 0x0)
		return;

	for (i = 0; i < elf_cnt; i++) {
		if (elf_info[i].file == NULL)
			continue;

		if (elf_info[i].addr_offset > cs->caller ||
		    elf_info[i].addr_end < cs->caller) {
			continue;
		}
		if (elf_info[i].strip) {
			strncpy(cs->entry_str, "## Stripped", sizeof(cs->entry_str)/sizeof(cs->entry_str[0]) - 1);
			cs->entry_str[sizeof(cs->entry_str)/sizeof(cs->entry_str[0]) - 1]='\0';
			return;
		}
		if (em_get_entry_near
		    (i, &elf_info[i].sh_symtab, &elf_info[i].sh_strtab, cs)) {
			goto found;
		}

		if (em_get_entry_near
		    (i, &elf_info[i].sh_dynsym, &elf_info[i].sh_dynstr, cs)) {
			goto found;
		}

	}

	/* cs->entry = 0x0; */
	strncpy(cs->entry_str, "## Unknown", sizeof(cs->entry_str)/sizeof(cs->entry_str[0]) - 1);
	cs->entry_str[sizeof(cs->entry_str)/sizeof(cs->entry_str[0]) - 1] = '\0';
	return;

      found:
	len = CALLSTACK_STR_SZ - strlen(cs->entry_str);
	if (fp_valid)
		strncat(cs->entry_str, "  ## GUESS", len);
}

static int
em_get_entry(unsigned int index, Elf64_Shdr * sym, Elf64_Shdr * str,
	     struct callstack *cs)
{
	int n, j;
	struct file *fp;
	Elf64_Sym ent;
	Elf64_Off offset = 0x0;
	Elf64_Word name = 0x0;
	unsigned long addr_base;

	fp = elf_info[index].file;
	n = sym->sh_size / sizeof(Elf64_Sym);
	addr_base = (index) ? elf_info[index].addr_offset : 0;

	if (fp->f_op->llseek(fp, sym->sh_offset, 0) < 0)
		return 0;
	for (j = 0; j < n; j++) {
		if (fp->f_op->llseek(fp, sym->sh_offset + j*sizeof(Elf64_Sym), 0) < 0)
			break;
		if (em_read_file(fp, (char *)&ent, sizeof(Elf64_Sym)) != sizeof(Elf64_Sym))
			break;

		if (ELF_ST_TYPE(ent.st_info) == STT_FUNC) {
			if((ent.st_value + addr_base) == cs->entry){
				offset = str->sh_offset;
				name = ent.st_name;
				em_get_entry_name(fp, offset + name, cs->entry_str);
				if(cs->entry_str[0] != '$') {
					cs->elf_info = &elf_info[index];
					goto found;
				}
			}
		}
	}
	return 0;

      found:
	return 1;
}

static int get_kernel_entry(unsigned long pc, struct callstack *cs);
static void em_get_entry_str(int elf_cnt, struct callstack *cs)
{
	int i;

	if (cs->entry == 0x0)
		goto fuzzy;

	if(get_kernel_entry(cs->entry, cs))
		return;

	for (i = 0; i < elf_cnt; i++) {
		if (elf_info[i].file == NULL)
			continue;

		if (elf_info[i].addr_offset > cs->caller ||
		    elf_info[i].addr_end < cs->caller) {
			continue;
		}

		if (elf_info[i].addr_offset > cs->entry ||
		    elf_info[i].addr_end < cs->entry) {
			continue;
		}

		if (em_get_entry
		    (i, &elf_info[i].sh_symtab, &elf_info[i].sh_strtab, cs)) {
			return;
		}
		if (em_get_entry
		    (i, &elf_info[i].sh_dynsym, &elf_info[i].sh_dynstr, cs)) {
			return;
		}
	}

fuzzy:
	/* did not find entry */
	em_find_symbol_near(elf_cnt, cs);
	return;
}

static void em_close_kernel_nm(void)
{
}

static void em_open_kernel_nm(void)
{
}

static int get_kernel_entry(unsigned long pc, struct callstack *cs)
{
	return 0;
}

static void em_open_modules(int *elf_cnt)
{
	*elf_cnt = 1;
}

static void em_close_elffile(unsigned int index)
{
	if (elf_info[index].file) {
		filp_close(elf_info[index].file, NULL);
		elf_info[index].file = NULL;
	}
}

static void em_close_elffiles(int elf_cnt)
{
	int i;

	for (i = 0; i < elf_cnt; i++) {
		em_close_elffile(i);
	}

	em_close_kernel_nm();
}

static int em_open_elffile(unsigned int index)
{
	int i;
	int strip = 2;
	Elf64_Ehdr ehdr;
	Elf64_Shdr shdr;
	Elf64_Shdr sh_shstrtab;
	char *shstrtab;
	struct file *fp;

	/*
	 * open elf file
	 */
	elf_info[index].file =
	    filp_open(elf_info[index].filename, O_RDONLY, 0444);

	if (IS_ERR(elf_info[index].file)) {
		elf_info[index].file = NULL;
		goto fail;
	}
	fp = elf_info[index].file;

	if (!fp->f_op || !fp->f_op->llseek
	    || (!fp->f_op->read && !fp->f_op->read_iter))
		goto close_fail;

	/*
	 * read elf header
	 */
	em_read_file(fp, (char *)&ehdr, sizeof(ehdr));
	if (memcmp(ehdr.e_ident, ELFMAG, SELFMAG) != 0)
		goto close_fail;
	if (!elf_check_arch(&ehdr))
		goto close_fail;

	/*
	 * read section header table
	 */
	fp->f_op->llseek(fp,
			 ehdr.e_shoff + sizeof(Elf64_Shdr) * ehdr.e_shstrndx,
			 0);
	em_read_file(fp, (char *)&sh_shstrtab, sizeof(sh_shstrtab));

	shstrtab = (char *)kmalloc(sh_shstrtab.sh_size, GFP_ATOMIC);
	if(shstrtab == NULL){
		goto fail;
	}

	fp->f_op->llseek(fp, sh_shstrtab.sh_offset, 0);
	em_read_file(fp, shstrtab, sh_shstrtab.sh_size);

	/*
	 * read shsymtab
	 */
	fp->f_op->llseek(fp, ehdr.e_shoff, 0);
	for (i = 0; i < ehdr.e_shnum; i++) {
		em_read_file(fp, (char *)&shdr, sizeof(shdr));
		if (strcmp(&shstrtab[shdr.sh_name], ".dynsym") == 0)
			elf_info[index].sh_dynsym = shdr;
		else if (strcmp(&shstrtab[shdr.sh_name], ".dynstr") == 0)
			elf_info[index].sh_dynstr = shdr;
		else if (strcmp(&shstrtab[shdr.sh_name], ".symtab") == 0) {
			elf_info[index].sh_symtab = shdr;
			strip--;
		} else if (strcmp(&shstrtab[shdr.sh_name], ".strtab") == 0) {
			elf_info[index].sh_strtab = shdr;
			strip--;
		}
	}

	if (!strip)
		elf_info[index].strip = strip;

	kfree(shstrtab);
	return 0;

      close_fail:
	em_close_elffile(index);
      fail:
	return -1;
}

static void init_struct_callstack(void)
{
	int i;

	for (i = 0; i < MAX_CALLSTACK; i++) {
		callstack[i].entry = 0x0;
		callstack[i].caller = 0x0;
		callstack[i].size = 0x0;
		callstack[i].elf_info = NULL;
		callstack[i].entry_str[0] = '\0';
	}
}

static void init_struct_elfinfo(void)
{
	int i;

	for (i = 0; i < ELF_INFO_MAX; i++) {
		elf_info[i].filename = NULL;
		elf_info[i].sh_dynsym.sh_size = 0;
		elf_info[i].sh_dynstr.sh_size = 0;
		elf_info[i].sh_symtab.sh_size = 0;
		elf_info[i].sh_strtab.sh_size = 0;
		elf_info[i].addr_offset = 0;
		elf_info[i].addr_end = 0;
		elf_info[i].strip = 1;
	}

}

static int em_open_elffiles(void)
{
	char *path;
	static char buf[LIB_NAME_SIZE];
	struct vm_area_struct *vm = NULL;
	char *short_name;
	int elf_cnt = 0;
	int i;

	/*
	 * initialize
	 */
	init_struct_callstack();
	init_struct_elfinfo();

	if (!not_interrupt)
		goto out;

	if (test_tsk_thread_flag(current, TIF_MEMDIE))
		goto out;

	em_open_kernel_nm();

	/*
	 * set elf_info
	 */
	elf_info[0].filename = em_get_execname();
	short_name = elf_info[0].filename;
	for (i = 0; elf_info[0].filename[i]; i++)
		if (elf_info[0].filename[i] == '/')
			short_name = &elf_info[0].filename[i + 1];
	if (current->mm != NULL)
		vm = current->mm->mmap;
	for (; vm != NULL; vm = vm->vm_next) {
		if ((vm->vm_flags & (VM_READ | VM_EXEC)) != (VM_READ | VM_EXEC))
			continue;
		if (vm->vm_flags & VM_WRITE)
			continue;
		if (vm->vm_file == NULL)
			continue;
		if (file_dentry(vm->vm_file)) {
			if (strcmp
			    (file_dentry(vm->vm_file)->d_name.name,
			     short_name) == 0) {
				elf_info[0].addr_offset = vm->vm_start;
				elf_info[0].addr_end = vm->vm_end;
			}
		}
	}

	em_open_modules(&elf_cnt);

	if (current->mm != NULL)
		vm = current->mm->mmap;
	for (i = 0; i < ARRAY_NUM(libname) && vm != NULL; vm = vm->vm_next) {
		if ((vm->vm_flags & (VM_READ | VM_EXEC)) != (VM_READ | VM_EXEC))
			continue;
		if (vm->vm_flags & VM_WRITE)	/* assume text is r-x and text
						   seg addr is base addr */
			continue;
#ifdef VM_EXECUTABLE
		if (vm->vm_flags & VM_EXECUTABLE)
			continue;
#endif
		if (vm->vm_file == NULL)
			continue;

		path = d_path(&vm->vm_file->f_path, buf, sizeof(buf));
		buf[sizeof(buf) - 1] = '\0';

		if (strncmp(path, "/lib/ld-linux-aarch64.so.1", sizeof(buf)) == 0)
			continue;

		strncpy(libname[i], path, LIB_NAME_SIZE);
		libname[i][LIB_NAME_SIZE - 1] = 0;

		elf_info[elf_cnt].filename = libname[i];
		elf_info[elf_cnt].addr_offset = vm->vm_start;
		elf_info[elf_cnt].addr_end = vm->vm_end;
		elf_cnt++;
		i++;
	}

	for (i = 0; i < elf_cnt; i++) {
		if (em_open_elffile(i) == -1)
			em_dump_write("\n\tWARNING: file not found: %s\n",
				      elf_info[i].filename);

		dbg("file : %s (%08lx %08lx)\n",
			 elf_info[i].filename, elf_info[i].addr_offset, elf_info[i].addr_end);
	}

out:
	return elf_cnt;
}

static inline ulong em_get_user(ulong *p)
{
	ulong v;

	if (__get_kuser(v, (ulong __user *)p))
		v = 0;

	return v;
}

static inline ulong em_put_user(ulong v, ulong *p)
{
	return __put_user(v, p);
}

static inline ulong arch_stack_pointer(ulong *frame)
{
	if (!frame)
		return 0UL;
	return em_get_user(frame);
}

static inline ulong arch_caller_address(ulong *frame)
{
	if (!frame)
		return 0UL;
	return em_get_user(frame + 1);
}

static inline ulong *arch_prev_frame(ulong *frame)
{
	if (!frame)
		return NULL;
	return (ulong *)em_get_user(frame);
}

#define GET_IP(r)	(instruction_pointer(r))
#define GET_LR(r)	((r)->regs[30])
#define GET_FP(r)	(frame_pointer(r))
#define GET_USP(r)	(user_stack_pointer(r))

static int em_get_callstack_fp(int elf_cnt, int no, int use_lr)
{
	int init;
	int i = no;
	ulong *fp = (ulong *)GET_FP(em_regs);
	ulong *fp_prev;

	/*
	 * set callstack
	 */
	init = 1;
	while (i < MAX_CALLSTACK) {
		switch (init) {
		case 1: /* examine PC */
			init++;
			if (!GET_IP(em_regs)) {
				use_lr = 1;
				continue;
			}
			callstack[i].caller = GET_IP(em_regs);
			callstack[i].entry = 0;
			break;
		case 2: /* examine LR */
			init++;
			if (!use_lr || !GET_LR(em_regs))
				continue;
			if (GET_LR(em_regs) == arch_caller_address(fp))
				continue;
			callstack[i].caller = GET_LR(em_regs);
			callstack[i].entry = 0;
			break;
		default: /* examin stack frame */
			init = 0;
			callstack[i].caller = arch_caller_address(fp);
			callstack[i].entry = 0;
			break;
		}

		if (not_interrupt)
			em_get_entry_str(elf_cnt, &callstack[i]);

		i ++;
		if (init) {
			continue;
		}
		fp_prev = fp;
		fp = arch_prev_frame(fp);
		if (!fp)
			break;
		if (fp == fp_prev)
			break;
	}

	return i;
}

#define DUMP_USER(regs) !user_mode(regs) && current->mm
#define USER_REGISTER task_pt_regs(current)

struct callstack *em_get_callstack(void)
{
	int elf_cnt;
	int no;

	elf_cnt = em_open_elffiles();

	fp_valid = 1;
	no = em_get_callstack_fp(elf_cnt, 0, 0);

	if (DUMP_USER(em_regs)) {
		struct pt_regs *em_regs_tmp = em_regs;
		callstack[no++].caller = CALLER_USER;
		em_regs = USER_REGISTER;
		no = em_get_callstack_fp(elf_cnt, no, 1);
		em_regs = em_regs_tmp;
	}

	em_close_elffiles(elf_cnt);

	callstack[no].caller = CALLER_END;

	return callstack;
}

static void em_dump_user(void (func)(int, char **), int argc, char **argv, const char *msg)
{
	struct pt_regs *em_regs_tmp = em_regs;
	em_regs = USER_REGISTER;
	em_dump_write("\n--------------- user %s dump ----------------", msg);
	func(argc, argv);
	em_dump_write("---------------------------------------------------\n");
	em_regs = em_regs_tmp;
}

static void em_do_dump_regs(int argc, char **argv)
{
	int i;
	char *mode;
	static char mode_list[][4] = { "EL0", "EL1", "???" };
	static char regnam[4];

	em_dump_write("\n[register dump]\n");

	for (i = 0; i < 29; i++) {
		scnprintf(regnam, sizeof regnam, "r%d", i);
		em_dump_write("%3s: 0x%016lx  ", regnam, em_regs->regs[i]);
		if ((i % 2) == 1)
			em_dump_write("\n");
	}
	em_dump_write(" fp: 0x%016lx   \n", GET_FP(em_regs));
	em_dump_write(" lr: 0x%016lx   sp: 0x%016lx\n",
		      em_regs->regs[30], GET_USP(em_regs));
	em_dump_write(" pc: 0x%016lx\n", GET_IP(em_regs));

	switch (processor_mode(em_regs)) {
	case PSR_MODE_EL0t: mode = mode_list[0]; break;
	case PSR_MODE_EL1t: mode = mode_list[1]; break;
	case PSR_MODE_EL1h: mode = mode_list[1]; break;
	default: mode = mode_list[2]; break;
	}
	em_dump_write("pstate: 0x%08lx: Flags: %c%c%c%c, "
		      "IRQ: o%s, FIQ: o%s, Mode: %s\n",
		      em_regs->pstate,
		      (em_regs->pstate & PSR_N_BIT) ? 'N' : 'n',
		      (em_regs->pstate & PSR_Z_BIT) ? 'Z' : 'z',
		      (em_regs->pstate & PSR_C_BIT) ? 'C' : 'c',
		      (em_regs->pstate & PSR_V_BIT) ? 'V' : 'v',
		      (em_regs->pstate & PSR_I_BIT) ? "ff" : "n",
		      (em_regs->pstate & PSR_F_BIT) ? "ff" : "n",
		      mode);
}

void em_dump_regs(int argc, char **argv)
{
	em_do_dump_regs(argc, argv);

	if (DUMP_USER(em_regs))
		em_dump_user(em_do_dump_regs, argc, argv, "register");
}

#define ISS_VALID (1 << 24)
#define ISS_MASK  (0x00ffffff)
#define ASID_SHIFT 48
#define ASID_MASK  (0xffffULL << ASID_SHIFT)

void em_dump_regs_detail(int argc, char **argv)
{
	u32 esr, iss, isr, sctlr, contxt, cpacr;
	u64 mpidr, far, vbar;
	u64 ttbr0, ttbr1, tcr, mair;
	u16 asid0, asid1;
	u64 tpidr_el0, tpidr_el1, tpidrro_el0;

	em_dump_regs(1, NULL);
	em_dump_write("\n[system register dump]\n\n");

	mpidr = read_sysreg(mpidr_el1);
	mpidr &= 0xff;
	sctlr = read_sysreg(sctlr_el1);

	/* Exception */
	esr = read_sysreg(esr_el1);
	iss = esr & ISS_MASK;
	far = read_sysreg(far_el1);
	isr = read_sysreg(isr_el1);
	vbar = read_sysreg(vbar_el1);

	/* MMU */
	tcr = read_sysreg(tcr_el1);
	ttbr0 = read_sysreg(ttbr0_el1);
	asid0 = ttbr0 >> ASID_SHIFT;
	ttbr0 &= ~ASID_MASK;
	ttbr1 = read_sysreg(ttbr1_el1);
	asid1 = ttbr1 >> ASID_SHIFT;
	ttbr1 &= ~ASID_MASK;
	mair = read_sysreg(mair_el1);

	contxt = read_sysreg(contextidr_el1);
	cpacr = read_sysreg(cpacr_el1);
	tpidr_el1 = read_sysreg(tpidr_el1);
	tpidr_el0 = read_sysreg(tpidr_el0);
	tpidrro_el0 = read_sysreg(tpidrro_el0);

	em_dump_write("*                Core ID: %x\n", mpidr);
	em_dump_write("*         System Control: %08x\n", sctlr);
	em_dump_write("*     Exception Syndrome: %08x, %s\n",
		      esr, esr_get_class_string(esr));
	if (esr & ISS_VALID) {
		em_dump_write("*                         %06x\n", iss);
	}

	em_dump_write("*          Fault Address: %016lx\n", far);
	em_dump_write("*       Interrupt Status: %08x\n", isr);
	em_dump_write("*            Vector Base: %016lx\n", vbar);
	em_dump_write("* Translation Table Base: %04x,%012lx\n", asid0, ttbr0);
	em_dump_write("*                         %04x,%012lx\n", asid1, ttbr1);
	em_dump_write("*    Translation Control: %016lx\n", tcr);
	em_dump_write("*       Memory Attribute: %016lx\n", mair);
	em_dump_write("*             Context ID: %08x\n", contxt);
	em_dump_write("*              Thread ID: %016lx(EL0)\n", tpidr_el0);
	em_dump_write("*                         %016lx(EL0,RO)\n",tpidrro_el0);
	em_dump_write("*                         %016lx(EL1)\n", tpidr_el1);
	em_dump_write("\n");
}

#define EM_USERSTACK_MAXDUMP	4 /* pages */

static void em_dump_till_end_of_page(unsigned long *sp, int user)
{
	unsigned long *tail, *sp_orig = sp;
	unsigned long stackdata;

	if (!user) {
		unsigned long stack_size, stack_page;

		stack_size = THREAD_SIZE;
		stack_page = (unsigned long)sp & ~(stack_size - 1);
		tail = (unsigned long *)(stack_page + stack_size);
	} else {
		tail = (unsigned long *)(((unsigned long)sp & PAGE_MASK) + PAGE_SIZE*EM_USERSTACK_MAXDUMP);
	}

	while (sp < tail) {
		if (__get_kuser(stackdata, (ulong __user *)sp)) {
			em_dump_write("\n (bad stack address)\n");
			break;
		}

		if (((unsigned long)sp-(unsigned long)sp_orig) % 0x10 == 0) {
			em_dump_write("\n0x%016lx: ", sp);
		}

		em_dump_write("0x%016lx ", stackdata);

		sp++;
	}
}

static void em_do_dump_stack(int argc, char **argv)
{
	unsigned long *sp = (unsigned long *)(em_regs->sp & ~0x0f);
	unsigned long *fp = (unsigned long *)(em_regs->regs[29] & ~0x0f);
	unsigned long *tail;
	unsigned long backchain;
	unsigned long stackdata;
	int frame = 1;

	tail = sp + PAGE_SIZE / 4;

	em_dump_write("\n[stack dump]\n");

	backchain = arch_stack_pointer(fp);
	while (sp < tail) {
		if (backchain == (unsigned long)sp) {
			em_dump_write("|");
			fp = arch_prev_frame(fp);
			if (!fp)
				break;

			backchain = arch_stack_pointer(fp);
			if (!backchain)
				break;
		} else {
			em_dump_write(" ");
		}

		if (backchain < (unsigned long)sp) {
			break;
		}

		if (__get_kuser(stackdata, (ulong __user *)sp)) {
			em_dump_write("\n (bad stack address)\n");
			break;
		}

		if (((unsigned long)tail-(unsigned long)sp) % 0x10 == 0) {
			if (frame) {
				em_dump_write("\n0x%016lx:|", sp);
				frame = 0;
			} else {
				em_dump_write("\n0x%016lx: ", sp);
			}
		}

		em_dump_write("0x%016lx", stackdata);

		sp++;
	}

	em_dump_write("\n");

	em_dump_write("\n #################em_dump_till_end_of_page###########\n");
	em_dump_till_end_of_page(sp, user_mode(em_regs));
	em_dump_write("\n");
}

void em_dump_stack(int argc, char **argv)
{
	em_do_dump_stack(argc, argv);

	if (DUMP_USER(em_regs))
		em_dump_user(em_do_dump_stack, argc, argv, "stack");
}

void em_show_syndrome(void)
{
	unsigned long esr,far;
	struct task_struct *tsk = current;

	em_dump_write("\n\n[Exception Syndrome]\n");

	if (user_mode(em_regs)) {
		switch (em_errcode) {
		case SIGILL:
			em_dump_write("Illegal Instruction (PC=0x%lx)\n",
				      GET_IP(em_regs));
			break;
		case SIGABRT:
			em_dump_write("Abort (PC=0x%lx)\n", GET_IP(em_regs));
			break;
		case SIGSEGV:
			em_dump_write("SEGV (PC=0x%lx)\n", GET_IP(em_regs));
			break;
		case 0:
			break;
		default:
			em_dump_write("signal %d\n", em_errcode);
			break;
		}
	}
	esr = tsk->thread.fault_code;
	far = tsk->thread.fault_address;
	if (!user_mode(em_regs)) {
		esr = em_errcode;
	}
	em_dump_write("%s (0x%08lx) at 0x%lx\n",
		      esr_get_class_string(esr), esr, far);
}

static unsigned long cpu_v2p(unsigned long v)
{
	unsigned long p;

	asm volatile("at s1e1r, %0" ::"r" (v));
	isb();
	p = read_sysreg(par_el1);
	if (p & 1)
		return -1;
	return p;
}

static unsigned long pgd_v2p(unsigned long v)
{
	pgd_t *pgd;
	p4d_t *p4d;
	pud_t *pud;
	pmd_t *pmd;
        pte_t *pte;

	if (v >= TASK_SIZE) {
		pgd = pgd_offset_k(v);
	}
	else {
		pgd = pgd_offset(current->active_mm, v);
	}
	if (pgd_none(*pgd))
		return -1;

	p4d = p4d_offset(pgd, v);
	if (p4d_none(*p4d))
		return -1;

	pud = pud_offset(p4d, v);
	if (pud_none(*pud))
		return -1;

	pmd = pmd_offset(pud, v);
	if (pmd_none(*pmd))
		return -1;
	if (pmd_sect(*pmd))
		return (pmd_pfn(*pmd) << PAGE_SHIFT) | (v & ~PMD_MASK);

	pte = pte_offset_kernel(pmd, v);
	if (pte_none(*pte))
		return -1;
	return (pte_pfn(*pte) << PAGE_SHIFT) | (v & ~(PAGE_MASK));
}

void em_cmd_v2p(int argc, char **argv)
{
	unsigned long addr;

	if (argc != 2)
		return;

	addr = simple_strtoul(argv[1], NULL, 16);
	em_dump_write("virt:0x%016lx \n\t-> phys(cpu):0x%016lx \n\t-> phys(pgd):0x%016lx\n", addr, cpu_v2p(addr), pgd_v2p(addr));
}

void em_dump_current_task(int argc, char **argv)
{
	em_dump_write("\n[current task]\n");
	em_dump_write("program: %s(%s) (pid: %d, cpu:%d, tid: %d, stackpage: %px)\n",
		      em_get_execname(), current->comm, current->pid,
		      raw_smp_processor_id(),
#ifdef CONFIG_SUBSYSTEM
	              current->itron_tid,
#else
		      -1,
#endif
		      current);
	em_dump_write("address: %016lx, error_code: %016lx\n",
		      current->thread.fault_address,
		      current->thread.fault_code);
}
