// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright 2022 Sony Corporation, SOCIONEXT INC.
 *
 */
#include <linux/types.h>
#include <linux/scu-clock.h>
#include <linux/clk-provider.h>
#include <linux/device.h>
#include <linux/io.h>
#include <linux/iopoll.h>

#include "clk-lin.h"

#define GCRG_DIVMAX		255

#define LCRG_DIV1		0
#define LCRG_DIV2		1
#define LCRG_DIV4		2
#define LCRG_DIV8		3
#define LCRG_DIV16		4
#define LCRG_DIV32		5
#define LCRG_DIV64		6

#define XCRG_ID_MASK		0x3f000
#define XCRG_ID_SHIFT		12

enum E_CRG_TYPE {
	E_CRG_TYPE_GCRG = 0,
	E_CRG_TYPE_LCRG,
};

static raw_spinlock_t xcrg_lock[NUMBER_OF_XCRG_INSTANCE];

struct xcrg_data {
	u32 div_mask;
	u32 kick_reg;
	u32 kick_val;
	u32 confi_val;
};

const struct xcrg_data xcrg[] = {
	{
		ODIVCHNUM_MASK,		//div_mask
		GCRGCMDEN,		//kick_reg
		CMDOCLK,		//kick_val
		CMDNOP,			//confi_val
	},
	{
		LDIVRC_MASK,		//div_mask
		LCRUPC,			//kick_reg
		UPC_RUNNING,		//kick_val
		UPC_NOT_RUNNING,	//confi_val
	}
};

struct lin_clk_div {
	struct clk_hw hw;
	void __iomem *base;
	u32 reg;
	u32 bit;
	u32 width;
	u8 flags;
	enum E_CRG_TYPE type;
};

#define to_lin_clk_div(_hw)	container_of(_hw, struct lin_clk_div, hw)

static unsigned long lin_divider_recalc_rate(struct clk_hw *hw,
					     unsigned long parent_rate)
{
	struct lin_clk_div *divider = to_lin_clk_div(hw);
	u32 val;

#ifdef CONFIG_CLOCK_FPGA_ENV
	enum E_CRG_TYPE type = divider->type;
	u32 xcrg_id;
	u32 div_ch_offs[] = {DIV_CH_OFS_GCRG, DIV_CH_OFS_LCRG};
	u32 div_ch;

	xcrg_id = (divider->reg & XCRG_ID_MASK) >> XCRG_ID_SHIFT;
	div_ch = ((divider->reg & DIV_CH_MASK) - div_ch_offs[type])
		 >> DIV_CH_SHIFT;
	val = g_fpga_env_divider[xcrg_id][div_ch] >> divider->bit;
#else
	val = readl(divider->base + divider->reg) >> divider->bit;
#endif
	val &= clk_div_mask(divider->width);

	return divider_recalc_rate(hw, parent_rate, val, NULL,
				   divider->flags, divider->width);
}

static long lin_divider_round_rate(struct clk_hw *hw, unsigned long rate,
				unsigned long *prate)
{
	struct lin_clk_div *divider = to_lin_clk_div(hw);

	/* if read only, just return current value */
	if (divider->flags & CLK_DIVIDER_READ_ONLY) {
		u32 val;
#ifdef CONFIG_CLOCK_FPGA_ENV
		enum E_CRG_TYPE type = divider->type;
		u32 xcrg_id;
		u32 div_ch_offs[] = {DIV_CH_OFS_GCRG, DIV_CH_OFS_LCRG};
		u32 div_ch;

		xcrg_id = (divider->reg & XCRG_ID_MASK) >> XCRG_ID_SHIFT;
		div_ch = ((divider->reg & DIV_CH_MASK) - div_ch_offs[type])
			  >> DIV_CH_SHIFT;
		val = g_fpga_env_divider[xcrg_id][div_ch] >> divider->bit;
#else
		val = readl(divider->base + divider->reg)
		      >> divider->bit;
#endif
		val &= clk_div_mask(divider->width);

		return divider_ro_round_rate(hw, rate, prate, NULL,
					     divider->width, divider->flags,
					     val);
	}

	return divider_round_rate(hw, rate, prate, NULL,
				  divider->width, divider->flags);
}

static int set_rate_xcrg(struct lin_clk_div *divider, int value)
{
	int ret = 0;
	u32 val, xcrg_id;
	unsigned long flags = 0;
	enum E_CRG_TYPE type = divider->type;
	u32 start = (divider->reg & XCRG_ID_MASK) + xcrg[type].kick_reg;
#ifdef CONFIG_CLOCK_FPGA_ENV
	//Skip polling
	u32 div_ch_offs[] = {DIV_CH_OFS_GCRG, DIV_CH_OFS_LCRG};
	u32 div_ch;
	(void)start;
#endif
	xcrg_id = (divider->reg & XCRG_ID_MASK) >> XCRG_ID_SHIFT;
#ifdef TESTCODE_LIN_CLOCK
	SCU_time_mesure_start();
#endif
	raw_spin_lock_irqsave(&xcrg_lock[xcrg_id], flags);

#ifdef CONFIG_CLOCK_FPGA_ENV
	div_ch = ((divider->reg & DIV_CH_MASK) - div_ch_offs[type])
		 >> DIV_CH_SHIFT;
	val = g_fpga_env_divider[xcrg_id][div_ch] & (~xcrg[type].div_mask);
#else
	val = readl(divider->base + start);
	if (val != xcrg[type].confi_val) {
		ret = -EBUSY;
		goto timeout;
	}
	val = readl(divider->base + divider->reg)
	      & (~xcrg[type].div_mask);
#endif
	val |= value;
#ifdef CONFIG_CLOCK_FPGA_ENV
	g_fpga_env_divider[xcrg_id][div_ch] = val;
#else
	writel(val, divider->base + divider->reg);
	writel(xcrg[type].kick_val, divider->base + start);
#endif
#ifndef CONFIG_CLOCK_FPGA_ENV
	mb(); // need before polling
	if (readl_poll_timeout_atomic((divider->base + start), val,
				       val == xcrg[type].confi_val,
				       POLL_INTERVAL_US, POLL_TIMEOUT_US)) {
		ret = -ETIMEDOUT;
		goto timeout;
	}
timeout:
#endif
	raw_spin_unlock_irqrestore(&xcrg_lock[xcrg_id], flags);
#ifdef TESTCODE_LIN_CLOCK
	SCU_time_mesure_stop();
#endif
	return ret;
}


static int lin_divider_set_rate(struct clk_hw *hw, unsigned long rate,
				unsigned long parent_rate)
{
	struct lin_clk_div *divider = to_lin_clk_div(hw);
	int value, ret;

	value = divider_get_val(rate, parent_rate, NULL,
				divider->width, divider->flags);
	if (value < 0)
		return value;
	ret = set_rate_xcrg(divider, value);
	return ret;
}

void lin_clk_div_init_locks(void)
{
	u32 xcrg_id;

	for (xcrg_id = 0; xcrg_id < NUMBER_OF_XCRG_INSTANCE; xcrg_id++)
		raw_spin_lock_init(&xcrg_lock[xcrg_id]);
}

static const struct clk_ops lin_clk_div_ops = {
	.recalc_rate = lin_divider_recalc_rate,
	.round_rate = lin_divider_round_rate,
	.set_rate = lin_divider_set_rate,
};

static const struct clk_ops lin_clk_div_ro_ops = {
	.recalc_rate = lin_divider_recalc_rate,
	.round_rate = lin_divider_round_rate,
};

struct clk_hw *lin_clk_register_div_data(struct device *dev,
			const struct lin_clk_div_data *data,
					     void __iomem *base)
{
	struct lin_clk_div *div;
	struct clk_init_data init;
	int ret;

	div = devm_kzalloc(dev, sizeof(*div), GFP_KERNEL);
	if (!div)
		return ERR_PTR(-ENOMEM);

	init.name = data->name;
	if (data->div_flag & CLK_DIVIDER_READ_ONLY)
		init.ops = &lin_clk_div_ro_ops;
	else
		init.ops = &lin_clk_div_ops;
	init.flags = CLK_SET_RATE_PARENT;
	init.parent_names = &data->parent_name;
	init.num_parents = 1;

	div->base = base;
	div->reg = data->reg;
	div->bit = data->bit;
	div->width = data->width;
	div->flags = data->div_flag;
	if (data->reg >= LCRG0)
		div->type = E_CRG_TYPE_LCRG;
	else
		div->type = E_CRG_TYPE_GCRG;
	div->hw.init = &init;

	ret = devm_clk_hw_register(dev, &div->hw);
	if (ret)
		return ERR_PTR(ret);

	return &div->hw;
}

/* LLD API */
static int get_val(enum E_CRG_TYPE type, uint32_t div, int *value)
{
	if (type == E_CRG_TYPE_GCRG) {
		if ((div > GCRG_DIVMAX + 1) || (div == 0))
			return -EINVAL;
		*value = div - 1;
	} else {
		switch (div) {
		case (1 << LCRG_DIV1):
			*value = LCRG_DIV1;
			break;
		case (1 << LCRG_DIV2):
			*value = LCRG_DIV2;
			break;
		case (1 << LCRG_DIV4):
			*value = LCRG_DIV4;
			break;
		case (1 << LCRG_DIV8):
			*value = LCRG_DIV8;
			break;
		case (1 << LCRG_DIV16):
			*value = LCRG_DIV16;
			break;
		case (1 << LCRG_DIV32):
			*value = LCRG_DIV32;
			break;
		case (1 << LCRG_DIV64):
			*value = LCRG_DIV64;
			break;
		default:
			return -EINVAL;
		}
	}
	return 0;
}

extern struct clk_hw_onecell_data *export_hw_data;
E_SCU_ERR scu_div_set(const E_SCU_CLKDIV id, uint32_t param)
{
	struct lin_clk_div *divider;
	int ret, value;

	if (id >= E_SCU_CLKDIV_MAX)
		return E_SCU_ERR_PARAM;
	if (!export_hw_data)
		return E_SCU_ERR_PARAM;
	divider = to_lin_clk_div(export_hw_data->hws[DIV_ID_TO_BINDING_ID(id)]);
	ret = get_val(divider->type, param, &value);
	if (ret){
		return E_SCU_ERR_PARAM;
	}
	ret = set_rate_xcrg(divider, value);
	if(ret == -EBUSY){
		return E_SCU_ERR_SYS;
	}
	else if( ret == -ETIMEDOUT){
		return E_SCU_ERR_SYS;
	}
	else
		return E_SCU_ERR_OK;
}
EXPORT_SYMBOL(scu_div_set);
