// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright 2022 Sony Corporation, SOCIONEXT INC.
 *
 */
#include <linux/clk-provider.h>
#include <linux/device.h>
#include <linux/io.h>
#include <linux/scu-clock.h>
#include "clk-lin.h"

struct lin_clk_gate {
	struct clk_hw hw;
	void __iomem *base;
	u32 reg;
	u32 bit;
};


#define to_lin_clk_gate(_hw)	container_of(_hw, struct lin_clk_gate, hw)

static int lin_clk_gate_endisable(struct clk_hw *hw, int set_clr)
{
	struct lin_clk_gate *gate = to_lin_clk_gate(hw);
	u32 val, offset;

	val = BIT(gate->bit);
	offset = gate->reg + set_clr;

	if(set_clr == CLR_REG){
		mb();
	}

	writel(val, gate->base + offset);

	if(set_clr == SET_REG){
		mb();
		readl(gate->base + offset);	//readback for timing adjustment
		mb();
	}

	return 0;
}

static int lin_clk_gate_enable(struct clk_hw *hw)
{
	return lin_clk_gate_endisable(hw, SET_REG);
}

static void lin_clk_gate_disable(struct clk_hw *hw)
{
	lin_clk_gate_endisable(hw, CLR_REG);
}

static const struct clk_ops lin_clk_gate_ops = {
	.enable = lin_clk_gate_enable,
	.disable = lin_clk_gate_disable,
};

void lin_clk_gate_init_locks(void)
{}

struct clk_hw *lin_clk_register_gate_data(struct device *dev,
			const struct lin_clk_gate_data *data,
					     void __iomem *base)
{
	struct lin_clk_gate *gate;
	struct clk_init_data init;
	int ret;

	gate = devm_kzalloc(dev, sizeof(*gate), GFP_KERNEL);
	if (!gate)
		return ERR_PTR(-ENOMEM);

	init.name = data->name;
	init.ops = &lin_clk_gate_ops;
	init.flags = CLK_SET_RATE_PARENT | CLK_IGNORE_UNUSED;
	init.parent_names = &data->parent_name;
	init.num_parents = 1;

	gate->base = base;
	gate->reg = data->reg;
	gate->bit = data->bit;
	gate->hw.init = &init;

	ret = devm_clk_hw_register(dev, &gate->hw);
	if (ret)
		return ERR_PTR(ret);

	return &gate->hw;
}

// LLD APIs
extern struct clk_hw_onecell_data *export_hw_data;
E_SCU_ERR scu_clk_enable(const E_SCU_CLK id)
{
	int ret = E_SCU_ERR_PARAM;

	if (id < E_SCU_CLK_MAX) {
		if (export_hw_data) {
			lin_clk_gate_enable(export_hw_data->hws[id]);
			ret = E_SCU_ERR_OK;
		}
	}
	return ret;
}
EXPORT_SYMBOL(scu_clk_enable);

E_SCU_ERR scu_clk_disable(const E_SCU_CLK id)
{
	int ret = E_SCU_ERR_PARAM;

	if (id < E_SCU_CLK_MAX) {
		if (export_hw_data) {
			lin_clk_gate_disable(export_hw_data->hws[id]);
			ret = E_SCU_ERR_OK;
		}
	}
	return ret;
}
EXPORT_SYMBOL(scu_clk_disable);

#ifdef TESTCODE_LIN_CLOCK
int scu_clk_state(const E_SCU_CLK id)
{
	struct lin_clk_gate *gate;
	struct clk_hw *hw;
	u32 val,offset;
	int ret = -1;;

	if ((id < E_SCU_CLK_MAX) && (export_hw_data)) {
		hw = export_hw_data->hws[id];
		gate = to_lin_clk_gate(hw);
		offset = gate->reg;
		val = readl(gate->base + offset);
		val &= BIT(gate->bit);
		ret = !!val;
	}

	return ret; // 1:clock enabled(ungated) 0: clock disabled(gated)
}
EXPORT_SYMBOL(scu_clk_state);
#endif
