// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright 2022 Sony Corporation, SOCIONEXT INC.
 *
 */
#include <linux/types.h>
#include <linux/scu-clock.h>
#include <linux/clk-provider.h>
#include <linux/device.h>
#include <linux/io.h>
#include <linux/iopoll.h>
#include "clk-lin.h"

struct lin_clk_mux {
	struct clk_hw hw;
	void __iomem *base;
	u32 reg;
	u32 mask;
	u8 shift;
	u8 flags;
	raw_spinlock_t lock;
};


#define to_lin_clk_mux(_hw)	container_of(_hw, struct lin_clk_mux, hw)

static u8 lin_mux_get_parent(struct clk_hw *hw)
{
	struct lin_clk_mux *mux = to_lin_clk_mux(hw);
	u32 val;
	val = readl(mux->base + mux->reg) >> mux->shift;
	val &= mux->mask;

	return clk_mux_val_to_index(hw, NULL, mux->flags, val);
}

static int lin_mux_set_parent(struct clk_hw *hw, u8 index)
{
	struct lin_clk_mux *mux = to_lin_clk_mux(hw);
	u32 val = clk_mux_index_to_val(NULL, mux->flags, index);
	void __iomem *reg = mux->base + mux->reg;
	u32 mask = mux->mask;
	u8 shift = mux->shift;
	unsigned long flags = 0;

#ifdef TESTCODE_LIN_CLOCK
	SCU_time_mesure_start();
#endif
	raw_spin_lock_irqsave(&mux->lock, flags);
	writel((mask << shift), reg + CLR_REG);
	if (val)
		writel((val << shift), reg + SET_REG);
	raw_spin_unlock_irqrestore(&mux->lock, flags);
#ifdef TESTCODE_LIN_CLOCK
	SCU_time_mesure_stop();
#endif

	return 0;
}

static const struct clk_ops lin_clk_mux_ops = {
	.get_parent = lin_mux_get_parent,
	.set_parent = lin_mux_set_parent,
	.determine_rate = __clk_mux_determine_rate,
};
void lin_clk_mux_init_locks(void)
{}

struct clk_hw *lin_clk_register_mux_data(struct device *dev,
			const struct lin_clk_mux_data *data,
					     void __iomem *base)
{
	struct lin_clk_mux *mux;
	struct clk_init_data init;
	int ret;

	mux = devm_kzalloc(dev, sizeof(*mux), GFP_KERNEL);
	if (!mux)
		return ERR_PTR(-ENOMEM);

	init.name = data->name;
	init.ops = &lin_clk_mux_ops;
	init.flags = CLK_SET_RATE_PARENT;
	init.parent_names = data->parent_names;
	init.num_parents = data->num_parents;

	mux->base = base;
	mux->reg = data->reg;
	mux->mask = data->mask;
	mux->shift = data->shift;
	mux->hw.init = &init;
	raw_spin_lock_init(&mux->lock);
	ret = devm_clk_hw_register(dev, &mux->hw);
	if (ret)
		return ERR_PTR(ret);

	return &mux->hw;
}

/* LLD API */
extern struct clk_hw_onecell_data *export_hw_data;
E_SCU_ERR scu_clk_select(const E_SCU_CLKSEL id, uint32_t param)
{
	switch (id) {
		case E_SCU_CLKSEL_UHS1_SD4CLK:
		if (param >= SCU_CLKSEL_UHS1_MAX)
			return E_SCU_ERR_PARAM;
		break;

		case E_SCU_CLKSEL_EMMC_SD4CLK:
		if (param >= SCU_CLKSEL_EMMC_MAX)
			return E_SCU_ERR_PARAM;
		break;

		default:
		return E_SCU_ERR_PARAM;
	}
	if (!export_hw_data)
		return E_SCU_ERR_PARAM;
	lin_mux_set_parent(export_hw_data->hws[MUX_ID_TO_BINDING_ID(id)],
			   (u8)param);
	return E_SCU_ERR_OK;
}
EXPORT_SYMBOL(scu_clk_select);
