// SPDX-License-Identifier: GPL-2.0
/*
 * clock controller for cxd
 *
 * Copyright 2022 Sony Corporation, SOCIONEXT INC.
 *
 */
#include <linux/types.h>
#include <dt-bindings/clock/cxd,lin-clock.h>
#include <linux/scu-clock.h>
#include <linux/clk-provider.h>
#include <linux/io.h>
#include <linux/init.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/platform_device.h>
#include "clk-lin.h"

#ifdef CONFIG_CLOCK_FPGA_ENV
const u32 g_fpga_env_gcrg_freq1[NUMBER_OF_GCRG_INSTANCE] = {
	//prm_init_pll_N
	//0    1   2   3   4   5   6   7   8   9  10   11   12   13  14  15
	108, 108, 99, 64, 64, 48, 66, 85, 99, 74, 64,  64, 108, 108, 64, 66,
	//16 17  18  19  20
	66,  66,  1, 66, 66,
};

const u32 g_fpga_env_gcrg_freq2[NUMBER_OF_GCRG_INSTANCE] = {
	// prm_init_pll_Q, prm_init_pll_P
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000,
};

u32 g_fpga_env_divider[NUMBER_OF_XCRG_INSTANCE][NUMBER_OF_XCRG_DIVIDER] = {
	// ODIVCHxNUM G0-21
	{11, 17, 35, 0}, 	// G0
	{8, 63, 63, 0},
	{63, 79, 0, 0},
	{3, 3, 0, 0},
	{49, 0, 0, 0},		// G4
	{59, 39, 0, 0},
	{1, 2, 0, 0},
	{1, 2, 3, 0},
	{2, 2, 1, 0},   	// G8
	{1, 2, 0, 0},
	{63, 0, 0, 0},
	{7, 0, 0, 0},
	{1, 11, 0, 0},   	//G12
	{1, 11, 0, 0},
	{1, 29, 29, 0},
	{1, 0, 0, 0},
	{1, 0, 0, 0},    	//G16
	{1, 0, 0, 0},
	{0}, 			  	//G18 Dummy
	{1, 0, 0, 0},
	{1, 0, 0, 0},		//G20
	{0}, {0}, {0}, {0}, {0}, {0}, {0}, {0}, {0}, {0}, // G21-30 padding
	// LDIVRC L0-14
	{2, 3, 4, 3}, 		// L0
	{2, 3, 3, 3, 2},
	{1, 1, 1, 1, 1,  1, 1, 1, 1, 1,  0, 0, 0, 1, 2,  3, 1, 1, 1, 1},
	{1, 1, 0, 1},
	{0, 1, 2}, 			// L4
	{0, 1, 2},
	{0, 1, 2, 1, 2, 0, 0},
	{0, 1, 2, 3},
	{0, 0, 0, 2, 3,  4, 5, 6, 0, 0,  0, 2},	//L8
	{0, 1},
	{0, 1},
	{0, 1},
	{0}, 			//L12 dummy
	{0, 1},
	{0, 1},			//L14
};
#endif

#ifdef TESTCODE_LIN_CLOCK
void __iomem * scu_driver_test_io_remap[MAX_REG]={NULL,NULL,NULL};
#endif

struct clk_hw_onecell_data *export_hw_data;

static const struct lin_clk_pll_data lin_clk_cxd_data_pll[] = {
	LIN_CLK_PLL("pll0", C_CLK_IN, 0),
	LIN_CLK_PLL("pll2_0", C_CLK_IN, 4),
	LIN_CLK_PLL("pll2_1", "gcrg4_0", 5),
	LIN_CLK_PLL("pll3", M_CLK_IN, 6),
	LIN_CLK_PLL("pll4", M_CLK_IN, 7),
	LIN_CLK_PLL("pll5", "m_clk_in/2", 8),
	LIN_CLK_PLL("pll6", M_CLK_IN, 9),
	LIN_CLK_PLL("pllsd", M_CLK_IN, 14),
	{.name = NULL},
};

static const struct lin_clk_fixed_factor_data lin_clk_cxd_data_fixed_factor[] = {
	LIN_CLK_FACTOR("lcrg8_5/12", -1, "lcrg8_5", 1, 12),
	LIN_CLK_FACTOR("lcrg2_15/10", -1, "lcrg2_15", 1, 10),
	LIN_CLK_FACTOR("lcrg2_15/100", -1, "lcrg2_15", 1, 100),
	LIN_CLK_FACTOR("lcrg2_15/99K", -1, "lcrg2_15", 1, 99000),
	LIN_CLK_FACTOR("m_clk_in/2", -1, "m_clk_in", 1, 2),
	{.name = NULL},
};

static const struct lin_clk_div_data lin_clk_cxd_data_div[] = {
	LIN_CLK_DIV_LCRG("lcrg0_0",  -1,                      "gcrg0_0",  0,  0, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg2_0",  -1,                      "gcrg6_0",  2,  0, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg2_1",  -1,                      "gcrg6_0",  2,  1, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg2_2",  -1,                      "gcrg6_0",  2,  2, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg2_3",  -1,                      "gcrg6_0",  2,  3, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg2_4",  -1,                      "gcrg6_0",  2,  4, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg2_5",  -1,                      "gcrg6_0",  2,  5, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg2_6",  -1,                      "gcrg6_0",  2,  6, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg2_7",  -1,                      "gcrg6_0",  2,  7, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg2_13", -1,                      "gcrg6_0",  2, 13, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg2_14", -1,                      "gcrg6_0",  2, 14, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg2_15", -1,                      "gcrg6_0",  2, 15, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg3_0",  SCU_CLKDIV_UHS1_PLL4,    "gcrg7_1",  3,  0, 0),
	LIN_CLK_DIV_LCRG("lcrg3_1",  SCU_CLKDIV_EMMC_PLL4,    "gcrg7_1",  3,  1, 0),
	LIN_CLK_DIV_LCRG("lcrg3_2",  -1,                      "gcrg7_1",  3,  2, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg4_0",  -1,                      "gcrg8_0",  4,  0, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg5_0",  -1,                      "gcrg8_1",  5,  0, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg5_2",  -1,                      "gcrg8_1",  5,  2, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg6_0",  -1,                      "gcrg8_2",  6,  0, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg6_3",  -1,                      "gcrg8_2",  6,  3, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg6_5",  SCU_CLKDIV_UHS1_PLL5,    "gcrg8_2",  6,  5, 0),
	LIN_CLK_DIV_LCRG("lcrg6_6",  SCU_CLKDIV_EMMC_PLL5,    "gcrg8_2",  6,  6, 0),
	LIN_CLK_DIV_LCRG("lcrg7_0",  -1,                      "gcrg9_0",  7,  0, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg8_0",  SCU_CLKDIV_UHS1_PLLSD,   "gcrg14_0", 8,  0, 0),
	LIN_CLK_DIV_LCRG("lcrg8_1",  SCU_CLKDIV_EMMC_PLLSD,   "gcrg14_0", 8,  1, 0),
	LIN_CLK_DIV_LCRG("lcrg8_2",  -1,                      "gcrg14_0", 8,  2, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg8_4",  -1,                      "gcrg14_0", 8,  4, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg8_5",  -1,                      "gcrg14_0", 8,  5, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_LCRG("lcrg8_7",  -1,                      "gcrg14_0", 8,  7, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_GCRG("gcrg0_0",  -1,                      "pll0",     0,  0, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_GCRG("gcrg4_0",  -1,                      "pll2_0",   4,  0, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_GCRG("gcrg5_0",  SCU_CLKDIV_AUDIO_MCLK,   "pll2_1",   5,  0, 0),
	LIN_CLK_DIV_GCRG("gcrg5_1",  SCU_CLKDIV_AUDIO_PWMCLK, "pll2_1",   5,  1, 0),
	LIN_CLK_DIV_GCRG("gcrg6_0",  -1,                      "pll3",     6,  0, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_GCRG("gcrg6_1",  -1,                      "pll3",     6,  1, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_GCRG("gcrg7_0",  -1,                      "pll4",     7,  0, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_GCRG("gcrg7_1",  -1,                      "pll4",     7,  1, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_GCRG("gcrg8_0",  -1,                      "pll5",     8,  0, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_GCRG("gcrg8_1",  -1,                      "pll5",     8,  1, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_GCRG("gcrg8_2",  -1,                      "pll5",     8,  2, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_GCRG("gcrg9_0",  -1,                      "pll6",     9,  0, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_GCRG("gcrg9_1",  -1,                      "pll6",     9,  1, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_GCRG("gcrg14_0", -1,                      "pllsd",   14,  0, CLK_DIVIDER_READ_ONLY),
	LIN_CLK_DIV_GCRG("gcrg14_1", SCU_CLKDIV_UHS2_0,       "pllsd",   14,  1, 0),
	LIN_CLK_DIV_GCRG("gcrg14_2", SCU_CLKDIV_UHS2_1,       "pllsd",   14,  2, 0),
	{.name = NULL},
};

static const struct lin_clk_mux_data lin_clk_cxd_data_mux[] = {
	LIN_CLK_MUX_UHS1("mux_uhs1_sd4clk", SCU_CLKSEL_UHS1_SD4CLK,
			 "lcrg8_0", "lcrg3_0", "lcrg6_5"),
	LIN_CLK_MUX_EMMC("mux_emmc_sd4clk", SCU_CLKSEL_EMMC_SD4CLK,
			 "lcrg8_1", "lcrg3_1", "lcrg6_6"),
	{.name = NULL},
};

static const struct lin_clk_gate_data lin_clk_cxd_data_gate[] = {
	LIN_CLK_GATE("cm_pmu_pclk", SCU_CLK_CM_PMU_PCLK, "lcrg2_15", 4, 0),
	LIN_CLK_GATE("bp_c_pclk", SCU_CLK_BP_C_PCLK, "lcrg2_15", 4, 4),
	LIN_CLK_GATE("bp_c_aclk", SCU_CLK_BP_C_ACLK, "lcrg2_7", 4, 5),
	LIN_CLK_GATE("bp_m_pclk", SCU_CLK_BP_M_PCLK, "lcrg2_15", 4, 6),
	LIN_CLK_GATE("bp_m_aclk", SCU_CLK_BP_M_ACLK, "lcrg2_0", 4, 7),
	LIN_CLK_GATE("cm_uart_pclk", SCU_CLK_CM_UART_PCLK, "lcrg8_4", 4, 8),
	LIN_CLK_GATE("cm_uart_sclk", SCU_CLK_CM_UART_SCLK_DELETED, "lcrg8_4", 4, 9),
	LIN_CLK_GATE("cm_tpu0_clk", SCU_CLK_CM_TPU0_CLK, "lcrg8_4", 4, 12),
	LIN_CLK_GATE("cm_tpu0_ps", SCU_CLK_CM_TPU0_PS, "lcrg8_5", 4, 13),
	LIN_CLK_GATE("cm_tpu1_clk", SCU_CLK_CM_TPU1_CLK, "lcrg8_4", 4, 14),
	LIN_CLK_GATE("cm_tpu1_ps", SCU_CLK_CM_TPU1_PS, "lcrg8_5", 4, 15),
	LIN_CLK_GATE("cm_tpu2_clk", SCU_CLK_CM_TPU2_CLK, "lcrg8_4", 4, 16),
	LIN_CLK_GATE("cm_tpu2_ps", SCU_CLK_CM_TPU2_PS, "lcrg8_5", 4, 17),
	LIN_CLK_GATE("cm_tpu3_clk", SCU_CLK_CM_TPU3_CLK, "lcrg8_4", 4, 18),
	LIN_CLK_GATE("cm_tpu3_ps", SCU_CLK_CM_TPU3_PS, "lcrg8_5", 4, 19),
	LIN_CLK_GATE("cm_tpu4_clk", SCU_CLK_CM_TPU4_CLK, "lcrg8_4", 4, 20),
	LIN_CLK_GATE("cm_tpu4_ps", SCU_CLK_CM_TPU4_PS, "lcrg8_5", 4, 21),
	LIN_CLK_GATE("cm_tpu5_clk", SCU_CLK_CM_TPU5_CLK, "lcrg8_4", 4, 22),
	LIN_CLK_GATE("cm_tpu5_ps", SCU_CLK_CM_TPU5_PS, "lcrg8_5", 4, 23),
	LIN_CLK_GATE("cm_gpio_pclk", SCU_CLK_CM_GPIO_PCLK, "lcrg2_15", 4, 24),
	LIN_CLK_GATE("cm_gpioint_pclk", SCU_CLK_CM_GPIOINT_PCLK, "lcrg2_15", 4, 25),
	LIN_CLK_GATE("cm_adc_pclk", SCU_CLK_CM_ADC_PCLK, "lcrg2_15", 4, 27),
	LIN_CLK_GATE("cm_sio_pclk", SCU_CLK_CM_SIO_PCLK, "lcrg8_4", 5, 0),
	LIN_CLK_GATE("cm_sio_ps", SCU_CLK_CM_SIO_PS, "lcrg8_5", 5, 1),
	LIN_CLK_GATE("cm_sram_aclk", SCU_CLK_CM_SRAM_ACLK, "lcrg2_13", 5, 4),
	LIN_CLK_GATE("pcie4lc_c0_pclk", SCU_CLK_PCIE4LC_C0_PCLK, "lcrg2_15", 5, 8),
	LIN_CLK_GATE("pcie4lc_c0_ck", SCU_CLK_PCIE4LC_C0_CK, "lcrg4_0", 5, 9),
	LIN_CLK_GATE("pcie4lc_c1_pclk", SCU_CLK_PCIE4LC_C1_PCLK, "lcrg2_15", 5, 10),
	LIN_CLK_GATE("pcie4lc_c1_ck", SCU_CLK_PCIE4LC_C1_CK, "lcrg4_0", 5, 11),
	LIN_CLK_GATE("pcie4lc_m_pclk", SCU_CLK_PCIE4LC_M_PCLK, "lcrg2_15", 5, 12),
	LIN_CLK_GATE("pcie4lc_m_ck", SCU_CLK_PCIE4LC_M_CK, "lcrg5_0", 5, 13),
	LIN_CLK_GATE("emmc_aclk", SCU_CLK_EMMC_ACLK, "lcrg2_3", 6, 0),
	LIN_CLK_GATE("emmc_pclk", SCU_CLK_EMMC_PCLK, "lcrg2_15", 6, 1),
	LIN_CLK_GATE("emmc_sd4clk", SCU_CLK_EMMC_SD4CLK, "mux_emmc_sd4clk", 6, 2),
	LIN_CLK_GATE("pcie4lf_m_aclk", SCU_CLK_PCIE4LF_M_ACLK, "lcrg2_3", 6, 4),
	LIN_CLK_GATE("pcie4lf_s_aclk", SCU_CLK_PCIE4LF_S_ACLK, "lcrg2_3", 6, 5),
	LIN_CLK_GATE("pcie4lf_dbi_aclk", SCU_CLK_PCIE4LF_DBI_ACLK, "lcrg2_3", 6, 6),
	LIN_CLK_GATE("pcie4lf_aux_clk", SCU_CLK_PCIE4LF_AUX_CLK, "lcrg2_15/10", 6, 7),
	LIN_CLK_GATE("pcie4lf_hclk", SCU_CLK_PCIE4LF_HCLK, "lcrg2_14", 6, 8),
	LIN_CLK_GATE("pcie4lf_phy_cr_para_clk", SCU_CLK_PCIE4LF_PHY_CR_PARA_CLK, "lcrg2_15", 6, 9),
	LIN_CLK_GATE("pcie4lf_phy_pclk", SCU_CLK_PCIE4LF_PHY_PCLK, "lcrg2_15", 6, 10),
	LIN_CLK_GATE("emmc_c_pclk", SCU_CLK_EMMC_C_PCLK, "lcrg2_15", 6, 12),
	LIN_CLK_GATE("emmc_c_ck", SCU_CLK_EMMC_C_CK, "lcrg4_0", 6, 13),
	LIN_CLK_GATE("emmc_m_pclk", SCU_CLK_EMMC_M_PCLK, "lcrg2_15", 6, 14),
	LIN_CLK_GATE("emmc_m_ck", SCU_CLK_EMMC_M_CK, "lcrg5_0", 6, 15),
	LIN_CLK_GATE("audio_dsp1_clk", SCU_CLK_AUDIO_DSP1_CLK, "lcrg7_0", 7, 0),
	LIN_CLK_GATE("audio_dsp1_bclk", SCU_CLK_AUDIO_DSP1_BCLK, "lcrg2_6", 7, 1),
	LIN_CLK_GATE("audio_dsp2_clk", SCU_CLK_AUDIO_DSP2_CLK, "gcrg9_1", 7, 4),
	LIN_CLK_GATE("audio_dsp2_bclk", SCU_CLK_AUDIO_DSP2_BCLK, "lcrg2_6", 7, 5),
	LIN_CLK_GATE("audio_if_pclk", SCU_CLK_AUDIO_IF_PCLK, "lcrg2_15", 7, 8),
	LIN_CLK_GATE("audio_if_aclk", SCU_CLK_AUDIO_IF_ACLK, "lcrg2_14", 7, 9),
	LIN_CLK_GATE("audio_if_mclk", SCU_CLK_AUDIO_IF_MCLK, "gcrg5_0", 7, 10),
	LIN_CLK_GATE("audio_if_sysclk", SCU_CLK_AUDIO_IF_SYSCLK, "lcrg2_15", 7, 11),
	LIN_CLK_GATE("audio_if_pwmclk", SCU_CLK_AUDIO_IF_PWMCLK, "gcrg5_1", 7, 12),
	LIN_CLK_GATE("audio_c0_pclk", SCU_CLK_AUDIO_C0_PCLK, "lcrg2_15", 7, 16),
	LIN_CLK_GATE("audio_c0_ck", SCU_CLK_AUDIO_C0_CK, "lcrg4_0", 7, 17),
	LIN_CLK_GATE("audio_m_pclk", SCU_CLK_AUDIO_M_PCLK, "lcrg2_15", 7, 20),
	LIN_CLK_GATE("audio_m_ck", SCU_CLK_AUDIO_M_CK, "lcrg5_0", 7, 21),
	LIN_CLK_GATE("otp_pclk", SCU_CLK_OTP_PCLK, "lcrg2_15", 8, 0),
	LIN_CLK_GATE("m_cpuint_pclk", SCU_CLK_M_CPUINT_PCLK, "lcrg2_15", 8, 4),
	LIN_CLK_GATE("ldec_aclk", SCU_CLK_LDEC_ACLK, "lcrg2_15", 8, 8),
	LIN_CLK_GATE("ldec_pclk", SCU_CLK_LDEC_PCLK, "lcrg2_2", 8, 9),
	LIN_CLK_GATE("fec_aclk", SCU_CLK_FEC_ACLK, "lcrg2_2", 8, 12),
	LIN_CLK_GATE("fec_pclk", SCU_CLK_FEC_PCLK, "lcrg2_15", 8, 13),
	LIN_CLK_GATE("crypto_e0_aclk", SCU_CLK_CRYPTO_E0_ACLK, "lcrg2_2", 8, 16),
	LIN_CLK_GATE("crypto_e0_s_pclk", SCU_CLK_CRYPTO_E0_S_PCLK, "lcrg2_2", 8, 17),
	LIN_CLK_GATE("crypto_e0_ns_pclk", SCU_CLK_CRYPTO_E0_NS_PCLK, "lcrg2_2", 8, 18),
	LIN_CLK_GATE("crypto_e0_clk", SCU_CLK_CRYPTO_E0_CLK, "lcrg6_0", 8, 19),
	LIN_CLK_GATE("crypto_e1_aclk", SCU_CLK_CRYPTO_E1_ACLK, "lcrg2_2", 8, 20),
	LIN_CLK_GATE("crypto_e1_s_pclk", SCU_CLK_CRYPTO_E1_S_PCLK, "lcrg2_2", 8, 21),
	LIN_CLK_GATE("crypto_e1_ns_pclk", SCU_CLK_CRYPTO_E1_NS_PCLK, "lcrg2_2", 8, 22),
	LIN_CLK_GATE("crypto_e1_clk", SCU_CLK_CRYPTO_E1_CLK, "lcrg6_0", 8, 23),
	LIN_CLK_GATE("crypto_xts_aclk", SCU_CLK_CRYPTO_XTS_ACLK, "lcrg2_2", 8, 24),
	LIN_CLK_GATE("crypto_xts_s_pclk", SCU_CLK_CRYPTO_XTS_S_PCLK, "lcrg2_2", 8, 25),
	LIN_CLK_GATE("crypto_xts_ns_pclk", SCU_CLK_CRYPTO_XTS_NS_PCLK, "lcrg2_2", 8, 26),
	LIN_CLK_GATE("m_gpio_pclk", SCU_CLK_M_GPIO_PCLK, "lcrg2_15", 8, 28),
	LIN_CLK_GATE("m_dmac_c_pclk", SCU_CLK_M_DMAC_C_PCLK, "lcrg2_15", 9, 0),
	LIN_CLK_GATE("m_dmac_c_ck", SCU_CLK_M_DMAC_C_CK, "lcrg4_0", 9, 1),
	LIN_CLK_GATE("tzcmp0_m_pclk", SCU_CLK_TZCMP0_M_PCLK, "lcrg2_15", 9, 2),
	LIN_CLK_GATE("tzcmp0_m_ck", SCU_CLK_TZCMP0_M_CK, "lcrg5_0", 9, 3),
	LIN_CLK_GATE("tzcmp3_m_pclk", SCU_CLK_TZCMP3_M_PCLK, "lcrg2_15", 9, 4),
	LIN_CLK_GATE("tzcmp3_m_ck", SCU_CLK_TZCMP3_M_CK, "lcrg5_0", 9, 5),
	LIN_CLK_GATE("m_main_c_pclk", SCU_CLK_M_MAIN_C_PCLK, "lcrg2_15", 9, 6),
	LIN_CLK_GATE("m_main_c_ck", SCU_CLK_M_MAIN_C_CK, "lcrg4_0", 9, 7),
	LIN_CLK_GATE("trng_rng_clk", SCU_CLK_TRNG_RNG_CLK, "lcrg2_15", 9, 8),
	LIN_CLK_GATE("tzc400_aclk0", SCU_CLK_TZC400_ACLK0, "lcrg5_0", 9, 10),
	LIN_CLK_GATE("tzc400_aclk1", SCU_CLK_TZC400_ACLK1, "lcrg2_1", 9, 11),
	LIN_CLK_GATE("tzc400_aclk2", SCU_CLK_TZC400_ACLK2, "lcrg2_2", 9, 12),
	LIN_CLK_GATE("tzc400_aclk3", SCU_CLK_TZC400_ACLK3, "lcrg5_0", 9, 13),
	LIN_CLK_GATE("tzc400_pclk", SCU_CLK_TZC400_PCLK, "lcrg2_15", 9, 14),
	LIN_CLK_GATE("m_peri_nic_aclk", SCU_CLK_M_PERI_NIC_ACLK, "lcrg2_13", 9, 16),
	LIN_CLK_GATE("m_peri_nic_gb_aclk", SCU_CLK_M_PERI_NIC_GB_ACLK, "lcrg2_1", 9, 17),
	LIN_CLK_GATE("m_peri_nic_gc_aclk", SCU_CLK_M_PERI_NIC_GC_ACLK, "lcrg2_2", 9, 18),
	LIN_CLK_GATE("m_peri_nic_hclk", SCU_CLK_M_PERI_NIC_HCLK, "lcrg2_14", 9, 19),
	LIN_CLK_GATE("m_peri_nic_m_pclk", SCU_CLK_M_PERI_NIC_M_PCLK, "lcrg5_2", 9, 20),
	LIN_CLK_GATE("m_peri_nic_pclk", SCU_CLK_M_PERI_NIC_PCLK, "lcrg2_15", 9, 21),
	LIN_CLK_GATE("m_peri_nic_u3clk", SCU_CLK_M_PERI_NIC_U3CLK, "lcrg6_3", 9, 22),
	LIN_CLK_GATE("s_sram_aclk", SCU_CLK_S_SRAM_ACLK, "lcrg2_13", 9, 24),
	LIN_CLK_GATE("m_wdt_wdogclk", SCU_CLK_M_WDT_WDOGCLK, "lcrg8_5/12", 9, 28),
	LIN_CLK_GATE("m_wdt_pclk", SCU_CLK_M_WDT_PCLK, "lcrg8_5", 9, 29),
	LIN_CLK_GATE("m_timer00_pclk", SCU_CLK_M_TIMER00_PCLK, "lcrg2_15", 10, 0),
	LIN_CLK_GATE("m_timer00_timer_clk", SCU_CLK_M_TIMER00_TIMER_CLK, "lcrg8_5/12", 10, 1),
	LIN_CLK_GATE("m_timer01_pclk", SCU_CLK_M_TIMER01_PCLK, "lcrg2_15", 10, 2),
	LIN_CLK_GATE("m_timer01_timer_clk", SCU_CLK_M_TIMER01_TIMER_CLK, "lcrg8_5/12", 10, 3),
	LIN_CLK_GATE("m_timer02_pclk", SCU_CLK_M_TIMER02_PCLK, "lcrg2_15", 10, 4),
	LIN_CLK_GATE("m_timer02_timer_clk", SCU_CLK_M_TIMER02_TIMER_CLK, "lcrg8_5/12", 10, 5),
	LIN_CLK_GATE("m_timer03_pclk", SCU_CLK_M_TIMER03_PCLK, "lcrg2_15", 10, 6),
	LIN_CLK_GATE("m_timer03_timer_clk", SCU_CLK_M_TIMER03_TIMER_CLK, "lcrg8_5/12", 10, 7),
	LIN_CLK_GATE("m_timer04_pclk", SCU_CLK_M_TIMER04_PCLK, "lcrg2_15", 10, 8),
	LIN_CLK_GATE("m_timer04_timer_clk", SCU_CLK_M_TIMER04_TIMER_CLK, "lcrg8_5/12", 10, 9),
	LIN_CLK_GATE("m_timer05_pclk", SCU_CLK_M_TIMER05_PCLK, "lcrg2_15", 10, 10),
	LIN_CLK_GATE("m_timer05_timer_clk", SCU_CLK_M_TIMER05_TIMER_CLK, "lcrg8_5/12", 10, 11),
	LIN_CLK_GATE("m_timer06_pclk", SCU_CLK_M_TIMER06_PCLK, "lcrg2_15", 10, 12),
	LIN_CLK_GATE("m_timer06_timer_clk", SCU_CLK_M_TIMER06_TIMER_CLK, "lcrg8_5/12", 10, 13),
	LIN_CLK_GATE("m_timer07_pclk", SCU_CLK_M_TIMER07_PCLK, "lcrg2_15", 10, 14),
	LIN_CLK_GATE("m_timer07_timer_clk", SCU_CLK_M_TIMER07_TIMER_CLK, "lcrg8_5/12", 10, 15),
	LIN_CLK_GATE("m_timer08_pclk", SCU_CLK_M_TIMER08_PCLK, "lcrg2_15", 10, 16),
	LIN_CLK_GATE("m_timer08_timer_clk", SCU_CLK_M_TIMER08_TIMER_CLK, "lcrg8_5/12", 10, 17),
	LIN_CLK_GATE("m_timer09_pclk", SCU_CLK_M_TIMER09_PCLK, "lcrg2_15", 10, 18),
	LIN_CLK_GATE("m_timer09_timer_clk", SCU_CLK_M_TIMER09_TIMER_CLK, "lcrg8_5/12", 10, 19),
	LIN_CLK_GATE("m_timer10_pclk", SCU_CLK_M_TIMER10_PCLK, "lcrg2_15", 10, 20),
	LIN_CLK_GATE("m_timer10_timer_clk", SCU_CLK_M_TIMER10_TIMER_CLK, "lcrg8_5/12", 10, 21),
	LIN_CLK_GATE("m_timer11_pclk", SCU_CLK_M_TIMER11_PCLK, "lcrg2_15", 10, 22),
	LIN_CLK_GATE("m_timer11_timer_clk", SCU_CLK_M_TIMER11_TIMER_CLK, "lcrg8_5/12", 10, 23),
	LIN_CLK_GATE("m_timer12_pclk", SCU_CLK_M_TIMER12_PCLK, "lcrg2_15", 10, 24),
	LIN_CLK_GATE("m_timer12_timer_clk", SCU_CLK_M_TIMER12_TIMER_CLK, "lcrg8_5/12", 10, 25),
	LIN_CLK_GATE("m_timer13_pclk", SCU_CLK_M_TIMER13_PCLK, "lcrg2_15", 10, 26),
	LIN_CLK_GATE("m_timer13_timer_clk", SCU_CLK_M_TIMER13_TIMER_CLK, "lcrg8_5/12", 10, 27),
	LIN_CLK_GATE("m_timer14_pclk", SCU_CLK_M_TIMER14_PCLK, "lcrg2_15", 10, 28),
	LIN_CLK_GATE("m_timer14_timer_clk", SCU_CLK_M_TIMER14_TIMER_CLK, "lcrg8_5/12", 10, 29),
	LIN_CLK_GATE("m_timer15_pclk", SCU_CLK_M_TIMER15_PCLK, "lcrg2_15", 10, 30),
	LIN_CLK_GATE("m_timer15_timer_clk", SCU_CLK_M_TIMER15_TIMER_CLK, "lcrg8_5/12", 10, 31),
	LIN_CLK_GATE("m_uart1_pclk", SCU_CLK_M_UART1_PCLK, "lcrg8_4", 11, 0),
	LIN_CLK_GATE("m_uart1_sclk", SCU_CLK_M_UART1_SCLK_DELETED, "lcrg8_4", 11, 1),
	LIN_CLK_GATE("m_uart2_pclk", SCU_CLK_M_UART2_PCLK, "lcrg8_4", 11, 2),
	LIN_CLK_GATE("m_uart2_sclk", SCU_CLK_M_UART2_SCLK_DELETED, "lcrg8_4", 11, 3),
	LIN_CLK_GATE("m_uart3_pclk", SCU_CLK_M_UART3_PCLK, "lcrg8_4", 11, 4),
	LIN_CLK_GATE("m_uart3_sclk", SCU_CLK_M_UART3_SCLK_DELETED, "lcrg8_4", 11, 5),
	LIN_CLK_GATE("m_sio_pclk", SCU_CLK_M_SIO_PCLK, "lcrg8_4", 11, 8),
	LIN_CLK_GATE("m_sio_ps", SCU_CLK_M_SIO_PS, "lcrg8_5", 11, 9),
	LIN_CLK_GATE("m_spi0_pclk", SCU_CLK_M_SPI0_PCLK, "lcrg2_13", 11, 12),
	LIN_CLK_GATE("m_spi0_sspclk", SCU_CLK_M_SPI0_SSPCLK, "lcrg2_13", 11, 13),
	LIN_CLK_GATE("m_spi1_pclk", SCU_CLK_M_SPI1_PCLK, "lcrg2_15", 11, 14),
	LIN_CLK_GATE("m_spi1_ssi_clk", SCU_CLK_M_SPI1_SSI_CLK, "lcrg2_13", 11, 15),
	LIN_CLK_GATE("m_spi2_pclk", SCU_CLK_M_SPI2_PCLK, "lcrg2_15", 11, 16),
	LIN_CLK_GATE("m_spi2_ssi_clk", SCU_CLK_M_SPI2_SSI_CLK, "lcrg2_13", 11, 17),
	LIN_CLK_GATE("m_spi3_pclk", SCU_CLK_M_SPI3_PCLK, "lcrg2_15", 11, 18),
	LIN_CLK_GATE("m_spi3_ssi_clk", SCU_CLK_M_SPI3_SSI_CLK, "lcrg2_13", 11, 19),
	LIN_CLK_GATE("s_spi_pclk", SCU_CLK_S_SPI_PCLK, "lcrg2_15", 11, 20),
	LIN_CLK_GATE("s_spi_ssi_clk", SCU_CLK_S_SPI_SSI_CLK, "lcrg2_13", 11, 21),
	LIN_CLK_GATE("m_i2c0_pclk", SCU_CLK_M_I2C0_PCLK, "lcrg2_15", 11, 24),
	LIN_CLK_GATE("m_i2c0_ic_clk", SCU_CLK_M_I2C0_IC_CLK, "lcrg2_15", 11, 25),
	LIN_CLK_GATE("m_i2c1_pclk", SCU_CLK_M_I2C1_PCLK, "lcrg2_15", 11, 26),
	LIN_CLK_GATE("m_i2c1_ic_clk", SCU_CLK_M_I2C1_IC_CLK, "lcrg2_15", 11, 27),
	LIN_CLK_GATE("m_i2c2_pclk", SCU_CLK_M_I2C2_PCLK, "lcrg2_15", 11, 28),
	LIN_CLK_GATE("m_i2c2_ic_clk", SCU_CLK_M_I2C2_IC_CLK, "lcrg2_15", 11, 29),
	LIN_CLK_GATE("media_c_pclk", SCU_CLK_MEDIA_C_PCLK, "lcrg2_15", 14, 0),
	LIN_CLK_GATE("media_c_ck", SCU_CLK_MEDIA_C_CK, "lcrg4_0", 14, 1),
	LIN_CLK_GATE("media_m_pclk", SCU_CLK_MEDIA_M_PCLK, "lcrg2_15", 14, 2),
	LIN_CLK_GATE("media_m_ck", SCU_CLK_MEDIA_M_CK_DELETED, "lcrg5_0", 14, 3),
	LIN_CLK_GATE("usb_m_pclk", SCU_CLK_USB_M_PCLK, "lcrg2_15", 14, 4),
	LIN_CLK_GATE("usb_m_ck", SCU_CLK_USB_M_CK_DELETED, "lcrg5_0", 14, 5),
	LIN_CLK_GATE("pcie2lm0_m_aclk", SCU_CLK_PCIE2LM0_M_ACLK, "lcrg2_4", 12, 0),
	LIN_CLK_GATE("pcie2lm0_s_aclk", SCU_CLK_PCIE2LM0_S_ACLK, "lcrg2_4", 12, 1),
	LIN_CLK_GATE("pcie2lm0_dbi_aclk", SCU_CLK_PCIE2LM0_DBI_ACLK, "lcrg2_4", 12, 2),
	LIN_CLK_GATE("pcie2lm0_aux_clk", SCU_CLK_PCIE2LM0_AUX_CLK, "lcrg2_15/10", 12, 3),
	LIN_CLK_GATE("pcie2lm0_hclk", SCU_CLK_PCIE2LM0_HCLK, "lcrg2_14", 12, 4),
	LIN_CLK_GATE("pcie2lm0_phy_cr_para_clk", SCU_CLK_PCIE2LM0_PHY_CR_PARA_CLK, "lcrg2_15", 12, 5),
	LIN_CLK_GATE("pcie2lm0_phy_pclk", SCU_CLK_PCIE2LM0_PHY_PCLK, "lcrg2_15", 12, 6),
	LIN_CLK_GATE("pcie2lm1_m_aclk", SCU_CLK_PCIE2LM1_M_ACLK, "lcrg2_4", 12, 8),
	LIN_CLK_GATE("pcie2lm1_s_aclk", SCU_CLK_PCIE2LM1_S_ACLK, "lcrg2_4", 12, 9),
	LIN_CLK_GATE("pcie2lm1_dbi_aclk", SCU_CLK_PCIE2LM1_DBI_ACLK, "lcrg2_4", 12, 10),
	LIN_CLK_GATE("pcie2lm1_aux_clk", SCU_CLK_PCIE2LM1_AUX_CLK, "lcrg2_15/10", 12, 11),
	LIN_CLK_GATE("pcie2lm1_hclk", SCU_CLK_PCIE2LM1_HCLK, "lcrg2_14", 12, 12),
	LIN_CLK_GATE("pcie2lm1_phy_cr_para_clk", SCU_CLK_PCIE2LM1_PHY_CR_PARA_CLK, "lcrg2_15", 12, 13),
	LIN_CLK_GATE("pcie2lm1_phy_pclk", SCU_CLK_PCIE2LM1_PHY_PCLK, "lcrg2_15", 12, 14),
	LIN_CLK_GATE("pcie1l_m_aclk", SCU_CLK_PCIE1L_M_ACLK, "lcrg2_4", 12, 16),
	LIN_CLK_GATE("pcie1l_s_aclk", SCU_CLK_PCIE1L_S_ACLK, "lcrg2_4", 12, 17),
	LIN_CLK_GATE("pcie1l_dbi_aclk", SCU_CLK_PCIE1L_DBI_ACLK, "lcrg2_4", 12, 18),
	LIN_CLK_GATE("pcie1l_aux_clk", SCU_CLK_PCIE1L_AUX_CLK, "lcrg2_15/10", 12, 19),
	LIN_CLK_GATE("pcie1l_hclk", SCU_CLK_PCIE1L_HCLK, "lcrg2_14", 12, 20),
	LIN_CLK_GATE("pcie1l_phy_cr_para_clk", SCU_CLK_PCIE1L_PHY_CR_PARA_CLK, "lcrg2_15", 12, 21),
	LIN_CLK_GATE("pcie1l_phy_pclk", SCU_CLK_PCIE1L_PHY_PCLK, "lcrg2_15", 12, 22),
	LIN_CLK_GATE("usb32_aclk", SCU_CLK_USB32_ACLK, "lcrg2_5", 13, 0),
	LIN_CLK_GATE("usb32_suspend_clk", SCU_CLK_USB32_SUSPEND_CLK, "lcrg2_15/100", 13, 1),
	LIN_CLK_GATE("usb32_tca_apb_clk", SCU_CLK_USB32_TCA_APB_CLK, "lcrg2_15", 13, 2),
	LIN_CLK_GATE("usb20_ohci_clk48", SCU_CLK_USB20_OHCI_CLK48, "lcrg8_5", 13, 3),
	LIN_CLK_GATE("usb20_ho_hclk", SCU_CLK_USB20_HO_HCLK, "lcrg2_14", 13, 4),
	LIN_CLK_GATE("usb20_hclk", SCU_CLK_USB20_HCLK, "lcrg2_14", 13, 5),
	LIN_CLK_GATE("uhs2_0_aclk", SCU_CLK_UHS2_0_ACLK, "lcrg2_14", 13, 8),
	LIN_CLK_GATE("uhs2_0_hclk", SCU_CLK_UHS2_0_HCLK, "lcrg2_14", 13, 9),
	LIN_CLK_GATE("uhs2_0_sd4clk", SCU_CLK_UHS2_0_SD4CLK, "gcrg14_1", 13, 10),
	LIN_CLK_GATE("uhs2_0_bclk2", SCU_CLK_UHS2_0_BCLK2_DELETED, "gcrg14_1", 13, 11),
	LIN_CLK_GATE("uhs2_1_aclk", SCU_CLK_UHS2_1_ACLK, "lcrg2_14", 13, 12),
	LIN_CLK_GATE("uhs2_1_hclk", SCU_CLK_UHS2_1_HCLK, "lcrg2_14", 13, 13),
	LIN_CLK_GATE("uhs2_1_sd4clk", SCU_CLK_UHS2_1_SD4CLK, "gcrg14_2", 13, 14),
	LIN_CLK_GATE("uhs2_1_bclk2", SCU_CLK_UHS2_1_BCLK2_DELETED, "gcrg14_2", 13, 15),
	LIN_CLK_GATE("uhs1_hclk", SCU_CLK_UHS1_HCLK, "lcrg2_14", 13, 16),
	LIN_CLK_GATE("uhs1_sd_sd4clk", SCU_CLK_UHS1_SD_SD4CLK, "mux_uhs1_sd4clk", 13, 17),
	LIN_CLK_GATE("usb20_mmu_cclk", SCU_CLK_USB20_MMU_CCLK, "lcrg2_5", 13, 20),
	LIN_CLK_GATE("usb20_mmu_tbu_bclk0", SCU_CLK_USB20_MMU_TBU_BCLK0, "lcrg2_14", 13, 21),
	LIN_CLK_GATE("uhs2_mmu_0_cclk", SCU_CLK_UHS2_MMU_0_CCLK, "lcrg2_4", 13, 22),
	LIN_CLK_GATE("uhs2_mmu_0_tbu_bclk0", SCU_CLK_UHS2_MMU_0_TBU_BCLK0, "lcrg2_14", 13, 23),
	LIN_CLK_GATE("uhs2_mmu_1_cclk", SCU_CLK_UHS2_MMU_1_CCLK, "lcrg2_4", 13, 24),
	LIN_CLK_GATE("uhs2_mmu_1_tbu_bclk0", SCU_CLK_UHS2_MMU_1_TBU_BCLK0, "lcrg2_14", 13, 25),
	LIN_CLK_GATE("uhs1_mmu_cclk", SCU_CLK_UHS1_MMU_CCLK, "lcrg2_4", 13, 26),
	LIN_CLK_GATE("uhs1_mmu_tbu_bclk0", SCU_CLK_UHS1_MMU_TBU_BCLK0, "lcrg2_14", 13, 27),
	LIN_CLK_GATE("ms_xmscki", SCU_CLK_MS_XMSCKI, "lcrg8_2", 14, 8),
	LIN_CLK_GATE("ms_hclk", SCU_CLK_MS_HCLK, "lcrg2_14", 14, 9),
	LIN_CLK_GATE("ms_dven", SCU_CLK_MS_DVEN, "lcrg2_15/99K", 14, 10),
	LIN_CLK_GATE("audio_c1_pclk", SCU_CLK_AUDIO_C1_PCLK, "lcrg2_15", 7, 18),
	LIN_CLK_GATE("audio_c1_ck", SCU_CLK_AUDIO_C1_CK, "lcrg4_0", 7, 19),
	LIN_CLK_GATE("sram_a0_aclk", SCU_CLK_SRAM_A0_ACLK, "lcrg2_6", 7, 24),
	LIN_CLK_GATE("sram_a1_aclk", SCU_CLK_SRAM_A1_ACLK, "lcrg2_6", 7, 25),
	LIN_CLK_GATE("tbl_sram_aclk", SCU_CLK_TBL_SRAM_ACLK, "lcrg2_4", 14, 12),
	LIN_CLK_GATE("fcs_hclk", SCU_CLK_FCS_HCLK, "lcrg2_14", 5, 16),
	LIN_CLK_GATE("fcs_pclk", SCU_CLK_FCS_PCLK, "lcrg2_15", 5, 17),
	LIN_CLK_GATE("lm_dsp_396", SCU_CLK_LM_DSP_396, "lcrg2_13", 1, 0),
	LIN_CLK_GATE("lm", SCU_CLK_LM, "lcrg5_0", 1, 1),
	LIN_CLK_GATE("lm_54", SCU_CLK_LM_54, "lcrg0_0", 1, 2),
	LIN_CLK_GATE("lm_v_core", SCU_CLK_LM_V_CORE, "lcrg6_0", 1, 3),
	LIN_CLK_GATE("lm_v_ctrl", SCU_CLK_LM_V_CTRL, "lcrg3_2", 1, 4),
	LIN_CLK_GATE("lm_v_ctrl_m", SCU_CLK_LM_V_CTRL_M, "gcrg7_0", 1, 5),
	LIN_CLK_GATE("lm_v_v", SCU_CLK_LM_V_V, "gcrg6_1", 1, 6),
	LIN_CLK_GATE("lm_12", SCU_CLK_LM_12, "lcrg8_7", 1, 7),
	LIN_CLK_GATE("pcie4lc_m_aclk", SCU_CLK_PCIE4LC_M_ACLK, "lcrg2_0", 24, 16),
	LIN_CLK_GATE("pcie4lc_s_aclk", SCU_CLK_PCIE4LC_S_ACLK, "lcrg2_0", 24, 17),
	LIN_CLK_GATE("pcie4lc_dbi_aclk", SCU_CLK_PCIE4LC_DBI_ACLK, "lcrg2_0", 24, 18),
	LIN_CLK_GATE("pcie4lc_aux_clk", SCU_CLK_PCIE4LC_AUX_CLK, "lcrg2_15/10", 24, 19),
	LIN_CLK_GATE("pcie4lc_hclk", SCU_CLK_PCIE4LC_HCLK, "lcrg2_14", 24, 20),
	LIN_CLK_GATE("pcie4lc_phy_cr_para_clk", SCU_CLK_PCIE4LC_PHY_CR_PARA_CLK, "lcrg2_15", 24, 21),
	LIN_CLK_GATE("pcie4lc_phy_pclk", SCU_CLK_PCIE4LC_PHY_PCLK, "lcrg2_15", 24, 22),
	LIN_CLK_GATE("copy0_aclk", SCU_CLK_COPY0_ACLK, "lcrg2_0", 24, 24),
	LIN_CLK_GATE("copy0_pclk", SCU_CLK_COPY0_PCLK, "lcrg2_15", 24, 25),
	LIN_CLK_GATE("copy1_aclk", SCU_CLK_COPY1_ACLK, "lcrg2_0", 24, 26),
	LIN_CLK_GATE("copy1_pclk", SCU_CLK_COPY1_PCLK, "lcrg2_15", 24, 27),
	LIN_CLK_GATE("copy2_aclk", SCU_CLK_COPY2_ACLK, "lcrg2_0", 24, 28),
	LIN_CLK_GATE("copy2_pclk", SCU_CLK_COPY2_PCLK, "lcrg2_15", 24, 29),
	LIN_CLK_GATE("copy3_aclk", SCU_CLK_COPY3_ACLK, "lcrg2_0", 24, 30),
	LIN_CLK_GATE("copy3_pclk", SCU_CLK_COPY3_PCLK, "lcrg2_15", 24, 31),
	LIN_CLK_GATE("m_dmacs_core", SCU_CLK_M_DMACS_CORE, "lcrg2_2", 26, 0),
	LIN_CLK_GATE("m_dmacs_pclk", SCU_CLK_M_DMACS_PCLK, "lcrg2_15", 26, 1),
	LIN_CLK_GATE("m_dmacs_hs_clk0", SCU_CLK_M_DMACS_HS_CLK0, "lcrg2_15", 26, 2),
	LIN_CLK_GATE("m_dmacs_hs_clk1", SCU_CLK_M_DMACS_HS_CLK1, "lcrg2_15", 26, 3),
	LIN_CLK_GATE("m_dmac1_core", SCU_CLK_M_DMAC1_CORE, "lcrg2_2", 26, 4),
	LIN_CLK_GATE("m_dmac1_pclk", SCU_CLK_M_DMAC1_PCLK, "lcrg2_15", 26, 5),
	LIN_CLK_GATE("m_dmac2_core", SCU_CLK_M_DMAC2_CORE, "lcrg2_2", 26, 6),
	LIN_CLK_GATE("m_dmac2_pclk", SCU_CLK_M_DMAC2_PCLK, "lcrg2_15", 26, 7),
	LIN_CLK_GATE("m_dmac3_core", SCU_CLK_M_DMAC3_CORE, "lcrg2_2", 26, 8),
	LIN_CLK_GATE("m_dmac3_pclk", SCU_CLK_M_DMAC3_PCLK, "lcrg2_15", 26, 9),
	LIN_CLK_GATE("m_dmac4_core", SCU_CLK_M_DMAC4_CORE, "lcrg2_2", 26, 10),
	LIN_CLK_GATE("m_dmac4_pclk", SCU_CLK_M_DMAC4_PCLK, "lcrg2_15", 26, 11),
	LIN_CLK_GATE("m_dmac4_hs_clk00", SCU_CLK_M_DMAC4_HS_CLK00, "lcrg2_13", 26, 12),
	LIN_CLK_GATE("m_dmac4_hs_clk01", SCU_CLK_M_DMAC4_HS_CLK01, "lcrg2_13", 26, 13),
	LIN_CLK_GATE("m_dmac4_hs_clk02", SCU_CLK_M_DMAC4_HS_CLK02, "lcrg2_15", 26, 14),
	LIN_CLK_GATE("m_dmac4_hs_clk03", SCU_CLK_M_DMAC4_HS_CLK03, "lcrg2_15", 26, 15),
	LIN_CLK_GATE("m_dmac4_hs_clk04", SCU_CLK_M_DMAC4_HS_CLK04, "lcrg2_15", 26, 16),
	LIN_CLK_GATE("m_dmac4_hs_clk05", SCU_CLK_M_DMAC4_HS_CLK05, "lcrg2_15", 26, 17),
	LIN_CLK_GATE("m_dmac4_hs_clk06", SCU_CLK_M_DMAC4_HS_CLK06, "lcrg2_15", 26, 18),
	LIN_CLK_GATE("m_dmac4_hs_clk07", SCU_CLK_M_DMAC4_HS_CLK07, "lcrg2_15", 26, 19),
	LIN_CLK_GATE("m_dmac4_hs_clk08", SCU_CLK_M_DMAC4_HS_CLK08, "lcrg2_15", 26, 20),
	LIN_CLK_GATE("m_dmac4_hs_clk09", SCU_CLK_M_DMAC4_HS_CLK09, "lcrg2_15", 26, 21),
	LIN_CLK_GATE("m_dmac4_hs_clk10", SCU_CLK_M_DMAC4_HS_CLK10, "lcrg2_15", 26, 22),
	LIN_CLK_GATE("m_dmac4_hs_clk11", SCU_CLK_M_DMAC4_HS_CLK11, "lcrg2_15", 26, 23),
	LIN_CLK_GATE("lm_dsp396", SCU_CLK_LM_DSP396, "lcrg2_13", 1, 8),
	LIN_CLK_GATE("lm_v_core_v", SCU_CLK_LM_V_CORE_V, "lcrg6_0", 1, 10),
	{.name = NULL},
};

struct lin_clk_const_tables {
	const struct lin_clk_pll_data *pll_table;
	const struct lin_clk_fixed_factor_data *fixed_table;
	const struct lin_clk_div_data *div_table;
	const struct lin_clk_mux_data *mux_table;
	const struct lin_clk_gate_data *gate_table;
};

static void register_ext_clks(struct device *dev, void __iomem *base)
{
	u32 val;
	unsigned long rate;
	struct clk_hw *hw;
	val = readl(base + CLKSEL1);
	if (val & CLK27_24_MASK)
		if (val & CLK72_27_MASK)
			rate = C_CLK_HD_IN72;
		else
			rate = C_CLK_HD_IN27;
	else
		rate = C_CLK_SYS_IN;

	hw = clk_hw_register_fixed_rate(dev, C_CLK_IN, NULL, 0, rate);
	WARN(IS_ERR(hw), "failed to register %s", C_CLK_IN);
	hw = clk_hw_register_fixed_rate(dev, M_CLK_IN, NULL, 0, M_CLK_USB_IN);
	WARN(IS_ERR(hw), "failed to register %s", M_CLK_IN);
}

static void register_pll(struct device *dev,
			 const struct lin_clk_pll_data *pll,
			 void __iomem *base)
{
	u32 nval, pval, qval, multi, div;
	struct clk_hw *hw;
#ifdef CONFIG_CLOCK_FPGA_ENV
	nval = g_fpga_env_gcrg_freq1[pll->gcrg_id];
#else
	nval = readl(base + GCRG(pll->gcrg_id) + GCRGPLLFREQ1);
#endif
	nval &= PLLFDIV_MASK;
#ifdef CONFIG_CLOCK_FPGA_ENV
	pval = g_fpga_env_gcrg_freq2[pll->gcrg_id];
#else
	pval = readl(base + GCRG(pll->gcrg_id) + GCRGPLLFREQ2);
#endif
	qval = (pval >> PLLFDEN_SHIFT) & PLLFDEN_MASK;
	pval &= PLLFNUM_MASK;

	if (pval < qval) {
		multi = nval * qval + pval;
		div = qval;
	} else {
		multi = nval;
		div = 1;
	}
	hw = devm_clk_hw_register_fixed_factor(dev, pll->name,
					pll->parent_name, CLK_SET_RATE_PARENT,
					multi, div);
	WARN(IS_ERR(hw), "failed to register %s", pll->name);
}

static void register_clks(struct device *dev,
			  void __iomem **base,
			  const struct lin_clk_const_tables *data,
			  struct clk_hw **hws)
{
	int id;
	struct clk_hw *hw;

	for (id = 0; data->pll_table[id].name; id++) {
		const struct lin_clk_pll_data *p =
						&data->pll_table[id];
		register_pll(dev, p,  base[DIV_REG]);
	}

	for (id = 0; data->fixed_table[id].name; id++) {
		const struct lin_clk_fixed_factor_data *p =
						&data->fixed_table[id];
		hw = devm_clk_hw_register_fixed_factor(dev, p->name,
					p->parent_name, CLK_SET_RATE_PARENT,
					p->mult, p->div);
		if (WARN(IS_ERR(hw), "failed to register %s", p->name))
			continue;
		if (p->idx >= 0)
			hws[p->idx] = hw;
	}

	for (id = 0; data->div_table[id].name; id++) {
		const struct lin_clk_div_data *p =
						&data->div_table[id];
		hw = lin_clk_register_div_data(dev, p, base[DIV_REG]);
		if (WARN(IS_ERR(hw), "failed to register %s", p->name))
			continue;
		if (p->idx >= 0)
			hws[p->idx] = hw;
	}

	for (id = 0; data->mux_table[id].name; id++) {
		const struct lin_clk_mux_data *p =
						&data->mux_table[id];
		hw = lin_clk_register_mux_data(dev, p, base[MUX_REG]);
		if (WARN(IS_ERR(hw), "failed to register %s", p->name))
			continue;
		if (p->idx >= 0)
			hws[p->idx] = hw;
	}

	for (id = 0; data->gate_table[id].name; id++) {
		const struct lin_clk_gate_data *p =
						&data->gate_table[id];
		hw = lin_clk_register_gate_data(dev, p, base[GATE_REG]);
		if (WARN(IS_ERR(hw), "failed to register %s", p->name))
			continue;
		if (p->idx >= 0)
			hws[p->idx] = hw;
	}
}

static int get_resource(struct platform_device *pdev, void __iomem **base)
{
	base[DIV_REG] = devm_platform_ioremap_resource_byname(pdev, "clkrst");
	if (IS_ERR(base[DIV_REG]))
		return PTR_ERR(base[DIV_REG]);
	base[GATE_REG] = devm_platform_ioremap_resource_byname(pdev, "gcb");
	if (IS_ERR(base[GATE_REG]))
		return PTR_ERR(base[GATE_REG]);
	base[MUX_REG] = devm_platform_ioremap_resource_byname(pdev, "clksel");
	if (IS_ERR(base[MUX_REG]))
		return PTR_ERR(base[MUX_REG]);
#ifdef TESTCODE_LIN_CLOCK
	scu_driver_test_io_remap[DIV_REG] = base[DIV_REG];
	scu_driver_test_io_remap[GATE_REG]= base[GATE_REG];
	scu_driver_test_io_remap[MUX_REG] = base[MUX_REG];
#endif
	return 0;
}

static int lin_clk_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct clk_hw_onecell_data *hw_data;
	const struct lin_clk_const_tables *data;
	void __iomem *base[MAX_REG];
	int ret;
	int clk_num = SCU_CLKMAX;

	dev_err( dev, "starting lin-clock driver");
	data = of_device_get_match_data(dev);
	if (WARN_ON(!data))
		return -EINVAL;

	ret = get_resource(pdev, base);
	if (ret) {
		dev_err(&pdev->dev, "missing IO resource\n");
		return -ENODEV;
	}

	hw_data = devm_kzalloc(dev, struct_size(hw_data, hws, clk_num),
			GFP_KERNEL);
	if (!hw_data)
		return -ENOMEM;
	hw_data->num = clk_num;

	while (--clk_num >= 0)
		hw_data->hws[clk_num] = ERR_PTR(-EINVAL);

	register_ext_clks(dev, base[MUX_REG]);
	register_clks(dev, base, data, hw_data->hws);
	lin_clk_div_init_locks();
	lin_clk_gate_init_locks();
	lin_clk_mux_init_locks();

	ret = of_clk_add_hw_provider(dev->of_node, of_clk_hw_onecell_get,
				      hw_data);
	if (!ret)
		export_hw_data = hw_data;
	return ret;
}


static const struct lin_clk_const_tables lin_clk_cxd_data = {
	.pll_table = lin_clk_cxd_data_pll,
	.fixed_table = lin_clk_cxd_data_fixed_factor,
	.div_table = lin_clk_cxd_data_div,
	.mux_table = lin_clk_cxd_data_mux,
	.gate_table = lin_clk_cxd_data_gate,
};

static const struct of_device_id lin_clk_dt_ids[] = {
	{
		.compatible = "scu,lin-clock",
		.data = &lin_clk_cxd_data,
	},
	{ /* sentinel */ },
};

static struct platform_driver lin_clk_driver = {
	.probe  = lin_clk_probe,
	.driver = {
	.name           = "scu-clock",
	.of_match_table = lin_clk_dt_ids,
	},
};

static int __init lin_clk_driver_init(void)
{
	return platform_driver_register(&lin_clk_driver);
}
arch_initcall(lin_clk_driver_init);

/* for test module */
#ifdef TESTCODE_LIN_CLOCK

#ifdef CONFIG_CLOCK_FPGA_ENV
u32 (*SCU_lin_get_ptr_g_fpga_env_divider(void))[21]
{
	return g_fpga_env_divider;
}
EXPORT_SYMBOL(SCU_lin_get_ptr_g_fpga_env_divider);
#else
u32 SCU_lin_clk_gate_reg_direct_read_div(u32 offset)
{
	u32 val;
	val = readl(scu_driver_test_io_remap[DIV_REG] + offset);
	return val;
}
EXPORT_SYMBOL(SCU_lin_clk_gate_reg_direct_read_div);

#endif

u32 SCU_lin_clk_gate_reg_direct_read_gate(u32 offset)
{
	u32 val;
	val = readl(scu_driver_test_io_remap[GATE_REG] + offset);
	return val;
}
EXPORT_SYMBOL(SCU_lin_clk_gate_reg_direct_read_gate);

u32 SCU_lin_clk_gate_reg_direct_read_mux(u32 offset)
{
	u32 val;
	val = readl(scu_driver_test_io_remap[MUX_REG] + offset);
	return val;
}
EXPORT_SYMBOL(SCU_lin_clk_gate_reg_direct_read_mux);

static u64 SCU_t1[2],SCU_t2[2];
static u64 SCU_max[2];

void SCU_time_mesure_reset(void)
{
	SCU_max[0]=0;
	SCU_max[1]=0;
}

void SCU_time_mesure_start(void)
{
	SCU_t1[smp_processor_id()]=ktime_get_ns();
}

void SCU_time_mesure_stop(void)
{
	u64 delta;
	u32 p;
	p=smp_processor_id();
	SCU_t2[p]=ktime_get_ns();
	if ( SCU_t1[p]!=0 )
		delta = SCU_t2[p]-SCU_t1[p];

	if (delta > SCU_max[p])
		SCU_max[p]=delta;
}

void SCU_time_mesure_print(void)
{
	u32 p;
	p=smp_processor_id();
	if (SCU_max[p])
		printk("max time@cpu%u=%u[ns]",p,(unsigned int)SCU_max[p]);
	else
		printk("max time@cpu%u=--[ns]",p);
}
EXPORT_SYMBOL(SCU_time_mesure_reset);
EXPORT_SYMBOL(SCU_time_mesure_start);
EXPORT_SYMBOL(SCU_time_mesure_stop);
EXPORT_SYMBOL(SCU_time_mesure_print);


#endif
