// SPDX-License-Identifier: GPL-2.0
// Copyright 2022 Sony Corporation, SOCIONEXT INC.
// Portions Copyright (C) 2024 Synopsys, Inc.  Used with permission. All rights reserved.

#include <linux/types.h>
#include <linux/spinlock.h>
#include <linux/iopoll.h>
#include <linux/dmac_lld.h>
#include "dw-axi-dmac-lld.h"
#include "dw-axi-dmac.h"

/**
 * LLD Channel Management Information
 */
struct lld_ch_mng_info_t {
	raw_spinlock_t		spinlock;	/* spinlock */
	bool			is_opened;	/* Channel is opened */
	LLD_DMAC_CH_CONF	ch_conf;	/* Configuration info */
	LLD_DMAC_CH_INFO	ch_info;	/* Function setting info */
	LLD_FUNC		int_func;	/* Callback function */
	void			*cookies;	/* Data passed to callback function */
	LLD_FUNC_FLAG		flag;		/* LLD_FUNC info */
	bool			is_interrupted;	/* Interrupt has occurred */
	uint64_t int_status;
	LLD_FACT		fact;		/* Interrupt fact */
};

static struct lld_ch_mng_info_t lld_ch_mng_info[LLD_CH_MAX + 1];

#ifdef TESTCODE_DMA_DRV_TRANS_PERF
struct lld_debug_info_t {
	uint64_t nsec_start;
	uint64_t nsec_end;
};

static struct lld_debug_info_t lld_debug_info = {.nsec_start = 0, .nsec_end = 0};
#endif /* TESTCODE_DMA_DRV_TRANS_PERF */

#ifdef TESTCODE_DMA_DRV_LOCK_PERF
struct lld_debug_info_t {
	uint64_t nsec_max_lock_time;
	char func_name_1[256];
	char func_name_2[256];
};

static struct lld_debug_info_t lld_debug_info = {.nsec_max_lock_time = 0, .func_name_1[0] = '\0', .func_name_2[0] = '\0'};
#endif /* TESTCODE_DMA_DRV_LOCK_PERF */


/**
 * LLD Channel Management Information
 */
/**
 * lld_dmac_ch_mng_get_lock() - Get lock for channel.
 *
 * @ch:		Target channel to get.
 *
 * @return	Address of spinlock.
 */
inline raw_spinlock_t *lld_dmac_ch_mng_get_lock(const LLD_CH ch)
{
	return &(lld_ch_mng_info[ch].spinlock);
}

/**
 * lld_dmac_ch_mng_open() - Set channel state to 'open'.
 *
 * @ch:		Target channel to set.
 *
 * @return	none.
 */
static inline void lld_dmac_ch_mng_open(const LLD_CH ch)
{
	lld_ch_mng_info[ch].is_opened = true;
}

/**
 * lld_dmac_ch_mng_close() - Set channel state to 'close'.
 *
 * @ch:		Target channel to set.
 *
 * @return	none.
 */
static inline void lld_dmac_ch_mng_close(const LLD_CH ch)
{
	lld_ch_mng_info[ch].is_opened = false;
}

/**
 * lld_dmac_ch_mng_is_opened() - Check the channel is 'opened'.
 *
 * @ch:		Target channel to check.
 *
 * @return	true on 'opened' state, false on 'closed' state.
 */
static inline bool lld_dmac_ch_mng_is_opened(const LLD_CH ch)
{
	return lld_ch_mng_info[ch].is_opened;
}

/**
 * lld_dmac_ch_mng_get_config() - Get configuration for the channel.
 *
 * @ch:		Target channel to get configuration.
 *
 * @return	Pointer to LLD_DMAC_CH_CONF.
 */
inline LLD_DMAC_CH_CONF *lld_dmac_ch_mng_get_config(const LLD_CH ch)
{
	return &(lld_ch_mng_info[ch].ch_conf);
}

/**
 * lld_dmac_ch_mng_set_config() - Set configuration for the channel.
 *
 * @ch:		Target channel to set configuration.
 * @conf:	Configuration information
 *
 * @return	none.
 */
static inline void lld_dmac_ch_mng_set_config(const LLD_CH ch, const LLD_DMAC_CH_CONF *conf)
{
	lld_ch_mng_info[ch].ch_conf.prio = conf->prio;
	lld_ch_mng_info[ch].ch_conf.aw_qos = conf->aw_qos;
	lld_ch_mng_info[ch].ch_conf.ar_qos = conf->ar_qos;
	lld_ch_mng_info[ch].ch_conf.src_outstand = conf->src_outstand;
	lld_ch_mng_info[ch].ch_conf.dst_outstand = conf->dst_outstand;
}

/**
 * lld_dmac_ch_mng_get_func_info() - Get fucntion information for the channel.
 *
 * @ch:		Target channel to get function information.
 *
 * @return	Pointer to LLD_DMAC_CH_INFO.
 */
static inline LLD_DMAC_CH_INFO *lld_dmac_ch_mng_get_func_info(const LLD_CH ch)
{
	return &(lld_ch_mng_info[ch].ch_info);
}

/**
 * lld_dmac_ch_mng_get_int_func() - Get callback fucntion for the channel.
 *
 * @ch:		Target channel to get callback function.
 *
 * @return	Pointer to callback function.
 */
inline LLD_FUNC lld_dmac_ch_mng_get_int_func(const LLD_CH ch)
{
	return lld_ch_mng_info[ch].int_func;
}

/**
 * lld_dmac_ch_mng_get_cookies() - Get cookies for the channel.
 *
 * @ch:		Target channel to get cookies.
 *
 * @return	Pointer to cookies.
 */
inline void *lld_dmac_ch_mng_get_cookies(const LLD_CH ch)
{
	return lld_ch_mng_info[ch].cookies;
}

/**
 * lld_dmac_ch_mng_get_int_status() - Get interrupt status for the channel.
 *
 * @ch:		Target channel to get interrupt status.
 *
 * @return	Interrupt status.
 */
inline uint64_t lld_dmac_ch_mng_get_int_status(const LLD_CH ch)
{
	return lld_ch_mng_info[ch].int_status;
}

/**
 * lld_dmac_ch_mng_set_int_status() - Set interrupt status for the channel.
 *
 * @ch:		Target channel to set interrupt status.
 *
 * @return	none.
 */
inline void lld_dmac_ch_mng_set_int_status(const LLD_CH ch, uint64_t int_status)
{
	lld_ch_mng_info[ch].int_status = int_status;
}

/**
 * lld_dmac_ch_mng_get_flag() - Get flag for the channel.
 *
 * @ch:		Target channel to get flag.
 *
 * @return	flag.
 */
inline LLD_FUNC_FLAG lld_dmac_ch_mng_get_flag(const LLD_CH ch)
{
	return lld_ch_mng_info[ch].flag;
}

/**
 * lld_dmac_ch_mng_set_func_info() - Set fucntion information for the channel.
 *
 * @ch:		Target channel to set function information.
 *
 * @return	none.
 */
static inline void lld_dmac_ch_mng_set_func_info(const LLD_CH ch, const LLD_DMAC_CH_INFO *info,
						 const LLD_FUNC int_func, void *cookies,
						 const LLD_FUNC_FLAG flag)
{
	lld_ch_mng_info[ch].ch_info.flowctl = info->flowctl;
	lld_ch_mng_info[ch].ch_info.srcid = info->srcid;
	lld_ch_mng_info[ch].ch_info.dstid = info->dstid;
	lld_ch_mng_info[ch].int_func = int_func;
	lld_ch_mng_info[ch].cookies = cookies;
	lld_ch_mng_info[ch].flag = flag;
}

/**
 * lld_dmac_ch_mng_clear_func_info() - Clear the fucntion information for the channel.
 *
 * @ch:		Target channel to clear function information.
 *
 * @return	none.
 */
static inline void lld_dmac_ch_mng_clear_func_info(const LLD_CH ch)
{
	lld_ch_mng_info[ch].ch_info.flowctl = 8;	/* Reset value is DMAX_CH(x)_TT_FC (=8) */
	lld_ch_mng_info[ch].ch_info.srcid = 0;
	lld_ch_mng_info[ch].ch_info.dstid = 0;
	lld_ch_mng_info[ch].int_func = 0;
	lld_ch_mng_info[ch].cookies = 0;
	lld_ch_mng_info[ch].flag = 0;
}


/**
 * lld_dmac_ch_mng_set_interrupt() - Set as interrupt has occurred.
 *
 * @ch:		Target channel to set.
 *
 * @return	None.
 */
inline void lld_dmac_ch_mng_set_interrupt(const LLD_CH ch)
{
	lld_ch_mng_info[ch].is_interrupted = true;
}

/**
 * lld_dmac_ch_mng_clear_interrupt() - Clear interrupt.
 *
 * @ch:		Target channel to clear.
 *
 * @return	None.
 */
static inline void lld_dmac_ch_mng_clear_interrupt(const LLD_CH ch)
{
	lld_ch_mng_info[ch].is_interrupted = false;
}

/**
 * lld_dmac_ch_mng_is_interrupted() - Whether interrupt has occurred.
 *
 * @ch:		Target channel to check.
 *
 * @return	true on interrupt has occurred, false on not occurred.
 */
static inline bool lld_dmac_ch_mng_is_interrupted(const LLD_CH ch)
{
	return lld_ch_mng_info[ch].is_interrupted;
}

/**
 * lld_dmac_ch_mng_set_fact() - Set interrupt fact.
 *
 * @ch:		Target channel to set.
 * @fact:	Interrupt fact.
 *
 * @return	None.
 */
inline void lld_dmac_ch_mng_set_fact(const LLD_CH ch, LLD_FACT fact)
{
	lld_ch_mng_info[ch].fact = fact;
}

/**
 * lld_dmac_ch_mng_get_fact() - Get interrupt fact.
 *
 * @ch:		Target channel to get.
 *
 * @return	Interrupt fact.
 */
static inline LLD_FACT lld_dmac_ch_mng_get_fact(const LLD_CH ch)
{
	return lld_ch_mng_info[ch].fact;
}

/**
 * lld_dmac_ch_mng_init_ch() - Initialize Channel Management Information for channel.
 *
 * @ch:		Target channel to initialize.
 *
 * @return	None.
 */
static inline void lld_dmac_ch_mng_init_ch(const LLD_CH ch)
{
	struct lld_ch_mng_info_t *info;

	/* Get LLD Channel Management Information for specified channel */
	info = &(lld_ch_mng_info[ch]);

	raw_spin_lock_init(&(info->spinlock));
	info->is_opened = false;
	/* Configuration of DMAC channel */
	info->ch_conf.prio = ((DMAC_MAX_CHANNELS - 1) - (ch % DMAC_MAX_CHANNELS));
	info->ch_conf.aw_qos = 0;
	info->ch_conf.ar_qos = 0;
	info->ch_conf.src_outstand = 0;
	info->ch_conf.dst_outstand = 0;
	/* Function setting information */
	info->ch_info.flowctl = 0;
	info->ch_info.srcid = 0;
	info->ch_info.dstid = 0;
	/* Callback function */
	info->int_func = 0;
	/* Data passed to callback function */
	info->cookies = 0;
	/* LLD_FUNC info */
	info->flag = 0;
	/* Interrupt */
	info->is_interrupted = false;
	/* Interrupt status */
	info->int_status = 0;
	/* Interrupt fact */
	info->fact = 0;
}

/**
 * Lower Level Functions
 */
/**
 * lld_dmac_is_valid_channel_number() - Whether the channel number is valid.
 *
 * @ch:		Target channel.
 *
 * @return	true on valid, false on invalid.
 */
static inline bool lld_dmac_is_valid_channel_number(const LLD_CH ch)
{
	/* The channel number is in valid range. */
	return ch <= LLD_CH_MAX;
}

/**
 * lld_dmac_is_valid_peripheral() - Whether the peripheral is valid.
 *
 * @id:		Peripheral ID.
 *
 * @return	true on valid, false on invalid.
 */
static inline bool lld_dmac_is_valid_peripheral(uint32_t id)
{
	/* The peripheral is in valid range. */
	return id <= LLD_DMAC_PERI_MAX;
}

/**
 * lld_dmac_is_valid_func_flag() - Whether the func flag is valid.
 *
 * @flag:	Func flag.
 *
 * @return	true on valid, false on invalid.
 */
static inline bool lld_dmac_is_valid_func_flag(LLD_FUNC_FLAG flag)
{
	/* The peripheral is in valid range. */
	return flag <= LLD_FUNC_MAX;
}

/**
 * lld_dmac_is_valid_priority() - Whether the priority is valid.
 *
 * @prio:	Priority.
 *
 * @return	true on valid, false on invalid.
 */
static inline bool lld_dmac_is_valid_priority(uint32_t priority)
{
	/* The priority is in valid range. */
	return priority <= LLD_DMAC_PRIORITY_MAX;
}

/**
 * lld_dmac_is_valid_qos() - Whether the AXI QoS is valid.
 *
 * @qos:	AXI QoS.
 *
 * @return	true on valid, false on invalid.
 */
static inline bool lld_dmac_is_valid_qos(uint32_t qos)
{
	/* AXI QoS is in valid range. */
	return qos <= LLD_DMAC_AXI_QOS_MAX;
}

/**
 * lld_dmac_is_valid_outstand() - Whether the outstanding is valid.
 *
 * @outstand:	Outstanding.
 *
 * @return	true on valid, false on invalid.
 */
static inline bool lld_dmac_is_valid_outstand(uint32_t outstand)
{
	/* The outstanding is in valid range. */
	return outstand <= LLD_DMAC_OUTSTAND_MAX;
}

/**
 * lld_dmac_is_valid_transfer_count() - Whether the transfer count is valid.
 *
 * @count:	transfer count.
 *
 * @return	true on valid, false on invalid.
 */
static inline bool lld_dmac_is_valid_transfer_count(uint32_t count)
{
	/* The transfer count is in valid range. */
	return count <= LLD_DMAC_TRANS_CNT_MAX;
}

/**
 * lld_dmac_is_valid_bt_length() - Whether the Burst Transaction Length is valid.
 *
 * @bsize:	Burst Transaction Length.
 *
 * @return	true on valid, false on invalid.
 */
static inline bool lld_dmac_is_valid_bt_length(uint32_t bsize)
{
	/* The Burst Transaction Length is in valid range. */
	return bsize <= LLD_DMAC_BSIZE_MAX;
}

/**
 * lld_dmac_is_valid_transfer_width() - Whether the Transfer Width is valid.
 *
 * @width:	Transfer Width.
 *
 * @return	true on valid, false on invalid.
 */
static inline bool lld_dmac_is_valid_transfer_width(uint32_t width)
{
	/* The Burst Transaction Length is in valid range. */
	return width <= LLD_DMAC_WIDTH_MAX;
}

/**
 * lld_dmac_is_set_func_info() - Whether the function info is set.
 *
 * @ch:		Target channel to check.
 *
 * @return	true on valid, false on invalid.
 */
static inline bool lld_dmac_is_set_func_info(const LLD_CH ch)
{
	/* Callback function for the channel is set. */
	return lld_dmac_ch_mng_get_int_func(ch) != 0;
}

/**
 * lld_dmac_is_available_lld_channel() - Whether the channel is available LLD.
 *		In this function, following checks will be done.
 *		  - Channel number is valid.
 *		  - Channel is assigned for LLD.
 *
 * @ch:		Target channel.
 *
 * @return	LLD_OK on available,
 *		LLD_DATA_ERR or LLD_NOT_OPEN_ERR on not available.
 */
static inline LLD_ER lld_dmac_is_available_lld_channel(const LLD_CH ch)
{
	/* The channel number is invalid */
	if (!lld_dmac_is_valid_channel_number(ch)) {
		return LLD_DATA_ERR;
	}

	/* Dma macro is not enabled */
	if (!(dma_is_enabled(ch))) {
		return LLD_NOT_OPEN_ERR;
	}

	/* The channel is assigned to DMA Engine */
	if (dma_is_controlled_by_dmaengine(ch)) {
		return LLD_DATA_ERR;
	}

	return LLD_OK;
}

/**
 * lld_dmac_valid_flowctl() - Whether the flowctl is valid.
 *
 * @info:	Pointer to the LLD_DMAC_CH_INFO structure.
 *
 * @return	LLD_OK on valid, LLD_DATA_ERR on invalid.
 */
static inline LLD_ER lld_dmac_valid_flowctl(const LLD_DMAC_CH_INFO *info)
{
	/* Flow control is out of range */
	if (info->flowctl > LLD_DMAC_FC_MAX) {
		return LLD_DATA_ERR;
	}

	/* Flow control is undefined value */
	if ((info->flowctl == 3) || (info->flowctl == 5)) {
		return LLD_DATA_ERR;
	}

	return LLD_OK;
}

/**
 * lld_dmac_get_src_auto_width() - Get auto data transfer width (src).
 *
 * @ch:		Target channel to get data.
 * @info:	Pointer to the LLD_DMAC_CH_INFO structure.
 *
 * @return	Auto data transfer width (src).
 */
static inline uint32_t lld_dmac_get_src_auto_width(const LLD_CH ch,
						   const LLD_DMAC_CH_INFO *info)
{
	uint32_t width;

	switch (info->flowctl) {
		case LLD_DMAC_PM_DMAC:
		case LLD_DMAC_PM_PERI:
			/* Get peripheral transfer width specified in Device Tree */
			width = dma_get_peri_width(ch, info->srcid);
			break;
		case LLD_DMAC_MP_PERI:
			/* Transfer width is 128bit */
			width = LLD_DMAC_WIDTH128;
			break;
		case LLD_DMAC_MM_DMAC:
		case LLD_DMAC_MP_DMAC:
		default:
			/* system error */
			printk(KERN_EMERG "%s: Invalid flowctl(%d)", __func__, info->flowctl);
			BUG_ON(1);
			break;
	}

	return width;
}

/**
 * lld_dmac_get_dst_auto_width() - Get auto data transfer width (dst).
 *
 * @ch:		Target channel to get data.
 * @info:	Pointer to the LLD_DMAC_CH_INFO structure.
 *
 * @return	Auto data transfer width (dst).
 */
static inline uint32_t lld_dmac_get_dst_auto_width(const LLD_CH ch,
						   const LLD_DMAC_CH_INFO *info)
{
	uint32_t width;

	switch (info->flowctl) {
		case LLD_DMAC_MP_DMAC:
		case LLD_DMAC_MP_PERI:
			/* Get peripheral transfer width specified in Device Tree */
			width = dma_get_peri_width(ch, info->dstid);
			break;
		case LLD_DMAC_MM_DMAC:
		case LLD_DMAC_PM_DMAC:
		case LLD_DMAC_PM_PERI:
			/* Transfer width is 128bit */
			width = LLD_DMAC_WIDTH128;
			break;
		default:
			/* system error */
			printk(KERN_EMERG "%s: Invalid flowctl(%d)", __func__, info->flowctl);
			BUG_ON(1);
			break;
	}

	return width;
}

/**
 * lld_dmac_set_config_func_to_register() - Set configuration and func info to register.
 *
 * @ch:		Target channel to set.
 * @conf:	Pointer to the LLD_DMAC_CH_CONF structure.
 * @info:	Pointer to the LLD_DMAC_CH_INFO structure.
 *
 * @return	None.
 */
static void lld_dmac_set_config_func_to_register(const LLD_CH ch, const LLD_DMAC_CH_CONF *conf, const LLD_DMAC_CH_INFO *info)
{
	void __iomem *chan_reg;
	uint64_t value_axi_qos;
	uint32_t value_cfg_hi;
	uint32_t value_cfg_lo;
	uint64_t value_cfg;

	/* Get base address of channel registers */
	chan_reg = dma_get_chx_reg_addr(ch);

	/* Set CH_AXI_QOS register value */
	value_axi_qos = ((conf->aw_qos << CH_CFG_AXI_AWQOS_POS) |
			 (conf->ar_qos << CH_CFG_AXI_ARQOS_POS));

	/* Set CH_CFG register value */
	/* Initial value of CHx_CFG is
	   LOCK_CH=0, LOCK_CH_L=0, DST_HWHS_POL=0, SRC_HWHS_POL=0,
	   HS_SEL_DST=0, HS_SEL_SRC=0,
	   DST_MULTBLK_TYPE=0, SRC_MULTBLK_TYPE=0 */
	value_cfg_hi = 0;
	value_cfg_lo = 0;

	/* Set CH_CFG register value with config info */
	value_cfg_hi |= ((conf->src_outstand << CH_CFG_H_SRC_OSR_LMT_POS) |
			 (conf->dst_outstand << CH_CFG_H_DST_OSR_LMT_POS) |
			 (conf->prio << CH_CFG_H_PRIORITY_POS));

	/* Set CH_CFG register value with func info */
	value_cfg_hi |= ((info->flowctl << CH_CFG_H_TT_FC_POS) |
			 (info->srcid << CH_CFG_H_SRC_PER_POS) |
			 (info->dstid << CH_CFG_H_DST_PER_POS));

	value_cfg = ((uint64_t)value_cfg_hi << 32) | value_cfg_lo;

	iowrite64(value_cfg, chan_reg + CH_CFG);
	iowrite32(value_axi_qos, chan_reg + CH_AXI_QOS);
}

/**
 * lld_dmac_set_transfer_to_register() - Set transfer info to register.
 *
 * @ch:		Target channel to set.
 * @tran_info:	Pointer to the LLD_DMAC_TRAN_INFO structure.
 * @info:	Pointer to the LLD_DMAC_CH_INFO structure.
 * @swidth:	Src data transfer width.
 * @dwidth:	Dst data transfer width.
 *
 * @return	None.
 */
static void lld_dmac_set_transfer_to_register(const LLD_CH ch,
					      const LLD_DMAC_TRAN_INFO *tran_info,
					      const LLD_DMAC_CH_INFO *info,
					      uint32_t swidth, uint32_t dwidth)
{
	void __iomem *chan_reg;
	uint32_t value_ctl_hi;
	uint32_t value_ctl_lo;
	uint64_t value_ctl;
	uint32_t ar_cache;
	uint32_t aw_cache;
	uint32_t awlen_en = 0;
	uint32_t arlen_en = 0;

	/* Get base address of channel registers */
	chan_reg = dma_get_chx_reg_addr(ch);

	/* Select cache signal */
	switch (info->flowctl) {
		case LLD_DMAC_MM_DMAC:
			ar_cache = DWAXIDMAC_AXI_CACHE_WRITEBACK_AND_WRITE_ALLOCATE;
			aw_cache = DWAXIDMAC_AXI_CACHE_WRITEBACK_AND_WRITE_ALLOCATE;
			break;
		case LLD_DMAC_MP_DMAC:
		case LLD_DMAC_MP_PERI:
			ar_cache = DWAXIDMAC_AXI_CACHE_WRITEBACK_AND_WRITE_ALLOCATE;
			aw_cache = DWAXIDMAC_AXI_CACHE_DEVICE_NON_BUF;
			break;
		case LLD_DMAC_PM_DMAC:
			ar_cache = DWAXIDMAC_AXI_CACHE_DEVICE_NON_BUF;
			aw_cache = DWAXIDMAC_AXI_CACHE_WRITEBACK_AND_WRITE_ALLOCATE;
			break;
		default:
			/* system error */
			printk(KERN_EMERG "%s: Invalid flowctl(%d)", __func__, info->flowctl);
			BUG_ON(1);
			break;
	}

	/* Set CH_CTL register value */
	/* Initial value of CHx_CTL is */
	/* SHADOWREG_OR_LLI_VALID=0, SHADOWREG_OR_LLI_LAST=0, IOC_BlkTfr=0,
	   DST_STAT_EN=0, SRC_STAT_EN=0, AWLEN=0, AWLEN_EN=0,
	   ARLEN=0, ARLEN_EN=0, AW_PROT=2, AR_PROT=2,
	   NonPosted_LastWrite_En=1, DMS=0, SMS=0 */
	value_ctl_hi = ((0 << CH_CTL_H_AWLEN_POS) | (0 << CH_CTL_H_ARLEN_POS) |
			(2 << CH_CTL_H_AW_PROT_POS) | (2 << CH_CTL_H_AR_PROT_POS));
	value_ctl_lo = ((1 << CH_CTL_L_LASTWRITE_EN_POS) |
			(aw_cache << CH_CTL_L_AW_CACHE_POS) |
			(ar_cache << CH_CTL_L_AR_CACHE_POS));

	/* Set CH_CTL register value with transfer info */
	value_ctl_lo |= ((tran_info->f1_si << CH_CTL_L_SRC_INC_POS) |
			 (tran_info->f1_di << CH_CTL_L_DST_INC_POS) |
			 (tran_info->f4_sbsize << CH_CTL_L_SRC_MSIZE_POS) |
			 (tran_info->f4_dbsize << CH_CTL_L_DST_MSIZE_POS) |
			 (swidth << CH_CTL_L_SRC_WIDTH_POS) |
			 (dwidth << CH_CTL_L_DST_WIDTH_POS));

	/* In case that LLD_DMAC_ADR_FIX is specified for Burst transaction */
	if (tran_info->f1_di == LLD_DMAC_ADR_FIX) {
		awlen_en = 1;
	}
	if (tran_info->f1_si == LLD_DMAC_ADR_FIX) {
		arlen_en = 1;
	}
	value_ctl_hi |= ((awlen_en << CH_CTL_H_AWLEN_EN_POS) |
			 (arlen_en << CH_CTL_H_ARLEN_EN_POS));

	value_ctl = ((uint64_t)value_ctl_hi << 32) | value_ctl_lo;

	iowrite64(tran_info->sadr, chan_reg + CH_SAR);

	iowrite64(tran_info->dadr, chan_reg + CH_DAR);

	iowrite32(tran_info->tsz, chan_reg + CH_BLOCK_TS);

	iowrite64(value_ctl, chan_reg + CH_CTL);

	/* SRC_MULTBLK_TYPE and DST_MULTBLK_TYPE in CHx_CFG
	   is already CONTINGUOUS */
}

/**
 * lld_dmac_set_llp_to_register() - Set Linked List Pointer to register.
 *
 * @ch:		Target channel to set.
 * @lli:	Pointer to the LLD_DMAC_IND_INFO structure.
 *
 * @return	None.
 */
static void lld_dmac_set_llp_to_register(const LLD_CH ch, const LLD_DMAC_IND_INFO lli)
{
	void __iomem *chan_reg;
	uint64_t value_lli;

	/* Get base address of channel registers */
	chan_reg = dma_get_chx_reg_addr(ch);

	/* LLP is treated as 64bit data */
	value_lli = *(uint64_t *)(&lli);

	/* Set CH_LLP register value */
	iowrite64(value_lli, chan_reg + CH_LLP);
}

/**
 * lld_dmac_set_mltblk_type_to_register() - Set Multi Block Transfer Type to register.
 *
 * @ch:		Target channel to set.
 * @src_mblk_type:	Source Multi Block Transfer Type.
 * @dst_mblk_type:	Destination Multi Block Transfer Type.
 *
 * @return	None.
 */
static void lld_dmac_set_mltblk_type_to_register(const LLD_CH ch, const uint8_t src_mblk_type, const uint8_t dst_mblk_type)
{
	void __iomem *chan_reg;
	uint64_t value_cfg;
	value_cfg = CH_CFG_MBLK_TYPE_UNMASK;

	/* Get base address of channel registers */
	chan_reg = dma_get_chx_reg_addr(ch);

	/* Set src_mblk_type and dst_mblk_type
		to SRC_MULTBLK_TYPE and DST_MULTBLK_TYPE in CHx_CFG */
	value_cfg &= ioread64(chan_reg + CH_CFG) ;
	value_cfg |= ((dst_mblk_type << CH_CFG_DST_MULTBLK_TYPE_POS) |
		      (src_mblk_type << CH_CFG_SRC_MULTBLK_TYPE_POS));
	iowrite64(value_cfg, chan_reg + CH_CFG);
}

/**
 * lld_dmac_interrupt_enable() - Enable interrupt for channel.
 *
 * @ch:		Target channel to set.
 *
 * @return	None.
 */
static void lld_dmac_interrupt_enable(const LLD_CH ch)
{
	void __iomem *chan_reg;

	/* Get base address of channel registers */
	chan_reg = dma_get_chx_reg_addr(ch);

	/* Enable interrupt status as axi_chan_irq_clear() */
	iowrite32(0xFFFFFFFF, chan_reg + CH_INTSTATUS_ENA);
}

/**
 * lld_dmac_interrupt_status_clear() - Clear interrupt status for channel.
 *
 * @ch:		Target channel to set.
 *
 * @return	None.
 */
static void lld_dmac_interrupt_status_clear(const LLD_CH ch)
{
	void __iomem *chan_reg;

	/* Get base address of channel registers */
	chan_reg = dma_get_chx_reg_addr(ch);

	/* Clear interrupt status as axi_chan_irq_clear() */
	iowrite32(0xFFFFFFFF, chan_reg + CH_INTCLEAR);

	/* Clear variable for interrupt status */
	lld_dmac_ch_mng_set_int_status(ch, D_CH_INT_STATUS_CLR);
}

/**
 * lld_dmac_ch_enable() - Enable DMA channel.
 *
 * @ch:		Target channel to enable.
 *
 * @return	None.
 */
static void lld_dmac_ch_enable(const LLD_CH ch)
{
	void __iomem *cmn_reg;
	uint32_t value;
	uint32_t chan_id;	/* channel id in DMAC IP */

	TESTCODE_DMAC_DRV_TRANS_VARS;

	/* Get base address of common registers */
	cmn_reg = dma_get_cmn_reg_addr(ch);

	/* channel id in DMAC IP */
	chan_id = ch % DMAC_MAX_CHANNELS;

	/* set enable bit and write enable bit */
	value = ((BIT(chan_id) << DMAC_CHAN_EN_SHIFT) |
		 (BIT(chan_id) << DMAC_CHAN_EN_WE_SHIFT));

	TESTCODE_DMAC_DRV_TRANS_GET_TIME;

	iowrite32(value, cmn_reg + DMAC_CHEN);

	TESTCODE_DMAC_DRV_TRANS_SET_NSEC_START;
}

/**
 * lld_dmac_ch_disable() - Disable DMA channel.
 *
 * @ch:		Target channel to disable.
 *
 * @return	None.
 */
static void lld_dmac_ch_disable(const LLD_CH ch)
{
	void __iomem *cmn_reg;
	uint32_t value;
	uint32_t chan_id;	/* channel id in DMAC IP */

	/* Get base address of common registers */
	cmn_reg = dma_get_cmn_reg_addr(ch);

	/* channel id in DMAC IP */
	chan_id = ch % DMAC_MAX_CHANNELS;

	/* set write enable bit (enable bit is cleared)*/
	value = (BIT(chan_id) << DMAC_CHAN_EN_WE_SHIFT);
	iowrite32(value, cmn_reg + DMAC_CHEN);
}

/**
 * lld_dmac_ch_is_enable() - Check the channel is DMA transferring.
 *
 * @ch:		Target channel to check.
 *
 * @return	true on transferring, false on not transferring.
 */
static bool lld_dmac_ch_is_enable(const LLD_CH ch)
{
	void __iomem *cmn_reg;
	uint32_t value;
	uint32_t chan_id;	/* channel id in DMAC IP */

	/* Get base address of common registers */
	cmn_reg = dma_get_cmn_reg_addr(ch);

	/* channel id in DMAC IP */
	chan_id = ch % DMAC_MAX_CHANNELS;

	value = ioread32(cmn_reg + DMAC_CHEN);

	/* whether the enable bit is on */
	return (value & (BIT(chan_id) << DMAC_CHAN_EN_SHIFT)) != 0;
}

/**
 * lld_dmac_get_destination_address() - Get destination address of the channel.
 *
 * @ch:		Target channel to get.
 *
 * @return	Destination address.
 */
static uint64_t lld_dmac_get_destination_address(const LLD_CH ch)
{
	void __iomem *chan_reg;
	uint64_t addr;

	/* Get base address of channel registers */
	chan_reg = dma_get_chx_reg_addr(ch);

	/* Get destination address */
	addr = ioread64(chan_reg + CH_DAR);

	return addr;
}

/**
 * lld_dmac_get_block_transfer_size() - Get block transfer size of the channel.
 *
 * @ch:		Target channel to get.
 *
 * @return	Block transfer size.
 */
static uint32_t lld_dmac_get_block_transfer_size(const LLD_CH ch)
{
	void __iomem *chan_reg;
	uint32_t value;

	/* Get base address of channel registers */
	chan_reg = dma_get_chx_reg_addr(ch);

	/* Get block transfer size */
	value = ioread32(chan_reg + CH_BLOCK_TS);

	return value;
}

/**
 * LLD Implementation
 */
/**
 * lld_dmac_open_impl() - Open the specified DMA channel (implementation).
 *
 * @ch:		Target channel to open.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
static LLD_ER lld_dmac_open_impl(const LLD_CH ch)
{
	/* The channel is already opened */
	if (lld_dmac_ch_mng_is_opened(ch)) {
		return LLD_DUPLICATE_OPEN_ERR;
	}

	/* Set the channel state to 'open' */
	lld_dmac_ch_mng_open(ch);

	dma_pm_runtime_get(ch);

	return LLD_OK;
}

/**
 * lld_dmac_close_impl() - Close the specified DMA channel (implementation).
 *
 * @ch:		Target channel to close.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
static LLD_ER lld_dmac_close_impl(const LLD_CH ch)
{
	/* The channel is already closed */
	if (!lld_dmac_ch_mng_is_opened(ch)) {
		/* Do nothing */
		return LLD_OK;
	}

	/* Set the channel state to 'close' */
	lld_dmac_ch_mng_close(ch);

	/* Clear the function information for the channel */
	lld_dmac_ch_mng_clear_func_info(ch);

	dma_pm_runtime_put(ch);

	return LLD_OK;
}

/**
 * lld_dmac_set_func_impl() - Set functions to the specified channel (implementation).
 *
 * @ch:		Target channel.
 * @info:	Pointer to the LLD_DMAC_CH_INFO structure.
 * @int_func:	Pointer to the callback function.
 * @cookies:	Identification parameter to be passed to the callback function.
 * @flag:	LLD_FUNC (context) information.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
static LLD_ER lld_dmac_set_func_impl(const LLD_CH ch, const LLD_DMAC_CH_INFO *info,
				     const LLD_FUNC int_func, void *cookies,
				     const LLD_FUNC_FLAG flag)
{
	/* The channel is not opened */
	if (!lld_dmac_ch_mng_is_opened(ch)) {
		return LLD_NOT_OPEN_ERR;
	}

	/* DMA Transfer is running */
	if (lld_dmac_ch_is_enable(ch)) {
		return LLD_HW_BUSY_ERR;
	}

	/* Set function info into lld_ch_mng_info */
	lld_dmac_ch_mng_set_func_info(ch, info, int_func, cookies, flag);

	/* Set configuration and function info to register */
	lld_dmac_set_config_func_to_register(ch, lld_dmac_ch_mng_get_config(ch),
					     lld_dmac_ch_mng_get_func_info(ch));

	return LLD_OK;
}

/**
 * lld_dmac_set_config_impl() - Configure the specified channel (implementation).
 *
 * @ch:		Target channel to configure.
 * @conf:	Pointer to the LLD_DMAC_CH_CONF structure.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
static LLD_ER lld_dmac_set_config_impl(const LLD_CH ch, const LLD_DMAC_CH_CONF *conf)
{
	/* The channel is not opened */
	if (!(dma_is_controlled_by_dmaengine(ch))) {
		if (!lld_dmac_ch_mng_is_opened(ch)) {
			return LLD_NOT_OPEN_ERR;
		}
	}

	/* DMA Transfer is running */
	if (lld_dmac_ch_is_enable(ch)) {
		return LLD_HW_BUSY_ERR;
	}

	/* Set configuration info into lld_ch_mng_info */
	lld_dmac_ch_mng_set_config(ch, conf);

	/* The values are not reflected to registers at this point. */

	return LLD_OK;
}

/**
 * lld_dmac_get_auto_transfer_width() - Get auto data transfer width.
 *
 * @ch:		Target channel to start.
 * @tran_info:	Pointer to the LLD_DMAC_TRAN_INFO structure.
 * @src_width:	Source transfer width.
 * @dst_width:	Destination transfer width.
 * @return	LLD_OK on success.
 */
static LLD_ER lld_dmac_get_auto_transfer_width(const LLD_CH ch, const LLD_DMAC_TRAN_INFO *tran_info,
																			uint32_t *src_width,  uint32_t *dst_width)
{
	LLD_DMAC_CH_INFO *info;
	uint32_t swidth, dwidth;

	/* The channel is not opened */
	if (!lld_dmac_ch_mng_is_opened(ch)) {
		return LLD_NOT_OPEN_ERR;
	}

	/* Function info is not set yet */
	if (!lld_dmac_is_set_func_info(ch)) {
		return LLD_INVALID_CALLBACK_ERR;
	}

	/* f1_sauto cannot be specified */
	info = lld_dmac_ch_mng_get_func_info(ch);
	if ((tran_info->f1_sauto == 1) &&
	    ((info->flowctl == LLD_DMAC_MM_DMAC) ||
	     (info->flowctl == LLD_DMAC_MP_DMAC))) {
		return LLD_DATA_ERR;
	}

	/* DMA Transfer is running */
	if (lld_dmac_ch_is_enable(ch)) {
		return LLD_HW_BUSY_ERR;
	}

	/* Decide transfer width */
	swidth = tran_info->f4_swidth;
	dwidth = tran_info->f4_dwidth;
	/* Automatic seting of transfer width */
	if (tran_info->f1_sauto == 1) {
		swidth = lld_dmac_get_src_auto_width(ch, info);
	}
	if (tran_info->f1_dauto == 1) {
		dwidth = lld_dmac_get_dst_auto_width(ch, info);
	}

	*src_width = swidth;
	*dst_width = dwidth;

	return LLD_OK;
}

/**
 * lld_dmac_start_impl() - Start DMA transfer (implementation).
 *
 * @ch:		Target channel to start.
 * @tran_info:	Pointer to the LLD_DMAC_TRAN_INFO structure.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
static LLD_ER lld_dmac_start_impl(const LLD_CH ch, const LLD_DMAC_TRAN_INFO *tran_info)
{
	LLD_ER ret;
	LLD_DMAC_CH_INFO *info;
	uint32_t src_width, dst_width;

	/* Get auto data transfer width. */
	ret = lld_dmac_get_auto_transfer_width(ch, tran_info, &src_width, &dst_width);
	if (ret != LLD_OK) {
		return ret;
	}

	/* Set transfer info to register */
	info = lld_dmac_ch_mng_get_func_info(ch);
	lld_dmac_set_transfer_to_register(ch, tran_info, info, src_width, dst_width);

	/* Set Multi Block Transfer Type for CONTIGUOUS to register. */
	lld_dmac_set_mltblk_type_to_register(ch,
		DWAXIDMAC_MBLK_TYPE_CONTIGUOUS, DWAXIDMAC_MBLK_TYPE_CONTIGUOUS);

	/* Clear Interrupt Status for channel */
	lld_dmac_interrupt_status_clear(ch);

	/* Enable Interrupt for channel */
	lld_dmac_interrupt_enable(ch);

	/* Start DMA transfer */
	lld_dmac_ch_enable(ch);

	return LLD_OK;
}

/**
 * lld_dmac_sc_start_impl() - Start DMA transfer with LLI (implementation).
 *
 * @ch:		Target channel to start.
 * @lli:	Pointer to the LLD_DMAC_IND_INFO structure.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
static LLD_ER lld_dmac_sc_start_impl(const LLD_CH ch, const LLD_DMAC_IND_INFO lli)
{
	/* The channel is not opened */
	if (!lld_dmac_ch_mng_is_opened(ch)) {
		return LLD_NOT_OPEN_ERR;
	}

	/* Function info is not set yet */
	if (!lld_dmac_is_set_func_info(ch)) {
		return LLD_INVALID_CALLBACK_ERR;
	}

	/* DMA Transfer is running */
	if (lld_dmac_ch_is_enable(ch)) {
		return LLD_HW_BUSY_ERR;
	}

	/* Set Linked List Pointer to register */
	lld_dmac_set_llp_to_register(ch, lli);

	/* Set Multi Block Transfer Type for LILNKED_LIST to register. */
	lld_dmac_set_mltblk_type_to_register(ch,
		DWAXIDMAC_MBLK_TYPE_LL, DWAXIDMAC_MBLK_TYPE_LL);

	/* Clear Interrupt Status for channel */
	lld_dmac_interrupt_status_clear(ch);

	/* Enable Interrupt for channel */
	lld_dmac_interrupt_enable(ch);

	/* Start DMA transfer */
	lld_dmac_ch_enable(ch);

	return LLD_OK;
}

/**
 * lld_dmac_stop_impl() - Stop DMA transfer (implementation).
 *
 * @ch:		Target channel to stop.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
static LLD_ER lld_dmac_stop_imp(const LLD_CH ch)
{
	/* The channel is not opened */
	if (!lld_dmac_ch_mng_is_opened(ch)) {
		return LLD_NOT_OPEN_ERR;
	}

	/* DMA Transfer is running */
	if (lld_dmac_ch_is_enable(ch)) {
		/* Stop DMA transfer */
		lld_dmac_ch_disable(ch);
	}

	/* Clear Interrupt Status for channel */
	lld_dmac_interrupt_status_clear(ch);

	return LLD_OK;
}

/**
 * lld_dmac_check_status_impl() - Get the operational state (implementation).
 *
 * @ch:		Target channel to check.
 *
 * @return	LLD_DMAC_TRAN_END or LLD_DMAC_TRAN_EXEC on success,
 *		otherwise error code.
 */
static int lld_dmac_check_status_impl(const LLD_CH ch)
{
	/* The channel is not opened */
	if (!lld_dmac_ch_mng_is_opened(ch)) {
		return LLD_NOT_OPEN_ERR;
	}

	/* Channel is DMA transferring */
	if (lld_dmac_ch_is_enable(ch)) {
		return LLD_DMAC_EXEC;
	}

	/* DMA transfer has finisned */
	return LLD_DMAC_TRAN_END;
}

/**
 * lld_dmac_get_trans_status_impl() - Get the state of transfer (implementation).
 *                                    (transfer address and transfer count)
 *
 * @ch:		Target channel to get.
 * @dadr:	Current destination address.
 * @tsize:	Current transfer count.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
static LLD_ER lld_dmac_get_trans_status_impl(const LLD_CH ch, uint64_t *dadr, uint32_t *tsize)
{
	/* The channel is not opened */
	if (!lld_dmac_ch_mng_is_opened(ch)) {
		return LLD_NOT_OPEN_ERR;
	}

	/* Set destination address */
	*dadr = lld_dmac_get_destination_address(ch);

	/* Set block transfer size */
	*tsize = lld_dmac_get_block_transfer_size(ch);

	return LLD_OK;
}

/**
 * lld_dmac_get_int_status_impl() - Get the interrupt status value which is
 *                                  stored when interrupt occurs (implementation).
 *
 * @ch:		Target channel to get.
 * @intstat:	Stored interrupt status.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
static LLD_ER lld_dmac_get_int_status_impl(const LLD_CH ch, uint64_t *intstat)
{
	/* The channel is not opened */
	if (!lld_dmac_ch_mng_is_opened(ch)) {
		return LLD_NOT_OPEN_ERR;
	}

	/* Set interrupt status */
	*intstat = lld_dmac_ch_mng_get_int_status(ch);

	return LLD_OK;
}

/**
 * LLD API
 */
/**
 * lld_dmac_open() - Open the specified DMA channel.
 *
 * @ch:		Target channel to open.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
LLD_ER lld_dmac_open(const LLD_CH ch)
{
	LLD_ER ret;
	unsigned long flags;
	raw_spinlock_t *lock;

	TESTCODE_DMAC_DRV_LOCK_VARS;

	/* The channel number is invalid */
	if (!lld_dmac_is_valid_channel_number(ch)) {
		return LLD_DATA_ERR;
	}

	/* Dma macro is not enabled */
	if (!(dma_is_enabled(ch))) {
		return LLD_NOT_OPEN_ERR;
	}

	/* The channel is assigned to DMA Engine */
	if (dma_is_controlled_by_dmaengine(ch)) {
		return LLD_DATA_ERR;
	}

	lock = lld_dmac_ch_mng_get_lock(ch);
	raw_spin_lock_irqsave(lock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	/* Execute open process */
	ret = lld_dmac_open_impl(ch);

	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("lld_dmac_open_impl");

	raw_spin_unlock_irqrestore(lock, flags);

	return ret;
}
EXPORT_SYMBOL(lld_dmac_open);

/**
 * lld_dmac_close() - Close the specified DMA channel.
 *
 * @ch:		Target channel to close.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
LLD_ER lld_dmac_close(const LLD_CH ch)
{
	LLD_ER ret;
	bool ch_enable;
	unsigned long flags;
	raw_spinlock_t *lock;

	TESTCODE_DMAC_DRV_LOCK_VARS;

	/* The channel number is invalid */
	if (!lld_dmac_is_valid_channel_number(ch)) {
		return LLD_DATA_ERR;
	}

	/* Dma macro is not enabled */
	if (!(dma_is_enabled(ch))) {
		return LLD_NOT_OPEN_ERR;
	}

	/* The channel is assigned to DMA Engine */
	if (dma_is_controlled_by_dmaengine(ch)) {
		return LLD_DATA_ERR;
	}

	lock = lld_dmac_ch_mng_get_lock(ch);

	/* DMA Transfer is running */
	raw_spin_lock_irqsave(lock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	ch_enable = lld_dmac_ch_is_enable(ch);

	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("lld_dmac_ch_is_enable");

	raw_spin_unlock_irqrestore(lock, flags);

	if (ch_enable) {
		void __iomem *cmn_reg;
		uint32_t chan_bit;
		uint32_t value;
		int ret;

		/* Get base address of common registers */
		cmn_reg = dma_get_cmn_reg_addr(ch);

		/* enable bit for channel */
		chan_bit = BIT(ch % DMAC_MAX_CHANNELS) << DMAC_CHAN_EN_SHIFT;

		/* wait until the transfer has finished
		   like dma_chan_terminate_all() */
		ret = readl_poll_timeout_atomic(cmn_reg + DMAC_CHEN, value,
						!(value & chan_bit), D_SLP_US_BETWEEN_READS_10, D_TIMEOUT_US_100);
		if (ret == -ETIMEDOUT) {
			return LLD_HW_BUSY_ERR;
		}
	}

	raw_spin_lock_irqsave(lock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	/* Execute close process */
	ret = lld_dmac_close_impl(ch);

	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("lld_dmac_close_impl");

	raw_spin_unlock_irqrestore(lock, flags);

	/* Configuration information for the chennel is not cleared */

	return ret;
}
EXPORT_SYMBOL(lld_dmac_close);

/**
 * lld_dmac_set_func() - Set functions to the specified channel.
 *
 * @ch:		Target channel.
 * @info:	Pointer to the LLD_DMAC_CH_INFO structure.
 * @int_func:	Pointer to the callback function.
 * @cookies:	Identification parameter to be passed to the callback function.
 * @flag:	LLD_FUNC (context) information.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
LLD_ER lld_dmac_set_func(const LLD_CH ch, const LLD_DMAC_CH_INFO *info,
			 const LLD_FUNC int_func, void *cookies,
			 const LLD_FUNC_FLAG flag)
{
	LLD_ER ret;
	unsigned long flags;
	raw_spinlock_t *lock;

	TESTCODE_DMAC_DRV_LOCK_VARS;

	/* The Channel is not available LLD channel */
	ret = lld_dmac_is_available_lld_channel(ch);
	if (ret != LLD_OK) {
		return ret;
	}

	if (info == 0) {
		return LLD_DATA_ERR;
	}

	/* Valid flow control */
	ret = lld_dmac_valid_flowctl(info);
	if (ret != LLD_OK) {
		return ret;
	}

	/* Peripheral ID is out of range */
	switch (info->flowctl) {
		/* Src is peripheral */
		case LLD_DMAC_PM_DMAC:
		case LLD_DMAC_PM_PERI:
			if (!lld_dmac_is_valid_peripheral(info->srcid)) {
				return LLD_DATA_ERR;
			}
			/* The channel cannot be used for peripheral transfer */
			if (!dma_is_peri_transferable(ch, info->srcid)) {
				return LLD_DATA_ERR;
			}
			break;
		/* Dst is peripheral */
		case LLD_DMAC_MP_DMAC:
		case LLD_DMAC_MP_PERI:
			if (!lld_dmac_is_valid_peripheral(info->dstid)) {
				return LLD_DATA_ERR;
			}
			/* The channel cannot be used for peripheral transfer */
			if (!dma_is_peri_transferable(ch, info->dstid)) {
				return LLD_DATA_ERR;
			}
			break;
		default:
			/* Nothing */
			break;
	}

	if (int_func == 0) {
		return LLD_DATA_ERR;
	}

	/* flag is out of range */
	if (!lld_dmac_is_valid_func_flag(flag)) {
		return LLD_DATA_ERR;
	}

	lock = lld_dmac_ch_mng_get_lock(ch);
	raw_spin_lock_irqsave(lock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	/* Execute set functions process */
	ret = lld_dmac_set_func_impl(ch, info, int_func, cookies, flag);

	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("lld_dmac_set_func_impl");

	raw_spin_unlock_irqrestore(lock, flags);

	return ret;
}
EXPORT_SYMBOL(lld_dmac_set_func);

/**
 * lld_dmac_set_config() - Configure the specified channel.
 *
 * @ch:		Target channel to configure.
 * @conf:	Pointer to the LLD_DMAC_CH_CONF structure.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
LLD_ER lld_dmac_set_config(const LLD_CH ch, const LLD_DMAC_CH_CONF *conf)
{
	LLD_ER ret;
	unsigned long flags;
	raw_spinlock_t *lock;

	TESTCODE_DMAC_DRV_LOCK_VARS;

	/* The channel number is invalid */
	if (!lld_dmac_is_valid_channel_number(ch)) {
		return LLD_DATA_ERR;
	}

	/* Dma macro is not enabled */
	if (!(dma_is_enabled(ch))) {
		return LLD_NOT_OPEN_ERR;
	}

	if (conf == 0) {
		return LLD_DATA_ERR;
	}

	/* Priority is out of range */
	if (!lld_dmac_is_valid_priority(conf->prio)) {
		return LLD_DATA_ERR;
	}

	/* AXI QoS is out of range */
	if (!lld_dmac_is_valid_qos(conf->aw_qos)) {
		return LLD_DATA_ERR;
	}
	if (!lld_dmac_is_valid_qos(conf->ar_qos)) {
		return LLD_DATA_ERR;
	}

	/* Outstanding is out of range */
	if (!lld_dmac_is_valid_outstand(conf->src_outstand)) {
		return LLD_DATA_ERR;
	}
	if (!lld_dmac_is_valid_outstand(conf->dst_outstand)) {
		return LLD_DATA_ERR;
	}

	lock = lld_dmac_ch_mng_get_lock(ch);
	raw_spin_lock_irqsave(lock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	/* Execute set configuartion process */
	ret = lld_dmac_set_config_impl(ch, conf);

	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("lld_dmac_set_config_impl");

	raw_spin_unlock_irqrestore(lock, flags);

	return ret;
}
EXPORT_SYMBOL(lld_dmac_set_config);

/**
 * lld_dmac_start() - Start DMA transfer.
 *
 * @ch:		Target channel to start.
 * @tran_info:	Pointer to the LLD_DMAC_TRAN_INFO structure.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
LLD_ER lld_dmac_start(const LLD_CH ch, const LLD_DMAC_TRAN_INFO *tran_info)
{
	LLD_ER ret;
	unsigned long flags;
	raw_spinlock_t *lock;

	TESTCODE_DMAC_DRV_LOCK_VARS;

	/* The Channel is not available LLD channel */
	ret = lld_dmac_is_available_lld_channel(ch);
	if (ret != LLD_OK) {
		return ret;
	}

	if (tran_info == 0) {
		return LLD_DATA_ERR;
	}

	/* Transfer count is out of range */
	if (!lld_dmac_is_valid_transfer_count(tran_info->tsz)) {
		return LLD_DATA_ERR;
	}

	/* Burst Transaction Length is out of range */
	if (!lld_dmac_is_valid_bt_length(tran_info->f4_sbsize)) {
		return LLD_DATA_ERR;
	}
	if (!lld_dmac_is_valid_bt_length(tran_info->f4_dbsize)) {
		return LLD_DATA_ERR;
	}

	/* Transfer Width is out of range */
	if (tran_info->f1_sauto != 1) {
		if (!lld_dmac_is_valid_transfer_width(tran_info->f4_swidth)) {
			return LLD_DATA_ERR;
		}
	}
	if (tran_info->f1_dauto != 1) {
		if (!lld_dmac_is_valid_transfer_width(tran_info->f4_dwidth)) {
			return LLD_DATA_ERR;
		}
	}

	lock = lld_dmac_ch_mng_get_lock(ch);
	raw_spin_lock_irqsave(lock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	/* Execute DMA transfer process */
	ret = lld_dmac_start_impl(ch, tran_info);

	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("lld_dmac_start_impl");

	raw_spin_unlock_irqrestore(lock, flags);

	return ret;
}
EXPORT_SYMBOL(lld_dmac_start);

/**
 * lld_dmac_sc_start() - Start DMA transfer with LLI.
 *
 * @ch:		Target channel to start.
 * @lli:	Pointer to the LLD_DMAC_IND_INFO structure.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
LLD_ER lld_dmac_sc_start(const LLD_CH ch, const LLD_DMAC_IND_INFO lli)
{
	LLD_ER ret;
	unsigned long flags;
	raw_spinlock_t *lock;

	TESTCODE_DMAC_DRV_LOCK_VARS;

	/* The Channel is not available LLD channel */
	ret = lld_dmac_is_available_lld_channel(ch);
	if (ret != LLD_OK) {
		return ret;
	}

	lock = lld_dmac_ch_mng_get_lock(ch);
	raw_spin_lock_irqsave(lock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	/* Execute DMA transfer process with LLI */
	ret = lld_dmac_sc_start_impl(ch, lli);

	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("lld_dmac_sc_start_impl");

	raw_spin_unlock_irqrestore(lock, flags);

	return ret;
}
EXPORT_SYMBOL(lld_dmac_sc_start);

/**
 * lld_dmac_stop() - Stop DMA transfer.
 *
 * @ch:		Target channel to stop.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
LLD_ER lld_dmac_stop(const LLD_CH ch)
{
	LLD_ER ret;
	unsigned long flags;
	raw_spinlock_t *lock;

	TESTCODE_DMAC_DRV_LOCK_VARS;

	/* The Channel is not available LLD channel */
	ret = lld_dmac_is_available_lld_channel(ch);
	if (ret != LLD_OK) {
		return ret;
	}

	lock = lld_dmac_ch_mng_get_lock(ch);
	raw_spin_lock_irqsave(lock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	/* Execute stop process */
	ret = lld_dmac_stop_imp(ch);

	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("lld_dmac_stop_imp");

	raw_spin_unlock_irqrestore(lock, flags);

	return ret;
}
EXPORT_SYMBOL(lld_dmac_stop);

/**
 * lld_dmac_check_status() - Get the operational state.
 *
 * @ch:		Target channel to check.
 *
 * @return	LLD_DMAC_TRAN_END or LLD_DMAC_TRAN_EXEC on success,
 *		otherwise error code.
 */
int lld_dmac_check_status(const LLD_CH ch)
{
	LLD_ER ret;
	int ret_status;
	unsigned long flags;
	raw_spinlock_t *lock;

	TESTCODE_DMAC_DRV_LOCK_VARS;

	/* The Channel is not available LLD channel */
	ret = lld_dmac_is_available_lld_channel(ch);
	if (ret != LLD_OK) {
		return ret;
	}

	lock = lld_dmac_ch_mng_get_lock(ch);
	raw_spin_lock_irqsave(lock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	/* Execute check status process */
	ret_status = lld_dmac_check_status_impl(ch);

	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("lld_dmac_check_status_impl");

	raw_spin_unlock_irqrestore(lock, flags);

	return ret_status;
}
EXPORT_SYMBOL(lld_dmac_check_status);

/**
 * lld_dmac_get_trans_status() - Get the state of transfer.
 *                               (transfer address and transfer count)
 *
 * @ch:		Target channel to get.
 * @dadr:	Current destination address.
 * @tsize:	Current transfer count.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
LLD_ER lld_dmac_get_trans_status(const LLD_CH ch, uint64_t *dadr, uint32_t *tsize)
{
	LLD_ER ret;
	unsigned long flags;
	raw_spinlock_t *lock;

	TESTCODE_DMAC_DRV_LOCK_VARS;

	/* The Channel is not available LLD channel */
	ret = lld_dmac_is_available_lld_channel(ch);
	if (ret != LLD_OK) {
		return ret;
	}

	if (dadr == 0) {
		return LLD_DATA_ERR;
	}

	if (tsize == 0) {
		return LLD_DATA_ERR;
	}

	lock = lld_dmac_ch_mng_get_lock(ch);
	raw_spin_lock_irqsave(lock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	/* Execute get transfer state process */
	ret = lld_dmac_get_trans_status_impl(ch, dadr, tsize);

	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("lld_dmac_get_trans_status_impl");

	raw_spin_unlock_irqrestore(lock, flags);

	return ret;
}
EXPORT_SYMBOL(lld_dmac_get_trans_status);

/**
 * lld_dmac_get_int_status() - Get the interrupt status value which is stored
 *                             when interrupt occurs.
 *
 * @ch:		Target channel to get.
 * @intstat:	Stored interrupt status.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
LLD_ER lld_dmac_get_int_status(const LLD_CH ch, uint64_t *intstat)
{
	LLD_ER ret;
	unsigned long flags;
	raw_spinlock_t *lock;

	TESTCODE_DMAC_DRV_LOCK_VARS;

	/* The Channel is not available LLD channel */
	ret = lld_dmac_is_available_lld_channel(ch);
	if (ret != LLD_OK) {
		return ret;
	}

	if (intstat == 0) {
		return LLD_DATA_ERR;
	}

	lock = lld_dmac_ch_mng_get_lock(ch);
	raw_spin_lock_irqsave(lock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	/* Execute get interrupt status process */
	ret = lld_dmac_get_int_status_impl(ch, intstat);

	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("lld_dmac_get_int_status_impl");

	raw_spin_unlock_irqrestore(lock, flags);

	return ret;
}
EXPORT_SYMBOL(lld_dmac_get_int_status);

#ifdef TESTCODE_DMA_DRV
/**
 * lld_dmac_reg_write() - Set register value.
 *
 * @reg_addr_offset:		register offset address.
 * @ch:		Target channel to set.
 *
 * @return	void
 */
void lld_dmac_reg_write(const uint32_t reg_addr_offset, const LLD_CH ch, uint64_t value)
{
	void __iomem *chan_reg;

	/* Get base address of channel registers */
	chan_reg = dma_get_chx_reg_addr(ch);

	/* Set register value */
	iowrite64(value, (chan_reg + reg_addr_offset));

	return;
}
EXPORT_SYMBOL(lld_dmac_reg_write);

/**
 * lld_dmac_reg_read() - Get register value.
 *
 * @reg_addr_offset:		register offset address.
 * @ch:		Target channel to get.
 *
 * @return	Register value
 */
uint64_t lld_dmac_reg_read(const uint32_t reg_addr_offset, const LLD_CH ch)
{
	void __iomem *chan_reg;
	uint64_t value;

	/* Get base address of channel registers */
	chan_reg = dma_get_chx_reg_addr(ch);

	/* Get register value */
	value = ioread64(chan_reg + reg_addr_offset);

	return value;
}
EXPORT_SYMBOL(lld_dmac_reg_read);

/**
 * lld_dmac_get_auto_width() - Get auto width.
 *
 * @ch:		Target channel to start.
 * @tran_info:	Pointer to the LLD_DMAC_TRAN_INFO structure.
 * @src_width: source transfer width.
 * @dst_width: destination transfer width.
 *
 * @return	LLD_OK on success, otherwise error code.
 */
LLD_ER lld_dmac_get_auto_width(const LLD_CH ch, const LLD_DMAC_TRAN_INFO *tran_info,
																			uint32_t *src_width,  uint32_t *dst_width)
{
	LLD_ER ret;
	LLD_DMAC_CH_INFO *info;
	uint32_t swidth, dwidth;

	/* Get auto data transfer width. */
	ret = lld_dmac_get_auto_transfer_width(ch, tran_info, &swidth, &dwidth);
	if (ret != LLD_OK) {
		return ret;
	}

	/* Set transfer info to register */
	info = lld_dmac_ch_mng_get_func_info(ch);
	lld_dmac_set_transfer_to_register(ch, tran_info, info, swidth, dwidth);

	*src_width = swidth;
	*dst_width = dwidth;

	return LLD_OK;
}
EXPORT_SYMBOL(lld_dmac_get_auto_width);

/**
 * lld_dmac_cmn_reg_read() - Get common register value.
 *
 * @reg_addr_offset:		register offset address.
 * @ch:		Target channel to get.
 *
 * @return	Register value
 */
uint64_t lld_dmac_cmn_reg_read(const uint32_t reg_addr_offset, const LLD_CH ch)
{
	void __iomem *cmn_reg;
	uint64_t value;

	/* Get base address of channel registers */
	cmn_reg = dma_get_cmn_reg_addr(ch);

	/* Get register value */
	value = ioread64(cmn_reg + reg_addr_offset);

	return value;
}
EXPORT_SYMBOL(lld_dmac_cmn_reg_read);
#endif

#ifdef TESTCODE_DMA_DRV_TRANS_PERF
/**
 * lld_dmac_debug_info_nsec_diff() - Get the nsec_diff.
 *
 * @return	nsec_diff.
 */
uint64_t lld_dmac_debug_info_get_nsec_diff(void)
{
	if (lld_debug_info.nsec_start > lld_debug_info.nsec_end)
		printk("%s: warning : nsec_start > nsec_end\n", __func__);

	return  lld_debug_info.nsec_end - lld_debug_info.nsec_start;
}
EXPORT_SYMBOL(lld_dmac_debug_info_get_nsec_diff);

/**
 * lld_dmac_debug_info_set_nsec_start() - Set nsec_start.
 *
 * @ts:		ktime_start.
 *
 * @return	none.
 */
void lld_dmac_debug_info_set_nsec_start(struct timespec64 ts)
{
	lld_debug_info.nsec_start = ts.tv_sec * 1000000000 + ts.tv_nsec;
}

/**
 * lld_dmac_debug_info_set_nsec_end() - Set nsec_end.
 *
 * @ts:		ktime_end.
 *
 * @return	none.
 */
void lld_dmac_debug_info_set_nsec_end(struct timespec64 ts)
{
	lld_debug_info.nsec_end = ts.tv_sec * 1000000000 + ts.tv_nsec;
}
#endif /* TESTCODE_DMA_DRV_TRANS_PERF */

#ifdef TESTCODE_DMA_DRV_LOCK_PERF
/**
 * lld_dmac_debug_info_init_max_lock_time() - Init max_lock_time.
 *
 * @return	none.
 */
void lld_dmac_debug_info_init_max_lock_time(void)
{
	lld_debug_info.nsec_max_lock_time = 0;
	lld_debug_info.func_name_1[0] = '\0';
	lld_debug_info.func_name_2[0] = '\0';
	printk("[Max Lock Time] Initialized.");
}
EXPORT_SYMBOL(lld_dmac_debug_info_init_max_lock_time);

/**
 * lld_dmac_debug_info_compare_and_set_max_lock_time() - Set max_lock_time.
 *
 * @ts_start:    ktime at lock start
 * @ts_end:      ktime at lock end
 * @func_name_1: function name infomation
 * @func_name_2: function name infomation
 *
 * @return	none.
 */
void lld_dmac_debug_info_compare_and_set_max_lock_time(struct timespec64 ts_start, struct timespec64 ts_end, const char *func_name_1, const char *func_name_2)
{
	uint64_t nsec_start;
	uint64_t nsec_end;
	uint64_t nsec_lock_time;

	nsec_start = ts_start.tv_sec * 1000000000 + ts_start.tv_nsec;
	nsec_end = ts_end.tv_sec * 1000000000 + ts_end.tv_nsec;

	if (nsec_start > nsec_end)
		printk("%s: warning : nsec_start > nsec_end\n", __func__);

	nsec_lock_time = nsec_end - nsec_start;

	if( nsec_lock_time > lld_debug_info.nsec_max_lock_time) {
		lld_debug_info.nsec_max_lock_time = nsec_lock_time;
		strcpy(lld_debug_info.func_name_1, func_name_1);
		strcpy(lld_debug_info.func_name_2, func_name_2);
		printk("[Max Lock Time]  %llu.%09llu [s] : %s %s\n", nsec_lock_time / 1000000000, nsec_lock_time % 1000000000,
			func_name_1, func_name_2);
	}
}

/**
 * lld_dmac_debug_info_get_max_lock_time() - Get max_lock_time.
 *
 * @return	max_lock_time.
 */
uint64_t lld_dmac_debug_info_get_max_lock_time(void)
{
	return lld_debug_info.nsec_max_lock_time;
}
EXPORT_SYMBOL(lld_dmac_debug_info_get_max_lock_time);
#endif /* TESTCODE_DMA_DRV_LOCK_PERF */

/**
 * lld_dmac_th_int_func() - furnction for Interrupt
 *
 * @ch:		Target channel for Interrupt
 *
 */
void lld_dmac_th_int_func(LLD_CH ch)
{
	bool interrupted;
	unsigned long flags;
	raw_spinlock_t *lock;

	TESTCODE_DMAC_DRV_LOCK_VARS;

	lock = lld_dmac_ch_mng_get_lock(ch);

	/* Interrupt has occured on the channel */
	raw_spin_lock_irqsave(lock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	interrupted = lld_dmac_ch_mng_is_interrupted(ch);

	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("lld_dmac_ch_mng_is_interrupted");

	raw_spin_unlock_irqrestore(lock, flags);

	if (interrupted) {
		LLD_FUNC int_func;
		LLD_FACT fact;
		void *cookies;

		raw_spin_lock_irqsave(lock, flags);

		TESTCODE_DMAC_DRV_LOCK_GET_START;

		/* Get callback, fact and cookies */
		int_func = lld_dmac_ch_mng_get_int_func(ch);
		fact = lld_dmac_ch_mng_get_fact(ch);
		cookies = lld_dmac_ch_mng_get_cookies(ch);

		/* Clear interrupt */
		lld_dmac_ch_mng_clear_interrupt(ch);

		TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("lld_dmac_ch_mng_get_fact/cookies/lld_dmac_ch_mng_clear_interrupt");

		raw_spin_unlock_irqrestore(lock, flags);

		/* Call callback */
		int_func(ch, fact, cookies);
	}

	return;
}

/**
 * lld_dmac_init() - Initialization function.
 *
 * @ch:		Target channel to set.
 *
 * @return	None.
 */
void lld_dmac_init(LLD_CH ch)
{
	/* Initialize hannel Management Information */
	lld_dmac_ch_mng_init_ch(ch);
}

/**
 * lld_dmac_set_priority() - Set priority in configuration for the channel.
 *
 * @ch:		Target channel to set.
 * @priority:	Priority.
 *
 * @return	None.
 */
void lld_dmac_set_priority(LLD_CH ch, u32 priority)
{
	lld_ch_mng_info[ch].ch_conf.prio = priority;
}
