// SPDX-License-Identifier:  GPL-2.0
// (C) 2017-2018 Synopsys, Inc. (www.synopsys.com)
// Copyright 2022 Sony Corporation, SOCIONEXT INC.
// Portions Copyright (C) 2024 Synopsys, Inc.  Used with permission. All rights reserved.

/*
 * Synopsys DesignWare AXI DMA Controller driver.
 *
 * Author: Eugeniy Paltsev <Eugeniy.Paltsev@synopsys.com>
 */
#include <linux/bitops.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/dmaengine.h>
#include <linux/dmapool.h>
#include <linux/dma-mapping.h>
#include <linux/err.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/iopoll.h>
#include <linux/io-64-nonatomic-lo-hi.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_dma.h>
#include <linux/platform_device.h>
#include <linux/pm_runtime.h>
#include <linux/property.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/pm.h>
#include <linux/reset.h>

#include "dw-axi-dmac.h"
#include "dw-axi-dmac-lld.h"
#include "../dmaengine.h"
#include "../virt-dma.h"

static void dma_load_registers(struct axi_dma_chip *chip);
static void dma_store_registers(struct axi_dma_chip *chip);

/*
 * The set of bus widths supported by the DMA controller. DW AXI DMAC supports
 * master data bus width up to 512 bits (for both AXI master interfaces), but
 * it depends on IP block configurarion.
 */
#define AXI_DMA_BUSWIDTHS		  \
	(DMA_SLAVE_BUSWIDTH_1_BYTE	| \
	DMA_SLAVE_BUSWIDTH_2_BYTES	| \
	DMA_SLAVE_BUSWIDTH_4_BYTES	| \
	DMA_SLAVE_BUSWIDTH_8_BYTES	| \
	DMA_SLAVE_BUSWIDTH_16_BYTES	| \
	DMA_SLAVE_BUSWIDTH_32_BYTES	| \
	DMA_SLAVE_BUSWIDTH_64_BYTES)

#define D_MASK_INT_STATUS	(0x00000000FFFFFFFFULL)

static struct axi_dma_chip *gDma_chips[DWAXIDMAC_CXD_ID_MAX + 1];

static inline void
axi_dma_iowrite32(struct axi_dma_chip *chip, u32 reg, u32 val)
{
	iowrite32(val, chip->regs + reg);
}

static inline u32 axi_dma_ioread32(struct axi_dma_chip *chip, u32 reg)
{
	return ioread32(chip->regs + reg);
}

static inline void
axi_chan_iowrite32(struct axi_dma_chan *chan, u32 reg, u32 val)
{
	iowrite32(val, chan->chan_regs + reg);
}

static inline u32 axi_chan_ioread32(struct axi_dma_chan *chan, u32 reg)
{
	return ioread32(chan->chan_regs + reg);
}

static inline uint64_t axi_chan_ioread64(struct axi_dma_chan *chan, u32 reg)
{
	return ioread64(chan->chan_regs + reg);
}

static inline void
axi_chan_iowrite64(struct axi_dma_chan *chan, u32 reg, u64 val)
{
	/*
	 * We split one 64 bit write for two 32 bit write as some HW doesn't
	 * support 64 bit access.
	 */
	iowrite32(lower_32_bits(val), chan->chan_regs + reg);
	iowrite32(upper_32_bits(val), chan->chan_regs + reg + 4);
}

static inline void axi_dma_disable(struct axi_dma_chip *chip)
{
	u32 val;

	val = axi_dma_ioread32(chip, DMAC_CFG);
	val &= ~DMAC_EN_MASK;
	axi_dma_iowrite32(chip, DMAC_CFG, val);
}

static inline void axi_dma_enable(struct axi_dma_chip *chip)
{
	u32 val;

	val = axi_dma_ioread32(chip, DMAC_CFG);
	val |= DMAC_EN_MASK;
	axi_dma_iowrite32(chip, DMAC_CFG, val);
}

static inline void axi_dma_irq_disable(struct axi_dma_chip *chip)
{
	u32 val;

	val = axi_dma_ioread32(chip, DMAC_CFG);
	val &= ~INT_EN_MASK;
	axi_dma_iowrite32(chip, DMAC_CFG, val);
}

static inline void axi_dma_irq_enable(struct axi_dma_chip *chip)
{
	u32 val;

	val = axi_dma_ioread32(chip, DMAC_CFG);
	val |= INT_EN_MASK;
	axi_dma_iowrite32(chip, DMAC_CFG, val);
}

static inline void axi_chan_irq_disable(struct axi_dma_chan *chan, u32 irq_mask)
{
	u32 val;

	if (likely(irq_mask == DWAXIDMAC_IRQ_ALL)) {
		axi_chan_iowrite32(chan, CH_INTSTATUS_ENA, DWAXIDMAC_IRQ_NONE);
	} else {
		val = axi_chan_ioread32(chan, CH_INTSTATUS_ENA);
		val &= ~irq_mask;
		axi_chan_iowrite32(chan, CH_INTSTATUS_ENA, val);
	}
}

static inline void axi_chan_irq_set(struct axi_dma_chan *chan, u32 irq_mask)
{
	axi_chan_iowrite32(chan, CH_INTSTATUS_ENA, irq_mask);
}

static inline void axi_chan_irq_sig_set(struct axi_dma_chan *chan, u32 irq_mask)
{
	axi_chan_iowrite32(chan, CH_INTSIGNAL_ENA, irq_mask);
}

static inline void axi_chan_irq_clear(struct axi_dma_chan *chan, u32 irq_mask)
{
	axi_chan_iowrite32(chan, CH_INTCLEAR, irq_mask);
}

static inline u32 axi_chan_irq_read(struct axi_dma_chan *chan)
{
	return axi_chan_ioread32(chan, CH_INTSTATUS);
}

static inline uint64_t axi_chan_irq_read64(struct axi_dma_chan *chan)
{
	return axi_chan_ioread64(chan, CH_INTSTATUS);
}

static inline void axi_chan_disable(struct axi_dma_chan *chan)
{
	u32 val;

	val = axi_dma_ioread32(chan->chip, DMAC_CHEN);
	val &= ~(BIT(chan->id) << DMAC_CHAN_EN_SHIFT);
	val |=   BIT(chan->id) << DMAC_CHAN_EN_WE_SHIFT;
	axi_dma_iowrite32(chan->chip, DMAC_CHEN, val);
}

static inline void axi_chan_enable(struct axi_dma_chan *chan)
{
	u32 val;

	TESTCODE_DMAC_DRV_TRANS_VARS;

	val = axi_dma_ioread32(chan->chip, DMAC_CHEN);
	val |= BIT(chan->id) << DMAC_CHAN_EN_SHIFT |
	       BIT(chan->id) << DMAC_CHAN_EN_WE_SHIFT;

	TESTCODE_DMAC_DRV_TRANS_GET_TIME;

	axi_dma_iowrite32(chan->chip, DMAC_CHEN, val);

	TESTCODE_DMAC_DRV_TRANS_SET_NSEC_START;
}

static inline bool axi_chan_is_hw_enable(struct axi_dma_chan *chan)
{
	u32 val;

	val = axi_dma_ioread32(chan->chip, DMAC_CHEN);

	return !!(val & (BIT(chan->id) << DMAC_CHAN_EN_SHIFT));
}

static void axi_dma_hw_init(struct axi_dma_chip *chip)
{
	u32 i;

	for (i = 0; i < chip->dw->hdata->nr_channels; i++) {
		axi_chan_irq_disable(&chip->dw->chan[i], DWAXIDMAC_IRQ_ALL);
		axi_chan_disable(&chip->dw->chan[i]);
	}
}

static u32 axi_chan_get_xfer_width(struct axi_dma_chan *chan, dma_addr_t src,
				   dma_addr_t dst, size_t len)
{
	u32 max_width = chan->chip->dw->hdata->m_data_width;

	return __ffs(src | dst | len | BIT(max_width));
}

static inline const char *axi_chan_name(struct axi_dma_chan *chan)
{
	return dma_chan_name(&chan->vc.chan);
}

static struct axi_dma_desc *axi_desc_alloc(u32 num)
{
	struct axi_dma_desc *desc;

	desc = kzalloc(sizeof(*desc), GFP_NOWAIT);
	if (!desc)
		return NULL;

	desc->hw_desc = kcalloc(num, sizeof(*desc->hw_desc), GFP_NOWAIT);
	if (!desc->hw_desc) {
		kfree(desc);
		return NULL;
	}
	desc->nr_hw_descs = num;

	return desc;
}

static struct axi_dma_lli *axi_desc_get(struct axi_dma_chan *chan,
					dma_addr_t *addr)
{
	struct axi_dma_lli *lli;
	dma_addr_t phys;

	lli = dma_pool_zalloc(chan->desc_pool, GFP_NOWAIT, &phys);
	if (unlikely(!lli)) {
		dev_err(chan2dev(chan), "%s: not enough descriptors available\n",
			axi_chan_name(chan));
		return NULL;
	}

	atomic_inc(&chan->descs_allocated);
	*addr = phys;

	return lli;
}

static void axi_desc_put(struct axi_dma_desc *desc)
{
	struct axi_dma_chan *chan = desc->chan;
	int count = desc->nr_hw_descs;
	struct axi_dma_hw_desc *hw_desc;
	int descs_put, put_cnt = 0;

	for (descs_put = 0; descs_put < count; descs_put++) {
		hw_desc = &desc->hw_desc[descs_put];
		if (hw_desc->llp) {
			dma_pool_free(chan->desc_pool, hw_desc->lli, hw_desc->llp);
			hw_desc->llp = 0;
			put_cnt++;
		}
	}

	kfree(desc->hw_desc);
	kfree(desc);
	atomic_sub(put_cnt, &chan->descs_allocated);
	dev_vdbg(chan2dev(chan), "%s: %d descs put, %d still allocated\n",
		axi_chan_name(chan), descs_put,
		atomic_read(&chan->descs_allocated));
}

static void vchan_desc_put(struct virt_dma_desc *vdesc)
{
	axi_desc_put(vd_to_axi_desc(vdesc));
}

static enum dma_status
dma_chan_tx_status(struct dma_chan *dchan, dma_cookie_t cookie,
		  struct dma_tx_state *txstate)
{
	struct axi_dma_chan *chan = dchan_to_axi_dma_chan(dchan);
	struct virt_dma_desc *vdesc;
	enum dma_status status;
	u32 completed_length;
	unsigned long flags;
	u32 completed_blocks;
	size_t bytes = 0;
	u32 length;
	u32 len;

	TESTCODE_DMAC_DRV_LOCK_VARS;

	status = dma_cookie_status(dchan, cookie, txstate);
	if (status == DMA_COMPLETE || !txstate)
		return status;

	spin_lock_irqsave(&chan->vc.lock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	vdesc = vchan_find_desc(&chan->vc, cookie);
	if (vdesc) {
		length = vd_to_axi_desc(vdesc)->length;
		completed_blocks = vd_to_axi_desc(vdesc)->completed_blocks;
		len = vd_to_axi_desc(vdesc)->hw_desc[0].len;
		completed_length = completed_blocks * len;
		bytes = length - completed_length;
	}

	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("vchan_find_desc");

	spin_unlock_irqrestore(&chan->vc.lock, flags);

	dma_set_residue(txstate, bytes);

	return status;
}

static void write_desc_llp(struct axi_dma_hw_desc *desc, dma_addr_t adr)
{
	desc->lli->llp = cpu_to_le64(adr);
}

static void write_chan_llp(struct axi_dma_chan *chan, dma_addr_t adr)
{
	axi_chan_iowrite64(chan, CH_LLP, adr);
}

static void dw_axi_dma_set_byte_halfword(struct axi_dma_chan *chan, bool set)
{
	u32 offset = DMAC_APB_BYTE_WR_CH_EN;
	u32 reg_width, val;

	if (!chan->chip->apb_regs) {
		dev_dbg(chan->chip->dev, "apb_regs not initialized\n");
		return;
	}

	reg_width = __ffs(chan->config.dst_addr_width);
	if (reg_width == DWAXIDMAC_TRANS_WIDTH_16)
		offset = DMAC_APB_HALFWORD_WR_CH_EN;

	val = ioread32(chan->chip->apb_regs + offset);

	if (set)
		val |= BIT(chan->id);
	else
		val &= ~BIT(chan->id);

	iowrite32(val, chan->chip->apb_regs + offset);
}
/* Called in chan locked context */
static void axi_chan_block_xfer_start(struct axi_dma_chan *chan,
				      struct axi_dma_desc *first)
{
	u32 reg, irq_mask;
	u8 lms = 0; /* Select AXI0 master for LLI fetching */
	LLD_CH lld_ch;
	LLD_DMAC_CH_CONF *lld_conf;
	unsigned long flags;

	TESTCODE_DMAC_DRV_LOCK_VARS;

	if (unlikely(axi_chan_is_hw_enable(chan))) {
		dev_err(chan2dev(chan), "%s is non-idle!\n",
			axi_chan_name(chan));

		return;
	}

	raw_spin_lock_irqsave(&chan->chip->spinlock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	axi_dma_enable(chan->chip);

	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("axi_dma_enable");

	raw_spin_unlock_irqrestore(&chan->chip->spinlock, flags);

	lld_ch = chan->chip->dw->hdata->cxd_id * DMAC_MAX_CHANNELS + chan->id;
	lld_conf = lld_dmac_ch_mng_get_config(lld_ch);

	reg = (DWAXIDMAC_MBLK_TYPE_LL << CH_CFG_L_DST_MULTBLK_TYPE_POS |
	       DWAXIDMAC_MBLK_TYPE_LL << CH_CFG_L_SRC_MULTBLK_TYPE_POS);
	axi_chan_iowrite32(chan, CH_CFG_L, reg);

	reg = (DWAXIDMAC_TT_FC_MEM_TO_MEM_DMAC << CH_CFG_H_TT_FC_POS |
		(lld_conf->prio << CH_CFG_H_PRIORITY_POS) |
		DWAXIDMAC_HS_SEL_HW << CH_CFG_H_HS_SEL_DST_POS |
		DWAXIDMAC_HS_SEL_HW << CH_CFG_H_HS_SEL_SRC_POS);
	switch (chan->direction) {
	case DMA_MEM_TO_DEV:
		dw_axi_dma_set_byte_halfword(chan, true);
		reg |= (chan->config.device_fc ?
			DWAXIDMAC_TT_FC_MEM_TO_PER_DST :
			DWAXIDMAC_TT_FC_MEM_TO_PER_DMAC)
			<< CH_CFG_H_TT_FC_POS;
		if (chan->chip->apb_regs) {
			reg |= (chan->id << CH_CFG_H_DST_PER_POS);
		} else {
			reg |= ((chan->hw_handshake_num << CH_CFG_H_DST_PER_POS) |
					(chan->chip->dw->hdata->cxd_outstand[(chan->hw_handshake_num * 2) + 1] << CH_CFG_H_SRC_OSR_LMT_POS) |
					(chan->chip->dw->hdata->cxd_outstand[chan->hw_handshake_num * 2] << CH_CFG_H_DST_OSR_LMT_POS));
		}
		break;
	case DMA_DEV_TO_MEM:
		reg |= (chan->config.device_fc ?
			DWAXIDMAC_TT_FC_PER_TO_MEM_SRC :
			DWAXIDMAC_TT_FC_PER_TO_MEM_DMAC)
			<< CH_CFG_H_TT_FC_POS;
		if (chan->chip->apb_regs) {
			reg |= (chan->id << CH_CFG_H_SRC_PER_POS);
		} else {
			reg |= ((chan->hw_handshake_num << CH_CFG_H_SRC_PER_POS) |
					(chan->chip->dw->hdata->cxd_outstand[chan->hw_handshake_num * 2] << CH_CFG_H_SRC_OSR_LMT_POS) |
					(chan->chip->dw->hdata->cxd_outstand[(chan->hw_handshake_num * 2) + 1] << CH_CFG_H_DST_OSR_LMT_POS));
		}
		break;
	default:
		reg |= ((lld_conf->src_outstand << CH_CFG_H_SRC_OSR_LMT_POS) |
				(lld_conf->dst_outstand << CH_CFG_H_DST_OSR_LMT_POS));
		break;
	}
	axi_chan_iowrite32(chan, CH_CFG_H, reg);

	reg = (lld_conf->ar_qos << CH_CFG_AXI_ARQOS_POS) |
		(lld_conf->aw_qos << CH_CFG_AXI_AWQOS_POS);
	axi_chan_iowrite32(chan, CH_AXI_QOS, reg);

	write_chan_llp(chan, first->hw_desc[0].llp | lms);

	irq_mask = DWAXIDMAC_IRQ_DMA_TRF | DWAXIDMAC_IRQ_ALL_ERR;
	axi_chan_irq_sig_set(chan, irq_mask);

	/* Generate 'suspend' status but don't generate interrupt */
	irq_mask |= DWAXIDMAC_IRQ_SUSPENDED;
	axi_chan_irq_set(chan, irq_mask);

	axi_chan_enable(chan);
}

static void axi_chan_start_first_queued(struct axi_dma_chan *chan)
{
	struct axi_dma_desc *desc;
	struct virt_dma_desc *vd;

	vd = vchan_next_desc(&chan->vc);
	if (!vd)
		return;

	desc = vd_to_axi_desc(vd);
	dev_vdbg(chan2dev(chan), "%s: started %u\n", axi_chan_name(chan),
		vd->tx.cookie);
	axi_chan_block_xfer_start(chan, desc);
}

static void dma_chan_issue_pending(struct dma_chan *dchan)
{
	struct axi_dma_chan *chan = dchan_to_axi_dma_chan(dchan);
	unsigned long flags;

	TESTCODE_DMAC_DRV_LOCK_VARS;

	spin_lock_irqsave(&chan->vc.lock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	if (vchan_issue_pending(&chan->vc))
		axi_chan_start_first_queued(chan);

	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("vchan_issue_pending");

	spin_unlock_irqrestore(&chan->vc.lock, flags);
}

static void dw_axi_dma_synchronize(struct dma_chan *dchan)
{
	struct axi_dma_chan *chan = dchan_to_axi_dma_chan(dchan);

	vchan_synchronize(&chan->vc);
}

static int dma_chan_alloc_chan_resources(struct dma_chan *dchan)
{
	struct axi_dma_chan *chan = dchan_to_axi_dma_chan(dchan);

	pm_runtime_get_sync(chan->chip->dev);

	/* ASSERT: channel is idle */
	if (axi_chan_is_hw_enable(chan)) {
		dev_err(chan2dev(chan), "%s is non-idle!\n",
			axi_chan_name(chan));
		pm_runtime_put_sync(chan->chip->dev);
		return -EBUSY;
	}

	/* LLI address must be aligned to a 64-byte boundary */
	chan->desc_pool = dma_pool_create(dev_name(chan2dev(chan)),
					  chan->chip->dev,
					  sizeof(struct axi_dma_lli),
					  64, 0);
	if (!chan->desc_pool) {
		dev_err(chan2dev(chan), "No memory for descriptors\n");
		pm_runtime_put_sync(chan->chip->dev);
		return -ENOMEM;
	}
	dev_vdbg(dchan2dev(dchan), "%s: allocating\n", axi_chan_name(chan));

	return 0;
}

static void dma_chan_free_chan_resources(struct dma_chan *dchan)
{
	struct axi_dma_chan *chan = dchan_to_axi_dma_chan(dchan);

	/* ASSERT: channel is idle */
	if (axi_chan_is_hw_enable(chan))
		dev_err(dchan2dev(dchan), "%s is non-idle!\n",
			axi_chan_name(chan));

	axi_chan_disable(chan);
	axi_chan_irq_disable(chan, DWAXIDMAC_IRQ_ALL);

	vchan_free_chan_resources(&chan->vc);

	dma_pool_destroy(chan->desc_pool);
	chan->desc_pool = NULL;
	dev_vdbg(dchan2dev(dchan),
		 "%s: free resources, descriptor still allocated: %u\n",
		 axi_chan_name(chan), atomic_read(&chan->descs_allocated));

	pm_runtime_put_sync(chan->chip->dev);
}

static void dw_axi_dma_set_hw_channel(struct axi_dma_chan *chan, bool set)
{
	struct axi_dma_chip *chip = chan->chip;
	unsigned long reg_value, val;

	if (!chip->apb_regs) {
		dev_dbg(chip->dev, "apb_regs not initialized\n");
		return;
	}

	/*
	 * An unused DMA channel has a default value of 0x3F.
	 * Lock the DMA channel by assign a handshake number to the channel.
	 * Unlock the DMA channel by assign 0x3F to the channel.
	 */
	if (set)
		val = chan->hw_handshake_num;
	else
		val = UNUSED_CHANNEL;

	reg_value = lo_hi_readq(chip->apb_regs + DMAC_APB_HW_HS_SEL_0);

	/* Channel is already allocated, set handshake as per channel ID */
	/* 64 bit write should handle for 8 channels */

	reg_value &= ~(DMA_APB_HS_SEL_MASK <<
			(chan->id * DMA_APB_HS_SEL_BIT_SIZE));
	reg_value |= (val << (chan->id * DMA_APB_HS_SEL_BIT_SIZE));
	lo_hi_writeq(reg_value, chip->apb_regs + DMAC_APB_HW_HS_SEL_0);

	return;
}

/*
 * If DW_axi_dmac sees CHx_CTL.ShadowReg_Or_LLI_Last bit of the fetched LLI
 * as 1, it understands that the current block is the final block in the
 * transfer and completes the DMA transfer operation at the end of current
 * block transfer.
 */
static void set_desc_last(struct axi_dma_hw_desc *desc)
{
	u32 val;

	val = le32_to_cpu(desc->lli->ctl_hi);
	val |= CH_CTL_H_LLI_LAST;
	desc->lli->ctl_hi = cpu_to_le32(val);
}

static void write_desc_sar(struct axi_dma_hw_desc *desc, dma_addr_t adr)
{
	desc->lli->sar = cpu_to_le64(adr);
}

static void write_desc_dar(struct axi_dma_hw_desc *desc, dma_addr_t adr)
{
	desc->lli->dar = cpu_to_le64(adr);
}

static void set_desc_src_master(struct axi_dma_hw_desc *desc)
{
	u32 val;

	/* Select AXI0 for source master */
	val = le32_to_cpu(desc->lli->ctl_lo);
	val &= ~CH_CTL_L_SRC_MAST;
	desc->lli->ctl_lo = cpu_to_le32(val);
}

static void set_desc_dest_master(struct axi_dma_hw_desc *hw_desc,
				 struct axi_dma_desc *desc)
{
	u32 val;

	/* Select AXI1 for source master if available */
	val = le32_to_cpu(hw_desc->lli->ctl_lo);
	if (desc->chan->chip->dw->hdata->nr_masters > 1)
		val |= CH_CTL_L_DST_MAST;
	else
		val &= ~CH_CTL_L_DST_MAST;

	hw_desc->lli->ctl_lo = cpu_to_le32(val);
}

static uint32_t dma_get_peri_msize(uint32_t max_burst)
{
	uint32_t peri_msize;

	switch (max_burst) {
		case DWAXIDMAC_CXD_MAX_BURST_1:
			peri_msize = DWAXIDMAC_BURST_TRANS_LEN_1;
			break;
		case DWAXIDMAC_CXD_MAX_BURST_4:
			peri_msize = DWAXIDMAC_BURST_TRANS_LEN_4;
			break;
		case DWAXIDMAC_CXD_MAX_BURST_8:
			peri_msize = DWAXIDMAC_BURST_TRANS_LEN_8;
			break;
		case DWAXIDMAC_CXD_MAX_BURST_16:
			peri_msize = DWAXIDMAC_BURST_TRANS_LEN_16;
			break;
		case DWAXIDMAC_CXD_MAX_BURST_32:
			peri_msize = DWAXIDMAC_BURST_TRANS_LEN_32;
			break;
		case DWAXIDMAC_CXD_MAX_BURST_64:
			peri_msize = DWAXIDMAC_BURST_TRANS_LEN_64;
			break;
		case DWAXIDMAC_CXD_MAX_BURST_128:
			peri_msize = DWAXIDMAC_BURST_TRANS_LEN_128;
			break;
		case DWAXIDMAC_CXD_MAX_BURST_256:
			peri_msize = DWAXIDMAC_BURST_TRANS_LEN_256;
			break;
		case DWAXIDMAC_CXD_MAX_BURST_512:
			peri_msize = DWAXIDMAC_BURST_TRANS_LEN_512;
			break;
		case DWAXIDMAC_CXD_MAX_BURST_1024:
			peri_msize = DWAXIDMAC_BURST_TRANS_LEN_1024;
			break;
		default:
			peri_msize = DWAXIDMAC_BURST_TRANS_LEN_UNDEFINED;
			break;
	}

	return peri_msize;
}

static int dw_axi_dma_set_hw_desc(struct axi_dma_chan *chan,
				  struct axi_dma_hw_desc *hw_desc,
				  dma_addr_t mem_addr, size_t len)
{
	unsigned int data_width = BIT(chan->chip->dw->hdata->m_data_width);
	unsigned int reg_width;
	unsigned int mem_width;
	dma_addr_t device_addr;
	size_t axi_block_ts;
	size_t block_ts;
	u32 ctllo, ctlhi;
	u32 burst_len;
	uint32_t perimsize;

	axi_block_ts = chan->chip->dw->hdata->block_size[chan->id];

	mem_width = __ffs(data_width | mem_addr | len);
	if (mem_width > DWAXIDMAC_TRANS_WIDTH_32)
		mem_width = DWAXIDMAC_TRANS_WIDTH_32;

	switch (chan->direction) {
	case DMA_MEM_TO_DEV:
		reg_width = __ffs(chan->config.dst_addr_width);
		perimsize = dma_get_peri_msize(chan->config.dst_maxburst);
		if (perimsize == DWAXIDMAC_BURST_TRANS_LEN_UNDEFINED) {
			return -EINVAL;
		}
		device_addr = chan->config.dst_addr;
		ctllo = reg_width << CH_CTL_L_DST_WIDTH_POS |
			mem_width << CH_CTL_L_SRC_WIDTH_POS |
			DWAXIDMAC_CH_CTL_L_NOINC << CH_CTL_L_DST_INC_POS |
			DWAXIDMAC_CH_CTL_L_INC << CH_CTL_L_SRC_INC_POS |
			perimsize << CH_CTL_L_DST_MSIZE_POS |
			DWAXIDMAC_BURST_TRANS_LEN_4 << CH_CTL_L_SRC_MSIZE_POS;
		block_ts = len >> mem_width;
		break;
	case DMA_DEV_TO_MEM:
		reg_width = __ffs(chan->config.src_addr_width);
		perimsize = dma_get_peri_msize(chan->config.src_maxburst);
		if (perimsize == DWAXIDMAC_BURST_TRANS_LEN_UNDEFINED) {
			return -EINVAL;
		}
		device_addr = chan->config.src_addr;
		ctllo = reg_width << CH_CTL_L_SRC_WIDTH_POS |
			mem_width << CH_CTL_L_DST_WIDTH_POS |
			DWAXIDMAC_CH_CTL_L_INC << CH_CTL_L_DST_INC_POS |
			DWAXIDMAC_CH_CTL_L_NOINC << CH_CTL_L_SRC_INC_POS |
			DWAXIDMAC_BURST_TRANS_LEN_4 << CH_CTL_L_DST_MSIZE_POS |
			perimsize << CH_CTL_L_SRC_MSIZE_POS;
		block_ts = len >> reg_width;
		break;
	default:
		return -EINVAL;
	}

	if (block_ts > axi_block_ts)
		return -EINVAL;

	hw_desc->lli = axi_desc_get(chan, &hw_desc->llp);
	if (unlikely(!hw_desc->lli))
		return -ENOMEM;

	ctlhi = CH_CTL_H_LLI_VALID;

	/* burst_len of DEV is 0 as fixed value */
	switch (chan->direction) {
	case DMA_MEM_TO_DEV:
		if (chan->chip->dw->hdata->restrict_axi_burst_len) {
			burst_len = chan->chip->dw->hdata->axi_rw_burst_len - 1;
			ctlhi |= CH_CTL_H_ARLEN_EN | burst_len << CH_CTL_H_ARLEN_POS;
		}
		ctlhi |= CH_CTL_H_AWLEN_EN | 0 << CH_CTL_H_AWLEN_POS;
		break;
	case DMA_DEV_TO_MEM:
		ctlhi |= CH_CTL_H_ARLEN_EN | 0 << CH_CTL_H_ARLEN_POS;
		if (chan->chip->dw->hdata->restrict_axi_burst_len) {
			burst_len = chan->chip->dw->hdata->axi_rw_burst_len - 1;
			ctlhi |= CH_CTL_H_AWLEN_EN | burst_len << CH_CTL_H_AWLEN_POS;
		}
		break;
	default:
		return -EINVAL;
	}

	hw_desc->lli->ctl_hi = cpu_to_le32(ctlhi);

	if (chan->direction == DMA_MEM_TO_DEV) {
		write_desc_sar(hw_desc, mem_addr);
		write_desc_dar(hw_desc, device_addr);
	} else {
		write_desc_sar(hw_desc, device_addr);
		write_desc_dar(hw_desc, mem_addr);
	}

	hw_desc->lli->block_ts_lo = cpu_to_le32(block_ts - 1);

	hw_desc->lli->ctl_lo = cpu_to_le32(ctllo);

	set_desc_src_master(hw_desc);

	hw_desc->len = len;
	return 0;
}

static size_t calculate_block_len(struct axi_dma_chan *chan,
				  dma_addr_t dma_addr, size_t buf_len,
				  enum dma_transfer_direction direction)
{
	u32 data_width, reg_width, mem_width;
	size_t axi_block_ts, block_len;

	axi_block_ts = chan->chip->dw->hdata->block_size[chan->id];

	switch (direction) {
	case DMA_MEM_TO_DEV:
		data_width = BIT(chan->chip->dw->hdata->m_data_width);
		mem_width = __ffs(data_width | dma_addr | buf_len);
		if (mem_width > DWAXIDMAC_TRANS_WIDTH_32)
			mem_width = DWAXIDMAC_TRANS_WIDTH_32;

		block_len = axi_block_ts << mem_width;
		break;
	case DMA_DEV_TO_MEM:
		reg_width = __ffs(chan->config.src_addr_width);
		block_len = axi_block_ts << reg_width;
		break;
	default:
		block_len = 0;
	}

	return block_len;
}

/* Helper functions for LLD API */
#define dma_get_id(lld_ch)	(lld_ch / DMAC_MAX_CHANNELS)

uint32_t dma_get_peri_width(LLD_CH lld_ch, uint32_t peri_id)
{
	uint32_t peri_width = 0;
	uint8_t macro_id;

	macro_id = dma_get_id(lld_ch);
	peri_width = gDma_chips[macro_id]->dw->hdata->cxd_peri_width[peri_id];
	return peri_width;
}

bool dma_is_peri_transferable(LLD_CH lld_ch, uint8_t peri_id)
{
	bool peri_transferable = false;
	uint32_t *peri_width;
	uint8_t macro_id;

	macro_id = dma_get_id(lld_ch);
	peri_width = gDma_chips[macro_id]->dw->hdata->cxd_peri_width;
	if (peri_width[peri_id] <= DWAXIDMAC_CXD_PERIWIDTH_MAX) {
		peri_transferable = true;
	}

	return peri_transferable;
}

bool dma_is_enabled(LLD_CH lld_ch)
{
	uint8_t macro_id;
	bool is_dma_enable = false;

	macro_id = dma_get_id(lld_ch);
	if (gDma_chips[macro_id] != NULL) {
		is_dma_enable = true;
	}

	return is_dma_enable;
}

bool dma_is_controlled_by_dmaengine(LLD_CH lld_ch)
{
	bool controlled_by_dmaengine = false;
	uint32_t cxd_dmaengine;
	u32 nr_channels;
	uint8_t macro_id, chan_id;

	macro_id = dma_get_id(lld_ch);
	chan_id = lld_ch - (macro_id * DMAC_MAX_CHANNELS);
	cxd_dmaengine = gDma_chips[macro_id]->dw->hdata->cxd_dmaengine;
	nr_channels = gDma_chips[macro_id]->dw->hdata->nr_channels;
	if (((cxd_dmaengine >> chan_id) & DWAXIDMAC_CXD_BITMASK_0x01) == DWAXIDMAC_CXD_BITMASK_0x01) {
		controlled_by_dmaengine = true;
	}
	return controlled_by_dmaengine;
}

void __iomem *dma_get_cmn_reg_addr(LLD_CH lld_ch)
{
	void __iomem *cmn_reg_addr;
	uint8_t macro_id;

	macro_id = dma_get_id(lld_ch);
	cmn_reg_addr = gDma_chips[macro_id]->regs;
	return cmn_reg_addr;
}

void __iomem *dma_get_chx_reg_addr(LLD_CH lld_ch)
{
	void __iomem *chx_reg_addr;
	uint8_t macro_id, chan_id;

	macro_id = dma_get_id(lld_ch);
	chan_id = lld_ch - (macro_id * DMAC_MAX_CHANNELS);
	chx_reg_addr = gDma_chips[macro_id]->dw->chan[chan_id].chan_regs;
	return chx_reg_addr;
}

void dma_pm_runtime_get(LLD_CH lld_ch)
{
	uint8_t macro_id;

	macro_id = dma_get_id(lld_ch);
	pm_runtime_get_sync(gDma_chips[macro_id]->dev);
}

void dma_pm_runtime_put(LLD_CH lld_ch)
{
	uint8_t macro_id;

	macro_id = dma_get_id(lld_ch);
	pm_runtime_put_sync(gDma_chips[macro_id]->dev);
}

static struct dma_async_tx_descriptor *
dw_axi_dma_chan_prep_cyclic(struct dma_chan *dchan, dma_addr_t dma_addr,
			    size_t buf_len, size_t period_len,
			    enum dma_transfer_direction direction,
			    unsigned long flags)
{
	struct axi_dma_chan *chan = dchan_to_axi_dma_chan(dchan);
	struct axi_dma_hw_desc *hw_desc = NULL;
	struct axi_dma_desc *desc = NULL;
	dma_addr_t src_addr = dma_addr;
	u32 num_periods, num_segments;
	size_t axi_block_len;
	u32 total_segments;
	u32 segment_len;
	unsigned int i;
	int status;
	u64 llp = 0;
	u8 lms = 0; /* Select AXI0 master for LLI fetching */

	num_periods = buf_len / period_len;

	axi_block_len = calculate_block_len(chan, dma_addr, buf_len, direction);
	if (axi_block_len == 0)
		return NULL;

	num_segments = DIV_ROUND_UP(period_len, axi_block_len);
	segment_len = DIV_ROUND_UP(period_len, num_segments);

	total_segments = num_periods * num_segments;

	desc = axi_desc_alloc(total_segments);
	if (unlikely(!desc))
		goto err_desc_get;

	chan->direction = direction;
	desc->chan = chan;
	chan->cyclic = true;
	desc->length = 0;
	desc->period_len = period_len;

	for (i = 0; i < total_segments; i++) {
		hw_desc = &desc->hw_desc[i];

		status = dw_axi_dma_set_hw_desc(chan, hw_desc, src_addr,
						segment_len);
		if (status < 0)
			goto err_desc_get;

		desc->length += hw_desc->len;
		/* Set end-of-link to the linked descriptor, so that cyclic
		 * callback function can be triggered during interrupt.
		 */
		set_desc_last(hw_desc);

		src_addr += segment_len;
	}

	llp = desc->hw_desc[0].llp;

	/* Managed transfer list */
	do {
		hw_desc = &desc->hw_desc[--total_segments];
		write_desc_llp(hw_desc, llp | lms);
		llp = hw_desc->llp;
	} while (total_segments);

	dw_axi_dma_set_hw_channel(chan, true);

	return vchan_tx_prep(&chan->vc, &desc->vd, flags);

err_desc_get:
	if (desc)
		axi_desc_put(desc);

	return NULL;
}

static struct dma_async_tx_descriptor *
dw_axi_dma_chan_prep_slave_sg(struct dma_chan *dchan, struct scatterlist *sgl,
			      unsigned int sg_len,
			      enum dma_transfer_direction direction,
			      unsigned long flags, void *context)
{
	struct axi_dma_chan *chan = dchan_to_axi_dma_chan(dchan);
	struct axi_dma_hw_desc *hw_desc = NULL;
	struct axi_dma_desc *desc = NULL;
	u32 num_segments, segment_len;
	unsigned int loop = 0;
	struct scatterlist *sg;
	size_t axi_block_len;
	u32 len, num_sgs = 0;
	unsigned int i;
	dma_addr_t mem;
	int status;
	u64 llp = 0;
	u8 lms = 0; /* Select AXI0 master for LLI fetching */

	if (unlikely(!is_slave_direction(direction) || !sg_len))
		return NULL;

	mem = sg_dma_address(sgl);
	len = sg_dma_len(sgl);

	axi_block_len = calculate_block_len(chan, mem, len, direction);
	if (axi_block_len == 0)
		return NULL;

	for_each_sg(sgl, sg, sg_len, i)
		num_sgs += DIV_ROUND_UP(sg_dma_len(sg), axi_block_len);

	desc = axi_desc_alloc(num_sgs);
	if (unlikely(!desc))
		goto err_desc_get;

	desc->chan = chan;
	desc->length = 0;
	chan->direction = direction;

	for_each_sg(sgl, sg, sg_len, i) {
		mem = sg_dma_address(sg);
		len = sg_dma_len(sg);
		num_segments = DIV_ROUND_UP(sg_dma_len(sg), axi_block_len);
		segment_len = DIV_ROUND_UP(sg_dma_len(sg), num_segments);

		do {
			hw_desc = &desc->hw_desc[loop++];
			status = dw_axi_dma_set_hw_desc(chan, hw_desc, mem, segment_len);
			if (status < 0)
				goto err_desc_get;

			desc->length += hw_desc->len;
			len -= segment_len;
			mem += segment_len;
		} while (len >= segment_len);
	}

	/* Set end-of-link to the last link descriptor of list */
	set_desc_last(&desc->hw_desc[num_sgs - 1]);

	/* Managed transfer list */
	do {
		hw_desc = &desc->hw_desc[--num_sgs];
		write_desc_llp(hw_desc, llp | lms);
		llp = hw_desc->llp;
	} while (num_sgs);

	dw_axi_dma_set_hw_channel(chan, true);

	return vchan_tx_prep(&chan->vc, &desc->vd, flags);

err_desc_get:
	if (desc)
		axi_desc_put(desc);

	return NULL;
}

static struct dma_async_tx_descriptor *
dma_chan_prep_dma_memcpy(struct dma_chan *dchan, dma_addr_t dst_adr,
			 dma_addr_t src_adr, size_t len, unsigned long flags)
{
	struct axi_dma_chan *chan = dchan_to_axi_dma_chan(dchan);
	size_t block_ts, max_block_ts, xfer_len;
	struct axi_dma_hw_desc *hw_desc = NULL;
	struct axi_dma_desc *desc = NULL;
	u32 xfer_width, reg, num;
	u64 llp = 0;
	u8 lms = 0; /* Select AXI0 master for LLI fetching */

	dev_dbg(chan2dev(chan), "%s: memcpy: src: %pad dst: %pad length: %zd flags: %#lx",
		axi_chan_name(chan), &src_adr, &dst_adr, len, flags);

	max_block_ts = chan->chip->dw->hdata->block_size[chan->id];
	xfer_width = axi_chan_get_xfer_width(chan, src_adr, dst_adr, len);
	num = DIV_ROUND_UP(len, max_block_ts << xfer_width);
	desc = axi_desc_alloc(num);
	if (unlikely(!desc))
		goto err_desc_get;

	desc->chan = chan;
	num = 0;
	desc->length = 0;
	while (len) {
		xfer_len = len;

		hw_desc = &desc->hw_desc[num];
		/*
		 * Take care for the alignment.
		 * Actually source and destination widths can be different, but
		 * make them same to be simpler.
		 */
		xfer_width = axi_chan_get_xfer_width(chan, src_adr, dst_adr, xfer_len);

		/*
		 * block_ts indicates the total number of data of width
		 * to be transferred in a DMA block transfer.
		 * BLOCK_TS register should be set to block_ts - 1
		 */
		block_ts = xfer_len >> xfer_width;
		if (block_ts > max_block_ts) {
			block_ts = max_block_ts;
			xfer_len = max_block_ts << xfer_width;
		}

		hw_desc->lli = axi_desc_get(chan, &hw_desc->llp);
		if (unlikely(!hw_desc->lli))
			goto err_desc_get;

		write_desc_sar(hw_desc, src_adr);
		write_desc_dar(hw_desc, dst_adr);
		hw_desc->lli->block_ts_lo = cpu_to_le32(block_ts - 1);

		reg = CH_CTL_H_LLI_VALID;
		if (chan->chip->dw->hdata->restrict_axi_burst_len) {
			u32 burst_len = chan->chip->dw->hdata->axi_rw_burst_len;

			reg |= (CH_CTL_H_ARLEN_EN |
				burst_len << CH_CTL_H_ARLEN_POS |
				CH_CTL_H_AWLEN_EN |
				burst_len << CH_CTL_H_AWLEN_POS);
		}
		hw_desc->lli->ctl_hi = cpu_to_le32(reg);

		reg = (DWAXIDMAC_BURST_TRANS_LEN_4 << CH_CTL_L_DST_MSIZE_POS |
		       DWAXIDMAC_BURST_TRANS_LEN_4 << CH_CTL_L_SRC_MSIZE_POS |
		       xfer_width << CH_CTL_L_DST_WIDTH_POS |
		       xfer_width << CH_CTL_L_SRC_WIDTH_POS |
		       DWAXIDMAC_CH_CTL_L_INC << CH_CTL_L_DST_INC_POS |
		       DWAXIDMAC_CH_CTL_L_INC << CH_CTL_L_SRC_INC_POS);
		hw_desc->lli->ctl_lo = cpu_to_le32(reg);

		set_desc_src_master(hw_desc);
		set_desc_dest_master(hw_desc, desc);

		hw_desc->len = xfer_len;
		desc->length += hw_desc->len;
		/* update the length and addresses for the next loop cycle */
		len -= xfer_len;
		dst_adr += xfer_len;
		src_adr += xfer_len;
		num++;
	}

	/* Set end-of-link to the last link descriptor of list */
	set_desc_last(&desc->hw_desc[num - 1]);
	/* Managed transfer list */
	do {
		hw_desc = &desc->hw_desc[--num];
		write_desc_llp(hw_desc, llp | lms);
		llp = hw_desc->llp;
	} while (num);

	return vchan_tx_prep(&chan->vc, &desc->vd, flags);

err_desc_get:
	if (desc)
		axi_desc_put(desc);
	return NULL;
}

static int dw_axi_dma_chan_slave_config(struct dma_chan *dchan,
					struct dma_slave_config *config)
{
	struct axi_dma_chan *chan = dchan_to_axi_dma_chan(dchan);

	/* If dma transfers from/to peripheral, check transfer width */
	if (config->direction == DMA_DEV_TO_MEM) {
		if ((config->src_addr_width == DMA_SLAVE_BUSWIDTH_UNDEFINED) ||
			(config->src_addr_width > DMA_SLAVE_BUSWIDTH_4_BYTES)) {
			return -ENOTSUPP;
		}
	} else if (config->direction == DMA_MEM_TO_DEV) {
		if ((config->dst_addr_width == DMA_SLAVE_BUSWIDTH_UNDEFINED) ||
			(config->dst_addr_width > DMA_SLAVE_BUSWIDTH_4_BYTES)) {
			return -ENOTSUPP;
		}
	}

	memcpy(&chan->config, config, sizeof(*config));

	return 0;
}

static void axi_chan_dump_lli(struct axi_dma_chan *chan,
			      struct axi_dma_hw_desc *desc)
{
	if (!desc->lli) {
		dev_err(dchan2dev(&chan->vc.chan), "NULL LLI\n");
		return;
	}

	dev_err(dchan2dev(&chan->vc.chan),
		"SAR: 0x%llx DAR: 0x%llx LLP: 0x%llx BTS 0x%x CTL: 0x%x:%08x",
		le64_to_cpu(desc->lli->sar),
		le64_to_cpu(desc->lli->dar),
		le64_to_cpu(desc->lli->llp),
		le32_to_cpu(desc->lli->block_ts_lo),
		le32_to_cpu(desc->lli->ctl_hi),
		le32_to_cpu(desc->lli->ctl_lo));
}

static void axi_chan_list_dump_lli(struct axi_dma_chan *chan,
				   struct axi_dma_desc *desc_head)
{
	int count = atomic_read(&chan->descs_allocated);
	int i;

	for (i = 0; i < count; i++)
		axi_chan_dump_lli(chan, &desc_head->hw_desc[i]);
}

static noinline void axi_chan_handle_err(struct axi_dma_chan *chan, u32 status)
{
	struct virt_dma_desc *vd;
	unsigned long flags;

	TESTCODE_DMAC_DRV_LOCK_VARS;

	spin_lock_irqsave(&chan->vc.lock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	axi_chan_disable(chan);

	/* The bad descriptor currently is in the head of vc list */
	vd = vchan_next_desc(&chan->vc);
	if (!vd) {
		dev_err(chan2dev(chan), "BUG: %s, IRQ with no descriptors\n",
			axi_chan_name(chan));
		goto out;
	}
	/* Remove the completed descriptor from issued list */
	list_del(&vd->node);

	/* WARN about bad descriptor */
	dev_err(chan2dev(chan),
		"Bad descriptor submitted for %s, cookie: %d, irq: 0x%08x\n",
		axi_chan_name(chan), vd->tx.cookie, status);
	axi_chan_list_dump_lli(chan, vd_to_axi_desc(vd));

	vchan_cookie_complete(vd);

	/* Try to restart the controller */
	axi_chan_start_first_queued(chan);

out:
	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("list_del/axi_chan_list_dump_lli");

	spin_unlock_irqrestore(&chan->vc.lock, flags);
}

static void axi_chan_block_xfer_complete(struct axi_dma_chan *chan)
{
	int count = atomic_read(&chan->descs_allocated);
	struct axi_dma_hw_desc *hw_desc;
	struct axi_dma_desc *desc;
	struct virt_dma_desc *vd;
	unsigned long flags;
	u64 llp;
	int i;

	TESTCODE_DMAC_DRV_LOCK_VARS;

	spin_lock_irqsave(&chan->vc.lock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	if (unlikely(axi_chan_is_hw_enable(chan))) {
		dev_err(chan2dev(chan), "BUG: %s caught DWAXIDMAC_IRQ_DMA_TRF, but channel not idle!\n",
			axi_chan_name(chan));
		axi_chan_disable(chan);
	}

	/* The completed descriptor currently is in the head of vc list */
	vd = vchan_next_desc(&chan->vc);
	if (!vd) {
		dev_err(chan2dev(chan), "BUG: %s, IRQ with no descriptors\n",
			axi_chan_name(chan));
		goto out;
	}

	if (chan->cyclic) {
		desc = vd_to_axi_desc(vd);
		if (desc) {
			llp = lo_hi_readq(chan->chan_regs + CH_LLP);
			for (i = 0; i < count; i++) {
				hw_desc = &desc->hw_desc[i];
				if (hw_desc->llp == llp) {
					axi_chan_irq_clear(chan, hw_desc->lli->status_lo);
					hw_desc->lli->ctl_hi |= CH_CTL_H_LLI_VALID;
					desc->completed_blocks = i;

					if (((hw_desc->len * (i + 1)) % desc->period_len) == 0)
						vchan_cyclic_callback(vd);
					break;
				}
			}

			axi_chan_enable(chan);
		}
	} else {
		/* Remove the completed descriptor from issued list before completing */
		list_del(&vd->node);
		vchan_cookie_complete(vd);
	}

out:
	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("vchan_next_desc");

	spin_unlock_irqrestore(&chan->vc.lock, flags);
}

static irqreturn_t dma_th_int_func(int irq, void *dev_id)
{
	LLD_CH ch;
	LLD_CH ch_lld;
	u32 cxd_dmaengine;
	uint64_t status;
	u32 chan_bit;
	unsigned long flags;

	struct axi_dma_chip *chip = dev_id;
	struct dw_axi_dma *dw = chip->dw;
	struct axi_dma_chan *chan;

#ifdef TESTCODE_DMA_DRV
#if !defined(TESTCODE_DMA_DRV_TRANS_PERF)
        printk("%s: Enter interrupt thread\n", __func__);
#endif /* !defined(TESTCODE_DMA_DRV_TRANS_PERF) */
#endif
	/* Get the channels controlled by DMA Engine */
	cxd_dmaengine = dw->hdata->cxd_dmaengine;

	for (ch = 0; ch < dw->hdata->nr_channels; ch++) {
		chan_bit = 0x1 << ch;

		chan = &dw->chan[ch];

		if (!(cxd_dmaengine & chan_bit)) {
			/* for LLD ch */
			ch_lld = (dw->hdata->cxd_id * DMAC_MAX_CHANNELS) + ch;
			lld_dmac_th_int_func(ch_lld);
		} else {
			/* for DMAEngine */
			spin_lock_irqsave(&chan->vc.lock, flags);
			status = axi_chan_irq_read(chan);
			axi_chan_irq_clear(chan, status);
			spin_unlock_irqrestore(&chan->vc.lock, flags);

			if (status & DWAXIDMAC_IRQ_ALL_ERR) {
				axi_chan_handle_err(chan, status);
			} else if (status & DWAXIDMAC_IRQ_DMA_TRF) {
				axi_chan_block_xfer_complete(chan);
			}
		}
	}

	/* Re-enable interrupts */
	raw_spin_lock_irqsave(&chip->spinlock, flags);
	axi_dma_irq_enable(chip);
	raw_spin_unlock_irqrestore(&chip->spinlock, flags);

	return IRQ_HANDLED;
}

static irqreturn_t dw_axi_dma_interrupt(int irq, void *dev_id)
{
	struct axi_dma_chip *chip = dev_id;
	struct dw_axi_dma *dw = chip->dw;
	struct axi_dma_chan *chan;

	u32 status, i;
	u32 combined_status, cxd_dmaengine;
	bool wake_thread;
	uint64_t status64;
	unsigned long flags;

	TESTCODE_DMAC_DRV_TRANS_VARS;
	TESTCODE_DMAC_DRV_TRANS_GET_TIME;
	TESTCODE_DMAC_DRV_TRANS_SET_NSEC_END;

	TESTCODE_DMAC_DRV_LOCK_VARS;

#ifdef TESTCODE_DMA_DRV
#if !defined(TESTCODE_DMA_DRV_TRANS_PERF)
        printk("%s: Enter interrupt handler\n", __func__);
#endif /* !defined(TESTCODE_DMA_DRV_TRANS_PERF) */
#endif
	/* Disable DMAC inerrupts. We'll enable them after processing chanels */
	raw_spin_lock_irqsave(&chip->spinlock, flags);

	TESTCODE_DMAC_DRV_LOCK_GET_START;

	axi_dma_irq_disable(chip);

	TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("axi_dma_irq_disable");

	raw_spin_unlock_irqrestore(&chip->spinlock, flags);

	/* Get Combined Interrupt Status Register */
	combined_status = axi_dma_ioread32(chip, DMAC_INTSTATUS);

	/* Get the channels controlled by DMA Engine */
	cxd_dmaengine = dw->hdata->cxd_dmaengine;

	/* initialize IRQ thread wakeup Enable */
	wake_thread = false;

	/* Poll, clear and process every chanel interrupt status */
	for (i = 0; i < dw->hdata->nr_channels; i++) {
		u32 chan_bit;

		chan = &dw->chan[i];
		status64 = axi_chan_irq_read64(chan);
		status = status64 & D_MASK_INT_STATUS;

		chan_bit = 0x1 << i;

		/* Channel is LLD and interrupt has occured on the channel */
		if (combined_status & chan_bit) {
			if (!(cxd_dmaengine & chan_bit)) {
				LLD_FACT fact;
				LLD_CH lld_ch;
				bool callback_is_necessary = false;

				/* Calculate LLD channel number */
				lld_ch = (dw->hdata->cxd_id * DMAC_MAX_CHANNELS) + i;

				/* Save interrupt status */
				lld_dmac_ch_mng_set_int_status(lld_ch, status64);

				/* Clear flag of the ch */
				axi_chan_irq_clear(chan, status);

				/* Some error has occurred */
				if (status & DWAXIDMAC_IRQ_ALL_ERR) {
					fact = LLD_DMAC_TRAN_ERR;
					callback_is_necessary = true;
				/* Transfer has finished */
				} else if (status & DWAXIDMAC_IRQ_DMA_TRF) {
					fact = LLD_DMAC_TRAN_END;
					callback_is_necessary = true;
				}
				if (callback_is_necessary) {
					LLD_FUNC_FLAG func_flag;
					raw_spinlock_t *lock;

					/* Get func flag */
					func_flag = lld_dmac_ch_mng_get_flag(lld_ch);

					/* Callback is executed with Threaded context */
					if (func_flag == LLD_FUNC_THREADED) {
						/* Set interrupt and fact */
						lock = lld_dmac_ch_mng_get_lock(lld_ch);
						raw_spin_lock_irqsave(lock, flags);

						TESTCODE_DMAC_DRV_LOCK_GET_START;

						lld_dmac_ch_mng_set_interrupt(lld_ch);
						lld_dmac_ch_mng_set_fact(lld_ch, fact);

						TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("lld_dmac_ch_mng_set_interrupt/lld_dmac_ch_mng_set_fact");
						raw_spin_unlock_irqrestore(lock, flags);

						/* Enable IRQ thread wakeup */
						wake_thread = true;
					} else {
						LLD_FUNC int_func;
						void *cookies;
						/* Get callback function */
						int_func = lld_dmac_ch_mng_get_int_func(lld_ch);

						/* Get cookies to be passed to callback */
						cookies = lld_dmac_ch_mng_get_cookies(lld_ch);

						/* execute callback function */
						if (int_func != NULL) {
							int_func(lld_ch, fact, cookies);
						}
					}
				}
			} else {
				/* Enable IRQ thread wakeup for Channel DMAEngine */
				wake_thread = true;
			}
		}

		dev_vdbg(chip->dev, "%s %u IRQ status: 0x%08x\n",
			axi_chan_name(chan), i, status);
	}

	if (wake_thread == false) {
		/* Re-enable interrupts */
		raw_spin_lock_irqsave(&chip->spinlock, flags);

		TESTCODE_DMAC_DRV_LOCK_GET_START;

		axi_dma_irq_enable(chip);

		TESTCODE_DMAC_DRV_LOCK_GET_END_AND_SET_MAX("axi_dma_irq_enable");

		raw_spin_unlock_irqrestore(&chip->spinlock, flags);
		return IRQ_HANDLED;
	} else {
		return IRQ_WAKE_THREAD;
	}
}

static int dma_chan_terminate_all(struct dma_chan *dchan)
{
	struct axi_dma_chan *chan = dchan_to_axi_dma_chan(dchan);
	u32 chan_active = BIT(chan->id) << DMAC_CHAN_EN_SHIFT;
	unsigned long flags;
	u32 val;
	int ret;
	LIST_HEAD(head);

	axi_chan_disable(chan);

	ret = readl_poll_timeout_atomic(chan->chip->regs + DMAC_CHEN, val,
					!(val & chan_active), 1000, 10000);
	if (ret == -ETIMEDOUT)
		dev_warn(dchan2dev(dchan),
			 "%s failed to stop\n", axi_chan_name(chan));

	if (chan->direction != DMA_MEM_TO_MEM)
		dw_axi_dma_set_hw_channel(chan, false);
	if (chan->direction == DMA_MEM_TO_DEV)
		dw_axi_dma_set_byte_halfword(chan, false);

	spin_lock_irqsave(&chan->vc.lock, flags);

	vchan_get_all_descriptors(&chan->vc, &head);

	chan->cyclic = false;
	spin_unlock_irqrestore(&chan->vc.lock, flags);

	vchan_dma_desc_free_list(&chan->vc, &head);

	dev_vdbg(dchan2dev(dchan), "terminated: %s\n", axi_chan_name(chan));

	return 0;
}

static int dma_chan_pause(struct dma_chan *dchan)
{
	struct axi_dma_chan *chan = dchan_to_axi_dma_chan(dchan);
	unsigned long flags;
	unsigned int timeout = 20; /* timeout iterations */
	u32 val;

	spin_lock_irqsave(&chan->vc.lock, flags);

	val = axi_dma_ioread32(chan->chip, DMAC_CHEN);
	val |= BIT(chan->id) << DMAC_CHAN_SUSP_SHIFT |
	       BIT(chan->id) << DMAC_CHAN_SUSP_WE_SHIFT;
	axi_dma_iowrite32(chan->chip, DMAC_CHEN, val);

	do  {
		if (axi_chan_irq_read(chan) & DWAXIDMAC_IRQ_SUSPENDED)
			break;

		udelay(2);
	} while (--timeout);

	axi_chan_irq_clear(chan, DWAXIDMAC_IRQ_SUSPENDED);

	chan->is_paused = true;

	spin_unlock_irqrestore(&chan->vc.lock, flags);

	return timeout ? 0 : -EAGAIN;
}

/* Called in chan locked context */
static inline void axi_chan_resume(struct axi_dma_chan *chan)
{
	u32 val;

	val = axi_dma_ioread32(chan->chip, DMAC_CHEN);
	val &= ~(BIT(chan->id) << DMAC_CHAN_SUSP_SHIFT);
	val |=  (BIT(chan->id) << DMAC_CHAN_SUSP_WE_SHIFT);
	axi_dma_iowrite32(chan->chip, DMAC_CHEN, val);

	chan->is_paused = false;
}

static int dma_chan_resume(struct dma_chan *dchan)
{
	struct axi_dma_chan *chan = dchan_to_axi_dma_chan(dchan);
	unsigned long flags;

	spin_lock_irqsave(&chan->vc.lock, flags);

	if (chan->is_paused)
		axi_chan_resume(chan);

	spin_unlock_irqrestore(&chan->vc.lock, flags);

	return 0;
}

static int axi_dma_suspend(struct axi_dma_chip *chip)
{
	uint8_t clkid;
	unsigned long flags;

	raw_spin_lock_irqsave(&chip->spinlock, flags);
	axi_dma_irq_disable(chip);
	axi_dma_disable(chip);
	raw_spin_unlock_irqrestore(&chip->spinlock, flags);

	/* Assert Resets */
	reset_control_assert(chip->core_rstn);
	reset_control_assert(chip->p_rstn);
	for (clkid = 0; clkid <= DWAXIDMAC_CXD_HS_RST_MAX; clkid++) {
		if (chip->hs_rstn[clkid] != NULL) {
			reset_control_assert(chip->hs_rstn[clkid]);
		}
	}

	/* Stop Clocks */
	clk_disable_unprepare(chip->core_clk);
	clk_disable_unprepare(chip->cfgr_clk);
	for (clkid = 0; clkid <= DWAXIDMAC_CXD_HS_CLK_MAX; clkid++) {
		if (chip->hs_clk[clkid] != NULL) {
			clk_disable_unprepare(chip->hs_clk[clkid]);
		}
	}

	return 0;
}

static int axi_dma_resume(struct axi_dma_chip *chip)
{
	int ret;
	uint8_t clkid;
	unsigned long flags;
	uint8_t ch;
	struct axi_dma_chan *chan;

	/* Supply Clocks */
	for (clkid = 0; clkid <= DWAXIDMAC_CXD_HS_CLK_MAX; clkid++) {
		if (chip->hs_clk[clkid] != NULL) {
			ret = clk_prepare_enable(chip->hs_clk[clkid]);
			if (ret < 0)
				return ret;
		}
	}

	ret = clk_prepare_enable(chip->cfgr_clk);
	if (ret < 0)
		return ret;

	ret = clk_prepare_enable(chip->core_clk);
	if (ret < 0)
		return ret;

	/* De-assert Resets */
	for (clkid = 0; clkid <= DWAXIDMAC_CXD_HS_RST_MAX; clkid++) {
		if (chip->hs_rstn[clkid] != NULL) {
			ret = reset_control_deassert(chip->hs_rstn[clkid]);
			if (ret < 0)
				return ret;
		}
	}

	ret = reset_control_deassert(chip->p_rstn);
	if (ret < 0)
		return ret;

	ret = reset_control_deassert(chip->core_rstn);
	if (ret < 0)
		return ret;

	if (chip->dw->in_system_suspend) {
		/* DMA initialization (INT CLEAR) */
		writeq_relaxed(DWAXIDMAC_LOAD_COMMON_INTCLEAR, chip->regs + DMAC_COMMON_INTCLEAR);
		for (ch = 0; ch < chip->dw->hdata->nr_channels; ch++) {
			chan = &chip->dw->chan[ch];
			writeq_relaxed(DWAXIDMAC_LOAD_CHX_INTCLEAR, chan->chan_regs + CH_INTCLEAR);
		}
	}

	raw_spin_lock_irqsave(&chip->spinlock, flags);
	axi_dma_enable(chip);
	axi_dma_irq_enable(chip);
	raw_spin_unlock_irqrestore(&chip->spinlock, flags);

	return 0;
}

static int __maybe_unused axi_dma_runtime_suspend(struct device *dev)
{
	uint8_t ch;
	struct axi_dma_chip *chip = dev_get_drvdata(dev);
	struct axi_dma_chan *chan;
	struct dma_chan *dchan;
	int ret;

	if (chip->dw->in_system_suspend) {
		/* Wait finishing transfers which is transfering */
		for (ch = 0; ch < chip->dw->hdata->nr_channels; ch++) {
			chan = &chip->dw->chan[ch];
			dchan = &chan->vc.chan;
			ret = dma_chan_terminate_all(dchan);
		}

		/* store registers */
		dma_store_registers(chip);
	}

	/* DMA disable for entering suspend */
	return axi_dma_suspend(chip);
}

static int __maybe_unused axi_dma_runtime_resume(struct device *dev)
{
	struct axi_dma_chip *chip = dev_get_drvdata(dev);
	int ret;

	/* DMA enable after resume */
	ret = axi_dma_resume(chip);
	if (ret < 0)
		return ret;

	if (chip->dw->in_system_suspend) {
		/* load registers */
		dma_load_registers(chip);
	}

	return 0;
}

static struct dma_chan *dw_axi_dma_of_xlate(struct of_phandle_args *dma_spec,
					    struct of_dma *ofdma)
{
	struct dw_axi_dma *dw = ofdma->of_dma_data;
	struct axi_dma_chan *chan;
	struct dma_chan *dchan;

	dchan = dma_get_any_slave_channel(&dw->dma);
	if (!dchan)
		return NULL;

	chan = dchan_to_axi_dma_chan(dchan);
	chan->hw_handshake_num = dma_spec->args[0];
	return dchan;
}

static int parse_device_properties(struct axi_dma_chip *chip)
{
	struct device *dev = chip->dev;
	u32 tmp, carr[DWAXIDMAC_CXD_PERIPHERAL_MAX + 1];
	u32 carr_outstand[((DWAXIDMAC_CXD_PERIPHERAL_MAX + 1) * 2)];
	int ret;

	ret = device_property_read_u32(dev, "dma-channels", &tmp);
	if (ret)
		return ret;
	if (tmp == 0 || tmp > DMAC_MAX_CHANNELS)
		return -EINVAL;

	chip->dw->hdata->nr_channels = tmp;

	ret = device_property_read_u32(dev, "cxd,dma-masters", &tmp);
	if (ret)
		return ret;
	if (tmp == 0 || tmp > DMAC_MAX_MASTERS)
		return -EINVAL;

	chip->dw->hdata->nr_masters = tmp;

	ret = device_property_read_u32(dev, "cxd,data-width", &tmp);
	if (ret)
		return ret;
	if (tmp > DWAXIDMAC_TRANS_WIDTH_MAX)
		return -EINVAL;

	chip->dw->hdata->m_data_width = tmp;

	ret = device_property_read_u32_array(dev, "cxd,block-size", carr,
					     chip->dw->hdata->nr_channels);
	if (ret)
		return ret;
	for (tmp = 0; tmp < chip->dw->hdata->nr_channels; tmp++) {
		if (carr[tmp] == 0 || carr[tmp] > DMAC_MAX_BLK_SIZE)
			return -EINVAL;

		chip->dw->hdata->block_size[tmp] = carr[tmp];
	}

	ret = device_property_read_u32_array(dev, "cxd,priority", carr,
					     chip->dw->hdata->nr_channels);
	if (ret)
		return ret;
	/* Priority value must be programmed within [0:nr_channels-1] range */
	for (tmp = 0; tmp < chip->dw->hdata->nr_channels; tmp++) {
		if (carr[tmp] >= chip->dw->hdata->nr_channels)
			return -EINVAL;

		chip->dw->hdata->priority[tmp] = carr[tmp];
	}

	/* axi-max-burst-len is optional property */
	ret = device_property_read_u32(dev, "cxd,axi-max-burst-len", &tmp);
	if (!ret) {
		if (tmp > DWAXIDMAC_ARWLEN_MAX + 1)
			return -EINVAL;
		if (tmp < DWAXIDMAC_ARWLEN_MIN + 1)
			return -EINVAL;

		chip->dw->hdata->restrict_axi_burst_len = true;
		chip->dw->hdata->axi_rw_burst_len = tmp;
	}

	/* enable controlled by DMA Engine API, valid:0x00 to 0xFF */
	ret = device_property_read_u32(dev, "dmaengine", &tmp);
	if (!ret)
		chip->dw->hdata->cxd_dmaengine = tmp;

	/* select LLD_CHxx index range, valid:0 to 3 */
	ret = device_property_read_u32(dev, "id", &tmp);
	if (!ret) {
		if (tmp > DWAXIDMAC_CXD_ID_MAX)
			return -EINVAL;
		chip->dw->hdata->cxd_id = tmp;
	}

	/* specify data width of just 13 dmac peripherals */
	ret = device_property_read_u32_array(dev, "peri-width", carr, (DWAXIDMAC_CXD_PERIPHERAL_MAX + 1));
	if (!ret) {
		for (tmp = 0; tmp <= DWAXIDMAC_CXD_PERIPHERAL_MAX; tmp++) {
			chip->dw->hdata->cxd_peri_width[tmp] = carr[tmp];
		}
	}

	/* specify data outstand of just multiple number of dmac peripherals */
	ret = device_property_read_u32_array(dev, "outstand", carr_outstand, ((DWAXIDMAC_CXD_PERIPHERAL_MAX + 1) * 2));
	if (!ret) {
		for (tmp = 0; tmp < ((DWAXIDMAC_CXD_PERIPHERAL_MAX + 1) * 2); tmp++) {
			if (carr_outstand[tmp] <= DWAXIDMAC_CXD_OUTSTAND_MAX) {
				chip->dw->hdata->cxd_outstand[tmp] = carr_outstand[tmp];
			} else {
				chip->dw->hdata->cxd_outstand[tmp] = 0;
			}
		}
	}

	return 0;
}

static int dw_probe(struct platform_device *pdev)
{
	struct device_node *node = pdev->dev.of_node;
	struct axi_dma_chip *chip;
	struct resource *mem;
	struct dw_axi_dma *dw;
	struct dw_axi_dma_hcfg *hdata;
	u32 i;
	uint8_t clkid;
	int ret;
	uint8_t *hsclkname[DWAXIDMAC_CXD_HS_CLK_MAX + 1] = {
			"hs-clk0", "hs-clk1", "hs-clk2", "hs-clk3", "hs-clk4",  "hs-clk5",
			"hs-clk6", "hs-clk7", "hs-clk8", "hs-clk9", "hs-clk10", "hs-clk11" };
	uint8_t *hsrstname[DWAXIDMAC_CXD_HS_RST_MAX + 1] = {
			"hs-rstn0", "hs-rstn1", "hs-rstn2", "hs-rstn3", "hs-rstn4",  "hs-rstn5",
			"hs-rstn6", "hs-rstn7", "hs-rstn8", "hs-rstn9", "hs-rstn10", "hs-rstn11" };
	LLD_CH lld_ch;

	chip = devm_kzalloc(&pdev->dev, sizeof(*chip), GFP_KERNEL);
	if (!chip)
		return -ENOMEM;

	dw = devm_kzalloc(&pdev->dev, sizeof(*dw), GFP_KERNEL);
	if (!dw)
		return -ENOMEM;

	hdata = devm_kzalloc(&pdev->dev, sizeof(*hdata), GFP_KERNEL);
	if (!hdata)
		return -ENOMEM;

	chip->dw = dw;
	chip->dev = &pdev->dev;
	chip->dw->hdata = hdata;

	raw_spin_lock_init(&chip->spinlock);

	chip->irq = platform_get_irq(pdev, 0);
	if (chip->irq < 0)
		return chip->irq;

	mem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	chip->regs = devm_ioremap_resource(chip->dev, mem);
	if (IS_ERR(chip->regs))
		return PTR_ERR(chip->regs);

	if (of_device_is_compatible(node, "intel,kmb-axi-dma")) {
		chip->apb_regs = devm_platform_ioremap_resource(pdev, 1);
		if (IS_ERR(chip->apb_regs))
			return PTR_ERR(chip->apb_regs);
	}

	/* get clock information */
	chip->core_clk = devm_clk_get(chip->dev, "core-clk");
	if (IS_ERR(chip->core_clk))
		return PTR_ERR(chip->core_clk);

	chip->cfgr_clk = devm_clk_get(chip->dev, "cfgr-clk");
	if (IS_ERR(chip->cfgr_clk))
		return PTR_ERR(chip->cfgr_clk);

	/* get reset information */
	chip->core_rstn = devm_reset_control_get(chip->dev, "core-rstn");
	if (IS_ERR(chip->core_rstn))
		return PTR_ERR(chip->core_rstn);

	chip->p_rstn = devm_reset_control_get(chip->dev, "p-rstn");
	if (IS_ERR(chip->p_rstn))
		return PTR_ERR(chip->p_rstn);

	for (clkid = 0; clkid <= DWAXIDMAC_CXD_HS_CLK_MAX; clkid++) {
		chip->hs_clk[clkid] = devm_clk_get(chip->dev, hsclkname[clkid]);
		if (IS_ERR(chip->hs_clk[clkid])) {
			chip->hs_clk[clkid] = NULL;
		}
		chip->hs_rstn[clkid] = devm_reset_control_get(chip->dev, hsrstname[clkid]);
		if (IS_ERR(chip->hs_rstn[clkid])) {
			chip->hs_rstn[clkid] = NULL;
		}
	}

	ret = parse_device_properties(chip);
	if (ret)
		return ret;

	for (i = 0; i < hdata->nr_channels; i++) {
		lld_ch = (LLD_CH)(hdata->cxd_id * DMAC_MAX_CHANNELS + i);
		lld_dmac_init(lld_ch);
		lld_dmac_set_priority(lld_ch, hdata->priority[i]);
	}

	dw->chan = devm_kcalloc(chip->dev, hdata->nr_channels,
				sizeof(*dw->chan), GFP_KERNEL);
	if (!dw->chan)
		return -ENOMEM;

	ret = devm_request_threaded_irq(chip->dev, chip->irq, dw_axi_dma_interrupt,
					dma_th_int_func, (IRQF_SHARED|IRQF_NO_THREAD),
					KBUILD_MODNAME, chip);
	if (ret)
		return ret;

	ret = dma_set_mask_and_coherent(&pdev->dev, DMA_BIT_MASK(64));
	if (ret) {
		dev_err(&pdev->dev, "DMA enable failed\n");
		return ret;
	}

	INIT_LIST_HEAD(&dw->dma.channels);
	for (i = 0; i < hdata->nr_channels; i++) {
		struct axi_dma_chan *chan = &dw->chan[i];

		chan->chip = chip;
		chan->id = i;
		chan->chan_regs = chip->regs + COMMON_REG_LEN + i * CHAN_REG_LEN;
		atomic_set(&chan->descs_allocated, 0);

		chan->vc.desc_free = vchan_desc_put;
		vchan_init(&chan->vc, &dw->dma);
	}

	/* Set capabilities */
	dma_cap_set(DMA_MEMCPY, dw->dma.cap_mask);
	dma_cap_set(DMA_SLAVE, dw->dma.cap_mask);
	dma_cap_set(DMA_CYCLIC, dw->dma.cap_mask);

	/* DMA capabilities */
	dw->dma.chancnt = hdata->nr_channels;
	dw->dma.max_burst = hdata->axi_rw_burst_len;
	dw->dma.src_addr_widths = AXI_DMA_BUSWIDTHS;
	dw->dma.dst_addr_widths = AXI_DMA_BUSWIDTHS;
	dw->dma.directions = BIT(DMA_MEM_TO_MEM);
	dw->dma.directions |= BIT(DMA_MEM_TO_DEV) | BIT(DMA_DEV_TO_MEM);
	dw->dma.residue_granularity = DMA_RESIDUE_GRANULARITY_BURST;

	dw->dma.dev = chip->dev;
	dw->dma.device_tx_status = dma_chan_tx_status;
	dw->dma.device_issue_pending = dma_chan_issue_pending;
	dw->dma.device_terminate_all = dma_chan_terminate_all;
	dw->dma.device_pause = dma_chan_pause;
	dw->dma.device_resume = dma_chan_resume;

	dw->dma.device_alloc_chan_resources = dma_chan_alloc_chan_resources;
	dw->dma.device_free_chan_resources = dma_chan_free_chan_resources;

	dw->dma.device_prep_dma_memcpy = dma_chan_prep_dma_memcpy;
	dw->dma.device_synchronize = dw_axi_dma_synchronize;
	dw->dma.device_config = dw_axi_dma_chan_slave_config;
	dw->dma.device_prep_slave_sg = dw_axi_dma_chan_prep_slave_sg;
	dw->dma.device_prep_dma_cyclic = dw_axi_dma_chan_prep_cyclic;

	/*
	 * Synopsis DesignWare AxiDMA datasheet mentioned Maximum
	 * supported blocks is 1024. Device register width is 4 bytes.
	 * Therefore, set constraint to 1024 * 4.
	 */
	dw->dma.dev->dma_parms = &dw->dma_parms;
	dma_set_max_seg_size(&pdev->dev, MAX_BLOCK_SIZE);
	platform_set_drvdata(pdev, chip);

	pm_runtime_enable(chip->dev);

	pm_runtime_get_sync(chip->dev);
	axi_dma_hw_init(chip);
	pm_runtime_put_sync(chip->dev);

	ret = dmaenginem_async_device_register(&dw->dma);
	if (ret)
		goto err_pm_disable;

	/* Register with OF helpers for DMA lookups */
	ret = of_dma_controller_register(pdev->dev.of_node,
					 dw_axi_dma_of_xlate, dw);
	if (ret < 0)
		dev_warn(&pdev->dev,
			 "Failed to register OF DMA controller, fallback to MEM_TO_MEM mode\n");

	dev_info(chip->dev, "DesignWare AXI DMA Controller, %d channels\n",
		 dw->hdata->nr_channels);

	gDma_chips[chip->dw->hdata->cxd_id] = chip;

	return 0;

err_pm_disable:
	pm_runtime_disable(chip->dev);

	gDma_chips[chip->dw->hdata->cxd_id] = NULL;

	return ret;
}

static int dw_remove(struct platform_device *pdev)
{
	struct axi_dma_chip *chip = platform_get_drvdata(pdev);
	struct dw_axi_dma *dw = chip->dw;
	struct axi_dma_chan *chan, *_chan;
	u32 i;
	uint8_t clkid;
	unsigned long flags;

	/* Enable clk before accessing to registers */
	for (clkid = 0; clkid <= DWAXIDMAC_CXD_HS_CLK_MAX; clkid++) {
		if (chip->hs_clk[clkid] != NULL) {
			clk_prepare_enable(chip->hs_clk[clkid]);
		}
	}
	clk_prepare_enable(chip->cfgr_clk);
	clk_prepare_enable(chip->core_clk);
	for (clkid = 0; clkid <= DWAXIDMAC_CXD_HS_RST_MAX; clkid++) {
		if (chip->hs_rstn[clkid] != NULL) {
			reset_control_deassert(chip->hs_rstn[clkid]);
		}
	}
	reset_control_deassert(chip->p_rstn);
	reset_control_deassert(chip->core_rstn);
	raw_spin_lock_irqsave(&chip->spinlock, flags);
	axi_dma_irq_disable(chip);
	raw_spin_unlock_irqrestore(&chip->spinlock, flags);
	for (i = 0; i < dw->hdata->nr_channels; i++) {
		axi_chan_disable(&chip->dw->chan[i]);
		axi_chan_irq_disable(&chip->dw->chan[i], DWAXIDMAC_IRQ_ALL);
	}
	raw_spin_lock_irqsave(&chip->spinlock, flags);
	axi_dma_disable(chip);
	raw_spin_unlock_irqrestore(&chip->spinlock, flags);

	pm_runtime_disable(chip->dev);
	axi_dma_suspend(chip);

	devm_free_irq(chip->dev, chip->irq, chip);

	of_dma_controller_free(chip->dev->of_node);

	list_for_each_entry_safe(chan, _chan, &dw->dma.channels,
			vc.chan.device_node) {
		list_del(&chan->vc.chan.device_node);
		tasklet_kill(&chan->vc.task);
	}

	gDma_chips[chip->dw->hdata->cxd_id] = NULL;

	return 0;
}

static void dma_store_registers(struct axi_dma_chip *chip)
{
	uint8_t regindex = 0;
	uint8_t ch;
	struct axi_dma_chan *chan;

	/* store common regs */
	chip->dw->cxd_store_common_regs[regindex++] = readq_relaxed(chip->regs + DMAC_CFG);
	chip->dw->cxd_store_common_regs[regindex++] = readq_relaxed(chip->regs + DMAC_COMMON_INTSTATUS_ENA);
	chip->dw->cxd_store_common_regs[regindex++] = readq_relaxed(chip->regs + DMAC_COMMON_INTSIGNAL_ENA);
	/* store chx regs */
	for (ch = 0; ch < chip->dw->hdata->nr_channels; ch++) {
		chan = &chip->dw->chan[ch];
		regindex = 0;
		chan->cxd_store_chx_regs[regindex++] = readq_relaxed(chan->chan_regs + CH_SAR);
		chan->cxd_store_chx_regs[regindex++] = readq_relaxed(chan->chan_regs + CH_DAR);
		chan->cxd_store_chx_regs[regindex++] = readq_relaxed(chan->chan_regs + CH_BLOCK_TS);
		chan->cxd_store_chx_regs[regindex++] = readq_relaxed(chan->chan_regs + CH_CTL);
		chan->cxd_store_chx_regs[regindex++] = readq_relaxed(chan->chan_regs + CH_CFG);
		chan->cxd_store_chx_regs[regindex++] = readq_relaxed(chan->chan_regs + CH_LLP);
		chan->cxd_store_chx_regs[regindex++] = readq_relaxed(chan->chan_regs + CH_SWHSSRC);
		chan->cxd_store_chx_regs[regindex++] = readq_relaxed(chan->chan_regs + CH_SWHSDST);
		chan->cxd_store_chx_regs[regindex++] = readq_relaxed(chan->chan_regs + CH_AXI_ID);
		chan->cxd_store_chx_regs[regindex++] = readq_relaxed(chan->chan_regs + CH_AXI_QOS);
		chan->cxd_store_chx_regs[regindex++] = readq_relaxed(chan->chan_regs + CH_INTSTATUS_ENA);
		chan->cxd_store_chx_regs[regindex]   = readq_relaxed(chan->chan_regs + CH_INTSIGNAL_ENA);
	}
}

static void dma_load_registers(struct axi_dma_chip *chip)
{
	uint8_t regindex = 0;
	uint8_t ch;
	struct axi_dma_chan *chan;

	/* load common regs */
	writeq_relaxed(chip->dw->cxd_store_common_regs[regindex++], chip->regs + DMAC_CFG);
	writeq_relaxed(chip->dw->cxd_store_common_regs[regindex++], chip->regs + DMAC_COMMON_INTSTATUS_ENA);
	writeq_relaxed(chip->dw->cxd_store_common_regs[regindex++], chip->regs + DMAC_COMMON_INTSIGNAL_ENA);
	/* load chx regs */
	for (ch = 0; ch < chip->dw->hdata->nr_channels; ch++) {
		chan = &chip->dw->chan[ch];
		regindex = 0;
		writeq_relaxed(chan->cxd_store_chx_regs[regindex++], chan->chan_regs + CH_SAR);
		writeq_relaxed(chan->cxd_store_chx_regs[regindex++], chan->chan_regs + CH_DAR);
		writeq_relaxed(chan->cxd_store_chx_regs[regindex++], chan->chan_regs + CH_BLOCK_TS);
		writeq_relaxed(chan->cxd_store_chx_regs[regindex++], chan->chan_regs + CH_CTL);
		writeq_relaxed(chan->cxd_store_chx_regs[regindex++], chan->chan_regs + CH_CFG);
		writeq_relaxed(chan->cxd_store_chx_regs[regindex++], chan->chan_regs + CH_LLP);
		writeq_relaxed(chan->cxd_store_chx_regs[regindex++], chan->chan_regs + CH_SWHSSRC);
		writeq_relaxed(chan->cxd_store_chx_regs[regindex++], chan->chan_regs + CH_SWHSDST);
		writeq_relaxed(chan->cxd_store_chx_regs[regindex++], chan->chan_regs + CH_AXI_ID);
		writeq_relaxed(chan->cxd_store_chx_regs[regindex++], chan->chan_regs + CH_AXI_QOS);
		writeq_relaxed(chan->cxd_store_chx_regs[regindex++], chan->chan_regs + CH_INTSTATUS_ENA);
		writeq_relaxed(chan->cxd_store_chx_regs[regindex],   chan->chan_regs + CH_INTSIGNAL_ENA);
	}
}

static int dw_suspend_late(struct device *dev)
{
	struct axi_dma_chip *chip = dev_get_drvdata(dev);

	chip->dw->in_system_suspend = true;

	pm_runtime_force_suspend(dev);

	/* print debug log */
	dev_dbg(chip->dev, "DMA Controllers Suspended.\n");

	return DWAXIDMAC_OK;
}

static int dw_resume_early(struct device *dev)
{
	struct axi_dma_chip *chip = dev_get_drvdata(dev);

	pm_runtime_force_resume(dev);

	chip->dw->in_system_suspend = false;

	/* print debug log */
	dev_dbg(chip->dev, "DMA Controllers Resumed.\n");

	return DWAXIDMAC_OK;
}

static const struct dev_pm_ops dw_axi_dma_pm_ops = {
	SET_LATE_SYSTEM_SLEEP_PM_OPS(dw_suspend_late, dw_resume_early)
	SET_RUNTIME_PM_OPS(axi_dma_runtime_suspend, axi_dma_runtime_resume, NULL)
};

static const struct of_device_id dw_dma_of_id_table[] = {
	{ .compatible = "snps,axi-dma-1.01a" },
	{ .compatible = "intel,kmb-axi-dma" },
	{}
};
MODULE_DEVICE_TABLE(of, dw_dma_of_id_table);

static struct platform_driver dw_driver = {
	.probe		= dw_probe,
	.remove		= dw_remove,
	.driver = {
		.name	= KBUILD_MODNAME,
		.of_match_table = dw_dma_of_id_table,
		.pm = &dw_axi_dma_pm_ops,
	},
};

static int __init dw_driver_init(void)
{
	LLD_CH lld_ch;

	/* Call LLD init at first because the initialization will be wanted to do
	 * whether DMA Engine's init is success or fail.
	 */
	for (lld_ch = LLD_CH0; lld_ch <= LLD_CH_MAX; lld_ch++) {
		lld_dmac_init(lld_ch);
	}

	return platform_driver_register(&dw_driver);
}
subsys_initcall(dw_driver_init);

MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("Synopsys DesignWare AXI DMA Controller platform driver");
MODULE_AUTHOR("Eugeniy Paltsev <Eugeniy.Paltsev@synopsys.com>");
