/*
 * drivers/cpuint/cpuint.c
 *
 * Copyright 2023 Sony Corporation
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/udif/module.h>
#include <linux/cpuint.h>
#include <mach/irqs.h>

#define CPUINT_SET		0x04
#define CPUINT_CLR		0x08
#define CPUINT_PORT(x)		((x) * 0x10U)
#define N_CPUINT_PORT		4

static inline void cpuint_wr(UDIF_VA addr, UDIF_U32 data)
{
	udif_iowrite32(data, addr);
}

typedef struct {
	UDIF_VA		va;
	bool		ready;
} CPUINT_INFO;
static CPUINT_INFO cpuint_dev[UDIF_NR_CPUINT];

int cpuint_irq(int port)
{
	if (port < 0 || N_CPUINT_PORT <= port) {
		return UDIF_ERR_PAR;
	}
	return IRQ_CPUINT(port);
}
EXPORT_SYMBOL(cpuint_irq);

int cpuint_assert(int port)
{
	if (port < 0 || N_CPUINT_PORT <= port) {
		return UDIF_ERR_PAR;
	}
	if (!READ_ONCE(cpuint_dev[0].ready)) {
		return UDIF_ERR_BUSY;
	}
	cpuint_wr(cpuint_dev[0].va+CPUINT_PORT(port)+CPUINT_SET, 1U);
	return UDIF_ERR_OK;
}
EXPORT_SYMBOL(cpuint_assert);

int cpuint_negate(int port)
{
	if (port < 0 || N_CPUINT_PORT <= port) {
		return UDIF_ERR_PAR;
	}
	if (!READ_ONCE(cpuint_dev[0].ready)) {
		return UDIF_ERR_BUSY;
	}
	cpuint_wr(cpuint_dev[0].va+CPUINT_PORT(port)+CPUINT_CLR, 1U);
	return UDIF_ERR_OK;
}
EXPORT_SYMBOL(cpuint_negate);

/*------------------------------------------------------------------*/
static UDIF_ERR cpuint_probe(const UDIF_DEVICE *dev, UDIF_CH ch, UDIF_VP data)
{
	if (!dev || ch >= UDIF_NR_CPUINT){
		return UDIF_ERR_PAR;
	}
	cpuint_dev[ch].va = udif_devio_virt(dev, ch);
	udif_devio_hclk(dev, ch, 1);
	udif_devio_hrst(dev, ch, 0);
	WRITE_ONCE(cpuint_dev[ch].ready, true);
	smp_mb();
	return UDIF_ERR_OK;
}

static UDIF_ERR cpuint_remove(const UDIF_DEVICE *dev, UDIF_CH ch, UDIF_VP data)
{
	if (!dev || ch >= UDIF_NR_CPUINT){
		return UDIF_ERR_PAR;
	}
	WRITE_ONCE(cpuint_dev[ch].ready, false);
	smp_mb();
	udif_devio_hrst(dev, ch, 1);
	udif_devio_hclk(dev, ch, 0);
	return UDIF_ERR_OK;
}

static UDIF_ERR cpuint_suspend(const UDIF_DEVICE *dev, UDIF_CH ch, UDIF_VP data)
{
	if (!dev || ch >= UDIF_NR_CPUINT){
		return UDIF_ERR_PAR;
	}
	WRITE_ONCE(cpuint_dev[ch].ready, false);
	smp_mb();
	return UDIF_ERR_OK;
}

static UDIF_ERR cpuint_resume(const UDIF_DEVICE *dev, UDIF_CH ch, UDIF_VP data)
{
	if (!dev || ch >= UDIF_NR_CPUINT){
		return UDIF_ERR_PAR;
	}
	udif_devio_hclk(dev, ch, 1);
	udif_devio_hrst(dev, ch, 0);
	WRITE_ONCE(cpuint_dev[ch].ready, true);
	smp_mb();
	return UDIF_ERR_OK;
}

static UDIF_DRIVER_OPS cpuint_ops = {
	.probe		= cpuint_probe,
	.remove		= cpuint_remove,
	.suspend	= cpuint_suspend,
	.resume		= cpuint_resume,
};

UDIF_IDS(cpuint_devs) = {
	UDIF_ID(UDIF_ID_CPUINT, UDIF_CH_MASK_DEFAULT),
};
UDIF_DEPS(cpuint_deps) = {};

UDIF_DECLARE_DRIVER(udif_cpuint, "cpuint", "1.0", &cpuint_ops, cpuint_devs, cpuint_deps, NULL);

static int __init __udif_init_cpuint(void)
{
	UDIF_DRIVER *drv = &udif_cpuint;

	return udif_driver_register(&drv, 1);
}
arch_initcall(__udif_init_cpuint);
