/*
 * driver/misc/cxd/dma/dw_axi_dmac.c
 *
 *   dw-axi-dmac LLD driver I/F
 *
 * Copyright 2022 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */

#include <linux/udif/types.h>
#include <linux/udif/malloc.h>
#include <linux/udif/cache.h>
#include <mach/noncache.h>

#include "internal.h"
#include <linux/dmac_lld_misc.h>

int hw_cxdma_check_args(req_t *req)
{
	if (req->size)
		return 0;

	/* Linked List mode */
	if (!IS_ALIGNED(req->src, DW_DMAC_LL_ALIGN)) {
		/* not aligned */
		return -1;
	}
	return 0;
}

static void req_to_lld(req_t *req, LLD_DMAC_CH_INFO *flow,
		       LLD_DMAC_TRAN_INFO *ptran)
{
	bool llmode = !req->size;
	bool auto_width = (DMA_PERI_AUTO_WIDTH == req->pconf.brst_size);

	if (DMA_TRANS_TYPE_PERI2MEM == req->pconf.direction) {
		flow->flowctl = LLD_DMAC_PM_PERI;
		flow->srcid = req->pconf.peripheral_id;
		flow->dstid = 0;
		if (!llmode) {
			ptran->sadr = req->src;
			ptran->dadr = req->dst;
			ptran->f1_si = req->pconf.src_inc;
			ptran->f1_di = req->pconf.dst_inc;
			ptran->f1_sauto  = auto_width;
			ptran->f4_swidth = req->pconf.brst_size;
			ptran->f4_sbsize = req->pconf.brst_len;
			ptran->f1_dauto  = 0;
			ptran->f4_dwidth = MACH_LLD_DMAC_WIDTH;
			ptran->f4_dbsize = MACH_LLD_DMAC_BSIZE;
		}
		return;
	}
	if (DMA_TRANS_TYPE_MEM2PERI == req->pconf.direction) {
		flow->flowctl = LLD_DMAC_MP_PERI;
		flow->srcid = 0;
		flow->dstid = req->pconf.peripheral_id;
		if (!llmode) {
			ptran->sadr = req->src;
			ptran->dadr = req->dst;
			ptran->f1_si = req->pconf.src_inc;
			ptran->f1_di = req->pconf.dst_inc;
			ptran->f1_sauto  = 0;
			ptran->f4_swidth = MACH_LLD_DMAC_WIDTH;
			ptran->f4_sbsize = MACH_LLD_DMAC_BSIZE;
			ptran->f1_dauto  = auto_width;
			ptran->f4_dwidth = req->pconf.brst_size;
			ptran->f4_dbsize = req->pconf.brst_len;
		}
		return;
	}
	flow->flowctl = LLD_DMAC_MM_DMAC;
	flow->srcid = 0;
	flow->dstid = 0;
}

static bool flow_cmp(LLD_DMAC_CH_INFO a, LLD_DMAC_CH_INFO b)
{
	return a.flowctl == b.flowctl && a.srcid == b.srcid
		&& a.dstid == b.dstid;
}

static void hw_cxdma_cb(LLD_CH ch, LLD_FACT fact, void *cookies)
{
	int err;

	err = (LLD_DMAC_TRAN_END == fact) ? 0 : -1;
	cxdma_cb((chan_t *)cookies, err);
}

int hw_cxdma_start(chan_t *chan)
{
	req_t *req = chan->cur;
	LLD_CH ch = (LLD_CH)chan->ch;
	LLD_ER ret;
	LLD_DMAC_CH_INFO flow;
	LLD_DMAC_IND_INFO lli;
	LLD_DMAC_LINKLIST *ll;
	uint64_t src_end, s, e;
	uint32_t sz, tsz;

	req_to_lld(req, &flow, &chan->hw.ptran);
	if (!flow_cmp(flow, chan->hw.flow)) { /* change */
		ret = lld_dmac_set_func((LLD_CH)ch, &flow, hw_cxdma_cb,
					chan, LLD_FUNC_THREADED);
		if (LLD_OK != ret) {
			chan_perr(ch, "%s:lld_dmac_set_func:ret=%d", __func__, ret);
		}
		chan->hw.flow = flow;
	}

	if (!req->size) { /* Linked List mode */
		lli.lli_adr = req->src >> DW_DMAC_LL_SHIFT;
		lli.lms = MACH_DW_DMAC_LLI_MASTER;
		lli.__1 = 0;
		ret = lld_dmac_sc_start(ch, lli);
		if (LLD_OK != ret) {
			chan_perr(ch, "lld_dmac_sc_start(user):ret=%d", ret);
			return -1;
		}
		return 0;
	}

	/*--- Non Linked List mode ---*/

	if (LLD_DMAC_MM_DMAC != flow.flowctl) { /* peri */
		ret = lld_dmac_start(ch, &chan->hw.ptran);
		if (LLD_OK != ret) {
			chan_perr(ch, "lld_dmac_start(P):ret=%d", ret);
			return -1;
		}
		return 0;
	}

	/*--- mem-to-mem ---*/

	/* end of src is aligned to transfer width ? */
	src_end = req->src + req->size;
	if (IS_ALIGNED(src_end, MACH_DW_DMAC_WIDTH)) {
		/* build a LLD_DMAC_TRAN_INFO */
		chan->hw.mtran.sadr = req->src;
		chan->hw.mtran.dadr = req->dst;
		chan->hw.mtran.f4_swidth = lld_dmac_swidth(req->src, src_end, req->size, &tsz);
		chan->hw.mtran.tsz = tsz - 1;
		ret = lld_dmac_start(ch, &chan->hw.mtran);
		if (LLD_OK != ret) {
			chan_perr(ch, "lld_dmac_start(MM):ret=%d", ret);
			return -1;
		}
		return 0;
	}

	/* unaligned */
	s = __lld_dmac_align(req->src);
	e = __lld_dmac_align(src_end - 1);
	if (s == e) { /* single word */
		chan->hw.mtran.sadr = req->src;
		chan->hw.mtran.dadr = req->dst;
		chan->hw.mtran.f4_swidth = lld_dmac_swidth(req->src, src_end, req->size, &tsz);
		chan->hw.mtran.tsz = tsz - 1;
		ret = lld_dmac_start(ch, &chan->hw.mtran);
		if (LLD_OK != ret) {
			chan_perr(ch, "lld_dmac_start(MM):ret=%d", ret);
			return -1;
		}
		return 0;
	}

	e = ALIGN_DOWN(src_end, MACH_DW_DMAC_WIDTH);
	sz = e - req->src;
	/*--- build a Linked List ---*/
	ll = chan->hw.ll;
	ll[0].sadr = req->src;
	ll[0].dadr = req->dst;
	lld_dmac_make_lli_swidth(&ll[0], 0UL, sz);
	/*---------------------------*/
	ll[1].sadr = e;
	ll[1].dadr = req->dst + sz;
	lld_dmac_make_lli_swidth(&ll[1], src_end, req->size - sz);
	/*---------------------------*/
	udif_cache_ctrl(ll, sizeof(LLD_DMAC_LINKLIST)*N_LLI, UDIF_CACHE_FLUSH);

	lli.lli_adr = __pa(ll) >> DW_DMAC_LL_SHIFT;
	lli.lms = MACH_DW_DMAC_LLI_MASTER;
	lli.__1 = 0;
	ret = lld_dmac_sc_start(ch, lli);
	if (LLD_OK != ret) {
		chan_perr(ch, "lld_dmac_sc_start(intra):ret=%d", ret);
		return -1;
	}
	return 0;
}

void hw_cxdma_stop(chan_t *chan)
{
	lld_dmac_stop(chan->ch);
}

void hw_cxdma_errstat(chan_t *chan, uint64_t *stat)
{
	LLD_ER ret;

	ret = lld_dmac_get_int_status(chan->ch, stat);
	if (LLD_OK != ret) {
		chan_perr(chan->ch, "lld_dmac_get_int_status:ret=%d", ret);
	}
}

#define MAX_LLIDUMP 10

typedef union {
	LLD_DMAC_CTR	ctr;
	uint64_t	data;
} u_lli_ctl;

typedef union {
	LLD_DMAC_IND_INFO	ind;
	uint64_t		data;
} u_lli_llp;

void hw_cxdma_dump(chan_t *chan)
{
	req_t *req = chan->cur;
	uint64_t next;
	LLD_DMAC_LINKLIST *ll;
	u_lli_ctl lli_ctl;
	u_lli_llp lli_llp;
	int i;

	if (!req) {
		chan_perr(chan->ch, "DUMP:cur=NULL");
		return;
	}
	chan_perr(chan->ch, "DUMP:%u:(%lx,%lx,%x),peri=%lx",
		  req->ch, req->src, req->dst, req->size, req->pconf.data);
	chan_perr(chan->ch, "DUMP:cb=%px,ck=%px", req->callback, req->cookies);
	if (req->size)
		return;
	next = req->src;
	for (i = 0; i < MAX_LLIDUMP; i++) {
		if (!IS_ALIGNED(next, DW_DMAC_LL_ALIGN)) {
			chan_perr(chan->ch, "LLI%d:LLP unaligned", i);
			break;
		}
		ll = (LLD_DMAC_LINKLIST *)PHYS_TO_VA(next);
		if (PA2NC_ERR == (uintptr_t)ll) {
			chan_perr(chan->ch, "LLI%d:illegal LLP", i);
			break;
		}
		lli_ctl.ctr = ll->ul_ctl;
		lli_llp.ind = ll->s_lli;
		chan_perr(chan->ch, "LLI%d:ctr=%lx,llp=%lx", i,
			  lli_ctl.data, lli_llp.data);
		chan_perr(chan->ch, "LLI%d:%lx,%lx,%x", i,
			  ll->sadr, ll->dadr, ll->tsz);
		if (!ll->ul_ctl.f1_lli_valid) {
			chan_perr(chan->ch, "LLI%d:not valid", i);
			break;
		}
		if (ll->ul_ctl.f1_lli_last) {
			chan_perr(chan->ch, "LLI%d:last", i);
			break;
		}
		next = ll->s_lli.lli_adr << DW_DMAC_LL_SHIFT;
	}
}

void hw_cxdma_config(chan_t *chan, struct hw_cxdma_conf *cfg)
{
	LLD_ER ret;
	LLD_DMAC_CH_CONF conf;

	ret = lld_dmac_open(chan->ch);
	if (LLD_OK != ret) {
		return;
	}
	conf.prio = cfg->priority;
	conf.src_outstand = cfg->src_outstanding;
	conf.dst_outstand = cfg->dst_outstanding;
	conf.aw_qos = conf.ar_qos = cfg->qos;
	ret = lld_dmac_set_config(chan->ch, &conf);
	if (LLD_OK != ret) {
		chan_perr(chan->ch, "lld_dmac_set_config:ret=%d", ret);
	}
	lld_dmac_close(chan->ch);
}

int hw_cxdma_open(chan_t *chan)
{
	UDIF_INT i;
	UDIF_SIZE sz;
	UDIF_VP *p;
	LLD_DMAC_LINKLIST *ll;
	LLD_ER ret;

	/* setup Linked List for internal use */
	sz = sizeof(LLD_DMAC_LINKLIST) * N_LLI;
	if (DW_DMAC_LL_ALIGN > L1_CACHE_BYTES) {
		sz += DW_DMAC_LL_ALIGN - L1_CACHE_BYTES;
	}
	p = udif_kmalloc(DRV_NAME, sz, GFP_KERNEL);
	if (!p) {
		chan_perr(chan->ch, "kmalloc failed");
		goto err1;
	}
	memset(p, 0, sz);
	chan->hw.mem = p;
	chan->hw.ll = (LLD_DMAC_LINKLIST *)PTR_ALIGN(p, DW_DMAC_LL_ALIGN);
	for (i = 0, ll = chan->hw.ll; i < N_LLI; i++, ll++) {
		if (lld_dmac_make_lli_fill(ll, N_LLI - i) != 0) {
			chan_perr(chan->ch, "lld_dmac_make_lli_fill failed\n");
			goto err2;
		}
		if (N_LLI - 1 == i) { /* last entry */
			if (lld_dmac_make_lli_last(ll) != 0) {
				chan_perr(chan->ch, "lld_dmac_make_lli_last failed\n");
				goto err2;
			}
			ll->ul_ctl.f3_swidth	= LLD_DMAC_WIDTH8;
		}
	}

	/* setup LLD_DMAC_TRAN_INFO for mem-to-mem */
	chan->hw.mtran.f1_si	 = LLD_DMAC_ADR_INC;
	chan->hw.mtran.f1_di	 = LLD_DMAC_ADR_INC;
	chan->hw.mtran.f4_swidth = MACH_LLD_DMAC_WIDTH;
	chan->hw.mtran.f4_dwidth = MACH_LLD_DMAC_WIDTH;
	chan->hw.mtran.f4_sbsize = MACH_LLD_DMAC_BSIZE;
	chan->hw.mtran.f4_dbsize = MACH_LLD_DMAC_BSIZE;
	chan->hw.mtran.f1_sauto	 = 0;
	chan->hw.mtran.f1_dauto  = 0;

	ret = lld_dmac_open(chan->ch);
	if (LLD_OK != ret) {
		chan_perr(chan->ch, "lld_dmac_open:ret=%d", ret);
		goto err2;
	}

	/* setup LLD_DMAC_CH_INFO */
	chan->hw.flow.flowctl = LLD_DMAC_MM_DMAC;
	chan->hw.flow.srcid = 0;
	chan->hw.flow.dstid = 0;

	ret = lld_dmac_set_func(chan->ch, &chan->hw.flow, hw_cxdma_cb,
				chan, LLD_FUNC_THREADED);
	if (LLD_OK != ret) {
		chan_perr(chan->ch, "%s:lld_dmac_set_func:ret=%d", __func__, ret);
		goto err3;
	}
	return 0;

err3:
	lld_dmac_close(chan->ch);
err2:
	udif_kfree(chan->hw.mem);
err1:
	return -1;
}

void hw_cxdma_close(chan_t *chan)
{
	LLD_ER ret;

	ret = lld_dmac_close(chan->ch);
	if (LLD_OK != ret) {
		chan_perr(chan->ch, "lld_dmac_close:ret=%d", ret);
	}
	if (chan->hw.mem) {
		udif_kfree(chan->hw.mem);
		chan->hw.mem = NULL;
		chan->hw.ll = NULL;
	}
}
