/*
 * driver/misc/cxd/dma/proc.c
 *
 * Copyright 2022 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */

#include <linux/udif/module.h>
#include <linux/udif/proc.h>

#include "internal.h"

#define PROC_LINE_SIZE	80

static const char *cxdma_show_state(uint32_t flag, void *cur)
{
	if (!(flag & CHAN_FLAG_EN))
		return "----";
	return (cur) ? "busy" : "idle";
}

static UDIF_SIZE cxdma_show(UDIF_PROC_READ *proc, UDIF_SIZE off, UDIF_UINT ch)
{
	chan_t *chan;
	unsigned int ch_flag;
	req_t *req;
	uint32_t pid;
	struct dma_stat stat;
	const char *state;
	unsigned long flags;

	chan = cxdma_get_chan(ch);
	if (!chan)
		return 0;

	LOCK(chan);
	ch_flag = chan->flag;
	req = chan->cur;
	if (req) {
		pid = req->drv.pid;
	}
	stat = chan->stat;
	UNLOCK(chan);
	state = cxdma_show_state(ch_flag, req);
	off += udif_proc_setbuf(proc, off, "%02d:%s %d %ld,%ld,%ld",
				ch, state, stat.pending_cnt,
				stat.done_cnt, stat.abort_cnt, stat.err_cnt);
	if (req) {
		off += udif_proc_setbuf(proc, off, ":%u,%px", pid, req);
	}
	off += udif_proc_setbuf(proc, off, "\n");
	return off;
}

static UDIF_INT cxdma_read_proc(UDIF_PROC_READ *proc)
{
	UDIF_SIZE off = 0, ret;
	static UDIF_UINT ch = 0;

	if (!proc) {
		return UDIF_ERR_PAR;
	}
	if (udif_proc_pos(proc) == 0) /* reset ch */
		ch = 0;
	if (udif_proc_bufsize(proc) < PROC_LINE_SIZE)
		return 0;
	while (off <= udif_proc_bufsize(proc) - PROC_LINE_SIZE
	       && ch < N_DMA_CH) {
		ret = cxdma_show(proc, off, ch);
		if (!ret)
			break;
		off = ret;
		ch++;
	}
	udif_proc_pos_inc(proc, off);
	if (ch >= N_DMA_CH)
		udif_proc_setend(proc);
	return off;
}

static UDIF_INT cxdma_write_proc(UDIF_PROC_WRITE *proc)
{
	return UDIF_ERR_PAR;
}

static UDIF_PROC cxdma_proc = {
	.name	= PROC_NAME,
	.read	= cxdma_read_proc,
	.write	= cxdma_write_proc,
};

void UDIF_INIT cxdma_proc_init(void)
{
	udif_create_proc(&cxdma_proc);
}

void UDIF_EXIT cxdma_proc_exit(void)
{
	udif_remove_proc(&cxdma_proc);
}
