/*
 * drivers/misc/cxd/pcie/access.c
 *
 *
 * Copyright 2024 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */

#include <linux/module.h>
#include <linux/snsc_boot_time.h>
#include <linux/udif/timer.h>
#include "internal.h"

static int pcie_backtrace;
module_param_named(backtrace, pcie_backtrace, int, S_IRUSR|S_IWUSR);

#define PCIE_STALL_THRESHOLD 1000 /* usec */

int pcie_write32(struct pci_dev *pdev, volatile void __iomem *addr, u32 val)
{
	int ret = -1;
	unsigned long flags;

	local_irq_save(flags);
	if (pcie_check_prsnt(pdev)) {
		writel(val, addr);
		ret = 0;
	}
	local_irq_restore(flags);
	return ret;
}

#ifdef CONFIG_PCIE_CXD_WORKAROUND
static void __pcie_cxd_workaround(struct pci_dev *dev)
{
	pci_bus_write_config_dword(dev->bus, dev->devfn, 0, 0);
}
#endif

int pcie_read32(struct pci_dev *pdev, volatile void __iomem *addr, u32 *val)
{
	int ret = -1;
	unsigned long flags;
	unsigned long long t1 = 0, t2 = 0, dt;

	local_irq_save(flags);
	if (pcie_check_prsnt(pdev)) {
		pcie_err_clr(pdev->bus);

		t1 = udif_read_freerun();
		*val = readl(addr);
		t2 = udif_read_freerun();

		if (pcie_err_stat(pdev->bus) < 0) {
			*val = ~0U;
		} else {
			ret = 0;
		}
#ifdef CONFIG_PCIE_CXD_WORKAROUND
		__pcie_cxd_workaround(pdev);
#endif
	}
	local_irq_restore(flags);

	dt = udif_cycles_to_usecs(t2 - t1);
	if (dt >= PCIE_STALL_THRESHOLD) {
		char buf[64];
		scnprintf(buf, sizeof buf, "PCI%d:STALL:%s:%llu",
			  pci_domain_nr(pdev->bus), __func__, dt);
		BOOT_TIME_ADD1(buf);
		printk(KERN_ERR "%s\n", buf);
		if (pcie_backtrace) {
			dump_stack();
		}
	}
	return ret;
}

int pcie_read64(struct pci_dev *pdev, volatile void __iomem *addr, u64 *val)
{
	int ret = -1;
	unsigned long flags;
	unsigned long long t1 = 0, t2 = 0, dt;

	local_irq_save(flags);
	if (pcie_check_prsnt(pdev)) {
		pcie_err_clr(pdev->bus);

		t1 = udif_read_freerun();
		*val = readq(addr);
		t2 = udif_read_freerun();

		if (pcie_err_stat(pdev->bus) < 0) {
			*val = ~0UL;
		} else {
			ret = 0;
		}
#ifdef CONFIG_PCIE_CXD_WORKAROUND
		__pcie_cxd_workaround(pdev);
#endif
	}
	local_irq_restore(flags);

	dt = udif_cycles_to_usecs(t2 - t1);
	if (dt >= PCIE_STALL_THRESHOLD) {
		char buf[64];
		scnprintf(buf, sizeof buf, "PCI%d:STALL:%s:%llu",
			  pci_domain_nr(pdev->bus), __func__, dt);
		BOOT_TIME_ADD1(buf);
		printk(KERN_ERR "%s\n", buf);
		if (pcie_backtrace) {
			dump_stack();
		}
	}
	return ret;
}
