/*
 * drivers/misc/cxd/pcie/aspm.c
 *   ASPM lock API
 *
 * Copyright 2022 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/moduleparam.h>
#include <linux/mutex.h>

#include "internal.h"

/* ASPM policy */
static int aspm[N_PCIE] = { 1, 1, 1, 1, 1 }; // POLICY_PERFORMANCE

static int aspm_set(const char *val, const struct kernel_param *kp)
{
	int domain, policy, ret;

	domain = (int *)kp->arg - aspm;
	if (domain < 0 || N_PCIE <= domain) {
		printk(KERN_ERR "ERROR:%s:internal error\n", __func__);
		return -EFAULT;
	}

	ret = kstrtoint(val, 0, &policy);
	if (ret)
		return ret;

	ret = pcie_aspm_set_domain_policy(domain, policy);
	if (ret) {
		printk(KERN_ERR "ERROR:%s:Invalid policy:%d\n",
		       __func__, policy);
		return -EINVAL;
	}
	return 0;
}

static const struct kernel_param_ops aspm_ops = {
	.set	= aspm_set,
	.get	= param_get_int,
};
module_param_cb(aspm0, &aspm_ops, &aspm[0], S_IRUSR|S_IWUSR);
module_param_cb(aspm1, &aspm_ops, &aspm[1], S_IRUSR|S_IWUSR);
module_param_cb(aspm2, &aspm_ops, &aspm[2], S_IRUSR|S_IWUSR);
module_param_cb(aspm3, &aspm_ops, &aspm[3], S_IRUSR|S_IWUSR);
module_param_cb(aspm4, &aspm_ops, &aspm[4], S_IRUSR|S_IWUSR);

int pcie_aspm_policy(int domain)
{
	if (domain < 0 || N_PCIE <= domain)
		return 1; // POLICY_PERFORMANCE
	return aspm[domain];
}
EXPORT_SYMBOL(pcie_aspm_policy);

/* caller must hold aspm_lock */
int pcie_aspm_change_policy(int domain, int policy)
{
	if (domain < 0 || N_PCIE <= domain)
		return -EINVAL;

	if (policy == aspm[domain])
		return 0;

	aspm[domain] = policy;
	return 1;
}

static struct mutex aspm_ctrl_mutex[N_PCIE] = {
	__MUTEX_INITIALIZER(aspm_ctrl_mutex[0]),
	__MUTEX_INITIALIZER(aspm_ctrl_mutex[1]),
	__MUTEX_INITIALIZER(aspm_ctrl_mutex[2]),
	__MUTEX_INITIALIZER(aspm_ctrl_mutex[3]),
	__MUTEX_INITIALIZER(aspm_ctrl_mutex[4])
};

int pcie_aspm_ctrl_trylock(int domain)
{
	if (domain < 0 || N_PCIE <= domain)
		return -EINVAL;
	return mutex_trylock(&aspm_ctrl_mutex[domain]);
}

int pcie_aspm_ctrl_lock(int domain)
{
	if (domain < 0 || N_PCIE <= domain)
		return -EINVAL;
	mutex_lock(&aspm_ctrl_mutex[domain]);
	return 0;
}

int pcie_aspm_ctrl_unlock(int domain)
{
	if (domain < 0 || N_PCIE <= domain)
		return -EINVAL;
	mutex_unlock(&aspm_ctrl_mutex[domain]);
	return 0;
}
EXPORT_SYMBOL_GPL(pcie_aspm_ctrl_trylock);
EXPORT_SYMBOL_GPL(pcie_aspm_ctrl_lock);
EXPORT_SYMBOL_GPL(pcie_aspm_ctrl_unlock);
