/*
 * drivers/misc/cxd/pcie/aximon.c
 *
 *
 * Copyright 2023 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/delay.h>
#include <linux/snsc_boot_time.h>
#include <mach/moduleparam.h>
#include <mach/platform.h>
#include <mach/irqs.h>
#include "internal.h"

/* AXI timeout[usec] */
static unsigned int axitmo[N_PCIE] = { 0, 0, 0, 0, 0 };
module_param_named(axitmo0, axitmo[0], uint, S_IRUSR|S_IWUSR);
module_param_named(axitmo1, axitmo[1], uint, S_IRUSR|S_IWUSR);
module_param_named(axitmo2, axitmo[2], uint, S_IRUSR|S_IWUSR);
module_param_named(axitmo3, axitmo[3], uint, S_IRUSR|S_IWUSR);
module_param_named(axitmo4, axitmo[4], uint, S_IRUSR|S_IWUSR);

#ifdef CONFIG_PCIE_CXD_FPGA
# define AXIMON_HCLK	6 /* MHz */
#else
# define AXIMON_HCLK	198 /* MHz */
#endif

#define AXIMON_OFFS		0x4000
#define AXIMON_COM_CTRL		0x00
# define AXIMON_COM_EXEC	 0x00000001U
# define AXIMON_COM_TRIG	 0x00010000U
#define AXIMON_COM_STAT		0x04
# define AXIMON_STAT_MASK	 0xFU
# define AXIMON_STAT_IDLE	 0x0U
# define AXIMON_STAT_MONITOR	 0x1U
#define AXIMON_IRQ_STS		0x10
#define AXIMON_IRQ_CLR		0x14
#define AXIMON_IRQ_ENB		0x18
# define AXIMON_IRQ_ALL		 0x77003331U
# define AXIMON_IRQ_DSSEL	 0x00000001U
# define AXIMON_IRQ_QOVER	 0x07000000U
# define AXIMON_IRQ_QFULL	 0x70000000U
#define AXIMON_TOUT_BASE_WR	0x40
#define AXIMON_TOUT_TIME_WR	0x44
#define AXIMON_TOUT_BASE_RD	0x48
#define AXIMON_TOUT_TIME_RD	0x4c
#define AXIMON_BUS_ENC0		0x50
#define AXIMON_BUS_ENC4		0x60

#if 0
struct aximon_info {
	volatile void __iomem * const base;
	const int irq;
};
#define AXIMON_INIT(x)	{ \
	.base = IO_ADDRESSP(CXD90XXX_PCIEREG##x), \
	.irq = IRQ_PCIE##x, \
	}
static struct aximon_info aximon[N_PCIE] = {
	AXIMON_INIT(0),
	AXIMON_INIT(1),
	AXIMON_INIT(2),
	AXIMON_INIT(3),
	AXIMON_INIT(4),
};
#endif

static inline void aximon_write32(volatile void __iomem *base, uint32_t reg, uint32_t data)
{
	writel_relaxed(data, base + AXIMON_OFFS + reg);
}

static inline uint32_t aximon_read32(volatile void __iomem *base, uint32_t reg)
{
	return readl_relaxed(base + AXIMON_OFFS + reg);
}

void pcie_aximon_start(int ch, void __iomem *base)
{
	char buf[64];

	if (ch < 0 || N_PCIE <= ch)
		return;
	if (!axitmo[ch])
		return;
	scnprintf(buf, sizeof buf, "PCI%d:AXIMON:start", ch);
	BOOT_TIME_ADD1(buf);
	aximon_write32(base, AXIMON_TOUT_BASE_WR, AXIMON_HCLK);
	aximon_write32(base, AXIMON_TOUT_TIME_WR, axitmo[ch]);
	aximon_write32(base, AXIMON_TOUT_BASE_RD, AXIMON_HCLK);
	aximon_write32(base, AXIMON_TOUT_TIME_RD, axitmo[ch]);
	aximon_write32(base, AXIMON_BUS_ENC0, 0); /* BRESP=RRESP=OKAY */
	aximon_write32(base, AXIMON_BUS_ENC4, 0); /* BRESP=RRESP=OKAY */
	/* start */
	aximon_write32(base, AXIMON_IRQ_CLR, AXIMON_IRQ_ALL);
	aximon_write32(base, AXIMON_IRQ_ENB, AXIMON_IRQ_DSSEL|AXIMON_IRQ_QOVER);
	aximon_write32(base, AXIMON_COM_CTRL, AXIMON_COM_EXEC|AXIMON_COM_TRIG);
}

void pcie_aximon_stop(int ch, void __iomem *base)
{
	char buf[64];

	if (ch < 0 || N_PCIE <= ch)
		return;
	if (!axitmo[ch])
		return;
	scnprintf(buf, sizeof buf, "PCI%d:AXIMON:stop", ch);
	BOOT_TIME_ADD1(buf);
	aximon_write32(base, AXIMON_IRQ_ENB, 0);
}

void pcie_aximon_intr(int ch, void __iomem *base)
{
	uint32_t stat;
	char buf[64];

	stat = aximon_read32(base, AXIMON_IRQ_STS);
	if (!stat) {
		return;
	}
	aximon_write32(base, AXIMON_IRQ_CLR, stat);
	mb();

	scnprintf(buf, sizeof buf, "PCI%d:AXIMON:INT=%x", ch, stat);
	BOOT_TIME_ADD1(buf);
}

int pcie_aximon_stat(int ch, void __iomem *base)
{
	uint32_t stat;

	if (ch < 0 || N_PCIE <= ch)
		return 0;
	if (!axitmo[ch])
		return 0;
	stat = aximon_read32(base, AXIMON_COM_STAT);
	stat &= AXIMON_STAT_MASK;
	if (stat > AXIMON_STAT_MONITOR) {
		return -1;
	}
	return 0;
}
