/*
 * drivers/misc/cxd/pcie/link.c
 *   Link Width and Speed API
 *
 *
 * Copyright 2022 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/module.h>
#include <linux/mutex.h>
#include <linux/semaphore.h>
#include <linux/fs.h>
#include <asm/uaccess.h>
#include <linux/proc_fs.h>
#include <linux/snsc_boot_time.h>
#include <linux/udif/module.h>
#include <linux/udif/delay.h>
#include <linux/udif/timer.h>

#include "internal.h"

static DEFINE_MUTEX(pcie_link_mutex);

int cxpcie_get_link(int ch, const struct pcie_link *link)
{
	struct pci_dev *dev = NULL;
	int current_link;

	if ((ch < 0) || (N_PCIE <= ch)) {
		printk(KERN_ERR "ERROR:%s: unexpected ch: ch=%d\n", __func__, ch);
		return -EINVAL;
	}

	mutex_lock(&pcie_link_mutex);

	/* find a RC and increment the reference count */
	dev = pci_get_domain_bus_and_slot(ch, 0, 0);
	if (!dev) {
		printk(KERN_ERR "ERROR:%s: could not find RC: ch=%d\n", __func__, ch);
		current_link = -ENODEV;
		goto end;
	}
	/* read current Link status */
	current_link = link->read(dev);
	if (current_link < 0) {
		printk(KERN_ERR "ERROR:%s: failed to read current Link %s : ch=%d, err=%d\n", __func__, link->name, ch, current_link);
	}

end:
	/* release a use of the pci device */
	if (dev)
		pci_dev_put(dev);

	mutex_unlock(&pcie_link_mutex);
	return current_link;
}

int cxpcie_set_link(int ch, unsigned int target_link, const struct pcie_link *link)
{
	struct pci_dev *dev = NULL;
	int current_link;
	int delay_us;
	unsigned int sleep_us;
	unsigned long long timeout;
	int err = 0;

	if ((ch < 0) || (N_PCIE <= ch)) {
		printk(KERN_ERR "ERROR:%s: unexpected ch: ch=%d\n", __func__, ch);
		return -EINVAL;
	}

	mutex_lock(&pcie_link_mutex);

	/* find a RC and increment the reference count */
	dev = pci_get_domain_bus_and_slot(ch, 0, 0);
	if (!dev) {
		printk(KERN_ERR "ERROR:%s: could not find RC: ch=%d\n", __func__, ch);
		err = -ENODEV;
		goto end;
	}
	/* read current Link status */
	current_link = link->read(dev);
	if (current_link < 0) {
		printk(KERN_ERR "ERROR:%s: failed to read current Link %s: ch=%d, err=%d\n", __func__, link->name, ch, current_link);
		err = current_link;
		goto end;
	}
	/* confirm Link status */
	if ((unsigned int)current_link == target_link) {
		err = 0;
		goto end; /* success */
	}
	/* change Link status */
	err = link->change(dev, ch, (unsigned int)current_link, target_link);
	if (err) {
		goto end;
	}
	delay_us = *(link->delay_us);
	if (delay_us > 0)
		udif_udelay(delay_us);

	/* wait */
	sleep_us = (unsigned int)(*(link->sleep_us));
	if (sleep_us < 1)
		sleep_us = 1;
	timeout = udif_read_freerun() + udif_usecs_to_cycles((unsigned long long)(*(link->timeout_us)));

	do {
		current_link = link->read(dev);
		if ((current_link < 0) || ((unsigned int)current_link == target_link)) {
			break;
		}
		if (udif_read_freerun() > timeout) {
			current_link = link->read(dev);
			break;
		}
		usleep_range(sleep_us, sleep_us * 2);
	} while (1);
	/* error */
	if (current_link < 0) {
		printk(KERN_ERR "ERROR:%s: failed to read current Link %s: ch=%d, err=%d\n", __func__, link->name, ch, current_link);
		err = current_link;
		goto end;
	}
	/* timeout */
	if ((unsigned int)current_link != target_link) {
		printk(KERN_ERR "ERROR:%s: current Link %s does not match the target: ch=%d, current=%d, target=%d\n", __func__, link->name, ch, current_link, target_link);
		err = -EIO;
		goto end;
	}
	/* success */
	err = 0;

end:
	/* release a use of the pci device */
	if (dev)
		pci_dev_put(dev);

	mutex_unlock(&pcie_link_mutex);
	return err;
}

int cxpcie_check_dev_state(int ch)
{
	int err = 0;
	struct pci_dev *dev_rc = pci_get_domain_bus_and_slot(ch, 0, 0);
	struct pci_dev *dev_ep = pci_get_domain_bus_and_slot(ch, 1, 0);

	if (!dev_rc || !dev_ep)
		err = -ENODEV;

	if (dev_rc)
		pci_dev_put(dev_rc);

	if (dev_ep)
		pci_dev_put(dev_ep);

	return err; /* 0: ready to get/set LinkWidth, !0: not ready */
}
