/*
 * drivers/misc/cxd/pcie/link_speed.c
 *   PCIe Link speed API
 *
 * Copyright 2022 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */

#include <linux/moduleparam.h>
#include "internal.h"

static int set_link_speed_delay_us   =    1; /*   1 us */
static int set_link_speed_sleep_us   =   15; /*  15 us */
static int set_link_speed_timeout_us = 1000; /* TBD  1 ms */
module_param_named(set_link_speed_delay_us  , set_link_speed_delay_us  , int, S_IRUSR|S_IWUSR);
module_param_named(set_link_speed_sleep_us  , set_link_speed_sleep_us  , int, S_IRUSR|S_IWUSR);
module_param_named(set_link_speed_timeout_us, set_link_speed_timeout_us, int, S_IRUSR|S_IWUSR);

/* read current Link Speed */
static int read_link_speed(struct pci_dev *dev)
{
	int ret;

	ret = pcie_dwc_get_speed(dev);
	if (ret < 0) {
		printk(KERN_ERR "ERROR:%s:pcie_dwc_get_speed:ret=%d\n", __func__, ret);
	}
	return ret;
}

/* change Link Speed */
static int change_link_speed(struct pci_dev *dev, int ch, unsigned int current_link_speed, unsigned int target_link_speed)
{
	int ret;

	ret = pcie_dwc_set_speed(dev, target_link_speed);
	if (ret < 0) {
		printk(KERN_ERR "ERROR:%s:link_speed=%d:pcie_dwc_set_speed:ret=%d\n", __func__, target_link_speed, ret);
	}
	return ret;
}

/* Link Speed: parameters */
static const struct pcie_link link_speed_param = {
	.name = "Speed",
	.read   = read_link_speed,
	.change = change_link_speed,
	.delay_us   = &set_link_speed_delay_us,
	.sleep_us   = &set_link_speed_sleep_us,
	.timeout_us = &set_link_speed_timeout_us,
};

/* Link Speed API: set */
int pcie_set_link_speed(int ch, PCIE_LINK_SPEED target_link_speed)
{
	return cxpcie_set_link(ch, (unsigned int)target_link_speed, &link_speed_param);
}
EXPORT_SYMBOL(pcie_set_link_speed);

/* Link Speed API: get */
int pcie_get_link_speed(int ch)
{
	return cxpcie_get_link(ch, &link_speed_param);
}
EXPORT_SYMBOL(pcie_get_link_speed);

int pcie_link_speed_check_dev_state(int ch)
{
	return cxpcie_check_dev_state(ch);
}
EXPORT_SYMBOL(pcie_link_speed_check_dev_state);

/* backward compt */
int pcie_link_speed_check_dev_state_ch0(void)
{
	return pcie_link_speed_check_dev_state(DEFAULT_CH);
}
EXPORT_SYMBOL(pcie_link_speed_check_dev_state_ch0);
