/*
 * drivers/misc/cxd/pcie/link_width.c
 *   PCIe Link width API
 *
 * Copyright 2022 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/moduleparam.h>
#include "internal.h"

static int set_link_width_delay_us   =    1; /*   1 us */
static int set_link_width_sleep_us   =   15; /*  15 us */
static int set_link_width_timeout_us = 1100; /* TBD 1.1 ms (> 1,050,864ns@4->2, 1,050,902ns@4->1, 12,864ns@2->4, 12,862ns@1->4) */
module_param_named(set_link_width_delay_us  , set_link_width_delay_us  , int, S_IRUSR|S_IWUSR);
module_param_named(set_link_width_sleep_us  , set_link_width_sleep_us  , int, S_IRUSR|S_IWUSR);
module_param_named(set_link_width_timeout_us, set_link_width_timeout_us, int, S_IRUSR|S_IWUSR);

/* read current Link Width */
static int read_link_width(struct pci_dev *dev)
{
	int ret;
	int width = -EINVAL;

	ret = pcie_dwc_get_lanes(dev);
	if (ret >= 0) {
		switch (ret) {
		case PCIE_LANE1:
			width = 1;
			break;
		case PCIE_LANE2:
			width = 2;
			break;
		case PCIE_LANE4:
			width = 4;
			break;
		default:
			ret = -EFAULT;
			break;
		}
	}
	if (ret < 0) {
		printk(KERN_ERR "ERROR:%s:pcie_dwc_get_lanes:ret=%d\n", __func__, ret);
		return ret;
	}
	return width;
}

static int change_link_width(struct pci_dev *dev, int ch, unsigned int current_link_width, unsigned int target_link_width)
{
	const int aspm_policy = pcie_aspm_policy(ch);
	enum PCIE_LANE lanes;
	int err;
	extern int pcie_aspm_set_domain_policy(int domain, int policy);

	/* convert parameter */
	switch (target_link_width) {
	case 1:
		lanes = PCIE_LANE1;
		break;
	case 2:
		lanes = PCIE_LANE2;
		break;
	case 3:
		lanes = PCIE_LANE4;
		break;
	default:
		return -EINVAL;
	}

	/* ASPM OFF */
	if (aspm_policy != 1) {
		int aspm_err;

		pcie_aspm_ctrl_lock(ch);
		aspm_err = pcie_aspm_set_domain_policy(ch, 1);
		if (aspm_err) {
			printk(KERN_ERR "ERROR:%s: ASPM OFF: ch=%d, policy=%d, err=%d\n", __func__, ch, aspm_policy, aspm_err);
			pcie_aspm_ctrl_unlock(ch);
			return aspm_err;
		}
	}
	/* change Link Width */
	err = pcie_dwc_set_lanes(dev, lanes);
	if (err) {
		printk(KERN_ERR "ERROR:%s: failed to change the Link Width: ch=%d, current=%d, target=%d, err=%d\n", __func__, ch, current_link_width, target_link_width, err);
	}
	/* ASPM ON */
	if (aspm_policy != 1) {
		int aspm_err = pcie_aspm_set_domain_policy(ch, aspm_policy);
		pcie_aspm_ctrl_unlock(ch);
		if (aspm_err) {
			printk(KERN_ERR "ERROR:%s: ASPM ON: ch=%d, policy=%d, err=%d\n", __func__, ch, aspm_policy, aspm_err);
			return aspm_err;
		}
	}
	return err;
}

static const struct pcie_link link_width_param = {
	.name = "Width",
	.read   = read_link_width,
	.change = change_link_width,
	.delay_us   = &set_link_width_delay_us,
	.sleep_us   = &set_link_width_sleep_us,
	.timeout_us = &set_link_width_timeout_us,
};

int pcie_get_link_width(int ch)
{
	return cxpcie_get_link(ch, &link_width_param);
}
EXPORT_SYMBOL(pcie_get_link_width);

int pcie_set_link_width(int ch, unsigned int target_link_width)
{
	return cxpcie_set_link(ch, target_link_width, &link_width_param);
}
EXPORT_SYMBOL(pcie_set_link_width);

int pcie_link_width_check_dev_state(int ch)
{
	return cxpcie_check_dev_state(ch);
}
EXPORT_SYMBOL(pcie_link_width_check_dev_state);

/* backward compat */
int pcie_link_width_check_dev_state_ch0(void)
{
	return pcie_link_width_check_dev_state(DEFAULT_CH);
}
EXPORT_SYMBOL(pcie_link_width_check_dev_state_ch0);
