/*
 * drivers/misc/cxd/pcie/lock.c
 *   lock API
 *
 * Copyright 2023 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/moduleparam.h>

#include "internal.h"

/* pci_lock */
static raw_spinlock_t pci_lock[N_PCIE] = {
	__RAW_SPIN_LOCK_UNLOCKED(pci_lock[0]),
	__RAW_SPIN_LOCK_UNLOCKED(pci_lock[1]),
	__RAW_SPIN_LOCK_UNLOCKED(pci_lock[2]),
	__RAW_SPIN_LOCK_UNLOCKED(pci_lock[3]),
	__RAW_SPIN_LOCK_UNLOCKED(pci_lock[4])
};

static int get_domain_nr(struct pci_bus *bus, const char *name)
{
	int domain = pci_domain_nr(bus);

	if (domain < 0 || N_PCIE <= domain) {
		printk(KERN_ERR "ERROR:%s:domain=%d\n", name, domain);
		domain = 0;
	}
	return domain;
}

void pci_lock_irqsave(struct pci_bus *bus, unsigned long *flags)
	__acquires(pci_lock)
{
	int n = get_domain_nr(bus, __func__);
	raw_spin_lock_irqsave(&pci_lock[n], *flags);
}

void pci_unlock_irqrestore(struct pci_bus *bus, unsigned long flags)
	__releases(pci_lock)
{
	int n = get_domain_nr(bus, __func__);
	raw_spin_unlock_irqrestore(&pci_lock[n], flags);
}

void pci_lock_irq(struct pci_bus *bus)
	__acquires(pci_lock)
{
	int n = get_domain_nr(bus, __func__);
	raw_spin_lock_irq(&pci_lock[n]);
}

void pci_unlock_irq(struct pci_bus *bus)
	__releases(pci_lock)
{
	int n = get_domain_nr(bus, __func__);
	raw_spin_unlock_irq(&pci_lock[n]);
}
