/*
 * drivers/misc/cxd/pcie/ltssm.c
 *
 *
 * Copyright 2024 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/module.h>
#include <linux/types.h>
#include <linux/proc_fs.h>
#include <linux/printk.h>
#include <linux/sched.h>
#include <uapi/linux/sched/types.h>
#include <linux/kthread.h>
#include <linux/delay.h>
#include <linux/io.h>
#include <linux/pci.h>
#include <linux/snsc_boot_time.h>
#include <linux/udif/timer.h>
#include <mach/platform.h>

static int lmon_ch;
module_param_named(lmon_ch, lmon_ch, int, 0644);
static int ring;
module_param_named(lmon_ring, ring, int, 0444);
static int n_entry;
module_param_named(lmon_n, n_entry, int, 0444);
static ulong phys;
module_param_named(lmon_phys, phys, ulong, 0444);
static int cpu = -1;
module_param_named(lmon_cpu, cpu, int, 0444);
static int policy = SCHED_IDLE;
module_param_named(lmon_pol, policy, int, 0444);
static int prio;
module_param_named(lmon_pri, prio, int, 0444);

static void __iomem *ik_base;
#define DL_STS		0x2c
#define LTSSM		0x30

struct ltssm_log {
	unsigned long t1, t2;
	unsigned char ltssm;
	unsigned char dl;
};
static struct ltssm_log *lmon_log;
static int put, cnt;

#define N_STATE 0x24
static char *stname[N_STATE] = {
	[0x00] = "DET_Q",
	[0x01] = "DET_ACT",
	[0x02] = "POLL_ACT",
	[0x03] = "POLL_COMPLIANCE",
	[0x04] = "POLL_CFG",
	[0x05] = "PRE_DET_Q",
	[0x06] = "DET_WAIT",
	[0x07] = "CFG_WIDTH_S",
	[0x08] = "CFG_WIDTH_A",
	[0x09] = "CFG_LANE_W",
	[0x0A] = "CFG_LANE_A",
	[0x0B] = "CFG_COMP",
	[0x0C] = "CFG_IDLE",
	[0x0D] = "RCVRY_LOCK",
	[0x0E] = "RCVRY_SPEED",
	[0x0F] = "RCVRY_RCVRCFG",
	[0x10] = "RCVRY_IDLE",
	[0x11] = "L0",
	[0x12] = "L0s",
	[0x13] = "L123_SEND_EIDLE",
	[0x14] = "L1_IDLE",
	[0x15] = "L2_IDLE",
	[0x16] = "L2_WAKE",
	[0x17] = "DISABLED_ENTRY",
	[0x18] = "DISABLED_IDLE",
	[0x19] = "DISABLED",
	[0x1A] = "LPBK_ENTRY",
	[0x1B] = "LPBK_ACT",
	[0x1C] = "LPBK_EXIT",
	[0x1D] = "LPBK_EXIT_TMO",
	[0x1E] = "HOT_RESET_ENT",
	[0x1F] = "HOT_RESET",
	[0x20] = "RCVRY_EQ0",
	[0x21] = "RCVRY_EQ1",
	[0x22] = "RCVRY_EQ2",
	[0x23] = "RCVRY_EQ3",
};

static void log_dump(void)
{
	int i, get;

	printk(KERN_ERR "PCIE%d:\n", lmon_ch);
	get = 0;
	if (n_entry == cnt) {
		get = put + 1;
		if (n_entry == get)
			get = 0;
	}
	for (i = 0; i < cnt; i++) {
		unsigned char st = lmon_log[get].ltssm;
		unsigned char dl = lmon_log[get].dl;
		printk(KERN_ERR "%llu-%llu: %02x", udif_cycles_to_usecs(lmon_log[get].t1), udif_cycles_to_usecs(lmon_log[get].t2), st);
		if (st < N_STATE) {
			printk(KERN_CONT " %s %x", stname[st], dl);
		}
		printk(KERN_CONT "\n");
		get++;
		if (n_entry == get)
			get = 0;
	}
}

static void log_init(void)
{
	put = cnt = 0;
}

static inline void log_add(unsigned long t, uint32_t ltssm, uint32_t dl)
{
	if (cnt) {
		put++;
		if (n_entry == put) {
			if (!ring)
				return;
			put = 0;
		}
	}
	lmon_log[put].t1 = t;
	lmon_log[put].t2 = t;
	lmon_log[put].ltssm = ltssm;
	lmon_log[put].dl = (dl >> 14)|(dl >> 7)|dl;
	if (cnt < n_entry)
		cnt++;
}

static int logger(void *data)
{
	unsigned long flags;
	unsigned long t;
	uint32_t sts, cur, prev, prev_saved;
	uint32_t dl;

	log_init();
	prev_saved = ~0;
	while (!kthread_should_stop()) {
		if (kthread_should_park()) {
			kthread_parkme();
			/* restart from here */
			log_init();
			prev_saved = ~0;
			continue;
		}
		if (!ring && cnt == n_entry) {
			/* FULL */
			kthread_parkme();
			/* restart from here */
			log_init();
			prev_saved = ~0;
			continue;
		}
		local_irq_save(flags);
		t = get_cycles();
		sts = readl_relaxed(ik_base + LTSSM);
		dl  = readl_relaxed(ik_base + DL_STS);
		local_irq_restore(flags);
		prev = (sts >> 16) & 0x3f;
		cur = sts & 0x3f;

		if (sts != prev_saved) {
			if (!cnt || (prev_saved & 0x3f) == prev) {
				log_add(t, cur, dl);
			} else {
				log_add(t, prev, dl);
				log_add(t, cur, dl);
			}
			prev_saved = sts;
		} else {
			lmon_log[put].t2 = t;
		}
	}

	log_dump();
	return 0;
}

static struct task_struct *lmon_tsk;

static ssize_t lmon_write(struct file *file, const char __user *userbuf, size_t count, loff_t *data)
{
	char buf[16];
	int n;

	n = count;
	if (count > sizeof(buf) - 1)
		n = sizeof(buf) - 1;
	if (copy_from_user(buf, userbuf, n)) {
		return -EFAULT;
	}
	if (!strncmp(buf, "on", 2)) {
		if (!IS_ERR_OR_NULL(lmon_tsk)) {
			kthread_unpark(lmon_tsk);
		}
	} else if (!strncmp(buf, "off", 3)) {
		if (!IS_ERR_OR_NULL(lmon_tsk)) {
			kthread_park(lmon_tsk);
		}
	} else if (!strncmp(buf, "dump", 4)) {
		if (!IS_ERR_OR_NULL(lmon_tsk)) {
			kthread_park(lmon_tsk);
			log_dump();
		}
	}
	return count;
}

struct proc_ops cxpcie_proc_ltssm_fops = {
	.proc_write = lmon_write,
};

void pcie_ltssm_start(int ch, void __iomem *base)
{
	long size;
	struct sched_attr param = {
		.sched_policy = policy,
		.sched_priority = prio,
	};

	if (!n_entry || ch != lmon_ch)
		return;
	ik_base = base;
	if (!IS_ERR_OR_NULL(lmon_tsk)) {
		kthread_unpark(lmon_tsk);
		return;
	}

	size = n_entry * sizeof (struct ltssm_log);
	if (phys) {
		lmon_log = ioremap_wc(phys, size);
	} else {
		lmon_log = vmalloc(size);
	}
	if (!lmon_log) {
		printk(KERN_ERR "%s:can not alloc buffer\n", __func__);
		goto err1;
	}
	memset(lmon_log, 0, size);

	lmon_tsk = kthread_create(logger, NULL, "LTSSM_LOGGER");
	if (IS_ERR(lmon_tsk)) {
		printk(KERN_ERR "%s:kthread create failed\n", __func__);
		goto err2;
	}
	if (cpu >= 0) {
		kthread_bind(lmon_tsk, cpu);
	}
	sched_setattr_nocheck(lmon_tsk, &param);
	wake_up_process(lmon_tsk);
	return;

err2:
	if (phys)
		iounmap(lmon_log);
	else
		vfree(lmon_log);
	lmon_log = NULL;
err1:
	n_entry = 0;
	return;
}

void pcie_ltssm_stop(int ch, void __iomem *base)
{
	if (!n_entry || ch != lmon_ch)
		return;
	if (!IS_ERR_OR_NULL(lmon_tsk)) {
		kthread_park(lmon_tsk);
	}
}

#if 0
void cxpcie_ltssm_cleanup(void)
{
	if (!IS_ERR_OR_NULL(lmon_tsk)) {
		kthread_stop(lmon_tsk);
		tsk = NULL;
	}
	if (lmon_log) {
		if (phys)
			iounmap(lmon_log);
		else
			vfree(lmon_log);
		lmon_log = NULL;
	}
}
#endif
