/*
 * drivers/misc/cxd/pcie/misc.c
 *
 *
 * Copyright 2023 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */

#include <linux/delay.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <mach/moduleparam.h>
#include <mach/platform.h>
#include <mach/regs-gpio.h>
#include "internal.h"

static const char *cxpcie_ch_name[PCIE_CH_MAX] = {
	"pcie_n",
	"pcie_m0",
	"pcie_m1",
	"pcie_c",
	"pcie_f"
};

const char *pcie_name(int ch)
{
	if (ch < 0 || ch >= PCIE_CH_MAX)
		return NULL;
	return cxpcie_ch_name[ch];
}

#define PCIEC_SIZE 0x1000000000UL /* 64GB */

int pcibios_assign_resource(struct pci_dev *dev, int resno, resource_size_t size, resource_size_t min_align)
{
	struct resource *res = dev->resource + resno;

	if (PCI_VENDOR_ID_SONY == dev->vendor
	    && PCI_DEVICE_ID_CXD90XXX == dev->device) {
		switch (resno) {
		case 0:
			if (PCIEC_SIZE == size) {
				printk(KERN_INFO "%s: %x %d %llx\n", __func__,
				       dev->vendor, resno, size);
				res->start = 0x000000000UL;
				res->end   = res->start + size - 1;
				return 0;
			}
			break;
		default:
			break;
		}
	}
	return -1;
}

/* AER */
static int noaer[N_PCIE] = { 0, 0, 0, 0, 0 };
module_param_named(noaer0, noaer[0], int, S_IRUSR|S_IWUSR);
module_param_named(noaer1, noaer[1], int, S_IRUSR|S_IWUSR);
module_param_named(noaer2, noaer[2], int, S_IRUSR|S_IWUSR);
module_param_named(noaer3, noaer[3], int, S_IRUSR|S_IWUSR);
module_param_named(noaer4, noaer[4], int, S_IRUSR|S_IWUSR);

int pci_aer_disable(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return 0;
	return noaer[ch];
}

static int vboost[N_PCIE] = { -1, -1, -1, -1, -1 };
module_param_named(vboost0, vboost[0], int, S_IRUSR|S_IWUSR);
module_param_named(vboost1, vboost[1], int, S_IRUSR|S_IWUSR);
module_param_named(vboost2, vboost[2], int, S_IRUSR|S_IWUSR);
module_param_named(vboost3, vboost[3], int, S_IRUSR|S_IWUSR);
module_param_named(vboost4, vboost[4], int, S_IRUSR|S_IWUSR);

int pcie_dwc_vboost(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return -1;
	return READ_ONCE(vboost[ch]);
}

int pcie_set_vboost(int ch, int val)
{
	if (ch < 0 || N_PCIE <= ch)
		return -1;
	WRITE_ONCE(vboost[ch], val);
	return 0;
}
EXPORT_SYMBOL(pcie_set_vboost);

static int tmo[N_PCIE] = { -1, -1, -1, -1, -1 };
/*  0:28ms-44ms,  1:65us-99us,  2:4.1ms-6.2ms,
    5:28ms-44ms,  6:86ms-131ms, 9:260ms-390ms,
   10:1.8s-2.8s, 13:5.4s-8.2s, 14:38s-58s , -1:not specify */
module_param_named(tmo0, tmo[0], int, S_IRUSR|S_IWUSR);
module_param_named(tmo1, tmo[1], int, S_IRUSR|S_IWUSR);
module_param_named(tmo2, tmo[2], int, S_IRUSR|S_IWUSR);
module_param_named(tmo3, tmo[3], int, S_IRUSR|S_IWUSR);
module_param_named(tmo4, tmo[4], int, S_IRUSR|S_IWUSR);

int pcie_dwc_tmo(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return -1;
	return tmo[ch];
}

static int max_speed[N_PCIE] = { -1, -1, -1, -1, -1 };
/* 1:Gen1, 2:Gen2, 3:Gen3, 4:Gen4, -1:auto */
module_param_named(speed0, max_speed[0], int, S_IRUSR|S_IWUSR);
module_param_named(speed1, max_speed[1], int, S_IRUSR|S_IWUSR);
module_param_named(speed2, max_speed[2], int, S_IRUSR|S_IWUSR);
module_param_named(speed3, max_speed[3], int, S_IRUSR|S_IWUSR);
module_param_named(speed4, max_speed[4], int, S_IRUSR|S_IWUSR);

int pcie_dwc_max_speed(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return -1;
	return max_speed[ch];
}

/* /PERST port */
static unsigned int perst[N_PCIE] = {
	PORT_UNDEF, PORT_UNDEF, PORT_UNDEF, PORT_UNDEF, PORT_UNDEF,
};
module_param_named(perst0, perst[0], port, S_IRUSR|S_IWUSR);
module_param_named(perst1, perst[1], port, S_IRUSR|S_IWUSR);
module_param_named(perst2, perst[2], port, S_IRUSR|S_IWUSR);
module_param_named(perst3, perst[3], port, S_IRUSR|S_IWUSR);
module_param_named(perst4, perst[4], port, S_IRUSR|S_IWUSR);

void pcie_set_perst(int ch, int assert)
{
	unsigned int port, bit;

	if (ch < 0 || N_PCIE <= ch)
		return;
	if (PORT_UNDEF == perst[ch])
		return;
	port = perst[ch] & 0xff;
	bit  = (perst[ch] >> 8) & 0xff;
	if (assert)
		writel_relaxed(BIT(bit), VA_GPIO(port)+GPIO_WDATA+GPIO_CLR);
	else
		writel_relaxed(BIT(bit), VA_GPIO(port)+GPIO_WDATA+GPIO_SET);
}

/* /PERST READ port */
static unsigned int perstR[N_PCIE] = {
	PORT_UNDEF, PORT_UNDEF, PORT_UNDEF, PORT_UNDEF, PORT_UNDEF,
};
module_param_named(perstR0, perstR[0], port, S_IRUSR|S_IWUSR);
module_param_named(perstR1, perstR[1], port, S_IRUSR|S_IWUSR);
module_param_named(perstR2, perstR[2], port, S_IRUSR|S_IWUSR);
module_param_named(perstR3, perstR[3], port, S_IRUSR|S_IWUSR);
module_param_named(perstR4, perstR[4], port, S_IRUSR|S_IWUSR);

int pcie_has_perst(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return 0;
	if (PORT_UNDEF == perstR[ch])
		return 0;
	return 1;
}

int __pcie_get_perst(int ch)
{
	unsigned int port, bit;
	int val;

	port = perstR[ch] & 0xff;
	bit  = (perstR[ch] >> 8) & 0xff;
	val = (readl_relaxed(VA_GPIO(port)+GPIO_WDATA) >> bit) & 0x1;
	return !val;
}

/* RETURN VALUE    0:negate, 1:assert, -1:not supported */
int pcie_get_perst(int ch)
{
	if (!pcie_has_perst(ch))
		return -1;
	return __pcie_get_perst(ch);
}

/* Tperst-clk */
#define PERST_DELAY 110
static unsigned int perst_delay[N_PCIE] = {
	PERST_DELAY, PERST_DELAY, PERST_DELAY, PERST_DELAY, PERST_DELAY,
};
module_param_named(perst_delay0, perst_delay[0], uint, S_IRUSR|S_IWUSR);
module_param_named(perst_delay1, perst_delay[1], uint, S_IRUSR|S_IWUSR);
module_param_named(perst_delay2, perst_delay[2], uint, S_IRUSR|S_IWUSR);
module_param_named(perst_delay3, perst_delay[3], uint, S_IRUSR|S_IWUSR);
module_param_named(perst_delay4, perst_delay[4], uint, S_IRUSR|S_IWUSR);

void pcie_perst_delay(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return;
	if (!perst_delay[ch])
		return;
	usleep_range(perst_delay[ch], perst_delay[ch] + 10);

	pcie_set_perst(ch, 0); /* negate /PERST */
}

int pcie_ext_err_stat(struct pci_bus *bus, void __iomem *dbi, void __iomem *link)
{
	int ch = pci_domain_nr(bus);

	if (ch < 0 || N_PCIE <= ch)
		return 0;

	if (pcie_aximon_stat(ch, link)) {
		return -1;
	}

	return 0;
}

/* AXI Ordering Control */
static uint axiorder[N_PCIE] = { 0, 0, 0, 0, 0 };
module_param_named(order0, axiorder[0], uint, S_IRUSR|S_IWUSR);
module_param_named(order1, axiorder[1], uint, S_IRUSR|S_IWUSR);
module_param_named(order2, axiorder[2], uint, S_IRUSR|S_IWUSR);
module_param_named(order3, axiorder[3], uint, S_IRUSR|S_IWUSR);
module_param_named(order4, axiorder[4], uint, S_IRUSR|S_IWUSR);

uint pcie_dwc_axiordering(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return 0;
	return axiorder[ch];
}
