/*
 * drivers/misc/cxd/pcie/proc.c
 *
 *
 * Copyright 2022 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/kernel.h>
#include <linux/fs.h>
#include <asm/uaccess.h>
#include <linux/proc_fs.h>
#include <linux/udif/module.h>

#include "internal.h"

#define PROC_PCIE_DIR "driver/pcie"
#define PROC_PCIE_RC "root_ctrl"
#define PROC_PCIE_LW "link_width"
#define PROC_PCIE_LWDS "link_width_dev_state"
#define PROC_PCIE_LS "link_speed"
#define PROC_PCIE_LSDS "link_speed_dev_state"
#define PROC_PCIE_LTSSM "ltssm"

static ssize_t cxpcie_proc_rc_read(struct file *file, char __user *buffer,
				   size_t count, loff_t *ppos)
{
	int ch = (uintptr_t)PDE_DATA(file_inode(file));
	char buf[64];
	size_t off = *ppos;

	buf[0] = '\0';
	scnprintf(buf, sizeof buf, "state: %d\n", pcie_cxd_status(ch));
	if (off >= strlen(buf))
		return 0;
	if (off + count > strlen(buf))
		count = strlen(buf) - off;
	if (copy_to_user(buffer, buf + off, count))
		return -EFAULT;
	*ppos += count;
	return count;
}

static ssize_t cxpcie_proc_rc_write(struct file *file, const char __user *buffer,
				    size_t count, loff_t *ppos)
{
	int ch = (uintptr_t)PDE_DATA(file_inode(file));
	char arg[16];
	long en;

	if (count >= sizeof arg)
		return -EOVERFLOW;
	if (copy_from_user(arg, buffer, count))
		return -EFAULT;
	arg[count] = '\0';
	en = simple_strtol(arg, NULL, 0);

	pcie_root_ctrl(ch, en);

	return count;
}

static ssize_t cxpcie_proc_link_read(struct file *file, char __user *buffer,
				     size_t count, loff_t *ppos,
				     int (*get_link)(int))
{
	int ch = (uintptr_t)PDE_DATA(file_inode(file));
	int current_link;
	char buf[4]; /* xxx\0 */
	int len;

	if (!buffer || (count < sizeof(buf)))
		return -EOVERFLOW;
	if (!ppos)
		return -EINVAL;

	if (*ppos > 0)
		return 0;

	current_link = (*get_link)(ch);
	if (current_link < 0)
		return current_link;

	len = num_to_str(buf, sizeof(buf) - 1, current_link, 0);
	if ((len == 0) || (sizeof(buf) - 1) < len)
		return -EOVERFLOW;

	buf[len++] = '\0';

	if (copy_to_user(buffer, buf, len))
		return -EFAULT;

	*ppos += len;
	return len;
}

static ssize_t cxpcie_proc_link_write(struct file *file, const char __user *buffer,
				    size_t count, loff_t *ppos,
				    int (*set_link)(int, unsigned int))
{
	int ch = (uintptr_t)PDE_DATA(file_inode(file));
	char arg[4]; /* xxx\0 */
	unsigned int target_link;
	int err;

	if (!buffer || (count >= sizeof(arg)))
		return -EOVERFLOW;
	if (copy_from_user(arg, buffer, count))
		return -EFAULT;
	arg[count] = '\0';
	target_link = simple_strtol(arg, NULL, 0);

	err = (*set_link)(ch, target_link);
	if (err)
		return err;

	return count;
}

static ssize_t cxpcie_proc_linkstate_read(struct file *file, char __user *buffer,
					     size_t count, loff_t *ppos,
					     int (*check_dev_state)(int ch))
{
	int ch = (uintptr_t)PDE_DATA(file_inode(file));
	int err;
	const char ready[] = "READY";
	const char nodev[] = "NODEV";
	const char *buf = NULL;
	int len;

	if (!buffer || (count < sizeof(ready)) || (count < sizeof(nodev)))
		return -EOVERFLOW;
	if (!ppos)
		return -EINVAL;

	if (*ppos > 0)
		return 0;

	err = (*check_dev_state)(ch);
	buf = (!err) ? ready : nodev;
	len = strlen(buf) + 1;

	if (copy_to_user(buffer, buf, len))
		return -EFAULT;

	*ppos += len;
	return len;
}

static ssize_t cxpcie_proc_linkwidth_read(struct file *file, char __user *buffer,
					   size_t count, loff_t *ppos)
{
	return cxpcie_proc_link_read(file, buffer, count, ppos, pcie_get_link_width);
}

static ssize_t cxpcie_proc_linkwidth_write(struct file *file, const char __user *buffer,
					    size_t count, loff_t *ppos)
{
	return cxpcie_proc_link_write(file, buffer, count, ppos, pcie_set_link_width);
}

static ssize_t cxpcie_proc_linkwidth_state_read(struct file *file, char __user *buffer,
						     size_t count, loff_t *ppos)
{
	return cxpcie_proc_linkstate_read(file, buffer, count, ppos, pcie_link_width_check_dev_state);
}

static ssize_t cxpcie_proc_linkspeed_read(struct file *file, char __user *buffer,
					   size_t count, loff_t *ppos)
{
	return cxpcie_proc_link_read(file, buffer, count, ppos, pcie_get_link_speed);
}

static ssize_t cxpcie_proc_linkspeed_write(struct file *file, const char __user *buffer,
					    size_t count, loff_t *ppos)
{
	return cxpcie_proc_link_write(file, buffer, count, ppos, pcie_set_link_speed);
}

static ssize_t cxpcie_proc_linkspeed_state_read(struct file *file, char __user *buffer,
						     size_t count, loff_t *ppos)
{
	return cxpcie_proc_linkstate_read(file, buffer, count, ppos, pcie_link_speed_check_dev_state);
}

static struct proc_dir_entry *cxpcie_proc;

static const struct proc_ops cxpcie_proc_rc_fops = {
	.proc_read  = cxpcie_proc_rc_read,
	.proc_write = cxpcie_proc_rc_write,
};

static const struct proc_ops cxpcie_proc_linkwidth_fops = {
	.proc_read  = cxpcie_proc_linkwidth_read,
	.proc_write = cxpcie_proc_linkwidth_write,
};

static const struct proc_ops cxpcie_proc_linkwidth_state_fops = {
	.proc_read  = cxpcie_proc_linkwidth_state_read,
};

static const struct proc_ops cxpcie_proc_linkspeed_fops = {
	.proc_read  = cxpcie_proc_linkspeed_read,
	.proc_write = cxpcie_proc_linkspeed_write,
};

static const struct proc_ops cxpcie_proc_linkspeed_state_fops = {
	.proc_read  = cxpcie_proc_linkspeed_state_read,
};

void cxpcie_proc_create(int ch)
{
	struct proc_dir_entry *dir, *ent;
	char name[16];
	const char *alias;

	if (!cxpcie_proc)
		return;
	scnprintf(name, sizeof name, "%u", ch);
	dir = proc_mkdir(name, cxpcie_proc);
	if (!dir)
		return;
	alias = pcie_name(ch);
	if (alias) {
		ent = proc_symlink(alias, cxpcie_proc, name);
	}
	ent = proc_create_data(PROC_PCIE_RC, 0, dir, &cxpcie_proc_rc_fops, (void *)(uintptr_t)ch);
	ent = proc_create_data(PROC_PCIE_LW, 0, dir, &cxpcie_proc_linkwidth_fops, (void *)(uintptr_t)ch);
	ent = proc_create_data(PROC_PCIE_LS, 0, dir, &cxpcie_proc_linkspeed_fops, (void *)(uintptr_t)ch);
	ent = proc_create_data(PROC_PCIE_LWDS, 0, dir, &cxpcie_proc_linkwidth_state_fops, (void *)(uintptr_t)ch);
	ent = proc_create_data(PROC_PCIE_LSDS, 0, dir, &cxpcie_proc_linkspeed_state_fops, (void *)(uintptr_t)ch);
	ent = proc_create_data(PROC_PCIE_LTSSM, 0, dir, &cxpcie_proc_ltssm_fops, (void *)(uintptr_t)ch);
}

void UDIF_INIT cxpcie_proc_init(void)
{
	cxpcie_proc = proc_mkdir(PROC_PCIE_DIR, NULL);
}
