/*
* drivers/misc/cxd/pcie/prsnt.c
*
*
* Copyright 2024 Sony Corporation
*
*  This program is free software; you can redistribute  it and/or modify it
*  under  the terms of  the GNU General  Public License as published by the
*  Free Software Foundation;  version 2 of the  License.
*
*  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
*  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
*  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
*  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
*  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
*  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
*  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
*  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
*  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*  You should have received a copy of the  GNU General Public License along
*  with this program; if not, write  to the Free Software Foundation, Inc.,
*  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
*
*/

#include <linux/snsc_boot_time.h>
#include <linux/udif/timer.h>
#include <linux/moduleparam.h>
#include <mach/moduleparam.h>
#include <mach/platform.h>
#include <mach/regs-gpio.h>
#include <mach/irqs.h>
#include "internal.h"

/*
 * Detect EJECT
 */
/* duration of /PRSNT=H */
static uint prsntmo[N_PCIE] = { 0, 0, 0, 0, 0 };
module_param_named(prsntmo0, prsntmo[0], uint, S_IRUSR|S_IWUSR);
module_param_named(prsntmo1, prsntmo[1], uint, S_IRUSR|S_IWUSR);
module_param_named(prsntmo2, prsntmo[2], uint, S_IRUSR|S_IWUSR);
module_param_named(prsntmo3, prsntmo[3], uint, S_IRUSR|S_IWUSR);
module_param_named(prsntmo4, prsntmo[4], uint, S_IRUSR|S_IWUSR);

static unsigned long long pcie_prsnt_tmo(int ch)
{
	return udif_usecs_to_cycles(prsntmo[ch]);
}

/* /PRSNT port */
static unsigned int prsnt[N_PCIE] = {
PORT_UNDEF, PORT_UNDEF, PORT_UNDEF, PORT_UNDEF, PORT_UNDEF,
};
module_param_named(prsnt0, prsnt[0], port, S_IRUSR|S_IWUSR);
module_param_named(prsnt1, prsnt[1], port, S_IRUSR|S_IWUSR);
module_param_named(prsnt2, prsnt[2], port, S_IRUSR|S_IWUSR);
module_param_named(prsnt3, prsnt[3], port, S_IRUSR|S_IWUSR);
module_param_named(prsnt4, prsnt[4], port, S_IRUSR|S_IWUSR);

static int pcie_has_prsnt(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return 0;
	if (PORT_UNDEF == prsnt[ch])
		return 0;
	if (!prsntmo[ch])
		return 0;
	return 1;
}

static int pcie_cxd_prsnt(int ch)
{
	unsigned int port, bit;
	int val;

	port = prsnt[ch] & 0xff;
	bit  = (prsnt[ch] >> 8) & 0xff;
	val = (readl_relaxed(VA_GPIO(port)+GPIO_RDATA) >> bit) & 0x1;
	return !val;
}

#if 0
static int pcie_prsnt_irq(int ch)
{
	unsigned int port, bit;
	int irq;

	if (!pcie_has_prsnt(ch))
		return -1;
	port = prsnt[ch] & 0xff;
	bit  = (prsnt[ch] >> 8) & 0xff;
	irq = gpiopin_to_irq(port, bit);
	return irq;
}
#endif

static int pcie_cxd_state[N_PCIE]; /* -1:normal, 0:eject */

void pcie_cxd_state_init(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return;
	pcie_cxd_state[ch] = -1;
	smp_mb();
}

static void pcie_cxd_set_eject(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return;
	pcie_cxd_state[ch] = 0;
	smp_mb();
}

int pcie_cxd_status(int ch)
{
	if (ch < 0 || N_PCIE <= ch)
		return -1;
	smp_mb();
	return pcie_cxd_state[ch];
}

static void pcie_tlog(int ch, char *msg)
{
	char buf[64] = "PCIx:";

	buf[3] = '0' + ch;
	strlcat(buf, msg, sizeof buf);
	BOOT_TIME_ADD1(buf);
}

/* check /PRSNT and /PERST */
/* Must be called with interrupts disabled */
/* (note) dev is not EP but RC. */
int pcie_check_prsnt(struct pci_dev *dev)
{
	int ch = pci_domain_nr(dev->bus);
	int has_perst, has_prsnt, ret;
	u64 tmo;
	extern int pcie_dwc_linkup(struct pci_dev *dev);

	has_perst = pcie_has_perst(ch);
	has_prsnt = pcie_has_prsnt(ch);
	if (!has_prsnt) {
		if (has_perst && __pcie_get_perst(ch)) {
			/* /PERST is asserted */
			pcie_cxd_set_eject(ch);
			pcie_tlog(ch, "/PERST=L");
			return 0;
		}
		return 1;
	}

	if (!pcie_cxd_status(ch)) { /* EJECT state */
		return 0;
	}

	/* detect continuous negation of /PRSNT */
	tmo = udif_read_freerun() + pcie_prsnt_tmo(ch);
	do {
		if (has_perst) {
			if (__pcie_get_perst(ch)) {
				/* /PERST is asserted */
				pcie_cxd_set_eject(ch);
				pcie_tlog(ch, "/PERST=L");
				return 0;
			}
		}

		if (pcie_cxd_prsnt(ch)) {
			/* /PRSNT is asserted */
			ret = pcie_dwc_linkup(dev);
			if (likely(ret)) {
				/* link up */
				return 1;
			}
			/* link down */
			pcie_cxd_set_eject(ch);
			pcie_tlog(ch, "LINKDN");
			return 0;
		}
	} while (udif_read_freerun() < tmo);

	/* EJECT */
	pcie_cxd_set_eject(ch);
	pcie_tlog(ch, "/PRSNT=H");
	return 0;
}
