/*
 * driver/misc/cxd/pciedma/connect.c
 *
 * Copyright 2022 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */

#include "internal.h"

static void pcidma_run(dmac_t *dmac)
{
	int ch;

	for (ch = 0; ch < PCIDMA_CH_MAX; ch++) {
		chan_t *chan = pcidma_get_chan(dmac->id, ch);
		unsigned long flags;

		LOCK(chan);
		if (chan->state || chan->cur) {
			UNLOCK(chan);
			chan_perr(dmac->id, ch, "%s:state=%d,cur=%p\n",
				  __func__, chan->state, chan->cur);
			BUG();
		}
		chan->state = 1; /* connected */
		pcidma_start(chan);
		UNLOCK(chan);
	}
}

static void pcidma_stop(dmac_t *dmac)
{
	int ch;

	for (ch = 0; ch < PCIDMA_CH_MAX; ch++) {
		chan_t *chan = pcidma_get_chan(dmac->id, ch);
		unsigned long flags;

		LOCK(chan);
		chan->state = 0; /* disconnected */
		UNLOCK(chan);
	}
}

void pcidma_shutdown(dmac_t *dmac)
{
	unsigned long flags;

	LOCK(dmac);
	if (dmac->state) {
		dmac->state = 0;
		pcidma_stop(dmac);
	}
	UNLOCK(dmac);
}

int pcidma_connect(uint pcie_ch, int connect)
{
	dmac_t *dmac;
	unsigned long flags;

	dmac = pcidma_get_dmac(pcie_ch);
	if (!dmac) {
		printk(KERN_ERR "ERROR:PCIDMA:%s:pci_id=%d\n", __func__, pcie_ch);
		return -1;
	}
	dmac_log(pcie_ch, "connect=%d", connect);
	LOCK(dmac);
	if (!dmac->state && connect) { /* connect */
		dmac->state = 1; /* connected */
		pcidma_run(dmac);
	} else if (dmac->state && !connect) { /* disconnect */
		dmac->state = 0; /* disconnected */
		pcidma_stop(dmac);
	}
	UNLOCK(dmac);
	return 0;
}
EXPORT_SYMBOL(pcidma_connect);
