/*
 * driver/misc/cxd/pciedma/dma_core.c
 *
 * Copyright 2022 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/errno.h>

#include "internal.h"

/*-----------------  Kernel API ----------------------*/
static inline void req_enqueue(chan_t *chan, req_t *req)
{
	udif_list_add_tail(&req->drv.list, &chan->pending_list);
	chan->stat.pending_cnt++;
}

static inline req_t *req_dequeue(chan_t *chan)
{
	req_t *next;

	if (udif_list_empty(&chan->pending_list))
		return NULL;
	next = udif_list_entry_next(&chan->pending_list, req_t, drv.list);
	udif_list_del(&next->drv.list);
	chan->stat.pending_cnt--;
	return next;
}

void pcidma_start(chan_t *chan)
{
	req_t *req;
	int ret;

	if (dma_busy(chan))
		return;
	if (dma_suspended(chan))
		return;
	if (!chan->state)
		return;

	do {
		req = req_dequeue(chan);
		if (!req)
			return;

		chan->cur = req;
		ret = hw_pcidma_start(chan);
		if (ret < 0) {
			chan->stat.err_cnt++;
		}
	} while (ret < 0);
}

void pcidma_cb(chan_t *chan, int err)
{
	req_t *cur;
	int suspend;
	unsigned long flags;

	if (err) {
		chan_errlog(chan->pcie_id, chan->dmac_ch, "abort");
	}

	LOCK(chan);
	chan->stat.done_cnt++;
	if (err)
		chan->stat.abort_cnt++;
	cur = chan->cur;
	suspend = dma_suspended(chan);
	UNLOCK(chan);

	if (unlikely(suspend))
		return;
	if (cur) {
		if (cur->callback) {
			(*cur->callback)(err, cur->cookies);
		} else {
			chan_perr(chan->pcie_id, chan->dmac_ch, "cb=NULL");
			BUG();
		}
	}

	LOCK(chan);
	chan->cur = NULL;
	pcidma_start(chan);
	UNLOCK(chan);
}

static inline int pcidma_check_args(req_t *req)
{
	if (!req)
		return -1;
	if (req->pcie_ch >= PCIDMA_PCIE_MAX)
		return -1;
	if (req->dmac_ch >= PCIDMA_CH_MAX)
		return -1;
	if (req->dir > PCIDMA_WR)
		return -1;
	if (!req->callback)
		return -1;
	if (!req->size) { /* LL-mode */
		if (!IS_ALIGNED(req->src, PCIDMA_LLI_ALIGN)) {
			/* not aligned */
			return -1;
		}
	}
	if (hw_pcidma_check_args(req) < 0)
		return -1;
	return 0;
}

int pcidma_transfer_async(req_t *req)
{
	int ret = 0;
	chan_t *chan;
	unsigned long flags;

	if (pcidma_check_args(req) < 0) {
		return -EINVAL;
	}

	chan = pcidma_get_chan(req->pcie_ch, req->dmac_ch);
	if (!chan)
		return -EINVAL;

	req->drv.pid = task_pid_nr(current);

	LOCK(chan);
	if (dma_enabled(chan) && !dma_suspended(chan)) {
		req_enqueue(chan, req);
		pcidma_start(chan);
		ret = 0;
	} else {
		ret = -EINVAL;
	}
	UNLOCK(chan);
	return ret;
}
EXPORT_SYMBOL(pcidma_transfer_async);
