/*
 * driver/misc/cxd/pciedma/internel.h
 *
 * Copyright 2022 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.
 */
#ifndef __CXD_PCIDMA_INTERNAL_H
#define __CXD_PCIDMA_INTERNAL_H

#include "config.h"
#include "dwc_pcie.h"

#include <linux/udif/spinlock.h>
#include <linux/udif/list.h>

/*---------- override struct pcidma_request ----------*/
struct __pcidma_private_t {
	UDIF_LIST list;
	uint32_t pid;
};

#include <linux/pcidma.h>

static_assert(sizeof(struct __pcidma_private_t) == sizeof(struct pcidma_private_t));
/*----------------------------------------------------*/

typedef struct pcidma_request req_t;

/*---------- PCIeDMAC CHANNEL control ----------*/
struct dma_stat {
	int pending_cnt;
	unsigned long go_cnt, done_cnt, abort_cnt, err_cnt;
};

typedef struct dmac_ctrl dmac_t;
struct dma_chan_ctrl {
	dmac_t *dmac;
	uint pcie_id, dmac_ch;
	/* driver */
	UDIF_RAW_SPINLOCK lock;
	uint32_t flag;
	int state; /* PCIe connection state */
	UDIF_LIST pending_list;
	req_t *cur;
	struct dma_stat stat;
	/*--- hw driver private ---*/
	struct hw_pcidma_info hw;
};
typedef struct dma_chan_ctrl chan_t;

#define LOCK(x)   udif_spin_lock_irqsave(&(x)->lock, flags)
#define UNLOCK(x) udif_spin_unlock_irqrestore(&(x)->lock, flags)

/* flag definition */
#define CHAN_FLAG_EN		0x00000001U
#define CHAN_FLAG_SUSPEND	0x00000002U

static inline bool dma_enabled(chan_t *chan)
{
	return !!(chan->flag & CHAN_FLAG_EN);
}

static inline void dma_flag_set_enable(chan_t *chan)
{
	chan->flag |= CHAN_FLAG_EN;
}

static inline void dma_flag_clr_enable(chan_t *chan)
{
	chan->flag &= ~CHAN_FLAG_EN;
}

static inline bool dma_suspended(chan_t *chan)
{
	return !!(chan->flag & CHAN_FLAG_SUSPEND);
}

static inline void dma_flag_set_suspend(chan_t *chan)
{
	chan->flag |= CHAN_FLAG_SUSPEND;
}

static inline void dma_flag_clr_suspend(chan_t *chan)
{
	chan->flag &= ~CHAN_FLAG_SUSPEND;
}

static inline bool dma_busy(chan_t *chan)
{
	return !!chan->cur;
}

extern chan_t *pcidma_get_chan(uint pcie_id, uint dmac_ch);

/*---------- PCIeDMAC CHIP control ----------*/
struct dmac_ctrl {
	uint id;
	UDIF_RAW_SPINLOCK lock;
	int state; /* PCIe connection state */
	chan_t chan[PCIDMA_CH_MAX];
#ifdef CONFIG_PROC_FS
	struct proc_dir_entry *proc;
#endif
};
typedef struct dmac_ctrl dmac_t;

extern dmac_t *pcidma_get_dmac(uint pcie_id);

extern void pcidma_start(chan_t *chan);
extern void pcidma_shutdown(dmac_t *dmac);

/* dev I/F */
extern int  hw_pcidma_check_args(req_t *req);
extern int  hw_pcidma_open(chan_t *chan);
extern void hw_pcidma_close(chan_t *chan);
extern int  hw_pcidma_start(chan_t *chan);
extern void hw_pcidma_stop(chan_t *chan);
extern void pcidma_cb(chan_t *chan, int err);

/* cdev I/F */
extern void pcidma_cdev_init(void);
extern void pcidma_cdev_exit(void);

/* proc I/F */
extern void pcidma_proc_init(void);
extern void pcidma_proc_exit(void);
extern void pcidma_proc_create(dmac_t *dmac);
extern void pcidma_proc_remove(dmac_t *dmac);

/* log I/F */
extern void __pcidma_log(uint flag, uint pcie_id, uint dmac_ch, const char *fmt, ...);
  /* flag */
#define LOG_PR		0x01 /* use printk */
#define LOG_BT		0x02 /* use snsc_boot_time */
#define LOG_NO_CH	0x10 /* suppress dmac_ch */
#define LOG_NO_ID	0x20 /* suppress pcie_id and dmac_ch */
#define LOG_ERR		0x80 /* ERR */

#define pcidma_perr(fmt, ...)						\
	__pcidma_log(LOG_ERR|LOG_PR|LOG_BT|LOG_NO_ID, 0, 0, fmt, ##__VA_ARGS__)
#define chan_perr(pcie_id, dmac_ch, fmt, ...)				\
	__pcidma_log(LOG_ERR|LOG_PR|LOG_BT, pcie_id, dmac_ch, fmt, ##__VA_ARGS__)
#define chan_errlog(pcie_id, dmac_ch, fmt, ...)				\
	__pcidma_log(LOG_ERR|LOG_BT, pcie_id, dmac_ch, fmt, ##__VA_ARGS__)
#define chan_log(pcie_id, dmac_ch, fmt, ...)				\
	__pcidma_log(LOG_BT, pcie_id, dmac_ch, fmt, ##__VA_ARGS__)
#define dmac_log(pcie_id, fmt, ...)					\
	__pcidma_log(LOG_BT|LOG_NO_CH, pcie_id, 0, fmt, ##__VA_ARGS__)

#endif /* __CXD_PCIDMA_INTERNAL_H */
