/*
 * driver/misc/cxd/pciedma/module.c
 *
 * Copyright 2022 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */

#include <linux/udif/module.h>

#include "internal.h"

static dmac_t dmac_ctrl[PCIDMA_PCIE_MAX];

dmac_t *pcidma_get_dmac(uint pcie_id)
{
	if (pcie_id >= PCIDMA_PCIE_MAX)
		return NULL;
	return &dmac_ctrl[pcie_id];
}

chan_t *pcidma_get_chan(uint pcie_id, uint dmac_ch)
{
	if (pcie_id >= PCIDMA_PCIE_MAX)
		return NULL;
	if (dmac_ch >= PCIDMA_CH_MAX)
		return NULL;
	return &dmac_ctrl[pcie_id].chan[dmac_ch];
}

static UDIF_INIT UDIF_ERR pcidma_init(UDIF_VP data)
{
	pcidma_cdev_init();
#ifdef CONFIG_PROC_FS
	pcidma_proc_init();
#endif
	return UDIF_ERR_OK;
}

static UDIF_EXIT UDIF_ERR pcidma_exit(UDIF_VP data)
{
	pcidma_cdev_exit();
#ifdef CONFIG_PROC_FS
	pcidma_proc_exit();
#endif
	return UDIF_ERR_OK;
}

static void dma_stat_init(struct dma_stat *stat)
{
	memset(stat, 0, sizeof *stat);
}

static UDIF_ERR pcidma_probe(const UDIF_DEVICE *dev, UDIF_CH __unuse, UDIF_VP data)
{
	int pci, ch;
	dmac_t *dmac;
	chan_t *chan;

	for (pci = 0; pci < PCIDMA_PCIE_MAX; pci++) {
		dmac = pcidma_get_dmac(pci);
		dmac->id = pci;
		udif_spin_lock_init(&dmac->lock);
		dmac->state = 0;
		for (ch = 0; ch < PCIDMA_CH_MAX; ch++) {
			/* initialize dma_chan_ctrl */
			chan = pcidma_get_chan(pci, ch);
			chan->dmac = dmac;
			chan->pcie_id = pci;
			chan->dmac_ch = ch;
			udif_spin_lock_init(&chan->lock);
			chan->flag = 0U;
			chan->state = 0;
			udif_list_head_init(&chan->pending_list);
			chan->cur = NULL;
			dma_stat_init(&chan->stat);

			if (hw_pcidma_open(chan) < 0) {
				continue;
			}
			dma_flag_set_enable(chan);
		}
#ifdef CONFIG_PROC_FS
		pcidma_proc_create(dmac);
#endif
	}
	return UDIF_ERR_OK;
}

static UDIF_ERR pcidma_remove(const UDIF_DEVICE *dev, UDIF_CH __unuse, UDIF_VP data)
{
	int pci, ch;

	for (pci = 0; pci < PCIDMA_PCIE_MAX; pci++) {
		dmac_t *dmac = pcidma_get_dmac(pci);
#ifdef CONFIG_PROC_FS
		pcidma_proc_remove(dmac);
#endif
		pcidma_shutdown(dmac);
		/* close */
		for (ch = 0; ch < PCIDMA_CH_MAX; ch++) {
			chan_t *chan = pcidma_get_chan(pci, ch);
			unsigned long flags;

			LOCK(chan);
			if (dma_enabled(chan)) {
				hw_pcidma_close(chan);
				dma_flag_clr_enable(chan);
			}
			UNLOCK(chan);
		}
	}
	return UDIF_ERR_OK;
}

#ifdef CONFIG_PM
static UDIF_ERR pcidma_suspend(const UDIF_DEVICE *dev, UDIF_CH __unuse, UDIF_VP data)
{
	int pci, ch;
	chan_t *chan;
	unsigned long flags;

	for (pci = 0; pci < PCIDMA_PCIE_MAX; pci++) {
		for (ch = 0; ch < PCIDMA_CH_MAX; ch++) {
			chan = pcidma_get_chan(pci, ch);
			LOCK(chan);
			if (dma_enabled(chan)) {
				dma_flag_set_suspend(chan);
				udif_list_head_init(&chan->pending_list);
				chan->cur = NULL;
			}
			UNLOCK(chan);
		}
	}
	return UDIF_ERR_OK;
}

static UDIF_ERR pcidma_resume(const UDIF_DEVICE *dev, UDIF_CH __unuse, UDIF_VP data)
{
	int pci, ch;
	chan_t *chan;
	unsigned long flags;

	for (pci = 0; pci < PCIDMA_PCIE_MAX; pci++) {
		for (ch = 0; ch < PCIDMA_CH_MAX; ch++) {
			chan = pcidma_get_chan(pci, ch);
			LOCK(chan);
			if (dma_enabled(chan)) {
				dma_flag_clr_suspend(chan);
			}
			UNLOCK(chan);
		}
	}
	return UDIF_ERR_OK;
}
#endif /* CONFIG_PM */

static UDIF_DRIVER_OPS pcidma_ops = {
	.init		= pcidma_init,
	.exit		= pcidma_exit,
	.probe		= pcidma_probe,
	.remove		= pcidma_remove,
#ifdef CONFIG_PM
	.suspend	= pcidma_suspend,
	.resume		= pcidma_resume,
#endif
};

UDIF_IDS(pcidma_ids) = {};
UDIF_DEPS(pcidma_deps) = {};
UDIF_DECLARE_DRIVER(udif_pcidma, DRV_NAME, DRV_VER, &pcidma_ops, pcidma_ids, pcidma_deps, NULL);

static int __init __udif_init_pcidma(void)
{
	UDIF_DRIVER *drv = &udif_pcidma;
	return udif_driver_register(&drv, 1);
}
late_initcall(__udif_init_pcidma);
