/*
 * driver/misc/cxd/pciedma/proc.c
 *
 * Copyright 2022 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */

#include <linux/seq_file.h>
#include <linux/proc_fs.h>

#include "internal.h"

#define PROC_PCIE_FMT "driver/pcie/%u/dma"

#define to_chan(p)	((uintptr_t)(p) - 1)
#define chan_to(n)	((void *)((uintptr_t)((n) + 1)))

static void *pcidma_seq_start(struct seq_file *m, loff_t *pos)
{
	uint ch = (uint)*pos;

	if (ch >= PCIDMA_CH_MAX)
		return NULL;
	return chan_to(ch);
}

static void pcidma_seq_stop(struct seq_file *m, void *arg) {}

static void *pcidma_seq_next(struct seq_file *m, void *v, loff_t *pos)
{
	uint ch = to_chan(v);

	(*pos)++;
	ch++;
	if (ch >= PCIDMA_CH_MAX)
		return NULL;
	return chan_to(ch);
}

static const char *pcidma_show_state(uint32_t flag, int state, void *cur)
{
	if (!(flag & CHAN_FLAG_EN))
		return "N/A ";
	if (!state)
		return "----";
	return (cur) ? "busy" : "idle";
}

static int pcidma_seq_show(struct seq_file *m, void *v)
{
	uint ch = to_chan(v);
	dmac_t *dmac = (dmac_t *)m->private;
	chan_t *chan = pcidma_get_chan(dmac->id, ch);
	uint32_t ch_flag;
	uint32_t ch_state;
	req_t *req;
	uint dir;
	uint32_t pid;
	struct dma_stat stat;
	const char *state;
	unsigned long flags;

	if (!chan) {
		pcidma_perr("%s:illegal ch:%u\n", __func__, ch);
		return 0;
	}
	seq_printf(m, "%u ", ch);
	LOCK(chan);
	ch_flag = chan->flag;
	ch_state = chan->state;
	req = chan->cur;
	if (req) {
		dir = req->dir;
		pid = req->drv.pid;
	}
	stat = chan->stat;
	UNLOCK(chan);
	state = pcidma_show_state(ch_flag, ch_state, req);
	seq_printf(m, "%s %d %ld,%ld,%ld", state, stat.pending_cnt,
		   stat.abort_cnt, stat.go_cnt, stat.done_cnt);
	if (req) {
		seq_printf(m, ":%u,%c,%px", pid, (PCIDMA_RD==dir)?'R':'W', req);
	}
	seq_printf(m, "\n");
	return 0;
}

static struct seq_operations pcidma_seq_ops = {
	.start	= pcidma_seq_start,
	.next	= pcidma_seq_next,
	.stop	= pcidma_seq_stop,
	.show	= pcidma_seq_show,
};

static int pcidma_proc_open(struct inode *inode, struct file *file)
{
	int ret;

	ret = seq_open(file, &pcidma_seq_ops);
	if (!ret) {
		struct seq_file *m = file->private_data;
		m->private = PDE_DATA(inode);
	}
	return ret;
}

static const struct proc_ops pcidma_proc_ops = {
	.proc_open	= pcidma_proc_open,
	.proc_read_iter	= seq_read_iter,
	.proc_lseek	= seq_lseek,
	.proc_release	= seq_release,
};

void pcidma_proc_create(dmac_t *dmac)
{
	char path[80];

	scnprintf(path, sizeof path, PROC_PCIE_FMT, dmac->id);
	dmac->proc = proc_create_data(path, 0, NULL, &pcidma_proc_ops, dmac);
}

void pcidma_proc_remove(dmac_t *dmac)
{
	if (dmac->proc) {
		proc_remove(dmac->proc);
		dmac->proc = NULL;
	}
}

void pcidma_proc_init(void) {}
void pcidma_proc_exit(void) {}
