/*
 * driver/misc/cxd/wdt/arm_sp805.c
 *
 *
 * Copyright 2022 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include "wdt.h"
#include <linux/arm_sp805.h>
#include <linux/scu-clock.h>
#include <linux/scu-reset.h>

/* MISC register */
#define WDT_RST_OUT	0x1

static inline u32 sp805_read(struct wdt_dev *wdev, int reg)
{
	return readl_relaxed(wdev->va[0] + reg);
}

static inline void sp805_write_raw(struct wdt_dev *wdev, int reg, u32 val)
{
	writel_relaxed(val, wdev->va[0] + reg);
}

static inline void sp805_lock(struct wdt_dev *wdev)
{
	sp805_write_raw(wdev, WDT_LOCK, 0);
}

static inline void sp805_unlock(struct wdt_dev *wdev)
{
	sp805_write_raw(wdev, WDT_LOCK, LOCK_UNLOCK);
}

static inline void sp805_write_lock(struct wdt_dev *wdev, int reg, u32 val)
{
	sp805_unlock(wdev);
	sp805_write_raw(wdev, reg, val);
	sp805_lock(wdev);
}

static inline void sp805_misc_write(struct wdt_dev *wdev, u32 val)
{
	writel_relaxed(val, wdev->va[UDIF_WDT_MISC]);
}

/*------------- XRESET_REQ output control -------------*/
#define CONNECT     1
#define DISCONNECT  0
static void xrstreq(struct wdt_dev *wdev, int connect)
{
	sp805_misc_write(wdev, (connect) ? WDT_RST_OUT : 0);
}

/*---------------------- timeout ----------------------*/
#define WDT_MAX_MSEC		(U32_MAX / WDT_CLK)

static unsigned int sp805_msec_to_tmo(unsigned int msec)
{
	if (msec > WDT_MAX_MSEC) {
		return U32_MAX;
	}
	return msec * WDT_CLK;
}

static unsigned int sp805_cnt_to_msec(unsigned int cnt)
{
	return cnt / WDT_CLK;
}

/*-------- clear interrupt and reload counter -----*/
static inline void sp805_intclr(struct wdt_dev *wdev)
{
	sp805_write_lock(wdev, WDT_INTCLR, 0);
}

/*-------------------- WDT API --------------------*/
static void sp805_stop(struct wdt_dev *wdev)
{
	sp805_write_lock(wdev, WDT_CTRL, 0);
}

static void sp805_restart(struct wdt_dev *wdev)
{
	/* reload counter */
	sp805_intclr(wdev);
}

static void sp805_start(struct wdt_dev *wdev, unsigned int timeout, int mode)
{
	unsigned int tmo = sp805_msec_to_tmo(timeout);
	u32 ctrl = CTRL_INTEN;

	if (WDT_MODE_RESET == mode)
		ctrl |= CTRL_RESEN;

	sp805_unlock(wdev);
	sp805_write_raw(wdev, WDT_CTRL, 0);
	sp805_write_raw(wdev, WDT_INTCLR, 0);
	sp805_write_raw(wdev, WDT_LOAD, tmo);
	sp805_write_raw(wdev, WDT_CTRL, ctrl);
	sp805_lock(wdev);
	wmb();

	xrstreq(wdev, CONNECT);
}

static unsigned int sp805_remain(struct wdt_dev *wdev)
{
	unsigned int cnt;

	cnt = sp805_read(wdev, WDT_VAL);
	return sp805_cnt_to_msec(cnt);
}

static int sp805_interrupt(struct wdt_dev *wdev)
{
	u32 stat;

	stat = sp805_read(wdev, WDT_MIS);
	if (!(stat & INTSTAT))
		return 0;

	sp805_intclr(wdev);
	return 1;
}

static struct wdt_ops sp805_ops = {
	.start	   = sp805_start,
	.stop	   = sp805_stop,
	.restart   = sp805_restart,
	.remain    = sp805_remain,
	.interrupt = sp805_interrupt,
};

struct wdt_ops *wdt_driver_init(void)
{
	return &sp805_ops;
}
