/*
 * driver/misc/cxd/wdt/dw_apb_wdt.c
 *
 * DW_apb_wdt driver
 *
 * Copyright 2021 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include "wdt.h"

#define WDT_CLK		4000 /* KHz */
#define WDT_CFG		(WDT_RPL256)
/* WDT_MISC register */
#define WDT_RST_OUT	0x1

/* WDT registers */
#define WDT_CR		0x00
# define WDT_EN		 0x1
# define WDT_RMOD_INT	 0x2
# define WDT_RPL256	 (0x7U << 2)
#define WDT_TORR	0x04
# define WDT_TOMASK	 0xf
# define WDT_TO_MIN	 16
# define WDT_TO_MAX	 31
#define WDT_CCVR	0x08
#define WDT_CRR		0x0c
# define WDT_RESTART	 0x76
#define WDT_STAT	0x10
# define WDT_INTSTAT	 0x1
#define WDT_EOI		0x14

static inline u32 dw_wdt_read(struct wdt_dev *wdev, int reg)
{
	return readl_relaxed(wdev->va[UDIF_WDT_REG] + reg);
}

static inline void dw_wdt_write(struct wdt_dev *wdev, int reg, u32 val)
{
	writel_relaxed(val, wdev->va[UDIF_WDT_REG] + reg);
}

static inline void dw_wdt_misc_write(struct wdt_dev *wdev, u32 val)
{
	writel_relaxed(val, wdev->va[UDIF_WDT_MISC]);
}

/*------------- XRESET_REQ output control -------------*/
#define CONNECT     1
#define DISCONNECT  0
static void xrstreq(struct wdt_dev *wdev, int conn)
{
	dw_wdt_misc_write(wdev, (conn) ? WDT_RST_OUT : 0);
}

/*---------------------- timeout ----------------------*/
#define WDT_MAX_MSEC		((1U << WDT_TO_MAX) / WDT_CLK)

static unsigned int dw_wdt_msec_to_tmo(unsigned int msec)
{
	unsigned int count;
	int i;

	if (msec > WDT_MAX_MSEC) {
		msec = WDT_MAX_MSEC;
	}
	count = WDT_CLK * msec; /* 0..2^31 */
	i = fls(count); /* 0..32 */
	if (i > 0) {
		if ((1U << (i - 1)) == count) /* 2^n ? */
			i--;

		i -= WDT_TO_MIN;
		if (i < 0)
			i = 0;
	}
	return i;
}

static unsigned int dw_wdt_tmo_to_msec(unsigned int tmo)
{
	return (1U << (WDT_TO_MIN + tmo)) / WDT_CLK;
}

/*-------------------- WDT API --------------------*/
static void dw_wdt_stop(struct wdt_dev *wdev)
{
	xrstreq(wdev, DISCONNECT);
}

static void dw_wdt_restart(struct wdt_dev *wdev)
{
	/* counter restart and clear wdt_sys_rst */
	dw_wdt_write(wdev, WDT_CRR, WDT_RESTART);
}

static void dw_wdt_start(struct wdt_dev *wdev, unsigned int timeout, int mode)
{
	unsigned int tmo = dw_wdt_msec_to_tmo(timeout);

	if (WDT_MODE_NORESET == mode)
		xrstreq(wdev, DISCONNECT);

	dw_wdt_write(wdev, WDT_TORR, tmo);
	dw_wdt_restart(wdev);
	dw_wdt_write(wdev, WDT_CR,
		  WDT_CFG|WDT_EN|((WDT_MODE_NORESET==mode)?WDT_RMOD_INT:0));
	wmb();

	if (WDT_MODE_RESET == mode)
		xrstreq(wdev, CONNECT);
}

static unsigned int dw_wdt_remain(struct wdt_dev *wdev)
{
	unsigned int count;

	count = dw_wdt_read(wdev, WDT_CCVR);
	return count / WDT_CLK; /* msec */
}

static int dw_wdt_interrupt(struct wdt_dev *wdev)
{
	u32 stat;

	stat = dw_wdt_read(wdev, WDT_STAT);
	if (!(stat & WDT_INTSTAT))
		return 0;

	stat = dw_wdt_read(wdev, WDT_EOI); /* clear intr */
	return 1;
}

static struct wdt_ops dw_wdt_ops = {
	.start	   = dw_wdt_start,
	.stop	   = dw_wdt_stop,
	.restart   = dw_wdt_restart,
	.remain    = dw_wdt_remain,
	.interrupt = dw_wdt_interrupt,
};

struct wdt_ops *wdt_driver_init(void)
{
	return &dw_wdt_ops;
}
