/*
 * driver/misc/cxd/wdt.c
 *
 * Watchdog timer driver
 *
 * Copyright 2022 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/module.h>
#include <linux/udif/module.h>
#include <linux/syscore_ops.h>
#include <linux/proc_fs.h>
#include <linux/snsc_boot_time.h>
#include <mach/errind.h>
#include <mach/gic_export.h>
#include <linux/wdt.h>
#include "wdt.h"

/* WDT mode */
#define WDT_OFF  0
#define WDT_ON   1
static unsigned int wdt_mode = WDT_OFF;
module_param_named(mode, wdt_mode, uint, S_IRUSR);

static int wdt_debug = 0;
module_param_named(debug, wdt_debug, int, S_IRUSR|S_IWUSR);

static unsigned int wdt_timeout = 0; /* msec */
module_param_named(timeout, wdt_timeout, uint, S_IRUSR);

/* wdt time extend to dump in msec */
#define WDT_EXTEDED_TIME_EMDUMP         (2000)
static unsigned int wdt_timeout_mp_excpt = WDT_EXTEDED_TIME_EMDUMP;
module_param_named(mp_excpt, wdt_timeout_mp_excpt, uint, S_IRUSR);

static void (*wdt_callback)(void) = NULL;

struct wdt_info {
	struct wdt_dev wdev;
	const UDIF_DEVICE *dev;
	UDIF_CH ch;
	UDIF_IRQ irq;
	int state;
	struct wdt_ops *ops;
};
static struct wdt_info wdt;

static inline void wdt_state_init(struct wdt_info *w)
{
	w->state = UDIF_NR_WDT;
	smp_mb();
}

static inline int wdt_ready(void)
{
	struct wdt_info *w = &wdt;

	smp_mb();
	return !w->state;
}

static void wdt_enable_irq(struct wdt_info *w)
{
	if (!w->irq)
		return;
	udif_maskirq(w->dev, w->ch, 1);
}

static void wdt_disable_irq(struct wdt_info *w)
{
	if (!w->irq)
		return;
	udif_maskirq(w->dev, w->ch, 0);
}

static void wdt_clear_irq(struct wdt_info *w)
{
	if (!w->irq)
		return;
	udif_clearirq(w->dev, w->ch);
}


/*----- wrapper -----*/
static void wdt_start(unsigned timeout, int mode)
{
	struct wdt_info *w = &wdt;

	if (w->ops && w->ops->start)
		w->ops->start(&w->wdev, timeout, mode);
}

static void wdt_stop(void)
{
	struct wdt_info *w = &wdt;

	if (w->ops && w->ops->stop)
		w->ops->stop(&w->wdev);
}

static void wdt_restart(void)
{
	struct wdt_info *w = &wdt;

	if (w->ops && w->ops->restart)
		w->ops->restart(&w->wdev);
}

static unsigned int wdt_remain(void)
{
	struct wdt_info *w = &wdt;

	if (w->ops && w->ops->remain)
		return w->ops->remain(&w->wdev);
	return 0;
}

static int wdt_interrupt(void)
{
	struct wdt_info *w = &wdt;

	if (w->ops && w->ops->interrupt)
		return w->ops->interrupt(&w->wdev);
	return 0;
}

/*
 * Kernel API
 */
void watchdog_start(void)
{
	if (WDT_OFF == wdt_mode)
		return;

	if (!wdt_ready())
		return;
	wdt_start(wdt_timeout, WDT_MODE_RESET);
}
EXPORT_SYMBOL(watchdog_start);

void watchdog_stop(void)
{
	if (WDT_OFF == wdt_mode)
		return;

	if (!wdt_ready())
		return;
	wdt_stop();
}
EXPORT_SYMBOL(watchdog_stop);

void watchdog_touch(void)
{
	if (WDT_OFF == wdt_mode)
		return;

	if (!wdt_ready())
		return;
	if (wdt_debug) {
		unsigned int remain;
		char buf[64];

		remain = wdt_remain();
		scnprintf(buf, sizeof buf, "WDT:%u [ms]", remain);
		BOOT_TIME_ADD1(buf);
	}

	wdt_restart();
}
EXPORT_SYMBOL(watchdog_touch);

/*
 * Touches watchdog from exception.c
 */
static DEFINE_RAW_SPINLOCK(wdte_lock);
static volatile int s_except = 0;

int watchdog_touch_exception(void)
{
	unsigned long flags;
	char msg[64];
	int cpu;
	unsigned int tmo;

	if (WDT_OFF == wdt_mode)
		return 0;

	if (!wdt_ready())
		return 0;

	if (s_except)
		return 1;

	raw_spin_lock_irqsave(&wdte_lock, flags);

	/* touch before calculation */
	watchdog_touch();

	/* modify wdt_timeout */
	tmo = wdt_timeout;
	if (wdt_timeout_mp_excpt > wdt_timeout) {
		tmo = wdt_timeout_mp_excpt;
	}
	wdt_start(tmo, WDT_MODE_RESET);
	/* log some info */
	printk(KERN_ERR "WDT:new_timeout=%u\n", tmo);
	cpu = smp_processor_id();
	scnprintf(msg, sizeof(msg), "watchdog extended on cpu%d", cpu);
	BOOT_TIME_ADD1(msg);
	/* we'd like to prevent making it chain */
	s_except = 1;

	raw_spin_unlock_irqrestore(&wdte_lock, flags);

	return 1;
}

void watchdog_reg_callback(void (*p)(void))
{
	wdt_callback = p;
}
EXPORT_SYMBOL(watchdog_reg_callback);


/*-------------------- WDT Driver -------------------- */
#define LOCK_ENABLE     1
#define LOCK_DISABLE    0

static DEFINE_SPINLOCK(wdt_lock);
static unsigned char wdt_lock_status = LOCK_DISABLE;

static int wdt_proc_open(struct inode *node, struct file *fp)
{
	if (!spin_trylock(&wdt_lock)) {
		printk(KERN_ERR "[WDT][OPEN1] WDT is Busy.\n");
		return -EBUSY;
	}

	if (wdt_lock_status & LOCK_ENABLE) {
		spin_unlock(&wdt_lock);
		printk(KERN_ERR "[WDT][OPEN2] WDT is Busy.\n");
		return -EBUSY;
	}

	fp->private_data = PDE_DATA(node);
	wdt_lock_status = LOCK_ENABLE;
	spin_unlock(&wdt_lock);

	return 0;
}

static int wdt_proc_release(struct inode *node, struct file *fp)
{
	wdt_lock_status = LOCK_DISABLE;
	smp_mb();

	return 0;
}

static long wdt_proc_ioctl(struct file *fp, unsigned int cmd, unsigned long arg)
{
	long ret = 0;
	unsigned int usec, msec;
	struct wdt_info *w = (struct wdt_info *)fp->private_data;

	printk(KERN_INFO "[INFO][%s] ioctl: cmd=%08X, arg=%u\n", __func__, cmd, (unsigned int)arg);
	switch(cmd) {
	case WDT_SET_STOP:
		/* WDT Reset Request disable. */
		wdt_disable_irq(w);
		wdt_stop();
		break;
	case WDT_SET_START:
		/* WDT Reset Request enable. */
		usec = arg;
		msec = usec/USEC_PER_MSEC;
		if (wdt_debug) {
			printk(KERN_INFO "[INFO][%s] tmout=%u[ms]\n",
			       __func__, msec);
		}
		wdt_disable_irq(w);
		wdt_start(msec, WDT_MODE_RESET);
		break;
	case WDT_SET_DEBUG:
		usec = arg;
		msec = usec/USEC_PER_MSEC;
		if (wdt_debug) {
			printk(KERN_INFO "[INFO][%s] tmout=%u[ms]\n",
			       __func__, msec);
		}
		wdt_stop();
		wdt_clear_irq(w);
		wdt_start(msec, WDT_MODE_NORESET);
		wdt_enable_irq(w);
		break;
	default:
		printk(KERN_ERR "[ERR][%s] cmd error. cmd=%08X\n",
		       __func__, cmd);
		ret = -1;
		break;
	}

	return ret;
}

static UDIF_ERR emg_intr(UDIF_INTR *hndl)
{
	struct wdt_info *w = udif_intr_data(hndl);

	if (!wdt_interrupt())
		return UDIF_ERR_OK;

	udif_disable_irq_nosync(w->irq); /* single shot */

	printk(KERN_ERR "START Emergency!\n");
	BOOT_TIME_ADD1("Emergency interrupt!!");
	if (wdt_callback) {
		wdt_callback();
	}
	errind_start(ERRIND_EMG);

	return UDIF_ERR_OK;
}

static const UDIF_DEVICE *wdtdev;

#ifdef CONFIG_PM
static int wdt_suspend(void)
{
	if (WDT_OFF == wdt_mode) {
		udif_disable_irq(udif_devint_irq(wdtdev, UDIF_WDT_REG));
	}
	return 0;
}

static void wdt_resume(void)
{
	udif_devio_hclk(wdtdev, 0, 1);
	udif_devio_devclk(wdtdev, 0, 1, 0);
	udif_devio_hrst(wdtdev, 0, 0);
	udif_devio_devrst(wdtdev, 0, 0);

	if (WDT_OFF == wdt_mode) {
		udif_enable_irq(udif_devint_irq(wdtdev, UDIF_WDT_REG));
		return;
	}

	watchdog_start();
}

static struct syscore_ops wdt_syscore_ops = {
	.suspend	= wdt_suspend,
	.resume		= wdt_resume,
};
#endif /* CONFIG_PM */

static const struct proc_ops wdt_proc_ops = {
	.proc_open	= wdt_proc_open,
	.proc_ioctl	= wdt_proc_ioctl,
	.proc_release	= wdt_proc_release,
};

__weak struct wdt_ops *wdt_driver_init(void){ return NULL; }

static UDIF_ERR wdt_init(UDIF_VP data)
{
	struct wdt_info *w = (struct wdt_info *)data;

	wdt_state_init(w);
	wdt.ops = wdt_driver_init();

	if (!proc_create_data(WDT_NAME, S_IRUSR|S_IWUSR, NULL, &wdt_proc_ops, w)) {
		printk(KERN_ERR "WDT:ERROR:%s:proc_create_data\n", __func__);
	}

#ifdef CONFIG_PM
	register_syscore_ops(&wdt_syscore_ops);
#endif

	return UDIF_ERR_OK;
}

static UDIF_ERR wdt_exit(UDIF_VP data)
{
	remove_proc_entry(WDT_NAME, NULL);
	return UDIF_ERR_OK;
}

static UDIF_ERR wdt_probe(const UDIF_DEVICE *dev, UDIF_CH i, UDIF_VP data)
{
	struct wdt_info *w = (struct wdt_info *)data;

	w->wdev.va[i] = udif_devio_virt(dev, i);
	w->state--;

	udif_devio_hclk(dev, i, 1);
	udif_devio_devclk(dev, i, 1, 0);
	udif_devio_hrst(dev, i, 0);
	udif_devio_devrst(dev, i, 0);

	if (WDT_OFF != wdt_mode) {
		wdtdev = dev;
		return UDIF_ERR_OK;
	}

	/* EMG mode (WDT_OFF) */
	if (UDIF_WDT_REG == i) {
		/* setup interrupt handler */
		w->dev = dev;
		w->ch = i;
		w->irq = udif_devint_irq(dev, i);
		if (udif_request_irq(dev, i, emg_intr, w)) {
			w->irq = 0;
			printk(KERN_ERR "WDT:ERROR:%s:request_irq %d failed.\n", __func__, w->irq);
		}
	}
	return UDIF_ERR_OK;
}

static UDIF_ERR wdt_remove(const UDIF_DEVICE *dev, UDIF_CH i, UDIF_VP data)
{
	struct wdt_info *w = (struct wdt_info *)data;

	wdt_state_init(w);

	udif_devio_devrst(dev, i, 1);
	udif_devio_hrst(dev, i, 1);
	udif_devio_devclk(dev, i, 0, 0);
	udif_devio_hclk(dev, i, 0);

	if (WDT_OFF != wdt_mode) {
		return UDIF_ERR_OK;
	}

	if (UDIF_WDT_REG == i && w->irq) {
		udif_free_irq(dev, i);
		w->irq = 0;
	}

	return UDIF_ERR_OK;
}

static UDIF_DRIVER_OPS wdt_ops = {
	.init		= wdt_init,
	.exit		= wdt_exit,
	.probe		= wdt_probe,
	.remove		= wdt_remove,
};

UDIF_IDS(wdt_devs) = {
	UDIF_ID(UDIF_ID_WDT, UDIF_CH_MASK_DEFAULT),
};
UDIF_DEPS(wdt_deps) = {};

UDIF_MODULE(wdt, "wdt", "3.0", wdt_ops, wdt_devs, wdt_deps, &wdt);
