// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright 2022 Sony Corporation, SOCIONEXT INC.
 */
/*
 * Portions Copyright (C) 2024 Synopsys, Inc.  Used with permission. All rights reserved.
 */

#include <linux/clk.h>
#include <linux/iopoll.h>
#include <linux/device.h>
#include <linux/reset.h>
#include <linux/irqdomain.h>
#include <linux/irqchip/chained_irq.h>
#include <linux/platform_device.h>
#include <linux/of_device.h>
#include <linux/pci.h>
#include <linux/pcie_dwc.h>
#include <linux/workqueue.h>
#include <linux/gpio/gpio.h>

#include "../../pci.h"
#include "../../pcie/portdrv.h"
#include "pcie-designware.h"


//#undef CONFIG_ARCH_CXD90XXX_FPGA















//#define SAMPLECODE_PME

void __weak pcie_log(const char *fmt, ...){}

#define PR_ERR(fmt, ...) \
		pr_err("ERROR : %s %s %d : " fmt "\n", __FILE__, __func__, __LINE__, \
				## __VA_ARGS__)

#define PR_DEBUG(fmt, ...) \
		pr_debug("DEBUG : %s %s %d : " fmt "\n", __FILE__, __func__, __LINE__, \
				## __VA_ARGS__)

#define DEV_ERR(fmt, ...) \
		dev_err(dev, "ERROR : %s %s %d : " fmt "\n", __FILE__, __func__, __LINE__, \
				## __VA_ARGS__)

#define PR_NOTICE(fmt, ...) \
		pr_notice("%s %s %d : " fmt "\n", __FILE__, __func__, __LINE__, ##__VA_ARGS__)

#define IK_PCIE_EN				0x0000
#define IK_PCIE_EN_LTSSM_ENABLE				0x00000001
#define IK_PCIE_EN_LTSSM_DISABLE			0x00000000
#define IK_PCIE_TYPE				0x0004
#define IK_PCIE_TYPE_RC					0x00000004
#define IK_PCIE_TYPE_EP					0x00000000
#define IK_PCIE_EN_CLR_SEL			0x000c
#define IK_PCIE_EN_CLR_SEL_LTSSM_EN_CLR_SEL		BIT(0)
#define IK_PCIE_RST_CNTL			0x0010
#define IK_PCIE_RST_CNTL_RST_PEX_O			BIT(16)
#define IK_PCIE_RST_CNTL_RST_PEX			BIT(1)
#define IK_PCIE_RST_CNTL_RST_PONX			BIT(0)
#define IK_PCIE_RST_CNTL_SET			0x0014
#define IK_PCIE_RST_CNTL_CLR			0x0018
#define IK_RST_STS				0x0020
#define IK_RST_STS_PERST_N_EXT				BIT(2)
#define IK_RST_REQ_STS				0x0024
#define IK_RST_REQ_STS_SMLH_REQ_RST			BIT(4)
#define IK_PCIE_DL_STS				0x002c
#define IK_PCIE_DL_STS_DATA_LINKUP_STS			BIT(16)
#define IK_LTSSM_STS				0x0030
#define IK_LTSSM_STS_STATE				GENMASK(5, 0)
#define IK_LTSSM_STS_STATE_L2I				0x15
#define IK_LTSSM_STS_STATE_HRESET			0x1f
#define IK_IRQ_CNCT_STS				0x0040
#define IK_IRQ_CNCT_CLR				0x0044
#define IK_IRQ_CNCT_EN				0x0048
#define IK_IRQ_CNCT_RAW				0x004c
#define IK_IRQ_CNCT_PERSTX_NEG				BIT(0)
#define IK_IRQ_CNCT_PERSTX_POS				BIT(1)
#define IK_IRQ_CNCT_DL_UP_NEG				BIT(12)
#define IK_IRQ_CNCT_DL_UP_POS				BIT(13)
#define IK_IRQ_CNCT_MSTR_EN_POS				BIT(17)
#define IK_IRQ_COM_STS				0x0050
#define IK_IRQ_COM_CLR				0x0054
#define IK_IRQ_COM_EN				0x0058
#define IK_IRQ_COM_RAW				0x005c
#define IK_IRQ_COM_PME_MSI_POS				BIT(1)
#define IK_IRQ_COM_TURNOFF_ACK				BIT(8)
#define IK_IRQ_COM_TURNOFF_REQ				BIT(9)
#define IK_IRQ_COM_INTA_POS_STS				BIT(25)
#define IK_IRQ_COM_INTB_POS_STS				BIT(27)
#define IK_IRQ_COM_INTC_POS_STS				BIT(29)
#define IK_IRQ_COM_INTD_POS_STS				BIT(31)
#define IK_IRQ_ERR_STS				0x0060
#define IK_IRQ_ERR_CLR				0x0064
#define IK_IRQ_ERR_EN				0x0068
#define IK_IRQ_ERR_RC_MSI				BIT(25)
#define IK_MSG_INTX_REQ				0x0100
#define IK_MSG_INTX_REQ_MSG_INTX_REQ			BIT(0)
#define IK_MSG_INTX_CLR				0x0104
#define IK_MSG_INTX_CLR_MSG_INTX_CLR			BIT(0)
#define IK_MSG_INTX_SEL				0x010c
#define IK_MSG_INTX_SEL_MSG_INTX_SEL			BIT(0)
#define IK_MSG_TURNOFF_REQ			0x0120
#define IK_MSG_TURNOFF_CLR			0x0124
#define IK_MSG_TURNOFF_FUNC0				BIT(0)
#define IK_MSG_ENTRY_L23_REQ			0x0130
#define IK_MSG_ENTRY_L23_CLR			0x0134
#define IK_MSG_ENTRY_L23_FUNC0				BIT(0)
#define IK_PCIE_CONF0				0x0f00
#define IK_PCIE_CONF0_DMACH_WR				GENMASK(7, 4)
#define IK_PCIE_CONF0_DMACH_WR_SHIFT			4
#define IK_PCIE_CONF0_DMACH_RD				GENMASK(11, 8)
#define IK_PCIE_CONF0_DMACH_RD_SHIFT			8
#define IK_PCIE_DBG				0x0ff0
#define IK_PCIE_DBG_DBI_XFER_PEND			BIT(8)
#define IK_PCIE_DBG_XFER_PEND				(BIT(0)|BIT(4)|BIT(8)|BIT(12))

#define IK_PHY_RST_CTRL				0x2000
#define IK_PHY_RST_CTRL_PHY_RSTX			BIT(0)
#define IK_PHY_PIPE_RST_CTRL			0x2004
#define IK_PHY_PIPE_RST_CTRL_PHY_PIPE_RSTX		BIT(0)
#define IK_PHY_SRAM_CTRL			0x2080
#define IK_PHY_SRAM_CTRL_PHY_SRAM_BYPASS		BIT(0)
#define IK_PHY_SRAM_CTRL_PHY_SRAM_EXT_LD_DONE		BIT(4)
#define IK_PHY_SRAM_CTRL_PHY_SRAM_INIT_DONE		BIT(16)
#define IK_PHY_OTHER_00				0x2300
#define IK_PHY_OTHER_00_EQ_AFE_X1_MASK			GENMASK(9, 5)
#define IK_PHY_OTHER_00_EQ_AFE_X1_VAL			0x000001a0
#define IK_PHY_OTHER_00_EQ_AFE_X2_MASK			GENMASK(14, 5)
#define IK_PHY_OTHER_00_EQ_AFE_X2_VAL			0x000035a0
#define IK_PHY_OTHER_00_EQ_AFE_X4_MASK			GENMASK(24, 5)
#define IK_PHY_OTHER_00_EQ_AFE_X4_VAL			0x00d6b5a0
#define IK_PHY_OTHER_02				0x2308
#define IK_PHY_OTHER_02_REF_USE_PAD			BIT(16)

#define LIN_PCIE_PHY_SUP_DIG_LVL_OVRD_IN	(0x20 * 4)
#define LIN_PCIE_PHY_TX_VBOOST_LVL_MASK			GENMASK(8, 6)
#define LIN_PCIE_PHY_TX_VBOOST_LVL_SHIFT		6
#define LIN_PCIE_PHY_TX_VBOOST_LVL_OVRD_EN		BIT(9)
#define LIN_PCIE_PHY_SRAM			0x30000
#define LIN_PCIE_PHY_SRAM_SIZE			0x10000

#define HDMA_EN				0x0000
#define HDMA_EN_ENABLE				0x1
#define HDMA_DOORBELL			0x0004
#define HDMA_DOORBELL_START			0x1
#define HDMA_DOORBELL_STOP			0x2
#define HDMA_LLP_LOW			0x0010
#define HDMA_LLP_HIGH			0x0014
#define HDMA_CYCLE			0x0018
#define HDMA_CYCLE_INIT				0x0
#define HDMA_XFERSIZE			0x001c
#define HDMA_SAR_LOW			0x0020
#define HDMA_SAR_HIGH			0x0024
#define HDMA_DAR_LOW			0x0028
#define HDMA_DAR_HIGH			0x002c
#define HDMA_CONTROL1			0x0034
#define HDMA_CONTROL1_SINGLE			0x0
#define HDMA_CONTROL1_LL			0x1
#define HDMA_INT_STATUS			0x0084
#define HDMA_INT_STATUS_KIND			0x07
#define HDMA_INT_STATUS_STOP			0x01
#define HDMA_INT_SETUP			0x0088
#define HDMA_INT_SETUP_NORMAL			0x0052
#define HDMA_INT_SETUP_DISABLE			0x0007
#define HDMA_INT_CLEAR			0x008c
#define HDMA_INT_CLEAR_ALL			0x7

#define HDMA_RD_OFST			0x100
#define HDMA_CH_OFST			0x200
#define HDMA_ERR_FLAG_POS			63

#define LIN_PCIE_EP_PERST_TIMEOUT		1000000
#define LIN_PCIE_PHY_DEASSERT_TIMEOUT		170
#define LIN_PCIE_PIPE_DEASSERT_TIMEOUT		14000
#define LIN_PCIE_L23_TIMEOUT			100000
#define LIN_PCIE_HRESET_TIMEOUT			1000000
#define LIN_PCIE_LTSSM_TIMEOUT			200000
#define LIN_PCIE_DBI_XFER_PEND_TIMEOUT		100
#define LIN_PCIE_DEAD_SLOT_TIMEOUT		100
#define LIN_PCIE_XFER_PEND_TIMEOUT		1000
#define LIN_PCIE_TURNOFF_ACK_TIMEOUT		10000
#define LIN_PCIE_GEN4_TRAININGS			3
#define LIN_PCIE_GEN4_LINK_TIMEOUT		100000

#define LIN_PCIE_MSI_IRQS			256

#define PCI_EXP_SLTCAP_PSN_LSB		19

#define LIN_PCIE_PCI32_SIZE		0xf8000000

#define to_lin_priv_from_pcie(x)		dev_get_drvdata((x)->dev)

enum lin_pcie_clocks {
	LIN_PCIE_CLK_M_ACLK,
	LIN_PCIE_CLK_S_ACLK,
	LIN_PCIE_CLK_DBI_ACLK,
	LIN_PCIE_CLK_AUX_CLK,
	LIN_PCIE_CLK_HCLK,
	LIN_PCIE_CLK_P_CR_CLK,
	LIN_PCIE_CLK_PCLK,
	LIN_PCIE_CLK_NUM
};

enum lin_pcie_rc_state {
	LIN_PCIE_STATE_INIT,
	LIN_PCIE_STATE_STOP,
	LIN_PCIE_STATE_ERR,
	LIN_PCIE_STATE_RC_PREP,
	LIN_PCIE_STATE_RC_START,
	LIN_PCIE_STATE_RC_SUSPENDING,
	LIN_PCIE_STATE_EP_START,
	LIN_PCIE_STATE_EP_STOP,
	LIN_PCIE_STATE_EP_SUSPEND,
	LIN_PCIE_STATE_NUM,
};

struct lin_pcie_dma_ch {
	struct lin_pcie_priv	*priv;
	u8	ch;
	int	wirq;
	int	rirq;
	bool	wch;
	bool	used;
	void	(*cb)(uint64_t err, void *cookies);
	void	*cookies;
};

struct lin_pcie_dma_ch_info {
	raw_spinlock_t spin;
	struct lin_pcie_dma_ch *dma_ch;
};
static struct lin_pcie_dma_ch_info lin_pcie_dma_ch_info[PCIE_CH_MAX][PCIDMA_CH_MAX];

struct lin_pcie_clkreq {
	unsigned int mode;
	u8	gpio_ch;
	u8	gpio_bit;
};

struct lin_pcie_priv {
	struct dw_pcie pci;
	void __iomem *link_base;
	void __iomem *dma_base;
	struct clk_bulk_data clks[LIN_PCIE_CLK_NUM];
	struct reset_control *rst;
	struct irq_domain *lgc_irqd;
	u32 ch;
	enum dw_pcie_device_mode mode;
	int cnct_irq;
	int com_irq;
	int err_irq;
	int aer_irq;
	int pme_irq;
	u16 pcie_ofst;
	u16 spci_ofst;

	u32 dma_avail_ch;
	int dma_nums;
	struct lin_pcie_dma_ch dma_ch[PCIDMA_CH_MAX];

	u32 phy_fw_id;
	bool phy_fw_upd;
	void __iomem *phy_base;

	struct lin_pcie_clkreq clkreq;

	bool rc_clks_enabled;
	bool rc_rst_deasserted;
	bool rc_irq_built;
	bool rc_host_inited;
	bool rc_irq_enabled;
	bool rc_clkreq_param_inited;

	bool omission_of_rc_probe;
	bool omission_of_rc_start;
	bool omission_of_rc_suspend;
	bool pci32;

	struct work_struct update_ep;

	bool ep_wq_inited;
	bool ep_clks_enabled;
	bool ep_rst_deasserted;
	bool ep_hw_set;
	bool ep_inited;
	bool ep_irq_enabled;
};

struct lin_pcie_ch_info {
	struct mutex mutex;
	raw_spinlock_t spin;
	raw_spinlock_t config_spin;
	enum lin_pcie_rc_state state;
	struct lin_pcie_priv *priv;
	enum dw_pcie_device_mode mode;
	struct device *dev;
	bool start_istr;
	struct task_struct *owner;
	unsigned int clkreq_mode;
	bool clkreq_ctrl_avail;



};
static struct lin_pcie_ch_info lin_pcie_ch_info[PCIE_CH_MAX];

static int lin_pcie_start_rc(struct lin_pcie_priv *priv);
static void lin_pcie_term_rc(struct lin_pcie_priv *priv);

static const char * const dma_wirq_name[] = {
	"dmaw0", "dmaw1", "dmaw2", "dmaw3", "dmaw4", "dmaw5", "dmaw6", "dmaw7"
};

static const char * const dma_rirq_name[] = {
	"dmar0", "dmar1", "dmar2", "dmar3", "dmar4", "dmar5", "dmar6", "dmar7"
};

struct firm_info {
	const u16 *fw;
	int n;
};













































int lin_pcie_read(struct pci_bus *bus, unsigned int devfn, int where, int size, u32 *val)
{
	struct pcie_port *pp;
	struct dw_pcie *pci;
	unsigned long flags;
	int ret = -1;
	void __iomem *config;





	if (!bus) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if (pci_is_root_bus(bus) && devfn) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if (devfn >= 0x100) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if ((where >= 0x1000) || (where < 0)) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if ((size != 1) && (size != 2) && (size != 4)) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if (pci_is_root_bus(bus)) {
		config = dw_pcie_own_conf_map_bus(bus, devfn, where);
		if (!config) {
			*val = ~0;
			PR_ERR();
			return PCIBIOS_DEVICE_NOT_FOUND;
		}

		ret = dw_pcie_read(config, size, val);
		if (ret) {
			PR_ERR();
			return ret;
		}

		return ret;
	}

	pp = bus->sysdata;
	pci = to_dw_pcie_from_pp(pp);

	if (!dw_pcie_link_up(pci))
		return -EINVAL;

	local_irq_save(flags);




	if (pcie_check_prsnt(bus->self)) {
		pcie_err_clr(bus);
		config = dw_pcie_other_conf_map_bus(bus, devfn, where);
		if (config) {
			ret = dw_pcie_read(config, size, val);
			if (!ret)
				ret = pcie_err_stat(bus);
#ifdef CONFIG_PCIE_CXD_WORKAROUND
			writel(0, config - where);
#endif
			if (ret)
				*val = ~0U;
		} else
			*val = ~0U;
	} else
		*val = ~0U;




	local_irq_restore(flags);





	if (ret)
		PR_ERR("%d", ret);

	return ret;
}




int lin_pcie_write(struct pci_bus *bus, unsigned int devfn, int where, int size, u32 val)
{
	struct pcie_port *pp;
	struct dw_pcie *pci;
	struct lin_pcie_priv *priv;
	unsigned long flags;
	unsigned long config_flags;
	int ret = -1;
	void __iomem *config;







	if (!bus) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if (pci_is_root_bus(bus) && devfn) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if (devfn >= 0x100) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if ((where >= 0x1000) || (where < 0)) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if ((size != 1) && (size != 2) && (size != 4)) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	pp = bus->sysdata;
	pci = to_dw_pcie_from_pp(pp);

	if (pci_is_root_bus(bus)) {
		priv = to_lin_priv_from_pcie(pci);

		config = dw_pcie_own_conf_map_bus(bus, devfn, where);
		if (!config) {
			PR_ERR();
			return PCIBIOS_DEVICE_NOT_FOUND;
		}

		raw_spin_lock_irqsave(&lin_pcie_ch_info[priv->ch].config_spin, config_flags);



		ret = dw_pcie_write(config, size, val);



		raw_spin_unlock_irqrestore(&lin_pcie_ch_info[priv->ch].config_spin, config_flags);




		if (ret) {
			PR_ERR();
			return ret;
		}

		return ret;
	}

	if (!dw_pcie_link_up(pci))
		return -EINVAL;

	local_irq_save(flags);



	if (pcie_check_prsnt(bus->self)) {
		pcie_err_clr(bus);
		config = dw_pcie_other_conf_map_bus(bus, devfn, where);
		if (config) {
			ret = dw_pcie_write(config, size, val);
			if (!ret) {
				ret = pcie_err_stat(bus);
				if (ret) {
					PR_ERR("%d %08x %04x %04x %08x %d %08x", ret,
							bus->domain_nr, bus->number, devfn,
							where, size, val);
				}
			} else {
				PR_ERR("%d %08x %04x %04x %08x %d %08x", ret, bus->domain_nr,
						bus->number, devfn, where, size, val);
			}
		} else
			PR_ERR("%08x %04x %04x %08x", bus->domain_nr, bus->number, devfn, where);
	}



	local_irq_restore(flags);





	return ret;
}




static int _pcie_dwc_set_atu(int pcie_ch, int atu_idx, u64 axi_addr, u64 pci_addr, u64 size)
{
	unsigned long flags;
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[pcie_ch];
	enum lin_pcie_rc_state state;
	struct lin_pcie_priv *priv;
	struct dw_pcie *pci;
	int ret;





	raw_spin_lock_irqsave(&ch_info->spin, flags);



	state = READ_ONCE(ch_info->state);
	priv = READ_ONCE(ch_info->priv);



	raw_spin_unlock_irqrestore(&ch_info->spin, flags);




	if ((state != LIN_PCIE_STATE_RC_START) && (state != LIN_PCIE_STATE_EP_START)) {
		ret = -ENODEV;
		PR_ERR("%d", ret);
		return ret;
	}

	pci = &priv->pci;

	if ((atu_idx < 0) || (atu_idx >= pci->num_ob_windows)) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	dw_pcie_disable_atu(pci, atu_idx, DW_PCIE_REGION_OUTBOUND);
	dw_pcie_prog_outbound_atu(pci, atu_idx, PCIE_ATU_TYPE_MEM, axi_addr, pci_addr,
			size);

	return 0;
}

int pcie_dwc_set_atu(int pcie_ch, int atu_idx, u64 axi_addr, u64 pci_addr, u64 size)
{
	int ret;

	if ((pcie_ch < 0) || (pcie_ch >= PCIE_CH_MAX)) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if (!size) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	mutex_lock(&lin_pcie_ch_info[pcie_ch].mutex);
	ret = _pcie_dwc_set_atu(pcie_ch, atu_idx, axi_addr, pci_addr, size);
	mutex_unlock(&lin_pcie_ch_info[pcie_ch].mutex);
	if (ret) {
		PR_ERR("%d", ret);
		return ret;
	}

	return 0;
}
EXPORT_SYMBOL(pcie_dwc_set_atu);

static int _pcie_dwc_get_atu(int pcie_ch, int atu_idx)
{
	unsigned long flags;
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[pcie_ch];
	enum lin_pcie_rc_state state;
	struct lin_pcie_priv *priv;
	struct dw_pcie *pci;
	int ret;
	u32 val;





	raw_spin_lock_irqsave(&ch_info->spin, flags);



	state = READ_ONCE(ch_info->state);
	priv = READ_ONCE(ch_info->priv);



	raw_spin_unlock_irqrestore(&ch_info->spin, flags);




	if ((state != LIN_PCIE_STATE_RC_START) && (state != LIN_PCIE_STATE_EP_START)) {
		ret = -ENODEV;
		PR_ERR("%d", ret);
		return ret;
	}

	pci = &priv->pci;

	if ((atu_idx < 0) || (atu_idx >= pci->num_ob_windows)) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	val = dw_pcie_readl_ob_unroll(pci, atu_idx, PCIE_ATU_UNR_REGION_CTRL2);
	if (val & PCIE_ATU_ENABLE)
		return 1;
	else
		return 0;
}

int pcie_dwc_get_atu(int pcie_ch, int atu_idx)
{
	int ret;

	if ((pcie_ch < 0) || (pcie_ch >= PCIE_CH_MAX)) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	mutex_lock(&lin_pcie_ch_info[pcie_ch].mutex);
	ret = _pcie_dwc_get_atu(pcie_ch, atu_idx);
	mutex_unlock(&lin_pcie_ch_info[pcie_ch].mutex);
	if (ret < 0) {
		PR_ERR("%d", ret);
		return ret;
	}

	return ret;
}
EXPORT_SYMBOL(pcie_dwc_get_atu);

static int _pcie_dwc_clr_atu(int pcie_ch, int atu_idx)
{
	unsigned long flags;
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[pcie_ch];
	enum lin_pcie_rc_state state;
	struct lin_pcie_priv *priv;
	struct dw_pcie *pci;
	int ret;





	raw_spin_lock_irqsave(&ch_info->spin, flags);



	state = READ_ONCE(ch_info->state);
	priv = READ_ONCE(ch_info->priv);



	raw_spin_unlock_irqrestore(&ch_info->spin, flags);




	if ((state != LIN_PCIE_STATE_RC_START) && (state != LIN_PCIE_STATE_EP_START)) {
		ret = -ENODEV;
		PR_ERR("%d", ret);
		return ret;
	}

	pci = &priv->pci;

	if ((atu_idx < 0) || (atu_idx >= pci->num_ob_windows)) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	dw_pcie_disable_atu(pci, atu_idx, DW_PCIE_REGION_OUTBOUND);

	return 0;
}

int pcie_dwc_clr_atu(int pcie_ch, int atu_idx)
{
	int ret;

	if ((pcie_ch < 0) || (pcie_ch >= PCIE_CH_MAX)) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	mutex_lock(&lin_pcie_ch_info[pcie_ch].mutex);
	ret = _pcie_dwc_clr_atu(pcie_ch, atu_idx);
	mutex_unlock(&lin_pcie_ch_info[pcie_ch].mutex);
	if (ret) {
		PR_ERR("%d", ret);
		return ret;
	}

	return 0;
}
EXPORT_SYMBOL(pcie_dwc_clr_atu);

static u32 dma_ch_reg_readl(struct lin_pcie_dma_ch *dma_ch, u32 where)
{
	return readl(dma_ch->priv->dma_base + where + dma_ch->ch * HDMA_CH_OFST +
			(dma_ch->wch ? 0 : HDMA_RD_OFST));
}

static void dma_ch_reg_writel(struct lin_pcie_dma_ch *dma_ch, u32 where, u32 val)
{
	writel(val, dma_ch->priv->dma_base + where + dma_ch->ch * HDMA_CH_OFST +
			(dma_ch->wch ? 0 : HDMA_RD_OFST));
}






static int _pcie_dwc_dma_start(struct lin_pcie_dma_ch *dma_ch, struct pcidma_param *req,
		void (*cb)(uint64_t err, void *cookies), void *cookies)
{
	int ret;

	if (dma_ch->used) {
		ret = -EBUSY;
		PR_ERR("%d", ret);
		return ret;
	}

	if (req->dir == PCIDMA_WR)
		dma_ch->wch = true;
	else
		dma_ch->wch = false;

	dma_ch->used = true;
	dma_ch->cb = cb;
	dma_ch->cookies = cookies;

	dma_ch_reg_writel(dma_ch, HDMA_EN, HDMA_EN_ENABLE);
	if (req->size) {
		dma_ch_reg_writel(dma_ch, HDMA_XFERSIZE, req->size);
		dma_ch_reg_writel(dma_ch, HDMA_SAR_LOW, lower_32_bits(req->src));
		dma_ch_reg_writel(dma_ch, HDMA_SAR_HIGH, upper_32_bits(req->src));
		dma_ch_reg_writel(dma_ch, HDMA_DAR_LOW, lower_32_bits(req->dst));
		dma_ch_reg_writel(dma_ch, HDMA_DAR_HIGH, upper_32_bits(req->dst));
		dma_ch_reg_writel(dma_ch, HDMA_CONTROL1, HDMA_CONTROL1_SINGLE);
	} else {
		dma_ch_reg_writel(dma_ch, HDMA_CYCLE, HDMA_CYCLE_INIT);
		dma_ch_reg_writel(dma_ch, HDMA_LLP_LOW, lower_32_bits(req->src));
		dma_ch_reg_writel(dma_ch, HDMA_LLP_HIGH, upper_32_bits(req->src));
		dma_ch_reg_writel(dma_ch, HDMA_CONTROL1, HDMA_CONTROL1_LL);
	}
	dma_ch_reg_writel(dma_ch, HDMA_INT_CLEAR, HDMA_INT_CLEAR_ALL);
	dma_ch_reg_writel(dma_ch, HDMA_INT_SETUP, HDMA_INT_SETUP_NORMAL);




	dma_ch_reg_writel(dma_ch, HDMA_DOORBELL, HDMA_DOORBELL_START);

	return 0;
}

int pcie_dwc_dma_start(struct pcidma_param *req, void (*cb)(uint64_t err, void *cookies),
		void *cookies)
{
	unsigned long flags;
	int ret;
	struct lin_pcie_dma_ch *dma_ch;





	if (!req) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if (req->pcie_ch >= PCIE_CH_MAX) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if (req->dmac_ch >= PCIDMA_CH_MAX) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if (req->dir >= PCIDMA_NUM) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if (!req->size && !req->src) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if (!cb) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	raw_spin_lock_irqsave(&lin_pcie_dma_ch_info[req->pcie_ch][req->dmac_ch].spin, flags);







	dma_ch = READ_ONCE(lin_pcie_dma_ch_info[req->pcie_ch][req->dmac_ch].dma_ch);
	if (dma_ch)
		ret = _pcie_dwc_dma_start(dma_ch, req, cb, cookies);
	else
		ret = -ENODEV;




	raw_spin_unlock_irqrestore(&lin_pcie_dma_ch_info[req->pcie_ch][req->dmac_ch].spin, flags);





	if (ret) {
		PR_ERR("%d", ret);
		return ret;
	}

	return 0;
}
EXPORT_SYMBOL(pcie_dwc_dma_start);

static void _pcie_dwc_dma_stop(struct lin_pcie_dma_ch *dma_ch)
{
	if (!dma_ch->used)
		return;
	dma_ch->used = false;

	dma_ch_reg_writel(dma_ch, HDMA_INT_SETUP, HDMA_INT_SETUP_DISABLE);
	dma_ch_reg_writel(dma_ch, HDMA_DOORBELL, HDMA_DOORBELL_STOP);
	dma_ch_reg_writel(dma_ch, HDMA_EN, 0);
}

int pcie_dwc_dma_stop(int pcie_ch, int dmac_ch)
{
	unsigned long flags;
	int ret;
	struct lin_pcie_dma_ch *dma_ch;





	if ((pcie_ch < 0) || (pcie_ch >= PCIE_CH_MAX)) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if ((dmac_ch < 0) || (dmac_ch >= PCIDMA_CH_MAX)) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	raw_spin_lock_irqsave(&lin_pcie_dma_ch_info[pcie_ch][dmac_ch].spin, flags);







	dma_ch = READ_ONCE(lin_pcie_dma_ch_info[pcie_ch][dmac_ch].dma_ch);
	if (dma_ch) {
		_pcie_dwc_dma_stop(dma_ch);
		ret = 0;
	} else
		ret = -ENODEV;




	raw_spin_unlock_irqrestore(&lin_pcie_dma_ch_info[pcie_ch][dmac_ch].spin, flags);





	if (ret) {
		PR_ERR("%d", ret);
		return ret;
	}

	return 0;
}
EXPORT_SYMBOL(pcie_dwc_dma_stop);












static int _dma_callback_fn(int irq, struct lin_pcie_dma_ch *dma_ch, u64 *err)
{
	u32 val;

	if (!dma_ch->used) {
		dma_ch_reg_writel(dma_ch, HDMA_INT_SETUP, HDMA_INT_SETUP_DISABLE);
		dma_ch_reg_writel(dma_ch, HDMA_EN, 0);

		return 1;
	}

	if ((dma_ch->wch && (irq != dma_ch->wirq)) || (!dma_ch->wch && (irq != dma_ch->rirq)))
		return 1;

	val = dma_ch_reg_readl(dma_ch, HDMA_INT_STATUS);
	if ((val & HDMA_INT_STATUS_KIND) == HDMA_INT_STATUS_STOP)
		*err = 0;
	else {
		PR_ERR("%d %d %08x", dma_ch->ch, dma_ch->wch, val);
		*err = BIT(HDMA_ERR_FLAG_POS) | val;
	}

	dma_ch->used = false;
	dma_ch_reg_writel(dma_ch, HDMA_INT_SETUP, HDMA_INT_SETUP_DISABLE);
	dma_ch_reg_writel(dma_ch, HDMA_EN, 0);

	return 0;
}

static irqreturn_t dma_callback_fn(int irq, void *context)
{
	struct lin_pcie_dma_ch_info *dma_ch_info = (struct lin_pcie_dma_ch_info *)context;
	struct lin_pcie_dma_ch *dma_ch;
	uint64_t err;
	unsigned long flags;
	int ret;



















	raw_spin_lock_irqsave(&dma_ch_info->spin, flags);




	ret = 1;
	dma_ch = READ_ONCE(dma_ch_info->dma_ch);
	if (dma_ch) {
		ret = _dma_callback_fn(irq, dma_ch, &err);
	}




	raw_spin_unlock_irqrestore(&dma_ch_info->spin, flags);










	if (!ret)
		dma_ch->cb(err, dma_ch->cookies);





	return IRQ_HANDLED;
}

static int lin_pcie_build_irq_for_dma(struct lin_pcie_priv *priv)
{
	struct dw_pcie *pci = &priv->pci;
	struct device *dev = pci->dev;
	struct platform_device *pdev = to_platform_device(dev);
	unsigned long dma_avail_ch = priv->dma_avail_ch;
	u32 bit;
	int ret;
	struct lin_pcie_dma_ch *dma_ch = NULL;

	for_each_set_bit(bit, &dma_avail_ch, PCIDMA_CH_MAX) {
		dma_ch = &priv->dma_ch[bit];
		dma_ch->priv = priv;
		dma_ch->ch = bit;

		dma_ch->wirq = platform_get_irq_byname(pdev, dma_wirq_name[bit]);
		if (dma_ch->wirq < 0) {
			ret = dma_ch->wirq;
			DEV_ERR("%d", ret);
			return ret;
		}

		ret = devm_request_irq(dev, dma_ch->wirq, dma_callback_fn, IRQF_TRIGGER_HIGH,
				dma_wirq_name[bit], &lin_pcie_dma_ch_info[priv->ch][bit]);
		if (ret) {
			DEV_ERR("%d", ret);
			return ret;
		}

		dma_ch->rirq = platform_get_irq_byname(pdev, dma_rirq_name[bit]);
		if (dma_ch->rirq < 0) {
			ret = dma_ch->rirq;
			DEV_ERR("%d", ret);
			return ret;
		}

		ret = devm_request_irq(dev, dma_ch->rirq, dma_callback_fn, IRQF_TRIGGER_HIGH,
				dma_rirq_name[bit], &lin_pcie_dma_ch_info[priv->ch][bit]);
		if (ret) {
			DEV_ERR("%d", ret);
			return ret;
		}
	}






	return 0;
}

static void _lin_pcie_make_dma_available(struct lin_pcie_priv *priv)
{
	int dma_idx;
	unsigned long flags;





	for (dma_idx = 0; dma_idx < PCIDMA_CH_MAX; dma_idx++) {
		if (priv->dma_avail_ch & BIT(dma_idx)) {
			raw_spin_lock_irqsave(&lin_pcie_dma_ch_info[priv->ch][dma_idx].spin, flags);




			WRITE_ONCE(lin_pcie_dma_ch_info[priv->ch][dma_idx].dma_ch,
					&priv->dma_ch[dma_idx]);




			raw_spin_unlock_irqrestore(&lin_pcie_dma_ch_info[priv->ch][dma_idx].spin,
					flags);




		}
	}
}

static void _lin_pcie_make_dma_unavailable(struct lin_pcie_priv *priv)
{
	int dma_idx;
	unsigned long flags;
	struct lin_pcie_dma_ch *dma_ch = NULL;





	for (dma_idx = 0; dma_idx < PCIDMA_CH_MAX; dma_idx++) {
		raw_spin_lock_irqsave(&lin_pcie_dma_ch_info[priv->ch][dma_idx].spin, flags);




		dma_ch = READ_ONCE(lin_pcie_dma_ch_info[priv->ch][dma_idx].dma_ch);
		if (dma_ch) {
			_pcie_dwc_dma_stop(dma_ch);
			WRITE_ONCE(lin_pcie_dma_ch_info[priv->ch][dma_idx].dma_ch, NULL);
		}




		raw_spin_unlock_irqrestore(&lin_pcie_dma_ch_info[priv->ch][dma_idx].spin, flags);




	}
}

static int _pcie_dwc_root_ctrl(int pcie_ch, int on)
{
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[pcie_ch];
	unsigned long flags;
	int ret;
	enum lin_pcie_rc_state state;
	struct lin_pcie_priv *priv;
	enum dw_pcie_device_mode mode;
	struct device *dev;





	raw_spin_lock_irqsave(&ch_info->spin, flags);




	state = READ_ONCE(ch_info->state);
	priv = READ_ONCE(ch_info->priv);
	mode = READ_ONCE(ch_info->mode);
	dev = READ_ONCE(ch_info->dev);

	if ((mode != DW_PCIE_RC_TYPE) && (mode != DW_PCIE_UNKNOWN_TYPE)) {
		ret = -EINVAL;
		PR_DEBUG("%d", ret);
	} else if (on == PCIE_CTRL_ON) {
		if (state == LIN_PCIE_STATE_RC_PREP)
			ret = 2;
		else if (state == LIN_PCIE_STATE_RC_START)
			ret = 1;
		else if ((state == LIN_PCIE_STATE_STOP) || (state == LIN_PCIE_STATE_INIT)) {
			ret = 0;
		} else {
			ret = -EBUSY;
			PR_ERR("%d", ret);
		}
	} else if (on == PCIE_CTRL_OFF) {
		if ((state == LIN_PCIE_STATE_ERR) || (state == LIN_PCIE_STATE_RC_PREP) ||
				(state == LIN_PCIE_STATE_RC_START) ||
				(state == LIN_PCIE_STATE_RC_SUSPENDING)) {
			ret = 0;
		} else if (state == LIN_PCIE_STATE_STOP)
			ret = 1;
		else {
			ret = -EBUSY;
			PR_ERR("%d", ret);
		}
	} else {
		if (state == LIN_PCIE_STATE_RC_PREP)
			ret = 0;
		else if ((state == LIN_PCIE_STATE_STOP) || (state == LIN_PCIE_STATE_RC_START))
			ret = 1;
		else {
			ret = -EBUSY;
			PR_ERR("%d", ret);
		}
	}




	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	if (ret < 0) {
		PR_DEBUG("%d", ret);
		return ret;
	} else if (ret == 1)
		return 0;
	else if (ret == 2) {
		ret = lin_pcie_start_rc(priv);
		if (ret) {
			PR_ERR("%d", ret);
			goto err_start_rc;
		}

		raw_spin_lock_irqsave(&ch_info->spin, flags);



		WRITE_ONCE(ch_info->state, LIN_PCIE_STATE_RC_START);
		_lin_pcie_make_dma_available(priv);



		raw_spin_unlock_irqrestore(&ch_info->spin, flags);





		return 0;
	}

	raw_spin_lock_irqsave(&ch_info->spin, flags);




	WRITE_ONCE(ch_info->owner, current);
	if (on == PCIE_CTRL_ON)
		WRITE_ONCE(ch_info->start_istr, true);




	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	if (on == PCIE_CTRL_ON)
		ret = device_attach(dev);
	else {
		device_release_driver(dev);
		ret = 1;
	}

	raw_spin_lock_irqsave(&ch_info->spin, flags);




	WRITE_ONCE(ch_info->owner, NULL);
	if (on == PCIE_CTRL_ON)
		WRITE_ONCE(ch_info->start_istr, false);




	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	if (ret <= 0) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	return 0;

err_start_rc:
	lin_pcie_term_rc(priv);
	raw_spin_lock_irqsave(&ch_info->spin, flags);



	if (priv)
		_lin_pcie_make_dma_unavailable(priv);
	WRITE_ONCE(ch_info->state, LIN_PCIE_STATE_ERR);



	raw_spin_unlock_irqrestore(&ch_info->spin, flags);




	PR_ERR("%d", ret);
	return ret;
}

int pcie_dwc_root_ctrl(int pcie_ch, int on)
{
	int ret;

	if ((pcie_ch < 0) || (pcie_ch >= PCIE_CH_MAX)) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if ((on < 0) || (on >= PCIE_CTRL_MAX)) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	mutex_lock(&lin_pcie_ch_info[pcie_ch].mutex);
	ret = _pcie_dwc_root_ctrl(pcie_ch, on);
	mutex_unlock(&lin_pcie_ch_info[pcie_ch].mutex);
	if (ret) {
		PR_DEBUG("%d", ret);
		return ret;
	}

	return 0;
}






struct firm_info * __weak pcie_dwc_firm(int firm_id)
{
	return NULL;
}











int __weak pcie_dwc_vboost(int pcie_ch)
{
	return -1;
}











int __weak pcie_dwc_tmo(int pcie_ch)
{
	return -1;
}











void __weak pcie_perst_delay(int pcie_ch)
{
}










int __weak pcie_aspm_policy(int pcie_ch)
{
	return aspm_policy;
}





int __weak pcie_dwc_max_speed(int pcie_ch)
{
	return -1;
}










int __weak pcie_check_prsnt(struct pci_dev *dev)
{
	return 1;
}




uint __weak pcie_dwc_axiordering(int pcie_ch)
{
	return 0;
}


int pcie_dwc_set_lanes(struct pci_dev *dev, int lanes)
{
	int ret;
	u32 target, val;

	if (!dev || !dev->bus) {
		ret = -ENODEV;
		PR_ERR("%d", ret);
		return ret;
	}

	if (dev->bus->number || dev->devfn) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	switch (lanes) {
	case PCIE_LANE1:
		target = PCI_EXP_LNKSTA_NLW_X1 >> PCI_EXP_LNKSTA_NLW_SHIFT;
		break;
	case PCIE_LANE2:
		target = PCI_EXP_LNKSTA_NLW_X2 >> PCI_EXP_LNKSTA_NLW_SHIFT;
		break;
	case PCIE_LANE4:
		target = PCI_EXP_LNKSTA_NLW_X4 >> PCI_EXP_LNKSTA_NLW_SHIFT;
		break;
	default:
		ret = -EPERM;
		PR_ERR("%d", ret);
		return ret;
	}

	pci_read_config_dword(dev, PCIE_PORT_MULTI_LANE_CTRL, &val);
	val &= ~(PORT_TARGET_LINK_WIDTH | PORT_DIRECT_LINK_WIDTH_CHANGE);
	val |= target;
	pci_write_config_dword(dev, PCIE_PORT_MULTI_LANE_CTRL, val);
	val |= PORT_DIRECT_LINK_WIDTH_CHANGE;
	pci_write_config_dword(dev, PCIE_PORT_MULTI_LANE_CTRL, val);

	return 0;
}




int pcie_dwc_get_lanes(struct pci_dev *dev)
{
	int ret;
	u16 linkstat;
	u32 target;

	if (!dev || !dev->bus) {
		ret = -ENODEV;
		PR_ERR("%d", ret);
		return ret;
	}

	if (dev->bus->number || dev->devfn) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	ret = pcie_capability_read_word(dev, PCI_EXP_LNKSTA, &linkstat);
	if (ret) {
		ret = -EIO;
		PR_ERR("%d", ret);
		return ret;
	}

	linkstat = linkstat & PCI_EXP_LNKSTA_NLW;
	switch (linkstat) {
	case PCI_EXP_LNKSTA_NLW_X1:
		target = PCIE_LANE1;
		break;
	case PCI_EXP_LNKSTA_NLW_X2:
		target = PCIE_LANE2;
		break;
	case PCI_EXP_LNKSTA_NLW_X4:
		target = PCIE_LANE4;
		break;
	default:
		ret = -EIO;
		PR_ERR("%d", ret);
		return ret;
	}

	return target;
}




int pcie_is_linkup(struct pci_dev *dev)
{
	int ret;
	u32 val;

	if (!dev || !dev->bus) {
		ret = -ENODEV;
		PR_ERR("%d", ret);
		return ret;
	}

	if (dev->bus->number || dev->devfn) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	pci_read_config_dword(dev, PCIE_PORT_DEBUG1, &val);

	return ((val & PCIE_PORT_DEBUG1_LINK_UP) && (!(val & PCIE_PORT_DEBUG1_LINK_IN_TRAINING)));
}




void pcie_err_clr(struct pci_bus *bus)
{
	int ret;
	struct pci_bus *root_bus;
	struct pci_dev *root_dev;
	void __iomem *config;

	if (!bus) {
		PR_ERR();
		return;
	}

	root_bus = bus;
	while (!pci_is_root_bus(root_bus))
		root_bus = root_bus->parent;

	root_dev = list_first_entry_or_null(&root_bus->devices, struct pci_dev, bus_list);
	if (!root_dev || !root_dev->bus) {
		PR_ERR();
		return;
	}
	if (root_dev->bus->number || root_dev->devfn) {
		PR_ERR();
		return;
	}
	if (!root_dev->aer_cap) {
		PR_ERR();
		return;
	}

	config = dw_pcie_own_conf_map_bus(root_bus, 0, 0);
	if (!config) {
		PR_ERR();
		return;
	}

	ret = dw_pcie_write(config + root_dev->aer_cap + PCI_ERR_UNCOR_STATUS, 4, 0xffffffff);
	if (ret) {
		PR_ERR();
		return;
	}

	return;
}





int __weak pcie_ext_err_stat(struct pci_bus *bus, void __iomem *dbi, void __iomem *link)
{
	return 0;
}











int pcie_err_stat(struct pci_bus *bus)
{
	int ret;
	struct pci_bus *root_bus;
	struct pci_dev *root_dev;
	void __iomem *config;
	u32 val;
	struct pcie_port *pp;
	struct dw_pcie *pci;
	struct lin_pcie_priv *priv;

	if (!bus) {
		PR_ERR();
		return 0;
	}

	root_bus = bus;
	while (!pci_is_root_bus(root_bus))
		root_bus = root_bus->parent;

	root_dev = list_first_entry_or_null(&root_bus->devices, struct pci_dev, bus_list);
	if (!root_dev || !root_dev->bus ) {
		PR_ERR();
		return 0;
	}
	if (root_dev->bus->number || root_dev->devfn) {
		PR_ERR();
		return 0;
	}
	if (!root_dev->aer_cap) {
		PR_ERR();
		return 0;
	}

	config = dw_pcie_own_conf_map_bus(root_bus, 0, 0);
	if (!config) {
		PR_ERR();
		return 0;
	}

	pp = bus->sysdata;
	pci = to_dw_pcie_from_pp(pp);
	priv = to_lin_priv_from_pcie(pci);

	ret = pcie_ext_err_stat(bus, config, priv->link_base);
	if (ret) {
		PR_ERR();
		return -1;
	}

	ret = dw_pcie_read(config + root_dev->aer_cap + PCI_ERR_UNCOR_STATUS, 4, &val);
	if (ret) {
		PR_ERR();
		return 0;
	}

	if (val & ~1) {
		struct pcie_port *pp = bus->sysdata;
		struct dw_pcie *pci = to_dw_pcie_from_pp(pp);
		int ch = of_get_pci_domain_nr(pci->dev->of_node);
		pcie_log("PCI%d:%s:0x%x", ch, __func__, val);
		PR_ERR("%08x", val);
		return  -1;
	}

	return 0;
}




int pcie_dwc_set_speed(struct pci_dev *dev, int speed)
{
	int ret;
	u32 target;
	struct pcie_port *pp;
	struct dw_pcie *pcie;
	struct lin_pcie_priv *priv;
	unsigned long flags;
	u32 reg32;





	if (!dev || !dev->bus) {
		ret = -ENODEV;
		PR_ERR("%d", ret);
		return ret;
	}

	if (dev->bus->number || dev->devfn) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	switch (speed) {
	case PCIE_SPEED1:
		target = PCI_EXP_LNKCTL2_TLS_2_5GT;
		break;
	case PCIE_SPEED2:
		target = PCI_EXP_LNKCTL2_TLS_5_0GT;
		break;
	case PCIE_SPEED3:
		target = PCI_EXP_LNKCTL2_TLS_8_0GT;
		break;
	case PCIE_SPEED4:
		target = PCI_EXP_LNKCTL2_TLS_16_0GT;
		break;
	default:
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	pp = dev->bus->sysdata;
	pcie = to_dw_pcie_from_pp(pp);
	priv = to_lin_priv_from_pcie(pcie);

	raw_spin_lock_irqsave(&lin_pcie_ch_info[priv->ch].config_spin, flags);




	reg32 = dw_pcie_readl_dbi(&priv->pci, priv->pcie_ofst + PCI_EXP_LNKCTL2);
	reg32 &= ~PCI_EXP_LNKCTL2_TLS;
	dw_pcie_writel_dbi(&priv->pci, priv->pcie_ofst + PCI_EXP_LNKCTL2, reg32 | target);

	reg32 = dw_pcie_readl_dbi(&priv->pci, PCIE_LINK_WIDTH_SPEED_CONTROL);
	reg32 &= ~PORT_LOGIC_SPEED_CHANGE;
	dw_pcie_writel_dbi(&priv->pci, PCIE_LINK_WIDTH_SPEED_CONTROL, reg32);
	reg32 |= PORT_LOGIC_SPEED_CHANGE;
	dw_pcie_writel_dbi(&priv->pci, PCIE_LINK_WIDTH_SPEED_CONTROL, reg32);




	raw_spin_unlock_irqrestore(&lin_pcie_ch_info[priv->ch].config_spin, flags);





	return 0;
}




int pcie_dwc_get_speed(struct pci_dev *dev)
{
	int ret;
	u16 linkstat;
	u32 target;

	if (!dev || !dev->bus) {
		ret = -ENODEV;
		PR_ERR("%d", ret);
		return ret;
	}

	if (dev->bus->number || dev->devfn) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	ret = pcie_capability_read_word(dev, PCI_EXP_LNKSTA, &linkstat);
	if (ret) {
		ret = -EIO;
		PR_ERR("%d", ret);
		return ret;
	}

	switch (linkstat & PCI_EXP_LNKSTA_CLS) {
	case PCI_EXP_LNKSTA_CLS_2_5GB:
		target = PCIE_SPEED1;
		break;
	case PCI_EXP_LNKSTA_CLS_5_0GB:
		target = PCIE_SPEED2;
		break;
	case PCI_EXP_LNKSTA_CLS_8_0GB:
		target = PCIE_SPEED3;
		break;
	case PCI_EXP_LNKSTA_CLS_16_0GB:
		target = PCIE_SPEED4;
		break;
	default:
		ret = -EIO;
		PR_ERR("%d", ret);
		return ret;
	}
	pcie_update_link_speed(dev->bus, linkstat);

	return target;
}





void __weak pcie_dwc_notify(struct device *dev, int pcie_ch, enum PCIE_NOTIFY event, void __iomem *link_base)
{
}










/* set CLKREQ# to Hi-Z */
static void lin_pcie_fix_clkreq_to_hiz(struct lin_pcie_priv *priv)
{
	if (!(priv->clkreq.mode & GPIOMODE_PERIPHERAL))
		return;

	mb();
	gpio_set_mode_bit(priv->clkreq.gpio_ch, priv->clkreq.gpio_bit, GPIOMODE_GPIO|GPIOMODE_INPUT);
	pcie_log("PCI%d:CLKREQ(%u,%u)=Hi-Z", priv->ch, priv->clkreq.gpio_ch, priv->clkreq.gpio_bit);
	mb();
}

static void lin_pcie_init_clkreq(struct lin_pcie_priv *priv)
{
	if (!(priv->clkreq.mode & GPIOMODE_PERIPHERAL))
		return;

	gpio_set_mode_bit(priv->clkreq.gpio_ch, priv->clkreq.gpio_bit, GPIOMODE_GPIO | GPIOMODE_OUTPUT_L);
	mb();
	pcie_log("PCI%d:CLKREQ(%u,%u)=GPIO_L", priv->ch, priv->clkreq.gpio_ch, priv->clkreq.gpio_bit);
}

static void lin_pcie_fix_clkreq_to_low(struct lin_pcie_priv *priv, bool active)
{
	if (!(priv->clkreq.mode & GPIOMODE_PERIPHERAL))
		return;

	mb();
	if (active) {
		gpio_set_mode_bit(priv->clkreq.gpio_ch, priv->clkreq.gpio_bit, GPIOMODE_GPIO);
		pcie_log("PCI%d:CLKREQ(%u,%u)=L", priv->ch, priv->clkreq.gpio_ch, priv->clkreq.gpio_bit);
	} else {
		gpio_set_mode_bit(priv->clkreq.gpio_ch, priv->clkreq.gpio_bit, GPIOMODE_PERIPHERAL);
		pcie_log("PCI%d:CLKREQ(%u,%u)=FUNC", priv->ch, priv->clkreq.gpio_ch, priv->clkreq.gpio_bit);
	}
	mb();
}

static void _pcie_dwc_fix_clkreq_to_low(int pcie_ch, bool active)
{
	if (!lin_pcie_ch_info[pcie_ch].clkreq_ctrl_avail) {
		PR_ERR("%d %d %d", pcie_ch, active, lin_pcie_ch_info[pcie_ch].state);
		return;
	}

	lin_pcie_fix_clkreq_to_low(lin_pcie_ch_info[pcie_ch].priv, active);
}

void pcie_dwc_fix_clkreq_to_low(int pcie_ch, bool active)
{
	unsigned long flags;





	if ((pcie_ch < 0) || (pcie_ch >= PCIE_CH_MAX)) {
		PR_ERR("%d", pcie_ch);
		return;
	}

	raw_spin_lock_irqsave(&lin_pcie_ch_info[pcie_ch].spin, flags);




	_pcie_dwc_fix_clkreq_to_low(pcie_ch, active);




	raw_spin_unlock_irqrestore(&lin_pcie_ch_info[pcie_ch].spin, flags);




}

static void lin_pcie_rc_com_irq_fn(struct irq_desc *desc)
{
	struct pcie_port *pp;
	struct dw_pcie *pci;
	struct lin_pcie_priv *priv;
	struct irq_chip *chip = irq_desc_get_chip(desc);
	unsigned long reg;
	u32 val;
	u32 bit;
#ifdef SAMPLECODE_PME
	int ret;
#endif








	chained_irq_enter(chip, desc);

	pp = irq_desc_get_handler_data(desc);
	pci = to_dw_pcie_from_pp(pp);
	priv = to_lin_priv_from_pcie(pci);

	val = readl(priv->link_base + IK_IRQ_COM_STS);

#ifdef SAMPLECODE_PME
	if (val & IK_IRQ_COM_PME_MSI_POS) {
		writel(IK_IRQ_COM_PME_MSI_POS, priv->link_base + IK_IRQ_COM_CLR);

		if (priv->pme_irq) {
			ret = irq_inject_interrupt(priv->pme_irq);
			if (ret) {
				PR_ERR("%d", ret);
			}
		}
	}
#endif

	reg = 0;
	if (val & IK_IRQ_COM_INTA_POS_STS)
		reg |= 1<<0;
	if (val & IK_IRQ_COM_INTB_POS_STS)
		reg |= 1<<1;
	if (val & IK_IRQ_COM_INTC_POS_STS)
		reg |= 1<<2;
	if (val & IK_IRQ_COM_INTD_POS_STS)
		reg |= 1<<3;

	for_each_set_bit(bit, &reg, PCI_NUM_INTX)
		generic_handle_domain_irq(priv->lgc_irqd, bit);

	dw_handle_msi_irq(pp);

	pcie_dwc_notify(priv->pci.dev, priv->ch, PCIE_NOTIFY_IRQCOM, priv->link_base);

	chained_irq_exit(chip, desc);





}

static void lin_pcie_intx_irq_ack(struct irq_data *d)
{
	struct pcie_port *pp = irq_data_get_irq_chip_data(d);
	struct dw_pcie *pci = to_dw_pcie_from_pp(pp);
	struct device *dev = pci->dev;
	struct lin_pcie_priv *priv = to_lin_priv_from_pcie(pci);
	u32 val;

	if (d->hwirq == 0)
		val = IK_IRQ_COM_INTA_POS_STS;
	else if (d->hwirq == 1)
		val = IK_IRQ_COM_INTB_POS_STS;
	else if (d->hwirq == 2)
		val = IK_IRQ_COM_INTC_POS_STS;
	else if (d->hwirq == 3)
		val = IK_IRQ_COM_INTD_POS_STS;
	else {
		DEV_ERR();
		return;
	}
	writel(val, priv->link_base + IK_IRQ_COM_CLR);
}

static void lin_pcie_intx_irq_mask(struct irq_data *d)
{
	struct pcie_port *pp = irq_data_get_irq_chip_data(d);
	struct dw_pcie *pci = to_dw_pcie_from_pp(pp);
	struct device *dev = pci->dev;
	struct lin_pcie_priv *priv = to_lin_priv_from_pcie(pci);
	u32 val;

	val = readl(priv->link_base + IK_IRQ_COM_EN);
	if (d->hwirq == 0)
		val &= ~IK_IRQ_COM_INTA_POS_STS;
	else if (d->hwirq == 1)
		val &= ~IK_IRQ_COM_INTB_POS_STS;
	else if (d->hwirq == 2)
		val &= ~IK_IRQ_COM_INTC_POS_STS;
	else if (d->hwirq == 3)
		val &= ~IK_IRQ_COM_INTD_POS_STS;
	else {
		DEV_ERR();
		return;
	}
	writel(val, priv->link_base + IK_IRQ_COM_EN);
}

static void lin_pcie_intx_irq_unmask(struct irq_data *d)
{
	struct pcie_port *pp = irq_data_get_irq_chip_data(d);
	struct dw_pcie *pci = to_dw_pcie_from_pp(pp);
	struct device *dev = pci->dev;
	struct lin_pcie_priv *priv = to_lin_priv_from_pcie(pci);
	u32 val;

	val = readl(priv->link_base + IK_IRQ_COM_EN);
	if (d->hwirq == 0) {
		val |= IK_IRQ_COM_INTA_POS_STS;
	} else if (d->hwirq == 1) {
		val |= IK_IRQ_COM_INTB_POS_STS;
	} else if (d->hwirq == 2) {
		val |= IK_IRQ_COM_INTC_POS_STS;
	} else if (d->hwirq == 3) {
		val |= IK_IRQ_COM_INTD_POS_STS;
	} else {
		DEV_ERR();
		return;
	}
	writel(val, priv->link_base + IK_IRQ_COM_EN);
}

static struct irq_chip lin_pcie_intx_chip = {
	.name = "lin-pcie-intx-chip",
	.irq_ack = lin_pcie_intx_irq_ack,
	.irq_mask = lin_pcie_intx_irq_mask,
	.irq_unmask = lin_pcie_intx_irq_unmask,
};

static int lin_pcie_intx_map(struct irq_domain *domain, unsigned int irq, irq_hw_number_t hwirq)
{
	irq_set_chip_and_handler(irq, &lin_pcie_intx_chip, handle_edge_irq);
	irq_set_chip_data(irq, domain->host_data);

	return 0;
}

static const struct irq_domain_ops lin_pcie_intx_domain_ops = {
	.map = lin_pcie_intx_map,
};

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
static irqreturn_t lin_pcie_rc_cnct_irq_fn(int irq, void *context)
{
	struct lin_pcie_priv *priv = (struct lin_pcie_priv *)context;
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[priv->ch];
	u32 val;










	val = readl(priv->link_base + IK_IRQ_CNCT_STS);
	if (!(val & IK_IRQ_CNCT_DL_UP_POS)) {








		return IRQ_HANDLED;
	}

	if (val & IK_IRQ_CNCT_DL_UP_POS) {
		u32 reg32;
		u16 tls;

		writel(IK_IRQ_CNCT_DL_UP_POS, priv->link_base + IK_IRQ_CNCT_CLR);

		raw_spin_lock(&ch_info->config_spin);



		tls = dw_pcie_readw_dbi(&priv->pci, priv->pcie_ofst + PCI_EXP_LNKCTL2) & PCI_EXP_LNKCTL2_TLS;
		if (tls != PCI_EXP_LNKCTL2_TLS_2_5GT) {
			reg32 = dw_pcie_readl_dbi(&priv->pci, PCIE_LINK_WIDTH_SPEED_CONTROL);
			reg32 &= ~PORT_LOGIC_SPEED_CHANGE;
			dw_pcie_writel_dbi(&priv->pci, PCIE_LINK_WIDTH_SPEED_CONTROL, reg32);
			reg32 |= PORT_LOGIC_SPEED_CHANGE;
			dw_pcie_writel_dbi(&priv->pci, PCIE_LINK_WIDTH_SPEED_CONTROL, reg32);
		}




		raw_spin_unlock(&ch_info->config_spin);




	}






	return IRQ_HANDLED;
}
#endif

static irqreturn_t lin_pcie_rc_err_irq_fn(int irq, void *context)
{
	struct lin_pcie_priv *priv = (struct lin_pcie_priv *)context;
	u32 val;
	int ret;








	val = readl(priv->link_base + IK_IRQ_ERR_STS);
	if (!(val & IK_IRQ_ERR_RC_MSI)) {








		return IRQ_HANDLED;
	}

	if (val & IK_IRQ_ERR_RC_MSI) {
		writel(IK_IRQ_ERR_RC_MSI, priv->link_base + IK_IRQ_ERR_CLR);

		if (priv->aer_irq) {
			ret = irq_inject_interrupt(priv->aer_irq);
			if (ret) {
				PR_ERR("%d", ret);
			}
		}
	}






	return IRQ_HANDLED;
}

static int lin_pcie_build_rc_irq(struct lin_pcie_priv *priv)
{
	struct dw_pcie *pci = &priv->pci;
	struct device *dev = pci->dev;
	struct platform_device *pdev = to_platform_device(dev);
	struct pcie_port *pp = &pci->pp;
	struct device_node *np = pci->dev->of_node;
	struct device_node *child;
	int ret;

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	priv->cnct_irq = platform_get_irq_byname(pdev, "cnct");
	if (priv->cnct_irq < 0) {
		ret = priv->cnct_irq;
		DEV_ERR("%d", ret);
		goto err;
	}

	irq_set_status_flags(priv->cnct_irq, IRQ_NOAUTOEN);

	ret = devm_request_irq(dev, priv->cnct_irq, lin_pcie_rc_cnct_irq_fn, IRQF_TRIGGER_HIGH |
			IRQF_NO_THREAD, "lin-pcie-rc-cnct", priv);
	if (ret) {
		DEV_ERR("%d", ret);
		goto err;
	}
#endif

	priv->err_irq = platform_get_irq_byname(pdev, "err");
	if (priv->err_irq < 0) {
		ret = priv->err_irq;
		DEV_ERR("%d", ret);
		goto err;
	}

	irq_set_status_flags(priv->err_irq, IRQ_NOAUTOEN);

	ret = devm_request_irq(dev, priv->err_irq, lin_pcie_rc_err_irq_fn, IRQF_TRIGGER_HIGH |
			IRQF_NO_THREAD, "lin-pcie-rc-err", priv);
	if (ret) {
		DEV_ERR("%d", ret);
		goto err;
	}

	pp->irq = platform_get_irq_byname(pdev, "com");
	if (pp->irq < 0) {
		ret = pp->irq;
		DEV_ERR("%d", ret);
		goto err;
	}
	pp->msi_irq = -ENODEV;

	child = of_get_child_by_name(np, "intx-controller");
	if (!child) {
		ret = -ENODATA;
		DEV_ERR("%d", ret);
		goto err;
	}

	priv->lgc_irqd = irq_domain_add_linear(child, PCI_NUM_INTX, &lin_pcie_intx_domain_ops, pp);
	if (!priv->lgc_irqd) {
		ret = -ENOMEM;
		DEV_ERR("%d", ret);
		goto err_add;
	}

	irq_set_chained_handler_and_data(pp->irq, lin_pcie_rc_com_irq_fn, pp);
	disable_irq(pp->irq);

	ret = lin_pcie_build_irq_for_dma(priv);
	if (ret)
		goto err_add;

	of_node_put(child);

	return 0;

err_add:
	of_node_put(child);
err:
	DEV_ERR("%d", ret);
	return ret;
}

static void lin_pcie_clear_rc_irq(struct lin_pcie_priv *priv)
{
#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	writel(0xffffffff, priv->link_base + IK_IRQ_CNCT_CLR);
#endif
	writel(0xffffffff, priv->link_base + IK_IRQ_ERR_CLR);
	writel(0xffffffff, priv->link_base + IK_IRQ_COM_CLR);
}

static void lin_pcie_enable_rc_irq(struct lin_pcie_priv *priv)
{
	struct pcie_port *pp = &priv->pci.pp;

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	writel(IK_IRQ_CNCT_DL_UP_POS, priv->link_base + IK_IRQ_CNCT_EN);
	enable_irq(priv->cnct_irq);
#endif

	writel(IK_IRQ_ERR_RC_MSI, priv->link_base + IK_IRQ_ERR_EN);
	enable_irq(priv->err_irq);

	writel(
#ifdef SAMPLECODE_PME
			IK_IRQ_COM_PME_MSI_POS |
#endif
			IK_IRQ_COM_INTA_POS_STS | IK_IRQ_COM_INTB_POS_STS |
			IK_IRQ_COM_INTC_POS_STS | IK_IRQ_COM_INTD_POS_STS,
			priv->link_base + IK_IRQ_COM_EN);
	enable_irq(pp->irq);
}

static void lin_pcie_disable_rc_irq(struct lin_pcie_priv *priv)
{
	struct pcie_port *pp = &priv->pci.pp;

	disable_irq(pp->irq);
	writel(0, priv->link_base + IK_IRQ_COM_EN);

	disable_irq(priv->err_irq);
	writel(0, priv->link_base + IK_IRQ_ERR_EN);

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	disable_irq(priv->cnct_irq);
	writel(0, priv->link_base + IK_IRQ_CNCT_EN);
#endif
}

static int lin_pcie_rc_host_init(struct pcie_port *pp)
{
	struct dw_pcie *pci = to_dw_pcie_from_pp(pp);
	struct device *dev = pci->dev;
	struct lin_pcie_priv *priv = to_lin_priv_from_pcie(pci);
	unsigned long reg;
	int tmo;
	const __be32 *addr;
	int ret;
	int lenp;
#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	int domain;
#endif

	priv->pcie_ofst = dw_pcie_find_capability(pci, PCI_CAP_ID_EXP);
	if (!priv->pcie_ofst) {
		ret = -ENODATA;
		DEV_ERR("%d", ret);
		return ret;
	}

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	priv->spci_ofst = dw_pcie_find_ext_capability(pci, PCI_EXT_CAP_ID_SECPCI);
	if (!priv->spci_ofst) {
		ret = -ENODATA;
		DEV_ERR("%d", ret);
		return ret;
	}

	if (priv->omission_of_rc_start) {
		pci->link_gen = 0;
	} else {
		pci->link_gen = pcie_dwc_max_speed(priv->ch);
		if (pci->link_gen < 0)
			pci->link_gen = of_pci_get_max_link_speed(dev->of_node);
	}
#else
	pci->link_gen = 1;
#endif

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	reg = dw_pcie_readw_dbi(pci, priv->pcie_ofst + PCI_EXP_FLAGS);
	reg |= PCI_EXP_FLAGS_SLOT;
	dw_pcie_dbi_ro_wr_en(pci);
	dw_pcie_writew_dbi(pci, priv->pcie_ofst + PCI_EXP_FLAGS, reg);
	dw_pcie_dbi_ro_wr_dis(pci);

	domain = of_get_pci_domain_nr(dev->of_node);
	if ((domain < 0) || (domain > 0x1ffe)) {
		ret = -EINVAL;
		DEV_ERR("%d", ret);
		return ret;
	}
	reg = dw_pcie_readl_dbi(pci, priv->pcie_ofst + PCI_EXP_SLTCAP);
	reg &= ~PCI_EXP_SLTCAP_PSN;
	reg |= (domain + 1) << PCI_EXP_SLTCAP_PSN_LSB;
	dw_pcie_dbi_ro_wr_en(pci);
	dw_pcie_writel_dbi(pci, priv->pcie_ofst + PCI_EXP_SLTCAP, reg);
	dw_pcie_dbi_ro_wr_dis(pci);
#endif

	tmo = pcie_dwc_tmo(priv->ch);
	if (tmo >= 0) {
		reg = dw_pcie_readl_dbi(pci, priv->pcie_ofst + PCI_EXP_DEVCTL2);
		reg &= ~PCI_EXP_DEVCTL2_COMP_TIMEOUT;
		reg |= tmo & PCI_EXP_DEVCTL2_COMP_TIMEOUT;
		dw_pcie_writel_dbi(pci, priv->pcie_ofst + PCI_EXP_DEVCTL2, reg);
	}

	{
		u32 ctrl = pcie_dwc_axiordering(domain);
		if (ctrl) {
			dw_pcie_writel_dbi(pci, PCIE_AMBA_ORDERING_CTRL, ctrl);
		}
	}

	addr = of_get_property(pci->dev->of_node, "msi-addr", &lenp);
	if (addr) {
		if (lenp != 8) {
			ret = -EINVAL;
			DEV_ERR("%d", ret);
			return ret;
		}

		pp->msi_target = (u64)be32_to_cpup(addr + 0) << 32;
		pp->msi_target |= be32_to_cpup(addr + 1);
		if (pp->msi_target % 4) {
			ret = -EINVAL;
			DEV_ERR("%d", ret);
			return ret;
		}
	}

	return 0;
}

static int lin_pcie_rc_host_init2(struct pcie_port *pp)
{
	int ret;
	struct dw_pcie *pci = to_dw_pcie_from_pp(pp);
	struct device *dev = pci->dev;
	struct lin_pcie_priv *priv = to_lin_priv_from_pcie(pci);
	struct device_node *np = pci->dev->of_node;
	int lenp;
	const __be32 *addr;
	u64 pci_addr;
	u64 cpu_addr;
	u64 size;
	int cnt;
	int nums;

	if (priv->omission_of_rc_start) {
		for (cnt = 0; cnt < pci->num_ib_windows; cnt++)
			dw_pcie_disable_atu(pci, cnt, DW_PCIE_REGION_INBOUND);


















	}

	cnt = 0;
	addr = of_get_property(np, "inbound-atu", &lenp);
	if (addr) {
		if (lenp % 24) {
			ret = -EINVAL;
			DEV_ERR("%d", ret);
			return ret;
		}

		nums = lenp / 24;
		if (priv->pci32)
			nums++;

		if (nums > pci->num_ib_windows) {
			ret = -EINVAL;
			DEV_ERR("%d", ret);
			return ret;
		}

		for (; cnt < lenp / 24; cnt++) {
			pci_addr = (u64)be32_to_cpup(addr + cnt * 6 + 0) << 32;
			pci_addr |= be32_to_cpup(addr + cnt * 6 + 1);
			cpu_addr = (u64)be32_to_cpup(addr + cnt * 6 + 2) << 32;
			cpu_addr |= be32_to_cpup(addr + cnt * 6 + 3);
			size = (u64)be32_to_cpup(addr + cnt * 6 + 4) << 32;
			size |= be32_to_cpup(addr + cnt * 6 + 5);
			dw_pcie_prog_inbound_atu_unroll2(pci, 0, cnt, PCIE_ATU_TYPE_MEM, pci_addr,
					cpu_addr, size);
		}
	}

	if (priv->pci32) {
		dw_pcie_prog_inbound_atu_unroll2(pci, 0, cnt, PCIE_ATU_TYPE_MEM, 0, PHYS_OFFSET,
				LIN_PCIE_PCI32_SIZE);
	}

	return 0;
}

static int lin_pcie_rc_host_init3(struct pcie_port *pp)
{
	int ret;
	struct dw_pcie *pci = to_dw_pcie_from_pp(pp);
	struct lin_pcie_priv *priv = to_lin_priv_from_pcie(pci);
	u32 reg32;
	u16 tls;
	int cnt;

	reg32 = readl(priv->link_base + IK_IRQ_CNCT_RAW);
	if (!(reg32 & IK_IRQ_CNCT_DL_UP_POS))
		return 0;

	writel(IK_IRQ_CNCT_DL_UP_POS, priv->link_base + IK_IRQ_CNCT_CLR);

	tls = dw_pcie_readw_dbi(pci, priv->pcie_ofst + PCI_EXP_LNKCTL2) & PCI_EXP_LNKCTL2_TLS;
	if (tls != PCI_EXP_LNKCTL2_TLS_2_5GT) {
		reg32 = dw_pcie_readl_dbi(pci, PCIE_LINK_WIDTH_SPEED_CONTROL);
		reg32 &= ~PORT_LOGIC_SPEED_CHANGE;
		dw_pcie_writel_dbi(pci, PCIE_LINK_WIDTH_SPEED_CONTROL, reg32);
		reg32 |= PORT_LOGIC_SPEED_CHANGE;
		dw_pcie_writel_dbi(pci, PCIE_LINK_WIDTH_SPEED_CONTROL, reg32);

		for (cnt = 0; cnt < LIN_PCIE_GEN4_TRAININGS; cnt++ ) {
			ret = readl_poll_timeout(pci->dbi_base + PCIE_PORT_DEBUG1, reg32,
					(reg32 & PCIE_PORT_DEBUG1_LINK_UP)
					&& !(reg32 & PCIE_PORT_DEBUG1_LINK_IN_TRAINING),
					LIN_PCIE_GEN4_LINK_TIMEOUT / 100, LIN_PCIE_GEN4_LINK_TIMEOUT);
			if (ret)
				return 0;
		}
	}

	return 0;
}

static const struct dw_pcie_host_ops lin_pcie_rc_host_ops = {
	.host_init = lin_pcie_rc_host_init,
	.host_init2 = lin_pcie_rc_host_init2,
	.host_init3 = lin_pcie_rc_host_init3,
};

static int lin_pcie_link_up(struct dw_pcie *pci)
{
	struct lin_pcie_priv *priv = to_lin_priv_from_pcie(pci);
	u32 val;
	u32 mask;

	val = readl(priv->link_base + IK_PCIE_DL_STS);
	mask = IK_PCIE_DL_STS_DATA_LINKUP_STS;

	return (val & mask) == mask;
}

static int lin_pcie_start_rc_link(struct dw_pcie *pci)
{
	struct lin_pcie_priv *priv = to_lin_priv_from_pcie(pci);
	int ret;
	u32 val;

	if (lin_pcie_link_up(pci))
		return 0;

	ret = readl_poll_timeout(priv->link_base + IK_PCIE_DBG, val,
			!(val & IK_PCIE_DBG_DBI_XFER_PEND), 1, LIN_PCIE_DBI_XFER_PEND_TIMEOUT);
	if (ret) {
		PR_ERR("%d", ret);
		return ret;
	}

	writel(IK_PCIE_EN_LTSSM_ENABLE, priv->link_base + IK_PCIE_EN);

	ret = readl_poll_timeout(priv->link_base + IK_RST_STS, val, val == 0xffffffff, 1,
			LIN_PCIE_DEAD_SLOT_TIMEOUT);
	if (ret) {
		PR_ERR("%d", ret);
		return ret;
	}

	return 0;
}

static int lin_pcie_start_ep_link(struct dw_pcie *pci)
{
	struct lin_pcie_priv *priv = to_lin_priv_from_pcie(pci);

	if (lin_pcie_link_up(pci))
		return 0;

	writel(IK_PCIE_EN_LTSSM_ENABLE, priv->link_base + IK_PCIE_EN);

	return 0;
}

static void lin_pcie_stop_rc_link(struct dw_pcie *pci)
{
	struct lin_pcie_priv *priv = to_lin_priv_from_pcie(pci);
#if !defined(CONFIG_PCIE_CXD_WORKAROUND)
	int ret;
	u32 val;
	u16 reg16;

	if (lin_pcie_link_up(pci)) {
		reg16 = dw_pcie_readw_dbi(pci, PCI_BRIDGE_CONTROL);
		reg16 |= PCI_BRIDGE_CTL_BUS_RESET;
		dw_pcie_writew_dbi(pci, PCI_BRIDGE_CONTROL, reg16);

		ret = readl_poll_timeout(priv->link_base + IK_LTSSM_STS, val,
				(val & IK_LTSSM_STS_STATE) == IK_LTSSM_STS_STATE_HRESET, 1,
				LIN_PCIE_HRESET_TIMEOUT);
		if (ret) {
			PR_ERR("%d", ret);
			goto err_reset;
		}

		reg16 &= ~PCI_BRIDGE_CTL_BUS_RESET;
		dw_pcie_writew_dbi(pci, PCI_BRIDGE_CONTROL, reg16);

		ret = readl_poll_timeout(priv->link_base + IK_RST_REQ_STS, val,
				!(val & IK_RST_REQ_STS_SMLH_REQ_RST), 1, LIN_PCIE_HRESET_TIMEOUT);
		if (ret) {
			PR_ERR("%d", ret);
			goto err;
		}

		writel(IK_PCIE_EN_LTSSM_ENABLE, priv->link_base + IK_PCIE_EN);
	}
#endif /* !CONFIG_PCIE_CXD_WORKAROUND */

	writel(IK_PCIE_EN_LTSSM_DISABLE, priv->link_base + IK_PCIE_EN);

	return;

#if !defined(CONFIG_PCIE_CXD_WORKAROUND)
err_reset:
	reg16 &= ~PCI_BRIDGE_CTL_BUS_RESET;
	dw_pcie_writew_dbi(pci, PCI_BRIDGE_CONTROL, reg16);
err:
	writel(IK_PCIE_EN_LTSSM_ENABLE, priv->link_base + IK_PCIE_EN);
	writel(IK_PCIE_EN_LTSSM_DISABLE, priv->link_base + IK_PCIE_EN);
#endif /* !CONFIG_PCIE_CXD_WORKAROUND */
}

static const struct dw_pcie_ops lin_pcie_ops_rc = {
	.start_link = lin_pcie_start_rc_link,
	.stop_link = lin_pcie_stop_rc_link,
	.link_up = lin_pcie_link_up,
};

static void lin_pcie_stop_ep_link(struct dw_pcie *pci)
{
	struct lin_pcie_priv *priv = to_lin_priv_from_pcie(pci);

	writel(IK_PCIE_EN_LTSSM_DISABLE, priv->link_base + IK_PCIE_EN);
}

static const struct dw_pcie_ops lin_pcie_ops_ep = {
	.start_link = lin_pcie_start_ep_link,
	.stop_link = lin_pcie_stop_ep_link,
	.link_up = lin_pcie_link_up,
};

static int lin_pcie_set_rc_base_hw(struct lin_pcie_priv *priv)
{
	struct device *dev = priv->pci.dev;
	int ret;
	u32 val;
	ktime_t start_time;
	ktime_t end_time;
#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	int cnt;
	int vboost;
#endif





#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	val = readl(priv->link_base + IK_PHY_OTHER_00);
	if (priv->ch == 0) {
		val &= ~IK_PHY_OTHER_00_EQ_AFE_X1_MASK;
		val |= IK_PHY_OTHER_00_EQ_AFE_X1_VAL;
	} else if ((priv->ch == 1) || (priv->ch == 2)) {
		val &= ~IK_PHY_OTHER_00_EQ_AFE_X2_MASK;
		val |= IK_PHY_OTHER_00_EQ_AFE_X2_VAL;
	} else {
		val &= ~IK_PHY_OTHER_00_EQ_AFE_X4_MASK;
		val |= IK_PHY_OTHER_00_EQ_AFE_X4_VAL;
	}
	writel(val, priv->link_base + IK_PHY_OTHER_00);

	val = readl(priv->link_base + IK_PHY_OTHER_02);
	val |= IK_PHY_OTHER_02_REF_USE_PAD;
	writel(val, priv->link_base + IK_PHY_OTHER_02);
#endif
	writel(IK_PCIE_EN_LTSSM_DISABLE, priv->link_base + IK_PCIE_EN);
	writel(IK_PCIE_TYPE_RC, priv->link_base + IK_PCIE_TYPE);
	writel(IK_PCIE_EN_CLR_SEL_LTSSM_EN_CLR_SEL, priv->link_base + IK_PCIE_EN_CLR_SEL);




	pcie_perst_delay(priv->ch);




	writel(IK_PCIE_RST_CNTL_RST_PEX_O,
			priv->link_base + IK_PCIE_RST_CNTL_SET);
	writel(IK_PCIE_RST_CNTL_RST_PEX, priv->link_base + IK_PCIE_RST_CNTL_SET);
	writel(IK_PCIE_RST_CNTL_RST_PONX, priv->link_base + IK_PCIE_RST_CNTL_SET);

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	vboost = pcie_dwc_vboost(priv->ch);
	if (vboost > 7) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		return ret;
	}

	if (priv->phy_fw_upd) {
		val = readl(priv->link_base + IK_PHY_SRAM_CTRL);
		val |= IK_PHY_SRAM_CTRL_PHY_SRAM_BYPASS;
		writel(val, priv->link_base + IK_PHY_SRAM_CTRL);
		val &= ~IK_PHY_SRAM_CTRL_PHY_SRAM_EXT_LD_DONE;
		writel(val, priv->link_base + IK_PHY_SRAM_CTRL);
		val &= ~IK_PHY_SRAM_CTRL_PHY_SRAM_BYPASS;
		writel(val, priv->link_base + IK_PHY_SRAM_CTRL);
	}
#endif

	start_time = ktime_get();

	writel(IK_PHY_RST_CTRL_PHY_RSTX, priv->link_base + IK_PHY_RST_CTRL);
#ifdef CONFIG_ARCH_CXD90XXX_FPGA
	udelay(100);
	writel(0, priv->link_base + IK_PHY_RST_CTRL);
	writel(IK_PHY_RST_CTRL_PHY_RSTX, priv->link_base + IK_PHY_RST_CTRL);
#endif

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	ret = readl_poll_timeout(priv->link_base + IK_PHY_SRAM_CTRL, val, val &
			IK_PHY_SRAM_CTRL_PHY_SRAM_INIT_DONE, 1, LIN_PCIE_PHY_DEASSERT_TIMEOUT * 2);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	if (vboost >= 0) {
		val = readl(priv->phy_base + LIN_PCIE_PHY_SUP_DIG_LVL_OVRD_IN);
		val &= ~LIN_PCIE_PHY_TX_VBOOST_LVL_MASK;
		val |= ((vboost << LIN_PCIE_PHY_TX_VBOOST_LVL_SHIFT) &
				LIN_PCIE_PHY_TX_VBOOST_LVL_MASK) | LIN_PCIE_PHY_TX_VBOOST_LVL_OVRD_EN;
		writel(val, priv->phy_base + LIN_PCIE_PHY_SUP_DIG_LVL_OVRD_IN);
	}

	if (priv->phy_fw_upd) {
		struct firm_info *fw_info;

		fw_info = pcie_dwc_firm(priv->phy_fw_id);
		if (!fw_info) {
			ret = -ENODATA;
			DEV_ERR("%d", ret);
			return ret;
		}
		if (fw_info->n <= 0 || fw_info->n > LIN_PCIE_PHY_SRAM_SIZE / 4) {
			ret = -EINVAL;
			DEV_ERR("%d", ret);
			return ret;
		}

		for (cnt = 0; cnt < fw_info->n; cnt++)
			writel(fw_info->fw[cnt], priv->phy_base + LIN_PCIE_PHY_SRAM + cnt * 4);

		val = readl(priv->link_base + IK_PHY_SRAM_CTRL);
		val |= IK_PHY_SRAM_CTRL_PHY_SRAM_EXT_LD_DONE;
		writel(val, priv->link_base + IK_PHY_SRAM_CTRL);
	}
#endif

	writel(IK_PHY_PIPE_RST_CTRL_PHY_PIPE_RSTX, priv->link_base + IK_PHY_PIPE_RST_CTRL);
	ret = readl_poll_timeout(priv->link_base + IK_RST_STS, val, val == 0xffffffff, 1,
			LIN_PCIE_PIPE_DEASSERT_TIMEOUT * 2);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	end_time = ktime_get();
	dev_notice(dev, "The completion of the phy initialization was detected after ");
	dev_notice(dev, "%lld [ns].\n", end_time - start_time);

	return 0;
}

static int lin_pcie_check_dma_nums(struct lin_pcie_priv *priv)
{
	struct device *dev = priv->pci.dev;
	int ret;
	u32 val;
	u32 wchs;
	u32 rchs;

	val = readl(priv->link_base + IK_PCIE_CONF0);
	wchs = (val & IK_PCIE_CONF0_DMACH_WR) >> IK_PCIE_CONF0_DMACH_WR_SHIFT;
	rchs = (val & IK_PCIE_CONF0_DMACH_RD) >> IK_PCIE_CONF0_DMACH_RD_SHIFT;

	if ((wchs != rchs) || ((wchs != 4) && (wchs != 8))) {
		ret = -ENODATA;
		PR_ERR("%d", ret);
		return ret;
	}

	priv->dma_nums = wchs;

	if (priv->dma_avail_ch & ~GENMASK(priv->dma_nums - 1, 0)) {
		ret = -EINVAL;
		DEV_ERR("%d", ret);
		return ret;
	}

	return 0;
}

static int lin_pcie_start_rc(struct lin_pcie_priv *priv)
{
	struct device *dev = priv->pci.dev;
	int ret;
	struct pci_dev *root_pdev;
#ifdef SAMPLECODE_PME
	struct device *pme_dev;
	struct pcie_device *pme_pedev;
#endif
	struct device *aer_dev;
	struct pcie_device *aer_pedev;
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[priv->ch];
	unsigned long flags;





	ret = reset_control_status(priv->rst);
	if (!ret) {
		priv->omission_of_rc_start = true;
	} else if (ret > 0) {
		priv->omission_of_rc_start = false;
	} else {
		DEV_ERR("%d", ret);
		return ret;
	}

	lin_pcie_init_clkreq(priv);

	raw_spin_lock_irqsave(&ch_info->spin, flags);




	lin_pcie_ch_info[priv->ch].clkreq_ctrl_avail = true;




	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	//Even if clks and reset have been enabled by FW, enable them for clks and reset frameworks
	ret = clk_bulk_prepare_enable(LIN_PCIE_CLK_NUM, priv->clks);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}
	priv->rc_clks_enabled = true;

	ret = reset_control_deassert(priv->rst);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}
	priv->rc_rst_deasserted = true;
	pcie_dwc_notify(dev, priv->ch, PCIE_NOTIFY_START, priv->link_base);

	ret = lin_pcie_check_dma_nums(priv);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	lin_pcie_clear_rc_irq(priv);

	if (!priv->omission_of_rc_start) {
		ret = lin_pcie_set_rc_base_hw(priv);
		if (ret) {
			DEV_ERR("%d", ret);
			return ret;
		}
	}

	ret = dw_pcie_host_init(&priv->pci.pp);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}
	priv->rc_host_inited = true;

	root_pdev = list_first_entry_or_null(&priv->pci.pp.bridge->bus->devices, struct pci_dev,
			bus_list);
	if (root_pdev) {
#ifdef SAMPLECODE_PME
		pme_dev = pcie_port_find_device(root_pdev, PCIE_PORT_SERVICE_PME);
		if (pme_dev) {
			pme_pedev = to_pcie_device(pme_dev);
			if (pme_pedev) {
				priv->pme_irq = pme_pedev->irq;
				if (!priv->pme_irq)
					DEV_ERR();
			} else
				DEV_ERR();
		} else
			DEV_ERR();
#endif

		aer_dev = pcie_port_find_device(root_pdev, PCIE_PORT_SERVICE_AER);
		if (aer_dev) {
			aer_pedev = to_pcie_device(aer_dev);
			if (aer_pedev) {
				priv->aer_irq = aer_pedev->irq;
				if (!priv->aer_irq)
					DEV_ERR();
			} else
				DEV_ERR();
#if 0
		} else
			DEV_ERR();
#else
		}
#endif
	} else
		DEV_ERR();

	lin_pcie_enable_rc_irq(priv);
	priv->rc_irq_enabled = true;

	return 0;
}

static int _lin_pcie_init_clkreq_param(struct lin_pcie_priv *priv, struct lin_pcie_ch_info *ch_info)
{
	struct dw_pcie *pci = &priv->pci;
	struct device *dev = pci->dev;
	struct platform_device *pdev = to_platform_device(dev);
	int ret;
	u32 clkreq;

	ret = of_property_read_u32(pdev->dev.of_node, "clkreq", &clkreq);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}
	priv->clkreq.gpio_ch = clkreq & 0xff;
	priv->clkreq.gpio_bit = (clkreq >> 8) & 0xff;

	if (!ch_info->clkreq_mode) {
		ret = gpio_get_mode_bit(priv->clkreq.gpio_ch, priv->clkreq.gpio_bit, &priv->clkreq.mode);
		if (ret) {
			DEV_ERR("%d", ret);
			return ret;
		}
		ch_info->clkreq_mode = priv->clkreq.mode;
	} else {
		priv->clkreq.mode = ch_info->clkreq_mode;
	}

	return 0;
}


static int lin_pcie_init_rc_priv(struct lin_pcie_priv *priv, bool start_istr)
{
	struct dw_pcie *pci = &priv->pci;
	struct device *dev = pci->dev;
	struct platform_device *pdev = to_platform_device(dev);
	struct resource *res;
	int ret;
	struct property *prop;
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[priv->ch];
	unsigned long flags;





	priv->pci.ops = &lin_pcie_ops_rc;
	priv->pci.pp.ops = &lin_pcie_rc_host_ops;
	priv->pci.pp.num_vectors = LIN_PCIE_MSI_IRQS;

	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "link");
	if (!res) {
		ret = -ENODATA;
		DEV_ERR("%d", ret);
		return ret;
	}

	priv->link_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(priv->link_base)) {
		ret = PTR_ERR(priv->link_base);
		DEV_ERR("%d", ret);
		return ret;
	}

	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "dma");
	if (!res) {
		ret = -ENODATA;
		DEV_ERR("%d", ret);
		return ret;
	}

	priv->dma_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(priv->dma_base)) {
		ret = PTR_ERR(priv->dma_base);
		DEV_ERR("%d", ret);
		return ret;
	}

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "phy");
	if (!res) {
		ret = -ENODATA;
		DEV_ERR("%d", ret);
		return ret;
	}

	priv->phy_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(priv->phy_base)) {
		ret = PTR_ERR(priv->phy_base);
		DEV_ERR("%d", ret);
		return ret;
	}
#endif

	priv->clks[LIN_PCIE_CLK_M_ACLK].id = "m_aclk";
	priv->clks[LIN_PCIE_CLK_S_ACLK].id = "s_aclk";
	priv->clks[LIN_PCIE_CLK_DBI_ACLK].id = "dbi_aclk";
	priv->clks[LIN_PCIE_CLK_AUX_CLK].id = "aux_clk";
	priv->clks[LIN_PCIE_CLK_HCLK].id = "hclk";
	priv->clks[LIN_PCIE_CLK_P_CR_CLK].id = "p_cr_clk";
	priv->clks[LIN_PCIE_CLK_PCLK].id = "pclk";
	ret = devm_clk_bulk_get(dev, LIN_PCIE_CLK_NUM, priv->clks);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	priv->rst = devm_reset_control_get_exclusive(dev, NULL);
	if (IS_ERR(priv->rst)) {
		ret = PTR_ERR(priv->rst);
		DEV_ERR("%d", ret);
		return ret;
	}

	ret = of_property_read_u32(pdev->dev.of_node, "available-dmach", &priv->dma_avail_ch);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	ret = lin_pcie_build_rc_irq(priv);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}
	priv->rc_irq_built = true;

	if (!start_istr) {
		prop = of_find_property(dev->of_node, "skip-auto-probe", NULL);
		if (prop)
			priv->omission_of_rc_probe = true;
		else
			priv->omission_of_rc_probe = false;
	} else {
		priv->omission_of_rc_probe = false;
	}

	prop = of_find_property(dev->of_node, "ignore-suspend", NULL);
	if (prop)
		priv->omission_of_rc_suspend = true;
	else
		priv->omission_of_rc_suspend = false;

	ret = of_property_read_u32(pdev->dev.of_node, "firm-id", &priv->phy_fw_id);
	if (!ret)
		priv->phy_fw_upd = true;
	else if (ret == -EINVAL)
		priv->phy_fw_upd = false;
	else {
		DEV_ERR("%d", ret);
		return ret;
	}

	prop = of_find_property(dev->of_node, "pci32", NULL);
	if (prop)
		priv->pci32 = true;
	else
		priv->pci32 = false;

	raw_spin_lock_irqsave(&ch_info->spin, flags);




	ret = _lin_pcie_init_clkreq_param(priv, ch_info);




	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}
	priv->rc_clkreq_param_inited = true;

	platform_set_drvdata(pdev, priv);

	return 0;
}

static void lin_pcie_enter_l23(struct lin_pcie_priv *priv)
{
	int ret;
	u32 val;
	struct device *dev = priv->pci.dev;

	if (!dw_pcie_link_up(&priv->pci)) {
		u16 reg16;
		unsigned long flags;
		raw_spin_lock_irqsave(&lin_pcie_ch_info[priv->ch].config_spin, flags);
		reg16 = dw_pcie_readw_dbi(&priv->pci, priv->pcie_ofst + PCI_EXP_LNKCTL);
		reg16 |= PCI_EXP_LNKCTL_LD;
		dw_pcie_writew_dbi(&priv->pci, priv->pcie_ofst + PCI_EXP_LNKCTL, reg16);
		raw_spin_unlock_irqrestore(&lin_pcie_ch_info[priv->ch].config_spin, flags);
		return;
	}

	writel(IK_IRQ_COM_TURNOFF_ACK, priv->link_base + IK_IRQ_COM_CLR);
	writel(IK_MSG_TURNOFF_FUNC0, priv->link_base + IK_MSG_TURNOFF_REQ);

	ret = readl_poll_timeout(priv->link_base + IK_IRQ_COM_RAW, val,
			val & IK_IRQ_COM_TURNOFF_ACK, 1, LIN_PCIE_TURNOFF_ACK_TIMEOUT);
	if (ret) {
		DEV_ERR("%d", ret);
		goto err;
	}

	ret = readl_poll_timeout(priv->link_base + IK_LTSSM_STS, val,
			!dw_pcie_link_up(&priv->pci) || ((val & IK_LTSSM_STS_STATE) == IK_LTSSM_STS_STATE_L2I),
			LIN_PCIE_L23_TIMEOUT / 100, LIN_PCIE_L23_TIMEOUT);
	if (ret || ((val & IK_LTSSM_STS_STATE) != IK_LTSSM_STS_STATE_L2I)) {
		DEV_ERR("%d", ret);
		goto err;
	}

	writel(IK_MSG_TURNOFF_FUNC0, priv->link_base + IK_MSG_TURNOFF_CLR);

	return;

err:
	writel(IK_MSG_TURNOFF_FUNC0, priv->link_base + IK_MSG_TURNOFF_CLR);
	lin_pcie_stop_rc_link(&priv->pci);
	DEV_ERR();
	return;
}

static void lin_pcie_term_rc(struct lin_pcie_priv *priv)
{
	struct pcie_port *pp;
	struct lin_pcie_ch_info *ch_info;
	unsigned long flags;





	if (!priv)
		return;

	ch_info = &lin_pcie_ch_info[priv->ch];
	pp = &priv->pci.pp;

	if (priv->rc_irq_enabled) {
		lin_pcie_disable_rc_irq(priv);
		priv->rc_irq_enabled = false;
	}

	if (priv->rc_host_inited) {
		dw_pcie_host_deinit(&priv->pci.pp);
		lin_pcie_enter_l23(priv);
		priv->rc_host_inited = false;
	}





	if (priv->rc_irq_built) {
		irq_set_chained_handler_and_data(pp->irq, NULL, NULL);
		priv->rc_irq_built = false;
	}

	if (priv->lgc_irqd) {
		irq_hw_number_t hwirq;
		unsigned int virq;

		for (hwirq = 0; hwirq < PCI_NUM_INTX; hwirq++) {
			virq = irq_find_mapping(priv->lgc_irqd, hwirq);
			if (virq > 0) {
				irq_dispose_mapping(virq);
			}
		}

		irq_domain_remove(priv->lgc_irqd);
		priv->lgc_irqd = NULL;
	}

	if (priv->rc_rst_deasserted) {
		u32 val;
		int ret;

		ret = readl_poll_timeout(priv->link_base + IK_PCIE_DBG, val,
				 !(val & IK_PCIE_DBG_XFER_PEND), 1, LIN_PCIE_XFER_PEND_TIMEOUT);
		if (ret) {
			PR_ERR("%d", ret);
		}

#ifdef CONFIG_ARCH_CXD90XXX_FPGA
		writel(IK_PCIE_RST_CNTL_RST_PEX_O | IK_PCIE_RST_CNTL_RST_PEX | IK_PCIE_RST_CNTL_RST_PONX,
				priv->link_base + IK_PCIE_RST_CNTL_CLR);
#endif

		pcie_dwc_notify(priv->pci.dev, priv->ch, PCIE_NOTIFY_TERM, priv->link_base);
		reset_control_assert(priv->rst);
		priv->rc_rst_deasserted = false;
	}

	if (priv->rc_clks_enabled) {
		clk_bulk_disable_unprepare(LIN_PCIE_CLK_NUM, priv->clks);
		priv->rc_clks_enabled = false;
	}

	if (priv->rc_clkreq_param_inited) {
		raw_spin_lock_irqsave(&ch_info->spin, flags);




		lin_pcie_ch_info[priv->ch].clkreq_ctrl_avail = false;




		raw_spin_unlock_irqrestore(&ch_info->spin, flags);





#if 0
		lin_pcie_fix_clkreq_to_low(priv, false);
#else
		lin_pcie_fix_clkreq_to_hiz(priv); /* CLKREQ:Hi-Z */
#endif
		priv->rc_clkreq_param_inited = false;
	}
}

static irqreturn_t lin_pcie_ep_cnct_irq_fn(int irq, void *context)
{
	struct lin_pcie_priv *priv = (struct lin_pcie_priv *)context;
	struct dw_pcie *pci = &priv->pci;
	struct device *dev = pci->dev;
	struct dw_pcie_ep *ep = &pci->ep;
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[priv->ch];
	u32 val;
	enum lin_pcie_rc_state state;










	val = readl(priv->link_base + IK_IRQ_CNCT_STS);
	if (!val) {








		return IRQ_HANDLED;
	}

	if (val & IK_IRQ_CNCT_PERSTX_NEG) {
		writel(IK_IRQ_CNCT_PERSTX_NEG, priv->link_base + IK_IRQ_CNCT_CLR);
		val &= ~IK_IRQ_CNCT_PERSTX_NEG;

		schedule_work(&priv->update_ep);
	}

	if (val & IK_IRQ_CNCT_PERSTX_POS) {
		writel(IK_IRQ_CNCT_PERSTX_POS, priv->link_base + IK_IRQ_CNCT_CLR);
		val &= ~IK_IRQ_CNCT_PERSTX_POS;

		schedule_work(&priv->update_ep);
	}

	if (val & IK_IRQ_CNCT_DL_UP_NEG) {
		writel(IK_IRQ_CNCT_DL_UP_NEG, priv->link_base + IK_IRQ_CNCT_CLR);
		dev_notice(dev, "Link Down was detected on PCIe EP.\n");
		val &= ~IK_IRQ_CNCT_DL_UP_NEG;
	}

	if (val & IK_IRQ_CNCT_DL_UP_POS) {
		writel(IK_IRQ_CNCT_DL_UP_POS, priv->link_base + IK_IRQ_CNCT_CLR);
		raw_spin_lock(&ch_info->spin);



		state = READ_ONCE(ch_info->state);
		if (state == LIN_PCIE_STATE_EP_START )
			dw_pcie_ep_linkup(ep);



		raw_spin_unlock(&ch_info->spin);







		val &= ~IK_IRQ_CNCT_DL_UP_POS;
	}

	if (val & IK_IRQ_CNCT_MSTR_EN_POS) {
		writel(IK_IRQ_CNCT_MSTR_EN_POS, priv->link_base + IK_IRQ_CNCT_CLR);
		dev_notice(dev, "Master Enable was detected on PCIe EP.\n");
		val &= ~IK_IRQ_CNCT_MSTR_EN_POS;
	}

	if (val) {
		DEV_ERR("%d", val);
		writel(val, priv->link_base + IK_IRQ_CNCT_CLR);
	}






	return IRQ_HANDLED;
}

static irqreturn_t lin_pcie_ep_com_irq_fn(int irq, void *context)
{
	struct lin_pcie_priv *priv = (struct lin_pcie_priv *)context;
	u32 val;








	val = readl(priv->link_base + IK_IRQ_COM_STS);
	if (!(val & IK_IRQ_COM_TURNOFF_REQ)) {








		return IRQ_HANDLED;
	}

	if (val & IK_IRQ_COM_TURNOFF_REQ) {
		writel(IK_IRQ_COM_TURNOFF_REQ, priv->link_base + IK_IRQ_COM_CLR);
		writel(IK_MSG_ENTRY_L23_FUNC0, priv->link_base + IK_MSG_ENTRY_L23_REQ);
	}






	return IRQ_HANDLED;
}

static int lin_pcie_build_ep_irq(struct lin_pcie_priv *priv)
{
	struct dw_pcie *pci = &priv->pci;
	struct device *dev = pci->dev;
	struct platform_device *pdev = to_platform_device(dev);
	int ret;

	priv->cnct_irq = platform_get_irq_byname(pdev, "cnct");
	if (priv->cnct_irq < 0) {
		ret = priv->cnct_irq;
		DEV_ERR("%d", ret);
		return ret;
	}

	irq_set_status_flags(priv->cnct_irq, IRQ_NOAUTOEN);

	ret = devm_request_irq(dev, priv->cnct_irq, lin_pcie_ep_cnct_irq_fn, IRQF_TRIGGER_HIGH |
			IRQF_NO_THREAD, "lin-pcie-ep-cnct", priv);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	priv->com_irq = platform_get_irq_byname(pdev, "com");
	if (priv->com_irq < 0) {
		ret = priv->com_irq;
		DEV_ERR("%d", ret);
		return ret;
	}

	irq_set_status_flags(priv->com_irq, IRQ_NOAUTOEN);

	ret = devm_request_irq(dev, priv->com_irq, lin_pcie_ep_com_irq_fn, IRQF_TRIGGER_HIGH |
			IRQF_NO_THREAD, "lin-pcie-ep-com", priv);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	ret = lin_pcie_build_irq_for_dma(priv);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	return 0;
}

static void lin_pcie_clear_ep_irq(struct lin_pcie_priv *priv)
{
	writel(0xffffffff, priv->link_base + IK_IRQ_CNCT_CLR);
	writel(0xffffffff, priv->link_base + IK_IRQ_COM_CLR);
}

static void lin_pcie_enable_ep_irq(struct lin_pcie_priv *priv)
{
	writel(IK_IRQ_CNCT_DL_UP_NEG | IK_IRQ_CNCT_DL_UP_POS | IK_IRQ_CNCT_MSTR_EN_POS |
			IK_IRQ_CNCT_PERSTX_NEG | IK_IRQ_CNCT_PERSTX_POS,
			priv->link_base + IK_IRQ_CNCT_EN);
	enable_irq(priv->cnct_irq);

	writel(IK_IRQ_COM_TURNOFF_REQ, priv->link_base + IK_IRQ_COM_EN);
	enable_irq(priv->com_irq);
}

static void lin_pcie_disable_ep_irq(struct lin_pcie_priv *priv)
{
	disable_irq(priv->com_irq);
	writel(0, priv->link_base + IK_IRQ_COM_EN);

	disable_irq(priv->cnct_irq);
	writel(0, priv->link_base + IK_IRQ_CNCT_EN);
}

static void lin_pcie_term_ep(struct lin_pcie_priv *priv)
{
	if (!priv)
		return;

	if (priv->ep_irq_enabled) {
		lin_pcie_disable_ep_irq(priv);
		priv->ep_irq_enabled = false;
	}

	if (priv->ep_wq_inited) {
		cancel_work_sync(&priv->update_ep);
		priv->ep_wq_inited = false;
	}

	if (priv->ep_inited) {
		dw_pcie_ep_exit2(&priv->pci.ep);
		priv->ep_inited = false;
	}

	if (priv->ep_hw_set)
		priv->ep_hw_set = false;

	if (priv->ep_rst_deasserted) {
		reset_control_assert(priv->rst);
		priv->ep_rst_deasserted = false;
	}

	if (priv->ep_clks_enabled) {
		clk_bulk_disable_unprepare(LIN_PCIE_CLK_NUM, priv->clks);
		priv->ep_clks_enabled = false;
	}
}

static void lin_pcie_term(struct lin_pcie_priv *priv)
{
	if (!priv)
		return;

	if (priv->mode == DW_PCIE_RC_TYPE)
		lin_pcie_term_rc(priv);
	else
		lin_pcie_term_ep(priv);
}

static int lin_pcie_probe_rc(struct platform_device *pdev, int ch)
{
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[ch];
	struct lin_pcie_priv *priv;
	struct device *dev = &pdev->dev;
	int ret;
	unsigned long flags;
	bool start_istr;





	priv = devm_kzalloc(dev, sizeof(*priv), GFP_KERNEL);
	if (!priv) {
		ret = -ENOMEM;
		DEV_ERR("%d", ret);
		goto err;
	}

	priv->mode = DW_PCIE_RC_TYPE;
	priv->ch = ch;
	priv->pci.dev = &pdev->dev;

	raw_spin_lock_irqsave(&ch_info->spin, flags);




	start_istr = READ_ONCE(ch_info->start_istr);




	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	ret = lin_pcie_init_rc_priv(priv, start_istr);
	if (ret) {
		DEV_ERR("%d", ret);
		goto err;
	}
	raw_spin_lock_irqsave(&ch_info->spin, flags);




	WRITE_ONCE(ch_info->priv, priv);




	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	if (priv->omission_of_rc_probe) {
		raw_spin_lock_irqsave(&ch_info->spin, flags);



		WRITE_ONCE(ch_info->mode, DW_PCIE_RC_TYPE);
		WRITE_ONCE(ch_info->state, LIN_PCIE_STATE_RC_PREP);



		raw_spin_unlock_irqrestore(&ch_info->spin, flags);





		lin_pcie_fix_clkreq_to_hiz(priv); /* CLKREQ:Hi-Z*/
		return 0;
	}

	ret = lin_pcie_start_rc(priv);
	if (ret) {
		DEV_ERR("%d", ret);
		goto err;
	}

	raw_spin_lock_irqsave(&ch_info->spin, flags);



	WRITE_ONCE(ch_info->mode, DW_PCIE_RC_TYPE);
	WRITE_ONCE(ch_info->state, LIN_PCIE_STATE_RC_START);
	_lin_pcie_make_dma_available(priv);



	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	return 0;

err:
	lin_pcie_term_rc(priv);
	DEV_ERR("%d", ret);
	return ret;
}

static void lin_pcie_ep_init(struct dw_pcie_ep *ep)
{
	struct dw_pcie *pci = to_dw_pcie_from_ep(ep);
	enum pci_barno bar;

	for (bar = BAR_0; bar <= PCI_STD_NUM_BARS; bar++)
		dw_pcie_ep_reset_bar(pci, bar);
}

static void lin_pcie_raise_legacy_irq(struct dw_pcie_ep *ep)
{
	struct dw_pcie *pci = to_dw_pcie_from_ep(ep);
	struct lin_pcie_priv *priv = to_lin_priv_from_pcie(pci);
	u32 val;

	val = dw_pcie_readl_dbi(pci, PCI_COMMAND);
	val &= ~PCI_COMMAND_INTX_DISABLE;
	dw_pcie_writel_dbi(pci, PCI_COMMAND, val);

	writel(IK_MSG_INTX_SEL_MSG_INTX_SEL, priv->link_base + IK_MSG_INTX_SEL);
	writel(IK_MSG_INTX_REQ_MSG_INTX_REQ, priv->link_base + IK_MSG_INTX_REQ);
	mdelay(1);
	writel(IK_MSG_INTX_CLR_MSG_INTX_CLR, priv->link_base + IK_MSG_INTX_CLR);
}

static int lin_pcie_raise_irq(struct dw_pcie_ep *ep, u8 func_no,
		enum pci_epc_irq_type type, u16 interrupt_num)
{
	struct dw_pcie *pci = to_dw_pcie_from_ep(ep);
	struct device *dev = pci->dev;
	int ret;

	switch (type) {
	case PCI_EPC_IRQ_LEGACY:
		lin_pcie_raise_legacy_irq(ep);
		break;
	case PCI_EPC_IRQ_MSI:
		ret = dw_pcie_ep_raise_msi_irq(ep, func_no, interrupt_num);
		if (ret) {
			DEV_ERR("%d", ret);
			return ret;
		}
		break;
	default:
		ret = -EBADRQC;
		DEV_ERR("%d", ret);
		return ret;
	}

	return 0;
}

static const struct pci_epc_features lin_pci_epc_features = {
	.linkup_notifier = true,
	.msi_capable = true,
};

static const struct pci_epc_features *lin_pcie_get_features(struct dw_pcie_ep *ep)
{
	return &lin_pci_epc_features;
}

static const struct dw_pcie_ep_ops lin_pcie_ep_ops = {
	.ep_init = lin_pcie_ep_init,
	.raise_irq = lin_pcie_raise_irq,
	.get_features = lin_pcie_get_features,
};

static int lin_pcie_set_ep_base_hw(struct lin_pcie_priv *priv)
{
	struct device *dev = priv->pci.dev;
	int ret;
	u32 val;
	ktime_t start_time;
	ktime_t end_time;
#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	int cnt;
	int vboost;
#endif

	ret = readl_poll_timeout(priv->link_base + IK_RST_STS, val, val & IK_RST_STS_PERST_N_EXT,
			1, LIN_PCIE_EP_PERST_TIMEOUT);
	if (ret == -ETIMEDOUT)
		return 1;
	else if (ret) {
		DEV_ERR("%d", ret);
		goto err;
	}

	writel(IK_PCIE_RST_CNTL_RST_PEX, priv->link_base + IK_PCIE_RST_CNTL_SET);
	writel(IK_PCIE_RST_CNTL_RST_PONX, priv->link_base + IK_PCIE_RST_CNTL_SET);

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	vboost = pcie_dwc_vboost(priv->ch);
	if (vboost > 7) {
		ret = -EINVAL;
		PR_ERR("%d", ret);
		goto err;
	}

	if (priv->phy_fw_upd) {
		val = readl(priv->link_base + IK_PHY_SRAM_CTRL);
		val |= IK_PHY_SRAM_CTRL_PHY_SRAM_BYPASS;
		writel(val, priv->link_base + IK_PHY_SRAM_CTRL);
		val &= ~IK_PHY_SRAM_CTRL_PHY_SRAM_EXT_LD_DONE;
		writel(val, priv->link_base + IK_PHY_SRAM_CTRL);
		val &= ~IK_PHY_SRAM_CTRL_PHY_SRAM_BYPASS;
		writel(val, priv->link_base + IK_PHY_SRAM_CTRL);
	}
#endif

	start_time = ktime_get();

	writel(IK_PHY_RST_CTRL_PHY_RSTX, priv->link_base + IK_PHY_RST_CTRL);
#ifdef CONFIG_ARCH_CXD90XXX_FPGA
	udelay(100);
	writel(0, priv->link_base + IK_PHY_RST_CTRL);
	writel(IK_PHY_RST_CTRL_PHY_RSTX, priv->link_base + IK_PHY_RST_CTRL);
#endif

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	ret = readl_poll_timeout(priv->link_base + IK_PHY_SRAM_CTRL, val, val &
			IK_PHY_SRAM_CTRL_PHY_SRAM_INIT_DONE, 1, LIN_PCIE_PHY_DEASSERT_TIMEOUT * 2);
	if (ret) {
		DEV_ERR("%d", ret);
		goto err;
	}

	if (vboost >= 0) {
		val = readl(priv->phy_base + LIN_PCIE_PHY_SUP_DIG_LVL_OVRD_IN);
		val &= ~LIN_PCIE_PHY_TX_VBOOST_LVL_MASK;
		val |= ((vboost << LIN_PCIE_PHY_TX_VBOOST_LVL_SHIFT) &
				LIN_PCIE_PHY_TX_VBOOST_LVL_MASK) | LIN_PCIE_PHY_TX_VBOOST_LVL_OVRD_EN;
		writel(val, priv->phy_base + LIN_PCIE_PHY_SUP_DIG_LVL_OVRD_IN);
	}

	if (priv->phy_fw_upd) {
		struct firm_info *fw_info;

		fw_info = pcie_dwc_firm(priv->phy_fw_id);
		if (!fw_info) {
			ret = -ENODATA;
			DEV_ERR("%d", ret);
			goto err;
		}
		if (fw_info->n <= 0 || fw_info->n > LIN_PCIE_PHY_SRAM_SIZE / 4) {
			ret = -EINVAL;
			DEV_ERR("%d", ret);
			goto err;
		}

		for (cnt = 0; cnt < fw_info->n; cnt++) {
			writel(fw_info->fw[cnt],
					priv->phy_base + LIN_PCIE_PHY_SRAM + cnt * 4);
		}

		val = readl(priv->link_base + IK_PHY_SRAM_CTRL);
		val |= IK_PHY_SRAM_CTRL_PHY_SRAM_EXT_LD_DONE;
		writel(val, priv->link_base + IK_PHY_SRAM_CTRL);
	}
#endif

	writel(IK_PHY_PIPE_RST_CTRL_PHY_PIPE_RSTX, priv->link_base + IK_PHY_PIPE_RST_CTRL);
	ret = readl_poll_timeout(priv->link_base + IK_RST_STS, val, val == 0xffffffff, 1,
			LIN_PCIE_PIPE_DEASSERT_TIMEOUT * 2);
	if (ret) {
		DEV_ERR("%d", ret);
		goto err;
	}

	end_time = ktime_get();
	dev_notice(dev, "The completion of the phy initialization was detected after ");
	dev_notice(dev, "%lld [ns].\n", end_time - start_time);

	return 0;

err:
	writel(IK_PCIE_EN_LTSSM_DISABLE, priv->link_base + IK_PCIE_EN);
	writel(IK_PCIE_RST_CNTL_RST_PEX | IK_PCIE_RST_CNTL_RST_PONX,
			priv->link_base + IK_PCIE_RST_CNTL_CLR);
	writel(IK_MSG_ENTRY_L23_FUNC0, priv->link_base + IK_MSG_ENTRY_L23_CLR);
	DEV_ERR("%d", ret);
	return ret;
}

static void lin_pcie_update_ep(struct work_struct *work)
{
	struct lin_pcie_priv *priv = container_of(work, struct lin_pcie_priv, update_ep);
	struct device *dev = priv->pci.dev;
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[priv->ch];
	unsigned long flags;
	enum lin_pcie_rc_state state;
	int ret;





	raw_spin_lock_irqsave(&ch_info->spin, flags);



	state = READ_ONCE(ch_info->state);
	if (state == LIN_PCIE_STATE_EP_START) {
		_lin_pcie_make_dma_unavailable(priv);
		WRITE_ONCE(ch_info->state, LIN_PCIE_STATE_EP_STOP);
	}



	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	if (state == LIN_PCIE_STATE_EP_START) {
		dw_pcie_ep_exit2(&priv->pci.ep);
		priv->ep_inited = false;

		writel(IK_PCIE_EN_LTSSM_DISABLE, priv->link_base + IK_PCIE_EN);
		writel(IK_PCIE_RST_CNTL_RST_PEX | IK_PCIE_RST_CNTL_RST_PONX,
				priv->link_base + IK_PCIE_RST_CNTL_CLR);
		writel(IK_MSG_ENTRY_L23_FUNC0, priv->link_base + IK_MSG_ENTRY_L23_CLR);
		priv->ep_hw_set = false;
	} else if(state != LIN_PCIE_STATE_EP_STOP) {
		goto err;
	}

	ret = lin_pcie_set_ep_base_hw(priv);
	if (ret == 1)
		return;
	else if (ret) {
		DEV_ERR("%d", ret);
		goto err;
	}
	priv->ep_hw_set = true;

	ret = dw_pcie_ep_init(&priv->pci.ep);
	if (ret) {
		DEV_ERR("%d", ret);
		goto err;
	}
	priv->ep_inited = true;

	raw_spin_lock_irqsave(&ch_info->spin, flags);



	state = READ_ONCE(ch_info->state);
	if (state == LIN_PCIE_STATE_EP_STOP) {
		WRITE_ONCE(ch_info->state, LIN_PCIE_STATE_EP_START);
		_lin_pcie_make_dma_available(priv);
	}



	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	if (state != LIN_PCIE_STATE_EP_STOP)
		goto err;

	return;

err:
	if (priv->ep_inited) {
		dw_pcie_ep_exit2(&priv->pci.ep);
		priv->ep_inited = false;
	}
	if (priv->ep_hw_set) {
		writel(IK_PCIE_EN_LTSSM_DISABLE, priv->link_base + IK_PCIE_EN);
		writel(IK_PCIE_RST_CNTL_RST_PEX | IK_PCIE_RST_CNTL_RST_PONX,
				priv->link_base + IK_PCIE_RST_CNTL_CLR);
		writel(IK_MSG_ENTRY_L23_FUNC0, priv->link_base + IK_MSG_ENTRY_L23_CLR);
		priv->ep_hw_set = false;
	}
}

static int lin_pcie_init_ep_priv(struct lin_pcie_priv *priv)
{
	struct dw_pcie *pci = &priv->pci;
	struct device *dev = pci->dev;
	struct platform_device *pdev = to_platform_device(dev);
	struct resource *res;
	int ret;
	int lenp;
	const __be32 *addr;

	priv->pci.ops = &lin_pcie_ops_ep;
	priv->pci.ep.ops = &lin_pcie_ep_ops;

	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "link");
	if (!res) {
		ret = -ENOENT;
		DEV_ERR("%d", ret);
		return ret;
	}

	priv->link_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(priv->link_base)) {
		ret = PTR_ERR(priv->link_base);
		DEV_ERR("%d", ret);
		return ret;
	}

	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "dma");
	if (!res) {
		ret = -ENODATA;
		DEV_ERR("%d", ret);
		return ret;
	}

	priv->dma_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(priv->dma_base)) {
		ret = PTR_ERR(priv->dma_base);
		DEV_ERR("%d", ret);
		return ret;
	}

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "phy");
	if (!res) {
		ret = -ENODATA;
		DEV_ERR("%d", ret);
		return ret;
	}

	priv->phy_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(priv->phy_base)) {
		ret = PTR_ERR(priv->phy_base);
		DEV_ERR("%d", ret);
		return ret;
	}
#endif

	priv->clks[LIN_PCIE_CLK_M_ACLK].id = "m_aclk";
	priv->clks[LIN_PCIE_CLK_S_ACLK].id = "s_aclk";
	priv->clks[LIN_PCIE_CLK_DBI_ACLK].id = "dbi_aclk";
	priv->clks[LIN_PCIE_CLK_AUX_CLK].id = "aux_clk";
	priv->clks[LIN_PCIE_CLK_HCLK].id = "hclk";
	priv->clks[LIN_PCIE_CLK_P_CR_CLK].id = "p_cr_clk";
	priv->clks[LIN_PCIE_CLK_PCLK].id = "pclk";
	ret = devm_clk_bulk_get(dev, LIN_PCIE_CLK_NUM, priv->clks);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	priv->rst = devm_reset_control_get_exclusive(dev, NULL);
	if (IS_ERR(priv->rst)) {
		ret = PTR_ERR(priv->rst);
		DEV_ERR("%d", ret);
		return ret;
	}

	ret = of_property_read_u32(pdev->dev.of_node, "available-dmach", &priv->dma_avail_ch);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	ret = lin_pcie_build_ep_irq(priv);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	ret = of_property_read_u32(pdev->dev.of_node, "firm-id", &priv->phy_fw_id);
	if (!ret)
		priv->phy_fw_upd = true;
	else if (ret == -EINVAL)
		priv->phy_fw_upd = false;
	else {
		DEV_ERR("%d", ret);
		return ret;
	}

	addr = of_get_property(pdev->dev.of_node, "mp-pcie-dram-inc", &lenp);
	if (addr) {
		if (lenp != 8) {
			ret = -EINVAL;
			DEV_ERR("%d", ret);
			return ret;
		}

		pci->ep.mp_pcie_dram_inc = (u64)be32_to_cpup(addr + 0) << 32;
		pci->ep.mp_pcie_dram_inc |= be32_to_cpup(addr + 1);
	}

	INIT_WORK(&priv->update_ep, lin_pcie_update_ep);
	priv->ep_wq_inited = true;

	platform_set_drvdata(pdev, priv);

	return 0;
}

static int lin_pcie_init_ep_hw(struct lin_pcie_priv *priv)
{
	struct device *dev = priv->pci.dev;
	int ret;
	u32 val;

	lin_pcie_clear_ep_irq(priv);

	ret = clk_bulk_prepare_enable(LIN_PCIE_CLK_NUM, priv->clks);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}
	priv->ep_clks_enabled = true;

	ret = reset_control_deassert(priv->rst);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}
	priv->ep_rst_deasserted = true;

	ret = lin_pcie_check_dma_nums(priv);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	val = readl(priv->link_base + IK_PHY_OTHER_00);
	if (priv->ch == 0) {
		val &= ~IK_PHY_OTHER_00_EQ_AFE_X1_MASK;
		val |= IK_PHY_OTHER_00_EQ_AFE_X1_VAL;
	} else if ((priv->ch == 1) || (priv->ch == 2)) {
		val &= ~IK_PHY_OTHER_00_EQ_AFE_X2_MASK;
		val |= IK_PHY_OTHER_00_EQ_AFE_X2_VAL;
	} else {
		val &= ~IK_PHY_OTHER_00_EQ_AFE_X4_MASK;
		val |= IK_PHY_OTHER_00_EQ_AFE_X4_VAL;
	}
	writel(val, priv->link_base + IK_PHY_OTHER_00);

	val = readl(priv->link_base + IK_PHY_OTHER_02);
	val |= IK_PHY_OTHER_02_REF_USE_PAD;
	writel(val, priv->link_base + IK_PHY_OTHER_02);
#endif
	writel(IK_PCIE_EN_LTSSM_DISABLE, priv->link_base + IK_PCIE_EN);
	writel(IK_PCIE_TYPE_EP, priv->link_base + IK_PCIE_TYPE);

	ret = readl_poll_timeout(priv->link_base + IK_RST_STS, val, val & IK_RST_STS_PERST_N_EXT,
			1, LIN_PCIE_EP_PERST_TIMEOUT);
	if (ret == -ETIMEDOUT)
		return 1;
	else if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	return 0;
}

static int lin_pcie_probe_ep(struct platform_device *pdev, int ch)
{
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[ch];
	struct device *dev = &pdev->dev;
	struct lin_pcie_priv *priv;
	unsigned long flags;
	int ret;
	bool update;





	priv = devm_kzalloc(dev, sizeof(*priv), GFP_KERNEL);
	if (!priv) {
		ret = -ENOMEM;
		DEV_ERR("%d", ret);
		goto err;
	}

	priv->mode = DW_PCIE_EP_TYPE;
	priv->ch = ch;
	priv->pci.dev = &pdev->dev;

	ret = lin_pcie_init_ep_priv(priv);
	if (ret) {
		DEV_ERR("%d", ret);
		goto err;
	}

	update = false;
	ret = lin_pcie_init_ep_hw(priv);
	if (!ret)
		update = true;
	else if (ret != 1) {
		DEV_ERR("%d", ret);
		goto err;
	}

	raw_spin_lock_irqsave(&ch_info->spin, flags);



	WRITE_ONCE(ch_info->priv, priv);
	WRITE_ONCE(ch_info->mode, DW_PCIE_EP_TYPE);
	WRITE_ONCE(ch_info->state, LIN_PCIE_STATE_EP_STOP);



	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	if (update)
		schedule_work(&priv->update_ep);
	lin_pcie_enable_ep_irq(priv);
	priv->ep_irq_enabled = true;

	return 0;

err:
	lin_pcie_term_ep(priv);
	DEV_ERR("%d", ret);
	return ret;
}

struct lin_pcie_of_data {
	enum dw_pcie_device_mode mode;
};

static int _lin_pcie_probe(struct platform_device *pdev, int ch)
{
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[ch];
	struct lin_pcie_priv *priv = NULL;
	struct device *dev = &pdev->dev;
	int ret;
	unsigned long flags;
	enum lin_pcie_rc_state state;
	const struct lin_pcie_of_data *of_data;





	of_data = of_device_get_match_data(dev);
	if (!of_data) {
		ret = -ENODATA;
		DEV_ERR("%d", ret);
		goto err;
	}
	if ((of_data->mode != DW_PCIE_RC_TYPE) && (of_data->mode != DW_PCIE_EP_TYPE)) {
		ret = -EBADRQC;
		DEV_ERR("%d", ret);
		goto err;
	}

	raw_spin_lock_irqsave(&ch_info->spin, flags);




	state = READ_ONCE(ch_info->state);
	priv = READ_ONCE(ch_info->priv);

	dev = READ_ONCE(ch_info->dev);
	if (!dev) {
		dev = get_device(&pdev->dev);
		WRITE_ONCE(ch_info->dev, dev);
	} else if (dev != &pdev->dev) {
		DEV_ERR("%d", ret);
		put_device(dev);
		dev = get_device(&pdev->dev);
		WRITE_ONCE(ch_info->dev, dev);
	}

	ret = 0;
	if (((state != LIN_PCIE_STATE_INIT) && (state != LIN_PCIE_STATE_STOP)) || priv) {
		ret = -EBUSY;
		DEV_ERR("%d", ret);
	}




	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	if (ret) {
		DEV_ERR("%d", ret);
		goto err;
	}

	if (of_data->mode == DW_PCIE_RC_TYPE) {
		ret = lin_pcie_probe_rc(pdev, ch);
		if (ret) {
			DEV_ERR("%d", ret);
			goto err;
		}
	} else {
		ret = lin_pcie_probe_ep(pdev, ch);
		if (ret) {
			DEV_ERR("%d", ret);
			goto err;
		}
	}

	return 0;

err:
	raw_spin_lock_irqsave(&ch_info->spin, flags);



	WRITE_ONCE(ch_info->state, LIN_PCIE_STATE_INIT);
	WRITE_ONCE(ch_info->priv, NULL);



	raw_spin_unlock_irqrestore(&ch_info->spin, flags);




	DEV_ERR("%d", ret);
	return ret;
}

static int lin_pcie_get_ch(struct platform_device *pdev, int *ch)
{
	struct device *dev = &pdev->dev;
	int ret;
	u32 val;

	ret = of_property_read_u32(pdev->dev.of_node, "lin-pcie-ch", &val);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	*ch = val;

	return 0;
}

static int lin_pcie_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	int ret;
	int ch;
	struct task_struct *owner;
	struct lin_pcie_ch_info *ch_info;
	bool locked;
	unsigned long flags;


















	ret = lin_pcie_get_ch(pdev, &ch);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	ch_info = &lin_pcie_ch_info[ch];

	raw_spin_lock_irqsave(&ch_info->spin, flags);




	owner = READ_ONCE(ch_info->owner);




	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	if (owner == current)
		locked = true;
	else
		locked = false;

	if (!locked)
		mutex_lock(&lin_pcie_ch_info[ch].mutex);
	ret = _lin_pcie_probe(pdev, ch);
	if (!locked)
		mutex_unlock(&lin_pcie_ch_info[ch].mutex);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	return 0;
}

static int _lin_pcie_remove(struct platform_device *pdev)
{
	struct lin_pcie_priv *priv = platform_get_drvdata(pdev);
	struct device *dev = &pdev->dev;
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[priv->ch];
	unsigned long flags;
	enum lin_pcie_rc_state state;
	int ret;





	raw_spin_lock_irqsave(&ch_info->spin, flags);




	state = READ_ONCE(ch_info->state);

	ret = 0;
	if (priv->mode == DW_PCIE_RC_TYPE) {
		if ((state != LIN_PCIE_STATE_ERR) && (state != LIN_PCIE_STATE_RC_PREP) &&
				(state != LIN_PCIE_STATE_RC_START) &&
				(state != LIN_PCIE_STATE_RC_SUSPENDING))
			DEV_ERR("%d", state);
	} else if(priv->mode == DW_PCIE_EP_TYPE) {
		if ((state != LIN_PCIE_STATE_ERR) && (state != LIN_PCIE_STATE_EP_START) &&
				(state != LIN_PCIE_STATE_EP_STOP) &&
				(state != LIN_PCIE_STATE_EP_SUSPEND))
			DEV_ERR("%d", state);
	} else {
		DEV_ERR("%d", state);
	}

	_lin_pcie_make_dma_unavailable(priv);
	WRITE_ONCE(ch_info->state, LIN_PCIE_STATE_STOP);




	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	lin_pcie_term(priv);

	raw_spin_lock_irqsave(&ch_info->spin, flags);




	WRITE_ONCE(ch_info->priv, NULL);




	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	return 0;
}

static int lin_pcie_remove(struct platform_device *pdev)
{
	struct lin_pcie_priv *priv = platform_get_drvdata(pdev);
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[priv->ch];
	struct device *dev = &pdev->dev;
	int ret;
	struct task_struct *owner;
	bool locked;
	unsigned long flags;





	raw_spin_lock_irqsave(&ch_info->spin, flags);




	owner = READ_ONCE(ch_info->owner);




	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	if (owner == current)
		locked = true;
	else
		locked = false;

	if (!locked)
		mutex_lock(&ch_info->mutex);
	ret = _lin_pcie_remove(pdev);
	if (!locked)
		mutex_unlock(&ch_info->mutex);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	return 0;
}

static int _lin_pcie_suspend(struct platform_device *pdev, pm_message_t pm_state)
{
	struct lin_pcie_priv *priv = platform_get_drvdata(pdev);
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[priv->ch];
	struct device *dev = &pdev->dev;
	unsigned long flags;
	enum lin_pcie_rc_state state;
	int ret;





	raw_spin_lock_irqsave(&ch_info->spin, flags);




	state = READ_ONCE(ch_info->state);

	ret = 0;
	if ((priv->mode == DW_PCIE_RC_TYPE) && priv->omission_of_rc_suspend &&
			((state == LIN_PCIE_STATE_RC_PREP) || (state == LIN_PCIE_STATE_RC_START))) {
		_lin_pcie_make_dma_unavailable(priv);
		WRITE_ONCE(ch_info->state, LIN_PCIE_STATE_RC_SUSPENDING);



	} else if ((priv->mode == DW_PCIE_EP_TYPE) && ((state == LIN_PCIE_STATE_EP_START) ||
				(state == LIN_PCIE_STATE_EP_STOP))) {
		if (state == LIN_PCIE_STATE_EP_START )
			_lin_pcie_make_dma_unavailable(priv);
		WRITE_ONCE(ch_info->state, LIN_PCIE_STATE_EP_SUSPEND);
	} else {
		_lin_pcie_make_dma_unavailable(priv);
		WRITE_ONCE(ch_info->state, LIN_PCIE_STATE_ERR);



		ret = -EBUSY;
		DEV_ERR("%d", ret);
	}




	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	if (ret < 0) {
		DEV_ERR("%d", ret);
		goto err;
	}

	if (priv->mode == DW_PCIE_EP_TYPE) {
		lin_pcie_term_ep(priv);
	}

	return 0;

err:
	lin_pcie_term(priv);
	raw_spin_lock_irqsave(&ch_info->spin, flags);



	_lin_pcie_make_dma_unavailable(priv);
	WRITE_ONCE(ch_info->state, LIN_PCIE_STATE_ERR);



	raw_spin_unlock_irqrestore(&ch_info->spin, flags);




	DEV_ERR("%d", ret);
	return ret;
}

static int lin_pcie_suspend(struct platform_device *pdev, pm_message_t pm_state)
{
	struct lin_pcie_priv *priv = platform_get_drvdata(pdev);
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[priv->ch];
	struct device *dev = &pdev->dev;
	int ret;

	mutex_lock(&ch_info->mutex);
	ret = _lin_pcie_suspend(pdev, pm_state);
	mutex_unlock(&ch_info->mutex);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	return 0;
}

static int _lin_pcie_resume(struct platform_device *pdev)
{
	struct lin_pcie_priv *priv = platform_get_drvdata(pdev);
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[priv->ch];
	struct device *dev = &pdev->dev;
	unsigned long flags;
	enum lin_pcie_rc_state state;
	int ret;
	bool update;





	raw_spin_lock_irqsave(&ch_info->spin, flags);




	state = READ_ONCE(ch_info->state);

	ret = 0;
	if ((priv->mode != DW_PCIE_EP_TYPE) || (state != LIN_PCIE_STATE_EP_SUSPEND)) {
		ret = -EBADRQC;
		DEV_ERR("%d", ret);
	}




	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	if (ret < 0) {
		DEV_ERR("%d", ret);
		goto err;
	}

	update = false;
	ret = lin_pcie_init_ep_hw(priv);
	if (!ret)
		update = true;
	else if (ret != 1) {
		DEV_ERR("%d", ret);
		goto err;
	}

	raw_spin_lock_irqsave(&ch_info->spin, flags);



	WRITE_ONCE(ch_info->state, LIN_PCIE_STATE_EP_STOP);



	raw_spin_unlock_irqrestore(&ch_info->spin, flags);





	if (update)
		schedule_work(&priv->update_ep);
	lin_pcie_enable_ep_irq(priv);
	priv->ep_irq_enabled = true;

	return 0;

err:
	lin_pcie_term(priv);
	raw_spin_lock_irqsave(&ch_info->spin, flags);



	_lin_pcie_make_dma_unavailable(priv);
	WRITE_ONCE(ch_info->state, LIN_PCIE_STATE_ERR);



	raw_spin_unlock_irqrestore(&ch_info->spin, flags);




	DEV_ERR("%d", ret);
	return ret;
}

static int lin_pcie_resume(struct platform_device *pdev)
{
	struct lin_pcie_priv *priv = platform_get_drvdata(pdev);
	struct lin_pcie_ch_info *ch_info = &lin_pcie_ch_info[priv->ch];
	struct device *dev = &pdev->dev;
	int ret;

	mutex_lock(&ch_info->mutex);
	ret = _lin_pcie_resume(pdev);
	mutex_unlock(&ch_info->mutex);
	if (ret) {
		DEV_ERR("%d", ret);
		return ret;
	}

	return 0;
}


static const struct lin_pcie_of_data lin_pcie_of_data_rc = {
	.mode = DW_PCIE_RC_TYPE,
};

static const struct lin_pcie_of_data lin_pcie_of_data_ep = {
	.mode = DW_PCIE_EP_TYPE,
};

static const struct of_device_id lin_pcie_match[] = {
	{
		.compatible = "cxd,pcie-rc",
		.data = &lin_pcie_of_data_rc,
	},
	{
		.compatible = "cxd,pcie-ep",
		.data = &lin_pcie_of_data_ep,
	},
	{ /* sentinel */ },
};

static struct platform_driver lin_pcie_driver = {
	.probe = lin_pcie_probe,
	.remove = lin_pcie_remove,
	.suspend = lin_pcie_suspend,
	.resume = lin_pcie_resume,
	.driver = {
		.name = "lin-pcie",
		.of_match_table = lin_pcie_match,
	},
};

static void lin_pcie_init_ch(void)
{
	int pcie_ch;
	int dma_idx;
	unsigned long flags;
	unsigned long dma_flags;

	for (pcie_ch = 0; pcie_ch < PCIE_CH_MAX; pcie_ch++) {
		mutex_init(&lin_pcie_ch_info[pcie_ch].mutex);
		raw_spin_lock_init(&lin_pcie_ch_info[pcie_ch].spin);
		raw_spin_lock_init(&lin_pcie_ch_info[pcie_ch].config_spin);
		raw_spin_lock_irqsave(&lin_pcie_ch_info[pcie_ch].spin, flags);
		lin_pcie_ch_info[pcie_ch].state = LIN_PCIE_STATE_INIT;
		lin_pcie_ch_info[pcie_ch].priv = NULL;
		lin_pcie_ch_info[pcie_ch].mode = DW_PCIE_UNKNOWN_TYPE;
		lin_pcie_ch_info[pcie_ch].dev = NULL;
		lin_pcie_ch_info[pcie_ch].start_istr = false;
		lin_pcie_ch_info[pcie_ch].owner = NULL;
		lin_pcie_ch_info[pcie_ch].clkreq_mode = 0;
		lin_pcie_ch_info[pcie_ch].clkreq_ctrl_avail = false;
		raw_spin_unlock_irqrestore(&lin_pcie_ch_info[pcie_ch].spin, flags);

		for (dma_idx = 0; dma_idx < PCIDMA_CH_MAX; dma_idx++) {
			raw_spin_lock_init(&lin_pcie_dma_ch_info[pcie_ch][dma_idx].spin);
			raw_spin_lock_irqsave(&lin_pcie_dma_ch_info[pcie_ch][dma_idx].spin, dma_flags);
			lin_pcie_dma_ch_info[pcie_ch][dma_idx].dma_ch = NULL;
			raw_spin_unlock_irqrestore(&lin_pcie_dma_ch_info[pcie_ch][dma_idx].spin, dma_flags);
		}
	}
}

static int __init lin_pcie_init(void)
{
	lin_pcie_init_ch();
	return platform_driver_register(&lin_pcie_driver);
}
device_initcall(lin_pcie_init);
