// SPDX-License-Identifier: GPL-2.0
/*
 * reset controller for cxd
 *
 * Copyright 2022 Sony Corporation, SOCIONEXT INC.
 *
 */

#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/platform_device.h>
#include <linux/reset-controller.h>
#include <linux/scu-reset.h>
#include <linux/delay.h>

#define ACTIVE_LOW		0
#define ACTIVE_HIGH		1

#define RESET_ON		0
#define RESET_OFF		1

#define SET_REG			4
#define CLR_REG			8

#define RSTCNT0			0
#define RSTCNT(n)		((n) * 0x10 + RSTCNT0)

#ifdef TESTCODE_LIN_RESET
void __iomem *SCU_reset_driver_test_io_remap = NULL;
#endif

struct lin_reset_data {
	unsigned int reg;
	unsigned int bit;
	unsigned int active_h;
	unsigned int delay_utime;
};

#define LIN_RESET(_regid, _bit, _act, _time)	\
	{					\
		.reg = RSTCNT(_regid),		\
		.bit = _bit,			\
		.active_h = _act,		\
		.delay_utime = _time,		\
	}

const struct lin_reset_data lin_cxd_reset_data[] = {
	LIN_RESET(4, 4, 0, 0), // E_SCU_RST_BP_C_PRESETN
	LIN_RESET(4, 5, 0, 0), // E_SCU_RST_BP_C_ARESETN
	LIN_RESET(4, 6, 0, 0), // E_SCU_RST_BP_M_PRESETN
	LIN_RESET(4, 7, 0, 0), // E_SCU_RST_BP_M_ARESETN
	LIN_RESET(4, 8, 0, 1), // E_SCU_RST_CM_UART_PRESETN
	LIN_RESET(4, 9, 0, 0), // E_SCU_RST_CM_UART_S_RST_N_DELETED
	LIN_RESET(4, 12, 0, 1), // E_SCU_RST_CM_TPU0_RESETN
	LIN_RESET(4, 14, 0, 1), // E_SCU_RST_CM_TPU1_RESETN
	LIN_RESET(4, 16, 0, 1), // E_SCU_RST_CM_TPU2_RESETN
	LIN_RESET(4, 18, 0, 1), // E_SCU_RST_CM_TPU3_RESETN
	LIN_RESET(4, 20, 0, 1), // E_SCU_RST_CM_TPU4_RESETN
	LIN_RESET(4, 22, 0, 1), // E_SCU_RST_CM_TPU5_RESETN
	LIN_RESET(4, 24, 0, 1), // E_SCU_RST_CM_GPIO_PRESETN
	LIN_RESET(4, 25, 0, 1), // E_SCU_RST_CM_GPIOINT_PRESETN
	LIN_RESET(4, 27, 0, 1), // E_SCU_RST_CM_ADC_PRESETN
	LIN_RESET(5, 0, 0, 0), // E_SCU_RST_CM_SIO_PRESETN
	LIN_RESET(5, 4, 0, 1), // E_SCU_RST_CM_SRAM_ARESETN
	LIN_RESET(5, 8, 0, 1), // E_SCU_RST_PCIE4LC_C0_ARESETN
	LIN_RESET(5, 10, 0, 1), // E_SCU_RST_PCIE4LC_C1_ARESETN
	LIN_RESET(5, 12, 0, 1), // E_SCU_RST_PCIE4LC_M_ARESETN
	LIN_RESET(6, 0, 0, 0), // E_SCU_RST_EMMC_XRESET
	LIN_RESET(6, 4, 0, 0), // E_SCU_RST_PCIE4LF_RST_N
	LIN_RESET(6, 12, 0, 1), // E_SCU_RST_EMMC_C_ARESETN
	LIN_RESET(6, 14, 0, 1), // E_SCU_RST_EMMC_M_ARESETN
	LIN_RESET(7, 0, 1, 0), // E_SCU_RST_AUDIO_DSP1_DRESET
	LIN_RESET(7, 1, 1, 0), // E_SCU_RST_AUDIO_DSP1_BRESET
	LIN_RESET(7, 4, 1, 0), // E_SCU_RST_AUDIO_DSP2_DRESET
	LIN_RESET(7, 5, 1, 0), // E_SCU_RST_AUDIO_DSP2_BRESET
	LIN_RESET(7, 8, 0, 0), // E_SCU_RST_AUDIO_IF_NRESET
	LIN_RESET(7, 16, 0, 1), // E_SCU_RST_AUDIO_C0_ARESETN
	LIN_RESET(7, 20, 0, 1), // E_SCU_RST_AUDIO_M_ARESETN
	LIN_RESET(8, 0, 0, 1), // E_SCU_RST_OTP_RESETN
	LIN_RESET(8, 4, 0, 0), // E_SCU_RST_M_CPUINT_PRESETN
	LIN_RESET(8, 8, 0, 1), // E_SCU_RST_LDEC_RESETN
	LIN_RESET(8, 12, 0, 1), // E_SCU_RST_FEC_RESETN
	LIN_RESET(8, 16, 0, 1), // E_SCU_RST_CRYPT_E0_RSTX
	LIN_RESET(8, 20, 0, 1), // E_SCU_RST_CRYPT_E1_RSTX
	LIN_RESET(8, 24, 0, 1), // E_SCU_RST_CRYPT_XTS_RSTX
	LIN_RESET(8, 28, 0, 1), // E_SCU_RST_M_GPIO_PRESETN
	LIN_RESET(9, 0, 0, 1), // E_SCU_RST_M_DMAC_C_ARESETN
	LIN_RESET(9, 2, 0, 1), // E_SCU_RST_TZCMP0_M_ARESETN
	LIN_RESET(9, 4, 0, 1), // E_SCU_RST_TZCMP3_M_ARESETN
	LIN_RESET(9, 6, 0, 1), // E_SCU_RST_M_MAIN_C_ARESETN
	LIN_RESET(9, 8, 0, 1), // E_SCU_RST_TRNG_RST_N
	LIN_RESET(9, 10, 0, 1), // E_SCU_RST_TZC400_ARESETN0
	LIN_RESET(9, 11, 0, 1), // E_SCU_RST_TZC400_ARESETN1
	LIN_RESET(9, 12, 0, 1), // E_SCU_RST_TZC400_ARESETN2
	LIN_RESET(9, 13, 0, 1), // E_SCU_RST_TZC400_ARESETN3
	LIN_RESET(9, 14, 0, 1), // E_SCU_RST_TZC400_PRESETN
	LIN_RESET(10, 0, 0, 1), // E_SCU_RST_M_TIMER00_PRESETN
	LIN_RESET(10, 2, 0, 1), // E_SCU_RST_M_TIMER01_PRESETN
	LIN_RESET(10, 4, 0, 1), // E_SCU_RST_M_TIMER02_PRESETN
	LIN_RESET(10, 6, 0, 1), // E_SCU_RST_M_TIMER03_PRESETN
	LIN_RESET(10, 8, 0, 1), // E_SCU_RST_M_TIMER04_PRESETN
	LIN_RESET(10, 10, 0, 1), // E_SCU_RST_M_TIMER05_PRESETN
	LIN_RESET(10, 12, 0, 1), // E_SCU_RST_M_TIMER06_PRESETN
	LIN_RESET(10, 14, 0, 1), // E_SCU_RST_M_TIMER07_PRESETN
	LIN_RESET(10, 16, 0, 1), // E_SCU_RST_M_TIMER08_PRESETN
	LIN_RESET(10, 18, 0, 1), // E_SCU_RST_M_TIMER09_PRESETN
	LIN_RESET(10, 20, 0, 1), // E_SCU_RST_M_TIMER10_PRESETN
	LIN_RESET(10, 22, 0, 1), // E_SCU_RST_M_TIMER11_PRESETN
	LIN_RESET(10, 24, 0, 1), // E_SCU_RST_M_TIMER12_PRESETN
	LIN_RESET(10, 26, 0, 1), // E_SCU_RST_M_TIMER13_PRESETN
	LIN_RESET(10, 28, 0, 1), // E_SCU_RST_M_TIMER14_PRESETN
	LIN_RESET(10, 30, 0, 1), // E_SCU_RST_M_TIMER15_PRESETN
	LIN_RESET(11, 0, 0, 1), // E_SCU_RST_M_UART1_PRESETN
	LIN_RESET(11, 1, 0, 0), // E_SCU_RST_M_UART1_S_RST_N_DELETED
	LIN_RESET(11, 2, 0, 1), // E_SCU_RST_M_UART2_PRESETN
	LIN_RESET(11, 3, 0, 0), // E_SCU_RST_M_UART2_S_RST_N_DELETED
	LIN_RESET(11, 4, 0, 1), // E_SCU_RST_M_UART3_PRESETN
	LIN_RESET(11, 5, 0, 0), // E_SCU_RST_M_UART3_S_RST_N_DELETED
	LIN_RESET(11, 8, 0, 0), // E_SCU_RST_M_SIO_PRESETN
	LIN_RESET(11, 12, 0, 1), // E_SCU_RST_M_SPI0_PRESETN
	LIN_RESET(11, 13, 0, 1), // E_SCU_RST_M_SPI0_SSPRST_N
	LIN_RESET(11, 14, 0, 1), // E_SCU_RST_M_SPI1_PRESETN
	LIN_RESET(11, 15, 0, 0), // E_SCU_RST_M_SPI1_SSI_RST_N
	LIN_RESET(11, 16, 0, 1), // E_SCU_RST_M_SPI2_PRESETN
	LIN_RESET(11, 17, 0, 0), // E_SCU_RST_M_SPI2_SSI_RST_N
	LIN_RESET(11, 18, 0, 1), // E_SCU_RST_M_SPI3_PRESETN
	LIN_RESET(11, 19, 0, 0), // E_SCU_RST_M_SPI3_SSI_RST_N
	LIN_RESET(11, 20, 0, 1), // E_SCU_RST_S_SPI_PRESETN
	LIN_RESET(11, 21, 0, 0), // E_SCU_RST_S_SPI_SSI_RST_N
	LIN_RESET(11, 24, 0, 1), // E_SCU_RST_M_I2C0_PRESETN
	LIN_RESET(11, 25, 0, 1), // E_SCU_RST_M_I2C0_IC_RST_N
	LIN_RESET(11, 26, 0, 1), // E_SCU_RST_M_I2C1_PRESETN
	LIN_RESET(11, 27, 0, 1), // E_SCU_RST_M_I2C1_IC_RST_N
	LIN_RESET(11, 28, 0, 1), // E_SCU_RST_M_I2C2_PRESETN
	LIN_RESET(11, 29, 0, 1), // E_SCU_RST_M_I2C2_IC_RST_N
	LIN_RESET(14, 0, 0, 1), // E_SCU_RST_MEDIA_C_ARESETN
	LIN_RESET(14, 2, 0, 1), // E_SCU_RST_MEDIA_M_ARESETN
	LIN_RESET(14, 4, 0, 1), // E_SCU_RST_USB_M_ARESETN
	LIN_RESET(12, 0, 0, 0), // E_SCU_RST_PCIE2LM0_RST_N
	LIN_RESET(12, 8, 0, 0), // E_SCU_RST_PCIE2LM1_RST_N
	LIN_RESET(12, 16, 0, 0), // E_SCU_RST_PCIE1L_RST_N
	LIN_RESET(13, 0, 0, 0), // E_SCU_RST_USB32_ARESETN
	LIN_RESET(13, 1, 1, 0), // E_SCU_RST_USB32_POR
	LIN_RESET(13, 2, 1, 0), // E_SCU_RST_USB32_PHY_RESET
	LIN_RESET(13, 3, 0, 1), // E_SCU_RST_USB32_APBRST
	LIN_RESET(13, 4, 0, 1), // E_SCU_RST_USB20_HO_HRESETN
	LIN_RESET(13, 5, 0, 1), // E_SCU_RST_USB20_HRESETN
	LIN_RESET(13, 6, 0, 1), // E_SCU_RST_USB20_PRST_N
	LIN_RESET(13, 7, 1, 0), // E_SCU_RST_USB20_POR
	LIN_RESET(13, 8, 0, 1), // E_SCU_RST_UHS2_0_XRESET
	LIN_RESET(13, 12, 0, 1), // E_SCU_RST_UHS2_1_XRESET
	LIN_RESET(13, 16, 0, 1), // E_SCU_RST_UHS1_XRESET
	LIN_RESET(13, 20, 0, 1), // E_SCU_RST_USB20_MMU_CRESETN
	LIN_RESET(13, 21, 0, 1), // E_SCU_RST_USB20_MMU_TBU_BRESETN0
	LIN_RESET(13, 22, 0, 1), // E_SCU_RST_UHS2_MMU_0_CRESETN
	LIN_RESET(13, 23, 0, 1), // E_SCU_RST_UHS2_MMU_0_TBU_BRESETN0
	LIN_RESET(13, 24, 0, 1), // E_SCU_RST_UHS2_MMU_1_CRESETN
	LIN_RESET(13, 25, 0, 1), // E_SCU_RST_UHS2_MMU_1_TBU_BRESETN0
	LIN_RESET(13, 26, 0, 1), // E_SCU_RST_UHS1_MMU_CRESETN
	LIN_RESET(13, 27, 0, 1), // E_SCU_RST_UHS1_MMU_TBU_BRESETN0
	LIN_RESET(14, 8, 0, 0), // E_SCU_RST_MS_HRESETN
	LIN_RESET(9, 16, 0, 1), // E_SCU_RST_M_PERI_NIC_ARESETN
	LIN_RESET(9, 17, 0, 1), // E_SCU_RST_M_PERI_NIC_GB_ARESETN
	LIN_RESET(9, 18, 0, 1), // E_SCU_RST_M_PERI_NIC_GC_ARESETN
	LIN_RESET(9, 19, 0, 1), // E_SCU_RST_M_PERI_NIC_HRESETN
	LIN_RESET(9, 20, 0, 1), // E_SCU_RST_M_PERI_NIC_M_PRESETN
	LIN_RESET(9, 21, 0, 1), // E_SCU_RST_M_PERI_NIC_PRESETN
	LIN_RESET(9, 22, 0, 1), // E_SCU_RST_M_PERI_NIC_U3RESETN
	LIN_RESET(9, 24, 0, 1), // E_SCU_RST_S_SRAM_ARESETN
	LIN_RESET(9, 28, 0, 1), // E_SCU_RST_M_WDT_WDOGRESN
	LIN_RESET(9, 29, 0, 1), // E_SCU_RST_M_WDT_PRESETN
	LIN_RESET(7, 18, 0, 1), // E_SCU_RST_AUDIO_C1_ARESETN
	LIN_RESET(7, 22, 0, 1), // E_SCU_RST_SRAM_A0_ARESETN
	LIN_RESET(7, 23, 0, 1), // E_SCU_RST_SRAM_A1_ARESETN
	LIN_RESET(14, 12, 0, 1), // E_SCU_RST_TBL_SRAM_ARESETN
	LIN_RESET(5, 16, 0, 0), // E_SCU_RST_FCS_ARESETN
	LIN_RESET(24, 16, 0, 0), // E_SCU_RST_PCIE4LC_RST_N
	LIN_RESET(24, 24, 0, 1), // E_SCU_RST_COPY0_RESETN
	LIN_RESET(24, 26, 0, 1), // E_SCU_RST_COPY1_RESETN
	LIN_RESET(24, 28, 0, 1), // E_SCU_RST_COPY2_RESETN
	LIN_RESET(24, 30, 0, 1), // E_SCU_RST_COPY3_RESETN
	LIN_RESET(26, 0, 0, 1), // E_SCU_RST_M_DMACS_RESETN
	LIN_RESET(26, 1, 0, 1), // E_SCU_RST_M_DMACS_PRESETN
	LIN_RESET(26, 2, 0, 1), // E_SCU_RST_M_DMACS_HS_RSTN0
	LIN_RESET(26, 3, 0, 1), // E_SCU_RST_M_DMACS_HS_RSTN1
	LIN_RESET(26, 4, 0, 1), // E_SCU_RST_M_DMAC1_RESETN
	LIN_RESET(26, 5, 0, 1), // E_SCU_RST_M_DMAC1_PRESETN
	LIN_RESET(26, 6, 0, 1), // E_SCU_RST_M_DMAC2_RESETN
	LIN_RESET(26, 7, 0, 1), // E_SCU_RST_M_DMAC2_PRESETN
	LIN_RESET(26, 8, 0, 1), // E_SCU_RST_M_DMAC3_RESETN
	LIN_RESET(26, 9, 0, 1), // E_SCU_RST_M_DMAC3_PRESETN
	LIN_RESET(26, 10, 0, 1), // E_SCU_RST_M_DMAC4_RESETN
	LIN_RESET(26, 11, 0, 1), // E_SCU_RST_M_DMAC4_PRESETN
	LIN_RESET(26, 12, 0, 1), // E_SCU_RST_M_DMAC4_HS_RSTN00
	LIN_RESET(26, 13, 0, 1), // E_SCU_RST_M_DMAC4_HS_RSTN01
	LIN_RESET(26, 14, 0, 1), // E_SCU_RST_M_DMAC4_HS_RSTN02
	LIN_RESET(26, 15, 0, 1), // E_SCU_RST_M_DMAC4_HS_RSTN03
	LIN_RESET(26, 16, 0, 1), // E_SCU_RST_M_DMAC4_HS_RSTN04
	LIN_RESET(26, 17, 0, 1), // E_SCU_RST_M_DMAC4_HS_RSTN05
	LIN_RESET(26, 18, 0, 1), // E_SCU_RST_M_DMAC4_HS_RSTN06
	LIN_RESET(26, 19, 0, 1), // E_SCU_RST_M_DMAC4_HS_RSTN07
	LIN_RESET(26, 20, 0, 1), // E_SCU_RST_M_DMAC4_HS_RSTN08
	LIN_RESET(26, 21, 0, 1), // E_SCU_RST_M_DMAC4_HS_RSTN09
	LIN_RESET(26, 22, 0, 1), // E_SCU_RST_M_DMAC4_HS_RSTN10
	LIN_RESET(26, 23, 0, 1), // E_SCU_RST_M_DMAC4_HS_RSTN11
};

struct lin_reset_priv {
	struct reset_controller_dev	rcdev;
	struct device			*dev;
	void __iomem			*base;
	const struct lin_reset_data	*data;
};

#define to_lin_reset_priv(_rcdev) \
			container_of(_rcdev, struct lin_reset_priv, rcdev)

static struct lin_reset_priv *g_priv;

static E_SCU_ERR check_reset_param(E_SCU_RST id, struct lin_reset_priv *priv)
{
	E_SCU_ERR ret;

	if (!priv)
		ret = E_SCU_ERR_PARAM;
	else if (id >= priv->rcdev.nr_resets)
		ret = E_SCU_ERR_PARAM;
	else
		ret = E_SCU_ERR_OK;
	return ret;
}

static void set_reset(E_SCU_RST id, int set, struct lin_reset_priv *priv)
{
	const struct lin_reset_data *data = priv->data;
	unsigned int val, reg;
	int sel_set_clr;

	val = BIT(data[id].bit);
	sel_set_clr = set ^ data[id].active_h;
	reg = data[id].reg + (sel_set_clr ? SET_REG : CLR_REG);

	if(set == RESET_ON){
		mb();
	}

	writel(val, priv->base + reg);

	if(set == RESET_OFF){
		mb();
		readl(priv->base + reg);	//readback for timing adjustment
		mb();
	}

	if (data[id].delay_utime) {
		wmb(); // need to reach register before udelay()
		udelay(data[id].delay_utime);
	}
}

static int lin_reset_assert(struct reset_controller_dev *rcdev,
				 unsigned long id)
{
	if (scu_rst_enable(id))
		return -EINVAL;
	else
		return 0;
}

static int lin_reset_deassert(struct reset_controller_dev *rcdev,
				   unsigned long id)
{
	if (scu_rst_disable(id))
		return -EINVAL;
	else
		return 0;
}

static int lin_reset_status(struct reset_controller_dev *rcdev,
				 unsigned long id)
{
	struct lin_reset_priv *priv = to_lin_reset_priv(rcdev);
	const struct lin_reset_data *data = priv->data;
	unsigned int val;

	if (check_reset_param(id, priv) != E_SCU_ERR_OK)
		return -EINVAL;

	val = readl(priv->base + data[id].reg);
	val = !(val & BIT(data[id].bit));
	val ^= data[id].active_h;

	return val;
}

static const struct reset_control_ops lin_reset_ops = {
	.assert = lin_reset_assert,
	.deassert = lin_reset_deassert,
	.status = lin_reset_status,
};

static int lin_reset_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct lin_reset_priv *priv;
	const struct lin_reset_data *data;
	struct resource *res;
	void __iomem *base;
	int ret;

	dev_err( dev, "starting lin-reset driver");

	data = of_device_get_match_data(dev);
	if (WARN_ON(!data))
		return -EINVAL;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		dev_err(&pdev->dev, "missing IO resource\n");
		return -ENODEV;
	}
	base = devm_ioremap_resource(dev, res);
	if (IS_ERR(base))
		return PTR_ERR(base);

	#ifdef TESTCODE_LIN_RESET
	SCU_reset_driver_test_io_remap = base;
	#endif

	priv = devm_kzalloc(dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	priv->rcdev.ops = &lin_reset_ops;
	priv->rcdev.owner = dev->driver->owner;
	priv->rcdev.of_node = dev->of_node;
	priv->rcdev.nr_resets = ARRAY_SIZE(lin_cxd_reset_data);
	priv->dev = dev;
	priv->base = base;
	priv->data = data;
	ret = devm_reset_controller_register(&pdev->dev, &priv->rcdev);
	if (!ret)
		g_priv = priv;
	return ret;
}

static const struct of_device_id lin_reset_dt_ids[] = {
	{
		.compatible = "scu,lin-reset",
		.data = lin_cxd_reset_data,
	},
	{ /* sentinel */ },
};

static struct platform_driver lin_reset_driver = {
	.probe  = lin_reset_probe,
	.driver = {
	.name           = "scu-reset",
	.of_match_table = lin_reset_dt_ids,
	},
};

static int __init lin_reset_driver_init(void)
{
	return platform_driver_register(&lin_reset_driver);
}
arch_initcall(lin_reset_driver_init);

/* LLD APIs */
E_SCU_ERR scu_rst_enable(const E_SCU_RST id)
{
	struct lin_reset_priv *priv = g_priv;
	E_SCU_ERR ret;

	ret = check_reset_param(id, priv);
	if (ret == E_SCU_ERR_OK)
		set_reset(id, RESET_ON, priv);
	return ret;
}
EXPORT_SYMBOL(scu_rst_enable);

E_SCU_ERR scu_rst_disable(const E_SCU_RST id)
{
	struct lin_reset_priv *priv = g_priv;
	E_SCU_ERR ret;

	ret = check_reset_param(id, priv);
	if (ret == E_SCU_ERR_OK)
		set_reset(id, RESET_OFF, priv);
	return ret;
}
EXPORT_SYMBOL(scu_rst_disable);

/* for test module */
#ifdef TESTCODE_LIN_RESET
u32 SCU_lin_reset_reg_direct_read(u32 offset)
{
	u32 val;
	val = readl(SCU_reset_driver_test_io_remap + offset);
	return val;
}
EXPORT_SYMBOL(SCU_lin_reset_reg_direct_read);

int scu_rst_state(const E_SCU_RST id)
{
	struct lin_reset_priv *priv = g_priv;
	const struct lin_reset_data *data = priv->data;
	E_SCU_ERR ret;
	unsigned int val;

	ret = check_reset_param(id, priv);
	if (ret == E_SCU_ERR_OK) {
		val = readl(priv->base + data[id].reg);
		val = !(val & BIT(data[id].bit));
		val ^= data[id].active_h;
	}
	return val;
}
EXPORT_SYMBOL(scu_rst_state);
#endif
