// SPDX-License-Identifier: GPL-2.0
/*
 *  snsc_rtc_build_timestamp: set system time using build timestamp
 *
 *  Copyright 2022 Sony Group Corporation
 */

#include <linux/rtc.h>

#include <linux/printk.h>
#include <linux/kernel.h>


int snsc_update_system_time(struct rtc_time *rtc_tm)
{

	int err = 0;
	struct timespec64 tv64 = {
		.tv_sec	 = 0,
		.tv_nsec = NSEC_PER_SEC >> 1,
	};
	struct rtc_time tm;
	unsigned long builtin_time_sec;

	err = kstrtol(snsc_build_timestamp_sec, 0, &builtin_time_sec);
	if (!err) {
		if (rtc_tm) {
			tv64.tv_sec = rtc_tm_to_time64(rtc_tm);
			/* dont use builtin timestamp and use rtc time */
			if (tv64.tv_sec > builtin_time_sec)
				err = -EINVAL;
		}
	}

	if (rtc_hctosys_ret != 0 && err == 0) {
		tv64.tv_sec = builtin_time_sec;
		rtc_time64_to_tm(tv64.tv_sec, &tm);
		err = do_settimeofday64(&tv64);
		pr_info("use built-in time and setting system clock to %d-%02d-%02d %02d:%02d:%02d UTC (%u)\n",
			tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday,
			tm.tm_hour, tm.tm_min, tm.tm_sec,
			(unsigned int) tv64.tv_sec);
	}

	return err;
}

static int __init snsc_rtc_build_timestamp(void)
{
	int err;

	if (rtc_hctosys_ret != -ENODEV)
		goto ret;

	err = snsc_update_system_time(NULL);
	rtc_hctosys_ret = err;

ret:
	return rtc_hctosys_ret;
}

late_initcall(snsc_rtc_build_timestamp);
