/*
 * mach-cxd90xxx/pm_core.c
 *
 *
 * Copyright 2022 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <mach/moduleparam.h>
#include <linux/init.h>
#include <linux/printk.h>
#include <linux/suspend.h>
#include <linux/warmboot.h>
#ifdef CONFIG_SNSC_SSBOOT
#include <linux/ssboot.h>
#endif
#include <linux/io.h>
#include <linux/wdt.h>
#include <linux/tmonitor.h>
#include <linux/em_export.h>
#include <asm/suspend.h>
#include <asm/system_misc.h>
#include <asm/cacheflush.h>
#include <asm/mmu_context.h>
#include <mach/pm.h>
#include <mach/platform.h>
#include <mach/noncache.h>
#include <mach/regs-gpio.h>

#ifdef CONFIG_OSAL_UDIF
extern int udif_driver_suspend(void);
extern int udif_driver_resume(void);
#else
# define udif_driver_suspend() (0)
# define udif_driver_resume() do {} while (0)
#endif /* CONFIG_OSAL_UDIF */

static int logbuf_noclear = 0;
module_param_named(logbuf_noclear, logbuf_noclear, int, S_IRUSR|S_IWUSR);

/*
 * resume vector
 */
static unsigned long cxd_resume_vector = ~0UL;
module_param_named(resume, cxd_resume_vector, ulongH, S_IRUSR);

/*
 * suspend error flag
 */
static unsigned long cxd_sus_error = ~0UL;
module_param_named(sus_error, cxd_sus_error, ulongH, S_IRUSR);

/*
 * PORT
 */
#define PORT_UNDEF (~0U)
/* XPOWER_OFF */
static unsigned int cxd_xpower_off = PORT_UNDEF;
module_param_named(poff, cxd_xpower_off, port, S_IRUSR);
/* XRESET_REQ */
#define XRSTREQ_PORT CXD_GPIO_PORT_C(0)
#define XRSTREQ_BIT  0

static void __noreturn cxd_reset(int force)
{
	local_irq_disable();
	/* OUTPUT */
	writel(BIT(XRSTREQ_BIT), VA_GPIO(XRSTREQ_PORT)+GPIO_DIR+GPIO_SET);
	/* L */
	writel(BIT(XRSTREQ_BIT), VA_GPIO(XRSTREQ_PORT)+GPIO_WDATA+GPIO_CLR);
	/* GPIO */
	writel(3U<<XRSTREQ_BIT*2, VA_GPIO(XRSTREQ_PORT)+GPIO_PORT+GPIO_CLR);
	mb();
	for (;;) {
			wfi();
	}
}

#if 0
static void cxd_pm_restart(enum reboot_mode reboot_mode, const char *cmd)
{
	cxd_reset(1);
}
#endif

void __noreturn (*pm_machine_reset)(int) = cxd_reset;
EXPORT_SYMBOL(pm_machine_reset);

void em_reboot(int force)
{
	flush_dcache_all();
	pm_machine_reset(force);
}

static void cxd_pm_setup(void)
{
	printk("resume entry: 0x%09lx\n", (unsigned long)virt_to_phys(cxd_resume));
	/* resume vector */
	if (~0UL == cxd_resume_vector) {
		printk("resume vector: not defined\n");
	} else {
		printk("resume vector: 0x%09lx\n", cxd_resume_vector);
		*(unsigned long *)phys_to_virt(cxd_resume_vector) = virt_to_phys(cxd_resume);
	}
	/* sus_error flag */
	if (~0UL == cxd_sus_error) {
		printk("sus_error address: not defined\n");
	} else {
		printk("sus_error address: 0x%09lx\n", cxd_sus_error);
		*(unsigned long *)phys_to_virt(cxd_sus_error) = 0UL;
	}
}

void __attribute__((weak)) cxd_early_resume(void)
{
	/* none */
}

#ifdef CONFIG_WARM_BOOT_IMAGE
/* check if memory data is kept during the last halt */
static unsigned int cxd_mem_alive = 0;

/* check if memory data is kept during the last halt */
unsigned int pm_is_mem_alive(void)
{
	return cxd_mem_alive;
}
EXPORT_SYMBOL(pm_is_mem_alive);

#ifdef CONFIG_SNSC_SSBOOT
int get_ssboot_stat(void)
{
	extern int cxd_ssboot_stat; /* in resume.S */

	return cxd_ssboot_stat;
}
EXPORT_SYMBOL(get_ssboot_stat);

void wb_reboot(void)
{
	em_reboot(1);
}
#endif /* CONFIG_SNSC_SSBOOT */
#endif /* CONFIG_WARM_BOOT_IMAGE */

#ifdef CONFIG_EJ_SMC_SUSPEND
#include <linux/arm-smccc.h>
#include <uapi/linux/psci.h>
static void __noreturn invoke_smc_suspend(void)
{
	phys_addr_t pa_cpu_resume = virt_to_phys(cxd_resume);
	struct arm_smccc_res res;

	arm_smccc_smc(PSCI_1_0_FN64_SYSTEM_SUSPEND, pa_cpu_resume,
		      0, 0, 0, 0, 0, 0, &res);
	for (;;) {
		wfi();
	}
	unreachable();
}

static void smc_suspend(void)
{
	extern int use_optee;

	if (!use_optee)
		return;
	invoke_smc_suspend();
	unreachable();
}
#endif /* CONFIG_EJ_SMC_SUSPEND */

#ifdef CONFIG_EJ_SIMPLE_SUSPEND
static int __noreturn cxd_suspend_finisher(unsigned long mode)
{
	extern void printascii(char *);

	raw_local_irq_disable();
	printascii("SUSPEND\r\n");
	flush_dcache_all();

#ifdef CONFIG_EJ_SMC_SUSPEND
	smc_suspend();
#endif
#if 0
	for (;;) {
		wfi();
	}
#endif
	asm volatile("hlt #1");
	unreachable();
}
#else /* CONFIG_EJ_SIMPLE_SUSPEND */
static int cxd_suspend_finisher(unsigned long mode)
{
	raw_local_irq_disable();
#ifdef CONFIG_WARM_BOOT_IMAGE
	cxd_mem_alive = 0; // for warmboot
	if (PM_ENTER_NORMAL == mode) {
		int ret;

		flush_dcache_all();
#ifdef CONFIG_SNSC_SSBOOT
		ret = wb_create_ssbi();
#else
		ret = wb_create_warmbootimage((unsigned long)cxd_resume, 0);
#endif /* CONFIG_SNSC_SSBOOT */
		if (ret)
			return ret;
	}

	cxd_mem_alive = 1; // for hotboot
#endif /* CONFIG_WARM_BOOT_IMAGE */

#ifdef CONFIG_EJ_SMC_SUSPEND
	flush_dcache_all();
	smc_suspend();
#endif
	printk("cxd_suspend failed!\n");
	return -EIO;
}
#endif /* CONFIG_EJ_SIMPLE_SUSPEND */

static void __noreturn cxd_pm_halt(void)
{
#if !defined(CONFIG_EJ_SIMPLE_SUSPEND)
	cxd_suspend_finisher(PM_ENTER_ERROR);
#endif
	for (;;) {
		wfi();
	}
	unreachable();
}

static void __noreturn pm_suspend_error(void)
{
	if (~0UL != cxd_sus_error) {
		*(unsigned long *)phys_to_virt(cxd_sus_error) = 1UL;
	}
	cxd_pm_halt();
	unreachable();
}

/*
 * Called after processes are frozen, but before we shut down devices.
 */
static int cxd_pm_begin(suspend_state_t state)
{
	watchdog_start();
	if (PM_SUSPEND_DISK == state) {
		watchdog_stop();
	}
	return warm_boot_pm_begin(state);
}

static int cxd_pm_prepare(void)
{
	watchdog_touch();
	if (unlikely(udif_driver_suspend())) {
		pm_suspend_error();
	}

	watchdog_touch();
	return warm_boot_pm_prepare();
}

static int cxd_pm_prepare_late(void)
{
	watchdog_touch();
	return 0;
}

#ifdef CONFIG_SNSC_BOOT_TIME
#include <linux/snsc_boot_time.h>
#endif

static int cxd_pm_enter(suspend_state_t state)
{
	int err = 0;
#ifdef CONFIG_SNSC_DEBUG_PROFILE
	extern void profile_soft_init(void);
	extern void profile_dump_record(void (*print)(char *));
#endif

	watchdog_touch();
#ifdef CONFIG_SNSC_DEBUG_PROFILE
	profile_dump_record(printascii);
#endif
	err = cpu_suspend(PM_ENTER_NORMAL, cxd_suspend_finisher);
#ifdef CONFIG_WARM_BOOT_IMAGE
	if (!err && PM_SUSPEND_DISK == pm_get_state() && !pm_is_mem_alive()) {
		if (!logbuf_noclear) {
			clear_logbuf();
		}
		tmonitor_restart();
		printk("WARMBOOT\n");
#ifdef CONFIG_SNSC_SSBOOT
		ssboot_setmode(get_ssboot_stat());
#endif
	}
#endif /* CONFIG_WARM_BOOT_IMAGE */

	cxd_early_resume();

#ifdef CONFIG_SNSC_BOOT_TIME
	boot_time_takeover();
#endif
#ifdef CONFIG_SNSC_DEBUG_PROFILE
	profile_soft_init();
#endif
	return err;
}

static void cxd_pm_wake(void)
{
	watchdog_touch();
}

/*
 * Called after devices are re-setup, but before processes are thawed.
 */
static void cxd_pm_finish(void)
{
	watchdog_touch();
	warm_boot_pm_finish();
	udif_driver_resume();
#ifdef CONFIG_SNSC_SSBOOT
	if (wb_create_ssbi_finish()) {
		extern int cxd_ssboot_stat; /* in resume.S */

		cxd_ssboot_stat = SSBOOT_SSBOOT; /* adjust */
		printk("SSBI Create Completed!\n");
	}
#endif /* CONFIG_SNSC_SSBOOT */
	watchdog_touch();
}

static void cxd_pm_end(void)
{
	warm_boot_pm_end();
	watchdog_stop();
}

static int cxd_pm_valid(suspend_state_t state)
{
#ifdef PM_SUSPEND_DISK
	return PM_SUSPEND_MEM == state || PM_SUSPEND_DISK == state;
#else
	return PM_SUSPEND_MEM == state;
#endif
}

static struct platform_suspend_ops cxd_pm_ops = {
	.valid		= cxd_pm_valid,
	.begin		= cxd_pm_begin,
	.prepare	= cxd_pm_prepare,
	.prepare_late	= cxd_pm_prepare_late,
	.enter		= cxd_pm_enter,
	.wake		= cxd_pm_wake,
	.finish		= cxd_pm_finish,
	.end		= cxd_pm_end,
};

static int __init cxd_pm_init(void)
{
	cxd_pm_setup();
	suspend_set_ops(&cxd_pm_ops);
#ifdef CONFIG_SNSC_SSBOOT
	wb_ssboot_vec[SSBOOT_CREATE_MINSSBI] = (unsigned long)cxd_resume;
	wb_ssboot_vec[SSBOOT_PROFILE] = (unsigned long)cxd_resume_profile;
	wb_ssboot_vec[SSBOOT_CREATE_OPTSSBI] = (unsigned long)cxd_resume_optimize;
	wb_ssboot_vec[SSBOOT_SSBOOT_PRE] = (unsigned long)cxd_resume_ssboot_pre;
	wb_ssboot_vec[SSBOOT_SSBOOT] = (unsigned long)cxd_resume_ssboot;
#endif /* CONFIG_SNSC_SSBOOT */
	return 0;
}

late_initcall(cxd_pm_init);
