/*
 * mach-cxd90xxx/udif/platform.c
 *
 *
 * Copyright 2022 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/udif/device.h>
#include <linux/udif/string.h>
#include <linux/udif/macros.h>
#include <linux/udif/export.h>
#include <mach/udif/platform.h>
#include <mach/platform.h>
#include <mach/irqs.h>
#include <mach/regs-clk.h>
#include <mach/regs-srst.h>

#define __UDIF_IOMEM_INIT(n,pa,sz)	UDIF_IOMEM_INIT(n,pa,sz,UDIF_IO_FLAGS_NONCACHED)

#define UDIF_INTRPT_INIT(n,i,f) \
{ \
	.name	= n, \
	.irq	= i, \
	.flags	= f, \
}


/* HW Timer stuff */
#define HWTMR_NAME		"hwtimer"
#define HWTMR_IO_BASE(ch)	CXD90XXX_TIMER_BASE(ch)
#define HWTMR_IO_SIZE		0x1000
#define HWTMR_IRQ(ch)		IRQ_TIMER(ch)
#define HWTMR_IRQ_FLAGS		IRQF_TIMER
#define HWTMR_PCLK		CXD90XXX_CLKEN(CLKEN_TIMER)
#define HWTMR_PCLK_SET		(HWTMR_PCLK + CLK_SET)
#define HWTMR_PCLK_CLR		(HWTMR_PCLK + CLK_CLR)
#define HWTMR_PCLK_BIT(ch)	CLKEN_TIMER_PCLK(ch)
#define HWTMR_DCLK		HWTMR_PCLK
#define HWTMR_DCLK_SET		(HWTMR_DCLK + CLK_SET)
#define HWTMR_DCLK_CLR		(HWTMR_DCLK + CLK_CLR)
#define HWTMR_DCLK_BIT(ch)	CLKEN_TIMER_DCLK(ch)
#define HWTMR_RST		CXD90XXX_RESET(SRST_TIMER)
#define HWTMR_RST_ASSERT	(HWTMR_RST + SRST_CLR)
#define HWTMR_RST_NEGATE	(HWTMR_RST + SRST_SET)
#define HWTMR_RST_BIT(ch)	SRST_TIMER_RST(ch)
#define HWTMR_RST_DLY		SRST_TIMER_DLY

static UDIF_CHANNELS hwtimer[UDIF_NR_HWTIMER] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(HWTMR_NAME "0", HWTMR_IO_BASE(0), HWTMR_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(HWTMR_NAME "0", HWTMR_IRQ(0), HWTMR_IRQ_FLAGS),
		.clock = UDIF_CLOCK_INIT(HWTMR_PCLK_SET, HWTMR_PCLK_CLR, HWTMR_PCLK_BIT(0),
					 HWTMR_DCLK_SET, HWTMR_DCLK_CLR, HWTMR_DCLK_BIT(0),
					 0, 0, 0, 0),
		.reset = UDIF_RESET_INIT(HWTMR_RST_ASSERT, HWTMR_RST_NEGATE, HWTMR_RST_BIT(0), HWTMR_RST_DLY,
					 0, 0, 0, 0),
	},
	[1] = {
		.iomem = __UDIF_IOMEM_INIT(HWTMR_NAME "1", HWTMR_IO_BASE(1), HWTMR_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(HWTMR_NAME "1", HWTMR_IRQ(1), HWTMR_IRQ_FLAGS),
		.clock = UDIF_CLOCK_INIT(HWTMR_PCLK_SET, HWTMR_PCLK_CLR, HWTMR_PCLK_BIT(1),
					 HWTMR_DCLK_SET, HWTMR_DCLK_CLR, HWTMR_DCLK_BIT(1),
					 0, 0, 0, 0),
		.reset = UDIF_RESET_INIT(HWTMR_RST_ASSERT, HWTMR_RST_NEGATE, HWTMR_RST_BIT(1), HWTMR_RST_DLY,
					 0, 0, 0, 0),
	},
	[2] = {
		.iomem = __UDIF_IOMEM_INIT(HWTMR_NAME "2", HWTMR_IO_BASE(2), HWTMR_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(HWTMR_NAME "2", HWTMR_IRQ(2), HWTMR_IRQ_FLAGS),
		.clock = UDIF_CLOCK_INIT(HWTMR_PCLK_SET, HWTMR_PCLK_CLR, HWTMR_PCLK_BIT(2),
					 HWTMR_DCLK_SET, HWTMR_DCLK_CLR, HWTMR_DCLK_BIT(2),
					 0, 0, 0, 0),
		.reset = UDIF_RESET_INIT(HWTMR_RST_ASSERT, HWTMR_RST_NEGATE, HWTMR_RST_BIT(2), HWTMR_RST_DLY,
					 0, 0, 0, 0),
	},
	[3] = {
		.iomem = __UDIF_IOMEM_INIT(HWTMR_NAME "3", HWTMR_IO_BASE(3), HWTMR_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(HWTMR_NAME "3", HWTMR_IRQ(3), HWTMR_IRQ_FLAGS),
		.clock = UDIF_CLOCK_INIT(HWTMR_PCLK_SET, HWTMR_PCLK_CLR, HWTMR_PCLK_BIT(3),
					 HWTMR_DCLK_SET, HWTMR_DCLK_CLR, HWTMR_DCLK_BIT(3),
					 0, 0, 0, 0),
		.reset = UDIF_RESET_INIT(HWTMR_RST_ASSERT, HWTMR_RST_NEGATE, HWTMR_RST_BIT(3), HWTMR_RST_DLY,
					 0, 0, 0, 0),
	},
};

/* WDT stuff */
#define WDT_NAME		"wdt"
#define WDT_REG_NAME            WDT_NAME "_reg"
#define WDT_MISC_NAME           WDT_NAME "_misc"
#define WDT_IO_BASE		CXD90XXX_WDT
#define WDT_IO_SIZE		0x1000
#define WDT_MISC_BASE		CXD90XXX_WDT_MISC
#define WDT_MISC_SIZE		0x100
#define WDT_IRQ_NAME		"EMERGENCY"
#define WDT_IRQ			IRQ_WDOG_INT
#define WDT_PCLK		CXD90XXX_CLKEN(CLKEN_WDT)
#define WDT_PCLK_SET		(WDT_PCLK + CLK_SET)
#define WDT_PCLK_CLR		(WDT_PCLK + CLK_CLR)
#define WDT_PCLK_BIT		CLKEN_WDT_PCLK
#define WDT_DCLK		WDT_PCLK
#define WDT_DCLK_SET		(WDT_DCLK + CLK_SET)
#define WDT_DCLK_CLR		(WDT_DCLK + CLK_CLR)
#define WDT_DCLK_BIT		CLKEN_WDT_DCLK
#define WDT_PRST		CXD90XXX_RESET(SRST_WDT)
#define WDT_PRST_ASSERT		(WDT_PRST + SRST_CLR)
#define WDT_PRST_NEGATE		(WDT_PRST + SRST_SET)
#define WDT_PRST_BIT		SRST_WDT_PRST
#define WDT_PRST_DLY		SRST_WDT_PRST_DLY
#define WDT_DRST		WDT_PRST
#define WDT_DRST_ASSERT		(WDT_DRST + SRST_CLR)
#define WDT_DRST_NEGATE		(WDT_DRST + SRST_SET)
#define WDT_DRST_BIT		SRST_WDT_DRST
#define WDT_DRST_DLY		0

static UDIF_CHANNELS wdt[UDIF_NR_WDT] = {
	[UDIF_WDT_REG] = {
		.iomem = __UDIF_IOMEM_INIT(WDT_REG_NAME, WDT_IO_BASE, WDT_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(WDT_IRQ_NAME, WDT_IRQ, 0),
		.clock = UDIF_CLOCK_INIT(WDT_PCLK_SET, WDT_PCLK_CLR, WDT_PCLK_BIT,
					 WDT_DCLK_SET, WDT_DCLK_CLR, WDT_DCLK_BIT,
					 0, 0, 0, 0),
		.reset = UDIF_RESET_INIT(WDT_PRST_ASSERT, WDT_PRST_NEGATE, WDT_PRST_BIT, WDT_PRST_DLY,
					 WDT_DRST_ASSERT, WDT_DRST_NEGATE, WDT_DRST_BIT, WDT_DRST_DLY),
	},
	[UDIF_WDT_MISC] = {
		.iomem = __UDIF_IOMEM_INIT(WDT_MISC_NAME, WDT_MISC_BASE, WDT_MISC_SIZE),
	},
};

/* gpio stuff */
#define GPIO_NAME		"gpio"
#define GPIO_PORT_NAME(x)	( "gpio_p" #x )
#define GPIO_IO_SIZE		0x0000100U
#define GPIO_C_PCLK		CXD90XXX_CLKEN(CLKEN_GPIO_C)
#define GPIO_C_PCLK_SET		(GPIO_C_PCLK + CLK_SET)
#define GPIO_C_PCLK_CLR		(GPIO_C_PCLK + CLK_CLR)
#define GPIO_C_PCLK_BIT		CLKEN_GPIO_C_PCLK
#define GPIO_C_RST		CXD90XXX_RESET(SRST_GPIO_C_GRP)
#define GPIO_C_RST_ASSERT	(GPIO_C_RST + SRST_CLR)
#define GPIO_C_RST_NEGATE	(GPIO_C_RST + SRST_SET)
#define GPIO_C_RST_BIT		SRST_GPIO_C_BIT
#define GPIO_C_RST_DLY		SRST_GPIO_C_DLY

#define GPIO_M_PCLK		CXD90XXX_CLKEN(CLKEN_GPIO_M)
#define GPIO_M_PCLK_SET		(GPIO_M_PCLK + CLK_SET)
#define GPIO_M_PCLK_CLR		(GPIO_M_PCLK + CLK_CLR)
#define GPIO_M_PCLK_BIT		CLKEN_GPIO_M_PCLK
#define GPIO_M_RST		CXD90XXX_RESET(SRST_GPIO_M_GRP)
#define GPIO_M_RST_ASSERT	(GPIO_M_RST + SRST_CLR)
#define GPIO_M_RST_NEGATE	(GPIO_M_RST + SRST_SET)
#define GPIO_M_RST_BIT		SRST_GPIO_M_BIT
#define GPIO_M_RST_DLY		SRST_GPIO_M_DLY

static UDIF_CHANNELS gpio[UDIF_NR_GPIO] = {
	[0]   = {
		.iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(0),CXD90XXX_GPIO_M(0), GPIO_IO_SIZE),
		.clock = UDIF_CLOCK_INIT(GPIO_M_PCLK_SET, GPIO_M_PCLK_CLR, GPIO_M_PCLK_BIT,
					 0, 0, 0,	0, 0, 0, 0),
		.reset = UDIF_RESET_INIT(GPIO_M_RST_ASSERT, GPIO_M_RST_NEGATE, GPIO_M_RST_BIT, GPIO_M_RST_DLY,
					 0, 0, 0, 0),
	},
	[1]   = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(1),CXD90XXX_GPIO_M(1), GPIO_IO_SIZE), },
	[2]   = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(2),CXD90XXX_GPIO_M(2), GPIO_IO_SIZE), },
	[3]   = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(3),CXD90XXX_GPIO_M(3), GPIO_IO_SIZE), },
	[4]   = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(4),CXD90XXX_GPIO_M(4), GPIO_IO_SIZE), },
	[5]   = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(5),CXD90XXX_GPIO_M(5), GPIO_IO_SIZE), },
	[6]   = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(6),CXD90XXX_GPIO_M(6), GPIO_IO_SIZE), },
	[7]   = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(7),CXD90XXX_GPIO_M(7), GPIO_IO_SIZE), },
	[8]   = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(8),CXD90XXX_GPIO_M(8), GPIO_IO_SIZE), },
	[9]   = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(9),CXD90XXX_GPIO_M(9), GPIO_IO_SIZE), },
	[10]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(10),CXD90XXX_GPIO_M(10),GPIO_IO_SIZE), },
	[11]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(11),CXD90XXX_GPIO_M(11),GPIO_IO_SIZE), },
	[12]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(12),CXD90XXX_GPIO_M(12),GPIO_IO_SIZE), },
	[13]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(13),CXD90XXX_GPIO_M(13),GPIO_IO_SIZE), },
	[14]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(14),CXD90XXX_GPIO_M(14),GPIO_IO_SIZE), },
	[15]  = {
		.iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(64),CXD90XXX_GPIO_C(0),GPIO_IO_SIZE),
		.clock = UDIF_CLOCK_INIT(GPIO_C_PCLK_SET, GPIO_C_PCLK_CLR, GPIO_C_PCLK_BIT,
					 0, 0, 0,	0, 0, 0, 0),
		.reset = UDIF_RESET_INIT(GPIO_C_RST_ASSERT, GPIO_C_RST_NEGATE, GPIO_C_RST_BIT, GPIO_C_RST_DLY,
					 0, 0, 0, 0),
	},
	[16]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(65),CXD90XXX_GPIO_C(1),GPIO_IO_SIZE), },
	[17]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(66),CXD90XXX_GPIO_C(2),GPIO_IO_SIZE), },
	[18]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(67),CXD90XXX_GPIO_C(3),GPIO_IO_SIZE), },
	[19]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(68),CXD90XXX_GPIO_C(4),GPIO_IO_SIZE), },
	[20]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(69),CXD90XXX_GPIO_C(5),GPIO_IO_SIZE), },
	[21]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(70),CXD90XXX_GPIO_C(6),GPIO_IO_SIZE), },
	[22]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(71),CXD90XXX_GPIO_C(7),GPIO_IO_SIZE), },
	[23]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(72),CXD90XXX_GPIO_C(8),GPIO_IO_SIZE), },
	[24]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(73),CXD90XXX_GPIO_C(9),GPIO_IO_SIZE), },
	[25]  = { .iomem = __UDIF_IOMEM_INIT(GPIO_PORT_NAME(74),CXD90XXX_GPIO_C(10),GPIO_IO_SIZE), },
};

/* ldec stuff */
#define LDEC_NAME		"ldec"
#define LDEC_IO_BASE		0xF4045000U
#define LDEC_IO_SIZE		0x00001000U
#define LDEC_IRQ		IRQ_LDEC
#define LDEC_PCLK		CXD90XXX_CLKEN(CLKEN_LDEC)
#define LDEC_PCLK_SET		(LDEC_PCLK + CLK_SET)
#define LDEC_PCLK_CLR		(LDEC_PCLK + CLK_CLR)
#define LDEC_PCLK_BIT		CLKEN_LDEC_PCLK
#define LDEC_ACLK		LDEC_PCLK
#define LDEC_ACLK_SET		(LDEC_ACLK + CLK_SET)
#define LDEC_ACLK_CLR		(LDEC_ACLK + CLK_CLR)
#define LDEC_ACLK_BIT		CLKEN_LDEC_ACLK
#define LDEC_RST		CXD90XXX_RESET(SRST_LDEC_GRP)
#define LDEC_RST_ASSERT		(LDEC_RST + SRST_CLR)
#define LDEC_RST_NEGATE		(LDEC_RST + SRST_SET)
#define LDEC_RST_BIT		SRST_LDEC_BIT
#define LDEC_RST_DLY		SRST_LDEC_DLY

static UDIF_CHANNELS ldec[UDIF_NR_LDEC] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(LDEC_NAME, LDEC_IO_BASE, LDEC_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(LDEC_NAME, LDEC_IRQ, 0),
		.clock = UDIF_CLOCK_INIT(LDEC_PCLK_SET, LDEC_PCLK_CLR, LDEC_PCLK_BIT,
					 LDEC_ACLK_SET, LDEC_ACLK_CLR, LDEC_ACLK_BIT,
					 0, 0, 0, 0),
		.reset = UDIF_RESET_INIT(LDEC_RST_ASSERT, LDEC_RST_NEGATE, LDEC_RST_BIT, LDEC_RST_DLY,
					 0, 0, 0, 0),
	},
};

/* MMC stuff */
#define MMC_NAME 		"mmc"
#define MMC_SDIF0_AHB_BASE	0xF0CD8000U
#define MMC_SDIF0_AHB_SIZE	0x00001000U
#define MMC_SDIF0_AXI_BASE	0xF0CDA000U
#define MMC_SDIF0_AXI_SIZE	0x00002000U
#define MMC_SDIF1_AHB_BASE	0xF0CD9000U
#define MMC_SDIF1_AHB_SIZE	0x00001000U
#define MMC_SDIF1_AXI_BASE	0xF0CDC000U
#define MMC_SDIF1_AXI_SIZE	0x00002000U
#define MMC_MMC_IO_BASE		0xF0CDE000U
#define MMC_MMC_IO_SIZE		0x00001000U
#define TE_IRQ(x)		IRQ_SDIF(x)

/* MMC is mapped  */
static UDIF_CHANNELS mmc[UDIF_NR_MMC] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(MMC_NAME "0", MMC_SDIF0_AHB_BASE, MMC_SDIF0_AHB_SIZE),
		.intr = UDIF_INTRPT_INIT(MMC_NAME "0", TE_IRQ(0)+0, 0),
	},
	[1] = {
		.iomem = __UDIF_IOMEM_INIT(MMC_NAME "1", MMC_SDIF0_AXI_BASE, MMC_SDIF0_AXI_SIZE),
		.intr = UDIF_INTRPT_INIT(MMC_NAME "1", TE_IRQ(0)+1, 0),
	},
	[2] = {
		.iomem = __UDIF_IOMEM_INIT(MMC_NAME "2", MMC_SDIF1_AHB_BASE, MMC_SDIF1_AHB_SIZE),
		.intr = UDIF_INTRPT_INIT(MMC_NAME "2", TE_IRQ(1)+0, 0),
	},
	[3] = {
		.iomem = __UDIF_IOMEM_INIT(MMC_NAME "3", MMC_SDIF1_AXI_BASE, MMC_SDIF1_AXI_SIZE),
		.intr = UDIF_INTRPT_INIT(MMC_NAME "3", TE_IRQ(1)+1, 0),
	},
	[4] = {
		.iomem = __UDIF_IOMEM_INIT(MMC_NAME "4", MMC_MMC_IO_BASE, MMC_MMC_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(MMC_NAME "4", TE_IRQ(2), 0),
	},
};

/* eMMC0 stuff */
#define EMMC0_NAME 		"emmc0"
#define EMMC0_IO_BASE		0xFC445000U
#define EMMC0_IO_SIZE		0x00002000U

static UDIF_CHANNELS emmc0[UDIF_NR_EMMC0] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(EMMC0_NAME, EMMC0_IO_BASE, EMMC0_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(EMMC0_NAME, IRQ_EMMC0, 0),
	},
};

/* MS stuff */
#define MS_NAME			"ms"
#define MS_IO_BASE		0xF0CDF000U
#define MS_IO_SIZE		0x1000
#define MS_IRQ			IRQ_MS0
#define MS_HCLK			CXD90XXX_CLKEN(CLKEN_MS)
#define MS_HCLK_SET		(MS_HCLK + CLK_SET)
#define MS_HCLK_CLR		(MS_HCLK + CLK_CLR)
#define MS_HCLK_BIT		CLKEN_MS_HCLK
#define MS_DCLK			MS_HCLK
#define MS_DCLK_SET		(MS_DCLK + CLK_SET)
#define MS_DCLK_CLR		(MS_DCLK + CLK_CLR)
#define MS_DCLK_BIT		CLKEN_MS_DCLK
#define MS_RST			CXD90XXX_RESET(SRST_MS_GRP)
#define MS_RST_ASSERT		(MS_RST + SRST_CLR)
#define MS_RST_NEGATE		(MS_RST + SRST_SET)
#define MS_RST_BIT		SRST_MS_BIT
#define MS_RST_DLY		SRST_MS_DLY

static UDIF_CHANNELS ms[UDIF_NR_MS] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(MS_NAME, MS_IO_BASE, MS_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(MS_NAME, MS_IRQ, 0),
		.clock = UDIF_CLOCK_INIT(MS_HCLK_SET, MS_HCLK_CLR, MS_HCLK_BIT,
					 MS_DCLK_SET, MS_DCLK_CLR, MS_DCLK_BIT,
					 0, 0, 0, 0),
		.reset = UDIF_RESET_INIT(MS_RST_ASSERT, MS_RST_NEGATE, MS_RST_BIT, MS_RST_DLY,
					 0, 0, 0, 0),
	},
};

/* SIO */
#define SIO_NAME        	"sio"
#define SIO_C_BASE		0xFC5A6000U
#define SIO_M_BASE		0xF1A3B000U
#define SIO_IO_SIZE     	0x1000
#define SIO_C_PCLK		CXD90XXX_CLKEN(CLKEN_SIO_C)
#define SIO_C_PCLK_SET		(SIO_C_PCLK + CLK_SET)
#define SIO_C_PCLK_CLR		(SIO_C_PCLK + CLK_CLR)
#define SIO_C_PCLK_BIT		CLKEN_SIO_C_PCLK
#define SIO_C_DCLK		SIO_C_PCLK
#define SIO_C_DCLK_SET		(SIO_C_DCLK + CLK_SET)
#define SIO_C_DCLK_CLR		(SIO_C_DCLK + CLK_CLR)
#define SIO_C_DCLK_BIT		CLKEN_SIO_C_DCLK
#define SIO_C_RST		CXD90XXX_RESET(SRST_SIO_C_GRP)
#define SIO_C_RST_ASSERT	(SIO_C_RST + SRST_CLR)
#define SIO_C_RST_NEGATE	(SIO_C_RST + SRST_SET)
#define SIO_C_RST_BIT		SRST_SIO_C_BIT
#define SIO_C_RST_DLY		SRST_SIO_C_DLY

#define SIO_M_PCLK		CXD90XXX_CLKEN(CLKEN_SIO_M)
#define SIO_M_PCLK_SET		(SIO_M_PCLK + CLK_SET)
#define SIO_M_PCLK_CLR		(SIO_M_PCLK + CLK_CLR)
#define SIO_M_PCLK_BIT		CLKEN_SIO_M_PCLK
#define SIO_M_DCLK		SIO_M_PCLK
#define SIO_M_DCLK_SET		(SIO_M_DCLK + CLK_SET)
#define SIO_M_DCLK_CLR		(SIO_M_DCLK + CLK_CLR)
#define SIO_M_DCLK_BIT		CLKEN_SIO_M_DCLK
#define SIO_M_RST		CXD90XXX_RESET(SRST_SIO_M_GRP)
#define SIO_M_RST_ASSERT	(SIO_M_RST + SRST_CLR)
#define SIO_M_RST_NEGATE	(SIO_M_RST + SRST_SET)
#define SIO_M_RST_BIT		SRST_SIO_M_BIT
#define SIO_M_RST_DLY		SRST_SIO_M_DLY

static UDIF_CHANNELS sio[UDIF_NR_SIO] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(SIO_NAME "0", SIO_C_BASE, SIO_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(SIO_NAME "0", IRQ_SIO_C, 0),
		.clock = UDIF_CLOCK_INIT(SIO_C_PCLK_SET, SIO_C_PCLK_CLR, SIO_C_PCLK_BIT,
					 SIO_C_DCLK_SET, SIO_C_DCLK_CLR, SIO_C_DCLK_BIT,
					 0, 0, 0, 0),
		.reset = UDIF_RESET_INIT(SIO_C_RST_ASSERT, SIO_C_RST_NEGATE, SIO_C_RST_BIT, SIO_C_RST_DLY,
					 0, 0, 0, 0),
	},
	[1] = {
		.iomem = __UDIF_IOMEM_INIT(SIO_NAME "1", SIO_M_BASE, SIO_IO_SIZE),
		.intr = UDIF_INTRPT_INIT(SIO_NAME "1", IRQ_SIO_M, 0),
		.clock = UDIF_CLOCK_INIT(SIO_M_PCLK_SET, SIO_M_PCLK_CLR, SIO_M_PCLK_BIT,
					 SIO_M_DCLK_SET, SIO_M_DCLK_CLR, SIO_M_DCLK_BIT,
					 0, 0, 0, 0),
		.reset = UDIF_RESET_INIT(SIO_M_RST_ASSERT, SIO_M_RST_NEGATE, SIO_M_RST_BIT, SIO_M_RST_DLY,
					 0, 0, 0, 0),
	},
};

/* SATA */
#define SATA_NAME		"sata"

/* cmmc */
#define CMMC_NAME		"cmmc"
#define CMMC_REG_BASE		0xF4061000U
#define CMMC_REG_SIZE		0x1000U
#define CMMC_FIFO_BASE		0xF4000000U
#define CMMC_FIFO_SIZE		0x40000U
#define CMMC_REG_NAME		CMMC_NAME "_reg"
#define CMMC_FIFO_NAME		CMMC_NAME "_buf"
#define CMMC_PCLK		CXD90XXX_CLKEN(CLKEN_CMMC)
#define CMMC_PCLK_SET		(CMMC_PCLK + CLK_SET)
#define CMMC_PCLK_CLR		(CMMC_PCLK + CLK_CLR)
#define CMMC_PCLK_BIT		CLKEN_CMMC_PCLK
#define CMMC_ACLK		CMMC_PCLK
#define CMMC_ACLK_SET		(CMMC_ACLK + CLK_SET)
#define CMMC_ACLK_CLR		(CMMC_ACLK + CLK_CLR)
#define CMMC_ACLK_BIT		CLKEN_CMMC_ACLK
#define CMMC_RST		CXD90XXX_RESET(SRST_CMMC_GRP)
#define CMMC_RST_ASSERT		(CMMC_RST + SRST_CLR)
#define CMMC_RST_NEGATE		(CMMC_RST + SRST_SET)
#define CMMC_RST_BIT		SRST_CMMC_BIT
#define CMMC_RST_DLY		SRST_CMMC_DLY

static UDIF_CHANNELS cmmc[UDIF_NR_CMMC] = {
	[UDIF_CMMC_CH_REG] = {
		.iomem = __UDIF_IOMEM_INIT(CMMC_REG_NAME, CMMC_REG_BASE, CMMC_REG_SIZE),
		.clock = UDIF_CLOCK_INIT(CMMC_PCLK_SET, CMMC_PCLK_CLR, CMMC_PCLK_BIT,
					 CMMC_ACLK_SET, CMMC_ACLK_CLR, CMMC_ACLK_BIT,
					 0, 0, 0, 0),
		.reset = UDIF_RESET_INIT(CMMC_RST_ASSERT, CMMC_RST_NEGATE, CMMC_RST_BIT, CMMC_RST_DLY,
					 0, 0, 0, 0),
	},
	[UDIF_CMMC_CH_INBUF] = {
		.iomem = __UDIF_IOMEM_INIT(CMMC_FIFO_NAME, CMMC_FIFO_BASE, CMMC_FIFO_SIZE),
	},
};

/* FUSB */
/* fusb stuff */
#define FUSB_NAME		"fusb"
/* fusb CH0 stuff */
#define FUSB_CH0_NAME		"usbgpio"
#define FUSB_CH0_IO_BASE	0xFC5E1E00
#define FUSB_CH0_IO_SIZE	0x100

static UDIF_CHANNELS fusb[UDIF_NR_FUSB] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(FUSB_CH0_NAME, FUSB_CH0_IO_BASE, FUSB_CH0_IO_SIZE),
	},
};

/* DDRC stuff */
#define DDRC_NAME 	"ddrc"
#define DDRC_BASE(x)	CXD90XXX_DDRPHY(x)
#define DDRC_SIZE	0x400000

static UDIF_CHANNELS ddrc[UDIF_NR_DDR] = {
	[0] = {	.iomem = __UDIF_IOMEM_INIT(DDRC_NAME "0", DDRC_BASE(0), DDRC_SIZE), },
	[1] = {	.iomem = __UDIF_IOMEM_INIT(DDRC_NAME "1", DDRC_BASE(1), DDRC_SIZE), },
};

/* FECCNT stuff */
#define FECCNT_NAME		"feccnt"
#define FECCNT_BASE		0xF4046000U
#define FECCNT_SIZE		0x1000
#define FECCNT_IRQ		270
#define FECCNT_PCLK		CXD90XXX_CLKEN(CLKEN_FECCNT)
#define FECCNT_PCLK_SET		(FECCNT_PCLK + CLK_SET)
#define FECCNT_PCLK_CLR		(FECCNT_PCLK + CLK_CLR)
#define FECCNT_PCLK_BIT		CLKEN_FECCNT_PCLK
#define FECCNT_ACLK		FECCNT_PCLK
#define FECCNT_ACLK_SET		(FECCNT_ACLK + CLK_SET)
#define FECCNT_ACLK_CLR		(FECCNT_ACLK + CLK_CLR)
#define FECCNT_ACLK_BIT		CLKEN_FECCNT_ACLK
#define FECCNT_RST		CXD90XXX_RESET(SRST_FECCNT_GRP)
#define FECCNT_RST_ASSERT	(FECCNT_RST + SRST_CLR)
#define FECCNT_RST_NEGATE	(FECCNT_RST + SRST_SET)
#define FECCNT_RST_BIT		SRST_FECCNT_BIT
#define FECCNT_RST_DLY		SRST_FECCNT_DLY

static UDIF_CHANNELS feccnt[UDIF_NR_FECCNT] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(FECCNT_NAME, FECCNT_BASE, FECCNT_SIZE),
		.intr = UDIF_INTRPT_INIT(FECCNT_NAME, FECCNT_IRQ, 0),
		.clock = UDIF_CLOCK_INIT(FECCNT_PCLK_SET, FECCNT_PCLK_CLR, FECCNT_PCLK_BIT,
					 FECCNT_ACLK_SET, FECCNT_ACLK_CLR, FECCNT_ACLK_BIT,
					 0, 0, 0, 0),
		.reset = UDIF_RESET_INIT(FECCNT_RST_ASSERT, FECCNT_RST_NEGATE, FECCNT_RST_BIT, FECCNT_RST_DLY,
					 0, 0, 0, 0),
	},
};

/* GPIOINT stuff */
#define GPIOINT_NAME		"gpioint"
#define GPIOINT_BASE		0xFC594000U
#define GPIOINT_SIZE		0x1000
#define GPIOINT_PCLK		CXD90XXX_CLKEN(CLKEN_GPIOINT_C)
#define GPIOINT_PCLK_SET	(GPIOINT_PCLK + CLK_SET)
#define GPIOINT_PCLK_CLR	(GPIOINT_PCLK + CLK_CLR)
#define GPIOINT_PCLK_BIT	CLKEN_GPIOINT_C_PCLK
#define GPIOINT_RST		CXD90XXX_RESET(SRST_GPIOINT_C_GRP)
#define GPIOINT_RST_ASSERT	(GPIOINT_RST + SRST_CLR)
#define GPIOINT_RST_NEGATE	(GPIOINT_RST + SRST_SET)
#define GPIOINT_RST_BIT		SRST_GPIOINT_C_BIT
#define GPIOINT_RST_DLY		SRST_GPIOINT_C_DLY

static UDIF_CHANNELS gpioint[UDIF_NR_GPIOINT] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(GPIOINT_NAME, GPIOINT_BASE, GPIOINT_SIZE),
		.clock = UDIF_CLOCK_INIT(GPIOINT_PCLK_SET, GPIOINT_PCLK_CLR, GPIOINT_PCLK_BIT,
					 0, 0, 0,   0, 0, 0, 0),
		.reset = UDIF_RESET_INIT(GPIOINT_RST_ASSERT, GPIOINT_RST_NEGATE, GPIOINT_RST_BIT, GPIOINT_RST_DLY,
					 0, 0, 0, 0),
	},
};

/* CPUINT stuff */
#define CPUINT_NAME		"cpuint"
#define CPUINT_BASE		0xFC595000U
#define CPUINT_SIZE		0x1000
#define CPUINT_PCLK		CXD90XXX_CLKEN(CLKEN_CPUINT_M)
#define CPUINT_PCLK_SET		(CPUINT_PCLK + CLK_SET)
#define CPUINT_PCLK_CLR		(CPUINT_PCLK + CLK_CLR)
#define CPUINT_PCLK_BIT		CLKEN_CPUINT_M_PCLK
#define CPUINT_RST		CXD90XXX_RESET(SRST_CPUINT_M_GRP)
#define CPUINT_RST_ASSERT	(CPUINT_RST + SRST_CLR)
#define CPUINT_RST_NEGATE	(CPUINT_RST + SRST_SET)
#define CPUINT_RST_BIT		SRST_CPUINT_M_BIT
#define CPUINT_RST_DLY		SRST_CPUINT_M_DLY

static UDIF_CHANNELS cpuint[UDIF_NR_CPUINT] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(CPUINT_NAME, CPUINT_BASE, CPUINT_SIZE),
		.clock = UDIF_CLOCK_INIT(CPUINT_PCLK_SET, CPUINT_PCLK_CLR, CPUINT_PCLK_BIT,
					 0, 0, 0,   0, 0, 0, 0),
		.reset = UDIF_RESET_INIT(CPUINT_RST_ASSERT, CPUINT_RST_NEGATE, CPUINT_RST_BIT, CPUINT_RST_DLY,
					 0, 0, 0, 0),
	},
};

/* ADSP stuff */
#define ADSP_NAME		"adsp"
#define ADSP_IRQ_NAME		"adsp_irq"
#define ADSP_BASE		0xFC490000U
#define ADSP_SIZE		0x400
#define ADSP_IRQ		315

static UDIF_CHANNELS adsp[UDIF_NR_ADSP] = {
	[0] = {
		.iomem = __UDIF_IOMEM_INIT(ADSP_NAME, ADSP_BASE, ADSP_SIZE),
		.intr = UDIF_INTRPT_INIT(ADSP_IRQ_NAME, ADSP_IRQ, 0),
	},
};


/*
 * All Devices
 */
UDIF_DEVICE udif_devices[UDIF_ID_NUM] = {
	[UDIF_ID_HWTIMER] = {
		.name	= HWTMR_NAME,
		.nr_ch	= NARRAY(hwtimer),
		.chs	= hwtimer,
	},
	[UDIF_ID_GPIO] = {
		.name	= GPIO_NAME,
		.nr_ch	= NARRAY(gpio),
		.chs	= gpio,
	},
	[UDIF_ID_LDEC] = {
		.name   = LDEC_NAME,
		.nr_ch  = NARRAY(ldec),
		.chs    = ldec,
	},
	[UDIF_ID_MMC] = {
		.name	= MMC_NAME,
		.nr_ch 	= NARRAY(mmc),
		.chs	= mmc,
	},
	[UDIF_ID_EMMC0] = {
		.name	= EMMC0_NAME,
		.nr_ch 	= NARRAY(emmc0),
		.chs	= emmc0,
	},
	[UDIF_ID_MS] = {
		.name   = MS_NAME,
		.nr_ch  = NARRAY(ms),
		.chs    = ms,
	},
        [UDIF_ID_SIO] = {
                .name   = SIO_NAME,
                .nr_ch = NARRAY(sio),
                .chs    = sio,
        },
	[UDIF_ID_SATA] = {
		.name   = SATA_NAME,
		.nr_ch = 0,
	},
	[UDIF_ID_CMMC] = {
		.name   = CMMC_NAME,
		.nr_ch  = NARRAY(cmmc),
		.chs    = cmmc,
	},
	[UDIF_ID_FUSB_OTG] = {
		.name   = FUSB_NAME,
		.nr_ch  = NARRAY(fusb),
		.chs    = fusb,
	},
	[UDIF_ID_WDT] = {
		.name	= WDT_NAME,
		.nr_ch	= NARRAY(wdt),
		.chs	= wdt,
	},
	[UDIF_ID_DDR] = {
		.name	= DDRC_NAME,
		.nr_ch	= NARRAY(ddrc),
		.chs	= ddrc,
	},
	[UDIF_ID_FECCNT] = {
		.name	= FECCNT_NAME,
		.nr_ch	= NARRAY(feccnt),
		.chs	= feccnt,
	},
	[UDIF_ID_GPIOINT] = {
		.name	= GPIOINT_NAME,
		.nr_ch	= NARRAY(gpioint),
		.chs	= gpioint,
	},
	[UDIF_ID_CPUINT] = {
		.name	= CPUINT_NAME,
		.nr_ch	= NARRAY(cpuint),
		.chs	= cpuint,
	},
	[UDIF_ID_ADSP] = {
		.name	= ADSP_NAME,
		.nr_ch	= NARRAY(adsp),
		.chs	= adsp,
	},
};

EXPORT_SYMBOL(udif_devices);

static int udif_platform_register(void)
{
	int i, ret = 0;

	for (i = 0; i < NARRAY(udif_devices); i++)
		ret = udif_device_register(&udif_devices[i]);

	return ret;
}

arch_initcall(udif_platform_register);

/*--------------- allocate virqs --------------------*/
#include <linux/irqdomain.h>
#include <dt-bindings/interrupt-controller/arm-gic.h>
#define TO_GIC_SPI(x)	((x) - IRQ_GIC_START)
#define N_PARAM 3
#define  PRM_CLS 0
#define  PRM_IRQ 1
#define  PRM_TYP 2

static void udif_alloc_irqs(struct irq_domain *domain, int irq_base, unsigned int nr_irqs)
{
	struct device_node *of_node;
	struct irq_fwspec fwspec;
	int nid;
	int ret;

	if (!nr_irqs)
		return;

	of_node = irq_domain_get_of_node(domain);
	nid = of_node_to_nid(of_node);
	fwspec.fwnode = domain->fwnode;
	fwspec.param_count = N_PARAM;
	fwspec.param[PRM_CLS] = GIC_SPI;
	fwspec.param[PRM_IRQ] = TO_GIC_SPI(irq_base);
	fwspec.param[PRM_TYP] = IRQ_TYPE_LEVEL_HIGH; /* dummy */
	ret = __irq_domain_alloc_irqs(domain, irq_base, nr_irqs, nid, &fwspec, false, NULL);
	if (ret < 0) {
		printk(KERN_ERR "%s: cannot map IRQ%d-%d\n", __func__,
		       irq_base, irq_base + nr_irqs - 1);
	}
}

void __init udif_map_irq(struct irq_domain *domain)
{
	int i;
	UDIF_DEVICE *dev;
	struct device_node *of_node;
	struct irq_fwspec fwspec;
	int nid;
	extern void cxd90xxx_gic_init(void);

	of_node = irq_domain_get_of_node(domain);
	nid = of_node_to_nid(of_node);
	fwspec.fwnode = domain->fwnode;
	fwspec.param_count = N_PARAM;
	fwspec.param[PRM_CLS] = GIC_SPI;
	fwspec.param[PRM_TYP] = IRQ_TYPE_LEVEL_HIGH; /* dummy */
	for (i = 0, dev = udif_devices; i < NARRAY(udif_devices); i++, dev++) {
		int ch;
		for (ch = 0; ch < dev->nr_ch; ch++) {
			int hwirq, ret;

			hwirq = dev->chs[ch].intr.irq;
			if (!hwirq)
				continue;

			fwspec.param[PRM_IRQ] = TO_GIC_SPI(hwirq);
			ret = __irq_domain_alloc_irqs(domain, hwirq, 1, nid, &fwspec, false, NULL);
			if (ret < 0) {
				printk(KERN_ERR "%s: cannot map IRQ%d\n",
				       __func__, hwirq);
			}
		}
	}
	/* GPIO INT */
	udif_alloc_irqs(domain, IRQ_GPIO(0), NR_GPIO_IRQ);
	/* CPUINT */
	udif_alloc_irqs(domain, IRQ_CPUINT(0), NR_CPUINT_IRQ);
	/* others */
	udif_alloc_irqs(domain, IRQ_GIC_START, 40);
	udif_alloc_irqs(domain, 113, 7);
	udif_alloc_irqs(domain, 150, 6);
	udif_alloc_irqs(domain, 303, 1);
	udif_alloc_irqs(domain, 80, 4);

	cxd90xxx_gic_init();
}
