// SPDX-License-Identifier: GPL-2.0
/*
 *  sysrq_dmesg: dump dmesg to console
 *
 *  Copyright 2021 Sony Group Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, see <https://www.gnu.org/licenses>.
 */

#include <linux/kernel.h>
#include <linux/workqueue.h>
#include <linux/kmsg_dump.h>
#include <linux/console.h>
#include <linux/sysrq.h>

#define SYSRQ_DMESG_LINE_BUF_SZ  512

/*
 * Dmesg dump callback function
 */
static void dmesg_dump_callback(struct work_struct *work)
{
	struct kmsg_dump_iter iter;
	size_t len;
	char buf[SYSRQ_DMESG_LINE_BUF_SZ];
	struct console *co;

	kmsg_dump_rewind(&iter);
	while (kmsg_dump_get_line(&iter, 1, buf, sizeof(buf), &len)){
		/*
		 * Since using printk() or pr_*() will append the message to the
		 * dmesg ring buffer, they cannot be used to display the retrieved
		 * message. Hence console_write() of serial drivers is used.
		 */
		console_lock();
		for (co = console_drivers; co; co = co->next) {
			if ((co->flags & CON_ENABLED) && co->write) {
				co->write(co, buf, len);
			}
		}
		console_unlock();
	}
}

/* Initialize Sysrq dmesg work */
static DECLARE_WORK(sysrq_dmesg_work, dmesg_dump_callback);

/*
 * Sysrq key interrupt handler
 */
static void sysrq_handle_dmesg_dump(int key)
{
	queue_work(system_unbound_wq, &sysrq_dmesg_work);
}

static struct sysrq_key_op sysrq_dmesg_dump_op = {
	.handler        = sysrq_handle_dmesg_dump,
	.help_msg       = "dump-dmesg(y)",
	.action_msg     = "Dump dmesg",
	.enable_mask    = SYSRQ_ENABLE_DUMP,
};

/*
 * Register sysrq key to dump dmesg
 */
static int __init sysrq_dmesg_init(void)
{
	int ret;

	ret = register_sysrq_key('y', &sysrq_dmesg_dump_op);
	if (ret){
		printk("Sysrq dmesg dump key registration failed\n");
		return ret;
	}

	return 0;
}
device_initcall(sysrq_dmesg_init);
