// SPDX-License-Identifier: GPL-2.0
/*
 * drivers/udif/bdev.c
 *
 * UDM
 *
 * Copyright 2024 Sony Corporation
 *
 */
#include <linux/genhd.h>
#include <linux/blkdev.h>
#include <linux/suspend.h>

void __udif_bdev_suspend(struct gendisk *disk)
{
#ifdef CONFIG_WARM_BOOT_IMAGE
	int i;

	if (pm_get_state() != PM_SUSPEND_DISK)
		return;

	for (i = 0; i < disk_max_parts(disk); i++) {
		dev_t dev;
		struct block_device *bdev;
		struct super_block *sb;

		dev = part_devt(disk, i);
		if (!dev)
			continue;
		bdev = blkdev_get_no_open(dev);
		if (!bdev)
			continue;
		sb = get_super(bdev);
		if (!sb)
			goto put_bdev;
		if (sb_rdonly(sb))
			goto put_sb;

		printk("sync %s partition %d\n", disk->disk_name, i);
		sync_filesystem(sb);

	put_sb:
		drop_super(sb);
	put_bdev:
		blkdev_put_no_open(bdev);
	}
#endif /* CONFIG_WARM_BOOT_IMAGE */
}

#ifdef CONFIG_SNSC_SAFE_SUSPEND
extern bool is_suspend_remount(struct super_block *sb);
#endif /* CONFIG_SNSC_SAFE_SUSPEND */

void __udif_bdev_resume(struct gendisk *disk)
{
#ifdef CONFIG_WARM_BOOT_IMAGE
	int i;

	if (pm_get_state() != PM_SUSPEND_DISK || pm_is_mem_alive())
		return;

	for (i = 0; i < disk_max_parts(disk); i++) {
		dev_t dev;
		struct block_device *bdev;
		struct super_block *sb;

		dev = part_devt(disk, i);
		if (!dev)
			continue;
		bdev = blkdev_get_no_open(dev);
		if (!bdev)
			continue;
		sb = get_super(bdev);
		if (!sb)
			goto put_bdev;
		if (sb_rdonly(sb)) {
#ifdef CONFIG_SNSC_SAFE_SUSPEND
		    if (!is_suspend_remount(sb)) {
#endif /* CONFIG_SNSC_SAFE_SUSPEND */
			drop_super(sb);
			goto put_bdev;
#ifdef CONFIG_SNSC_SAFE_SUSPEND
		    }
#endif /* CONFIG_SNSC_SAFE_SUSPEND */
		}
		drop_super(sb);

		printk("invalidate %s partition %d\n", disk->disk_name, i);
		__invalidate_device(bdev, true);

	put_bdev:
		blkdev_put_no_open(bdev);
	}
#endif /* CONFIG_WARM_BOOT_IMAGE */
}
EXPORT_SYMBOL(__udif_bdev_suspend);
EXPORT_SYMBOL(__udif_bdev_resume);
