/*
 * drivers/udif/proc.c
 *
 * UDM
 *
 * Copyright 2018 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
/*
 * proc/fs/generic.c --- generic routines for the proc-fs
 *
 * This file contains generic proc-fs routines for handling
 * directories and files.
 *
 * Copyright (C) 1991, 1992 Linus Torvalds.
 * Copyright (C) 1997 Theodore Ts'o
 */
#include <linux/version.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/module.h>
#include <linux/udif/proc.h>
#include <linux/udif/macros.h>

#define DIR_NAME "udm"

#define UDIF_DECLARE_PROC_READ(name, b, c, p, e, d) \
struct UDIF_PROC_READ name = { \
	.kaddr	= b, \
	.count	= c, \
	.pos	= p, \
	.eof	= e, \
	.data	= d, \
}

#define UDIF_DECLARE_PROC_WRITE(name, b, c, d) \
struct UDIF_PROC_WRITE name = { \
	.uaddr	= b, \
	.count	= c, \
	.data	= d, \
}

static struct proc_dir_entry *udif_proc_dir = NULL;

struct udif_seq_private {
	UDIF_PROC *proc;
	UDIF_OFF pos;
	int eof;
	int ret;
};

static void *m_start(struct seq_file *m, loff_t *ppos)
{
	struct udif_seq_private *priv = m->private;
	UDIF_PROC *proc = priv->proc;

	if (!proc || !proc->read)
		return NULL;
	if (!*ppos) {
		priv->pos = 0;
		priv->eof = 0;
	}
	return (priv->eof) ? NULL : ppos;
}

static void *m_next(struct seq_file *m, void *v, loff_t *ppos)
{
	struct udif_seq_private *priv = m->private;

	(*ppos)++;
	return (priv->ret <= 0 || priv->eof) ? NULL : ppos;
}

static void m_stop(struct seq_file *m, void *v)
{
	return;
}

static int m_show(struct seq_file *m, void *v)
{
	struct udif_seq_private *priv = m->private;
	UDIF_PROC *proc = priv->proc;
	struct UDIF_PROC_READ rd;
	char *buf;

	if (priv->eof)
		return 0;
	rd.count = seq_get_buf(m, &buf);
	rd.kaddr = buf;
	rd.pos   = &priv->pos;
	rd.eof   = &priv->eof;
	rd.data  = proc->data;
	priv->ret = proc->read(&rd);
	if (priv->ret <= 0)
		return priv->ret;
	seq_commit(m, priv->ret);
	return 0;
}

static const struct seq_operations udif_seq_ops = {
	.start	= m_start,
	.next	= m_next,
	.stop	= m_stop,
	.show	= m_show,
};

static int udif_proc_open(struct inode *inode, struct file *filp)
{
	UDIF_PROC *proc = PDE_DATA(inode);
	struct udif_seq_private *priv;

	priv = __seq_open_private(filp, &udif_seq_ops,
				  sizeof(struct udif_seq_private));
	if (!priv)
		return -ENOMEM;
	/* priv is allocated by kzalloc */
	priv->proc = proc;
	return 0;
}

static ssize_t
udif_proc_write(struct file *file, const char __user *buffer,
		size_t count, loff_t *ppos)
{
	UDIF_PROC *proc = PDE_DATA(file_inode(file));
	int ret = count;

	if (likely(proc) && proc->write) {
		UDIF_DECLARE_PROC_WRITE(wr, buffer, count, proc->data);
		ret = proc->write(&wr);
	}

	return ret;
}

static const struct proc_ops proc_fops = {
	.proc_open	= udif_proc_open,
	.proc_read_iter	= seq_read_iter,
	.proc_write	= udif_proc_write,
	.proc_lseek	= seq_lseek,
	.proc_release= seq_release_private,
};

UDIF_ERR udif_create_proc(UDIF_PROC *proc)
{
	struct proc_dir_entry *ent;

	UDIF_PARM_CHK(!proc, "proc is NULL", UDIF_ERR_PAR);

	if (unlikely(!udif_proc_dir)) {
		UDIF_PERR("not initialize udif_proc_dir\n");
		return UDIF_ERR_IO;
	}

	ent = proc_create_data(proc->name, S_IRUGO, udif_proc_dir,
			       &proc_fops, proc);
	if (unlikely(!ent)) {
		UDIF_PERR("cannot create proc entry %s\n", proc->name);
		return UDIF_ERR_NOMEM;
	}

	return UDIF_ERR_OK;
}

UDIF_ERR udif_remove_proc(UDIF_PROC *proc)
{
	UDIF_PARM_CHK(!proc, "proc is NULL", UDIF_ERR_PAR);

	if (unlikely(!udif_proc_dir)) {
		UDIF_PERR("not initialize udif_proc_dir\n");
		return UDIF_ERR_IO;
	}

	remove_proc_entry(proc->name, udif_proc_dir);

	return UDIF_ERR_OK;
}

EXPORT_SYMBOL(udif_create_proc);
EXPORT_SYMBOL(udif_remove_proc);

static int __init udif_init_proc(void)
{
	udif_proc_dir = proc_mkdir(DIR_NAME, NULL);
	if (unlikely(!udif_proc_dir)) {
		UDIF_PERR("cannot proc_mkdir\n");
		return -ENOMEM;
	}

	return 0;
}

postcore_initcall(udif_init_proc);

#if 0
static void __exit udif_exit_proc(void)
{
	remove_proc_entry(DIR_NAME, NULL);
	udif_proc_dir = NULL;
}
module_exit(udif_exit_proc);
#endif
