// SPDX-License-Identifier: GPL-2.0
/*
 * usb2_smmu.c
 *
 * Copyright 2022 Sony Corporation, SOCIONEXT INC.
 *
 */

#include <linux/dma-mapping.h>
#include <linux/mutex.h>
#include <linux/errno.h>
#include <linux/list.h>
#include <linux/wait.h>
#include <linux/semaphore.h>
#include <linux/slab.h>
#include <linux/usb.h>
#include <linux/usb/hcd.h>
#include <linux/platform_device.h>
#include "sdebug.h"
#include "usb2_smmu.h"

//#define TESTCODE_USB_DWC2_SMMU_LOCK
#ifdef TESTCODE_USB_DWC2_SMMU_LOCK
int __attribute__((optimize("O0")))l_start_smmu_alloc(char *arg, ...) { return 0; }
int __attribute__((optimize("O0")))l_end_smmu_alloc(char *arg, ...) { return 0; }
EXPORT_SYMBOL_GPL(l_start_smmu_alloc);
EXPORT_SYMBOL_GPL(l_end_smmu_alloc);
int __attribute__((optimize("O0")))l_start_smmu_free_a(char *arg, ...) { return 0; }
int __attribute__((optimize("O0")))l_end_smmu_free_a(char *arg, ...) { return 0; }
EXPORT_SYMBOL_GPL(l_start_smmu_free_a);
EXPORT_SYMBOL_GPL(l_end_smmu_free_a);
int __attribute__((optimize("O0")))l_start_smmu_free(char *arg, ...) { return 0; }
int __attribute__((optimize("O0")))l_end_smmu_free(char *arg, ...) { return 0; }
EXPORT_SYMBOL_GPL(l_start_smmu_free);
EXPORT_SYMBOL_GPL(l_end_smmu_free);
#endif

#ifdef CONFIG_USB2_SMMU_LIB
// debug macro ------------------------------------------------
//#define SMMU_DEBUG_SECURE		// set also secure alias reg
//#define SMMU_DEBUG_BYPASS		// bypass, not translation
//#define SMMU_DEBUG_ALLCO_TABLE	// print information of control allocated table

// defines ----------------------------------------------------
#define SMMU_PAGE_64K		(16384U)	// 16K granule
#define SMMU_PAGE_4K		(4096U)	// 4K granule
#define SMMU_L1_ENTRY_CNT	(4U)			// 4 pieces of pointer to L2 table
#define SMMU_ENTRY_SIZE		(8U)			// byte/entry

#define SMMU_32RANGE_ALL_PAGECNT		(2048U)	// count of 2MB entry at 32bit usb2 address space, (4GB / 2MB - 1)

#define SMMU_64RANGE_BASE		(0x200000000UL)
#define SMMU_64RANGE_SIZE		(0x200000000UL)

#define PAGECNT_ONELINE		(64U)	// each bit of uint64_t value means allocated or not allocated

#define SMMU_BYTES_PER_ENTRY		(2*1024*1024U)	// page size of SMMU level2 translation entry
#define SMMU_BYTES_PER_ENTRY_SHIFT	(21)		// address space(2MB) on one of SMMU level2 translation entry
#define SMMU_ENTRY_ADDR_MASK		(~(SMMU_BYTES_PER_ENTRY -1))
#define UINT64_ADR_MASK				(0x3F)	// addr mask on uint64_t

#define SMMU_PAGE_OFFSET_GR1		(1)	// count of PAGE from SMMU_BASE to GR1 address space
#define SMMU_PAGE_OFFSET_SSD		(4)	// count of PAGE from SMMU_BASE to SSD address space

//descriptor format
#define PT_BLOCK    (0x1ULL)        // Block
#define PT_TABLE    (0x3ULL)        // Table/Page
// shareability
#define PT_OSH      (2<<8)      // outter shareable
#define PT_ISH      (3<<8)      // inner shareable
// accessibility
#define PT_AF       (1<<10)     // accessed flag
#define PT_RW       (0<<7)      // read-write
#define PT_RO       (1<<7)      // read-only
#define PT_KERNEL   (0<<6)      // privileged, supervisor EL1 access only
#define PT_USER     (1<<6)      // unprivileged, EL0 access allowed
#define PT_NS       (1UL<<5)    // non-secure
#define PT_NX       (1UL<<4)    // no execute
// defined in MAIR register
#define PT_MEM      (0<<2)      // normal memory
#define PT_DEV      (1<<2)      // device memory
#define PT_NC       (2<<2)      // non-cachable


//#define SMMU_RECORD_BLOCK_L1_ADDR_MASK	(0xFFFFC0000000)	// [47:30]
#define SMMU_RECORD_TABLE_L1_ADDR_MASK	(0xFFFFFFFFF000)	// [47:12]
#define SMMU_RECORD_BLOCK_L2_ADDR_MASK	(0xFFFFFFE00000)	// [47:21]

#define SMMU_MAKE_RECORD_BLOCK_L1_NS(ADDR)  ((ADDR&SMMU_RECORD_BLOCK_L1_ADDR_MASK) | PT_AF | PT_OSH | (PT_RO | PT_USER) | (PT_NX | PT_DEV) | PT_BLOCK | PT_NS)
#define SMMU_MAKE_RECORD_TABLE_L1(TBLADDR)    ((TBLADDR & SMMU_RECORD_TABLE_L1_ADDR_MASK) | PT_TABLE)
#define SMMU_MAKE_RECORD_BLOCK_L2_NS(ADDR)       ((ADDR&SMMU_RECORD_BLOCK_L2_ADDR_MASK) | PT_AF | PT_OSH | PT_RO | PT_USER | PT_NS | PT_DEV | PT_BLOCK | PT_NS)
#define SMMU_MAKE_RECORD_BLOCK_L2(ADDR)       ((ADDR&SMMU_RECORD_BLOCK_L2_ADDR_MASK) | PT_AF | PT_OSH | PT_RO | PT_USER | PT_NS | PT_DEV | PT_BLOCK)


// define smmu register -------------
// GR0 space
#define GR0_SCR			(0x000) 	// SMMU_SCR
#define GR0_SACR		(0x010)		// SMMU_SACR
#define GR0_IDR1		(0x024)		// SMMU_IDR1
#define		GR0_IDR1_PAGESIZE_MASK		(0x80000000)
#define 	GR0_IDR1_NUMPAGENDXB_MASK	(0x70000000)
#define 	GR0_IDR1_NUMPAGENDXB_SHIFT	(28)
#define GR0_GFAR        (0x040)    // SMMU_GFSR : Global Fault Address reg.
#define GR0_GFSR        (0x048)    // SMMU_GFSR : Global Fault Status reg.
#define     BIT_GR0_GFSR_SMCF            (0x00000004)    // Stream Match Fault
#define GR0_STLBIALL	(0x060)		// SMMU_STILBIALL
#define GR0_TLBIALLNSNH	(0x068)		// SMMU_TILBIALLNSNH
#define GR0_TLBGSSYNC   (0x070)     // SMMU_TLBGSTATUS : Synchronize TLB Invalidate
#define GR0_TLBGSTATUS  (0x074)     // SMMU_TLBGSTATUS : TLB Status
#define GR0_DBGRPTRTBU  (0x080)    // TBU-TLB Debug Read Pointer reg
#define GR0_DBGRDATATBU (0x084)    // TBU-TLB Debug Read Data reg
#define GR0_DBGRPTRTCU  (0x088)    // TCU-TLB Debug Read Pointer reg
#define GR0_DBGRDATATCU (0x08C)    // TBU-TLB Debug Read Data reg
#define GR0_NSACR		(0x410)		// SMMU_NSACR
#define GR0_SMR0		(0x800)		// SMMU_SMR0
#define GR0_SMR1		(0x804)		// SMMU_SMR1
#define GR0_S2CR0		(0xC00)		// SMMU_SMR0
#define GR0_S2CR1		(0xC04)		// SMMU_SMR1

// GR1 space
#define GR1_CBAR0		(0x0000)	// SMMU_CBAR0
#define GR1_CBAR1		(0x0004)	// SMMU_CBAR1
#define GR1_CBA2R0		(0x0800)	// SMMU_CBAR0
#define GR1_CBA2R1		(0x0804)	// SMMU_CBAR0

// SSD space
#define SSD_SSDR0		(0x0000)	// SMMU_SSDR0

// CB0 space
#define CB_TTBR0		(0x0020)	// SMMU_CB_TTBR0
#define CB_TCR			(0x0030)	// SMMU_CB_TCR
#define CB_MAIR0		(0x0038)	// SMMU_CB_MAIR0
#define CB_MAIR1		(0x0038)	// SMMU_CB_MAIR1
#define CB_FSR          (0x0058)    // SMMU_CB_FSR : Fault Status reg.
#define     BIT_CB_FSR_TF                 (0x00000002) // Translation Fault
#define CB_FAR          (0x0060)    // SMMU_CB_FAR : Fault Address reg.
#define CB_FSYNR0       (0x0068)    // SMMU_CB_FAR : Fault Syndrome reg.
#define CB_TLBIIPAS2	(0x0630)	// SMMU_CB_TLBIIPAS2
#define CB_TLBSYNC		(0x07F0)	// SMMU_CB_TLBSYNC
#define CB_TLBSTATUS	(0x07F4)	// SMMU_CB_TLBSTATUS

#ifdef CONFIG_USB2_SMMU_LIB
#define D_2MB_MSK	(0xFFFFFFFFFFE00000)
#define D_4GB_MSK	(0x00000000FFFFFFFF)
#define D_G_L2ID(x)	(x / PAGECNT_ONELINE)
#define D_G_L2BIT(x)	(1UL << (x & UINT64_ADR_MASK))
struct map_area64 {
	struct list_head head;
	uint64_t addr64;
	uint32_t size;
	uint32_t addr32;
};

struct allocated_area {
	struct list_head head;
	struct list_head list_map_area64;
	uint64_t addr64_2mb_align;
	uint32_t need_2m_blk;
	uint32_t start_id;
};
#endif
// structure ----------------------------------------------------


// SMMU control info
struct smmu_ctrl {
    void		*hndl;		// SMMU controller addrss
    void		*sram_hndl;	// SRAM address for table RAM
    void		*sram_hndl_pa;	// SRAM address for table RAM
    void	*GR0;		// top of GR0 address space
    void	*GR1;		// top of GR1 address space
#ifdef SMMU_DEBUG_SECURE
    void	*SSD;		// top of SSD address space
#endif
    void	*CB;		// top of CB address space

	struct mutex mlock;

	uint64_t	*smmu_l1_table;		// L1 table(va) (4 entry)
	uint64_t	*smmu_l1_table_pa;		// L1 table(pa)
	
#ifdef CONFIG_USB2_SMMU_LIB
	struct list_head allocated_list;
	uint64_t usingblock[SMMU_32RANGE_ALL_PAGECNT / sizeof(uint64_t)];
	spinlock_t update_lock;
#endif
};


// variable -----------------------------------------------------
// SMMU control information

#ifdef SMMU_SECURE_DEBUG
static volatile bool bSecure = true;
#endif


static void dwc2_smmu_hw_init(struct smmu_ctrl *smmu_ctrl_info, uint64_t ttbr0)
{
    uint32_t reg32;
    uint32_t page_size, page_num;
    void *hndl = smmu_ctrl_info->hndl;
    void *GR0 = hndl;
    void *GR1;
#ifdef SMMU_DEBUG_SECURE
    void *SSD;
#endif
    void *CB;

    smmu_ctrl_info->GR0 = GR0;

    reg32 = readl_relaxed(GR0 + GR0_IDR1);

    if ((reg32 & GR0_IDR1_PAGESIZE_MASK) == GR0_IDR1_PAGESIZE_MASK)
        page_size = SMMU_PAGE_64K;
    else
        page_size = SMMU_PAGE_4K;

    page_num = 1 << (((reg32 & GR0_IDR1_NUMPAGENDXB_MASK) >> GR0_IDR1_NUMPAGENDXB_SHIFT) + 1);
    GR1	= hndl + page_size * SMMU_PAGE_OFFSET_GR1;	// +01000h
    smmu_ctrl_info->GR1 = GR1;
#ifdef SMMU_DEBUG_SECURE
    SSD	= hndl + page_size * SMMU_PAGE_OFFSET_SSD;	// +04000h
    smmu_ctrl_info->SSD = SSD;
#endif
    CB	= hndl + page_size * page_num;
    smmu_ctrl_info->CB = CB;
    
    s_print(S_DEBUG_INFO, "SMMU: [IDR1] page_size=%d, page_num=%d\n", page_size, page_num);

	mutex_lock(&smmu_ctrl_info->mlock);

    // (GR0)SMMU_sACR : Auxiliary Configuration Reg.
    // [31..27]	reserved
    // [26]		CACHE_LOCK		1	ACTLR in non-secure contexts is R/W or RO, 
    // [25]		DP4K_TBUDISB	0	4KB page size dependency check disable, enable
    // [24]		DP4K_TCUDISB	0	4KB page size dependency check disable, enable
    // [23..11]	reserved
    // [10]		S2CRB_TLBEN		1	Stream to Context Register Bypass TLB Enabled
    // [9]		MMUDISB_TLBEN	1	MMU Disabled Bypass TLB Enabled
    // [8]		SMTNMB_TLBEN	1	Stream Match Table No Match TLB Enabled, enable
    // [7..5]	reserved
    // [4]		IPA2PA_CEN		1	Walk Cache 2 Enabled, enable
    // [3]		S2WC2EN			1	Walk Cache 1 Enabled, enable
    // [2]		S1WC2EN			1	Pre-fetch buffer Enabled, enable
    // [1..0]	reserved
    writel_relaxed(0x0400071C, GR0 + GR0_SACR);
#ifdef SMMU_DEBUG_SECURE
    if (bSecure) {
        writel_relaxed(0x0400071C, GR0 + GR0_NSACR);
    }
#endif

    // (GR0)SMMU_SMR : Stream Match registers
    // SMMU_sCR0.EXIDENABLE=0 is expected
    //  [31]	VALID	1	valid
    //  [31:16]	EXMASK	0xFFFE
    //	[15]
    //	[15:0]	EXID	0 or 1 (or only 0)
    writel_relaxed(0xFFFE0000, GR0 + GR0_SMR0);
    writel_relaxed(0xFFFE0001, GR0 + GR0_SMR1);

    // (GR0)SMMU_S2CR : Stream to Context registers
    // [31:30]
    // [29:28]	TRANSIENTCFG (Not Implement? )
    // [27:26]	INSTCFG	0b00	Instruction Fetch Attribute Configuration, default
    // [25:24]	PRIVCFG	0		Privileged Attribute Configuration, default
    // [23:22]	WACFG   0b00	Write Allocation Configuration, default
    // [21:20]	RACFG   0b00	Read Allocation Configuration, default
    // [19:18]	NSCFG	0b00	Non-secure Configuration, default
    // [17:16]	TYPE	0b00	initial context type, translation context bank index
    // [15:12]	MemAttr	0b1111	Memory Attributes, inner/outer write-back cacheable
    // [11]		MTCFG	0		Memory Type Configuration, default
    // [10]
    // [9:8]	SHCFG	0b00	Shared Configuration, default
    // [7:0]	CBNDX	0		Context bank index, use 0th
    writel_relaxed(0x0800F000, GR0 + GR0_S2CR0);    // INSTCFG=0b10
    writel_relaxed(0x0800F000, GR0 + GR0_S2CR1);    // INSTCFG=0b10

    // (GR1) SMMU_CBARn  : Context Bank Attribute Reg.
    // Type=0b01 (stage1 with stage2 bypass)
    // [31..24]    IRPTNDX     0        Interrupt Index, 0
    // [23..22]    WACFG       0        Write-Allocation Configuration, default
    // [21..20]    RACFG       0        Read-Allocation Configuration, default
    // [19..18]    Barrier     0        Shareability Upgrade, No effect
    // [17..16]    TYPE        0b00     stage2
    // [15..12]    MemAttr     0b0000   Memory Attribute
    // [11]        FB          0        Force Broadcast, not force
    // [10]        HYPC/E2HC   0
    // [9..8]      BPSHCFG     0b00     bypass shared configuration
    // [7..0]      VMID        0        Virtual Machine ID, 0
    writel_relaxed(0x00000000, GR1 + GR1_CBAR0);
    //writel_relaxed(0x00000000, GR1 + GR1_CBAR1);

    // (GR1) SMMU_CBA2Rn : Context Bank Attribute Reg.
    // [31..16]		VMID16
    // [15..2]		reserved
    // [1]			MONC		0	Non-monitor
    // [0]			VA64		1	Virtual address is 64bit
    writel_relaxed(0x00000000, GR1 + GR1_CBA2R0);
    writel_relaxed(0x00000000, GR1 + GR1_CBA2R1);

    //==============================================================================================
    // (CB) Translation Table Base Register
    //----------------------------------------------------------------------------------------------
    // SMMU_CB0_TTBR0	RW		(+0x020,0x024)
    //==============================================================================================
    writel_relaxed((uint32_t)ttbr0, CB + CB_TTBR0);			// lower 32bit
    writel_relaxed((uint32_t)(ttbr0 >> 32), CB + CB_TTBR0 + 4);	// upper 32bit

    // (CB) SMMU_CB0_TCR : Translation Control Reg.
    // [31..30]    TG1         0b10    page granule size for TTBR1, 4KB granule
    // [29..28]    SH1         0b00    shareable for TTBR1
    // [27..26]    ORGN1       0        outer cacheability
    // [25..24]    IRGN1       0        inner cacheability
    // [23]        EPD1        0        translation walk disable, enable
    // [22]        A1          0        ASID select, TTBR0
    // [21..19]    reserved
    // [28..16]    T1SZ        0b000   size offset for TTBR1, maximum(not used)
    // [15
    // [14]        NSCFG0      0       page granule size for TTBR0, 4KB granule
    // [13..12]    SH0         0b11    shareable for TTBR0, outer sharerable
    // [11..10]    ORGN0       3        outer cacheability
    // [9..8]      IRGN0       3        inner cacheability
    // [7]         EPD0        0        translation walk disable, enable
    // [6..3]      reserved
    // [2...0]     T0SZ        0b000   size offset for TTBR0, maximu
    // stage 2 --
    // [31]        EAE         1
    //             reserved
    // [13..12]    SH0         0b11    shareable for TTBR0, outer sharerable
    // [11..10]    ORGN0       3       outer cacheability
    // [9..8]      IRGN0       3       inner cacheability
    // [7..6]      SL0         1       level 1
    //             reserved
    // [3..0]      T0SZ        8       Size of TTBR0
    writel_relaxed(0x80003F49, CB + CB_TCR);

    // (CB) SMMU_CB0_MAIRs : Memory Attribute Indirection Reg.
    // AttrN Strongly-ordered or device, inner non-cacheable normal memory
    writel_relaxed(0x04040404, CB + CB_MAIR0);

    //==============================================================================================
    // (GR0 )Invalidate entire TLB Register
    //----------------------------------------------------------------------------------------------
    // SMMU_STLBIALL    S/NS	WO	(+0x060)
    // SMMU_TLBIALLNSNH NS		WO	(+0x068)
    //==============================================================================================
    writel_relaxed(0x00000000, GR0 + GR0_STLBIALL);
    writel_relaxed(0x00000000, GR0 + GR0_TLBIALLNSNH);

    // (GR0 )SMMU_sCR0 : Secure Configuration Register 0
    // +0x000 : SMMU_SCR0 	(S/NS)
    // +0x400 : SMMU_NSCR0	(S)
    // [31:30]	RESERVED
    // [29:28]	NSCFG		0	Non-secure configuration, use default NS attribute
    // [27:26]	WACFG		0	write allocation configuration, default
    // [25:24]	RACFG		0	read allocation configuration, default
    // [23:22]	SHCFG		0	shared configuration, default
    // [21]		SMCFCFG		1	stream match conflict fault configuration, raise stream match conflict fault
    // [20]		MTCFG		0	memory type configuration, default
    // [19:16]	MemAttr		0xF	memory attribute, inner/outer write-back cacheable
    // [15:14]	BSU			0	barrier shareability, no effect
    // [13]		FB			0	force broadcast, as presented
    // [12]		PTM			1	Private TLB Maintenance, privately managed
    // [11]
    // [10]		USFCFG		0	unidentified stream fault configuration, pass through
    // [9]		GSE			0	global stall enable, don't enforce
    // [8]		STALLD		0	disable per-context stalling, permit
    // [7:6]	TRANSIENTCFG 0	transiend configuration, default
    // [5]		GCFGFIE		1	Global Configuration Fault Interrupt Enable.
    // [4]		GCFGFRE		1	Global Configuration Fault Report Enable
    // [3]		EXIDENABLE	0	Extended ID Enable, Use the SMMUv1 format
    // [2]		GFIE		1	Global Fault Interrupt Enable
    // [1]		GFRE		1	Global Fault Report Enable
    // [0]		CLIENTPD	0	Client Port Disable, SMMU translation (not bypass)
    writel_relaxed(0x002F1536, GR0 + GR0_SCR);

	mutex_unlock(&smmu_ctrl_info->mlock);
}


static void dwc2_smmu_hw_enable(struct smmu_ctrl *smmu_ctrl_info)
{
	mutex_lock(&smmu_ctrl_info->mlock);

    // (CB) SMMU_CBn_SCTLR : System Control Register
    // [31]
    // [30]		UCI		1		User Cache Maintenace Operatoin Enable, enable
    // [29:28]	NSCFG	0		non-secure configuration, default
    // [27:26]	WACFG	0		Write Allocate Configuration, default
    // [25:24]	RACFG	0		Read Allocate Configuration, default
    // [23:22]	SHCFG	0		shared Configuration, default
    // [21]
    // [20]		MTCFG	0		memory type configuration, default
    // [19:16]	MemAttr	0xF		memory attribute, inner/outer write-back cacheable
    // [15:14]	TRANSIENTCFG 0	transiend allocate configuration, default
    // [13]
    // [12]		ASIDPNE	0		Address space identifier private namespace enable, not use
    // [11]
    // [10]		UWXN	0		unprivileged writable execute never, not enabled
    // [9]		WNX		0		writable execute never, not enabled
    // [8]		HUPCF	0		Hit underprivious context fault, stall/terminate
    // [7]		CFCFG	0		Context Fault Configuration, terminate
    // [6]		CFIE	1		Context Fault Interrupt Enable, enable
    // [5]		CFRE	1		Context Fault Report Enable, enable
    // [4]		E		0		Endianess, little
    // [3]		AFFD	0		Access Flag Fault Disable, enable
    // [2]		AFE		1		Access Flag Enable, AP[0] in translation table is access flag
    // [1]		TRE		1		TEX Remap Enable, enable
    // [0]		M		1		MME Enable, enabled
    writel_relaxed(0x400F0067, smmu_ctrl_info->CB + GR0_SCR);

#ifdef SMMU_DEBUG_SECURE
    //==============================================================================================
    // Security State determination address space Register
    //----------------------------------------------------------------------------------------------
    // SMMU_IDR1.SSDTP bit indicates existence
    if (bSecure)
        writel_relaxed(0xFFFFFFFF, smmu_ctrl_info->SSD + SSD_SSDR0);
#endif

	mutex_unlock(&smmu_ctrl_info->mlock);
}
static void dwc2_smmu_hw_disable(struct smmu_ctrl *smmu_ctrl_info)
{
	mutex_lock(&smmu_ctrl_info->mlock);

    // (CB) SMMU_CBn_SCTLR : System Control Register
    // [31]
    // [30]		UCI		1		User Cache Maintenace Operatoin Enable, enable
    // [29:28]	NSCFG	0		non-secure configuration, default
    // [27:26]	WACFG	0		Write Allocate Configuration, default
    // [25:24]	RACFG	0		Read Allocate Configuration, default
    // [23:22]	SHCFG	0		shared Configuration, default
    // [21]
    // [20]		MTCFG	0		memory type configuration, default
    // [19:16]	MemAttr	0xF		memory attribute, inner/outer write-back cacheable
    // [15:14]	TRANSIENTCFG 0	transiend allocate configuration, default
    // [13]
    // [12]		ASIDPNE	0		Address space identifier private namespace enable, not use
    // [11]
    // [10]		UWXN	0		unprivileged writable execute never, not enabled
    // [9]		WNX		0		writable execute never, not enabled
    // [8]		HUPCF	0		Hit underprivious context fault, stall/terminate
    // [7]		CFCFG	0		Context Fault Configuration, terminate
    // [6]		CFIE	1		Context Fault Interrupt Enable, enable
    // [5]		CFRE	1		Context Fault Report Enable, enable
    // [4]		E		0		Endianess, little
    // [3]		AFFD	0		Access Flag Fault Disable, enable
    // [2]		AFE		1		Access Flag Enable, AP[0] in translation table is access flag
    // [1]		TRE		1		TEX Remap Enable, enable
    // [0]		M		1		MME Enable, disable
    writel_relaxed(0x400F0066, smmu_ctrl_info->CB + GR0_SCR);

#ifdef SMMU_DEBUG_SECURE
    //==============================================================================================
    // Security State determination address space Register
    //----------------------------------------------------------------------------------------------
    // SMMU_IDR1.SSDTP bit indicates existence
    if (bSecure)
        writel_relaxed(0xFFFFFFFF, smmu_ctrl_info->SSD + SSD_SSDR0);
#endif

	mutex_unlock(&smmu_ctrl_info->mlock);
}


static int dwc2_smmu_hw_invalidate_va(struct smmu_ctrl *smmu_ctrl_info, uint32_t va)
{
#ifndef SMMU_SKIP_INVALIDATE
    uint32_t reg32, cnt = 0;

    // Comment out until normal operation can be confirmed

    // for stage2
    reg32 = (va & 0xFFFFF000) >> 12;

//	mutex_lock(&smmu_ctrl_info->mlock);	// lock

    writel_relaxed(reg32, smmu_ctrl_info->CB + CB_TLBIIPAS2);  // lower 32bit
    writel_relaxed(0, smmu_ctrl_info->CB + CB_TLBIIPAS2+4);    // upper 32bit (all 0)

	writel_relaxed(0, smmu_ctrl_info->CB + CB_TLBSYNC);		// Sync start

//	mutex_unlock(&smmu_ctrl_info->mlock);	// unlock

	mb();

	do {
//		mutex_lock(&smmu_ctrl_info->mlock);	// lock
		reg32 = readl_relaxed(smmu_ctrl_info->CB + CB_TLBSTATUS);
//		mutex_unlock(&smmu_ctrl_info->mlock);	// unlock
		if (reg32 == 0) {
			break;
		}
		udelay(5);
		cnt++;
		if (cnt > 1000) {
			s_print(S_DEBUG_INFO, "SMMU: Timeout of release TLBSTATUS\n");
			return -ETIMEDOUT;
		}
	} while(1);
#endif
	return 0;
}


static void dwc2_smmu_make_table_level2(volatile uint64_t* pTTB_LEVEL2, uint32_t nRegionIndex, uint64_t pa)
{
    pTTB_LEVEL2[nRegionIndex] = SMMU_MAKE_RECORD_BLOCK_L2_NS(pa);
}


int32_t dwc2_smmu_init(struct dwc2_hsotg *hsotg, void *smmu_hndl)
{
    uint32_t i;
	struct smmu_ctrl *smmu_ctrl_info;
	volatile uint64_t tbl_val;
	dma_addr_t dma_handle;
	struct cxd_dwc2 *cxd_glue = hsotg->cxd_glue;

	smmu_ctrl_info = (struct smmu_ctrl *)kzalloc(sizeof(struct smmu_ctrl), GFP_ATOMIC | __GFP_ZERO);
	if (!smmu_ctrl_info) {
		s_print(S_DEBUG_ERROR, "%s: kmalloc() failed\n", __func__);
		return -ENOMEM;
	}

	smmu_ctrl_info->hndl 		= smmu_hndl;

	smmu_ctrl_info->sram_hndl = ioremap(cxd_glue->smmu_ram, cxd_glue->smmu_len);
	memset_io(smmu_ctrl_info->sram_hndl, 0, cxd_glue->smmu_len);
	smmu_ctrl_info->sram_hndl_pa = (void*)(cxd_glue->smmu_ram);

	mutex_init(&smmu_ctrl_info->mlock);

    // L1 table
    smmu_ctrl_info->smmu_l1_table = dma_alloc_coherent(
			hsotg->dev, SMMU_L1_ENTRY_CNT * SMMU_ENTRY_SIZE, &dma_handle, GFP_KERNEL);
	if (!smmu_ctrl_info->smmu_l1_table) {
		s_print(S_DEBUG_ERROR, "%s: dma_alloc_coherent() failed\n", __func__);
		kfree(smmu_ctrl_info);
		return -ENOMEM;
	}
	smmu_ctrl_info->smmu_l1_table_pa = (uint64_t*)dma_handle;

	// clear table area
	memset(smmu_ctrl_info->smmu_l1_table, 0, SMMU_L1_ENTRY_CNT * SMMU_ENTRY_SIZE);

	// set L1 table enbry
	for (i = 0;i < SMMU_L1_ENTRY_CNT;i++) {
		smmu_ctrl_info->smmu_l1_table[i] = SMMU_MAKE_RECORD_TABLE_L1( (((uint64_t)smmu_ctrl_info->sram_hndl_pa) + SMMU_PAGE_4K * i) );
	}

	// Not need to flush because L1 table is located on coherent memory.

    mb();

    for (i = 0;i < SMMU_L1_ENTRY_CNT;i++) {
        tbl_val = smmu_ctrl_info->smmu_l1_table[i];
        //s_print("*** L1 table[%u]: 0x%llx\n", i, (uint64_t)smmu_ctrl_info->smmu_l1_table[i]);
    }

	dwc2_smmu_hw_init(smmu_ctrl_info, (uint64_t)smmu_ctrl_info->smmu_l1_table_pa);
	dwc2_smmu_hw_enable(smmu_ctrl_info);

	hsotg->smmu_ctrl = smmu_ctrl_info;

#ifdef CONFIG_USB2_SMMU_LIB
	s_print(S_DEBUG_ERROR, "NEW smmu init start\n");
	INIT_LIST_HEAD(&smmu_ctrl_info->allocated_list);
	spin_lock_init(&smmu_ctrl_info->update_lock);
	memset(smmu_ctrl_info->usingblock, 0, sizeof(smmu_ctrl_info->usingblock));
#endif

    return 0;
}
EXPORT_SYMBOL_GPL(dwc2_smmu_init);

void dwc2_smmu_exit(struct dwc2_hsotg *hsotg)
{
	struct smmu_ctrl *smmu_ctrl_info = hsotg->smmu_ctrl;

	dwc2_smmu_hw_disable(smmu_ctrl_info);
	iounmap(smmu_ctrl_info->sram_hndl);
	dma_free_coherent(hsotg->dev, SMMU_L1_ENTRY_CNT * SMMU_ENTRY_SIZE,
			smmu_ctrl_info->smmu_l1_table, (dma_addr_t)(smmu_ctrl_info->smmu_l1_table_pa));
	kfree(smmu_ctrl_info);

	hsotg->smmu_ctrl = NULL;
    s_print(S_DEBUG_INFO, " %s\n",__func__);
}
EXPORT_SYMBOL_GPL(dwc2_smmu_exit);


void dwc2_smmu_debug_print_TLB_status(struct dwc2_hsotg *hsotg)
{
#define TBU_UNIT_NO    (64)
	struct smmu_ctrl *smmu_ctrl_info;
    uint32_t i, j, cnt;
    volatile uint32_t reg32;

    s_print(S_DEBUG_INFO, "SMMU[DBG]--------\n");

	smmu_ctrl_info = hsotg->smmu_ctrl;

    mutex_lock(&smmu_ctrl_info->mlock);    // lock

    writel_relaxed(0, smmu_ctrl_info->GR0 + GR0_DBGRPTRTBU);    // set read pointer to top

    for (i = 0, cnt = 0; i < TBU_UNIT_NO; i++) {

        // w0
        reg32 = readl_relaxed(smmu_ctrl_info->GR0 + GR0_DBGRDATATBU);
        if (reg32 & 0x1) {
            // word is invalid
            cnt++;
            continue;
        }

        s_print(S_DEBUG_INFO, "[TBU-TLB][%d]-----\n", i);

        // w0
        s_print(S_DEBUG_INFO, "  0x%08x\n", reg32);

        // w1-w6
        for (j = 0; j < 6; j++) {
            reg32 = readl_relaxed(smmu_ctrl_info->GR0 + GR0_DBGRDATATBU);
            s_print(S_DEBUG_INFO, "  0x%08x\n", reg32);
        }
    }

    mutex_unlock(&smmu_ctrl_info->mlock);	// unlock

    if (cnt == TBU_UNIT_NO) {
        s_print(S_DEBUG_INFO, "[TBU-TLB] all entry(%d) is Invalid\n", cnt);
    }

    s_print(S_DEBUG_INFO, "SMMU[DBG]--------\n");
}
EXPORT_SYMBOL_GPL(dwc2_smmu_debug_print_TLB_status);


void dwc2_smmu_debug_print_info(struct dwc2_hsotg *hsotg)
{
	struct smmu_ctrl *smmu_ctrl_info;
    uint32_t reg1, i;
    uint64_t faddr, reg64;
    uint32_t fault = 0;

    s_print(S_DEBUG_INFO, "[Debug Info] (start)------------\n");

	smmu_ctrl_info = hsotg->smmu_ctrl;

    mutex_lock(&smmu_ctrl_info->mlock);    // lock

    // GR0
    reg1 = readl_relaxed(smmu_ctrl_info->GR0 + GR0_GFSR);
    if (reg1 != 0) {
        s_print(S_DEBUG_INFO, "[GR0] GFSR(0x%x) = 0x%x\n", GR0_GFSR, reg1);

        faddr = readl_relaxed(smmu_ctrl_info->GR0 + GR0_GFAR);
        faddr |= readl_relaxed(smmu_ctrl_info->GR0 + GR0_GFAR + 4);
        s_print(S_DEBUG_INFO, "[GR0] GFAR(0x%x) = 0x%llx\n", GR0_GFAR, faddr);
    }

    // CB
    reg1 = readl_relaxed(smmu_ctrl_info->CB + CB_FSR);
    if (reg1 != 0) {
        s_print(S_DEBUG_INFO, "[CB ] FSR(0x%x) = 0x%x\n", CB_FSR, reg1);

        faddr = readl_relaxed(smmu_ctrl_info->CB + CB_FAR);
        faddr |= ((uint64_t)readl_relaxed(smmu_ctrl_info->CB + CB_FAR + 4) << 32);
        s_print(S_DEBUG_INFO, "[CB ] FAR(0x%x) = 0x%llx\n", CB_FAR, faddr);
    }
    if (reg1 & BIT_CB_FSR_TF) {
        fault = 1;
    }

    reg1 = readl_relaxed(smmu_ctrl_info->CB + CB_FSYNR0);
    if (reg1 != 0) {
        s_print(S_DEBUG_INFO, "[CB ] FSYNR0(0x%x) = 0x%x\n", CB_FSYNR0, reg1);
    }

    s_print(S_DEBUG_INFO, "[Debug Info] (table setting)----\n");
    reg64 = readl_relaxed(smmu_ctrl_info->CB + CB_TTBR0);
    reg64 |= ((uint64_t)readl_relaxed(smmu_ctrl_info->CB + CB_TTBR0+4)) << 32;
    s_print(S_DEBUG_INFO, "[Debug Info] [L1]----\n");
    s_print(S_DEBUG_INFO, "[CB ] TTBR0(0x%x) = 0x%llx (L1 table address)\n", CB_TTBR0, reg64);
    for (i = 0; i < 4; i++) {
        s_print(S_DEBUG_INFO, "[CB ] L1[%u] table = 0x%llx\n", i, smmu_ctrl_info->smmu_l1_table[i]);
    }
    s_print(S_DEBUG_INFO, "[Debug Info] [L2]----\n");
    s_print(S_DEBUG_INFO, "[CB ] smmu_table_ram_top(pa) = 0x%llx\n", (uint64_t)smmu_ctrl_info->sram_hndl_pa);
    for (i = 0; i < 4; i++) {
        reg64 = readl_relaxed(smmu_ctrl_info->sram_hndl + 8 * i);
        reg64 |= ((uint64_t)readl_relaxed(smmu_ctrl_info->sram_hndl + (8 * i) + 4)) << 32;
        s_print(S_DEBUG_INFO, "[CB ] L2[%u] table = 0x%llx\n", i, reg64);
    }
    for (i = 16; i < 20; i++) {
        reg64 = readl_relaxed(smmu_ctrl_info->sram_hndl + 8 * i);
        reg64 |= ((uint64_t)readl_relaxed(smmu_ctrl_info->sram_hndl + (8 * i) + 4)) << 32;
        s_print(S_DEBUG_INFO, "[CB ] L2[%u] table = 0x%llx\n", i, reg64);
    }

    mutex_unlock(&smmu_ctrl_info->mlock);	// unlock

    s_print(S_DEBUG_INFO, "[Debug Info] (stop)-------------\n");

    if (fault) {
        s_print(S_DEBUG_INFO, "[Debug Info] fault ----\n");
        dwc2_smmu_debug_print_TLB_status(hsotg);
    }
}
EXPORT_SYMBOL_GPL(dwc2_smmu_debug_print_info);
#endif

#ifdef CONFIG_USB2_SMMU_LIB
/**
 * set_block() - set allocated area of virtual space
 * @priv: device private data
 * @id: index of virtual space
 *
 * Return: no
 */
static __inline void set_block(struct smmu_ctrl *priv, uint32_t id)
{
	uint64_t *usingblock = &priv->usingblock[0];
	uint64_t bit = D_G_L2BIT(id);
	uint32_t i = D_G_L2ID(id);

	usingblock[i] |= bit;
	//s_print(S_DEBUG_INFO, "test id=%4d,i=%d 0x%016llx\n",id, i, usingblock[i]);
	return;
}
/**
 * clear_block() - clear allocated area of virtual space
 * @priv: device private data
 * @id: index of virtual space
 *
 * Return: no
 */
static __inline void clear_block(struct smmu_ctrl *priv, uint32_t id)
{
	uint64_t *usingblock = &priv->usingblock[0];
	uint64_t bit = D_G_L2BIT(id);
	uint32_t i = D_G_L2ID(id);

	usingblock[i] &= ~bit;
	//s_print(S_DEBUG_INFO, "test id=%4d,i=%d 0x%016llx (bit 0x%llx)\n",id, i, usingblock[i], ~bit);
	return;
}
/**
 * is_is_allocated() - Check whether the id is in allocated
 * @priv: device private data
 * @id: index of virtual space
 *
 * Return: true is allocated. false is free
 */
static __inline bool id_is_allocated(struct smmu_ctrl *priv, uint32_t id)
{
	uint64_t *usingblock = &priv->usingblock[0];
	uint64_t mask;
	uint32_t i;

	i = D_G_L2ID(id);
	mask = D_G_L2BIT(id);
	//s_print(S_DEBUG_INFO, "seek: id=%4d,i=%d 0x%016llx\n",id, i, mask);
	if (usingblock[i] & mask)
		return true;
	else
		return false;
}
/**
 * check_free_allocinfo() - Check if the requested block is allocatable.
 * @priv: device private data
 * @needblock: The number of requested block
 *
 * Return: first fit index of free area
 */
static __inline int32_t check_free_allocinfo(struct smmu_ctrl *priv, uint32_t needblock)
{
	uint32_t count;
	uint32_t id;
	int32_t first;

	first = -ENOMEM;
	count = 0;
	for (id = 0; id < SMMU_32RANGE_ALL_PAGECNT; id++) {
		if (!id_is_allocated(priv, id)) {
			if (!count)
				first = id;
			count++;
		} else
			count = 0;
		if (count == needblock)
			return first;
	}
	return -ENOMEM;
}

/**
 * add_map_area64_info() - Add map_area64 to list
 * @priv: device private data
 * @addr64: The requested 64bit address
 * @size: The requested size for 64bit address
 * @own: virtual space object to add new map_area64
 * @map_area: New map_area64 object
 *
 * Return: no
 */
static void add_map_area64_info(struct smmu_ctrl *priv, uint64_t addr64, uint32_t size, struct allocated_area *own, struct map_area64 *map_area)
{
	map_area->addr64 = addr64;
	map_area->size = size;
	list_add(&map_area->head, &own->list_map_area64);

	return;
}
#ifdef DEBUG
/**
 * check_same_address() - for debug
 * @priv: device private data
 * @addr64: The requested 64bit address
 *
 * find the allocated address
 * As a premise, linux DMA API does not allocate same address.
 *
 * Return: the address find or not
 */
static bool check_same_address(struct smmu_ctrl *priv, uint64_t addr64)
{
	struct allocated_area *allocinfo;
	struct map_area64 *area64info;
	struct list_head *aobj, *alist;
	struct list_head *obj64, *list64;
	bool find;

	find = false;
	list_for_each_safe(aobj, alist, &priv->allocated_list) {
		allocinfo = container_of(aobj, struct allocated_area, head);
		list_for_each_safe(obj64, list64, &allocinfo->list_map_area64) {
			area64info = container_of(obj64, struct map_area64, head);
			if (area64info->addr64 == addr64) {
				find = true;
				dev_warn_once(NULL, "same address detected\n");
				return find;
			}
		}
	}
	return find;
}
#endif/* DEBUG */

static uint32_t get_smmu_virt_addr(uint32_t i, uint64_t addr64)
{
	uint32_t addr32 = (i << SMMU_BYTES_PER_ENTRY_SHIFT) + (addr64 & ~SMMU_ENTRY_ADDR_MASK);
	//s_print(S_DEBUG_INFO, "%s:%d L2[%d], 64: 0x%llx, 32: 0x%x \n", __func__, __LINE__,i, addr64, addr32);
	return addr32;
}
/**
 * dwc2_smmu_is_range_of_alloc() - SMMU API
 * @addr: The requested 64bit address
 *
 *
 * Return: true is 64bit address, false is 32bit address.
 */
bool dwc2_smmu_is_range_of_alloc(dma_addr_t addr)
{

	if ((SMMU_64RANGE_BASE <= addr) && (addr < (SMMU_64RANGE_BASE + SMMU_64RANGE_SIZE)))
		return true;
	else if ((addr >> 32) == 0)
		return false;
	else
		dev_warn_once(NULL, "out of range address for smmu alloc\n");

	return false;
}
EXPORT_SYMBOL_GPL(dwc2_smmu_is_range_of_alloc);
/**
 * dwc2_smmu_alloc() - SMMU API
 * @hsotg: device private data
 * @addr64: The requested 64bit address
 * @size: The size for requested 64bit address
 * @p_addr32: set the virtual spcase address for 32bit hardware
 *
 * Allocate the requested address and return 32bit virtual address for hardware.
 * Not allocate, if the requested address is already in the allocated virtual space.
 *
 * Return: success or not
 */
int dwc2_smmu_alloc(struct dwc2_hsotg *hsotg, uint64_t addr64, uint32_t size, uint32_t *p_addr32)
{
	struct smmu_ctrl *priv = hsotg->smmu_ctrl;
	struct allocated_area *allocinfo;
	struct map_area64 *map_area;
	int ret;
	uint32_t i;
	int32_t start_id;
	bool find;
	struct allocated_area *own;
	uint32_t need_blk;
	uint64_t sb, eb;
	unsigned long flags;

#ifdef DEBUG
	check_same_address(priv, addr64);
#endif

	if (addr64 == 0 && size == 0)
		return -EINVAL;

	if (!priv) {
		s_print(S_DEBUG_INFO, "alloc: private data is NULL\n");
		goto e_nomem;
	}

	own = kmalloc(sizeof(struct allocated_area), GFP_ATOMIC);
	if (own == NULL)
		goto e_nomem;

	map_area = kmalloc(sizeof(struct map_area64), GFP_ATOMIC);
	if (map_area == NULL)
		goto e_nomem_map_area64;


	spin_lock_irqsave(&priv->update_lock, flags);
#ifdef TESTCODE_USB_DWC2_SMMU_LOCK
	l_start_smmu_alloc("start");
#endif
	list_for_each_entry(allocinfo, &priv->allocated_list, head) {
		if (allocinfo->addr64_2mb_align <= addr64 &&
			(addr64 + size) <= (allocinfo->addr64_2mb_align + allocinfo->need_2m_blk * SMMU_BYTES_PER_ENTRY)) {
			/* already mapped64 */
			uint32_t offset_blk;
			add_map_area64_info(priv, addr64, size, allocinfo, map_area);
			find = true;
			offset_blk = (addr64 - allocinfo->addr64_2mb_align) / SMMU_BYTES_PER_ENTRY;
			*p_addr32 = get_smmu_virt_addr(allocinfo->start_id + offset_blk, addr64);
			map_area->addr32 = *p_addr32;
			break;
		} else {
			find = false;
		}
	}
	if (!find) {
		/* add new virtual space */
		/* calc needed block for virtual space */
		sb = (addr64 & D_4GB_MSK) >> SMMU_BYTES_PER_ENTRY_SHIFT;
		eb = ((addr64 + size - 1) & D_4GB_MSK) >> SMMU_BYTES_PER_ENTRY_SHIFT;
		need_blk = eb - sb + 1;

		start_id = check_free_allocinfo(priv, need_blk);
		if (start_id < 0)
			goto e_nomem_lock;

		/* set parameter of virtual space */
		own->addr64_2mb_align = addr64 & D_2MB_MSK;
		own->need_2m_blk = need_blk;
		own->start_id = start_id;
		INIT_LIST_HEAD(&own->list_map_area64);

		//s_print(S_DEBUG_INFO, "%s:%d %d %d\n", __func__, __LINE__,own->start_id,  own->need_2m_blk);
		for (i = own->start_id; i < own->start_id + own->need_2m_blk; i++) {
			set_block(priv, i);
			dwc2_smmu_make_table_level2(priv->sram_hndl, i, addr64 + ((i - own->start_id) * SMMU_BYTES_PER_ENTRY));
			ret = dwc2_smmu_hw_invalidate_va(priv, i * SMMU_BYTES_PER_ENTRY);
			if (ret != 0)
				goto e_nomem_lock;
		}
		add_map_area64_info(priv, addr64, size, own, map_area);/* add 64bit space info to list */
		list_add(&own->head, &priv->allocated_list);

		*p_addr32 = get_smmu_virt_addr(own->start_id, addr64);
		map_area->addr32 = *p_addr32;
	}
#ifdef TESTCODE_USB_DWC2_SMMU_LOCK
	l_end_smmu_alloc("end");
#endif
	spin_unlock_irqrestore(&priv->update_lock, flags);

	if(find == true) {
		kfree(own);
	}

	return 0;

e_nomem_lock:
	spin_unlock_irqrestore(&priv->update_lock, flags);
	kfree(map_area);
e_nomem_map_area64:
	kfree(own);
e_nomem:
	s_print(S_DEBUG_INFO, "%s:%d error\n", __func__, __LINE__);
	return -ENOMEM;
}
EXPORT_SYMBOL_GPL(dwc2_smmu_alloc);
/**
 * dwc2_get_free_addr() - SMMU API
 * @hsotg: device private data
 * call before dwc2_smmu_free is called
 * get 64 bit address from addr (64 or 32 bit address)
 */
int64_t dwc2_get_free_addr(struct dwc2_hsotg *hsotg, uint64_t addr)
{
	struct smmu_ctrl *priv = hsotg->smmu_ctrl;
	struct allocated_area *allocinfo;
	struct map_area64 *area64info;
	struct list_head *aobj, *alist;
	struct list_head *obj64, *list64;
	unsigned long flags;
	int64_t ret_addr = -ENOMEM;
	uint64_t cmpval;

	spin_lock_irqsave(&priv->update_lock, flags);
#ifdef TESTCODE_USB_DWC2_SMMU_LOCK
	l_start_smmu_free_a("start");
#endif
	list_for_each_safe(aobj, alist, &priv->allocated_list) {
		allocinfo = container_of(aobj, struct allocated_area, head);
		list_for_each_safe(obj64, list64, &allocinfo->list_map_area64) {
			area64info = container_of(obj64, struct map_area64, head);
			if (addr < SMMU_64RANGE_BASE)
				cmpval = (uint64_t)area64info->addr32;
			else
				cmpval = area64info->addr64;
			if (cmpval == addr) {
				ret_addr = area64info->addr64;
				goto done;
			}
		}
	}
	s_print(S_DEBUG_INFO, "%s:%d cannot find address 0x%llx\n", __func__, __LINE__, addr);
done:
#ifdef TESTCODE_USB_DWC2_SMMU_LOCK
	l_end_smmu_free_a("end");
#endif
	spin_unlock_irqrestore(&priv->update_lock, flags);
	return ret_addr;
}
EXPORT_SYMBOL_GPL(dwc2_get_free_addr);
/**
 * dwc2_smmu_free() - SMMU API
 * @hsotg: device private data
 * @addr64: The requested 64bit address
 *
 * free the requested address
 *
 * Return: no
 */
void dwc2_smmu_free(struct dwc2_hsotg *hsotg, uint64_t addr)
{
	struct smmu_ctrl *priv = hsotg->smmu_ctrl;
	struct allocated_area *allocinfo;
	struct map_area64 *area64info;
	struct list_head *aobj, *alist;
	struct list_head *obj64, *list64;
	uint32_t i;
	bool find;
	unsigned long flags;
	uint64_t cmpval;

	if (!priv) {
		s_print(S_DEBUG_INFO, "free: private data is NULL\n");
		return;
	}
	find = false;
	spin_lock_irqsave(&priv->update_lock, flags);
#ifdef TESTCODE_USB_DWC2_SMMU_LOCK
	l_start_smmu_free("start");
#endif
	list_for_each_safe(aobj, alist, &priv->allocated_list) {
		allocinfo = container_of(aobj, struct allocated_area, head);
		list_for_each_safe(obj64, list64, &allocinfo->list_map_area64) {
			area64info = container_of(obj64, struct map_area64, head);
			if (addr >= SMMU_64RANGE_BASE)
				cmpval = area64info->addr64;
			else
				cmpval = (uint64_t)area64info->addr32;
			if (cmpval == addr) {
				find = true;
				list_del(&area64info->head);
				kfree(area64info);
				if (find) {
					if (list_empty(&allocinfo->list_map_area64)) {
						for (i = allocinfo->start_id; i < allocinfo->start_id + allocinfo->need_2m_blk; i++)
							clear_block(priv, i);
						list_del(&allocinfo->head);
						kfree(allocinfo);
					}
				}
				goto done;
			}
		}
	}
done:
#ifdef TESTCODE_USB_DWC2_SMMU_LOCK
	l_end_smmu_free("end");
#endif
	spin_unlock_irqrestore(&priv->update_lock, flags);
}
EXPORT_SYMBOL_GPL(dwc2_smmu_free);

#endif
