// SPDX-License-Identifier: GPL-2.0
/*
 * drd.c - DesignWare USB2 DRD Controller Dual-role support
 *
 * Copyright (C) 2020 STMicroelectronics
 * Copyright 2022 Sony Corporation, SOCIONEXT INC.
 *
 * Author(s): Amelie Delaunay <amelie.delaunay@st.com>
 */

#include <linux/clk.h>
#include <linux/iopoll.h>
#include <linux/platform_device.h>
#include <linux/usb/role.h>
#include <linux/extcon.h>
#include "core.h"
#include "hcd.h"
#if defined(CONFIG_USB2_SMMU_LIB) && defined(CONFIG_ARCH_CXD90XXX)
#include "dwc2-cxd.h"
#include "../cxd/usb2_smmu.h"
#include "../cxd/sdebug.h"
#endif

#define D_UDEV_STRINITVAL 0
#define DWC2PROBEWAIT	20

#ifdef TESTCODE_USB_DWC2_TIME_DRD
	static struct timespec64 testcode_tm_start;
#endif
static void __dwc2_set_mode(struct work_struct *work)
{
	struct dwc2_hsotg *dwc = (container_of((work), struct dwc2_hsotg, drd_work));
	unsigned long flags;
	int ret;
	char uchger_state[50] = { D_UDEV_STRINITVAL };
	char *envp[] = { uchger_state, NULL };
	struct cxd_dwc2 *cxd_glue = dwc->cxd_glue;
	int retry;
#ifdef TESTCODE_USB_DWC2_TIME_DRD
	struct timespec64 end;
	long diff_ns;
	long diff_s;

	ktime_get_real_ts64(&testcode_tm_start);
#endif
	mutex_lock(&dwc->drd_mutex);

	if (!dwc->desired_dr_role)
		goto out;

	if (dwc->desired_dr_role == dwc->current_dr_role)
		goto out;

	if (dwc->desired_dr_role == DWC2_GCTL_PRTCAP_OTG && dwc->edev)
		goto out;

	ret = 0;
	switch (dwc->current_dr_role) {
	case DWC2_GCTL_PRTCAP_HOST:
		dwc2_hcd_remove(dwc);
		break;
	case DWC2_GCTL_PRTCAP_DEVICE_DISCON:
		break;
	case DWC2_GCTL_PRTCAP_DEVICE:
		dwc2_hsotg_remove(dwc);
		dwc->gadget_enabled = 0;
		break;
	case DWC2_GCTL_PRTCAP_HOST_DISCON:
		break;
	default:
		if (dwc->desired_dr_role == DWC2_GCTL_PRTCAP_HOST) {
			/* Wait for probed dwc2.
			 *  EHCI and OHCI uses dwc2_hsotg for smmu but start probing before dwc2 private data is set.
			 */
			retry = DWC2PROBEWAIT;
			while(!platform_get_drvdata(to_platform_device(dwc->dev)) && retry > 0) {
				mdelay(1);
				retry--;
			}
		}
		dwc2_cxd_dev_prepare_stop(cxd_glue);
		break;
	}

	if (ret)
		s_print(S_DEBUG_ERROR, "failed to prepare of starting\n");

	spin_lock_irqsave(&dwc->lock, flags);

	dwc->current_dr_role = dwc->desired_dr_role;

	spin_unlock_irqrestore(&dwc->lock, flags);

	switch (dwc->desired_dr_role) {
	case DWC2_GCTL_PRTCAP_HOST:
		dwc2_cxd_host_prepare_start(cxd_glue);
#if defined(CONFIG_USB2_SMMU_LIB) && defined(CONFIG_ARCH_CXD90XXX)
		s_print(S_DEBUG_ERROR, "HOST smmu start\n");
		dwc2_smmu_init(dwc, cxd_glue->smmu_base);
#endif
		ret = dwc2_hcd_init(dwc);
		if (ret)
			s_print(S_DEBUG_ERROR, "failed to initialize host\n");
		/* udev */
		snprintf(uchger_state, ARRAY_SIZE(uchger_state), "USB2_DRD=%s", "USB2_DRDHOST");
		kobject_uevent_env(&dwc->dev->kobj, KOBJ_CHANGE, envp);
		break;
	case DWC2_GCTL_PRTCAP_DEVICE_DISCON:
#if defined(CONFIG_USB2_SMMU_LIB) && defined(CONFIG_ARCH_CXD90XXX)
		s_print(S_DEBUG_ERROR, "DEVICE smmu exit\n");
		dwc2_smmu_exit(dwc);
#endif
		dwc2_cxd_dev_prepare_stop(cxd_glue);
		/* udev */
		snprintf(uchger_state, ARRAY_SIZE(uchger_state), "USB2_DRD=%s", "USB2_DEVDISCON");
		kobject_uevent_env(&dwc->dev->kobj, KOBJ_CHANGE, envp);
		break;
	case DWC2_GCTL_PRTCAP_DEVICE:
		dwc2_cxd_dev_prepare_start(cxd_glue);
#if defined(CONFIG_USB2_SMMU_LIB) && defined(CONFIG_ARCH_CXD90XXX)
		s_print(S_DEBUG_ERROR, "DEVICE smmu start\n");
		dwc2_smmu_init(dwc, cxd_glue->smmu_base);
#endif
		dwc2_gadget_init(dwc);
		dwc->gadget_enabled = 1;
		/* udev */
		snprintf(uchger_state, ARRAY_SIZE(uchger_state), "USB2_DRD=%s", "USB2_DRDDEV");
		kobject_uevent_env(&dwc->dev->kobj, KOBJ_CHANGE, envp);
		break;
	case DWC2_GCTL_PRTCAP_HOST_DISCON:
#if defined(CONFIG_USB2_SMMU_LIB) && defined(CONFIG_ARCH_CXD90XXX)
		s_print(S_DEBUG_ERROR, "HOST smmu exit\n");
		dwc2_smmu_exit(dwc);
#endif
		dwc2_cxd_host_prepare_stop(cxd_glue);
		/* udev */
		snprintf(uchger_state, ARRAY_SIZE(uchger_state), "USB2_DRD=%s", "USB2_HOSTDISCON");
		kobject_uevent_env(&dwc->dev->kobj, KOBJ_CHANGE, envp);
		break;
	default:
		break;
	}

out:
	dwc2_cxd_set_drd_done();
	mutex_unlock(&dwc->drd_mutex);
#ifdef TESTCODE_USB_DWC2_TIME_DRD
		ktime_get_real_ts64(&end);
		diff_ns = end.tv_nsec - testcode_tm_start.tv_nsec;
		diff_s = end.tv_sec - testcode_tm_start.tv_sec;
		if (diff_ns < 0) {
			diff_ns = 1000000000 + end.tv_nsec - testcode_tm_start.tv_nsec;
			diff_s -= 1;
		}
		s_print(S_DEBUG_INFO, "%s: drd time %ld.%09ld [s]\n", __func__, diff_s, diff_ns);
#endif
}


static void dwc2_set_mode(struct dwc2_hsotg *dwc, u32 mode)
{
	unsigned long flags;

	if (dwc->dr_mode != USB_DR_MODE_OTG)
		return;

	spin_lock_irqsave(&dwc->lock, flags);
	dwc->desired_dr_role = mode;
	spin_unlock_irqrestore(&dwc->lock, flags);

	queue_work(system_freezable_wq, &dwc->drd_work);
}

static void dwc2_drd_update(struct dwc2_hsotg *dwc)
{
	int id;
	int vbus;
	struct cxd_dwc2 *cxd_glue = dwc->cxd_glue;

	if (dwc->edev) {
		id = extcon_get_state(dwc->edev, EXTCON_USB_HOST);
		if (id < 0)
			id = 0;

		if (id == 0) {
			// device
			vbus = extcon_get_state(dwc->edev, EXTCON_USB);
			if (vbus < 0)
				vbus = 0;

			if (vbus)
				dwc2_set_mode(dwc, DWC2_GCTL_PRTCAP_DEVICE);
			else
				dwc2_cxd_dev_prepare_stop(cxd_glue);
		} else
			dwc2_set_mode(dwc, DWC2_GCTL_PRTCAP_HOST);
	}
}

static int dwc2_drd_notifier(struct notifier_block *nb,
			     unsigned long event, void *ptr)
{
	struct dwc2_hsotg *dwc = container_of(nb, struct dwc2_hsotg, edev_nb);

	dwc2_set_mode(dwc, event ? DWC2_GCTL_PRTCAP_HOST : DWC2_GCTL_PRTCAP_HOST_DISCON);

	return NOTIFY_DONE;
}

static int dwc2_drd_notifier_vbus(struct notifier_block *nb,
			     unsigned long event, void *ptr)
{
	struct dwc2_hsotg *dwc = container_of(nb, struct dwc2_hsotg, edev_nb_vbus);

	dwc2_set_mode(dwc, event ? DWC2_GCTL_PRTCAP_DEVICE : DWC2_GCTL_PRTCAP_DEVICE_DISCON);

	return NOTIFY_DONE;
}

#ifndef CONFIG_ARCH_CXD90XXX
static void dwc2_ovr_init(struct dwc2_hsotg *hsotg)
{
	unsigned long flags;
	u32 gotgctl;

	spin_lock_irqsave(&hsotg->lock, flags);

	gotgctl = dwc2_readl(hsotg, GOTGCTL);
	gotgctl |= GOTGCTL_BVALOEN | GOTGCTL_AVALOEN | GOTGCTL_VBVALOEN;
	gotgctl |= GOTGCTL_DBNCE_FLTR_BYPASS;
	gotgctl &= ~(GOTGCTL_BVALOVAL | GOTGCTL_AVALOVAL | GOTGCTL_VBVALOVAL);
	dwc2_writel(hsotg, gotgctl, GOTGCTL);

	spin_unlock_irqrestore(&hsotg->lock, flags);

	dwc2_force_mode(hsotg, (hsotg->dr_mode == USB_DR_MODE_HOST));
}

static int dwc2_ovr_avalid(struct dwc2_hsotg *hsotg, bool valid)
{
	u32 gotgctl = dwc2_readl(hsotg, GOTGCTL);

	/* Check if A-Session is already in the right state */
	if ((valid && (gotgctl & GOTGCTL_ASESVLD)) ||
	    (!valid && !(gotgctl & GOTGCTL_ASESVLD)))
		return -EALREADY;

	gotgctl &= ~GOTGCTL_BVALOVAL;
	if (valid)
		gotgctl |= GOTGCTL_AVALOVAL | GOTGCTL_VBVALOVAL;
	else
		gotgctl &= ~(GOTGCTL_AVALOVAL | GOTGCTL_VBVALOVAL);
	dwc2_writel(hsotg, gotgctl, GOTGCTL);

	return 0;
}

static int dwc2_ovr_bvalid(struct dwc2_hsotg *hsotg, bool valid)
{
	u32 gotgctl = dwc2_readl(hsotg, GOTGCTL);

	/* Check if B-Session is already in the right state */
	if ((valid && (gotgctl & GOTGCTL_BSESVLD)) ||
	    (!valid && !(gotgctl & GOTGCTL_BSESVLD)))
		return -EALREADY;

	gotgctl &= ~GOTGCTL_AVALOVAL;
	if (valid)
		gotgctl |= GOTGCTL_BVALOVAL | GOTGCTL_VBVALOVAL;
	else
		gotgctl &= ~(GOTGCTL_BVALOVAL | GOTGCTL_VBVALOVAL);
	dwc2_writel(hsotg, gotgctl, GOTGCTL);

	return 0;
}
static int dwc2_drd_role_sw_set(struct usb_role_switch *sw, enum usb_role role)
{
	struct dwc2_hsotg *hsotg = usb_role_switch_get_drvdata(sw);
	unsigned long flags;
	int already = 0;

	/* Skip session not in line with dr_mode */
	if ((role == USB_ROLE_DEVICE && hsotg->dr_mode == USB_DR_MODE_HOST) ||
	    (role == USB_ROLE_HOST && hsotg->dr_mode == USB_DR_MODE_PERIPHERAL))
		return -EINVAL;

#if IS_ENABLED(CONFIG_USB_DWC2_PERIPHERAL) || \
	IS_ENABLED(CONFIG_USB_DWC2_DUAL_ROLE)
	/* Skip session if core is in test mode */
	if (role == USB_ROLE_NONE && hsotg->test_mode) {
		dev_dbg(hsotg->dev, "Core is in test mode\n");
		return -EBUSY;
	}
#endif

	/*
	 * In case of USB_DR_MODE_PERIPHERAL, clock is disabled at the end of
	 * the probe and enabled on udc_start.
	 * If role-switch set is called before the udc_start, we need to enable
	 * the clock to read/write GOTGCTL and GUSBCFG registers to override
	 * mode and sessions. It is the case if cable is plugged at boot.
	 */
	if (!hsotg->ll_hw_enabled && hsotg->clk) {
		int ret = clk_prepare_enable(hsotg->clk);

		if (ret)
			return ret;
	}

	spin_lock_irqsave(&hsotg->lock, flags);

	if (role == USB_ROLE_HOST) {
		already = dwc2_ovr_avalid(hsotg, true);
	} else if (role == USB_ROLE_DEVICE) {
		already = dwc2_ovr_bvalid(hsotg, true);
		/* This clear DCTL.SFTDISCON bit */
		dwc2_hsotg_core_connect(hsotg);
	} else {
		if (dwc2_is_device_mode(hsotg)) {
			if (!dwc2_ovr_bvalid(hsotg, false))
				/* This set DCTL.SFTDISCON bit */
				dwc2_hsotg_core_disconnect(hsotg);
		} else {
			dwc2_ovr_avalid(hsotg, false);
		}
	}

	spin_unlock_irqrestore(&hsotg->lock, flags);

	if (!already && hsotg->dr_mode == USB_DR_MODE_OTG)
		/* This will raise a Connector ID Status Change Interrupt */
		dwc2_force_mode(hsotg, role == USB_ROLE_HOST);

	if (!hsotg->ll_hw_enabled && hsotg->clk)
		clk_disable_unprepare(hsotg->clk);

	dev_dbg(hsotg->dev, "%s-session valid\n",
		role == USB_ROLE_NONE ? "No" :
		role == USB_ROLE_HOST ? "A" : "B");

	return 0;
}
#endif

int dwc2_drd_init(struct dwc2_hsotg *hsotg)
{
	int ret;
	struct device *dev = hsotg->dev;

#ifndef CONFIG_ARCH_CXD90XXX
	struct usb_role_switch_desc role_sw_desc = {0};
	struct usb_role_switch *role_sw;

	if (!device_property_read_bool(hsotg->dev, "usb-role-switch"))
		return 0;

	role_sw_desc.driver_data = hsotg;
	role_sw_desc.fwnode = dev_fwnode(hsotg->dev);
	role_sw_desc.set = dwc2_drd_role_sw_set;
	role_sw_desc.allow_userspace_control = true;

	role_sw = usb_role_switch_register(hsotg->dev, &role_sw_desc);
	if (IS_ERR(role_sw)) {
		ret = PTR_ERR(role_sw);
		dev_err(hsotg->dev,
			"failed to register role switch: %d\n", ret);
		return ret;
	}

	hsotg->role_sw = role_sw;

	/* Enable override and initialize values */
	dwc2_ovr_init(hsotg);
#endif

	INIT_WORK(&hsotg->drd_work, __dwc2_set_mode);

	if (device_property_read_bool(dev, "extcon")) {
		hsotg->edev = extcon_get_edev_by_phandle(dev, 0);
		if (IS_ERR(hsotg->edev))
			return PTR_ERR(hsotg->edev);
	}


	hsotg->edev_nb.notifier_call = dwc2_drd_notifier;
	ret = extcon_register_notifier(hsotg->edev, EXTCON_USB_HOST, &hsotg->edev_nb);
	if (ret < 0) {
		s_print(S_DEBUG_ERROR, "couldn't register cable notifier\n");
		return ret;
	}
	hsotg->edev_nb_vbus.notifier_call = dwc2_drd_notifier_vbus;
	ret = extcon_register_notifier(hsotg->edev, EXTCON_USB, &hsotg->edev_nb_vbus);
	if (ret < 0) {
		s_print(S_DEBUG_ERROR, "couldn't register cable notifier\n");
		return ret;
	}

	dwc2_drd_update(hsotg);
	s_print(S_DEBUG_INFO, "%s called\n", __func__);
	return 0;
}

void dwc2_drd_suspend(struct dwc2_hsotg *hsotg)
{
	u32 gintsts, gintmsk;

	if (hsotg->role_sw && !hsotg->params.external_id_pin_ctl) {
		gintmsk = dwc2_readl(hsotg, GINTMSK);
		gintmsk &= ~GINTSTS_CONIDSTSCHNG;
		dwc2_writel(hsotg, gintmsk, GINTMSK);
		gintsts = dwc2_readl(hsotg, GINTSTS);
		dwc2_writel(hsotg, gintsts | GINTSTS_CONIDSTSCHNG, GINTSTS);
	}
	if (hsotg->gadget_enabled)
		dwc2_hsotg_remove(hsotg);
}

void dwc2_drd_resume(struct dwc2_hsotg *hsotg)
{
	u32 gintsts, gintmsk;

	if (hsotg->role_sw && !hsotg->params.external_id_pin_ctl) {
		gintsts = dwc2_readl(hsotg, GINTSTS);
		dwc2_writel(hsotg, gintsts | GINTSTS_CONIDSTSCHNG, GINTSTS);
		gintmsk = dwc2_readl(hsotg, GINTMSK);
		gintmsk |= GINTSTS_CONIDSTSCHNG;
		dwc2_writel(hsotg, gintmsk, GINTMSK);
	}
	if (hsotg->gadget_enabled)
		dwc2_gadget_init(hsotg);
}

void dwc2_drd_exit(struct dwc2_hsotg *hsotg)
{
	if (hsotg->role_sw)
		usb_role_switch_unregister(hsotg->role_sw);

	if (hsotg->edev) {
		extcon_unregister_notifier(hsotg->edev, EXTCON_USB_HOST, &hsotg->edev_nb);
		extcon_unregister_notifier(hsotg->edev, EXTCON_USB, &hsotg->edev_nb_vbus);
	}
	cancel_work_sync(&hsotg->drd_work);
}
