/* SPDX-License-Identifier:     GPL-2.0 */
/* Copyright 2022 Sony Corporation, SOCIONEXT INC. */
#include <linux/bits.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/mfd/syscon.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_platform.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/regmap.h>
#include <linux/clk.h>
#include <linux/clk-provider.h>
#include <linux/reset.h>
#include <linux/usb/of.h>
#include <linux/usb/otg.h>

#include "dwc2-cxd.h"
#include "../cxd/sdebug.h"

/* usb2 axi2avb registers */
#define D_H2XBB_PIF_WBEATNB		0x8
#define		 D_H2XBB_PIF_AXI8BEAT		0x7
#define D_H2XBB_PIF_RBEATNB		0x10
#define D_H2XBB_PIF_HO_WBEATNB	0x8
#define		 D_H2XBB_PIF_HO_AXI16BEAT	0x4
#define D_H2XBB_PIF_HO_RBEATNB	0x10
///* usb2 dev misc registers offset from 0x2050 */
#define D_OFS_USB20_HF_SEL		0x0
#define	 D_USB20_HF_SEL_VALDEV		0x0
#define	 D_USB20_HF_SEL_VALHOST		0x1
/* usb2 host misc registers offset from 0x20a0 */
#define D_OFS_USB20_OHCI_AHB_MON	0x0
#define   D_BIT_OHCI_AHB_MON_READY		BIT(16)
/* usb2 host misc registers offset from 0x2140 */
#define D_OFS_USB20_PHY_CTRL0	0x0
#define   D_BIT_PHY_CTRL_ANPD		BIT(0)
#ifdef CONFIG_ARCH_CXD90XXX_FPGA
/* usb2 host fj */
#define D_OFS_USB20_FJ_LINK_MODE	0x20
#define 	D_USB20_FJ_LINK_MODE_INC16_8_4	0xe
#endif

// timeout or wait
#define TIMEOUT_OHCI_AHB_MON_READY_US	(250)
#define WAIT_OHCI_AHB_MON_READY_US		(20)

static inline u32 cxd_dwc2_readl(void __iomem *base, u32 offset)
{
	return readl_relaxed(base + offset);
}

static inline void cxd_dwc2_writel(void __iomem *base, u32 offset, u32 value)
{
	writel_relaxed(value, base + offset);
}

#ifdef CONFIG_ARCH_CXD90XXX_FPGA
static void argo_test_func_cxd_dwc2_h2xbb_init(struct cxd_dwc2 *dwc2_data)
{
	uint32_t reg_val;
	/* H2XBB dev setting */
	reg_val = D_H2XBB_PIF_AXI8BEAT;
	cxd_dwc2_writel(dwc2_data->axi2avb_glue_base, D_H2XBB_PIF_WBEATNB, reg_val);
	cxd_dwc2_writel(dwc2_data->axi2avb_glue_base, D_H2XBB_PIF_RBEATNB, reg_val);
	/* H2XBB host setting */
	reg_val = D_H2XBB_PIF_HO_AXI16BEAT;
	cxd_dwc2_writel(dwc2_data->axi2avb_ho_glue_base, D_H2XBB_PIF_HO_WBEATNB, reg_val);
	cxd_dwc2_writel(dwc2_data->axi2avb_ho_glue_base, D_H2XBB_PIF_HO_RBEATNB, reg_val);
}
#endif

static int dwc2_cxd_probe(struct platform_device *pdev)
{
	struct cxd_dwc2 *dwc2_data;
	struct resource *res;
	struct device *dev = &pdev->dev;
	struct device_node *node = dev->of_node, *child;
	int ret = 0;

	dwc2_data = devm_kzalloc(dev, sizeof(*dwc2_data), GFP_KERNEL);
	if (!dwc2_data)
		return -ENOMEM;

	dwc2_data->dev = dev;
	dwc2_data->hclk = devm_clk_get_optional(dev, "hclk");
	if (IS_ERR(dwc2_data->hclk))
		return PTR_ERR(dwc2_data->hclk);

	dwc2_data->ho_hclk = devm_clk_get_optional(dev, "ho_hclk");
	if (IS_ERR(dwc2_data->ho_hclk))
		return PTR_ERR(dwc2_data->ho_hclk);

	dwc2_data->ohci_clk48 = devm_clk_get_optional(dev, "ohci_clk48");
	if (IS_ERR(dwc2_data->ohci_clk48))
		return PTR_ERR(dwc2_data->ohci_clk48);

#if defined(CONFIG_USB2_SMMU_LIB)
	dwc2_data->mmu_cclk = devm_clk_get_optional(dev, "mmu_cclk");
	if (IS_ERR(dwc2_data->mmu_cclk))
		return PTR_ERR(dwc2_data->mmu_cclk);

	dwc2_data->mmu_tbu_bclk0 = devm_clk_get_optional(dev, "mmu_tbu_bclk0");
	if (IS_ERR(dwc2_data->mmu_tbu_bclk0))
		return PTR_ERR(dwc2_data->mmu_tbu_bclk0);
#endif
	dwc2_data->psave_on = D_PSAVE_ON;

	dwc2_data->por_rst = devm_reset_control_get(&pdev->dev, "por_rst");
	if (IS_ERR(dwc2_data->por_rst))
		return PTR_ERR(dwc2_data->por_rst);

	dwc2_data->prst_n = devm_reset_control_get(&pdev->dev, "prst_n");
	if (IS_ERR(dwc2_data->prst_n))
		return PTR_ERR(dwc2_data->prst_n);

	dwc2_data->hresetn = devm_reset_control_get(&pdev->dev, "hresetn");
	if (IS_ERR(dwc2_data->hresetn))
		return PTR_ERR(dwc2_data->hresetn);

	dwc2_data->ho_hresetn = devm_reset_control_get(&pdev->dev, "ho_hresetn");
	if (IS_ERR(dwc2_data->ho_hresetn))
		return PTR_ERR(dwc2_data->ho_hresetn);

#if defined(CONFIG_USB2_SMMU_LIB)
	dwc2_data->mmu_cresetn = devm_reset_control_get(&pdev->dev, "mmu_cresetn");
	if (IS_ERR(dwc2_data->mmu_cresetn))
		return PTR_ERR(dwc2_data->mmu_cresetn);

	dwc2_data->mmu_tbu_bresetn0 = devm_reset_control_get(&pdev->dev, "mmu_tbu_bresetn0");
	if (IS_ERR(dwc2_data->mmu_tbu_bresetn0))
		return PTR_ERR(dwc2_data->mmu_tbu_bresetn0);
#endif

	/* SNI glue device */
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "reg-glue");
	dwc2_data->glue_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(dwc2_data->glue_base))
		return PTR_ERR(dwc2_data->glue_base);
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "ho-reg-glue");
	dwc2_data->ho_glue_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(dwc2_data->ho_glue_base))
		return PTR_ERR(dwc2_data->ho_glue_base);
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "anpd-reg-glue");
	dwc2_data->anpd_glue_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(dwc2_data->anpd_glue_base))
		return PTR_ERR(dwc2_data->anpd_glue_base);
#if defined(CONFIG_ARCH_CXD90XXX_FPGA) || defined(D_WA_DDR_COHERENCY_ISSUE)
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "axi2avb-reg-glue");
	dwc2_data->axi2avb_glue_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(dwc2_data->axi2avb_glue_base))
		return PTR_ERR(dwc2_data->axi2avb_glue_base);
#endif
	/* SNI glue host */
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "host-reg-glue");
	dwc2_data->host_glue_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(dwc2_data->host_glue_base))
		return PTR_ERR(dwc2_data->host_glue_base);
#ifdef CONFIG_ARCH_CXD90XXX_FPGA
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "axi2avb-ho-reg-glue");
	dwc2_data->axi2avb_ho_glue_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(dwc2_data->axi2avb_ho_glue_base))
		return PTR_ERR(dwc2_data->axi2avb_ho_glue_base);
#endif

#if defined(CONFIG_USB2_SMMU_LIB)
	/* smmu glue device */
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "smmu_base");
	dwc2_data->smmu_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(dwc2_data->smmu_base))
		return PTR_ERR(dwc2_data->smmu_base);
	if (of_property_read_u64(node, "smmu-ram", &dwc2_data->smmu_ram))
		return -ENODEV;
	if (of_property_read_u32(node, "smmu-len", &dwc2_data->smmu_len))
		return -ENODEV;
#endif
#if defined(D_WA_DDR_COHERENCY_ISSUE)
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "axi2avb-ohci-reg-glue");
	dwc2_data->axi2avb_ohci_glue_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(dwc2_data->axi2avb_ohci_glue_base))
		return PTR_ERR(dwc2_data->axi2avb_ohci_glue_base);
#endif

	dwc2_cxd_init_suppl();
	platform_set_drvdata(pdev, dwc2_data);

	ret = dwc2_cxd_dev_prepare_start(dwc2_data);
	if (ret)
		return ret;

	/* for dwc2 core */
	child = of_get_child_by_name(node, "dwc2");
	if (!child) {
		s_print(S_DEBUG_ERROR, "failed to find dwc2 core node\n");
		return -ENODEV;
	}

	/* Allocate and initialize the core */
	ret = of_platform_populate(node, NULL, NULL, dev);
	if (ret) {
		s_print(S_DEBUG_ERROR, "failed to add dwc2 core\n");
		return ret;
	}

	dwc2_data->child_pdev = of_find_device_by_node(child);
	if (!dwc2_data->child_pdev) {
		s_print(S_DEBUG_ERROR, "failed to find dwc2 core device\n");
		return -ENODEV;
	}

	return ret;
}

static void anpd_set(struct cxd_dwc2 *dwc, uint32_t set)
{
	int reg;

	reg = cxd_dwc2_readl(dwc->anpd_glue_base, D_OFS_USB20_PHY_CTRL0);
	if (set == D_PSAVE_ON)
		reg |= D_BIT_PHY_CTRL_ANPD;
	else
		reg &= ~D_BIT_PHY_CTRL_ANPD;
	cxd_dwc2_writel(dwc->anpd_glue_base, D_OFS_USB20_PHY_CTRL0, reg);
}

int dwc2_cxd_dev_prepare_start(struct cxd_dwc2 *dwc)
{
	int err = 0;
	uint32_t reg;

	dwc->psave_on = D_PSAVE_OFF;
	anpd_set(dwc, D_PSAVE_OFF);
#if defined(CONFIG_USB2_SMMU_LIB)
	err = clk_prepare_enable(dwc->mmu_tbu_bclk0);
	if (err) {
		s_print(S_DEBUG_WARNING, "could not enable mmu_tbu_bclk0: %d\n", err);
		goto err_tbu_bclk0;
	}

	err = clk_prepare_enable(dwc->mmu_cclk);
	if (err) {
		s_print(S_DEBUG_WARNING, "could not enable mmu_cclk: %d\n", err);
		goto err_cclk;
	}
#endif
	err = clk_prepare_enable(dwc->hclk);
	if (err) {
		s_print(S_DEBUG_WARNING, "could not enable hclk: %d\n", err);
		goto err_hclk;
	}

	reg = D_USB20_HF_SEL_VALDEV;
	cxd_dwc2_writel(dwc->glue_base, D_OFS_USB20_HF_SEL, reg);

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	//ES
	udelay(10);
	dwc2_cxd_set_phy_reg();
	reset_control_deassert(dwc->por_rst);
	#if defined(CONFIG_USB2_SMMU_LIB)
	reset_control_deassert(dwc->mmu_tbu_bresetn0);
	reset_control_deassert(dwc->mmu_cresetn);
	#endif
	udelay(46);
	reset_control_deassert(dwc->prst_n);
	udelay(1);
	reset_control_deassert(dwc->hresetn);
	udelay(1);
//FIXME w/a mmu_tbl_sram_aclk
	reg = cxd_dwc2_readl(ioremap(0xfc5e1000, 0x1000), 0xe0);
	reg |= 0x1000;
	cxd_dwc2_writel(ioremap(0xfc5e1000, 0x1000), 0xe0, reg);
//FIXME w/a mmu_tbl_sram_aresetn
	reg = cxd_dwc2_readl(ioremap(0xfc5e1000, 0x1000), 0x4e0);
	reg |= 0x1000;
	cxd_dwc2_writel(ioremap(0xfc5e1000, 0x1000), 0x4e0, reg);
#else
	reset_control_deassert(dwc->por_rst);
	reset_control_deassert(dwc->ho_hresetn);
	reset_control_deassert(dwc->hresetn);
	reset_control_deassert(dwc->prst_n);
#if defined(CONFIG_USB2_SMMU_LIB)
	reset_control_deassert(dwc->mmu_cresetn);
	reset_control_deassert(dwc->mmu_tbu_bresetn0);
#endif
	argo_test_func_cxd_dwc2_h2xbb_init(dwc);
#endif

	s_print(S_DEBUG_INFO, "%s\n", __func__);

	return err;

err_hclk:
#if defined(CONFIG_USB2_SMMU_LIB)
	clk_disable_unprepare(dwc->mmu_cclk);
err_cclk:
	clk_disable_unprepare(dwc->mmu_tbu_bclk0);
err_tbu_bclk0:
#endif
	return err;
}
EXPORT_SYMBOL_GPL(dwc2_cxd_dev_prepare_start);

void dwc2_cxd_dev_prepare_stop(struct cxd_dwc2 *dwc)
{
	if (dwc->psave_on == D_PSAVE_ON)
		return;
#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	//ES
	#if defined(CONFIG_USB2_SMMU_LIB)
	reset_control_assert(dwc->mmu_tbu_bresetn0);
	reset_control_assert(dwc->mmu_cresetn);
	#endif
	reset_control_assert(dwc->por_rst);
	reset_control_assert(dwc->prst_n);
	reset_control_assert(dwc->hresetn);

	#if defined(CONFIG_USB2_SMMU_LIB)
	clk_disable_unprepare(dwc->mmu_tbu_bclk0);
	clk_disable_unprepare(dwc->mmu_cclk);
	#endif
	clk_disable_unprepare(dwc->hclk);
#else
	reset_control_assert(dwc->por_rst);
	reset_control_assert(dwc->ho_hresetn);
	reset_control_assert(dwc->hresetn);
	reset_control_assert(dwc->prst_n);
#if defined(CONFIG_USB2_SMMU_LIB)
	reset_control_assert(dwc->mmu_cresetn);
	reset_control_assert(dwc->mmu_tbu_bresetn0);
#endif
	clk_disable_unprepare(dwc->hclk);
#if defined(CONFIG_USB2_SMMU_LIB)
	clk_disable_unprepare(dwc->mmu_tbu_bclk0);
	clk_disable_unprepare(dwc->mmu_cclk);
#endif
#endif
	dwc->psave_on = D_PSAVE_ON;
	anpd_set(dwc, D_PSAVE_ON);
	s_print(S_DEBUG_INFO, "%s\n", __func__);
}
EXPORT_SYMBOL_GPL(dwc2_cxd_dev_prepare_stop);

int dwc2_cxd_host_prepare_start(struct cxd_dwc2 *dwc)
{
	int err = 0;
	uint32_t reg;
#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	int32_t timeout;
#endif

	anpd_set(dwc, D_PSAVE_OFF);
#if defined(CONFIG_USB2_SMMU_LIB)
	err = clk_prepare_enable(dwc->mmu_tbu_bclk0);
	if (err) {
		s_print(S_DEBUG_WARNING, "could not enable mmu_tbu_bclk0: %d\n", err);
		goto err_bclk0;
	}
	err = clk_prepare_enable(dwc->mmu_cclk);
	if (err) {
		s_print(S_DEBUG_WARNING, "could not enable mmu_cclk: %d\n", err);
		goto err_cclk;
	}
#endif
	err = clk_prepare_enable(dwc->ho_hclk);
	if (err) {
		s_print(S_DEBUG_WARNING, "could not enable ho_hclk: %d\n", err);
		goto err_hclk;
	}
	err = clk_prepare_enable(dwc->ohci_clk48);
	if (err) {
		s_print(S_DEBUG_WARNING, "could not enable ohci_clk48: %d\n", err);
		goto err_clk48;
	}

	reg = D_USB20_HF_SEL_VALHOST;
	cxd_dwc2_writel(dwc->glue_base, D_OFS_USB20_HF_SEL, reg);

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	//ES
	dwc2_cxd_set_phy_reg();
	#if defined(CONFIG_USB2_SMMU_LIB)
	reset_control_deassert(dwc->mmu_tbu_bresetn0);
	reset_control_deassert(dwc->mmu_cresetn);
	#endif
	reset_control_deassert(dwc->ho_hresetn);
#else
	reset_control_deassert(dwc->por_rst);
	reset_control_deassert(dwc->ho_hresetn);
	reset_control_deassert(dwc->hresetn);
	reset_control_deassert(dwc->prst_n);
#if defined(CONFIG_USB2_SMMU_LIB)
	reset_control_deassert(dwc->mmu_cresetn);
	reset_control_deassert(dwc->mmu_tbu_bresetn0);
#endif
	argo_test_func_cxd_dwc2_h2xbb_init(dwc);

	reg = D_USB20_FJ_LINK_MODE_INC16_8_4;
	cxd_dwc2_writel(dwc->axi2avb_ho_glue_base, D_OFS_USB20_FJ_LINK_MODE, reg);
#endif

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	udelay(57);
	timeout = TIMEOUT_OHCI_AHB_MON_READY_US;
	while (1) {
		reg = cxd_dwc2_readl(dwc->ho_glue_base, D_OFS_USB20_OHCI_AHB_MON);
		reg &= D_BIT_OHCI_AHB_MON_READY;
		if (reg) {
			break;
		}

		if (timeout <= 0) {
			s_print(S_DEBUG_WARNING, "could not check regaccess_ready=1\n");
			err = -EIO;
			goto err_eio;
		}

		udelay(WAIT_OHCI_AHB_MON_READY_US);
		timeout -= WAIT_OHCI_AHB_MON_READY_US;
	}
#endif

	s_print(S_DEBUG_INFO, "%s\n", __func__);
	return err;

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
err_eio:
	clk_disable_unprepare(dwc->ohci_clk48);
#endif
err_clk48:
	clk_disable_unprepare(dwc->ho_hclk);
err_hclk:
#if defined(CONFIG_USB2_SMMU_LIB)
	clk_disable_unprepare(dwc->mmu_cclk);
err_cclk:
	clk_disable_unprepare(dwc->mmu_tbu_bclk0);
err_bclk0:
#endif
	return err;
}
EXPORT_SYMBOL_GPL(dwc2_cxd_host_prepare_start);

void dwc2_cxd_host_prepare_stop(struct cxd_dwc2 *dwc)
{
#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	#if defined(CONFIG_USB2_SMMU_LIB)
	reset_control_assert(dwc->mmu_tbu_bresetn0);
	reset_control_assert(dwc->mmu_cresetn);
	#endif
	reset_control_assert(dwc->ho_hresetn);

	#if defined(CONFIG_USB2_SMMU_LIB)
	clk_disable_unprepare(dwc->mmu_tbu_bclk0);
	clk_disable_unprepare(dwc->mmu_cclk);
	#endif
	clk_disable_unprepare(dwc->ho_hclk);
	clk_disable_unprepare(dwc->ohci_clk48);
#else
	reset_control_assert(dwc->prst_n);
	reset_control_assert(dwc->ho_hresetn);
	reset_control_assert(dwc->hresetn);
	reset_control_assert(dwc->por_rst);
#if defined(CONFIG_USB2_SMMU_LIB)
	reset_control_assert(dwc->mmu_cresetn);
	reset_control_assert(dwc->mmu_tbu_bresetn0);
#endif
	clk_disable_unprepare(dwc->ho_hclk);
	clk_disable_unprepare(dwc->ohci_clk48);
#if defined(CONFIG_USB2_SMMU_LIB)
	clk_disable_unprepare(dwc->mmu_tbu_bclk0);
	clk_disable_unprepare(dwc->mmu_cclk);
	#endif
#endif
	anpd_set(dwc, D_PSAVE_ON);
	s_print(S_DEBUG_INFO, "%s\n", __func__);
}
EXPORT_SYMBOL_GPL(dwc2_cxd_host_prepare_stop);

static int dwc2_cxd_remove(struct platform_device *pdev)
{
	struct cxd_dwc2 *data = platform_get_drvdata(pdev);

#if defined(CONFIG_USB2_SMMU_LIB)
	reset_control_assert(data->mmu_cresetn);
	reset_control_assert(data->mmu_tbu_bresetn0);
#endif
	of_platform_depopulate(&pdev->dev);

	return 0;
}
static int dwc2_cxd_suspend(struct device *dev)
{
	return 0;
}
static int dwc2_cxd_resume(struct device *dev)
{
	return 0;
}

static const struct dev_pm_ops cxd_dwc2_pm = {
	SET_SYSTEM_SLEEP_PM_OPS(dwc2_cxd_suspend, dwc2_cxd_resume)
};

static const struct of_device_id cxd_dwc2_match[] = {
	{ .compatible = "cxd,dwc2" },
	{ /* sentinel */ },
};

MODULE_DEVICE_TABLE(of, cxd_dwc2_match);

static struct platform_driver cxd_dwc2_driver = {
	.probe = dwc2_cxd_probe,
	.remove = dwc2_cxd_remove,
	.driver = {
		.name = "usb-cxd-dwc2",
		.of_match_table = cxd_dwc2_match,
		.pm = &cxd_dwc2_pm,
	},
};

module_platform_driver(cxd_dwc2_driver);

MODULE_LICENSE("GPL");
