/* SPDX-License-Identifier:     GPL-2.0 */
/* Copyright 2022 Sony Corporation, SOCIONEXT INC. */
/* Portions Copyright (C) 2024 Synopsys, Inc.  Used with permission. All rights reserved. */
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/mfd/syscon.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_platform.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/regmap.h>
#include <linux/clk.h>
#include <linux/clk-provider.h>
#include <linux/reset.h>
#include <linux/usb/of.h>
#include <linux/usb/otg.h>
#include <linux/spinlock.h>

#include "core.h"
#include "dwc3-cxd.h"
#include "../cxd/sdebug.h"
#include "../dwc3/core.h"

#ifdef TESTCODE_USB_DWC3_LOCK
int __attribute__((optimize("O0")))l_start_dwc3_tca_cc(char *arg, ...) { return 0; }
int __attribute__((optimize("O0")))l_end_dwc3_tca_cc(char *arg, ...) { return 0; }
EXPORT_SYMBOL_GPL(l_start_dwc3_tca_cc);
EXPORT_SYMBOL_GPL(l_end_dwc3_tca_cc);
#endif

/* usb32 misc registers offset from 0x1eb0(PHY_SRAM_CTL=0) */
#define D_OFS_USB32_PHY_SRAM_CTRL		0x0
#define 	D_PHY0_SRAM_BYPASS			BIT(0)
#define 	D_SRAM_EXT_LD_DONE			BIT(1)
#define D_OFS_USB32_PHY_SRAM_MON		0x10
#define D_OFS_USB32_ANPD				0x20
#define 	D_ANPD_ON					BIT(0)
#define 	D_ANPD_OFF					0

/* tca */
#define D_OFS_TCA_INTR_EN				0x4
#define		D_BIT_XA_ACT_EVT_EN			BIT(0)
#define		D_BIT_XA_TO_EVT_EN			BIT(1)
#define D_OFS_TCA_INTR_STS				0x8
#define		D_BIT_INTR_STS_CLR			0xffffffff
#define		D_BIT_XA_ACK_EVT			BIT(0)
#define		D_BIT_XA_TO_ACK_EVT			BIT(1)
#define D_OFS_TCA_TCPC					0x14
#define		D_BIT_TCPC_MUX_CONTROL		0x00000003
#define		D_SET_TCPC_MUX_GEN2X1		0x00000001
#define		D_BIT_TCPC_CONN_ORIEN		BIT(2)
#define		D_BIT_TCPC_LOW_POWER_EN		BIT(3)
#define		D_BIT_TCPC_VALID			BIT(4)
#define D_TCA_SYSMODE_CFG				0x18
#define		D_BIT_TYPEC_DISABLE		BIT(3)

#define	D_POLARITY_INIT					0

#define D_RETRY_NUM					13
#define D_RETRY_NUM_PHY0_SRAM_INIT_DONE			300
#define D_RETRY_NUM_XA_ACK_EVT				26
#define D_WAIT_MARGIN					1
#define D_WAIT_MARGIN_FOR_SRAM_INIT_DONE		30
#define D_WAIT_FOR_BEFOR_USB32IP			20
#define D_WAIT_FOR_TCA_APB_CLK_CYC			1
#define D_WAIT_FOR_BEFOR_RESET_DEASSERT			15
#define D_WAIT_FOR_PHY0_SRAM_INIT_DONE			1
#define D_WAIT_FOR_TCA_INIT_STEP2			1
#define D_WAIT_FOR_TCA_INIT_STEP7			1



static inline u32 cxd_dwc3_readl(void __iomem *base, u32 offset)
{
	return readl(base + offset);
}

static inline void cxd_dwc3_writel(void __iomem *base, u32 offset, u32 value)
{
	writel_relaxed(value, base + offset);
}

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
void dwc3_cxd_is_disconnect(struct cxd_dwc3 *dwc, uint32_t is_disconn)
{
	uint32_t reg_val;

	spin_lock(&dwc->lock);
	reg_val = cxd_dwc3_readl(dwc->phy_base, D_TCA_SYSMODE_CFG);
	if (is_disconn)
		reg_val |= D_BIT_TYPEC_DISABLE;
	else
		reg_val &= ~D_BIT_TYPEC_DISABLE;

	cxd_dwc3_writel(dwc->phy_base, D_TCA_SYSMODE_CFG, reg_val);
	spin_unlock(&dwc->lock);
}
EXPORT_SYMBOL(dwc3_cxd_is_disconnect);
#endif

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
static void step5(struct cxd_dwc3 *dwc, uint32_t polarity)
{
	uint32_t reg_val;

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	dwc3_cxd_is_disconnect(dwc, D_IS_CONNECT);
#endif
	spin_lock(&dwc->tcalock);
#ifdef TESTCODE_USB_DWC3_LOCK
	l_start_dwc3_tca_cc("start");
#endif
	reg_val = cxd_dwc3_readl(dwc->phy_base, D_OFS_TCA_TCPC);
	reg_val &= ~D_BIT_TCPC_LOW_POWER_EN;
	reg_val |= (D_SET_TCPC_MUX_GEN2X1 | D_BIT_TCPC_VALID);
	if (polarity)
		reg_val |= D_BIT_TCPC_CONN_ORIEN;/* connector flip */
	else
		reg_val &= ~D_BIT_TCPC_CONN_ORIEN;/* connector normal */
#ifdef TESTCODE_USB_DWC3_LOCK
	l_end_dwc3_tca_cc("enc");
#endif
	spin_unlock(&dwc->tcalock);

	cxd_dwc3_writel(dwc->phy_base, D_OFS_TCA_TCPC, reg_val);
}

static void step6to9(struct cxd_dwc3 *dwc)
{
	volatile uint32_t reg_val;
	uint32_t retry;

	/* step6 */
	retry = D_RETRY_NUM + D_WAIT_MARGIN;
	do {
		mdelay(D_WAIT_FOR_TCA_INIT_STEP7);/* step7 */
		/* step8 */
		reg_val = cxd_dwc3_readl(dwc->phy_base, D_OFS_TCA_INTR_STS);
		if (reg_val & D_BIT_XA_TO_ACK_EVT) {
			dev_err(dwc->dev, "tca_init step6 failed \n");
			return;
		}
		reg_val &= D_BIT_XA_ACK_EVT;

		--retry;
		if (retry == 0) {
			dev_err(dwc->dev, "both xa_act event and timeout did not occur\n");
			return;
		}
	} while(!reg_val);
	/* Step9 */
	cxd_dwc3_writel(dwc->phy_base, D_OFS_TCA_INTR_STS, D_BIT_XA_ACK_EVT);
	return;
}

static void tca_init(struct cxd_dwc3 *dwc, uint32_t polarity)
{
	volatile uint32_t reg_val;
	uint32_t retry;
	/* step2 */

	retry = D_RETRY_NUM_XA_ACK_EVT + D_WAIT_MARGIN;
	do {
		reg_val = cxd_dwc3_readl(dwc->phy_base, D_OFS_TCA_INTR_STS);
		reg_val &= D_BIT_XA_ACK_EVT;
		mdelay(D_WAIT_FOR_TCA_INIT_STEP2);
		--retry;
		if (retry == 0) {
			dev_err(dwc->dev, "time out for \"PHY ready\"\n");
			return;
		}
	} while(!reg_val);
	/* step3 */
	cxd_dwc3_writel(dwc->phy_base, D_OFS_TCA_INTR_STS, D_BIT_INTR_STS_CLR);
	/* step4 */
	cxd_dwc3_writel(dwc->phy_base, D_OFS_TCA_INTR_EN, 0);
	/* step5 */
	step5(dwc, D_POLARITY_INIT);
	/* step6 to step9 */
	step6to9(dwc);
		}

void dwc3_tca_cc_control_event(struct cxd_dwc3 *dwc, uint32_t polarity)
{
	step5(dwc, polarity);
	step6to9(dwc);
}
EXPORT_SYMBOL(dwc3_tca_cc_control_event);
#endif

static int dwc3_cxd_clk_enable(struct cxd_dwc3 *data)
{
	int err;

	err = clk_prepare_enable(data->tca_apb_clk);
	if (err) {
		s_print(S_DEBUG_INFO, " could not enable tca_apb_clk: %d\\n", err);
		goto err;
	}
	err = clk_prepare_enable(data->suspend_clk);
	if (err) {
		s_print(S_DEBUG_INFO, " could not enable suspend_clk: %d\\n", err);
		goto err_suspend_clk;
	}
	err = clk_prepare_enable(data->aclk);
	if (err) {
		s_print(S_DEBUG_INFO, " could not enable aclk: %d\\n", err);
		goto err_aclk;
	}
	return 0;

err_aclk:
	clk_disable_unprepare(data->suspend_clk);
err_suspend_clk:
	clk_disable_unprepare(data->tca_apb_clk);
err:
	return err;
}

static int dwc3_cxd_probe(struct platform_device *pdev)
{
	struct cxd_dwc3 *dwc3_data;
	struct resource *res;
	struct device *dev = &pdev->dev;
	struct device_node *node = dev->of_node, *child;
	struct platform_device *child_pdev;
	int ret;
#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	uint32_t reg_val;
	uint32_t retry;
#endif
	
	ret = 0;
	dwc3_data = devm_kzalloc(dev, sizeof(*dwc3_data), GFP_KERNEL);
	if (!dwc3_data)
		return -ENOMEM;

	dwc3_data->dev = dev;

	/* SNI glue */
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "reg-glue");
	dwc3_data->glue_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(dwc3_data->glue_base))
		return PTR_ERR(dwc3_data->glue_base);

	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "reg-phy");
	dwc3_data->phy_base = devm_ioremap_resource(dev, res);
	if (IS_ERR(dwc3_data->phy_base))
		return PTR_ERR(dwc3_data->phy_base);

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	// ANPD=0(normal)
	cxd_dwc3_writel(dwc3_data->glue_base, D_OFS_USB32_ANPD, D_ANPD_OFF);
	// SRAM operation
	reg_val = cxd_dwc3_readl(dwc3_data->glue_base, D_OFS_USB32_PHY_SRAM_CTRL);
	reg_val &= ~D_PHY0_SRAM_BYPASS;
	cxd_dwc3_writel(dwc3_data->glue_base, D_OFS_USB32_PHY_SRAM_CTRL, reg_val);
#endif

	spin_lock_init(&dwc3_data->lock);
	spin_lock_init(&dwc3_data->tcalock);
	dwc3_cxd_init_suppl();
	dwc3_data->tca_apb_clk = devm_clk_get_optional(dev, "tca_apb_clk");
	if (IS_ERR(dwc3_data->tca_apb_clk)) {
		ret = PTR_ERR(dwc3_data->tca_apb_clk);
		goto err_clk;
	}
	dwc3_data->suspend_clk = devm_clk_get_optional(dev, "suspend_clk");
	if (IS_ERR(dwc3_data->suspend_clk)) {
		ret = PTR_ERR(dwc3_data->suspend_clk);
		goto err_clk;
	}
	dwc3_data->aclk = devm_clk_get_optional(dev, "aclk");
	if (IS_ERR(dwc3_data->aclk)) {
		ret = PTR_ERR(dwc3_data->aclk);
		goto err_clk;
	}
	ret = dwc3_cxd_clk_enable(dwc3_data);
	if (ret)
		goto err_clk;

	udelay(D_WAIT_FOR_BEFOR_RESET_DEASSERT + D_WAIT_MARGIN);
	dwc3_data->phy_reset = devm_reset_control_get(&pdev->dev, "phy_reset");
	if (IS_ERR(dwc3_data->phy_reset)) {
		ret = PTR_ERR(dwc3_data->phy_reset);
		goto err_phy_reset;
	}
	reset_control_deassert(dwc3_data->phy_reset);

	dwc3_data->por_rst = devm_reset_control_get(&pdev->dev, "por_rst");
	if (IS_ERR(dwc3_data->por_rst)) {
		ret = PTR_ERR(dwc3_data->por_rst);
		goto err_por_rst;
	}
	reset_control_deassert(dwc3_data->por_rst);

	dwc3_data->aresetn = devm_reset_control_get(&pdev->dev, "aresetn");
	if (IS_ERR(dwc3_data->aresetn)) {
		ret = PTR_ERR(dwc3_data->aresetn);
		goto err_aresetn;
	}
	reset_control_deassert(dwc3_data->aresetn);
	dwc3_data->psave_on = D_PSAVE_OFF;

	udelay(D_WAIT_FOR_TCA_APB_CLK_CYC);
	dwc3_data->apbrst = devm_reset_control_get(&pdev->dev, "apbrst");
	if (IS_ERR(dwc3_data->apbrst)) {
		ret = PTR_ERR(dwc3_data->apbrst);
		goto err_apbrst;
	}
	reset_control_deassert(dwc3_data->apbrst);


#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	retry = D_RETRY_NUM_PHY0_SRAM_INIT_DONE + D_WAIT_MARGIN_FOR_SRAM_INIT_DONE;
	do {
		reg_val = cxd_dwc3_readl(dwc3_data->glue_base, D_OFS_USB32_PHY_SRAM_MON);
		udelay(D_WAIT_FOR_PHY0_SRAM_INIT_DONE);//from PHY_RESET to PHY0_SRAM_INIT_DONE
		--retry;
		if (retry == 0) {
			dev_err(&pdev->dev, "failed to done phy_sram_mon\n");
			goto undo_softreset;
		}
	} while(!reg_val);
	udelay(D_WAIT_FOR_TCA_APB_CLK_CYC);
	reg_val = D_SRAM_EXT_LD_DONE;
	cxd_dwc3_writel(dwc3_data->glue_base, D_OFS_USB32_PHY_SRAM_CTRL, reg_val);

	tca_init(dwc3_data, D_POLARITY_INIT);
	mdelay(D_WAIT_FOR_BEFOR_USB32IP + D_WAIT_MARGIN);
#endif

	platform_set_drvdata(pdev, dwc3_data);
	/* for dwc3 core */
	child = of_get_child_by_name(node, "dwc3");
	if (!child) {
		dev_err(&pdev->dev, "failed to find dwc3 core node\n");
		ret = -ENODEV;
		goto undo_softreset;
	}

	/* Allocate and initialize the core */
	ret = of_platform_populate(node, NULL, NULL, dev);
	if (ret) {
		dev_err(dev, "failed to add dwc3 core\n");
		goto undo_softreset;
	}

	child_pdev = of_find_device_by_node(child);
	if (!child_pdev) {
		dev_err(dev, "failed to find dwc3 core device\n");
		ret = -ENODEV;
		goto undo_softreset;
	}

	return ret;
	

undo_softreset:
	reset_control_assert(dwc3_data->apbrst);
err_apbrst:
	reset_control_assert(dwc3_data->aresetn);
err_aresetn:
	reset_control_assert(dwc3_data->por_rst);
err_por_rst:
	reset_control_assert(dwc3_data->phy_reset);
err_phy_reset:
	clk_disable_unprepare(dwc3_data->tca_apb_clk);
	clk_disable_unprepare(dwc3_data->suspend_clk);
	clk_disable_unprepare(dwc3_data->aclk);
err_clk:

	return ret;
}

int dwc3_cxd_psave_off(struct cxd_dwc3 *data)
{
#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	uint32_t reg_val;
	uint32_t retry;
#endif
	int ret;

	data->psave_on = D_PSAVE_OFF;
	cxd_dwc3_writel(data->glue_base, D_OFS_USB32_ANPD, D_ANPD_OFF);
	// SRAM operation
	reg_val = cxd_dwc3_readl(data->glue_base, D_OFS_USB32_PHY_SRAM_CTRL);
	reg_val &= ~D_PHY0_SRAM_BYPASS;
	cxd_dwc3_writel(data->glue_base, D_OFS_USB32_PHY_SRAM_CTRL, reg_val);

	ret = dwc3_cxd_clk_enable(data);
	if (ret) {
		s_print(S_DEBUG_INFO, " failed to go to power save\n");
		return ret;
	}

	udelay(D_WAIT_FOR_BEFOR_RESET_DEASSERT + D_WAIT_MARGIN);
	reset_control_deassert(data->phy_reset);
	reset_control_deassert(data->por_rst);
	reset_control_deassert(data->aresetn);
	udelay(D_WAIT_FOR_TCA_APB_CLK_CYC);
	reset_control_deassert(data->apbrst);

#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	retry = D_RETRY_NUM_PHY0_SRAM_INIT_DONE + D_WAIT_MARGIN_FOR_SRAM_INIT_DONE;
	do {
		reg_val = cxd_dwc3_readl(data->glue_base, D_OFS_USB32_PHY_SRAM_MON);
		udelay(D_WAIT_FOR_PHY0_SRAM_INIT_DONE);//from PHY_RESET to PHY0_SRAM_INIT_DONE
		--retry;
		if (retry == 0) {
			dev_err(data->dev, "failed to done phy_sram_mon\n");
			goto undo_softreset;
		}
	} while(!reg_val);
	udelay(D_WAIT_FOR_TCA_APB_CLK_CYC);
	dwc3_cxd_set_phy_reg();
	reg_val = D_SRAM_EXT_LD_DONE;
	cxd_dwc3_writel(data->glue_base, D_OFS_USB32_PHY_SRAM_CTRL, reg_val);

	tca_init(data, D_POLARITY_INIT);
	mdelay(D_WAIT_FOR_BEFOR_USB32IP + D_WAIT_MARGIN);
#endif
	s_print(S_DEBUG_INFO, " %s\n", __func__);

	return 0;
#ifndef CONFIG_ARCH_CXD90XXX_FPGA
undo_softreset:
	reset_control_assert(data->apbrst);
	reset_control_assert(data->aresetn);
	reset_control_assert(data->por_rst);
	reset_control_assert(data->phy_reset);
	clk_disable_unprepare(data->tca_apb_clk);
	clk_disable_unprepare(data->suspend_clk);
	clk_disable_unprepare(data->aclk);

	return ret;
#endif

}
EXPORT_SYMBOL(dwc3_cxd_psave_off);

static int prepare_stop(struct cxd_dwc3 *data)
{
	int rc;

	rc = reset_control_assert(data->apbrst);
	rc += reset_control_assert(data->phy_reset);
	rc += reset_control_assert(data->por_rst);
	rc += reset_control_assert(data->aresetn);
	if (rc != 0)
		return rc;
	clk_disable_unprepare(data->tca_apb_clk);
	clk_disable_unprepare(data->suspend_clk);
	clk_disable_unprepare(data->aclk);

	return 0;
}

void dwc3_cxd_psave_on(struct cxd_dwc3 *data)
{
	data->psave_on = D_PSAVE_ON;
#ifndef CONFIG_ARCH_CXD90XXX_FPGA
	dwc3_cxd_is_disconnect(data, D_IS_DISCONN);
#endif
	prepare_stop(data);
	cxd_dwc3_writel(data->glue_base, D_OFS_USB32_ANPD, D_ANPD_ON);
	s_print(S_DEBUG_INFO, " %s\n", __func__);
}
EXPORT_SYMBOL(dwc3_cxd_psave_on);


static int dwc3_cxd_remove(struct platform_device *pdev)
{
	struct cxd_dwc3 *data = platform_get_drvdata(pdev);

	prepare_stop(data);
	of_platform_depopulate(&pdev->dev);

	return 0;
}
static int dwc3_cxd_suspend(struct device *dev)
{
	struct cxd_dwc3 *dwc3_data = dev_get_drvdata(dev);

	dwc3_cxd_psave_on(dwc3_data);
	return 0;
}
static int dwc3_cxd_resume(struct device *dev)
{
	struct cxd_dwc3 *dwc3_data = dev_get_drvdata(dev);
	struct device_node *node = dev->of_node, *child;
	struct dwc3 *dwc;
	struct platform_device *child_pdev;
	union extcon_property_value polarity;
	uint32_t reg_val;

	// ANPD=0(normal)
	cxd_dwc3_writel(dwc3_data->glue_base, D_OFS_USB32_ANPD, D_ANPD_OFF);
	// SRAM operation
	reg_val = cxd_dwc3_readl(dwc3_data->glue_base, D_OFS_USB32_PHY_SRAM_CTRL);
	reg_val &= ~D_PHY0_SRAM_BYPASS;
	cxd_dwc3_writel(dwc3_data->glue_base, D_OFS_USB32_PHY_SRAM_CTRL, reg_val);

	child = of_get_child_by_name(node, "dwc3");
	if (!child) {
		s_print(S_DEBUG_ERROR, "Cannot find dwc3 child node\n");
		return -ENODEV;
	}

	child_pdev = of_find_device_by_node(child);
	if (!child_pdev) {
		s_print(S_DEBUG_ERROR, "cannot get child node data\n");
		return -ENODEV;
	}

	dwc = platform_get_drvdata(child_pdev);

	if ((dwc->current_dr_role == DWC3_GCTL_PRTCAP_DEVICE) ||
		(dwc->current_dr_role == DWC3_GCTL_PRTCAP_HOST) ||
		(dwc->current_dr_role == DWC3_GCTL_PRTCAP_EXT_HOST_DISCON)) {
		dwc3_cxd_psave_off(dwc->cxd_glue);
		extcon_get_property(dwc->edev, EXTCON_USB, EXTCON_PROP_USB_TYPEC_POLARITY, &polarity);
#ifndef CONFIG_ARCH_CXD90XXX_FPGA
		dwc3_tca_cc_control_event(dwc->cxd_glue, polarity.intval);
#endif
	}

	return 0;
}

static const struct dev_pm_ops cxd_dwc3_pm = {
	SET_SYSTEM_SLEEP_PM_OPS(dwc3_cxd_suspend, dwc3_cxd_resume)
};

static const struct of_device_id cxd_dwc3_match[] = {
	{ .compatible = "cxd,dwc3" },
	{ /* sentinel */ },
};

MODULE_DEVICE_TABLE(of, cxd_dwc3_match);

static struct platform_driver cxd_dwc3_driver = {
	.probe = dwc3_cxd_probe,
	.remove = dwc3_cxd_remove,
	.driver = {
		.name = "usb-cxd-dwc3",
		.of_match_table = cxd_dwc3_match,
		.pm = &cxd_dwc3_pm,
	},
};

module_platform_driver(cxd_dwc3_driver);

MODULE_LICENSE("GPL");
