/*
 * drivers/usb/f_usb/udif_usb_otg.c
 *
 * Copyright (C) 2011-2012 FUJITSU SEMICONDUCTOR LIMITED
 * Copyright 2018 Sony Imaging Products and Solutions Incorporated.
 *
 * ALL RIGHTS RESERVED, COPYRIGHT (C) SOCIONEXT INC. 2015
 * LICENSED MATERIAL - PROGRAM PROPERTY OF SOCIONEXT INC.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*--------------------------------------------------------------------------*
 * drivers/usb/scd/scd_device.c
 *
 * OTG and, Platform bus driver operations
 *
 * Copyright 2011 Sony Corporation
 *
 * This file is part of the HS-OTG Controller Driver SCD.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *---------------------------------------------------------------------------*/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/timer.h>
#include <linux/jiffies.h>
#include <linux/gpio/gpio.h>
#include <mach/irqs.h>
#include <mach/gic_export.h>

#include <linux/udif/module.h>
#include <linux/udif/device.h>
#include <linux/udif/types.h>
#include <linux/udif/irq.h>
#include <linux/udif/io.h>
#include <linux/udif/spinlock.h>
#include <linux/udif/list.h>
#include <linux/udif/tasklet.h>
#include <linux/udif/malloc.h>
#include <linux/udif/mutex.h>
#include <linux/udif/delay.h>
#include <linux/udif/proc.h>
#include <linux/usb/f_usb/usb_otg_dwc3_extcon.h>
#include <linux/usb/f_usb/usb_otg_dwc2_extcon.h>

#include "linux/usb/f_usb/usb_otg_notify.h"
#include "udif_usb_otg.h"
#include "udif_usb_otg_if.h"

//#define FUSB_FUNC_TRACE					/**<  */

#ifdef FUSB_FUNC_TRACE
#define fusb_func_trace(b) printk("CALL %s %s(%d)\n",(b),__FUNCTION__,__LINE__)
#define fusb_print_dbg(fmt, arg...) printk(KERN_DEBUG fmt " %s(%d)\n", ##arg, __FUNCTION__,__LINE__)
#define fusb_print_err(fmt, arg...) printk(KERN_ERR fmt " %s(%d)\n", ##arg, __FUNCTION__,__LINE__)
#else
#define fusb_func_trace(b)
#define fusb_print_dbg(fmt, arg...)
/*#define fusb_print_err(fmt, arg...)*/
#define fusb_print_err(fmt, arg...) printk(KERN_ERR fmt " %s(%d)\n", ##arg, __FUNCTION__,__LINE__)
#endif

#ifdef NDEBUG
#define fusb_assert_NULL(b)
#else /* #ifdef NDEBUG */
#define fusb_assert_NULL(b)								\
	if (!(b))									\
	{										\
		printk(KERN_ERR " %s(%d) 0x%x *** ASSERT \n",__FILE__,__LINE__,(b));	\
		BUG();									\
	}				 /**<  */
#endif


/**
 * @brief usb_str
 *
 *
 */
static char* usb_str = "fusb_otg";
UDIF_MODULE_PARAM(usb_str, charp, 0664U);

UDIF_IDS(fusb_notify_otg_ids) = {
	UDIF_ID(UDIF_ID_FUSB_OTG, 0xFFU)	/* 0b:OTG Register I/O, 1b:USB_VBUS, 2b:USB_ID */
};

static UDIF_DRIVER_OPS fusb_notify_otg_cb = {
	.init       = fusb_notify_otg_init,
	.exit       = fusb_notify_otg_exit,
	.probe      = fusb_notify_otg_probe,
	.remove     = fusb_notify_otg_remove,
	.suspend    = fusb_notify_otg_suspend,
	.resume     = fusb_notify_otg_resume,
	.shutdown   = fusb_notify_otg_shutdown,
};

UDIF_DEPS(fusb_deps) = {};
UDIF_MODULE(fusb_otg, "fusb_otg", "1.0", fusb_notify_otg_cb, fusb_notify_otg_ids, fusb_deps, NULL);

static struct fusb_notify_control *g_fusb_notify_control = NULL;

static UDIF_LIST_HEAD(fusb_anchor);

static UDIF_DECLARE_SPINLOCK(fusb_lock);

static int fusb_create_proc_flag = FUSB_OTG_FLAG_OFF;

static enum usb_otg_vbus_stat fusb_read_vbus0(void);
static enum usb_otg_vbus_stat fusb_read_vbus1(void);
static int fusb_read_proc(UDIF_PROC_READ *proc)
{
	int len = 0;
	struct fusb_notify_control *fusb;

	if (proc == NULL) {
		fusb_print_err("argument proc is set NULL");
		return UDIF_ERR_PAR;
	}

	len += udif_proc_setbuf(proc, len, "USB Controller Driver Ver : %d\n", FUSB_VERSION);

	len += udif_proc_setbuf(proc, len, "  internal latch:\n");
	fusb = g_fusb_notify_control;
	if (fusb) {
		if (FUSB_OTG_FLAG_ON != fusb->start_comp_flag) {
			len += udif_proc_setbuf(proc, len, "    not started\n");
		}
		len += udif_proc_setbuf(proc, len, "    usb_id[0]=%d\n", fusb->cid[USB_OTG_VBUS_PORT0]);
		len += udif_proc_setbuf(proc, len, "    usb_id[1]=%d\n", fusb->cid[USB_OTG_VBUS_PORT1]);
		len += udif_proc_setbuf(proc, len, "    vbus[USB_OTG_VBUS_PORT0]  =%d\n", fusb->vbus[USB_OTG_VBUS_PORT0]);
		len += udif_proc_setbuf(proc, len, "    vbus[USB_OTG_VBUS_PORT1]  =%d\n", fusb->vbus[USB_OTG_VBUS_PORT1]);
	}
	else {
		len += udif_proc_setbuf(proc, len, "    not started\n");
	}
	len += udif_proc_setbuf(proc, len, "  readouts:\n");
	len += udif_proc_setbuf(proc, len, "    vbus[USB_OTG_VBUS_PORT0]  =%d\n", fusb_read_vbus0());
	len += udif_proc_setbuf(proc, len, "    vbus[USB_OTG_VBUS_PORT1]  =%d\n", fusb_read_vbus1());

	udif_proc_setend(proc);
	return len;
}

/**
 * @brief fusb_proc
 *
 *
 */
static UDIF_PROC fusb_proc = {
	.name  = FUSB_PROC_NAME,
	.read  = fusb_read_proc,
};

static UDIF_ERR fusb_notify_otg_init(UDIF_VP data)
{
	struct fusb_notify_control *fusb;
	UDIF_ERR ret;
	UDIF_UINT size;

	fusb_func_trace("START");

	size = sizeof(struct fusb_notify_control);
	fusb = udif_kmalloc("fusb_notiry", size, 0);
	if (fusb == NULL) {
		fusb_print_err("udif_kmalloc is failed");
		return UDIF_ERR_NOMEM;
	}

	memset(fusb, 0, size);
	fusb->otg_regs      = NULL;
	udif_spin_lock_init(&fusb->lock);
	udif_tasklet_init(&fusb->event_tasklet, fusb_signal_event, (UDIF_ULONG)fusb);
	udif_list_head_init(&fusb->event_anchor);
	fusb->cid[USB_OTG_VBUS_PORT0] = FUSB_B_DEVICE;
	fusb->cid[USB_OTG_VBUS_PORT1] = FUSB_B_DEVICE;
	fusb->vbus[USB_OTG_VBUS_PORT0] = FUSB_VBUS_OFF;
	fusb->vbus[USB_OTG_VBUS_PORT1] = FUSB_VBUS_OFF;
	fusb->gadget_con    = FUSB_EVENT_DISCONNECT;
	fusb->init_comp_flag = FUSB_OTG_FLAG_ON;
	fusb->start_comp_flag = FUSB_OTG_FLAG_OFF;

	timer_setup(&fusb->con_timer, NULL, 0);
	fusb->con_timer.function    = fusb_con_timer_func;

	g_fusb_notify_control = fusb;

	if ((ret = udif_create_proc(&fusb_proc)) != UDIF_ERR_OK) {
		fusb_print_err("udif_create_proc is failed");
		return ret;
	}
	fusb_create_proc_flag = FUSB_OTG_FLAG_ON;

	fusb_func_trace("END");

	return UDIF_ERR_OK;
}

static UDIF_ERR fusb_notify_otg_exit(UDIF_VP data)
{
	struct fusb_notify_control *fusb;
	UDIF_ERR ret;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	if (fusb_create_proc_flag == FUSB_OTG_FLAG_ON) {
		if ((ret = udif_remove_proc(&fusb_proc)) != UDIF_ERR_OK) {
			fusb_print_err("udif_remove_proc is failed");
		} else {
			fusb_create_proc_flag = FUSB_OTG_FLAG_OFF;
		}
	}

	if (g_fusb_notify_control == NULL) {
		return UDIF_ERR_OK;
	}

	g_fusb_notify_control = NULL;

	udif_tasklet_kill(&fusb->event_tasklet);
	udif_kfree((UDIF_VP)fusb);

	fusb_func_trace("END");

	return UDIF_ERR_OK;
}

static enum usb_otg_vbus_stat fusb_read_vbus0(void)
{
	if (fusb_otg_dwc3_read_vbus() == true) {
		return USB_OTG_VBUS_STAT_VALID;
	} else {
		return USB_OTG_VBUS_STAT_OFF;
	}
}

static enum usb_otg_vbus_stat fusb_read_vbus1(void)
{
	if (fusb_otg_dwc2_read_vbus() == true) {
		return USB_OTG_VBUS_STAT_VALID;
	} else {
		return USB_OTG_VBUS_STAT_OFF;
	}
}

static UDIF_ERR fusb_notify_otg_probe(const UDIF_DEVICE * dev, UDIF_CH ch, UDIF_VP data)
{
	struct fusb_notify_control *fusb;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	switch(ch) {
	case FUSB_OTG_REG_CH:
		fusb->otg_regs = udif_devio_virt(dev, ch);
		break;
	case FUSB_USB_VBUS_CH:
		break;
	case FUSB_USB_ID_CH:
		break;
	default :
		fusb_print_err("Out of setting channel");
		return UDIF_ERR_PAR;
	}

	fusb_func_trace("END");

	return UDIF_ERR_OK;
}

static UDIF_ERR fusb_notify_otg_remove(const UDIF_DEVICE * dev, UDIF_CH ch, UDIF_VP data)
{
	struct fusb_notify_control *fusb;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	switch(ch) {
	case FUSB_OTG_REG_CH:
		fusb->otg_regs = NULL;
		break;
	case FUSB_USB_VBUS_CH:
		break;
	case FUSB_USB_ID_CH:
		break;
	default :
		fusb_print_err("Out of setting channel");
		return UDIF_ERR_PAR;
	}

	fusb_func_trace("END");

	return UDIF_ERR_OK;
}

static UDIF_ERR fusb_notify_otg_suspend(const UDIF_DEVICE * dev, UDIF_CH ch, UDIF_VP data)
{
	struct fusb_notify_control *fusb;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return UDIF_ERR_NOENT;
	}

	fusb->init_comp_flag = FUSB_OTG_FLAG_OFF;

	if (fusb->start_comp_flag == FUSB_OTG_FLAG_OFF) {
		return UDIF_ERR_OK;
	}

	switch(ch) {
	case FUSB_OTG_REG_CH:
		fusb_otg_dwc3_disable_irq();
		fusb_otg_dwc2_disable_irq();
		break;
	case FUSB_USB_VBUS_CH:
		break;
	case FUSB_USB_ID_CH:
		break;
	default :
		break;
	}

	fusb_func_trace("END");

	return UDIF_ERR_OK;
}

static UDIF_ERR fusb_notify_otg_resume(const UDIF_DEVICE * dev, UDIF_CH ch, UDIF_VP data)
{
	struct fusb_notify_control *fusb;
	enum usb_otg_vbus_stat vbus[USB_OTG_VBUS_PORT_MAX];

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return UDIF_ERR_NOENT;
	}

	if (fusb->start_comp_flag == FUSB_OTG_FLAG_OFF)
		return UDIF_ERR_OK;

	switch(ch) {
	case FUSB_OTG_REG_CH:
		fusb_otg_dwc3_enable_irq();
		fusb_otg_dwc2_enable_irq();
		vbus[USB_OTG_VBUS_PORT0] = fusb_read_vbus0();
		vbus[USB_OTG_VBUS_PORT1] = fusb_read_vbus1();
		fusb_notify_vbus(fusb, vbus[USB_OTG_VBUS_PORT0], USB_OTG_VBUS_PORT0, false);
		fusb_notify_vbus(fusb, vbus[USB_OTG_VBUS_PORT1], USB_OTG_VBUS_PORT1, false);

		break;
	case FUSB_USB_VBUS_CH:
		break;
	case FUSB_USB_ID_CH:
		break;
	default :
		break;
	}

	fusb_func_trace("END");

	return UDIF_ERR_OK;
}

static UDIF_ERR fusb_notify_otg_shutdown(const UDIF_DEVICE * dev, UDIF_CH ch, UDIF_VP data)
{
	struct fusb_notify_control *fusb;
	UDIF_ERR ret;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	if (fusb_create_proc_flag == FUSB_OTG_FLAG_ON) {
		if ((ret = udif_remove_proc(&fusb_proc)) != UDIF_ERR_OK) {
			fusb_print_err("udif_remove_proc is failed");
		} else {
			fusb_create_proc_flag = FUSB_OTG_FLAG_OFF;
		}
	}

	fusb->start_comp_flag = FUSB_OTG_FLAG_OFF;

	udif_tasklet_kill(&fusb->event_tasklet);
	udif_kfree((UDIF_VP)fusb);

	fusb_func_trace("END");

	return UDIF_ERR_OK;
}

static void fusb_signal_event(UDIF_ULONG data)
{
	struct fusb_notify_control *fusb;

	UDIF_LIST temp_anchor;
	struct fusb_otg_event_container *ec;
	struct fusb_otg_event_container *temp_ec;

	if (data == 0) {
		return;
	}

	fusb = (struct fusb_notify_control *)data;

	udif_list_head_init(&temp_anchor);
	fusb_lock_irq(&fusb->lock, fusb->flags);
	udif_list_for_each_entry_safe(ec, temp_ec, &fusb->event_anchor, event_list) {
		udif_list_del(&ec->event_list);
		udif_list_head_init(&ec->event_list);
		udif_list_add_tail(&ec->event_list, &temp_anchor);
	}
	fusb_unlock_irq(&fusb->lock, fusb->flags);

	udif_list_for_each_entry_safe(ec, temp_ec, &temp_anchor, event_list) {
		udif_list_del(&ec->event_list);

		if (fusb != NULL) {
			fusb_print_dbg("usb_otg_core_notify. event type = %d \n",ec->event.type);
			usb_otg_core_notify(fusb->notify, &ec->event);
		}
		udif_kfree(ec);
	}

	return;
}

static void fusb_signal_event_interrupt(struct fusb_notify_control *fusb, union usb_otg_event *event)
{
	struct fusb_otg_event_container *event_container;

	event_container = (struct fusb_otg_event_container *)udif_kmalloc("fusb_usb_event", sizeof(struct fusb_otg_event_container), 0);
	if (event_container == NULL) {
		fusb_print_err("udif_kmalloc is failed");
		return;
	}

	memcpy(&event_container->event, event, sizeof(union usb_otg_event));
	udif_list_head_init(&event_container->event_list);
	fusb_lock_irq(&fusb->lock, fusb->flags);
	udif_list_add_tail(&event_container->event_list, &fusb->event_anchor);
	fusb_unlock_irq(&fusb->lock, fusb->flags);

	udif_tasklet_schedule(&fusb->event_tasklet);

	return;
}

void fusb_interrupt_usb_vbus0(enum usb_otg_vbus_stat vbus0)
{
	struct fusb_notify_control *fusb;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	if(fusb != NULL){
		fusb_notify_vbus(fusb, vbus0, USB_OTG_VBUS_PORT0, false);
	}

	fusb_func_trace("END");

}

void fusb_interrupt_usb_vbus1(enum usb_otg_vbus_stat vbus1)
{
	struct fusb_notify_control *fusb;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	if(fusb != NULL){
		fusb_notify_vbus(fusb, vbus1, USB_OTG_VBUS_PORT1, false);
	}

	fusb_func_trace("END");

}

static void fusb_notify_cid(struct fusb_notify_control *fusb, int current_cid, enum usb_otg_port_type port, bool forced)
{
	union usb_otg_event otg_event;
	if ((fusb->cid[port] != current_cid) || forced) {
		otg_event.type              = USB_OTG_EVENT_TYPE_CID;
		otg_event.cid.value         = current_cid;
		otg_event.cid.port          = port;
		fusb_signal_event_interrupt(fusb, &otg_event);
		fusb->cid[port] = current_cid;
		fusb_print_dbg("current_cid = %d \n",current_cid);
	}
}

static void fusb_notify_vbus(struct fusb_notify_control *fusb, enum usb_otg_vbus_stat current_vbus, enum usb_otg_port_type port, bool forced)
{
	union usb_otg_event otg_event;
	if ((fusb->vbus[port] != current_vbus) || forced) {
		otg_event.type              = USB_OTG_EVENT_TYPE_VBUS;
		if (current_vbus == USB_OTG_VBUS_STAT_VALID) {
			otg_event.vbus.vbus_stat = USB_OTG_VBUS_STAT_VALID;
		} else {
			otg_event.vbus.vbus_stat = USB_OTG_VBUS_STAT_OFF;
		}
		otg_event.vbus.port         = port;
		fusb_signal_event_interrupt(fusb, &otg_event);
		fusb->vbus[port] = current_vbus;
		fusb_print_dbg("current_vbus = %d \n",current_vbus);
		if( fusb->vbus[port] == USB_OTG_VBUS_STAT_OFF ){
			fusb->gadget_con = FUSB_EVENT_DISCONNECT;
		}
	}
}

int fusb_try_start_notify(unsigned int port)
{
	struct fusb_notify_control *fusb;
	enum usb_otg_vbus_stat vbus[USB_OTG_VBUS_PORT_MAX];

	fusb = g_fusb_notify_control;
	fusb_assert_NULL(fusb != NULL);

	if (port == FUSB_DWC_PORT_TYPE_USB3) {
		fusb_otg_dwc3_enable_irq();
		vbus[USB_OTG_VBUS_PORT0]    = fusb_read_vbus0();
		fusb_notify_cid(fusb, FUSB_B_DEVICE, USB_OTG_VBUS_PORT0, true);
		fusb_notify_vbus(fusb, vbus[USB_OTG_VBUS_PORT0], USB_OTG_VBUS_PORT0, true);
        }
	else if (port == FUSB_DWC_PORT_TYPE_USB2) {
		fusb_otg_dwc2_enable_irq();
		vbus[USB_OTG_VBUS_PORT1]    = fusb_read_vbus1();
		fusb_notify_cid(fusb, FUSB_B_DEVICE, USB_OTG_VBUS_PORT1, true);
		fusb_notify_vbus(fusb, vbus[USB_OTG_VBUS_PORT1], USB_OTG_VBUS_PORT1, true);
	}
	else {
		fusb_print_err("port error\n");
		return -EINVAL;
	}

	fusb->start_comp_flag = FUSB_OTG_FLAG_ON;

/*	current_otg_mode = USB_OTG_CONTROL_IDLE;*/

	return 0;
}
#ifdef CONFIG_USB_FUSB_OTG_TEST
EXPORT_SYMBOL_GPL(fusb_try_start_notify);
#endif

int fusb_try_stop_notify(unsigned int port)
{
	struct fusb_notify_control *fusb;

	fusb = g_fusb_notify_control;
	fusb_assert_NULL(fusb != NULL);

	fusb->start_comp_flag = FUSB_OTG_FLAG_OFF;

	if (port == FUSB_DWC_PORT_TYPE_USB3) {
		fusb_otg_dwc3_disable_irq();
	}
	else if (port == FUSB_DWC_PORT_TYPE_USB2) {
		fusb_otg_dwc2_disable_irq();
	}
	else {
		fusb_print_err("port error\n");
	}

	udif_tasklet_kill(&fusb->event_tasklet);

/*	current_otg_mode = USB_OTG_CONTROL_STOP;*/

	return FUSB_OTG_OK;
}
#ifdef CONFIG_USB_FUSB_OTG_TEST
EXPORT_SYMBOL_GPL(fusb_try_stop_notify);
#endif

int usb_otg_control_register_core_notify(int (*notify) (void * p))
{
	struct fusb_notify_control *fusb;

	fusb_lock(&fusb_lock);

	fusb_func_trace("START");

	if (notify == NULL) {
		fusb_print_err("argument notify callback is set NULL");
		fusb_unlock(&fusb_lock);
		return -EINVAL;
	}

	fusb = g_fusb_notify_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		fusb_unlock(&fusb_lock);
		return -ENOENT;
	}

	fusb->notify = notify;

	fusb_unlock(&fusb_lock);

	fusb_func_trace("END");

	return 0;
}
EXPORT_SYMBOL(usb_otg_control_register_core_notify);         /**<  */

int usb_otg_control_unregister_core_notify(int (*notify) (void * p))
{
	struct fusb_notify_control *fusb;

	fusb_func_trace("START");

	if (notify == NULL) {
		fusb_print_err("argument notify callback is set NULL");
		return -EINVAL;
	}

	fusb = g_fusb_notify_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return -ENOENT;
	}

	fusb_lock(&fusb_lock);

	if (fusb->notify != notify) {
		fusb_print_err("fusb->notify not match");
		fusb_unlock(&fusb_lock);
		return -EINVAL;
	}
	fusb->notify = NULL;

	fusb_unlock(&fusb_lock);

	fusb_func_trace("END");

	return 0;
}
EXPORT_SYMBOL(usb_otg_control_unregister_core_notify);       /**<  */

enum usb_otg_vbus_stat fusb_otg_get_vbus(void)
{
	struct fusb_notify_control *fusb;
	fusb = g_fusb_notify_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return USB_OTG_VBUS_STAT_OFF;
	}

	if (fusb->vbus[USB_OTG_VBUS_PORT0] == FUSB_VBUS_ON) { // tentative.
		return USB_OTG_VBUS_STAT_VALID;
	}

	return USB_OTG_VBUS_STAT_OFF;
}
EXPORT_SYMBOL(fusb_otg_get_vbus);                     /**<  */

int fusb_otg_notify_vbuserror(void)
{
	struct fusb_notify_control *fusb;
	union usb_otg_event otg_event;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return -ENOENT;
	}

	otg_event.type = USB_OTG_EVENT_TYPE_VBUSERROR;
	fusb_signal_event_interrupt(fusb, &otg_event);

	fusb_func_trace("END");

	return FUSB_OTG_OK;
}
EXPORT_SYMBOL(fusb_otg_notify_vbuserror);             /**<  */

static void fusb_con_timer_func(struct timer_list *t)
{
	struct fusb_notify_control *fusb = from_timer(fusb, t, con_timer);
	union usb_otg_event otg_event;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return;
	}

	if ((fusb->gadget_con != FUSB_EVENT_CONNECT) && (fusb->con_timer_data == FUSB_EVENT_CONNECT)) {
		fusb->gadget_con = FUSB_EVENT_CONNECT;
		otg_event.type = USB_OTG_EVENT_TYPE_CON;
		otg_event.con.value = 1U;
		fusb_signal_event_interrupt(fusb, &otg_event);
	}

	fusb_func_trace("END");

	return;
}

int fusb_otg_notify_connect(unsigned int con)
{
	struct fusb_notify_control *fusb;
	union usb_otg_event otg_event;
	unsigned long expires;

	if ((con != FUSB_EVENT_DISCONNECT) && (con != FUSB_EVENT_CONNECT)) {
		fusb_print_err("Out of setting argument");
		return -EINVAL;
	}

	fusb = g_fusb_notify_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return -ENOENT;
	}

	fusb->con_timer_data = con;
	if (fusb->gadget_con != con) {
		if (con == FUSB_EVENT_CONNECT) {
			expires = jiffies + msecs_to_jiffies(FUSB_DEVICE_CONNETCT_TIMER);
			mod_timer(&fusb->con_timer, expires);
		} else {
			fusb->gadget_con = con;
			otg_event.type = USB_OTG_EVENT_TYPE_CON;
			otg_event.con.value = 0;
			fusb_signal_event_interrupt(fusb, &otg_event);
		}
	}

	return FUSB_OTG_OK;
}
EXPORT_SYMBOL(fusb_otg_notify_connect);               /**<  */

MODULE_LICENSE ("GPL"); 				/**<  */
