/*
 * drivers/usb/f_usb/usb_otg.c
 *
 * Copyright (C) 2011-2012 FUJITSU SEMICONDUCTOR LIMITED
 * Copyright 2018 Sony Imaging Products and Solutions Incorporated.
 *
 * ALL RIGHTS RESERVED, COPYRIGHT (C) SOCIONEXT INC. 2015
 * LICENSED MATERIAL - PROGRAM PROPERTY OF SOCIONEXT INC.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*--------------------------------------------------------------------------*
 * drivers/usb/scd/scd_device.c
 *
 * OTG and, Platform bus driver operations
 *
 * Copyright 2011 Sony Corporation
 *
 * This file is part of the HS-OTG Controller Driver SCD.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *---------------------------------------------------------------------------*/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/semaphore.h>
#include <linux/mutex.h>

#include <linux/io.h>
#include <asm/delay.h>

#include <linux/extcon.h>
#include <linux/usb/f_usb/usb_otg_dwc3_extcon.h>
#include <linux/usb/f_usb/usb_otg_dwc2_extcon.h>
#include <linux/usb/f_usb/usb_otg_control.h>
#include <linux/usb/ch9.h>
#include "usb_otg.h"
#include "udif_usb_otg_if.h"
#include "usb_otg_resource.h"

//#define FUSB_FUNC_TRACE					/**<  */

#ifdef FUSB_FUNC_TRACE
#define fusb_func_trace(b) printk("CALL %s %s(%d)\n",(b),__FUNCTION__,__LINE__)
#define fusb_print_dbg(fmt, arg...) printk(KERN_DEBUG fmt " %s(%d)\n", ##arg, __FUNCTION__,__LINE__)
#define fusb_print_err(fmt, arg...) printk(KERN_ERR fmt " %s(%d)\n", ##arg, __FUNCTION__,__LINE__)
#else
#define fusb_func_trace(b)
#define fusb_print_dbg(fmt, arg...)
/*#define fusb_print_err(fmt, arg...)*/
#define fusb_print_err(fmt, arg...) printk(KERN_ERR fmt " %s(%d)\n", ##arg, __FUNCTION__,__LINE__)
#endif

#ifdef NDEBUG
#define fusb_assert_NULL(b)
#else /* #ifdef NDEBUG */
#define fusb_assert_NULL(b)								\
	if (!(b))									\
	{										\
		printk(KERN_ERR " %s(%d) 0x%x *** ASSERT \n",__FILE__,__LINE__,(b));	\
		BUG();									\
	}				 /**<  */
#endif


/**
 * @brief usb_str
 *
 *
 */

static struct fusb_device_control *g_fusb_device_control = NULL;

static DEFINE_SPINLOCK(fusb_lock);

static struct usb_otg_control_ops otg_control_ops = {
	.get_port_info      = fusb_get_port_info,
	.select_port        = fusb_select_port,
	.start_control      = fusb_start_control,
	.stop_control       = fusb_stop_control,
	.start_gadget       = fusb_start_gadget,
	.stop_gadget        = fusb_stop_gadget,
	.enable_rchost      = fusb_enable_rchost,
	.disable_rchost     = fusb_disable_rchost,
	.stop_rchost        = fusb_stop_rchost,
	.start_host         = fusb_start_host,
	.stop_host          = fusb_stop_host,
	.start_rcgadget     = fusb_start_rcgadget,
	.stop_rcgadget      = fusb_stop_rcgadget,
	.get_mode           = fusb_get_mode,
	.request_session    = fusb_request_session,
	.set_speed          = fusb_set_speed,
	.get_speed          = fusb_get_speed,
	.set_test_mode      = fusb_set_test_mode,
	.get_test_mode      = fusb_get_test_mode,
	.set_tpc_setting    = fusb_set_tpc_setting,
	.set_phy            = fusb_set_phy,
};

/**
 * @brief fusb_otg registoration
 *
 *
 */


#define PLAT_FUSB_NAME    "extcon-fusb_otg"

#ifdef CONFIG_USB_FUSB_OTG_EXTCON
static int __init fusb_dev_init(void)
{
	return 0;
}
static void __exit fusb_dev_cleanup(void)
{
}
#else
static struct platform_device *pdev = NULL;
#if !defined(PLATFORM_DEVID_AUTO)
#define DEVID  (-1)
#else
#define DEVID  PLATFORM_DEVID_AUTO    /* -2 */
#endif

static int __init fusb_dev_init(void)
{
	int ret;

	pdev = platform_device_alloc(PLAT_FUSB_NAME, DEVID);
	if (!pdev) {
		printk(KERN_ERR "%s platform_device_alloc fail\n", __FUNCTION__);
		return -ENOMEM;
	}

	ret = platform_device_add(pdev);   /* platform_device_register */
	if (ret) {
		printk(KERN_ERR "%s platform_devce_add fail\n", __FUNCTION__);
	}

	return ret;
}
static void __exit fusb_dev_cleanup(void)
{
	if (pdev)
		platform_device_unregister(pdev);
	pdev = NULL;
}
#endif

static int fusb_otg_probe(struct platform_device *pdev);
static int fusb_otg_remove(struct platform_device *pdev);
static int fusb_otg_suspend(struct device *dev);
static int fusb_otg_resume(struct device *dev);
static const struct dev_pm_ops fusb_otg_dev_pm_ops = {
	SET_SYSTEM_SLEEP_PM_OPS(fusb_otg_suspend, fusb_otg_resume)
};
static const struct of_device_id usb_extcon_dt_match[] = {
	{ .compatible = "linux,"PLAT_FUSB_NAME, },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, usb_extcon_dt_match);
static struct platform_driver fusb_otg_driver = {
        .probe          = fusb_otg_probe,
        .remove         = fusb_otg_remove,
        .driver         = {
                .name   = PLAT_FUSB_NAME,
                .pm     = &fusb_otg_dev_pm_ops,
                .of_match_table = usb_extcon_dt_match,
        },
};

static int fusb_susres_debug = 0;

static int __init fusb_otg_init(void)
{
	struct fusb_device_control *fusb;

	fusb_func_trace("START");

	fusb = kmalloc(sizeof(struct fusb_device_control), GFP_KERNEL);
	if (fusb == NULL) {
		fusb_print_err("kmalloc is failed");
		return -ENOMEM;
	}

	if (fusb_dev_init()) {
		kfree(fusb);
		return -ENOMEM;
	}

	memset(fusb, 0, sizeof(struct fusb_device_control));
	fusb->otgc.ops      = &otg_control_ops;
	fusb->otg_mode[FUSB_DWC_PORT_TYPE_USB3]    = USB_OTG_CONTROL_STOP;
	fusb->otg_mode[FUSB_DWC_PORT_TYPE_USB2]    = USB_OTG_CONTROL_STOP;
	fusb->init_comp_flag = FUSB_OTG_FLAG_ON;
	fusb->host_resume_comp_flag = 0;
	fusb->port_type = FUSB_DWC_PORT_TYPE_USB3;
	fusb->cid[FUSB_DWC_PORT_TYPE_USB3] = FUSB_B_DEVICE;
	fusb->cid[FUSB_DWC_PORT_TYPE_USB2] = FUSB_B_DEVICE;

	g_fusb_device_control = fusb;

	fusb_func_trace("END");

	return platform_driver_register(&fusb_otg_driver);
}
subsys_initcall(fusb_otg_init);

static void __exit fusb_otg_exit(void)
{
	struct fusb_device_control *fusb;

	fusb_func_trace("START");

	if (g_fusb_device_control == NULL)
		return;

	platform_driver_unregister(&fusb_otg_driver);
	fusb_dev_cleanup();

	fusb = g_fusb_device_control;
	g_fusb_device_control = NULL;

	kfree(fusb);

	fusb_func_trace("END");
}
module_exit(fusb_otg_exit);

/* controller registers to check its status */
#define FUSB_XHCI_PORTSC_OFFSET    (0x400)
struct fusb_controller_port_t {
	uint32_t portsc;
	uint32_t portpmsc;
	uint32_t portli;
	uint32_t porthlpmc;
};

struct fusb_controller_status_t {
	int                           setf;
	/* host */
	uint32_t                      *usbsts;
	struct fusb_controller_port_t *portsc;
	int                           maxports;
	/* gadget */
	uint32_t                      *dctl;
	uint32_t                      *dsts;
};

static struct fusb_controller_status_t fusb_controller_status = { 0, };

static int fusb_otg_probe(struct platform_device *pdev)
{
	struct fusb_device_control *fusb;
	int ret;

	fusb_func_trace("START");
	fusb_controller_status.setf = 0;
	fusb = g_fusb_device_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return -ENODEV;
	}

	ret = fusb_otg_dwc3_extcon_setup();
	ret = fusb_otg_dwc2_extcon_setup();

	fusb_func_trace("END");

	return ret;
}

static int fusb_otg_remove(struct platform_device *pdev)
{
	struct fusb_device_control *fusb;

	fusb_func_trace("START");

	fusb = g_fusb_device_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return -ENODEV;
	}

	fusb_otg_dwc3_extcon_teardown();
	fusb_otg_dwc2_extcon_teardown();

	fusb_func_trace("END");

	return 0;
}

static int fusb_otg_suspend(struct device *dev)
{
	struct fusb_device_control *fusb;

	fusb_func_trace("START");

	fusb = g_fusb_device_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return -ENODEV;
	}

	fusb->init_comp_flag = FUSB_OTG_FLAG_OFF;

	if ((current_otg_mode(FUSB_DWC_PORT_TYPE_USB3) == USB_OTG_CONTROL_STOP) && (current_otg_mode(FUSB_DWC_PORT_TYPE_USB2) == USB_OTG_CONTROL_STOP)) {
		return 0;
	}

	/* that is all here */
	fusb_func_trace("END");

	return 0;
}

static int fusb_otg_resume(struct device *dev)
{
	struct fusb_device_control *fusb;

	fusb_func_trace("START");

	fusb = g_fusb_device_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return -ENODEV;
	}

	if (fusb_susres_debug) {
		struct fusb_controller_status_t *csts = &fusb_controller_status;
		printk(KERN_INFO "%s maxports=%d, usbsts=%p, portsc=%p\n", __FUNCTION__, csts->maxports, csts->usbsts, csts->portsc);
		printk(KERN_INFO "%s dctl=%p, dsts=%p, setf=%d\n", __FUNCTION__, csts->dctl, csts->dsts, csts->setf);
	}

	if ((current_otg_mode(FUSB_DWC_PORT_TYPE_USB3) == USB_OTG_CONTROL_STOP) && (current_otg_mode(FUSB_DWC_PORT_TYPE_USB2) == USB_OTG_CONTROL_STOP)) {
		return 0;
	}

	/* that is all here */
	fusb_func_trace("END");

	return 0;
}

/* apis */

static int fusb_get_port_info(struct usb_otg_control *otg_control,
			      struct usb_otg_control_port_info *info)
{
	fusb_func_trace("START");

	if ((otg_control == NULL)||(info == NULL)) {
		fusb_print_err("argument is set NULL");
		return -EINVAL;
	}

	info->nr            = 0;
	info->current_port  = 0;

	fusb_func_trace("END");

	return FUSB_OTG_OK;
}

static int fusb_select_port(struct usb_otg_control *otg_control, unsigned int pn)
{
	struct fusb_device_control *fusb;

	fusb_func_trace("START");

	fusb = otgc_to_fusb(otg_control);
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return -ENOENT;
	}

	if (pn == 0) {
		fusb->port_type = FUSB_DWC_PORT_TYPE_USB3;
	} else if (pn == 1) {
		fusb->port_type = FUSB_DWC_PORT_TYPE_USB2;
	} else {
		fusb_print_err("error port no");
	}

	fusb_func_trace("END");

	return FUSB_OTG_OK;
}

static int fusb_start_control(struct usb_otg_control *otg_control, unsigned int port)
{
	int status;
	struct fusb_device_control *fusb;

	fusb_func_trace("START");

	if (otg_control == NULL) {
		fusb_print_err("argument is set NULL");
		return -EINVAL;
	}

	fusb = otgc_to_fusb(otg_control);
	if (current_otg_mode(port) != USB_OTG_CONTROL_STOP) {
		fusb_print_err("OTG MODE is not STOP MODE");
		return -EBUSY;
	}

	status = fusb_try_start_control(fusb, port);

	fusb_func_trace("END");

	return status ;
}

static int fusb_stop_control(struct usb_otg_control *otg_control, unsigned int port)
{
	int status;
	struct fusb_device_control *fusb;

	fusb_func_trace("START");

	if (otg_control == NULL) {
		fusb_print_err("argument is set NULL");
		return -EINVAL;
	}

	fusb = otgc_to_fusb(otg_control);
	if (current_otg_mode(port) != USB_OTG_CONTROL_IDLE) {
		fusb_print_err("OTG MODE is not IDLE MODE");
		return -EBUSY;
	}

	status = fusb_try_stop_control(fusb, port);

	fusb_func_trace("END");

	return status;
}

static int fusb_start_gadget(struct usb_otg_control *otg_control, unsigned int port)
{
	int status;
	struct fusb_device_control *fusb;

	fusb_func_trace("START");

	if (otg_control == NULL) {
		fusb_print_err("argument is set NULL");
		return -EINVAL;
	}

	fusb = otgc_to_fusb(otg_control);
	if (current_otg_mode(port) != USB_OTG_CONTROL_IDLE) {
		fusb_print_err("OTG MODE is not IDLE MODE");
		return -EBUSY;
	}

	status = fusb_try_start_gadget(fusb, port);

	fusb_func_trace("END");

	return status;
}

static int fusb_stop_gadget(struct usb_otg_control *otg_control, unsigned int port)
{
	int status;
	struct fusb_device_control *fusb;

	fusb_func_trace("START");

	if (otg_control == NULL) {
		fusb_print_err("argument is set NULL");
		return -EINVAL;
	}

	fusb = otgc_to_fusb(otg_control);
	if (current_otg_mode(port) != USB_OTG_CONTROL_GADGET) {
		fusb_print_err("OTG MODE is not GADGET MODE");
		return -EBUSY;
	}

	status = fusb_try_stop_gadget(fusb, port);

	fusb_func_trace("END");

	return status;
}

static int fusb_enable_rchost(struct usb_otg_control *otg_control, unsigned int port)
{
	fusb_func_trace(" ");

	return FUSB_OTG_OK;
}

static int fusb_disable_rchost(struct usb_otg_control *otg_control, unsigned int port)
{
	fusb_func_trace(" ");

	return FUSB_OTG_OK;
}

static int fusb_stop_rchost(struct usb_otg_control *otg_control, unsigned int port)
{
	fusb_func_trace(" ");

	return FUSB_OTG_OK;
}

static int fusb_start_host(struct usb_otg_control *otg_control, unsigned int port)
{
	struct fusb_device_control *fusb;

	fusb_func_trace("START");

	if (otg_control == NULL) {
		fusb_print_err("argument is set NULL");
		return -EINVAL;
	}

	fusb = otgc_to_fusb(otg_control);
	if (current_otg_mode(port) != USB_OTG_CONTROL_IDLE) {
		fusb_print_err("OTG MODE is not IDLE MODE");
		return -EBUSY;
	}

	current_otg_mode(port) = USB_OTG_CONTROL_HOST;

	if (fusb->cid[port] == FUSB_A_DEVICE) {
		fusb_try_start_host(fusb, port);
	}

	fusb_func_trace("END");

	return FUSB_OTG_OK;
}

static int fusb_stop_host(struct usb_otg_control *otg_control, unsigned int port)
{
	int status;
	struct fusb_device_control *fusb;

	fusb_func_trace("START");

	if (otg_control == NULL) {
		fusb_print_err("argument is set NULL");
		return -EINVAL;
	}

	fusb = otgc_to_fusb(otg_control);
	if (current_otg_mode(port) != USB_OTG_CONTROL_HOST) {
		fusb_print_err("OTG MODE is not HOST MODE");
		return -EBUSY;
	}

	status = fusb_try_stop_host(fusb, port);

	fusb_func_trace("END");

	return status;
}

static int fusb_start_rcgadget(struct usb_otg_control *otg_control, unsigned int port)
{
	fusb_func_trace(" ");

	return FUSB_OTG_OK;
}

static int fusb_stop_rcgadget(struct usb_otg_control *otg_control, unsigned int port)
{
	fusb_func_trace(" ");

	return FUSB_OTG_OK;
}

static int fusb_get_mode(struct usb_otg_control *otg_control, unsigned int port)
{
	struct fusb_device_control *fusb;

	fusb_func_trace("START");

	if (otg_control == NULL) {
		fusb_print_err("argument is set NULL");
		return -EINVAL;
	}

	fusb = otgc_to_fusb(otg_control);

	fusb_func_trace("END");

	return current_otg_mode(port);
}

static int fusb_request_session(struct usb_otg_control *otg_control, unsigned int port)
{
	fusb_func_trace(" ");

	return FUSB_OTG_OK;
}

#define PATH_DWC3_SPEED_PARAM "/sys/module/dwc3/parameters/dwc3_gadget_speed"
static int fusb_set_speed(struct usb_otg_control *otg_control, unsigned int port, unsigned int speed)
{
	struct file *fp;
	ssize_t result;
	char sp;

	fusb_func_trace("START");

	if (otg_control == NULL) {
		fusb_print_err("argument is set NULL");
		return -EINVAL;
	}

	if (speed >= USB_OTG_SPEED_UNKNOWN) {
		fusb_print_err("Out of setting argument");
		return -EINVAL;
	}

	switch (speed) {
		case USB_OTG_SPEED_SS:
			sp = '0' + USB_SPEED_SUPER;
			break;
		case USB_OTG_SPEED_HS:
			sp = '0' + USB_SPEED_HIGH;
			break;
		case USB_OTG_SPEED_FS:
			sp = '0' + USB_SPEED_FULL;
			break;
		case USB_OTG_SPEED_SSP:
		case USB_OTG_SPEED_NATIVE:
		default:
			sp = '0' + USB_SPEED_SUPER_PLUS;
	}

	if (port == FUSB_DWC_PORT_TYPE_USB3) {
		fp = filp_open(PATH_DWC3_SPEED_PARAM, O_RDWR | O_SYNC, 644);
		if (IS_ERR(fp)) {
			fusb_print_err("sys open error");
		}
	else {
		fusb_print_dbg("speed:%d->%c", speed, sp);
		result = kernel_write( fp, &sp, sizeof(sp), 0);
		if (result < sizeof(sp)) {
				fusb_print_err("sys write error");
			}
			filp_close(fp, NULL);
		}
	}

	fusb_func_trace("END");

	return FUSB_OTG_OK;
}

static enum usb_otg_control_speed fusb_get_speed(struct usb_otg_control *otg_control, unsigned int port)
{
	enum usb_otg_control_speed speed;
	struct file *fp;
	ssize_t result;
	char sp;

	fusb_func_trace("START");

	if (otg_control == NULL) {
		fusb_print_err("argument is set NULL");
		return -EINVAL;
	}

	speed = USB_OTG_SPEED_HS;
	if (port == FUSB_DWC_PORT_TYPE_USB3) {
		fp = filp_open(PATH_DWC3_SPEED_PARAM, O_RDWR | O_SYNC, 644);
		if (IS_ERR(fp)) {
			fusb_print_err("sys open error");
		}
		else {
			result = kernel_read(fp, &sp, sizeof(sp), 0);
			if (result < sizeof(sp)) {
				fusb_print_err("sys read error");
			}
			else {
				switch (sp - '0') {
					case USB_SPEED_FULL:
						speed = USB_OTG_SPEED_FS;
						break;
					case USB_SPEED_SUPER:
						speed = USB_OTG_SPEED_SS;
						break;
					case USB_SPEED_SUPER_PLUS:
						speed = USB_OTG_SPEED_SSP;
						break;
					case USB_SPEED_HIGH:
					default:
						speed = USB_OTG_SPEED_HS;
					}
			}
			filp_close(fp, NULL);
		}
	}

	fusb_print_dbg("speed:%d", speed);

	fusb_func_trace("END");

	return speed;
}

static int fusb_set_test_mode(struct usb_otg_control *otg_control,
			      enum usb_otg_control_test_mode test_mode)
{
	fusb_func_trace(" ");

	return FUSB_OTG_OK;
}

static enum usb_otg_control_test_mode fusb_get_test_mode(struct usb_otg_control *otg_control)
{
	fusb_func_trace(" ");

	return USB_OTG_TEST_MODE_NORMAL;
}

static int fusb_set_tpc_setting(struct usb_otg_control *otg_control, int orientation, int ss_operation)
{
	bool prm_orientation = false;
	bool prm_ss_operation = false;
	struct fusb_device_control *fusb;

	fusb_func_trace(" ");

	fusb = g_fusb_device_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return -ENOENT;
	}

	fusb->port_type = FUSB_DWC_PORT_TYPE_USB3;
	if (orientation == USB_OTG_TYPEC_ORIENTATION_NORMAL) {
		prm_orientation = false;
	} else if (orientation == USB_OTG_TYPEC_ORIENTATION_FLIPPED) {
		prm_orientation = true;
	} else {
		fusb_print_err("param orientation=%d", orientation);
		return -EINVAL;
	}

	if (ss_operation == USB_OTG_USB_SS_OPERATION_ENABLE) {
		prm_ss_operation = true;
	} else if (ss_operation == USB_OTG_USB_SS_OPERATION_DISABLE) {
		prm_ss_operation = false;
	} else {
		fusb_print_err("param ss_operation=%d", ss_operation);
		return -EINVAL;
	}

	fusb_otg_set_dwc3_tpc_setting(prm_orientation, prm_ss_operation);
	return 0;
}

cxd_phy_param_t backup_phy_param;
EXPORT_SYMBOL(backup_phy_param);

static int fusb_set_phy(unsigned int port, const void *data)
{

	fusb_func_trace("");

	memcpy(&backup_phy_param, data, sizeof(cxd_phy_param_t));
	return 0;
}

static int fusb_try_start_control(struct fusb_device_control *fusb, unsigned int port)
{
	fusb_assert_NULL(fusb != NULL);

	fusb_try_start_notify(port);

	current_otg_mode(port) = USB_OTG_CONTROL_IDLE;

	return 0;
}

static int fusb_try_stop_control(struct fusb_device_control *fusb, unsigned int port)
{
	fusb_assert_NULL(fusb != NULL);

	fusb_try_stop_notify(port);

	current_otg_mode(port) = USB_OTG_CONTROL_STOP;

	return FUSB_OTG_OK;
}

static int fusb_try_start_gadget(struct fusb_device_control *fusb, unsigned int port)
{

	if ((fusb->cid[port] != FUSB_B_DEVICE) || (fusb->vbus[port] != FUSB_VBUS_ON)) {
		fusb_print_err("Not Condition Start Gadget cid=%d, vbus=%d", fusb->cid[port], fusb->vbus[port]);
		return -EBUSY;
	}

	if (port == FUSB_DWC_PORT_TYPE_USB3) {
		fusb_otg_set_dwc3_drd_state(DWC3_DRD_GADGET);
	} else if (port == FUSB_DWC_PORT_TYPE_USB2) {
		fusb_otg_set_dwc2_drd_state(DWC2_DRD_GADGET);
	} else {
		fusb_print_err("fusb_otg_get_drd_state port error");
	}

	printk(KERN_INFO "%s gadget started\n", __FUNCTION__);

	current_otg_mode(port) = USB_OTG_CONTROL_GADGET;

	return FUSB_OTG_OK;
}

static int fusb_try_stop_gadget(struct fusb_device_control *fusb, unsigned int port)
{
	if (port == FUSB_DWC_PORT_TYPE_USB3) {
		fusb_otg_set_dwc3_drd_state(DWC3_DRD_IDLE);
	}
	else if (port == FUSB_DWC_PORT_TYPE_USB2) {
		fusb_otg_set_dwc2_drd_state(DWC2_DRD_IDLE);
	}
	current_otg_mode(port) = USB_OTG_CONTROL_IDLE;
	printk(KERN_INFO "%s gadget stopped\n", __FUNCTION__);
	return FUSB_OTG_OK;
}

#ifdef CONFIG_ARCH_CXD900XX_FPGA
#define INTERVAL_WAIT_FOR_START (300)
#define MAX_LOOP_WAIT_FOR_START (3000000/INTERVAL_WAIT_FOR_START)    /* 3s */
#else
#define INTERVAL_WAIT_FOR_START (10)
#define MAX_LOOP_WAIT_FOR_START (1000000/INTERVAL_WAIT_FOR_START)    /* 1s */
#endif
static int fusb_try_start_host(struct fusb_device_control *fusb, unsigned int port)
{
	if (port == FUSB_DWC_PORT_TYPE_USB3) {
		fusb_otg_set_dwc3_drd_state(DWC3_DRD_HOST);
	}
	else if (port == FUSB_DWC_PORT_TYPE_USB2){
		fusb_otg_set_dwc2_drd_state(DWC2_DRD_HOST);
	} else {
		fusb_print_err("Host's port error");
	}

	printk(KERN_INFO "%s host started", __FUNCTION__);

	return FUSB_OTG_OK;
}

static int fusb_try_stop_host(struct fusb_device_control *fusb, unsigned int port)
{
	current_otg_mode(port) = USB_OTG_CONTROL_IDLE;

	if (port == FUSB_DWC_PORT_TYPE_USB3) {
		fusb_otg_set_dwc3_drd_state(DWC3_DRD_IDLE);
	} else if (port == FUSB_DWC_PORT_TYPE_USB2) {
		fusb_otg_set_dwc2_drd_state(DWC2_DRD_IDLE);
	} else {
		fusb_print_err("Host's port error");
	}

	printk(KERN_INFO "%s gadget started\n", __FUNCTION__);

	return FUSB_OTG_OK;
}

int usb_otg_control_register_core(struct usb_otg_core* otg_core)
{
	int status;
	struct fusb_device_control *fusb;

	fusb_lock(&fusb_lock);

	fusb_func_trace("START");

	if (otg_core == NULL) {
		fusb_print_err("argument otg_core is set NULL");
		fusb_unlock(&fusb_lock);
		return -EINVAL;
	}

	fusb = g_fusb_device_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		fusb_unlock(&fusb_lock);
		return -ENOENT;
	}

	fusb->otgc.otg_core = otg_core;

	status = usb_otg_core_bind(otg_core, &fusb->otgc);
	if (status < 0) {
		fusb_print_dbg("usb_otge_core_bind is failed status=%d \n", status);
		fusb->otgc.otg_core = NULL;
	}

	fusb_unlock(&fusb_lock);

	fusb_func_trace("END");

	return status;
}
EXPORT_SYMBOL(usb_otg_control_register_core);         /**<  */

int usb_otg_control_unregister_core(struct usb_otg_core *otg_core)
{
	int status;
	struct fusb_device_control *fusb;

	fusb_func_trace("START");

	if (otg_core == NULL) {
		fusb_print_err("argument otg_core is set NULL");
		return -EINVAL;
	}

	fusb = g_fusb_device_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return -ENOENT;
	}

	fusb_lock(&fusb_lock);

	if (fusb->otgc.otg_core != otg_core) {
		fusb_print_err("fusb->otgc.otg_core NULL pointer");
		fusb_unlock(&fusb_lock);
		return -EINVAL;
	}
	fusb->otgc.otg_core = NULL;
	status = usb_otg_core_unbind(otg_core, &fusb->otgc);

	fusb_unlock(&fusb_lock);

	fusb_func_trace("END");

	return status;
}
EXPORT_SYMBOL(usb_otg_control_unregister_core);       /**<  */

void *fusb_otg_get_base_addr(unsigned int flag)
{
	struct fusb_device_control *fusb;
	fusb = g_fusb_device_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return NULL;
	}

	if (flag == FUSB_ADDR_TYPE_HOST) {
		return (void *)fusb->host_regs;
	} else if (flag == FUSB_ADDR_TYPE_GADGET) {
		return (void *)fusb->gadget_regs;
	} else if (flag == FUSB_ADDR_TYPE_GLOBAL) {
		return (void *)fusb->global_regs;
	}

	fusb_print_err("Out of setting argument");
	return NULL;
}
EXPORT_SYMBOL(fusb_otg_get_base_addr);                /**<  */

void fusb_otg_dwc3_bind(struct usb_otg_dwc3_ops *ops)
{
	struct fusb_device_control *fusb;

	fusb_func_trace("START");

	fusb = g_fusb_device_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return;
	}

	fusb->dwc3_ops = ops;
	fusb_func_trace("END");

	return;
}
EXPORT_SYMBOL(fusb_otg_dwc3_bind);                  /**<  */

void fusb_otg_resume_initial_set(void)
{
	struct fusb_device_control *fusb;

	fusb_func_trace("START");

	fusb = g_fusb_device_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return;
	}

	if (fusb->init_comp_flag == FUSB_OTG_FLAG_OFF) {
		fusb->init_comp_flag = FUSB_OTG_FLAG_ON;
	}
	fusb_func_trace("END");
	return;
}
EXPORT_SYMBOL(fusb_otg_resume_initial_set);		/**<  */


#ifdef CONFIG_USB_FUSB_DWC3_REGS
int fusb_setup_controller_regs(struct resource *p)
{
	if (res_cregs.start || res_cregs.end)
		printk(KERN_WARNING "%s controller_regs is overwritten now(%llx->%llx, %llx->%llx)\n",
		       __FUNCTION__, res_cregs.start, p->start, res_cregs.end, p->end);
	res_cregs = *p;
	return 0;
}
void fusb_setup_controller_name(struct device *dev)
{
	devname = dev_name(dev);
}
#endif

int usb_otg_set_vbus(int vbus)
{
	struct fusb_device_control *fusb = NULL;
	int current_vbus;

	fusb_func_trace("START");

	fusb = g_fusb_device_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return -ENOENT;
	}

	if (vbus != USB_OTG_VBUS_STAT_OFF && vbus != USB_OTG_VBUS_STAT_VALID) {
		fusb_print_err("Out of param, vbus=%d", vbus);
		return -EINVAL;
    }

	if (USB_OTG_VBUS_STAT_VALID == vbus)
		current_vbus = FUSB_VBUS_ON;
	else
		current_vbus = FUSB_VBUS_OFF;

	/* Current vbus save. */
	fusb->vbus[fusb->port_type] = current_vbus;

	fusb_func_trace("END");

	return FUSB_OTG_OK;
}
#ifdef CONFIG_USB_FUSB_OTG_TEST
EXPORT_SYMBOL_GPL(usb_otg_set_vbus);
#endif

int usb_otg_set_cid(int cid)
{
	struct fusb_device_control *fusb;

	fusb_func_trace("START");

	if ((cid != USB_OTG_CID_STAT_A) && (cid != USB_OTG_CID_STAT_B)) {
		fusb_print_err("Out of param, cid=%d", cid);
		return -EINVAL;
	}

	fusb = g_fusb_device_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return -ENOENT;
	}

	fusb_print_dbg("cid:%d, mode:%x", cid, current_otg_mode(fusb->port_type));
	/* Current cid save. */
	if (cid == USB_OTG_CID_STAT_A) {
		fusb->cid[fusb->port_type] = FUSB_A_DEVICE;
		if (current_otg_mode(fusb->port_type) == USB_OTG_CONTROL_HOST) {
			fusb_try_start_host(fusb, fusb->port_type);
		}
	}
	else
		fusb->cid[fusb->port_type] = FUSB_B_DEVICE;

	fusb_func_trace("END");

	return FUSB_OTG_OK;
}
#ifdef CONFIG_USB_FUSB_OTG_TEST
EXPORT_SYMBOL_GPL(usb_otg_set_cid);
#endif

MODULE_LICENSE ("GPL"); 				/**<  */
