/*
 * drivers/usb/f_usb/usb_otg_dwc2_extcon.c
 *
 * Copyright 2022 Sony Corporation
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/slab.h>
#include <linux/module.h>
#include <linux/init.h>
#include <asm/io.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/extcon.h>
#include <linux/extcon-provider.h>
#include <linux/usb/f_usb/usb_otg_dwc2_extcon.h>
#include "../dwc2/dwc2-cxd.h"

#define COMPAT "cxd,dwc2_extcon"
#define NAME_DRV "dwc2_extcon"

#define INTNAME_VBUS "dwc2_vbus"
#define G_EDEV	g_dwc2

#define D_VBUS_UNLIMIT				0x0
#define D_EXT_MON					0x10
#define		D_EXT_MON_ID		BIT(1)
#define		D_EXT_MON_VBUS		BIT(0)
#define D_INTINV					0x20
#define 	D_INTINV_ID			BIT(1)
#define 	D_INTINV_VBUS		BIT(0)
#define D_INT_MSK					0x30
#define 	D_INT_MSK_ID		BIT(1)
#define 	D_INT_MSK_VBUS		BIT(0)
#define 	D_INT_MSK_BOTH_EN	0
#define D_INT_CLR					0x40
#define 	D_INT_CLR_ID		BIT(1)
#define 	D_INT_CLR_VBUS		BIT(0)

struct extcon_dev *G_EDEV;
unsigned int g_dwc2_irq;
void __iomem *g_extcon1_reg;

struct pdata {
	struct device *dev;
	void __iomem *reg;
	spinlock_t lock;
	struct extcon_dev *edev;
};

static struct pdata *m_info = NULL;

static const unsigned int usb_otg_extcon_cable[] = {
	EXTCON_USB,
	EXTCON_USB_HOST,
	EXTCON_NONE,
};

static const struct of_device_id dev_match[] = {
	{ .compatible = COMPAT},
	{ /* end of list */ },
};

static ssize_t drd_state_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct pdata *priv = dev_get_drvdata(dev);
	struct usb_en{
		uint8_t dev:1;
		uint8_t host:1;
		uint8_t reserved:6;
	} val = {0};

	if (extcon_get_state(priv->edev, EXTCON_USB) > 0) {
		val.dev = 1;
	}
	if (extcon_get_state(priv->edev, EXTCON_USB_HOST) > 0) {
		val.host = 1;
	}

	return snprintf(buf, 25, "device=%s, host=%s\n", val.dev ? "ON" : "off", val.host ? "ON" : "off");
}

static ssize_t drd_switch_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	unsigned long val;

	if(kstrtol(buf, 10, &val) == -EINVAL)
		pr_info("\nkstrtol err\n");

	if (val == 0) {
		fusb_otg_set_dwc2_drd_state(DWC2_DRD_HOST);
	} else if (val == 1) {
		fusb_otg_set_dwc2_drd_state(DWC2_DRD_GADGET);
	} else {
		fusb_otg_set_dwc2_drd_state(DWC2_DRD_IDLE);
	}
	return count;
}

static DEVICE_ATTR_WO(drd_switch);
static DEVICE_ATTR_RO(drd_state);
static struct attribute* dwc2_attrs[] = {
	&dev_attr_drd_switch.attr,
	&dev_attr_drd_state.attr,
	NULL
};

static const struct attribute_group attr_grp = {
	.attrs = dwc2_attrs,
};


/* DRD */
int fusb_otg_set_dwc2_drd_state(enum DWC2_DRD state)
{
	enum DWC2_DRD prev_state;

	if (NULL == m_info)
		return -EAGAIN;

	fusb_otg_get_dwc2_drd_state(&prev_state);
	if (state == DWC2_DRD_HOST) {
		if (prev_state != DWC2_DRD_HOST) {
			dwc2_cxd_start_drd();
			extcon_set_state_sync(m_info->edev, EXTCON_USB, false);
			extcon_set_state_sync(m_info->edev, EXTCON_USB_HOST, false);
			extcon_set_state_sync(m_info->edev, EXTCON_USB_HOST, true);
			dwc2_cxd_wait_drd_done();
		}
	} else if (state == DWC2_DRD_GADGET) {
		if (prev_state != DWC2_DRD_GADGET) {
			dwc2_cxd_start_drd();
			extcon_set_state_sync(m_info->edev, EXTCON_USB_HOST, false);
			extcon_set_state_sync(m_info->edev, EXTCON_USB, false);
			extcon_set_state_sync(m_info->edev, EXTCON_USB, true);
			dwc2_cxd_wait_drd_done();
		}
	} else {
		if(prev_state != DWC2_DRD_IDLE) {
			dwc2_cxd_start_drd();
			extcon_set_state_sync(m_info->edev, EXTCON_USB, false);
			extcon_set_state_sync(m_info->edev, EXTCON_USB_HOST, false);
		}
	}

	return 0;
}
EXPORT_SYMBOL(fusb_otg_set_dwc2_drd_state);

int fusb_otg_get_dwc2_drd_state(enum DWC2_DRD *state)
{
	if (NULL == m_info)
		return -EAGAIN;

	if(extcon_get_state(m_info->edev, EXTCON_USB_HOST)) {
		*state = DWC2_DRD_HOST;
	} else if(extcon_get_state(m_info->edev, EXTCON_USB)) {
		*state = DWC2_DRD_GADGET;
	} else {
		*state = DWC2_DRD_IDLE;
	}

	return 0;
}

bool fusb_otg_dwc2_read_vbus(void)
{
	uint32_t val;
	val = readl(g_extcon1_reg + D_EXT_MON);
	return (bool)(val & D_EXT_MON_VBUS);
}

void fusb_otg_dwc2_enable_irq(void)
{
	enable_irq(g_dwc2_irq);
}

void fusb_otg_dwc2_disable_irq(void)
{
	disable_irq(g_dwc2_irq);
}

static irqreturn_t dwc2_vbus_interrupt(int irq, void *data)
{
	struct pdata *info = data;
	uint32_t val;

	writel_relaxed(D_INT_CLR_VBUS, info->reg + D_INT_CLR);
	val = fusb_otg_dwc2_read_vbus();
	if (val == true) {
		fusb_interrupt_usb_vbus1(USB_OTG_VBUS_STAT_VALID);
	} else {
		fusb_interrupt_usb_vbus1(USB_OTG_VBUS_STAT_OFF);
	}
	return IRQ_HANDLED;
}

static int extcon_probe(struct platform_device *pdev_probe)
{
	int ret;
	struct pdata *pd;
	struct platform_device *pdev;
	struct device *dev;
	uint32_t val;
	struct resource *res;

	ret = 0;

	pd = kzalloc(sizeof(*pd),GFP_KERNEL);
	pdev = pdev_probe;
	platform_set_drvdata(pdev, pd);
	dev = &pdev->dev;
	pd->dev = dev;
	m_info = pd;

	ret = sysfs_create_group(&pdev->dev.kobj, &attr_grp);
	G_EDEV = pd->edev = devm_extcon_dev_allocate(dev, usb_otg_extcon_cable);
	if (IS_ERR(pd->edev)) {
		dev_err(dev, "failed to allocate memory for extcon\n");
		goto fail2;
	}
	ret = devm_extcon_dev_register(dev, pd->edev);
	if (ret) {
		dev_err(pd->dev, "failed to register extcon device\n");
		goto fail2;
	}
	
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "reg-glue");
	pd->reg = devm_ioremap_resource(dev, res);
	if (IS_ERR(pd->reg))
		return PTR_ERR(pd->reg);

	g_extcon1_reg = pd->reg;
	res = platform_get_resource_byname(pdev, IORESOURCE_IRQ, "vbus");
	g_dwc2_irq = res->start;
	if (devm_request_threaded_irq(pd->dev, res->start, &dwc2_vbus_interrupt,
		NULL, IRQF_TRIGGER_RISING, INTNAME_VBUS, pd))
		return -ENOMEM;
	fusb_otg_dwc2_disable_irq();

	/* enable interrupt */
	val = D_INT_MSK_BOTH_EN;
	writel_relaxed(val, pd->reg + D_INT_MSK);

	return 0;

fail2:
	platform_device_unregister(pdev);
	kfree(pd);
	return -1;
}

static int extcon_remove(struct platform_device *pdev)
{
	return 0;
}

static struct platform_driver driver_info = {
	.driver = {
		.name = NAME_DRV,
		.of_match_table = of_match_ptr(dev_match),
	},
	.probe = extcon_probe,
	.remove = extcon_remove,
};

int32_t fusb_otg_dwc2_extcon_setup(void)
{
	return platform_driver_register(&driver_info);
};

int fusb_otg_dwc2_extcon_teardown(void)
{
#if 0  /* there is no extcon driver that frees any memories */
	struct fusb_otg_extcon_info *info = platform_get_drvdata(pdev);
	struct device *dev = &pdev->dev;

	devm_extcon_dev_unregister(dev, info->edev);
	devm_extcon_dev_free(dev, info->edev);
	devm_kfree(dev, info);
#endif
	platform_driver_unregister(&driver_info);
	m_info = NULL;
	return 0;
}

