/*
 * drivers/usb/f_usb/usb_otg_dwc3_extcon.c
 *
 * Copyright 2022 Sony Corporation
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/slab.h>
#include <linux/module.h>
#include <linux/init.h>
#include <asm/io.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/extcon.h>
#include <linux/extcon-provider.h>
#include <linux/usb/f_usb/usb_otg_dwc3_extcon.h>
#include "../dwc3/dwc3-cxd.h"

#define COMPAT "cxd,dwc3_extcon"
#define NAME_DRV "dwc3_extcon"

#define EXP_SYMBOL_FUNC dwc3_extcon_get_device
#define INTNAME_VBUS "dwc3_vbus"
#define G_EDEV	g_dwc3

#define D_VBUS_UNLIMIT				0x0
#define D_EXT_MON					0x10
#define		D_EXT_MON_ID		BIT(1)
#define		D_EXT_MON_VBUS		BIT(0)
#define D_INTINV					0x20
#define 	D_INTINV_ID			BIT(1)
#define 	D_INTINV_VBUS		BIT(0)
#define D_INT_MSK					0x30
#define 	D_INT_MSK_ID		BIT(1)
#define 	D_INT_MSK_VBUS		BIT(0)
#define 	D_INT_MSK_BOTH_EN	0
#define D_INT_CLR					0x40
#define 	D_INT_CLR_ID		BIT(1)
#define 	D_INT_CLR_VBUS		BIT(0)

enum extcon_port_no {
	EXTCON_DWC3 = 0,
	EXTCON_DWC2 = 1,
};

struct extcon_dev *G_EDEV;

unsigned int g_dwc3_irq;

void __iomem *g_extcon0_reg;

struct pdata {
	struct device *dev;
	void __iomem *reg;
	spinlock_t lock;
	struct extcon_dev *edev;
};

static struct pdata *m_info = NULL;

static const unsigned int usb_otg_extcon_cable[] = {
	EXTCON_USB,
	EXTCON_USB_HOST,
	EXTCON_NONE,
};

static const struct of_device_id dev_match[] = {
	{ .compatible = COMPAT},
	{ /* end of list */ },
};

static DEFINE_MUTEX(extcon_lock);

static ssize_t polarity_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct pdata *priv = dev_get_drvdata(dev);
	unsigned long val;
	union extcon_property_value state;

	if(kstrtol(buf, 10, &val) == -EINVAL)
		pr_info("\nkstrtol err\n");

	state.intval = (int)val;
	extcon_set_property(priv->edev, EXTCON_USB, EXTCON_PROP_USB_TYPEC_POLARITY, state);
	extcon_set_property(priv->edev, EXTCON_USB_HOST, EXTCON_PROP_USB_TYPEC_POLARITY, state);
	return count;
}

static ssize_t drd_state_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct pdata *priv = dev_get_drvdata(dev);
	struct usb_en {
		uint8_t dev:1;
		uint8_t host:1;
		uint8_t reserved:6;
	} val = {0};
	union extcon_property_value property = {0};
	int ret = 0;

	if (extcon_get_state(priv->edev, EXTCON_USB) > 0) {
		val.dev = 1;
		ret = extcon_get_property(priv->edev, EXTCON_USB, EXTCON_PROP_USB_TYPEC_POLARITY,&property);
	}
	if (extcon_get_state(priv->edev, EXTCON_USB_HOST) > 0) {
		val.host = 1;
		ret = extcon_get_property(priv->edev, EXTCON_USB_HOST, EXTCON_PROP_USB_TYPEC_POLARITY,&property);
	}

	if (ret)
		dev_err(priv->dev, "get polarity property failed\n");

	return snprintf(buf, 40, "device=%s, host=%s, polarity=%s\n", val.dev ? "ON" : "off", val.host ? "ON" : "off", property.intval ? "flip" : "normal");
}

static ssize_t drd_switch_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	unsigned long val;

	if(kstrtol(buf, 10, &val) == -EINVAL)
		pr_info("\nkstrtol err\n");

	if (val == 0 ) {
		fusb_otg_set_dwc3_drd_state(DWC3_DRD_HOST);
	} else if (val == 1) {
		fusb_otg_set_dwc3_drd_state(DWC3_DRD_GADGET);
	} else {
		fusb_otg_set_dwc3_drd_state(DWC3_DRD_IDLE);
	}
	return count;
}

static DEVICE_ATTR_WO(polarity);
static DEVICE_ATTR_WO(drd_switch);
static DEVICE_ATTR_RO(drd_state);
static struct attribute* dwc3_attrs[] = {
	&dev_attr_polarity.attr,
	&dev_attr_drd_switch.attr,
	&dev_attr_drd_state.attr,
	NULL
};

static const struct attribute_group attr_grp = {
	.attrs = dwc3_attrs,
};

/* DRD */
int fusb_otg_set_dwc3_drd_state(enum DWC3_DRD state)
{
	enum DWC3_DRD prev_state;

	if (NULL == m_info)
		return -EAGAIN;

	mutex_lock(&extcon_lock);
	fusb_otg_get_dwc3_drd_state(&prev_state);
	if (state == DWC3_DRD_HOST) {
		if (prev_state != DWC3_DRD_HOST) {
			dwc3_cxd_start_drd();
			extcon_set_state_sync(m_info->edev, EXTCON_USB, false);
			extcon_set_state_sync(m_info->edev, EXTCON_USB_HOST, false);
			extcon_set_state_sync(m_info->edev, EXTCON_USB_HOST, true);
			dwc3_cxd_wait_drd_done();
		}
	} else if (state == DWC3_DRD_GADGET) {
		if (prev_state != DWC3_DRD_GADGET) {
			dwc3_cxd_start_drd();
			extcon_set_state_sync(m_info->edev, EXTCON_USB_HOST, false);
			extcon_set_state_sync(m_info->edev, EXTCON_USB, false);
			extcon_set_state_sync(m_info->edev, EXTCON_USB, true);
			dwc3_cxd_wait_drd_done();
		}
	} else {
		if (prev_state != DWC3_DRD_IDLE) {
			dwc3_cxd_start_drd();
			extcon_set_state_sync(m_info->edev, EXTCON_USB, false);
			extcon_set_state_sync(m_info->edev, EXTCON_USB_HOST, false);
		}
	}
	mutex_unlock(&extcon_lock);

	return 0;
}
EXPORT_SYMBOL(fusb_otg_set_dwc3_drd_state);

int fusb_otg_get_dwc3_drd_state(enum DWC3_DRD *state)
{
	if (NULL == m_info)
		return -EAGAIN;

	if(extcon_get_state(m_info->edev, EXTCON_USB_HOST)) {
		*state = DWC3_DRD_HOST;
	}
	else if(extcon_get_state(m_info->edev, EXTCON_USB)) {
		*state = DWC3_DRD_GADGET;
	}
	else {
		*state = DWC3_DRD_IDLE;
	}

	return 0;
}

/* tpc setting all */
int fusb_otg_set_dwc3_tpc_setting(bool orientation, bool ss_operation)
{
	int ret;

	ret = fusb_otg_set_dwc3_orientation(orientation);
	if (0 > ret) {
		dev_err(m_info->dev, "set orientation err=%d\n", ret);
		return ret;
	}

	ret = fusb_otg_set_dwc3_ss_operation(ss_operation);
	if (0 > ret) {
		dev_err(m_info->dev, "set ss operation err=%d\n", ret);
		return ret;
	}
        return 0;
}
EXPORT_SYMBOL(fusb_otg_set_dwc3_tpc_setting);

/* type-c orientation */
int fusb_otg_set_dwc3_orientation(bool orientation)
{
	int ret;
	union extcon_property_value prop_val;

	if (NULL == m_info)
		return -EAGAIN;

	prop_val.intval = 0;
	if (orientation)
		prop_val.intval = 1;

	if (fusb_otg_dwc3_read_vbus()) {
		mutex_lock(&extcon_lock);
		dwc3_cxd_start_drd();
		ret = extcon_set_property_sync(m_info->edev, EXTCON_USB, EXTCON_PROP_USB_TYPEC_POLARITY, prop_val);
		dwc3_cxd_wait_drd_done();
		mutex_unlock(&extcon_lock);
	} else {
		ret = extcon_set_property(m_info->edev, EXTCON_USB_HOST, EXTCON_PROP_USB_TYPEC_POLARITY, prop_val);
	}

	if (0 > ret) {
		dev_err(m_info->dev, "extcon_set_property prop_val=%d, err=%d\n", prop_val.intval, ret);
	}

	return ret;
}

/* ss operation */
int fusb_otg_set_dwc3_ss_operation(bool ss_operation)
{
	int ret;
	ret = 0; // not supported
	return ret;
}

bool fusb_otg_dwc3_read_vbus(void)
{
	uint32_t val;
	val = readl(g_extcon0_reg + D_EXT_MON);
	return (bool)(val & D_EXT_MON_VBUS);
}

void fusb_otg_dwc3_enable_irq(void)
{
	enable_irq(g_dwc3_irq);
}

void fusb_otg_dwc3_disable_irq(void)
{
	disable_irq(g_dwc3_irq);
}

static irqreturn_t dwc3_vbus_interrupt(int irq, void *data)
{
	struct pdata *info = data;
	bool val;

	writel_relaxed(D_INT_CLR_VBUS, info->reg + D_INT_CLR);
	val = fusb_otg_dwc3_read_vbus();
	if (val == true) {
		fusb_interrupt_usb_vbus0(USB_OTG_VBUS_STAT_VALID);
	} else {
		fusb_interrupt_usb_vbus0(USB_OTG_VBUS_STAT_OFF);
	}

	return IRQ_HANDLED;
}

static int extcon_probe(struct platform_device *pdev_probe)
{
	int ret;
	struct pdata *pd;
	struct platform_device *pdev;
	struct device *dev;
	uint32_t val;
	struct resource *res;

	ret = 0;

	pd = kzalloc(sizeof(*pd),GFP_KERNEL);
	pdev = pdev_probe;
	platform_set_drvdata(pdev, pd);
	dev = &pdev->dev;
	pd->dev = dev;
	m_info = pd;

	ret = sysfs_create_group(&pdev->dev.kobj, &attr_grp);
	G_EDEV = pd->edev = devm_extcon_dev_allocate(dev, usb_otg_extcon_cable);
	if (IS_ERR(pd->edev)) {
		dev_err(dev, "failed to allocate memory for extcon\n");
		goto fail2;
	}
	ret = devm_extcon_dev_register(dev, pd->edev);
	if (ret) {
		dev_err(pd->dev, "failed to register extcon device\n");
		goto fail2;
	}
	
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "reg-glue");
	pd->reg = devm_ioremap_resource(dev, res);
	if (IS_ERR(pd->reg))
		return PTR_ERR(pd->reg);

	g_extcon0_reg = pd->reg;
	res = platform_get_resource_byname(pdev, IORESOURCE_IRQ, "vbus");
	g_dwc3_irq = res->start;
	if (devm_request_threaded_irq(pd->dev, res->start, &dwc3_vbus_interrupt,
		NULL, IRQF_TRIGGER_RISING, INTNAME_VBUS, pd))
		return -ENOMEM;

	fusb_otg_dwc3_disable_irq();
	
	/* enable interrupt */
	val = D_INT_MSK_BOTH_EN;
	writel_relaxed(val, pd->reg + D_INT_MSK);

	extcon_set_property_capability(pd->edev, EXTCON_USB,EXTCON_PROP_USB_TYPEC_POLARITY);
	extcon_set_property_capability(pd->edev, EXTCON_USB_HOST,EXTCON_PROP_USB_TYPEC_POLARITY);

	return 0;

fail2:
	platform_device_unregister(pdev);
	kfree(pd);
	return -1;
}

static int extcon_remove(struct platform_device *pdev)
{
	return 0;
}

static struct platform_driver driver_info = {
	.driver = {
		.name = NAME_DRV,
		.of_match_table = of_match_ptr(dev_match),
	},
	.probe = extcon_probe,
	.remove = extcon_remove,
};

int32_t fusb_otg_dwc3_extcon_setup(void)
{
	return platform_driver_register(&driver_info);
};

int fusb_otg_dwc3_extcon_teardown(void)
{
#if 0  /* there is no extcon driver that frees any memories */
	struct fusb_otg_extcon_info *info = platform_get_drvdata(pdev);
	struct device *dev = &pdev->dev;

	devm_extcon_dev_unregister(dev, info->edev);
	devm_extcon_dev_free(dev, info->edev);
	devm_kfree(dev, info);
#endif
	platform_driver_unregister(&driver_info);
	m_info = NULL;
	return 0;
}

