// SPDX-License-Identifier: GPL-2.0+
/*
 * f_uac2.c -- USB Audio Class 2.0 Function
 *
 * Copyright (C) 2011
 *    Yadwinder Singh (yadi.brar01@gmail.com)
 *    Jaswinder Singh (jaswinder.singh@linaro.org)
 *
 * Copyright (C) 2020
 *    Ruslan Bilovol (ruslan.bilovol@gmail.com)
 */

#include <linux/usb/audio.h>
#include <linux/usb/audio-v2.h>
#include <linux/module.h>
#include <linux/delay.h>

#include "u_audio.h"
#include "u_uac2.h"

/* UAC2 spec: 4.1 Audio Channel Cluster Descriptor */
#define UAC2_CHANNEL_MASK 0x07FFFFFF

/*
 * The driver implements a simple UAC_2 topology.
 * USB-OUT -> IT_1 -> FU -> OT_3 -> ALSA_Capture
 * ALSA_Playback -> IT_2 -> FU -> OT_4 -> USB-IN
 * Capture and Playback sampling rates are independently
 *  controlled by two clock sources :
 *    CLK_5 := c_srate, and CLK_6 := p_srate
 */
//#define USB_OUT_CLK_ID	(out_clk_src_desc.bClockID)
//#define USB_IN_CLK_ID	(in_clk_src_desc.bClockID)
#if defined(VOLUME_MUTE_SUPPORT)
#define USB_OUT_FU_ID	(out_feature_unit_desc->bUnitID)
#define USB_IN_FU_ID	(in_feature_unit_desc->bUnitID)
#endif

#define USB_OUT_IT_ID	1
#define IO_IN_IT_ID	2
#define IO_OUT_OT_ID	3
#define USB_IN_OT_ID	4
#define USB_OUT_CLK_ID	5
#define USB_IN_CLK_ID	6
#define FEATURE_UNIT_ID	7

#define CONTROL_ABSENT	0
#define CONTROL_RDONLY	1
#define CONTROL_RDWR	3

#define CLK_FREQ_CTRL	0
#define CLK_VLD_CTRL	2

#if defined(VOLUME_MUTE_SUPPORT)
#define FU_MUTE_CTRL	0
#define FU_VOL_CTRL	2
#endif

#define COPY_CTRL	0
#define CONN_CTRL	2
#define OVRLD_CTRL	4
#define CLSTR_CTRL	6
#define UNFLW_CTRL	8
#define OVFLW_CTRL	10

#define EPIN_EN(_opts) ((_opts)->p_chmask != 0)
//#define EPOUT_EN(_opts) ((_opts)->c_chmask != 0)
#if defined(FEEDBACK_ENDPOINT_SUPPORT)
#define EPOUT_FBACK_IN_EN(_opts) ((_opts)->c_sync == USB_ENDPOINT_SYNC_ASYNC)
#endif

#if defined(VOLUME_MUTE_SUPPORT)
#define FUIN_EN(_opts) (EPIN_EN(_opts) \
				&& ((_opts)->p_mute_present \
				|| (_opts)->p_volume_present))
#define FUOUT_EN(_opts) (EPOUT_EN(_opts) \
				&& ((_opts)->c_mute_present \
				|| (_opts)->c_volume_present))
#endif

struct f_uac2 {
	struct g_audio g_audio;
	u8 ac_intf, as_in_intf, as_out_intf;
	u8 ac_alt, as_in_alt, as_out_alt;	/* needed for get_alt() */

#if defined(VOLUME_MUTE_SUPPORT)
	struct usb_ctrlrequest setup_cr;	/* will be used in data stage */

	/* Interrupt IN endpoint of AC interface */
	struct usb_ep	*int_ep;
	atomic_t	int_count;
#endif
};

static inline struct f_uac2 *func_to_uac2(struct usb_function *f)
{
	return container_of(f, struct f_uac2, g_audio.func);
}

static inline
struct f_uac2_opts *g_audio_to_uac2_opts(struct g_audio *agdev)
{
	return container_of(agdev->func.fi, struct f_uac2_opts, func_inst);
}

#if defined(VOLUME_MUTE_SUPPORT)
static int afunc_notify(struct g_audio *agdev, int unit_id, int cs);
#endif

/* --------- USB Function Interface ------------- */

enum {
	STR_ASSOC,
	STR_IF_CTRL,
	STR_CLKSRC_IN,
	STR_CLKSRC_OUT,
	STR_USB_IT,
	STR_IO_IT,
	STR_USB_OT,
	STR_IO_OT,
#if defined(VOLUME_MUTE_SUPPORT)
	STR_FU_IN,
	STR_FU_OUT,
#endif
	STR_AS_OUT_ALT0,
	STR_AS_OUT_ALT1,
	STR_AS_IN_ALT0,
	STR_AS_IN_ALT1,
	STR_FEAT_DESC_0,
};

static char clksrc_in[8];
//static char clksrc_out[8];
/*
static struct usb_string strings_fn[] = {
	[STR_ASSOC].s = "Source/Sink",
	[STR_IF_CTRL].s = "Topology Control",
	[STR_CLKSRC_IN].s = clksrc_in,
	[STR_CLKSRC_OUT].s = clksrc_out,
	[STR_USB_IT].s = "USBH Out",
	[STR_IO_IT].s = "USBD Out",
	[STR_USB_OT].s = "USBH In",
	[STR_IO_OT].s = "USBD In",
#if defined(VOLUME_MUTE_SUPPORT)
	[STR_FU_IN].s = "Capture Volume",
	[STR_FU_OUT].s = "Playback Volume",
#endif
	[STR_AS_OUT_ALT0].s = "Playback Inactive",
	[STR_AS_OUT_ALT1].s = "Playback Active",
	[STR_AS_IN_ALT0].s = "Capture Inactive",
	[STR_AS_IN_ALT1].s = "Capture Active",
	[STR_FEAT_DESC_0].s = "Feature Unit",
	{ },
};
*/
//static struct usb_gadget_strings str_fn = {
//	.language = 0x0409,	/* en-us */
//	.strings = strings_fn,
//};
/*
static struct usb_gadget_strings *fn_strings[] = {
	&str_fn,
	NULL,
};
*/
static struct usb_interface_assoc_descriptor iad_desc = {
	.bLength = sizeof iad_desc,
	.bDescriptorType = USB_DT_INTERFACE_ASSOCIATION,

	.bFirstInterface = 0,
	.bInterfaceCount = 2,
	.bFunctionClass = USB_CLASS_AUDIO,
	.bFunctionSubClass = USB_SUBCLASS_AUDIOSTREAMING,
	.bFunctionProtocol = 0, //UAC_VERSION_2,
};

/* Audio Control Interface */
static struct usb_interface_descriptor std_ac_if_desc = {
	.bLength = sizeof std_ac_if_desc,
	.bDescriptorType = USB_DT_INTERFACE,

	.bAlternateSetting = 0,
#if defined(VOLUME_MUTE_SUPPORT)
	/* .bNumEndpoints = DYNAMIC */
#else
	.bNumEndpoints = 0,
#endif
	.bInterfaceClass = USB_CLASS_AUDIO,
	.bInterfaceSubClass = USB_SUBCLASS_AUDIOCONTROL,
	.bInterfaceProtocol = 0, // UAC_VERSION_2,
};

/* Clock source for IN traffic */
static struct uac_clock_source_descriptor in_clk_src_desc = {
	.bLength = sizeof in_clk_src_desc,
	.bDescriptorType = USB_DT_CS_INTERFACE,

	.bDescriptorSubtype = UAC2_CLOCK_SOURCE,
	.bClockID = USB_IN_CLK_ID,
	.bmAttributes = UAC_CLOCK_SOURCE_TYPE_INT_FIXED,
	.bmControls = (CONTROL_RDONLY << CLK_FREQ_CTRL),
	.bAssocTerminal = 0,
};

/* Clock source for OUT traffic */
static struct uac_clock_source_descriptor out_clk_src_desc = {
	.bLength = sizeof out_clk_src_desc,
	.bDescriptorType = USB_DT_CS_INTERFACE,

	.bDescriptorSubtype = UAC2_CLOCK_SOURCE,
	.bClockID = USB_OUT_CLK_ID,
	.bmAttributes = UAC_CLOCK_SOURCE_TYPE_INT_FIXED,
	.bmControls = (CONTROL_RDONLY << CLK_FREQ_CTRL),
	.bAssocTerminal = 0,
};

/* Input Terminal for USB_OUT */
static struct uac2_input_terminal_descriptor usb_out_it_desc = {
	.bLength = sizeof usb_out_it_desc,
	.bDescriptorType = USB_DT_CS_INTERFACE,

	.bDescriptorSubtype = UAC_INPUT_TERMINAL,
	.bTerminalID = USB_OUT_IT_ID,
	.wTerminalType = cpu_to_le16(UAC_TERMINAL_STREAMING),
	.bAssocTerminal = 0,
	.bCSourceID = USB_OUT_CLK_ID,
	.iChannelNames = 0,
	.bmControls = cpu_to_le16(CONTROL_RDWR << COPY_CTRL),
};

/* Input Terminal for I/O-In */
static struct uac_input_terminal_descriptor io_in_it_desc = {
	.bLength = sizeof io_in_it_desc,
	.bDescriptorType = USB_DT_CS_INTERFACE,

	.bDescriptorSubtype = UAC_INPUT_TERMINAL,
	.bTerminalID = IO_IN_IT_ID,
	.wTerminalType = cpu_to_le16(UAC_INPUT_TERMINAL_MICROPHONE),
	.bAssocTerminal = 0,
	.wChannelConfig =	3,  //MOD 3->0
//	.bCSourceID = USB_IN_CLK_ID,
//	.iChannelNames = 0,
//	.bmControls = cpu_to_le16(CONTROL_RDWR << COPY_CTRL),
};

/* Ouput Terminal for USB_IN */
static struct uac1_output_terminal_descriptor usb_in_ot_desc = {
	.bLength = sizeof usb_in_ot_desc,
	.bDescriptorType = USB_DT_CS_INTERFACE,

	.bDescriptorSubtype = UAC_OUTPUT_TERMINAL,
	.bTerminalID = USB_IN_OT_ID,
	.wTerminalType = cpu_to_le16(UAC_TERMINAL_STREAMING),
	.bAssocTerminal = 0,
	.bSourceID = IO_IN_IT_ID,
//	.bCSourceID = USB_IN_CLK_ID,
//	.bmControls = cpu_to_le16(CONTROL_RDWR << COPY_CTRL),
};

/* Ouput Terminal for I/O-Out */
static struct uac2_output_terminal_descriptor io_out_ot_desc = {
	.bLength = sizeof io_out_ot_desc,
	.bDescriptorType = USB_DT_CS_INTERFACE,

	.bDescriptorSubtype = UAC_OUTPUT_TERMINAL,
	.bTerminalID = IO_OUT_OT_ID,
	.wTerminalType = cpu_to_le16(UAC_OUTPUT_TERMINAL_SPEAKER),
	.bAssocTerminal = 0,
	.bSourceID = USB_OUT_IT_ID,
	.bCSourceID = USB_OUT_CLK_ID,
	.bmControls = cpu_to_le16(CONTROL_RDWR << COPY_CTRL),
};


#if defined(VOLUME_MUTE_SUPPORT)
static struct uac2_feature_unit_descriptor *in_feature_unit_desc;
static struct uac2_feature_unit_descriptor *out_feature_unit_desc;
#endif

DECLARE_UAC_AC_HEADER_DESCRIPTOR(1);

static struct uac1_ac_header_descriptor_1 ac_hdr_desc = {
	.bLength = sizeof ac_hdr_desc,
	.bDescriptorType = USB_DT_CS_INTERFACE,

	.bDescriptorSubtype = UAC_MS_HEADER,
	.bcdADC = cpu_to_le16(0x100),
//	.bCategory = UAC2_FUNCTION_IO_BOX,
	.wTotalLength = cpu_to_le16(0x0027),
	.bInCollection =	1,
	.baInterfaceNr = {
	/* Interface number of the first AudioStream interface */
		[0] =		1,
	}
};

#if defined(VOLUME_MUTE_SUPPORT)
/* AC IN Interrupt Endpoint */
static struct usb_endpoint_descriptor fs_ep_int_desc = {
	.bLength = USB_DT_ENDPOINT_SIZE,
	.bDescriptorType = USB_DT_ENDPOINT,

	.bEndpointAddress = USB_DIR_IN,
	.bmAttributes = USB_ENDPOINT_XFER_INT,
	.wMaxPacketSize = cpu_to_le16(6),
	.bInterval = 1,
};

static struct usb_endpoint_descriptor hs_ep_int_desc = {
	.bLength = USB_DT_ENDPOINT_SIZE,
	.bDescriptorType = USB_DT_ENDPOINT,

	.bmAttributes = USB_ENDPOINT_XFER_INT,
	.wMaxPacketSize = cpu_to_le16(6),
	.bInterval = 4,
};

static struct usb_endpoint_descriptor ss_ep_int_desc = {
	.bLength = USB_DT_ENDPOINT_SIZE,
	.bDescriptorType = USB_DT_ENDPOINT,

	.bEndpointAddress = USB_DIR_IN,
	.bmAttributes = USB_ENDPOINT_XFER_INT,
	.wMaxPacketSize = cpu_to_le16(6),
	.bInterval = 4,
};
#endif

#if defined(VOLUME_MUTE_SUPPORT)
static struct usb_ss_ep_comp_descriptor ss_ep_int_desc_comp = {
	.bLength = sizeof(ss_ep_int_desc_comp),
	.bDescriptorType = USB_DT_SS_ENDPOINT_COMP,
	.wBytesPerInterval = cpu_to_le16(6),
};
#endif

/* Audio Streaming OUT Interface - Alt0 */
static struct usb_interface_descriptor std_as_out_if0_desc = {
	.bLength = sizeof std_as_out_if0_desc,
	.bDescriptorType = USB_DT_INTERFACE,

	.bAlternateSetting = 0,
	.bNumEndpoints = 0,
	.bInterfaceClass = USB_CLASS_AUDIO,
	.bInterfaceSubClass = USB_SUBCLASS_AUDIOSTREAMING,
	.bInterfaceProtocol = 0,//UAC_VERSION_2,
};

/* Audio Streaming OUT Interface - Alt1 */
static struct usb_interface_descriptor std_as_out_if1_desc = {
	.bLength = sizeof std_as_out_if1_desc,
	.bDescriptorType = USB_DT_INTERFACE,

	.bAlternateSetting = 1,
	.bNumEndpoints = 1,
	.bInterfaceClass = USB_CLASS_AUDIO,
	.bInterfaceSubClass = USB_SUBCLASS_AUDIOSTREAMING,
	.bInterfaceProtocol = 0,//UAC_VERSION_2,
};

/* Audio Stream OUT Intface Desc */
static struct uac2_as_header_descriptor as_out_hdr_desc = {
	.bLength = sizeof as_out_hdr_desc,
	.bDescriptorType = USB_DT_CS_INTERFACE,

	.bDescriptorSubtype = UAC_AS_GENERAL,
	.bTerminalLink = USB_OUT_IT_ID,
	.bmControls = 0,
	.bFormatType = UAC_FORMAT_TYPE_I,
	.bmFormats = cpu_to_le32(UAC_FORMAT_TYPE_I_PCM),
	.iChannelNames = 0,
};

/* Audio USB_OUT Format */
static struct uac2_format_type_i_descriptor as_out_fmt1_desc = {
	.bLength = sizeof as_out_fmt1_desc,
	.bDescriptorType = USB_DT_CS_INTERFACE,
	.bDescriptorSubtype = UAC_FORMAT_TYPE,
	.bFormatType = UAC_FORMAT_TYPE_I,
};

/* STD AS ISO OUT Endpoint */
//static struct usb_endpoint_descriptor fs_epout_desc = {
//	.bLength = USB_DT_ENDPOINT_SIZE,
//	.bDescriptorType = USB_DT_ENDPOINT,
//
//	.bEndpointAddress = USB_DIR_OUT,
//	/* .bmAttributes = DYNAMIC */
//	.wMaxPacketSize = cpu_to_le16(1023),
//	.bInterval = 1,
//};

//static struct usb_endpoint_descriptor hs_epout_desc = {
//	.bLength = USB_DT_ENDPOINT_SIZE,
//	.bDescriptorType = USB_DT_ENDPOINT,
//
//	/* .bmAttributes = DYNAMIC */
//	/* .wMaxPacketSize = DYNAMIC */
//	.bInterval = 4,
//};

//static struct usb_endpoint_descriptor ss_epout_desc = {
//	.bLength = USB_DT_ENDPOINT_SIZE,
//	.bDescriptorType = USB_DT_ENDPOINT,
//
//	.bEndpointAddress = USB_DIR_OUT,
//	/* .bmAttributes = DYNAMIC */
//	/* .wMaxPacketSize = DYNAMIC */
//	.bInterval = 4,
//};

//static struct usb_ss_ep_comp_descriptor ss_epout_desc_comp = {
//	.bLength		= sizeof(ss_epout_desc_comp),
//	.bDescriptorType	= USB_DT_SS_ENDPOINT_COMP,
//	.bMaxBurst		= 0,
//	.bmAttributes		= 0,
//	/* wBytesPerInterval = DYNAMIC */
//};

/* CS AS ISO OUT Endpoint */
static struct uac2_iso_endpoint_descriptor as_iso_out_desc = {
	.bLength = sizeof as_iso_out_desc,
	.bDescriptorType = USB_DT_CS_ENDPOINT,

	.bDescriptorSubtype = UAC_EP_GENERAL,
	.bmAttributes = 0,
	.bmControls = 0,
	.bLockDelayUnits = 0,
	.wLockDelay = 0,
};

#if defined(FEEDBACK_ENDPOINT_SUPPORT)
/* STD AS ISO IN Feedback Endpoint */
static struct usb_endpoint_descriptor fs_epin_fback_desc = {
	.bLength = USB_DT_ENDPOINT_SIZE,
	.bDescriptorType = USB_DT_ENDPOINT,

	.bEndpointAddress = USB_DIR_IN,
	.bmAttributes = USB_ENDPOINT_XFER_ISOC | USB_ENDPOINT_USAGE_FEEDBACK,
	.wMaxPacketSize = cpu_to_le16(3),
	.bInterval = 1,
};

static struct usb_endpoint_descriptor hs_epin_fback_desc = {
	.bLength = USB_DT_ENDPOINT_SIZE,
	.bDescriptorType = USB_DT_ENDPOINT,

	.bmAttributes = USB_ENDPOINT_XFER_ISOC | USB_ENDPOINT_USAGE_FEEDBACK,
	.wMaxPacketSize = cpu_to_le16(4),
	.bInterval = 4,
};

static struct usb_endpoint_descriptor ss_epin_fback_desc = {
	.bLength = USB_DT_ENDPOINT_SIZE,
	.bDescriptorType = USB_DT_ENDPOINT,

	.bEndpointAddress = USB_DIR_IN,
	.bmAttributes = USB_ENDPOINT_XFER_ISOC | USB_ENDPOINT_USAGE_FEEDBACK,
	.wMaxPacketSize = cpu_to_le16(4),
	.bInterval = 4,
};

static struct usb_ss_ep_comp_descriptor ss_epin_fback_desc_comp = {
	.bLength		= sizeof(ss_epin_fback_desc_comp),
	.bDescriptorType	= USB_DT_SS_ENDPOINT_COMP,
	.bMaxBurst		= 0,
	.bmAttributes		= 0,
	.wBytesPerInterval	= cpu_to_le16(4),
};
#endif

/* Audio Streaming IN Interface - Alt0 */
static struct usb_interface_descriptor std_as_in_if0_desc = {
	.bLength = sizeof std_as_in_if0_desc,
	.bDescriptorType = USB_DT_INTERFACE,

	.bAlternateSetting = 0,
	.bNumEndpoints = 0,
	.bInterfaceClass = USB_CLASS_AUDIO,
	.bInterfaceSubClass = USB_SUBCLASS_AUDIOSTREAMING,
	.bInterfaceProtocol = 0,
};

/* Audio Streaming IN Interface - Alt1 */
static struct usb_interface_descriptor std_as_in_if1_desc = {
	.bLength = sizeof std_as_in_if1_desc,
	.bDescriptorType = USB_DT_INTERFACE,

	.bAlternateSetting = 1,
	.bNumEndpoints = 1,
	.bInterfaceClass = USB_CLASS_AUDIO,
	.bInterfaceSubClass = USB_SUBCLASS_AUDIOSTREAMING,
	.bInterfaceProtocol = 0,
};

/* Audio Stream IN Intface Desc */
static struct uac1_as_header_descriptor as_in_hdr_desc = {
	.bLength = sizeof as_in_hdr_desc,
	.bDescriptorType = USB_DT_CS_INTERFACE,

	.bDescriptorSubtype = UAC_AS_GENERAL,
	.bTerminalLink = USB_IN_OT_ID,
	.bDelay =		1,
	.wFormatTag =		UAC_FORMAT_TYPE_I_PCM,

//	.bmControls = 0,
//	.bFormatType = UAC_FORMAT_TYPE_I,
//	.bmFormats = cpu_to_le32(UAC_FORMAT_TYPE_I_PCM),
//	.iChannelNames = 0,
};
DECLARE_UAC_FORMAT_TYPE_I_DISCRETE_DESC(1);

/* Audio USB_IN Format */
static struct uac_format_type_i_discrete_descriptor_1 as_in_fmt1_desc = {
	.bLength = UAC_FORMAT_TYPE_I_DISCRETE_DESC_SIZE(1),
	.bDescriptorType = USB_DT_CS_INTERFACE,
	.bDescriptorSubtype = UAC_FORMAT_TYPE,
	.bFormatType = UAC_FORMAT_TYPE_I,
	.bNrChannels = 2,
	.bSubframeSize =	2,
	.bBitResolution =	16,
	.bSamFreqType =		1,
};

/* STD AS ISO IN Endpoint */
static struct usb_endpoint_descriptor fs_epin_desc = {
	.bLength = USB_DT_ENDPOINT_SIZE,
	.bDescriptorType = USB_DT_ENDPOINT,

	.bEndpointAddress = USB_DIR_IN,
	.bmAttributes = USB_ENDPOINT_XFER_ISOC | USB_ENDPOINT_SYNC_ASYNC,
	/* .wMaxPacketSize = DYNAMIC */
	.bInterval = 1,
};

static struct usb_endpoint_descriptor hs_epin_desc = {
	.bLength = USB_DT_ENDPOINT_SIZE,
	.bDescriptorType = USB_DT_ENDPOINT,

	.bmAttributes = USB_ENDPOINT_XFER_ISOC | USB_ENDPOINT_SYNC_ASYNC,
	/* .wMaxPacketSize = DYNAMIC */
	.bInterval = 4,
};

//static struct usb_endpoint_descriptor ss_epin_desc = {
//	.bLength = USB_DT_ENDPOINT_SIZE,
//	.bDescriptorType = USB_DT_ENDPOINT,
//
//	.bEndpointAddress = USB_DIR_IN,
//	.bmAttributes = USB_ENDPOINT_XFER_ISOC | USB_ENDPOINT_SYNC_ASYNC,
//	/* .wMaxPacketSize = DYNAMIC */
//	.bInterval = 4,
//};

//static struct usb_ss_ep_comp_descriptor ss_epin_desc_comp = {
//	.bLength		= sizeof(ss_epin_desc_comp),
//	.bDescriptorType	= USB_DT_SS_ENDPOINT_COMP,
//	.bMaxBurst		= 0,
//	.bmAttributes		= 0,
//	/* wBytesPerInterval = DYNAMIC */
//};

/* CS AS ISO IN Endpoint */
static struct uac_iso_endpoint_descriptor as_iso_in_desc = {
	.bLength = 0x07, //sizeof as_iso_in_desc,
	.bDescriptorType = USB_DT_CS_ENDPOINT,

	.bDescriptorSubtype = UAC_EP_GENERAL,
	.bmAttributes = 0,
//	.bmControls = 0,
	.bLockDelayUnits = 0,
	.wLockDelay = 0,
};

DECLARE_UAC_FEATURE_UNIT_DESCRIPTOR(0); //MOD 0->2

static struct uac_feature_unit_descriptor_0 feature_unit_desc = {
	.bLength		= UAC_DT_FEATURE_UNIT_SIZE(0), //MOD 0->2
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubtype	= UAC_FEATURE_UNIT,
	.bUnitID		= FEATURE_UNIT_ID,
	.bSourceID		= IO_IN_IT_ID,
	.bControlSize		= 2,
	.bmaControls[0]		= 0,  //MOD
//	.bmaControls[1]		= 0,//(UAC_FU_MUTE | UAC_FU_VOLUME), //MOD
//	.bmaControls[2]		= 0,//(UAC_FU_MUTE | UAC_FU_VOLUME), //MOD
};

static struct usb_audio_control_selector feature_unit = {
	.list = LIST_HEAD_INIT(feature_unit.list),
	.id = FEATURE_UNIT_ID,
	.name = "Mute & Volume Control",
	.type = UAC_FEATURE_UNIT,
	.desc = (struct usb_descriptor_header *)&feature_unit_desc,
};

static struct usb_ss_ep_comp_descriptor uac_ss_control_comp = {
	.bLength		= sizeof(uac_ss_control_comp),
	.bDescriptorType	= USB_DT_SS_ENDPOINT_COMP,
	/* The following 3 values can be tweaked if necessary. */
	.bMaxBurst		= 0,
	.bmAttributes		= 0,
	.wBytesPerInterval	= 0x00,
};

static struct usb_endpoint_descriptor uac_ss_streaming_ep = {
	.bLength		= USB_DT_ENDPOINT_SIZE,
	.bDescriptorType	= USB_DT_ENDPOINT,

	.bEndpointAddress	= USB_DIR_IN,
	.bmAttributes		= USB_ENDPOINT_SYNC_ASYNC
				| USB_ENDPOINT_XFER_ISOC,
	.wMaxPacketSize = 0x0400,//0xC0,
	.bInterval = 0x04,
//	.bmAttributes		= USB_ENDPOINT_XFER_BULK,
	/* The wMaxPacketSize and bInterval values will be initialized from
	 * module parameters.
	 */
};

static struct usb_ss_ep_comp_descriptor uac_ss_streaming_comp = {
	.bLength		= 0x06,//sizeof(uac_ss_streaming_comp),
	.bDescriptorType	= USB_DT_SS_ENDPOINT_COMP,
	.bMaxBurst = 0,
	.bmAttributes = 0,
	.wBytesPerInterval = 0,
	/* The bMaxBurst, bmAttributes and wBytesPerInterval values will be
	 * initialized from module parameters.
	 */
};

static struct usb_descriptor_header *fs_audio_desc[] = {
	(struct usb_descriptor_header *)&iad_desc,
	(struct usb_descriptor_header *)&std_ac_if_desc,

	(struct usb_descriptor_header *)&ac_hdr_desc,
//	(struct usb_descriptor_header *)&in_clk_src_desc,
//	(struct usb_descriptor_header *)&out_clk_src_desc,
//	(struct usb_descriptor_header *)&usb_out_it_desc,
#if defined(VOLUME_MUTE_SUPPORT)
	(struct usb_descriptor_header *)&out_feature_unit_desc,
#endif
	(struct usb_descriptor_header *)&io_in_it_desc,
	(struct usb_descriptor_header *)&usb_in_ot_desc,
#if defined(VOLUME_MUTE_SUPPORT)
	(struct usb_descriptor_header *)&in_feature_unit_desc,
#endif
//	(struct usb_descriptor_header *)&io_out_ot_desc,
	(struct usb_descriptor_header *)&feature_unit_desc,    //ADD Feature unit
#if defined(VOLUME_MUTE_SUPPORT)
	(struct usb_descriptor_header *)&fs_ep_int_desc,
#endif
//	(struct usb_descriptor_header *)&std_as_out_if0_desc,
//	(struct usb_descriptor_header *)&std_as_out_if1_desc,

//	(struct usb_descriptor_header *)&as_out_hdr_desc,
//	(struct usb_descriptor_header *)&as_out_fmt1_desc,
//	(struct usb_descriptor_header *)&fs_epout_desc,
//	(struct usb_descriptor_header *)&as_iso_out_desc,
#if defined(FEEDBACK_ENDPOINT_SUPPORT)
	(struct usb_descriptor_header *)&fs_epin_fback_desc,
#endif

	(struct usb_descriptor_header *)&std_as_in_if0_desc,
	(struct usb_descriptor_header *)&std_as_in_if1_desc,

	(struct usb_descriptor_header *)&as_in_hdr_desc,
	(struct usb_descriptor_header *)&as_in_fmt1_desc,
	(struct usb_descriptor_header *)&fs_epin_desc,
	(struct usb_descriptor_header *)&as_iso_in_desc,
	NULL,
};

static struct usb_descriptor_header *hs_audio_desc[] = {
	(struct usb_descriptor_header *)&iad_desc,
	(struct usb_descriptor_header *)&std_ac_if_desc,

	(struct usb_descriptor_header *)&ac_hdr_desc,
//	(struct usb_descriptor_header *)&in_clk_src_desc,
//	(struct usb_descriptor_header *)&out_clk_src_desc,
//	(struct usb_descriptor_header *)&usb_out_it_desc,
#if defined(VOLUME_MUTE_SUPPORT)
	(struct usb_descriptor_header *)&out_feature_unit_desc,
#endif
	(struct usb_descriptor_header *)&io_in_it_desc,
	(struct usb_descriptor_header *)&usb_in_ot_desc,
#if defined(VOLUME_MUTE_SUPPORT)
	(struct usb_descriptor_header *)&in_feature_unit_desc,
#endif
//	(struct usb_descriptor_header *)&io_out_ot_desc,
	(struct usb_descriptor_header *)&feature_unit_desc,    //ADD Feature unit

#if defined(VOLUME_MUTE_SUPPORT)
	(struct usb_descriptor_header *)&hs_ep_int_desc,
#endif
//	(struct usb_descriptor_header *)&std_as_out_if0_desc,
//	(struct usb_descriptor_header *)&std_as_out_if1_desc,

//	(struct usb_descriptor_header *)&as_out_hdr_desc,
//	(struct usb_descriptor_header *)&as_out_fmt1_desc,
//	(struct usb_descriptor_header *)&hs_epout_desc,
//	(struct usb_descriptor_header *)&as_iso_out_desc,
#if defined(FEEDBACK_ENDPOINT_SUPPORT)
	(struct usb_descriptor_header *)&hs_epin_fback_desc,
#endif

	(struct usb_descriptor_header *)&std_as_in_if0_desc,
	(struct usb_descriptor_header *)&std_as_in_if1_desc,

	(struct usb_descriptor_header *)&as_in_hdr_desc,
	(struct usb_descriptor_header *)&as_in_fmt1_desc,
	(struct usb_descriptor_header *)&hs_epin_desc,
	(struct usb_descriptor_header *)&as_iso_in_desc,
	NULL,
};

static struct usb_descriptor_header *ss_audio_desc[] = {
	(struct usb_descriptor_header *)&iad_desc,
	(struct usb_descriptor_header *)&std_ac_if_desc,

	(struct usb_descriptor_header *)&ac_hdr_desc,
//	(struct usb_descriptor_header *)&in_clk_src_desc,
//	(struct usb_descriptor_header *)&out_clk_src_desc,
//	(struct usb_descriptor_header *)&usb_out_it_desc,
//	(struct usb_descriptor_header *)&uac_ss_control_comp,
#if defined(VOLUME_MUTE_SUPPORT)
	(struct usb_descriptor_header *)&out_feature_unit_desc,
#endif
	(struct usb_descriptor_header *)&io_in_it_desc,
	(struct usb_descriptor_header *)&usb_in_ot_desc,
#if defined(VOLUME_MUTE_SUPPORT)
	(struct usb_descriptor_header *)&in_feature_unit_desc,
#endif
//	(struct usb_descriptor_header *)&io_out_ot_desc,
	(struct usb_descriptor_header *)&feature_unit_desc,    //ADD Feature unit

#if defined(VOLUME_MUTE_SUPPORT)
	(struct usb_descriptor_header *)&ss_ep_int_desc,
	(struct usb_descriptor_header *)&ss_ep_int_desc_comp,
#endif

//	(struct usb_descriptor_header *)&std_as_out_if0_desc,
//	(struct usb_descriptor_header *)&std_as_out_if1_desc,

//	(struct usb_descriptor_header *)&as_out_hdr_desc,
//	(struct usb_descriptor_header *)&as_out_fmt1_desc,
//	(struct usb_descriptor_header *)&ss_epout_desc,
//	(struct usb_descriptor_header *)&ss_epout_desc_comp,
//	(struct usb_descriptor_header *)&as_iso_out_desc,
#if defined(FEEDBACK_ENDPOINT_SUPPORT)
	(struct usb_descriptor_header *)&ss_epin_fback_desc,
	(struct usb_descriptor_header *)&ss_epin_fback_desc_comp,
#endif

	(struct usb_descriptor_header *)&std_as_in_if0_desc,
	(struct usb_descriptor_header *)&std_as_in_if1_desc,

	(struct usb_descriptor_header *)&as_in_hdr_desc,
	(struct usb_descriptor_header *)&as_in_fmt1_desc,
	(struct usb_descriptor_header *)&uac_ss_streaming_ep,
	(struct usb_descriptor_header *)&uac_ss_streaming_comp,
	(struct usb_descriptor_header *)&as_iso_in_desc,
	NULL,
};

#if defined(VOLUME_MUTE_SUPPORT)
struct cntrl_cur_lay2 {
	__le16	wCUR;
};

struct cntrl_range_lay2 {
	__le16	wNumSubRanges;
	__le16	wMIN;
	__le16	wMAX;
	__le16	wRES;
} __packed;
#endif

struct cntrl_cur_lay3 {
	__le32	dCUR;
};

struct cntrl_range_lay3 {
	__le16	wNumSubRanges;
	__le32	dMIN;
	__le32	dMAX;
	__le32	dRES;
} __packed;

static int set_ep_max_packet_size(const struct f_uac2_opts *uac2_opts,
	struct usb_endpoint_descriptor *ep_desc,
	enum usb_device_speed speed, bool is_playback)
{
	int chmask, srate, ssize;
	u16 max_size_bw, max_size_ep;
	unsigned int factor;

	switch (speed) {
	case USB_SPEED_FULL:
		max_size_ep = 1023;
		factor = 1000;
		break;

	case USB_SPEED_HIGH:
	case USB_SPEED_SUPER:
		max_size_ep = 1024;
		factor = 8000;
		break;

	default:
		return -EINVAL;
	}

	if (is_playback) {
		chmask = uac2_opts->p_chmask;
		srate = uac2_opts->p_srate;
		ssize = uac2_opts->p_ssize;
	} else {
		chmask = uac2_opts->c_chmask;
		srate = uac2_opts->c_srate;
		ssize = uac2_opts->c_ssize;
	}

#if defined(FEEDBACK_ENDPOINT_SUPPORT)
	if (!is_playback && (uac2_opts->c_sync == USB_ENDPOINT_SYNC_ASYNC)) {
		// Win10 requires max packet size + 1 frame
		srate = srate * (1000 + uac2_opts->fb_max) / 1000;
#else
	if (!is_playback) {
#endif
		// updated srate is always bigger, therefore DIV_ROUND_UP always yields +1
		max_size_bw = num_channels(chmask) * ssize *
			(DIV_ROUND_UP(srate, factor / (1 << (ep_desc->bInterval - 1))));
	} else {
		// adding 1 frame provision for Win10
		max_size_bw = num_channels(chmask) * ssize *
			(DIV_ROUND_UP(srate, factor / (1 << (ep_desc->bInterval - 1))) + 1);
	}
	ep_desc->wMaxPacketSize = cpu_to_le16(min_t(u16, max_size_bw,
						    max_size_ep));

	return 0;
}

#if defined(VOLUME_MUTE_SUPPORT)
static struct uac2_feature_unit_descriptor *build_fu_desc(int chmask)
{
	struct uac2_feature_unit_descriptor *fu_desc;
	int channels = num_channels(chmask);
	int fu_desc_size = UAC2_DT_FEATURE_UNIT_SIZE(channels);

	fu_desc = kzalloc(fu_desc_size, GFP_KERNEL);
	if (!fu_desc)
		return NULL;

	fu_desc->bLength = fu_desc_size;
	fu_desc->bDescriptorType = USB_DT_CS_INTERFACE;

	fu_desc->bDescriptorSubtype = UAC_FEATURE_UNIT;

	/* bUnitID, bSourceID and bmaControls will be defined later */

	return fu_desc;
}
#endif

/* Use macro to overcome line length limitation */
#define USBDHDR(p) (struct usb_descriptor_header *)(p)

#if 0
static void setup_headers(struct f_uac2_opts *opts,
			  struct usb_descriptor_header **headers,
			  enum usb_device_speed speed)
{
//	struct usb_ss_ep_comp_descriptor *epout_desc_comp = NULL;
	struct usb_ss_ep_comp_descriptor *epin_desc_comp = NULL;
#if defined(FEEDBACK_ENDPOINT_SUPPORT)
	struct usb_ss_ep_comp_descriptor *epin_fback_desc_comp = NULL;
#endif
	struct usb_endpoint_descriptor *epin_desc;

#if defined(FEEDBACK_ENDPOINT_SUPPORT)
	struct usb_endpoint_descriptor *epin_fback_desc;
#endif
#if defined(VOLUME_MUTE_SUPPORT)
	struct usb_endpoint_descriptor *ep_int_desc;
	struct usb_ss_ep_comp_descriptor *ep_int_desc_comp = NULL;
#endif
	int i;

	switch (speed) {
	case USB_SPEED_FULL:
//		epout_desc = &fs_epout_desc;
		epin_desc = &fs_epin_desc;
#if defined(FEEDBACK_ENDPOINT_SUPPORT)
		epin_fback_desc = &fs_epin_fback_desc;
#endif
#if defined(VOLUME_MUTE_SUPPORT)
		ep_int_desc = &fs_ep_int_desc;
#endif
		break;
	case USB_SPEED_HIGH:
//		epout_desc = &hs_epout_desc;
		epin_desc = &hs_epin_desc;
#if defined(FEEDBACK_ENDPOINT_SUPPORT)
		epin_fback_desc = &hs_epin_fback_desc;
#endif
#if defined(VOLUME_MUTE_SUPPORT)
		ep_int_desc = &hs_ep_int_desc;
#endif
		break;
	default:
//		epout_desc = &ss_epout_desc;
		epin_desc = &uac_ss_streaming_ep;
//		epout_desc_comp = &ss_epout_desc_comp;
		epin_desc_comp = &uac_ss_streaming_comp;
#if defined(FEEDBACK_ENDPOINT_SUPPORT)
		epin_fback_desc = &ss_epin_fback_desc;
		epin_fback_desc_comp = &ss_epin_fback_desc_comp;
#endif
#if defined(VOLUME_MUTE_SUPPORT)
		ep_int_desc = &ss_ep_int_desc;
		ep_int_desc_comp = &ss_ep_int_desc_comp;
#endif
	}

	i = 0;
	headers[i++] = USBDHDR(&iad_desc);
	headers[i++] = USBDHDR(&std_ac_if_desc);
	headers[i++] = USBDHDR(&ac_hdr_desc);
//	if (EPIN_EN(opts))
//		headers[i++] = USBDHDR(&in_clk_src_desc);
//	if (EPOUT_EN(opts)) {
//		headers[i++] = USBDHDR(&out_clk_src_desc);
//		headers[i++] = USBDHDR(&usb_out_it_desc);
//	}

#if defined(VOLUME_MUTE_SUPPORT)
    if (FUOUT_EN(opts))
      headers[i++] = USBDHDR(out_feature_unit_desc);
    }
#endif

	if (EPIN_EN(opts)) {
		headers[i++] = USBDHDR(&io_in_it_desc);
		headers[i++] = USBDHDR(&usb_in_ot_desc);
#if defined(VOLUME_MUTE_SUPPORT)
		if (FUIN_EN(opts))
			headers[i++] = USBDHDR(in_feature_unit_desc);
#else
		headers[i++] = USBDHDR(&feature_unit_desc);
#endif
	}
	
//	if (EPOUT_EN(opts))
//		headers[i++] = USBDHDR(&io_out_ot_desc);

#if defined(VOLUME_MUTE_SUPPORT)
	if (FUOUT_EN(opts) || FUIN_EN(opts))
		headers[i++] = USBDHDR(ep_int_desc);
		if (ep_int_desc_comp)
			headers[i++] = USBDHDR(ep_int_desc_comp);
	}
#endif
//	if (EPOUT_EN(opts)) {
//		headers[i++] = USBDHDR(&std_as_out_if0_desc);
//		headers[i++] = USBDHDR(&std_as_out_if1_desc);
//		headers[i++] = USBDHDR(&as_out_hdr_desc);
//		headers[i++] = USBDHDR(&as_out_fmt1_desc);
//		headers[i++] = USBDHDR(epout_desc);
//		if (epout_desc_comp)
//			headers[i++] = USBDHDR(epout_desc_comp);
//
//		headers[i++] = USBDHDR(&as_iso_out_desc);
//#if defined(FEEDBACK_ENDPOINT_SUPPORT)
//		if (EPOUT_FBACK_IN_EN(opts)) {
//			headers[i++] = USBDHDR(epin_fback_desc);
//			if (epin_fback_desc_comp)
//				headers[i++] = USBDHDR(epin_fback_desc_comp);
//		}
//#endif
//	}
	if (EPIN_EN(opts)) {
		headers[i++] = USBDHDR(&std_as_in_if0_desc);
		headers[i++] = USBDHDR(&std_as_in_if1_desc);
		headers[i++] = USBDHDR(&as_in_hdr_desc);
		headers[i++] = USBDHDR(&as_in_fmt1_desc);
		headers[i++] = USBDHDR(epin_desc);
		if (epin_desc_comp)
			headers[i++] = USBDHDR(epin_desc_comp);

		headers[i++] = USBDHDR(&as_iso_in_desc);
	}
	headers[i] = NULL;
}

static void setup_descriptor(struct f_uac2_opts *opts)
{
	/* patch descriptors */
//	int i = 1; /* ID's start with 1 */

//	if (EPOUT_EN(opts))
//		usb_out_it_desc.bTerminalID = i++;
	if (EPIN_EN(opts))
		io_in_it_desc.bTerminalID = IO_IN_IT_ID;
//	if (EPOUT_EN(opts))
//		io_out_ot_desc.bTerminalID = i++;
	if (EPIN_EN(opts))
		usb_in_ot_desc.bTerminalID = USB_IN_OT_ID;
#if defined(VOLUME_MUTE_SUPPORT)
	if (FUOUT_EN(opts))
		out_feature_unit_desc->bUnitID = i++;
	if (FUIN_EN(opts))
		in_feature_unit_desc->bUnitID = i++;
#endif
//	if (EPOUT_EN(opts))
//		out_clk_src_desc.bClockID = i++;
//	if (EPIN_EN(opts))
//		in_clk_src_desc.bClockID = i++;

//	usb_out_it_desc.bCSourceID = out_clk_src_desc.bClockID;

#if defined(VOLUME_MUTE_SUPPORT)
	if (FUIN_EN(opts)) {
		usb_in_ot_desc.bSourceID = in_feature_unit_desc->bUnitID;
		in_feature_unit_desc->bSourceID = io_in_it_desc.bTerminalID;
	} else {
		usb_in_ot_desc.bSourceID = io_in_it_desc.bTerminalID;
	}
#else
	usb_in_ot_desc.bSourceID = io_in_it_desc.bTerminalID;
#endif

//	usb_in_ot_desc.bCSourceID = in_clk_src_desc.bClockID;
//	io_in_it_desc.bCSourceID = in_clk_src_desc.bClockID;
//	io_out_ot_desc.bCSourceID = out_clk_src_desc.bClockID;

#if defined(VOLUME_MUTE_SUPPORT)
	if (FUOUT_EN(opts)) {
		io_out_ot_desc.bSourceID = out_feature_unit_desc->bUnitID;
		out_feature_unit_desc->bSourceID = usb_out_it_desc.bTerminalID;
	} else {
		io_out_ot_desc.bSourceID = usb_out_it_desc.bTerminalID;
	}
#endif

//	io_out_ot_desc.bSourceID = usb_out_it_desc.bTerminalID;
//	as_out_hdr_desc.bTerminalLink = usb_out_it_desc.bTerminalID;
	as_in_hdr_desc.bTerminalLink = usb_in_ot_desc.bTerminalID;

	feature_unit_desc.bSourceID = io_in_it_desc.bTerminalID;

//	iad_desc.bInterfaceCount = 1;
	ac_hdr_desc.wTotalLength = cpu_to_le16(sizeof(ac_hdr_desc));

	if (EPIN_EN(opts)) {
		u16 len = le16_to_cpu(ac_hdr_desc.wTotalLength);

//		len += sizeof(in_clk_src_desc);
		len += sizeof(usb_in_ot_desc);

#if defined(VOLUME_MUTE_SUPPORT)
		if (FUIN_EN(opts))
			len += in_feature_unit_desc->bLength;
#endif

		len += sizeof(io_in_it_desc);
		len += feature_unit_desc.bLength;
		ac_hdr_desc.wTotalLength = cpu_to_le16(len);
//		iad_desc.bInterfaceCount++;
	}
//	if (EPOUT_EN(opts)) {
//		u16 len = le16_to_cpu(ac_hdr_desc.wTotalLength);
//
//		len += sizeof(out_clk_src_desc);
//		len += sizeof(usb_out_it_desc);
#if defined(VOLUME_MUTE_SUPPORT)
		if (FUOUT_EN(opts))
			len += out_feature_unit_desc->bLength;
#endif

//		len += sizeof(io_out_ot_desc);
//		ac_hdr_desc.wTotalLength = cpu_to_le16(len);
//		iad_desc.bInterfaceCount++;
//	}

	setup_headers(opts, fs_audio_desc, USB_SPEED_FULL);
	setup_headers(opts, hs_audio_desc, USB_SPEED_HIGH);
	setup_headers(opts, ss_audio_desc, USB_SPEED_SUPER);
}
#endif

static int afunc_validate_opts(struct g_audio *agdev, struct device *dev)
{
	struct f_uac2_opts *opts = g_audio_to_uac2_opts(agdev);

	if (!opts->p_chmask && !opts->c_chmask) {
		dev_err(dev, "Error: no playback and capture channels\n");
		return -EINVAL;
	} else if (opts->p_chmask & ~UAC2_CHANNEL_MASK) {
		dev_err(dev, "Error: unsupported playback channels mask\n");
		return -EINVAL;
	} else if (opts->c_chmask & ~UAC2_CHANNEL_MASK) {
		dev_err(dev, "Error: unsupported capture channels mask\n");
		return -EINVAL;
	} else if ((opts->p_ssize < 1) || (opts->p_ssize > 4)) {
		dev_err(dev, "Error: incorrect playback sample size\n");
		return -EINVAL;
	} else if ((opts->c_ssize < 1) || (opts->c_ssize > 4)) {
		dev_err(dev, "Error: incorrect capture sample size\n");
		return -EINVAL;
	} else if (!opts->p_srate) {
		dev_err(dev, "Error: incorrect playback sampling rate\n");
		return -EINVAL;
	} else if (!opts->c_srate) {
		dev_err(dev, "Error: incorrect capture sampling rate\n");
		return -EINVAL;
	}

#if defined(VOLUME_MUTE_SUPPORT)
	if (opts->p_volume_max <= opts->p_volume_min) {
		dev_err(dev, "Error: incorrect playback volume max/min\n");
			return -EINVAL;
	} else if (opts->c_volume_max <= opts->c_volume_min) {
		dev_err(dev, "Error: incorrect capture volume max/min\n");
			return -EINVAL;
	} else if (opts->p_volume_res <= 0) {
		dev_err(dev, "Error: negative/zero playback volume resolution\n");
			return -EINVAL;
	} else if (opts->c_volume_res <= 0) {
		dev_err(dev, "Error: negative/zero capture volume resolution\n");
			return -EINVAL;
	}

	if ((opts->p_volume_max - opts->p_volume_min) % opts->p_volume_res) {
		dev_err(dev, "Error: incorrect playback volume resolution\n");
			return -EINVAL;
	} else if ((opts->c_volume_max - opts->c_volume_min) % opts->c_volume_res) {
		dev_err(dev, "Error: incorrect capture volume resolution\n");
			return -EINVAL;
	}
#endif

	return 0;
}

static int
afunc_bind(struct usb_configuration *cfg, struct usb_function *fn)
{
	struct f_uac2 *uac2 = func_to_uac2(fn);
	struct g_audio *agdev = func_to_g_audio(fn);
	struct usb_composite_dev *cdev = cfg->cdev;
	struct usb_gadget *gadget = cdev->gadget;
	struct device *dev = &gadget->dev;
	struct f_uac2_opts *uac2_opts = g_audio_to_uac2_opts(agdev);
	//struct usb_string *us;
	int ret;
	u8 *sam_freq;
	unsigned int rate;

	ret = afunc_validate_opts(agdev, dev);
	if (ret) {
		 dev_err(dev, "%s:%d afunc_validate_opts Error!\n", __func__, __LINE__);
		return ret;
	}

	dev_info(dev,"%s: call\n", __func__);

/* Not use interface name.
	us = usb_gstrings_attach(cdev, fn_strings, ARRAY_SIZE(strings_fn));
	if (IS_ERR(us))
		return PTR_ERR(us);

#if defined(VOLUME_MUTE_SUPPORT)
	if (FUOUT_EN(uac2_opts)) {
		out_feature_unit_desc = build_fu_desc(uac2_opts->c_chmask);
		if (!out_feature_unit_desc)
			return -ENOMEM;
	}
	if (FUIN_EN(uac2_opts)) {
		in_feature_unit_desc = build_fu_desc(uac2_opts->p_chmask);
		if (!in_feature_unit_desc) {
			ret = -ENOMEM;
			goto err_free_fu;
		}
	}
#endif

	iad_desc.iFunction = us[STR_ASSOC].id;
	std_ac_if_desc.iInterface = us[STR_IF_CTRL].id;
	in_clk_src_desc.iClockSource = us[STR_CLKSRC_IN].id;
	out_clk_src_desc.iClockSource = us[STR_CLKSRC_OUT].id;
	usb_out_it_desc.iTerminal = us[STR_USB_IT].id;
	io_in_it_desc.iTerminal = us[STR_IO_IT].id;
	usb_in_ot_desc.iTerminal = us[STR_USB_OT].id;
	io_out_ot_desc.iTerminal = us[STR_IO_OT].id;
	std_as_out_if0_desc.iInterface = us[STR_AS_OUT_ALT0].id;
	std_as_out_if1_desc.iInterface = us[STR_AS_OUT_ALT1].id;
	std_as_in_if0_desc.iInterface = us[STR_AS_IN_ALT0].id;
	std_as_in_if1_desc.iInterface = us[STR_AS_IN_ALT1].id;

#if defined(VOLUME_MUTE_SUPPORT)
	if (FUOUT_EN(uac2_opts)) {
		u8 *i_feature = (u8 *)out_feature_unit_desc +
				out_feature_unit_desc->bLength - 1;
		*i_feature = us[STR_FU_OUT].id;
	}
	if (FUIN_EN(uac2_opts)) {
		u8 *i_feature = (u8 *)in_feature_unit_desc +
				in_feature_unit_desc->bLength - 1;
		*i_feature = us[STR_FU_IN].id;
	}
#else 
	feature_unit_desc.iFeature = us[STR_FEAT_DESC_0].id;  //ADD
#endif
*/

	/* Initialize the configurable parameters */
//	usb_out_it_desc.bNrChannels = num_channels(uac2_opts->c_chmask);
//	usb_out_it_desc.bmChannelConfig = cpu_to_le32(uac2_opts->c_chmask);
	io_in_it_desc.bNrChannels = num_channels(uac2_opts->p_chmask);
//	io_in_it_desc.bmChannelConfig = cpu_to_le32(uac2_opts->p_chmask);
//	as_out_hdr_desc.bNrChannels = num_channels(uac2_opts->c_chmask);
//	as_out_hdr_desc.bmChannelConfig = cpu_to_le32(uac2_opts->c_chmask);
//	as_in_hdr_desc.bNrChannels = num_channels(uac2_opts->p_chmask);
//	as_in_hdr_desc.bmChannelConfig = cpu_to_le32(uac2_opts->p_chmask);
//	as_out_fmt1_desc.bSubslotSize = uac2_opts->c_ssize;
//	as_out_fmt1_desc.bBitResolution = uac2_opts->c_ssize * 8;
//	as_in_fmt1_desc.bSubslotSize = uac2_opts->p_ssize;
	as_in_fmt1_desc.bBitResolution = uac2_opts->p_ssize * 8;

#if defined(VOLUME_MUTE_SUPPORT)
	if (FUOUT_EN(uac2_opts)) {
		__le32 *bma = (__le32 *)&out_feature_unit_desc->bmaControls[0];
		u32 control = 0;

		if (uac2_opts->c_mute_present)
			control |= CONTROL_RDWR << FU_MUTE_CTRL;
		if (uac2_opts->c_volume_present)
			control |= CONTROL_RDWR << FU_VOL_CTRL;
		*bma = cpu_to_le32(control);
	}
	if (FUIN_EN(uac2_opts)) {
		__le32 *bma = (__le32 *)&in_feature_unit_desc->bmaControls[0];
		u32 control = 0;

		if (uac2_opts->p_mute_present)
			control |= CONTROL_RDWR << FU_MUTE_CTRL;
		if (uac2_opts->p_volume_present)
			control |= CONTROL_RDWR << FU_VOL_CTRL;
		*bma = cpu_to_le32(control);
	}
#endif

	snprintf(clksrc_in, sizeof(clksrc_in), "%uHz", uac2_opts->p_srate);
//	snprintf(clksrc_out, sizeof(clksrc_out), "%uHz", uac2_opts->c_srate);
	rate = uac2_opts->p_srate;
	sam_freq = as_in_fmt1_desc.tSamFreq[0];
	memcpy(sam_freq, &rate, 3);

	ret = usb_interface_id(cfg, fn);
	if (ret < 0) {
		dev_err(dev, "%s:%d Error!\n", __func__, __LINE__);
#if defined(VOLUME_MUTE_SUPPORT)
		goto err_free_fu;
#else
		return ret;
#endif
	}
	iad_desc.bFirstInterface = ret;

	std_ac_if_desc.bInterfaceNumber = ret;
	uac2->ac_intf = ret;
	uac2->ac_alt = 0;

//	if (EPOUT_EN(uac2_opts)) {
//		ret = usb_interface_id(cfg, fn);
//		if (ret < 0) {
//			dev_err(dev, "%s:%d Error!\n", __func__, __LINE__);
//#if defined(VOLUME_MUTE_SUPPORT)
//			goto err_free_fu;
//#else
//			return ret;
//#endif
//		}
//		std_as_out_if0_desc.bInterfaceNumber = ret;
//		std_as_out_if1_desc.bInterfaceNumber = ret;
//		std_as_out_if1_desc.bNumEndpoints = 1;
//		uac2->as_out_intf = ret;
//		uac2->as_out_alt = 0;

//		if (EPOUT_FBACK_IN_EN(uac2_opts)) {
//			fs_epout_desc.bmAttributes =
//			  USB_ENDPOINT_XFER_ISOC | USB_ENDPOINT_SYNC_ASYNC;
//			hs_epout_desc.bmAttributes =
//			  USB_ENDPOINT_XFER_ISOC | USB_ENDPOINT_SYNC_ASYNC;
//			ss_epout_desc.bmAttributes =
//			  USB_ENDPOINT_XFER_ISOC | USB_ENDPOINT_SYNC_ASYNC;
//			std_as_out_if1_desc.bNumEndpoints++;
//		} else {
//			fs_epout_desc.bmAttributes =
//			  USB_ENDPOINT_XFER_ISOC | USB_ENDPOINT_SYNC_ADAPTIVE;
//			hs_epout_desc.bmAttributes =
//			  USB_ENDPOINT_XFER_ISOC | USB_ENDPOINT_SYNC_ADAPTIVE;
//			ss_epout_desc.bmAttributes =
//			  USB_ENDPOINT_XFER_ISOC | USB_ENDPOINT_SYNC_ADAPTIVE;
//		}
//	}

	if (EPIN_EN(uac2_opts)) {
		ret = usb_interface_id(cfg, fn);
		if (ret < 0) {
			dev_err(dev, "%s:%d Error!\n", __func__, __LINE__);
#if defined(VOLUME_MUTE_SUPPORT)
			goto err_free_fu;
#else
			return ret;
#endif
		}
		std_as_in_if0_desc.bInterfaceNumber = ret;
		std_as_in_if1_desc.bInterfaceNumber = ret;
		uac2->as_in_intf = ret;
		uac2->as_in_alt = 0;
	}

#if defined(VOLUME_MUTE_SUPPORT)
	if (FUOUT_EN(uac2_opts) || FUIN_EN(uac2_opts)) {
		uac2->int_ep = usb_ep_autoconfig(gadget, &fs_ep_int_desc);
		if (!uac2->int_ep) {
			dev_err(dev, "%s:%d Error!\n", __func__, __LINE__);
			ret = -ENODEV;
			goto err_free_fu;
		}

		std_ac_if_desc.bNumEndpoints = 1;
	}
#endif

	/* Calculate wMaxPacketSize according to audio bandwidth */
	ret = set_ep_max_packet_size(uac2_opts, &fs_epin_desc, USB_SPEED_FULL,
				     true);
	if (ret < 0) {
		dev_err(dev, "%s:%d Error!\n", __func__, __LINE__);
		return ret;
	}

//	ret = set_ep_max_packet_size(uac2_opts, &fs_epout_desc, USB_SPEED_FULL,
//				     false);
//	if (ret < 0) {
//		dev_err(dev, "%s:%d Error!\n", __func__, __LINE__);
//		return ret;
//	}

	ret = set_ep_max_packet_size(uac2_opts, &hs_epin_desc, USB_SPEED_HIGH,
				     true);
	if (ret < 0) {
		dev_err(dev, "%s:%d Error!\n", __func__, __LINE__);
		return ret;
	}

//	ret = set_ep_max_packet_size(uac2_opts, &hs_epout_desc, USB_SPEED_HIGH,
//				     false);
//	if (ret < 0) {
//		dev_err(dev, "%s:%d Error!\n", __func__, __LINE__);
//		return ret;
//	}

	ret = set_ep_max_packet_size(uac2_opts, &uac_ss_streaming_ep, USB_SPEED_SUPER,
				     true);
	if (ret < 0) {
		dev_err(dev, "%s:%d Error!\n", __func__, __LINE__);
		return ret;
	}


//	if (EPOUT_EN(uac2_opts)) {
//		agdev->out_ep = usb_ep_autoconfig(gadget, &fs_epout_desc);
//		if (!agdev->out_ep) {
//			dev_err(dev, "%s:%d Error!\n", __func__, __LINE__);
//#if defined(VOLUME_MUTE_SUPPORT)
//			ret = -ENODEV;
//			goto err_free_fu;
//#else
//			return -ENODEV;
//#endif
//		}
//#if defined(FEEDBACK_ENDPOINT_SUPPORT)
//		if (EPOUT_FBACK_IN_EN(uac2_opts)) {
//			agdev->in_ep_fback = usb_ep_autoconfig(gadget,
//						       &fs_epin_fback_desc);
//			if (!agdev->in_ep_fback) {
//				dev_err(dev, "%s:%d Error!\n",
//					__func__, __LINE__);
//#if defined(VOLUME_MUTE_SUPPORT)
//			    ret = -ENODEV;
//			    goto err_free_fu;
//#else
//				return -ENODEV;
//#endif
//			}
//		}
//#endif
//	}

	if (EPIN_EN(uac2_opts)) {
		if (gadget_is_superspeed(cfg->cdev->gadget)) {
			printk("super dayo\n");
			agdev->in_ep = usb_ep_autoconfig_ss(cdev->gadget, &uac_ss_streaming_ep,
					  &uac_ss_streaming_comp);
		}
		else {
			agdev->in_ep = usb_ep_autoconfig(gadget, &fs_epin_desc);
		}
		if (!agdev->in_ep) {
			dev_err(dev, "%s:%d Error!\n", __func__, __LINE__);
#if defined(VOLUME_MUTE_SUPPORT)
			    ret = -ENODEV;
			    goto err_free_fu;
#else
				return -ENODEV;
#endif
		}
	}

	agdev->in_ep_maxpsize = max_t(u16,
				le16_to_cpu(fs_epin_desc.wMaxPacketSize),
				le16_to_cpu(hs_epin_desc.wMaxPacketSize));
//	agdev->out_ep_maxpsize = max_t(u16,
//				le16_to_cpu(fs_epout_desc.wMaxPacketSize),
//				le16_to_cpu(hs_epout_desc.wMaxPacketSize));

	agdev->in_ep_maxpsize = max_t(u16, agdev->in_ep_maxpsize,
				le16_to_cpu(uac_ss_streaming_ep.wMaxPacketSize));
//	agdev->out_ep_maxpsize = max_t(u16, agdev->out_ep_maxpsize,
//				le16_to_cpu(ss_epout_desc.wMaxPacketSize));

#if defined(VOLUME_MUTE_SUPPORT)
	// HS and SS endpoint addresses are copied from autoconfigured FS descriptors
	hs_ep_int_desc.bEndpointAddress = fs_ep_int_desc.bEndpointAddress;
#endif

	uac_ss_streaming_comp.wBytesPerInterval = uac_ss_streaming_ep.wMaxPacketSize;
//	ss_epout_desc_comp.wBytesPerInterval = ss_epout_desc.wMaxPacketSize;

	/* set auto-configured Endpoint Address */
	if (gadget_is_superspeed(cfg->cdev->gadget)) {
		fs_epin_desc.bEndpointAddress = uac_ss_streaming_ep.bEndpointAddress;
		hs_epin_desc.bEndpointAddress = uac_ss_streaming_ep.bEndpointAddress;
	}
	else {
		hs_epin_desc.bEndpointAddress        = fs_epin_desc.bEndpointAddress;
		uac_ss_streaming_ep.bEndpointAddress = fs_epin_desc.bEndpointAddress;
	}
#if defined(FEEDBACK_ENDPOINT_SUPPORT)
	hs_epin_fback_desc.bEndpointAddress = fs_epin_fback_desc.bEndpointAddress;
	ss_epin_fback_desc.bEndpointAddress = fs_epin_fback_desc.bEndpointAddress;
#endif

#if defined(VOLUME_MUTE_SUPPORT)
	ss_ep_int_desc.bEndpointAddress = fs_ep_int_desc.bEndpointAddress;
#endif

	////
	uac_ss_streaming_ep.wMaxPacketSize = 0x0400;
	uac_ss_streaming_ep.bInterval = 4;
	uac_ss_streaming_comp.bmAttributes = 0;
	uac_ss_streaming_comp.bMaxBurst = 0;
	uac_ss_streaming_comp.wBytesPerInterval =0xC0;

	/////

	agdev->in_ep_maxpsize = uac_ss_streaming_ep.wMaxPacketSize;
    // 独自対応としてDescriptorの設定はデフォルト値を使う
//    setup_descriptor(uac2_opts);

//	ret = usb_assign_descriptors(fn, fs_audio_desc, hs_audio_desc, ss_audio_desc,
//				     ss_audio_desc);
	ret = usb_assign_descriptors(fn, fs_audio_desc, hs_audio_desc, ss_audio_desc,
				     NULL);

	if (ret)
#if defined(VOLUME_MUTE_SUPPORT)
		goto err_free_fu;
#else
		return ret;
#endif

	agdev->gadget = gadget;

	agdev->params.p_chmask = uac2_opts->p_chmask;
	agdev->params.p_srate = uac2_opts->p_srate;
	agdev->params.p_ssize = uac2_opts->p_ssize;
#if defined(VOLUME_MUTE_SUPPORT)
	if (FUIN_EN(uac2_opts)) {
		agdev->params.p_fu.id = USB_IN_FU_ID;
		agdev->params.p_fu.mute_present = uac2_opts->p_mute_present;
		agdev->params.p_fu.volume_present = uac2_opts->p_volume_present;
		agdev->params.p_fu.volume_min = uac2_opts->p_volume_min;
		agdev->params.p_fu.volume_max = uac2_opts->p_volume_max;
		agdev->params.p_fu.volume_res = uac2_opts->p_volume_res;
	}
#endif
	agdev->params.c_chmask = uac2_opts->c_chmask;
	agdev->params.c_srate = uac2_opts->c_srate;
	agdev->params.c_ssize = uac2_opts->c_ssize;
#if defined(VOLUME_MUTE_SUPPORT)
	if (FUOUT_EN(uac2_opts)) {
		agdev->params.c_fu.id = USB_OUT_FU_ID;
		agdev->params.c_fu.mute_present = uac2_opts->c_mute_present;
		agdev->params.c_fu.volume_present = uac2_opts->c_volume_present;
		agdev->params.c_fu.volume_min = uac2_opts->c_volume_min;
		agdev->params.c_fu.volume_max = uac2_opts->c_volume_max;
		agdev->params.c_fu.volume_res = uac2_opts->c_volume_res;
	}
#endif
	agdev->params.req_number = uac2_opts->req_number;
#if defined(FEEDBACK_ENDPOINT_SUPPORT)
	agdev->params.fb_max = uac2_opts->fb_max;
#endif

#if defined(VOLUME_MUTE_SUPPORT)
	if (FUOUT_EN(uac2_opts) || FUIN_EN(uac2_opts))
    agdev->notify = afunc_notify;
#endif

	ret = g_audio_setup(agdev, "UAC2 PCM", "UAC2_Gadget");

	if (ret)
		goto err_free_descs;
	return 0;

err_free_descs:
	usb_free_all_descriptors(fn);
	agdev->gadget = NULL;

#if defined(VOLUME_MUTE_SUPPORT)
err_free_fu:
	kfree(out_feature_unit_desc);
	out_feature_unit_desc = NULL;
	kfree(in_feature_unit_desc);
	in_feature_unit_desc = NULL;
	return ret;
#else
	return ret;
#endif
}

#if defined(VOLUME_MUTE_SUPPORT)
static void
afunc_notify_complete(struct usb_ep *_ep, struct usb_request *req)
{
	struct g_audio *agdev = req->context;
	struct f_uac2 *uac2 = func_to_uac2(&agdev->func);

	atomic_dec(&uac2->int_count);
	kfree(req->buf);
	usb_ep_free_request(_ep, req);
}

static int
afunc_notify(struct g_audio *agdev, int unit_id, int cs)
{
	struct f_uac2 *uac2 = func_to_uac2(&agdev->func);
	struct usb_request *req;
	struct uac2_interrupt_data_msg *msg;
	u16 w_index, w_value;
	int ret;

	if (!uac2->int_ep->enabled)
		return 0;

	if (atomic_inc_return(&uac2->int_count) > UAC2_DEF_INT_REQ_NUM) {
		atomic_dec(&uac2->int_count);
		return 0;
	}

	req = usb_ep_alloc_request(uac2->int_ep, GFP_ATOMIC);
	if (req == NULL) {
		ret = -ENOMEM;
		goto err_dec_int_count;
	}

	msg = kzalloc(sizeof(*msg), GFP_ATOMIC);
	if (msg == NULL) {
		ret = -ENOMEM;
		goto err_free_request;
	}

	w_index = unit_id << 8 | uac2->ac_intf;
	w_value = cs << 8;

	msg->bInfo = 0; /* Non-vendor, interface interrupt */
	msg->bAttribute = UAC2_CS_CUR;
	msg->wIndex = cpu_to_le16(w_index);
	msg->wValue = cpu_to_le16(w_value);

	req->length = sizeof(*msg);
	req->buf = msg;
	req->context = agdev;
	req->complete = afunc_notify_complete;

	ret = usb_ep_queue(uac2->int_ep, req, GFP_ATOMIC);

	if (ret)
		goto err_free_msg;

	return 0;

err_free_msg:
	kfree(msg);
err_free_request:
	usb_ep_free_request(uac2->int_ep, req);
err_dec_int_count:
	atomic_dec(&uac2->int_count);

	return ret;
}
#endif


static int
afunc_set_alt(struct usb_function *fn, unsigned intf, unsigned alt)
{
	struct usb_composite_dev *cdev = fn->config->cdev;
	struct f_uac2 *uac2 = func_to_uac2(fn);
#if defined(VOLUME_MUTE_SUPPORT)
	struct g_audio *agdev = func_to_g_audio(fn);
#endif
	struct usb_gadget *gadget = cdev->gadget;
	struct device *dev = &gadget->dev;
	int ret = 0;

	dev_info(dev,"%s:(IF%u,alt%u)\n", __func__,intf, alt);
	dev_info(dev,"%s:(UAC2 IF%u,alt%u)\n", __func__, uac2->ac_intf, alt);
	/* No i/f has more than 2 alt settings  */
	if (alt > 1) {
		dev_err(dev, "%s:%d Error!\n", __func__, __LINE__);
		return -EINVAL;
	}

	if (intf == uac2->ac_intf) {
		/* Control I/f has only 1 AltSetting - 0 */
		if (alt) {
			dev_err(dev, "%s:%d Error!\n", __func__, __LINE__);
			return -EINVAL;
		}

#if defined(VOLUME_MUTE_SUPPORT)
		/* restart interrupt endpoint */
		if (uac2->int_ep) {
			usb_ep_disable(uac2->int_ep);
			config_ep_by_speed(gadget, &agdev->func, uac2->int_ep);
			usb_ep_enable(uac2->int_ep);
		}
#endif

		return 0;
	}

//	if (intf == uac2->as_out_intf) {
//		uac2->as_out_alt = alt;
//
//		if (alt)
//			ret = u_audio_start_capture(&uac2->g_audio);
//		else
//			u_audio_stop_capture(&uac2->g_audio);
//	} else if (intf == uac2->as_in_intf) {
    if (intf == uac2->as_in_intf) {
		uac2->as_in_alt = alt;

		if (alt)
			ret = u_audio_start_playback(&uac2->g_audio, alt);
		else
			u_audio_stop_playback(&uac2->g_audio);
	} else {
		dev_err(dev, "%s:%d Error!\n", __func__, __LINE__);
		return -EINVAL;
	}

	return ret;
}

static int
afunc_get_alt(struct usb_function *fn, unsigned intf)
{
	struct f_uac2 *uac2 = func_to_uac2(fn);
	struct g_audio *agdev = func_to_g_audio(fn);

	if (intf == uac2->ac_intf)
		return uac2->ac_alt;
//	else if (intf == uac2->as_out_intf)
//		return uac2->as_out_alt;
	else if (intf == uac2->as_in_intf)
		return uac2->as_in_alt;
	else
		dev_err(&agdev->gadget->dev,
			"%s:%d Invalid Interface %d!\n",
			__func__, __LINE__, intf);

	return -EINVAL;
}

static void
afunc_disable(struct usb_function *fn)
{
	struct f_uac2 *uac2 = func_to_uac2(fn);

	uac2->as_in_alt = 0;
//	uac2->as_out_alt = 0;
//	u_audio_stop_capture(&uac2->g_audio);
	u_audio_stop_playback(&uac2->g_audio);
#if defined(VOLUME_MUTE_SUPPORT)
	if (uac2->int_ep)
		usb_ep_disable(uac2->int_ep);
#endif
}

static int
in_rq_cur(struct usb_function *fn, const struct usb_ctrlrequest *cr)
{
	struct usb_request *req = fn->config->cdev->req;
	struct g_audio *agdev = func_to_g_audio(fn);
#if defined(VOLUME_MUTE_SUPPORT)
	struct f_uac2_opts *opts = g_audio_to_uac2_opts(agdev);
#else
	struct f_uac2_opts *opts;
#endif
	u16 w_length = le16_to_cpu(cr->wLength);
	u16 w_index = le16_to_cpu(cr->wIndex);
	u16 w_value = le16_to_cpu(cr->wValue);
	u8 entity_id = (w_index >> 8) & 0xff;
	u8 control_selector = w_value >> 8;
	int value = -EOPNOTSUPP;
	int p_srate, c_srate;

#if !defined(VOLUME_MUTE_SUPPORT)
	opts = g_audio_to_uac2_opts(agdev);
#endif
	p_srate = opts->p_srate;
	c_srate = opts->c_srate;

#if defined(VOLUME_MUTE_SUPPORT)
	if ((entity_id == USB_IN_CLK_ID) || (entity_id == USB_OUT_CLK_ID)) {
		if (control_selector == UAC2_CS_CONTROL_SAM_FREQ) {
			struct cntrl_cur_lay3 c;

			memset(&c, 0, sizeof(struct cntrl_cur_lay3));

			if (entity_id == USB_IN_CLK_ID)
				c.dCUR = cpu_to_le32(p_srate);
			else if (entity_id == USB_OUT_CLK_ID)
				c.dCUR = cpu_to_le32(c_srate);

			value = min_t(unsigned int, w_length, sizeof(c));
			memcpy(req->buf, &c, value);
		} else if (control_selector == UAC2_CS_CONTROL_CLOCK_VALID) {
			*(u8 *)req->buf = 1;
			value = min_t(unsigned int, w_length, 1);
		} else {
			dev_err(&agdev->gadget->dev,
				"%s:%d control_selector=%d TODO!\n",
				__func__, __LINE__, control_selector);
		}
	} else if ((FUIN_EN(opts) && (entity_id == USB_IN_FU_ID)) ||
			(FUOUT_EN(opts) && (entity_id == USB_OUT_FU_ID))) {
		unsigned int is_playback = 0;

		if (FUIN_EN(opts) && (entity_id == USB_IN_FU_ID))
			is_playback = 1;

		if (control_selector == UAC_FU_MUTE) {
			unsigned int mute;

			u_audio_get_mute(agdev, is_playback, &mute);

			*(u8 *)req->buf = mute;
			value = min_t(unsigned int, w_length, 1);
		} else if (control_selector == UAC_FU_VOLUME) {
			struct cntrl_cur_lay2 c;
			s16 volume;

			memset(&c, 0, sizeof(struct cntrl_cur_lay2));

			u_audio_get_volume(agdev, is_playback, &volume);
			c.wCUR = cpu_to_le16(volume);

			value = min_t(unsigned int, w_length, sizeof(c));
			memcpy(req->buf, &c, value);
		} else {
			dev_err(&agdev->gadget->dev,
				"%s:%d control_selector=%d TODO!\n",
				__func__, __LINE__, control_selector);
		}
	} else {
		dev_err(&agdev->gadget->dev,
			"%s:%d entity_id=%d control_selector=%d TODO!\n",
			__func__, __LINE__, entity_id, control_selector);
	}
#else
	if (control_selector == UAC2_CS_CONTROL_SAM_FREQ) {
		struct cntrl_cur_lay3 c;
		memset(&c, 0, sizeof(struct cntrl_cur_lay3));

		if (entity_id == USB_IN_CLK_ID)
			c.dCUR = cpu_to_le32(p_srate);
		else if (entity_id == USB_OUT_CLK_ID)
			c.dCUR = cpu_to_le32(c_srate);

		value = min_t(unsigned, w_length, sizeof c);
		memcpy(req->buf, &c, value);
	} else if (control_selector == UAC2_CS_CONTROL_CLOCK_VALID) {
		*(u8 *)req->buf = 1;
		value = min_t(unsigned, w_length, 1);
	} else {
		dev_err(&agdev->gadget->dev,
			"%s:%d control_selector=%d TODO!\n",
			__func__, __LINE__, control_selector);
	}
#endif

	return value;
}

static int
in_rq_range(struct usb_function *fn, const struct usb_ctrlrequest *cr)
{
	struct usb_request *req = fn->config->cdev->req;
	struct g_audio *agdev = func_to_g_audio(fn);
#if defined(VOLUME_MUTE_SUPPORT)
	struct f_uac2_opts *opts = g_audio_to_uac2_opts(agdev);
#else
	struct f_uac2_opts *opts;
#endif
	u16 w_length = le16_to_cpu(cr->wLength);
	u16 w_index = le16_to_cpu(cr->wIndex);
	u16 w_value = le16_to_cpu(cr->wValue);
	u8 entity_id = (w_index >> 8) & 0xff;
	u8 control_selector = w_value >> 8;
#if !defined(VOLUME_MUTE_SUPPORT)
	struct cntrl_range_lay3 r;
#endif
	int value = -EOPNOTSUPP;
	int p_srate, c_srate;

#if !defined(VOLUME_MUTE_SUPPORT)
	opts = g_audio_to_uac2_opts(agdev);
#endif
	p_srate = opts->p_srate;
	c_srate = opts->c_srate;

#if defined(VOLUME_MUTE_SUPPORT)
	if ((entity_id == USB_IN_CLK_ID) || (entity_id == USB_OUT_CLK_ID)) {
		if (control_selector == UAC2_CS_CONTROL_SAM_FREQ) {
			struct cntrl_range_lay3 r;

			if (entity_id == USB_IN_CLK_ID)
				r.dMIN = cpu_to_le32(p_srate);
			else if (entity_id == USB_OUT_CLK_ID)
				r.dMIN = cpu_to_le32(c_srate);
			else
				return -EOPNOTSUPP;

			r.dMAX = r.dMIN;
			r.dRES = 0;
			r.wNumSubRanges = cpu_to_le16(1);

			value = min_t(unsigned int, w_length, sizeof(r));
			memcpy(req->buf, &r, value);
		} else {
			dev_err(&agdev->gadget->dev,
				"%s:%d control_selector=%d TODO!\n",
				__func__, __LINE__, control_selector);
		}
	} else if ((FUIN_EN(opts) && (entity_id == USB_IN_FU_ID)) ||
			(FUOUT_EN(opts) && (entity_id == USB_OUT_FU_ID))) {
		unsigned int is_playback = 0;

		if (FUIN_EN(opts) && (entity_id == USB_IN_FU_ID))
			is_playback = 1;

		if (control_selector == UAC_FU_VOLUME) {
			struct cntrl_range_lay2 r;
			s16 max_db, min_db, res_db;

			if (is_playback) {
				max_db = opts->p_volume_max;
				min_db = opts->p_volume_min;
				res_db = opts->p_volume_res;
			} else {
				max_db = opts->c_volume_max;
				min_db = opts->c_volume_min;
				res_db = opts->c_volume_res;
			}

			r.wMAX = cpu_to_le16(max_db);
			r.wMIN = cpu_to_le16(min_db);
			r.wRES = cpu_to_le16(res_db);
			r.wNumSubRanges = cpu_to_le16(1);

			value = min_t(unsigned int, w_length, sizeof(r));
			memcpy(req->buf, &r, value);
		} else {
			dev_err(&agdev->gadget->dev,
				"%s:%d control_selector=%d TODO!\n",
				__func__, __LINE__, control_selector);
		}
	} else {
		dev_err(&agdev->gadget->dev,
			"%s:%d entity_id=%d control_selector=%d TODO!\n",
			__func__, __LINE__, entity_id, control_selector);
	}
#else
	if (control_selector == UAC2_CS_CONTROL_SAM_FREQ) {
		if (entity_id == USB_IN_CLK_ID)
			r.dMIN = cpu_to_le32(p_srate);
		else if (entity_id == USB_OUT_CLK_ID)
			r.dMIN = cpu_to_le32(c_srate);
		else
			return -EOPNOTSUPP;

		r.dMAX = r.dMIN;
		r.dRES = 0;
		r.wNumSubRanges = cpu_to_le16(1);

		value = min_t(unsigned, w_length, sizeof r);
		memcpy(req->buf, &r, value);
	} else {
		dev_err(&agdev->gadget->dev,
			"%s:%d control_selector=%d TODO!\n",
			__func__, __LINE__, control_selector);
	}
#endif

	return value;
}

static int
ac_rq_in(struct usb_function *fn, const struct usb_ctrlrequest *cr)
{
	
	struct usb_request *req = fn->config->cdev->req;
	int value = -1;
	
	if (cr->bRequest == UAC2_CS_CUR)
		return in_rq_cur(fn, cr);
	else if (cr->bRequest == UAC2_CS_RANGE)
		return in_rq_range(fn, cr);
	else if (cr->bRequest >= 0x80  || cr->bRequest <=0x90)
	{
		switch(cr->bRequest){
		case 0x81:
			value = 0xffc0;
			break;
		case 0x82:
			value = 0xe3a0;
			break;
		case 0x83:
			value = 0xfff0;
			break;
		case 0x84:
			value = 0x0030;
			break;
		default:
			value = 1;
			break;
		}
		memcpy(req->buf, &value, cr->wLength);
		return cr->wLength;
	}
	else
		return -EOPNOTSUPP;
}

#if defined(VOLUME_MUTE_SUPPORT)
static void
out_rq_cur_complete(struct usb_ep *ep, struct usb_request *req)
{
	struct g_audio *agdev = req->context;
	struct usb_composite_dev *cdev = agdev->func.config->cdev;
	struct f_uac2_opts *opts = g_audio_to_uac2_opts(agdev);
	struct f_uac2 *uac2 = func_to_uac2(&agdev->func);
	struct usb_ctrlrequest *cr = &uac2->setup_cr;
	u16 w_index = le16_to_cpu(cr->wIndex);
	u16 w_value = le16_to_cpu(cr->wValue);
	u8 entity_id = (w_index >> 8) & 0xff;
	u8 control_selector = w_value >> 8;

	if (req->status != 0) {
		dev_dbg(&cdev->gadget->dev, "completion err %d\n", req->status);
		return;
	}

	if ((FUIN_EN(opts) && (entity_id == USB_IN_FU_ID)) ||
		(FUOUT_EN(opts) && (entity_id == USB_OUT_FU_ID))) {
		unsigned int is_playback = 0;

		if (FUIN_EN(opts) && (entity_id == USB_IN_FU_ID))
			is_playback = 1;

		if (control_selector == UAC_FU_MUTE) {
			u8 mute = *(u8 *)req->buf;

			u_audio_set_mute(agdev, is_playback, mute);

			return;
		} else if (control_selector == UAC_FU_VOLUME) {
			struct cntrl_cur_lay2 *c = req->buf;
			s16 volume;

			volume = le16_to_cpu(c->wCUR);
			u_audio_set_volume(agdev, is_playback, volume);

			return;
		} else {
			dev_err(&agdev->gadget->dev,
				"%s:%d control_selector=%d TODO!\n",
				__func__, __LINE__, control_selector);
			usb_ep_set_halt(ep);
		}
	}
}
#endif

static int
out_rq_cur(struct usb_function *fn, const struct usb_ctrlrequest *cr)
{
#if defined(VOLUME_MUTE_SUPPORT)
	struct usb_request *req = fn->config->cdev->req;
	struct g_audio *agdev = func_to_g_audio(fn);
	struct f_uac2_opts *opts = g_audio_to_uac2_opts(agdev);
	struct f_uac2 *uac2 = func_to_uac2(fn);
	u16 w_length = le16_to_cpu(cr->wLength);
	u16 w_index = le16_to_cpu(cr->wIndex);
	u16 w_value = le16_to_cpu(cr->wValue);
	u8 entity_id = (w_index >> 8) & 0xff;
	u8 control_selector = w_value >> 8;

	if ((entity_id == USB_IN_CLK_ID) || (entity_id == USB_OUT_CLK_ID)) {
		if (control_selector == UAC2_CS_CONTROL_SAM_FREQ)
			return w_length;
	} else if ((FUIN_EN(opts) && (entity_id == USB_IN_FU_ID)) ||
			(FUOUT_EN(opts) && (entity_id == USB_OUT_FU_ID))) {
		memcpy(&uac2->setup_cr, cr, sizeof(*cr));
		req->context = agdev;
		req->complete = out_rq_cur_complete;

		return w_length;
	} else {
		dev_err(&agdev->gadget->dev,
			"%s:%d entity_id=%d control_selector=%d TODO!\n",
			__func__, __LINE__, entity_id, control_selector);
	}
#else
	u16 w_length = le16_to_cpu(cr->wLength);
	u16 w_value = le16_to_cpu(cr->wValue);
	u8 control_selector = w_value >> 8;

	if (control_selector == UAC2_CS_CONTROL_SAM_FREQ)
		return w_length;
#endif

	return -EOPNOTSUPP;
}

static int
setup_rq_inf(struct usb_function *fn, const struct usb_ctrlrequest *cr)
{
	struct f_uac2 *uac2 = func_to_uac2(fn);
	struct g_audio *agdev = func_to_g_audio(fn);
	u16 w_index = le16_to_cpu(cr->wIndex);
	u8 intf = w_index & 0xff;

	if (intf != uac2->ac_intf) {
		dev_err(&agdev->gadget->dev,
			"%s:%d Error!\n", __func__, __LINE__);
		return -EOPNOTSUPP;
	}

	if (cr->bRequestType & USB_DIR_IN)
		return ac_rq_in(fn, cr);
	else if (cr->bRequest == UAC2_CS_CUR)
		return out_rq_cur(fn, cr);

	return -EOPNOTSUPP;
}

static int
afunc_setup(struct usb_function *fn, const struct usb_ctrlrequest *cr)
{
	struct usb_composite_dev *cdev = fn->config->cdev;
	struct g_audio *agdev = func_to_g_audio(fn);
	struct usb_request *req = cdev->req;
	u16 w_length = le16_to_cpu(cr->wLength);
	int value = -EOPNOTSUPP;

	/* Only Class specific requests are supposed to reach here */
	if ((cr->bRequestType & USB_TYPE_MASK) != USB_TYPE_CLASS)
		return -EOPNOTSUPP;

	if ((cr->bRequestType & USB_RECIP_MASK) == USB_RECIP_INTERFACE)
		value = setup_rq_inf(fn, cr);
	else
		dev_err(&agdev->gadget->dev, "%s:%d Error!\n",
				__func__, __LINE__);

	if (value >= 0) {
		req->length = value;
		req->zero = value < w_length;
		value = usb_ep_queue(cdev->gadget->ep0, req, GFP_ATOMIC);
		if (value < 0) {
			dev_err(&agdev->gadget->dev,
				"%s:%d Error!\n", __func__, __LINE__);
			req->status = 0;
		}
	}

	return value;
}

static inline struct f_uac2_opts *to_f_uac2_opts(struct config_item *item)
{
	return container_of(to_config_group(item), struct f_uac2_opts,
			    func_inst.group);
}

static void f_uac2_attr_release(struct config_item *item)
{
	struct f_uac2_opts *opts = to_f_uac2_opts(item);

	usb_put_function_instance(&opts->func_inst);
}

static struct configfs_item_operations f_uac2_item_ops = {
	.release	= f_uac2_attr_release,
};

#if defined(VOLUME_MUTE_SUPPORT)
#define uac2_kstrtou32 kstrtou32
#define uac2_kstrtos16 kstrtos16
#define uac2_kstrtobool(s, base, res) kstrtobool((s), (res))

static const char *u32_fmt = "%u\n";
static const char *s16_fmt = "%hd\n";
static const char *bool_fmt = "%u\n";
#endif

#if defined(VOLUME_MUTE_SUPPORT)
#define UAC2_ATTRIBUTE(type, name)					\
static ssize_t f_uac2_opts_##name##_show(struct config_item *item,	\
					 char *page)			\
{									\
	struct f_uac2_opts *opts = to_f_uac2_opts(item);		\
	int result;							\
									\
	mutex_lock(&opts->lock);					\
	result = sprintf(page, type##_fmt, opts->name);			\
	mutex_unlock(&opts->lock);					\
									\
	return result;							\
}									\
									\
static ssize_t f_uac2_opts_##name##_store(struct config_item *item,	\
					  const char *page, size_t len)	\
{									\
	struct f_uac2_opts *opts = to_f_uac2_opts(item);		\
	int ret;							\
	type num;							\
									\
	mutex_lock(&opts->lock);					\
	if (opts->refcnt) {						\
		ret = -EBUSY;						\
		goto end;						\
	}								\
									\
	ret = uac2_kstrto##type(page, 0, &num);				\
	if (ret)							\
		goto end;						\
									\
	opts->name = num;						\
	ret = len;							\
									\
end:									\
	mutex_unlock(&opts->lock);					\
	return ret;							\
}									\
									\
CONFIGFS_ATTR(f_uac2_opts_, name)
#else
#define UAC2_ATTRIBUTE(name)						\
static ssize_t f_uac2_opts_##name##_show(struct config_item *item,	\
					 char *page)			\
{									\
	struct f_uac2_opts *opts = to_f_uac2_opts(item);		\
	int result;							\
									\
	mutex_lock(&opts->lock);					\
	result = sprintf(page, "%u\n", opts->name);			\
	mutex_unlock(&opts->lock);					\
									\
	return result;							\
}									\
									\
static ssize_t f_uac2_opts_##name##_store(struct config_item *item,	\
					  const char *page, size_t len)	\
{									\
	struct f_uac2_opts *opts = to_f_uac2_opts(item);		\
	int ret;							\
	u32 num;							\
									\
	mutex_lock(&opts->lock);					\
	if (opts->refcnt) {						\
		ret = -EBUSY;						\
		goto end;						\
	}								\
									\
	ret = kstrtou32(page, 0, &num);					\
	if (ret)							\
		goto end;						\
									\
	opts->name = num;						\
	ret = len;							\
									\
end:									\
	mutex_unlock(&opts->lock);					\
	return ret;							\
}									\
									\
CONFIGFS_ATTR(f_uac2_opts_, name)
#endif

#if defined(FEEDBACK_ENDPOINT_SUPPORT)
#define UAC2_ATTRIBUTE_SYNC(name)					\
static ssize_t f_uac2_opts_##name##_show(struct config_item *item,	\
					 char *page)			\
{									\
	struct f_uac2_opts *opts = to_f_uac2_opts(item);		\
	int result;							\
	char *str;							\
									\
	mutex_lock(&opts->lock);					\
	switch (opts->name) {						\
	case USB_ENDPOINT_SYNC_ASYNC:					\
		str = "async";						\
		break;							\
	case USB_ENDPOINT_SYNC_ADAPTIVE:				\
		str = "adaptive";					\
		break;							\
	default:							\
		str = "unknown";					\
		break;							\
	}								\
	result = sprintf(page, "%s\n", str);				\
	mutex_unlock(&opts->lock);					\
									\
	return result;							\
}									\
									\
static ssize_t f_uac2_opts_##name##_store(struct config_item *item,	\
					  const char *page, size_t len)	\
{									\
	struct f_uac2_opts *opts = to_f_uac2_opts(item);		\
	int ret = 0;							\
									\
	mutex_lock(&opts->lock);					\
	if (opts->refcnt) {						\
		ret = -EBUSY;						\
		goto end;						\
	}								\
									\
	if (!strncmp(page, "async", 5))					\
		opts->name = USB_ENDPOINT_SYNC_ASYNC;			\
	else if (!strncmp(page, "adaptive", 8))				\
		opts->name = USB_ENDPOINT_SYNC_ADAPTIVE;		\
	else {								\
		ret = -EINVAL;						\
		goto end;						\
	}								\
									\
	ret = len;							\
									\
end:									\
	mutex_unlock(&opts->lock);					\
	return ret;							\
}									\
									\
CONFIGFS_ATTR(f_uac2_opts_, name)
#endif

#if defined(VOLUME_MUTE_SUPPORT)
UAC2_ATTRIBUTE(u32, p_chmask);
UAC2_ATTRIBUTE(u32, p_srate);
UAC2_ATTRIBUTE(u32, p_ssize);
UAC2_ATTRIBUTE(u32, c_chmask);
UAC2_ATTRIBUTE(u32, c_srate);
#if defined(FEEDBACK_ENDPOINT_SUPPORT)
UAC2_ATTRIBUTE_SYNC(c_sync);
#endif
UAC2_ATTRIBUTE(u32, c_ssize);
UAC2_ATTRIBUTE(u32, req_number);

UAC2_ATTRIBUTE(bool, p_mute_present);
UAC2_ATTRIBUTE(bool, p_volume_present);
UAC2_ATTRIBUTE(s16, p_volume_min);
UAC2_ATTRIBUTE(s16, p_volume_max);
UAC2_ATTRIBUTE(s16, p_volume_res);

UAC2_ATTRIBUTE(bool, c_mute_present);
UAC2_ATTRIBUTE(bool, c_volume_present);
UAC2_ATTRIBUTE(s16, c_volume_min);
UAC2_ATTRIBUTE(s16, c_volume_max);
UAC2_ATTRIBUTE(s16, c_volume_res);
UAC2_ATTRIBUTE(u32, fb_max);
#else
UAC2_ATTRIBUTE(p_chmask);
UAC2_ATTRIBUTE(p_srate);
UAC2_ATTRIBUTE(p_ssize);
UAC2_ATTRIBUTE(c_chmask);
UAC2_ATTRIBUTE(c_srate);
#if defined(FEEDBACK_ENDPOINT_SUPPORT)
UAC2_ATTRIBUTE_SYNC(c_sync);
#endif
UAC2_ATTRIBUTE(c_ssize);
UAC2_ATTRIBUTE(req_number);
#if defined(FEEDBACK_ENDPOINT_SUPPORT)
UAC2_ATTRIBUTE(fb_max);
#endif
#endif

static struct configfs_attribute *f_uac2_attrs[] = {
	&f_uac2_opts_attr_p_chmask,
	&f_uac2_opts_attr_p_srate,
	&f_uac2_opts_attr_p_ssize,
	&f_uac2_opts_attr_c_chmask,
	&f_uac2_opts_attr_c_srate,
	&f_uac2_opts_attr_c_ssize,
#if defined(FEEDBACK_ENDPOINT_SUPPORT)
	&f_uac2_opts_attr_c_sync,
#endif
	&f_uac2_opts_attr_req_number,
#if defined(FEEDBACK_ENDPOINT_SUPPORT)
	&f_uac2_opts_attr_fb_max,
#endif

#if defined(VOLUME_MUTE_SUPPORT)
	&f_uac2_opts_attr_p_mute_present,
	&f_uac2_opts_attr_p_volume_present,
	&f_uac2_opts_attr_p_volume_min,
	&f_uac2_opts_attr_p_volume_max,
	&f_uac2_opts_attr_p_volume_res,

	&f_uac2_opts_attr_c_mute_present,
	&f_uac2_opts_attr_c_volume_present,
	&f_uac2_opts_attr_c_volume_min,
	&f_uac2_opts_attr_c_volume_max,
	&f_uac2_opts_attr_c_volume_res,
#endif

	NULL,
};

static const struct config_item_type f_uac2_func_type = {
	.ct_item_ops	= &f_uac2_item_ops,
	.ct_attrs	= f_uac2_attrs,
	.ct_owner	= THIS_MODULE,
};

static void afunc_free_inst(struct usb_function_instance *f)
{
	struct f_uac2_opts *opts;

	opts = container_of(f, struct f_uac2_opts, func_inst);
	kfree(opts);
}

static struct usb_function_instance *afunc_alloc_inst(void)
{
	struct f_uac2_opts *opts;

	opts = kzalloc(sizeof(*opts), GFP_KERNEL);
	if (!opts)
		return ERR_PTR(-ENOMEM);

	mutex_init(&opts->lock);
	opts->func_inst.free_func_inst = afunc_free_inst;

	config_group_init_type_name(&opts->func_inst.group, "",
				    &f_uac2_func_type);

	opts->p_chmask = UAC2_DEF_PCHMASK;
	opts->p_srate = UAC2_DEF_PSRATE;
	opts->p_ssize = UAC2_DEF_PSSIZE;
	opts->c_chmask = UAC2_DEF_CCHMASK;
	opts->c_srate = UAC2_DEF_CSRATE;
	opts->c_ssize = UAC2_DEF_CSSIZE;
#if defined(FEEDBACK_ENDPOINT_SUPPORT)
	opts->c_sync = UAC2_DEF_CSYNC;
#endif

#if defined(VOLUME_MUTE_SUPPORT)
	opts->p_mute_present = UAC2_DEF_MUTE_PRESENT;
	opts->p_volume_present = UAC2_DEF_VOLUME_PRESENT;
	opts->p_volume_min = UAC2_DEF_MIN_DB;
	opts->p_volume_max = UAC2_DEF_MAX_DB;
	opts->p_volume_res = UAC2_DEF_RES_DB;

	opts->c_mute_present = UAC2_DEF_MUTE_PRESENT;
	opts->c_volume_present = UAC2_DEF_VOLUME_PRESENT;
	opts->c_volume_min = UAC2_DEF_MIN_DB;
	opts->c_volume_max = UAC2_DEF_MAX_DB;
	opts->c_volume_res = UAC2_DEF_RES_DB;
#endif

	opts->req_number = UAC2_DEF_REQ_NUM;
#if defined(FEEDBACK_ENDPOINT_SUPPORT)
	opts->fb_max = UAC2_DEF_FB_MAX;
#endif
	return &opts->func_inst;
}

static void afunc_free(struct usb_function *f)
{
	struct g_audio *agdev;
	struct f_uac2_opts *opts;

	agdev = func_to_g_audio(f);
	opts = container_of(f->fi, struct f_uac2_opts, func_inst);
	kfree(agdev);
	mutex_lock(&opts->lock);
	--opts->refcnt;
	mutex_unlock(&opts->lock);
}

static void afunc_unbind(struct usb_configuration *c, struct usb_function *f)
{
	struct g_audio *agdev = func_to_g_audio(f);
	struct usb_composite_dev *cdev = f->config->cdev;
	struct usb_gadget *gadget = cdev->gadget;
	struct device *dev = &gadget->dev;

	dev_info(dev,"%s: call \n", __func__);

	g_audio_cleanup(agdev);
	usb_free_all_descriptors(f);

	agdev->gadget = NULL;

#if defined(VOLUME_MUTE_SUPPORT)
	kfree(out_feature_unit_desc);
	out_feature_unit_desc = NULL;
	kfree(in_feature_unit_desc);
	in_feature_unit_desc = NULL;
#endif
}

static struct usb_function *afunc_alloc(struct usb_function_instance *fi)
{
	struct f_uac2	*uac2;
	struct f_uac2_opts *opts;

	uac2 = kzalloc(sizeof(*uac2), GFP_KERNEL);
	if (uac2 == NULL)
		return ERR_PTR(-ENOMEM);

	opts = container_of(fi, struct f_uac2_opts, func_inst);
	mutex_lock(&opts->lock);
	++opts->refcnt;
	mutex_unlock(&opts->lock);

	uac2->g_audio.func.name = "uac2_func";
	uac2->g_audio.func.bind = afunc_bind;
	uac2->g_audio.func.unbind = afunc_unbind;
	uac2->g_audio.func.set_alt = afunc_set_alt;
	uac2->g_audio.func.get_alt = afunc_get_alt;
	uac2->g_audio.func.disable = afunc_disable;
	uac2->g_audio.func.setup = afunc_setup;
	uac2->g_audio.func.free_func = afunc_free;

	return &uac2->g_audio.func;
}

DECLARE_USB_FUNCTION_INIT(uac2, afunc_alloc_inst, afunc_alloc);
MODULE_LICENSE("GPL");
#if defined(VOLUME_MUTE_SUPPORT)
MODULE_AUTHOR("Ruslan Bilovol");
#endif
MODULE_AUTHOR("Yadwinder Singh");
MODULE_AUTHOR("Jaswinder Singh");
