// SPDX-License-Identifier: GPL-2.0+
/*
 *	webcam.c -- USB webcam gadget driver
 *
 *	Copyright (C) 2009-2010
 *	    Laurent Pinchart (laurent.pinchart@ideasonboard.com)
 */

#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/module.h>
#include <linux/usb/video.h>

#include "uvc.h"
#include "u_uvc.h"

USB_GADGET_COMPOSITE_OPTIONS();
#ifndef USB_VIDEO_ONLY /* not combine uac */
//add audio

#ifndef CONFIG_GADGET_UAC1
#include "u_uac2.h"
/* Playback(USB-IN) Default Stereo - Fl/Fr */
static int p_chmask = UAC2_DEF_PCHMASK;
module_param(p_chmask, uint, S_IRUGO);
MODULE_PARM_DESC(p_chmask, "Playback Channel Mask");

/* Playback Default 48 KHz */
static int p_srate = UAC2_DEF_PSRATE;
module_param(p_srate, uint, S_IRUGO);
MODULE_PARM_DESC(p_srate, "Playback Sampling Rate");

/* Playback Default 16bits/sample */
static int p_ssize = UAC2_DEF_PSSIZE;
module_param(p_ssize, uint, S_IRUGO);
MODULE_PARM_DESC(p_ssize, "Playback Sample Size(bytes)");

/* Capture(USB-OUT) Default Stereo - Fl/Fr */
static int c_chmask = UAC2_DEF_CCHMASK;
module_param(c_chmask, uint, S_IRUGO);
MODULE_PARM_DESC(c_chmask, "Capture Channel Mask");

/* Capture Default 64 KHz */
static int c_srate = UAC2_DEF_CSRATE;
module_param(c_srate, uint, S_IRUGO);
MODULE_PARM_DESC(c_srate, "Capture Sampling Rate");

/* Capture Default 16bits/sample */
static int c_ssize = UAC2_DEF_CSSIZE;
module_param(c_ssize, uint, S_IRUGO);
MODULE_PARM_DESC(c_ssize, "Capture Sample Size(bytes)");
#else
#include "u_uac1.h"

static char *fn_play = FILE_PCM_PLAYBACK;
module_param(fn_play, charp, S_IRUGO);
MODULE_PARM_DESC(fn_play, "Playback PCM device file name");

static char *fn_cap = FILE_PCM_CAPTURE;
module_param(fn_cap, charp, S_IRUGO);
MODULE_PARM_DESC(fn_cap, "Capture PCM device file name");

static char *fn_cntl = FILE_CONTROL;
module_param(fn_cntl, charp, S_IRUGO);
MODULE_PARM_DESC(fn_cntl, "Control device file name");

static int req_buf_size = UAC1_OUT_EP_MAX_PACKET_SIZE;
module_param(req_buf_size, int, S_IRUGO);
MODULE_PARM_DESC(req_buf_size, "ISO OUT endpoint request buffer size");

static int req_count = UAC1_REQ_COUNT;
module_param(req_count, int, S_IRUGO);
MODULE_PARM_DESC(req_count, "ISO OUT endpoint request count");

static int audio_buf_size = UAC1_AUDIO_BUF_SIZE;
module_param(audio_buf_size, int, S_IRUGO);
MODULE_PARM_DESC(audio_buf_size, "Audio buffer size");
#endif

#ifndef CONFIG_GADGET_UAC1
static struct usb_function_instance *fi_uac2;
static struct usb_function *f_uac2;
#else
static struct usb_function_instance *fi_uac1;
static struct usb_function *f_uac1;
#endif
#endif

static int yuv_frame_num = 0;
static int yuv_frame_index[YUV_FRAME_NUM];
module_param_array(yuv_frame_index, int, &yuv_frame_num, S_IRUGO|S_IWUSR);

static int mjpg_frame_num = 0;
static int mjpg_frame_index[MJPG_FRAME_NUM];
module_param_array(mjpg_frame_index, int, &mjpg_frame_num, S_IRUGO|S_IWUSR);

//add audio

/*-------------------------------------------------------------------------*/

/* module parameters specific to the Video streaming endpoint */
static unsigned int streaming_interval = 1;
module_param(streaming_interval, uint, S_IRUGO|S_IWUSR);
MODULE_PARM_DESC(streaming_interval, "1 - 16");

static unsigned int streaming_maxpacket = 1024;
module_param(streaming_maxpacket, uint, S_IRUGO|S_IWUSR);
MODULE_PARM_DESC(streaming_maxpacket, "1 - 1023 (FS), 1 - 3072 (hs/ss)");

static unsigned int streaming_maxburst = 0;
module_param(streaming_maxburst, uint, S_IRUGO|S_IWUSR);
MODULE_PARM_DESC(streaming_maxburst, "0 - 15 (ss only)");

/* --------------------------------------------------------------------------
 * Device descriptor
 */

#define WEBCAM_VENDOR_ID		0x054C	/* Linux Foundation */
#define WEBCAM_PRODUCT_ID		0x0102	/* Webcam A/V gadget */
#define WEBCAM_DEVICE_BCD		0x0010	/* 0.10 */

static char webcam_vendor_label[] = "Linux Foundation";
static char webcam_product_label[] = "Webcam gadget";
static char webcam_serial_label[] = "000000000000";
static char webcam_config_label[] = "Video";

/* string IDs are assigned dynamically */

#define STRING_DESCRIPTION_IDX		USB_GADGET_FIRST_AVAIL_IDX

static struct usb_string webcam_strings[] = {
	[USB_GADGET_MANUFACTURER_IDX].s = webcam_vendor_label,
	[USB_GADGET_PRODUCT_IDX].s = webcam_product_label,
	[USB_GADGET_SERIAL_IDX].s = webcam_serial_label,
/* Not use configuration name.
	[STRING_DESCRIPTION_IDX].s = webcam_config_label,
*/
	{  }
};

static struct usb_gadget_strings webcam_stringtab = {
	.language = 0x0409,	/* en-us */
	.strings = webcam_strings,
};

static struct usb_gadget_strings *webcam_device_strings[] = {
	&webcam_stringtab,
	NULL,
};

static struct usb_function_instance *fi_uvc;
static struct usb_function *f_uvc;

static struct usb_device_descriptor webcam_device_descriptor = {
	.bLength		= USB_DT_DEVICE_SIZE,
	.bDescriptorType	= USB_DT_DEVICE,
	/* .bcdUSB = DYNAMIC */
	.bDeviceClass		= USB_CLASS_MISC,
	.bDeviceSubClass	= 0x02,
	.bDeviceProtocol	= 0x01,
	.bMaxPacketSize0	= 0, /* dynamic */
	.idVendor		= cpu_to_le16(WEBCAM_VENDOR_ID),
	.idProduct		= cpu_to_le16(WEBCAM_PRODUCT_ID),
	.bcdDevice		= cpu_to_le16(WEBCAM_DEVICE_BCD),
	.iManufacturer		= 0, /* dynamic */
	.iProduct		= 0, /* dynamic */
	.iSerialNumber		= 0, /* dynamic */
	.bNumConfigurations	= 0, /* dynamic */
};

DECLARE_UVC_HEADER_DESCRIPTOR(1);

static const struct UVC_HEADER_DESCRIPTOR(1) uvc_control_header = {
	.bLength		= UVC_DT_HEADER_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VC_HEADER,
	.bcdUVC			= cpu_to_le16(0x0110),
	.wTotalLength		= 0, /* dynamic */
	.dwClockFrequency	= cpu_to_le32(0x0BB80),
	.bInCollection		= 0, /* dynamic */
	.baInterfaceNr[0]	= 0, /* dynamic */
};

static const struct uvc_camera_terminal_descriptor uvc_camera_terminal = {
	.bLength		= UVC_DT_CAMERA_TERMINAL_SIZE(3),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VC_INPUT_TERMINAL,
	.bTerminalID		= 1,
	.wTerminalType		= cpu_to_le16(0x0201),
	.bAssocTerminal		= 0,
	.iTerminal		= 0,
	.wObjectiveFocalLengthMin	= cpu_to_le16(0),
	.wObjectiveFocalLengthMax	= cpu_to_le16(0),
	.wOcularFocalLength		= cpu_to_le16(0),
	.bControlSize		= 3,
	.bmControls[0]		= 0, //2
	.bmControls[1]		= 0,
	.bmControls[2]		= 0,
};

static const struct uvc_processing_unit_descriptor uvc_processing = {
	.bLength		= UVC_DT_PROCESSING_UNIT_SIZE(2),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VC_PROCESSING_UNIT,
	.bUnitID		= 2,
	.bSourceID		= 1,
	.wMaxMultiplier		= cpu_to_le16(16*1024),
	.bControlSize		= 2,
	.bmControls[0]		= 0,
	.bmControls[1]		= 0,
	.iProcessing		= 0,
	.bmVideoStandards	= 0,
};

static const struct uvc_output_terminal_descriptor uvc_output_terminal = {
	.bLength		= UVC_DT_OUTPUT_TERMINAL_SIZE,
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VC_OUTPUT_TERMINAL,
	.bTerminalID		= 3,
	.wTerminalType		= cpu_to_le16(0x0101),
	.bAssocTerminal		= 0,
	.bSourceID		= 2,
	.iTerminal		= 0,
};

DECLARE_UVC_INPUT_HEADER_DESCRIPTOR(1, 1);

static const struct UVC_INPUT_HEADER_DESCRIPTOR(1, 1) uvc_input_header = {
	.bLength		= UVC_DT_INPUT_HEADER_SIZE(1, 1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_INPUT_HEADER,
	.bNumFormats		= 1,
	.wTotalLength		= 0, /* dynamic */
	.bEndpointAddress	= 0, /* dynamic */
	.bmInfo			= 0,
	.bTerminalLink		= 3,
	.bStillCaptureMethod	= 0,
	.bTriggerSupport	= 0,
	.bTriggerUsage		= 0,
	.bControlSize		= 1,
	.bmaControls[0][0]	= 0,
};

DECLARE_UVC_INPUT_HEADER_DESCRIPTOR(1, 2);
static const struct UVC_INPUT_HEADER_DESCRIPTOR(1, 2) uvc_input_header_ss = {
	.bLength		= UVC_DT_INPUT_HEADER_SIZE(1, 2),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_INPUT_HEADER,
	.bNumFormats		= 2,
	.wTotalLength		= 0, /* dynamic */
	.bEndpointAddress	= 0, /* dynamic */
	.bmInfo			= 0,
	.bTerminalLink		= 3,
	.bStillCaptureMethod	= 0,
	.bTriggerSupport	= 0,
	.bTriggerUsage		= 0,
	.bControlSize		= 1,
	.bmaControls[0][0]	= 0,
	.bmaControls[1][0]	= 0,
};
#if 0
// Not used in HS.
static struct uvc_format_uncompressed uvc_format_yuv = {
	.bLength		= UVC_DT_FORMAT_UNCOMPRESSED_SIZE,
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FORMAT_UNCOMPRESSED,
	.bFormatIndex		= 2,
	.bNumFrameDescriptors	= 1,
	.guidFormat		=
		{ 'N',  'V',  '1',  '2', 0x00, 0x00, 0x10, 0x00,
		 0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71},
	.bBitsPerPixel		= 12,
	.bDefaultFrameIndex	= 1,
	.bAspectRatioX		= 0,
	.bAspectRatioY		= 0,
	.bmInterfaceFlags	= 0,
	.bCopyProtect		= 0,
};
#endif
static struct uvc_format_uncompressed uvc_format_yuv_ss = {
	.bLength		= UVC_DT_FORMAT_UNCOMPRESSED_SIZE,
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FORMAT_UNCOMPRESSED,
	.bFormatIndex		= 1,
	.bNumFrameDescriptors	= 1,
	.guidFormat		=
		{ 'N',  'V',  '1',  '2', 0x00, 0x00, 0x10, 0x00,
		 0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71},
	.bBitsPerPixel		= 12,
	.bDefaultFrameIndex	= 1,
	.bAspectRatioX		= 0,
	.bAspectRatioY		= 0,
	.bmInterfaceFlags	= 0,
	.bCopyProtect		= 0,
};

DECLARE_UVC_FRAME_UNCOMPRESSED(1);

static const struct UVC_FRAME_UNCOMPRESSED(1) uvc_frame_yuv_720p_30 = {
	.bLength		= UVC_DT_FRAME_UNCOMPRESSED_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_UNCOMPRESSED,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(1280),
	.wHeight		= cpu_to_le16(720),
	.dwMinBitRate		= cpu_to_le32(442368000),
	.dwMaxBitRate		= cpu_to_le32(442368000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(1843200),
	.dwDefaultFrameInterval	= cpu_to_le32(333333),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(333333),
};

static const struct UVC_FRAME_UNCOMPRESSED(1) uvc_frame_yuv_720p_25 = {
	.bLength		= UVC_DT_FRAME_UNCOMPRESSED_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_UNCOMPRESSED,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(1280),
	.wHeight		= cpu_to_le16(720),
	.dwMinBitRate		= cpu_to_le32(368640000),
	.dwMaxBitRate		= cpu_to_le32(368640000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(1843200),
	.dwDefaultFrameInterval	= cpu_to_le32(400000),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(400000),
};

static const struct UVC_FRAME_UNCOMPRESSED(1) uvc_frame_yuv_1080p_30 = {
	.bLength		= UVC_DT_FRAME_UNCOMPRESSED_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_UNCOMPRESSED,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(1920),
	.wHeight		= cpu_to_le16(1080),
	.dwMinBitRate		= cpu_to_le32(995328000),
	.dwMaxBitRate		= cpu_to_le32(995328000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(1843200),
	.dwDefaultFrameInterval	= cpu_to_le32(333333),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(333333),
};

static const struct UVC_FRAME_UNCOMPRESSED(1) uvc_frame_yuv_1080p_25 = {
	.bLength		= UVC_DT_FRAME_UNCOMPRESSED_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_UNCOMPRESSED,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(1920),
	.wHeight		= cpu_to_le16(1080),
	.dwMinBitRate		= cpu_to_le32(829440000),
	.dwMaxBitRate		= cpu_to_le32(829440000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(1843200),
	.dwDefaultFrameInterval	= cpu_to_le32(400000),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(400000),
};


static const struct UVC_FRAME_UNCOMPRESSED(1) uvc_frame_yuv_1080p_60 = {
	.bLength		= UVC_DT_FRAME_UNCOMPRESSED_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_UNCOMPRESSED,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(1920),
	.wHeight		= cpu_to_le16(1080),
	.dwMinBitRate		= cpu_to_le32(1990656000),
	.dwMaxBitRate		= cpu_to_le32(1990656000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(1843200),
	.dwDefaultFrameInterval	= cpu_to_le32(166666),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(166666),
};

static const struct UVC_FRAME_UNCOMPRESSED(1) uvc_frame_yuv_1080p_50 = {
	.bLength		= UVC_DT_FRAME_UNCOMPRESSED_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_UNCOMPRESSED,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(1920),
	.wHeight		= cpu_to_le16(1080),
	.dwMinBitRate		= cpu_to_le32(1658880000),
	.dwMaxBitRate		= cpu_to_le32(1658880000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(1843200),
	.dwDefaultFrameInterval	= cpu_to_le32(200000),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(200000),
};

static const struct UVC_FRAME_UNCOMPRESSED(1) uvc_frame_yuv_2160p_30 = {
	.bLength		= UVC_DT_FRAME_UNCOMPRESSED_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_UNCOMPRESSED,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(3840),
	.wHeight		= cpu_to_le16(2160),
	.dwMinBitRate		= cpu_to_le32(3981312000),
	.dwMaxBitRate		= cpu_to_le32(3981312000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(8294400),
	.dwDefaultFrameInterval	= cpu_to_le32(333333),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(333333),
};

static const struct UVC_FRAME_UNCOMPRESSED(1) uvc_frame_yuv_2160p_25 = {
	.bLength		= UVC_DT_FRAME_UNCOMPRESSED_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_UNCOMPRESSED,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(3840),
	.wHeight		= cpu_to_le16(2160),
	.dwMinBitRate		= cpu_to_le32(3317760000),
	.dwMaxBitRate		= cpu_to_le32(3317760000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(8294400),
	.dwDefaultFrameInterval	= cpu_to_le32(400000),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(400000),
};
static const struct UVC_FRAME_UNCOMPRESSED(1) uvc_frame_yuv_2160p_15 = {
	.bLength		= UVC_DT_FRAME_UNCOMPRESSED_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_UNCOMPRESSED,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(3840),
	.wHeight		= cpu_to_le16(2160),
	.dwMinBitRate		= cpu_to_le32(1990656000),
	.dwMaxBitRate		= cpu_to_le32(1990656000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(614400),
	.dwDefaultFrameInterval	= cpu_to_le32(666666),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(666666),
};

static const struct UVC_FRAME_UNCOMPRESSED(1) uvc_frame_yuv_2160p_12_5 = {
	.bLength		= UVC_DT_FRAME_UNCOMPRESSED_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_UNCOMPRESSED,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(3840),
	.wHeight		= cpu_to_le16(2160),
	.dwMinBitRate		= cpu_to_le32(1658880000),
	.dwMaxBitRate		= cpu_to_le32(1658880000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(614400),
	.dwDefaultFrameInterval	= cpu_to_le32(800000),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(800000),
};

static struct uvc_format_mjpeg uvc_format_mjpg = {
	.bLength		= UVC_DT_FORMAT_MJPEG_SIZE,
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FORMAT_MJPEG,
	.bFormatIndex		= 1,
	.bNumFrameDescriptors	= 1,
	.bmFlags		= 0,
	.bDefaultFrameIndex	= 1,
	.bAspectRatioX		= 0,
	.bAspectRatioY		= 0,
	.bmInterfaceFlags	= 0,
	.bCopyProtect		= 0,
};
static struct uvc_format_mjpeg uvc_format_mjpg_ss = {
	.bLength		= UVC_DT_FORMAT_MJPEG_SIZE,
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FORMAT_MJPEG,
	.bFormatIndex		= 2,
	.bNumFrameDescriptors	= 1,
	.bmFlags		= 0,
	.bDefaultFrameIndex	= 1,
	.bAspectRatioX		= 0,
	.bAspectRatioY		= 0,
	.bmInterfaceFlags	= 0,
	.bCopyProtect		= 0,
};

DECLARE_UVC_FRAME_MJPEG(1);

static const struct UVC_FRAME_MJPEG(1) uvc_frame_mjpg_720p_30 = {
	.bLength		=   UVC_DT_FRAME_MJPEG_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_MJPEG,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(1280),
	.wHeight		= cpu_to_le16(720),
	.dwMinBitRate		= cpu_to_le32(442368000),
	.dwMaxBitRate		= cpu_to_le32(442368000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(2138400),
	.dwDefaultFrameInterval	= cpu_to_le32(333333),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(333333),
};

static const struct UVC_FRAME_MJPEG(1) uvc_frame_mjpg_720p_25 = {
	.bLength		=   UVC_DT_FRAME_MJPEG_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_MJPEG,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(1280),
	.wHeight		= cpu_to_le16(720),
	.dwMinBitRate		= cpu_to_le32(368640000),
	.dwMaxBitRate		= cpu_to_le32(368640000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(2138400),
	.dwDefaultFrameInterval	= cpu_to_le32(400000),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(400000),
};

static const struct UVC_FRAME_MJPEG(1) uvc_frame_mjpg_1080p_30 = {
	.bLength		=   UVC_DT_FRAME_MJPEG_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_MJPEG,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(1920),
	.wHeight		= cpu_to_le16(1080),
	.dwMinBitRate		= cpu_to_le32(995328000),
	.dwMaxBitRate		= cpu_to_le32(995328000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(2138400),
	.dwDefaultFrameInterval	= cpu_to_le32(333333),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(333333),
};

static const struct UVC_FRAME_MJPEG(1) uvc_frame_mjpg_1080p_25 = {
	.bLength		=   UVC_DT_FRAME_MJPEG_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_MJPEG,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(1920),
	.wHeight		= cpu_to_le16(1080),
	.dwMinBitRate		= cpu_to_le32(829440000),
	.dwMaxBitRate		= cpu_to_le32(829440000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(2138400),
	.dwDefaultFrameInterval	= cpu_to_le32(400000),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(400000),
};

static const struct UVC_FRAME_MJPEG(1) uvc_frame_mjpg_1080p_60 = {
	.bLength		=   UVC_DT_FRAME_MJPEG_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_MJPEG,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(1920),
	.wHeight		= cpu_to_le16(1080),
	.dwMinBitRate		= cpu_to_le32(1990656000),
	.dwMaxBitRate		= cpu_to_le32(1990656000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(614400),
	.dwDefaultFrameInterval	= cpu_to_le32(166666),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(166666),
};

static const struct UVC_FRAME_MJPEG(1) uvc_frame_mjpg_1080p_50 = {
	.bLength		=   UVC_DT_FRAME_MJPEG_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_MJPEG,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(1920),
	.wHeight		= cpu_to_le16(1080),
	.dwMinBitRate		= cpu_to_le32(1658880000),
	.dwMaxBitRate		= cpu_to_le32(1658880000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(614400),
	.dwDefaultFrameInterval	= cpu_to_le32(200000),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(200000),
};

static const struct UVC_FRAME_MJPEG(1) uvc_frame_mjpg_2160p_15 = {
	.bLength		=   UVC_DT_FRAME_MJPEG_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_MJPEG,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(3840),
	.wHeight		= cpu_to_le16(2160),
	.dwMinBitRate		= cpu_to_le32(1990656000),
	.dwMaxBitRate		= cpu_to_le32(1990656000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(17694720),
	.dwDefaultFrameInterval	= cpu_to_le32(666666),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(666666),
};

static const struct UVC_FRAME_MJPEG(1) uvc_frame_mjpg_2160p_12_5 = {
	.bLength		=   UVC_DT_FRAME_MJPEG_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_MJPEG,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(3840),
	.wHeight		= cpu_to_le16(2160),
	.dwMinBitRate		= cpu_to_le32(1658880000),
	.dwMaxBitRate		= cpu_to_le32(1658880000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(17694720),
	.dwDefaultFrameInterval	= cpu_to_le32(800000),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(800000),
};

static const struct UVC_FRAME_MJPEG(1) uvc_frame_mjpg_2160p_30 = {
	.bLength		=   UVC_DT_FRAME_MJPEG_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_MJPEG,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(3840),
	.wHeight		= cpu_to_le16(2160),
	.dwMinBitRate		= cpu_to_le32(3981312000),
	.dwMaxBitRate		= cpu_to_le32(3981312000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(17694720),
	.dwDefaultFrameInterval	= cpu_to_le32(333333),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(333333),
};

static const struct UVC_FRAME_MJPEG(1) uvc_frame_mjpg_2160p_25 = {
	.bLength		=   UVC_DT_FRAME_MJPEG_SIZE(1),
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_FRAME_MJPEG,
	.bFrameIndex		= 1,
	.bmCapabilities		= 0,
	.wWidth			= cpu_to_le16(3840),
	.wHeight		= cpu_to_le16(2160),
	.dwMinBitRate		= cpu_to_le32(3317760000),
	.dwMaxBitRate		= cpu_to_le32(3317760000),
	.dwMaxVideoFrameBufferSize	= cpu_to_le32(17694720),
	.dwDefaultFrameInterval	= cpu_to_le32(400000),
	.bFrameIntervalType	= 1,
	.dwFrameInterval[0]	= cpu_to_le32(400000),
};

static const struct uvc_color_matching_descriptor uvc_color_matching = {
	.bLength		= UVC_DT_COLOR_MATCHING_SIZE,
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_COLORFORMAT,
	.bColorPrimaries	= 1,//BT.709, sRGB (default)
	.bTransferCharacteristics	= 1,//BT.709 (default)
	.bMatrixCoefficients	= 1,//BT. 709
};

static const struct uvc_color_matching_descriptor uvc_color_matching_ss = {
	.bLength		= UVC_DT_COLOR_MATCHING_SIZE,
	.bDescriptorType	= USB_DT_CS_INTERFACE,
	.bDescriptorSubType	= UVC_VS_COLORFORMAT,
	.bColorPrimaries	= 1,//BT.709, sRGB (default)
	.bTransferCharacteristics	= 1,//BT.709 (default)
	.bMatrixCoefficients	= 1,//BT. 709
};
static const struct uvc_descriptor_header * const uvc_fs_control_cls[] = {
	(const struct uvc_descriptor_header *) &uvc_control_header,
	(const struct uvc_descriptor_header *) &uvc_camera_terminal,
	(const struct uvc_descriptor_header *) &uvc_processing,
	(const struct uvc_descriptor_header *) &uvc_output_terminal,
	NULL,
};

static const struct uvc_descriptor_header * const uvc_ss_control_cls[] = {
	(const struct uvc_descriptor_header *) &uvc_control_header,
	(const struct uvc_descriptor_header *) &uvc_camera_terminal,
	(const struct uvc_descriptor_header *) &uvc_processing,
	(const struct uvc_descriptor_header *) &uvc_output_terminal,
	NULL,
};

static const struct uvc_descriptor_header * const uvc_default_streaming_cls[] = {
	(const struct uvc_descriptor_header *) &uvc_input_header,
	(const struct uvc_descriptor_header *) &uvc_format_mjpg,
	(const struct uvc_descriptor_header *) &uvc_frame_mjpg_720p_30,
	(const struct uvc_descriptor_header *) &uvc_color_matching,
	NULL,
};

static const struct uvc_descriptor_header * uvc_hs_streaming_cls[UVC_FORMAT_LENGTH];

static const struct uvc_descriptor_header * uvc_ss_streaming_cls[UVC_FORMAT_LENGTH];

#define AUDIO_VENDOR_NUM		0x1d6b	/* Linux Foundation */
#define AUDIO_PRODUCT_NUM		0x0101	/* Linux-USB Audio Gadget */

/*-------------------------------------------------------------------------*/

static struct usb_device_descriptor device_desc = {
	.bLength =		sizeof device_desc,
	.bDescriptorType =	USB_DT_DEVICE,

	.bcdUSB =		__constant_cpu_to_le16(0x200),

#ifdef CONFIG_GADGET_UAC1
	.bDeviceClass =		USB_CLASS_PER_INTERFACE,
	.bDeviceSubClass =	0,
	.bDeviceProtocol =	0,
#else
	.bDeviceClass =		USB_CLASS_MISC,
	.bDeviceSubClass =	0x02,
	.bDeviceProtocol =	0x01,
#endif
	/* .bMaxPacketSize0 = f(hardware) */

	/* Vendor and product id defaults change according to what configs
	 * we support.  (As does bNumConfigurations.)  These values can
	 * also be overridden by module parameters.
	 */
	.idVendor =		__constant_cpu_to_le16(AUDIO_VENDOR_NUM),
	.idProduct =		__constant_cpu_to_le16(AUDIO_PRODUCT_NUM),
	/* .bcdDevice = f(hardware) */
	/* .iManufacturer = DYNAMIC */
	/* .iProduct = DYNAMIC */
	/* NO SERIAL NUMBER */
	.bNumConfigurations =	0,
};


/* --------------------------------------------------------------------------
 * USB configuration
 */

static int audio_do_config(struct usb_configuration *c)
{
#ifndef USB_VIDEO_ONLY /* not combine uac */
	int status;

	/* FIXME alloc iConfiguration string, set it in c->strings */

#ifdef CONFIG_GADGET_UAC1
	f_uac1 = usb_get_function(fi_uac1);
	if (IS_ERR(f_uac1)) {
		status = PTR_ERR(f_uac1);
		return status;
	}

	status = usb_add_function(c, f_uac1);
	if (status < 0) {
		usb_put_function(f_uac1);
		return status;
	}
#else
	f_uac2 = usb_get_function(fi_uac2);
	if (IS_ERR(f_uac2)) {
		status = PTR_ERR(f_uac2);
		return status;
	}

	status = usb_add_function(c, f_uac2);
	if (status < 0) {
		usb_put_function(f_uac2);
		return status;
	}
#endif
#endif

	return 0;
}

static int
webcam_config_bind(struct usb_configuration *c)
{
	int status = 0;
	int ret;
	
    ret = audio_do_config(c);
	f_uvc = usb_get_function(fi_uvc);
	if (IS_ERR(f_uvc))
		return PTR_ERR(f_uvc);

	status = usb_add_function(c, f_uvc);
	if (status < 0)
		usb_put_function(f_uvc);

	return status;
}

static struct usb_configuration webcam_config_driver = {
	.label			= webcam_config_label,
	.bConfigurationValue	= 1,
	.iConfiguration		= 0, /* dynamic */
	.bmAttributes		= USB_CONFIG_ATT_SELFPOWER,
	.MaxPower		= CONFIG_USB_GADGET_VBUS_DRAW,
};

static int
setup_hs_descriptor(void)
{
	int index = 0;
	if(mjpg_frame_num == 0)
		goto error;
	
	uvc_hs_streaming_cls[index] = (const struct uvc_descriptor_header *) &uvc_input_header;
	index++;
	
	// MJPEG.
	if((mjpg_frame_num != 0) && (mjpg_frame_index[0] != -1)){
		uvc_hs_streaming_cls[index] = (const struct uvc_descriptor_header *) &uvc_format_mjpg;
		index++;
		
		// HS is Mjpeg 720 only.
		switch(mjpg_frame_index[0]){
		case UVC_FORMAT_FHD30P:
		case UVC_FORMAT_FHD60P:
		case UVC_FORMAT_4K30P:
		case UVC_FORMAT_4K15P:
		case UVC_FORMAT_HD720_30P:
			uvc_hs_streaming_cls[index] = (const struct uvc_descriptor_header *) &uvc_frame_mjpg_720p_30;

			index++;
			break;
		case UVC_FORMAT_FHD25P:
		case UVC_FORMAT_FHD50P:
		case UVC_FORMAT_4K25P:
		case UVC_FORMAT_4K12_5P:
		case UVC_FORMAT_HD720_25P:
			uvc_hs_streaming_cls[index] = (const struct uvc_descriptor_header *) &uvc_frame_mjpg_720p_25;
			index++;
			break;
		default:
			goto error;
		}
		uvc_hs_streaming_cls[index] = (const struct uvc_descriptor_header *) &uvc_color_matching;		
		index++;
		
		uvc_format_mjpg.bNumFrameDescriptors = mjpg_frame_num;
	}	
	uvc_hs_streaming_cls[index] = NULL;	
	return 0;
	
error:
	return -1;
}

static int
setup_ss_descriptor(void)
{
	int index = 0;
	if((yuv_frame_num == 0) && (mjpg_frame_num == 0))
		goto error;
	
	uvc_ss_streaming_cls[index] = (const struct uvc_descriptor_header *) &uvc_input_header_ss;
	
	// YUV
	if((yuv_frame_num != 0) && (yuv_frame_index[0] != -1)){
		int i;
		uvc_ss_streaming_cls[index + 1] = (const struct uvc_descriptor_header *) &uvc_format_yuv_ss;
		index++;
		
		for(i = 1; i <= yuv_frame_num; i++){
			switch(yuv_frame_index[i - 1]){
			case UVC_FORMAT_FHD30P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_yuv_1080p_30;
				break;
			case UVC_FORMAT_FHD60P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_yuv_1080p_60;
				break;
			case UVC_FORMAT_4K30P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_yuv_2160p_30;
				break;
			case UVC_FORMAT_4K15P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_yuv_2160p_15;
				break;
			case UVC_FORMAT_FHD25P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_yuv_1080p_25;
				break;
			case UVC_FORMAT_FHD50P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_yuv_1080p_50;
				break;
			case UVC_FORMAT_4K25P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_yuv_2160p_25;
				break;
			case UVC_FORMAT_4K12_5P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_yuv_2160p_12_5;
				break;
			default:
				goto error;
			}
		}
		
		uvc_ss_streaming_cls[index + yuv_frame_num + 1] = (const struct uvc_descriptor_header *) &uvc_color_matching_ss;
		index += yuv_frame_num + 1;
		
		uvc_format_yuv_ss.bNumFrameDescriptors = yuv_frame_num;
	}
	
	// MJPEG
	if((mjpg_frame_num != 0) && (mjpg_frame_index[0] != -1)){
		int i;
		uvc_ss_streaming_cls[index + 1] = (const struct uvc_descriptor_header *) &uvc_format_mjpg_ss;
		index++;
		
		for(i = 1; i <= mjpg_frame_num; i++){
			switch(mjpg_frame_index[i - 1]){
			case UVC_FORMAT_FHD30P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_mjpg_1080p_30;
				break;
			case UVC_FORMAT_FHD60P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_mjpg_1080p_60;
				break;
			case UVC_FORMAT_4K30P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_mjpg_2160p_30;
				break;
			case UVC_FORMAT_4K15P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_mjpg_2160p_15;
				break;
			case UVC_FORMAT_HD720_30P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_mjpg_720p_30;
				break;
			case UVC_FORMAT_FHD25P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_mjpg_1080p_25;
				break;
			case UVC_FORMAT_FHD50P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_mjpg_1080p_50;
				break;
			case UVC_FORMAT_4K25P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_mjpg_2160p_25;
				break;
			case UVC_FORMAT_4K12_5P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_mjpg_2160p_12_5;
				break;
			case UVC_FORMAT_HD720_25P:
				uvc_ss_streaming_cls[index + i] = (const struct uvc_descriptor_header *) &uvc_frame_mjpg_720p_25;
				break;
			default:
				goto error;
			}
		}

		uvc_ss_streaming_cls[index + mjpg_frame_num + 1] = (const struct uvc_descriptor_header *) &uvc_color_matching_ss;
		index += mjpg_frame_num + 1;
		
		uvc_format_mjpg_ss.bNumFrameDescriptors = mjpg_frame_num;
	}
	
	uvc_ss_streaming_cls[index + 1] = NULL;
	
	return 0;
	
error:
	return -1;
}


static int
webcam_unbind(struct usb_composite_dev *cdev)
{
	if (!IS_ERR_OR_NULL(f_uvc))
		usb_put_function(f_uvc);
	if (!IS_ERR_OR_NULL(fi_uvc))
		usb_put_function_instance(fi_uvc);

#ifndef USB_VIDEO_ONLY /* not combine uac */
#ifdef CONFIG_GADGET_UAC1
	if (!IS_ERR_OR_NULL(f_uac1))
		usb_put_function(f_uac1);
	if (!IS_ERR_OR_NULL(fi_uac1))
		usb_put_function_instance(fi_uac1);
#else
	if (!IS_ERR_OR_NULL(f_uac2))
		usb_put_function(f_uac2);
	if (!IS_ERR_OR_NULL(fi_uac2))
		usb_put_function_instance(fi_uac2);
#endif
#endif
	return 0;
}

static int
webcam_bind(struct usb_composite_dev *cdev)
{
	struct f_uvc_opts *uvc_opts;
	int ret;
#ifndef USB_VIDEO_ONLY /* not combine uac */
/*audio bind*/
#ifndef CONFIG_GADGET_UAC1
	struct f_uac2_opts	*uac2_opts;
#else
	struct f_uac1_opts	*uac1_opts;
#endif
#endif

	fi_uvc = usb_get_function_instance("uvc");
	if (IS_ERR(fi_uvc))
		return PTR_ERR(fi_uvc);

	uvc_opts = container_of(fi_uvc, struct f_uvc_opts, func_inst);

	uvc_opts->streaming_interval = streaming_interval;
	uvc_opts->streaming_maxpacket = streaming_maxpacket;
	uvc_opts->streaming_maxburst = streaming_maxburst;

	uvc_opts->fs_control = uvc_fs_control_cls;
	uvc_opts->ss_control = uvc_ss_control_cls;
	uvc_opts->fs_streaming = uvc_default_streaming_cls;

	if (setup_hs_descriptor() < 0){
		uvc_opts->hs_streaming = uvc_default_streaming_cls;
	}
	else{
		uvc_opts->hs_streaming = uvc_hs_streaming_cls;
	}

	if (setup_ss_descriptor() < 0){
		uvc_opts->ss_streaming = uvc_default_streaming_cls;
	}
	else{
		uvc_opts->ss_streaming = uvc_ss_streaming_cls;
	}

#ifndef USB_VIDEO_ONLY /* not combine uac */
#ifndef CONFIG_GADGET_UAC1
	fi_uac2 = usb_get_function_instance("uac2");
	if (IS_ERR(fi_uac2))
		return PTR_ERR(fi_uac2);
#else
	fi_uac1 = usb_get_function_instance("uac1");
	if (IS_ERR(fi_uac1))
		return PTR_ERR(fi_uac1);
#endif

#ifndef CONFIG_GADGET_UAC1
	uac2_opts = container_of(fi_uac2, struct f_uac2_opts, func_inst);
	uac2_opts->p_chmask = p_chmask;
	uac2_opts->p_srate = p_srate;
	uac2_opts->p_ssize = p_ssize;
	uac2_opts->c_chmask = c_chmask;
	uac2_opts->c_srate = c_srate;
	uac2_opts->c_ssize = c_ssize;
#else
	uac1_opts = container_of(fi_uac1, struct f_uac1_opts, func_inst);
	uac1_opts->fn_play = fn_play;
	uac1_opts->fn_cap = fn_cap;
	uac1_opts->fn_cntl = fn_cntl;
	uac1_opts->req_buf_size = req_buf_size;
	uac1_opts->req_count = req_count;
	uac1_opts->audio_buf_size = audio_buf_size;
#endif
#endif

	/* Allocate string descriptor numbers ... note that string contents
	 * can be overridden by the composite_dev glue.
	 */
	ret = usb_string_ids_tab(cdev, webcam_strings);
	if (ret < 0)
		goto error;
	webcam_device_descriptor.iManufacturer =
		webcam_strings[USB_GADGET_MANUFACTURER_IDX].id;
	webcam_device_descriptor.iProduct =
		webcam_strings[USB_GADGET_PRODUCT_IDX].id;
	webcam_device_descriptor.iSerialNumber =
		webcam_strings[USB_GADGET_SERIAL_IDX].id;
/* Not use configuration name.
	webcam_config_driver.iConfiguration =
		webcam_strings[STRING_DESCRIPTION_IDX].id;
*/

	gadget_pfs_overwrite_desc_device(&webcam_device_descriptor);

	/* Register our configuration. */
	if ((ret = usb_add_config(cdev, &webcam_config_driver,
					webcam_config_bind)) < 0)
		goto error;

	usb_composite_overwrite_options(cdev, &coverwrite);
	gadget_pfs_overwrite_desc_string(webcam_strings);
	INFO(cdev, "Webcam Video Gadget\n");
	return 0;

error:
	usb_put_function_instance(fi_uvc);
	return ret;
}

/* --------------------------------------------------------------------------
 * Driver
 */

static struct usb_composite_driver webcam_driver = {
	.name		= "g_webcam",
	.dev		= &webcam_device_descriptor,
	.strings	= webcam_device_strings,
	.max_speed	= USB_SPEED_SUPER,
	.bind		= webcam_bind,
	.unbind		= webcam_unbind,
};

module_usb_composite_driver(webcam_driver);

MODULE_AUTHOR("Laurent Pinchart");
MODULE_DESCRIPTION("Webcam Video Gadget");
MODULE_LICENSE("GPL");

