
/*******************************************************************************//**
 * @file    usbg_buspower_cmn.c
 * @brief   USB Gadget BusPower Driver - common
 * 
 * Copyright 2011 Sony Corporation
 * Copyright 2018 Sony Imaging Products and Solutions Incorporated.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 *******************************************************************************/

/*=============================================================================
/  Includes
/==============================================================================*/

#include <asm/errno.h>
#include <asm/unaligned.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/version.h>

#include <linux/usb/ch9.h>
#include <linux/device.h>
#include <linux/usb/gadget.h>

#include <linux/usb/gcore/usb_event.h>
#include <linux/usb/gcore/usb_gadgetcore.h>

#include <linux/usb/specific_gadget/usbg_cmn.h>
#include <linux/usb/specific_gadget/usbg_type.h>
#include <linux/usb/specific_gadget/usbg_still.h>
#include <linux/usb/specific_gadget/usb_kdbg.h>

#define FNAME "usbg_buspower_cmn.c"
#include "usbg_buspower_pvt.h"


/*=============================================================================
/  Enumerations
/==============================================================================*/


/*=============================================================================
/  Structs
/==============================================================================*/


/*=============================================================================
/  Prototypes
/==============================================================================*/


/*=============================================================================
/  Variable Declarations
/==============================================================================*/

/**
 *    @brief        USB Gadget BusPower driver instance
 */
static struct func_data *usbg_buspower_this = NULL;


/*=============================================================================
/  Functions
/==============================================================================*/

/**
 *    @brief        Create driver instance
 *    @retval       0            success
 *    @retval       -ENOMEM      short of memory
 */
int usbg_buspower_create_this(void)
{
    
    struct func_data*    p_this;
    struct func_data_ex* p_this_ex;
    int ret = 0;
    
    USBG_BUSPWR_PDEBUG( "Calling: %s() ", __FUNCTION__ );
    
    if( usbg_buspower_this != NULL ) {
        
        // Instance is already created
        // Nothint to do
        USBG_BUSPWR_PDEBUG( "%s(): Instance is already created.", __FUNCTION__ );
        goto SUB_RET;
        
    }
    
    // Alloc driver instance area
    p_this = kmalloc( sizeof(struct func_data), GFP_ATOMIC );
    
    if( !p_this ) {
        
        USBG_BUSPWR_PERR( "%s(): Could not create p_this instance.", __FUNCTION__ );
        ret = -ENOMEM;
        goto SUB_RET;
        
    }
    
    // Alloc driver instance area ( Driver extention )
    p_this_ex = kmalloc( sizeof(struct func_data_ex), GFP_ATOMIC );
    
    if( !p_this_ex ) {
        
        USBG_BUSPWR_PERR( "%s(): Could not create p_this_ex instance.", __FUNCTION__ );
        kfree( p_this );
        ret = -ENOMEM;
        goto SUB_RET;
        
    }
    
    // Fill with "0"
    memset( p_this,    0, sizeof( struct func_data )   );
    memset( p_this_ex, 0, sizeof( struct func_data_ex ));

    // new usb request stop flag
    p_this_ex->stop_usb_request = false;

    // Link informations this->this_ex
    p_this->vp_user = p_this_ex;
    usbg_buspower_this = p_this;
    
SUB_RET:
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s() , return val is %d ", __FUNCTION__, ret );
    
    return ret;
    
}

/**
 *    @brief        Free driver instance
 */
void usbg_buspower_delete_this( void )
{
    
    struct func_data*    p_this;
    struct func_data_ex* p_this_ex;
    
    USBG_BUSPWR_PDEBUG( "Calling: %s() ", __FUNCTION__ );
    
    p_this    = usbg_buspower_get_this();
    p_this_ex = usbg_buspower_get_this_ex();
    
    // free driver informations
    if( p_this_ex ) {
        
        kfree( p_this_ex );
        p_this_ex = NULL;
        
    }
    
    if( p_this ) {
        
        kfree( p_this );
        p_this = NULL;
        
    }
    
    usbg_buspower_this = NULL;
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
    return;
    
}

/**
 *    @brief        Get driver instance
 *    @retval       Driver instance
 */
struct func_data *usbg_buspower_get_this( void )
{
    
    USBG_BUSPWR_PDEBUG( "Calling: %s()", __FUNCTION__ );
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
    return usbg_buspower_this;
    
}

/**
 *    @brief        Get driver extention instance
 *    @retval       Driver extention instance
 */
struct func_data_ex *usbg_buspower_get_this_ex( void )
{
    
    struct func_data_ex* pRet = NULL;
    
    USBG_BUSPWR_PDEBUG( "Calling: %s() ", __FUNCTION__ );
    
    if( !usbg_buspower_this ) {
        
        goto SUB_RET;
        
    }
    
    pRet = (struct func_data_ex *)usbg_buspower_this->vp_user;
    
SUB_RET:
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
    return pRet;
    
}

/**
 *    @brief        Link driver instance to struct file
 *    @param        file    struct file
 */
void usbg_buspower_link_this_to_file( struct file *file )
{
    
    USBG_BUSPWR_PDEBUG( "Calling: %s() ", __FUNCTION__ );
    
    if( file ) {
        
        file->private_data = usbg_buspower_this;
        
    }
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
}

/**
 *    @brief        Put U8 data
 *    @param        p        address
 *    @param        val      data
 */
void usbg_buspower_put_u8( u8 **p, u8 val )
{
    
    USBG_BUSPWR_PDEBUG( "Calling: %s() ", __FUNCTION__ );
    
    *(u8 *)(*p) = val;
    (*p) += 1;
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
}

/**
 *    @brief        Call user area callback function
 *    @param        cb_id    callback ID
 *    @param        evt_id   event ID
 *    @param        arg      arguments
 *    @param        size     argument size
 */
void usbg_buspower_call_user_cb( int cb_id, int evt_id, void *arg, int size )
{
    
    struct func_data *p_this;
    void *user_cb;
    usb_hndl_t handle;
    
    USBG_BUSPWR_PDEBUG( "Calling: %s() ", __FUNCTION__ );
    
    p_this = usbg_buspower_get_this();
    handle = (usb_hndl_t)p_this->info.hndl;
    
    switch( cb_id ) {
        case USBG_BUSPWR_CBID_START:
            user_cb = p_this->info.event.start;
            break;
        case USBG_BUSPWR_CBID_STOP:
            user_cb = p_this->info.event.stop;
            break;
        case USBG_BUSPWR_CBID_EP_SET_HALT:
            user_cb = p_this->info.event.ep_set_halt;
            break;
        case USBG_BUSPWR_CBID_EP_CLR_HALT:
            user_cb = p_this->info.event.ep_clear_halt;
            break;
        case USBG_BUSPWR_CBID_SUSPEND:
            user_cb = p_this->info.event.suspend;
            break;
        case USBG_BUSPWR_CBID_RESUME:
            user_cb = p_this->info.event.resume;
            break;
        case USBG_BUSPWR_CBID_CLASS:
            user_cb = p_this->info.event.class_req;
            break;
        case USBG_BUSPWR_CBID_VENDOR:
            user_cb = p_this->info.event.vendor_req;
            break;
        default:
            USBG_BUSPWR_PDEBUG( "%s(): Unknown Callback ID.", __FUNCTION__ );
            return;
    }
    
    usb_event_add_queue( USB_EVENT_PRI_NORMAL,
            user_cb, handle, evt_id, size, arg );
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
}

/**
 *    @brief        Get ep0 buffer
 *    @retval       ep0 buffer address
 */
void *usbg_buspower_get_ep0_buffer(void)
{
    
    struct func_data_ex* p_this_ex = usbg_buspower_get_this_ex();
    void *pRet = NULL;
    
    USBG_BUSPWR_PDEBUG( "Calling: %s() ", __FUNCTION__ );
    
    if( p_this_ex ) {
        
        pRet = p_this_ex->ep0_buffer;
        
    }
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
    return pRet;
    
}

/**
 *    @brief        Get ep struct
 *    @param        ep_id    Endpoint ID
 *    @retval       struct ep
 */
struct usb_ep *usbg_buspower_get_ep( int ep_id )
{
    
    struct usb_ep *pRet = NULL;
    
    USBG_BUSPWR_PDEBUG( "Calling: %s() ", __FUNCTION__ );
    
    if( ep_id >= USBG_BUSPWR_EP_IDX_MAX ) {
        
        // invalid EP no.
        USBG_BUSPWR_PERR( "%s(): Invalid Endpoint No. ep_id=%d", __FUNCTION__, ep_id );
        goto SUB_RET;
        
    }
    
    if( !usbg_buspower_this || !usbg_buspower_this->ptbl_ep ) {
        
        USBG_BUSPWR_PERR( "%s(): p_this->ptbl_ep==NULL.", __FUNCTION__ );
        goto SUB_RET;
        
    }
    
    pRet = usbg_buspower_this->ptbl_ep[ ep_id ];
    
SUB_RET:
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
    return pRet;
    
}

/**
 *    @brief        Get ep information
 *    @param        ep    struct ep
 *    @retval       ep information
 */
struct usbg_buspower_ep_info *usbg_buspower_get_ep_info(struct usb_ep *ep)
{
    
    struct usbg_buspower_ep_info *pRet = NULL;
    
    USBG_BUSPWR_PDEBUG( "Calling: %s() ", __FUNCTION__ );
    
    if( !ep ){
        
        goto SUB_RET;
        
    }
    
    pRet = (struct usbg_buspower_ep_info*)ep->driver_data;
   
SUB_RET:
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
    return pRet;
    
}

/**
 *    @brief        Dequeue request
 *    @param        ep        struct ep
 */
void usbg_buspower_ep_dequeue( struct usb_ep *ep )
{
    
    struct usbg_buspower_ep_info *ep_inf = usbg_buspower_get_ep_info( ep );
    
    USBG_BUSPWR_PDEBUG( "Calling: %s() ", __FUNCTION__ );
    
    USBG_BUSPWR_REQUESTING_LOCK_ON();
    
    if( ep_inf && ep_inf->req && ep_inf->requesting ) {
        
        USBG_BUSPWR_REQUESTING_LOCK_OFF();
        USBG_BUSPWR_PDEBUG( "%s(): request!=NULL.", __FUNCTION__ );
        usb_ep_dequeue( ep, ep_inf->req);
        
    }else{
        
        USBG_BUSPWR_REQUESTING_LOCK_OFF();
        USBG_BUSPWR_PDEBUG( "%s(): request==NULL.", __FUNCTION__ );
        
    }
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
    return;
    
}

/**
 *    @brief        Dequeue request from all ep
 */
void usbg_buspower_ep_dequeue_all( void )
{
    
    int i;
    struct usb_ep **ep_list;
    
    USBG_BUSPWR_PDEBUG( "Calling: %s() ", __FUNCTION__ );
    
    if( !usbg_buspower_this ) {
        
        // no driver instance
        return;
        
    }
    
    ep_list = usbg_buspower_this->ptbl_ep;
    
    if( !ep_list ) {
        
        // no ep table
        return;
        
    }
    
    for( i = 0; i < USBG_BUSPWR_EP_IDX_MAX; i++ ) {
        
        if( ep_list[i] ) {
            
            usbg_buspower_ep_dequeue( ep_list[i] );
            
        }
        
    }
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
    return;
    
}

/**
 *    @brief        Requesting?
 *    @param        ep_id    Endpoint ID
 */
int  usbg_buspower_is_requesting( int ep_id )
{
    
    int ret = FALSE;
    struct func_data_ex *p_this_ex = usbg_buspower_get_this_ex();
    
    USBG_BUSPWR_PDEBUG( "Calling: %s() ", __FUNCTION__ );
    
    if( !p_this_ex || !( p_this_ex->ep_info ) ) {
        
        USBG_BUSPWR_PDEBUG( "%s(): p_this_ex->ep_info is NULL.", __FUNCTION__ );
        
    }else{
        
        USBG_BUSPWR_REQUESTING_LOCK_ON();
        
        if( p_this_ex->ep_info[ ep_id ].requesting ){
            
            USBG_BUSPWR_REQUESTING_LOCK_OFF();
            
            ret = TRUE;
            goto SUB_RET;
            
        }
        
        USBG_BUSPWR_REQUESTING_LOCK_OFF();
        
    }
    
SUB_RET:
    USBG_BUSPWR_PDEBUG( "Leaving: %s() , return val is %d ", __FUNCTION__, ret );
    
    return ret;
    
}

/**
 *    @brief        Get usb_request area
 *    @retval       usb_request address
 */
struct usb_request *usbg_buspower_get_request( struct usb_ep *ep )
{
    
    struct usb_request *pRet = NULL;
    struct usbg_buspower_ep_info *ep_inf = usbg_buspower_get_ep_info( ep );
    
    USBG_BUSPWR_PDEBUG( "Calling: %s() ", __FUNCTION__ );
    
    if( ep_inf ) {
        
        pRet = ep_inf->req;
        
    }else{
        
        USBG_BUSPWR_PDEBUG( "%s(): request==NULL.", __FUNCTION__ );
        
    }
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
    return pRet;
    
}

/**
 *    @brief        Get event buffer
 *    @retval       Data buffer address
 */
void *usbg_buspower_get_data_buffer( void )
{
    
    void *pRet = NULL;
    struct func_data_ex* p_this_ex = usbg_buspower_get_this_ex();
    
    USBG_BUSPWR_PDEBUG( "Calling: %s() ", __FUNCTION__ );
    
    if( p_this_ex ) {
        
        pRet = p_this_ex->data_buffer;
        
    }
    
    USBG_BUSPWR_PDEBUG( "Leaving: %s()", __FUNCTION__ );
    
    return pRet;
    
}

/**
 *    @brief        Set requesting flag
 *    @param        ep_id    Endpoint ID
 *    @param        flag     requesting flag to be set. TRUE or FALSE.
 */
void usbg_buspower_set_requesting_flag( int ep_id, int flag )
{
    
    struct func_data_ex *p_this_ex = usbg_buspower_get_this_ex();
    
    if( !p_this_ex || !( p_this_ex->ep_info ) ) {
        
        USBG_BUSPWR_PDEBUG( "%s(): p_this_ex->ep_info is NULL.", __FUNCTION__ );
        
    }else{
        
        if( ( flag == TRUE ) && usbg_buspower_is_requesting( ep_id ) ){
            
            USBG_BUSPWR_PDEBUG( "%s(): request already queued.", __FUNCTION__ );
            
        }
        
        USBG_BUSPWR_REQUESTING_LOCK_ON();
        
        p_this_ex->ep_info[ ep_id ].requesting = flag;
        
        USBG_BUSPWR_REQUESTING_LOCK_OFF();
        
    }
    
}

/**
 *    @brief        Start request
 *    @param        ep_id    Endpoint ID
 */
void usbg_buspower_request_start( int ep_id )
{
    
    usbg_buspower_set_requesting_flag( ep_id, TRUE );
    
}

/**
 *    @brief        Stop request
 *    @param        ep_id    Endpoint ID
 */
void usbg_buspower_request_stop( int ep_id )
{
    
    usbg_buspower_set_requesting_flag( ep_id, FALSE );
    
}

/**
 *    @brief        new ep_queue allowed check
 *    @retval       TRUE    ep_queue not allowed
 *    @retval       FALSE   ep_queue allowed
 */
bool  usbg_buspower_is_stop_usb_reqest(void)
{
    struct func_data_ex *p_this_ex = usbg_buspower_get_this_ex();

    return p_this_ex->stop_usb_request;
}
